#nullable enable

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using HarmonyLib;
using Verse;
using RimWorld;
using rjw.Modules.Shared.Logs;
using static RimWorld.LovePartnerRelationUtility;

using ReservationManager = Verse.AI.ReservationManager;
using Reservation = Verse.AI.ReservationManager.Reservation;
using AllowedSex = rjw.RJWPreferenceSettings.AllowedSex;

using Benchy = rjw.Modules.Benchmark.Benchy;
using LudeonTK;

namespace rjw.Modules.Attraction
{
	[StaticConstructorOnStartup]
	public static class AttractionUtility
	{
		// TODO list:
		// Add insectoid specific preferences.
		//   After seeing a number of weird inconsistencies in the whole insect pregnancy
		//   system, I will probably hold off on this until I've done a refactor to that
		//   code to smooth those out.
		// Make changes to the `RaceGroupDef` helper.
		//   In particular, humans with the furskin gene need the `Fur` tag.
		//   Putting this off too.  It seems there's some API issues that shoehorn it
		//     into only working with `PawnKindDef`.  The public API would need to work
		//     only off a `Pawn` instead, since `PawnKindDef` does not contain concrete
		//     information on the genes a pawn has (only what they COULD have).  This
		//     would make it a breaking change.
		// Change hookup and random-rape logic to use weighted random selection.
		//   This is a few methods:
		//     `FindMates` - A generator yielding pawns for fucking.
		//     `FindVictims` - A generator yielding pawns for rape.
		//     `FindRapists` - A generator yielding pawns that would rape the victim.
		//     `FindMate`, `FindVictim`, and `FindRapist` which basically call the above
		//       but with `.FirstOrDefault()` afterwards.
		// Move attraction curves into a config def.
		// Some day, make a quirk for Nymphomaniac, since it has a lot of special cases
		//   built into the attraction system that would be better handled in the same
		//   way we handle Zoophile and Necrophile.
		// Remove `SexAppraiser_OLD`.

		static AttractionUtility()
		{
			Type AP = typeof(AttractionPreference);
			StandardPreferenceApplicators = GenTypes.AllTypes
				.Where(t => !t.IsAbstract && AP.IsAssignableFrom(t))
				.SelectMany(t => t.GetMethods(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic))
				.Where(mi => mi.HasAttribute<StandardPreferenceAttribute>())
				.Select(ToDelegate)
				.OfType<ApplyPreference>()
				.ToArray();

			// Reports problems with marked applicators without crashing the game.
			static ApplyPreference? ToDelegate(MethodInfo mi)
			{
				try
				{
					return (ApplyPreference)Delegate.CreateDelegate(typeof(ApplyPreference), mi);
				}
				catch (Exception e)
				{
					var name = $"{mi.DeclaringType.FullName}.{mi.Name}";
					Log.Error($"Failed to create `ApplyPreference` delegate for: {name}", e);
					return null;
				}
			}
		}

		private static readonly ILog Log = LogManager.GetLogger(nameof(AttractionUtility));

		/// <summary>
		/// Curve that maps a pawn's `Need_Sex.CurLevel` to a scalar to apply.
		/// </summary>
		private static readonly SimpleCurve needSexCurve = new()
		{
			// Frustrated...
			new(0.0f, 1.6f),
			new(0.05f, 1.6f),
			// Horny...
			new(0.25f, 1.3f),
			// Neutral...
			new(0.5f, 1.1f),
			// Satisfied...
			new(0.75f, 1f),
			new(1f, 1f)
		};

		/// <summary>
		/// Curve that maps the percentage-toward-max-reservations on a pawn to a
		/// scalar to apply.
		/// </summary>
		static readonly SimpleCurve reservationsCurve = new()
		{
			new(0f, 1f),
			new(0.3f, 0.4f),
			new(1f, 0f)
		};

		/// <summary>
		/// <para>When `Evaluate` is called without any modes provided, this is used.</para>
		/// <para>This ordering should hit the major "no-go" checks the quickest, allowing
		/// an early abort and saving some processing time.</para>
		/// </summary>
		private static readonly AttractionMode[] defaultModes = new[]
		{
			AttractionMode.Romantic,
			AttractionMode.Physical,
			AttractionMode.Social,
			AttractionMode.Age
		};

		/// <summary>
		/// Wildmen have a hardcoded wildness of 75%; `Pawn.SpecialDisplayStats`
		/// adds wildness as a special case in RimWorld's source.
		/// </summary>
		public const float WildManWildness = 0.75f;

		/// <summary>
		/// <para>This delegate used for registering standard preferences for automatic
		/// application at the start of a request.</para>
		/// <para>Create a static method that matches this delegate and apply the
		/// <see cref="StandardPreferenceAttribute" /> to it, and that method will
		/// be registered into the system automatically.</para>
		/// <para>In your applicator method, just do any neccessary checks and then
		/// either add it to the request or don't.  Simple!</para>
		/// </summary>
		/// <param name="request">The request.</param>
		public delegate void ApplyPreference(ref AttractionRequest request);

		/// <summary>
		/// A list of functions registered to apply standard pawn preferences.
		/// </summary>
		private static readonly ApplyPreference[] StandardPreferenceApplicators;

		/// <summary>
		/// The number of standard applicators registered.  This acts as a decent
		/// value to initialize the capacity for a request's preferences.
		/// </summary>
		public static int StandardApplicatorsCount => StandardPreferenceApplicators.Length;

		/// <summary>
		/// <para>Evaluates the attraction between two pawns in a general sense.</para>
		/// <para>If you're interested in using the attraction for sex, you should
		/// provide the related purpose.</para>
		/// </summary>
		/// <param name="pawn">The pawn observing.</param>
		/// <param name="target">The pawn being observed.</param>
		/// <returns>The attraction weight.</returns>
		public static float Evaluate(Pawn pawn, Pawn target)
		{
			var request = new AttractionRequest(pawn, target);
			return Evaluate(ref request, defaultModes);
		}

		/// <summary>
		/// Evaluates the attraction between two pawns for the given purpose.
		/// </summary>
		/// <param name="pawn">The pawn observing.</param>
		/// <param name="target">The pawn being observed.</param>
		/// <param name="purpose">The ultimate purpose of the request.</param>
		/// <returns>The attraction weight.</returns>
		public static float Evaluate(Pawn pawn, Pawn target, AttractionPurpose purpose)
		{
			var request = new AttractionRequest(purpose, pawn, target);
			return Evaluate(ref request, defaultModes);
		}

		/// <summary>
		/// <para>Evaluates an attraction request, producing a weight that indicates
		/// how much the request's pawn is into the target.</para>
		/// <para>This variant runs through all possible "attraction modes".</para>
		/// </summary>
		/// <param name="request">The request to evaluate.</param>
		/// <returns>The attraction weight.</returns>
		public static float Evaluate(ref AttractionRequest request) =>
			Evaluate(ref request, defaultModes);

		/// <summary>
		/// <para>Evaluates an attraction request, producing a weight that indicates
		/// how much the request's pawn is into the target.</para>
		/// <para>This variant allows you to request only specific "attraction modes"
		/// if you're only interested in certain factors.</para>
		/// </summary>
		/// <param name="request">The request to evaluate.</param>
		/// <param name="modes">The modes to run through.</param>
		/// <returns>The attraction weight.</returns>
		public static float Evaluate(ref AttractionRequest request, params AttractionMode[] modes)
		{
			if (modes.Length == 0) return 0f;
			if (!PreflightChecks(ref request)) return 0f;

			SetupRequest(ref request);

			var factor = 1f;
			using (var evaluator = request.GetEvaluator())
			{
				foreach (var mode in modes)
				{
					factor *= evaluator.Evaluate(mode, ref request);
					// Abort early if 0 or less.
					if (factor <= 0f) return 0f;
				}
			}

			return Finalize(ref request, factor);
		}

		/// <summary>
		/// Sets up the preferences for this request.
		/// </summary>
		/// <param name="request">The request to setup.</param>
		private static void SetupRequest(ref AttractionRequest request)
		{
			// Allow the standard preferences to set themselves up for this request.
			foreach (var applyTo in StandardPreferenceApplicators)
				applyTo(ref request);
		}

		/// <summary>
		/// Handles various preflight checks to ensure the request does not permit
		/// something it should not be permitting.
		/// </summary>
		/// <param name="request">The current request.</param>
		/// <returns>Whether the pawns are cleared for takeoff.</returns>
		private static bool PreflightChecks(ref AttractionRequest request)
		{
			// All this is currently only for sexual interactions.
			if (request.Purpose.IsNotForSex())
				return true;

			var pawn = request.Pawn;
			var target = request.Target;

			// Must be reachable.
			if (pawn.Dead || pawn.Suspended || target.Suspended)
				return false;

			// Bestiality needs to be enabled.
			if (!RJWSettings.bestiality_enabled && request.Category.IsBestiality())
				return false;

			// Necrophilia needs to be enabled.
			if (!RJWSettings.necrophilia_enabled && target.Dead)
				return false;

			// Must not violate the user's gender settings.
			if (!IsGenderOk(pawn, target))
				return false;

			// Must be old enough to fuck.
			if (!IsAgeOk(pawn, target))
				return false;

			// Not healthy enough to have sex (unless the pawn doesn't care).
			if (!pawn.IsAnimal() && !xxx.is_healthy_enough(pawn))
				if (!xxx.is_psychopath(pawn))
					return false;

			// Don't keep a hungry pawn from sustenance (unless the pawn doesn't care).
			if (xxx.is_starved(target) && target.Faction == pawn.Faction)
				if (!xxx.is_psychopath(pawn) && !xxx.is_rapist(pawn))
					return false;

			// No sex while on death's door (unless the pawn doesn't care).
			if (!request.ignoreBleeding && xxx.is_human(pawn) && !xxx.is_not_dying(target))
				if (!xxx.is_psychopath(pawn) && !xxx.is_rapist(pawn) && !xxx.is_bloodlust(pawn))
					return false;

			return true;
		}

		/// <summary>
		/// <para>Checks to see if two pawns are okay to fuck, based on age.</para>
		/// <para>TODO: this should be made private once `SexAppraiser.would_fuck`
		/// is no longer in wide-spread use.</para>
		/// </summary>
		/// <param name="pawn">The first pawn.</param>
		/// <param name="target">The second pawn.</param>
		/// <returns>Whether they're good to go.</returns>
		internal static bool IsAgeOk(Pawn pawn, Pawn target)
		{
			return AgeOkFor(pawn) && AgeOkFor(target);

			static bool AgeOkFor(Pawn pawn)
			{
				if (xxx.is_mechanoid(pawn) || xxx.can_do_loving(pawn))
					return true;
				if (xxx.is_human(pawn))
					return pawn.ageTracker.Growth >= 1f;
				if (pawn.GetRJWPawnData()?.RaceSupportDef?.adultAge is int adultAge && adultAge != 0)
					return pawn.ageTracker.AgeBiologicalYearsFloat >= adultAge;
				return false;
			}
		}

		/// <summary>
		/// <para>Check's RJW's settings regarding gender limitations.</para>
		/// <para>This only looks at genders, not their actual sex-parts.</para>
		/// </summary>
		/// <param name="pawn">The pawn observing.</param>
		/// <param name="target">The observed pawn.</param>
		/// <returns>Whether the pairing does not violate any rules.</returns>
		internal static bool IsGenderOk(Pawn pawn, Pawn target) =>
			(pawn.gender, target.gender) switch
			{
				(Gender.Male, Gender.Male) =>
					RJWPreferenceSettings.Malesex != AllowedSex.Nohomo,
				(Gender.Male, Gender.Female) =>
					RJWPreferenceSettings.Malesex != AllowedSex.Homo,
				(Gender.Female, Gender.Female) =>
					RJWPreferenceSettings.FeMalesex!= AllowedSex.Nohomo,
				(Gender.Female, Gender.Male) =>
					RJWPreferenceSettings.FeMalesex != AllowedSex.Homo,
				_ => true
			};
		
		internal static float Finalize(ref AttractionRequest request, float factor)
		{
			if (factor <= 0f) return 0f;
			if (!request.finalize) return factor;

			var pawn = request.Pawn;
			var target = request.Target;

			if (request.Purpose.IsForSex())
			{
				// Anything without the need (mostly animals) is treated as neutral horniness.
				var curLevel = pawn.needs.TryGetNeed<Need_Sex>()?.CurLevel ?? 0.5f;
				factor *= needSexCurve.Evaluate(curLevel);

				// Bias the factor if they're nymphs or have taken hump shroom.
				if (xxx.is_nympho(pawn) || pawn.health.hediffSet.HasHediff(RJWHediffDefOf.HumpShroomEffect))
					factor = 0.2f + 0.8f * factor;

				// Reduce the factor based on number of other pawns reserving the target.
				var reservedCount = target.Dead ? 1f : ReservedCount(target);
				var reservedPercentage = reservedCount / xxx.max_rapists_per_prisoner;
				factor *= reservationsCurve.Evaluate(reservedPercentage);
			}

			return factor;
		}

		/// <summary>
		/// Normalizes an attraction weight between 0 and 1, making the weight look more
		/// like a legacy `would_fuck` value.
		/// </summary>
		/// <param name="weight">The weight to normalize.</param>
		/// <returns>The normalized weight.</returns>
		public static float Normalize(float weight)
		{
			if (weight <= 1f) return weight * 0.5f;
			return GenMath.LerpDouble(0f, 1f, 1f, 0.5f, 1f / weight);
		}

		/// <summary>
		/// Attempts to the standardize the body of a pawn.
		/// </summary>
		/// <param name="pawn">The pawn whose body to standardize.</param>
		/// <returns>The standardized body.</returns>
		public static StandardizedBody StandardizeBody(Pawn pawn)
		{
			if (pawn.story?.bodyType is { } bodyType)
			{
				// Fast direct match to vanilla defs.
				if (bodyType == BodyTypeDefOf.Baby) return StandardizedBody.Baby;
				if (bodyType == BodyTypeDefOf.Child) return StandardizedBody.Child;
				if (bodyType == BodyTypeDefOf.Female) return StandardizedBody.Female;
				if (bodyType == BodyTypeDefOf.Male) return StandardizedBody.Male;
				if (bodyType == BodyTypeDefOf.Thin) return StandardizedBody.Thin;
				if (bodyType == BodyTypeDefOf.Hulk) return StandardizedBody.Hulk;
				if (bodyType == BodyTypeDefOf.Fat) return StandardizedBody.Fat;

				// Heuristic match to a standard achetype.
				var lowerName = bodyType.defName.ToLower();
				if (lowerName.Contains("baby")) return StandardizedBody.Baby;
				if (lowerName.Contains("child")) return StandardizedBody.Child;
				if (lowerName.Contains("girl")) return StandardizedBody.Child;
				if (lowerName.Contains("boy")) return StandardizedBody.Child;
				if (lowerName.Contains("female")) return StandardizedBody.Female;
				if (lowerName.Contains("male")) return StandardizedBody.Male;
				if (lowerName.Contains("thin")) return StandardizedBody.Thin;
				if (lowerName.Contains("hulk")) return StandardizedBody.Hulk;
				if (lowerName.Contains("fat")) return StandardizedBody.Fat;
			}

			if (xxx.is_mechanoid(pawn)) return StandardizedBody.Mechanoid;
			if (xxx.is_insect(pawn)) return StandardizedBody.Insectoid;
			if (pawn.IsAnimal()) return StandardizedBody.Animal;
			return StandardizedBody.Unknown;
		}

		/// <summary>
		/// Gets the status of this pawn's relation in context of the target.
		/// </summary>
		/// <param name="pawn">The pawn.</param>
		/// <param name="target">The other pawn.</param>
		/// <returns>A value indicating the status of the relationship.</returns>
		public static AttractionStatus GetStatus(Pawn pawn, Pawn target)
		{
			var spouses = pawn.GetSpouses(false);
			if (spouses.Count == 0) return AttractionStatus.Open;
			if (spouses.Contains(target)) return AttractionStatus.Marital;
			return AttractionStatus.Extramarital;
		}

		/// <summary>
		/// <para>Checks to see if the request's pawn would be willing to fool around.</para>
		/// <para>This assumes the pawn has already been checked as being in a relationship.</para>
		/// </summary>
		/// <param name="request">The request.</param>
		/// <returns>Whether the request's pawn can fool around.</returns>
		public static bool CanFoolAround(ref AttractionRequest request) =>
			CanFoolAround(request.Pawn, true);

		/// <summary>
		/// Checks to see if the given pawn would be willing to fool around.
		/// </summary>
		/// <param name="pawn">The pawn to check.</param>
		/// <param name="assumeInRelationship">Skips this check if `true`.</param>
		/// <returns>Whether the pawn can fool around.</returns>
		public static bool CanFoolAround(Pawn pawn, bool assumeInRelationship = false)
		{
			// Alternative modes that relax the cheating rules.
			if (RJWSettings.WildMode || RJWSettings.HippieMode) return true;

			// If they have no lover or spouse, they can always fool around.
			if (!assumeInRelationship && !HasAnyLovePartner(pawn)) return true;

			// Would they share their bed with anyone?
			var sharedBed
				= pawn.GetSpouseCount(false) > 0 ? HistoryEventDefOf.SharedBed_NonSpouse
				: HistoryEventDefOf.SharedBed;
			if (IdeoUtility.DoerWillingToDo(sharedBed, pawn)) return true;

			// Would they court someone new as their next spouse?
			var spouseCount = pawn.GetHistoryEventForSpouseCountPlusOne();
			if (IdeoUtility.DoerWillingToDo(spouseCount, pawn)) return true;

			// At this point, their ideology wouldn't allow it, but there are other mods
			// that add polyamory as a trait.  One would hope they'd change the above checks
			// to actually implement the feature, but just in case its a naive implementation,
			// we'll check to see if the pawn is in an exclusively polyamorous relationship.
			return !xxx.HasNonPolyPartner(pawn);
		}

		/// <summary>
		/// Checks to see if the request's target has done something to the request's
		/// pawn that makes them deserving of revenge rape.
		/// </summary>
		/// <param name="request">The request.</param>
		/// <returns>If revenge rape is deserved.</returns>
		public static bool CanHaveRevengeRape(ref AttractionRequest request)
		{
			var pawn = request.Pawn;
			var target = request.Target;

			// No revenge rape involving animals...  Yet.
			if (request.Category.InvolvesAnimals()) return false;

			// If the target cheated on this pawn, they qualify for revenge rape.
			foreach (var rel in request.Relations)
				if (IsExLovePartnerRelation(rel))
					if (pawn.HasMemoryWith(target, ThoughtDefOf.CheatedOnMe))
						return true;

			// This next part can fail if the target has no relations tracker,
			// like with mechanoids.
			if (target.relations is null) return false;

			// To spice things up, we'll also check if they're the "other lover".
			foreach (var rel in ExistingLovePartners(target, false))
				if (ExLovePartnerRelationExists(pawn, rel.otherPawn))
					if (pawn.HasMemoryWith(rel.otherPawn, ThoughtDefOf.CheatedOnMe))
						return true;

			return false;
		}

		/// <summary>
		/// Checks to see if the two pawns have on-going or lingering hostility
		/// towards one another.
		/// </summary>
		/// <param name="pawn">The first pawn.</param>
		/// <param name="target">The second pawn.</param>
		/// <returns>Whether there is hostility.</returns>
		public static bool HasHostility(Pawn pawn, Pawn target)
		{
			// Immediate, personal hostility.
			if (target.HostileTo(pawn) || pawn.HostileTo(target)) return true;

			// Faction hostility.
			if (target.Faction is { } tFaction)
				if (pawn.HostileTo(tFaction))
					return true;
			
			if (pawn.Faction is { } pFaction)
				if (target.HostileTo(pFaction))
					return true;

			// Lingering hostility from social fighting.
			if (!pawn.IsAnimal() && !target.IsAnimal())
				if (pawn.HasMemoryWith(target, ThoughtDefOf.HadAngeringFight))
					return true;

			return false;
		}

		/// <summary>
		/// Checks to see if the pawns of the request have on-going or lingering
		/// hostility towards one another.
		/// </summary>
		/// <param name="request">The request.</param>
		/// <returns>Whether there is hostility.</returns>
		public static bool HasHostility(ref AttractionRequest request) =>
			HasHostility(request.Pawn, request.Target);

		/// <summary>
		/// Gets the attraction category for the two pawns.
		/// </summary>
		/// <param name="observer">The observing pawn.</param>
		/// <param name="target">The pawn being observed</param>
		/// <returns>The attraction category.</returns>
		public static AttractionCategory GetCategory(Pawn observer, Pawn target) =>
			(observer.IsAnimal(), target.IsAnimal()) switch
			{
				(false, false) => AttractionCategory.BetweenHumans,
				(true, true) => AttractionCategory.BetweenAnimals,
				(true, false) => AttractionCategory.AnimalToHuman,
				(false, true) => AttractionCategory.HumanToAnimal
			};

		private static readonly FieldInfo AccessForReservations =
			AccessTools.Field(typeof(ReservationManager), "reservations");

		private static int ReservedCount(Pawn pawn)
		{
			if (pawn is null) return 0;

			var reserver = pawn.Map.reservationManager;
			var reservations = (List<Reservation>)AccessForReservations.GetValue(reserver);
			if (reservations.Count == 0) return 0;

			var ret = 0;
			foreach (var t in reservations)
				if (t.Target.Pawn == pawn)
					ret += 1;
			return ret;
		}
		
		/// <summary>
		/// <para>Debug tool to display a table of attraction factors of a pawn
		/// to all other pawns on the current map.</para>
		/// <para>This appears in the debug actions menu under the RJW category.</para>
		/// </summary>
		[DebugAction("RJW", actionType = DebugActionType.ToolMapForPawns)]
		internal static void AttractionFactorsForPawn(Pawn pawn)
		{
			var dataSource = pawn.Map.mapPawns.AllPawnsSpawned
				.Where(t => t != pawn)
				.OrderBy(t => xxx.get_pawnname(t))
				.SelectMany(ToData)
				.ToList();

			var table = new TableDataGetter<(Pawn pawn, Pawn target, AttractionPurpose purpose)>[]
			{
				new("pawn", (d) => xxx.get_pawnname(d.pawn)),
				new("target", (d) => xxx.get_pawnname(d.target)),
				new("purpose", (d) => d.purpose.ToString()),
				new("ageOK", (d) => IsAgeOk(d.pawn, d.target)),
				new("preflightOK", DoPreflight),
				new("finalized", (d) => DoEval(d, true)),
				new("combined", (d) => DoEval(d, false)),
				new("age", (d) => GetFactor(d, AttractionMode.Age)),
				new("physical", (d) => GetFactor(d, AttractionMode.Physical)),
				new("romantic", (d) => GetFactor(d, AttractionMode.Romantic)),
				new("social", (d) => GetFactor(d, AttractionMode.Social))
			};

			DebugTables.MakeTablesDialog(dataSource, table);

			// Helper functions...

			IEnumerable<(Pawn pawn, Pawn target, AttractionPurpose purpose)> ToData(Pawn target)
			{
				yield return (pawn, target, AttractionPurpose.General);
				yield return (pawn, target, AttractionPurpose.ForFucking);
				yield return (pawn, target, AttractionPurpose.ForRape);
			}

			static string DoPreflight((Pawn pawn, Pawn target, AttractionPurpose purpose) d)
			{
				var request = new AttractionRequest(d.purpose, d.pawn, d.target);
				return PreflightChecks(ref request).ToString();
			}

			static string DoEval((Pawn pawn, Pawn target, AttractionPurpose purpose) d, bool finalize)
			{
				using var _benchy = Benchy.Watch("AttractionUtility.Evaluate");
				var request = new AttractionRequest(d.purpose, d.pawn, d.target)
				{
					finalize = finalize
				};
				return Evaluate(ref request).ToString("F3");
			}

			static string GetFactor((Pawn pawn, Pawn target, AttractionPurpose purpose) d, AttractionMode mode)
			{
				var request = new AttractionRequest(d.purpose, d.pawn, d.target)
				{
					finalize = false
				};

				// We want to skip the preflight checks, so we're doing our own evaluation.
				SetupRequest(ref request);

				using var evaluator = request.GetEvaluator();
				return evaluator.Evaluate(mode, ref request).ToString("F3");
			}
		}

		private enum PawnMenuType
		{
			Colonist,
			Prisoner,
			Slave,
			ColonyAnimal,
			Visitor,
			TamedAnimal,
			Wild
		}

		[DebugAction("RJW", actionType = DebugActionType.Action, allowedGameStates = AllowedGameStates.PlayingOnMap)]
		internal static void ExplainAttraction()
		{
			
			var pawnsByType = Find.CurrentMap.mapPawns.AllPawnsSpawned
				.OrderBy((pawn) => xxx.get_pawnname(pawn))
				.GroupBy((pawn) => MenuType(pawn))
				.ToDictionary((g) => g.Key, (g) => g.ToList());
			var types = (PawnMenuType[])Enum.GetValues(typeof(PawnMenuType));

			PickObserver();

			// Helper functions...

			void PickObserver()
			{
				var observerList = new List<DebugMenuOption>();
				foreach (var type in types)
				{
					if (!pawnsByType.TryGetValue(type, out var pawnList)) continue;
					if (pawnList.Count == 0) continue;
					foreach (var pawn in pawnList)
					{
						observerList.Add(new DebugMenuOption(
							PawnLabelWithType(pawn, type),
							DebugMenuOptionMode.Action,
							PickTarget(pawn)
						));
					}
				}

				if (observerList.Count > 0)
					Find.WindowStack.Add(new Dialog_DebugOptionListLister(observerList));
				else
					Messages.Message("No pawns on map.", MessageTypeDefOf.NegativeEvent, false);
			}

			Action PickTarget(Pawn selectedPawn) => () =>
			{
				var targetList = new List<DebugMenuOption>();
				foreach (var type in types)
				{
					if (!pawnsByType.TryGetValue(type, out var pawnList)) continue;
					if (pawnList.Count == 0) continue;
					foreach (var targetPawn in pawnList)
					{
						if (targetPawn == selectedPawn) continue;
						targetList.Add(new DebugMenuOption(
							PawnLabelWithType(targetPawn, type),
							DebugMenuOptionMode.Action,
							PickPurpose(selectedPawn, targetPawn)
						));
					}
				}

				if (targetList.Count > 0)
					Find.WindowStack.Add(new Dialog_DebugOptionListLister(targetList));
				else
					Messages.Message("No other pawns on map.", MessageTypeDefOf.NegativeEvent, false);
			};

			Action PickPurpose(Pawn observer, Pawn target) => () =>
			{
				var purposeList = new List<DebugMenuOption>();
				var purposes = (AttractionPurpose[])Enum.GetValues(typeof(AttractionPurpose));
				foreach (var purpose in purposes)
				{
					purposeList.Add(new DebugMenuOption(
						GenText.SplitCamelCase(purpose.ToString()),
						DebugMenuOptionMode.Action,
						() => ExplainAttraction(observer, target, purpose)
					));
				}
				Find.WindowStack.Add(new Dialog_DebugOptionListLister(purposeList));
			};

			static string PawnLabelWithType(Pawn pawn, PawnMenuType type) =>
				$"{GenText.SplitCamelCase(type.ToString())} {GetPawnLabel(pawn)}";

			static PawnMenuType MenuType(Pawn x)
			{
				if (x.IsHumanLike())
				{
					if (x.IsColonist) return PawnMenuType.Colonist;
					if (x.IsPrisonerOfColony) return PawnMenuType.Prisoner;
					if (x.IsSlaveOfColony) return PawnMenuType.Slave;
					if (x.IsWildMan()) return PawnMenuType.Wild;
					return PawnMenuType.Visitor;
				}
				if (x.IsTameAnimal())
				{
					if (x.Faction == Faction.OfPlayer) return PawnMenuType.ColonyAnimal;
					return PawnMenuType.TamedAnimal;
				}
				return PawnMenuType.Wild;
			}
		}

		/// <summary>
		/// Open a new window with attraction detalization
		/// </summary>
		public static void ExplainAttraction(Pawn observer, Pawn target, AttractionPurpose purpose)
		{
			var sb = new StringBuilder();
			sb.Append("Observer: ").AppendLine(GetPawnLabel(observer));
			sb.Append("Target: ").AppendLine(GetPawnLabel(target));
			sb.Append("Purpose: ").AppendLine(GenText.SplitCamelCase(purpose.ToString()));
			sb.AppendLine();

			var request = new AttractionRequest(purpose, observer, target);
			SetupRequest(ref request);

			// Run through the modes, showing the explanations.  The preferences will
			// mute the color of their explanation if they didn't end up changing the
			// factor.
			var modes = new AttractionMode[] {
				AttractionMode.Age,
				AttractionMode.Physical,
				AttractionMode.Romantic,
				AttractionMode.Social
			};
			var factors = new List<float>();
			using (var evaluator = request.GetEvaluator())
			{
				foreach (var mode in modes)
				{
					var textList = evaluator.Explain(mode, ref request, out var modeFactor);
					factors.Add(modeFactor);

					sb.AppendTagged(GenText.SplitCamelCase(mode.ToString()).AsTipTitle());
					sb.Append(": ").AppendLine(FactorStr(modeFactor));
					foreach (var text in textList) sb.AppendLineTagged(text);
					sb.AppendLine();
				}
			}

			// Demonstrate how the factors are combined.
			var combinedFactor = factors.Aggregate(1f, (acc, factor) => acc * factor);
			sb.AppendTagged("Combined".AsTipTitle());
			sb.Append(": ").Append(string.Join(" * ", factors.Select(FactorStr)));
			sb.Append(" = ").AppendLine(FactorStr(combinedFactor));
			sb.AppendLine();

			// Apply the finalizer to get the actual result value.
			sb.AppendTagged("Finalized".AsTipTitle());
			sb.Append(": ").Append(FactorStr(Finalize(ref request, combinedFactor)));

			Find.WindowStack.Add(new Dialog_MessageBox(sb.ToString().Trim()));

			static string FactorStr(float factor) =>
				factor.ToStringByStyle(ToStringStyle.FloatMaxThree);
		}

		private static string GetPawnLabel(Pawn pawn)
		{
			var name = xxx.get_pawnname(pawn);
			var sex = GenderHelper.GetSex(pawn).ToString().CapitalizeFirst();
			var race = GetRaceLabel(pawn);
			var age = pawn.ageTracker.AgeBiologicalYears;
			if (name.ToLower() == race.ToLower())
				return $"{name} ({sex}, {age})";
			return $"{name} ({sex} {race}, {age})";
		}

		private static string GetRaceLabel(Pawn pawn)
		{
			if (pawn.IsHumanLike() && ModsConfig.BiotechActive)
				if (pawn.genes is { } genes && genes.Xenotype != XenotypeDefOf.Baseliner)
					return genes.XenotypeLabelCap;
			if (pawn.IsHumanLike())
				return pawn.def.LabelCap;
			return GenText.ToTitleCaseSmart(pawn.KindLabel);
		}
	}
}