#nullable enable

using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Verse;
using RimWorld;
using rjw.Modules.Shared.Logs;
using static RimWorld.LovePartnerRelationUtility;

namespace rjw.Modules.Attraction
{
	public ref struct AttractionRequest
	{
		/// <summary>
		/// Delegate for customizing the attraction preferences of a request.
		/// </summary>
		/// <param name="request">The request to modify.</param>
		public delegate void ModifyPreferences(ref AttractionRequest request);

		/// <summary>
		/// The pawn doing the observing.
		/// </summary>
		public readonly Pawn Pawn;

		/// <summary>
		/// The pawn being observed.
		/// </summary>
		public readonly Pawn Target;

		/// <summary>
		/// <para>How the attraction is to be used.</para>
		/// <para>Factors can be calculated differently based on this value.</para>
		/// </summary>
		public readonly AttractionPurpose Purpose;

		/// <summary>
		/// How the two pawns are categorized, in terms of animal vs human.
		/// </summary>
		public readonly AttractionCategory Category;

		/// <summary>
		/// <para>Indicates if these two have any kind of love relation.</para>
		/// <para>Although zoophiles treat the bond as a love relationship, this
		/// only counts the human standard relationships.</para>
		/// </summary>
		public bool HasLoveRelationWithTarget =>
			hasLoveRelation ??= Relations.Any(IsLovePartnerRelation);
		private bool? hasLoveRelation = null;

		/// <summary>
		/// <para>A cached list of the relations the pawn has with the target.</para>
		/// <para>Please do not mutate this list.</para>
		/// </summary>
		public List<PawnRelationDef> Relations =>
			relations ??= Pawn.GetRelations(Target).ToList();
		private List<PawnRelationDef>? relations = null;

		/// <summary>
		/// Whether this instance is blocking attempts to manipulate preferences.
		/// </summary>
		public bool IsLocked => IsLocked;
		private bool isLocked = false;

		/// <summary>
		/// Indicates the status of this pawn's relation in context of the target.
		/// </summary>
		public AttractionStatus Status =>
			status ??= AttractionUtility.GetStatus(Pawn, Target);
		private AttractionStatus? status = null;

		/// <summary>
		/// <para>Option to ignore the bleeding pre-flight check.</para>
		/// <para>Only considered when the purpose is sex.</para>
		/// </summary>
		public bool ignoreBleeding = false;

		/// <summary>
		/// Option that skips finalizing the factor when set to false.
		/// </summary>
		public bool finalize = true;

		/// <summary>
		/// Optional modifier that can be set to customize preferences before quirks.
		/// </summary>
		public ModifyPreferences? preQuirkModifier = null;

		/// <summary>
		/// Optional modifier that can be set to customize preferences after quirks.
		/// </summary>
		public ModifyPreferences? postQuirkModifier = null;

		/// <summary>
		/// Logger instance for the preference.
		/// </summary>
		private readonly ILog Log;

		/// <summary>
		/// The main storage for pawn preferences.
		/// </summary>
		private readonly Dictionary<string, AttractionPreference> Preferences =
			new(AttractionUtility.StandardApplicatorsCount);

		/// <summary>
		/// Creates a new request with the given purpose.
		/// </summary>
		/// <param name="purpose">The purpose to use.</param>
		/// <param name="pawn">The pawn doing the observing.</param>
		/// <param name="target">The pawn being observed.</param>
		public AttractionRequest(AttractionPurpose purpose, Pawn pawn, Pawn target)
		{
			Log = LogManager.GetLogger("AttractionRequest");
			Purpose = purpose;
			Pawn = pawn;
			Target = target;
			Category = AttractionUtility.GetCategory(pawn, target);
		}

		/// <summary>
		/// Creates a general request.
		/// </summary>
		/// <param name="pawn">The pawn doing the observing.</param>
		/// <param name="target">The pawn being observed.</param>
		public AttractionRequest(Pawn pawn, Pawn target)
		: this(AttractionPurpose.General, pawn, target) { }

		public bool HasPreference(string key) =>
			Preferences.ContainsKey(key);

		public AttractionPreference? GetPreference(string key)
		{
			if (isLocked)
			{
				Log.Error($"Tried to get preference with key `{key}` while locked.");
				return null;
			}
			return Preferences.TryGetValue(key, out var pref) ? pref : null;
		}

		public T? GetPreference<T>(string key) where T : AttractionPreference =>
			GetPreference(key) as T;

		public void SetPreference(AttractionPreference pref)
		{
			// The only allowed operation allowed while locked is adding new preferences.
			if (isLocked && HasPreference(pref.Key))
			{
				Log.Error($"Tried to replace preference with key `{pref.Key}` while locked.");
				return;
			}
			Preferences[pref.Key] = pref;
		}

		public void RemovePreference(string key)
		{
			if (isLocked)
			{
				Log.Error($"Tried to remove preference with key `{key}` while locked.");
				return;
			}
			Preferences.Remove(key);
		}

		/// <summary>
		/// Gets an evaluator instance.  Call this with a `using` block after setting
		/// up the request.
		/// </summary>
		/// <returns>The evaluator instance.</returns>
		public PreferenceEvaluator GetEvaluator() => new(Preferences.Values);

		public override string ToString() =>
			$"{xxx.get_pawnname(Pawn)} -> {xxx.get_pawnname(Target)}";
	}

	/// <summary>
	/// This controls access to a shared resource used to reduce GC churn while
	/// evaluating a request.
	/// </summary>
	public readonly ref struct PreferenceEvaluator
	{
		/// <summary>
		/// Internal preference storage.
		/// </summary>
		[ThreadStatic]
		private static readonly List<AttractionPreference> SortedStorage = new();

		public PreferenceEvaluator(IEnumerable<AttractionPreference> prefs)
		{
			SortedStorage.AddRange(prefs);
			SortedStorage.Sort(AttractionPreference.Comparer);
		}

		public void Dispose() =>
			SortedStorage.Clear();

		/// <summary>
		/// <para>Performs an evaluation for the given mode.</para>
		/// <para>Unfortunately, ref fields are not a thing yet, so you have to
		/// pass in the request here instead of the constructor.</para>
		/// </summary>
		/// <param name="mode">The mode to evaluate.</param>
		/// <param name="request">The attraction request.</param>
		/// <returns>The factor for the mode.</returns>
		public float Evaluate(AttractionMode mode, ref AttractionRequest request)
		{
			// Since addition and multiplication are communicative, as long
			// as we do each separately, we should always get the same value.
			// The `PrefCache` is guaranteed to have an entry for each possible
			// combination, so `TryGetValue` is not needed.

			var factor = 1f;

			// We'll do the additive stuff first.
			foreach (var pref in SortedStorage)
				if (pref.Mode == mode && pref.Operation == FactorOperation.Add)
					pref.Evaluate(ref request, ref factor);

			// No reason to do the multiplicative ones if this is the case.
			if (factor <= 0f) return 0f;

			// Now the multiplicative stuff.
			foreach (var pref in SortedStorage)
			{
				if (pref.Mode == mode && pref.Operation == FactorOperation.Multiply)
				{
					pref.Evaluate(ref request, ref factor);
					// Abort early if we hit 0.
					if (factor <= 0f) return 0f;
				}
			}

			// Finally, the add-by-factor pseudo-multiplication stuff.
			foreach (var pref in SortedStorage)
				if (pref.Mode == mode && pref.Operation == FactorOperation.AddByFactor)
					pref.Evaluate(ref request, ref factor);

			return factor;
		}

		/// <summary>
		/// <para>Performs an evaluation for the given mode, but generates a list of
		/// strings that explain the process in detail.</para>
		/// </summary>
		/// <param name="mode">The mode to evaluate.</param>
		/// <param name="request">The attraction request.</param>
		/// <param name="factor">A variable to hold the factor.</param>
		/// <returns>The list of explanations.</returns>
		public List<TaggedString> Explain(
			AttractionMode mode,
			ref AttractionRequest request,
			out float factor)
		{
			factor = 1f;
			var list = new List<TaggedString>();

			foreach (var pref in SortedStorage)
				if (pref.Mode == mode && pref.Operation == FactorOperation.Add)
					list.Add(pref.Explain(ref request, ref factor));

			factor = Mathf.Max(0f, factor);

			foreach (var pref in SortedStorage)
			{
				if (pref.Mode == mode && pref.Operation == FactorOperation.Multiply)
				{
					list.Add(pref.Explain(ref request, ref factor));
					factor = Mathf.Max(0f, factor);
				}
			}

			foreach (var pref in SortedStorage)
				if (pref.Mode == mode && pref.Operation == FactorOperation.AddByFactor)
					list.Add(pref.Explain(ref request, ref factor));

			factor = Mathf.Max(0f, factor);
			return list;
		}
	}
}