#nullable enable

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using UnityEngine;
using Verse;
using rjw.Modules.Shared.Logs;

namespace rjw.Modules.Attraction
{
	/// <summary>
	/// How to apply the value looked up from the curve to the factor.
	/// </summary>
	public enum FactorOperation {
		/// <summary>
		/// Adds the value to the factor.
		/// </summary>
		Add,
		/// <summary>
		/// Multiplies the value with the factor.
		/// </summary>
		Multiply,
		/// <summary>
		/// <para>Adds the value to the factor, but the current factor was used
		/// to produce the value.</para>
		/// <para>The `GetOperand` function provides the current factor, which
		/// allows you to do an operation like `factor += factor * scalar`.
		/// Use this operation to let it know so the operation can be applied
		/// in a safe order.</para>
		/// </summary>
		AddByFactor
	}

	/// <summary>
	/// <para>An attribute to mark applicators forstandard preferences.  These
	/// are preferences that should be applied to all requests automatically.</para>
	/// <para>The method needs to be static and have a signature that fits the
	/// <see cref="AttractionUtility.ApplyPreference" /> delegate.  It should check
	/// the request to see if the preference should apply and add itself to the
	/// request if so.</para>
	/// </summary>
	[AttributeUsage(AttributeTargets.Method)]
	public sealed class StandardPreferenceAttribute : Attribute { }

	/// <summary>
	/// Sorts by priority descending then the key so we have a deterministic
	/// evaluation order between runs of the game.
	/// </summary>
	public class PrefComparer : IComparer<AttractionPreference>
	{
		public int Compare(AttractionPreference a, AttractionPreference b)
		{
			if (a.Priority.CompareTo(b.Priority) is var prio)
				if (prio is not 0)
					return -prio;
			return string.CompareOrdinal(a.Key, b.Key);
		}
	}

	/// <summary>
	/// <para>Base class for pawn preferences.</para>
	/// <para>Implementers should write child classes in a way that allows them to
	/// be easily and arbitrarily tweaked for the implementation of quirks.</para>
	/// </summary>
	public abstract class AttractionPreference
	{
		/// <summary>
		/// Global comparer instance.
		/// </summary>
		public static readonly PrefComparer Comparer = new();

		/// <summary>
		/// Regex that removes standard preference prefixes like "P_" and "R_".
		/// </summary>
		private static readonly Regex PrefixRemover = new(@"^[APSR]_");

		private static readonly Color ColorWhite = GenColor.FromHex("FFFFFF");
		private static readonly Color ColorMuted = GenColor.FromHex("808080");
		private static readonly Color ColorImproved = GenColor.FromHex("66CC00");
		private static readonly Color ColorDegraded = GenColor.FromHex("FF3300");

		/// <summary>
		/// Logger instance for the preference.
		/// </summary>
		protected ILog Log;

		/// <summary>
		/// <para>The attraction mode to which this preference applies.</para>
		/// <para>The preference will only be applied if this mode was requested.</para>
		/// </summary>
		public AttractionMode Mode { get; private set; }

		/// <summary>
		/// <para>A string used to look up this preference from the request.</para>
		/// <para>Quirks can grab a well-known preference with this key and tweak
		/// it to apply the nuances of the quirk.</para>
		/// </summary>
		public string Key { get; private set; }

		/// <summary>
		/// What operation to apply to the factor using the value.
		/// </summary>
		public FactorOperation Operation { get; private set; }

		/// <summary>
		/// <para>The run priority.</para>
		/// <para>Mostly useful for multiplicative preferences that are very likely
		/// to set the factor to zero.  Giving these a higher priority can optimize
		/// evaluation by stopping evaluation early.</para>
		/// </summary>
		public int Priority { get; protected set; } = 0;

		/// <summary>
		/// Initializes the preference.  Required to be called by child classes.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		protected AttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation)
		{
			Log = LogManager.GetLogger($"PawnPreference::{GetType().Name}");
			Mode = mode;
			Key = key;
			Operation = operation;
		}

		/// <summary>
		/// Evaluates this preference, adjusting the `factor` as needed.
		/// </summary>
		/// <param name="request">The attraction request.</param>
		/// <param name="factor">The current factor.</param>
		public void Evaluate(ref AttractionRequest request, ref float factor)
		{
			// Make sure child classes get a factor that is 0 or greater.
			var safeFactor = Mathf.Max(0f, factor);

			var operand = GetOperand(ref request, safeFactor);
			factor = Operation switch
			{
				FactorOperation.Add or FactorOperation.AddByFactor =>
					factor + operand,
				FactorOperation.Multiply when operand < 0f =>
					LogProblem(factor, "Multiplying by a negative number is not allowed."),
				FactorOperation.Multiply =>
					factor * operand,
				var unknown =>
					LogProblem(factor, $"Unknown FactorOperation: {(int)unknown}")
			};
		}

		/// <summary>
		/// Generates a string that describes how this preference is changing the factor.
		/// This should update the factor like a normal evaluation.
		/// </summary>
		/// <param name="request">The attraction request.</param>
		/// <param name="factor">The current factor.</param>
		/// <returns>The explanation string.</returns>
		public virtual TaggedString Explain(ref AttractionRequest request, ref float factor)
		{
			var noopValue = NoopValue(Operation);
			var prevFactor = factor;
			var safeFactor = Mathf.Max(0f, prevFactor);
			var operand = GetOperand(ref request, safeFactor);
			var operandText = Operation switch
			{
				FactorOperation.Add or FactorOperation.AddByFactor =>
					operand.ToStringByStyle(ToStringStyle.FloatMaxThree, ToStringNumberSense.Offset),
				_ =>
					$"{operand.ToStringByStyle(ToStringStyle.FloatMaxThree)}x"
			};
			Evaluate(ref request, ref factor);

			// If the factor didn't change, mute the color.
			var labelColor = factor == prevFactor ? ColorMuted : ColorWhite;
			// And we'll color the operand based on how it tried to change it.
			var operandColor
				= operand == noopValue ? labelColor
				: operand < noopValue ? ColorDegraded * labelColor
				: ColorImproved * labelColor;

			var label = PrefixRemover.Replace(Key, "");
			return new StringBuilder()
				.Append(GenText.SplitCamelCase(label).Replace("_", ", "))
				.Append(": ").AppendTagged(operandText.Colorize(operandColor))
				.Append(" ⇒ ").Append(factor.ToStringByStyle(ToStringStyle.FloatMaxThree))
				.ToString()
				.Colorize(labelColor);
		}

		private float LogProblem(float factor, string msg)
		{
			Log.Error(msg);
			return factor;
		}

		/// <summary>
		/// Gets the operand to adjust the factor with.
		/// </summary>
		/// <param name="request">The attraction request.</param>
		/// <param name="factor">The current factor.</param>
		/// <returns>The operand.</returns>
		protected abstract float GetOperand(ref AttractionRequest request, float factor);

		/// <summary>
		/// Gets a value that would not change the factor.
		/// </summary>
		/// <returns>The value.</returns>
		protected static float NoopValue(FactorOperation op) =>
			op == FactorOperation.Multiply ? 1f : 0f;
	}

	/// <summary>
	/// Pawn preference that adjusts the factor using a fixed value.
	/// </summary>
	public class FlatAttractionPreference : AttractionPreference
	{
		/// <summary>
		/// The flat value.
		/// </summary>
		public float Value { get; set; }

		/// <summary>
		/// Instantiates a new flat preference.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="value">The value to be applied.</param>
		public FlatAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			float value) : base(mode, key, operation)
		{
			Value = value;
		}

		protected override float GetOperand(ref AttractionRequest request, float factor) =>
			Value;
	}

	/// <summary>
	/// Pawn preference that changes the factor based on the result of a predicate.
	/// </summary>
	public class PredicatedAttractionPreference : AttractionPreference
	{
		public delegate bool PredicateFn(ref AttractionRequest request);

		public delegate float WhenFn(ref AttractionRequest request, float factor);

		/// <summary>
		/// Function that performs a test to see if the pawn has a positive or
		/// negative response to the target.
		/// </summary>
		public PredicateFn Predicate { get; set; }

		/// <summary>
		/// <para>Function evaluated when the `Test` returns true.</para>
		/// <para>When null, no action will be done for this case.</para>
		/// </summary>
		public WhenFn? WhenTrue { get; set; }

		/// <summary>
		/// <para>Function evaluated when the `Test` returns false.</para>
		/// <para>When null, no action will be done for this case.</para>
		/// </summary>
		public WhenFn? WhenFalse { get; set; }

		/// <summary>
		/// <para>Instantiates a new predicated preference.</para>
		/// <para>Tip: you can use named-parameters for `whenTrue` and `whenFalse` to
		/// set only one or the other, leaving the one omitted as a noop.</para>
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="predicateFn">The function that checks the request.</param>
		/// <param name="whenTrue">What to do to the factor when the predicate is true.</param>
		/// <param name="whenFalse">What to do to the factor when the predicate is false.</param>
		public PredicatedAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			PredicateFn predicateFn,
			WhenFn? whenTrue = null,
			WhenFn? whenFalse = null) : base(mode, key, operation)
		{
			Predicate = predicateFn;
			WhenTrue = whenTrue;
			WhenFalse = whenFalse;
		}

		protected override float GetOperand(ref AttractionRequest request, float factor)
		{
			var whenFn = Predicate(ref request) ? WhenTrue : WhenFalse;
			if (whenFn is null) return NoopValue(Operation);
			return whenFn.Invoke(ref request, factor);
		}

		/// <summary>
		/// Creates a `WhenFn` delegate using a fixed value.
		/// </summary>
		/// <param name="value">The other operand.</param>
		/// <returns>A delegate.</returns>
		public static WhenFn? FixedValue(float value) =>
			(ref AttractionRequest request, float factor) => value;
	}

	/// <summary>
	/// <para>Pawn preference that checks a particular value and then changes the
	/// factor based on the value.</para>
	/// <para>This is basically LINQ's `Select` operator, but for preferences.</para>
	/// </summary>
	/// <typeparam name="T">The type to be mapped.</typeparam>
	public class SelectAttractionPreference<T> : AttractionPreference
	{
		public delegate T FetchFn(ref AttractionRequest request);

		public delegate float SelectFn(T value, ref AttractionRequest request);

		/// <summary>
		/// The function that gets the value to use in the selection.
		/// </summary>
		public FetchFn Fetch { get; set; }

		/// <summary>
		/// The function that checks the value against its options and updates the
		/// factor accordingly.
		/// </summary>
		public SelectFn Select { get; set; }

		/// <summary>
		/// Instantiates a new choice preference.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="fetchFn">The function used to get the value to select on.</param>
		/// <param name="selectFn">The function that takes the value and updates the factor.</param>
		public SelectAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			FetchFn fetchFn,
			SelectFn selectFn) : base(mode, key, operation)
		{
			Fetch = fetchFn;
			Select = selectFn;
		}

		protected override float GetOperand(ref AttractionRequest request, float factor) =>
			Select(Fetch(ref request), ref request);
	}

	/// <summary>
	/// Pawn preference that picks a value to change the factor from a limited 
	/// selection of possibilities, represented by enum values.
	/// </summary>
	/// <typeparam name="T">The type of the enum.</typeparam>
	public class OptionAttractionPreference<T> : AttractionPreference where T : struct, Enum
	{
		public delegate T FetchFn(ref AttractionRequest request);

		/// <summary>
		/// The function that gets the value to lookup.
		/// </summary>
		public FetchFn Fetch { get; set; }

		/// <summary>
		/// <para>The options, mapping from the value to a float.</para>
		/// <para>If the value is not found, the default will be used instead.</para>
		/// </summary>
		public Dictionary<T, float> Options { get; set; }

		/// <summary>
		/// The default value to use when no suitable option is found.
		/// </summary>
		public float Default { get; set; }

		/// <summary>
		/// Instantiates a new option preference.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="fetchFn">The function used to get the value to pick against.</param>
		/// <param name="options">The options used to assemble the dictionary.</param>
		/// <param name="defaultVal">The value to use in case of a lookup failure.</param>
		public OptionAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			FetchFn fetchFn,
			IEnumerable<KeyValuePair<T, float>> options,
			float defaultVal) : base(mode, key, operation)
		{
			Fetch = fetchFn;
			Options = options.ToDictionary((v) => v.Key, (v) => v.Value);
			Default = defaultVal;
		}

		/// <summary>
		/// Instantiates a new option preference using the given operation with
		/// a default value that does not change the factor.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="fetchFn">The function used to get the value to pick against.</param>
		/// <param name="options">The options used to assemble the dictionary.</param>
		public OptionAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			FetchFn fetchFn,
			IEnumerable<KeyValuePair<T, float>> options)
		: this(mode, key, operation, fetchFn, options, NoopValue(operation)) { }

		/// <summary>
		/// Instantiates a new option preference using a multiplicative operation
		/// and a default value that does not change the factor.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="fetchFn">The function used to get the value to pick against.</param>
		/// <param name="options">The options used to assemble the dictionary.</param>
		public OptionAttractionPreference(
			AttractionMode mode,
			string key,
			FetchFn fetchFn,
			IEnumerable<KeyValuePair<T, float>> options)
		: this(mode, key, FactorOperation.Multiply, fetchFn, options) { }

		protected override float GetOperand(ref AttractionRequest request, float factor) =>
			Options.TryGetValue(Fetch(ref request), out var result) ? result : Default;
	}

	public class CurveAttractionPreference : AttractionPreference
	{
		public delegate float FetchFn(ref AttractionRequest request);

		/// <summary>
		/// Fetches the `x` value to evaluate the curve against.
		/// </summary>
		public FetchFn Fetch { get; set; }

		/// <summary>
		/// The curve that will be used to lookup a value to apply to the factor.
		/// </summary>
		public SimpleCurve Curve { get; private set; }

		/// <summary>
		/// Instantiates a new curve preference.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="operation">How to use the value to change the factor.</param>
		/// <param name="fetchFn">Function that gets the value to evaluate against the curve.</param>
		/// <param name="curve">The curve used for lookups.</param>
		public CurveAttractionPreference(
			AttractionMode mode,
			string key,
			FactorOperation operation,
			FetchFn fetchFn,
			IEnumerable<CurvePoint> points) : base(mode, key, operation)
		{
			Fetch = fetchFn;
			Curve = new(points);
		}

		/// <summary>
		/// Instantiates a new curve preference with the default multiply operation.
		/// </summary>
		/// <param name="mode">The attraction mode this preference applies to.</param>
		/// <param name="key">The unique lookup key for this preference.</param>
		/// <param name="fetchFn">Function that gets the value to evaluate against the curve.</param>
		/// <param name="curve">The curve used for lookups.</param>
		public CurveAttractionPreference(
			AttractionMode mode,
			string key,
			FetchFn fetchFn,
			IEnumerable<CurvePoint> points) : base(mode, key, FactorOperation.Multiply)
		{
			Fetch = fetchFn;
			Curve = new(points);
		}

		/// <summary>
		/// <para>Replaces all the points in the curve with the given points.</para>
		/// <para>If you're using another `SimpleCurve` as a source, especially one that
		/// is `static readonly`, this ensures that your original curve isn't accidentally
		/// mutated by other quirks or something.</para>
		/// </summary>
		/// <param name="points">The points to set.</param>
		public void ReplaceCurve(IEnumerable<CurvePoint> points)
		{
			if (!points.Any())
			{
				Log.Error("An attempt was made to replace the curve with an empty one.");
				return;
			}

			Curve.Points.Clear();
			foreach (var point in points) Curve.Add(point);
		}

		protected override float GetOperand(ref AttractionRequest request, float factor) =>
			Curve.Evaluate(Fetch(ref request));
	}
}