#nullable enable

using Verse;
using Verse.AI;
using RimWorld;
using System.Collections.Generic;
using System.Linq;

namespace rjw
{
	public class JobGiver_ComfortPrisonerRape : ThinkNode_JobGiver
	{
		/// <summary>
		/// The minimum attraction of the rapist to the comfort prisoner to
		/// enable their use.  The attraction of the prisoner to the rapist
		/// is not of interest to us.
		/// </summary>
		const float MinimumAttraction = 0.1f;

		/// <summary>
		/// Finds all pawns on the pawn's map that might be good rape victims
		/// for the given pawn.
		/// </summary>
		/// <param name="pawn">The pawn looking for a rape.</param>
		/// <returns>An enumeration of potential victims.</returns>
		public static IEnumerable<Pawn> PotentialVictimsFor(Pawn pawn)
		{
			foreach (var victim in DesignatorsData.rjwComfort)
			{
				// Basic eligibility checks.
				if (victim == pawn) continue;
				if (victim.Map != pawn.Map) continue;
				if (victim.Suspended) continue;
				if (victim.Drafted) continue;
				if (victim.IsForbidden(pawn)) continue;
				if (!xxx.can_get_raped(victim)) continue;
				if (!Pather_Utility.cells_to_target_rape(pawn, victim.Position)) continue;

				// Animals only consider targets they can see, instead of seeking them out.
				if (pawn.IsAnimal() && !pawn.CanSee(victim)) continue;

				yield return victim;
			}
		}

		public static Pawn? FindComfortPrisoner(Pawn pawn)
		{
			if (!DesignatorsData.rjwComfort.Any()) return null;

			string pawnName = xxx.get_pawnname(pawn);
			if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({pawnName})");

			var targets = PotentialVictimsFor(pawn);

			if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({pawnName}): found {targets.Count()}");

			foreach (var result in SexAppraiser.FindVictims(pawn, targets))
			{
				var victim = result.Target;
				var weight = result.ObserverAttraction;
				if (weight < MinimumAttraction)
				{
					if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({pawnName}): {xxx.get_pawnname(victim)} not attractive enough");
					continue;
				}

				if (!pawn.CanReserveAndReach(victim, PathEndMode.OnCell, Danger.Some, 1, 0))
				{
					if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({pawnName}): cannot path to {xxx.get_pawnname(victim)}");
					continue;
				}

				if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({pawnName}): using {xxx.get_pawnname(victim)} with score {weight}");
				return victim;
			}

			return null;
		}

		protected override Job? TryGiveJob(Pawn pawn)
		{
			if (RJWSettings.DebugRape) ModLog.Message($"JobGiver_ComfortPrisonerRape::TryGiveJob({xxx.get_pawnname(pawn)}) called");

			if (!SexAppraiser.InMoodForRape(pawn))
			{
				if (RJWSettings.DebugRape) ModLog.Message($"JobGiver_ComfortPrisonerRape({xxx.get_pawnname(pawn)}): not in mood for rape.");
				return null;
			}

			if (!RJWSettings.WildMode)
			{
				// don't allow pawns marked as comfort prisoners to rape others
				if (RJWSettings.DebugRape) ModLog.Message($"JobGiver_ComfortPrisonerRape::TryGiveJob({xxx.get_pawnname(pawn)}): is healthy = {xxx.is_healthy(pawn)}, is cp = {pawn.IsDesignatedComfort()}, is ready = {SexUtility.ReadyForLovin(pawn)}, is frustrated = {xxx.is_frustrated(pawn)}");
				if (!xxx.is_healthy(pawn) || pawn.IsDesignatedComfort() || (!SexUtility.ReadyForLovin(pawn) && !xxx.is_frustrated(pawn))) return null;
			}

			if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({xxx.get_pawnname(pawn)}): can rape = {xxx.can_rape(pawn)}, is drafted = {pawn.Drafted}");
			if (pawn.Drafted || !xxx.can_rape(pawn)) return null;

			// It's unnecessary to include other job checks. Pawns seem to only look for new jobs when between jobs or laying down idle.
			// But let's check for devmode, just in case someone's using the JobGiver tool in the debug menu to figure out why things aren't working
			// "I already have a job (Wait_Wander)" is tells you nothing about why they can't find targets.
			if (pawn.jobs.curJob is { } curJob && curJob.def != JobDefOf.LayDown && !RJWSettings.DevMode)
			{
				if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({xxx.get_pawnname(pawn)}): I already have a job ({pawn.CurJobDef})");
				return null;
			}

			// Faction check.  This blocks guests and vistors from using the
			// comfort prisoners.  Is this what is intended?
			if (pawn.Faction?.IsPlayer is not true && !pawn.IsPrisonerOfColony)
			{
				if (RJWSettings.DebugRape) ModLog.Message($"FindComfortPrisoner({xxx.get_pawnname(pawn)}): not considered part of colony");
				return null;
			}

			var target = FindComfortPrisoner(pawn);
			if (target is null)
			{
				if (RJWSettings.DebugRape) ModLog.Message($"JobGiver_ComfortPrisonerRape::TryGiveJob({xxx.get_pawnname(pawn)}): no target found");
				return null;
			}

			var jobDef = xxx.is_animal(target) ? xxx.bestiality : xxx.RapeCP;
			if (RJWSettings.DebugRape) ModLog.Message($"JobGiver_ComfortPrisonerRape::TryGiveJob({xxx.get_pawnname(pawn)}): using target {xxx.get_pawnname(target)} and job {jobDef.defName}");

			return JobMaker.MakeJob(jobDef, target);
		}
	}
}
