using Verse;
using Verse.AI;
using System.Collections.Generic;
using System.Linq;
using RimWorld;
using Multiplayer.API;
using System;

using Seeded = rjw.Modules.Rand.Seeded;
using rjw.Modules.Attraction;
using rjw.Modules.Shared.Extensions;

namespace rjw
{
	/// <summary>
	/// Helper for sex with (humanlikes)
	/// </summary>
	public class CasualSex_Helper
	{
		public static readonly HashSet<string> quickieAllowedJobs = new(DefDatabase<StringListDef>.GetNamed("QuickieInterruptibleJobs").strings);
		/// <summary>
		/// Salt for RNG when finding a partner.
		/// </summary>
		const int PartnerRNGSalt = -787835064;

		/// <summary>
		/// Salt for RNG when finding a location.
		/// </summary>
		const int LocationRNGSalt = -2112521797;

		/// <summary>
		/// The maximum distance to search for a random sex location.
		/// </summary>
		const int MaxLocationDistance = 40;

		public enum QuickieCloseness { Close, InRange, OutOfRange }

		public static bool CanHaveSex(Pawn target)
		{
			return xxx.can_fuck(target) || xxx.can_be_fucked(target);
		}

		/// <summary>
		/// Determines the rough proximity of the observer and target.
		/// </summary>
		/// <param name="observer">The observer.</param>
		/// <param name="target">The target.</param>
		/// <returns>A fuzzy value indicating how close they are.</returns>
		public static QuickieCloseness GetCloseness(Pawn observer, Pawn target) =>
			observer.Position.DistanceTo(target.Position) switch
			{
				<= 4f =>
					QuickieCloseness.Close,
				var dist when dist > RJWSettings.maxDistanceCellsCasual =>
					QuickieCloseness.OutOfRange,
				_ =>
					QuickieCloseness.InRange
			};

		/// <summary>
		/// Finds all pawns on the pawn's map that might be good hookup partners
		/// for the given pawn.
		/// </summary>
		/// <param name="pawn">The pawn looking for a hookup.</param>
		/// <param name="bedsex">Whether this should be restricted to targets in bed.</param>
		/// <returns>An enumeration of potential hookup partners.</returns>
		public static IEnumerable<Pawn> PotentialPartnersFor(Pawn pawn, bool bedsex = false)
		{
			foreach (var partner in pawn.Map.mapPawns.AllPawnsSpawned)
			{
				// Animals are handled by the bestiality jobs.
				if (partner.IsAnimal()) continue;
				// Probably a droid or some other modded pawn.
				if (partner.relations is null) continue;

				// Basic eligibility checks.
				if (partner == pawn) continue;
				if (GetCloseness(pawn, partner) is QuickieCloseness.OutOfRange) continue;
				if (partner.IsForbidden(pawn)) continue;
				if (!xxx.IsTargetPawnOkay(partner)) continue;
				if (!CanHaveSex(partner)) continue;
				if (partner.HostileTo(pawn)) continue;

				// They must be able to mutually reserve each other.
				if (!pawn.CanReserveAndReach(partner, PathEndMode.OnCell, Danger.Some, 1, 0)) continue;
				if (!partner.CanReserve(pawn, 1, 0)) continue;

				// Validate the partner fits the request for a bed.
				switch (bedsex)
				{
					case true when Bed_Utility.is_laying_down_alone(partner): break;
					case true when Bed_Utility.in_same_bed(partner, pawn): break;
					case false when !partner.InBed(): break;
					default: continue;
				}

				// Make sure they are not doing something we shouldn't interrupt.
				if (!bedsex && partner.jobs?.curJob is Job curJob)
					if (curJob.playerForced || !quickieAllowedJobs.Contains(curJob.def.defName))
						continue;
				
				// They need to be actually allowed to hookup right now.
				if (!CanTargetHookup(pawn, partner)) continue;

				yield return partner;
			}
		}

		[SyncMethod]
		public static Pawn FindPartner(Pawn pawn, bool bedsex = false)
		{
			string pawnName = xxx.get_pawnname(pawn);
			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): starting.");

			bool pawnIsNympho = xxx.is_nympho(pawn);
			bool pawnCanPickAnyone = RJWSettings.WildMode;// || (pawnIsNympho && RJWHookupSettings.NymphosCanPickAnyone);

			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): nympho:{pawnIsNympho}, ignores rules:{pawnCanPickAnyone}");

			if (!RJWHookupSettings.ColonistsCanHookup && pawn.IsFreeColonist && !pawnCanPickAnyone)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): is a colonist and colonist hookups are disabled in mod settings");
				return null;
			}

			var targets = PotentialPartnersFor(pawn, bedsex).ToList();

			if (targets.Count <= 0)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): no eligible targets");
				return null;
			}

			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): considering {targets.Count} targets");

			// This will set the seed for this whole function.
			using var _rng = Seeded.ForHour(pawn, PartnerRNGSalt);

			// HippieMode means: everyone is allowed to have sex with everyone else ("free love")
			//	- no prioritization of partners
			//	- not necessary to be horny
			//	- never cheating
			// -> just pick a partner

			if (!RJWSettings.HippieMode)
			{
				// Prefer established lovers, if any exist.
				var splitPartners = targets.ToLookup(target => pawn.GetRelations(target).Any(LovePartnerRelationUtility.IsLovePartnerRelation));
				var lovers = splitPartners[true].ToList();
				targets = splitPartners[false].ToList();

				if (lovers.Count == 0)
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): no eligible love partners");
					// There may just be no currently eligible love partners.  We still need to check
					// if they can hook up and if it would be cheating.
					goto checkHookup;
				}

				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): considering {lovers.Count} partners first");

				// Use the default `ForFucking` for observer and candidates.
				var appraisalParams = new AppraisalParams(pawn, lovers);

				// Using `FindPartners`, would exclude any lover who is below average.
				// We'll run a custom search that is still weighted toward the more attractive
				// options, but can select any of the possible partners instead.
				var results = SexAppraiser.RNG.FindResults(SexAppraiser.CandidatesAsTargets(in appraisalParams));

				// If this is not empty, this will use the first option.
				foreach (var result in results)
				{
					// The partner needs to be actually reachable.
					// if (!Pather_Utility.can_path_to_target(pawn, result.Target)) continue;

					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): banging lover {xxx.get_pawnname(result.Target)}");
					return result.Target;
				}

			checkHookup:
				// No lovers available... see if the pawn fancies a hookup.  Nymphos and frustrated pawns always do!
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): no lovers in targets; checking CanPawnHookup");
				if (!RNG.CanPawnHookup(pawn, pawnIsNympho))
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): no hookup today");
					return null;
				}

				// No cheating from casual hookups... would probably make colony relationship management too annoying
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): checking CanFoolAround");
				
				// Check if any hookup would be cheating.
				if (!AttractionUtility.CanFoolAround(pawn, true))
				{
					if (RJWHookupSettings.NymphosCanCheat && pawnIsNympho && xxx.is_frustrated(pawn))
					{
						if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): I'm a nympho and I'm so frustrated that I'm going to cheat");
						// No return here so they continue searching for hookup
					}
					else if (pawn.HasBeerGoggles())
					{
						if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): I want to bang and im too drunk to care if its cheating");
					}
					else
					{
						if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): I want to bang but that's cheating");
						return null;
					}
				}
			}

			// Use a special appraisal function to find a partner.
			foreach (AppraisalResult result in RNG.FindQuickiePartner(pawn, targets))
			{
				// Make sure the configured minimum is enforced.
				if (!CasualHookupAllowedViaSettings(result)) continue;
				// We put this off as long as possible, but finally do a pathing check.
				if (!Pather_Utility.can_path_to_target(pawn, result.Target)) continue;

				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): found rando {xxx.get_pawnname(result.Target)} with score {result.CombinedAttraction}");
				return result.Target;
			}

			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindPartner({pawnName}): couldn't find anyone to bang");
			return null;
		}

		public static bool CasualHookupAllowedViaSettings(AppraisalResult result)
		{
			GetCasualHookupSettingsForPawn(result.Observer, out float observerMinFuckability, out int observerMinRelationship, out float observerMinAttractiveness);
			GetCasualHookupSettingsForPawn(result.Target, out float targetMinFuckability, out int targetMinRelationship, out float targetMinAttractiveness);

			if (result.ObserverAttraction < observerMinFuckability)
				return false;

			if (result.TargetAttraction < targetMinFuckability)
				return false;

			if (xxx.is_animal(result.Target))
				return true;

			// Humanlike checks
			int relations = result.Observer.relations.OpinionOf(result.Target);
			if (relations < observerMinRelationship)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindBestPartner({result.Observer}): hookup {xxx.get_pawnname(result.Target)}, i don't like them:({relations})");
				return false;
			}

			relations = result.Target.relations.OpinionOf(result.Observer);
			if (relations < targetMinRelationship)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindBestPartner({result.Observer}): hookup {xxx.get_pawnname(result.Target)}, don't like me:({relations})");
				return false;
			}

			float attraction = result.Observer.relations.SecondaryRomanceChanceFactor(result.Target);
			if (attraction < observerMinAttractiveness)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindBestPartner({result.Observer}): hookup {xxx.get_pawnname(result.Target)}, i don't find them attractive:({attraction})");
				return false;
			}
			attraction = result.Target.relations.SecondaryRomanceChanceFactor(result.Observer);
			if (attraction < targetMinAttractiveness)
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" FindBestPartner({result.Observer}): hookup {xxx.get_pawnname(result.Target)}, doesn't find me attractive:({attraction})");
				return false;
			}

			return true;
		}

		// Menstruation patches this method
		public static void GetCasualHookupSettingsForPawn(Pawn pawn, out float minimumFuckability, out int minimumRelationship, out float minimumAttractiveness)
		{
			if (xxx.is_nympho(pawn))
			{
				minimumFuckability = 0f;
				minimumRelationship = 0;
				minimumAttractiveness = 0f;
				return;
			}

			minimumFuckability = RJWHookupSettings.MinimumFuckabilityToHookup;
			minimumRelationship = RJWHookupSettings.MinimumRelationshipToHookup;
			minimumAttractiveness = RJWHookupSettings.MinimumAttractivenessToHookup;
		}

		public static bool CanTargetHookup(Pawn pawn, Pawn target)
		{
			var pawnName = xxx.get_pawnname(pawn);
			var targetName = xxx.get_pawnname(target);

			// Already checked in `PotentialPartnersFor`, but this is a public function.
			if (target.relations is not { } tRelations) return false;

			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" CanTargetHookup({pawnName}): checking hookup {targetName}");

			// Check to see if the mod settings for hookups allow this pairing
			if (!RJWSettings.WildMode && !HookupAllowedViaSettings(pawn, target)) return false;

			// If the pawn has had sex recently and isn't horny right now, skip them.
			if (!SexUtility.ReadyForLovin(target) && !xxx.is_hornyorfrustrated(target))
			{
				if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" CanTargetHookup({pawnName}): hookup {targetName} isn't ready for lovin'");
				return false;
			}

			if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" CanTargetHookup({pawnName}): hookup {targetName} is sufficiently available");
			return true;
		}

		/// <summary> Checks to see if the mod settings allow the two pawns to hookup. </summary>
		public static bool HookupAllowedViaSettings(Pawn pawn, Pawn targetPawn)
		{
			// Can prisoners hook up?
			if (pawn.IsPrisonerOfColony || pawn.IsPrisoner || xxx.is_slave(pawn))
			{
				if (!RJWHookupSettings.PrisonersCanHookupWithNonPrisoner && !(targetPawn.IsPrisonerOfColony || targetPawn.IsPrisoner || xxx.is_slave(targetPawn)))
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting PrisonersCanHookupWithNonPrisoner");
					return false;
				}

				if (!RJWHookupSettings.PrisonersCanHookupWithPrisoner && (targetPawn.IsPrisonerOfColony || targetPawn.IsPrisoner || xxx.is_slave(targetPawn)))
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting PrisonersCanHookupWithPrisoner");
					return false;
				}
			}
			else
			{
				// Can non prisoners hook up with prisoners?
				if (!RJWHookupSettings.CanHookupWithPrisoner && (targetPawn.IsPrisonerOfColony || targetPawn.IsPrisoner || xxx.is_slave(targetPawn)))
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting CanHookupWithPrisoner");
					return false;
				}
			}

			// Can slave hook up?
			//if (xxx.is_slave(pawn))
			//{
			//	if (!RJWHookupSettings.SlaveCanHookup)
			//	{
			//		if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting SlaveCanHookup");
			//		return false;
			//	}
			//}

			// Can colonist hook up with visitors?
			if (pawn.IsFreeColonist && !xxx.is_slave(pawn))
			{
				if (!RJWHookupSettings.ColonistsCanHookupWithVisitor && targetPawn.Faction != Faction.OfPlayer && !targetPawn.IsPrisonerOfColony)
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting ColonistsCanHookupWithVisitor");
					return false;
				}
			}

			// Can visitors hook up?
			if (pawn.Faction != Faction.OfPlayer && !pawn.IsPrisonerOfColony)
			{
				// visitors vs colonist
				if (!RJWHookupSettings.VisitorsCanHookupWithColonists && targetPawn.IsColonist)
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting VisitorsCanHookupWithColonists");
					return false;
				}

				// visitors vs visitors
				if (!RJWHookupSettings.VisitorsCanHookupWithVisitors && targetPawn.Faction != Faction.OfPlayer)
				{
					if (RJWSettings.DebugLogJoinInBed) ModLog.Message($" find_partner({xxx.get_pawnname(pawn)}): not hooking up with {xxx.get_pawnname(targetPawn)} due to mod setting VisitorsCanHookupWithVisitors");
					return false;
				}
			}

			// TODO: Not sure if this handles all the pawn-on-animal cases.

			return true;
		}

		[SyncMethod]
		public static IntVec3 FindSexLocation(Pawn pawn, Pawn partner = null)
		{
			IntVec3 position = pawn.Position;
			int bestPosition = -100;
			IntVec3 cell = pawn.Position;

			List<Pawn> all_pawns = pawn.Map.mapPawns.AllPawnsSpawned.Where(x
				=> x.Position.DistanceTo(pawn.Position) < 100
				&& xxx.is_human(x)
				&& x != pawn
				&& x != partner
				).ToList();

			//ModLog.Message(" Pawn is " + xxx.get_pawnname(pawn) + ", current cell is " + cell);

			List<IntVec3> randomCells;
			using (Seeded.ForHour(pawn, LocationRNGSalt))
			{
				randomCells = RNG.RandomCellsAround(pawn.Position)
					.Take(50)
					.Distinct()
					.Where(x =>
						x.Standable(pawn.Map)
						&& x.InAllowedArea(pawn)
						&& x.GetDangerFor(pawn, pawn.Map) != Danger.Deadly
						&& !x.ContainsTrap(pawn.Map)
						&& !x.ContainsStaticFire(pawn.Map)
					)
					.ToList();
			}

			//ModLog.Message(" Found " + random_cells.Count + " valid cells.");

			static int GetRoomId(Room r) => r.ID;
			var randomCellsRoomsDoors = randomCells
				.Select((c) => c.GetRoom(pawn.Map))
				.DistinctBy(GetRoomId)
				.Where((r) => !r.PsychologicallyOutdoors)
				.ToDictionary(
					GetRoomId,
					(r) => r.ContainedAndAdjacentThings.Count(x =>
						x.def.IsDoor
						// dubs hyg toilets stall doors(false,false)?
						&& x.def.holdsRoof && x.def.blockLight
					)
				);

			foreach (IntVec3 randomCell in randomCells)
			{
				if (!Pather_Utility.cells_to_target_casual(pawn, randomCell))
					continue;// too far

				int doors = randomCellsRoomsDoors.TryGetValue(randomCell.GetRoom(pawn.Map).ID);
				int score = ScoreSexSpot(all_pawns, doors, randomCell, pawn, partner);

				if (score <= bestPosition) continue;

				bestPosition = score;
				cell = randomCell;
			}

			return cell;

			//ModLog.Message(" Best cell is " + cell);
		}

		/// <summary>
		/// Get how much pawn likes the idea of having sex in a particular cell.
		/// </summary>
		/// <param name="otherPawns">Other nearby pawns to consider</param>
		/// <param name="doorCount">If cell is in the room, how many doors the room has</param>
		/// <param name="cell">The spot coordinates</param>
		/// <param name="pawn">Pawn that selects a spot</param>
		/// <param name="partner">Partner, may be null in a case of masturbation</param>
		/// <param name="isExhibitionist">Is pawn exhibitionist. Currently set by the submods</param>
		/// <returns>The relative score. It has no meaning by itself</returns>
		public static int ScoreSexSpot(List<Pawn> otherPawns, int doorCount, IntVec3 cell, Pawn pawn, Pawn partner = null, bool isExhibitionist = false)
		{
			int score = 0;
			Room room = cell.GetRoom(pawn.Map);

			bool might_be_seen = MightBeSeen(otherPawns, cell, pawn, partner);

			if (isExhibitionist)
			{
				if (might_be_seen)
					score += 5;
				else
					score -= 10;
			}
			else
			{
				if (might_be_seen)
					score -= 30;
			}

			FloatRange temperature = pawn.ComfortableTemperatureRange();

			if (temperature.Includes(cell.GetTemperature(pawn.Map)))
				score += 20;
			else
				score -= 20;

			if (cell.Roofed(pawn.Map))
				score += 5;

			if (cell.HasEatSurface(pawn.Map))
				score += 5; // Hide in vegetation.

			Danger danger = cell.GetDangerFor(pawn, pawn.Map);

			if (danger == Danger.Some)
				score -= 25;
			else if (danger == Danger.None)
				score += 5;

			TerrainDef terrain = cell.GetTerrain(pawn.Map);

			// TODO: Check Use terrain.IsWater instead
			if (terrain == TerrainDefOf.WaterShallow ||
				terrain == TerrainDefOf.WaterMovingShallow ||
				terrain == TerrainDefOf.WaterOceanShallow)
				score -= 20;

			if (cell.GetThingList(pawn.Map).Any(x => x.def.IsWithinCategory(ThingCategoryDefOf.Corpses)))
				if (xxx.is_necrophiliac(pawn))
					score += 20;
				else
					score -= 20;

			if (cell.GetThingList(pawn.Map).Any(x => x.def.IsWithinCategory(ThingCategoryDefOf.Foods)))
				score -= 10;

			if (room == pawn.Position.GetRoom(pawn.MapHeld))
				score -= 10;

			if (room.PsychologicallyOutdoors)
				score += 5;

			if (room.IsHuge)
				score -= 5;

			if (room.ContainedBeds.Any())
			{
				score += 5;
				if (room.ContainedBeds.Any(x => x.Position == cell))
					score += 5;
			}

			if (!room.Owners.Any())
				score += 10;
			else if (room.Owners.Contains(pawn))
				score += 20;

			if (room.IsDoorway && !isExhibitionist)
				score -= 100;

			if (room.Role == RoomRoleDefOf.Bedroom)
				score += 10;

			else if (room.Role == RoomRoleDefOf.Barracks || room.Role == RoomRoleDefOf.PrisonBarracks || room.Role == RoomRoleDefOf.PrisonCell
				|| room.Role == VanillaRoomRoleDefOf.Laboratory || room.Role == VanillaRoomRoleDefOf.RecRoom
				|| room.Role == VanillaRoomRoleDefOf.DiningRoom || room.Role == RoomRoleDefOf.Hospital
				)
				if (isExhibitionist)
					score += 10;
				else
					score -= 5;

			if (room.GetStat(RoomStatDefOf.Cleanliness) < 0.01f)
				score -= 5;

			if (room.GetStat(RoomStatDefOf.GraveVisitingJoyGainFactor) > 0.1f)
				if (xxx.is_necrophiliac(pawn))
				{
					score += 5;
				}
				else
				{
					score -= 5;
				}

			if (doorCount > 1)
				if (isExhibitionist)
				{
					score += 2 * doorCount;
				}
				else
				{
					score -= 5 * doorCount;
				}
			
			return score;
		}

		public static bool MightBeSeen(List<Pawn> otherPawns, IntVec3 cell, Pawn pawn, Pawn partner = null)
		{
			return otherPawns.Any(x
					=> x != partner
					&& x.Awake()
					&& x.Position.DistanceTo(cell) < 50
					&& GenSight.LineOfSight(x.Position, cell, pawn.Map)
					);
		}

		/// <summary>
		/// These methods employ RNG.  You should be using a deterministic seed
		/// (using <see cref="Seeded" />) or using the Multiplayer API's
		/// `SyncMethod` attribute for methods responding to a player action.
		/// </summary>
		public static class RNG
		{
			public static bool CanPawnHookup(Pawn pawn, bool pawnIsNympho)
			{
				if (pawnIsNympho) return true;
				if (RJWSettings.WildMode) return true;
				if (xxx.is_frustrated(pawn)) return true;
				if (!xxx.is_horny(pawn)) return false;

				return Rand.Value < RJWHookupSettings.HookupChanceForNonNymphos;
			}

			public static IEnumerable<AppraisalResult> FindQuickiePartner(
				Pawn observer,
				IEnumerable<Pawn> candidates)
			{
				// Split into near and far targets.
				var splitTargets = candidates.ToLookup((t) => GetCloseness(observer, t));

				// Fast-path: try the near targets first; just use the best one.
				var nearParams = new AppraisalParams(observer, splitTargets[QuickieCloseness.Close]);
				var nearResults = SexAppraiser.CandidatesAsTargets(in nearParams);
				foreach (var result in SexAppraiser.FindBestResults(nearResults))
					yield return result;

				// Then the more distant targets.
				var farParams = new AppraisalParams(observer, splitTargets[QuickieCloseness.InRange]);
				var farResults = SexAppraiser.CandidatesAsTargets(in farParams);
				foreach (var result in SexAppraiser.RNG.FindBiasedResults(farResults))
					yield return result;
			}

			/// <summary>
			/// <para>Infinitely yields random locations around the given position.</para>
			/// <para>THIS ENUMERABLE DOES NOT TERMINATE; use with `Take` or some other
			/// method to limit how many results it generates.</para>
			/// </summary>
			/// <param name="position">The position.</param>
			/// <returns>An inifinite enumerable of positions.</returns>
			public static IEnumerable<IntVec3> RandomCellsAround(IntVec3 position)
			{
				while (true)
				{
					float angle = Rand.Range(0f, 360f);
					int dist = Rand.RangeInclusive(1, MaxLocationDistance);
					var offset = IntVec3.FromPolar(angle, dist);

					yield return position + offset;
				}
			}
		}
	}
}
