using RimWorld;
using System.Collections.Generic;
using System.Linq;
using Verse;
using System;

namespace rjw.RMB
{
	/// <summary>
	/// Generator of RMB categories for breeding
	/// </summary>
	static class RMB_RapeAnimal
	{
		/// <summary>
		/// Add breeding and reverse breeding options to <paramref name="opts"/>
		/// </summary>
		/// <param name="pawn">Selected pawn</param>
		/// <param name="opts">List of RMB options</param>
		/// <param name="target">Target of the right click</param>
		public static void AddFloatMenuOptions(Pawn pawn, List<FloatMenuOption> opts, LocalTargetInfo target)
		{
			AcceptanceReport canCreateEntries = DoBasicChecks(pawn, target.Pawn);

			if (!canCreateEntries)
			{
				if (RJWSettings.DevMode && !canCreateEntries.Reason.NullOrEmpty())
					opts.Add(new FloatMenuOption(canCreateEntries.Reason, null));

				return;
			}

			canCreateEntries = DoChecks(pawn, target.Pawn);

			if (!canCreateEntries)
			{
				opts.Add(new FloatMenuOption("RJW_RMB_RapeAnimal".Translate(target.Pawn) + ": " + canCreateEntries.Reason, null));
				return;
			}

			FloatMenuOption opt = GenerateCategoryOption(pawn, target);
			if (opt != null)
				opts.Add(opt);

			opt = GenerateCategoryOption(pawn, target, true);
			if (opt != null)
				opts.Add(opt);
		}

		/// <summary>
		/// Check for the things that should be obvious to the player or does not change for particular pawns.
		/// </summary>
		/// <returns>
		/// AcceptanceReport. Reason is an untranslated string and should only be shown in the DevMode
		/// </returns>
		private static AcceptanceReport DoBasicChecks(Pawn pawn, Pawn target)
		{
			if (!RJWSettings.rape_enabled)
			{
				return "No breeding: Rape is disabled by the mod settings";
			}

			if (!RJWSettings.bestiality_enabled)
			{
				return "No breeding: Bestiality is disabled by the mod settings";
			}

			if (!xxx.can_rape(pawn, true))
			{
				return "No breeding: Pawn can't rape";
			}

			if (!xxx.can_be_fucked(target))
			{
				return "No breeding: Target can't have sex";
			}

			if (target.HostileTo(pawn) && !target.Downed)
			{
				return "No breeding: Hostile must be downed";
			}

			return true;
		}

		/// <summary>
		/// Check for the things that can change for particular pawns.
		/// </summary>
		/// <returns>
		/// AcceptanceReport. Reason is a translated string and should not be null.
		/// </returns>
		private static AcceptanceReport DoChecks(Pawn pawn, Pawn target)
		{
			if (!pawn.IsDesignatedHero() && !pawn.IsHeroOwner())
			{
				if (SexAppraiser.would_fuck(pawn, target) < 0.1f)
					return "RJW_RMB_ReasonUnappealingTarget".Translate();
			}

			return true;
		}

		/// <summary>
		/// Generate one FloatMenuOption
		/// </summary>
		/// <param name="pawn"></param>
		/// <param name="target"></param>
		/// <param name="reverse"></param>
		/// <returns>Category-level item that opens a sub-menu on click</returns>
		private static FloatMenuOption GenerateCategoryOption(Pawn pawn, LocalTargetInfo target, bool reverse = false)
		{
			string text = null;

			if (reverse)
			{
				text = "RJW_RMB_RapeAnimal_Reverse".Translate(target.Pawn);
			}
			else
			{
				text = "RJW_RMB_RapeAnimal".Translate(target.Pawn);
			}

			Action action = null;

			if (target.Pawn.HostileTo(pawn))
			{
				// Needs testing
				// Before separation from RMB_Rape animals, in theory, could get here.
				// But would the RapeEnemy job driver work properly?
				// Or maybe downed animals loose hostile status?
				action = delegate ()
				{
					JobDef job = xxx.RapeEnemy;
					var validinteractions = RMB_Menu.GenerateNonSoloSexRoleOptions(pawn, target, job, true, reverse);
					if (validinteractions.Any())
						FloatMenuUtility.MakeMenu(validinteractions, (FloatMenuOption opt) => opt.Label, (FloatMenuOption opt) => opt.action);
					else
						Messages.Message("No valid interactions found for " + text, MessageTypeDefOf.RejectInput, false);
				};
			}
			else if (!xxx.can_fuck(pawn) && !xxx.can_be_fucked(pawn))
			{
				if (RJWSettings.DevMode && !reverse)
				{
					return new FloatMenuOption("No breeding: Pawn can't fuck", null);
				}

				return null;
			}
			else
			{
				action = delegate ()
				{
					JobDef job = xxx.bestiality;
					var validinteractions = RMB_Menu.GenerateNonSoloSexRoleOptions(pawn, target, job, true, reverse);
					if (validinteractions.Any())
						FloatMenuUtility.MakeMenu(validinteractions, (FloatMenuOption opt) => opt.Label, (FloatMenuOption opt) => opt.action);
					else
						Messages.Message("No valid interactions found for " + text, MessageTypeDefOf.RejectInput, false);
				};
			}

			return FloatMenuUtility.DecoratePrioritizedTask(new FloatMenuOption(text, action, MenuOptionPriority.High), pawn, target);
		}
	}
}
