/*
 * Canopus HQX decoder
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef AVCODEC_HQXVLC_H
#define AVCODEC_HQXVLC_H

#include <stdint.h>

#include "vlc.h"

#include "libavutil/attributes.h"
#include "libavutil/macros.h"

#define HQX_CBP_VLC_BITS 5
#define HQX_DC_VLC_BITS 9

enum HQXACMode {
    HQX_AC_Q0 = 0,
    HQX_AC_Q8,
    HQX_AC_Q16,
    HQX_AC_Q32,
    HQX_AC_Q64,
    HQX_AC_Q128,
    NUM_HQX_AC
};

typedef struct HQXAC {
    int bits;
    const RL_VLC_ELEM *lut;
} HQXAC;

static const uint8_t cbp_vlc_bits[16] = {
    0x04, 0x1C, 0x1D, 0x09, 0x1E, 0x0B, 0x1B, 0x08,
    0x1F, 0x1A, 0x0C, 0x07, 0x0A, 0x06, 0x05, 0x00,
};

static const uint8_t cbp_vlc_lens[16] = {
    4, 5, 5, 4, 5, 4, 5, 4, 5, 5, 4, 4, 4, 4, 4, 2,
};

static const uint16_t dc9_vlc_bits[512] = {
    0x0010, 0x0008, 0x0022, 0x0024, 0x0026, 0x0028, 0x002A, 0x002C,
    0x002E, 0x0030, 0x0032, 0x0034, 0x0074, 0x0076, 0x0078, 0x007A,
    0x007C, 0x0000, 0x0002, 0x0004, 0x0006, 0x0008, 0x000A, 0x000C,
    0x000E, 0x0050, 0x0052, 0x0054, 0x0056, 0x0058, 0x005A, 0x005C,
    0x005E, 0x0020, 0x0022, 0x0024, 0x0026, 0x0028, 0x002A, 0x002C,
    0x002E, 0x0030, 0x0032, 0x0034, 0x0036, 0x0038, 0x003A, 0x003C,
    0x003E, 0x00C0, 0x00C2, 0x00C4, 0x00C6, 0x00C8, 0x00CA, 0x00CC,
    0x00CE, 0x00D0, 0x00D2, 0x00D4, 0x00D6, 0x00D8, 0x00DA, 0x00DC,
    0x00DE, 0x01C0, 0x01C2, 0x01C4, 0x01C6, 0x01C8, 0x01CA, 0x01CC,
    0x01CE, 0x01D0, 0x01D2, 0x01D4, 0x01D6, 0x01D8, 0x01DA, 0x01DC,
    0x01DE, 0x01E0, 0x01E2, 0x01E4, 0x01E6, 0x01E8, 0x01EA, 0x01EC,
    0x01EE, 0x01F0, 0x01F2, 0x01F4, 0x01F6, 0x01F8, 0x01FA, 0x01FC,
    0x01FE, 0x06C0, 0x06C2, 0x06C4, 0x06C6, 0x06C8, 0x06CA, 0x06CC,
    0x06CE, 0x06D0, 0x06D2, 0x06D4, 0x06D6, 0x06D8, 0x06DA, 0x06DC,
    0x06DE, 0x06E0, 0x06E2, 0x06E4, 0x06E6, 0x06E8, 0x06EA, 0x06EC,
    0x06EE, 0x06F0, 0x06F2, 0x06F4, 0x06F6, 0x06F8, 0x06FA, 0x06FC,
    0x06FE, 0x0E00, 0x0E02, 0x0E04, 0x0E06, 0x0E08, 0x0E0A, 0x0E0C,
    0x0E0E, 0x0E10, 0x0E12, 0x0E14, 0x0E16, 0x0E18, 0x0E1A, 0x0E1C,
    0x0E1E, 0x0E20, 0x0E22, 0x0E24, 0x0E26, 0x0E28, 0x0E2A, 0x0E2C,
    0x0E2E, 0x0E30, 0x0E32, 0x0E34, 0x0E36, 0x0E38, 0x0E3A, 0x0E3C,
    0x0E3E, 0x0E40, 0x0E42, 0x0E44, 0x0E46, 0x0E48, 0x0E4A, 0x0E4C,
    0x0E4E, 0x0E50, 0x0E52, 0x0E54, 0x0E56, 0x0E58, 0x0E5A, 0x0E5C,
    0x0E5E, 0x0E60, 0x0E62, 0x0E64, 0x0E66, 0x0E68, 0x0E6A, 0x0E6C,
    0x0E6E, 0x0E70, 0x0E72, 0x0E74, 0x0E76, 0x0E78, 0x0E7A, 0x0E7C,
    0x0E7E, 0x1F80, 0x1F82, 0x1F84, 0x1F86, 0x1F88, 0x1F8A, 0x1F8C,
    0x1F8E, 0x1F90, 0x1F92, 0x1F94, 0x1F96, 0x1F98, 0x1F9A, 0x1F9C,
    0x1F9E, 0x1FA0, 0x1FA2, 0x1FA4, 0x1FA6, 0x1FA8, 0x1FAA, 0x1FAC,
    0x1FAE, 0x1FB0, 0x1FB2, 0x1FB4, 0x1FB6, 0x1FB8, 0x1FBA, 0x1FBC,
    0x1FBE, 0x1FC0, 0x1FC2, 0x1FC4, 0x1FC6, 0x1FC8, 0x1FCA, 0x1FCC,
    0x1FCE, 0x1FD0, 0x1FD2, 0x1FD4, 0x1FD6, 0x1FD8, 0x1FDA, 0x1FDC,
    0x1FDE, 0x1FE0, 0x1FE2, 0x1FE4, 0x1FE6, 0x1FE8, 0x1FEA, 0x1FEC,
    0x1FEE, 0x1FF0, 0x1FF2, 0x1FF4, 0x1FF6, 0x1FF8, 0x1FFA, 0x1FFC,
    0x0FFF, 0x1FFD, 0x1FFB, 0x1FF9, 0x1FF7, 0x1FF5, 0x1FF3, 0x1FF1,
    0x1FEF, 0x1FED, 0x1FEB, 0x1FE9, 0x1FE7, 0x1FE5, 0x1FE3, 0x1FE1,
    0x1FDF, 0x1FDD, 0x1FDB, 0x1FD9, 0x1FD7, 0x1FD5, 0x1FD3, 0x1FD1,
    0x1FCF, 0x1FCD, 0x1FCB, 0x1FC9, 0x1FC7, 0x1FC5, 0x1FC3, 0x1FC1,
    0x1FBF, 0x1FBD, 0x1FBB, 0x1FB9, 0x1FB7, 0x1FB5, 0x1FB3, 0x1FB1,
    0x1FAF, 0x1FAD, 0x1FAB, 0x1FA9, 0x1FA7, 0x1FA5, 0x1FA3, 0x1FA1,
    0x1F9F, 0x1F9D, 0x1F9B, 0x1F99, 0x1F97, 0x1F95, 0x1F93, 0x1F91,
    0x1F8F, 0x1F8D, 0x1F8B, 0x1F89, 0x1F87, 0x1F85, 0x1F83, 0x1F81,
    0x0E7F, 0x0E7D, 0x0E7B, 0x0E79, 0x0E77, 0x0E75, 0x0E73, 0x0E71,
    0x0E6F, 0x0E6D, 0x0E6B, 0x0E69, 0x0E67, 0x0E65, 0x0E63, 0x0E61,
    0x0E5F, 0x0E5D, 0x0E5B, 0x0E59, 0x0E57, 0x0E55, 0x0E53, 0x0E51,
    0x0E4F, 0x0E4D, 0x0E4B, 0x0E49, 0x0E47, 0x0E45, 0x0E43, 0x0E41,
    0x0E3F, 0x0E3D, 0x0E3B, 0x0E39, 0x0E37, 0x0E35, 0x0E33, 0x0E31,
    0x0E2F, 0x0E2D, 0x0E2B, 0x0E29, 0x0E27, 0x0E25, 0x0E23, 0x0E21,
    0x0E1F, 0x0E1D, 0x0E1B, 0x0E19, 0x0E17, 0x0E15, 0x0E13, 0x0E11,
    0x0E0F, 0x0E0D, 0x0E0B, 0x0E09, 0x0E07, 0x0E05, 0x0E03, 0x0E01,
    0x06FF, 0x06FD, 0x06FB, 0x06F9, 0x06F7, 0x06F5, 0x06F3, 0x06F1,
    0x06EF, 0x06ED, 0x06EB, 0x06E9, 0x06E7, 0x06E5, 0x06E3, 0x06E1,
    0x06DF, 0x06DD, 0x06DB, 0x06D9, 0x06D7, 0x06D5, 0x06D3, 0x06D1,
    0x06CF, 0x06CD, 0x06CB, 0x06C9, 0x06C7, 0x06C5, 0x06C3, 0x06C1,
    0x01FF, 0x01FD, 0x01FB, 0x01F9, 0x01F7, 0x01F5, 0x01F3, 0x01F1,
    0x01EF, 0x01ED, 0x01EB, 0x01E9, 0x01E7, 0x01E5, 0x01E3, 0x01E1,
    0x01DF, 0x01DD, 0x01DB, 0x01D9, 0x01D7, 0x01D5, 0x01D3, 0x01D1,
    0x01CF, 0x01CD, 0x01CB, 0x01C9, 0x01C7, 0x01C5, 0x01C3, 0x01C1,
    0x00DF, 0x00DD, 0x00DB, 0x00D9, 0x00D7, 0x00D5, 0x00D3, 0x00D1,
    0x00CF, 0x00CD, 0x00CB, 0x00C9, 0x00C7, 0x00C5, 0x00C3, 0x00C1,
    0x003F, 0x003D, 0x003B, 0x0039, 0x0037, 0x0035, 0x0033, 0x0031,
    0x002F, 0x002D, 0x002B, 0x0029, 0x0027, 0x0025, 0x0023, 0x0021,
    0x005F, 0x005D, 0x005B, 0x0059, 0x0057, 0x0055, 0x0053, 0x0051,
    0x000F, 0x000D, 0x000B, 0x0009, 0x0007, 0x0005, 0x0003, 0x0001,
    0x007D, 0x007B, 0x0079, 0x0077, 0x0075, 0x0035, 0x0033, 0x0031,
    0x002F, 0x002D, 0x002B, 0x0029, 0x0027, 0x0025, 0x0023, 0x0009,
};

static const uint8_t dc9_vlc_lens[512] = {
     5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,
     7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,  7,  7,  7,  7,  7,
     7,  7,  7,  7,  7,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  5,
};

static const uint16_t dc10_vlc_bits[1024] = {
    0x0014, 0x002A, 0x002C, 0x002E, 0x0064, 0x0066, 0x0068, 0x006A,
    0x006C, 0x006E, 0x0070, 0x0072, 0x0074, 0x0076, 0x0078, 0x007A,
    0x007C, 0x0040, 0x0042, 0x0044, 0x0046, 0x0048, 0x004A, 0x004C,
    0x004E, 0x0050, 0x0052, 0x0054, 0x0056, 0x0058, 0x005A, 0x005C,
    0x005E, 0x0000, 0x0002, 0x0004, 0x0006, 0x0008, 0x000A, 0x000C,
    0x000E, 0x0010, 0x0012, 0x0014, 0x0016, 0x0018, 0x001A, 0x001C,
    0x001E, 0x00C0, 0x00C2, 0x00C4, 0x00C6, 0x00C8, 0x00CA, 0x00CC,
    0x00CE, 0x00D0, 0x00D2, 0x00D4, 0x00D6, 0x00D8, 0x00DA, 0x00DC,
    0x00DE, 0x0040, 0x0042, 0x0044, 0x0046, 0x0048, 0x004A, 0x004C,
    0x004E, 0x0050, 0x0052, 0x0054, 0x0056, 0x0058, 0x005A, 0x005C,
    0x005E, 0x0060, 0x0062, 0x0064, 0x0066, 0x0068, 0x006A, 0x006C,
    0x006E, 0x0070, 0x0072, 0x0074, 0x0076, 0x0078, 0x007A, 0x007C,
    0x007E, 0x01C0, 0x01C2, 0x01C4, 0x01C6, 0x01C8, 0x01CA, 0x01CC,
    0x01CE, 0x01D0, 0x01D2, 0x01D4, 0x01D6, 0x01D8, 0x01DA, 0x01DC,
    0x01DE, 0x01E0, 0x01E2, 0x01E4, 0x01E6, 0x01E8, 0x01EA, 0x01EC,
    0x01EE, 0x01F0, 0x01F2, 0x01F4, 0x01F6, 0x01F8, 0x01FA, 0x01FC,
    0x01FE, 0x0400, 0x0402, 0x0404, 0x0406, 0x0408, 0x040A, 0x040C,
    0x040E, 0x0410, 0x0412, 0x0414, 0x0416, 0x0418, 0x041A, 0x041C,
    0x041E, 0x0420, 0x0422, 0x0424, 0x0426, 0x0428, 0x042A, 0x042C,
    0x042E, 0x0430, 0x0432, 0x0434, 0x0436, 0x0438, 0x043A, 0x043C,
    0x043E, 0x0440, 0x0442, 0x0444, 0x0446, 0x0448, 0x044A, 0x044C,
    0x044E, 0x0450, 0x0452, 0x0454, 0x0456, 0x0458, 0x045A, 0x045C,
    0x045E, 0x0460, 0x0462, 0x0464, 0x0466, 0x0468, 0x046A, 0x046C,
    0x046E, 0x0470, 0x0472, 0x0474, 0x0476, 0x0478, 0x047A, 0x047C,
    0x047E, 0x0C00, 0x0C02, 0x0C04, 0x0C06, 0x0C08, 0x0C0A, 0x0C0C,
    0x0C0E, 0x0C10, 0x0C12, 0x0C14, 0x0C16, 0x0C18, 0x0C1A, 0x0C1C,
    0x0C1E, 0x0C20, 0x0C22, 0x0C24, 0x0C26, 0x0C28, 0x0C2A, 0x0C2C,
    0x0C2E, 0x0C30, 0x0C32, 0x0C34, 0x0C36, 0x0C38, 0x0C3A, 0x0C3C,
    0x0C3E, 0x0C40, 0x0C42, 0x0C44, 0x0C46, 0x0C48, 0x0C4A, 0x0C4C,
    0x0C4E, 0x0C50, 0x0C52, 0x0C54, 0x0C56, 0x0C58, 0x0C5A, 0x0C5C,
    0x0C5E, 0x0C60, 0x0C62, 0x0C64, 0x0C66, 0x0C68, 0x0C6A, 0x0C6C,
    0x0C6E, 0x0C70, 0x0C72, 0x0C74, 0x0C76, 0x0C78, 0x0C7A, 0x0C7C,
    0x0C7E, 0x0900, 0x0902, 0x0904, 0x0906, 0x0908, 0x090A, 0x090C,
    0x090E, 0x0910, 0x0912, 0x0914, 0x0916, 0x0918, 0x091A, 0x091C,
    0x091E, 0x0920, 0x0922, 0x0924, 0x0926, 0x0928, 0x092A, 0x092C,
    0x092E, 0x0930, 0x0932, 0x0934, 0x0936, 0x0938, 0x093A, 0x093C,
    0x093E, 0x0940, 0x0942, 0x0944, 0x0946, 0x0948, 0x094A, 0x094C,
    0x094E, 0x0950, 0x0952, 0x0954, 0x0956, 0x0958, 0x095A, 0x095C,
    0x095E, 0x0960, 0x0962, 0x0964, 0x0966, 0x0968, 0x096A, 0x096C,
    0x096E, 0x0970, 0x0972, 0x0974, 0x0976, 0x0978, 0x097A, 0x097C,
    0x097E, 0x0980, 0x0982, 0x0984, 0x0986, 0x0988, 0x098A, 0x098C,
    0x098E, 0x0990, 0x0992, 0x0994, 0x0996, 0x0998, 0x099A, 0x099C,
    0x099E, 0x09A0, 0x09A2, 0x09A4, 0x09A6, 0x09A8, 0x09AA, 0x09AC,
    0x09AE, 0x09B0, 0x09B2, 0x09B4, 0x09B6, 0x09B8, 0x09BA, 0x09BC,
    0x09BE, 0x09C0, 0x09C2, 0x09C4, 0x09C6, 0x09C8, 0x09CA, 0x09CC,
    0x09CE, 0x09D0, 0x09D2, 0x09D4, 0x09D6, 0x09D8, 0x09DA, 0x09DC,
    0x09DE, 0x09E0, 0x09E2, 0x09E4, 0x09E6, 0x09E8, 0x09EA, 0x09EC,
    0x09EE, 0x09F0, 0x09F2, 0x09F4, 0x09F6, 0x09F8, 0x09FA, 0x09FC,
    0x09FE, 0x3F00, 0x3F02, 0x3F04, 0x3F06, 0x3F08, 0x3F0A, 0x3F0C,
    0x3F0E, 0x3F10, 0x3F12, 0x3F14, 0x3F16, 0x3F18, 0x3F1A, 0x3F1C,
    0x3F1E, 0x3F20, 0x3F22, 0x3F24, 0x3F26, 0x3F28, 0x3F2A, 0x3F2C,
    0x3F2E, 0x3F30, 0x3F32, 0x3F34, 0x3F36, 0x3F38, 0x3F3A, 0x3F3C,
    0x3F3E, 0x3F40, 0x3F42, 0x3F44, 0x3F46, 0x3F48, 0x3F4A, 0x3F4C,
    0x3F4E, 0x3F50, 0x3F52, 0x3F54, 0x3F56, 0x3F58, 0x3F5A, 0x3F5C,
    0x3F5E, 0x3F60, 0x3F62, 0x3F64, 0x3F66, 0x3F68, 0x3F6A, 0x3F6C,
    0x3F6E, 0x3F70, 0x3F72, 0x3F74, 0x3F76, 0x3F78, 0x3F7A, 0x3F7C,
    0x3F7E, 0x3F80, 0x3F82, 0x3F84, 0x3F86, 0x3F88, 0x3F8A, 0x3F8C,
    0x3F8E, 0x3F90, 0x3F92, 0x3F94, 0x3F96, 0x3F98, 0x3F9A, 0x3F9C,
    0x3F9E, 0x3FA0, 0x3FA2, 0x3FA4, 0x3FA6, 0x3FA8, 0x3FAA, 0x3FAC,
    0x3FAE, 0x3FB0, 0x3FB2, 0x3FB4, 0x3FB6, 0x3FB8, 0x3FBA, 0x3FBC,
    0x3FBE, 0x3FC0, 0x3FC2, 0x3FC4, 0x3FC6, 0x3FC8, 0x3FCA, 0x3FCC,
    0x3FCE, 0x3FD0, 0x3FD2, 0x3FD4, 0x3FD6, 0x3FD8, 0x3FDA, 0x3FDC,
    0x3FDE, 0x3FE0, 0x3FE2, 0x3FE4, 0x3FE6, 0x3FE8, 0x3FEA, 0x3FEC,
    0x3FEE, 0x3FF0, 0x3FF2, 0x3FF4, 0x3FF6, 0x3FF8, 0x3FFA, 0x3FFC,
    0x1FFF, 0x3FFD, 0x3FFB, 0x3FF9, 0x3FF7, 0x3FF5, 0x3FF3, 0x3FF1,
    0x3FEF, 0x3FED, 0x3FEB, 0x3FE9, 0x3FE7, 0x3FE5, 0x3FE3, 0x3FE1,
    0x3FDF, 0x3FDD, 0x3FDB, 0x3FD9, 0x3FD7, 0x3FD5, 0x3FD3, 0x3FD1,
    0x3FCF, 0x3FCD, 0x3FCB, 0x3FC9, 0x3FC7, 0x3FC5, 0x3FC3, 0x3FC1,
    0x3FBF, 0x3FBD, 0x3FBB, 0x3FB9, 0x3FB7, 0x3FB5, 0x3FB3, 0x3FB1,
    0x3FAF, 0x3FAD, 0x3FAB, 0x3FA9, 0x3FA7, 0x3FA5, 0x3FA3, 0x3FA1,
    0x3F9F, 0x3F9D, 0x3F9B, 0x3F99, 0x3F97, 0x3F95, 0x3F93, 0x3F91,
    0x3F8F, 0x3F8D, 0x3F8B, 0x3F89, 0x3F87, 0x3F85, 0x3F83, 0x3F81,
    0x3F7F, 0x3F7D, 0x3F7B, 0x3F79, 0x3F77, 0x3F75, 0x3F73, 0x3F71,
    0x3F6F, 0x3F6D, 0x3F6B, 0x3F69, 0x3F67, 0x3F65, 0x3F63, 0x3F61,
    0x3F5F, 0x3F5D, 0x3F5B, 0x3F59, 0x3F57, 0x3F55, 0x3F53, 0x3F51,
    0x3F4F, 0x3F4D, 0x3F4B, 0x3F49, 0x3F47, 0x3F45, 0x3F43, 0x3F41,
    0x3F3F, 0x3F3D, 0x3F3B, 0x3F39, 0x3F37, 0x3F35, 0x3F33, 0x3F31,
    0x3F2F, 0x3F2D, 0x3F2B, 0x3F29, 0x3F27, 0x3F25, 0x3F23, 0x3F21,
    0x3F1F, 0x3F1D, 0x3F1B, 0x3F19, 0x3F17, 0x3F15, 0x3F13, 0x3F11,
    0x3F0F, 0x3F0D, 0x3F0B, 0x3F09, 0x3F07, 0x3F05, 0x3F03, 0x3F01,
    0x09FF, 0x09FD, 0x09FB, 0x09F9, 0x09F7, 0x09F5, 0x09F3, 0x09F1,
    0x09EF, 0x09ED, 0x09EB, 0x09E9, 0x09E7, 0x09E5, 0x09E3, 0x09E1,
    0x09DF, 0x09DD, 0x09DB, 0x09D9, 0x09D7, 0x09D5, 0x09D3, 0x09D1,
    0x09CF, 0x09CD, 0x09CB, 0x09C9, 0x09C7, 0x09C5, 0x09C3, 0x09C1,
    0x09BF, 0x09BD, 0x09BB, 0x09B9, 0x09B7, 0x09B5, 0x09B3, 0x09B1,
    0x09AF, 0x09AD, 0x09AB, 0x09A9, 0x09A7, 0x09A5, 0x09A3, 0x09A1,
    0x099F, 0x099D, 0x099B, 0x0999, 0x0997, 0x0995, 0x0993, 0x0991,
    0x098F, 0x098D, 0x098B, 0x0989, 0x0987, 0x0985, 0x0983, 0x0981,
    0x097F, 0x097D, 0x097B, 0x0979, 0x0977, 0x0975, 0x0973, 0x0971,
    0x096F, 0x096D, 0x096B, 0x0969, 0x0967, 0x0965, 0x0963, 0x0961,
    0x095F, 0x095D, 0x095B, 0x0959, 0x0957, 0x0955, 0x0953, 0x0951,
    0x094F, 0x094D, 0x094B, 0x0949, 0x0947, 0x0945, 0x0943, 0x0941,
    0x093F, 0x093D, 0x093B, 0x0939, 0x0937, 0x0935, 0x0933, 0x0931,
    0x092F, 0x092D, 0x092B, 0x0929, 0x0927, 0x0925, 0x0923, 0x0921,
    0x091F, 0x091D, 0x091B, 0x0919, 0x0917, 0x0915, 0x0913, 0x0911,
    0x090F, 0x090D, 0x090B, 0x0909, 0x0907, 0x0905, 0x0903, 0x0901,
    0x0C7F, 0x0C7D, 0x0C7B, 0x0C79, 0x0C77, 0x0C75, 0x0C73, 0x0C71,
    0x0C6F, 0x0C6D, 0x0C6B, 0x0C69, 0x0C67, 0x0C65, 0x0C63, 0x0C61,
    0x0C5F, 0x0C5D, 0x0C5B, 0x0C59, 0x0C57, 0x0C55, 0x0C53, 0x0C51,
    0x0C4F, 0x0C4D, 0x0C4B, 0x0C49, 0x0C47, 0x0C45, 0x0C43, 0x0C41,
    0x0C3F, 0x0C3D, 0x0C3B, 0x0C39, 0x0C37, 0x0C35, 0x0C33, 0x0C31,
    0x0C2F, 0x0C2D, 0x0C2B, 0x0C29, 0x0C27, 0x0C25, 0x0C23, 0x0C21,
    0x0C1F, 0x0C1D, 0x0C1B, 0x0C19, 0x0C17, 0x0C15, 0x0C13, 0x0C11,
    0x0C0F, 0x0C0D, 0x0C0B, 0x0C09, 0x0C07, 0x0C05, 0x0C03, 0x0C01,
    0x047F, 0x047D, 0x047B, 0x0479, 0x0477, 0x0475, 0x0473, 0x0471,
    0x046F, 0x046D, 0x046B, 0x0469, 0x0467, 0x0465, 0x0463, 0x0461,
    0x045F, 0x045D, 0x045B, 0x0459, 0x0457, 0x0455, 0x0453, 0x0451,
    0x044F, 0x044D, 0x044B, 0x0449, 0x0447, 0x0445, 0x0443, 0x0441,
    0x043F, 0x043D, 0x043B, 0x0439, 0x0437, 0x0435, 0x0433, 0x0431,
    0x042F, 0x042D, 0x042B, 0x0429, 0x0427, 0x0425, 0x0423, 0x0421,
    0x041F, 0x041D, 0x041B, 0x0419, 0x0417, 0x0415, 0x0413, 0x0411,
    0x040F, 0x040D, 0x040B, 0x0409, 0x0407, 0x0405, 0x0403, 0x0401,
    0x01FF, 0x01FD, 0x01FB, 0x01F9, 0x01F7, 0x01F5, 0x01F3, 0x01F1,
    0x01EF, 0x01ED, 0x01EB, 0x01E9, 0x01E7, 0x01E5, 0x01E3, 0x01E1,
    0x01DF, 0x01DD, 0x01DB, 0x01D9, 0x01D7, 0x01D5, 0x01D3, 0x01D1,
    0x01CF, 0x01CD, 0x01CB, 0x01C9, 0x01C7, 0x01C5, 0x01C3, 0x01C1,
    0x007F, 0x007D, 0x007B, 0x0079, 0x0077, 0x0075, 0x0073, 0x0071,
    0x006F, 0x006D, 0x006B, 0x0069, 0x0067, 0x0065, 0x0063, 0x0061,
    0x005F, 0x005D, 0x005B, 0x0059, 0x0057, 0x0055, 0x0053, 0x0051,
    0x004F, 0x004D, 0x004B, 0x0049, 0x0047, 0x0045, 0x0043, 0x0041,
    0x00DF, 0x00DD, 0x00DB, 0x00D9, 0x00D7, 0x00D5, 0x00D3, 0x00D1,
    0x00CF, 0x00CD, 0x00CB, 0x00C9, 0x00C7, 0x00C5, 0x00C3, 0x00C1,
    0x001F, 0x001D, 0x001B, 0x0019, 0x0017, 0x0015, 0x0013, 0x0011,
    0x000F, 0x000D, 0x000B, 0x0009, 0x0007, 0x0005, 0x0003, 0x0001,
    0x005F, 0x005D, 0x005B, 0x0059, 0x0057, 0x0055, 0x0053, 0x0051,
    0x004F, 0x004D, 0x004B, 0x0049, 0x0047, 0x0045, 0x0043, 0x0041,
    0x007D, 0x007B, 0x0079, 0x0077, 0x0075, 0x0073, 0x0071, 0x006F,
    0x006D, 0x006B, 0x0069, 0x0067, 0x0065, 0x002F, 0x002D, 0x002B,
};

static const uint8_t dc10_vlc_lens[1024] = {
     5,  6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
     7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    13, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  6,  6,  6,
};

static const uint16_t dc11_vlc_bits[2048] = {
    0x0032, 0x0066, 0x0068, 0x006A, 0x006C, 0x006E, 0x0070, 0x0072,
    0x0074, 0x00F0, 0x00F2, 0x00F4, 0x00F6, 0x00F8, 0x00FA, 0x00FC,
    0x00FE, 0x0040, 0x0042, 0x0044, 0x0046, 0x0048, 0x004A, 0x004C,
    0x004E, 0x0050, 0x0052, 0x0054, 0x0056, 0x0058, 0x005A, 0x005C,
    0x005E, 0x00C0, 0x00C2, 0x00C4, 0x00C6, 0x00C8, 0x00CA, 0x00CC,
    0x00CE, 0x00D0, 0x00D2, 0x00D4, 0x00D6, 0x00D8, 0x00DA, 0x00DC,
    0x00DE, 0x00E0, 0x00E2, 0x00E4, 0x00E6, 0x00E8, 0x00EA, 0x00EC,
    0x00EE, 0x00F0, 0x00F2, 0x00F4, 0x00F6, 0x00F8, 0x00FA, 0x00FC,
    0x00FE, 0x0000, 0x0002, 0x0004, 0x0006, 0x0008, 0x000A, 0x000C,
    0x000E, 0x0010, 0x0012, 0x0014, 0x0016, 0x0018, 0x001A, 0x001C,
    0x001E, 0x0020, 0x0022, 0x0024, 0x0026, 0x0028, 0x002A, 0x002C,
    0x002E, 0x0030, 0x0032, 0x0034, 0x0036, 0x0038, 0x003A, 0x003C,
    0x003E, 0x0200, 0x0202, 0x0204, 0x0206, 0x0208, 0x020A, 0x020C,
    0x020E, 0x0210, 0x0212, 0x0214, 0x0216, 0x0218, 0x021A, 0x021C,
    0x021E, 0x0220, 0x0222, 0x0224, 0x0226, 0x0228, 0x022A, 0x022C,
    0x022E, 0x0230, 0x0232, 0x0234, 0x0236, 0x0238, 0x023A, 0x023C,
    0x023E, 0x0080, 0x0082, 0x0084, 0x0086, 0x0088, 0x008A, 0x008C,
    0x008E, 0x0090, 0x0092, 0x0094, 0x0096, 0x0098, 0x009A, 0x009C,
    0x009E, 0x00A0, 0x00A2, 0x00A4, 0x00A6, 0x00A8, 0x00AA, 0x00AC,
    0x00AE, 0x00B0, 0x00B2, 0x00B4, 0x00B6, 0x00B8, 0x00BA, 0x00BC,
    0x00BE, 0x00C0, 0x00C2, 0x00C4, 0x00C6, 0x00C8, 0x00CA, 0x00CC,
    0x00CE, 0x00D0, 0x00D2, 0x00D4, 0x00D6, 0x00D8, 0x00DA, 0x00DC,
    0x00DE, 0x00E0, 0x00E2, 0x00E4, 0x00E6, 0x00E8, 0x00EA, 0x00EC,
    0x00EE, 0x00F0, 0x00F2, 0x00F4, 0x00F6, 0x00F8, 0x00FA, 0x00FC,
    0x00FE, 0x0480, 0x0482, 0x0484, 0x0486, 0x0488, 0x048A, 0x048C,
    0x048E, 0x0490, 0x0492, 0x0494, 0x0496, 0x0498, 0x049A, 0x049C,
    0x049E, 0x04A0, 0x04A2, 0x04A4, 0x04A6, 0x04A8, 0x04AA, 0x04AC,
    0x04AE, 0x04B0, 0x04B2, 0x04B4, 0x04B6, 0x04B8, 0x04BA, 0x04BC,
    0x04BE, 0x04C0, 0x04C2, 0x04C4, 0x04C6, 0x04C8, 0x04CA, 0x04CC,
    0x04CE, 0x04D0, 0x04D2, 0x04D4, 0x04D6, 0x04D8, 0x04DA, 0x04DC,
    0x04DE, 0x04E0, 0x04E2, 0x04E4, 0x04E6, 0x04E8, 0x04EA, 0x04EC,
    0x04EE, 0x04F0, 0x04F2, 0x04F4, 0x04F6, 0x04F8, 0x04FA, 0x04FC,
    0x04FE, 0x0A00, 0x0A02, 0x0A04, 0x0A06, 0x0A08, 0x0A0A, 0x0A0C,
    0x0A0E, 0x0A10, 0x0A12, 0x0A14, 0x0A16, 0x0A18, 0x0A1A, 0x0A1C,
    0x0A1E, 0x0A20, 0x0A22, 0x0A24, 0x0A26, 0x0A28, 0x0A2A, 0x0A2C,
    0x0A2E, 0x0A30, 0x0A32, 0x0A34, 0x0A36, 0x0A38, 0x0A3A, 0x0A3C,
    0x0A3E, 0x0A40, 0x0A42, 0x0A44, 0x0A46, 0x0A48, 0x0A4A, 0x0A4C,
    0x0A4E, 0x0A50, 0x0A52, 0x0A54, 0x0A56, 0x0A58, 0x0A5A, 0x0A5C,
    0x0A5E, 0x0A60, 0x0A62, 0x0A64, 0x0A66, 0x0A68, 0x0A6A, 0x0A6C,
    0x0A6E, 0x0A70, 0x0A72, 0x0A74, 0x0A76, 0x0A78, 0x0A7A, 0x0A7C,
    0x0A7E, 0x0A80, 0x0A82, 0x0A84, 0x0A86, 0x0A88, 0x0A8A, 0x0A8C,
    0x0A8E, 0x0A90, 0x0A92, 0x0A94, 0x0A96, 0x0A98, 0x0A9A, 0x0A9C,
    0x0A9E, 0x0AA0, 0x0AA2, 0x0AA4, 0x0AA6, 0x0AA8, 0x0AAA, 0x0AAC,
    0x0AAE, 0x0AB0, 0x0AB2, 0x0AB4, 0x0AB6, 0x0AB8, 0x0ABA, 0x0ABC,
    0x0ABE, 0x0AC0, 0x0AC2, 0x0AC4, 0x0AC6, 0x0AC8, 0x0ACA, 0x0ACC,
    0x0ACE, 0x0AD0, 0x0AD2, 0x0AD4, 0x0AD6, 0x0AD8, 0x0ADA, 0x0ADC,
    0x0ADE, 0x0AE0, 0x0AE2, 0x0AE4, 0x0AE6, 0x0AE8, 0x0AEA, 0x0AEC,
    0x0AEE, 0x0AF0, 0x0AF2, 0x0AF4, 0x0AF6, 0x0AF8, 0x0AFA, 0x0AFC,
    0x0AFE, 0x1800, 0x1802, 0x1804, 0x1806, 0x1808, 0x180A, 0x180C,
    0x180E, 0x1810, 0x1812, 0x1814, 0x1816, 0x1818, 0x181A, 0x181C,
    0x181E, 0x1820, 0x1822, 0x1824, 0x1826, 0x1828, 0x182A, 0x182C,
    0x182E, 0x1830, 0x1832, 0x1834, 0x1836, 0x1838, 0x183A, 0x183C,
    0x183E, 0x1840, 0x1842, 0x1844, 0x1846, 0x1848, 0x184A, 0x184C,
    0x184E, 0x1850, 0x1852, 0x1854, 0x1856, 0x1858, 0x185A, 0x185C,
    0x185E, 0x1860, 0x1862, 0x1864, 0x1866, 0x1868, 0x186A, 0x186C,
    0x186E, 0x1870, 0x1872, 0x1874, 0x1876, 0x1878, 0x187A, 0x187C,
    0x187E, 0x1880, 0x1882, 0x1884, 0x1886, 0x1888, 0x188A, 0x188C,
    0x188E, 0x1890, 0x1892, 0x1894, 0x1896, 0x1898, 0x189A, 0x189C,
    0x189E, 0x18A0, 0x18A2, 0x18A4, 0x18A6, 0x18A8, 0x18AA, 0x18AC,
    0x18AE, 0x18B0, 0x18B2, 0x18B4, 0x18B6, 0x18B8, 0x18BA, 0x18BC,
    0x18BE, 0x18C0, 0x18C2, 0x18C4, 0x18C6, 0x18C8, 0x18CA, 0x18CC,
    0x18CE, 0x18D0, 0x18D2, 0x18D4, 0x18D6, 0x18D8, 0x18DA, 0x18DC,
    0x18DE, 0x18E0, 0x18E2, 0x18E4, 0x18E6, 0x18E8, 0x18EA, 0x18EC,
    0x18EE, 0x18F0, 0x18F2, 0x18F4, 0x18F6, 0x18F8, 0x18FA, 0x18FC,
    0x18FE, 0x1600, 0x1602, 0x1604, 0x1606, 0x1608, 0x160A, 0x160C,
    0x160E, 0x1610, 0x1612, 0x1614, 0x1616, 0x1618, 0x161A, 0x161C,
    0x161E, 0x1620, 0x1622, 0x1624, 0x1626, 0x1628, 0x162A, 0x162C,
    0x162E, 0x1630, 0x1632, 0x1634, 0x1636, 0x1638, 0x163A, 0x163C,
    0x163E, 0x1640, 0x1642, 0x1644, 0x1646, 0x1648, 0x164A, 0x164C,
    0x164E, 0x1650, 0x1652, 0x1654, 0x1656, 0x1658, 0x165A, 0x165C,
    0x165E, 0x1660, 0x1662, 0x1664, 0x1666, 0x1668, 0x166A, 0x166C,
    0x166E, 0x1670, 0x1672, 0x1674, 0x1676, 0x1678, 0x167A, 0x167C,
    0x167E, 0x1680, 0x1682, 0x1684, 0x1686, 0x1688, 0x168A, 0x168C,
    0x168E, 0x1690, 0x1692, 0x1694, 0x1696, 0x1698, 0x169A, 0x169C,
    0x169E, 0x16A0, 0x16A2, 0x16A4, 0x16A6, 0x16A8, 0x16AA, 0x16AC,
    0x16AE, 0x16B0, 0x16B2, 0x16B4, 0x16B6, 0x16B8, 0x16BA, 0x16BC,
    0x16BE, 0x16C0, 0x16C2, 0x16C4, 0x16C6, 0x16C8, 0x16CA, 0x16CC,
    0x16CE, 0x16D0, 0x16D2, 0x16D4, 0x16D6, 0x16D8, 0x16DA, 0x16DC,
    0x16DE, 0x16E0, 0x16E2, 0x16E4, 0x16E6, 0x16E8, 0x16EA, 0x16EC,
    0x16EE, 0x16F0, 0x16F2, 0x16F4, 0x16F6, 0x16F8, 0x16FA, 0x16FC,
    0x16FE, 0x1700, 0x1702, 0x1704, 0x1706, 0x1708, 0x170A, 0x170C,
    0x170E, 0x1710, 0x1712, 0x1714, 0x1716, 0x1718, 0x171A, 0x171C,
    0x171E, 0x1720, 0x1722, 0x1724, 0x1726, 0x1728, 0x172A, 0x172C,
    0x172E, 0x1730, 0x1732, 0x1734, 0x1736, 0x1738, 0x173A, 0x173C,
    0x173E, 0x1740, 0x1742, 0x1744, 0x1746, 0x1748, 0x174A, 0x174C,
    0x174E, 0x1750, 0x1752, 0x1754, 0x1756, 0x1758, 0x175A, 0x175C,
    0x175E, 0x1760, 0x1762, 0x1764, 0x1766, 0x1768, 0x176A, 0x176C,
    0x176E, 0x1770, 0x1772, 0x1774, 0x1776, 0x1778, 0x177A, 0x177C,
    0x177E, 0x1780, 0x1782, 0x1784, 0x1786, 0x1788, 0x178A, 0x178C,
    0x178E, 0x1790, 0x1792, 0x1794, 0x1796, 0x1798, 0x179A, 0x179C,
    0x179E, 0x17A0, 0x17A2, 0x17A4, 0x17A6, 0x17A8, 0x17AA, 0x17AC,
    0x17AE, 0x17B0, 0x17B2, 0x17B4, 0x17B6, 0x17B8, 0x17BA, 0x17BC,
    0x17BE, 0x17C0, 0x17C2, 0x17C4, 0x17C6, 0x17C8, 0x17CA, 0x17CC,
    0x17CE, 0x17D0, 0x17D2, 0x17D4, 0x17D6, 0x17D8, 0x17DA, 0x17DC,
    0x17DE, 0x17E0, 0x17E2, 0x17E4, 0x17E6, 0x17E8, 0x17EA, 0x17EC,
    0x17EE, 0x17F0, 0x17F2, 0x17F4, 0x17F6, 0x17F8, 0x17FA, 0x17FC,
    0x17FE, 0x7600, 0x7602, 0x7604, 0x7606, 0x7608, 0x760A, 0x760C,
    0x760E, 0x7610, 0x7612, 0x7614, 0x7616, 0x7618, 0x761A, 0x761C,
    0x761E, 0x7620, 0x7622, 0x7624, 0x7626, 0x7628, 0x762A, 0x762C,
    0x762E, 0x7630, 0x7632, 0x7634, 0x7636, 0x7638, 0x763A, 0x763C,
    0x763E, 0x7640, 0x7642, 0x7644, 0x7646, 0x7648, 0x764A, 0x764C,
    0x764E, 0x7650, 0x7652, 0x7654, 0x7656, 0x7658, 0x765A, 0x765C,
    0x765E, 0x7660, 0x7662, 0x7664, 0x7666, 0x7668, 0x766A, 0x766C,
    0x766E, 0x7670, 0x7672, 0x7674, 0x7676, 0x7678, 0x767A, 0x767C,
    0x767E, 0x7680, 0x7682, 0x7684, 0x7686, 0x7688, 0x768A, 0x768C,
    0x768E, 0x7690, 0x7692, 0x7694, 0x7696, 0x7698, 0x769A, 0x769C,
    0x769E, 0x76A0, 0x76A2, 0x76A4, 0x76A6, 0x76A8, 0x76AA, 0x76AC,
    0x76AE, 0x76B0, 0x76B2, 0x76B4, 0x76B6, 0x76B8, 0x76BA, 0x76BC,
    0x76BE, 0x76C0, 0x76C2, 0x76C4, 0x76C6, 0x76C8, 0x76CA, 0x76CC,
    0x76CE, 0x76D0, 0x76D2, 0x76D4, 0x76D6, 0x76D8, 0x76DA, 0x76DC,
    0x76DE, 0x76E0, 0x76E2, 0x76E4, 0x76E6, 0x76E8, 0x76EA, 0x76EC,
    0x76EE, 0x76F0, 0x76F2, 0x76F4, 0x76F6, 0x76F8, 0x76FA, 0x76FC,
    0x76FE, 0x7700, 0x7702, 0x7704, 0x7706, 0x7708, 0x770A, 0x770C,
    0x770E, 0x7710, 0x7712, 0x7714, 0x7716, 0x7718, 0x771A, 0x771C,
    0x771E, 0x7720, 0x7722, 0x7724, 0x7726, 0x7728, 0x772A, 0x772C,
    0x772E, 0x7730, 0x7732, 0x7734, 0x7736, 0x7738, 0x773A, 0x773C,
    0x773E, 0x7740, 0x7742, 0x7744, 0x7746, 0x7748, 0x774A, 0x774C,
    0x774E, 0x7750, 0x7752, 0x7754, 0x7756, 0x7758, 0x775A, 0x775C,
    0x775E, 0x7760, 0x7762, 0x7764, 0x7766, 0x7768, 0x776A, 0x776C,
    0x776E, 0x7770, 0x7772, 0x7774, 0x7776, 0x7778, 0x777A, 0x777C,
    0x777E, 0x7780, 0x7782, 0x7784, 0x7786, 0x7788, 0x778A, 0x778C,
    0x778E, 0x7790, 0x7792, 0x7794, 0x7796, 0x7798, 0x779A, 0x779C,
    0x779E, 0x77A0, 0x77A2, 0x77A4, 0x77A6, 0x77A8, 0x77AA, 0x77AC,
    0x77AE, 0x77B0, 0x77B2, 0x77B4, 0x77B6, 0x77B8, 0x77BA, 0x77BC,
    0x77BE, 0x77C0, 0x77C2, 0x77C4, 0x77C6, 0x77C8, 0x77CA, 0x77CC,
    0x77CE, 0x77D0, 0x77D2, 0x77D4, 0x77D6, 0x77D8, 0x77DA, 0x77DC,
    0x77DE, 0x77E0, 0x77E2, 0x77E4, 0x77E6, 0x77E8, 0x77EA, 0x77EC,
    0x77EE, 0x77F0, 0x77F2, 0x77F4, 0x77F6, 0x77F8, 0x77FA, 0x77FC,
    0x3BFF, 0x77FD, 0x77FB, 0x77F9, 0x77F7, 0x77F5, 0x77F3, 0x77F1,
    0x77EF, 0x77ED, 0x77EB, 0x77E9, 0x77E7, 0x77E5, 0x77E3, 0x77E1,
    0x77DF, 0x77DD, 0x77DB, 0x77D9, 0x77D7, 0x77D5, 0x77D3, 0x77D1,
    0x77CF, 0x77CD, 0x77CB, 0x77C9, 0x77C7, 0x77C5, 0x77C3, 0x77C1,
    0x77BF, 0x77BD, 0x77BB, 0x77B9, 0x77B7, 0x77B5, 0x77B3, 0x77B1,
    0x77AF, 0x77AD, 0x77AB, 0x77A9, 0x77A7, 0x77A5, 0x77A3, 0x77A1,
    0x779F, 0x779D, 0x779B, 0x7799, 0x7797, 0x7795, 0x7793, 0x7791,
    0x778F, 0x778D, 0x778B, 0x7789, 0x7787, 0x7785, 0x7783, 0x7781,
    0x777F, 0x777D, 0x777B, 0x7779, 0x7777, 0x7775, 0x7773, 0x7771,
    0x776F, 0x776D, 0x776B, 0x7769, 0x7767, 0x7765, 0x7763, 0x7761,
    0x775F, 0x775D, 0x775B, 0x7759, 0x7757, 0x7755, 0x7753, 0x7751,
    0x774F, 0x774D, 0x774B, 0x7749, 0x7747, 0x7745, 0x7743, 0x7741,
    0x773F, 0x773D, 0x773B, 0x7739, 0x7737, 0x7735, 0x7733, 0x7731,
    0x772F, 0x772D, 0x772B, 0x7729, 0x7727, 0x7725, 0x7723, 0x7721,
    0x771F, 0x771D, 0x771B, 0x7719, 0x7717, 0x7715, 0x7713, 0x7711,
    0x770F, 0x770D, 0x770B, 0x7709, 0x7707, 0x7705, 0x7703, 0x7701,
    0x76FF, 0x76FD, 0x76FB, 0x76F9, 0x76F7, 0x76F5, 0x76F3, 0x76F1,
    0x76EF, 0x76ED, 0x76EB, 0x76E9, 0x76E7, 0x76E5, 0x76E3, 0x76E1,
    0x76DF, 0x76DD, 0x76DB, 0x76D9, 0x76D7, 0x76D5, 0x76D3, 0x76D1,
    0x76CF, 0x76CD, 0x76CB, 0x76C9, 0x76C7, 0x76C5, 0x76C3, 0x76C1,
    0x76BF, 0x76BD, 0x76BB, 0x76B9, 0x76B7, 0x76B5, 0x76B3, 0x76B1,
    0x76AF, 0x76AD, 0x76AB, 0x76A9, 0x76A7, 0x76A5, 0x76A3, 0x76A1,
    0x769F, 0x769D, 0x769B, 0x7699, 0x7697, 0x7695, 0x7693, 0x7691,
    0x768F, 0x768D, 0x768B, 0x7689, 0x7687, 0x7685, 0x7683, 0x7681,
    0x767F, 0x767D, 0x767B, 0x7679, 0x7677, 0x7675, 0x7673, 0x7671,
    0x766F, 0x766D, 0x766B, 0x7669, 0x7667, 0x7665, 0x7663, 0x7661,
    0x765F, 0x765D, 0x765B, 0x7659, 0x7657, 0x7655, 0x7653, 0x7651,
    0x764F, 0x764D, 0x764B, 0x7649, 0x7647, 0x7645, 0x7643, 0x7641,
    0x763F, 0x763D, 0x763B, 0x7639, 0x7637, 0x7635, 0x7633, 0x7631,
    0x762F, 0x762D, 0x762B, 0x7629, 0x7627, 0x7625, 0x7623, 0x7621,
    0x761F, 0x761D, 0x761B, 0x7619, 0x7617, 0x7615, 0x7613, 0x7611,
    0x760F, 0x760D, 0x760B, 0x7609, 0x7607, 0x7605, 0x7603, 0x7601,
    0x17FF, 0x17FD, 0x17FB, 0x17F9, 0x17F7, 0x17F5, 0x17F3, 0x17F1,
    0x17EF, 0x17ED, 0x17EB, 0x17E9, 0x17E7, 0x17E5, 0x17E3, 0x17E1,
    0x17DF, 0x17DD, 0x17DB, 0x17D9, 0x17D7, 0x17D5, 0x17D3, 0x17D1,
    0x17CF, 0x17CD, 0x17CB, 0x17C9, 0x17C7, 0x17C5, 0x17C3, 0x17C1,
    0x17BF, 0x17BD, 0x17BB, 0x17B9, 0x17B7, 0x17B5, 0x17B3, 0x17B1,
    0x17AF, 0x17AD, 0x17AB, 0x17A9, 0x17A7, 0x17A5, 0x17A3, 0x17A1,
    0x179F, 0x179D, 0x179B, 0x1799, 0x1797, 0x1795, 0x1793, 0x1791,
    0x178F, 0x178D, 0x178B, 0x1789, 0x1787, 0x1785, 0x1783, 0x1781,
    0x177F, 0x177D, 0x177B, 0x1779, 0x1777, 0x1775, 0x1773, 0x1771,
    0x176F, 0x176D, 0x176B, 0x1769, 0x1767, 0x1765, 0x1763, 0x1761,
    0x175F, 0x175D, 0x175B, 0x1759, 0x1757, 0x1755, 0x1753, 0x1751,
    0x174F, 0x174D, 0x174B, 0x1749, 0x1747, 0x1745, 0x1743, 0x1741,
    0x173F, 0x173D, 0x173B, 0x1739, 0x1737, 0x1735, 0x1733, 0x1731,
    0x172F, 0x172D, 0x172B, 0x1729, 0x1727, 0x1725, 0x1723, 0x1721,
    0x171F, 0x171D, 0x171B, 0x1719, 0x1717, 0x1715, 0x1713, 0x1711,
    0x170F, 0x170D, 0x170B, 0x1709, 0x1707, 0x1705, 0x1703, 0x1701,
    0x16FF, 0x16FD, 0x16FB, 0x16F9, 0x16F7, 0x16F5, 0x16F3, 0x16F1,
    0x16EF, 0x16ED, 0x16EB, 0x16E9, 0x16E7, 0x16E5, 0x16E3, 0x16E1,
    0x16DF, 0x16DD, 0x16DB, 0x16D9, 0x16D7, 0x16D5, 0x16D3, 0x16D1,
    0x16CF, 0x16CD, 0x16CB, 0x16C9, 0x16C7, 0x16C5, 0x16C3, 0x16C1,
    0x16BF, 0x16BD, 0x16BB, 0x16B9, 0x16B7, 0x16B5, 0x16B3, 0x16B1,
    0x16AF, 0x16AD, 0x16AB, 0x16A9, 0x16A7, 0x16A5, 0x16A3, 0x16A1,
    0x169F, 0x169D, 0x169B, 0x1699, 0x1697, 0x1695, 0x1693, 0x1691,
    0x168F, 0x168D, 0x168B, 0x1689, 0x1687, 0x1685, 0x1683, 0x1681,
    0x167F, 0x167D, 0x167B, 0x1679, 0x1677, 0x1675, 0x1673, 0x1671,
    0x166F, 0x166D, 0x166B, 0x1669, 0x1667, 0x1665, 0x1663, 0x1661,
    0x165F, 0x165D, 0x165B, 0x1659, 0x1657, 0x1655, 0x1653, 0x1651,
    0x164F, 0x164D, 0x164B, 0x1649, 0x1647, 0x1645, 0x1643, 0x1641,
    0x163F, 0x163D, 0x163B, 0x1639, 0x1637, 0x1635, 0x1633, 0x1631,
    0x162F, 0x162D, 0x162B, 0x1629, 0x1627, 0x1625, 0x1623, 0x1621,
    0x161F, 0x161D, 0x161B, 0x1619, 0x1617, 0x1615, 0x1613, 0x1611,
    0x160F, 0x160D, 0x160B, 0x1609, 0x1607, 0x1605, 0x1603, 0x1601,
    0x18FF, 0x18FD, 0x18FB, 0x18F9, 0x18F7, 0x18F5, 0x18F3, 0x18F1,
    0x18EF, 0x18ED, 0x18EB, 0x18E9, 0x18E7, 0x18E5, 0x18E3, 0x18E1,
    0x18DF, 0x18DD, 0x18DB, 0x18D9, 0x18D7, 0x18D5, 0x18D3, 0x18D1,
    0x18CF, 0x18CD, 0x18CB, 0x18C9, 0x18C7, 0x18C5, 0x18C3, 0x18C1,
    0x18BF, 0x18BD, 0x18BB, 0x18B9, 0x18B7, 0x18B5, 0x18B3, 0x18B1,
    0x18AF, 0x18AD, 0x18AB, 0x18A9, 0x18A7, 0x18A5, 0x18A3, 0x18A1,
    0x189F, 0x189D, 0x189B, 0x1899, 0x1897, 0x1895, 0x1893, 0x1891,
    0x188F, 0x188D, 0x188B, 0x1889, 0x1887, 0x1885, 0x1883, 0x1881,
    0x187F, 0x187D, 0x187B, 0x1879, 0x1877, 0x1875, 0x1873, 0x1871,
    0x186F, 0x186D, 0x186B, 0x1869, 0x1867, 0x1865, 0x1863, 0x1861,
    0x185F, 0x185D, 0x185B, 0x1859, 0x1857, 0x1855, 0x1853, 0x1851,
    0x184F, 0x184D, 0x184B, 0x1849, 0x1847, 0x1845, 0x1843, 0x1841,
    0x183F, 0x183D, 0x183B, 0x1839, 0x1837, 0x1835, 0x1833, 0x1831,
    0x182F, 0x182D, 0x182B, 0x1829, 0x1827, 0x1825, 0x1823, 0x1821,
    0x181F, 0x181D, 0x181B, 0x1819, 0x1817, 0x1815, 0x1813, 0x1811,
    0x180F, 0x180D, 0x180B, 0x1809, 0x1807, 0x1805, 0x1803, 0x1801,
    0x0AFF, 0x0AFD, 0x0AFB, 0x0AF9, 0x0AF7, 0x0AF5, 0x0AF3, 0x0AF1,
    0x0AEF, 0x0AED, 0x0AEB, 0x0AE9, 0x0AE7, 0x0AE5, 0x0AE3, 0x0AE1,
    0x0ADF, 0x0ADD, 0x0ADB, 0x0AD9, 0x0AD7, 0x0AD5, 0x0AD3, 0x0AD1,
    0x0ACF, 0x0ACD, 0x0ACB, 0x0AC9, 0x0AC7, 0x0AC5, 0x0AC3, 0x0AC1,
    0x0ABF, 0x0ABD, 0x0ABB, 0x0AB9, 0x0AB7, 0x0AB5, 0x0AB3, 0x0AB1,
    0x0AAF, 0x0AAD, 0x0AAB, 0x0AA9, 0x0AA7, 0x0AA5, 0x0AA3, 0x0AA1,
    0x0A9F, 0x0A9D, 0x0A9B, 0x0A99, 0x0A97, 0x0A95, 0x0A93, 0x0A91,
    0x0A8F, 0x0A8D, 0x0A8B, 0x0A89, 0x0A87, 0x0A85, 0x0A83, 0x0A81,
    0x0A7F, 0x0A7D, 0x0A7B, 0x0A79, 0x0A77, 0x0A75, 0x0A73, 0x0A71,
    0x0A6F, 0x0A6D, 0x0A6B, 0x0A69, 0x0A67, 0x0A65, 0x0A63, 0x0A61,
    0x0A5F, 0x0A5D, 0x0A5B, 0x0A59, 0x0A57, 0x0A55, 0x0A53, 0x0A51,
    0x0A4F, 0x0A4D, 0x0A4B, 0x0A49, 0x0A47, 0x0A45, 0x0A43, 0x0A41,
    0x0A3F, 0x0A3D, 0x0A3B, 0x0A39, 0x0A37, 0x0A35, 0x0A33, 0x0A31,
    0x0A2F, 0x0A2D, 0x0A2B, 0x0A29, 0x0A27, 0x0A25, 0x0A23, 0x0A21,
    0x0A1F, 0x0A1D, 0x0A1B, 0x0A19, 0x0A17, 0x0A15, 0x0A13, 0x0A11,
    0x0A0F, 0x0A0D, 0x0A0B, 0x0A09, 0x0A07, 0x0A05, 0x0A03, 0x0A01,
    0x04FF, 0x04FD, 0x04FB, 0x04F9, 0x04F7, 0x04F5, 0x04F3, 0x04F1,
    0x04EF, 0x04ED, 0x04EB, 0x04E9, 0x04E7, 0x04E5, 0x04E3, 0x04E1,
    0x04DF, 0x04DD, 0x04DB, 0x04D9, 0x04D7, 0x04D5, 0x04D3, 0x04D1,
    0x04CF, 0x04CD, 0x04CB, 0x04C9, 0x04C7, 0x04C5, 0x04C3, 0x04C1,
    0x04BF, 0x04BD, 0x04BB, 0x04B9, 0x04B7, 0x04B5, 0x04B3, 0x04B1,
    0x04AF, 0x04AD, 0x04AB, 0x04A9, 0x04A7, 0x04A5, 0x04A3, 0x04A1,
    0x049F, 0x049D, 0x049B, 0x0499, 0x0497, 0x0495, 0x0493, 0x0491,
    0x048F, 0x048D, 0x048B, 0x0489, 0x0487, 0x0485, 0x0483, 0x0481,
    0x00FF, 0x00FD, 0x00FB, 0x00F9, 0x00F7, 0x00F5, 0x00F3, 0x00F1,
    0x00EF, 0x00ED, 0x00EB, 0x00E9, 0x00E7, 0x00E5, 0x00E3, 0x00E1,
    0x00DF, 0x00DD, 0x00DB, 0x00D9, 0x00D7, 0x00D5, 0x00D3, 0x00D1,
    0x00CF, 0x00CD, 0x00CB, 0x00C9, 0x00C7, 0x00C5, 0x00C3, 0x00C1,
    0x00BF, 0x00BD, 0x00BB, 0x00B9, 0x00B7, 0x00B5, 0x00B3, 0x00B1,
    0x00AF, 0x00AD, 0x00AB, 0x00A9, 0x00A7, 0x00A5, 0x00A3, 0x00A1,
    0x009F, 0x009D, 0x009B, 0x0099, 0x0097, 0x0095, 0x0093, 0x0091,
    0x008F, 0x008D, 0x008B, 0x0089, 0x0087, 0x0085, 0x0083, 0x0081,
    0x023F, 0x023D, 0x023B, 0x0239, 0x0237, 0x0235, 0x0233, 0x0231,
    0x022F, 0x022D, 0x022B, 0x0229, 0x0227, 0x0225, 0x0223, 0x0221,
    0x021F, 0x021D, 0x021B, 0x0219, 0x0217, 0x0215, 0x0213, 0x0211,
    0x020F, 0x020D, 0x020B, 0x0209, 0x0207, 0x0205, 0x0203, 0x0201,
    0x003F, 0x003D, 0x003B, 0x0039, 0x0037, 0x0035, 0x0033, 0x0031,
    0x002F, 0x002D, 0x002B, 0x0029, 0x0027, 0x0025, 0x0023, 0x0021,
    0x001F, 0x001D, 0x001B, 0x0019, 0x0017, 0x0015, 0x0013, 0x0011,
    0x000F, 0x000D, 0x000B, 0x0009, 0x0007, 0x0005, 0x0003, 0x0001,
    0x00FF, 0x00FD, 0x00FB, 0x00F9, 0x00F7, 0x00F5, 0x00F3, 0x00F1,
    0x00EF, 0x00ED, 0x00EB, 0x00E9, 0x00E7, 0x00E5, 0x00E3, 0x00E1,
    0x00DF, 0x00DD, 0x00DB, 0x00D9, 0x00D7, 0x00D5, 0x00D3, 0x00D1,
    0x00CF, 0x00CD, 0x00CB, 0x00C9, 0x00C7, 0x00C5, 0x00C3, 0x00C1,
    0x005F, 0x005D, 0x005B, 0x0059, 0x0057, 0x0055, 0x0053, 0x0051,
    0x004F, 0x004D, 0x004B, 0x0049, 0x0047, 0x0045, 0x0043, 0x0041,
    0x00FF, 0x00FD, 0x00FB, 0x00F9, 0x00F7, 0x00F5, 0x00F3, 0x00F1,
    0x0075, 0x0073, 0x0071, 0x006F, 0x006D, 0x006B, 0x0069, 0x0067,
};

static const uint8_t dc11_vlc_lens[2048] = {
     6,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    14, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  7,  7,  7,  7,  7,  7,  7,  7,
};


static HQXAC hqx_ac[NUM_HQX_AC] = {
    { 10 }, { 11 }, { 11 }, { 11 }, { 12 }, { 11 },
};

// level is in -255..255 range, run 0..64, so it fits into 16 bits.
// We offset run by 1 in order to include the implicit run of 1.
#define E(level, run) ((level * 128) | (run + 1))

static const int16_t hqx_ac_run_level[] = {
    // AC table Q0 - 815 elements
    E(   1,  0), E(  -1,  0), E(   2,  0), E(  -2,  0), E(   3,  0),
    E(  -3,  0), E(   4,  0), E(  -4,  0), E(   1,  1), E(  -1,  1),
    E(   5,  0), E(  -5,  0), E(   6,  0), E(  -6,  0), E(   7,  0),
    E(  -7,  0), E(   8,  0), E(  -8,  0), E(   0,  0), E(   1,  0),
    E(  -1,  0), E(   2,  0), E(  -2,  0), E(   3,  0), E(  -3,  0),
    E(   4,  0), E(  -4,  0), E(   5,  0), E(  -5,  0), E(   6,  0),
    E(  -6,  0), E(   7,  0), E(  -7,  0), E(   8,  0), E(  -8,  0),
    E(   9,  0), E(  -9,  0), E(  10,  0), E( -10,  0), E(  11,  0),
    E( -11,  0), E(  12,  0), E( -12,  0), E(  13,  0), E( -13,  0),
    E(  14,  0), E( -14,  0), E(  15,  0), E( -15,  0), E(  16,  0),
    E( -16,  0), E(  17,  0), E( -17,  0), E(  18,  0), E( -18,  0),
    E(  19,  0), E( -19,  0), E(  20,  0), E( -20,  0), E(  21,  0),
    E( -21,  0), E(  22,  0), E( -22,  0), E(  23,  0), E( -23,  0),
    E(  24,  0), E( -24,  0), E(  25,  0), E( -25,  0), E(  26,  0),
    E( -26,  0), E(  27,  0), E( -27,  0), E(  28,  0), E( -28,  0),
    E(  29,  0), E( -29,  0), E(  30,  0), E( -30,  0), E(  31,  0),
    E( -31,  0), E(  32,  0), E( -32,  0), E(  33,  0), E( -33,  0),
    E(  34,  0), E( -34,  0), E(  35,  0), E( -35,  0), E(  36,  0),
    E( -36,  0), E(  37,  0), E( -37,  0), E(  38,  0), E( -38,  0),
    E(  39,  0), E( -39,  0), E(  40,  0), E( -40,  0), E(  41,  0),
    E( -41,  0), E(  42,  0), E( -42,  0), E(  43,  0), E( -43,  0),
    E(  44,  0), E( -44,  0), E(  45,  0), E( -45,  0), E(  46,  0),
    E( -46,  0), E(  47,  0), E( -47,  0), E(  48,  0), E( -48,  0),
    E(  49,  0), E( -49,  0), E(  50,  0), E( -50,  0), E(  51,  0),
    E( -51,  0), E(  52,  0), E( -52,  0), E(  53,  0), E( -53,  0),
    E(  54,  0), E( -54,  0), E(  55,  0), E( -55,  0), E(  56,  0),
    E( -56,  0), E(  57,  0), E( -57,  0), E(  58,  0), E( -58,  0),
    E(  59,  0), E( -59,  0), E(  60,  0), E( -60,  0), E(  61,  0),
    E( -61,  0), E(  62,  0), E( -62,  0), E(  63,  0), E( -63,  0),
    E(  64,  0), E( -64,  0), E(  65,  0), E( -65,  0), E(  66,  0),
    E( -66,  0), E(  67,  0), E( -67,  0), E(  68,  0), E( -68,  0),
    E(  69,  0), E( -69,  0), E(  70,  0), E( -70,  0), E(  71,  0),
    E( -71,  0), E(  72,  0), E( -72,  0), E(  73,  0), E( -73,  0),
    E(  74,  0), E( -74,  0), E(  75,  0), E( -75,  0), E(  76,  0),
    E( -76,  0), E(  77,  0), E( -77,  0), E(  78,  0), E( -78,  0),
    E(  79,  0), E( -79,  0), E(  80,  0), E( -80,  0), E(  81,  0),
    E( -81,  0), E(  82,  0), E( -82,  0), E(  83,  0), E( -83,  0),
    E(  84,  0), E( -84,  0), E(  85,  0), E( -85,  0), E(  86,  0),
    E( -86,  0), E(  87,  0), E( -87,  0), E(  88,  0), E( -88,  0),
    E(  89,  0), E( -89,  0), E(  90,  0), E( -90,  0), E(  91,  0),
    E( -91,  0), E(  92,  0), E( -92,  0), E(  93,  0), E( -93,  0),
    E(  94,  0), E( -94,  0), E(  95,  0), E( -95,  0), E(  96,  0),
    E( -96,  0), E(  97,  0), E( -97,  0), E(  98,  0), E( -98,  0),
    E(  99,  0), E( -99,  0), E( 100,  0), E(-100,  0), E( 101,  0),
    E(-101,  0), E( 102,  0), E(-102,  0), E( 103,  0), E(-103,  0),
    E( 104,  0), E(-104,  0), E( 105,  0), E(-105,  0), E( 106,  0),
    E(-106,  0), E( 107,  0), E(-107,  0), E( 108,  0), E(-108,  0),
    E( 109,  0), E(-109,  0), E( 110,  0), E(-110,  0), E( 111,  0),
    E(-111,  0), E( 112,  0), E(-112,  0), E( 113,  0), E(-113,  0),
    E( 114,  0), E(-114,  0), E( 115,  0), E(-115,  0), E( 116,  0),
    E(-116,  0), E( 117,  0), E(-117,  0), E( 118,  0), E(-118,  0),
    E( 119,  0), E(-119,  0), E( 120,  0), E(-120,  0), E( 121,  0),
    E(-121,  0), E( 122,  0), E(-122,  0), E( 123,  0), E(-123,  0),
    E( 124,  0), E(-124,  0), E( 125,  0), E(-125,  0), E( 126,  0),
    E(-126,  0), E( 127,  0), E(-127,  0), E( 128,  0), E(-128,  0),
    E( 129,  0), E(-129,  0), E( 130,  0), E(-130,  0), E( 131,  0),
    E(-131,  0), E( 132,  0), E(-132,  0), E( 133,  0), E(-133,  0),
    E( 134,  0), E(-134,  0), E( 135,  0), E(-135,  0), E( 136,  0),
    E(-136,  0), E( 137,  0), E(-137,  0), E( 138,  0), E(-138,  0),
    E( 139,  0), E(-139,  0), E( 140,  0), E(-140,  0), E( 141,  0),
    E(-141,  0), E( 142,  0), E(-142,  0), E( 143,  0), E(-143,  0),
    E( 144,  0), E(-144,  0), E( 145,  0), E(-145,  0), E( 146,  0),
    E(-146,  0), E( 147,  0), E(-147,  0), E( 148,  0), E(-148,  0),
    E( 149,  0), E(-149,  0), E( 150,  0), E(-150,  0), E( 151,  0),
    E(-151,  0), E( 152,  0), E(-152,  0), E( 153,  0), E(-153,  0),
    E( 154,  0), E(-154,  0), E( 155,  0), E(-155,  0), E( 156,  0),
    E(-156,  0), E( 157,  0), E(-157,  0), E( 158,  0), E(-158,  0),
    E( 159,  0), E(-159,  0), E( 160,  0), E(-160,  0), E( 161,  0),
    E(-161,  0), E( 162,  0), E(-162,  0), E( 163,  0), E(-163,  0),
    E( 164,  0), E(-164,  0), E( 165,  0), E(-165,  0), E( 166,  0),
    E(-166,  0), E( 167,  0), E(-167,  0), E( 168,  0), E(-168,  0),
    E( 169,  0), E(-169,  0), E( 170,  0), E(-170,  0), E( 171,  0),
    E(-171,  0), E( 172,  0), E(-172,  0), E( 173,  0), E(-173,  0),
    E( 174,  0), E(-174,  0), E( 175,  0), E(-175,  0), E( 176,  0),
    E(-176,  0), E( 177,  0), E(-177,  0), E( 178,  0), E(-178,  0),
    E( 179,  0), E(-179,  0), E( 180,  0), E(-180,  0), E( 181,  0),
    E(-181,  0), E( 182,  0), E(-182,  0), E( 183,  0), E(-183,  0),
    E( 184,  0), E(-184,  0), E( 185,  0), E(-185,  0), E( 186,  0),
    E(-186,  0), E( 187,  0), E(-187,  0), E( 188,  0), E(-188,  0),
    E( 189,  0), E(-189,  0), E( 190,  0), E(-190,  0), E( 191,  0),
    E(-191,  0), E( 192,  0), E(-192,  0), E( 193,  0), E(-193,  0),
    E( 194,  0), E(-194,  0), E( 195,  0), E(-195,  0), E( 196,  0),
    E(-196,  0), E( 197,  0), E(-197,  0), E( 198,  0), E(-198,  0),
    E( 199,  0), E(-199,  0), E( 200,  0), E(-200,  0), E( 201,  0),
    E(-201,  0), E( 202,  0), E(-202,  0), E( 203,  0), E(-203,  0),
    E( 204,  0), E(-204,  0), E( 205,  0), E(-205,  0), E( 206,  0),
    E(-206,  0), E( 207,  0), E(-207,  0), E( 208,  0), E(-208,  0),
    E( 209,  0), E(-209,  0), E( 210,  0), E(-210,  0), E( 211,  0),
    E(-211,  0), E( 212,  0), E(-212,  0), E( 213,  0), E(-213,  0),
    E( 214,  0), E(-214,  0), E( 215,  0), E(-215,  0), E( 216,  0),
    E(-216,  0), E( 217,  0), E(-217,  0), E( 218,  0), E(-218,  0),
    E( 219,  0), E(-219,  0), E( 220,  0), E(-220,  0), E( 221,  0),
    E(-221,  0), E( 222,  0), E(-222,  0), E( 223,  0), E(-223,  0),
    E( 224,  0), E(-224,  0), E( 225,  0), E(-225,  0), E( 226,  0),
    E(-226,  0), E( 227,  0), E(-227,  0), E( 228,  0), E(-228,  0),
    E( 229,  0), E(-229,  0), E( 230,  0), E(-230,  0), E( 231,  0),
    E(-231,  0), E( 232,  0), E(-232,  0), E( 233,  0), E(-233,  0),
    E( 234,  0), E(-234,  0), E( 235,  0), E(-235,  0), E( 236,  0),
    E(-236,  0), E( 237,  0), E(-237,  0), E( 238,  0), E(-238,  0),
    E( 239,  0), E(-239,  0), E( 240,  0), E(-240,  0), E( 241,  0),
    E(-241,  0), E( 242,  0), E(-242,  0), E( 243,  0), E(-243,  0),
    E( 244,  0), E(-244,  0), E( 245,  0), E(-245,  0), E( 246,  0),
    E(-246,  0), E( 247,  0), E(-247,  0), E( 248,  0), E(-248,  0),
    E( 249,  0), E(-249,  0), E( 250,  0), E(-250,  0), E( 251,  0),
    E(-251,  0), E( 252,  0), E(-252,  0), E( 253,  0), E(-253,  0),
    E( 254,  0), E(-254,  0), E( 255,  0), E(-255,  0), E(   0, 64),
    E(   9,  0), E(  -9,  0), E(  10,  0), E( -10,  0), E(  11,  0),
    E( -11,  0), E(  12,  0), E( -12,  0), E(  13,  0), E( -13,  0),
    E(  14,  0), E( -14,  0), E(   2,  1), E(  -2,  1), E(   1,  2),
    E(  -1,  2), E(  15,  0), E( -15,  0), E(  16,  0), E( -16,  0),
    E(  17,  0), E( -17,  0), E(  18,  0), E( -18,  0), E(  19,  0),
    E( -19,  0), E(  20,  0), E( -20,  0), E(  21,  0), E( -21,  0),
    E(   3,  1), E(  -3,  1), E(   4,  1), E(  -4,  1), E(   1,  3),
    E(  -1,  3), E(   1,  4), E(  -1,  4), E(   0,  0), E(  22,  0),
    E( -22,  0), E(  23,  0), E( -23,  0), E(  24,  0), E( -24,  0),
    E(  25,  0), E( -25,  0), E(  26,  0), E( -26,  0), E(  27,  0),
    E( -27,  0), E(  28,  0), E( -28,  0), E(  29,  0), E( -29,  0),
    E(  30,  0), E( -30,  0), E(  31,  0), E( -31,  0), E(  32,  0),
    E( -32,  0), E(  33,  0), E( -33,  0), E(   5,  1), E(  -5,  1),
    E(   6,  1), E(  -6,  1), E(   2,  2), E(  -2,  2), E(   1,  5),
    E(  -1,  5), E(   1,  6), E(  -1,  6), E(  34,  0), E( -34,  0),
    E(  35,  0), E( -35,  0), E(  36,  0), E( -36,  0), E(  37,  0),
    E( -37,  0), E(  38,  0), E( -38,  0), E(  39,  0), E( -39,  0),
    E(  40,  0), E( -40,  0), E(  41,  0), E( -41,  0), E(  42,  0),
    E( -42,  0), E(  43,  0), E( -43,  0), E(  44,  0), E( -44,  0),
    E(  45,  0), E( -45,  0), E(  46,  0), E( -46,  0), E(  47,  0),
    E( -47,  0), E(  48,  0), E( -48,  0), E(  49,  0), E( -49,  0),
    E(  50,  0), E( -50,  0), E(   0,  1), E(   7,  1), E(  -7,  1),
    E(   8,  1), E(  -8,  1), E(   9,  1), E(  -9,  1), E(  10,  1),
    E( -10,  1), E(   0,  2), E(   3,  2), E(  -3,  2), E(   0,  3),
    E(   2,  3), E(  -2,  3), E(   1,  7), E(  -1,  7), E(   1,  8),
    E(  -1,  8), E(   0,  0), E(   0,  1), E(   0,  2), E(   0,  3),
    E(   0,  4), E(   0,  5), E(   0,  6), E(   0,  7), E(   0,  8),
    E(   0,  9), E(   0, 10), E(   0, 11), E(   0, 12), E(   0, 13),
    E(   0, 14), E(   0, 15), E(   0, 16), E(   0, 17), E(   0, 18),
    E(   0, 19), E(   0, 20), E(   0, 21), E(   0, 22), E(   0, 23),
    E(   0, 24), E(   0, 25), E(   0, 26), E(   0, 27), E(   0, 28),
    E(   0, 29), E(   0, 30), E(   0, 31), E(   0, 32), E(   0, 33),
    E(   0, 34), E(   0, 35), E(   0, 36), E(   0, 37), E(   0, 38),
    E(   0, 39), E(   0, 40), E(   0, 41), E(   0, 42), E(   0, 43),
    E(   0, 44), E(   0, 45), E(   0, 46), E(   0, 47), E(   0, 48),
    E(   0, 49), E(   0, 50), E(   0, 51), E(   0, 52), E(   0, 53),
    E(   0, 54), E(   0, 55), E(   0, 56), E(   0, 57), E(   0, 58),
    E(   0, 59), E(   0, 60), E(   0, 61), E(   0, 62), E(   0, 63),
    E(  51,  0), E( -51,  0), E(  52,  0), E( -52,  0), E(  53,  0),
    E( -53,  0), E(  54,  0), E( -54,  0), E(  55,  0), E( -55,  0),
    E(  56,  0), E( -56,  0), E(  57,  0), E( -57,  0), E(  58,  0),
    E( -58,  0), E(  59,  0), E( -59,  0), E(  60,  0), E( -60,  0),
    E(  61,  0), E( -61,  0), E(  62,  0), E( -62,  0), E(  63,  0),
    E( -63,  0), E(  11,  1), E( -11,  1), E(  12,  1), E( -12,  1),
    E(  13,  1), E( -13,  1), E(  14,  1), E( -14,  1), E(   4,  2),
    E(  -4,  2), E(   5,  2), E(  -5,  2), E(   6,  2), E(  -6,  2),
    E(   3,  3), E(  -3,  3), E(   0,  4), E(   2,  4), E(  -2,  4),
    E(   0,  5), E(   0,  6), E(   1,  9), E(  -1,  9), E(   1, 10),
    E(  -1, 10), E(  15,  1), E( -15,  1), E(  16,  1), E( -16,  1),
    E(  17,  1), E( -17,  1), E(  18,  1), E( -18,  1), E(   7,  2),
    E(  -7,  2), E(   8,  2), E(  -8,  2), E(   9,  2), E(  -9,  2),
    E(  10,  2), E( -10,  2), E(   4,  3), E(  -4,  3), E(   5,  3),
    E(  -5,  3), E(   6,  3), E(  -6,  3), E(   2,  5), E(  -2,  5),
    E(   0,  7), E(   0,  8), E(   0,  9), E(   0, 10), E(   1, 11),
    E(  -1, 11), E(   1, 12), E(  -1, 12), E(   1, 13), E(  -1, 13),
    E(   1, 14), E(  -1, 14), E(  19,  1), E( -19,  1), E(  20,  1),
    E( -20,  1), E(   3,  4), E(  -3,  4), E(   2,  6), E(  -2,  6),
    // AC table Q8 - 907 elements
    E(   1,  0), E(  -1,  0), E(   2,  0), E(  -2,  0), E(   3,  0),
    E(  -3,  0), E(   4,  0), E(  -4,  0), E(   0, 64), E(   5,  0),
    E(  -5,  0), E(   6,  0), E(  -6,  0), E(   7,  0), E(  -7,  0),
    E(   8,  0), E(  -8,  0), E(   1,  1), E(  -1,  1), E(   2,  1),
    E(  -2,  1), E(   9,  0), E(  -9,  0), E(  10,  0), E( -10,  0),
    E(  11,  0), E( -11,  0), E(  12,  0), E( -12,  0), E(   3,  1),
    E(  -3,  1), E(   4,  1), E(  -4,  1), E(   1,  2), E(  -1,  2),
    E(   0,  0), E(   1,  0), E(  -1,  0), E(   2,  0), E(  -2,  0),
    E(   3,  0), E(  -3,  0), E(   4,  0), E(  -4,  0), E(   5,  0),
    E(  -5,  0), E(   6,  0), E(  -6,  0), E(   7,  0), E(  -7,  0),
    E(   8,  0), E(  -8,  0), E(   9,  0), E(  -9,  0), E(  10,  0),
    E( -10,  0), E(  11,  0), E( -11,  0), E(  12,  0), E( -12,  0),
    E(  13,  0), E( -13,  0), E(  14,  0), E( -14,  0), E(  15,  0),
    E( -15,  0), E(  16,  0), E( -16,  0), E(  17,  0), E( -17,  0),
    E(  18,  0), E( -18,  0), E(  19,  0), E( -19,  0), E(  20,  0),
    E( -20,  0), E(  21,  0), E( -21,  0), E(  22,  0), E( -22,  0),
    E(  23,  0), E( -23,  0), E(  24,  0), E( -24,  0), E(  25,  0),
    E( -25,  0), E(  26,  0), E( -26,  0), E(  27,  0), E( -27,  0),
    E(  28,  0), E( -28,  0), E(  29,  0), E( -29,  0), E(  30,  0),
    E( -30,  0), E(  31,  0), E( -31,  0), E(  32,  0), E( -32,  0),
    E(  33,  0), E( -33,  0), E(  34,  0), E( -34,  0), E(  35,  0),
    E( -35,  0), E(  36,  0), E( -36,  0), E(  37,  0), E( -37,  0),
    E(  38,  0), E( -38,  0), E(  39,  0), E( -39,  0), E(  40,  0),
    E( -40,  0), E(  41,  0), E( -41,  0), E(  42,  0), E( -42,  0),
    E(  43,  0), E( -43,  0), E(  44,  0), E( -44,  0), E(  45,  0),
    E( -45,  0), E(  46,  0), E( -46,  0), E(  47,  0), E( -47,  0),
    E(  48,  0), E( -48,  0), E(  49,  0), E( -49,  0), E(  50,  0),
    E( -50,  0), E(  51,  0), E( -51,  0), E(  52,  0), E( -52,  0),
    E(  53,  0), E( -53,  0), E(  54,  0), E( -54,  0), E(  55,  0),
    E( -55,  0), E(  56,  0), E( -56,  0), E(  57,  0), E( -57,  0),
    E(  58,  0), E( -58,  0), E(  59,  0), E( -59,  0), E(  60,  0),
    E( -60,  0), E(  61,  0), E( -61,  0), E(  62,  0), E( -62,  0),
    E(  63,  0), E( -63,  0), E(  64,  0), E( -64,  0), E(  65,  0),
    E( -65,  0), E(  66,  0), E( -66,  0), E(  67,  0), E( -67,  0),
    E(  68,  0), E( -68,  0), E(  69,  0), E( -69,  0), E(  70,  0),
    E( -70,  0), E(  71,  0), E( -71,  0), E(  72,  0), E( -72,  0),
    E(  73,  0), E( -73,  0), E(  74,  0), E( -74,  0), E(  75,  0),
    E( -75,  0), E(  76,  0), E( -76,  0), E(  77,  0), E( -77,  0),
    E(  78,  0), E( -78,  0), E(  79,  0), E( -79,  0), E(  80,  0),
    E( -80,  0), E(  81,  0), E( -81,  0), E(  82,  0), E( -82,  0),
    E(  83,  0), E( -83,  0), E(  84,  0), E( -84,  0), E(  85,  0),
    E( -85,  0), E(  86,  0), E( -86,  0), E(  87,  0), E( -87,  0),
    E(  88,  0), E( -88,  0), E(  89,  0), E( -89,  0), E(  90,  0),
    E( -90,  0), E(  91,  0), E( -91,  0), E(  92,  0), E( -92,  0),
    E(  93,  0), E( -93,  0), E(  94,  0), E( -94,  0), E(  95,  0),
    E( -95,  0), E(  96,  0), E( -96,  0), E(  97,  0), E( -97,  0),
    E(  98,  0), E( -98,  0), E(  99,  0), E( -99,  0), E( 100,  0),
    E(-100,  0), E( 101,  0), E(-101,  0), E( 102,  0), E(-102,  0),
    E( 103,  0), E(-103,  0), E( 104,  0), E(-104,  0), E( 105,  0),
    E(-105,  0), E( 106,  0), E(-106,  0), E( 107,  0), E(-107,  0),
    E( 108,  0), E(-108,  0), E( 109,  0), E(-109,  0), E( 110,  0),
    E(-110,  0), E( 111,  0), E(-111,  0), E( 112,  0), E(-112,  0),
    E( 113,  0), E(-113,  0), E( 114,  0), E(-114,  0), E( 115,  0),
    E(-115,  0), E( 116,  0), E(-116,  0), E( 117,  0), E(-117,  0),
    E( 118,  0), E(-118,  0), E( 119,  0), E(-119,  0), E( 120,  0),
    E(-120,  0), E( 121,  0), E(-121,  0), E( 122,  0), E(-122,  0),
    E( 123,  0), E(-123,  0), E( 124,  0), E(-124,  0), E( 125,  0),
    E(-125,  0), E( 126,  0), E(-126,  0), E( 127,  0), E(-127,  0),
    E( 128,  0), E(-128,  0), E( 129,  0), E(-129,  0), E( 130,  0),
    E(-130,  0), E( 131,  0), E(-131,  0), E( 132,  0), E(-132,  0),
    E( 133,  0), E(-133,  0), E( 134,  0), E(-134,  0), E( 135,  0),
    E(-135,  0), E( 136,  0), E(-136,  0), E( 137,  0), E(-137,  0),
    E( 138,  0), E(-138,  0), E( 139,  0), E(-139,  0), E( 140,  0),
    E(-140,  0), E( 141,  0), E(-141,  0), E( 142,  0), E(-142,  0),
    E( 143,  0), E(-143,  0), E( 144,  0), E(-144,  0), E( 145,  0),
    E(-145,  0), E( 146,  0), E(-146,  0), E( 147,  0), E(-147,  0),
    E( 148,  0), E(-148,  0), E( 149,  0), E(-149,  0), E( 150,  0),
    E(-150,  0), E( 151,  0), E(-151,  0), E( 152,  0), E(-152,  0),
    E( 153,  0), E(-153,  0), E( 154,  0), E(-154,  0), E( 155,  0),
    E(-155,  0), E( 156,  0), E(-156,  0), E( 157,  0), E(-157,  0),
    E( 158,  0), E(-158,  0), E( 159,  0), E(-159,  0), E( 160,  0),
    E(-160,  0), E( 161,  0), E(-161,  0), E( 162,  0), E(-162,  0),
    E( 163,  0), E(-163,  0), E( 164,  0), E(-164,  0), E( 165,  0),
    E(-165,  0), E( 166,  0), E(-166,  0), E( 167,  0), E(-167,  0),
    E( 168,  0), E(-168,  0), E( 169,  0), E(-169,  0), E( 170,  0),
    E(-170,  0), E( 171,  0), E(-171,  0), E( 172,  0), E(-172,  0),
    E( 173,  0), E(-173,  0), E( 174,  0), E(-174,  0), E( 175,  0),
    E(-175,  0), E( 176,  0), E(-176,  0), E( 177,  0), E(-177,  0),
    E( 178,  0), E(-178,  0), E( 179,  0), E(-179,  0), E( 180,  0),
    E(-180,  0), E( 181,  0), E(-181,  0), E( 182,  0), E(-182,  0),
    E( 183,  0), E(-183,  0), E( 184,  0), E(-184,  0), E( 185,  0),
    E(-185,  0), E( 186,  0), E(-186,  0), E( 187,  0), E(-187,  0),
    E( 188,  0), E(-188,  0), E( 189,  0), E(-189,  0), E( 190,  0),
    E(-190,  0), E( 191,  0), E(-191,  0), E( 192,  0), E(-192,  0),
    E( 193,  0), E(-193,  0), E( 194,  0), E(-194,  0), E( 195,  0),
    E(-195,  0), E( 196,  0), E(-196,  0), E( 197,  0), E(-197,  0),
    E( 198,  0), E(-198,  0), E( 199,  0), E(-199,  0), E( 200,  0),
    E(-200,  0), E( 201,  0), E(-201,  0), E( 202,  0), E(-202,  0),
    E( 203,  0), E(-203,  0), E( 204,  0), E(-204,  0), E( 205,  0),
    E(-205,  0), E( 206,  0), E(-206,  0), E( 207,  0), E(-207,  0),
    E( 208,  0), E(-208,  0), E( 209,  0), E(-209,  0), E( 210,  0),
    E(-210,  0), E( 211,  0), E(-211,  0), E( 212,  0), E(-212,  0),
    E( 213,  0), E(-213,  0), E( 214,  0), E(-214,  0), E( 215,  0),
    E(-215,  0), E( 216,  0), E(-216,  0), E( 217,  0), E(-217,  0),
    E( 218,  0), E(-218,  0), E( 219,  0), E(-219,  0), E( 220,  0),
    E(-220,  0), E( 221,  0), E(-221,  0), E( 222,  0), E(-222,  0),
    E( 223,  0), E(-223,  0), E( 224,  0), E(-224,  0), E( 225,  0),
    E(-225,  0), E( 226,  0), E(-226,  0), E( 227,  0), E(-227,  0),
    E( 228,  0), E(-228,  0), E( 229,  0), E(-229,  0), E( 230,  0),
    E(-230,  0), E( 231,  0), E(-231,  0), E( 232,  0), E(-232,  0),
    E( 233,  0), E(-233,  0), E( 234,  0), E(-234,  0), E( 235,  0),
    E(-235,  0), E( 236,  0), E(-236,  0), E( 237,  0), E(-237,  0),
    E( 238,  0), E(-238,  0), E( 239,  0), E(-239,  0), E( 240,  0),
    E(-240,  0), E( 241,  0), E(-241,  0), E( 242,  0), E(-242,  0),
    E( 243,  0), E(-243,  0), E( 244,  0), E(-244,  0), E( 245,  0),
    E(-245,  0), E( 246,  0), E(-246,  0), E( 247,  0), E(-247,  0),
    E( 248,  0), E(-248,  0), E( 249,  0), E(-249,  0), E( 250,  0),
    E(-250,  0), E( 251,  0), E(-251,  0), E( 252,  0), E(-252,  0),
    E( 253,  0), E(-253,  0), E( 254,  0), E(-254,  0), E( 255,  0),
    E(-255,  0), E(  13,  0), E( -13,  0), E(  14,  0), E( -14,  0),
    E(  15,  0), E( -15,  0), E(  16,  0), E( -16,  0), E(  17,  0),
    E( -17,  0), E(  18,  0), E( -18,  0), E(   5,  1), E(  -5,  1),
    E(   6,  1), E(  -6,  1), E(   2,  2), E(  -2,  2), E(   1,  3),
    E(  -1,  3), E(   0,  0), E(  19,  0), E( -19,  0), E(  20,  0),
    E( -20,  0), E(  21,  0), E( -21,  0), E(  22,  0), E( -22,  0),
    E(  23,  0), E( -23,  0), E(  24,  0), E( -24,  0), E(  25,  0),
    E( -25,  0), E(   7,  1), E(  -7,  1), E(   8,  1), E(  -8,  1),
    E(   3,  2), E(  -3,  2), E(   2,  3), E(  -2,  3), E(   1,  4),
    E(  -1,  4), E(   1,  5), E(  -1,  5), E(  26,  0), E( -26,  0),
    E(  27,  0), E( -27,  0), E(  28,  0), E( -28,  0), E(  29,  0),
    E( -29,  0), E(  30,  0), E( -30,  0), E(  31,  0), E( -31,  0),
    E(  32,  0), E( -32,  0), E(  33,  0), E( -33,  0), E(  34,  0),
    E( -34,  0), E(  35,  0), E( -35,  0), E(  36,  0), E( -36,  0),
    E(   0,  1), E(   9,  1), E(  -9,  1), E(  10,  1), E( -10,  1),
    E(  11,  1), E( -11,  1), E(  12,  1), E( -12,  1), E(   0,  2),
    E(   4,  2), E(  -4,  2), E(   5,  2), E(  -5,  2), E(   6,  2),
    E(  -6,  2), E(   0,  3), E(   3,  3), E(  -3,  3), E(   4,  3),
    E(  -4,  3), E(   0,  4), E(   2,  4), E(  -2,  4), E(   0,  5),
    E(   1,  6), E(  -1,  6), E(   1,  7), E(  -1,  7), E(   1,  8),
    E(  -1,  8), E(  37,  0), E( -37,  0), E(  38,  0), E( -38,  0),
    E(  39,  0), E( -39,  0), E(  40,  0), E( -40,  0), E(  41,  0),
    E( -41,  0), E(  42,  0), E( -42,  0), E(  43,  0), E( -43,  0),
    E(  44,  0), E( -44,  0), E(  45,  0), E( -45,  0), E(  46,  0),
    E( -46,  0), E(  47,  0), E( -47,  0), E(  48,  0), E( -48,  0),
    E(  13,  1), E( -13,  1), E(  14,  1), E( -14,  1), E(  15,  1),
    E( -15,  1), E(  16,  1), E( -16,  1), E(   7,  2), E(  -7,  2),
    E(   8,  2), E(  -8,  2), E(   5,  3), E(  -5,  3), E(   6,  3),
    E(  -6,  3), E(   3,  4), E(  -3,  4), E(   4,  4), E(  -4,  4),
    E(   2,  5), E(  -2,  5), E(   0,  6), E(   2,  6), E(  -2,  6),
    E(   0,  7), E(   0,  8), E(   0,  9), E(   1,  9), E(  -1,  9),
    E(   1, 10), E(  -1, 10), E(   1, 11), E(  -1, 11), E(   1, 12),
    E(  -1, 12), E(  49,  0), E( -49,  0), E(  50,  0), E( -50,  0),
    E(  51,  0), E( -51,  0), E(  52,  0), E( -52,  0), E(  53,  0),
    E( -53,  0), E(  54,  0), E( -54,  0), E(  55,  0), E( -55,  0),
    E(  56,  0), E( -56,  0), E(  57,  0), E( -57,  0), E(  58,  0),
    E( -58,  0), E(  59,  0), E( -59,  0), E(  60,  0), E( -60,  0),
    E(  61,  0), E( -61,  0), E(  62,  0), E( -62,  0), E(  63,  0),
    E( -63,  0), E(  17,  1), E( -17,  1), E(  18,  1), E( -18,  1),
    E(  19,  1), E( -19,  1), E(  20,  1), E( -20,  1), E(  21,  1),
    E( -21,  1), E(  22,  1), E( -22,  1), E(  23,  1), E( -23,  1),
    E(  24,  1), E( -24,  1), E(   9,  2), E(  -9,  2), E(  10,  2),
    E( -10,  2), E(  11,  2), E( -11,  2), E(  12,  2), E( -12,  2),
    E(   7,  3), E(  -7,  3), E(   8,  3), E(  -8,  3), E(   5,  4),
    E(  -5,  4), E(   6,  4), E(  -6,  4), E(   3,  5), E(  -3,  5),
    E(   4,  5), E(  -4,  5), E(   5,  5), E(  -5,  5), E(   6,  5),
    E(  -6,  5), E(   3,  6), E(  -3,  6), E(   4,  6), E(  -4,  6),
    E(   2,  7), E(  -2,  7), E(   2,  8), E(  -2,  8), E(   2,  9),
    E(  -2,  9), E(   0, 10), E(   2, 10), E(  -2, 10), E(   0, 11),
    E(   0, 12), E(   0, 13), E(   1, 13), E(  -1, 13), E(   1, 14),
    E(  -1, 14), E(   0,  0), E(   0,  1), E(   0,  2), E(   0,  3),
    E(   0,  4), E(   0,  5), E(   0,  6), E(   0,  7), E(   0,  8),
    E(   0,  9), E(   0, 10), E(   0, 11), E(   0, 12), E(   0, 13),
    E(   0, 14), E(   0, 15), E(   0, 16), E(   0, 17), E(   0, 18),
    E(   0, 19), E(   0, 20), E(   0, 21), E(   0, 22), E(   0, 23),
    E(   0, 24), E(   0, 25), E(   0, 26), E(   0, 27), E(   0, 28),
    E(   0, 29), E(   0, 30), E(   0, 31), E(   0, 32), E(   0, 33),
    E(   0, 34), E(   0, 35), E(   0, 36), E(   0, 37), E(   0, 38),
    E(   0, 39), E(   0, 40), E(   0, 41), E(   0, 42), E(   0, 43),
    E(   0, 44), E(   0, 45), E(   0, 46), E(   0, 47), E(   0, 48),
    E(   0, 49), E(   0, 50), E(   0, 51), E(   0, 52), E(   0, 53),
    E(   0, 54), E(   0, 55), E(   0, 56), E(   0, 57), E(   0, 58),
    E(   0, 59), E(   0, 60), E(   0, 61), E(   0, 62), E(   0, 63),
    E(  25,  1), E( -25,  1), E(  26,  1), E( -26,  1), E(  27,  1),
    E( -27,  1), E(  28,  1), E( -28,  1), E(  29,  1), E( -29,  1),
    E(  30,  1), E( -30,  1), E(  31,  1), E( -31,  1), E(  32,  1),
    E( -32,  1), E(  13,  2), E( -13,  2), E(  14,  2), E( -14,  2),
    E(  15,  2), E( -15,  2), E(  16,  2), E( -16,  2), E(   9,  3),
    E(  -9,  3), E(  10,  3), E( -10,  3), E(  11,  3), E( -11,  3),
    E(   7,  4), E(  -7,  4), E(   3,  7), E(  -3,  7), E(   4,  7),
    E(  -4,  7), E(   3,  8), E(  -3,  8), E(   4,  8), E(  -4,  8),
    E(   3,  9), E(  -3,  9), E(   2, 11), E(  -2, 11), E(   2, 12),
    E(  -2, 12), E(   0, 14),
    // AC table Q16 - 512 elements
    E(   1,  0), E(  -1,  0), E(   2,  0), E(  -2,  0), E(   3,  0),
    E(  -3,  0), E(   4,  0), E(  -4,  0), E(   1,  1), E(  -1,  1),
    E(   0, 64), E(   5,  0), E(  -5,  0), E(   6,  0), E(  -6,  0),
    E(   2,  1), E(  -2,  1), E(   1,  2), E(  -1,  2), E(   7,  0),
    E(  -7,  0), E(   8,  0), E(  -8,  0), E(   9,  0), E(  -9,  0),
    E(   3,  1), E(  -3,  1), E(   1,  3), E(  -1,  3), E(   1,  4),
    E(  -1,  4), E(  10,  0), E( -10,  0), E(  11,  0), E( -11,  0),
    E(  12,  0), E( -12,  0), E(   4,  1), E(  -4,  1), E(   2,  2),
    E(  -2,  2), E(   1,  5), E(  -1,  5), E(   1,  6), E(  -1,  6),
    E(  13,  0), E( -13,  0), E(  14,  0), E( -14,  0), E(  15,  0),
    E( -15,  0), E(  16,  0), E( -16,  0), E(  17,  0), E( -17,  0),
    E(   5,  1), E(  -5,  1), E(   2,  3), E(  -2,  3), E(   1,  7),
    E(  -1,  7), E(   1,  8), E(  -1,  8), E(   1,  9), E(  -1,  9),
    E(   1, 10), E(  -1, 10), E(   0,  0), E(  18,  0), E( -18,  0),
    E(  19,  0), E( -19,  0), E(  20,  0), E( -20,  0), E(  21,  0),
    E( -21,  0), E(  22,  0), E( -22,  0), E(   6,  1), E(  -6,  1),
    E(   7,  1), E(  -7,  1), E(   3,  2), E(  -3,  2), E(   2,  4),
    E(  -2,  4), E(   2,  5), E(  -2,  5), E(   1, 11), E(  -1, 11),
    E(   1, 12), E(  -1, 12), E(   1, 13), E(  -1, 13), E(   0,  0),
    E(   0,  1), E(   0,  2), E(   0,  3), E(   0,  4), E(   0,  5),
    E(   0,  6), E(   0,  7), E(   0,  8), E(   0,  9), E(   0, 10),
    E(   0, 11), E(   0, 12), E(   0, 13), E(   0, 14), E(   0, 15),
    E(   0, 16), E(   0, 17), E(   0, 18), E(   0, 19), E(   0, 20),
    E(   0, 21), E(   0, 22), E(   0, 23), E(   0, 24), E(   0, 25),
    E(   0, 26), E(   0, 27), E(   0, 28), E(   0, 29), E(   0, 30),
    E(   0, 31), E(   0, 32), E(   0, 33), E(   0, 34), E(   0, 35),
    E(   0, 36), E(   0, 37), E(   0, 38), E(   0, 39), E(   0, 40),
    E(   0, 41), E(   0, 42), E(   0, 43), E(   0, 44), E(   0, 45),
    E(   0, 46), E(   0, 47), E(   0, 48), E(   0, 49), E(   0, 50),
    E(   0, 51), E(   0, 52), E(   0, 53), E(   0, 54), E(   0, 55),
    E(   0, 56), E(   0, 57), E(   0, 58), E(   0, 59), E(   0, 60),
    E(   0, 61), E(   0, 62), E(   0, 63), E(   0,  0), E(   1,  0),
    E(  -1,  0), E(   2,  0), E(  -2,  0), E(   3,  0), E(  -3,  0),
    E(   4,  0), E(  -4,  0), E(   5,  0), E(  -5,  0), E(   6,  0),
    E(  -6,  0), E(   7,  0), E(  -7,  0), E(   8,  0), E(  -8,  0),
    E(   9,  0), E(  -9,  0), E(  10,  0), E( -10,  0), E(  11,  0),
    E( -11,  0), E(  12,  0), E( -12,  0), E(  13,  0), E( -13,  0),
    E(  14,  0), E( -14,  0), E(  15,  0), E( -15,  0), E(  16,  0),
    E( -16,  0), E(  17,  0), E( -17,  0), E(  18,  0), E( -18,  0),
    E(  19,  0), E( -19,  0), E(  20,  0), E( -20,  0), E(  21,  0),
    E( -21,  0), E(  22,  0), E( -22,  0), E(  23,  0), E( -23,  0),
    E(  24,  0), E( -24,  0), E(  25,  0), E( -25,  0), E(  26,  0),
    E( -26,  0), E(  27,  0), E( -27,  0), E(  28,  0), E( -28,  0),
    E(  29,  0), E( -29,  0), E(  30,  0), E( -30,  0), E(  31,  0),
    E( -31,  0), E(  32,  0), E( -32,  0), E(  33,  0), E( -33,  0),
    E(  34,  0), E( -34,  0), E(  35,  0), E( -35,  0), E(  36,  0),
    E( -36,  0), E(  37,  0), E( -37,  0), E(  38,  0), E( -38,  0),
    E(  39,  0), E( -39,  0), E(  40,  0), E( -40,  0), E(  41,  0),
    E( -41,  0), E(  42,  0), E( -42,  0), E(  43,  0), E( -43,  0),
    E(  44,  0), E( -44,  0), E(  45,  0), E( -45,  0), E(  46,  0),
    E( -46,  0), E(  47,  0), E( -47,  0), E(  48,  0), E( -48,  0),
    E(  49,  0), E( -49,  0), E(  50,  0), E( -50,  0), E(  51,  0),
    E( -51,  0), E(  52,  0), E( -52,  0), E(  53,  0), E( -53,  0),
    E(  54,  0), E( -54,  0), E(  55,  0), E( -55,  0), E(  56,  0),
    E( -56,  0), E(  57,  0), E( -57,  0), E(  58,  0), E( -58,  0),
    E(  59,  0), E( -59,  0), E(  60,  0), E( -60,  0), E(  61,  0),
    E( -61,  0), E(  62,  0), E( -62,  0), E(  63,  0), E( -63,  0),
    E(  64,  0), E( -64,  0), E(  65,  0), E( -65,  0), E(  66,  0),
    E( -66,  0), E(  67,  0), E( -67,  0), E(  68,  0), E( -68,  0),
    E(  69,  0), E( -69,  0), E(  70,  0), E( -70,  0), E(  71,  0),
    E( -71,  0), E(  72,  0), E( -72,  0), E(  73,  0), E( -73,  0),
    E(  74,  0), E( -74,  0), E(  75,  0), E( -75,  0), E(  76,  0),
    E( -76,  0), E(  77,  0), E( -77,  0), E(  78,  0), E( -78,  0),
    E(  79,  0), E( -79,  0), E(  80,  0), E( -80,  0), E(  81,  0),
    E( -81,  0), E(  82,  0), E( -82,  0), E(  83,  0), E( -83,  0),
    E(  84,  0), E( -84,  0), E(  85,  0), E( -85,  0), E(  86,  0),
    E( -86,  0), E(  87,  0), E( -87,  0), E(  88,  0), E( -88,  0),
    E(  89,  0), E( -89,  0), E(  90,  0), E( -90,  0), E(  91,  0),
    E( -91,  0), E(  92,  0), E( -92,  0), E(  93,  0), E( -93,  0),
    E(  94,  0), E( -94,  0), E(  95,  0), E( -95,  0), E(  96,  0),
    E( -96,  0), E(  97,  0), E( -97,  0), E(  98,  0), E( -98,  0),
    E(  99,  0), E( -99,  0), E( 100,  0), E(-100,  0), E( 101,  0),
    E(-101,  0), E( 102,  0), E(-102,  0), E( 103,  0), E(-103,  0),
    E( 104,  0), E(-104,  0), E( 105,  0), E(-105,  0), E( 106,  0),
    E(-106,  0), E( 107,  0), E(-107,  0), E( 108,  0), E(-108,  0),
    E( 109,  0), E(-109,  0), E( 110,  0), E(-110,  0), E( 111,  0),
    E(-111,  0), E( 112,  0), E(-112,  0), E( 113,  0), E(-113,  0),
    E( 114,  0), E(-114,  0), E( 115,  0), E(-115,  0), E( 116,  0),
    E(-116,  0), E( 117,  0), E(-117,  0), E( 118,  0), E(-118,  0),
    E( 119,  0), E(-119,  0), E( 120,  0), E(-120,  0), E( 121,  0),
    E(-121,  0), E( 122,  0), E(-122,  0), E( 123,  0), E(-123,  0),
    E( 124,  0), E(-124,  0), E( 125,  0), E(-125,  0), E( 126,  0),
    E(-126,  0), E( 127,  0), E(-127,  0), E(  23,  0), E( -23,  0),
    E(  24,  0), E( -24,  0), E(  25,  0), E( -25,  0), E(  26,  0),
    E( -26,  0), E(  27,  0), E( -27,  0), E(  28,  0), E( -28,  0),
    E(   8,  1), E(  -8,  1), E(   9,  1), E(  -9,  1), E(   4,  2),
    E(  -4,  2), E(   3,  3), E(  -3,  3), E(   3,  4), E(  -3,  4),
    E(   2,  6), E(  -2,  6), E(   2,  7), E(  -2,  7), E(  29,  0),
    E( -29,  0), E(  30,  0), E( -30,  0), E(  31,  0), E( -31,  0),
    E(  32,  0), E( -32,  0), E(  33,  0), E( -33,  0), E(  34,  0),
    E( -34,  0), E(  35,  0), E( -35,  0), E(   0,  1), E(  10,  1),
    E( -10,  1), E(  11,  1), E( -11,  1), E(  12,  1), E( -12,  1),
    E(   0,  2), E(   5,  2), E(  -5,  2), E(   0,  3), E(   4,  3),
    E(  -4,  3), E(   3,  5), E(  -3,  5), E(   2,  8), E(  -2,  8),
    E(   2,  9), E(  -2,  9), E(   1, 14), E(  -1, 14), E(   1, 15),
    E(  -1, 15), E(  36,  0), E( -36,  0), E(  37,  0), E( -37,  0),
    E(  38,  0), E( -38,  0), E(  39,  0), E( -39,  0), E(  40,  0),
    E( -40,  0), E(  13,  1), E( -13,  1), E(  14,  1), E( -14,  1),
    E(  15,  1), E( -15,  1), E(   6,  2), E(  -6,  2), E(   7,  2),
    E(  -7,  2), E(   5,  3), E(  -5,  3), E(   0,  4), E(   4,  4),
    E(  -4,  4), E(   0,  5), E(   0,  6), E(   3,  6), E(  -3,  6),
    E(   0,  7), E(   3,  7), E(  -3,  7), E(   2, 10), E(  -2, 10),
    E(   1, 16), E(  -1, 16),
    // AC table Q32 - 354 elements
    E(  1,  0), E( -1,  0), E(  2,  0), E( -2,  0), E(  0, 64), E(  3,  0),
    E( -3,  0), E(  1,  1), E( -1,  1), E(  4,  0), E( -4,  0), E(  5,  0),
    E( -5,  0), E(  2,  1), E( -2,  1), E(  1,  2), E( -1,  2), E(  1,  3),
    E( -1,  3), E(  6,  0), E( -6,  0), E(  7,  0), E( -7,  0), E(  3,  1),
    E( -3,  1), E(  1,  4), E( -1,  4), E(  1,  5), E( -1,  5), E(  8,  0),
    E( -8,  0), E(  9,  0), E( -9,  0), E( 10,  0), E(-10,  0), E(  4,  1),
    E( -4,  1), E(  2,  2), E( -2,  2), E(  1,  6), E( -1,  6), E(  1,  7),
    E( -1,  7), E(  1,  8), E( -1,  8), E( 11,  0), E(-11,  0), E( 12,  0),
    E(-12,  0), E( 13,  0), E(-13,  0), E(  5,  1), E( -5,  1), E(  2,  3),
    E( -2,  3), E(  1,  9), E( -1,  9), E(  1, 10), E( -1, 10), E( 14,  0),
    E(-14,  0), E( 15,  0), E(-15,  0), E( 16,  0), E(-16,  0), E(  6,  1),
    E( -6,  1), E(  7,  1), E( -7,  1), E(  3,  2), E( -3,  2), E(  3,  3),
    E( -3,  3), E(  2,  4), E( -2,  4), E(  2,  5), E( -2,  5), E(  1, 11),
    E( -1, 11), E(  1, 12), E( -1, 12), E(  1, 13), E( -1, 13), E(  0,  0),
    E(  0,  1), E(  0,  2), E(  0,  3), E(  0,  4), E(  0,  5), E(  0,  6),
    E(  0,  7), E(  0,  8), E(  0,  9), E(  0, 10), E(  0, 11), E(  0, 12),
    E(  0, 13), E(  0, 14), E(  0, 15), E(  0, 16), E(  0, 17), E(  0, 18),
    E(  0, 19), E(  0, 20), E(  0, 21), E(  0, 22), E(  0, 23), E(  0, 24),
    E(  0, 25), E(  0, 26), E(  0, 27), E(  0, 28), E(  0, 29), E(  0, 30),
    E(  0, 31), E(  0, 32), E(  0, 33), E(  0, 34), E(  0, 35), E(  0, 36),
    E(  0, 37), E(  0, 38), E(  0, 39), E(  0, 40), E(  0, 41), E(  0, 42),
    E(  0, 43), E(  0, 44), E(  0, 45), E(  0, 46), E(  0, 47), E(  0, 48),
    E(  0, 49), E(  0, 50), E(  0, 51), E(  0, 52), E(  0, 53), E(  0, 54),
    E(  0, 55), E(  0, 56), E(  0, 57), E(  0, 58), E(  0, 59), E(  0, 60),
    E(  0, 61), E(  0, 62), E(  0, 63), E(  0,  0), E( 17,  0), E(-17,  0),
    E( 18,  0), E(-18,  0), E( 19,  0), E(-19,  0), E( 20,  0), E(-20,  0),
    E(  8,  1), E( -8,  1), E(  9,  1), E( -9,  1), E(  4,  2), E( -4,  2),
    E(  3,  4), E( -3,  4), E(  2,  6), E( -2,  6), E(  2,  7), E( -2,  7),
    E(  2,  8), E( -2,  8), E(  1, 14), E( -1, 14), E(  0,  0), E(  1,  0),
    E( -1,  0), E(  2,  0), E( -2,  0), E(  3,  0), E( -3,  0), E(  4,  0),
    E( -4,  0), E(  5,  0), E( -5,  0), E(  6,  0), E( -6,  0), E(  7,  0),
    E( -7,  0), E(  8,  0), E( -8,  0), E(  9,  0), E( -9,  0), E( 10,  0),
    E(-10,  0), E( 11,  0), E(-11,  0), E( 12,  0), E(-12,  0), E( 13,  0),
    E(-13,  0), E( 14,  0), E(-14,  0), E( 15,  0), E(-15,  0), E( 16,  0),
    E(-16,  0), E( 17,  0), E(-17,  0), E( 18,  0), E(-18,  0), E( 19,  0),
    E(-19,  0), E( 20,  0), E(-20,  0), E( 21,  0), E(-21,  0), E( 22,  0),
    E(-22,  0), E( 23,  0), E(-23,  0), E( 24,  0), E(-24,  0), E( 25,  0),
    E(-25,  0), E( 26,  0), E(-26,  0), E( 27,  0), E(-27,  0), E( 28,  0),
    E(-28,  0), E( 29,  0), E(-29,  0), E( 30,  0), E(-30,  0), E( 31,  0),
    E(-31,  0), E( 32,  0), E(-32,  0), E( 33,  0), E(-33,  0), E( 34,  0),
    E(-34,  0), E( 35,  0), E(-35,  0), E( 36,  0), E(-36,  0), E( 37,  0),
    E(-37,  0), E( 38,  0), E(-38,  0), E( 39,  0), E(-39,  0), E( 40,  0),
    E(-40,  0), E( 41,  0), E(-41,  0), E( 42,  0), E(-42,  0), E( 43,  0),
    E(-43,  0), E( 44,  0), E(-44,  0), E( 45,  0), E(-45,  0), E( 46,  0),
    E(-46,  0), E( 47,  0), E(-47,  0), E( 48,  0), E(-48,  0), E( 49,  0),
    E(-49,  0), E( 50,  0), E(-50,  0), E( 51,  0), E(-51,  0), E( 52,  0),
    E(-52,  0), E( 53,  0), E(-53,  0), E( 54,  0), E(-54,  0), E( 55,  0),
    E(-55,  0), E( 56,  0), E(-56,  0), E( 57,  0), E(-57,  0), E( 58,  0),
    E(-58,  0), E( 59,  0), E(-59,  0), E( 60,  0), E(-60,  0), E( 61,  0),
    E(-61,  0), E( 62,  0), E(-62,  0), E( 63,  0), E(-63,  0), E( 21,  0),
    E(-21,  0), E( 22,  0), E(-22,  0), E( 23,  0), E(-23,  0), E(  0,  1),
    E( 10,  1), E(-10,  1), E( 11,  1), E(-11,  1), E(  0,  2), E(  5,  2),
    E( -5,  2), E(  6,  2), E( -6,  2), E(  0,  3), E(  4,  3), E( -4,  3),
    E(  0,  4), E(  3,  5), E( -3,  5), E(  3,  6), E( -3,  6), E(  2,  9),
    E( -2,  9), E(  1, 15), E( -1, 15), E( 24,  0), E(-24,  0), E( 25,  0),
    E(-25,  0), E( 26,  0), E(-26,  0), E( 12,  1), E(-12,  1), E( 13,  1),
    E(-13,  1), E(  5,  3), E( -5,  3), E(  4,  4), E( -4,  4), E(  0,  5),
    E(  4,  5), E( -4,  5), E(  0,  6), E(  0,  7), E(  3,  7), E( -3,  7),
    E(  0,  8), E(  3,  8), E( -3,  8), E(  0,  9), E(  1, 16), E( -1, 16),
    // AC table Q64 - 257 elements
    E(  1,  0), E( -1,  0), E(  0, 64), E(  2,  0), E( -2,  0), E(  3,  0),
    E( -3,  0), E(  1,  1), E( -1,  1), E(  4,  0), E( -4,  0), E(  2,  1),
    E( -2,  1), E(  1,  2), E( -1,  2), E(  5,  0), E( -5,  0), E(  1,  3),
    E( -1,  3), E(  1,  4), E( -1,  4), E(  6,  0), E( -6,  0), E(  3,  1),
    E( -3,  1), E(  2,  2), E( -2,  2), E(  1,  5), E( -1,  5), E(  1,  6),
    E( -1,  6), E(  1,  7), E( -1,  7), E(  7,  0), E( -7,  0), E(  8,  0),
    E( -8,  0), E(  4,  1), E( -4,  1), E(  2,  3), E( -2,  3), E(  1,  8),
    E( -1,  8), E(  1,  9), E( -1,  9), E(  9,  0), E( -9,  0), E( 10,  0),
    E(-10,  0), E(  5,  1), E( -5,  1), E(  3,  2), E( -3,  2), E(  2,  4),
    E( -2,  4), E(  2,  5), E( -2,  5), E(  1, 10), E( -1, 10), E(  1, 11),
    E( -1, 11), E(  0,  0), E(  0,  1), E(  0,  2), E(  0,  3), E(  0,  4),
    E(  0,  5), E(  0,  6), E(  0,  7), E(  0,  8), E(  0,  9), E(  0, 10),
    E(  0, 11), E(  0, 12), E(  0, 13), E(  0, 14), E(  0, 15), E(  0, 16),
    E(  0, 17), E(  0, 18), E(  0, 19), E(  0, 20), E(  0, 21), E(  0, 22),
    E(  0, 23), E(  0, 24), E(  0, 25), E(  0, 26), E(  0, 27), E(  0, 28),
    E(  0, 29), E(  0, 30), E(  0, 31), E(  0, 32), E(  0, 33), E(  0, 34),
    E(  0, 35), E(  0, 36), E(  0, 37), E(  0, 38), E(  0, 39), E(  0, 40),
    E(  0, 41), E(  0, 42), E(  0, 43), E(  0, 44), E(  0, 45), E(  0, 46),
    E(  0, 47), E(  0, 48), E(  0, 49), E(  0, 50), E(  0, 51), E(  0, 52),
    E(  0, 53), E(  0, 54), E(  0, 55), E(  0, 56), E(  0, 57), E(  0, 58),
    E(  0, 59), E(  0, 60), E(  0, 61), E(  0, 62), E(  0, 63), E(  0,  0),
    E( 11,  0), E(-11,  0), E( 12,  0), E(-12,  0), E(  6,  1), E( -6,  1),
    E(  7,  1), E( -7,  1), E(  3,  3), E( -3,  3), E(  3,  4), E( -3,  4),
    E(  3,  5), E( -3,  5), E(  2,  6), E( -2,  6), E(  2,  7), E( -2,  7),
    E(  1, 12), E( -1, 12), E(  1, 13), E( -1, 13), E(  1, 14), E( -1, 14),
    E( 13,  0), E(-13,  0), E( 14,  0), E(-14,  0), E(  0,  1), E(  8,  1),
    E( -8,  1), E(  4,  2), E( -4,  2), E(  4,  3), E( -4,  3), E(  2,  8),
    E( -2,  8), E(  2,  9), E( -2,  9), E(  1, 15), E( -1, 15), E(  0,  0),
    E(  1,  0), E( -1,  0), E(  2,  0), E( -2,  0), E(  3,  0), E( -3,  0),
    E(  4,  0), E( -4,  0), E(  5,  0), E( -5,  0), E(  6,  0), E( -6,  0),
    E(  7,  0), E( -7,  0), E(  8,  0), E( -8,  0), E(  9,  0), E( -9,  0),
    E( 10,  0), E(-10,  0), E( 11,  0), E(-11,  0), E( 12,  0), E(-12,  0),
    E( 13,  0), E(-13,  0), E( 14,  0), E(-14,  0), E( 15,  0), E(-15,  0),
    E( 16,  0), E(-16,  0), E( 17,  0), E(-17,  0), E( 18,  0), E(-18,  0),
    E( 19,  0), E(-19,  0), E( 20,  0), E(-20,  0), E( 21,  0), E(-21,  0),
    E( 22,  0), E(-22,  0), E( 23,  0), E(-23,  0), E( 24,  0), E(-24,  0),
    E( 25,  0), E(-25,  0), E( 26,  0), E(-26,  0), E( 27,  0), E(-27,  0),
    E( 28,  0), E(-28,  0), E( 29,  0), E(-29,  0), E( 30,  0), E(-30,  0),
    E( 31,  0), E(-31,  0), E( 15,  0), E(-15,  0), E(  9,  1), E( -9,  1),
    E( 10,  1), E(-10,  1), E(  0,  2), E(  5,  2), E( -5,  2), E(  0,  3),
    E(  5,  3), E( -5,  3), E(  0,  4), E(  4,  4), E( -4,  4), E(  0,  5),
    E(  4,  5), E( -4,  5), E(  0,  6), E(  3,  6), E( -3,  6), E(  3,  7),
    E( -3,  7), E(  1, 16), E( -1, 16), E(  1, 17), E( -1, 17),
    // AC table Q128 - 194 elements
    E(  1,  0), E( -1,  0), E(  0, 64), E(  1,  1), E( -1,  1), E(  2,  0),
    E( -2,  0), E(  3,  0), E( -3,  0), E(  2,  1), E( -2,  1), E(  1,  2),
    E( -1,  2), E(  1,  3), E( -1,  3), E(  1,  4), E( -1,  4), E(  4,  0),
    E( -4,  0), E(  1,  5), E( -1,  5), E(  1,  6), E( -1,  6), E(  1,  7),
    E( -1,  7), E(  5,  0), E( -5,  0), E(  3,  1), E( -3,  1), E(  2,  2),
    E( -2,  2), E(  2,  3), E( -2,  3), E(  1,  8), E( -1,  8), E(  1,  9),
    E( -1,  9), E(  6,  0), E( -6,  0), E(  4,  1), E( -4,  1), E(  2,  4),
    E( -2,  4), E(  2,  5), E( -2,  5), E(  1, 10), E( -1, 10), E(  7,  0),
    E( -7,  0), E(  5,  1), E( -5,  1), E(  3,  2), E( -3,  2), E(  3,  3),
    E( -3,  3), E(  2,  6), E( -2,  6), E(  0,  0), E(  0,  1), E(  0,  2),
    E(  0,  3), E(  0,  4), E(  0,  5), E(  0,  6), E(  0,  7), E(  0,  8),
    E(  0,  9), E(  0, 10), E(  0, 11), E(  0, 12), E(  0, 13), E(  0, 14),
    E(  0, 15), E(  0, 16), E(  0, 17), E(  0, 18), E(  0, 19), E(  0, 20),
    E(  0, 21), E(  0, 22), E(  0, 23), E(  0, 24), E(  0, 25), E(  0, 26),
    E(  0, 27), E(  0, 28), E(  0, 29), E(  0, 30), E(  0, 31), E(  0, 32),
    E(  0, 33), E(  0, 34), E(  0, 35), E(  0, 36), E(  0, 37), E(  0, 38),
    E(  0, 39), E(  0, 40), E(  0, 41), E(  0, 42), E(  0, 43), E(  0, 44),
    E(  0, 45), E(  0, 46), E(  0, 47), E(  0, 48), E(  0, 49), E(  0, 50),
    E(  0, 51), E(  0, 52), E(  0, 53), E(  0, 54), E(  0, 55), E(  0, 56),
    E(  0, 57), E(  0, 58), E(  0, 59), E(  0, 60), E(  0, 61), E(  0, 62),
    E(  0, 63), E(  6,  1), E( -6,  1), E(  7,  1), E( -7,  1), E(  3,  4),
    E( -3,  4), E(  3,  5), E( -3,  5), E(  2,  7), E( -2,  7), E(  2,  8),
    E( -2,  8), E(  2,  9), E( -2,  9), E(  1, 11), E( -1, 11), E(  1, 12),
    E( -1, 12), E(  1, 13), E( -1, 13), E(  0,  0), E(  8,  0), E( -8,  0),
    E(  9,  0), E( -9,  0), E(  8,  1), E( -8,  1), E(  4,  2), E( -4,  2),
    E(  4,  3), E( -4,  3), E(  3,  6), E( -3,  6), E(  1, 14), E( -1, 14),
    E(  1, 15), E( -1, 15), E(  1, 16), E( -1, 16), E(  0,  1), E(  0,  2),
    E(  0,  3), E(  0,  0), E(  1,  0), E( -1,  0), E(  2,  0), E( -2,  0),
    E(  3,  0), E( -3,  0), E(  4,  0), E( -4,  0), E(  5,  0), E( -5,  0),
    E(  6,  0), E( -6,  0), E(  7,  0), E( -7,  0), E(  8,  0), E( -8,  0),
    E(  9,  0), E( -9,  0), E( 10,  0), E(-10,  0), E( 11,  0), E(-11,  0),
    E( 12,  0), E(-12,  0), E( 13,  0), E(-13,  0), E( 14,  0), E(-14,  0),
    E( 15,  0), E(-15,  0),
};

static const uint8_t hqx_ac_lens[] = {
    // AC table Q0 - 815 elements
     4,  4,  4,  4,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,
    13, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14, 14, 14, 14,  5,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,
     7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10,  9, 10, 10, 10, 10, 10, 10, 10, 10,  9, 10,
    10,  9, 10, 10, 10, 10, 10, 10, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 10, 11, 11, 10, 10, 11, 11, 11, 11, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13,
    13, 13, 13, 13, 13,
    // AC table Q8 - 907 elements
     4,  4,  4,  4,  5,  5,  5,  5,  4,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7, 14,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10,  9, 10, 10, 10, 10, 10, 10, 10, 10,  9, 10, 10, 10, 10, 10,
    10,  9, 10, 10, 10, 10,  9, 10, 10,  9, 10, 10, 10, 10, 10, 10, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 10, 11, 11, 10, 10, 10, 11, 11, 11, 11,
    11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 11, 12, 12, 11, 11, 11,
    12, 12, 12, 12, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 12,
    // AC table Q16 - 512 elements
     3,  3,  4,  4,  5,  5,  5,  5,  5,  5,  4,  6,  6,  6,  6,  6,  6,  6,
     6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 16, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 11, 12, 12, 12, 12, 12, 12, 11, 12, 12, 11, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13,
    13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12, 13, 13, 12, 12, 13,
    13, 12, 13, 13, 13, 13, 13, 13,
    // AC table Q32 - 354 elements
     3,  3,  4,  4,  3,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,
     6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,
     8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,  9,  9,  9,  9,  9,  9,
     9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 16, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 12, 12, 12, 12, 12, 12, 11,
    12, 12, 12, 12, 11, 12, 12, 12, 12, 11, 12, 12, 11, 12, 12, 12, 12, 12,
    12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 12,
    13, 13, 12, 12, 13, 13, 12, 13, 13, 12, 13, 13,
    // AC table Q64 - 257 elements
     3,  3,  2,  4,  4,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  7,  7,  7,
     7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  9,  9,  9,
     9,  9,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 10, 10, 10, 10, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 10,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 11, 12, 12, 12, 12, 12, 12, 12,
    12, 12, 12, 12, 12, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 13, 13, 13, 13,
    13, 13, 12, 13, 13, 12, 13, 13, 12, 13, 13, 12, 13, 13, 12, 13, 13, 13,
    13, 13, 13, 13, 13,
    // AC table Q128 - 194 elements
     3,  3,  2,  4,  4,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  7,
     7,  7,  7,  7,  7,  7,  7,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
     8,  9,  9,  9,  9,  9,  9,  9,  9,  9,  9, 10, 10, 10, 10, 10, 10, 10,
    10, 10, 10, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15,
    15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12,
    12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12,
    12, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
    17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
};

static const uint16_t hqx_ac_nb_elems[] = { 815, 907, 512, 354, 257, 194 };

static VLCElem cbp_vlc[(1 << HQX_CBP_VLC_BITS) + 896 /* dc9 */ + 1344 /* dc10 */
                       + 15630 /* RL_VLC_ELEMS for hqx_ac */];

static const VLCElem *dc_vlc[2];

#define INIT_DC_TABLE(idx, name)                                              \
    do {                                                                      \
         dc_vlc[idx] = ff_vlc_init_tables(&state, HQX_DC_VLC_BITS,            \
                                          FF_ARRAY_ELEMS(name ## _vlc_lens),  \
                                          name ## _vlc_lens, 1, 1,            \
                                          name ## _vlc_bits, 2, 2, 0);        \
    } while (0)

static av_cold av_unused void hqx_init_static(void)
{
    VLCInitState state = VLC_INIT_STATE(cbp_vlc);
    const uint8_t *lens = hqx_ac_lens;
    const int16_t *run_level = hqx_ac_run_level;

    ff_vlc_init_tables(&state, HQX_CBP_VLC_BITS, FF_ARRAY_ELEMS(cbp_vlc_lens),
                       cbp_vlc_lens, 1, 1, cbp_vlc_bits, 1, 1, 0);

    INIT_DC_TABLE(0, dc9);
    INIT_DC_TABLE(1, dc10);

    for (int i = 0; i < NUM_HQX_AC; ++i) {
        RL_VLC_ELEM *lut = state.table;
        unsigned nb_codes = state.size;

        hqx_ac[i].lut =
            ff_vlc_init_tables_from_lengths(&state, hqx_ac[i].bits,
                                            hqx_ac_nb_elems[i], lens, 1,
                                            run_level, 2, 2, 0, 0);

        nb_codes -= state.size;

        for (unsigned j = 0; j < nb_codes; ++j) {
            // lut[j] is in VLC (not RL_VLC) state
            int sym = lut[j].sym;
            int len = lut[j].len;
            int level;

            if (len < 0) {
                level = sym;
            } else {
                level      = sym >> 7;
                lut[j].run = sym & 0x7f;
            }
            // lut[j] is now in RL_VLC state
            lut[j].len8  = len;
            lut[j].level = level;
        }
        lens      += hqx_ac_nb_elems[i];
        run_level += hqx_ac_nb_elems[i];
    }
}

#endif /* AVCODEC_HQXVLC_H*/
