within ThermoSysPro.Combustion.CombustionChambers;
model GridFurnace "Combustion furnace"
  parameter Real X2eap=0.1 "Primary air fraction in zone 2";
  parameter Real X3eap=0.1 "Primary air fraction in zone 3";
  parameter Real XCleom=0 "Cl mass fraction in the biomass";
  parameter Real XFeom=0 "F mass fraction in the biomass";
  parameter Real XMACHeom=0 "machefers mass fraction in the biomass";
  parameter Modelica.SIunits.Density rhoCENDom=0.5 "Density of the ashes in the biomass";
  parameter ThermoSysPro.Units.AbsoluteTemperature Tfrecirc=600 "Temperature of the recirculated flue gases";
  parameter Real XfCO2recirc=0.3 "CO2 mass fraction in the recirculated flue gases";
  parameter Real XfH2Orecirc=0.1 "H2O mass fraction in the recirculated flue gases";
  parameter Real XfO2recirc=0.2 "O2 mass fraction in the recirculated flue gases";
  parameter Real XfSO2recirc=0 "SO2 mass fraction in the recirculated flue gases";
  parameter Real Xrecirc=0.1 "Recirculated flue gases fraction in Qsf";
  parameter Modelica.SIunits.SpecificHeatCapacity CpMACH=500 "Clinker average specific heat capacity";
  parameter Modelica.SIunits.SpecificHeatCapacity CpMACHs2=500 "Clinker specific heat capacity at the outlet of zone 2";
  parameter Modelica.SIunits.SpecificHeatCapacity CpMACHs3=500 "Clinker specific heat capacity at the outlet of zone 3";
  parameter Modelica.SIunits.SpecificHeatCapacity CpMACHs4=500 "Clinker specific heat capacity at the outlet of zone 4";
  parameter ThermoSysPro.Units.AbsoluteTemperature TsjeMACH=293 "Clinker temperature at the outlet of the water seal";
  parameter ThermoSysPro.Units.AbsoluteTemperature Teeje=293 "Water temperature at the outlet of the water seal";
  parameter Real XsjeH2OMACH=0.2 "Clinker humidity at the outlet of the water seal";
  parameter Real rendje=1 "Water seal efficiency";
  parameter Integer jointeau=0 "1: with water seal - 0: without water seal";
  parameter Real XCvol=0 "Volatile carbon fraction";
  parameter Real XCimb=0 "Unburnt carbon fraction";
  parameter ThermoSysPro.Units.AbsoluteTemperature T1sfm=500 "Flue gases temperature at the outlet of zone 1";
  parameter Real Eray2=0.1 "Energy fraction radiated towards zone 2 of the furnace";
  parameter Real Eray5=0.1 "Energy fraction radiated towards zone 5 of the furnac";
  parameter Real perte=0 "Loss percent of LHV";
  parameter Modelica.SIunits.SpecificHeatCapacity Cp3CO=500 "CO specific heat capacity in zone 3";
  Modelica.SIunits.MassFlowRate Qsf(start=10) "Flue gases mass flow rate at the outlet";
  ThermoSysPro.Units.AbsolutePressure Psf(start=100000.0) "Flue gases pressure at the outlet";
  ThermoSysPro.Units.AbsoluteTemperature Tsf(start=1000) "Flue gases temperature at the outlet";
  Real XsfN2(start=0.6) "Flue gases N2 mass fraction at the outlet";
  Real XsfCO2(start=0.1) "Flue gases CO2 mass fraction at the outlet";
  Real XsfH2O(start=0.1) "Flue gases H2O mass fraction at the outlet";
  Real XsfO2(start=0.1) "Flue gases O2 mass fraction at the outlet";
  Real XsfSO2(start=0.1) "Flue gases SO2 mass fraction at the outlet";
  Modelica.SIunits.MassFlowRate Qeap(start=10) "Primary air mass flow rate";
  ThermoSysPro.Units.AbsolutePressure Peap(start=100000.0) "Primary air pressure";
  ThermoSysPro.Units.AbsoluteTemperature Teap(start=300) "Primary air temperature";
  Real XeapN2(start=0.6) "Primary air N2 mass fraction";
  Real XeapCO2(start=0.1) "Primary air CO2 mass fraction";
  Real XeapH2O(start=0.1) "Primary air H2O mass fraction";
  Real XeapO2(start=0.1) "Primary air O2 mass fraction";
  Real XeapSO2(start=0.1) "Primary air SO2 mass fraction";
  Modelica.SIunits.MassFlowRate Qeas(start=10) "Secondary air mass flow rate";
  ThermoSysPro.Units.AbsolutePressure Peas(start=100000.0) "Secondary air pressure";
  ThermoSysPro.Units.AbsoluteTemperature Teas(start=300) "Secondary air temperature";
  Real XeasN2(start=0.6) "Secondary air N2 mass fraction";
  Real XeasCO2(start=0.1) "Secondary air CO2 mass fraction";
  Real XeasH2O(start=0.1) "Secondary air H2O mass fraction";
  Real XeasO2(start=0.1) "Secondaryr O2 mass fraction";
  Real XeasSO2(start=0.1) "Secondary SO2 mass fraction";
  Modelica.SIunits.MassFlowRate Qeom(start=10) "Biomass mass flow rate";
  ThermoSysPro.Units.AbsoluteTemperature Teom(start=300) "Biomass temperature";
  Real PCIom(start=1000000.0) "Biomass LHV (J/kg)";
  Real XCeom(start=0.1) "Biomass C mass fraction";
  Real XHeom(start=0.1) "Biomass H mass fraction";
  Real XOeom(start=0.1) "Biomass O mass fraction";
  Real XNeom(start=0.01) "Biomass N mass fraction";
  Real XSeom(start=0.1) "Biomass S mass fraction";
  Real XCENDeom(start=0.1) "Biomass ashes mass fraction";
  Real XH2Oeom(start=0.1) "Biomass humidity";
  Modelica.SIunits.SpecificHeatCapacity Cpom(start=1000) "Biomass specific heat capacity";
  Modelica.SIunits.MassFlowRate Qerefo(start=10) "Cooling water mass flow rate";
  ThermoSysPro.Units.SpecificEnthalpy Herefo(start=10000.0) "Cooling water specific enthalpy";
  Modelica.SIunits.MassFlowRate Qfrecirc(start=10) "Recirculated flue gases mass flow rate";
  Real XsfN2recirc(start=0.1) "Recirculated flue gases N2 mass fraction";
  Real PCIMACH(start=10000000.0) "Clinker LHV";
  Modelica.SIunits.MassFlowRate QsMACH(start=10) "Clinker mass flow rate";
  ThermoSysPro.Units.AbsoluteTemperature TsMACH(start=500) "Clinket temperature";
  Real FVN(start=0.1) "Volatile ashes mass fraction";
  Modelica.SIunits.Density rhocend(start=500) "Ashes density in the flue gases";
  Modelica.SIunits.Power Wsr(start=10000000.0) "Radiated power";
  Real excair(start=0.1) "Combustion excess air";
  Real X1eap(start=0.1) "Primary air fraction in zone 1";
  Modelica.SIunits.MassFlowRate Q1eap(start=10) "Primary mass flow rate in zone 1";
  Modelica.SIunits.MassFlowRate Q2eap(start=10) "Primary mass flow rate in zone 2";
  Modelica.SIunits.MassFlowRate Q3eap(start=10) "Primary mass flow rate in zone 3";
  Real XCvol2(start=0.1) "C mass fraction burnt in zone 2";
  Real XMACHimb(start=0.1) "C mass raction unburnt in the clinker";
  Real XCvol3(start=0.1) "C mass fraction burnt in zone 3";
  ThermoSysPro.Units.SpecificEnthalpy Heap(start=1000.0) "Humid air specific enthalpy at the primary air temperature";
  ThermoSysPro.Units.SpecificEnthalpy Heas(start=1000.0) "Humid air specific enthalpy at the secondary air temperature";
  ThermoSysPro.Units.SpecificEnthalpy H1a(start=1000.0) "Primary air specific enthalpy at T1sfm";
  ThermoSysPro.Units.SpecificEnthalpy Hefrecirc(start=1000.0) "Specific enthalpy of the incoming recirculated flue gases";
  Modelica.SIunits.MassFlowRate Qeasm(start=10) "Mass flow rate of the secondary air / recirculated flue gases mixture";
  ThermoSysPro.Units.SpecificEnthalpy Heasm(start=1000.0) "Specific enthalpy of the secondary air / recirculated flue gases mixture";
  ThermoSysPro.Units.AbsoluteTemperature Teasm(start=500) "Temperature of the secondary air / recirculated flue gases mixture";
  Real XeasmO2(start=0.1) "O2 mass fraction in the secondary air / recirculated flue gases mixture";
  Real XeasmCO2(start=0.1) "CO2 mass fraction in the secondary air / recirculated flue gases mixture";
  Real XeasmH2O(start=0.1) "H2O mass fraction in the secondary air / recirculated flue gases mixture";
  Real XeasmSO2(start=0.1) "SO2 mass fraction in the secondary air / recirculated flue gases mixture";
  Real XeasmN2(start=0.1) "N2 mass fraction in the secondary air / recirculated flue gases mixture";
  Integer mode=0 "IF97 region. 1:liquid - 2:steam - 4:saturation line - 0:automatic";
  ThermoSysPro.Units.SpecificEnthalpy Heauom(start=1000.0) "Biomass water specific enthalpy";
  ThermoSysPro.Units.AbsolutePressure Psateom(start=100000.0) "Water saturation presure at Teom";
  ThermoSysPro.Units.SpecificEnthalpy Hvteom(start=10000.0) "Steam saturation specific enthalpy at Teom";
  ThermoSysPro.Units.SpecificEnthalpy Hlteom(start=10000.0) "Water saturation specific enthalpy at Teom";
  ThermoSysPro.Units.SpecificEnthalpy Hvapteom(start=10000.0) "Phase transition energy at Teom";
  ThermoSysPro.Units.SpecificEnthalpy Hs1vom(start=1000.0) "Water specific enthalpy of the outgoing biomass at T1sfm vapor";
  ThermoSysPro.Units.SpecificEnthalpy Heom(start=1000.0) "Biomass specific enthalpy at the inlet";
  Modelica.SIunits.Power Wff(start=1000000.0) "Flue gases formation energy";
  Modelica.SIunits.Power Wp(start=1000000.0) "Biomass pyrolysis power";
  Modelica.SIunits.Power Wimbp(start=1000000.0) "Power saved in the combustion flue gases due to the non-destruction of unburnt C";
  Modelica.SIunits.Power Wimbm(start=1000000.0) "Power lost by the combustion flue gases due to the non-combustion of unburnt C";
  ThermoSysPro.Units.SpecificEnthalpy Hpyr(start=1000.0) "Pyrolysis specific enthalpy";
  ThermoSysPro.Units.SpecificEnthalpy H1om(start=1000.0) "Specific enthalpy of the dry biomass at the pyrolysis temperature of the biomass";
  Modelica.SIunits.MassFlowRate Q1H2O(start=10) "H2O mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q1O2(start=10) "O2 mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q1N2(start=10) "N2 mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q1CO2(start=10) "CO2 mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q1SO2(start=10) "SO2 mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q1g(start=10) "Total mass flow rate at the oultet of zone 1";
  Modelica.SIunits.MassFlowRate Q2eom(start=10) "Biomass mass flow rate at the inlet of zone 2";
  Real PCI1om(start=1000000.0) "LHV after drying";
  Real X1MACHom(start=0.1) "Clinker mass fraction in the biomass after drying";
  Real XC1vol2(start=0.1) "Burnt C mass fraction in zone 2 after drying";
  Real XC1vol3(start=0.1) "Burnt C mass fraction in zone 3 after drying";
  Real X1MACHimb(start=0.1) "Unburnt C mass fraction in the clinker after drying";
  Real X1H(start=0.1) "H mass fraction in the biomass after drying";
  Real X1O(start=0.1) "O mass fraction in the biomass after drying";
  Real X1N(start=0.1) "N mass fraction in the biomass after drying";
  Real X1Cl(start=0.1) "Cl mass fraction in the biomass after drying";
  Real X1F(start=0.1) "F mass fraction in the biomass after drying";
  Real X1S(start=0.1) "S mass fraction in the biomass after drying";
  Real X1CEND(start=0.1) "Ashes mass fraction in the biomass after drying";
  Modelica.SIunits.MassFlowRate Qcendom(start=10) "Ashes mass flow rate";
  Real Xfcend(start=0.1) "Ashes mass fraction in the flue gases";
  Modelica.SIunits.Power P1g(start=1000000.0) "Power saved in zone 1";
  Modelica.SIunits.MassFlowRate Q2eo(start=10) "Mass flow rate of the oxygen carried by the air and the biomass at the inlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2HCl(start=10) "Combustion HCl mass fraction in zone 2";
  Modelica.SIunits.MassFlowRate Q2HF(start=10) "Combustion HF mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2SO2(start=10) "Combustion SO2 mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2H2O(start=10) "Combustion H2O mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2CO(start=10) "Combustion CO mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2N2(start=10) "Combustion N2 mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2O2(start=10) "Combustion O2 mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2cend(start=10) "Combustion ashes mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2CO2(start=10) "Combustion CO2 mass flow rate at the outlet of zone 2";
  Modelica.SIunits.MassFlowRate Q2g(start=10) "Elements total mass flow rate after combustion";
  Real Epsivol(start=0.1) "Volatile matter mass fraction produced in zone 2";
  Modelica.SIunits.MassFlowRate Q3eom(start=10) "Solid matter at the inlet of zone 3";
  Real X2MACHom(start=0.1) "Clinker mass fraction of the biomass after volatilisation";
  Real XC2vol3(start=0.1) "C mass fraction burnt in zone 3 after volatilisation";
  Real X2MACHimb(start=0.1) "Unburnt C mas fraction in the clinker after volatilisation";
  Real PCICsol(start=1000000.0) "LHV of the solid outgoing carbon";
  Real PCICvol(start=1000000.0) "LHV of the volatile carbon transformed into CO";
  ThermoSysPro.Units.SpecificEnthalpy H2(start=1000.0) "Enthalpy released in zone 2";
  Modelica.SIunits.Power P2g(start=1000000.0) "Power released by the combustion in zone 2";
  Modelica.SIunits.Power P1o(start=1000000.0) "Power captured by the biomass";
  Modelica.SIunits.Power P1v(start=1000000.0) "Power captured by the steam";
  Modelica.SIunits.Power P1a(start=1000000.0) "Power captured by the air";
  Modelica.SIunits.Power P1r(start=1000000.0) "Power captured by the cooling water";
  Real Eray0(start=0.2) "Fraction of the radiated power from zone 2 unused for the drying";
  Real X2O2(start=0.1) "O2 mass fraction at the oultet of zone 2";
  Real X2SO2(start=0.1) "SO2 mass fraction at the oultet of zone 2";
  Real X2CO2(start=0.1) "CO2 mass fraction at the oultet of zone 2";
  Real X2H2O(start=0.1) "H2O mass fraction at the oultet of zone 2";
  Real X2N2(start=0.1) "N2 mass fraction at the oultet of zone 2";
  ThermoSysPro.Units.SpecificEnthalpy H2g(start=1200000) "Flue gases specific enthalpy at the oultet of zone 2";
  ThermoSysPro.Units.AbsoluteTemperature T2(start=1000) "Flue gases temperature at the oultet of zone 2";
  Modelica.SIunits.MassFlowRate Q3od(start=10) "O mass flow rate available for the oxydation of the clinker in CO and/or CO2";
  Modelica.SIunits.MassFlowRate Q3cd(start=10) "C mass flow rate available for the oxydation of the clinker in CO and/or CO2";
  Real taux3oc(start=0.1) "Ratio of the mass flow rates O/C available for the oxydation of the clinker";
  Modelica.SIunits.MassFlowRate Q3CO2(start=10) "CO2 mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q3CO(start=10) "CO mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q3O2(start=10) "O2 mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q3N2(start=10) "N2 mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q3H2O(start=10) "H2O mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q3SO2(start=10) "SO2 mass flow rate at the outlet of zone 3";
  Modelica.SIunits.Power P3s(start=1000000.0) "Power captured by solid matter in zone 3";
  Modelica.SIunits.Power P3g(start=1000000.0) "Power captured by gaseous matter in zone 3";
  ThermoSysPro.Units.SpecificEnthalpy H3s(start=1000.0) "Gaseous matter specific enthalpy at T3g";
  ThermoSysPro.Units.SpecificEnthalpy H3g(start=1000.0) "Solid matter specific enthalpy at T3o";
  Real XC2vol31(start=0.1) "C mass fraction burnt in zone 3 after volatilisation";
  Real XC2vol4(start=0.1) "C mass fraction burnt at the inlet of zone 4";
  Modelica.SIunits.MassFlowRate Q3g(start=10) "Flue gases mass flow rate at the outlet of zone 3";
  Modelica.SIunits.MassFlowRate Q4eom(start=10) "Clinker mass flow rate at the inlet of zone 4";
  Real X4MACHom(start=0.1) "Mass flow rate of the biomass clinker after C volatilisation in zone 3";
  Real X4MACHimb(start=0.1) "Unburnt C mass fraction in the clinker after C volatilisation in zone 3";
  Modelica.SIunits.SpecificHeatCapacity Cp3a(start=1000) "Average specific heat capacity at the inlet of zone 3";
  ThermoSysPro.Units.AbsoluteTemperature T3o(start=500) "Clinker temperature at the outlet of zone 3";
  constant Modelica.SIunits.SpecificHeatCapacity Cp3g=1100 "Average flue gases specific heat capacity at T3g";
  Modelica.SIunits.Power P3ac(start=1000000.0) "Air power heated at (T2 + T3o)/2";
  Modelica.SIunits.Power P3co(start=1000000.0) "CO power heated at (T2 + T3o)/2";
  Modelica.SIunits.Power P3(start=1000000.0) "Total flue gases power at the outlet of zone 3";
  ThermoSysPro.Units.AbsoluteTemperature T4o(start=600) "Clinker temperature at the outlet for the water seal";
  ThermoSysPro.Units.AbsoluteTemperature T4er(start=600) "Water temperature at the inlet of the water seal";
  Real X4H2O(start=0.1) "H2O mass fraction in the clinker at the outlet of the water seal";
  Modelica.SIunits.SpecificHeatCapacity Cp4liq(start=1000) "Water specific heat capacity at TEej";
  Modelica.SIunits.Power P4m(start=1000000.0) "Power lost by the clinker during vaporisation";
  Modelica.SIunits.Power P4h(start=1000000.0) "Power associated to the clinker humidity";
  Modelica.SIunits.MassFlowRate Q4v(start=10) "Steam mass flow rate generated by the water seal";
  ThermoSysPro.Units.SpecificEnthalpy H4(start=1000.0) "Enthalpy in zone 4";
  Modelica.SIunits.Power P4v(start=1000000.0) "Power captured by the steam in zone 4";
  constant ThermoSysPro.Units.SpecificEnthalpy Hvapo=2501600 "Vaporisation energy";
  Modelica.SIunits.MassFlowRate QO2p(start=10) "Flue gases O2 mass flow rate at the outlet";
  Modelica.SIunits.MassFlowRate Qairp(start=10) "Excess air mass flow rate for data Qo2p";
  Modelica.SIunits.MassFlowRate Qairs(start=10) "Stoechiometric air mass flow rate";
  Modelica.SIunits.MassFlowRate Q5eH2O(start=10) "Steam mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eCO(start=10) "CO mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eCO2(start=10) "CO2 mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eO2(start=10) "O2 mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eN2(start=10) "N2 mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eSO2(start=10) "SO2 mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eHCl(start=10) "HCl mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eHF(start=10) "HF mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5ecend(start=10) "Ashes mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5eam(start=10) "Total mass flow rate at the inlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5od(start=10) "O mass flow rate available in zone 5";
  Modelica.SIunits.MassFlowRate Q5cd(start=10) "C mass flow rate available in zone 5";
  Modelica.SIunits.MassFlowRate Q5hd(start=10) "H mass flow rate available in zone 5";
  Modelica.SIunits.MassFlowRate Q5ost(start=10) "Stoechiomtrique O mass flow rate for zone 5";
  Real exc5(start=0.1) "Air excess for zone 5";
  Modelica.SIunits.Power P5(start=1000000.0) "Power released by oxydation in zone 5";
  Modelica.SIunits.MassFlowRate Q5sCO2(start=10) "CO2 mass flow rate at the outlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5sO2(start=10) "O2 mass flow rate at the outlet of zone 5";
  Modelica.SIunits.MassFlowRate Q5sCO(start=10) "CO mass flow rate at the outlet of zone 5";
  Modelica.SIunits.Power P5s(start=1000000.0) "Power released by oxydation in zone 5";
  Real XsfCO(start=0.1) "Flue gases CO mass fraction at the outlet of zone 5";
  Real XsfN21(start=0.1) "Flue gases N2 mass fraction at the outlet of zone 5";
  Real XsfHCl(start=0.1) "Flue gases HCl mass fraction at the outlet of zone 5";
  Real XsfHF(start=0.1) "Flue gases HF mass fraction at the outlet of zone 5";
  Real XsfCEND(start=0.1) "Ashes mass fraction at the outlet of zone 5";
  Modelica.SIunits.Power P5a(start=1000000.0) "Power brought by secondary air";
  Modelica.SIunits.Power P5t(start=1000000.0) "Power accumulated in the flue gases";
  Real Xcor(start=0.1) "Corrective factor for the flue gases mass fractions";
  Real X5sH2OC(start=0.1) "Corrected flue gases H2O mass fraction";
  Real X5sCO2C(start=0.1) "Corrected flue gases CO2 mass fraction";
  Real X5sO2C(start=0.1) "Corrected flue gases O2 mass fraction";
  Real X5sSO2C(start=0.1) "Corrected flue gases SO2 mass fraction";
  Real X5sN2C(start=0.1) "Corrected flue gases N2 mass fraction";
  Modelica.SIunits.Density rhonorm(start=1000) "Density of the outgoing flue gases Masse at 0 deg C and 1 atm";
  Real FVN0(start=0.1) "Ashes normal volume fraction for the computation of FVN";
  ThermoSysPro.FlueGases.Connectors.FlueGasesInlet Ca2 annotation(Placement(transformation(x=-50.0, y=60.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false), iconTransformation(x=-50.0, y=60.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Combustion.Connectors.FuelInlet Com annotation(Placement(transformation(x=-90.0, y=-10.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false), iconTransformation(x=-90.0, y=-10.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.WaterSteam.Connectors.FluidInlet port_eau_refroid annotation(Placement(transformation(x=80.0, y=30.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false), iconTransformation(x=80.0, y=30.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Properties.WaterSteam.Common.ThermoProperties_pT pro1 annotation(Placement(transformation(x=-90.0, y=90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Properties.WaterSteam.Common.ThermoProperties_pT pro2 annotation(Placement(transformation(x=-90.0, y=70.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Properties.WaterSteam.Common.ThermoProperties_pT pro3 annotation(Placement(transformation(x=-90.0, y=50.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Properties.WaterSteam.Common.ThermoProperties_pT pro4 annotation(Placement(transformation(x=-90.0, y=30.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.Properties.WaterSteam.Common.ThermoProperties_pT pro5 annotation(Placement(transformation(x=-50.0, y=90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.FlueGases.Connectors.FlueGasesInlet Ca1 annotation(Placement(transformation(x=0.0, y=-90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false), iconTransformation(x=0.0, y=-90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
  ThermoSysPro.FlueGases.Connectors.FlueGasesOutlet Cfg "Flue gases outlet" annotation(Placement(transformation(x=0.0, y=90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false), iconTransformation(x=0.0, y=90.0, scale=0.1, aspectRatio=1.0, flipHorizontal=false, flipVertical=false)));
protected
  constant ThermoSysPro.Units.SpecificEnthalpy H0v=2501551.43 "Vaporisation energy at 0C";
  constant ThermoSysPro.Units.SpecificEnthalpy HfCO2=32791664.0 "CO2 formation enthalpy";
  constant ThermoSysPro.Units.SpecificEnthalpy HfCO=9201000.0 "CO formation enthalpy";
  constant ThermoSysPro.Units.SpecificEnthalpy HfH2Og=13433333.0 "H2Og formation enthalpy";
  constant ThermoSysPro.Units.SpecificEnthalpy HfSO2=4630265.0 "SO2 formation enthalpy";
  constant ThermoSysPro.Units.SpecificEnthalpy HfH2Ol=15883300.0 "H2Ol formation enthalpy";
equation
  Qsf=Cfg.Q;
  Tsf=Cfg.T;
  Psf=Cfg.P;
  XsfCO2=Cfg.Xco2;
  XsfH2O=Cfg.Xh2o;
  XsfO2=Cfg.Xo2;
  XsfSO2=Cfg.Xso2;
  Qeap=Ca1.Q;
  Peap=Ca1.P;
  Peap=Psf;
  Teap=Ca1.T;
  XeapCO2=Ca1.Xco2;
  XeapH2O=Ca1.Xh2o;
  XeapO2=Ca1.Xo2;
  XeapN2=1 - Ca1.Xco2 - Ca1.Xh2o - Ca1.Xo2 - Ca1.Xso2;
  XeapSO2=Ca1.Xso2;
  Qeas=Ca2.Q;
  Peas=Ca2.P;
  Peas=Psf;
  Teas=Ca2.T;
  XeasCO2=Ca2.Xco2;
  XeasH2O=Ca2.Xh2o;
  XeasO2=Ca2.Xo2;
  XeasN2=1 - Ca2.Xco2 - Ca2.Xh2o - Ca2.Xo2 - Ca2.Xso2;
  XeasSO2=Ca2.Xso2;
  Qeom=Com.Q;
  Teom=Com.T;
  PCIom=Com.LHV;
  XCeom=Com.Xc;
  XHeom=Com.Xh;
  XOeom=Com.Xo;
  XNeom=Com.Xn;
  XSeom=Com.Xs;
  XCENDeom=Com.Xashes;
  XH2Oeom=Com.hum;
  Cpom=Com.cp;
  Qerefo=port_eau_refroid.Q;
  Herefo=port_eau_refroid.h;
  port_eau_refroid.h=port_eau_refroid.h_vol;
  X1eap=1 - X2eap - X3eap;
  Q1eap=Qeap*X1eap;
  Q2eap=Qeap*X2eap;
  Q3eap=Qeap*X3eap;
  Qfrecirc=Qsf*Xrecirc;
  XCvol2=XCeom*XCvol;
  XMACHimb=XCeom*XCimb;
  XCvol3=XCeom*(1 - XCvol - XCimb);
  Heap=ThermoSysPro.Properties.FlueGases.FlueGases_h(Peap, Teap, XeapCO2, XeapH2O, XeapO2, XeapSO2);
  Heas=ThermoSysPro.Properties.FlueGases.FlueGases_h(Peap, Teas, XeasCO2, XeasH2O, XeasO2, XeasSO2);
  H1a=ThermoSysPro.Properties.FlueGases.FlueGases_h(Peap, T1sfm, XeapCO2, XeapH2O, XeapO2, XeapSO2);
  XsfN2recirc=1 - XfCO2recirc - XfH2Orecirc - XfO2recirc - XfSO2recirc;
  Hefrecirc=ThermoSysPro.Properties.FlueGases.FlueGases_h(Psf, Tfrecirc, XfCO2recirc, XfH2Orecirc, XfO2recirc, XfSO2recirc);
  Qeasm=Qeas + Qfrecirc;
  if Qeasm <= 0 then
    XeasmO2=0;
    XeasmCO2=0;
    XeasmH2O=0;
    XeasmSO2=0;
    XeasmN2=0;
    Heasm=1000.0;
    Teasm=274.15;
  else
    XeasmO2=(XfO2recirc*Qfrecirc + XeasO2*Qeas)/Qeasm;
    XeasmCO2=XfCO2recirc*Qfrecirc/Qeasm;
    XeasmH2O=(XfH2Orecirc*Qfrecirc + XeasH2O*Qeas)/Qeasm;
    XeasmSO2=XfSO2recirc*Qfrecirc/Qeasm;
    XeasmN2=1 - XeasmO2 - XeasmSO2 - XeasmH2O - XeasmSO2;
    Heasm=(Qeasm*Heas + Qfrecirc*Hefrecirc)/Qeasm;
    Heasm=ThermoSysPro.Properties.FlueGases.FlueGases_h(Psf, Teasm, XeasmCO2, XeasmH2O, XeasmO2, XeasmSO2);
  end if;
  pro1=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Peap, Teom, mode);
  Heauom=pro1.h;
  Psateom=ThermoSysPro.Properties.WaterSteam.BaseIF97.Basic.psat(Teom);
  pro2=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Psateom, Teom, 2);
  Hvteom=pro2.h;
  pro3=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Psateom, Teom, 1);
  Hlteom=pro3.h;
  Hvapteom=Hvteom - Hlteom;
  pro4=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Psf, T1sfm, 0);
  Hs1vom=pro4.h;
  Heom=Cpom*(Teom - 273.15);
  Wff=Qeom*(HfCO2*XCeom + HfH2Og*XHeom*9 + HfSO2*XSeom/32.1*64.1 + HfH2Og*XH2Oeom);
  Wp=Wff - Qeom*PCIom/(1 - perte) - Qeom*HfH2Ol*XH2Oeom;
  Wimbp=Wp*XCimb;
  Wimbm=Qeom*XCeom*HfCO2*XCimb;
  Hpyr=Wp/(Qeom*(1 - XH2Oeom));
  H1om=Hpyr + Heom;
  Q1H2O=Qeom*XH2Oeom + Q1eap*XeapH2O + Qerefo;
  Q1O2=Q1eap*XeapO2;
  Q1N2=Q1eap*XeapN2;
  Q1CO2=Q1eap*XeapCO2;
  Q1SO2=Q1eap*XeapSO2;
  Q1g=Q1H2O + Q1O2 + Q1N2 + Q1CO2 + Q1SO2;
  Q2eom=Qeom*(1 - XH2Oeom);
  PCI1om=PCIom/(1 - XH2Oeom);
  X1MACHom=XMACHeom/(1 - XH2Oeom);
  XC1vol2=XCvol2/(1 - XH2Oeom);
  XC1vol3=XCvol3/(1 - XH2Oeom);
  X1MACHimb=XMACHimb/(1 - XH2Oeom);
  X1H=XHeom/(1 - XH2Oeom);
  X1O=XOeom/(1 - XH2Oeom);
  X1N=XNeom/(1 - XH2Oeom);
  X1Cl=XCleom/(1 - XH2Oeom);
  X1F=XFeom/(1 - XH2Oeom);
  X1S=XSeom/(1 - XH2Oeom);
  X1CEND=XCENDeom/(1 - XH2Oeom);
  Qcendom=XCENDeom*Qeom;
  Xfcend=Qcendom/Qsf;
  P1g=Qeom*XH2Oeom*Hs1vom + Qeom*XH2Oeom*Hvapteom + Q1eap*H1a + Qerefo*Hs1vom;
  Q2eo=Q2eom*X1O + Q2eap*XeapO2;
  Q2HCl=36.5/35.5*Q2eom*X1Cl;
  Q2HF=20/19*Q2eom*X1F;
  Q2SO2=64/32*Q2eom*X1S + Q2eap*XeapSO2;
  Q2H2O=18/2*Q2eom*(X1H - 1/35.5*X1Cl - 1/19*X1F) + Q2eap*XeapH2O;
  Q2CO=28/12*Q2eom*XC1vol2;
  Q2N2=Q2eo*X1N + Q2eap*XeapN2;
  Q2O2=Q2eo - Q2eom*(X1S + 16/2*(X1H - 1/35.5*X1Cl - 1/19*X1F) + 16/12*XC1vol2);
  Q2cend=Q2eom*X1CEND;
  Q2CO2=Q2eap*XeapCO2;
  Q2g=Q2HCl + Q2HF + Q2SO2 + Q2H2O + Q2CO + Q2N2 + Q2O2 + Q2cend + Q2CO2;
  Epsivol=XC1vol2 + X1H + X1O + X1S + X1Cl + X1F + X1N + X1CEND;
  Q3eom=Q2eom*(1 - Epsivol);
  X2MACHom=X1MACHom/(1 - Epsivol);
  XC2vol3=XC1vol3/(1 - Epsivol);
  X2MACHimb=X1MACHimb/(1 - Epsivol);
  PCICsol=(XC1vol3 + X1MACHimb)*HfCO2;
  PCICvol=XC1vol2*(HfCO2 - HfCO);
  H2=PCI1om - PCICsol - PCICvol;
  P2g=H2*Q2eom + (Q2H2O - Q2eap*XeapH2O)*H0v + Wimbp;
  P1o=Qeom*(1 - XH2Oeom)*(H1om - Heom);
  P1v=Qeom*XH2Oeom*(Hs1vom - Hlteom);
  P1a=Q1eap*(H1a - Heap);
  P1r=Qerefo*(Hs1vom - Herefo);
  Eray0=(P1o + P1v + P1a + P1r)/P2g;
  X2O2=Q2O2/Q2g;
  X2SO2=Q2SO2/Q2g;
  X2CO2=Q2CO/Q2g;
  X2H2O=Q2H2O/Q2g;
  X2N2=1 - (X2O2 + X2SO2 + X2CO2 + X2H2O);
  H2g=(Q2eap*Heap + Q2eom*H1om + P2g*(1 - Eray0 - Eray2) - Q3eom*CpMACHs2*(T2 - 273.15))/Q2g;
  H2g=ThermoSysPro.Properties.FlueGases.FlueGases_h(Peap, T2, X2CO2, X2H2O, X2O2, X2SO2);
  Q3od=Q3eap*XeapO2;
  Q3cd=Q3eom*XC2vol3;
  taux3oc=Q3od/Q3cd;
  if taux3oc >= 32/12 then
    Q3CO2=44/12*Q3cd + Q3eap*XeapCO2;
    Q3CO=0;
    Q3O2=Q3od - 32/12*Q3cd;
    Q3N2=Q3eap*XeapN2;
    Q3H2O=Q3eap*XeapH2O;
    Q3SO2=Q3eap*XeapSO2;
    P3s=Q3cd*HfCO;
    P3g=Q3cd*(HfCO2 - HfCO);
    H3s=P3s/Q3eom;
    H3g=P3g/Q3eom;
    XC2vol31=XC2vol3;
    XC2vol4=XC2vol3;
  elseif taux3oc >= 16/12 and taux3oc < 32/12 then
    Q3CO2=44/12*(Q3od - 16/12*Q3cd) + Q3eap*XeapCO2;
    Q3CO=28/12*Q3cd - 28/16*(Q3od - 16/12*Q3cd);
    Q3O2=0;
    Q3N2=Q3eap*XeapN2;
    Q3H2O=Q3eap*XeapH2O;
    Q3SO2=Q3eap*XeapSO2;
    P3s=Q3cd*HfCO;
    P3g=(12/16*Q3od - Q3cd)*(HfCO2 - HfCO);
    H3s=P3s/Q3eom;
    H3g=P3g/Q3eom;
    XC2vol31=XC2vol3;
    XC2vol4=XC2vol3;
  else
    Q3CO2=0;
    Q3CO=28/16*Q3od;
    Q3O2=0;
    Q3N2=Q3eap*XeapN2;
    Q3H2O=Q3eap*XeapH2O;
    Q3SO2=Q3eap*XeapSO2;
    P3s=12/16*Q3od*HfCO;
    P3g=0;
    H3s=P3s/Q3eom;
    H3g=P3g/Q3eom;
    XC2vol31=XC2vol3*12/16*taux3oc;
    XC2vol4=XC2vol3;
  end if;
  Q3g=Q3CO2 + Q3CO + Q3H2O + Q3O2 + Q3N2 + Q3SO2;
  Q4eom=Q3eom*(1 - XC2vol31);
  X4MACHom=X2MACHom/(1 - XC2vol31);
  X4MACHimb=X2MACHimb/(1 - XC2vol31) + (XC2vol4 - XC2vol31)/(1 - XC2vol31);
  Cp3a=ThermoSysPro.Properties.FlueGases.FlueGases_h(Peap, (T2 + Teap)/2, X2CO2, X2H2O, X2O2, X2SO2)/((T2 + Teap)/2);
  T3o - 273.15=(Q3eap*Heap + Q3eom*CpMACHs2*(T2 - 273.15) - (Q3eap - 16/12*Q3eom*XC2vol31)*Cp3a*(T2 - 273.15)/2 - 28/12*Q3eom*XC2vol31*Cp3CO*(T2 - 273.15)/2 + P3s)/(Q3eap*Cp3a/2 + 28/12*Q3eom*XC2vol31*Cp3CO/2 + Q4eom*CpMACHs3);
  P3ac=(Q3eap - 16/12*Q3eom*XC2vol31)*Cp3a*(T2 - 273.15 + (T3o - 273.15))/2;
  P3co=28/12*Q3eom*XC2vol31*Cp3CO*(T2 - 273.15 + (T3o - 273.15))/2;
  P3=P3g + P3ac + P3co;
  if jointeau == 1 then
    T4o=TsjeMACH;
    TsMACH=TsjeMACH;
    T4er=Teeje;
    X4H2O=XsjeH2OMACH;
    Cp4liq=4180;
    P4m=Q4eom*CpMACHs4*(T3o - 273.15 - (T4o - 273.15));
    P4h=Q4eom*X4H2O*Cp4liq*(T4o - 273.15 - (T4er - 273.15));
    Q4v=rendje*(P4m - P4h)/(Cp4liq*(373.15 - (T4er - 273.15)) + Hvapo);
    pro5=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Peap, 373.15, 2);
    H4=pro5.h;
    P4v=Q4v*pro5.h;
  else
    T4o=273.15;
    TsMACH=T3o;
    T4er=273.15;
    X4H2O=0;
    Cp4liq=0;
    P4m=0;
    P4h=0;
    Q4v=0;
    pro5=ThermoSysPro.Properties.WaterSteam.IF97.Water_PT(Peap, 373.15, 2);
    H4=pro5.h;
    P4v=0;
  end if;
  QsMACH=Q4eom;
  PCIMACH=(Wimbm - Wimbp)/QsMACH;
  QO2p=Qsf*XsfO2;
  Qairp=QO2p/XeapO2*(1 - XeapH2O);
  Qairs=Qeap + Qeas - Qairp;
  excair=Qairp/Qairs*100;
  Q5eH2O=Q1H2O + Q2H2O + Q3H2O + Q4v + Qeasm*XeasmH2O;
  Q5eCO=Q2CO + Q3CO;
  Q5eCO2=Q3CO2 + Q1CO2 + Q2CO2 + Qeasm*XeasmCO2;
  Q5eO2=Q1O2 + Q2O2 + Q3O2 + Qeasm*XeasmO2;
  Q5eN2=Q1N2 + Q2N2 + Q3N2 + Qeasm*XeasmN2;
  Q5eSO2=Q1SO2 + Q2SO2 + Q3SO2 + Qeasm*XeasmSO2;
  Q5eHCl=Q2HCl;
  Q5eHF=Q2HF;
  Q5ecend=Q2cend;
  Q5eam=Q5eH2O + Q5eCO + Q5eCO2 + Q5eO2 + Q5eN2 + Q5eSO2 + Q5eHCl + Q5eHF + Q5ecend;
  Q5od=Q5eO2 + 16/18*Q5eH2O + 32/44*Q5eCO2 + 16/28*Q5eCO;
  Q5cd=12/44*Q5eCO2 + 12/28*Q5eCO;
  Q5hd=2/18*Q5eH2O;
  Q5ost=16/2*Q5hd + 32/12*Q5cd;
  exc5=Q5od/Q5ost;
  if exc5 > 1 then
    P5=12/28*Q5eCO*(HfCO2 - HfCO);
    Q5sCO2=Q5eCO2 + 44/28*Q5eCO;
    Q5sO2=Q5eO2 - 16/28*Q5eCO;
    Q5sCO=0;
  else
    P5=Q5eO2*12/16*(HfCO2 - HfCO);
    Q5sCO2=Q5eCO2 + 44/16*Q5eO2;
    Q5sO2=0;
    Q5sCO=Q5eCO - 28/16*Q5eO2;
  end if;
  Qsf=Q5eH2O + Q5sCO + Q5sCO2 + Q5sO2 + Q5eN2 + Q5eSO2 + Q5eHCl + Q5eHF + Q5ecend;
  P5s=P5*(1 - Eray5);
  XsfH2O=Q5eH2O/Qsf;
  XsfCO=Q5sCO/Qsf;
  XsfCO2=Q5sCO2/Qsf;
  XsfO2=Q5sO2/Qsf;
  XsfSO2=Q5eSO2/Qsf;
  XsfN21=Q5eN2/Qsf;
  XsfHCl=Q5eHCl/Qsf;
  XsfHF=Q5eHF/Qsf;
  XsfCEND=Q5ecend/Qsf;
  XsfN2=1 - (XsfCO2 + XsfH2O + XsfO2 + XsfSO2);
  P5a=Qeasm*Heasm;
  P5t=P1g + H2g*Q2g + P3 + P4v + P5s + P5a;
  Wsr=P5s*Eray5/(1 - Eray5) + P2g*Eray2;
  P5t/Qsf=ThermoSysPro.Properties.FlueGases.FlueGases_h(Psf, Tsf, XsfCO2, XsfH2O, XsfO2, XsfSO2);
  rhocend=rhoCENDom;
  Xcor=1/(1 - XsfH2O);
  X5sH2OC=0;
  X5sCO2C=XsfCO2*Xcor;
  X5sO2C=XsfO2*Xcor;
  X5sSO2C=XsfSO2*Xcor;
  X5sN2C=1 - (X5sH2OC + X5sCO2C + X5sO2C + X5sSO2C);
  rhonorm=ThermoSysPro.Properties.FlueGases.FlueGases_rho(101325.0, 273.15, X5sCO2C, X5sH2OC, X5sO2C, X5sSO2C);
  FVN0=Qcendom/rhocend/(Qsf/rhonorm);
  0=if FVN0 < 0 or FVN > 0.1 then FVN - 0.001 else FVN - FVN0;
  annotation(Diagram(coordinateSystem(extent={{-100,-100},{100,100}}), graphics={Polygon(points={{-80,20},{-80,-80},{100,-80},{100,20},{62,20},{20,40},{20,80},{-40,80},{-40,40},{-80,20}}, lineColor={0,0,255}, fillColor={255,255,0}, fillPattern=FillPattern.Solid),Polygon(points={{-80,0},{60,-40},{80,-80},{-80,-80},{-80,0}}, lineColor={0,0,255}, fillColor={160,160,160}, fillPattern=FillPattern.Solid)}), Icon(coordinateSystem(extent={{-100,-100},{100,100}}), graphics={Polygon(points={{-80,20},{-80,-80},{100,-80},{100,20},{62,20},{20,40},{20,80},{-40,80},{-40,40},{-80,20}}, lineColor={0,0,255}, fillColor={255,255,0}, fillPattern=FillPattern.Solid),Polygon(points={{-80,0},{60,-40},{80,-80},{-80,-80},{-80,0}}, lineColor={0,0,255}, fillColor={160,160,160}, fillPattern=FillPattern.Solid)}), DymolaStoredErrors, Documentation(revisions="<html>
<u><p><b>Authors</u> : </p></b>
<ul style='margin-top:0cm' type=disc>
<li>
    Baligh El Hefni</li>
</ul>
</html>
", info="<html>
<p><b>Copyright &copy; EDF 2002 - 2010</b></p>
</HTML>
<html>
<p><b>ThermoSysPro Version 2.0</b></p>
</HTML>
"));
end GridFurnace;
