﻿#
# This script handles common telemetry tasks for Install.ps1 and Add-AppDevPackage.ps1.
#

function IsVsTelemetryRegOptOutSet()
{
    $VsTelemetryRegOptOutKeys = @(
        "HKLM:\SOFTWARE\Policies\Microsoft\VisualStudio\SQM",
        "HKLM:\SOFTWARE\Wow6432Node\Microsoft\VSCommon\16.0\SQM",
        "HKLM:\SOFTWARE\Microsoft\VSCommon\16.0\SQM",
        "HKLM:\SOFTWARE\Wow6432Node\Microsoft\VSCommon\17.0\SQM",
        "HKLM:\SOFTWARE\Microsoft\VSCommon\17.0\SQM"
    )

    foreach ($optOutKey in $VsTelemetryRegOptOutKeys)
    {
        if (Test-Path $optOutKey)
        {
            # If any of these keys have the DWORD value OptIn set to 0 that means that the user
            # has explicitly opted out of logging telemetry from Visual Studio.
            $val = (Get-ItemProperty $optOutKey)."OptIn"
            if ($val -eq 0)
            {
                return $true
            }
        }
    }

    return $false
}

try
{
    $TelemetryOptedOut = IsVsTelemetryRegOptOutSet
    if (!$TelemetryOptedOut)
    {
        $TelemetryAssembliesFolder = $args[0]
        $EventName = $args[1]
        $ReturnCode = $args[2]
        $ProcessorArchitecture = $args[3]

        foreach ($file in Get-ChildItem $TelemetryAssembliesFolder -Filter "*.dll")
        {
            [Reflection.Assembly]::LoadFile((Join-Path $TelemetryAssembliesFolder $file)) | Out-Null
        }

        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.IsOptedIn = $True
        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.Start()

        $TelEvent = New-Object "Microsoft.VisualStudio.Telemetry.TelemetryEvent" -ArgumentList $EventName
        if ($null -ne $ReturnCode)
        {
            $TelEvent.Properties["VS.DesignTools.SideLoadingScript.ReturnCode"] = $ReturnCode
        }

        if ($null -ne $ProcessorArchitecture)
        {
            $TelEvent.Properties["VS.DesignTools.SideLoadingScript.ProcessorArchitecture"] = $ProcessorArchitecture
        }

        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.PostEvent($TelEvent)
        [Microsoft.VisualStudio.Telemetry.TelemetryService]::DefaultSession.Dispose()
    }
}
catch
{
    # Ignore telemetry errors
}
# SIG # Begin signature block
# MIIplQYJKoZIhvcNAQcCoIIphjCCKYICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBs5RUy/PKaA/U5
# /+cdRqsBJInUnLtOhcpeK3MamAG/SqCCDeUwgga9MIIEpaADAgECAhMzAAAAHEif
# gd+hsLd3AAAAAAAcMA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEG
# A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
# cm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0
# aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0yNDA4MDgyMTM2MjNaFw0zNTA2MjMy
# MjA0MDFaMF8xCzAJBgNVBAYTAlVTMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
# YXRpb24xMDAuBgNVBAMTJ01pY3Jvc29mdCBXaW5kb3dzIENvZGUgU2lnbmluZyBQ
# Q0EgMjAyNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJp9a30nwXYq
# Lq7j1TT/zCtt7vxU+CCj+7BkifS/B2gXKGU7OV9SXRJGP1yFs5p6jpsYi4cYzF56
# AV0AEmmEjV8wT2lvPU5BhN3wV30HqYPIYEj5P3WXf0kXD9fvjUf1GAtXEriJ8w7A
# LNaVEm9Rs4ePA0ZsYHaCbU5kBUJQDXv76hafOcQgdFCA3I3zYtfzX2vOwx87uDOa
# CuyKORZih9c3zTf+TLC5QYLyhVMBnDXEHDOrvaw92DSyIqpdgRWpufzqDFy1egVj
# koXZhb+9pZ9heUzNXTXhOoXzexh6YzAL4flBWm+Bc1hQyESenEvBJznV+25u3h77
# jjgMUY44+WXQ4u9qddDe/U5SeAaKRvvibmi4z7QRpLvZsla0CPiOUGz00Do5sfkC
# 0EwlsSzfM3+8A9rsyFVOgWDVPzt98OJP2EoaEOq8GE9GCoN2i7/4C2FCwff1BSCT
# JWZO1Wcr2MteJE6UxGV+ihA8nN51YPKD2dYGoewrXvRzC/1HoUeSvlZf0mf9GHEt
# vvkbJVRRo6PBf0md5t87Vb1mM/fIp1eypyaxmXkgpcBwuylsOq2kSVOJ5wBPoaEs
# sJkeMcKnEuuu++UKdDHlS0DtsYjN0QnOucvTdSsdvhzKOSjJF3XVqr9f2C945LXT
# 5rxKIHUIEDBcNYU6BKDDH6rfpKOOCSilAgMBAAGjggFGMIIBQjAOBgNVHQ8BAf8E
# BAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFB6C3w7XjLPXAjSDDtqr
# rWW5r7jsMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMA8GA1UdEwEB/wQFMAMB
# Af8wHwYDVR0jBBgwFoAU1fZWy4/oolxiaNE9lJBb186aGMQwVgYDVR0fBE8wTTBL
# oEmgR4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMv
# TWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggr
# BgEFBQcwAoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNS
# b29DZXJBdXRfMjAxMC0wNi0yMy5jcnQwDQYJKoZIhvcNAQEMBQADggIBAENf+N8/
# u+mUjDtc9btoA52RBc0XVDSBMQBMqxu56hXHBwuctUWs1XBqDDMIFCHu9c6Y/UF+
# TN8EIgjnujApKYmHP4f4EM3ARSmlzrpF5ozOJx0BA5FUv1jmpdf/2ZbqpvCxlxv/
# G1R4KjrSmmqPHzs6igw3b7RTbj7BxIS8fOIkwYWQhB2fLjlg+3HSrDGPFIhpIJWV
# amMIR7a72OGonjdf45rspwqIHuynZU4avy9ruB/Rhhbwm+fMb8BMecIaTmkohx/E
# ZZ5GNWcN6oTYW3G2BM3B3YznWkl9t4shP60fMue+2ksdHGWSE8EVTdSmGUdj0jrU
# c46lGVFJISF3/MxcxnlNeP1Khyr+ZzT4Ets/I7mufpaLnLalzMR2zIuhGOAWWswe
# sbjtFzkVUFgDR2SW903I0XKlbPEA6q8epHGJ9roxh85nsEKcBNUw4Scp68KCqSpF
# BaKiyV1skd+l8U50WNePMb9Bzz0OfASal8v5sQG+DW01kN+I+RKUIbM5I50wJjiH
# ymQFNDsbobFx9I95mCEEPU7fUZ3VT/HOUVbkmX7ltIC/eQAu5GO8fu+ceETMybvb
# oxUM4dYNC+PzooUxfmC0DuKRwB21bX9+acuIBkxIm4Ed3O19w1VLoA7UNOUuJ7z6
# NQ2W/+q7cnfOPl2QVL4qlgCblUT2vmQpllV3MIIHIDCCBQigAwIBAgITMwAAAIbn
# cZS5Tf8J+wAAAAAAhjANBgkqhkiG9w0BAQwFADBfMQswCQYDVQQGEwJVUzEeMBwG
# A1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTAwLgYDVQQDEydNaWNyb3NvZnQg
# V2luZG93cyBDb2RlIFNpZ25pbmcgUENBIDIwMjQwHhcNMjUwNTA4MTgyNDUzWhcN
# MjYwNTA2MTgyNDUzWjB0MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMR4wHAYDVQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQCkffWVt1N9gK2cozqGbR1wGCUvR9RBe8CyPJxR
# BCdNuD07Q43NQPiX0rSkJoyYurzxnc82BJmk0UKdH4B929bxJkK1pjAN+Wn9Jedb
# ITMAaIP1Wmw60SC0Hs6wXKeRM9nqOTbkBhp2wKVxkDQppnfqZROMn6EtLcgEfpTU
# Qk/IHxaIbxqbDnRLY31LUareoRlUf0tuLNf42ZAgUDyEtVOjri5pe4AVyPsrPuIh
# EHLXzKrpnuqrK6nSfTgsr7b7fwL4xqd13rhG1DS30LK6JfCAVw7HPbD/7m/RQOhp
# +ZMPhlZZfLWvnqu97cmp3j3+NKRFzYCF6U3VNutdON/AhLn4NN0+Sz6Mm6eixBcS
# ARuYwV1K62vUzyTiK252LQg7XSqwUDcdCTXru+2bt9aH8kosQWgDr8i2Xc9jyZUj
# jEwMlUKzxunqz7tQ80OzTSAgz2ykW0o16CTTEV4/Pb/hLWFlPhXph+jJx+MkhT38
# yr3f2uPwVCuP9eMZSuEafKZc+TOX1Gsr2BFIwxdP8ICJTH7MpvwAv4G17so84xNG
# GvRq7TpS9Ly6ubUJ409709Jnos43dD7fXnE5XmRoILvFDUCo3tnt9Zshnx7wfAsg
# +8phXHOd6YiYgTG773s1HGPvMlwZCT+HPFX7W5ziIdNQC22in37/qrQ7wdKg4UMm
# ZIY4wwIDAQABo4IBvjCCAbowDgYDVR0PAQH/BAQDAgeAMB8GA1UdJQQYMBYGCisG
# AQQBgjc9BgEGCCsGAQUFBwMDMAwGA1UdEwEB/wQCMAAwHQYDVR0OBBYEFFCjGj3V
# OqaQ27YLTqVOylBylCiAMFQGA1UdEQRNMEukSTBHMS0wKwYDVQQLEyRNaWNyb3Nv
# ZnQgSXJlbGFuZCBPcGVyYXRpb25zIExpbWl0ZWQxFjAUBgNVBAUTDTIzMDg2NSs1
# MDQ1OTEwHwYDVR0jBBgwFoAUHoLfDteMs9cCNIMO2qutZbmvuOwwagYDVR0fBGMw
# YTBfoF2gW4ZZaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWlj
# cm9zb2Z0JTIwV2luZG93cyUyMENvZGUlMjBTaWduaW5nJTIwUENBJTIwMjAyNC5j
# cmwwdwYIKwYBBQUHAQEEazBpMGcGCCsGAQUFBzAChltodHRwOi8vd3d3Lm1pY3Jv
# c29mdC5jb20vcGtpb3BzL2NlcnRzL01pY3Jvc29mdCUyMFdpbmRvd3MlMjBDb2Rl
# JTIwU2lnbmluZyUyMFBDQSUyMDIwMjQuY3J0MA0GCSqGSIb3DQEBDAUAA4ICAQAM
# OWRf42CxONGV43y2AkPRXmTlBZytzMdgL8Aa6W9w+1UNxP8sSs6YlC9ADqTlehqh
# DVKZjTzRj/7ENx+Lzvu+uc4sVvYfRb4iNYwsj798zooF008RAOVvJ1Zz4hnL13mk
# yW9Pe3OA0Wm824FlnhgrV1N3OHij09S0x4xXv4BGVLL5OVxkiH8+kKquKApvPDod
# c+ZDfzocEwK0ORABs12RXDuoePES8XBRZ/WUCN/BPle7ZGMgYcfPWQ+qREn64GcL
# HvufdK5mYmQlKnazA2CIzvwdTyPwfqTTBeUk0MkHtiZfcE98xXVYlO9J3A7q6K7w
# xSuDrEGheVwRoEbhYOfLp5xN9cE11LLXbLDF2j8MDBTjY/sigH9lESII89vAQmhN
# x2z3/6tvou017ex3pFVb2qEia3OMv/+6Pb3UXbFf0EYshPjTkYIChpYSgZ6ctKZZ
# x7C6PFcztRon+JKsyDbAjjmjNV0VB94wXz5he0VV4Tq7NUQs5SgfCqZqxoXGLuTY
# X9gfp1tMStsJqb/yYPpmKM476KpKVstwoz+vwY+lwfPhcRhpxJvjXV0tt4x57ThO
# /TctTdV5SzuaE8ttOfUWzLCbcveKJ3F/6cBdO6nIMj4W8fp4S2xu45DToWeLb35+
# 608fp/yrVLJw+MXwtop7qDwm+6qb/MYQoy8Tk8XvojGCGwYwghsCAgEBMHYwXzEL
# MAkGA1UEBhMCVVMxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEwMC4G
# A1UEAxMnTWljcm9zb2Z0IFdpbmRvd3MgQ29kZSBTaWduaW5nIFBDQSAyMDI0AhMz
# AAAAhudxlLlN/wn7AAAAAACGMA0GCWCGSAFlAwQCAQUAoIGuMBkGCSqGSIb3DQEJ
# AzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8G
# CSqGSIb3DQEJBDEiBCANSiN/A9m3PfXxWvXOr5Y6vf4QWtWaOKTWzL+fiYGStDBC
# BgorBgEEAYI3AgEMMTQwMqAUgBIATQBpAGMAcgBvAHMAbwBmAHShGoAYaHR0cDov
# L3d3dy5taWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUABIICAAmc/9oZmmaScseq
# Y2rUbVS2WuoyvaKPx0Q4dhTXdKvhD/0uLFkDFlCQgSmLZwkz7/ffqwunSYFaUmzh
# Usu8T9hY7VKtFAFz7fy/rukMT7jrmLqbH91n8Tx7O6DAgs/g3AL5ZME6jOXWmznb
# /JkDVRRskMT1s2IiQGlRc6w+YJX32NNEwnKLRIYZsCOTsUKccaCGFvxa7XuXfjgD
# U1LeOpyaCkoZaPO2x2M8TSAuuvPXeRC25+o3MdATuYO5FSzGNN8DXTVA+seNX0YV
# 6mCO5ykIyRI5uv9dWQWS08tPQrNGzu8LHfnBakMap12/yAAAWq8w/+FhVWaX8AXU
# 09zNbBPkrpiIqQ1NE325eWAhUNN8PWI6RKpGppyDmrirV2cufoL/fan8do382858
# 65o6dBf86dE/Pt9H3wu7k+obSPTqOTpQFzbZI6CnMnp00UVch8JmKVuZz/tEmg0b
# J1I/YE6zDEb5jTO/7aFUJvpS9E+QWid+8N5XpMRvtLySXbMEyol0QKg/J1fEKRaj
# PrBHT7OZk+d5AHKNwL8+q3Nxw3bsANvlSTCC5gFW3Uhmpre2QhYvgALJKEdcgivH
# Gelg15gxKGllRAlG2kgjv1yLWfzFvxDdXaZT1h+KJmcyqZHzWsXbMFYVLsFsyAk1
# z36FCfzkOIkVe0MfqVWEaE0ATU2joYIXsDCCF6wGCisGAQQBgjcDAwExghecMIIX
# mAYJKoZIhvcNAQcCoIIXiTCCF4UCAQMxDzANBglghkgBZQMEAgEFADCCAVoGCyqG
# SIb3DQEJEAEEoIIBSQSCAUUwggFBAgEBBgorBgEEAYRZCgMBMDEwDQYJYIZIAWUD
# BAIBBQAEIKsESOethaxTYSrk5P3nSZtGNVju7Xc00KLfj9BNCvR0AgZok+tNNlUY
# EzIwMjUwODE0MTgzODM4LjczOVowBIACAfSggdmkgdYwgdMxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVs
# YW5kIE9wZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNO
# OjM2MDUtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
# ZXJ2aWNloIIR/jCCBygwggUQoAMCAQICEzMAAAH3WCB1BMr7wvQAAQAAAfcwDQYJ
# KoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
# EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
# bjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwHhcNMjQw
# NzI1MTgzMTA2WhcNMjUxMDIyMTgzMTA2WjCB0zELMAkGA1UEBhMCVVMxEzARBgNV
# BAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
# c29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UECxMkTWljcm9zb2Z0IElyZWxhbmQgT3Bl
# cmF0aW9ucyBMaW1pdGVkMScwJQYDVQQLEx5uU2hpZWxkIFRTUyBFU046MzYwNS0w
# NUUwLUQ5NDcxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0YW1wIFNlcnZpY2Uw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDQ50dME2ibr+5cpoQo/2s8
# hORPpDEXXW2PMHQ2TVvIOk+sVMeFreHHBJ1NyvxjRreToRCXCrLpE7PjZ7RHl4Nb
# 50KhBqmKkMgVQ5ineg26rBf/F6kBMSRjXszJcXHqtBbY1xZQlbdCjYC4nQc61uVK
# ki1Bk8aYecaqS38MHjkXDGTpWhK/E1xAqEoROS7Ou3xToNFxxCbUV2GY8qAPOBx8
# M8zmj4afNuIy7rLTr0DgQeYsyaR5xKRW8GZxnxWfMUdMOQYt2mcNXkVeNU5sCBtI
# zRyephIZ9GntUYcFGrKixy9HhtxD4JX2kONsnpLmtmfW4DyFGGPT0ezfcdF6+3ih
# YBVgYi2ASwb4GsJhumBYwMQhWcCA9kSI8BojzAEZ6YTh94SS7PtMDCCREFxTMuBD
# i68+pEPUD4mS3br6kOpZhKfQwDyPTNpxCT2r8C9yI9cP0i3Z7P6aoTOAVFGwkYu1
# x/0eSy8rwmx3ojnMVKGWqLlunN/Vjg06I06HlDBbWki8DmKuVqXuoWGQB555mqai
# nz643FlfEUJAbdHezmldbz0WIKH2uZetWo4LCBxcUglABCSWUqwj5Qmoar2uZEAE
# nPmUcpMViYXBwznYpZaM3HfPqh3DPaH6zFrF7BOh70aq0PHf9pT7Ko1FwHzDS1Jd
# R/7KU3i6TnEcSkunH5k02wIDAQABo4IBSTCCAUUwHQYDVR0OBBYEFN9GpDM/eb09
# la4t/Wnz+Z4V+SaYMB8GA1UdIwQYMBaAFJ+nFV0AXmJdg/Tl0mWnG1M1GelyMF8G
# A1UdHwRYMFYwVKBSoFCGTmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMv
# Y3JsL01pY3Jvc29mdCUyMFRpbWUtU3RhbXAlMjBQQ0ElMjAyMDEwKDEpLmNybDBs
# BggrBgEFBQcBAQRgMF4wXAYIKwYBBQUHMAKGUGh0dHA6Ly93d3cubWljcm9zb2Z0
# LmNvbS9wa2lvcHMvY2VydHMvTWljcm9zb2Z0JTIwVGltZS1TdGFtcCUyMFBDQSUy
# MDIwMTAoMSkuY3J0MAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUH
# AwgwDgYDVR0PAQH/BAQDAgeAMA0GCSqGSIb3DQEBCwUAA4ICAQA3RqNp8gt4vpJA
# gwgwBczVB3rFqhyLaY6ulHy8pbLJOwvdvzcDtcYuIBtDFOuqde9VZZ42y3lhAPyx
# o75ROA4sl1N19QAOEtegr5GXCN+d2KYglP0wf21RhcvMlcqFkzT2i4/A2yufxg4s
# il0CLlM/I3wKXXU4ZlKU/2vwme+iZbTQCgng+X2uWDQbmVxCScBeodr2dB1anVnF
# eo137QmwqaVHy1wA1ffcKUz02doKUkTEtAeIp4dRRa2rIsyXrlNbrBEzteUXtj49
# OcLx241afi4ueD4439nf0Y7qoGPsgRnGirijdq8SH1trjdRTpODNVloGbxVoDTBL
# BR7+mqlM5gVY3rZcveCX8kLanN8g/E/rpd9EsjFp+MFVebwpUOfZwwv0i9ErTaz3
# jVjn5FHiBIA6EuJBDoDTdU1G6n6ykxrST5dM8CL7ZowfnFrVmNv8ry71/0zTlTT9
# tQwlckM/77KxakltVEOIcbuzNpxr6vceJQ+NAnJCXY2I5xhMZX8NwussIErbMbnT
# cUZvTg3kp/XReADAVpeWh3kH14qH3k+dcrHYs0GAvAbzlqeWGEbHEFDmYWwkaQGf
# Q9k+0DNnJ+v3qrHOmnakf0MklyMoIOsyZnOJdrOlrlVU3foI7WQNTgAGRJhNc4zx
# GYle5CbuZQXdtaaP6GMAlvinPqFPlTCCB3EwggVZoAMCAQICEzMAAAAVxedrngKb
# SZkAAAAAABUwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
# ZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29mdCBSb290IENlcnRpZmlj
# YXRlIEF1dGhvcml0eSAyMDEwMB4XDTIxMDkzMDE4MjIyNVoXDTMwMDkzMDE4MzIy
# NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
# B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UE
# AxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQDk4aZM57RyIQt5osvXJHm9DtWC0/3unAcH0qlsTnXI
# yjVX9gF/bErg4r25PhdgM/9cT8dm95VTcVrifkpa/rg2Z4VGIwy1jRPPdzLAEBjo
# YH1qUoNEt6aORmsHFPPFdvWGUNzBRMhxXFExN6AKOG6N7dcP2CZTfDlhAnrEqv1y
# aa8dq6z2Nr41JmTamDu6GnszrYBbfowQHJ1S/rboYiXcag/PXfT+jlPP1uyFVk3v
# 3byNpOORj7I5LFGc6XBpDco2LXCOMcg1KL3jtIckw+DJj361VI/c+gVVmG1oO5pG
# ve2krnopN6zL64NF50ZuyjLVwIYwXE8s4mKyzbnijYjklqwBSru+cakXW2dg3viS
# kR4dPf0gz3N9QZpGdc3EXzTdEonW/aUgfX782Z5F37ZyL9t9X4C626p+Nuw2TPYr
# bqgSUei/BQOj0XOmTTd0lBw0gg/wEPK3Rxjtp+iZfD9M269ewvPV2HM9Q07BMzlM
# jgK8QmguEOqEUUbi0b1qGFphAXPKZ6Je1yh2AuIzGHLXpyDwwvoSCtdjbwzJNmSL
# W6CmgyFdXzB0kZSU2LlQ+QuJYfM2BjUYhEfb3BvR/bLUHMVr9lxSUV0S2yW6r1AF
# emzFER1y7435UsSFF5PAPBXbGjfHCBUYP3irRbb1Hode2o+eFnJpxq57t7c+auIu
# rQIDAQABo4IB3TCCAdkwEgYJKwYBBAGCNxUBBAUCAwEAATAjBgkrBgEEAYI3FQIE
# FgQUKqdS/mTEmr6CkTxGNSnPEP8vBO4wHQYDVR0OBBYEFJ+nFV0AXmJdg/Tl0mWn
# G1M1GelyMFwGA1UdIARVMFMwUQYMKwYBBAGCN0yDfQEBMEEwPwYIKwYBBQUHAgEW
# M2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvRG9jcy9SZXBvc2l0b3J5
# Lmh0bTATBgNVHSUEDDAKBggrBgEFBQcDCDAZBgkrBgEEAYI3FAIEDB4KAFMAdQBi
# AEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV
# 9lbLj+iiXGJo0T2UkFvXzpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
# Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAx
# MC0wNi0yMy5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8v
# d3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2
# LTIzLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAnVV9/Cqt4SwfZwExJFvhnnJL/Klv
# 6lwUtj5OR2R4sQaTlz0xM7U518JxNj/aZGx80HU5bbsPMeTCj/ts0aGUGCLu6WZn
# OlNN3Zi6th542DYunKmCVgADsAW+iehp4LoJ7nvfam++Kctu2D9IdQHZGN5tggz1
# bSNU5HhTdSRXud2f8449xvNo32X2pFaq95W2KFUn0CS9QKC/GbYSEhFdPSfgQJY4
# rPf5KYnDvBewVIVCs/wMnosZiefwC2qBwoEZQhlSdYo2wh3DYXMuLGt7bj8sCXgU
# 6ZGyqVvfSaN0DLzskYDSPeZKPmY7T7uG+jIa2Zb0j/aRAfbOxnT99kxybxCrdTDF
# NLB62FD+CljdQDzHVG2dY3RILLFORy3BFARxv2T5JL5zbcqOCb2zAVdJVGTZc9d/
# HltEAY5aGZFrDZ+kKNxnGSgkujhLmm77IVRrakURR6nxt67I6IleT53S0Ex2tVdU
# CbFpAUR+fKFhbHP+CrvsQWY9af3LwUFJfn6Tvsv4O+S3Fb+0zj6lMVGEvL8CwYKi
# excdFYmNcP7ntdAoGokLjzbaukz5m/8K6TT4JDVnK+ANuOaMmdbhIurwJ0I9JZTm
# dHRbatGePu1+oDEzfbzL6Xu/OHBE0ZDxyKs6ijoIYn/ZcGNTTY3ugm2lBRDBcQZq
# ELQdVTNYs6FwZvKhggNZMIICQQIBATCCAQGhgdmkgdYwgdMxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQK
# ExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTArBgNVBAsTJE1pY3Jvc29mdCBJcmVs
# YW5kIE9wZXJhdGlvbnMgTGltaXRlZDEnMCUGA1UECxMeblNoaWVsZCBUU1MgRVNO
# OjM2MDUtMDVFMC1EOTQ3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBT
# ZXJ2aWNloiMKAQEwBwYFKw4DAhoDFQBvbwoMb/Fds0GOYzv+erDduCsQ5qCBgzCB
# gKR+MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
# EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNV
# BAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMA0GCSqGSIb3DQEBCwUA
# AgUA7EhMVjAiGA8yMDI1MDgxNDExNTE1MFoYDzIwMjUwODE1MTE1MTUwWjB3MD0G
# CisGAQQBhFkKBAExLzAtMAoCBQDsSExWAgEAMAoCAQACAi+sAgH/MAcCAQACAhJm
# MAoCBQDsSZ3WAgEAMDYGCisGAQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwKgCjAI
# AgEAAgMHoSChCjAIAgEAAgMBhqAwDQYJKoZIhvcNAQELBQADggEBAJ32FpF39Tc6
# DDVUKLQaFlma6oLXADjJmN3XLtZITGARUYMpgWywptqYoln+JQ1SdI/y2Aw53QX6
# 1jnJk2r3Bfc4fF1lgvcvZGhc0+CtFySMZ9M4idbCfw+C7tj4fT4av5rVokW8RIP3
# gAVIlEI1wdjZD8U8tZYJnfvkxlJ7ooyauQWjB0AEmzHOlkQQmyz4LRPso0MUK63s
# pzFJVhZfESDOh6+K+JSnuWnHU6DZLNjx0lZL680s7ZH39RjMqrPjqfdPhcn/HezO
# 3yVxw0sXwOiD4usxdiwbRQW7aC/RMptrhvOmk7rvb1WiPDMKX8qxtTiqfe5Tblpn
# hTJwYI8EG94xggQNMIIECQIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
# MjAxMAITMwAAAfdYIHUEyvvC9AABAAAB9zANBglghkgBZQMEAgEFAKCCAUowGgYJ
# KoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCA/W4ainHj4
# sYUls4+drgOpQDGwotlnuD/EBe+pdNnFizCB+gYLKoZIhvcNAQkQAi8xgeowgecw
# geQwgb0EICHamNprdxrR5xi6G7rS5gc/8gqc9t51tVAnlKggflniMIGYMIGApH4w
# fDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
# ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMd
# TWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAAH3WCB1BMr7wvQAAQAA
# AfcwIgQgkUSyh0HBudEKupDdoGjEt+KpBZvdAV0qwEkyaTEnLvgwDQYJKoZIhvcN
# AQELBQAEggIAQ7rM6A/B+lBJFAFh/KxY7GF2nQoIEa1FDGAOo7/I/Y3/PXGGA1Ii
# u5v8Dd9NFO4yhStrVLPkfUmG+AmB4X5ntiGAs9Hrz8ADO7h93R00yYG/+A3o6o0P
# +wcNeu8w8xtRV8LiVcKaT69+cZhYca6E6kv+o4VZmm9vnjPH55ml5lI7rO78BcKh
# Z5NBaDJKczd0XRw0MGDkAGEiLFgm0jfiJYSAvG42bMZu2T6pY+gQ+EEASXwubmRz
# SDpI6Js14/JQV2dX38j72u9PK6SQ1Y9hqAvvuKznxwSWAvZvJy9cANckqycN0EJc
# wwimshrbprYT+V4XXI9UsmUnuAqKlVs5SxqHr5NF0SGbhNTB2pIUXh72gkmzEMmX
# i0O1yE0nPhvt/NY2K0OnBC76Z35v9yDnyQeDuhIvDmj/tBImVx6rYD7cpjq5sXQg
# Lac9ek21rJhGuqbXToLYQemhGWQv3lsSWeQvt6mAoAxQye2b604uERjxfT7ZsN6S
# BsJGOJRWuIO7k879hX+FbcGKNO24x7Z0GX5N6BXqN4R3vMwpX4fMCt4ceXCtcWNj
# t+cZeYz1ussOlO3FfxYRIh5DXr1YkoJYLd3ZyiEBKT6tNfy1ELchY1gIilRj6eo5
# RmrrMJtVhNPIgiIsIdP8IfGmcSK3OsRhBHxF2eTSOGQSJTSzcLYpLjU=
# SIG # End signature block
