/********************************************************************
    Copyright (c) 2013-2015 - Mogara

    This file is part of QSanguosha-Hegemony.

    This game is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 3.0
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    See the LICENSE file for more details.

    Mogara
    *********************************************************************/

#include "uiutils.h"

#include <QPixmap>
#include <QImage>
#include <QFile>
#include <QDir>
#include <QDesktopServices>
#include <QMutex>
#include <ft2build.h>

#define NEW_FONT_PIXEL(x, y, channel) (newImage[((y) * cols + (x)) * 4 + channel])
#define FONT_PIXEL(x, y) (bitmap.buffer[(y) * rowStep + (x)])

void QSanUiUtils::paintShadow(QPainter *painter, const QImage &image, QColor shadowColor, int radius, double decade, const int x, const int y)
{
    const uchar *oldImage = image.bits();
    int cols = image.width();
    int rows = image.height();
    int alpha = shadowColor.alpha();
    uchar *newImage = new uchar[cols * rows * 4];
#define _NEW_PIXEL_CHANNEL(x, y, channel) (newImage[(y * cols + x) * 4 + channel])
#define _NEW_PIXEL(x, y) _NEW_PIXEL_CHANNEL(x, y, 3)
#define _OLD_PIXEL(x, y) (oldImage[(y * cols + x) * 4 + 3])
    for (int Y = 0; Y < rows; Y++) {
        for (int X = 0; X < cols; X++) {
            _NEW_PIXEL_CHANNEL(X, Y, 0) = shadowColor.blue();
            _NEW_PIXEL_CHANNEL(X, Y, 1) = shadowColor.green();
            _NEW_PIXEL_CHANNEL(X, Y, 2) = shadowColor.red();
            _NEW_PIXEL_CHANNEL(X, Y, 3) = 0;
        }
    }

    for (int Y = 0; Y < rows; Y++) {
        for (int X = 0; X < cols; X++) {
            uchar oldVal = _OLD_PIXEL(X, Y);
            if (oldVal == 0) continue;
            for (int dy = -radius; dy <= radius; dy++) {
                for (int dx = -radius; dx <= radius; dx++) {
                    int wx = X + dx;
                    int wy = Y + dy;
                    int dist = dx * dx + dy * dy;
                    if (wx < 0 || wy < 0 || wx >= cols || wy >= rows) continue;
                    if (dx * dx + dy * dy > radius * radius) continue;
                    int newVal = alpha - decade * dist;
                    Q_ASSERT((wy * cols + wx) * 4 < cols * rows * 4);
                    _NEW_PIXEL(wx, wy) = (uchar)qMax((int)_NEW_PIXEL(wx, wy), newVal);
                }
            }
        }
    }
#undef _NEW_PIXEL_CHANNEL
#undef _NEW_PIXEL
#undef _OLD_PIXEL
    QImage result(newImage, cols, rows, QImage::Format_ARGB32);
    painter->drawImage(x, y, result);
    delete[] newImage;
    newImage = NULL;
}

void QSanUiUtils::paintShadow(QPainter *painter, const QImage &image, QColor shadowColor, int radius, double decade, const QRect boundingBox)
{
    QPoint TL = boundingBox.topLeft();
    paintShadow(painter, image, shadowColor, radius, decade, TL.x(), TL.y());
}

void QSanUiUtils::makeGray(QPixmap &pixmap)
{
    QImage img = pixmap.toImage();
    for (int i = 0; i < img.width(); i++) {
        for (int j = 0; j < img.height(); j++) {
            QColor color = QColor::fromRgba(img.pixel(i, j));
            int gray = qGray(color.rgb());
            img.setPixel(i, j, qRgba(gray, gray, gray, color.alpha()));
        }
    }
    pixmap = QPixmap::fromImage(img);
}

#include FT_FREETYPE_H
#include FT_BITMAP_H
#include FT_OUTLINE_H

static FT_Library  _ftlib;
static bool _ftLibInitialized = false;

bool QSanUiUtils::QSanFreeTypeFont::init()
{
    FT_Error error = FT_Init_FreeType(&_ftlib);
    if (error) {
        qWarning("error loading library");
        return false;
    } else {
        _ftLibInitialized = true;
        return true;
    }
}

void QSanUiUtils::QSanFreeTypeFont::quit()
{
    if (_ftLibInitialized) {
        FT_Done_FreeType(_ftlib);
    }
}

QString QSanUiUtils::QSanFreeTypeFont::resolveFont(const QString &fontName)
{
    QString result;
    if (QFile::exists(fontName))
        result = fontName;
    else {
        QStringList dirsToResolve;
        QStringList extsToTry;
        QString sysfolder = QStandardPaths::writableLocation(QStandardPaths::FontsLocation);
        dirsToResolve.push_back(sysfolder);
        dirsToResolve.push_back(QDir::currentPath());
        dirsToResolve.push_back("./font");
        extsToTry.push_back("ttf");
        extsToTry.push_back("ttc");
        foreach (const QString &sdir, dirsToResolve) {
            foreach (const QString &ext, extsToTry) {
                QDir dir(sdir);
                QString filePath = dir.filePath(QString("%1.%2").arg(fontName).arg(ext));
                if (QFile::exists(filePath)) {
                    result = filePath;
                    break;
                }
            }
        }
    }
    return result;
}

static QSanUiUtils::QSanFreeTypeFont::QSanFont qsanfont(FT_Face face)
{
    return reinterpret_cast<QSanUiUtils::QSanFreeTypeFont::QSanFont>(face);
}

static FT_Face qsanfont(QSanUiUtils::QSanFreeTypeFont::QSanFont font)
{
    return reinterpret_cast<FT_Face>(font);
}

QSanUiUtils::QSanFreeTypeFont::QSanFont QSanUiUtils::QSanFreeTypeFont::loadFont(const QString &fontName)
{
    if (!_ftLibInitialized && !init())
        return NULL;
    FT_Face face = NULL;
    QString resolvedPath = resolveFont(fontName);
    QByteArray arr = resolvedPath.toLatin1();
    const char *fontPath = arr.constData();
    FT_Error error = FT_New_Face(_ftlib, fontPath, 0, &face);
    if (error == FT_Err_Unknown_File_Format)
        qWarning("Unsupported font format: %s.", fontPath);
    else if (error)
        qWarning("Cannot open font file: %s.", fontPath);
    else
        return qsanfont(face);
    return 0;
}

static QMutex _paintTextMutex;

bool QSanUiUtils::QSanFreeTypeFont::paintQString(QPainter *painter, QString text, QSanUiUtils::QSanFreeTypeFont::QSanFont font, QColor color,
    QSize &fontSize, int spacing, int weight, QRect boundingBox,
    Qt::Orientation orient, Qt::Alignment align)
{
    if (!_ftLibInitialized || font == NULL || painter == NULL || text.isNull())
        return false;

    QVector<uint> charcodes = text.toUcs4();
    int len = charcodes.size();
    int pixelsAdded = (weight >> 6) * 2;
    int xstep, ystep;
    Qt::Alignment hAlign = align & Qt::AlignHorizontal_Mask;
    Qt::Alignment vAlign = align & Qt::AlignVertical_Mask;
    if (hAlign == 0) hAlign = Qt::AlignHCenter;
    if (vAlign == 0) vAlign = Qt::AlignVCenter;

    QPoint topLeft = boundingBox.topLeft();
    boundingBox.moveTopLeft(QPoint(0, 0));

    if (orient == Qt::Vertical) {
        xstep = 0;
        if (fontSize.width() > boundingBox.width())
            fontSize.setWidth(boundingBox.width());
        ystep = spacing + fontSize.height();
        // AlignJustify means the text should fill out the whole rect space
        // so we increase the step
        if (align & Qt::AlignJustify) {
            ystep = boundingBox.height() / len;
            if (fontSize.height() + spacing > ystep)
                fontSize.setHeight(ystep - spacing);
        }
    } else {
        ystep = 0;
        if (fontSize.height() > boundingBox.height())
            fontSize.setHeight(boundingBox.height());
        xstep = spacing + fontSize.width();
        // AlignJustifx means the text should fill out the whole rect space
        // so we increase the step
        if (align & Qt::AlignJustify) {
            xstep = boundingBox.width() / len;
            if (fontSize.width() + spacing > xstep)
                fontSize.setWidth(xstep - spacing);
        }
    }
    if (fontSize.width() <= 0 || fontSize.height() <= 0) return false;
    // we allocate larger area than necessary in case we need bold font
    int rows = boundingBox.height() + pixelsAdded + 3;
    int cols = boundingBox.width() + pixelsAdded + 3;
    int imageSize = rows * cols;
    int imageBytes = imageSize * 4;
    uchar *newImage = new uchar[imageBytes];

    for (int i = 0; i < imageBytes;) {
        newImage[i++] = color.blue();
        newImage[i++] = color.green();
        newImage[i++] = color.red();
        newImage[i++] = 0;
    }

#if (defined(_NEW_PIXEL) || defined(_FONT_PIXEL))
#error("macro _NEW_PIXEL or _FONT_PIXEL already in use")
#endif
#define _NEW_PIXEL(x, y, channel) (newImage[((y) * cols + (x)) * 4 + channel])
#define _FONT_PIXEL(x, y) (bitmap.buffer[(y) * rowStep + (x)])
    // we do not do kerning for vertical layout for now
    bool useKerning = ((orient == Qt::Horizontal) && !(align & Qt::AlignJustify));

    _paintTextMutex.lock();
    FT_Face face = qsanfont(font);
    FT_GlyphSlot slot = face->glyph;
    FT_Error error;
    error = FT_Set_Pixel_Sizes(face, fontSize.width(), fontSize.height());
    FT_UInt previous = 0;
    int currentX = 0;
    int currentY = 0;
    for (int i = 0; i < len; i++) {
        FT_Vector  delta;
        FT_UInt glyph_index = FT_Get_Char_Index(face, charcodes[i]);
        error = FT_Load_Glyph(face, glyph_index, FT_LOAD_DEFAULT | FT_LOAD_NO_BITMAP);
        if (error) continue;

        if (useKerning && previous && glyph_index) {
            error = FT_Get_Kerning(face, previous, glyph_index, FT_KERNING_DEFAULT, &delta);
            currentX += delta.x >> 6;
        }
        previous = glyph_index;

        FT_Bitmap bitmap;
        if (weight == 0) {
            FT_Load_Glyph(face, glyph_index, FT_LOAD_RENDER);
        } else {
            FT_Outline_Embolden(&slot->outline, weight);
            FT_Render_Glyph(face->glyph, FT_RENDER_MODE_NORMAL);
        }

        bitmap = slot->bitmap;
        Q_ASSERT(bitmap.pitch == bitmap.width || bitmap.pitch == (bitmap.width - 1) / 8 + 1);
        bool mono = true;
        if (bitmap.pitch == bitmap.width)
            mono = false;

        int fontRows = bitmap.rows;
        int fontCols = bitmap.width;
        int rowStep = bitmap.pitch;
        int tmpYOffset = fontSize.height() - slot->bitmap_top;
        currentY = currentY + tmpYOffset;

        if (orient == Qt::Vertical)
            currentX = (fontSize.width() - bitmap.width) / 2;

        // now paint the bitmap to the new region;
        if (currentX < 0) currentX = 0;
        if (currentY < 0) currentY = 0;
        for (int y = 0; y < fontRows; y++) {
            if (currentY + y >= rows)
                break;
            uchar *fontPtr = &_FONT_PIXEL(0, y);
            uchar *imagePtr = &_NEW_PIXEL(currentX, currentY + y, 3);
            int fontClippedCols;
            if (fontCols + currentX < cols)
                fontClippedCols = fontCols;
            else
                fontClippedCols = cols - 1 - currentX;
            if (!mono) {
                for (int x = 0; x < fontClippedCols; x++) {
                    *imagePtr = *fontPtr;
                    fontPtr++;
                    imagePtr += 4;
                }
            } else {
                int mask = 0x80;
                for (int x = 0; x < fontClippedCols; x++) {
                    if (*fontPtr & mask)
                        *imagePtr = 255;
                    mask = mask >> 1;
                    if (mask == 0) {
                        fontPtr++;
                        mask = 0x80;
                    }
                    imagePtr += 4;
                }
            }
        }
        if (useKerning)
            currentX += (slot->advance.x >> 6) + spacing;
        else
            currentX += xstep;
        currentY = currentY - tmpYOffset + ystep;
    }
#undef _NEW_PIXEL
#undef _FONT_PIXEL
    _paintTextMutex.unlock();

    int xstart = 0, ystart = 0;
    if (orient == Qt::Vertical) {
        if (hAlign & Qt::AlignLeft)
            xstart = spacing;
        else if (hAlign & Qt::AlignHCenter)
            xstart = (boundingBox.width() - fontSize.width()) / 2;
        else if (hAlign & Qt::AlignRight)
            xstart = boundingBox.right() - spacing - fontSize.width();
        else {
            xstart = 0;
            Q_ASSERT(false);
        }

        if (vAlign & Qt::AlignTop)
            ystart = spacing;
        else if (vAlign & Qt::AlignVCenter || align & Qt::AlignJustify)
            ystart = (boundingBox.height() - currentY) / 2;
        else if (vAlign & Qt::AlignBottom)
            ystart = boundingBox.height() - currentY - spacing;
        else {
            ystart = 0;
            Q_ASSERT(false);
        }
    } else {
        if (vAlign & Qt::AlignTop)
            ystart = spacing;
        else if (vAlign & Qt::AlignVCenter)
            ystart = (boundingBox.height() - fontSize.height()) / 2;
        else if (vAlign & Qt::AlignBottom)
            ystart = boundingBox.bottom() - spacing - fontSize.height();
        else {
            ystart = 0;
            Q_ASSERT(false);
        }

        if (hAlign & Qt::AlignLeft)
            xstart = spacing;
        else if (hAlign & Qt::AlignHCenter || align & Qt::AlignJustify)
            xstart = (boundingBox.width() - currentX) / 2;
        else if (hAlign & Qt::AlignRight)
            xstart = boundingBox.right() - currentX - spacing;
        else {
            xstart = 0;
            Q_ASSERT(false);
        }
    }
    if (xstart < 0) xstart = 0;
    if (ystart < 0) ystart = 0;
    QImage result(newImage, cols, rows, QImage::Format_ARGB32);
    painter->drawImage(topLeft.x() + xstart, topLeft.y() + ystart, result);
    delete[] newImage;
    newImage = NULL;
    return true;
}

bool QSanUiUtils::QSanFreeTypeFont::paintQStringMultiLine(QPainter *painter, QString text,
    int *font, QColor color,
    QSize &fontSize, int spacing, int weight, QRect boundingBox,
    Qt::Alignment align)
{
    if (!_ftLibInitialized || font == NULL || painter == NULL)
        return false;

    QVector<uint> charcodes = text.toUcs4();
    int len = charcodes.size();
    int charsPerLine = boundingBox.width() / fontSize.width();
    int numLines = (len - 1) / charsPerLine + 1;

    QPoint topLeft = boundingBox.topLeft();
    boundingBox.moveTopLeft(QPoint(0, 0));

    int xstep;
    // AlignJustifx means the text should fill out the whole rect space
    // so we increase the step
    if (align & Qt::AlignJustify)
    {
        xstep = boundingBox.width() / len;
        if (fontSize.width() + spacing > xstep)
        {
            fontSize.setWidth(xstep - spacing);
        }
    }
    else
    {
        xstep = spacing + fontSize.width();
    }

    if (fontSize.height() * numLines > boundingBox.height())
    {
        fontSize.setHeight(boundingBox.height() / numLines - spacing);
    }

    int ystep = fontSize.height() + spacing;

    if (fontSize.width() <= 0 || fontSize.height() <= 0)
    {
        return false;
    }

    // we allocate larger area than necessary in case we need bold font
    int pixelsAdded = (weight >> 6) * 2;
    int rows = boundingBox.height() + pixelsAdded + 3;
    int cols = boundingBox.width() + pixelsAdded + 3;
    int imageSize = rows * cols;
    int imageBytes = imageSize * 4;
    uchar *const newImage = new uchar[imageBytes];

    for (int i = 0; i < imageBytes;)
    {
        newImage[i++] = color.blue();
        newImage[i++] = color.green();
        newImage[i++] = color.red();
        newImage[i++] = 0;
    }

    _paintTextMutex.lock();

    // we do not do kerning for vertical layout for now
    bool useKerning = (!(align & Qt::AlignJustify));
    FT_UInt previous = 0;
    int currentX = 0;
    int currentY = 0;
    int maxX = 0;
    int maxY = 0;
    FT_Face face = (FT_Face)font;

    FT_GlyphSlot slot = face->glyph;
    FT_Error error = FT_Set_Pixel_Sizes(face, fontSize.width(), fontSize.height());
    for (int i = 0, j = 0; i < len; ++i, ++j)
    {
        if (QChar('\n').unicode() == charcodes[i])
        {
            currentY += ystep;
            currentX = 0;
            j = -1;

            continue;
        }

        int line = j / charsPerLine;
        int cursor = j % charsPerLine;
        // whenever we start a new line, reset X and increment Y
        if (cursor == 0 && line > 0)
        {
            currentY += ystep;
            currentX = 0;
            j = 0;
        }

        FT_Vector delta;
        FT_UInt glyph_index = FT_Get_Char_Index(face, charcodes[i]);
        error = FT_Load_Glyph(face, glyph_index, FT_LOAD_DEFAULT | FT_LOAD_NO_BITMAP);
        if (error)
        {
            continue;
        }

        if (useKerning && previous && glyph_index)
        {
            error = FT_Get_Kerning(face, previous, glyph_index,
                FT_KERNING_DEFAULT, &delta);
            currentX += delta.x >> 6;
        }
        previous = glyph_index;

        if (weight == 0)
        {
            FT_Load_Glyph(face, glyph_index, FT_LOAD_RENDER);
        }
        else
        {
            FT_Outline_Embolden(&slot->outline, weight);
            FT_Render_Glyph(face->glyph, FT_RENDER_MODE_NORMAL);
        }

        FT_Bitmap bitmap = slot->bitmap;
        int fontRows = bitmap.rows;
        int fontCols = bitmap.width;
        int rowStep = bitmap.pitch;
        int tmpYOffset = fontSize.height() - slot->bitmap_top;
        currentY = currentY + tmpYOffset;

        Q_ASSERT(bitmap.pitch == bitmap.width ||
            bitmap.pitch == (bitmap.width - 1) / 8 + 1);

        //@todo put it back
        bool mono = true;
        if (bitmap.pitch == bitmap.width)
        {
            mono = false;
        }

        // now paint the bitmap to the new region
        Q_ASSERT(currentX >= 0 && currentY >= 0);

        for (int y = 0; y < fontRows; ++y)
        {
            if (currentY + y >= rows)
            {
                break;
            }

            const uchar *fontPtr = &FONT_PIXEL(0, y);
            uchar *imagePtr = &NEW_FONT_PIXEL(currentX, currentY + y, 3);

            int fontClippedCols;
            if (fontCols + currentX < cols)
            {
                fontClippedCols = fontCols;
            }
            else
            {
                fontClippedCols = cols - 1 - currentX;
            }

            if (!mono)
            {
                for (int x = 0; x < fontClippedCols; ++x)
                {
                    *imagePtr = *fontPtr;
                    ++fontPtr;
                    imagePtr += 4;
                }
            }
            else
            {
                int mask = 0x80;
                for (int x = 0; x < fontClippedCols; ++x)
                {
                    if (*fontPtr & mask)
                    {
                        *imagePtr = 255;
                    }

                    mask = mask >> 1;
                    if (mask == 0)
                    {
                        ++fontPtr;
                        mask = 0x80;
                    }

                    imagePtr += 4;
                }
            }
        }

        if (useKerning)
        {
            currentX += (slot->metrics.width >> 6) + spacing;
        }
        else
        {
            currentX += xstep;
        }

        if (currentX > maxX)
        {
            maxX = currentX;
        }

        currentY -= tmpYOffset;
    }

    _paintTextMutex.unlock();

    maxY = currentY + ystep;

    Qt::Alignment hAlign = align & Qt::AlignHorizontal_Mask;
    Qt::Alignment vAlign = align & Qt::AlignVertical_Mask;

    int xstart, ystart;
    if (hAlign & Qt::AlignLeft)
    {
        xstart = spacing;
    }
    else if (hAlign & Qt::AlignHCenter || align & Qt::AlignJustify)
    {
        xstart = (boundingBox.width() - maxX) / 2;
    }
    else if (hAlign & Qt::AlignRight)
    {
        xstart = boundingBox.right() - maxX - spacing;
    }
    else
    {
        xstart = 0;
        Q_ASSERT(false);
    }

    if (vAlign & Qt::AlignTop)
    {
        ystart = spacing;
    }
    else if (vAlign & Qt::AlignVCenter)
    {
        ystart = (boundingBox.height() - maxY) / 2;
    }
    else if (vAlign & Qt::AlignBottom)
    {
        ystart = boundingBox.height() - maxY - spacing;
    }
    else
    {
        ystart = 0;
        Q_ASSERT(false);
    }

    if (xstart < 0)
    {
        xstart = 0;
    }
    if (ystart < 0)
    {
        ystart = 0;
    }

    QImage result(newImage, cols, rows, QImage::Format_ARGB32);
    painter->drawImage(topLeft.x() + xstart, topLeft.y() + ystart, result);

    delete[] newImage;

    return true;
}

