﻿using Kermalis.PokemonBattleEngine.Data;
using Kermalis.PokemonBattleEngine.Data.Utils;
using Kermalis.PokemonGameEngine.Core;
using Kermalis.PokemonGameEngine.Item;
using Kermalis.PokemonGameEngine.Pkmn.Pokedata;
using Kermalis.PokemonGameEngine.World;
using System;
using System.Collections.Generic;

namespace Kermalis.PokemonGameEngine.Pkmn
{
    internal static class Evolution
    {
        private static readonly Queue<(PartyPokemon, EvolutionData.EvoData, bool)> _pendingEvolutions = new(PkmnConstants.PartyCapacity);

        // Only level-up evolutions can be cancelled
        public static void AddPendingEvolution(PartyPokemon pkmn, EvolutionData.EvoData evo, bool canCancel)
        {
            _pendingEvolutions.Enqueue((pkmn, evo, canCancel));
        }
        public static bool TryGetNextPendingEvolution(out (PartyPokemon, EvolutionData.EvoData, bool) evo)
        {
            return _pendingEvolutions.TryDequeue(out evo);
        }

        private static bool HeldEverstonePreventsEvolution(PartyPokemon pkmn)
        {
            return pkmn.Item == ItemType.Everstone && pkmn.Species != PBESpecies.Kadabra;
        }
        private static bool IsNight()
        {
            DateTime time = DateTime.Now;
            Month month = OverworldTime.GetMonth((Month)time.Month);
            Season season = OverworldTime.GetSeason(month);
            int hour = OverworldTime.GetHour(time.Hour);
            return OverworldTime.GetTimeOfDay(season, hour) == TimeOfDay.Night;
        }
        private static bool IsNosepassMagnetonLocation()
        {
            MapSection mapSection = Overworld.GetPlayerMapSection();
            return mapSection == MapSection.TestCave;
        }
        // TODO
        private static bool IsLeafeonLocation()
        {
            return false;
        }
        // TODO
        private static bool IsGlaceonLocation()
        {
            return false;
        }

        // Ignores Shedinja_LevelUp & Beauty_LevelUp
        public static EvolutionData.EvoData GetLevelUpEvolution(Party party, PartyPokemon pkmn)
        {
            if (HeldEverstonePreventsEvolution(pkmn))
            {
                return null;
            }

            bool isNight = IsNight();

            // Cache attack and defense calcs for Tyrogue
            ushort attack = 0;
            ushort defense = 0;
            void CalcAttackDefense()
            {
                if (attack != 0 && defense != 0)
                {
                    return;
                }
                var bs = BaseStats.Get(pkmn.Species, pkmn.Form, true);
                attack = PBEDataUtils.CalculateStat(bs, PBEStat.Attack, pkmn.Nature, pkmn.EffortValues.Attack, pkmn.IndividualValues.Attack, pkmn.Level, PkmnConstants.PBESettings);
                defense = PBEDataUtils.CalculateStat(bs, PBEStat.Defense, pkmn.Nature, pkmn.EffortValues.Defense, pkmn.IndividualValues.Defense, pkmn.Level, PkmnConstants.PBESettings);
            }

            var data = new EvolutionData(pkmn.Species, pkmn.Form);
            foreach (EvolutionData.EvoData evo in data.Evolutions)
            {
                bool isMatch;
                switch (evo.Method)
                {
                    case EvoMethod.Friendship_LevelUp: isMatch = pkmn.Friendship >= evo.Param; break;
                    case EvoMethod.Friendship_Day_LevelUp: isMatch = !isNight && pkmn.Friendship >= evo.Param; break;
                    case EvoMethod.Friendship_Night_LevelUp: isMatch = isNight && pkmn.Friendship >= evo.Param; break;
                    case EvoMethod.LevelUp:
                    case EvoMethod.Ninjask_LevelUp: isMatch = pkmn.Level >= evo.Param; break;
                    case EvoMethod.Silcoon_LevelUp: isMatch = pkmn.Level >= evo.Param && ((pkmn.PID >> 0x10) % 10) <= 4; break;
                    case EvoMethod.Cascoon_LevelUp: isMatch = pkmn.Level >= evo.Param && ((pkmn.PID >> 0x10) % 10) > 4; break;
                    case EvoMethod.Item_Day_LevelUp: isMatch = !isNight && pkmn.Item == (ItemType)evo.Param; break;
                    case EvoMethod.Item_Night_LevelUp: isMatch = isNight && pkmn.Item == (ItemType)evo.Param; break;
                    case EvoMethod.Move_LevelUp: isMatch = pkmn.Moveset.Contains((PBEMove)evo.Param); break;
                    case EvoMethod.Male_LevelUp: isMatch = pkmn.Level >= evo.Param && pkmn.Gender == PBEGender.Male; break;
                    case EvoMethod.Female_LevelUp: isMatch = pkmn.Level >= evo.Param && pkmn.Gender == PBEGender.Female; break;
                    case EvoMethod.NosepassMagneton_Location_LevelUp: isMatch = IsNosepassMagnetonLocation(); break;
                    case EvoMethod.Leafeon_Location_LevelUp: isMatch = IsLeafeonLocation(); break;
                    case EvoMethod.Glaceon_Location_LevelUp: isMatch = IsGlaceonLocation(); break;
                    case EvoMethod.ATK_GT_DEF_LevelUp:
                    {
                        CalcAttackDefense();
                        isMatch = pkmn.Level >= evo.Param && attack > defense;
                        break;
                    }
                    case EvoMethod.ATK_EE_DEF_LevelUp:
                    {
                        CalcAttackDefense();
                        isMatch = pkmn.Level >= evo.Param && attack == defense;
                        break;
                    }
                    case EvoMethod.ATK_LT_DEF_LevelUp:
                    {
                        CalcAttackDefense();
                        isMatch = pkmn.Level >= evo.Param && attack < defense;
                        break;
                    }
                    case EvoMethod.PartySpecies_LevelUp:
                    {
                        isMatch = false;
                        foreach (PartyPokemon p in party)
                        {
                            if (p != pkmn && !p.IsEgg && p.Species == (PBESpecies)evo.Param)
                            {
                                isMatch = true;
                                break;
                            }
                        }
                        break;
                    }
                    default: isMatch = false; break;
                }
                if (isMatch)
                {
                    return evo;
                }
            }
            return null;
        }

        public static EvolutionData.EvoData GetItemEvolution(PartyPokemon pkmn, ItemType item)
        {
            // Everstone doesn't affect item evolution
            bool isNight = IsNight();

            var data = new EvolutionData(pkmn.Species, pkmn.Form);
            foreach (EvolutionData.EvoData evo in data.Evolutions)
            {
                if (item != (ItemType)evo.Param)
                {
                    continue;
                }
                bool isMatch;
                switch (evo.Method)
                {
                    case EvoMethod.Stone: isMatch = true; break;
                    case EvoMethod.Male_Stone: isMatch = pkmn.Gender == PBEGender.Male; break;
                    case EvoMethod.Female_Stone: isMatch = pkmn.Gender == PBEGender.Female; break;
                    case EvoMethod.Item_Day_LevelUp: isMatch = !isNight; break;
                    case EvoMethod.Item_Night_LevelUp: isMatch = isNight; break;
                    default: isMatch = false; break;
                }
                if (isMatch)
                {
                    return evo;
                }
            }
            return null;
        }

        public static EvolutionData.EvoData GetTradeEvolution(PartyPokemon pkmn, PBESpecies otherSpecies)
        {
            if (HeldEverstonePreventsEvolution(pkmn))
            {
                return null;
            }

            var data = new EvolutionData(pkmn.Species, pkmn.Form);
            foreach (EvolutionData.EvoData evo in data.Evolutions)
            {
                bool isMatch;
                switch (evo.Method)
                {
                    case EvoMethod.Trade: isMatch = true; break;
                    case EvoMethod.Item_Trade: isMatch = pkmn.Item == (ItemType)evo.Param; break;
                    case EvoMethod.ShelmetKarrablast:
                    {
                        isMatch = (pkmn.Species == PBESpecies.Shelmet && otherSpecies == PBESpecies.Karrablast)
                            || (pkmn.Species == PBESpecies.Karrablast && otherSpecies == PBESpecies.Shelmet);
                        break;
                    }
                    default: isMatch = false; break;
                }
                if (isMatch)
                {
                    return evo;
                }
            }
            return null;
        }

        public static void TryCreateShedinja(PartyPokemon nincada)
        {
            Party party = Game.Instance.Save.PlayerParty;
            if (party.Count >= PkmnConstants.PartyCapacity)
            {
                return;
            }

            Inventory<InventorySlotNew> inv = Game.Instance.Save.PlayerInventory;
            if (!inv.TryRemove(ItemType.PokeBall, 1))
            {
                return;
            }

            var shedinja = PartyPokemon.CreateShedinja(nincada);
            party.Add(shedinja);
            Game.Instance.Save.Pokedex.SetCaught(shedinja.Species, shedinja.Form, shedinja.Gender, shedinja.Shiny, shedinja.PID);
        }
    }
}
