/*
 * Copyright (c) 2024 Nordic Semiconductor ASA
 *
 * SPDX-License-Identifier: LicenseRef-Nordic-5-Clause
 */

#include <zephyr/ztest.h>
#include <suit_storage_mpi.h>
#include "test_common.h"

/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
static uint8_t nordic_vid[] = {
	0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
	0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4
};

/* RFC4122 uuid5(nordic_vid, 'nRF54H20_nordic_top') */
static uint8_t nordic_top_cid[] = {
	0xf0, 0x3d, 0x38, 0x5e, 0xa7, 0x31, 0x56, 0x05,
	0xb1, 0x5d, 0x03, 0x7f, 0x6d, 0xa6, 0x09, 0x7f
};

/* RFC4122 uuid5(nordic_vid, 'nRF54H20_sec') */
static uint8_t nordic_sdfw_cid[] = {
	0xd9, 0x6b, 0x40, 0xb7, 0x09, 0x2b, 0x5c, 0xd1,
	0xa5, 0x9f, 0x9a, 0xf8, 0x0c, 0x33, 0x7e, 0xba
};

/* RFC4122 uuid5(nordic_vid, 'nRF54H20_sys') */
static uint8_t nordic_scfw_cid[] = {
	0xc0, 0x8a, 0x25, 0xd7, 0x35, 0xe6, 0x59, 0x2c,
	0xb7, 0xad, 0x43, 0xac, 0xc8, 0xd1, 0xd1, 0xc8
};

/* RFC4122 uuid5(nordic_vid, 'test_sample_root') */
static uint8_t app_root_cid[] = {
	0x97, 0x05, 0x48, 0x23, 0x4c, 0x3d, 0x59, 0xa1,
	0x89, 0x86, 0xa5, 0x46, 0x60, 0xa1, 0x4b, 0x0a,
};

/* RFC4122 uuid5(nordic_vid, 'test_sample_rad') */
static uint8_t rad_cid[] = {
	0xf3, 0xe2, 0xb4, 0xe0, 0xd3, 0x73, 0x51, 0xdd,
	0x99, 0x2b, 0xba, 0x1d, 0x84, 0xf1, 0x16, 0x9a,
};

uint8_t nvv_empty[64] = {
	/* 0xFF * 32 */
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	/* SHA-256 */
	0xAF, 0x96, 0x13, 0x76, 0x0F, 0x72, 0x63, 0x5F,
	0xBD, 0xB4, 0x4A, 0x5A, 0x0A, 0x63, 0xC3, 0x9F,
	0x12, 0xAF, 0x30, 0xF9, 0x50, 0xA6, 0xEE, 0x5C,
	0x97, 0x1B, 0xE1, 0x88, 0xE8, 0x9C, 0x40, 0x51,
};

uint8_t nvv_sample[64] = {
	/* sample 8 values */
	0x00, 0x00, 0x00, 0x00, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0x55, 0xAA, 0x55, 0x00, 0x00, 0x00, 0x01,
	0x01, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
	0xAB, 0xCD, 0xEF, 0x98, 0xFF, 0xFF, 0xFF, 0xFF,
	/* SHA-256 */
	0x73, 0xb4, 0x25, 0x20, 0xdf, 0x43, 0x7e, 0xc7,
	0x63, 0xfb, 0xb6, 0x05, 0x70, 0x23, 0x2f, 0xb3,
	0x6a, 0x65, 0xbe, 0x14, 0xcb, 0x2e, 0x83, 0x6a,
	0x3b, 0xc9, 0xfd, 0x87, 0xf6, 0xde, 0x79, 0x05,
};

void erase_area_nordic(void)
{
	/* Clear the whole nordic area */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_erase(fdev, SUIT_STORAGE_NORDIC_OFFSET, SUIT_STORAGE_NORDIC_SIZE);

	zassert_equal(0, err, "Unable to erase nordic area before test execution");
}

void erase_area_rad(void)
{
	/* Clear the whole radio area */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_erase(fdev, SUIT_STORAGE_RAD_OFFSET, SUIT_STORAGE_RAD_SIZE);

	zassert_equal(0, err, "Unable to erase radio core area before test execution");
}

void erase_area_app(void)
{
	/* Clear the whole application area */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_erase(fdev, SUIT_STORAGE_APP_OFFSET, SUIT_STORAGE_APP_SIZE);

	zassert_equal(0, err, "Unable to erase application area before test execution");
}

void erase_area_app_nvv(void)
{
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_erase(fdev, SUIT_STORAGE_APP_NVV_OFFSET, SUIT_STORAGE_APP_NVV_SIZE / 2);

	zassert_equal(0, err, "Unable to erase NNV area before test execution");
}

void erase_area_app_nvv_backup(void)
{
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_erase(fdev, SUIT_STORAGE_APP_NVV_OFFSET + SUIT_STORAGE_APP_NVV_SIZE / 2,
			      SUIT_STORAGE_APP_NVV_SIZE / 2);

	zassert_equal(0, err, "Unable to erase NNV backup before test execution");
}

void write_area_app_empty_nvv_backup(void)
{
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_APP_NVV_OFFSET + SUIT_STORAGE_APP_NVV_SIZE / 2,
			      nvv_empty, sizeof(nvv_empty));

	zassert_equal(0, err, "Unable to store empty NVV contents before test execution");
}

void write_area_app_nvv_backup(void)
{
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_APP_NVV_OFFSET + SUIT_STORAGE_APP_NVV_SIZE / 2,
			      nvv_sample, sizeof(nvv_sample));

	zassert_equal(0, err, "Unable to store empty NVV contents before test execution");
}

void write_area_app_nvv(void)
{
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_APP_NVV_OFFSET, nvv_sample, sizeof(nvv_sample));

	zassert_equal(0, err, "Unable to store sample NVV contents before test execution");
}

void write_empty_area_app(void)
{
	/* Digest of the content defined in assert_empty_mpi_area_app(). */
	uint8_t app_digest[] = {
		0xd6, 0xc4, 0x94, 0x17, 0xb1, 0xca, 0x0a, 0x67,
		0x14, 0xdc, 0xde, 0x2b, 0x40, 0x01, 0x0c, 0xb7,
		0x49, 0x6d, 0x05, 0xdf, 0x7f, 0x8c, 0x8b, 0x1b,
		0x98, 0x14, 0x09, 0x7e, 0x9d, 0x62, 0xc8, 0xe1,
	};

	/* Write the digest of application area filled with 0xFF */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_APP_OFFSET + SUIT_STORAGE_APP_MPI_SIZE, app_digest,
			      sizeof(app_digest));

	zassert_equal(0, err, "Unable to store application MPI digest before test execution");
}

void write_empty_area_rad(void)
{
	/* Digest of the content defined in assert_empty_mpi_area_rad(). */
	uint8_t rad_digest[] = {
		0xd1, 0x69, 0xf6, 0x75, 0x42, 0x29, 0xc2, 0x00,
		0xba, 0x48, 0x38, 0xa3, 0x8e, 0x48, 0x94, 0xc0,
		0x3c, 0x47, 0xce, 0x89, 0x39, 0xdb, 0x4b, 0x7a,
		0x11, 0xc2, 0x24, 0x92, 0x1b, 0x98, 0x25, 0x21,
	};

	/* Write the digest of radio area filled with 0xFF */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_RAD_OFFSET + SUIT_STORAGE_RAD_MPI_SIZE, rad_digest,
			      sizeof(rad_digest));

	zassert_equal(0, err, "Unable to store radio MPI digest before test execution");
}

void write_area_app_root(void)
{
	uint8_t mpi_root[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_root') */
		0x97, 0x05, 0x48, 0x23, 0x4c, 0x3d, 0x59, 0xa1,
		0x89, 0x86, 0xa5, 0x46, 0x60, 0xa1, 0x4b, 0x0a,
	};
	/* Digest of the content defined in assert_valid_mpi_area_app(). */
	uint8_t app_digest[] = {
		0xa9, 0xa0, 0xee, 0x40, 0x5f, 0xad, 0x2b, 0xeb,
		0x66, 0x50, 0xdd, 0xa9, 0x97, 0x11, 0x72, 0x98,
		0x2b, 0x17, 0x45, 0x90, 0x16, 0xe1, 0xc7, 0xf5,
		0xc1, 0xdc, 0x3f, 0xb4, 0x58, 0x96, 0x1e, 0x44,
	};

	/* Write the sample application area (just the root MPI) and corresponding digest */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_APP_OFFSET, mpi_root, sizeof(mpi_root));

	zassert_equal(0, err,
		      "Unable to store application root MPI contents before test execution");

	err = flash_write(fdev, SUIT_STORAGE_APP_OFFSET + SUIT_STORAGE_APP_MPI_SIZE, app_digest,
			  sizeof(app_digest));
	zassert_equal(0, err, "Unable to store application MPI digest before test execution");
}

void write_area_rad(void)
{
	uint8_t mpi_rad[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_rad') */
		0xf3, 0xe2, 0xb4, 0xe0, 0xd3, 0x73, 0x51, 0xdd,
		0x99, 0x2b, 0xba, 0x1d, 0x84, 0xf1, 0x16, 0x9a,
	};
	/* Digest of the content defined in assert_valid_mpi_area_rad(). */
	uint8_t rad_digest[] = {
		0x29, 0x8b, 0x7b, 0x01, 0x42, 0x24, 0xcf, 0x2f,
		0x48, 0xe5, 0xf3, 0x6d, 0x0f, 0xa5, 0x66, 0x6d,
		0x17, 0x42, 0x32, 0xf7, 0x03, 0x9f, 0xf8, 0xa4,
		0x8d, 0x52, 0xc6, 0x8d, 0xea, 0xdc, 0x33, 0x3b,
	};

	/* Write the sample radio area (just one MPI entry) and corresponding digest */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, SUIT_STORAGE_RAD_OFFSET, mpi_rad, sizeof(mpi_rad));

	zassert_equal(0, err, "Unable to store radio root MPI contents before test execution");

	err = flash_write(fdev, SUIT_STORAGE_RAD_OFFSET + SUIT_STORAGE_RAD_MPI_SIZE, rad_digest,
			  sizeof(rad_digest));
	zassert_equal(0, err, "Unable to store radio MPI digest before test execution");
}

void write_area_nordic_root(void)
{
	uintptr_t mpi_root_backup_offset =
		SUIT_STORAGE_NORDIC_OFFSET + SUIT_STORAGE_RAD_MPI_SIZE + SUIT_STORAGE_DIGEST_SIZE;
	uint8_t mpi_root[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_root') */
		0x97, 0x05, 0x48, 0x23, 0x4c, 0x3d, 0x59, 0xa1,
		0x89, 0x86, 0xa5, 0x46, 0x60, 0xa1, 0x4b, 0x0a,
	};
	/* Digest of the content defined in assert_valid_mpi_area_app(). */
	uint8_t app_digest[] = {
		0xa9, 0xa0, 0xee, 0x40, 0x5f, 0xad, 0x2b, 0xeb,
		0x66, 0x50, 0xdd, 0xa9, 0x97, 0x11, 0x72, 0x98,
		0x2b, 0x17, 0x45, 0x90, 0x16, 0xe1, 0xc7, 0xf5,
		0xc1, 0xdc, 0x3f, 0xb4, 0x58, 0x96, 0x1e, 0x44,
	};

	/* Write the sample application backup area (just the root MPI) and corresponding digest */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, mpi_root_backup_offset, mpi_root, sizeof(mpi_root));

	zassert_equal(0, err,
		      "Unable to store application backup root MPI contents before test execution");

	err = flash_write(fdev, mpi_root_backup_offset + SUIT_STORAGE_APP_MPI_SIZE, app_digest,
			  sizeof(app_digest));
	zassert_equal(0, err,
		      "Unable to store application backup MPI digest before test execution");
}

void write_area_nordic_rad(void)
{
	uintptr_t mpi_rad_backup_offset = SUIT_STORAGE_NORDIC_OFFSET;
	uint8_t mpi_rad[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_rad') */
		0xf3, 0xe2, 0xb4, 0xe0, 0xd3, 0x73, 0x51, 0xdd,
		0x99, 0x2b, 0xba, 0x1d, 0x84, 0xf1, 0x16, 0x9a,
	};
	/* Digest of the content defined in assert_valid_mpi_area_rad(). */
	uint8_t rad_digest[] = {
		0x29, 0x8b, 0x7b, 0x01, 0x42, 0x24, 0xcf, 0x2f,
		0x48, 0xe5, 0xf3, 0x6d, 0x0f, 0xa5, 0x66, 0x6d,
		0x17, 0x42, 0x32, 0xf7, 0x03, 0x9f, 0xf8, 0xa4,
		0x8d, 0x52, 0xc6, 0x8d, 0xea, 0xdc, 0x33, 0x3b,
	};

	/* Write the sample radio backup area (just one MPI entry) and corresponding digest */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, mpi_rad_backup_offset, mpi_rad, sizeof(mpi_rad));

	zassert_equal(0, err,
		      "Unable to store radio backup root MPI contents before test execution");

	err = flash_write(fdev, mpi_rad_backup_offset + SUIT_STORAGE_RAD_MPI_SIZE, rad_digest,
			  sizeof(rad_digest));
	zassert_equal(0, err, "Unable to store radio backup MPI digest before test execution");
}

void write_area_nordic_old_root(void)
{
	uintptr_t mpi_root_backup_offset =
		SUIT_STORAGE_NORDIC_OFFSET + SUIT_STORAGE_RAD_MPI_SIZE + SUIT_STORAGE_DIGEST_SIZE;
	uint8_t mpi_root[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* 0xAA * 16 */
		0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
		0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
	};
	uint8_t app_digest[] = {
		0xca, 0x4f, 0x80, 0xdc, 0xd1, 0x9b, 0xb2, 0x24,
		0x8e, 0x2b, 0x13, 0xeb, 0x6c, 0xf2, 0xae, 0xd0,
		0x96, 0x28, 0xc2, 0x0c, 0x69, 0xe5, 0xf5, 0x23,
		0x0c, 0x92, 0xd6, 0x2d, 0x36, 0x69, 0x27, 0x58,
	};

	/* Write the sample application backup area (just the root MPI with altered class) and
	 * corresponding digest
	 */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, mpi_root_backup_offset, mpi_root, sizeof(mpi_root));

	zassert_equal(0, err,
		      "Unable to store application backup root MPI contents before test execution");

	err = flash_write(fdev, mpi_root_backup_offset + SUIT_STORAGE_APP_MPI_SIZE, app_digest,
			  sizeof(app_digest));
	zassert_equal(0, err,
		      "Unable to store application backup MPI digest before test execution");
}

void write_area_nordic_old_rad(void)
{
	uintptr_t mpi_rad_backup_offset = SUIT_STORAGE_NORDIC_OFFSET;
	uint8_t mpi_rad[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* 0xBB * 16 */
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
		0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	};
	uint8_t rad_digest[] = {
		0x7a, 0x07, 0x30, 0x81, 0xea, 0x7b, 0x3b, 0x08,
		0xca, 0x55, 0xbe, 0x48, 0x8d, 0x75, 0x22, 0x35,
		0x6f, 0x1f, 0x82, 0x16, 0x28, 0xae, 0x68, 0x71,
		0xc3, 0xc1, 0x8e, 0x5b, 0xb9, 0xf0, 0x2f, 0xdd,
	};

	/* Write the sample radio backup area (just the sample MPI with altered class) and
	 * corresponding digest
	 */
	const struct device *fdev = SUIT_PLAT_INTERNAL_NVM_DEV;

	zassert_not_null(fdev, "Unable to find a driver to modify MPI area");

	int err = flash_write(fdev, mpi_rad_backup_offset, mpi_rad, sizeof(mpi_rad));

	zassert_equal(0, err,
		      "Unable to store radio backup sample MPI contents before test execution");

	err = flash_write(fdev, mpi_rad_backup_offset + SUIT_STORAGE_APP_MPI_SIZE, rad_digest,
			  sizeof(rad_digest));
	zassert_equal(0, err, "Unable to store radio backup MPI digest before test execution");
}

void assert_nordic_classes(void)
{
	suit_manifest_class_info_t class_infos[CONFIG_SUIT_STORAGE_N_ENVELOPES];
	size_t class_infos_len = CONFIG_SUIT_STORAGE_N_ENVELOPES;

	/* ASSERT that it is possible to fetch list of supported manifest classes and roles */
	int err = suit_storage_mpi_class_ids_get(class_infos, &class_infos_len);

	zassert_equal(err, SUIT_PLAT_SUCCESS,
		      "Failed to fetch list of supported manifest classes (%d).", err);
	/* ... and MPI reports at least 3 class IDs */
	zassert_true(class_infos_len >= 3,
		     "Invalid number of supported manifest classes (%d < %d).", class_infos_len, 3);
	/* ... and the Nordic top manifest class is supported */
	zassert_mem_equal(class_infos[0].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[0].class_id, nordic_top_cid, sizeof(nordic_top_cid));
	zassert_equal(class_infos[0].role, SUIT_MANIFEST_SEC_TOP, "Invalid class role returned: %d",
		      class_infos[0].role);
	/* ... and the Nordic secure domain update & recovery manifest class is supported */
	zassert_mem_equal(class_infos[1].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[1].class_id, nordic_sdfw_cid, sizeof(nordic_sdfw_cid));
	zassert_equal(class_infos[1].role, SUIT_MANIFEST_SEC_SDFW,
		      "Invalid class role returned: %d", class_infos[3].role);
	/* ... and the Nordic system controller manifest class is supported */
	zassert_mem_equal(class_infos[2].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[2].class_id, nordic_scfw_cid, sizeof(nordic_scfw_cid));
	zassert_equal(class_infos[2].role, SUIT_MANIFEST_SEC_SYSCTRL,
		      "Invalid class role returned: %d", class_infos[3].role);
}

void assert_sample_root_class(void)
{
	suit_manifest_class_info_t class_infos[CONFIG_SUIT_STORAGE_N_ENVELOPES];
	size_t class_infos_len = CONFIG_SUIT_STORAGE_N_ENVELOPES;

	/* ASSERT that it is possible to fetch list of supported manifest classes and roles */
	int err = suit_storage_mpi_class_ids_get(class_infos, &class_infos_len);

	zassert_equal(err, SUIT_PLAT_SUCCESS,
		      "Failed to fetch list of supported manifest classes (%d).", err);
	/* ... and MPI reports 4 supported class IDs */
	zassert_equal(class_infos_len, 4,
		      "Invalid number of supported manifest classes (%d != %d).", class_infos_len,
		      4);
	/* ... and the sample application root manifest class is supported */
	zassert_mem_equal(class_infos[3].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[3].class_id, app_root_cid, sizeof(app_root_cid));
	zassert_equal(class_infos[3].role, SUIT_MANIFEST_APP_ROOT,
		      "Invalid class role returned: %d", class_infos[3].role);
}

void assert_sample_root_rad_class(void)
{
	suit_manifest_class_info_t class_infos[CONFIG_SUIT_STORAGE_N_ENVELOPES];
	size_t class_infos_len = CONFIG_SUIT_STORAGE_N_ENVELOPES;

	/* ASSERT that it is possible to fetch list of supported manifest classes and roles */
	int err = suit_storage_mpi_class_ids_get(class_infos, &class_infos_len);

	zassert_equal(err, SUIT_PLAT_SUCCESS,
		      "Failed to fetch list of supported manifest classes (%d).", err);
	/* ... and MPI reports 5 supported class IDs */
	zassert_equal(class_infos_len, 5,
		      "Invalid number of supported manifest classes (%d != %d).", class_infos_len,
		      5);
	/* ... and the sample application root manifest class is supported */
	zassert_mem_equal(class_infos[3].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[3].class_id, rad_cid, sizeof(rad_cid));
	zassert_equal(class_infos[3].role, SUIT_MANIFEST_RAD_RECOVERY,
		      "Invalid class role returned: %d", class_infos[3].role);
	/* ... and the sample application root manifest class is supported */
	zassert_mem_equal(class_infos[4].vendor_id, nordic_vid, sizeof(nordic_vid));
	zassert_mem_equal(class_infos[4].class_id, app_root_cid, sizeof(app_root_cid));
	zassert_equal(class_infos[4].role, SUIT_MANIFEST_APP_ROOT,
		      "Invalid class role returned: %d", class_infos[4].role);
}

void assert_empty_mpi_area_app(uint8_t *addr, size_t size)
{
	uint8_t empty_mpi_app[] = {
		/* 0xFF * 240 */
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* SHA-256 */
		0xd6, 0xc4, 0x94, 0x17, 0xb1, 0xca, 0x0a, 0x67,
		0x14, 0xdc, 0xde, 0x2b, 0x40, 0x01, 0x0c, 0xb7,
		0x49, 0x6d, 0x05, 0xdf, 0x7f, 0x8c, 0x8b, 0x1b,
		0x98, 0x14, 0x09, 0x7e, 0x9d, 0x62, 0xc8, 0xe1,
	};

	zassert_equal(size, sizeof(empty_mpi_app), "Incorrect application area size (0x%x != 0x%x)",
		      size, sizeof(empty_mpi_app));
	zassert_mem_equal(addr, empty_mpi_app, sizeof(empty_mpi_app),
			  "MPI application area does not match empty contents (addr: 0x%lx)",
			  (uintptr_t)addr);
}

void assert_valid_mpi_area_app(uint8_t *addr, size_t size)
{
	uint8_t valid_mpi_app[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_root') */
		0x97, 0x05, 0x48, 0x23, 0x4c, 0x3d, 0x59, 0xa1,
		0x89, 0x86, 0xa5, 0x46, 0x60, 0xa1, 0x4b, 0x0a,
		/* 0xFF * 192 */
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* SHA-256 */
		0xa9, 0xa0, 0xee, 0x40, 0x5f, 0xad, 0x2b, 0xeb,
		0x66, 0x50, 0xdd, 0xa9, 0x97, 0x11, 0x72, 0x98,
		0x2b, 0x17, 0x45, 0x90, 0x16, 0xe1, 0xc7, 0xf5,
		0xc1, 0xdc, 0x3f, 0xb4, 0x58, 0x96, 0x1e, 0x44,
	};

	zassert_equal(size, sizeof(valid_mpi_app), "Incorrect application area size (0x%x != 0x%x)",
		      size, sizeof(valid_mpi_app));
	zassert_mem_equal(addr, valid_mpi_app, sizeof(valid_mpi_app),
			  "MPI application area does not match empty contents (addr: 0x%lx)",
			  (uintptr_t)addr);
}

void assert_empty_mpi_area_rad(uint8_t *addr, size_t size)
{
	uint8_t empty_mpi_rad[] = {
		/* 0xFF * 144 */
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* SHA-256 */
		0xd1, 0x69, 0xf6, 0x75, 0x42, 0x29, 0xc2, 0x00,
		0xba, 0x48, 0x38, 0xa3, 0x8e, 0x48, 0x94, 0xc0,
		0x3c, 0x47, 0xce, 0x89, 0x39, 0xdb, 0x4b, 0x7a,
		0x11, 0xc2, 0x24, 0x92, 0x1b, 0x98, 0x25, 0x21,
	};

	zassert_equal(size, sizeof(empty_mpi_rad), "Incorrect radio area size (0x%x != 0x%x)", size,
		      sizeof(empty_mpi_rad));
	zassert_mem_equal(addr, empty_mpi_rad, sizeof(empty_mpi_rad),
			  "MPI radio area does not match empty contents (addr: 0x%lx)",
			  (uintptr_t)addr);
}

void assert_valid_mpi_area_rad(uint8_t *addr, size_t size)
{
	uint8_t valid_mpi_rad[] = {
		0x01, /* version */
		0x01, /* downgrade prevention disabled */
		0x02, /* Independent update allowed */
		0x01, /* signature check disabled */
		/* reserved (12) */
		0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* RFC4122 uuid5(uuid.NAMESPACE_DNS, 'nordicsemi.com') */
		0x76, 0x17, 0xda, 0xa5, 0x71, 0xfd, 0x5a, 0x85,
		0x8f, 0x94, 0xe2, 0x8d, 0x73, 0x5c, 0xe9, 0xf4,
		/* RFC4122 uuid5(nordic_vid, 'test_sample_rad') */
		0xf3, 0xe2, 0xb4, 0xe0, 0xd3, 0x73, 0x51, 0xdd,
		0x99, 0x2b, 0xba, 0x1d, 0x84, 0xf1, 0x16, 0x9a,
		/* 0xFF * 96 */
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
		/* SHA-256 */
		0x29, 0x8b, 0x7b, 0x01, 0x42, 0x24, 0xcf, 0x2f,
		0x48, 0xe5, 0xf3, 0x6d, 0x0f, 0xa5, 0x66, 0x6d,
		0x17, 0x42, 0x32, 0xf7, 0x03, 0x9f, 0xf8, 0xa4,
		0x8d, 0x52, 0xc6, 0x8d, 0xea, 0xdc, 0x33, 0x3b,
	};

	zassert_equal(size, sizeof(valid_mpi_rad), "Incorrect radio area size (0x%x != 0x%x)", size,
		      sizeof(valid_mpi_rad));
	zassert_mem_equal(addr, valid_mpi_rad, sizeof(valid_mpi_rad),
			  "MPI radio area does not match empty contents (addr: 0x%lx)",
			  (uintptr_t)addr);
}
