/*
 * Copyright (c) 2024 Nordic Semiconductor ASA
 *
 * SPDX-License-Identifier: LicenseRef-Nordic-5-Clause
 */

#pragma once

#include <platform/PlatformManager.h>

/**
 * @brief nRF Connect Matter event handler API.
 *
 * This module allows to safely register and unregister Matter event handler functions from the application code.
 * It also offers a default handler that is utilized in all nrfconnect SDK Matter samples and applications,
 * and that can be re-used also in a production firmware.
 *
 * A user can register multiple event handlers that are kept in the Matter server as a linked list.
 *
 * Note that only public events (kFlag_IsPublic) are propagated to the Matter application scope.
 * To learn details about Matter public events please refer to the src/include/platform/CHIPDeviceEvent.h file
 * in the Matter SDK.
 */
namespace Nrf::Matter
{
/**
 * @brief Safely register custom Matter event handler
 *
 * This function adds user provided Matter event handler in a thread safe manner.
 *
 * @param handler Matter event handler to register
 * @param data Matter event data
 * @return CHIP_NO_ERROR on success, other error code otherwise
 */
CHIP_ERROR RegisterEventHandler(chip::DeviceLayer::PlatformManager::EventHandlerFunct handler, intptr_t data);

/**
 * @brief Safely unregister custom Matter event handler
 *
 * This function removes previously added Matter event handler in a thread safe manner.
 *
 * @param handler Matter event handler to unregister
 * @param data Matter event data
 * @return CHIP_NO_ERROR on success, other error code otherwise
 */
void UnregisterEventHandler(chip::DeviceLayer::PlatformManager::EventHandlerFunct handler, intptr_t data);

/**
 * @brief Default Matter event handler
 *
 * This function implements the default, Nordic specific handling of public events generated by Matter stack.
 * Note that even if DefaultEventHandler is registered to serve Matter events (for instance, when initializing
 * Matter server), its behavior can be extended with further handlers that can be registered with
 * RegisterEventHandler function (even after the Matter server is initialized).
 *
 * @param event Matter event data
 * @param data address to the optional context data
 */
void DefaultEventHandler(const chip::DeviceLayer::ChipDeviceEvent *event, intptr_t data);

} // namespace Nrf::Matter
