/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_common_tables.c
 * Description:  common tables like fft twiddle factors, Bitreverse, reciprocal etc
 *
 * $Date:        23 April 2021
 * $Revision:    V1.9.0
 *
 * Target Processor: Cortex-M and Cortex-A cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2021 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math_types.h"
#include "arm_common_tables.h"

/**
  @ingroup ComplexFFT
 */

/**
  @addtogroup CFFT_CIFFT Complex FFT Tables
  @{
 */

/**
  @par
  Pseudo code for Generation of Bit reversal Table is
  @par
  <pre>for (l = 1; l <= N/4; l++)
  {
    for (i = 0; i< logN2; i++)
    {
      a[i] = l & (1 << i);
    }
    for (j = 0; j < logN2; j++)
    {
      if (a[j] != 0)
      y[l] += (1 << ((logN2 - 1) - j));
    }
    y[l] = y[l] >> 1;
   } </pre>
  @par
  where N = 4096, logN2 = 12
  @par
  N is the maximum FFT Size supported
*/

/**
  @brief  Table for bit reversal process
*/
#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_FFT_ALLOW_TABLES)

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_BITREV_1024)
const uint16_t armBitRevTable[1024] = {
   0x400, 0x200, 0x600, 0x100, 0x500, 0x300, 0x700, 0x080, 0x480, 0x280,
   0x680, 0x180, 0x580, 0x380, 0x780, 0x040, 0x440, 0x240, 0x640, 0x140,
   0x540, 0x340, 0x740, 0x0c0, 0x4c0, 0x2c0, 0x6c0, 0x1c0, 0x5c0, 0x3c0,
   0x7c0, 0x020, 0x420, 0x220, 0x620, 0x120, 0x520, 0x320, 0x720, 0x0a0,
   0x4a0, 0x2a0, 0x6a0, 0x1a0, 0x5a0, 0x3a0, 0x7a0, 0x060, 0x460, 0x260,
   0x660, 0x160, 0x560, 0x360, 0x760, 0x0e0, 0x4e0, 0x2e0, 0x6e0, 0x1e0,
   0x5e0, 0x3e0, 0x7e0, 0x010, 0x410, 0x210, 0x610, 0x110, 0x510, 0x310,
   0x710, 0x090, 0x490, 0x290, 0x690, 0x190, 0x590, 0x390, 0x790, 0x050,
   0x450, 0x250, 0x650, 0x150, 0x550, 0x350, 0x750, 0x0d0, 0x4d0, 0x2d0,
   0x6d0, 0x1d0, 0x5d0, 0x3d0, 0x7d0, 0x030, 0x430, 0x230, 0x630, 0x130,
   0x530, 0x330, 0x730, 0x0b0, 0x4b0, 0x2b0, 0x6b0, 0x1b0, 0x5b0, 0x3b0,
   0x7b0, 0x070, 0x470, 0x270, 0x670, 0x170, 0x570, 0x370, 0x770, 0x0f0,
   0x4f0, 0x2f0, 0x6f0, 0x1f0, 0x5f0, 0x3f0, 0x7f0, 0x008, 0x408, 0x208,
   0x608, 0x108, 0x508, 0x308, 0x708, 0x088, 0x488, 0x288, 0x688, 0x188,
   0x588, 0x388, 0x788, 0x048, 0x448, 0x248, 0x648, 0x148, 0x548, 0x348,
   0x748, 0x0c8, 0x4c8, 0x2c8, 0x6c8, 0x1c8, 0x5c8, 0x3c8, 0x7c8, 0x028,
   0x428, 0x228, 0x628, 0x128, 0x528, 0x328, 0x728, 0x0a8, 0x4a8, 0x2a8,
   0x6a8, 0x1a8, 0x5a8, 0x3a8, 0x7a8, 0x068, 0x468, 0x268, 0x668, 0x168,
   0x568, 0x368, 0x768, 0x0e8, 0x4e8, 0x2e8, 0x6e8, 0x1e8, 0x5e8, 0x3e8,
   0x7e8, 0x018, 0x418, 0x218, 0x618, 0x118, 0x518, 0x318, 0x718, 0x098,
   0x498, 0x298, 0x698, 0x198, 0x598, 0x398, 0x798, 0x058, 0x458, 0x258,
   0x658, 0x158, 0x558, 0x358, 0x758, 0x0d8, 0x4d8, 0x2d8, 0x6d8, 0x1d8,
   0x5d8, 0x3d8, 0x7d8, 0x038, 0x438, 0x238, 0x638, 0x138, 0x538, 0x338,
   0x738, 0x0b8, 0x4b8, 0x2b8, 0x6b8, 0x1b8, 0x5b8, 0x3b8, 0x7b8, 0x078,
   0x478, 0x278, 0x678, 0x178, 0x578, 0x378, 0x778, 0x0f8, 0x4f8, 0x2f8,
   0x6f8, 0x1f8, 0x5f8, 0x3f8, 0x7f8, 0x004, 0x404, 0x204, 0x604, 0x104,
   0x504, 0x304, 0x704, 0x084, 0x484, 0x284, 0x684, 0x184, 0x584, 0x384,
   0x784, 0x044, 0x444, 0x244, 0x644, 0x144, 0x544, 0x344, 0x744, 0x0c4,
   0x4c4, 0x2c4, 0x6c4, 0x1c4, 0x5c4, 0x3c4, 0x7c4, 0x024, 0x424, 0x224,
   0x624, 0x124, 0x524, 0x324, 0x724, 0x0a4, 0x4a4, 0x2a4, 0x6a4, 0x1a4,
   0x5a4, 0x3a4, 0x7a4, 0x064, 0x464, 0x264, 0x664, 0x164, 0x564, 0x364,
   0x764, 0x0e4, 0x4e4, 0x2e4, 0x6e4, 0x1e4, 0x5e4, 0x3e4, 0x7e4, 0x014,
   0x414, 0x214, 0x614, 0x114, 0x514, 0x314, 0x714, 0x094, 0x494, 0x294,
   0x694, 0x194, 0x594, 0x394, 0x794, 0x054, 0x454, 0x254, 0x654, 0x154,
   0x554, 0x354, 0x754, 0x0d4, 0x4d4, 0x2d4, 0x6d4, 0x1d4, 0x5d4, 0x3d4,
   0x7d4, 0x034, 0x434, 0x234, 0x634, 0x134, 0x534, 0x334, 0x734, 0x0b4,
   0x4b4, 0x2b4, 0x6b4, 0x1b4, 0x5b4, 0x3b4, 0x7b4, 0x074, 0x474, 0x274,
   0x674, 0x174, 0x574, 0x374, 0x774, 0x0f4, 0x4f4, 0x2f4, 0x6f4, 0x1f4,
   0x5f4, 0x3f4, 0x7f4, 0x00c, 0x40c, 0x20c, 0x60c, 0x10c, 0x50c, 0x30c,
   0x70c, 0x08c, 0x48c, 0x28c, 0x68c, 0x18c, 0x58c, 0x38c, 0x78c, 0x04c,
   0x44c, 0x24c, 0x64c, 0x14c, 0x54c, 0x34c, 0x74c, 0x0cc, 0x4cc, 0x2cc,
   0x6cc, 0x1cc, 0x5cc, 0x3cc, 0x7cc, 0x02c, 0x42c, 0x22c, 0x62c, 0x12c,
   0x52c, 0x32c, 0x72c, 0x0ac, 0x4ac, 0x2ac, 0x6ac, 0x1ac, 0x5ac, 0x3ac,
   0x7ac, 0x06c, 0x46c, 0x26c, 0x66c, 0x16c, 0x56c, 0x36c, 0x76c, 0x0ec,
   0x4ec, 0x2ec, 0x6ec, 0x1ec, 0x5ec, 0x3ec, 0x7ec, 0x01c, 0x41c, 0x21c,
   0x61c, 0x11c, 0x51c, 0x31c, 0x71c, 0x09c, 0x49c, 0x29c, 0x69c, 0x19c,
   0x59c, 0x39c, 0x79c, 0x05c, 0x45c, 0x25c, 0x65c, 0x15c, 0x55c, 0x35c,
   0x75c, 0x0dc, 0x4dc, 0x2dc, 0x6dc, 0x1dc, 0x5dc, 0x3dc, 0x7dc, 0x03c,
   0x43c, 0x23c, 0x63c, 0x13c, 0x53c, 0x33c, 0x73c, 0x0bc, 0x4bc, 0x2bc,
   0x6bc, 0x1bc, 0x5bc, 0x3bc, 0x7bc, 0x07c, 0x47c, 0x27c, 0x67c, 0x17c,
   0x57c, 0x37c, 0x77c, 0x0fc, 0x4fc, 0x2fc, 0x6fc, 0x1fc, 0x5fc, 0x3fc,
   0x7fc, 0x002, 0x402, 0x202, 0x602, 0x102, 0x502, 0x302, 0x702, 0x082,
   0x482, 0x282, 0x682, 0x182, 0x582, 0x382, 0x782, 0x042, 0x442, 0x242,
   0x642, 0x142, 0x542, 0x342, 0x742, 0x0c2, 0x4c2, 0x2c2, 0x6c2, 0x1c2,
   0x5c2, 0x3c2, 0x7c2, 0x022, 0x422, 0x222, 0x622, 0x122, 0x522, 0x322,
   0x722, 0x0a2, 0x4a2, 0x2a2, 0x6a2, 0x1a2, 0x5a2, 0x3a2, 0x7a2, 0x062,
   0x462, 0x262, 0x662, 0x162, 0x562, 0x362, 0x762, 0x0e2, 0x4e2, 0x2e2,
   0x6e2, 0x1e2, 0x5e2, 0x3e2, 0x7e2, 0x012, 0x412, 0x212, 0x612, 0x112,
   0x512, 0x312, 0x712, 0x092, 0x492, 0x292, 0x692, 0x192, 0x592, 0x392,
   0x792, 0x052, 0x452, 0x252, 0x652, 0x152, 0x552, 0x352, 0x752, 0x0d2,
   0x4d2, 0x2d2, 0x6d2, 0x1d2, 0x5d2, 0x3d2, 0x7d2, 0x032, 0x432, 0x232,
   0x632, 0x132, 0x532, 0x332, 0x732, 0x0b2, 0x4b2, 0x2b2, 0x6b2, 0x1b2,
   0x5b2, 0x3b2, 0x7b2, 0x072, 0x472, 0x272, 0x672, 0x172, 0x572, 0x372,
   0x772, 0x0f2, 0x4f2, 0x2f2, 0x6f2, 0x1f2, 0x5f2, 0x3f2, 0x7f2, 0x00a,
   0x40a, 0x20a, 0x60a, 0x10a, 0x50a, 0x30a, 0x70a, 0x08a, 0x48a, 0x28a,
   0x68a, 0x18a, 0x58a, 0x38a, 0x78a, 0x04a, 0x44a, 0x24a, 0x64a, 0x14a,
   0x54a, 0x34a, 0x74a, 0x0ca, 0x4ca, 0x2ca, 0x6ca, 0x1ca, 0x5ca, 0x3ca,
   0x7ca, 0x02a, 0x42a, 0x22a, 0x62a, 0x12a, 0x52a, 0x32a, 0x72a, 0x0aa,
   0x4aa, 0x2aa, 0x6aa, 0x1aa, 0x5aa, 0x3aa, 0x7aa, 0x06a, 0x46a, 0x26a,
   0x66a, 0x16a, 0x56a, 0x36a, 0x76a, 0x0ea, 0x4ea, 0x2ea, 0x6ea, 0x1ea,
   0x5ea, 0x3ea, 0x7ea, 0x01a, 0x41a, 0x21a, 0x61a, 0x11a, 0x51a, 0x31a,
   0x71a, 0x09a, 0x49a, 0x29a, 0x69a, 0x19a, 0x59a, 0x39a, 0x79a, 0x5a,
   0x45a, 0x25a, 0x65a, 0x15a, 0x55a, 0x35a, 0x75a, 0x0da, 0x4da, 0x2da,
   0x6da, 0x1da, 0x5da, 0x3da, 0x7da, 0x03a, 0x43a, 0x23a, 0x63a, 0x13a,
   0x53a, 0x33a, 0x73a, 0x0ba, 0x4ba, 0x2ba, 0x6ba, 0x1ba, 0x5ba, 0x3ba,
   0x7ba, 0x07a, 0x47a, 0x27a, 0x67a, 0x17a, 0x57a, 0x37a, 0x77a, 0x0fa,
   0x4fa, 0x2fa, 0x6fa, 0x1fa, 0x5fa, 0x3fa, 0x7fa, 0x006, 0x406, 0x206,
   0x606, 0x106, 0x506, 0x306, 0x706, 0x086, 0x486, 0x286, 0x686, 0x186,
   0x586, 0x386, 0x786, 0x046, 0x446, 0x246, 0x646, 0x146, 0x546, 0x346,
   0x746, 0x0c6, 0x4c6, 0x2c6, 0x6c6, 0x1c6, 0x5c6, 0x3c6, 0x7c6, 0x026,
   0x426, 0x226, 0x626, 0x126, 0x526, 0x326, 0x726, 0x0a6, 0x4a6, 0x2a6,
   0x6a6, 0x1a6, 0x5a6, 0x3a6, 0x7a6, 0x066, 0x466, 0x266, 0x666, 0x166,
   0x566, 0x366, 0x766, 0x0e6, 0x4e6, 0x2e6, 0x6e6, 0x1e6, 0x5e6, 0x3e6,
   0x7e6, 0x016, 0x416, 0x216, 0x616, 0x116, 0x516, 0x316, 0x716, 0x096,
   0x496, 0x296, 0x696, 0x196, 0x596, 0x396, 0x796, 0x056, 0x456, 0x256,
   0x656, 0x156, 0x556, 0x356, 0x756, 0x0d6, 0x4d6, 0x2d6, 0x6d6, 0x1d6,
   0x5d6, 0x3d6, 0x7d6, 0x036, 0x436, 0x236, 0x636, 0x136, 0x536, 0x336,
   0x736, 0x0b6, 0x4b6, 0x2b6, 0x6b6, 0x1b6, 0x5b6, 0x3b6, 0x7b6, 0x076,
   0x476, 0x276, 0x676, 0x176, 0x576, 0x376, 0x776, 0x0f6, 0x4f6, 0x2f6,
   0x6f6, 0x1f6, 0x5f6, 0x3f6, 0x7f6, 0x00e, 0x40e, 0x20e, 0x60e, 0x10e,
   0x50e, 0x30e, 0x70e, 0x08e, 0x48e, 0x28e, 0x68e, 0x18e, 0x58e, 0x38e,
   0x78e, 0x04e, 0x44e, 0x24e, 0x64e, 0x14e, 0x54e, 0x34e, 0x74e, 0x0ce,
   0x4ce, 0x2ce, 0x6ce, 0x1ce, 0x5ce, 0x3ce, 0x7ce, 0x02e, 0x42e, 0x22e,
   0x62e, 0x12e, 0x52e, 0x32e, 0x72e, 0x0ae, 0x4ae, 0x2ae, 0x6ae, 0x1ae,
   0x5ae, 0x3ae, 0x7ae, 0x06e, 0x46e, 0x26e, 0x66e, 0x16e, 0x56e, 0x36e,
   0x76e, 0x0ee, 0x4ee, 0x2ee, 0x6ee, 0x1ee, 0x5ee, 0x3ee, 0x7ee, 0x01e,
   0x41e, 0x21e, 0x61e, 0x11e, 0x51e, 0x31e, 0x71e, 0x09e, 0x49e, 0x29e,
   0x69e, 0x19e, 0x59e, 0x39e, 0x79e, 0x05e, 0x45e, 0x25e, 0x65e, 0x15e,
   0x55e, 0x35e, 0x75e, 0x0de, 0x4de, 0x2de, 0x6de, 0x1de, 0x5de, 0x3de,
   0x7de, 0x03e, 0x43e, 0x23e, 0x63e, 0x13e, 0x53e, 0x33e, 0x73e, 0x0be,
   0x4be, 0x2be, 0x6be, 0x1be, 0x5be, 0x3be, 0x7be, 0x07e, 0x47e, 0x27e,
   0x67e, 0x17e, 0x57e, 0x37e, 0x77e, 0x0fe, 0x4fe, 0x2fe, 0x6fe, 0x1fe,
   0x5fe, 0x3fe, 0x7fe, 0x001
};
#endif

/**
  @brief  Double Precision Floating-point Twiddle factors Table Generation
*/

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_16)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(double)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(double)N);
  } </pre>
  @par
  where N = 16, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_16[32] = {
    0x3ff0000000000000,    0x0000000000000000,     //       1,       0'
    0x3fed906bcf328d46,    0x3fd87de2a6aea963,     // 0.92388, 0.38268'
    0x3fe6a09e667f3bcc,    0x3fe6a09e667f3bcc,     // 0.70711, 0.70711'
    0x3fd87de2a6aea963,    0x3fed906bcf328d46,     // 0.38268, 0.92388'
    0x0000000000000000,    0x3ff0000000000000,     //       0,       1'
    0xbfd87de2a6aea963,    0x3fed906bcf328d46,     //-0.38268, 0.92388'
    0xbfe6a09e667f3bcc,    0x3fe6a09e667f3bcc,     //-0.70711, 0.70711'
    0xbfed906bcf328d46,    0x3fd87de2a6aea963,     //-0.92388, 0.38268'
    0xbff0000000000000,    0x0000000000000000,     //      -1,       0'
    0xbfed906bcf328d46,    0xbfd87de2a6aea963,     //-0.92388,-0.38268'
    0xbfe6a09e667f3bcc,    0xbfe6a09e667f3bcc,     //-0.70711,-0.70711'
    0xbfd87de2a6aea963,    0xbfed906bcf328d46,     //-0.38268,-0.92388'
    0x0000000000000000,    0xbff0000000000000,     //       0,      -1'
    0x3fd87de2a6aea963,    0xbfed906bcf328d46,     // 0.38268,-0.92388'
    0x3fe6a09e667f3bcc,    0xbfe6a09e667f3bcc,     // 0.70711,-0.70711'
    0x3fed906bcf328d46,    0xbfd87de2a6aea963,     // 0.92388,-0.38268'
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_32)

/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/N);
  } </pre>
  @par
  where N = 32, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_32[64] = {
    0x3ff0000000000000,     0x0000000000000000,     //       1,       0'
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     // 0.98079, 0.19509'
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     // 0.92388, 0.38268'
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     // 0.83147, 0.55557'
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     // 0.70711, 0.70711'
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     // 0.55557, 0.83147'
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     // 0.38268, 0.92388'
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     // 0.19509, 0.98079'
    0x0000000000000000,     0x3ff0000000000000,     //       0,       1'
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     //-0.19509, 0.98079'
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     //-0.38268, 0.92388'
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //-0.55557, 0.83147'
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //-0.70711, 0.70711'
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //-0.83147, 0.55557'
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     //-0.92388, 0.38268'
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     //-0.98079, 0.19509'
    0xbff0000000000000,     0x0000000000000000,     //      -1,       0'
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     //-0.98079,-0.19509'
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     //-0.92388,-0.38268'
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //-0.83147,-0.55557'
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //-0.70711,-0.70711'
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //-0.55557,-0.83147'
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     //-0.38268,-0.92388'
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     //-0.19509,-0.98079'
    0x0000000000000000,     0xbff0000000000000,     //       0,      -1'
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     // 0.19509,-0.98079'
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     // 0.38268,-0.92388'
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     // 0.55557,-0.83147'
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     // 0.70711,-0.70711'
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     // 0.83147,-0.55557'
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     // 0.92388,-0.38268'
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     // 0.98079,-0.19509'
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_64)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 64, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_64[128] = {
    0x3ff0000000000000,     0x0000000000000000,     //        1,        0'
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //  0.99518, 0.098017'
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  0.98079,  0.19509'
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //  0.95694,  0.29028'
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //  0.92388,  0.38268'
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  0.88192,   0.4714'
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  0.83147,  0.55557'
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //  0.77301,  0.63439'
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  0.70711,  0.70711'
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //  0.63439,  0.77301'
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  0.55557,  0.83147'
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   0.4714,  0.88192'
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //  0.38268,  0.92388'
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //  0.29028,  0.95694'
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  0.19509,  0.98079'
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     // 0.098017,  0.99518'
    0x0000000000000000,     0x3ff0000000000000,     //        0,        1'
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     //-0.098017,  0.99518'
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     // -0.19509,  0.98079'
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     // -0.29028,  0.95694'
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     // -0.38268,  0.92388'
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //  -0.4714,  0.88192'
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     // -0.55557,  0.83147'
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     // -0.63439,  0.77301'
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     // -0.70711,  0.70711'
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     // -0.77301,  0.63439'
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     // -0.83147,  0.55557'
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     // -0.88192,   0.4714'
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     // -0.92388,  0.38268'
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     // -0.95694,  0.29028'
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     // -0.98079,  0.19509'
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     // -0.99518, 0.098017'
    0xbff0000000000000,     0x0000000000000000,     //       -1,        0'
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     // -0.99518,-0.098017'
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     // -0.98079, -0.19509'
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     // -0.95694, -0.29028'
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     // -0.92388, -0.38268'
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     // -0.88192,  -0.4714'
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     // -0.83147, -0.55557'
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     // -0.77301, -0.63439'
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     // -0.70711, -0.70711'
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     // -0.63439, -0.77301'
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     // -0.55557, -0.83147'
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //  -0.4714, -0.88192'
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     // -0.38268, -0.92388'
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     // -0.29028, -0.95694'
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     // -0.19509, -0.98079'
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     //-0.098017, -0.99518'
    0x0000000000000000,     0xbff0000000000000,     //        0,       -1'
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     // 0.098017, -0.99518'
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  0.19509, -0.98079'
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //  0.29028, -0.95694'
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //  0.38268, -0.92388'
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   0.4714, -0.88192'
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  0.55557, -0.83147'
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //  0.63439, -0.77301'
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  0.70711, -0.70711'
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //  0.77301, -0.63439'
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  0.83147, -0.55557'
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  0.88192,  -0.4714'
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //  0.92388, -0.38268'
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //  0.95694, -0.29028'
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  0.98079, -0.19509'
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //  0.99518,-0.098017'
    };

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_128)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 128, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_128[256] = {
    0x3ff0000000000000,     0x0000000000000000,     //        1,        0'
    0x3feff621e3796d7e,     0x3fa91f65f10dd814,     //   0.9988, 0.049068'
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //  0.99518, 0.098017'
    0x3fefa7557f08a517,     0x3fc2c8106e8e613a,     //  0.98918,  0.14673'
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  0.98079,  0.19509'
    0x3fef0a7efb9230d7,     0x3fcf19f97b215f1a,     //  0.97003,  0.24298'
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //  0.95694,  0.29028'
    0x3fee212104f686e5,     0x3fd58f9a75ab1fdd,     //  0.94154,  0.33689'
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //  0.92388,  0.38268'
    0x3feced7af43cc773,     0x3fdb5d1009e15cc0,     //  0.90399,  0.42756'
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  0.88192,   0.4714'
    0x3feb728345196e3e,     0x3fe073879922ffed,     //  0.85773,   0.5141'
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  0.83147,  0.55557'
    0x3fe9b3e047f38741,     0x3fe30ff7fce17035,     //  0.80321,   0.5957'
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //  0.77301,  0.63439'
    0x3fe7b5df226aafb0,     0x3fe57d69348cec9f,     //  0.74095,  0.67156'
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  0.70711,  0.70711'
    0x3fe57d69348cec9f,     0x3fe7b5df226aafb0,     //  0.67156,  0.74095'
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //  0.63439,  0.77301'
    0x3fe30ff7fce17035,     0x3fe9b3e047f38741,     //   0.5957,  0.80321'
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  0.55557,  0.83147'
    0x3fe073879922ffed,     0x3feb728345196e3e,     //   0.5141,  0.85773'
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   0.4714,  0.88192'
    0x3fdb5d1009e15cc0,     0x3feced7af43cc773,     //  0.42756,  0.90399'
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //  0.38268,  0.92388'
    0x3fd58f9a75ab1fdd,     0x3fee212104f686e5,     //  0.33689,  0.94154'
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //  0.29028,  0.95694'
    0x3fcf19f97b215f1a,     0x3fef0a7efb9230d7,     //  0.24298,  0.97003'
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  0.19509,  0.98079'
    0x3fc2c8106e8e613a,     0x3fefa7557f08a517,     //  0.14673,  0.98918'
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     // 0.098017,  0.99518'
    0x3fa91f65f10dd814,     0x3feff621e3796d7e,     // 0.049068,   0.9988'
    0x0000000000000000,     0x3ff0000000000000,     //        0,        1'
    0xbfa91f65f10dd814,     0x3feff621e3796d7e,     //-0.049068,   0.9988'
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     //-0.098017,  0.99518'
    0xbfc2c8106e8e613a,     0x3fefa7557f08a517,     // -0.14673,  0.98918'
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     // -0.19509,  0.98079'
    0xbfcf19f97b215f1a,     0x3fef0a7efb9230d7,     // -0.24298,  0.97003'
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     // -0.29028,  0.95694'
    0xbfd58f9a75ab1fdd,     0x3fee212104f686e5,     // -0.33689,  0.94154'
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     // -0.38268,  0.92388'
    0xbfdb5d1009e15cc0,     0x3feced7af43cc773,     // -0.42756,  0.90399'
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //  -0.4714,  0.88192'
    0xbfe073879922ffed,     0x3feb728345196e3e,     //  -0.5141,  0.85773'
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     // -0.55557,  0.83147'
    0xbfe30ff7fce17035,     0x3fe9b3e047f38741,     //  -0.5957,  0.80321'
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     // -0.63439,  0.77301'
    0xbfe57d69348cec9f,     0x3fe7b5df226aafb0,     // -0.67156,  0.74095'
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     // -0.70711,  0.70711'
    0xbfe7b5df226aafb0,     0x3fe57d69348cec9f,     // -0.74095,  0.67156'
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     // -0.77301,  0.63439'
    0xbfe9b3e047f38741,     0x3fe30ff7fce17035,     // -0.80321,   0.5957'
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     // -0.83147,  0.55557'
    0xbfeb728345196e3e,     0x3fe073879922ffed,     // -0.85773,   0.5141'
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     // -0.88192,   0.4714'
    0xbfeced7af43cc773,     0x3fdb5d1009e15cc0,     // -0.90399,  0.42756'
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     // -0.92388,  0.38268'
    0xbfee212104f686e5,     0x3fd58f9a75ab1fdd,     // -0.94154,  0.33689'
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     // -0.95694,  0.29028'
    0xbfef0a7efb9230d7,     0x3fcf19f97b215f1a,     // -0.97003,  0.24298'
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     // -0.98079,  0.19509'
    0xbfefa7557f08a517,     0x3fc2c8106e8e613a,     // -0.98918,  0.14673'
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     // -0.99518, 0.098017'
    0xbfeff621e3796d7e,     0x3fa91f65f10dd814,     //  -0.9988, 0.049068'
    0xbff0000000000000,     0x0000000000000000,     //       -1,        0'
    0xbfeff621e3796d7e,     0xbfa91f65f10dd814,     //  -0.9988,-0.049068'
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     // -0.99518,-0.098017'
    0xbfefa7557f08a517,     0xbfc2c8106e8e613a,     // -0.98918, -0.14673'
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     // -0.98079, -0.19509'
    0xbfef0a7efb9230d7,     0xbfcf19f97b215f1a,     // -0.97003, -0.24298'
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     // -0.95694, -0.29028'
    0xbfee212104f686e5,     0xbfd58f9a75ab1fdd,     // -0.94154, -0.33689'
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     // -0.92388, -0.38268'
    0xbfeced7af43cc773,     0xbfdb5d1009e15cc0,     // -0.90399, -0.42756'
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     // -0.88192,  -0.4714'
    0xbfeb728345196e3e,     0xbfe073879922ffed,     // -0.85773,  -0.5141'
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     // -0.83147, -0.55557'
    0xbfe9b3e047f38741,     0xbfe30ff7fce17035,     // -0.80321,  -0.5957'
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     // -0.77301, -0.63439'
    0xbfe7b5df226aafb0,     0xbfe57d69348cec9f,     // -0.74095, -0.67156'
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     // -0.70711, -0.70711'
    0xbfe57d69348cec9f,     0xbfe7b5df226aafb0,     // -0.67156, -0.74095'
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     // -0.63439, -0.77301'
    0xbfe30ff7fce17035,     0xbfe9b3e047f38741,     //  -0.5957, -0.80321'
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     // -0.55557, -0.83147'
    0xbfe073879922ffed,     0xbfeb728345196e3e,     //  -0.5141, -0.85773'
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //  -0.4714, -0.88192'
    0xbfdb5d1009e15cc0,     0xbfeced7af43cc773,     // -0.42756, -0.90399'
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     // -0.38268, -0.92388'
    0xbfd58f9a75ab1fdd,     0xbfee212104f686e5,     // -0.33689, -0.94154'
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     // -0.29028, -0.95694'
    0xbfcf19f97b215f1a,     0xbfef0a7efb9230d7,     // -0.24298, -0.97003'
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     // -0.19509, -0.98079'
    0xbfc2c8106e8e613a,     0xbfefa7557f08a517,     // -0.14673, -0.98918'
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     //-0.098017, -0.99518'
    0xbfa91f65f10dd814,     0xbfeff621e3796d7e,     //-0.049068,  -0.9988'
    0x0000000000000000,     0xbff0000000000000,     //        0,       -1'
    0x3fa91f65f10dd814,     0xbfeff621e3796d7e,     // 0.049068,  -0.9988'
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     // 0.098017, -0.99518'
    0x3fc2c8106e8e613a,     0xbfefa7557f08a517,     //  0.14673, -0.98918'
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  0.19509, -0.98079'
    0x3fcf19f97b215f1a,     0xbfef0a7efb9230d7,     //  0.24298, -0.97003'
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //  0.29028, -0.95694'
    0x3fd58f9a75ab1fdd,     0xbfee212104f686e5,     //  0.33689, -0.94154'
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //  0.38268, -0.92388'
    0x3fdb5d1009e15cc0,     0xbfeced7af43cc773,     //  0.42756, -0.90399'
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   0.4714, -0.88192'
    0x3fe073879922ffed,     0xbfeb728345196e3e,     //   0.5141, -0.85773'
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  0.55557, -0.83147'
    0x3fe30ff7fce17035,     0xbfe9b3e047f38741,     //   0.5957, -0.80321'
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //  0.63439, -0.77301'
    0x3fe57d69348cec9f,     0xbfe7b5df226aafb0,     //  0.67156, -0.74095'
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  0.70711, -0.70711'
    0x3fe7b5df226aafb0,     0xbfe57d69348cec9f,     //  0.74095, -0.67156'
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //  0.77301, -0.63439'
    0x3fe9b3e047f38741,     0xbfe30ff7fce17035,     //  0.80321,  -0.5957'
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  0.83147, -0.55557'
    0x3feb728345196e3e,     0xbfe073879922ffed,     //  0.85773,  -0.5141'
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  0.88192,  -0.4714'
    0x3feced7af43cc773,     0xbfdb5d1009e15cc0,     //  0.90399, -0.42756'
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //  0.92388, -0.38268'
    0x3fee212104f686e5,     0xbfd58f9a75ab1fdd,     //  0.94154, -0.33689'
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //  0.95694, -0.29028'
    0x3fef0a7efb9230d7,     0xbfcf19f97b215f1a,     //  0.97003, -0.24298'
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  0.98079, -0.19509'
    0x3fefa7557f08a517,     0xbfc2c8106e8e613a,     //  0.98918, -0.14673'
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //  0.99518,-0.098017'
    0x3feff621e3796d7e,     0xbfa91f65f10dd814,     //   0.9988,-0.049068'
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_256)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 256, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_256[512] = {
    0x3ff0000000000000,     0x0000000000000000,     //        1,        0
    0x3feffd886084cd0d,     0x3f992155f7a3667e,     //   0.9997, 0.024541
    0x3feff621e3796d7e,     0x3fa91f65f10dd814,     //   0.9988, 0.049068
    0x3fefe9cdad01883a,     0x3fb2d52092ce19f6,     //  0.99729, 0.073565
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //  0.99518, 0.098017
    0x3fefc26470e19fd3,     0x3fbf564e56a9730e,     //  0.99248,  0.12241
    0x3fefa7557f08a517,     0x3fc2c8106e8e613a,     //  0.98918,  0.14673
    0x3fef8764fa714ba9,     0x3fc5e214448b3fc6,     //  0.98528,  0.17096
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  0.98079,  0.19509
    0x3fef38f3ac64e589,     0x3fcc0b826a7e4f63,     //   0.9757,   0.2191
    0x3fef0a7efb9230d7,     0x3fcf19f97b215f1a,     //  0.97003,  0.24298
    0x3feed740e7684963,     0x3fd111d262b1f677,     //  0.96378,  0.26671
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //  0.95694,  0.29028
    0x3fee6288ec48e112,     0x3fd4135c94176602,     //  0.94953,  0.31368
    0x3fee212104f686e5,     0x3fd58f9a75ab1fdd,     //  0.94154,  0.33689
    0x3feddb13b6ccc23d,     0x3fd7088530fa459e,     //  0.93299,   0.3599
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //  0.92388,  0.38268
    0x3fed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //  0.91421,  0.40524
    0x3feced7af43cc773,     0x3fdb5d1009e15cc0,     //  0.90399,  0.42756
    0x3fec954b213411f5,     0x3fdcc66e9931c45d,     //  0.89322,  0.44961
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  0.88192,   0.4714
    0x3febd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //  0.87009,   0.4929
    0x3feb728345196e3e,     0x3fe073879922ffed,     //  0.85773,   0.5141
    0x3feb090a58150200,     0x3fe11eb3541b4b22,     //  0.84485,    0.535
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  0.83147,  0.55557
    0x3fea29a7a0462782,     0x3fe26d054cdd12df,     //  0.81758,  0.57581
    0x3fe9b3e047f38741,     0x3fe30ff7fce17035,     //  0.80321,   0.5957
    0x3fe93a22499263fc,     0x3fe3affa292050b9,     //  0.78835,  0.61523
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //  0.77301,  0.63439
    0x3fe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //  0.75721,  0.65317
    0x3fe7b5df226aafb0,     0x3fe57d69348cec9f,     //  0.74095,  0.67156
    0x3fe72d0837efff97,     0x3fe610b7551d2cde,     //  0.72425,  0.68954
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  0.70711,  0.70711
    0x3fe610b7551d2cde,     0x3fe72d0837efff97,     //  0.68954,  0.72425
    0x3fe57d69348cec9f,     0x3fe7b5df226aafb0,     //  0.67156,  0.74095
    0x3fe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //  0.65317,  0.75721
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //  0.63439,  0.77301
    0x3fe3affa292050b9,     0x3fe93a22499263fc,     //  0.61523,  0.78835
    0x3fe30ff7fce17035,     0x3fe9b3e047f38741,     //   0.5957,  0.80321
    0x3fe26d054cdd12df,     0x3fea29a7a0462782,     //  0.57581,  0.81758
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  0.55557,  0.83147
    0x3fe11eb3541b4b22,     0x3feb090a58150200,     //    0.535,  0.84485
    0x3fe073879922ffed,     0x3feb728345196e3e,     //   0.5141,  0.85773
    0x3fdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //   0.4929,  0.87009
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   0.4714,  0.88192
    0x3fdcc66e9931c45d,     0x3fec954b213411f5,     //  0.44961,  0.89322
    0x3fdb5d1009e15cc0,     0x3feced7af43cc773,     //  0.42756,  0.90399
    0x3fd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //  0.40524,  0.91421
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //  0.38268,  0.92388
    0x3fd7088530fa459e,     0x3feddb13b6ccc23d,     //   0.3599,  0.93299
    0x3fd58f9a75ab1fdd,     0x3fee212104f686e5,     //  0.33689,  0.94154
    0x3fd4135c94176602,     0x3fee6288ec48e112,     //  0.31368,  0.94953
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //  0.29028,  0.95694
    0x3fd111d262b1f677,     0x3feed740e7684963,     //  0.26671,  0.96378
    0x3fcf19f97b215f1a,     0x3fef0a7efb9230d7,     //  0.24298,  0.97003
    0x3fcc0b826a7e4f63,     0x3fef38f3ac64e589,     //   0.2191,   0.9757
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  0.19509,  0.98079
    0x3fc5e214448b3fc6,     0x3fef8764fa714ba9,     //  0.17096,  0.98528
    0x3fc2c8106e8e613a,     0x3fefa7557f08a517,     //  0.14673,  0.98918
    0x3fbf564e56a9730e,     0x3fefc26470e19fd3,     //  0.12241,  0.99248
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     // 0.098017,  0.99518
    0x3fb2d52092ce19f6,     0x3fefe9cdad01883a,     // 0.073565,  0.99729
    0x3fa91f65f10dd814,     0x3feff621e3796d7e,     // 0.049068,   0.9988
    0x3f992155f7a3667e,     0x3feffd886084cd0d,     // 0.024541,   0.9997
    0x0000000000000000,     0x3ff0000000000000,     //        0,        1
    0xbf992155f7a3667e,     0x3feffd886084cd0d,     //-0.024541,   0.9997
    0xbfa91f65f10dd814,     0x3feff621e3796d7e,     //-0.049068,   0.9988
    0xbfb2d52092ce19f6,     0x3fefe9cdad01883a,     //-0.073565,  0.99729
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     //-0.098017,  0.99518
    0xbfbf564e56a9730e,     0x3fefc26470e19fd3,     // -0.12241,  0.99248
    0xbfc2c8106e8e613a,     0x3fefa7557f08a517,     // -0.14673,  0.98918
    0xbfc5e214448b3fc6,     0x3fef8764fa714ba9,     // -0.17096,  0.98528
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     // -0.19509,  0.98079
    0xbfcc0b826a7e4f63,     0x3fef38f3ac64e589,     //  -0.2191,   0.9757
    0xbfcf19f97b215f1a,     0x3fef0a7efb9230d7,     // -0.24298,  0.97003
    0xbfd111d262b1f677,     0x3feed740e7684963,     // -0.26671,  0.96378
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     // -0.29028,  0.95694
    0xbfd4135c94176602,     0x3fee6288ec48e112,     // -0.31368,  0.94953
    0xbfd58f9a75ab1fdd,     0x3fee212104f686e5,     // -0.33689,  0.94154
    0xbfd7088530fa459e,     0x3feddb13b6ccc23d,     //  -0.3599,  0.93299
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     // -0.38268,  0.92388
    0xbfd9ef7943a8ed8a,     0x3fed4134d14dc93a,     // -0.40524,  0.91421
    0xbfdb5d1009e15cc0,     0x3feced7af43cc773,     // -0.42756,  0.90399
    0xbfdcc66e9931c45d,     0x3fec954b213411f5,     // -0.44961,  0.89322
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //  -0.4714,  0.88192
    0xbfdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //  -0.4929,  0.87009
    0xbfe073879922ffed,     0x3feb728345196e3e,     //  -0.5141,  0.85773
    0xbfe11eb3541b4b22,     0x3feb090a58150200,     //   -0.535,  0.84485
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     // -0.55557,  0.83147
    0xbfe26d054cdd12df,     0x3fea29a7a0462782,     // -0.57581,  0.81758
    0xbfe30ff7fce17035,     0x3fe9b3e047f38741,     //  -0.5957,  0.80321
    0xbfe3affa292050b9,     0x3fe93a22499263fc,     // -0.61523,  0.78835
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     // -0.63439,  0.77301
    0xbfe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     // -0.65317,  0.75721
    0xbfe57d69348cec9f,     0x3fe7b5df226aafb0,     // -0.67156,  0.74095
    0xbfe610b7551d2cde,     0x3fe72d0837efff97,     // -0.68954,  0.72425
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     // -0.70711,  0.70711
    0xbfe72d0837efff97,     0x3fe610b7551d2cde,     // -0.72425,  0.68954
    0xbfe7b5df226aafb0,     0x3fe57d69348cec9f,     // -0.74095,  0.67156
    0xbfe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     // -0.75721,  0.65317
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     // -0.77301,  0.63439
    0xbfe93a22499263fc,     0x3fe3affa292050b9,     // -0.78835,  0.61523
    0xbfe9b3e047f38741,     0x3fe30ff7fce17035,     // -0.80321,   0.5957
    0xbfea29a7a0462782,     0x3fe26d054cdd12df,     // -0.81758,  0.57581
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     // -0.83147,  0.55557
    0xbfeb090a58150200,     0x3fe11eb3541b4b22,     // -0.84485,    0.535
    0xbfeb728345196e3e,     0x3fe073879922ffed,     // -0.85773,   0.5141
    0xbfebd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     // -0.87009,   0.4929
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     // -0.88192,   0.4714
    0xbfec954b213411f5,     0x3fdcc66e9931c45d,     // -0.89322,  0.44961
    0xbfeced7af43cc773,     0x3fdb5d1009e15cc0,     // -0.90399,  0.42756
    0xbfed4134d14dc93a,     0x3fd9ef7943a8ed8a,     // -0.91421,  0.40524
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     // -0.92388,  0.38268
    0xbfeddb13b6ccc23d,     0x3fd7088530fa459e,     // -0.93299,   0.3599
    0xbfee212104f686e5,     0x3fd58f9a75ab1fdd,     // -0.94154,  0.33689
    0xbfee6288ec48e112,     0x3fd4135c94176602,     // -0.94953,  0.31368
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     // -0.95694,  0.29028
    0xbfeed740e7684963,     0x3fd111d262b1f677,     // -0.96378,  0.26671
    0xbfef0a7efb9230d7,     0x3fcf19f97b215f1a,     // -0.97003,  0.24298
    0xbfef38f3ac64e589,     0x3fcc0b826a7e4f63,     //  -0.9757,   0.2191
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     // -0.98079,  0.19509
    0xbfef8764fa714ba9,     0x3fc5e214448b3fc6,     // -0.98528,  0.17096
    0xbfefa7557f08a517,     0x3fc2c8106e8e613a,     // -0.98918,  0.14673
    0xbfefc26470e19fd3,     0x3fbf564e56a9730e,     // -0.99248,  0.12241
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     // -0.99518, 0.098017
    0xbfefe9cdad01883a,     0x3fb2d52092ce19f6,     // -0.99729, 0.073565
    0xbfeff621e3796d7e,     0x3fa91f65f10dd814,     //  -0.9988, 0.049068
    0xbfeffd886084cd0d,     0x3f992155f7a3667e,     //  -0.9997, 0.024541
    0xbff0000000000000,     0x0000000000000000,     //       -1,        0
    0xbfeffd886084cd0d,     0xbf992155f7a3667e,     //  -0.9997,-0.024541
    0xbfeff621e3796d7e,     0xbfa91f65f10dd814,     //  -0.9988,-0.049068
    0xbfefe9cdad01883a,     0xbfb2d52092ce19f6,     // -0.99729,-0.073565
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     // -0.99518,-0.098017
    0xbfefc26470e19fd3,     0xbfbf564e56a9730e,     // -0.99248, -0.12241
    0xbfefa7557f08a517,     0xbfc2c8106e8e613a,     // -0.98918, -0.14673
    0xbfef8764fa714ba9,     0xbfc5e214448b3fc6,     // -0.98528, -0.17096
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     // -0.98079, -0.19509
    0xbfef38f3ac64e589,     0xbfcc0b826a7e4f63,     //  -0.9757,  -0.2191
    0xbfef0a7efb9230d7,     0xbfcf19f97b215f1a,     // -0.97003, -0.24298
    0xbfeed740e7684963,     0xbfd111d262b1f677,     // -0.96378, -0.26671
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     // -0.95694, -0.29028
    0xbfee6288ec48e112,     0xbfd4135c94176602,     // -0.94953, -0.31368
    0xbfee212104f686e5,     0xbfd58f9a75ab1fdd,     // -0.94154, -0.33689
    0xbfeddb13b6ccc23d,     0xbfd7088530fa459e,     // -0.93299,  -0.3599
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     // -0.92388, -0.38268
    0xbfed4134d14dc93a,     0xbfd9ef7943a8ed8a,     // -0.91421, -0.40524
    0xbfeced7af43cc773,     0xbfdb5d1009e15cc0,     // -0.90399, -0.42756
    0xbfec954b213411f5,     0xbfdcc66e9931c45d,     // -0.89322, -0.44961
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     // -0.88192,  -0.4714
    0xbfebd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     // -0.87009,  -0.4929
    0xbfeb728345196e3e,     0xbfe073879922ffed,     // -0.85773,  -0.5141
    0xbfeb090a58150200,     0xbfe11eb3541b4b22,     // -0.84485,   -0.535
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     // -0.83147, -0.55557
    0xbfea29a7a0462782,     0xbfe26d054cdd12df,     // -0.81758, -0.57581
    0xbfe9b3e047f38741,     0xbfe30ff7fce17035,     // -0.80321,  -0.5957
    0xbfe93a22499263fc,     0xbfe3affa292050b9,     // -0.78835, -0.61523
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     // -0.77301, -0.63439
    0xbfe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     // -0.75721, -0.65317
    0xbfe7b5df226aafb0,     0xbfe57d69348cec9f,     // -0.74095, -0.67156
    0xbfe72d0837efff97,     0xbfe610b7551d2cde,     // -0.72425, -0.68954
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     // -0.70711, -0.70711
    0xbfe610b7551d2cde,     0xbfe72d0837efff97,     // -0.68954, -0.72425
    0xbfe57d69348cec9f,     0xbfe7b5df226aafb0,     // -0.67156, -0.74095
    0xbfe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     // -0.65317, -0.75721
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     // -0.63439, -0.77301
    0xbfe3affa292050b9,     0xbfe93a22499263fc,     // -0.61523, -0.78835
    0xbfe30ff7fce17035,     0xbfe9b3e047f38741,     //  -0.5957, -0.80321
    0xbfe26d054cdd12df,     0xbfea29a7a0462782,     // -0.57581, -0.81758
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     // -0.55557, -0.83147
    0xbfe11eb3541b4b22,     0xbfeb090a58150200,     //   -0.535, -0.84485
    0xbfe073879922ffed,     0xbfeb728345196e3e,     //  -0.5141, -0.85773
    0xbfdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //  -0.4929, -0.87009
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //  -0.4714, -0.88192
    0xbfdcc66e9931c45d,     0xbfec954b213411f5,     // -0.44961, -0.89322
    0xbfdb5d1009e15cc0,     0xbfeced7af43cc773,     // -0.42756, -0.90399
    0xbfd9ef7943a8ed8a,     0xbfed4134d14dc93a,     // -0.40524, -0.91421
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     // -0.38268, -0.92388
    0xbfd7088530fa459e,     0xbfeddb13b6ccc23d,     //  -0.3599, -0.93299
    0xbfd58f9a75ab1fdd,     0xbfee212104f686e5,     // -0.33689, -0.94154
    0xbfd4135c94176602,     0xbfee6288ec48e112,     // -0.31368, -0.94953
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     // -0.29028, -0.95694
    0xbfd111d262b1f677,     0xbfeed740e7684963,     // -0.26671, -0.96378
    0xbfcf19f97b215f1a,     0xbfef0a7efb9230d7,     // -0.24298, -0.97003
    0xbfcc0b826a7e4f63,     0xbfef38f3ac64e589,     //  -0.2191,  -0.9757
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     // -0.19509, -0.98079
    0xbfc5e214448b3fc6,     0xbfef8764fa714ba9,     // -0.17096, -0.98528
    0xbfc2c8106e8e613a,     0xbfefa7557f08a517,     // -0.14673, -0.98918
    0xbfbf564e56a9730e,     0xbfefc26470e19fd3,     // -0.12241, -0.99248
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     //-0.098017, -0.99518
    0xbfb2d52092ce19f6,     0xbfefe9cdad01883a,     //-0.073565, -0.99729
    0xbfa91f65f10dd814,     0xbfeff621e3796d7e,     //-0.049068,  -0.9988
    0xbf992155f7a3667e,     0xbfeffd886084cd0d,     //-0.024541,  -0.9997
    0x0000000000000000,     0xbff0000000000000,     //        0,       -1
    0x3f992155f7a3667e,     0xbfeffd886084cd0d,     // 0.024541,  -0.9997
    0x3fa91f65f10dd814,     0xbfeff621e3796d7e,     // 0.049068,  -0.9988
    0x3fb2d52092ce19f6,     0xbfefe9cdad01883a,     // 0.073565, -0.99729
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     // 0.098017, -0.99518
    0x3fbf564e56a9730e,     0xbfefc26470e19fd3,     //  0.12241, -0.99248
    0x3fc2c8106e8e613a,     0xbfefa7557f08a517,     //  0.14673, -0.98918
    0x3fc5e214448b3fc6,     0xbfef8764fa714ba9,     //  0.17096, -0.98528
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  0.19509, -0.98079
    0x3fcc0b826a7e4f63,     0xbfef38f3ac64e589,     //   0.2191,  -0.9757
    0x3fcf19f97b215f1a,     0xbfef0a7efb9230d7,     //  0.24298, -0.97003
    0x3fd111d262b1f677,     0xbfeed740e7684963,     //  0.26671, -0.96378
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //  0.29028, -0.95694
    0x3fd4135c94176602,     0xbfee6288ec48e112,     //  0.31368, -0.94953
    0x3fd58f9a75ab1fdd,     0xbfee212104f686e5,     //  0.33689, -0.94154
    0x3fd7088530fa459e,     0xbfeddb13b6ccc23d,     //   0.3599, -0.93299
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //  0.38268, -0.92388
    0x3fd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //  0.40524, -0.91421
    0x3fdb5d1009e15cc0,     0xbfeced7af43cc773,     //  0.42756, -0.90399
    0x3fdcc66e9931c45d,     0xbfec954b213411f5,     //  0.44961, -0.89322
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   0.4714, -0.88192
    0x3fdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //   0.4929, -0.87009
    0x3fe073879922ffed,     0xbfeb728345196e3e,     //   0.5141, -0.85773
    0x3fe11eb3541b4b22,     0xbfeb090a58150200,     //    0.535, -0.84485
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  0.55557, -0.83147
    0x3fe26d054cdd12df,     0xbfea29a7a0462782,     //  0.57581, -0.81758
    0x3fe30ff7fce17035,     0xbfe9b3e047f38741,     //   0.5957, -0.80321
    0x3fe3affa292050b9,     0xbfe93a22499263fc,     //  0.61523, -0.78835
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //  0.63439, -0.77301
    0x3fe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //  0.65317, -0.75721
    0x3fe57d69348cec9f,     0xbfe7b5df226aafb0,     //  0.67156, -0.74095
    0x3fe610b7551d2cde,     0xbfe72d0837efff97,     //  0.68954, -0.72425
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  0.70711, -0.70711
    0x3fe72d0837efff97,     0xbfe610b7551d2cde,     //  0.72425, -0.68954
    0x3fe7b5df226aafb0,     0xbfe57d69348cec9f,     //  0.74095, -0.67156
    0x3fe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //  0.75721, -0.65317
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //  0.77301, -0.63439
    0x3fe93a22499263fc,     0xbfe3affa292050b9,     //  0.78835, -0.61523
    0x3fe9b3e047f38741,     0xbfe30ff7fce17035,     //  0.80321,  -0.5957
    0x3fea29a7a0462782,     0xbfe26d054cdd12df,     //  0.81758, -0.57581
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  0.83147, -0.55557
    0x3feb090a58150200,     0xbfe11eb3541b4b22,     //  0.84485,   -0.535
    0x3feb728345196e3e,     0xbfe073879922ffed,     //  0.85773,  -0.5141
    0x3febd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //  0.87009,  -0.4929
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  0.88192,  -0.4714
    0x3fec954b213411f5,     0xbfdcc66e9931c45d,     //  0.89322, -0.44961
    0x3feced7af43cc773,     0xbfdb5d1009e15cc0,     //  0.90399, -0.42756
    0x3fed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //  0.91421, -0.40524
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //  0.92388, -0.38268
    0x3feddb13b6ccc23d,     0xbfd7088530fa459e,     //  0.93299,  -0.3599
    0x3fee212104f686e5,     0xbfd58f9a75ab1fdd,     //  0.94154, -0.33689
    0x3fee6288ec48e112,     0xbfd4135c94176602,     //  0.94953, -0.31368
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //  0.95694, -0.29028
    0x3feed740e7684963,     0xbfd111d262b1f677,     //  0.96378, -0.26671
    0x3fef0a7efb9230d7,     0xbfcf19f97b215f1a,     //  0.97003, -0.24298
    0x3fef38f3ac64e589,     0xbfcc0b826a7e4f63,     //   0.9757,  -0.2191
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  0.98079, -0.19509
    0x3fef8764fa714ba9,     0xbfc5e214448b3fc6,     //  0.98528, -0.17096
    0x3fefa7557f08a517,     0xbfc2c8106e8e613a,     //  0.98918, -0.14673
    0x3fefc26470e19fd3,     0xbfbf564e56a9730e,     //  0.99248, -0.12241
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //  0.99518,-0.098017
    0x3fefe9cdad01883a,     0xbfb2d52092ce19f6,     //  0.99729,-0.073565
    0x3feff621e3796d7e,     0xbfa91f65f10dd814,     //   0.9988,-0.049068
    0x3feffd886084cd0d,     0xbf992155f7a3667e,     //   0.9997,-0.024541
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_512)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 512, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_512[1024] = {
    0x3ff0000000000000,     0x0000000000000000,     //        1,        0
    0x3fefff62169b92db,     0x3f8921d1fcdec784,     //  0.99992, 0.012272
    0x3feffd886084cd0d,     0x3f992155f7a3667e,     //   0.9997, 0.024541
    0x3feffa72effef75d,     0x3fa2d865759455cd,     //  0.99932, 0.036807
    0x3feff621e3796d7e,     0x3fa91f65f10dd814,     //   0.9988, 0.049068
    0x3feff095658e71ad,     0x3faf656e79f820e0,     //  0.99812, 0.061321
    0x3fefe9cdad01883a,     0x3fb2d52092ce19f6,     //  0.99729, 0.073565
    0x3fefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     //  0.99631, 0.085797
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //  0.99518, 0.098017
    0x3fefce15fd6da67b,     0x3fbc3785c79ec2d5,     //  0.99391,  0.11022
    0x3fefc26470e19fd3,     0x3fbf564e56a9730e,     //  0.99248,  0.12241
    0x3fefb5797195d741,     0x3fc139f0cedaf576,     //   0.9909,  0.13458
    0x3fefa7557f08a517,     0x3fc2c8106e8e613a,     //  0.98918,  0.14673
    0x3fef97f924c9099b,     0x3fc45576b1293e5a,     //   0.9873,  0.15886
    0x3fef8764fa714ba9,     0x3fc5e214448b3fc6,     //  0.98528,  0.17096
    0x3fef7599a3a12077,     0x3fc76dd9de50bf31,     //  0.98311,  0.18304
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  0.98079,  0.19509
    0x3fef4e603b0b2f2d,     0x3fca82a025b00451,     //  0.97832,  0.20711
    0x3fef38f3ac64e589,     0x3fcc0b826a7e4f63,     //   0.9757,   0.2191
    0x3fef2252f7763ada,     0x3fcd934fe5454311,     //  0.97294,  0.23106
    0x3fef0a7efb9230d7,     0x3fcf19f97b215f1a,     //  0.97003,  0.24298
    0x3feef178a3e473c2,     0x3fd04fb80e37fdae,     //  0.96698,  0.25487
    0x3feed740e7684963,     0x3fd111d262b1f677,     //  0.96378,  0.26671
    0x3feebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     //  0.96043,  0.27852
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //  0.95694,  0.29028
    0x3fee817bab4cd10d,     0x3fd35410c2e18152,     //  0.95331,  0.30201
    0x3fee6288ec48e112,     0x3fd4135c94176602,     //  0.94953,  0.31368
    0x3fee426a4b2bc17e,     0x3fd4d1e24278e76a,     //  0.94561,  0.32531
    0x3fee212104f686e5,     0x3fd58f9a75ab1fdd,     //  0.94154,  0.33689
    0x3fedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     //  0.93734,  0.34842
    0x3feddb13b6ccc23d,     0x3fd7088530fa459e,     //  0.93299,   0.3599
    0x3fedb6526238a09b,     0x3fd7c3a9311dcce7,     //  0.92851,  0.37132
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //  0.92388,  0.38268
    0x3fed696173c9e68b,     0x3fd9372a63bc93d7,     //  0.91911,  0.39399
    0x3fed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //  0.91421,  0.40524
    0x3fed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     //  0.90917,  0.41643
    0x3feced7af43cc773,     0x3fdb5d1009e15cc0,     //  0.90399,  0.42756
    0x3fecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     //  0.89867,  0.43862
    0x3fec954b213411f5,     0x3fdcc66e9931c45d,     //  0.89322,  0.44961
    0x3fec678b3488739b,     0x3fdd79775b86e389,     //  0.88764,  0.46054
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  0.88192,   0.4714
    0x3fec08c426725549,     0x3fdedc1952ef78d5,     //  0.87607,  0.48218
    0x3febd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //  0.87009,   0.4929
    0x3feba5aa673590d2,     0x3fe01cfc874c3eb7,     //  0.86397,  0.50354
    0x3feb728345196e3e,     0x3fe073879922ffed,     //  0.85773,   0.5141
    0x3feb3e4d3ef55712,     0x3fe0c9704d5d898f,     //  0.85136,  0.52459
    0x3feb090a58150200,     0x3fe11eb3541b4b22,     //  0.84485,    0.535
    0x3fead2bc9e21d511,     0x3fe1734d63dedb49,     //  0.83822,  0.54532
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  0.83147,  0.55557
    0x3fea63091b02fae2,     0x3fe21a799933eb58,     //  0.82459,  0.56573
    0x3fea29a7a0462782,     0x3fe26d054cdd12df,     //  0.81758,  0.57581
    0x3fe9ef43ef29af94,     0x3fe2bedb25faf3ea,     //  0.81046,   0.5858
    0x3fe9b3e047f38741,     0x3fe30ff7fce17035,     //  0.80321,   0.5957
    0x3fe9777ef4c7d742,     0x3fe36058b10659f3,     //  0.79584,  0.60551
    0x3fe93a22499263fc,     0x3fe3affa292050b9,     //  0.78835,  0.61523
    0x3fe8fbcca3ef940d,     0x3fe3fed9534556d4,     //  0.78074,  0.62486
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //  0.77301,  0.63439
    0x3fe87c400fba2ebf,     0x3fe49a449b9b0938,     //  0.76517,  0.64383
    0x3fe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //  0.75721,  0.65317
    0x3fe7f8ece3571771,     0x3fe5328292a35596,     //  0.74914,  0.66242
    0x3fe7b5df226aafb0,     0x3fe57d69348cec9f,     //  0.74095,  0.67156
    0x3fe771e75f037261,     0x3fe5c77bbe65018c,     //  0.73265,   0.6806
    0x3fe72d0837efff97,     0x3fe610b7551d2cde,     //  0.72425,  0.68954
    0x3fe6e74454eaa8ae,     0x3fe6591925f0783e,     //  0.71573,  0.69838
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  0.70711,  0.70711
    0x3fe6591925f0783e,     0x3fe6e74454eaa8ae,     //  0.69838,  0.71573
    0x3fe610b7551d2cde,     0x3fe72d0837efff97,     //  0.68954,  0.72425
    0x3fe5c77bbe65018c,     0x3fe771e75f037261,     //   0.6806,  0.73265
    0x3fe57d69348cec9f,     0x3fe7b5df226aafb0,     //  0.67156,  0.74095
    0x3fe5328292a35596,     0x3fe7f8ece3571771,     //  0.66242,  0.74914
    0x3fe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //  0.65317,  0.75721
    0x3fe49a449b9b0938,     0x3fe87c400fba2ebf,     //  0.64383,  0.76517
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //  0.63439,  0.77301
    0x3fe3fed9534556d4,     0x3fe8fbcca3ef940d,     //  0.62486,  0.78074
    0x3fe3affa292050b9,     0x3fe93a22499263fc,     //  0.61523,  0.78835
    0x3fe36058b10659f3,     0x3fe9777ef4c7d742,     //  0.60551,  0.79584
    0x3fe30ff7fce17035,     0x3fe9b3e047f38741,     //   0.5957,  0.80321
    0x3fe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //   0.5858,  0.81046
    0x3fe26d054cdd12df,     0x3fea29a7a0462782,     //  0.57581,  0.81758
    0x3fe21a799933eb58,     0x3fea63091b02fae2,     //  0.56573,  0.82459
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  0.55557,  0.83147
    0x3fe1734d63dedb49,     0x3fead2bc9e21d511,     //  0.54532,  0.83822
    0x3fe11eb3541b4b22,     0x3feb090a58150200,     //    0.535,  0.84485
    0x3fe0c9704d5d898f,     0x3feb3e4d3ef55712,     //  0.52459,  0.85136
    0x3fe073879922ffed,     0x3feb728345196e3e,     //   0.5141,  0.85773
    0x3fe01cfc874c3eb7,     0x3feba5aa673590d2,     //  0.50354,  0.86397
    0x3fdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //   0.4929,  0.87009
    0x3fdedc1952ef78d5,     0x3fec08c426725549,     //  0.48218,  0.87607
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   0.4714,  0.88192
    0x3fdd79775b86e389,     0x3fec678b3488739b,     //  0.46054,  0.88764
    0x3fdcc66e9931c45d,     0x3fec954b213411f5,     //  0.44961,  0.89322
    0x3fdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     //  0.43862,  0.89867
    0x3fdb5d1009e15cc0,     0x3feced7af43cc773,     //  0.42756,  0.90399
    0x3fdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     //  0.41643,  0.90917
    0x3fd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //  0.40524,  0.91421
    0x3fd9372a63bc93d7,     0x3fed696173c9e68b,     //  0.39399,  0.91911
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //  0.38268,  0.92388
    0x3fd7c3a9311dcce7,     0x3fedb6526238a09b,     //  0.37132,  0.92851
    0x3fd7088530fa459e,     0x3feddb13b6ccc23d,     //   0.3599,  0.93299
    0x3fd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     //  0.34842,  0.93734
    0x3fd58f9a75ab1fdd,     0x3fee212104f686e5,     //  0.33689,  0.94154
    0x3fd4d1e24278e76a,     0x3fee426a4b2bc17e,     //  0.32531,  0.94561
    0x3fd4135c94176602,     0x3fee6288ec48e112,     //  0.31368,  0.94953
    0x3fd35410c2e18152,     0x3fee817bab4cd10d,     //  0.30201,  0.95331
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //  0.29028,  0.95694
    0x3fd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     //  0.27852,  0.96043
    0x3fd111d262b1f677,     0x3feed740e7684963,     //  0.26671,  0.96378
    0x3fd04fb80e37fdae,     0x3feef178a3e473c2,     //  0.25487,  0.96698
    0x3fcf19f97b215f1a,     0x3fef0a7efb9230d7,     //  0.24298,  0.97003
    0x3fcd934fe5454311,     0x3fef2252f7763ada,     //  0.23106,  0.97294
    0x3fcc0b826a7e4f63,     0x3fef38f3ac64e589,     //   0.2191,   0.9757
    0x3fca82a025b00451,     0x3fef4e603b0b2f2d,     //  0.20711,  0.97832
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  0.19509,  0.98079
    0x3fc76dd9de50bf31,     0x3fef7599a3a12077,     //  0.18304,  0.98311
    0x3fc5e214448b3fc6,     0x3fef8764fa714ba9,     //  0.17096,  0.98528
    0x3fc45576b1293e5a,     0x3fef97f924c9099b,     //  0.15886,   0.9873
    0x3fc2c8106e8e613a,     0x3fefa7557f08a517,     //  0.14673,  0.98918
    0x3fc139f0cedaf576,     0x3fefb5797195d741,     //  0.13458,   0.9909
    0x3fbf564e56a9730e,     0x3fefc26470e19fd3,     //  0.12241,  0.99248
    0x3fbc3785c79ec2d5,     0x3fefce15fd6da67b,     //  0.11022,  0.99391
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     // 0.098017,  0.99518
    0x3fb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     // 0.085797,  0.99631
    0x3fb2d52092ce19f6,     0x3fefe9cdad01883a,     // 0.073565,  0.99729
    0x3faf656e79f820e0,     0x3feff095658e71ad,     // 0.061321,  0.99812
    0x3fa91f65f10dd814,     0x3feff621e3796d7e,     // 0.049068,   0.9988
    0x3fa2d865759455cd,     0x3feffa72effef75d,     // 0.036807,  0.99932
    0x3f992155f7a3667e,     0x3feffd886084cd0d,     // 0.024541,   0.9997
    0x3f8921d1fcdec784,     0x3fefff62169b92db,     // 0.012272,  0.99992
    0x0000000000000000,     0x3ff0000000000000,     //        0,        1
    0xbf8921d1fcdec784,     0x3fefff62169b92db,     //-0.012272,  0.99992
    0xbf992155f7a3667e,     0x3feffd886084cd0d,     //-0.024541,   0.9997
    0xbfa2d865759455cd,     0x3feffa72effef75d,     //-0.036807,  0.99932
    0xbfa91f65f10dd814,     0x3feff621e3796d7e,     //-0.049068,   0.9988
    0xbfaf656e79f820e0,     0x3feff095658e71ad,     //-0.061321,  0.99812
    0xbfb2d52092ce19f6,     0x3fefe9cdad01883a,     //-0.073565,  0.99729
    0xbfb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     //-0.085797,  0.99631
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     //-0.098017,  0.99518
    0xbfbc3785c79ec2d5,     0x3fefce15fd6da67b,     // -0.11022,  0.99391
    0xbfbf564e56a9730e,     0x3fefc26470e19fd3,     // -0.12241,  0.99248
    0xbfc139f0cedaf576,     0x3fefb5797195d741,     // -0.13458,   0.9909
    0xbfc2c8106e8e613a,     0x3fefa7557f08a517,     // -0.14673,  0.98918
    0xbfc45576b1293e5a,     0x3fef97f924c9099b,     // -0.15886,   0.9873
    0xbfc5e214448b3fc6,     0x3fef8764fa714ba9,     // -0.17096,  0.98528
    0xbfc76dd9de50bf31,     0x3fef7599a3a12077,     // -0.18304,  0.98311
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     // -0.19509,  0.98079
    0xbfca82a025b00451,     0x3fef4e603b0b2f2d,     // -0.20711,  0.97832
    0xbfcc0b826a7e4f63,     0x3fef38f3ac64e589,     //  -0.2191,   0.9757
    0xbfcd934fe5454311,     0x3fef2252f7763ada,     // -0.23106,  0.97294
    0xbfcf19f97b215f1a,     0x3fef0a7efb9230d7,     // -0.24298,  0.97003
    0xbfd04fb80e37fdae,     0x3feef178a3e473c2,     // -0.25487,  0.96698
    0xbfd111d262b1f677,     0x3feed740e7684963,     // -0.26671,  0.96378
    0xbfd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     // -0.27852,  0.96043
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     // -0.29028,  0.95694
    0xbfd35410c2e18152,     0x3fee817bab4cd10d,     // -0.30201,  0.95331
    0xbfd4135c94176602,     0x3fee6288ec48e112,     // -0.31368,  0.94953
    0xbfd4d1e24278e76a,     0x3fee426a4b2bc17e,     // -0.32531,  0.94561
    0xbfd58f9a75ab1fdd,     0x3fee212104f686e5,     // -0.33689,  0.94154
    0xbfd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     // -0.34842,  0.93734
    0xbfd7088530fa459e,     0x3feddb13b6ccc23d,     //  -0.3599,  0.93299
    0xbfd7c3a9311dcce7,     0x3fedb6526238a09b,     // -0.37132,  0.92851
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     // -0.38268,  0.92388
    0xbfd9372a63bc93d7,     0x3fed696173c9e68b,     // -0.39399,  0.91911
    0xbfd9ef7943a8ed8a,     0x3fed4134d14dc93a,     // -0.40524,  0.91421
    0xbfdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     // -0.41643,  0.90917
    0xbfdb5d1009e15cc0,     0x3feced7af43cc773,     // -0.42756,  0.90399
    0xbfdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     // -0.43862,  0.89867
    0xbfdcc66e9931c45d,     0x3fec954b213411f5,     // -0.44961,  0.89322
    0xbfdd79775b86e389,     0x3fec678b3488739b,     // -0.46054,  0.88764
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //  -0.4714,  0.88192
    0xbfdedc1952ef78d5,     0x3fec08c426725549,     // -0.48218,  0.87607
    0xbfdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //  -0.4929,  0.87009
    0xbfe01cfc874c3eb7,     0x3feba5aa673590d2,     // -0.50354,  0.86397
    0xbfe073879922ffed,     0x3feb728345196e3e,     //  -0.5141,  0.85773
    0xbfe0c9704d5d898f,     0x3feb3e4d3ef55712,     // -0.52459,  0.85136
    0xbfe11eb3541b4b22,     0x3feb090a58150200,     //   -0.535,  0.84485
    0xbfe1734d63dedb49,     0x3fead2bc9e21d511,     // -0.54532,  0.83822
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     // -0.55557,  0.83147
    0xbfe21a799933eb58,     0x3fea63091b02fae2,     // -0.56573,  0.82459
    0xbfe26d054cdd12df,     0x3fea29a7a0462782,     // -0.57581,  0.81758
    0xbfe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //  -0.5858,  0.81046
    0xbfe30ff7fce17035,     0x3fe9b3e047f38741,     //  -0.5957,  0.80321
    0xbfe36058b10659f3,     0x3fe9777ef4c7d742,     // -0.60551,  0.79584
    0xbfe3affa292050b9,     0x3fe93a22499263fc,     // -0.61523,  0.78835
    0xbfe3fed9534556d4,     0x3fe8fbcca3ef940d,     // -0.62486,  0.78074
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     // -0.63439,  0.77301
    0xbfe49a449b9b0938,     0x3fe87c400fba2ebf,     // -0.64383,  0.76517
    0xbfe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     // -0.65317,  0.75721
    0xbfe5328292a35596,     0x3fe7f8ece3571771,     // -0.66242,  0.74914
    0xbfe57d69348cec9f,     0x3fe7b5df226aafb0,     // -0.67156,  0.74095
    0xbfe5c77bbe65018c,     0x3fe771e75f037261,     //  -0.6806,  0.73265
    0xbfe610b7551d2cde,     0x3fe72d0837efff97,     // -0.68954,  0.72425
    0xbfe6591925f0783e,     0x3fe6e74454eaa8ae,     // -0.69838,  0.71573
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     // -0.70711,  0.70711
    0xbfe6e74454eaa8ae,     0x3fe6591925f0783e,     // -0.71573,  0.69838
    0xbfe72d0837efff97,     0x3fe610b7551d2cde,     // -0.72425,  0.68954
    0xbfe771e75f037261,     0x3fe5c77bbe65018c,     // -0.73265,   0.6806
    0xbfe7b5df226aafb0,     0x3fe57d69348cec9f,     // -0.74095,  0.67156
    0xbfe7f8ece3571771,     0x3fe5328292a35596,     // -0.74914,  0.66242
    0xbfe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     // -0.75721,  0.65317
    0xbfe87c400fba2ebf,     0x3fe49a449b9b0938,     // -0.76517,  0.64383
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     // -0.77301,  0.63439
    0xbfe8fbcca3ef940d,     0x3fe3fed9534556d4,     // -0.78074,  0.62486
    0xbfe93a22499263fc,     0x3fe3affa292050b9,     // -0.78835,  0.61523
    0xbfe9777ef4c7d742,     0x3fe36058b10659f3,     // -0.79584,  0.60551
    0xbfe9b3e047f38741,     0x3fe30ff7fce17035,     // -0.80321,   0.5957
    0xbfe9ef43ef29af94,     0x3fe2bedb25faf3ea,     // -0.81046,   0.5858
    0xbfea29a7a0462782,     0x3fe26d054cdd12df,     // -0.81758,  0.57581
    0xbfea63091b02fae2,     0x3fe21a799933eb58,     // -0.82459,  0.56573
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     // -0.83147,  0.55557
    0xbfead2bc9e21d511,     0x3fe1734d63dedb49,     // -0.83822,  0.54532
    0xbfeb090a58150200,     0x3fe11eb3541b4b22,     // -0.84485,    0.535
    0xbfeb3e4d3ef55712,     0x3fe0c9704d5d898f,     // -0.85136,  0.52459
    0xbfeb728345196e3e,     0x3fe073879922ffed,     // -0.85773,   0.5141
    0xbfeba5aa673590d2,     0x3fe01cfc874c3eb7,     // -0.86397,  0.50354
    0xbfebd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     // -0.87009,   0.4929
    0xbfec08c426725549,     0x3fdedc1952ef78d5,     // -0.87607,  0.48218
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     // -0.88192,   0.4714
    0xbfec678b3488739b,     0x3fdd79775b86e389,     // -0.88764,  0.46054
    0xbfec954b213411f5,     0x3fdcc66e9931c45d,     // -0.89322,  0.44961
    0xbfecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     // -0.89867,  0.43862
    0xbfeced7af43cc773,     0x3fdb5d1009e15cc0,     // -0.90399,  0.42756
    0xbfed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     // -0.90917,  0.41643
    0xbfed4134d14dc93a,     0x3fd9ef7943a8ed8a,     // -0.91421,  0.40524
    0xbfed696173c9e68b,     0x3fd9372a63bc93d7,     // -0.91911,  0.39399
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     // -0.92388,  0.38268
    0xbfedb6526238a09b,     0x3fd7c3a9311dcce7,     // -0.92851,  0.37132
    0xbfeddb13b6ccc23d,     0x3fd7088530fa459e,     // -0.93299,   0.3599
    0xbfedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     // -0.93734,  0.34842
    0xbfee212104f686e5,     0x3fd58f9a75ab1fdd,     // -0.94154,  0.33689
    0xbfee426a4b2bc17e,     0x3fd4d1e24278e76a,     // -0.94561,  0.32531
    0xbfee6288ec48e112,     0x3fd4135c94176602,     // -0.94953,  0.31368
    0xbfee817bab4cd10d,     0x3fd35410c2e18152,     // -0.95331,  0.30201
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     // -0.95694,  0.29028
    0xbfeebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     // -0.96043,  0.27852
    0xbfeed740e7684963,     0x3fd111d262b1f677,     // -0.96378,  0.26671
    0xbfeef178a3e473c2,     0x3fd04fb80e37fdae,     // -0.96698,  0.25487
    0xbfef0a7efb9230d7,     0x3fcf19f97b215f1a,     // -0.97003,  0.24298
    0xbfef2252f7763ada,     0x3fcd934fe5454311,     // -0.97294,  0.23106
    0xbfef38f3ac64e589,     0x3fcc0b826a7e4f63,     //  -0.9757,   0.2191
    0xbfef4e603b0b2f2d,     0x3fca82a025b00451,     // -0.97832,  0.20711
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     // -0.98079,  0.19509
    0xbfef7599a3a12077,     0x3fc76dd9de50bf31,     // -0.98311,  0.18304
    0xbfef8764fa714ba9,     0x3fc5e214448b3fc6,     // -0.98528,  0.17096
    0xbfef97f924c9099b,     0x3fc45576b1293e5a,     //  -0.9873,  0.15886
    0xbfefa7557f08a517,     0x3fc2c8106e8e613a,     // -0.98918,  0.14673
    0xbfefb5797195d741,     0x3fc139f0cedaf576,     //  -0.9909,  0.13458
    0xbfefc26470e19fd3,     0x3fbf564e56a9730e,     // -0.99248,  0.12241
    0xbfefce15fd6da67b,     0x3fbc3785c79ec2d5,     // -0.99391,  0.11022
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     // -0.99518, 0.098017
    0xbfefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     // -0.99631, 0.085797
    0xbfefe9cdad01883a,     0x3fb2d52092ce19f6,     // -0.99729, 0.073565
    0xbfeff095658e71ad,     0x3faf656e79f820e0,     // -0.99812, 0.061321
    0xbfeff621e3796d7e,     0x3fa91f65f10dd814,     //  -0.9988, 0.049068
    0xbfeffa72effef75d,     0x3fa2d865759455cd,     // -0.99932, 0.036807
    0xbfeffd886084cd0d,     0x3f992155f7a3667e,     //  -0.9997, 0.024541
    0xbfefff62169b92db,     0x3f8921d1fcdec784,     // -0.99992, 0.012272
    0xbff0000000000000,     0x0000000000000000,     //       -1,        0
    0xbfefff62169b92db,     0xbf8921d1fcdec784,     // -0.99992,-0.012272
    0xbfeffd886084cd0d,     0xbf992155f7a3667e,     //  -0.9997,-0.024541
    0xbfeffa72effef75d,     0xbfa2d865759455cd,     // -0.99932,-0.036807
    0xbfeff621e3796d7e,     0xbfa91f65f10dd814,     //  -0.9988,-0.049068
    0xbfeff095658e71ad,     0xbfaf656e79f820e0,     // -0.99812,-0.061321
    0xbfefe9cdad01883a,     0xbfb2d52092ce19f6,     // -0.99729,-0.073565
    0xbfefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     // -0.99631,-0.085797
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     // -0.99518,-0.098017
    0xbfefce15fd6da67b,     0xbfbc3785c79ec2d5,     // -0.99391, -0.11022
    0xbfefc26470e19fd3,     0xbfbf564e56a9730e,     // -0.99248, -0.12241
    0xbfefb5797195d741,     0xbfc139f0cedaf576,     //  -0.9909, -0.13458
    0xbfefa7557f08a517,     0xbfc2c8106e8e613a,     // -0.98918, -0.14673
    0xbfef97f924c9099b,     0xbfc45576b1293e5a,     //  -0.9873, -0.15886
    0xbfef8764fa714ba9,     0xbfc5e214448b3fc6,     // -0.98528, -0.17096
    0xbfef7599a3a12077,     0xbfc76dd9de50bf31,     // -0.98311, -0.18304
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     // -0.98079, -0.19509
    0xbfef4e603b0b2f2d,     0xbfca82a025b00451,     // -0.97832, -0.20711
    0xbfef38f3ac64e589,     0xbfcc0b826a7e4f63,     //  -0.9757,  -0.2191
    0xbfef2252f7763ada,     0xbfcd934fe5454311,     // -0.97294, -0.23106
    0xbfef0a7efb9230d7,     0xbfcf19f97b215f1a,     // -0.97003, -0.24298
    0xbfeef178a3e473c2,     0xbfd04fb80e37fdae,     // -0.96698, -0.25487
    0xbfeed740e7684963,     0xbfd111d262b1f677,     // -0.96378, -0.26671
    0xbfeebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     // -0.96043, -0.27852
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     // -0.95694, -0.29028
    0xbfee817bab4cd10d,     0xbfd35410c2e18152,     // -0.95331, -0.30201
    0xbfee6288ec48e112,     0xbfd4135c94176602,     // -0.94953, -0.31368
    0xbfee426a4b2bc17e,     0xbfd4d1e24278e76a,     // -0.94561, -0.32531
    0xbfee212104f686e5,     0xbfd58f9a75ab1fdd,     // -0.94154, -0.33689
    0xbfedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     // -0.93734, -0.34842
    0xbfeddb13b6ccc23d,     0xbfd7088530fa459e,     // -0.93299,  -0.3599
    0xbfedb6526238a09b,     0xbfd7c3a9311dcce7,     // -0.92851, -0.37132
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     // -0.92388, -0.38268
    0xbfed696173c9e68b,     0xbfd9372a63bc93d7,     // -0.91911, -0.39399
    0xbfed4134d14dc93a,     0xbfd9ef7943a8ed8a,     // -0.91421, -0.40524
    0xbfed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     // -0.90917, -0.41643
    0xbfeced7af43cc773,     0xbfdb5d1009e15cc0,     // -0.90399, -0.42756
    0xbfecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     // -0.89867, -0.43862
    0xbfec954b213411f5,     0xbfdcc66e9931c45d,     // -0.89322, -0.44961
    0xbfec678b3488739b,     0xbfdd79775b86e389,     // -0.88764, -0.46054
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     // -0.88192,  -0.4714
    0xbfec08c426725549,     0xbfdedc1952ef78d5,     // -0.87607, -0.48218
    0xbfebd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     // -0.87009,  -0.4929
    0xbfeba5aa673590d2,     0xbfe01cfc874c3eb7,     // -0.86397, -0.50354
    0xbfeb728345196e3e,     0xbfe073879922ffed,     // -0.85773,  -0.5141
    0xbfeb3e4d3ef55712,     0xbfe0c9704d5d898f,     // -0.85136, -0.52459
    0xbfeb090a58150200,     0xbfe11eb3541b4b22,     // -0.84485,   -0.535
    0xbfead2bc9e21d511,     0xbfe1734d63dedb49,     // -0.83822, -0.54532
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     // -0.83147, -0.55557
    0xbfea63091b02fae2,     0xbfe21a799933eb58,     // -0.82459, -0.56573
    0xbfea29a7a0462782,     0xbfe26d054cdd12df,     // -0.81758, -0.57581
    0xbfe9ef43ef29af94,     0xbfe2bedb25faf3ea,     // -0.81046,  -0.5858
    0xbfe9b3e047f38741,     0xbfe30ff7fce17035,     // -0.80321,  -0.5957
    0xbfe9777ef4c7d742,     0xbfe36058b10659f3,     // -0.79584, -0.60551
    0xbfe93a22499263fc,     0xbfe3affa292050b9,     // -0.78835, -0.61523
    0xbfe8fbcca3ef940d,     0xbfe3fed9534556d4,     // -0.78074, -0.62486
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     // -0.77301, -0.63439
    0xbfe87c400fba2ebf,     0xbfe49a449b9b0938,     // -0.76517, -0.64383
    0xbfe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     // -0.75721, -0.65317
    0xbfe7f8ece3571771,     0xbfe5328292a35596,     // -0.74914, -0.66242
    0xbfe7b5df226aafb0,     0xbfe57d69348cec9f,     // -0.74095, -0.67156
    0xbfe771e75f037261,     0xbfe5c77bbe65018c,     // -0.73265,  -0.6806
    0xbfe72d0837efff97,     0xbfe610b7551d2cde,     // -0.72425, -0.68954
    0xbfe6e74454eaa8ae,     0xbfe6591925f0783e,     // -0.71573, -0.69838
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     // -0.70711, -0.70711
    0xbfe6591925f0783e,     0xbfe6e74454eaa8ae,     // -0.69838, -0.71573
    0xbfe610b7551d2cde,     0xbfe72d0837efff97,     // -0.68954, -0.72425
    0xbfe5c77bbe65018c,     0xbfe771e75f037261,     //  -0.6806, -0.73265
    0xbfe57d69348cec9f,     0xbfe7b5df226aafb0,     // -0.67156, -0.74095
    0xbfe5328292a35596,     0xbfe7f8ece3571771,     // -0.66242, -0.74914
    0xbfe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     // -0.65317, -0.75721
    0xbfe49a449b9b0938,     0xbfe87c400fba2ebf,     // -0.64383, -0.76517
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     // -0.63439, -0.77301
    0xbfe3fed9534556d4,     0xbfe8fbcca3ef940d,     // -0.62486, -0.78074
    0xbfe3affa292050b9,     0xbfe93a22499263fc,     // -0.61523, -0.78835
    0xbfe36058b10659f3,     0xbfe9777ef4c7d742,     // -0.60551, -0.79584
    0xbfe30ff7fce17035,     0xbfe9b3e047f38741,     //  -0.5957, -0.80321
    0xbfe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //  -0.5858, -0.81046
    0xbfe26d054cdd12df,     0xbfea29a7a0462782,     // -0.57581, -0.81758
    0xbfe21a799933eb58,     0xbfea63091b02fae2,     // -0.56573, -0.82459
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     // -0.55557, -0.83147
    0xbfe1734d63dedb49,     0xbfead2bc9e21d511,     // -0.54532, -0.83822
    0xbfe11eb3541b4b22,     0xbfeb090a58150200,     //   -0.535, -0.84485
    0xbfe0c9704d5d898f,     0xbfeb3e4d3ef55712,     // -0.52459, -0.85136
    0xbfe073879922ffed,     0xbfeb728345196e3e,     //  -0.5141, -0.85773
    0xbfe01cfc874c3eb7,     0xbfeba5aa673590d2,     // -0.50354, -0.86397
    0xbfdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //  -0.4929, -0.87009
    0xbfdedc1952ef78d5,     0xbfec08c426725549,     // -0.48218, -0.87607
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //  -0.4714, -0.88192
    0xbfdd79775b86e389,     0xbfec678b3488739b,     // -0.46054, -0.88764
    0xbfdcc66e9931c45d,     0xbfec954b213411f5,     // -0.44961, -0.89322
    0xbfdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     // -0.43862, -0.89867
    0xbfdb5d1009e15cc0,     0xbfeced7af43cc773,     // -0.42756, -0.90399
    0xbfdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     // -0.41643, -0.90917
    0xbfd9ef7943a8ed8a,     0xbfed4134d14dc93a,     // -0.40524, -0.91421
    0xbfd9372a63bc93d7,     0xbfed696173c9e68b,     // -0.39399, -0.91911
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     // -0.38268, -0.92388
    0xbfd7c3a9311dcce7,     0xbfedb6526238a09b,     // -0.37132, -0.92851
    0xbfd7088530fa459e,     0xbfeddb13b6ccc23d,     //  -0.3599, -0.93299
    0xbfd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     // -0.34842, -0.93734
    0xbfd58f9a75ab1fdd,     0xbfee212104f686e5,     // -0.33689, -0.94154
    0xbfd4d1e24278e76a,     0xbfee426a4b2bc17e,     // -0.32531, -0.94561
    0xbfd4135c94176602,     0xbfee6288ec48e112,     // -0.31368, -0.94953
    0xbfd35410c2e18152,     0xbfee817bab4cd10d,     // -0.30201, -0.95331
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     // -0.29028, -0.95694
    0xbfd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     // -0.27852, -0.96043
    0xbfd111d262b1f677,     0xbfeed740e7684963,     // -0.26671, -0.96378
    0xbfd04fb80e37fdae,     0xbfeef178a3e473c2,     // -0.25487, -0.96698
    0xbfcf19f97b215f1a,     0xbfef0a7efb9230d7,     // -0.24298, -0.97003
    0xbfcd934fe5454311,     0xbfef2252f7763ada,     // -0.23106, -0.97294
    0xbfcc0b826a7e4f63,     0xbfef38f3ac64e589,     //  -0.2191,  -0.9757
    0xbfca82a025b00451,     0xbfef4e603b0b2f2d,     // -0.20711, -0.97832
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     // -0.19509, -0.98079
    0xbfc76dd9de50bf31,     0xbfef7599a3a12077,     // -0.18304, -0.98311
    0xbfc5e214448b3fc6,     0xbfef8764fa714ba9,     // -0.17096, -0.98528
    0xbfc45576b1293e5a,     0xbfef97f924c9099b,     // -0.15886,  -0.9873
    0xbfc2c8106e8e613a,     0xbfefa7557f08a517,     // -0.14673, -0.98918
    0xbfc139f0cedaf576,     0xbfefb5797195d741,     // -0.13458,  -0.9909
    0xbfbf564e56a9730e,     0xbfefc26470e19fd3,     // -0.12241, -0.99248
    0xbfbc3785c79ec2d5,     0xbfefce15fd6da67b,     // -0.11022, -0.99391
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     //-0.098017, -0.99518
    0xbfb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     //-0.085797, -0.99631
    0xbfb2d52092ce19f6,     0xbfefe9cdad01883a,     //-0.073565, -0.99729
    0xbfaf656e79f820e0,     0xbfeff095658e71ad,     //-0.061321, -0.99812
    0xbfa91f65f10dd814,     0xbfeff621e3796d7e,     //-0.049068,  -0.9988
    0xbfa2d865759455cd,     0xbfeffa72effef75d,     //-0.036807, -0.99932
    0xbf992155f7a3667e,     0xbfeffd886084cd0d,     //-0.024541,  -0.9997
    0xbf8921d1fcdec784,     0xbfefff62169b92db,     //-0.012272, -0.99992
    0x0000000000000000,     0xbff0000000000000,     //        0,       -1
    0x3f8921d1fcdec784,     0xbfefff62169b92db,     // 0.012272, -0.99992
    0x3f992155f7a3667e,     0xbfeffd886084cd0d,     // 0.024541,  -0.9997
    0x3fa2d865759455cd,     0xbfeffa72effef75d,     // 0.036807, -0.99932
    0x3fa91f65f10dd814,     0xbfeff621e3796d7e,     // 0.049068,  -0.9988
    0x3faf656e79f820e0,     0xbfeff095658e71ad,     // 0.061321, -0.99812
    0x3fb2d52092ce19f6,     0xbfefe9cdad01883a,     // 0.073565, -0.99729
    0x3fb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     // 0.085797, -0.99631
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     // 0.098017, -0.99518
    0x3fbc3785c79ec2d5,     0xbfefce15fd6da67b,     //  0.11022, -0.99391
    0x3fbf564e56a9730e,     0xbfefc26470e19fd3,     //  0.12241, -0.99248
    0x3fc139f0cedaf576,     0xbfefb5797195d741,     //  0.13458,  -0.9909
    0x3fc2c8106e8e613a,     0xbfefa7557f08a517,     //  0.14673, -0.98918
    0x3fc45576b1293e5a,     0xbfef97f924c9099b,     //  0.15886,  -0.9873
    0x3fc5e214448b3fc6,     0xbfef8764fa714ba9,     //  0.17096, -0.98528
    0x3fc76dd9de50bf31,     0xbfef7599a3a12077,     //  0.18304, -0.98311
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  0.19509, -0.98079
    0x3fca82a025b00451,     0xbfef4e603b0b2f2d,     //  0.20711, -0.97832
    0x3fcc0b826a7e4f63,     0xbfef38f3ac64e589,     //   0.2191,  -0.9757
    0x3fcd934fe5454311,     0xbfef2252f7763ada,     //  0.23106, -0.97294
    0x3fcf19f97b215f1a,     0xbfef0a7efb9230d7,     //  0.24298, -0.97003
    0x3fd04fb80e37fdae,     0xbfeef178a3e473c2,     //  0.25487, -0.96698
    0x3fd111d262b1f677,     0xbfeed740e7684963,     //  0.26671, -0.96378
    0x3fd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     //  0.27852, -0.96043
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //  0.29028, -0.95694
    0x3fd35410c2e18152,     0xbfee817bab4cd10d,     //  0.30201, -0.95331
    0x3fd4135c94176602,     0xbfee6288ec48e112,     //  0.31368, -0.94953
    0x3fd4d1e24278e76a,     0xbfee426a4b2bc17e,     //  0.32531, -0.94561
    0x3fd58f9a75ab1fdd,     0xbfee212104f686e5,     //  0.33689, -0.94154
    0x3fd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     //  0.34842, -0.93734
    0x3fd7088530fa459e,     0xbfeddb13b6ccc23d,     //   0.3599, -0.93299
    0x3fd7c3a9311dcce7,     0xbfedb6526238a09b,     //  0.37132, -0.92851
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //  0.38268, -0.92388
    0x3fd9372a63bc93d7,     0xbfed696173c9e68b,     //  0.39399, -0.91911
    0x3fd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //  0.40524, -0.91421
    0x3fdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     //  0.41643, -0.90917
    0x3fdb5d1009e15cc0,     0xbfeced7af43cc773,     //  0.42756, -0.90399
    0x3fdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     //  0.43862, -0.89867
    0x3fdcc66e9931c45d,     0xbfec954b213411f5,     //  0.44961, -0.89322
    0x3fdd79775b86e389,     0xbfec678b3488739b,     //  0.46054, -0.88764
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   0.4714, -0.88192
    0x3fdedc1952ef78d5,     0xbfec08c426725549,     //  0.48218, -0.87607
    0x3fdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //   0.4929, -0.87009
    0x3fe01cfc874c3eb7,     0xbfeba5aa673590d2,     //  0.50354, -0.86397
    0x3fe073879922ffed,     0xbfeb728345196e3e,     //   0.5141, -0.85773
    0x3fe0c9704d5d898f,     0xbfeb3e4d3ef55712,     //  0.52459, -0.85136
    0x3fe11eb3541b4b22,     0xbfeb090a58150200,     //    0.535, -0.84485
    0x3fe1734d63dedb49,     0xbfead2bc9e21d511,     //  0.54532, -0.83822
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  0.55557, -0.83147
    0x3fe21a799933eb58,     0xbfea63091b02fae2,     //  0.56573, -0.82459
    0x3fe26d054cdd12df,     0xbfea29a7a0462782,     //  0.57581, -0.81758
    0x3fe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //   0.5858, -0.81046
    0x3fe30ff7fce17035,     0xbfe9b3e047f38741,     //   0.5957, -0.80321
    0x3fe36058b10659f3,     0xbfe9777ef4c7d742,     //  0.60551, -0.79584
    0x3fe3affa292050b9,     0xbfe93a22499263fc,     //  0.61523, -0.78835
    0x3fe3fed9534556d4,     0xbfe8fbcca3ef940d,     //  0.62486, -0.78074
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //  0.63439, -0.77301
    0x3fe49a449b9b0938,     0xbfe87c400fba2ebf,     //  0.64383, -0.76517
    0x3fe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //  0.65317, -0.75721
    0x3fe5328292a35596,     0xbfe7f8ece3571771,     //  0.66242, -0.74914
    0x3fe57d69348cec9f,     0xbfe7b5df226aafb0,     //  0.67156, -0.74095
    0x3fe5c77bbe65018c,     0xbfe771e75f037261,     //   0.6806, -0.73265
    0x3fe610b7551d2cde,     0xbfe72d0837efff97,     //  0.68954, -0.72425
    0x3fe6591925f0783e,     0xbfe6e74454eaa8ae,     //  0.69838, -0.71573
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  0.70711, -0.70711
    0x3fe6e74454eaa8ae,     0xbfe6591925f0783e,     //  0.71573, -0.69838
    0x3fe72d0837efff97,     0xbfe610b7551d2cde,     //  0.72425, -0.68954
    0x3fe771e75f037261,     0xbfe5c77bbe65018c,     //  0.73265,  -0.6806
    0x3fe7b5df226aafb0,     0xbfe57d69348cec9f,     //  0.74095, -0.67156
    0x3fe7f8ece3571771,     0xbfe5328292a35596,     //  0.74914, -0.66242
    0x3fe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //  0.75721, -0.65317
    0x3fe87c400fba2ebf,     0xbfe49a449b9b0938,     //  0.76517, -0.64383
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //  0.77301, -0.63439
    0x3fe8fbcca3ef940d,     0xbfe3fed9534556d4,     //  0.78074, -0.62486
    0x3fe93a22499263fc,     0xbfe3affa292050b9,     //  0.78835, -0.61523
    0x3fe9777ef4c7d742,     0xbfe36058b10659f3,     //  0.79584, -0.60551
    0x3fe9b3e047f38741,     0xbfe30ff7fce17035,     //  0.80321,  -0.5957
    0x3fe9ef43ef29af94,     0xbfe2bedb25faf3ea,     //  0.81046,  -0.5858
    0x3fea29a7a0462782,     0xbfe26d054cdd12df,     //  0.81758, -0.57581
    0x3fea63091b02fae2,     0xbfe21a799933eb58,     //  0.82459, -0.56573
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  0.83147, -0.55557
    0x3fead2bc9e21d511,     0xbfe1734d63dedb49,     //  0.83822, -0.54532
    0x3feb090a58150200,     0xbfe11eb3541b4b22,     //  0.84485,   -0.535
    0x3feb3e4d3ef55712,     0xbfe0c9704d5d898f,     //  0.85136, -0.52459
    0x3feb728345196e3e,     0xbfe073879922ffed,     //  0.85773,  -0.5141
    0x3feba5aa673590d2,     0xbfe01cfc874c3eb7,     //  0.86397, -0.50354
    0x3febd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //  0.87009,  -0.4929
    0x3fec08c426725549,     0xbfdedc1952ef78d5,     //  0.87607, -0.48218
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  0.88192,  -0.4714
    0x3fec678b3488739b,     0xbfdd79775b86e389,     //  0.88764, -0.46054
    0x3fec954b213411f5,     0xbfdcc66e9931c45d,     //  0.89322, -0.44961
    0x3fecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     //  0.89867, -0.43862
    0x3feced7af43cc773,     0xbfdb5d1009e15cc0,     //  0.90399, -0.42756
    0x3fed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     //  0.90917, -0.41643
    0x3fed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //  0.91421, -0.40524
    0x3fed696173c9e68b,     0xbfd9372a63bc93d7,     //  0.91911, -0.39399
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //  0.92388, -0.38268
    0x3fedb6526238a09b,     0xbfd7c3a9311dcce7,     //  0.92851, -0.37132
    0x3feddb13b6ccc23d,     0xbfd7088530fa459e,     //  0.93299,  -0.3599
    0x3fedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     //  0.93734, -0.34842
    0x3fee212104f686e5,     0xbfd58f9a75ab1fdd,     //  0.94154, -0.33689
    0x3fee426a4b2bc17e,     0xbfd4d1e24278e76a,     //  0.94561, -0.32531
    0x3fee6288ec48e112,     0xbfd4135c94176602,     //  0.94953, -0.31368
    0x3fee817bab4cd10d,     0xbfd35410c2e18152,     //  0.95331, -0.30201
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //  0.95694, -0.29028
    0x3feebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     //  0.96043, -0.27852
    0x3feed740e7684963,     0xbfd111d262b1f677,     //  0.96378, -0.26671
    0x3feef178a3e473c2,     0xbfd04fb80e37fdae,     //  0.96698, -0.25487
    0x3fef0a7efb9230d7,     0xbfcf19f97b215f1a,     //  0.97003, -0.24298
    0x3fef2252f7763ada,     0xbfcd934fe5454311,     //  0.97294, -0.23106
    0x3fef38f3ac64e589,     0xbfcc0b826a7e4f63,     //   0.9757,  -0.2191
    0x3fef4e603b0b2f2d,     0xbfca82a025b00451,     //  0.97832, -0.20711
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  0.98079, -0.19509
    0x3fef7599a3a12077,     0xbfc76dd9de50bf31,     //  0.98311, -0.18304
    0x3fef8764fa714ba9,     0xbfc5e214448b3fc6,     //  0.98528, -0.17096
    0x3fef97f924c9099b,     0xbfc45576b1293e5a,     //   0.9873, -0.15886
    0x3fefa7557f08a517,     0xbfc2c8106e8e613a,     //  0.98918, -0.14673
    0x3fefb5797195d741,     0xbfc139f0cedaf576,     //   0.9909, -0.13458
    0x3fefc26470e19fd3,     0xbfbf564e56a9730e,     //  0.99248, -0.12241
    0x3fefce15fd6da67b,     0xbfbc3785c79ec2d5,     //  0.99391, -0.11022
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //  0.99518,-0.098017
    0x3fefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     //  0.99631,-0.085797
    0x3fefe9cdad01883a,     0xbfb2d52092ce19f6,     //  0.99729,-0.073565
    0x3feff095658e71ad,     0xbfaf656e79f820e0,     //  0.99812,-0.061321
    0x3feff621e3796d7e,     0xbfa91f65f10dd814,     //   0.9988,-0.049068
    0x3feffa72effef75d,     0xbfa2d865759455cd,     //  0.99932,-0.036807
    0x3feffd886084cd0d,     0xbf992155f7a3667e,     //   0.9997,-0.024541
    0x3fefff62169b92db,     0xbf8921d1fcdec784,     //  0.99992,-0.012272
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_1024)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 1024, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_1024[2048] = {
    0x3ff0000000000000,    0x0000000000000000,     //         1,         0
    0x3fefffd8858e8a92,    0x3f7921f0fe670071,     //   0.99998, 0.0061359
    0x3fefff62169b92db,    0x3f8921d1fcdec784,     //   0.99992,  0.012272
    0x3feffe9cb44b51a1,    0x3f92d936bbe30efd,     //   0.99983,  0.018407
    0x3feffd886084cd0d,    0x3f992155f7a3667e,     //    0.9997,  0.024541
    0x3feffc251df1d3f8,    0x3f9f693731d1cf01,     //   0.99953,  0.030675
    0x3feffa72effef75d,    0x3fa2d865759455cd,     //   0.99932,  0.036807
    0x3feff871dadb81df,    0x3fa5fc00d290cd43,     //   0.99908,  0.042938
    0x3feff621e3796d7e,    0x3fa91f65f10dd814,     //    0.9988,  0.049068
    0x3feff3830f8d575c,    0x3fac428d12c0d7e3,     //   0.99848,  0.055195
    0x3feff095658e71ad,    0x3faf656e79f820e0,     //   0.99812,  0.061321
    0x3fefed58ecb673c4,    0x3fb1440134d709b2,     //   0.99772,  0.067444
    0x3fefe9cdad01883a,    0x3fb2d52092ce19f6,     //   0.99729,  0.073565
    0x3fefe5f3af2e3940,    0x3fb4661179272096,     //   0.99682,  0.079682
    0x3fefe1cafcbd5b09,    0x3fb5f6d00a9aa419,     //   0.99631,  0.085797
    0x3fefdd539ff1f456,    0x3fb787586a5d5b21,     //   0.99577,  0.091909
    0x3fefd88da3d12526,    0x3fb917a6bc29b42c,     //   0.99518,  0.098017
    0x3fefd37914220b84,    0x3fbaa7b724495c04,     //   0.99456,   0.10412
    0x3fefce15fd6da67b,    0x3fbc3785c79ec2d5,     //   0.99391,   0.11022
    0x3fefc8646cfeb721,    0x3fbdc70ecbae9fc8,     //   0.99321,   0.11632
    0x3fefc26470e19fd3,    0x3fbf564e56a9730e,     //   0.99248,   0.12241
    0x3fefbc1617e44186,    0x3fc072a047ba831d,     //   0.99171,    0.1285
    0x3fefb5797195d741,    0x3fc139f0cedaf576,     //    0.9909,   0.13458
    0x3fefae8e8e46cfbb,    0x3fc20116d4ec7bce,     //   0.99006,   0.14066
    0x3fefa7557f08a517,    0x3fc2c8106e8e613a,     //   0.98918,   0.14673
    0x3fef9fce55adb2c8,    0x3fc38edbb0cd8d14,     //   0.98826,    0.1528
    0x3fef97f924c9099b,    0x3fc45576b1293e5a,     //    0.9873,   0.15886
    0x3fef8fd5ffae41db,    0x3fc51bdf8597c5f2,     //   0.98631,   0.16491
    0x3fef8764fa714ba9,    0x3fc5e214448b3fc6,     //   0.98528,   0.17096
    0x3fef7ea629e63d6e,    0x3fc6a81304f64ab2,     //   0.98421,     0.177
    0x3fef7599a3a12077,    0x3fc76dd9de50bf31,     //   0.98311,   0.18304
    0x3fef6c3f7df5bbb7,    0x3fc83366e89c64c5,     //   0.98196,   0.18907
    0x3fef6297cff75cb0,    0x3fc8f8b83c69a60a,     //   0.98079,   0.19509
    0x3fef58a2b1789e84,    0x3fc9bdcbf2dc4366,     //   0.97957,    0.2011
    0x3fef4e603b0b2f2d,    0x3fca82a025b00451,     //   0.97832,   0.20711
    0x3fef43d085ff92dd,    0x3fcb4732ef3d6722,     //   0.97703,   0.21311
    0x3fef38f3ac64e589,    0x3fcc0b826a7e4f63,     //    0.9757,    0.2191
    0x3fef2dc9c9089a9d,    0x3fcccf8cb312b286,     //   0.97434,   0.22508
    0x3fef2252f7763ada,    0x3fcd934fe5454311,     //   0.97294,   0.23106
    0x3fef168f53f7205d,    0x3fce56ca1e101a1b,     //    0.9715,   0.23702
    0x3fef0a7efb9230d7,    0x3fcf19f97b215f1a,     //   0.97003,   0.24298
    0x3feefe220c0b95ec,    0x3fcfdcdc1adfedf8,     //   0.96852,   0.24893
    0x3feef178a3e473c2,    0x3fd04fb80e37fdae,     //   0.96698,   0.25487
    0x3feee482e25a9dbc,    0x3fd0b0d9cfdbdb90,     //   0.96539,   0.26079
    0x3feed740e7684963,    0x3fd111d262b1f677,     //   0.96378,   0.26671
    0x3feec9b2d3c3bf84,    0x3fd172a0d7765177,     //   0.96212,   0.27262
    0x3feebbd8c8df0b74,    0x3fd1d3443f4cdb3d,     //   0.96043,   0.27852
    0x3feeadb2e8e7a88e,    0x3fd233bbabc3bb72,     //    0.9587,   0.28441
    0x3fee9f4156c62dda,    0x3fd294062ed59f05,     //   0.95694,   0.29028
    0x3fee9084361df7f3,    0x3fd2f422daec0386,     //   0.95514,   0.29615
    0x3fee817bab4cd10d,    0x3fd35410c2e18152,     //   0.95331,   0.30201
    0x3fee7227db6a9744,    0x3fd3b3cefa0414b7,     //   0.95144,   0.30785
    0x3fee6288ec48e112,    0x3fd4135c94176602,     //   0.94953,   0.31368
    0x3fee529f04729ffc,    0x3fd472b8a5571054,     //   0.94759,    0.3195
    0x3fee426a4b2bc17e,    0x3fd4d1e24278e76a,     //   0.94561,   0.32531
    0x3fee31eae870ce25,    0x3fd530d880af3c24,     //   0.94359,   0.33111
    0x3fee212104f686e5,    0x3fd58f9a75ab1fdd,     //   0.94154,   0.33689
    0x3fee100cca2980ac,    0x3fd5ee27379ea693,     //   0.93946,   0.34266
    0x3fedfeae622dbe2b,    0x3fd64c7ddd3f27c6,     //   0.93734,   0.34842
    0x3feded05f7de47da,    0x3fd6aa9d7dc77e16,     //   0.93518,   0.35416
    0x3feddb13b6ccc23d,    0x3fd7088530fa459e,     //   0.93299,    0.3599
    0x3fedc8d7cb410260,    0x3fd766340f2418f6,     //   0.93077,   0.36561
    0x3fedb6526238a09b,    0x3fd7c3a9311dcce7,     //   0.92851,   0.37132
    0x3feda383a9668988,    0x3fd820e3b04eaac4,     //   0.92621,   0.37701
    0x3fed906bcf328d46,    0x3fd87de2a6aea963,     //   0.92388,   0.38268
    0x3fed7d0b02b8ecf9,    0x3fd8daa52ec8a4af,     //   0.92151,   0.38835
    0x3fed696173c9e68b,    0x3fd9372a63bc93d7,     //   0.91911,   0.39399
    0x3fed556f52e93eb1,    0x3fd993716141bdfe,     //   0.91668,   0.39962
    0x3fed4134d14dc93a,    0x3fd9ef7943a8ed8a,     //   0.91421,   0.40524
    0x3fed2cb220e0ef9f,    0x3fda4b4127dea1e4,     //   0.91171,   0.41084
    0x3fed17e7743e35dc,    0x3fdaa6c82b6d3fc9,     //   0.90917,   0.41643
    0x3fed02d4feb2bd92,    0x3fdb020d6c7f4009,     //    0.9066,     0.422
    0x3feced7af43cc773,    0x3fdb5d1009e15cc0,     //   0.90399,   0.42756
    0x3fecd7d9898b32f6,    0x3fdbb7cf2304bd01,     //   0.90135,   0.43309
    0x3fecc1f0f3fcfc5c,    0x3fdc1249d8011ee7,     //   0.89867,   0.43862
    0x3fecabc169a0b901,    0x3fdc6c7f4997000a,     //   0.89597,   0.44412
    0x3fec954b213411f5,    0x3fdcc66e9931c45d,     //   0.89322,   0.44961
    0x3fec7e8e52233cf3,    0x3fdd2016e8e9db5b,     //   0.89045,   0.45508
    0x3fec678b3488739b,    0x3fdd79775b86e389,     //   0.88764,   0.46054
    0x3fec5042012b6907,    0x3fddd28f1481cc58,     //    0.8848,   0.46598
    0x3fec38b2f180bdb1,    0x3fde2b5d3806f63b,     //   0.88192,    0.4714
    0x3fec20de3fa971b0,    0x3fde83e0eaf85113,     //   0.87901,    0.4768
    0x3fec08c426725549,    0x3fdedc1952ef78d5,     //   0.87607,   0.48218
    0x3febf064e15377dd,    0x3fdf3405963fd068,     //   0.87309,   0.48755
    0x3febd7c0ac6f952a,    0x3fdf8ba4dbf89aba,     //   0.87009,    0.4929
    0x3febbed7c49380ea,    0x3fdfe2f64be7120f,     //   0.86705,   0.49823
    0x3feba5aa673590d2,    0x3fe01cfc874c3eb7,     //   0.86397,   0.50354
    0x3feb8c38d27504e9,    0x3fe0485626ae221a,     //   0.86087,   0.50883
    0x3feb728345196e3e,    0x3fe073879922ffed,     //   0.85773,    0.5141
    0x3feb5889fe921405,    0x3fe09e907417c5e1,     //   0.85456,   0.51936
    0x3feb3e4d3ef55712,    0x3fe0c9704d5d898f,     //   0.85136,   0.52459
    0x3feb23cd470013b4,    0x3fe0f426bb2a8e7d,     //   0.84812,    0.5298
    0x3feb090a58150200,    0x3fe11eb3541b4b22,     //   0.84485,     0.535
    0x3feaee04b43c1474,    0x3fe14915af336ceb,     //   0.84155,   0.54017
    0x3fead2bc9e21d511,    0x3fe1734d63dedb49,     //   0.83822,   0.54532
    0x3feab7325916c0d4,    0x3fe19d5a09f2b9b8,     //   0.83486,   0.55046
    0x3fea9b66290ea1a3,    0x3fe1c73b39ae68c8,     //   0.83147,   0.55557
    0x3fea7f58529fe69d,    0x3fe1f0f08bbc861b,     //   0.82805,   0.56066
    0x3fea63091b02fae2,    0x3fe21a799933eb58,     //   0.82459,   0.56573
    0x3fea4678c8119ac8,    0x3fe243d5fb98ac1f,     //    0.8211,   0.57078
    0x3fea29a7a0462782,    0x3fe26d054cdd12df,     //   0.81758,   0.57581
    0x3fea0c95eabaf937,    0x3fe2960727629ca8,     //   0.81404,   0.58081
    0x3fe9ef43ef29af94,    0x3fe2bedb25faf3ea,     //   0.81046,    0.5858
    0x3fe9d1b1f5ea80d6,    0x3fe2e780e3e8ea16,     //   0.80685,   0.59076
    0x3fe9b3e047f38741,    0x3fe30ff7fce17035,     //   0.80321,    0.5957
    0x3fe995cf2ed80d22,    0x3fe338400d0c8e57,     //   0.79954,   0.60062
    0x3fe9777ef4c7d742,    0x3fe36058b10659f3,     //   0.79584,   0.60551
    0x3fe958efe48e6dd7,    0x3fe3884185dfeb22,     //   0.79211,   0.61038
    0x3fe93a22499263fc,    0x3fe3affa292050b9,     //   0.78835,   0.61523
    0x3fe91b166fd49da2,    0x3fe3d78238c58343,     //   0.78456,   0.62006
    0x3fe8fbcca3ef940d,    0x3fe3fed9534556d4,     //   0.78074,   0.62486
    0x3fe8dc45331698cc,    0x3fe425ff178e6bb1,     //   0.77689,   0.62964
    0x3fe8bc806b151741,    0x3fe44cf325091dd6,     //   0.77301,   0.63439
    0x3fe89c7e9a4dd4ab,    0x3fe473b51b987347,     //    0.7691,   0.63912
    0x3fe87c400fba2ebf,    0x3fe49a449b9b0938,     //   0.76517,   0.64383
    0x3fe85bc51ae958cc,    0x3fe4c0a145ec0004,     //    0.7612,   0.64851
    0x3fe83b0e0bff976e,    0x3fe4e6cabbe3e5e9,     //   0.75721,   0.65317
    0x3fe81a1b33b57acc,    0x3fe50cc09f59a09b,     //   0.75319,   0.65781
    0x3fe7f8ece3571771,    0x3fe5328292a35596,     //   0.74914,   0.66242
    0x3fe7d7836cc33db2,    0x3fe5581038975137,     //   0.74506,     0.667
    0x3fe7b5df226aafb0,    0x3fe57d69348cec9f,     //   0.74095,   0.67156
    0x3fe79400574f55e4,    0x3fe5a28d2a5d7250,     //   0.73682,   0.67609
    0x3fe771e75f037261,    0x3fe5c77bbe65018c,     //   0.73265,    0.6806
    0x3fe74f948da8d28d,    0x3fe5ec3495837074,     //   0.72846,   0.68508
    0x3fe72d0837efff97,    0x3fe610b7551d2cde,     //   0.72425,   0.68954
    0x3fe70a42b3176d7a,    0x3fe63503a31c1be8,     //      0.72,   0.69397
    0x3fe6e74454eaa8ae,    0x3fe6591925f0783e,     //   0.71573,   0.69838
    0x3fe6c40d73c18275,    0x3fe67cf78491af10,     //   0.71143,   0.70275
    0x3fe6a09e667f3bcc,    0x3fe6a09e667f3bcc,     //   0.70711,   0.70711
    0x3fe67cf78491af10,    0x3fe6c40d73c18275,     //   0.70275,   0.71143
    0x3fe6591925f0783e,    0x3fe6e74454eaa8ae,     //   0.69838,   0.71573
    0x3fe63503a31c1be8,    0x3fe70a42b3176d7a,     //   0.69397,      0.72
    0x3fe610b7551d2cde,    0x3fe72d0837efff97,     //   0.68954,   0.72425
    0x3fe5ec3495837074,    0x3fe74f948da8d28d,     //   0.68508,   0.72846
    0x3fe5c77bbe65018c,    0x3fe771e75f037261,     //    0.6806,   0.73265
    0x3fe5a28d2a5d7250,    0x3fe79400574f55e4,     //   0.67609,   0.73682
    0x3fe57d69348cec9f,    0x3fe7b5df226aafb0,     //   0.67156,   0.74095
    0x3fe5581038975137,    0x3fe7d7836cc33db2,     //     0.667,   0.74506
    0x3fe5328292a35596,    0x3fe7f8ece3571771,     //   0.66242,   0.74914
    0x3fe50cc09f59a09b,    0x3fe81a1b33b57acc,     //   0.65781,   0.75319
    0x3fe4e6cabbe3e5e9,    0x3fe83b0e0bff976e,     //   0.65317,   0.75721
    0x3fe4c0a145ec0004,    0x3fe85bc51ae958cc,     //   0.64851,    0.7612
    0x3fe49a449b9b0938,    0x3fe87c400fba2ebf,     //   0.64383,   0.76517
    0x3fe473b51b987347,    0x3fe89c7e9a4dd4ab,     //   0.63912,    0.7691
    0x3fe44cf325091dd6,    0x3fe8bc806b151741,     //   0.63439,   0.77301
    0x3fe425ff178e6bb1,    0x3fe8dc45331698cc,     //   0.62964,   0.77689
    0x3fe3fed9534556d4,    0x3fe8fbcca3ef940d,     //   0.62486,   0.78074
    0x3fe3d78238c58343,    0x3fe91b166fd49da2,     //   0.62006,   0.78456
    0x3fe3affa292050b9,    0x3fe93a22499263fc,     //   0.61523,   0.78835
    0x3fe3884185dfeb22,    0x3fe958efe48e6dd7,     //   0.61038,   0.79211
    0x3fe36058b10659f3,    0x3fe9777ef4c7d742,     //   0.60551,   0.79584
    0x3fe338400d0c8e57,    0x3fe995cf2ed80d22,     //   0.60062,   0.79954
    0x3fe30ff7fce17035,    0x3fe9b3e047f38741,     //    0.5957,   0.80321
    0x3fe2e780e3e8ea16,    0x3fe9d1b1f5ea80d6,     //   0.59076,   0.80685
    0x3fe2bedb25faf3ea,    0x3fe9ef43ef29af94,     //    0.5858,   0.81046
    0x3fe2960727629ca8,    0x3fea0c95eabaf937,     //   0.58081,   0.81404
    0x3fe26d054cdd12df,    0x3fea29a7a0462782,     //   0.57581,   0.81758
    0x3fe243d5fb98ac1f,    0x3fea4678c8119ac8,     //   0.57078,    0.8211
    0x3fe21a799933eb58,    0x3fea63091b02fae2,     //   0.56573,   0.82459
    0x3fe1f0f08bbc861b,    0x3fea7f58529fe69d,     //   0.56066,   0.82805
    0x3fe1c73b39ae68c8,    0x3fea9b66290ea1a3,     //   0.55557,   0.83147
    0x3fe19d5a09f2b9b8,    0x3feab7325916c0d4,     //   0.55046,   0.83486
    0x3fe1734d63dedb49,    0x3fead2bc9e21d511,     //   0.54532,   0.83822
    0x3fe14915af336ceb,    0x3feaee04b43c1474,     //   0.54017,   0.84155
    0x3fe11eb3541b4b22,    0x3feb090a58150200,     //     0.535,   0.84485
    0x3fe0f426bb2a8e7d,    0x3feb23cd470013b4,     //    0.5298,   0.84812
    0x3fe0c9704d5d898f,    0x3feb3e4d3ef55712,     //   0.52459,   0.85136
    0x3fe09e907417c5e1,    0x3feb5889fe921405,     //   0.51936,   0.85456
    0x3fe073879922ffed,    0x3feb728345196e3e,     //    0.5141,   0.85773
    0x3fe0485626ae221a,    0x3feb8c38d27504e9,     //   0.50883,   0.86087
    0x3fe01cfc874c3eb7,    0x3feba5aa673590d2,     //   0.50354,   0.86397
    0x3fdfe2f64be7120f,    0x3febbed7c49380ea,     //   0.49823,   0.86705
    0x3fdf8ba4dbf89aba,    0x3febd7c0ac6f952a,     //    0.4929,   0.87009
    0x3fdf3405963fd068,    0x3febf064e15377dd,     //   0.48755,   0.87309
    0x3fdedc1952ef78d5,    0x3fec08c426725549,     //   0.48218,   0.87607
    0x3fde83e0eaf85113,    0x3fec20de3fa971b0,     //    0.4768,   0.87901
    0x3fde2b5d3806f63b,    0x3fec38b2f180bdb1,     //    0.4714,   0.88192
    0x3fddd28f1481cc58,    0x3fec5042012b6907,     //   0.46598,    0.8848
    0x3fdd79775b86e389,    0x3fec678b3488739b,     //   0.46054,   0.88764
    0x3fdd2016e8e9db5b,    0x3fec7e8e52233cf3,     //   0.45508,   0.89045
    0x3fdcc66e9931c45d,    0x3fec954b213411f5,     //   0.44961,   0.89322
    0x3fdc6c7f4997000a,    0x3fecabc169a0b901,     //   0.44412,   0.89597
    0x3fdc1249d8011ee7,    0x3fecc1f0f3fcfc5c,     //   0.43862,   0.89867
    0x3fdbb7cf2304bd01,    0x3fecd7d9898b32f6,     //   0.43309,   0.90135
    0x3fdb5d1009e15cc0,    0x3feced7af43cc773,     //   0.42756,   0.90399
    0x3fdb020d6c7f4009,    0x3fed02d4feb2bd92,     //     0.422,    0.9066
    0x3fdaa6c82b6d3fc9,    0x3fed17e7743e35dc,     //   0.41643,   0.90917
    0x3fda4b4127dea1e4,    0x3fed2cb220e0ef9f,     //   0.41084,   0.91171
    0x3fd9ef7943a8ed8a,    0x3fed4134d14dc93a,     //   0.40524,   0.91421
    0x3fd993716141bdfe,    0x3fed556f52e93eb1,     //   0.39962,   0.91668
    0x3fd9372a63bc93d7,    0x3fed696173c9e68b,     //   0.39399,   0.91911
    0x3fd8daa52ec8a4af,    0x3fed7d0b02b8ecf9,     //   0.38835,   0.92151
    0x3fd87de2a6aea963,    0x3fed906bcf328d46,     //   0.38268,   0.92388
    0x3fd820e3b04eaac4,    0x3feda383a9668988,     //   0.37701,   0.92621
    0x3fd7c3a9311dcce7,    0x3fedb6526238a09b,     //   0.37132,   0.92851
    0x3fd766340f2418f6,    0x3fedc8d7cb410260,     //   0.36561,   0.93077
    0x3fd7088530fa459e,    0x3feddb13b6ccc23d,     //    0.3599,   0.93299
    0x3fd6aa9d7dc77e16,    0x3feded05f7de47da,     //   0.35416,   0.93518
    0x3fd64c7ddd3f27c6,    0x3fedfeae622dbe2b,     //   0.34842,   0.93734
    0x3fd5ee27379ea693,    0x3fee100cca2980ac,     //   0.34266,   0.93946
    0x3fd58f9a75ab1fdd,    0x3fee212104f686e5,     //   0.33689,   0.94154
    0x3fd530d880af3c24,    0x3fee31eae870ce25,     //   0.33111,   0.94359
    0x3fd4d1e24278e76a,    0x3fee426a4b2bc17e,     //   0.32531,   0.94561
    0x3fd472b8a5571054,    0x3fee529f04729ffc,     //    0.3195,   0.94759
    0x3fd4135c94176602,    0x3fee6288ec48e112,     //   0.31368,   0.94953
    0x3fd3b3cefa0414b7,    0x3fee7227db6a9744,     //   0.30785,   0.95144
    0x3fd35410c2e18152,    0x3fee817bab4cd10d,     //   0.30201,   0.95331
    0x3fd2f422daec0386,    0x3fee9084361df7f3,     //   0.29615,   0.95514
    0x3fd294062ed59f05,    0x3fee9f4156c62dda,     //   0.29028,   0.95694
    0x3fd233bbabc3bb72,    0x3feeadb2e8e7a88e,     //   0.28441,    0.9587
    0x3fd1d3443f4cdb3d,    0x3feebbd8c8df0b74,     //   0.27852,   0.96043
    0x3fd172a0d7765177,    0x3feec9b2d3c3bf84,     //   0.27262,   0.96212
    0x3fd111d262b1f677,    0x3feed740e7684963,     //   0.26671,   0.96378
    0x3fd0b0d9cfdbdb90,    0x3feee482e25a9dbc,     //   0.26079,   0.96539
    0x3fd04fb80e37fdae,    0x3feef178a3e473c2,     //   0.25487,   0.96698
    0x3fcfdcdc1adfedf8,    0x3feefe220c0b95ec,     //   0.24893,   0.96852
    0x3fcf19f97b215f1a,    0x3fef0a7efb9230d7,     //   0.24298,   0.97003
    0x3fce56ca1e101a1b,    0x3fef168f53f7205d,     //   0.23702,    0.9715
    0x3fcd934fe5454311,    0x3fef2252f7763ada,     //   0.23106,   0.97294
    0x3fcccf8cb312b286,    0x3fef2dc9c9089a9d,     //   0.22508,   0.97434
    0x3fcc0b826a7e4f63,    0x3fef38f3ac64e589,     //    0.2191,    0.9757
    0x3fcb4732ef3d6722,    0x3fef43d085ff92dd,     //   0.21311,   0.97703
    0x3fca82a025b00451,    0x3fef4e603b0b2f2d,     //   0.20711,   0.97832
    0x3fc9bdcbf2dc4366,    0x3fef58a2b1789e84,     //    0.2011,   0.97957
    0x3fc8f8b83c69a60a,    0x3fef6297cff75cb0,     //   0.19509,   0.98079
    0x3fc83366e89c64c5,    0x3fef6c3f7df5bbb7,     //   0.18907,   0.98196
    0x3fc76dd9de50bf31,    0x3fef7599a3a12077,     //   0.18304,   0.98311
    0x3fc6a81304f64ab2,    0x3fef7ea629e63d6e,     //     0.177,   0.98421
    0x3fc5e214448b3fc6,    0x3fef8764fa714ba9,     //   0.17096,   0.98528
    0x3fc51bdf8597c5f2,    0x3fef8fd5ffae41db,     //   0.16491,   0.98631
    0x3fc45576b1293e5a,    0x3fef97f924c9099b,     //   0.15886,    0.9873
    0x3fc38edbb0cd8d14,    0x3fef9fce55adb2c8,     //    0.1528,   0.98826
    0x3fc2c8106e8e613a,    0x3fefa7557f08a517,     //   0.14673,   0.98918
    0x3fc20116d4ec7bce,    0x3fefae8e8e46cfbb,     //   0.14066,   0.99006
    0x3fc139f0cedaf576,    0x3fefb5797195d741,     //   0.13458,    0.9909
    0x3fc072a047ba831d,    0x3fefbc1617e44186,     //    0.1285,   0.99171
    0x3fbf564e56a9730e,    0x3fefc26470e19fd3,     //   0.12241,   0.99248
    0x3fbdc70ecbae9fc8,    0x3fefc8646cfeb721,     //   0.11632,   0.99321
    0x3fbc3785c79ec2d5,    0x3fefce15fd6da67b,     //   0.11022,   0.99391
    0x3fbaa7b724495c04,    0x3fefd37914220b84,     //   0.10412,   0.99456
    0x3fb917a6bc29b42c,    0x3fefd88da3d12526,     //  0.098017,   0.99518
    0x3fb787586a5d5b21,    0x3fefdd539ff1f456,     //  0.091909,   0.99577
    0x3fb5f6d00a9aa419,    0x3fefe1cafcbd5b09,     //  0.085797,   0.99631
    0x3fb4661179272096,    0x3fefe5f3af2e3940,     //  0.079682,   0.99682
    0x3fb2d52092ce19f6,    0x3fefe9cdad01883a,     //  0.073565,   0.99729
    0x3fb1440134d709b2,    0x3fefed58ecb673c4,     //  0.067444,   0.99772
    0x3faf656e79f820e0,    0x3feff095658e71ad,     //  0.061321,   0.99812
    0x3fac428d12c0d7e3,    0x3feff3830f8d575c,     //  0.055195,   0.99848
    0x3fa91f65f10dd814,    0x3feff621e3796d7e,     //  0.049068,    0.9988
    0x3fa5fc00d290cd43,    0x3feff871dadb81df,     //  0.042938,   0.99908
    0x3fa2d865759455cd,    0x3feffa72effef75d,     //  0.036807,   0.99932
    0x3f9f693731d1cf01,    0x3feffc251df1d3f8,     //  0.030675,   0.99953
    0x3f992155f7a3667e,    0x3feffd886084cd0d,     //  0.024541,    0.9997
    0x3f92d936bbe30efd,    0x3feffe9cb44b51a1,     //  0.018407,   0.99983
    0x3f8921d1fcdec784,    0x3fefff62169b92db,     //  0.012272,   0.99992
    0x3f7921f0fe670071,    0x3fefffd8858e8a92,     // 0.0061359,   0.99998
    0x0000000000000000,    0x3ff0000000000000,     //         0,         1
    0xbf7921f0fe670071,    0x3fefffd8858e8a92,     //-0.0061359,   0.99998
    0xbf8921d1fcdec784,    0x3fefff62169b92db,     // -0.012272,   0.99992
    0xbf92d936bbe30efd,    0x3feffe9cb44b51a1,     // -0.018407,   0.99983
    0xbf992155f7a3667e,    0x3feffd886084cd0d,     // -0.024541,    0.9997
    0xbf9f693731d1cf01,    0x3feffc251df1d3f8,     // -0.030675,   0.99953
    0xbfa2d865759455cd,    0x3feffa72effef75d,     // -0.036807,   0.99932
    0xbfa5fc00d290cd43,    0x3feff871dadb81df,     // -0.042938,   0.99908
    0xbfa91f65f10dd814,    0x3feff621e3796d7e,     // -0.049068,    0.9988
    0xbfac428d12c0d7e3,    0x3feff3830f8d575c,     // -0.055195,   0.99848
    0xbfaf656e79f820e0,    0x3feff095658e71ad,     // -0.061321,   0.99812
    0xbfb1440134d709b2,    0x3fefed58ecb673c4,     // -0.067444,   0.99772
    0xbfb2d52092ce19f6,    0x3fefe9cdad01883a,     // -0.073565,   0.99729
    0xbfb4661179272096,    0x3fefe5f3af2e3940,     // -0.079682,   0.99682
    0xbfb5f6d00a9aa419,    0x3fefe1cafcbd5b09,     // -0.085797,   0.99631
    0xbfb787586a5d5b21,    0x3fefdd539ff1f456,     // -0.091909,   0.99577
    0xbfb917a6bc29b42c,    0x3fefd88da3d12526,     // -0.098017,   0.99518
    0xbfbaa7b724495c04,    0x3fefd37914220b84,     //  -0.10412,   0.99456
    0xbfbc3785c79ec2d5,    0x3fefce15fd6da67b,     //  -0.11022,   0.99391
    0xbfbdc70ecbae9fc8,    0x3fefc8646cfeb721,     //  -0.11632,   0.99321
    0xbfbf564e56a9730e,    0x3fefc26470e19fd3,     //  -0.12241,   0.99248
    0xbfc072a047ba831d,    0x3fefbc1617e44186,     //   -0.1285,   0.99171
    0xbfc139f0cedaf576,    0x3fefb5797195d741,     //  -0.13458,    0.9909
    0xbfc20116d4ec7bce,    0x3fefae8e8e46cfbb,     //  -0.14066,   0.99006
    0xbfc2c8106e8e613a,    0x3fefa7557f08a517,     //  -0.14673,   0.98918
    0xbfc38edbb0cd8d14,    0x3fef9fce55adb2c8,     //   -0.1528,   0.98826
    0xbfc45576b1293e5a,    0x3fef97f924c9099b,     //  -0.15886,    0.9873
    0xbfc51bdf8597c5f2,    0x3fef8fd5ffae41db,     //  -0.16491,   0.98631
    0xbfc5e214448b3fc6,    0x3fef8764fa714ba9,     //  -0.17096,   0.98528
    0xbfc6a81304f64ab2,    0x3fef7ea629e63d6e,     //    -0.177,   0.98421
    0xbfc76dd9de50bf31,    0x3fef7599a3a12077,     //  -0.18304,   0.98311
    0xbfc83366e89c64c5,    0x3fef6c3f7df5bbb7,     //  -0.18907,   0.98196
    0xbfc8f8b83c69a60a,    0x3fef6297cff75cb0,     //  -0.19509,   0.98079
    0xbfc9bdcbf2dc4366,    0x3fef58a2b1789e84,     //   -0.2011,   0.97957
    0xbfca82a025b00451,    0x3fef4e603b0b2f2d,     //  -0.20711,   0.97832
    0xbfcb4732ef3d6722,    0x3fef43d085ff92dd,     //  -0.21311,   0.97703
    0xbfcc0b826a7e4f63,    0x3fef38f3ac64e589,     //   -0.2191,    0.9757
    0xbfcccf8cb312b286,    0x3fef2dc9c9089a9d,     //  -0.22508,   0.97434
    0xbfcd934fe5454311,    0x3fef2252f7763ada,     //  -0.23106,   0.97294
    0xbfce56ca1e101a1b,    0x3fef168f53f7205d,     //  -0.23702,    0.9715
    0xbfcf19f97b215f1a,    0x3fef0a7efb9230d7,     //  -0.24298,   0.97003
    0xbfcfdcdc1adfedf8,    0x3feefe220c0b95ec,     //  -0.24893,   0.96852
    0xbfd04fb80e37fdae,    0x3feef178a3e473c2,     //  -0.25487,   0.96698
    0xbfd0b0d9cfdbdb90,    0x3feee482e25a9dbc,     //  -0.26079,   0.96539
    0xbfd111d262b1f677,    0x3feed740e7684963,     //  -0.26671,   0.96378
    0xbfd172a0d7765177,    0x3feec9b2d3c3bf84,     //  -0.27262,   0.96212
    0xbfd1d3443f4cdb3d,    0x3feebbd8c8df0b74,     //  -0.27852,   0.96043
    0xbfd233bbabc3bb72,    0x3feeadb2e8e7a88e,     //  -0.28441,    0.9587
    0xbfd294062ed59f05,    0x3fee9f4156c62dda,     //  -0.29028,   0.95694
    0xbfd2f422daec0386,    0x3fee9084361df7f3,     //  -0.29615,   0.95514
    0xbfd35410c2e18152,    0x3fee817bab4cd10d,     //  -0.30201,   0.95331
    0xbfd3b3cefa0414b7,    0x3fee7227db6a9744,     //  -0.30785,   0.95144
    0xbfd4135c94176602,    0x3fee6288ec48e112,     //  -0.31368,   0.94953
    0xbfd472b8a5571054,    0x3fee529f04729ffc,     //   -0.3195,   0.94759
    0xbfd4d1e24278e76a,    0x3fee426a4b2bc17e,     //  -0.32531,   0.94561
    0xbfd530d880af3c24,    0x3fee31eae870ce25,     //  -0.33111,   0.94359
    0xbfd58f9a75ab1fdd,    0x3fee212104f686e5,     //  -0.33689,   0.94154
    0xbfd5ee27379ea693,    0x3fee100cca2980ac,     //  -0.34266,   0.93946
    0xbfd64c7ddd3f27c6,    0x3fedfeae622dbe2b,     //  -0.34842,   0.93734
    0xbfd6aa9d7dc77e16,    0x3feded05f7de47da,     //  -0.35416,   0.93518
    0xbfd7088530fa459e,    0x3feddb13b6ccc23d,     //   -0.3599,   0.93299
    0xbfd766340f2418f6,    0x3fedc8d7cb410260,     //  -0.36561,   0.93077
    0xbfd7c3a9311dcce7,    0x3fedb6526238a09b,     //  -0.37132,   0.92851
    0xbfd820e3b04eaac4,    0x3feda383a9668988,     //  -0.37701,   0.92621
    0xbfd87de2a6aea963,    0x3fed906bcf328d46,     //  -0.38268,   0.92388
    0xbfd8daa52ec8a4af,    0x3fed7d0b02b8ecf9,     //  -0.38835,   0.92151
    0xbfd9372a63bc93d7,    0x3fed696173c9e68b,     //  -0.39399,   0.91911
    0xbfd993716141bdfe,    0x3fed556f52e93eb1,     //  -0.39962,   0.91668
    0xbfd9ef7943a8ed8a,    0x3fed4134d14dc93a,     //  -0.40524,   0.91421
    0xbfda4b4127dea1e4,    0x3fed2cb220e0ef9f,     //  -0.41084,   0.91171
    0xbfdaa6c82b6d3fc9,    0x3fed17e7743e35dc,     //  -0.41643,   0.90917
    0xbfdb020d6c7f4009,    0x3fed02d4feb2bd92,     //    -0.422,    0.9066
    0xbfdb5d1009e15cc0,    0x3feced7af43cc773,     //  -0.42756,   0.90399
    0xbfdbb7cf2304bd01,    0x3fecd7d9898b32f6,     //  -0.43309,   0.90135
    0xbfdc1249d8011ee7,    0x3fecc1f0f3fcfc5c,     //  -0.43862,   0.89867
    0xbfdc6c7f4997000a,    0x3fecabc169a0b901,     //  -0.44412,   0.89597
    0xbfdcc66e9931c45d,    0x3fec954b213411f5,     //  -0.44961,   0.89322
    0xbfdd2016e8e9db5b,    0x3fec7e8e52233cf3,     //  -0.45508,   0.89045
    0xbfdd79775b86e389,    0x3fec678b3488739b,     //  -0.46054,   0.88764
    0xbfddd28f1481cc58,    0x3fec5042012b6907,     //  -0.46598,    0.8848
    0xbfde2b5d3806f63b,    0x3fec38b2f180bdb1,     //   -0.4714,   0.88192
    0xbfde83e0eaf85113,    0x3fec20de3fa971b0,     //   -0.4768,   0.87901
    0xbfdedc1952ef78d5,    0x3fec08c426725549,     //  -0.48218,   0.87607
    0xbfdf3405963fd068,    0x3febf064e15377dd,     //  -0.48755,   0.87309
    0xbfdf8ba4dbf89aba,    0x3febd7c0ac6f952a,     //   -0.4929,   0.87009
    0xbfdfe2f64be7120f,    0x3febbed7c49380ea,     //  -0.49823,   0.86705
    0xbfe01cfc874c3eb7,    0x3feba5aa673590d2,     //  -0.50354,   0.86397
    0xbfe0485626ae221a,    0x3feb8c38d27504e9,     //  -0.50883,   0.86087
    0xbfe073879922ffed,    0x3feb728345196e3e,     //   -0.5141,   0.85773
    0xbfe09e907417c5e1,    0x3feb5889fe921405,     //  -0.51936,   0.85456
    0xbfe0c9704d5d898f,    0x3feb3e4d3ef55712,     //  -0.52459,   0.85136
    0xbfe0f426bb2a8e7d,    0x3feb23cd470013b4,     //   -0.5298,   0.84812
    0xbfe11eb3541b4b22,    0x3feb090a58150200,     //    -0.535,   0.84485
    0xbfe14915af336ceb,    0x3feaee04b43c1474,     //  -0.54017,   0.84155
    0xbfe1734d63dedb49,    0x3fead2bc9e21d511,     //  -0.54532,   0.83822
    0xbfe19d5a09f2b9b8,    0x3feab7325916c0d4,     //  -0.55046,   0.83486
    0xbfe1c73b39ae68c8,    0x3fea9b66290ea1a3,     //  -0.55557,   0.83147
    0xbfe1f0f08bbc861b,    0x3fea7f58529fe69d,     //  -0.56066,   0.82805
    0xbfe21a799933eb58,    0x3fea63091b02fae2,     //  -0.56573,   0.82459
    0xbfe243d5fb98ac1f,    0x3fea4678c8119ac8,     //  -0.57078,    0.8211
    0xbfe26d054cdd12df,    0x3fea29a7a0462782,     //  -0.57581,   0.81758
    0xbfe2960727629ca8,    0x3fea0c95eabaf937,     //  -0.58081,   0.81404
    0xbfe2bedb25faf3ea,    0x3fe9ef43ef29af94,     //   -0.5858,   0.81046
    0xbfe2e780e3e8ea16,    0x3fe9d1b1f5ea80d6,     //  -0.59076,   0.80685
    0xbfe30ff7fce17035,    0x3fe9b3e047f38741,     //   -0.5957,   0.80321
    0xbfe338400d0c8e57,    0x3fe995cf2ed80d22,     //  -0.60062,   0.79954
    0xbfe36058b10659f3,    0x3fe9777ef4c7d742,     //  -0.60551,   0.79584
    0xbfe3884185dfeb22,    0x3fe958efe48e6dd7,     //  -0.61038,   0.79211
    0xbfe3affa292050b9,    0x3fe93a22499263fc,     //  -0.61523,   0.78835
    0xbfe3d78238c58343,    0x3fe91b166fd49da2,     //  -0.62006,   0.78456
    0xbfe3fed9534556d4,    0x3fe8fbcca3ef940d,     //  -0.62486,   0.78074
    0xbfe425ff178e6bb1,    0x3fe8dc45331698cc,     //  -0.62964,   0.77689
    0xbfe44cf325091dd6,    0x3fe8bc806b151741,     //  -0.63439,   0.77301
    0xbfe473b51b987347,    0x3fe89c7e9a4dd4ab,     //  -0.63912,    0.7691
    0xbfe49a449b9b0938,    0x3fe87c400fba2ebf,     //  -0.64383,   0.76517
    0xbfe4c0a145ec0004,    0x3fe85bc51ae958cc,     //  -0.64851,    0.7612
    0xbfe4e6cabbe3e5e9,    0x3fe83b0e0bff976e,     //  -0.65317,   0.75721
    0xbfe50cc09f59a09b,    0x3fe81a1b33b57acc,     //  -0.65781,   0.75319
    0xbfe5328292a35596,    0x3fe7f8ece3571771,     //  -0.66242,   0.74914
    0xbfe5581038975137,    0x3fe7d7836cc33db2,     //    -0.667,   0.74506
    0xbfe57d69348cec9f,    0x3fe7b5df226aafb0,     //  -0.67156,   0.74095
    0xbfe5a28d2a5d7250,    0x3fe79400574f55e4,     //  -0.67609,   0.73682
    0xbfe5c77bbe65018c,    0x3fe771e75f037261,     //   -0.6806,   0.73265
    0xbfe5ec3495837074,    0x3fe74f948da8d28d,     //  -0.68508,   0.72846
    0xbfe610b7551d2cde,    0x3fe72d0837efff97,     //  -0.68954,   0.72425
    0xbfe63503a31c1be8,    0x3fe70a42b3176d7a,     //  -0.69397,      0.72
    0xbfe6591925f0783e,    0x3fe6e74454eaa8ae,     //  -0.69838,   0.71573
    0xbfe67cf78491af10,    0x3fe6c40d73c18275,     //  -0.70275,   0.71143
    0xbfe6a09e667f3bcc,    0x3fe6a09e667f3bcc,     //  -0.70711,   0.70711
    0xbfe6c40d73c18275,    0x3fe67cf78491af10,     //  -0.71143,   0.70275
    0xbfe6e74454eaa8ae,    0x3fe6591925f0783e,     //  -0.71573,   0.69838
    0xbfe70a42b3176d7a,    0x3fe63503a31c1be8,     //     -0.72,   0.69397
    0xbfe72d0837efff97,    0x3fe610b7551d2cde,     //  -0.72425,   0.68954
    0xbfe74f948da8d28d,    0x3fe5ec3495837074,     //  -0.72846,   0.68508
    0xbfe771e75f037261,    0x3fe5c77bbe65018c,     //  -0.73265,    0.6806
    0xbfe79400574f55e4,    0x3fe5a28d2a5d7250,     //  -0.73682,   0.67609
    0xbfe7b5df226aafb0,    0x3fe57d69348cec9f,     //  -0.74095,   0.67156
    0xbfe7d7836cc33db2,    0x3fe5581038975137,     //  -0.74506,     0.667
    0xbfe7f8ece3571771,    0x3fe5328292a35596,     //  -0.74914,   0.66242
    0xbfe81a1b33b57acc,    0x3fe50cc09f59a09b,     //  -0.75319,   0.65781
    0xbfe83b0e0bff976e,    0x3fe4e6cabbe3e5e9,     //  -0.75721,   0.65317
    0xbfe85bc51ae958cc,    0x3fe4c0a145ec0004,     //   -0.7612,   0.64851
    0xbfe87c400fba2ebf,    0x3fe49a449b9b0938,     //  -0.76517,   0.64383
    0xbfe89c7e9a4dd4ab,    0x3fe473b51b987347,     //   -0.7691,   0.63912
    0xbfe8bc806b151741,    0x3fe44cf325091dd6,     //  -0.77301,   0.63439
    0xbfe8dc45331698cc,    0x3fe425ff178e6bb1,     //  -0.77689,   0.62964
    0xbfe8fbcca3ef940d,    0x3fe3fed9534556d4,     //  -0.78074,   0.62486
    0xbfe91b166fd49da2,    0x3fe3d78238c58343,     //  -0.78456,   0.62006
    0xbfe93a22499263fc,    0x3fe3affa292050b9,     //  -0.78835,   0.61523
    0xbfe958efe48e6dd7,    0x3fe3884185dfeb22,     //  -0.79211,   0.61038
    0xbfe9777ef4c7d742,    0x3fe36058b10659f3,     //  -0.79584,   0.60551
    0xbfe995cf2ed80d22,    0x3fe338400d0c8e57,     //  -0.79954,   0.60062
    0xbfe9b3e047f38741,    0x3fe30ff7fce17035,     //  -0.80321,    0.5957
    0xbfe9d1b1f5ea80d6,    0x3fe2e780e3e8ea16,     //  -0.80685,   0.59076
    0xbfe9ef43ef29af94,    0x3fe2bedb25faf3ea,     //  -0.81046,    0.5858
    0xbfea0c95eabaf937,    0x3fe2960727629ca8,     //  -0.81404,   0.58081
    0xbfea29a7a0462782,    0x3fe26d054cdd12df,     //  -0.81758,   0.57581
    0xbfea4678c8119ac8,    0x3fe243d5fb98ac1f,     //   -0.8211,   0.57078
    0xbfea63091b02fae2,    0x3fe21a799933eb58,     //  -0.82459,   0.56573
    0xbfea7f58529fe69d,    0x3fe1f0f08bbc861b,     //  -0.82805,   0.56066
    0xbfea9b66290ea1a3,    0x3fe1c73b39ae68c8,     //  -0.83147,   0.55557
    0xbfeab7325916c0d4,    0x3fe19d5a09f2b9b8,     //  -0.83486,   0.55046
    0xbfead2bc9e21d511,    0x3fe1734d63dedb49,     //  -0.83822,   0.54532
    0xbfeaee04b43c1474,    0x3fe14915af336ceb,     //  -0.84155,   0.54017
    0xbfeb090a58150200,    0x3fe11eb3541b4b22,     //  -0.84485,     0.535
    0xbfeb23cd470013b4,    0x3fe0f426bb2a8e7d,     //  -0.84812,    0.5298
    0xbfeb3e4d3ef55712,    0x3fe0c9704d5d898f,     //  -0.85136,   0.52459
    0xbfeb5889fe921405,    0x3fe09e907417c5e1,     //  -0.85456,   0.51936
    0xbfeb728345196e3e,    0x3fe073879922ffed,     //  -0.85773,    0.5141
    0xbfeb8c38d27504e9,    0x3fe0485626ae221a,     //  -0.86087,   0.50883
    0xbfeba5aa673590d2,    0x3fe01cfc874c3eb7,     //  -0.86397,   0.50354
    0xbfebbed7c49380ea,    0x3fdfe2f64be7120f,     //  -0.86705,   0.49823
    0xbfebd7c0ac6f952a,    0x3fdf8ba4dbf89aba,     //  -0.87009,    0.4929
    0xbfebf064e15377dd,    0x3fdf3405963fd068,     //  -0.87309,   0.48755
    0xbfec08c426725549,    0x3fdedc1952ef78d5,     //  -0.87607,   0.48218
    0xbfec20de3fa971b0,    0x3fde83e0eaf85113,     //  -0.87901,    0.4768
    0xbfec38b2f180bdb1,    0x3fde2b5d3806f63b,     //  -0.88192,    0.4714
    0xbfec5042012b6907,    0x3fddd28f1481cc58,     //   -0.8848,   0.46598
    0xbfec678b3488739b,    0x3fdd79775b86e389,     //  -0.88764,   0.46054
    0xbfec7e8e52233cf3,    0x3fdd2016e8e9db5b,     //  -0.89045,   0.45508
    0xbfec954b213411f5,    0x3fdcc66e9931c45d,     //  -0.89322,   0.44961
    0xbfecabc169a0b901,    0x3fdc6c7f4997000a,     //  -0.89597,   0.44412
    0xbfecc1f0f3fcfc5c,    0x3fdc1249d8011ee7,     //  -0.89867,   0.43862
    0xbfecd7d9898b32f6,    0x3fdbb7cf2304bd01,     //  -0.90135,   0.43309
    0xbfeced7af43cc773,    0x3fdb5d1009e15cc0,     //  -0.90399,   0.42756
    0xbfed02d4feb2bd92,    0x3fdb020d6c7f4009,     //   -0.9066,     0.422
    0xbfed17e7743e35dc,    0x3fdaa6c82b6d3fc9,     //  -0.90917,   0.41643
    0xbfed2cb220e0ef9f,    0x3fda4b4127dea1e4,     //  -0.91171,   0.41084
    0xbfed4134d14dc93a,    0x3fd9ef7943a8ed8a,     //  -0.91421,   0.40524
    0xbfed556f52e93eb1,    0x3fd993716141bdfe,     //  -0.91668,   0.39962
    0xbfed696173c9e68b,    0x3fd9372a63bc93d7,     //  -0.91911,   0.39399
    0xbfed7d0b02b8ecf9,    0x3fd8daa52ec8a4af,     //  -0.92151,   0.38835
    0xbfed906bcf328d46,    0x3fd87de2a6aea963,     //  -0.92388,   0.38268
    0xbfeda383a9668988,    0x3fd820e3b04eaac4,     //  -0.92621,   0.37701
    0xbfedb6526238a09b,    0x3fd7c3a9311dcce7,     //  -0.92851,   0.37132
    0xbfedc8d7cb410260,    0x3fd766340f2418f6,     //  -0.93077,   0.36561
    0xbfeddb13b6ccc23d,    0x3fd7088530fa459e,     //  -0.93299,    0.3599
    0xbfeded05f7de47da,    0x3fd6aa9d7dc77e16,     //  -0.93518,   0.35416
    0xbfedfeae622dbe2b,    0x3fd64c7ddd3f27c6,     //  -0.93734,   0.34842
    0xbfee100cca2980ac,    0x3fd5ee27379ea693,     //  -0.93946,   0.34266
    0xbfee212104f686e5,    0x3fd58f9a75ab1fdd,     //  -0.94154,   0.33689
    0xbfee31eae870ce25,    0x3fd530d880af3c24,     //  -0.94359,   0.33111
    0xbfee426a4b2bc17e,    0x3fd4d1e24278e76a,     //  -0.94561,   0.32531
    0xbfee529f04729ffc,    0x3fd472b8a5571054,     //  -0.94759,    0.3195
    0xbfee6288ec48e112,    0x3fd4135c94176602,     //  -0.94953,   0.31368
    0xbfee7227db6a9744,    0x3fd3b3cefa0414b7,     //  -0.95144,   0.30785
    0xbfee817bab4cd10d,    0x3fd35410c2e18152,     //  -0.95331,   0.30201
    0xbfee9084361df7f3,    0x3fd2f422daec0386,     //  -0.95514,   0.29615
    0xbfee9f4156c62dda,    0x3fd294062ed59f05,     //  -0.95694,   0.29028
    0xbfeeadb2e8e7a88e,    0x3fd233bbabc3bb72,     //   -0.9587,   0.28441
    0xbfeebbd8c8df0b74,    0x3fd1d3443f4cdb3d,     //  -0.96043,   0.27852
    0xbfeec9b2d3c3bf84,    0x3fd172a0d7765177,     //  -0.96212,   0.27262
    0xbfeed740e7684963,    0x3fd111d262b1f677,     //  -0.96378,   0.26671
    0xbfeee482e25a9dbc,    0x3fd0b0d9cfdbdb90,     //  -0.96539,   0.26079
    0xbfeef178a3e473c2,    0x3fd04fb80e37fdae,     //  -0.96698,   0.25487
    0xbfeefe220c0b95ec,    0x3fcfdcdc1adfedf8,     //  -0.96852,   0.24893
    0xbfef0a7efb9230d7,    0x3fcf19f97b215f1a,     //  -0.97003,   0.24298
    0xbfef168f53f7205d,    0x3fce56ca1e101a1b,     //   -0.9715,   0.23702
    0xbfef2252f7763ada,    0x3fcd934fe5454311,     //  -0.97294,   0.23106
    0xbfef2dc9c9089a9d,    0x3fcccf8cb312b286,     //  -0.97434,   0.22508
    0xbfef38f3ac64e589,    0x3fcc0b826a7e4f63,     //   -0.9757,    0.2191
    0xbfef43d085ff92dd,    0x3fcb4732ef3d6722,     //  -0.97703,   0.21311
    0xbfef4e603b0b2f2d,    0x3fca82a025b00451,     //  -0.97832,   0.20711
    0xbfef58a2b1789e84,    0x3fc9bdcbf2dc4366,     //  -0.97957,    0.2011
    0xbfef6297cff75cb0,    0x3fc8f8b83c69a60a,     //  -0.98079,   0.19509
    0xbfef6c3f7df5bbb7,    0x3fc83366e89c64c5,     //  -0.98196,   0.18907
    0xbfef7599a3a12077,    0x3fc76dd9de50bf31,     //  -0.98311,   0.18304
    0xbfef7ea629e63d6e,    0x3fc6a81304f64ab2,     //  -0.98421,     0.177
    0xbfef8764fa714ba9,    0x3fc5e214448b3fc6,     //  -0.98528,   0.17096
    0xbfef8fd5ffae41db,    0x3fc51bdf8597c5f2,     //  -0.98631,   0.16491
    0xbfef97f924c9099b,    0x3fc45576b1293e5a,     //   -0.9873,   0.15886
    0xbfef9fce55adb2c8,    0x3fc38edbb0cd8d14,     //  -0.98826,    0.1528
    0xbfefa7557f08a517,    0x3fc2c8106e8e613a,     //  -0.98918,   0.14673
    0xbfefae8e8e46cfbb,    0x3fc20116d4ec7bce,     //  -0.99006,   0.14066
    0xbfefb5797195d741,    0x3fc139f0cedaf576,     //   -0.9909,   0.13458
    0xbfefbc1617e44186,    0x3fc072a047ba831d,     //  -0.99171,    0.1285
    0xbfefc26470e19fd3,    0x3fbf564e56a9730e,     //  -0.99248,   0.12241
    0xbfefc8646cfeb721,    0x3fbdc70ecbae9fc8,     //  -0.99321,   0.11632
    0xbfefce15fd6da67b,    0x3fbc3785c79ec2d5,     //  -0.99391,   0.11022
    0xbfefd37914220b84,    0x3fbaa7b724495c04,     //  -0.99456,   0.10412
    0xbfefd88da3d12526,    0x3fb917a6bc29b42c,     //  -0.99518,  0.098017
    0xbfefdd539ff1f456,    0x3fb787586a5d5b21,     //  -0.99577,  0.091909
    0xbfefe1cafcbd5b09,    0x3fb5f6d00a9aa419,     //  -0.99631,  0.085797
    0xbfefe5f3af2e3940,    0x3fb4661179272096,     //  -0.99682,  0.079682
    0xbfefe9cdad01883a,    0x3fb2d52092ce19f6,     //  -0.99729,  0.073565
    0xbfefed58ecb673c4,    0x3fb1440134d709b2,     //  -0.99772,  0.067444
    0xbfeff095658e71ad,    0x3faf656e79f820e0,     //  -0.99812,  0.061321
    0xbfeff3830f8d575c,    0x3fac428d12c0d7e3,     //  -0.99848,  0.055195
    0xbfeff621e3796d7e,    0x3fa91f65f10dd814,     //   -0.9988,  0.049068
    0xbfeff871dadb81df,    0x3fa5fc00d290cd43,     //  -0.99908,  0.042938
    0xbfeffa72effef75d,    0x3fa2d865759455cd,     //  -0.99932,  0.036807
    0xbfeffc251df1d3f8,    0x3f9f693731d1cf01,     //  -0.99953,  0.030675
    0xbfeffd886084cd0d,    0x3f992155f7a3667e,     //   -0.9997,  0.024541
    0xbfeffe9cb44b51a1,    0x3f92d936bbe30efd,     //  -0.99983,  0.018407
    0xbfefff62169b92db,    0x3f8921d1fcdec784,     //  -0.99992,  0.012272
    0xbfefffd8858e8a92,    0x3f7921f0fe670071,     //  -0.99998, 0.0061359
    0xbff0000000000000,    0x0000000000000000,     //        -1,         0
    0xbfefffd8858e8a92,    0xbf7921f0fe670071,     //  -0.99998,-0.0061359
    0xbfefff62169b92db,    0xbf8921d1fcdec784,     //  -0.99992, -0.012272
    0xbfeffe9cb44b51a1,    0xbf92d936bbe30efd,     //  -0.99983, -0.018407
    0xbfeffd886084cd0d,    0xbf992155f7a3667e,     //   -0.9997, -0.024541
    0xbfeffc251df1d3f8,    0xbf9f693731d1cf01,     //  -0.99953, -0.030675
    0xbfeffa72effef75d,    0xbfa2d865759455cd,     //  -0.99932, -0.036807
    0xbfeff871dadb81df,    0xbfa5fc00d290cd43,     //  -0.99908, -0.042938
    0xbfeff621e3796d7e,    0xbfa91f65f10dd814,     //   -0.9988, -0.049068
    0xbfeff3830f8d575c,    0xbfac428d12c0d7e3,     //  -0.99848, -0.055195
    0xbfeff095658e71ad,    0xbfaf656e79f820e0,     //  -0.99812, -0.061321
    0xbfefed58ecb673c4,    0xbfb1440134d709b2,     //  -0.99772, -0.067444
    0xbfefe9cdad01883a,    0xbfb2d52092ce19f6,     //  -0.99729, -0.073565
    0xbfefe5f3af2e3940,    0xbfb4661179272096,     //  -0.99682, -0.079682
    0xbfefe1cafcbd5b09,    0xbfb5f6d00a9aa419,     //  -0.99631, -0.085797
    0xbfefdd539ff1f456,    0xbfb787586a5d5b21,     //  -0.99577, -0.091909
    0xbfefd88da3d12526,    0xbfb917a6bc29b42c,     //  -0.99518, -0.098017
    0xbfefd37914220b84,    0xbfbaa7b724495c04,     //  -0.99456,  -0.10412
    0xbfefce15fd6da67b,    0xbfbc3785c79ec2d5,     //  -0.99391,  -0.11022
    0xbfefc8646cfeb721,    0xbfbdc70ecbae9fc8,     //  -0.99321,  -0.11632
    0xbfefc26470e19fd3,    0xbfbf564e56a9730e,     //  -0.99248,  -0.12241
    0xbfefbc1617e44186,    0xbfc072a047ba831d,     //  -0.99171,   -0.1285
    0xbfefb5797195d741,    0xbfc139f0cedaf576,     //   -0.9909,  -0.13458
    0xbfefae8e8e46cfbb,    0xbfc20116d4ec7bce,     //  -0.99006,  -0.14066
    0xbfefa7557f08a517,    0xbfc2c8106e8e613a,     //  -0.98918,  -0.14673
    0xbfef9fce55adb2c8,    0xbfc38edbb0cd8d14,     //  -0.98826,   -0.1528
    0xbfef97f924c9099b,    0xbfc45576b1293e5a,     //   -0.9873,  -0.15886
    0xbfef8fd5ffae41db,    0xbfc51bdf8597c5f2,     //  -0.98631,  -0.16491
    0xbfef8764fa714ba9,    0xbfc5e214448b3fc6,     //  -0.98528,  -0.17096
    0xbfef7ea629e63d6e,    0xbfc6a81304f64ab2,     //  -0.98421,    -0.177
    0xbfef7599a3a12077,    0xbfc76dd9de50bf31,     //  -0.98311,  -0.18304
    0xbfef6c3f7df5bbb7,    0xbfc83366e89c64c5,     //  -0.98196,  -0.18907
    0xbfef6297cff75cb0,    0xbfc8f8b83c69a60a,     //  -0.98079,  -0.19509
    0xbfef58a2b1789e84,    0xbfc9bdcbf2dc4366,     //  -0.97957,   -0.2011
    0xbfef4e603b0b2f2d,    0xbfca82a025b00451,     //  -0.97832,  -0.20711
    0xbfef43d085ff92dd,    0xbfcb4732ef3d6722,     //  -0.97703,  -0.21311
    0xbfef38f3ac64e589,    0xbfcc0b826a7e4f63,     //   -0.9757,   -0.2191
    0xbfef2dc9c9089a9d,    0xbfcccf8cb312b286,     //  -0.97434,  -0.22508
    0xbfef2252f7763ada,    0xbfcd934fe5454311,     //  -0.97294,  -0.23106
    0xbfef168f53f7205d,    0xbfce56ca1e101a1b,     //   -0.9715,  -0.23702
    0xbfef0a7efb9230d7,    0xbfcf19f97b215f1a,     //  -0.97003,  -0.24298
    0xbfeefe220c0b95ec,    0xbfcfdcdc1adfedf8,     //  -0.96852,  -0.24893
    0xbfeef178a3e473c2,    0xbfd04fb80e37fdae,     //  -0.96698,  -0.25487
    0xbfeee482e25a9dbc,    0xbfd0b0d9cfdbdb90,     //  -0.96539,  -0.26079
    0xbfeed740e7684963,    0xbfd111d262b1f677,     //  -0.96378,  -0.26671
    0xbfeec9b2d3c3bf84,    0xbfd172a0d7765177,     //  -0.96212,  -0.27262
    0xbfeebbd8c8df0b74,    0xbfd1d3443f4cdb3d,     //  -0.96043,  -0.27852
    0xbfeeadb2e8e7a88e,    0xbfd233bbabc3bb72,     //   -0.9587,  -0.28441
    0xbfee9f4156c62dda,    0xbfd294062ed59f05,     //  -0.95694,  -0.29028
    0xbfee9084361df7f3,    0xbfd2f422daec0386,     //  -0.95514,  -0.29615
    0xbfee817bab4cd10d,    0xbfd35410c2e18152,     //  -0.95331,  -0.30201
    0xbfee7227db6a9744,    0xbfd3b3cefa0414b7,     //  -0.95144,  -0.30785
    0xbfee6288ec48e112,    0xbfd4135c94176602,     //  -0.94953,  -0.31368
    0xbfee529f04729ffc,    0xbfd472b8a5571054,     //  -0.94759,   -0.3195
    0xbfee426a4b2bc17e,    0xbfd4d1e24278e76a,     //  -0.94561,  -0.32531
    0xbfee31eae870ce25,    0xbfd530d880af3c24,     //  -0.94359,  -0.33111
    0xbfee212104f686e5,    0xbfd58f9a75ab1fdd,     //  -0.94154,  -0.33689
    0xbfee100cca2980ac,    0xbfd5ee27379ea693,     //  -0.93946,  -0.34266
    0xbfedfeae622dbe2b,    0xbfd64c7ddd3f27c6,     //  -0.93734,  -0.34842
    0xbfeded05f7de47da,    0xbfd6aa9d7dc77e16,     //  -0.93518,  -0.35416
    0xbfeddb13b6ccc23d,    0xbfd7088530fa459e,     //  -0.93299,   -0.3599
    0xbfedc8d7cb410260,    0xbfd766340f2418f6,     //  -0.93077,  -0.36561
    0xbfedb6526238a09b,    0xbfd7c3a9311dcce7,     //  -0.92851,  -0.37132
    0xbfeda383a9668988,    0xbfd820e3b04eaac4,     //  -0.92621,  -0.37701
    0xbfed906bcf328d46,    0xbfd87de2a6aea963,     //  -0.92388,  -0.38268
    0xbfed7d0b02b8ecf9,    0xbfd8daa52ec8a4af,     //  -0.92151,  -0.38835
    0xbfed696173c9e68b,    0xbfd9372a63bc93d7,     //  -0.91911,  -0.39399
    0xbfed556f52e93eb1,    0xbfd993716141bdfe,     //  -0.91668,  -0.39962
    0xbfed4134d14dc93a,    0xbfd9ef7943a8ed8a,     //  -0.91421,  -0.40524
    0xbfed2cb220e0ef9f,    0xbfda4b4127dea1e4,     //  -0.91171,  -0.41084
    0xbfed17e7743e35dc,    0xbfdaa6c82b6d3fc9,     //  -0.90917,  -0.41643
    0xbfed02d4feb2bd92,    0xbfdb020d6c7f4009,     //   -0.9066,    -0.422
    0xbfeced7af43cc773,    0xbfdb5d1009e15cc0,     //  -0.90399,  -0.42756
    0xbfecd7d9898b32f6,    0xbfdbb7cf2304bd01,     //  -0.90135,  -0.43309
    0xbfecc1f0f3fcfc5c,    0xbfdc1249d8011ee7,     //  -0.89867,  -0.43862
    0xbfecabc169a0b901,    0xbfdc6c7f4997000a,     //  -0.89597,  -0.44412
    0xbfec954b213411f5,    0xbfdcc66e9931c45d,     //  -0.89322,  -0.44961
    0xbfec7e8e52233cf3,    0xbfdd2016e8e9db5b,     //  -0.89045,  -0.45508
    0xbfec678b3488739b,    0xbfdd79775b86e389,     //  -0.88764,  -0.46054
    0xbfec5042012b6907,    0xbfddd28f1481cc58,     //   -0.8848,  -0.46598
    0xbfec38b2f180bdb1,    0xbfde2b5d3806f63b,     //  -0.88192,   -0.4714
    0xbfec20de3fa971b0,    0xbfde83e0eaf85113,     //  -0.87901,   -0.4768
    0xbfec08c426725549,    0xbfdedc1952ef78d5,     //  -0.87607,  -0.48218
    0xbfebf064e15377dd,    0xbfdf3405963fd068,     //  -0.87309,  -0.48755
    0xbfebd7c0ac6f952a,    0xbfdf8ba4dbf89aba,     //  -0.87009,   -0.4929
    0xbfebbed7c49380ea,    0xbfdfe2f64be7120f,     //  -0.86705,  -0.49823
    0xbfeba5aa673590d2,    0xbfe01cfc874c3eb7,     //  -0.86397,  -0.50354
    0xbfeb8c38d27504e9,    0xbfe0485626ae221a,     //  -0.86087,  -0.50883
    0xbfeb728345196e3e,    0xbfe073879922ffed,     //  -0.85773,   -0.5141
    0xbfeb5889fe921405,    0xbfe09e907417c5e1,     //  -0.85456,  -0.51936
    0xbfeb3e4d3ef55712,    0xbfe0c9704d5d898f,     //  -0.85136,  -0.52459
    0xbfeb23cd470013b4,    0xbfe0f426bb2a8e7d,     //  -0.84812,   -0.5298
    0xbfeb090a58150200,    0xbfe11eb3541b4b22,     //  -0.84485,    -0.535
    0xbfeaee04b43c1474,    0xbfe14915af336ceb,     //  -0.84155,  -0.54017
    0xbfead2bc9e21d511,    0xbfe1734d63dedb49,     //  -0.83822,  -0.54532
    0xbfeab7325916c0d4,    0xbfe19d5a09f2b9b8,     //  -0.83486,  -0.55046
    0xbfea9b66290ea1a3,    0xbfe1c73b39ae68c8,     //  -0.83147,  -0.55557
    0xbfea7f58529fe69d,    0xbfe1f0f08bbc861b,     //  -0.82805,  -0.56066
    0xbfea63091b02fae2,    0xbfe21a799933eb58,     //  -0.82459,  -0.56573
    0xbfea4678c8119ac8,    0xbfe243d5fb98ac1f,     //   -0.8211,  -0.57078
    0xbfea29a7a0462782,    0xbfe26d054cdd12df,     //  -0.81758,  -0.57581
    0xbfea0c95eabaf937,    0xbfe2960727629ca8,     //  -0.81404,  -0.58081
    0xbfe9ef43ef29af94,    0xbfe2bedb25faf3ea,     //  -0.81046,   -0.5858
    0xbfe9d1b1f5ea80d6,    0xbfe2e780e3e8ea16,     //  -0.80685,  -0.59076
    0xbfe9b3e047f38741,    0xbfe30ff7fce17035,     //  -0.80321,   -0.5957
    0xbfe995cf2ed80d22,    0xbfe338400d0c8e57,     //  -0.79954,  -0.60062
    0xbfe9777ef4c7d742,    0xbfe36058b10659f3,     //  -0.79584,  -0.60551
    0xbfe958efe48e6dd7,    0xbfe3884185dfeb22,     //  -0.79211,  -0.61038
    0xbfe93a22499263fc,    0xbfe3affa292050b9,     //  -0.78835,  -0.61523
    0xbfe91b166fd49da2,    0xbfe3d78238c58343,     //  -0.78456,  -0.62006
    0xbfe8fbcca3ef940d,    0xbfe3fed9534556d4,     //  -0.78074,  -0.62486
    0xbfe8dc45331698cc,    0xbfe425ff178e6bb1,     //  -0.77689,  -0.62964
    0xbfe8bc806b151741,    0xbfe44cf325091dd6,     //  -0.77301,  -0.63439
    0xbfe89c7e9a4dd4ab,    0xbfe473b51b987347,     //   -0.7691,  -0.63912
    0xbfe87c400fba2ebf,    0xbfe49a449b9b0938,     //  -0.76517,  -0.64383
    0xbfe85bc51ae958cc,    0xbfe4c0a145ec0004,     //   -0.7612,  -0.64851
    0xbfe83b0e0bff976e,    0xbfe4e6cabbe3e5e9,     //  -0.75721,  -0.65317
    0xbfe81a1b33b57acc,    0xbfe50cc09f59a09b,     //  -0.75319,  -0.65781
    0xbfe7f8ece3571771,    0xbfe5328292a35596,     //  -0.74914,  -0.66242
    0xbfe7d7836cc33db2,    0xbfe5581038975137,     //  -0.74506,    -0.667
    0xbfe7b5df226aafb0,    0xbfe57d69348cec9f,     //  -0.74095,  -0.67156
    0xbfe79400574f55e4,    0xbfe5a28d2a5d7250,     //  -0.73682,  -0.67609
    0xbfe771e75f037261,    0xbfe5c77bbe65018c,     //  -0.73265,   -0.6806
    0xbfe74f948da8d28d,    0xbfe5ec3495837074,     //  -0.72846,  -0.68508
    0xbfe72d0837efff97,    0xbfe610b7551d2cde,     //  -0.72425,  -0.68954
    0xbfe70a42b3176d7a,    0xbfe63503a31c1be8,     //     -0.72,  -0.69397
    0xbfe6e74454eaa8ae,    0xbfe6591925f0783e,     //  -0.71573,  -0.69838
    0xbfe6c40d73c18275,    0xbfe67cf78491af10,     //  -0.71143,  -0.70275
    0xbfe6a09e667f3bcc,    0xbfe6a09e667f3bcc,     //  -0.70711,  -0.70711
    0xbfe67cf78491af10,    0xbfe6c40d73c18275,     //  -0.70275,  -0.71143
    0xbfe6591925f0783e,    0xbfe6e74454eaa8ae,     //  -0.69838,  -0.71573
    0xbfe63503a31c1be8,    0xbfe70a42b3176d7a,     //  -0.69397,     -0.72
    0xbfe610b7551d2cde,    0xbfe72d0837efff97,     //  -0.68954,  -0.72425
    0xbfe5ec3495837074,    0xbfe74f948da8d28d,     //  -0.68508,  -0.72846
    0xbfe5c77bbe65018c,    0xbfe771e75f037261,     //   -0.6806,  -0.73265
    0xbfe5a28d2a5d7250,    0xbfe79400574f55e4,     //  -0.67609,  -0.73682
    0xbfe57d69348cec9f,    0xbfe7b5df226aafb0,     //  -0.67156,  -0.74095
    0xbfe5581038975137,    0xbfe7d7836cc33db2,     //    -0.667,  -0.74506
    0xbfe5328292a35596,    0xbfe7f8ece3571771,     //  -0.66242,  -0.74914
    0xbfe50cc09f59a09b,    0xbfe81a1b33b57acc,     //  -0.65781,  -0.75319
    0xbfe4e6cabbe3e5e9,    0xbfe83b0e0bff976e,     //  -0.65317,  -0.75721
    0xbfe4c0a145ec0004,    0xbfe85bc51ae958cc,     //  -0.64851,   -0.7612
    0xbfe49a449b9b0938,    0xbfe87c400fba2ebf,     //  -0.64383,  -0.76517
    0xbfe473b51b987347,    0xbfe89c7e9a4dd4ab,     //  -0.63912,   -0.7691
    0xbfe44cf325091dd6,    0xbfe8bc806b151741,     //  -0.63439,  -0.77301
    0xbfe425ff178e6bb1,    0xbfe8dc45331698cc,     //  -0.62964,  -0.77689
    0xbfe3fed9534556d4,    0xbfe8fbcca3ef940d,     //  -0.62486,  -0.78074
    0xbfe3d78238c58343,    0xbfe91b166fd49da2,     //  -0.62006,  -0.78456
    0xbfe3affa292050b9,    0xbfe93a22499263fc,     //  -0.61523,  -0.78835
    0xbfe3884185dfeb22,    0xbfe958efe48e6dd7,     //  -0.61038,  -0.79211
    0xbfe36058b10659f3,    0xbfe9777ef4c7d742,     //  -0.60551,  -0.79584
    0xbfe338400d0c8e57,    0xbfe995cf2ed80d22,     //  -0.60062,  -0.79954
    0xbfe30ff7fce17035,    0xbfe9b3e047f38741,     //   -0.5957,  -0.80321
    0xbfe2e780e3e8ea16,    0xbfe9d1b1f5ea80d6,     //  -0.59076,  -0.80685
    0xbfe2bedb25faf3ea,    0xbfe9ef43ef29af94,     //   -0.5858,  -0.81046
    0xbfe2960727629ca8,    0xbfea0c95eabaf937,     //  -0.58081,  -0.81404
    0xbfe26d054cdd12df,    0xbfea29a7a0462782,     //  -0.57581,  -0.81758
    0xbfe243d5fb98ac1f,    0xbfea4678c8119ac8,     //  -0.57078,   -0.8211
    0xbfe21a799933eb58,    0xbfea63091b02fae2,     //  -0.56573,  -0.82459
    0xbfe1f0f08bbc861b,    0xbfea7f58529fe69d,     //  -0.56066,  -0.82805
    0xbfe1c73b39ae68c8,    0xbfea9b66290ea1a3,     //  -0.55557,  -0.83147
    0xbfe19d5a09f2b9b8,    0xbfeab7325916c0d4,     //  -0.55046,  -0.83486
    0xbfe1734d63dedb49,    0xbfead2bc9e21d511,     //  -0.54532,  -0.83822
    0xbfe14915af336ceb,    0xbfeaee04b43c1474,     //  -0.54017,  -0.84155
    0xbfe11eb3541b4b22,    0xbfeb090a58150200,     //    -0.535,  -0.84485
    0xbfe0f426bb2a8e7d,    0xbfeb23cd470013b4,     //   -0.5298,  -0.84812
    0xbfe0c9704d5d898f,    0xbfeb3e4d3ef55712,     //  -0.52459,  -0.85136
    0xbfe09e907417c5e1,    0xbfeb5889fe921405,     //  -0.51936,  -0.85456
    0xbfe073879922ffed,    0xbfeb728345196e3e,     //   -0.5141,  -0.85773
    0xbfe0485626ae221a,    0xbfeb8c38d27504e9,     //  -0.50883,  -0.86087
    0xbfe01cfc874c3eb7,    0xbfeba5aa673590d2,     //  -0.50354,  -0.86397
    0xbfdfe2f64be7120f,    0xbfebbed7c49380ea,     //  -0.49823,  -0.86705
    0xbfdf8ba4dbf89aba,    0xbfebd7c0ac6f952a,     //   -0.4929,  -0.87009
    0xbfdf3405963fd068,    0xbfebf064e15377dd,     //  -0.48755,  -0.87309
    0xbfdedc1952ef78d5,    0xbfec08c426725549,     //  -0.48218,  -0.87607
    0xbfde83e0eaf85113,    0xbfec20de3fa971b0,     //   -0.4768,  -0.87901
    0xbfde2b5d3806f63b,    0xbfec38b2f180bdb1,     //   -0.4714,  -0.88192
    0xbfddd28f1481cc58,    0xbfec5042012b6907,     //  -0.46598,   -0.8848
    0xbfdd79775b86e389,    0xbfec678b3488739b,     //  -0.46054,  -0.88764
    0xbfdd2016e8e9db5b,    0xbfec7e8e52233cf3,     //  -0.45508,  -0.89045
    0xbfdcc66e9931c45d,    0xbfec954b213411f5,     //  -0.44961,  -0.89322
    0xbfdc6c7f4997000a,    0xbfecabc169a0b901,     //  -0.44412,  -0.89597
    0xbfdc1249d8011ee7,    0xbfecc1f0f3fcfc5c,     //  -0.43862,  -0.89867
    0xbfdbb7cf2304bd01,    0xbfecd7d9898b32f6,     //  -0.43309,  -0.90135
    0xbfdb5d1009e15cc0,    0xbfeced7af43cc773,     //  -0.42756,  -0.90399
    0xbfdb020d6c7f4009,    0xbfed02d4feb2bd92,     //    -0.422,   -0.9066
    0xbfdaa6c82b6d3fc9,    0xbfed17e7743e35dc,     //  -0.41643,  -0.90917
    0xbfda4b4127dea1e4,    0xbfed2cb220e0ef9f,     //  -0.41084,  -0.91171
    0xbfd9ef7943a8ed8a,    0xbfed4134d14dc93a,     //  -0.40524,  -0.91421
    0xbfd993716141bdfe,    0xbfed556f52e93eb1,     //  -0.39962,  -0.91668
    0xbfd9372a63bc93d7,    0xbfed696173c9e68b,     //  -0.39399,  -0.91911
    0xbfd8daa52ec8a4af,    0xbfed7d0b02b8ecf9,     //  -0.38835,  -0.92151
    0xbfd87de2a6aea963,    0xbfed906bcf328d46,     //  -0.38268,  -0.92388
    0xbfd820e3b04eaac4,    0xbfeda383a9668988,     //  -0.37701,  -0.92621
    0xbfd7c3a9311dcce7,    0xbfedb6526238a09b,     //  -0.37132,  -0.92851
    0xbfd766340f2418f6,    0xbfedc8d7cb410260,     //  -0.36561,  -0.93077
    0xbfd7088530fa459e,    0xbfeddb13b6ccc23d,     //   -0.3599,  -0.93299
    0xbfd6aa9d7dc77e16,    0xbfeded05f7de47da,     //  -0.35416,  -0.93518
    0xbfd64c7ddd3f27c6,    0xbfedfeae622dbe2b,     //  -0.34842,  -0.93734
    0xbfd5ee27379ea693,    0xbfee100cca2980ac,     //  -0.34266,  -0.93946
    0xbfd58f9a75ab1fdd,    0xbfee212104f686e5,     //  -0.33689,  -0.94154
    0xbfd530d880af3c24,    0xbfee31eae870ce25,     //  -0.33111,  -0.94359
    0xbfd4d1e24278e76a,    0xbfee426a4b2bc17e,     //  -0.32531,  -0.94561
    0xbfd472b8a5571054,    0xbfee529f04729ffc,     //   -0.3195,  -0.94759
    0xbfd4135c94176602,    0xbfee6288ec48e112,     //  -0.31368,  -0.94953
    0xbfd3b3cefa0414b7,    0xbfee7227db6a9744,     //  -0.30785,  -0.95144
    0xbfd35410c2e18152,    0xbfee817bab4cd10d,     //  -0.30201,  -0.95331
    0xbfd2f422daec0386,    0xbfee9084361df7f3,     //  -0.29615,  -0.95514
    0xbfd294062ed59f05,    0xbfee9f4156c62dda,     //  -0.29028,  -0.95694
    0xbfd233bbabc3bb72,    0xbfeeadb2e8e7a88e,     //  -0.28441,   -0.9587
    0xbfd1d3443f4cdb3d,    0xbfeebbd8c8df0b74,     //  -0.27852,  -0.96043
    0xbfd172a0d7765177,    0xbfeec9b2d3c3bf84,     //  -0.27262,  -0.96212
    0xbfd111d262b1f677,    0xbfeed740e7684963,     //  -0.26671,  -0.96378
    0xbfd0b0d9cfdbdb90,    0xbfeee482e25a9dbc,     //  -0.26079,  -0.96539
    0xbfd04fb80e37fdae,    0xbfeef178a3e473c2,     //  -0.25487,  -0.96698
    0xbfcfdcdc1adfedf8,    0xbfeefe220c0b95ec,     //  -0.24893,  -0.96852
    0xbfcf19f97b215f1a,    0xbfef0a7efb9230d7,     //  -0.24298,  -0.97003
    0xbfce56ca1e101a1b,    0xbfef168f53f7205d,     //  -0.23702,   -0.9715
    0xbfcd934fe5454311,    0xbfef2252f7763ada,     //  -0.23106,  -0.97294
    0xbfcccf8cb312b286,    0xbfef2dc9c9089a9d,     //  -0.22508,  -0.97434
    0xbfcc0b826a7e4f63,    0xbfef38f3ac64e589,     //   -0.2191,   -0.9757
    0xbfcb4732ef3d6722,    0xbfef43d085ff92dd,     //  -0.21311,  -0.97703
    0xbfca82a025b00451,    0xbfef4e603b0b2f2d,     //  -0.20711,  -0.97832
    0xbfc9bdcbf2dc4366,    0xbfef58a2b1789e84,     //   -0.2011,  -0.97957
    0xbfc8f8b83c69a60a,    0xbfef6297cff75cb0,     //  -0.19509,  -0.98079
    0xbfc83366e89c64c5,    0xbfef6c3f7df5bbb7,     //  -0.18907,  -0.98196
    0xbfc76dd9de50bf31,    0xbfef7599a3a12077,     //  -0.18304,  -0.98311
    0xbfc6a81304f64ab2,    0xbfef7ea629e63d6e,     //    -0.177,  -0.98421
    0xbfc5e214448b3fc6,    0xbfef8764fa714ba9,     //  -0.17096,  -0.98528
    0xbfc51bdf8597c5f2,    0xbfef8fd5ffae41db,     //  -0.16491,  -0.98631
    0xbfc45576b1293e5a,    0xbfef97f924c9099b,     //  -0.15886,   -0.9873
    0xbfc38edbb0cd8d14,    0xbfef9fce55adb2c8,     //   -0.1528,  -0.98826
    0xbfc2c8106e8e613a,    0xbfefa7557f08a517,     //  -0.14673,  -0.98918
    0xbfc20116d4ec7bce,    0xbfefae8e8e46cfbb,     //  -0.14066,  -0.99006
    0xbfc139f0cedaf576,    0xbfefb5797195d741,     //  -0.13458,   -0.9909
    0xbfc072a047ba831d,    0xbfefbc1617e44186,     //   -0.1285,  -0.99171
    0xbfbf564e56a9730e,    0xbfefc26470e19fd3,     //  -0.12241,  -0.99248
    0xbfbdc70ecbae9fc8,    0xbfefc8646cfeb721,     //  -0.11632,  -0.99321
    0xbfbc3785c79ec2d5,    0xbfefce15fd6da67b,     //  -0.11022,  -0.99391
    0xbfbaa7b724495c04,    0xbfefd37914220b84,     //  -0.10412,  -0.99456
    0xbfb917a6bc29b42c,    0xbfefd88da3d12526,     // -0.098017,  -0.99518
    0xbfb787586a5d5b21,    0xbfefdd539ff1f456,     // -0.091909,  -0.99577
    0xbfb5f6d00a9aa419,    0xbfefe1cafcbd5b09,     // -0.085797,  -0.99631
    0xbfb4661179272096,    0xbfefe5f3af2e3940,     // -0.079682,  -0.99682
    0xbfb2d52092ce19f6,    0xbfefe9cdad01883a,     // -0.073565,  -0.99729
    0xbfb1440134d709b2,    0xbfefed58ecb673c4,     // -0.067444,  -0.99772
    0xbfaf656e79f820e0,    0xbfeff095658e71ad,     // -0.061321,  -0.99812
    0xbfac428d12c0d7e3,    0xbfeff3830f8d575c,     // -0.055195,  -0.99848
    0xbfa91f65f10dd814,    0xbfeff621e3796d7e,     // -0.049068,   -0.9988
    0xbfa5fc00d290cd43,    0xbfeff871dadb81df,     // -0.042938,  -0.99908
    0xbfa2d865759455cd,    0xbfeffa72effef75d,     // -0.036807,  -0.99932
    0xbf9f693731d1cf01,    0xbfeffc251df1d3f8,     // -0.030675,  -0.99953
    0xbf992155f7a3667e,    0xbfeffd886084cd0d,     // -0.024541,   -0.9997
    0xbf92d936bbe30efd,    0xbfeffe9cb44b51a1,     // -0.018407,  -0.99983
    0xbf8921d1fcdec784,    0xbfefff62169b92db,     // -0.012272,  -0.99992
    0xbf7921f0fe670071,    0xbfefffd8858e8a92,     //-0.0061359,  -0.99998
    0x0000000000000000,    0xbff0000000000000,     //         0,        -1
    0x3f7921f0fe670071,    0xbfefffd8858e8a92,     // 0.0061359,  -0.99998
    0x3f8921d1fcdec784,    0xbfefff62169b92db,     //  0.012272,  -0.99992
    0x3f92d936bbe30efd,    0xbfeffe9cb44b51a1,     //  0.018407,  -0.99983
    0x3f992155f7a3667e,    0xbfeffd886084cd0d,     //  0.024541,   -0.9997
    0x3f9f693731d1cf01,    0xbfeffc251df1d3f8,     //  0.030675,  -0.99953
    0x3fa2d865759455cd,    0xbfeffa72effef75d,     //  0.036807,  -0.99932
    0x3fa5fc00d290cd43,    0xbfeff871dadb81df,     //  0.042938,  -0.99908
    0x3fa91f65f10dd814,    0xbfeff621e3796d7e,     //  0.049068,   -0.9988
    0x3fac428d12c0d7e3,    0xbfeff3830f8d575c,     //  0.055195,  -0.99848
    0x3faf656e79f820e0,    0xbfeff095658e71ad,     //  0.061321,  -0.99812
    0x3fb1440134d709b2,    0xbfefed58ecb673c4,     //  0.067444,  -0.99772
    0x3fb2d52092ce19f6,    0xbfefe9cdad01883a,     //  0.073565,  -0.99729
    0x3fb4661179272096,    0xbfefe5f3af2e3940,     //  0.079682,  -0.99682
    0x3fb5f6d00a9aa419,    0xbfefe1cafcbd5b09,     //  0.085797,  -0.99631
    0x3fb787586a5d5b21,    0xbfefdd539ff1f456,     //  0.091909,  -0.99577
    0x3fb917a6bc29b42c,    0xbfefd88da3d12526,     //  0.098017,  -0.99518
    0x3fbaa7b724495c04,    0xbfefd37914220b84,     //   0.10412,  -0.99456
    0x3fbc3785c79ec2d5,    0xbfefce15fd6da67b,     //   0.11022,  -0.99391
    0x3fbdc70ecbae9fc8,    0xbfefc8646cfeb721,     //   0.11632,  -0.99321
    0x3fbf564e56a9730e,    0xbfefc26470e19fd3,     //   0.12241,  -0.99248
    0x3fc072a047ba831d,    0xbfefbc1617e44186,     //    0.1285,  -0.99171
    0x3fc139f0cedaf576,    0xbfefb5797195d741,     //   0.13458,   -0.9909
    0x3fc20116d4ec7bce,    0xbfefae8e8e46cfbb,     //   0.14066,  -0.99006
    0x3fc2c8106e8e613a,    0xbfefa7557f08a517,     //   0.14673,  -0.98918
    0x3fc38edbb0cd8d14,    0xbfef9fce55adb2c8,     //    0.1528,  -0.98826
    0x3fc45576b1293e5a,    0xbfef97f924c9099b,     //   0.15886,   -0.9873
    0x3fc51bdf8597c5f2,    0xbfef8fd5ffae41db,     //   0.16491,  -0.98631
    0x3fc5e214448b3fc6,    0xbfef8764fa714ba9,     //   0.17096,  -0.98528
    0x3fc6a81304f64ab2,    0xbfef7ea629e63d6e,     //     0.177,  -0.98421
    0x3fc76dd9de50bf31,    0xbfef7599a3a12077,     //   0.18304,  -0.98311
    0x3fc83366e89c64c5,    0xbfef6c3f7df5bbb7,     //   0.18907,  -0.98196
    0x3fc8f8b83c69a60a,    0xbfef6297cff75cb0,     //   0.19509,  -0.98079
    0x3fc9bdcbf2dc4366,    0xbfef58a2b1789e84,     //    0.2011,  -0.97957
    0x3fca82a025b00451,    0xbfef4e603b0b2f2d,     //   0.20711,  -0.97832
    0x3fcb4732ef3d6722,    0xbfef43d085ff92dd,     //   0.21311,  -0.97703
    0x3fcc0b826a7e4f63,    0xbfef38f3ac64e589,     //    0.2191,   -0.9757
    0x3fcccf8cb312b286,    0xbfef2dc9c9089a9d,     //   0.22508,  -0.97434
    0x3fcd934fe5454311,    0xbfef2252f7763ada,     //   0.23106,  -0.97294
    0x3fce56ca1e101a1b,    0xbfef168f53f7205d,     //   0.23702,   -0.9715
    0x3fcf19f97b215f1a,    0xbfef0a7efb9230d7,     //   0.24298,  -0.97003
    0x3fcfdcdc1adfedf8,    0xbfeefe220c0b95ec,     //   0.24893,  -0.96852
    0x3fd04fb80e37fdae,    0xbfeef178a3e473c2,     //   0.25487,  -0.96698
    0x3fd0b0d9cfdbdb90,    0xbfeee482e25a9dbc,     //   0.26079,  -0.96539
    0x3fd111d262b1f677,    0xbfeed740e7684963,     //   0.26671,  -0.96378
    0x3fd172a0d7765177,    0xbfeec9b2d3c3bf84,     //   0.27262,  -0.96212
    0x3fd1d3443f4cdb3d,    0xbfeebbd8c8df0b74,     //   0.27852,  -0.96043
    0x3fd233bbabc3bb72,    0xbfeeadb2e8e7a88e,     //   0.28441,   -0.9587
    0x3fd294062ed59f05,    0xbfee9f4156c62dda,     //   0.29028,  -0.95694
    0x3fd2f422daec0386,    0xbfee9084361df7f3,     //   0.29615,  -0.95514
    0x3fd35410c2e18152,    0xbfee817bab4cd10d,     //   0.30201,  -0.95331
    0x3fd3b3cefa0414b7,    0xbfee7227db6a9744,     //   0.30785,  -0.95144
    0x3fd4135c94176602,    0xbfee6288ec48e112,     //   0.31368,  -0.94953
    0x3fd472b8a5571054,    0xbfee529f04729ffc,     //    0.3195,  -0.94759
    0x3fd4d1e24278e76a,    0xbfee426a4b2bc17e,     //   0.32531,  -0.94561
    0x3fd530d880af3c24,    0xbfee31eae870ce25,     //   0.33111,  -0.94359
    0x3fd58f9a75ab1fdd,    0xbfee212104f686e5,     //   0.33689,  -0.94154
    0x3fd5ee27379ea693,    0xbfee100cca2980ac,     //   0.34266,  -0.93946
    0x3fd64c7ddd3f27c6,    0xbfedfeae622dbe2b,     //   0.34842,  -0.93734
    0x3fd6aa9d7dc77e16,    0xbfeded05f7de47da,     //   0.35416,  -0.93518
    0x3fd7088530fa459e,    0xbfeddb13b6ccc23d,     //    0.3599,  -0.93299
    0x3fd766340f2418f6,    0xbfedc8d7cb410260,     //   0.36561,  -0.93077
    0x3fd7c3a9311dcce7,    0xbfedb6526238a09b,     //   0.37132,  -0.92851
    0x3fd820e3b04eaac4,    0xbfeda383a9668988,     //   0.37701,  -0.92621
    0x3fd87de2a6aea963,    0xbfed906bcf328d46,     //   0.38268,  -0.92388
    0x3fd8daa52ec8a4af,    0xbfed7d0b02b8ecf9,     //   0.38835,  -0.92151
    0x3fd9372a63bc93d7,    0xbfed696173c9e68b,     //   0.39399,  -0.91911
    0x3fd993716141bdfe,    0xbfed556f52e93eb1,     //   0.39962,  -0.91668
    0x3fd9ef7943a8ed8a,    0xbfed4134d14dc93a,     //   0.40524,  -0.91421
    0x3fda4b4127dea1e4,    0xbfed2cb220e0ef9f,     //   0.41084,  -0.91171
    0x3fdaa6c82b6d3fc9,    0xbfed17e7743e35dc,     //   0.41643,  -0.90917
    0x3fdb020d6c7f4009,    0xbfed02d4feb2bd92,     //     0.422,   -0.9066
    0x3fdb5d1009e15cc0,    0xbfeced7af43cc773,     //   0.42756,  -0.90399
    0x3fdbb7cf2304bd01,    0xbfecd7d9898b32f6,     //   0.43309,  -0.90135
    0x3fdc1249d8011ee7,    0xbfecc1f0f3fcfc5c,     //   0.43862,  -0.89867
    0x3fdc6c7f4997000a,    0xbfecabc169a0b901,     //   0.44412,  -0.89597
    0x3fdcc66e9931c45d,    0xbfec954b213411f5,     //   0.44961,  -0.89322
    0x3fdd2016e8e9db5b,    0xbfec7e8e52233cf3,     //   0.45508,  -0.89045
    0x3fdd79775b86e389,    0xbfec678b3488739b,     //   0.46054,  -0.88764
    0x3fddd28f1481cc58,    0xbfec5042012b6907,     //   0.46598,   -0.8848
    0x3fde2b5d3806f63b,    0xbfec38b2f180bdb1,     //    0.4714,  -0.88192
    0x3fde83e0eaf85113,    0xbfec20de3fa971b0,     //    0.4768,  -0.87901
    0x3fdedc1952ef78d5,    0xbfec08c426725549,     //   0.48218,  -0.87607
    0x3fdf3405963fd068,    0xbfebf064e15377dd,     //   0.48755,  -0.87309
    0x3fdf8ba4dbf89aba,    0xbfebd7c0ac6f952a,     //    0.4929,  -0.87009
    0x3fdfe2f64be7120f,    0xbfebbed7c49380ea,     //   0.49823,  -0.86705
    0x3fe01cfc874c3eb7,    0xbfeba5aa673590d2,     //   0.50354,  -0.86397
    0x3fe0485626ae221a,    0xbfeb8c38d27504e9,     //   0.50883,  -0.86087
    0x3fe073879922ffed,    0xbfeb728345196e3e,     //    0.5141,  -0.85773
    0x3fe09e907417c5e1,    0xbfeb5889fe921405,     //   0.51936,  -0.85456
    0x3fe0c9704d5d898f,    0xbfeb3e4d3ef55712,     //   0.52459,  -0.85136
    0x3fe0f426bb2a8e7d,    0xbfeb23cd470013b4,     //    0.5298,  -0.84812
    0x3fe11eb3541b4b22,    0xbfeb090a58150200,     //     0.535,  -0.84485
    0x3fe14915af336ceb,    0xbfeaee04b43c1474,     //   0.54017,  -0.84155
    0x3fe1734d63dedb49,    0xbfead2bc9e21d511,     //   0.54532,  -0.83822
    0x3fe19d5a09f2b9b8,    0xbfeab7325916c0d4,     //   0.55046,  -0.83486
    0x3fe1c73b39ae68c8,    0xbfea9b66290ea1a3,     //   0.55557,  -0.83147
    0x3fe1f0f08bbc861b,    0xbfea7f58529fe69d,     //   0.56066,  -0.82805
    0x3fe21a799933eb58,    0xbfea63091b02fae2,     //   0.56573,  -0.82459
    0x3fe243d5fb98ac1f,    0xbfea4678c8119ac8,     //   0.57078,   -0.8211
    0x3fe26d054cdd12df,    0xbfea29a7a0462782,     //   0.57581,  -0.81758
    0x3fe2960727629ca8,    0xbfea0c95eabaf937,     //   0.58081,  -0.81404
    0x3fe2bedb25faf3ea,    0xbfe9ef43ef29af94,     //    0.5858,  -0.81046
    0x3fe2e780e3e8ea16,    0xbfe9d1b1f5ea80d6,     //   0.59076,  -0.80685
    0x3fe30ff7fce17035,    0xbfe9b3e047f38741,     //    0.5957,  -0.80321
    0x3fe338400d0c8e57,    0xbfe995cf2ed80d22,     //   0.60062,  -0.79954
    0x3fe36058b10659f3,    0xbfe9777ef4c7d742,     //   0.60551,  -0.79584
    0x3fe3884185dfeb22,    0xbfe958efe48e6dd7,     //   0.61038,  -0.79211
    0x3fe3affa292050b9,    0xbfe93a22499263fc,     //   0.61523,  -0.78835
    0x3fe3d78238c58343,    0xbfe91b166fd49da2,     //   0.62006,  -0.78456
    0x3fe3fed9534556d4,    0xbfe8fbcca3ef940d,     //   0.62486,  -0.78074
    0x3fe425ff178e6bb1,    0xbfe8dc45331698cc,     //   0.62964,  -0.77689
    0x3fe44cf325091dd6,    0xbfe8bc806b151741,     //   0.63439,  -0.77301
    0x3fe473b51b987347,    0xbfe89c7e9a4dd4ab,     //   0.63912,   -0.7691
    0x3fe49a449b9b0938,    0xbfe87c400fba2ebf,     //   0.64383,  -0.76517
    0x3fe4c0a145ec0004,    0xbfe85bc51ae958cc,     //   0.64851,   -0.7612
    0x3fe4e6cabbe3e5e9,    0xbfe83b0e0bff976e,     //   0.65317,  -0.75721
    0x3fe50cc09f59a09b,    0xbfe81a1b33b57acc,     //   0.65781,  -0.75319
    0x3fe5328292a35596,    0xbfe7f8ece3571771,     //   0.66242,  -0.74914
    0x3fe5581038975137,    0xbfe7d7836cc33db2,     //     0.667,  -0.74506
    0x3fe57d69348cec9f,    0xbfe7b5df226aafb0,     //   0.67156,  -0.74095
    0x3fe5a28d2a5d7250,    0xbfe79400574f55e4,     //   0.67609,  -0.73682
    0x3fe5c77bbe65018c,    0xbfe771e75f037261,     //    0.6806,  -0.73265
    0x3fe5ec3495837074,    0xbfe74f948da8d28d,     //   0.68508,  -0.72846
    0x3fe610b7551d2cde,    0xbfe72d0837efff97,     //   0.68954,  -0.72425
    0x3fe63503a31c1be8,    0xbfe70a42b3176d7a,     //   0.69397,     -0.72
    0x3fe6591925f0783e,    0xbfe6e74454eaa8ae,     //   0.69838,  -0.71573
    0x3fe67cf78491af10,    0xbfe6c40d73c18275,     //   0.70275,  -0.71143
    0x3fe6a09e667f3bcc,    0xbfe6a09e667f3bcc,     //   0.70711,  -0.70711
    0x3fe6c40d73c18275,    0xbfe67cf78491af10,     //   0.71143,  -0.70275
    0x3fe6e74454eaa8ae,    0xbfe6591925f0783e,     //   0.71573,  -0.69838
    0x3fe70a42b3176d7a,    0xbfe63503a31c1be8,     //      0.72,  -0.69397
    0x3fe72d0837efff97,    0xbfe610b7551d2cde,     //   0.72425,  -0.68954
    0x3fe74f948da8d28d,    0xbfe5ec3495837074,     //   0.72846,  -0.68508
    0x3fe771e75f037261,    0xbfe5c77bbe65018c,     //   0.73265,   -0.6806
    0x3fe79400574f55e4,    0xbfe5a28d2a5d7250,     //   0.73682,  -0.67609
    0x3fe7b5df226aafb0,    0xbfe57d69348cec9f,     //   0.74095,  -0.67156
    0x3fe7d7836cc33db2,    0xbfe5581038975137,     //   0.74506,    -0.667
    0x3fe7f8ece3571771,    0xbfe5328292a35596,     //   0.74914,  -0.66242
    0x3fe81a1b33b57acc,    0xbfe50cc09f59a09b,     //   0.75319,  -0.65781
    0x3fe83b0e0bff976e,    0xbfe4e6cabbe3e5e9,     //   0.75721,  -0.65317
    0x3fe85bc51ae958cc,    0xbfe4c0a145ec0004,     //    0.7612,  -0.64851
    0x3fe87c400fba2ebf,    0xbfe49a449b9b0938,     //   0.76517,  -0.64383
    0x3fe89c7e9a4dd4ab,    0xbfe473b51b987347,     //    0.7691,  -0.63912
    0x3fe8bc806b151741,    0xbfe44cf325091dd6,     //   0.77301,  -0.63439
    0x3fe8dc45331698cc,    0xbfe425ff178e6bb1,     //   0.77689,  -0.62964
    0x3fe8fbcca3ef940d,    0xbfe3fed9534556d4,     //   0.78074,  -0.62486
    0x3fe91b166fd49da2,    0xbfe3d78238c58343,     //   0.78456,  -0.62006
    0x3fe93a22499263fc,    0xbfe3affa292050b9,     //   0.78835,  -0.61523
    0x3fe958efe48e6dd7,    0xbfe3884185dfeb22,     //   0.79211,  -0.61038
    0x3fe9777ef4c7d742,    0xbfe36058b10659f3,     //   0.79584,  -0.60551
    0x3fe995cf2ed80d22,    0xbfe338400d0c8e57,     //   0.79954,  -0.60062
    0x3fe9b3e047f38741,    0xbfe30ff7fce17035,     //   0.80321,   -0.5957
    0x3fe9d1b1f5ea80d6,    0xbfe2e780e3e8ea16,     //   0.80685,  -0.59076
    0x3fe9ef43ef29af94,    0xbfe2bedb25faf3ea,     //   0.81046,   -0.5858
    0x3fea0c95eabaf937,    0xbfe2960727629ca8,     //   0.81404,  -0.58081
    0x3fea29a7a0462782,    0xbfe26d054cdd12df,     //   0.81758,  -0.57581
    0x3fea4678c8119ac8,    0xbfe243d5fb98ac1f,     //    0.8211,  -0.57078
    0x3fea63091b02fae2,    0xbfe21a799933eb58,     //   0.82459,  -0.56573
    0x3fea7f58529fe69d,    0xbfe1f0f08bbc861b,     //   0.82805,  -0.56066
    0x3fea9b66290ea1a3,    0xbfe1c73b39ae68c8,     //   0.83147,  -0.55557
    0x3feab7325916c0d4,    0xbfe19d5a09f2b9b8,     //   0.83486,  -0.55046
    0x3fead2bc9e21d511,    0xbfe1734d63dedb49,     //   0.83822,  -0.54532
    0x3feaee04b43c1474,    0xbfe14915af336ceb,     //   0.84155,  -0.54017
    0x3feb090a58150200,    0xbfe11eb3541b4b22,     //   0.84485,    -0.535
    0x3feb23cd470013b4,    0xbfe0f426bb2a8e7d,     //   0.84812,   -0.5298
    0x3feb3e4d3ef55712,    0xbfe0c9704d5d898f,     //   0.85136,  -0.52459
    0x3feb5889fe921405,    0xbfe09e907417c5e1,     //   0.85456,  -0.51936
    0x3feb728345196e3e,    0xbfe073879922ffed,     //   0.85773,   -0.5141
    0x3feb8c38d27504e9,    0xbfe0485626ae221a,     //   0.86087,  -0.50883
    0x3feba5aa673590d2,    0xbfe01cfc874c3eb7,     //   0.86397,  -0.50354
    0x3febbed7c49380ea,    0xbfdfe2f64be7120f,     //   0.86705,  -0.49823
    0x3febd7c0ac6f952a,    0xbfdf8ba4dbf89aba,     //   0.87009,   -0.4929
    0x3febf064e15377dd,    0xbfdf3405963fd068,     //   0.87309,  -0.48755
    0x3fec08c426725549,    0xbfdedc1952ef78d5,     //   0.87607,  -0.48218
    0x3fec20de3fa971b0,    0xbfde83e0eaf85113,     //   0.87901,   -0.4768
    0x3fec38b2f180bdb1,    0xbfde2b5d3806f63b,     //   0.88192,   -0.4714
    0x3fec5042012b6907,    0xbfddd28f1481cc58,     //    0.8848,  -0.46598
    0x3fec678b3488739b,    0xbfdd79775b86e389,     //   0.88764,  -0.46054
    0x3fec7e8e52233cf3,    0xbfdd2016e8e9db5b,     //   0.89045,  -0.45508
    0x3fec954b213411f5,    0xbfdcc66e9931c45d,     //   0.89322,  -0.44961
    0x3fecabc169a0b901,    0xbfdc6c7f4997000a,     //   0.89597,  -0.44412
    0x3fecc1f0f3fcfc5c,    0xbfdc1249d8011ee7,     //   0.89867,  -0.43862
    0x3fecd7d9898b32f6,    0xbfdbb7cf2304bd01,     //   0.90135,  -0.43309
    0x3feced7af43cc773,    0xbfdb5d1009e15cc0,     //   0.90399,  -0.42756
    0x3fed02d4feb2bd92,    0xbfdb020d6c7f4009,     //    0.9066,    -0.422
    0x3fed17e7743e35dc,    0xbfdaa6c82b6d3fc9,     //   0.90917,  -0.41643
    0x3fed2cb220e0ef9f,    0xbfda4b4127dea1e4,     //   0.91171,  -0.41084
    0x3fed4134d14dc93a,    0xbfd9ef7943a8ed8a,     //   0.91421,  -0.40524
    0x3fed556f52e93eb1,    0xbfd993716141bdfe,     //   0.91668,  -0.39962
    0x3fed696173c9e68b,    0xbfd9372a63bc93d7,     //   0.91911,  -0.39399
    0x3fed7d0b02b8ecf9,    0xbfd8daa52ec8a4af,     //   0.92151,  -0.38835
    0x3fed906bcf328d46,    0xbfd87de2a6aea963,     //   0.92388,  -0.38268
    0x3feda383a9668988,    0xbfd820e3b04eaac4,     //   0.92621,  -0.37701
    0x3fedb6526238a09b,    0xbfd7c3a9311dcce7,     //   0.92851,  -0.37132
    0x3fedc8d7cb410260,    0xbfd766340f2418f6,     //   0.93077,  -0.36561
    0x3feddb13b6ccc23d,    0xbfd7088530fa459e,     //   0.93299,   -0.3599
    0x3feded05f7de47da,    0xbfd6aa9d7dc77e16,     //   0.93518,  -0.35416
    0x3fedfeae622dbe2b,    0xbfd64c7ddd3f27c6,     //   0.93734,  -0.34842
    0x3fee100cca2980ac,    0xbfd5ee27379ea693,     //   0.93946,  -0.34266
    0x3fee212104f686e5,    0xbfd58f9a75ab1fdd,     //   0.94154,  -0.33689
    0x3fee31eae870ce25,    0xbfd530d880af3c24,     //   0.94359,  -0.33111
    0x3fee426a4b2bc17e,    0xbfd4d1e24278e76a,     //   0.94561,  -0.32531
    0x3fee529f04729ffc,    0xbfd472b8a5571054,     //   0.94759,   -0.3195
    0x3fee6288ec48e112,    0xbfd4135c94176602,     //   0.94953,  -0.31368
    0x3fee7227db6a9744,    0xbfd3b3cefa0414b7,     //   0.95144,  -0.30785
    0x3fee817bab4cd10d,    0xbfd35410c2e18152,     //   0.95331,  -0.30201
    0x3fee9084361df7f3,    0xbfd2f422daec0386,     //   0.95514,  -0.29615
    0x3fee9f4156c62dda,    0xbfd294062ed59f05,     //   0.95694,  -0.29028
    0x3feeadb2e8e7a88e,    0xbfd233bbabc3bb72,     //    0.9587,  -0.28441
    0x3feebbd8c8df0b74,    0xbfd1d3443f4cdb3d,     //   0.96043,  -0.27852
    0x3feec9b2d3c3bf84,    0xbfd172a0d7765177,     //   0.96212,  -0.27262
    0x3feed740e7684963,    0xbfd111d262b1f677,     //   0.96378,  -0.26671
    0x3feee482e25a9dbc,    0xbfd0b0d9cfdbdb90,     //   0.96539,  -0.26079
    0x3feef178a3e473c2,    0xbfd04fb80e37fdae,     //   0.96698,  -0.25487
    0x3feefe220c0b95ec,    0xbfcfdcdc1adfedf8,     //   0.96852,  -0.24893
    0x3fef0a7efb9230d7,    0xbfcf19f97b215f1a,     //   0.97003,  -0.24298
    0x3fef168f53f7205d,    0xbfce56ca1e101a1b,     //    0.9715,  -0.23702
    0x3fef2252f7763ada,    0xbfcd934fe5454311,     //   0.97294,  -0.23106
    0x3fef2dc9c9089a9d,    0xbfcccf8cb312b286,     //   0.97434,  -0.22508
    0x3fef38f3ac64e589,    0xbfcc0b826a7e4f63,     //    0.9757,   -0.2191
    0x3fef43d085ff92dd,    0xbfcb4732ef3d6722,     //   0.97703,  -0.21311
    0x3fef4e603b0b2f2d,    0xbfca82a025b00451,     //   0.97832,  -0.20711
    0x3fef58a2b1789e84,    0xbfc9bdcbf2dc4366,     //   0.97957,   -0.2011
    0x3fef6297cff75cb0,    0xbfc8f8b83c69a60a,     //   0.98079,  -0.19509
    0x3fef6c3f7df5bbb7,    0xbfc83366e89c64c5,     //   0.98196,  -0.18907
    0x3fef7599a3a12077,    0xbfc76dd9de50bf31,     //   0.98311,  -0.18304
    0x3fef7ea629e63d6e,    0xbfc6a81304f64ab2,     //   0.98421,    -0.177
    0x3fef8764fa714ba9,    0xbfc5e214448b3fc6,     //   0.98528,  -0.17096
    0x3fef8fd5ffae41db,    0xbfc51bdf8597c5f2,     //   0.98631,  -0.16491
    0x3fef97f924c9099b,    0xbfc45576b1293e5a,     //    0.9873,  -0.15886
    0x3fef9fce55adb2c8,    0xbfc38edbb0cd8d14,     //   0.98826,   -0.1528
    0x3fefa7557f08a517,    0xbfc2c8106e8e613a,     //   0.98918,  -0.14673
    0x3fefae8e8e46cfbb,    0xbfc20116d4ec7bce,     //   0.99006,  -0.14066
    0x3fefb5797195d741,    0xbfc139f0cedaf576,     //    0.9909,  -0.13458
    0x3fefbc1617e44186,    0xbfc072a047ba831d,     //   0.99171,   -0.1285
    0x3fefc26470e19fd3,    0xbfbf564e56a9730e,     //   0.99248,  -0.12241
    0x3fefc8646cfeb721,    0xbfbdc70ecbae9fc8,     //   0.99321,  -0.11632
    0x3fefce15fd6da67b,    0xbfbc3785c79ec2d5,     //   0.99391,  -0.11022
    0x3fefd37914220b84,    0xbfbaa7b724495c04,     //   0.99456,  -0.10412
    0x3fefd88da3d12526,    0xbfb917a6bc29b42c,     //   0.99518, -0.098017
    0x3fefdd539ff1f456,    0xbfb787586a5d5b21,     //   0.99577, -0.091909
    0x3fefe1cafcbd5b09,    0xbfb5f6d00a9aa419,     //   0.99631, -0.085797
    0x3fefe5f3af2e3940,    0xbfb4661179272096,     //   0.99682, -0.079682
    0x3fefe9cdad01883a,    0xbfb2d52092ce19f6,     //   0.99729, -0.073565
    0x3fefed58ecb673c4,    0xbfb1440134d709b2,     //   0.99772, -0.067444
    0x3feff095658e71ad,    0xbfaf656e79f820e0,     //   0.99812, -0.061321
    0x3feff3830f8d575c,    0xbfac428d12c0d7e3,     //   0.99848, -0.055195
    0x3feff621e3796d7e,    0xbfa91f65f10dd814,     //    0.9988, -0.049068
    0x3feff871dadb81df,    0xbfa5fc00d290cd43,     //   0.99908, -0.042938
    0x3feffa72effef75d,    0xbfa2d865759455cd,     //   0.99932, -0.036807
    0x3feffc251df1d3f8,    0xbf9f693731d1cf01,     //   0.99953, -0.030675
    0x3feffd886084cd0d,    0xbf992155f7a3667e,     //    0.9997, -0.024541
    0x3feffe9cb44b51a1,    0xbf92d936bbe30efd,     //   0.99983, -0.018407
    0x3fefff62169b92db,    0xbf8921d1fcdec784,     //   0.99992, -0.012272
    0x3fefffd8858e8a92,    0xbf7921f0fe670071,     //   0.99998,-0.0061359
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_2048)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 2048, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_2048[4096] = {
    0x3ff0000000000000,     0x0000000000000000,     //         1,         0
    0x3feffff621621d02,     0x3f6921f8becca4ba,     //         1,  0.003068
    0x3fefffd8858e8a92,     0x3f7921f0fe670071,     //   0.99998, 0.0061359
    0x3fefffa72c978c4f,     0x3f82d96b0e509703,     //   0.99996, 0.0092038
    0x3fefff62169b92db,     0x3f8921d1fcdec784,     //   0.99992,  0.012272
    0x3fefff0943c53bd1,     0x3f8f6a296ab997ca,     //   0.99988,  0.015339
    0x3feffe9cb44b51a1,     0x3f92d936bbe30efd,     //   0.99983,  0.018407
    0x3feffe1c6870cb77,     0x3f95fd4d21fab226,     //   0.99977,  0.021474
    0x3feffd886084cd0d,     0x3f992155f7a3667e,     //    0.9997,  0.024541
    0x3feffce09ce2a679,     0x3f9c454f4ce53b1c,     //   0.99962,  0.027608
    0x3feffc251df1d3f8,     0x3f9f693731d1cf01,     //   0.99953,  0.030675
    0x3feffb55e425fdae,     0x3fa14685db42c17e,     //   0.99943,  0.033741
    0x3feffa72effef75d,     0x3fa2d865759455cd,     //   0.99932,  0.036807
    0x3feff97c4208c014,     0x3fa46a396ff86179,     //    0.9992,  0.039873
    0x3feff871dadb81df,     0x3fa5fc00d290cd43,     //   0.99908,  0.042938
    0x3feff753bb1b9164,     0x3fa78dbaa5874685,     //   0.99894,  0.046003
    0x3feff621e3796d7e,     0x3fa91f65f10dd814,     //    0.9988,  0.049068
    0x3feff4dc54b1bed3,     0x3faab101bd5f8317,     //   0.99864,  0.052132
    0x3feff3830f8d575c,     0x3fac428d12c0d7e3,     //   0.99848,  0.055195
    0x3feff21614e131ed,     0x3fadd406f9808ec8,     //    0.9983,  0.058258
    0x3feff095658e71ad,     0x3faf656e79f820e0,     //   0.99812,  0.061321
    0x3fefef0102826191,     0x3fb07b614e463064,     //   0.99793,  0.064383
    0x3fefed58ecb673c4,     0x3fb1440134d709b2,     //   0.99772,  0.067444
    0x3fefeb9d2530410f,     0x3fb20c9674ed444c,     //   0.99751,  0.070505
    0x3fefe9cdad01883a,     0x3fb2d52092ce19f6,     //   0.99729,  0.073565
    0x3fefe7ea85482d60,     0x3fb39d9f12c5a299,     //   0.99706,  0.076624
    0x3fefe5f3af2e3940,     0x3fb4661179272096,     //   0.99682,  0.079682
    0x3fefe3e92be9d886,     0x3fb52e774a4d4d0a,     //   0.99657,   0.08274
    0x3fefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     //   0.99631,  0.085797
    0x3fefdf9922f73307,     0x3fb6bf1b3e79b129,     //   0.99604,  0.088854
    0x3fefdd539ff1f456,     0x3fb787586a5d5b21,     //   0.99577,  0.091909
    0x3fefdafa7514538c,     0x3fb84f8712c130a0,     //   0.99548,  0.094963
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //   0.99518,  0.098017
    0x3fefd60d2da75c9e,     0x3fb9dfb6eb24a85c,     //   0.99488,   0.10107
    0x3fefd37914220b84,     0x3fbaa7b724495c04,     //   0.99456,   0.10412
    0x3fefd0d158d86087,     0x3fbb6fa6ec38f64c,     //   0.99424,   0.10717
    0x3fefce15fd6da67b,     0x3fbc3785c79ec2d5,     //   0.99391,   0.11022
    0x3fefcb4703914354,     0x3fbcff533b307dc1,     //   0.99356,   0.11327
    0x3fefc8646cfeb721,     0x3fbdc70ecbae9fc8,     //   0.99321,   0.11632
    0x3fefc56e3b7d9af6,     0x3fbe8eb7fde4aa3e,     //   0.99285,   0.11937
    0x3fefc26470e19fd3,     0x3fbf564e56a9730e,     //   0.99248,   0.12241
    0x3fefbf470f0a8d88,     0x3fc00ee8ad6fb85b,     //    0.9921,   0.12545
    0x3fefbc1617e44186,     0x3fc072a047ba831d,     //   0.99171,    0.1285
    0x3fefb8d18d66adb7,     0x3fc0d64dbcb26786,     //   0.99131,   0.13154
    0x3fefb5797195d741,     0x3fc139f0cedaf576,     //    0.9909,   0.13458
    0x3fefb20dc681d54d,     0x3fc19d8940be24e7,     //   0.99049,   0.13762
    0x3fefae8e8e46cfbb,     0x3fc20116d4ec7bce,     //   0.99006,   0.14066
    0x3fefaafbcb0cfddc,     0x3fc264994dfd340a,     //   0.98962,    0.1437
    0x3fefa7557f08a517,     0x3fc2c8106e8e613a,     //   0.98918,   0.14673
    0x3fefa39bac7a1791,     0x3fc32b7bf94516a7,     //   0.98872,   0.14976
    0x3fef9fce55adb2c8,     0x3fc38edbb0cd8d14,     //   0.98826,    0.1528
    0x3fef9bed7cfbde29,     0x3fc3f22f57db4893,     //   0.98778,   0.15583
    0x3fef97f924c9099b,     0x3fc45576b1293e5a,     //    0.9873,   0.15886
    0x3fef93f14f85ac08,     0x3fc4b8b17f79fa88,     //   0.98681,   0.16189
    0x3fef8fd5ffae41db,     0x3fc51bdf8597c5f2,     //   0.98631,   0.16491
    0x3fef8ba737cb4b78,     0x3fc57f008654cbde,     //    0.9858,   0.16794
    0x3fef8764fa714ba9,     0x3fc5e214448b3fc6,     //   0.98528,   0.17096
    0x3fef830f4a40c60c,     0x3fc6451a831d830d,     //   0.98475,   0.17398
    0x3fef7ea629e63d6e,     0x3fc6a81304f64ab2,     //   0.98421,     0.177
    0x3fef7a299c1a322a,     0x3fc70afd8d08c4ff,     //   0.98366,   0.18002
    0x3fef7599a3a12077,     0x3fc76dd9de50bf31,     //   0.98311,   0.18304
    0x3fef70f6434b7eb7,     0x3fc7d0a7bbd2cb1b,     //   0.98254,   0.18606
    0x3fef6c3f7df5bbb7,     0x3fc83366e89c64c5,     //   0.98196,   0.18907
    0x3fef677556883cee,     0x3fc8961727c41804,     //   0.98138,   0.19208
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //   0.98079,   0.19509
    0x3fef5da6ed43685d,     0x3fc95b49e9b62af9,     //   0.98018,    0.1981
    0x3fef58a2b1789e84,     0x3fc9bdcbf2dc4366,     //   0.97957,    0.2011
    0x3fef538b1faf2d07,     0x3fca203e1b1831da,     //   0.97895,   0.20411
    0x3fef4e603b0b2f2d,     0x3fca82a025b00451,     //   0.97832,   0.20711
    0x3fef492206bcabb4,     0x3fcae4f1d5f3b9ab,     //   0.97768,   0.21011
    0x3fef43d085ff92dd,     0x3fcb4732ef3d6722,     //   0.97703,   0.21311
    0x3fef3e6bbc1bbc65,     0x3fcba96334f15dad,     //   0.97637,   0.21611
    0x3fef38f3ac64e589,     0x3fcc0b826a7e4f63,     //    0.9757,    0.2191
    0x3fef33685a3aaef0,     0x3fcc6d90535d74dc,     //   0.97503,   0.22209
    0x3fef2dc9c9089a9d,     0x3fcccf8cb312b286,     //   0.97434,   0.22508
    0x3fef2817fc4609ce,     0x3fcd31774d2cbdee,     //   0.97364,   0.22807
    0x3fef2252f7763ada,     0x3fcd934fe5454311,     //   0.97294,   0.23106
    0x3fef1c7abe284708,     0x3fcdf5163f01099a,     //   0.97223,   0.23404
    0x3fef168f53f7205d,     0x3fce56ca1e101a1b,     //    0.9715,   0.23702
    0x3fef1090bc898f5f,     0x3fceb86b462de348,     //   0.97077,      0.24
    0x3fef0a7efb9230d7,     0x3fcf19f97b215f1a,     //   0.97003,   0.24298
    0x3fef045a14cf738c,     0x3fcf7b7480bd3801,     //   0.96928,   0.24596
    0x3feefe220c0b95ec,     0x3fcfdcdc1adfedf8,     //   0.96852,   0.24893
    0x3feef7d6e51ca3c0,     0x3fd01f1806b9fdd2,     //   0.96775,    0.2519
    0x3feef178a3e473c2,     0x3fd04fb80e37fdae,     //   0.96698,   0.25487
    0x3feeeb074c50a544,     0x3fd0804e05eb661e,     //   0.96619,   0.25783
    0x3feee482e25a9dbc,     0x3fd0b0d9cfdbdb90,     //   0.96539,   0.26079
    0x3feeddeb6a078651,     0x3fd0e15b4e1749cd,     //   0.96459,   0.26375
    0x3feed740e7684963,     0x3fd111d262b1f677,     //   0.96378,   0.26671
    0x3feed0835e999009,     0x3fd1423eefc69378,     //   0.96295,   0.26967
    0x3feec9b2d3c3bf84,     0x3fd172a0d7765177,     //   0.96212,   0.27262
    0x3feec2cf4b1af6b2,     0x3fd1a2f7fbe8f243,     //   0.96128,   0.27557
    0x3feebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     //   0.96043,   0.27852
    0x3feeb4cf515b8811,     0x3fd2038583d727bd,     //   0.95957,   0.28146
    0x3feeadb2e8e7a88e,     0x3fd233bbabc3bb72,     //    0.9587,   0.28441
    0x3feea68393e65800,     0x3fd263e6995554ba,     //   0.95783,   0.28735
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //   0.95694,   0.29028
    0x3fee97ec36016b30,     0x3fd2c41a4e954520,     //   0.95605,   0.29322
    0x3fee9084361df7f3,     0x3fd2f422daec0386,     //   0.95514,   0.29615
    0x3fee89095bad6025,     0x3fd3241fb638baaf,     //   0.95423,   0.29908
    0x3fee817bab4cd10d,     0x3fd35410c2e18152,     //   0.95331,   0.30201
    0x3fee79db29a5165a,     0x3fd383f5e353b6aa,     //   0.95238,   0.30493
    0x3fee7227db6a9744,     0x3fd3b3cefa0414b7,     //   0.95144,   0.30785
    0x3fee6a61c55d53a7,     0x3fd3e39be96ec271,     //   0.95049,   0.31077
    0x3fee6288ec48e112,     0x3fd4135c94176602,     //   0.94953,   0.31368
    0x3fee5a9d550467d3,     0x3fd44310dc8936f0,     //   0.94856,   0.31659
    0x3fee529f04729ffc,     0x3fd472b8a5571054,     //   0.94759,    0.3195
    0x3fee4a8dff81ce5e,     0x3fd4a253d11b82f3,     //    0.9466,   0.32241
    0x3fee426a4b2bc17e,     0x3fd4d1e24278e76a,     //   0.94561,   0.32531
    0x3fee3a33ec75ce85,     0x3fd50163dc197047,     //    0.9446,   0.32821
    0x3fee31eae870ce25,     0x3fd530d880af3c24,     //   0.94359,   0.33111
    0x3fee298f4439197a,     0x3fd5604012f467b4,     //   0.94257,     0.334
    0x3fee212104f686e5,     0x3fd58f9a75ab1fdd,     //   0.94154,   0.33689
    0x3fee18a02fdc66d9,     0x3fd5bee78b9db3b6,     //   0.94051,   0.33978
    0x3fee100cca2980ac,     0x3fd5ee27379ea693,     //   0.93946,   0.34266
    0x3fee0766d9280f54,     0x3fd61d595c88c203,     //    0.9384,   0.34554
    0x3fedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     //   0.93734,   0.34842
    0x3fedf5e36a9ba59c,     0x3fd67b949cad63ca,     //   0.93627,   0.35129
    0x3feded05f7de47da,     0x3fd6aa9d7dc77e16,     //   0.93518,   0.35416
    0x3fede4160f6d8d81,     0x3fd6d998638a0cb5,     //   0.93409,   0.35703
    0x3feddb13b6ccc23d,     0x3fd7088530fa459e,     //   0.93299,    0.3599
    0x3fedd1fef38a915a,     0x3fd73763c9261092,     //   0.93188,   0.36276
    0x3fedc8d7cb410260,     0x3fd766340f2418f6,     //   0.93077,   0.36561
    0x3fedbf9e4395759a,     0x3fd794f5e613dfae,     //   0.92964,   0.36847
    0x3fedb6526238a09b,     0x3fd7c3a9311dcce7,     //   0.92851,   0.37132
    0x3fedacf42ce68ab9,     0x3fd7f24dd37341e3,     //   0.92736,   0.37416
    0x3feda383a9668988,     0x3fd820e3b04eaac4,     //   0.92621,   0.37701
    0x3fed9a00dd8b3d46,     0x3fd84f6aaaf3903f,     //   0.92505,   0.37985
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //   0.92388,   0.38268
    0x3fed86c48445a450,     0x3fd8ac4b86d5ed44,     //    0.9227,   0.38552
    0x3fed7d0b02b8ecf9,     0x3fd8daa52ec8a4af,     //   0.92151,   0.38835
    0x3fed733f508c0dff,     0x3fd908ef81ef7bd1,     //   0.92032,   0.39117
    0x3fed696173c9e68b,     0x3fd9372a63bc93d7,     //   0.91911,   0.39399
    0x3fed5f7172888a7f,     0x3fd96555b7ab948f,     //    0.9179,   0.39681
    0x3fed556f52e93eb1,     0x3fd993716141bdfe,     //   0.91668,   0.39962
    0x3fed4b5b1b187524,     0x3fd9c17d440df9f2,     //   0.91545,   0.40243
    0x3fed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //   0.91421,   0.40524
    0x3fed36fc7bcbfbdc,     0x3fda1d6543b50ac0,     //   0.91296,   0.40804
    0x3fed2cb220e0ef9f,     0x3fda4b4127dea1e4,     //   0.91171,   0.41084
    0x3fed2255c6e5a4e1,     0x3fda790cd3dbf31a,     //   0.91044,   0.41364
    0x3fed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     //   0.90917,   0.41643
    0x3fed0d672f59d2b9,     0x3fdad473125cdc08,     //   0.90789,   0.41922
    0x3fed02d4feb2bd92,     0x3fdb020d6c7f4009,     //    0.9066,     0.422
    0x3fecf830e8ce467b,     0x3fdb2f971db31972,     //    0.9053,   0.42478
    0x3feced7af43cc773,     0x3fdb5d1009e15cc0,     //   0.90399,   0.42756
    0x3fece2b32799a060,     0x3fdb8a7814fd5693,     //   0.90267,   0.43033
    0x3fecd7d9898b32f6,     0x3fdbb7cf2304bd01,     //   0.90135,   0.43309
    0x3fecccee20c2de9f,     0x3fdbe51517ffc0d9,     //   0.90002,   0.43586
    0x3fecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     //   0.89867,   0.43862
    0x3fecb6e20a00da99,     0x3fdc3f6d47263129,     //   0.89732,   0.44137
    0x3fecabc169a0b901,     0x3fdc6c7f4997000a,     //   0.89597,   0.44412
    0x3feca08f19b9c449,     0x3fdc997fc3865388,     //    0.8946,   0.44687
    0x3fec954b213411f5,     0x3fdcc66e9931c45d,     //   0.89322,   0.44961
    0x3fec89f587029c13,     0x3fdcf34baee1cd21,     //   0.89184,   0.45235
    0x3fec7e8e52233cf3,     0x3fdd2016e8e9db5b,     //   0.89045,   0.45508
    0x3fec7315899eaad7,     0x3fdd4cd02ba8609c,     //   0.88905,   0.45781
    0x3fec678b3488739b,     0x3fdd79775b86e389,     //   0.88764,   0.46054
    0x3fec5bef59fef85a,     0x3fdda60c5cfa10d8,     //   0.88622,   0.46326
    0x3fec5042012b6907,     0x3fddd28f1481cc58,     //    0.8848,   0.46598
    0x3fec44833141c004,     0x3fddfeff66a941de,     //   0.88336,   0.46869
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //   0.88192,    0.4714
    0x3fec2cd14931e3f1,     0x3fde57a86d3cd824,     //   0.88047,    0.4741
    0x3fec20de3fa971b0,     0x3fde83e0eaf85113,     //   0.87901,    0.4768
    0x3fec14d9dc465e58,     0x3fdeb00695f25620,     //   0.87755,   0.47949
    0x3fec08c426725549,     0x3fdedc1952ef78d5,     //   0.87607,   0.48218
    0x3febfc9d25a1b147,     0x3fdf081906bff7fd,     //   0.87459,   0.48487
    0x3febf064e15377dd,     0x3fdf3405963fd068,     //   0.87309,   0.48755
    0x3febe41b611154c1,     0x3fdf5fdee656cda3,     //    0.8716,   0.49023
    0x3febd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //   0.87009,    0.4929
    0x3febcb54cb0d2327,     0x3fdfb7575c24d2de,     //   0.86857,   0.49557
    0x3febbed7c49380ea,     0x3fdfe2f64be7120f,     //   0.86705,   0.49823
    0x3febb249a0b6c40d,     0x3fe00740c82b82e0,     //   0.86551,   0.50089
    0x3feba5aa673590d2,     0x3fe01cfc874c3eb7,     //   0.86397,   0.50354
    0x3feb98fa1fd9155e,     0x3fe032ae55edbd95,     //   0.86242,   0.50619
    0x3feb8c38d27504e9,     0x3fe0485626ae221a,     //   0.86087,   0.50883
    0x3feb7f6686e792ea,     0x3fe05df3ec31b8b6,     //    0.8593,   0.51147
    0x3feb728345196e3e,     0x3fe073879922ffed,     //   0.85773,    0.5141
    0x3feb658f14fdbc47,     0x3fe089112032b08c,     //   0.85615,   0.51673
    0x3feb5889fe921405,     0x3fe09e907417c5e1,     //   0.85456,   0.51936
    0x3feb4b7409de7925,     0x3fe0b405878f85ec,     //   0.85296,   0.52198
    0x3feb3e4d3ef55712,     0x3fe0c9704d5d898f,     //   0.85136,   0.52459
    0x3feb3115a5f37bf4,     0x3fe0ded0b84bc4b5,     //   0.84974,    0.5272
    0x3feb23cd470013b4,     0x3fe0f426bb2a8e7d,     //   0.84812,    0.5298
    0x3feb16742a4ca2f5,     0x3fe1097248d0a956,     //   0.84649,    0.5324
    0x3feb090a58150200,     0x3fe11eb3541b4b22,     //   0.84485,     0.535
    0x3feafb8fd89f57b6,     0x3fe133e9cfee254e,     //   0.84321,   0.53759
    0x3feaee04b43c1474,     0x3fe14915af336ceb,     //   0.84155,   0.54017
    0x3feae068f345ecef,     0x3fe15e36e4dbe2bc,     //   0.83989,   0.54275
    0x3fead2bc9e21d511,     0x3fe1734d63dedb49,     //   0.83822,   0.54532
    0x3feac4ffbd3efac8,     0x3fe188591f3a46e5,     //   0.83655,   0.54789
    0x3feab7325916c0d4,     0x3fe19d5a09f2b9b8,     //   0.83486,   0.55046
    0x3feaa9547a2cb98e,     0x3fe1b250171373be,     //   0.83317,   0.55302
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //   0.83147,   0.55557
    0x3fea8d676e545ad2,     0x3fe1dc1b64dc4872,     //   0.82976,   0.55812
    0x3fea7f58529fe69d,     0x3fe1f0f08bbc861b,     //   0.82805,   0.56066
    0x3fea7138de9d60f5,     0x3fe205baa17560d6,     //   0.82632,    0.5632
    0x3fea63091b02fae2,     0x3fe21a799933eb58,     //   0.82459,   0.56573
    0x3fea54c91090f524,     0x3fe22f2d662c13e1,     //   0.82285,   0.56826
    0x3fea4678c8119ac8,     0x3fe243d5fb98ac1f,     //    0.8211,   0.57078
    0x3fea38184a593bc6,     0x3fe258734cbb7110,     //   0.81935,    0.5733
    0x3fea29a7a0462782,     0x3fe26d054cdd12df,     //   0.81758,   0.57581
    0x3fea1b26d2c0a75e,     0x3fe2818bef4d3cba,     //   0.81581,   0.57831
    0x3fea0c95eabaf937,     0x3fe2960727629ca8,     //   0.81404,   0.58081
    0x3fe9fdf4f13149de,     0x3fe2aa76e87aeb58,     //   0.81225,   0.58331
    0x3fe9ef43ef29af94,     0x3fe2bedb25faf3ea,     //   0.81046,    0.5858
    0x3fe9e082edb42472,     0x3fe2d333d34e9bb7,     //   0.80866,   0.58828
    0x3fe9d1b1f5ea80d6,     0x3fe2e780e3e8ea16,     //   0.80685,   0.59076
    0x3fe9c2d110f075c3,     0x3fe2fbc24b441015,     //   0.80503,   0.59323
    0x3fe9b3e047f38741,     0x3fe30ff7fce17035,     //   0.80321,    0.5957
    0x3fe9a4dfa42b06b2,     0x3fe32421ec49a620,     //   0.80138,   0.59816
    0x3fe995cf2ed80d22,     0x3fe338400d0c8e57,     //   0.79954,   0.60062
    0x3fe986aef1457594,     0x3fe34c5252c14de1,     //   0.79769,   0.60307
    0x3fe9777ef4c7d742,     0x3fe36058b10659f3,     //   0.79584,   0.60551
    0x3fe9683f42bd7fe1,     0x3fe374531b817f8d,     //   0.79398,   0.60795
    0x3fe958efe48e6dd7,     0x3fe3884185dfeb22,     //   0.79211,   0.61038
    0x3fe94990e3ac4a6c,     0x3fe39c23e3d63029,     //   0.79023,   0.61281
    0x3fe93a22499263fc,     0x3fe3affa292050b9,     //   0.78835,   0.61523
    0x3fe92aa41fc5a815,     0x3fe3c3c44981c517,     //   0.78646,   0.61765
    0x3fe91b166fd49da2,     0x3fe3d78238c58343,     //   0.78456,   0.62006
    0x3fe90b7943575efe,     0x3fe3eb33eabe0680,     //   0.78265,   0.62246
    0x3fe8fbcca3ef940d,     0x3fe3fed9534556d4,     //   0.78074,   0.62486
    0x3fe8ec109b486c49,     0x3fe41272663d108c,     //   0.77882,   0.62725
    0x3fe8dc45331698cc,     0x3fe425ff178e6bb1,     //   0.77689,   0.62964
    0x3fe8cc6a75184655,     0x3fe4397f5b2a4380,     //   0.77495,   0.63202
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //   0.77301,   0.63439
    0x3fe8ac871ede1d88,     0x3fe4605a692b32a2,     //   0.77106,   0.63676
    0x3fe89c7e9a4dd4ab,     0x3fe473b51b987347,     //    0.7691,   0.63912
    0x3fe88c66e7481ba1,     0x3fe48703306091fe,     //   0.76714,   0.64148
    0x3fe87c400fba2ebf,     0x3fe49a449b9b0938,     //   0.76517,   0.64383
    0x3fe86c0a1d9aa195,     0x3fe4ad79516722f0,     //   0.76319,   0.64618
    0x3fe85bc51ae958cc,     0x3fe4c0a145ec0004,     //    0.7612,   0.64851
    0x3fe84b7111af83f9,     0x3fe4d3bc6d589f80,     //   0.75921,   0.65085
    0x3fe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //   0.75721,   0.65317
    0x3fe82a9c13f545ff,     0x3fe4f9cc25cca486,     //    0.7552,   0.65549
    0x3fe81a1b33b57acc,     0x3fe50cc09f59a09b,     //   0.75319,   0.65781
    0x3fe8098b756e52fa,     0x3fe51fa81cd99aa6,     //   0.75117,   0.66011
    0x3fe7f8ece3571771,     0x3fe5328292a35596,     //   0.74914,   0.66242
    0x3fe7e83f87b03686,     0x3fe5454ff5159dfb,     //    0.7471,   0.66471
    0x3fe7d7836cc33db2,     0x3fe5581038975137,     //   0.74506,     0.667
    0x3fe7c6b89ce2d333,     0x3fe56ac35197649e,     //   0.74301,   0.66928
    0x3fe7b5df226aafb0,     0x3fe57d69348cec9f,     //   0.74095,   0.67156
    0x3fe7a4f707bf97d2,     0x3fe59001d5f723df,     //   0.73889,   0.67383
    0x3fe79400574f55e4,     0x3fe5a28d2a5d7250,     //   0.73682,   0.67609
    0x3fe782fb1b90b35b,     0x3fe5b50b264f7448,     //   0.73474,   0.67835
    0x3fe771e75f037261,     0x3fe5c77bbe65018c,     //   0.73265,    0.6806
    0x3fe760c52c304764,     0x3fe5d9dee73e345c,     //   0.73056,   0.68285
    0x3fe74f948da8d28d,     0x3fe5ec3495837074,     //   0.72846,   0.68508
    0x3fe73e558e079942,     0x3fe5fe7cbde56a0f,     //   0.72636,   0.68732
    0x3fe72d0837efff97,     0x3fe610b7551d2cde,     //   0.72425,   0.68954
    0x3fe71bac960e41bf,     0x3fe622e44fec22ff,     //   0.72213,   0.69176
    0x3fe70a42b3176d7a,     0x3fe63503a31c1be8,     //      0.72,   0.69397
    0x3fe6f8ca99c95b75,     0x3fe64715437f535b,     //   0.71787,   0.69618
    0x3fe6e74454eaa8ae,     0x3fe6591925f0783e,     //   0.71573,   0.69838
    0x3fe6d5afef4aafcc,     0x3fe66b0f3f52b386,     //   0.71358,   0.70057
    0x3fe6c40d73c18275,     0x3fe67cf78491af10,     //   0.71143,   0.70275
    0x3fe6b25ced2fe29c,     0x3fe68ed1eaa19c71,     //   0.70927,   0.70493
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //   0.70711,   0.70711
    0x3fe68ed1eaa19c71,     0x3fe6b25ced2fe29c,     //   0.70493,   0.70927
    0x3fe67cf78491af10,     0x3fe6c40d73c18275,     //   0.70275,   0.71143
    0x3fe66b0f3f52b386,     0x3fe6d5afef4aafcc,     //   0.70057,   0.71358
    0x3fe6591925f0783e,     0x3fe6e74454eaa8ae,     //   0.69838,   0.71573
    0x3fe64715437f535b,     0x3fe6f8ca99c95b75,     //   0.69618,   0.71787
    0x3fe63503a31c1be8,     0x3fe70a42b3176d7a,     //   0.69397,      0.72
    0x3fe622e44fec22ff,     0x3fe71bac960e41bf,     //   0.69176,   0.72213
    0x3fe610b7551d2cde,     0x3fe72d0837efff97,     //   0.68954,   0.72425
    0x3fe5fe7cbde56a0f,     0x3fe73e558e079942,     //   0.68732,   0.72636
    0x3fe5ec3495837074,     0x3fe74f948da8d28d,     //   0.68508,   0.72846
    0x3fe5d9dee73e345c,     0x3fe760c52c304764,     //   0.68285,   0.73056
    0x3fe5c77bbe65018c,     0x3fe771e75f037261,     //    0.6806,   0.73265
    0x3fe5b50b264f7448,     0x3fe782fb1b90b35b,     //   0.67835,   0.73474
    0x3fe5a28d2a5d7250,     0x3fe79400574f55e4,     //   0.67609,   0.73682
    0x3fe59001d5f723df,     0x3fe7a4f707bf97d2,     //   0.67383,   0.73889
    0x3fe57d69348cec9f,     0x3fe7b5df226aafb0,     //   0.67156,   0.74095
    0x3fe56ac35197649e,     0x3fe7c6b89ce2d333,     //   0.66928,   0.74301
    0x3fe5581038975137,     0x3fe7d7836cc33db2,     //     0.667,   0.74506
    0x3fe5454ff5159dfb,     0x3fe7e83f87b03686,     //   0.66471,    0.7471
    0x3fe5328292a35596,     0x3fe7f8ece3571771,     //   0.66242,   0.74914
    0x3fe51fa81cd99aa6,     0x3fe8098b756e52fa,     //   0.66011,   0.75117
    0x3fe50cc09f59a09b,     0x3fe81a1b33b57acc,     //   0.65781,   0.75319
    0x3fe4f9cc25cca486,     0x3fe82a9c13f545ff,     //   0.65549,    0.7552
    0x3fe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //   0.65317,   0.75721
    0x3fe4d3bc6d589f80,     0x3fe84b7111af83f9,     //   0.65085,   0.75921
    0x3fe4c0a145ec0004,     0x3fe85bc51ae958cc,     //   0.64851,    0.7612
    0x3fe4ad79516722f0,     0x3fe86c0a1d9aa195,     //   0.64618,   0.76319
    0x3fe49a449b9b0938,     0x3fe87c400fba2ebf,     //   0.64383,   0.76517
    0x3fe48703306091fe,     0x3fe88c66e7481ba1,     //   0.64148,   0.76714
    0x3fe473b51b987347,     0x3fe89c7e9a4dd4ab,     //   0.63912,    0.7691
    0x3fe4605a692b32a2,     0x3fe8ac871ede1d88,     //   0.63676,   0.77106
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //   0.63439,   0.77301
    0x3fe4397f5b2a4380,     0x3fe8cc6a75184655,     //   0.63202,   0.77495
    0x3fe425ff178e6bb1,     0x3fe8dc45331698cc,     //   0.62964,   0.77689
    0x3fe41272663d108c,     0x3fe8ec109b486c49,     //   0.62725,   0.77882
    0x3fe3fed9534556d4,     0x3fe8fbcca3ef940d,     //   0.62486,   0.78074
    0x3fe3eb33eabe0680,     0x3fe90b7943575efe,     //   0.62246,   0.78265
    0x3fe3d78238c58343,     0x3fe91b166fd49da2,     //   0.62006,   0.78456
    0x3fe3c3c44981c517,     0x3fe92aa41fc5a815,     //   0.61765,   0.78646
    0x3fe3affa292050b9,     0x3fe93a22499263fc,     //   0.61523,   0.78835
    0x3fe39c23e3d63029,     0x3fe94990e3ac4a6c,     //   0.61281,   0.79023
    0x3fe3884185dfeb22,     0x3fe958efe48e6dd7,     //   0.61038,   0.79211
    0x3fe374531b817f8d,     0x3fe9683f42bd7fe1,     //   0.60795,   0.79398
    0x3fe36058b10659f3,     0x3fe9777ef4c7d742,     //   0.60551,   0.79584
    0x3fe34c5252c14de1,     0x3fe986aef1457594,     //   0.60307,   0.79769
    0x3fe338400d0c8e57,     0x3fe995cf2ed80d22,     //   0.60062,   0.79954
    0x3fe32421ec49a620,     0x3fe9a4dfa42b06b2,     //   0.59816,   0.80138
    0x3fe30ff7fce17035,     0x3fe9b3e047f38741,     //    0.5957,   0.80321
    0x3fe2fbc24b441015,     0x3fe9c2d110f075c3,     //   0.59323,   0.80503
    0x3fe2e780e3e8ea16,     0x3fe9d1b1f5ea80d6,     //   0.59076,   0.80685
    0x3fe2d333d34e9bb7,     0x3fe9e082edb42472,     //   0.58828,   0.80866
    0x3fe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //    0.5858,   0.81046
    0x3fe2aa76e87aeb58,     0x3fe9fdf4f13149de,     //   0.58331,   0.81225
    0x3fe2960727629ca8,     0x3fea0c95eabaf937,     //   0.58081,   0.81404
    0x3fe2818bef4d3cba,     0x3fea1b26d2c0a75e,     //   0.57831,   0.81581
    0x3fe26d054cdd12df,     0x3fea29a7a0462782,     //   0.57581,   0.81758
    0x3fe258734cbb7110,     0x3fea38184a593bc6,     //    0.5733,   0.81935
    0x3fe243d5fb98ac1f,     0x3fea4678c8119ac8,     //   0.57078,    0.8211
    0x3fe22f2d662c13e1,     0x3fea54c91090f524,     //   0.56826,   0.82285
    0x3fe21a799933eb58,     0x3fea63091b02fae2,     //   0.56573,   0.82459
    0x3fe205baa17560d6,     0x3fea7138de9d60f5,     //    0.5632,   0.82632
    0x3fe1f0f08bbc861b,     0x3fea7f58529fe69d,     //   0.56066,   0.82805
    0x3fe1dc1b64dc4872,     0x3fea8d676e545ad2,     //   0.55812,   0.82976
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //   0.55557,   0.83147
    0x3fe1b250171373be,     0x3feaa9547a2cb98e,     //   0.55302,   0.83317
    0x3fe19d5a09f2b9b8,     0x3feab7325916c0d4,     //   0.55046,   0.83486
    0x3fe188591f3a46e5,     0x3feac4ffbd3efac8,     //   0.54789,   0.83655
    0x3fe1734d63dedb49,     0x3fead2bc9e21d511,     //   0.54532,   0.83822
    0x3fe15e36e4dbe2bc,     0x3feae068f345ecef,     //   0.54275,   0.83989
    0x3fe14915af336ceb,     0x3feaee04b43c1474,     //   0.54017,   0.84155
    0x3fe133e9cfee254e,     0x3feafb8fd89f57b6,     //   0.53759,   0.84321
    0x3fe11eb3541b4b22,     0x3feb090a58150200,     //     0.535,   0.84485
    0x3fe1097248d0a956,     0x3feb16742a4ca2f5,     //    0.5324,   0.84649
    0x3fe0f426bb2a8e7d,     0x3feb23cd470013b4,     //    0.5298,   0.84812
    0x3fe0ded0b84bc4b5,     0x3feb3115a5f37bf4,     //    0.5272,   0.84974
    0x3fe0c9704d5d898f,     0x3feb3e4d3ef55712,     //   0.52459,   0.85136
    0x3fe0b405878f85ec,     0x3feb4b7409de7925,     //   0.52198,   0.85296
    0x3fe09e907417c5e1,     0x3feb5889fe921405,     //   0.51936,   0.85456
    0x3fe089112032b08c,     0x3feb658f14fdbc47,     //   0.51673,   0.85615
    0x3fe073879922ffed,     0x3feb728345196e3e,     //    0.5141,   0.85773
    0x3fe05df3ec31b8b6,     0x3feb7f6686e792ea,     //   0.51147,    0.8593
    0x3fe0485626ae221a,     0x3feb8c38d27504e9,     //   0.50883,   0.86087
    0x3fe032ae55edbd95,     0x3feb98fa1fd9155e,     //   0.50619,   0.86242
    0x3fe01cfc874c3eb7,     0x3feba5aa673590d2,     //   0.50354,   0.86397
    0x3fe00740c82b82e0,     0x3febb249a0b6c40d,     //   0.50089,   0.86551
    0x3fdfe2f64be7120f,     0x3febbed7c49380ea,     //   0.49823,   0.86705
    0x3fdfb7575c24d2de,     0x3febcb54cb0d2327,     //   0.49557,   0.86857
    0x3fdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //    0.4929,   0.87009
    0x3fdf5fdee656cda3,     0x3febe41b611154c1,     //   0.49023,    0.8716
    0x3fdf3405963fd068,     0x3febf064e15377dd,     //   0.48755,   0.87309
    0x3fdf081906bff7fd,     0x3febfc9d25a1b147,     //   0.48487,   0.87459
    0x3fdedc1952ef78d5,     0x3fec08c426725549,     //   0.48218,   0.87607
    0x3fdeb00695f25620,     0x3fec14d9dc465e58,     //   0.47949,   0.87755
    0x3fde83e0eaf85113,     0x3fec20de3fa971b0,     //    0.4768,   0.87901
    0x3fde57a86d3cd824,     0x3fec2cd14931e3f1,     //    0.4741,   0.88047
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //    0.4714,   0.88192
    0x3fddfeff66a941de,     0x3fec44833141c004,     //   0.46869,   0.88336
    0x3fddd28f1481cc58,     0x3fec5042012b6907,     //   0.46598,    0.8848
    0x3fdda60c5cfa10d8,     0x3fec5bef59fef85a,     //   0.46326,   0.88622
    0x3fdd79775b86e389,     0x3fec678b3488739b,     //   0.46054,   0.88764
    0x3fdd4cd02ba8609c,     0x3fec7315899eaad7,     //   0.45781,   0.88905
    0x3fdd2016e8e9db5b,     0x3fec7e8e52233cf3,     //   0.45508,   0.89045
    0x3fdcf34baee1cd21,     0x3fec89f587029c13,     //   0.45235,   0.89184
    0x3fdcc66e9931c45d,     0x3fec954b213411f5,     //   0.44961,   0.89322
    0x3fdc997fc3865388,     0x3feca08f19b9c449,     //   0.44687,    0.8946
    0x3fdc6c7f4997000a,     0x3fecabc169a0b901,     //   0.44412,   0.89597
    0x3fdc3f6d47263129,     0x3fecb6e20a00da99,     //   0.44137,   0.89732
    0x3fdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     //   0.43862,   0.89867
    0x3fdbe51517ffc0d9,     0x3fecccee20c2de9f,     //   0.43586,   0.90002
    0x3fdbb7cf2304bd01,     0x3fecd7d9898b32f6,     //   0.43309,   0.90135
    0x3fdb8a7814fd5693,     0x3fece2b32799a060,     //   0.43033,   0.90267
    0x3fdb5d1009e15cc0,     0x3feced7af43cc773,     //   0.42756,   0.90399
    0x3fdb2f971db31972,     0x3fecf830e8ce467b,     //   0.42478,    0.9053
    0x3fdb020d6c7f4009,     0x3fed02d4feb2bd92,     //     0.422,    0.9066
    0x3fdad473125cdc08,     0x3fed0d672f59d2b9,     //   0.41922,   0.90789
    0x3fdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     //   0.41643,   0.90917
    0x3fda790cd3dbf31a,     0x3fed2255c6e5a4e1,     //   0.41364,   0.91044
    0x3fda4b4127dea1e4,     0x3fed2cb220e0ef9f,     //   0.41084,   0.91171
    0x3fda1d6543b50ac0,     0x3fed36fc7bcbfbdc,     //   0.40804,   0.91296
    0x3fd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //   0.40524,   0.91421
    0x3fd9c17d440df9f2,     0x3fed4b5b1b187524,     //   0.40243,   0.91545
    0x3fd993716141bdfe,     0x3fed556f52e93eb1,     //   0.39962,   0.91668
    0x3fd96555b7ab948f,     0x3fed5f7172888a7f,     //   0.39681,    0.9179
    0x3fd9372a63bc93d7,     0x3fed696173c9e68b,     //   0.39399,   0.91911
    0x3fd908ef81ef7bd1,     0x3fed733f508c0dff,     //   0.39117,   0.92032
    0x3fd8daa52ec8a4af,     0x3fed7d0b02b8ecf9,     //   0.38835,   0.92151
    0x3fd8ac4b86d5ed44,     0x3fed86c48445a450,     //   0.38552,    0.9227
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //   0.38268,   0.92388
    0x3fd84f6aaaf3903f,     0x3fed9a00dd8b3d46,     //   0.37985,   0.92505
    0x3fd820e3b04eaac4,     0x3feda383a9668988,     //   0.37701,   0.92621
    0x3fd7f24dd37341e3,     0x3fedacf42ce68ab9,     //   0.37416,   0.92736
    0x3fd7c3a9311dcce7,     0x3fedb6526238a09b,     //   0.37132,   0.92851
    0x3fd794f5e613dfae,     0x3fedbf9e4395759a,     //   0.36847,   0.92964
    0x3fd766340f2418f6,     0x3fedc8d7cb410260,     //   0.36561,   0.93077
    0x3fd73763c9261092,     0x3fedd1fef38a915a,     //   0.36276,   0.93188
    0x3fd7088530fa459e,     0x3feddb13b6ccc23d,     //    0.3599,   0.93299
    0x3fd6d998638a0cb5,     0x3fede4160f6d8d81,     //   0.35703,   0.93409
    0x3fd6aa9d7dc77e16,     0x3feded05f7de47da,     //   0.35416,   0.93518
    0x3fd67b949cad63ca,     0x3fedf5e36a9ba59c,     //   0.35129,   0.93627
    0x3fd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     //   0.34842,   0.93734
    0x3fd61d595c88c203,     0x3fee0766d9280f54,     //   0.34554,    0.9384
    0x3fd5ee27379ea693,     0x3fee100cca2980ac,     //   0.34266,   0.93946
    0x3fd5bee78b9db3b6,     0x3fee18a02fdc66d9,     //   0.33978,   0.94051
    0x3fd58f9a75ab1fdd,     0x3fee212104f686e5,     //   0.33689,   0.94154
    0x3fd5604012f467b4,     0x3fee298f4439197a,     //     0.334,   0.94257
    0x3fd530d880af3c24,     0x3fee31eae870ce25,     //   0.33111,   0.94359
    0x3fd50163dc197047,     0x3fee3a33ec75ce85,     //   0.32821,    0.9446
    0x3fd4d1e24278e76a,     0x3fee426a4b2bc17e,     //   0.32531,   0.94561
    0x3fd4a253d11b82f3,     0x3fee4a8dff81ce5e,     //   0.32241,    0.9466
    0x3fd472b8a5571054,     0x3fee529f04729ffc,     //    0.3195,   0.94759
    0x3fd44310dc8936f0,     0x3fee5a9d550467d3,     //   0.31659,   0.94856
    0x3fd4135c94176602,     0x3fee6288ec48e112,     //   0.31368,   0.94953
    0x3fd3e39be96ec271,     0x3fee6a61c55d53a7,     //   0.31077,   0.95049
    0x3fd3b3cefa0414b7,     0x3fee7227db6a9744,     //   0.30785,   0.95144
    0x3fd383f5e353b6aa,     0x3fee79db29a5165a,     //   0.30493,   0.95238
    0x3fd35410c2e18152,     0x3fee817bab4cd10d,     //   0.30201,   0.95331
    0x3fd3241fb638baaf,     0x3fee89095bad6025,     //   0.29908,   0.95423
    0x3fd2f422daec0386,     0x3fee9084361df7f3,     //   0.29615,   0.95514
    0x3fd2c41a4e954520,     0x3fee97ec36016b30,     //   0.29322,   0.95605
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //   0.29028,   0.95694
    0x3fd263e6995554ba,     0x3feea68393e65800,     //   0.28735,   0.95783
    0x3fd233bbabc3bb72,     0x3feeadb2e8e7a88e,     //   0.28441,    0.9587
    0x3fd2038583d727bd,     0x3feeb4cf515b8811,     //   0.28146,   0.95957
    0x3fd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     //   0.27852,   0.96043
    0x3fd1a2f7fbe8f243,     0x3feec2cf4b1af6b2,     //   0.27557,   0.96128
    0x3fd172a0d7765177,     0x3feec9b2d3c3bf84,     //   0.27262,   0.96212
    0x3fd1423eefc69378,     0x3feed0835e999009,     //   0.26967,   0.96295
    0x3fd111d262b1f677,     0x3feed740e7684963,     //   0.26671,   0.96378
    0x3fd0e15b4e1749cd,     0x3feeddeb6a078651,     //   0.26375,   0.96459
    0x3fd0b0d9cfdbdb90,     0x3feee482e25a9dbc,     //   0.26079,   0.96539
    0x3fd0804e05eb661e,     0x3feeeb074c50a544,     //   0.25783,   0.96619
    0x3fd04fb80e37fdae,     0x3feef178a3e473c2,     //   0.25487,   0.96698
    0x3fd01f1806b9fdd2,     0x3feef7d6e51ca3c0,     //    0.2519,   0.96775
    0x3fcfdcdc1adfedf8,     0x3feefe220c0b95ec,     //   0.24893,   0.96852
    0x3fcf7b7480bd3801,     0x3fef045a14cf738c,     //   0.24596,   0.96928
    0x3fcf19f97b215f1a,     0x3fef0a7efb9230d7,     //   0.24298,   0.97003
    0x3fceb86b462de348,     0x3fef1090bc898f5f,     //      0.24,   0.97077
    0x3fce56ca1e101a1b,     0x3fef168f53f7205d,     //   0.23702,    0.9715
    0x3fcdf5163f01099a,     0x3fef1c7abe284708,     //   0.23404,   0.97223
    0x3fcd934fe5454311,     0x3fef2252f7763ada,     //   0.23106,   0.97294
    0x3fcd31774d2cbdee,     0x3fef2817fc4609ce,     //   0.22807,   0.97364
    0x3fcccf8cb312b286,     0x3fef2dc9c9089a9d,     //   0.22508,   0.97434
    0x3fcc6d90535d74dc,     0x3fef33685a3aaef0,     //   0.22209,   0.97503
    0x3fcc0b826a7e4f63,     0x3fef38f3ac64e589,     //    0.2191,    0.9757
    0x3fcba96334f15dad,     0x3fef3e6bbc1bbc65,     //   0.21611,   0.97637
    0x3fcb4732ef3d6722,     0x3fef43d085ff92dd,     //   0.21311,   0.97703
    0x3fcae4f1d5f3b9ab,     0x3fef492206bcabb4,     //   0.21011,   0.97768
    0x3fca82a025b00451,     0x3fef4e603b0b2f2d,     //   0.20711,   0.97832
    0x3fca203e1b1831da,     0x3fef538b1faf2d07,     //   0.20411,   0.97895
    0x3fc9bdcbf2dc4366,     0x3fef58a2b1789e84,     //    0.2011,   0.97957
    0x3fc95b49e9b62af9,     0x3fef5da6ed43685d,     //    0.1981,   0.98018
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //   0.19509,   0.98079
    0x3fc8961727c41804,     0x3fef677556883cee,     //   0.19208,   0.98138
    0x3fc83366e89c64c5,     0x3fef6c3f7df5bbb7,     //   0.18907,   0.98196
    0x3fc7d0a7bbd2cb1b,     0x3fef70f6434b7eb7,     //   0.18606,   0.98254
    0x3fc76dd9de50bf31,     0x3fef7599a3a12077,     //   0.18304,   0.98311
    0x3fc70afd8d08c4ff,     0x3fef7a299c1a322a,     //   0.18002,   0.98366
    0x3fc6a81304f64ab2,     0x3fef7ea629e63d6e,     //     0.177,   0.98421
    0x3fc6451a831d830d,     0x3fef830f4a40c60c,     //   0.17398,   0.98475
    0x3fc5e214448b3fc6,     0x3fef8764fa714ba9,     //   0.17096,   0.98528
    0x3fc57f008654cbde,     0x3fef8ba737cb4b78,     //   0.16794,    0.9858
    0x3fc51bdf8597c5f2,     0x3fef8fd5ffae41db,     //   0.16491,   0.98631
    0x3fc4b8b17f79fa88,     0x3fef93f14f85ac08,     //   0.16189,   0.98681
    0x3fc45576b1293e5a,     0x3fef97f924c9099b,     //   0.15886,    0.9873
    0x3fc3f22f57db4893,     0x3fef9bed7cfbde29,     //   0.15583,   0.98778
    0x3fc38edbb0cd8d14,     0x3fef9fce55adb2c8,     //    0.1528,   0.98826
    0x3fc32b7bf94516a7,     0x3fefa39bac7a1791,     //   0.14976,   0.98872
    0x3fc2c8106e8e613a,     0x3fefa7557f08a517,     //   0.14673,   0.98918
    0x3fc264994dfd340a,     0x3fefaafbcb0cfddc,     //    0.1437,   0.98962
    0x3fc20116d4ec7bce,     0x3fefae8e8e46cfbb,     //   0.14066,   0.99006
    0x3fc19d8940be24e7,     0x3fefb20dc681d54d,     //   0.13762,   0.99049
    0x3fc139f0cedaf576,     0x3fefb5797195d741,     //   0.13458,    0.9909
    0x3fc0d64dbcb26786,     0x3fefb8d18d66adb7,     //   0.13154,   0.99131
    0x3fc072a047ba831d,     0x3fefbc1617e44186,     //    0.1285,   0.99171
    0x3fc00ee8ad6fb85b,     0x3fefbf470f0a8d88,     //   0.12545,    0.9921
    0x3fbf564e56a9730e,     0x3fefc26470e19fd3,     //   0.12241,   0.99248
    0x3fbe8eb7fde4aa3e,     0x3fefc56e3b7d9af6,     //   0.11937,   0.99285
    0x3fbdc70ecbae9fc8,     0x3fefc8646cfeb721,     //   0.11632,   0.99321
    0x3fbcff533b307dc1,     0x3fefcb4703914354,     //   0.11327,   0.99356
    0x3fbc3785c79ec2d5,     0x3fefce15fd6da67b,     //   0.11022,   0.99391
    0x3fbb6fa6ec38f64c,     0x3fefd0d158d86087,     //   0.10717,   0.99424
    0x3fbaa7b724495c04,     0x3fefd37914220b84,     //   0.10412,   0.99456
    0x3fb9dfb6eb24a85c,     0x3fefd60d2da75c9e,     //   0.10107,   0.99488
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     //  0.098017,   0.99518
    0x3fb84f8712c130a0,     0x3fefdafa7514538c,     //  0.094963,   0.99548
    0x3fb787586a5d5b21,     0x3fefdd539ff1f456,     //  0.091909,   0.99577
    0x3fb6bf1b3e79b129,     0x3fefdf9922f73307,     //  0.088854,   0.99604
    0x3fb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     //  0.085797,   0.99631
    0x3fb52e774a4d4d0a,     0x3fefe3e92be9d886,     //   0.08274,   0.99657
    0x3fb4661179272096,     0x3fefe5f3af2e3940,     //  0.079682,   0.99682
    0x3fb39d9f12c5a299,     0x3fefe7ea85482d60,     //  0.076624,   0.99706
    0x3fb2d52092ce19f6,     0x3fefe9cdad01883a,     //  0.073565,   0.99729
    0x3fb20c9674ed444c,     0x3fefeb9d2530410f,     //  0.070505,   0.99751
    0x3fb1440134d709b2,     0x3fefed58ecb673c4,     //  0.067444,   0.99772
    0x3fb07b614e463064,     0x3fefef0102826191,     //  0.064383,   0.99793
    0x3faf656e79f820e0,     0x3feff095658e71ad,     //  0.061321,   0.99812
    0x3fadd406f9808ec8,     0x3feff21614e131ed,     //  0.058258,    0.9983
    0x3fac428d12c0d7e3,     0x3feff3830f8d575c,     //  0.055195,   0.99848
    0x3faab101bd5f8317,     0x3feff4dc54b1bed3,     //  0.052132,   0.99864
    0x3fa91f65f10dd814,     0x3feff621e3796d7e,     //  0.049068,    0.9988
    0x3fa78dbaa5874685,     0x3feff753bb1b9164,     //  0.046003,   0.99894
    0x3fa5fc00d290cd43,     0x3feff871dadb81df,     //  0.042938,   0.99908
    0x3fa46a396ff86179,     0x3feff97c4208c014,     //  0.039873,    0.9992
    0x3fa2d865759455cd,     0x3feffa72effef75d,     //  0.036807,   0.99932
    0x3fa14685db42c17e,     0x3feffb55e425fdae,     //  0.033741,   0.99943
    0x3f9f693731d1cf01,     0x3feffc251df1d3f8,     //  0.030675,   0.99953
    0x3f9c454f4ce53b1c,     0x3feffce09ce2a679,     //  0.027608,   0.99962
    0x3f992155f7a3667e,     0x3feffd886084cd0d,     //  0.024541,    0.9997
    0x3f95fd4d21fab226,     0x3feffe1c6870cb77,     //  0.021474,   0.99977
    0x3f92d936bbe30efd,     0x3feffe9cb44b51a1,     //  0.018407,   0.99983
    0x3f8f6a296ab997ca,     0x3fefff0943c53bd1,     //  0.015339,   0.99988
    0x3f8921d1fcdec784,     0x3fefff62169b92db,     //  0.012272,   0.99992
    0x3f82d96b0e509703,     0x3fefffa72c978c4f,     // 0.0092038,   0.99996
    0x3f7921f0fe670071,     0x3fefffd8858e8a92,     // 0.0061359,   0.99998
    0x3f6921f8becca4ba,     0x3feffff621621d02,     //  0.003068,         1
    0x0000000000000000,     0x3ff0000000000000,     //         0,         1
    0xbf6921f8becca4ba,     0x3feffff621621d02,     // -0.003068,         1
    0xbf7921f0fe670071,     0x3fefffd8858e8a92,     //-0.0061359,   0.99998
    0xbf82d96b0e509703,     0x3fefffa72c978c4f,     //-0.0092038,   0.99996
    0xbf8921d1fcdec784,     0x3fefff62169b92db,     // -0.012272,   0.99992
    0xbf8f6a296ab997ca,     0x3fefff0943c53bd1,     // -0.015339,   0.99988
    0xbf92d936bbe30efd,     0x3feffe9cb44b51a1,     // -0.018407,   0.99983
    0xbf95fd4d21fab226,     0x3feffe1c6870cb77,     // -0.021474,   0.99977
    0xbf992155f7a3667e,     0x3feffd886084cd0d,     // -0.024541,    0.9997
    0xbf9c454f4ce53b1c,     0x3feffce09ce2a679,     // -0.027608,   0.99962
    0xbf9f693731d1cf01,     0x3feffc251df1d3f8,     // -0.030675,   0.99953
    0xbfa14685db42c17e,     0x3feffb55e425fdae,     // -0.033741,   0.99943
    0xbfa2d865759455cd,     0x3feffa72effef75d,     // -0.036807,   0.99932
    0xbfa46a396ff86179,     0x3feff97c4208c014,     // -0.039873,    0.9992
    0xbfa5fc00d290cd43,     0x3feff871dadb81df,     // -0.042938,   0.99908
    0xbfa78dbaa5874685,     0x3feff753bb1b9164,     // -0.046003,   0.99894
    0xbfa91f65f10dd814,     0x3feff621e3796d7e,     // -0.049068,    0.9988
    0xbfaab101bd5f8317,     0x3feff4dc54b1bed3,     // -0.052132,   0.99864
    0xbfac428d12c0d7e3,     0x3feff3830f8d575c,     // -0.055195,   0.99848
    0xbfadd406f9808ec8,     0x3feff21614e131ed,     // -0.058258,    0.9983
    0xbfaf656e79f820e0,     0x3feff095658e71ad,     // -0.061321,   0.99812
    0xbfb07b614e463064,     0x3fefef0102826191,     // -0.064383,   0.99793
    0xbfb1440134d709b2,     0x3fefed58ecb673c4,     // -0.067444,   0.99772
    0xbfb20c9674ed444c,     0x3fefeb9d2530410f,     // -0.070505,   0.99751
    0xbfb2d52092ce19f6,     0x3fefe9cdad01883a,     // -0.073565,   0.99729
    0xbfb39d9f12c5a299,     0x3fefe7ea85482d60,     // -0.076624,   0.99706
    0xbfb4661179272096,     0x3fefe5f3af2e3940,     // -0.079682,   0.99682
    0xbfb52e774a4d4d0a,     0x3fefe3e92be9d886,     //  -0.08274,   0.99657
    0xbfb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     // -0.085797,   0.99631
    0xbfb6bf1b3e79b129,     0x3fefdf9922f73307,     // -0.088854,   0.99604
    0xbfb787586a5d5b21,     0x3fefdd539ff1f456,     // -0.091909,   0.99577
    0xbfb84f8712c130a0,     0x3fefdafa7514538c,     // -0.094963,   0.99548
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     // -0.098017,   0.99518
    0xbfb9dfb6eb24a85c,     0x3fefd60d2da75c9e,     //  -0.10107,   0.99488
    0xbfbaa7b724495c04,     0x3fefd37914220b84,     //  -0.10412,   0.99456
    0xbfbb6fa6ec38f64c,     0x3fefd0d158d86087,     //  -0.10717,   0.99424
    0xbfbc3785c79ec2d5,     0x3fefce15fd6da67b,     //  -0.11022,   0.99391
    0xbfbcff533b307dc1,     0x3fefcb4703914354,     //  -0.11327,   0.99356
    0xbfbdc70ecbae9fc8,     0x3fefc8646cfeb721,     //  -0.11632,   0.99321
    0xbfbe8eb7fde4aa3e,     0x3fefc56e3b7d9af6,     //  -0.11937,   0.99285
    0xbfbf564e56a9730e,     0x3fefc26470e19fd3,     //  -0.12241,   0.99248
    0xbfc00ee8ad6fb85b,     0x3fefbf470f0a8d88,     //  -0.12545,    0.9921
    0xbfc072a047ba831d,     0x3fefbc1617e44186,     //   -0.1285,   0.99171
    0xbfc0d64dbcb26786,     0x3fefb8d18d66adb7,     //  -0.13154,   0.99131
    0xbfc139f0cedaf576,     0x3fefb5797195d741,     //  -0.13458,    0.9909
    0xbfc19d8940be24e7,     0x3fefb20dc681d54d,     //  -0.13762,   0.99049
    0xbfc20116d4ec7bce,     0x3fefae8e8e46cfbb,     //  -0.14066,   0.99006
    0xbfc264994dfd340a,     0x3fefaafbcb0cfddc,     //   -0.1437,   0.98962
    0xbfc2c8106e8e613a,     0x3fefa7557f08a517,     //  -0.14673,   0.98918
    0xbfc32b7bf94516a7,     0x3fefa39bac7a1791,     //  -0.14976,   0.98872
    0xbfc38edbb0cd8d14,     0x3fef9fce55adb2c8,     //   -0.1528,   0.98826
    0xbfc3f22f57db4893,     0x3fef9bed7cfbde29,     //  -0.15583,   0.98778
    0xbfc45576b1293e5a,     0x3fef97f924c9099b,     //  -0.15886,    0.9873
    0xbfc4b8b17f79fa88,     0x3fef93f14f85ac08,     //  -0.16189,   0.98681
    0xbfc51bdf8597c5f2,     0x3fef8fd5ffae41db,     //  -0.16491,   0.98631
    0xbfc57f008654cbde,     0x3fef8ba737cb4b78,     //  -0.16794,    0.9858
    0xbfc5e214448b3fc6,     0x3fef8764fa714ba9,     //  -0.17096,   0.98528
    0xbfc6451a831d830d,     0x3fef830f4a40c60c,     //  -0.17398,   0.98475
    0xbfc6a81304f64ab2,     0x3fef7ea629e63d6e,     //    -0.177,   0.98421
    0xbfc70afd8d08c4ff,     0x3fef7a299c1a322a,     //  -0.18002,   0.98366
    0xbfc76dd9de50bf31,     0x3fef7599a3a12077,     //  -0.18304,   0.98311
    0xbfc7d0a7bbd2cb1b,     0x3fef70f6434b7eb7,     //  -0.18606,   0.98254
    0xbfc83366e89c64c5,     0x3fef6c3f7df5bbb7,     //  -0.18907,   0.98196
    0xbfc8961727c41804,     0x3fef677556883cee,     //  -0.19208,   0.98138
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  -0.19509,   0.98079
    0xbfc95b49e9b62af9,     0x3fef5da6ed43685d,     //   -0.1981,   0.98018
    0xbfc9bdcbf2dc4366,     0x3fef58a2b1789e84,     //   -0.2011,   0.97957
    0xbfca203e1b1831da,     0x3fef538b1faf2d07,     //  -0.20411,   0.97895
    0xbfca82a025b00451,     0x3fef4e603b0b2f2d,     //  -0.20711,   0.97832
    0xbfcae4f1d5f3b9ab,     0x3fef492206bcabb4,     //  -0.21011,   0.97768
    0xbfcb4732ef3d6722,     0x3fef43d085ff92dd,     //  -0.21311,   0.97703
    0xbfcba96334f15dad,     0x3fef3e6bbc1bbc65,     //  -0.21611,   0.97637
    0xbfcc0b826a7e4f63,     0x3fef38f3ac64e589,     //   -0.2191,    0.9757
    0xbfcc6d90535d74dc,     0x3fef33685a3aaef0,     //  -0.22209,   0.97503
    0xbfcccf8cb312b286,     0x3fef2dc9c9089a9d,     //  -0.22508,   0.97434
    0xbfcd31774d2cbdee,     0x3fef2817fc4609ce,     //  -0.22807,   0.97364
    0xbfcd934fe5454311,     0x3fef2252f7763ada,     //  -0.23106,   0.97294
    0xbfcdf5163f01099a,     0x3fef1c7abe284708,     //  -0.23404,   0.97223
    0xbfce56ca1e101a1b,     0x3fef168f53f7205d,     //  -0.23702,    0.9715
    0xbfceb86b462de348,     0x3fef1090bc898f5f,     //     -0.24,   0.97077
    0xbfcf19f97b215f1a,     0x3fef0a7efb9230d7,     //  -0.24298,   0.97003
    0xbfcf7b7480bd3801,     0x3fef045a14cf738c,     //  -0.24596,   0.96928
    0xbfcfdcdc1adfedf8,     0x3feefe220c0b95ec,     //  -0.24893,   0.96852
    0xbfd01f1806b9fdd2,     0x3feef7d6e51ca3c0,     //   -0.2519,   0.96775
    0xbfd04fb80e37fdae,     0x3feef178a3e473c2,     //  -0.25487,   0.96698
    0xbfd0804e05eb661e,     0x3feeeb074c50a544,     //  -0.25783,   0.96619
    0xbfd0b0d9cfdbdb90,     0x3feee482e25a9dbc,     //  -0.26079,   0.96539
    0xbfd0e15b4e1749cd,     0x3feeddeb6a078651,     //  -0.26375,   0.96459
    0xbfd111d262b1f677,     0x3feed740e7684963,     //  -0.26671,   0.96378
    0xbfd1423eefc69378,     0x3feed0835e999009,     //  -0.26967,   0.96295
    0xbfd172a0d7765177,     0x3feec9b2d3c3bf84,     //  -0.27262,   0.96212
    0xbfd1a2f7fbe8f243,     0x3feec2cf4b1af6b2,     //  -0.27557,   0.96128
    0xbfd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     //  -0.27852,   0.96043
    0xbfd2038583d727bd,     0x3feeb4cf515b8811,     //  -0.28146,   0.95957
    0xbfd233bbabc3bb72,     0x3feeadb2e8e7a88e,     //  -0.28441,    0.9587
    0xbfd263e6995554ba,     0x3feea68393e65800,     //  -0.28735,   0.95783
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     //  -0.29028,   0.95694
    0xbfd2c41a4e954520,     0x3fee97ec36016b30,     //  -0.29322,   0.95605
    0xbfd2f422daec0386,     0x3fee9084361df7f3,     //  -0.29615,   0.95514
    0xbfd3241fb638baaf,     0x3fee89095bad6025,     //  -0.29908,   0.95423
    0xbfd35410c2e18152,     0x3fee817bab4cd10d,     //  -0.30201,   0.95331
    0xbfd383f5e353b6aa,     0x3fee79db29a5165a,     //  -0.30493,   0.95238
    0xbfd3b3cefa0414b7,     0x3fee7227db6a9744,     //  -0.30785,   0.95144
    0xbfd3e39be96ec271,     0x3fee6a61c55d53a7,     //  -0.31077,   0.95049
    0xbfd4135c94176602,     0x3fee6288ec48e112,     //  -0.31368,   0.94953
    0xbfd44310dc8936f0,     0x3fee5a9d550467d3,     //  -0.31659,   0.94856
    0xbfd472b8a5571054,     0x3fee529f04729ffc,     //   -0.3195,   0.94759
    0xbfd4a253d11b82f3,     0x3fee4a8dff81ce5e,     //  -0.32241,    0.9466
    0xbfd4d1e24278e76a,     0x3fee426a4b2bc17e,     //  -0.32531,   0.94561
    0xbfd50163dc197047,     0x3fee3a33ec75ce85,     //  -0.32821,    0.9446
    0xbfd530d880af3c24,     0x3fee31eae870ce25,     //  -0.33111,   0.94359
    0xbfd5604012f467b4,     0x3fee298f4439197a,     //    -0.334,   0.94257
    0xbfd58f9a75ab1fdd,     0x3fee212104f686e5,     //  -0.33689,   0.94154
    0xbfd5bee78b9db3b6,     0x3fee18a02fdc66d9,     //  -0.33978,   0.94051
    0xbfd5ee27379ea693,     0x3fee100cca2980ac,     //  -0.34266,   0.93946
    0xbfd61d595c88c203,     0x3fee0766d9280f54,     //  -0.34554,    0.9384
    0xbfd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     //  -0.34842,   0.93734
    0xbfd67b949cad63ca,     0x3fedf5e36a9ba59c,     //  -0.35129,   0.93627
    0xbfd6aa9d7dc77e16,     0x3feded05f7de47da,     //  -0.35416,   0.93518
    0xbfd6d998638a0cb5,     0x3fede4160f6d8d81,     //  -0.35703,   0.93409
    0xbfd7088530fa459e,     0x3feddb13b6ccc23d,     //   -0.3599,   0.93299
    0xbfd73763c9261092,     0x3fedd1fef38a915a,     //  -0.36276,   0.93188
    0xbfd766340f2418f6,     0x3fedc8d7cb410260,     //  -0.36561,   0.93077
    0xbfd794f5e613dfae,     0x3fedbf9e4395759a,     //  -0.36847,   0.92964
    0xbfd7c3a9311dcce7,     0x3fedb6526238a09b,     //  -0.37132,   0.92851
    0xbfd7f24dd37341e3,     0x3fedacf42ce68ab9,     //  -0.37416,   0.92736
    0xbfd820e3b04eaac4,     0x3feda383a9668988,     //  -0.37701,   0.92621
    0xbfd84f6aaaf3903f,     0x3fed9a00dd8b3d46,     //  -0.37985,   0.92505
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     //  -0.38268,   0.92388
    0xbfd8ac4b86d5ed44,     0x3fed86c48445a450,     //  -0.38552,    0.9227
    0xbfd8daa52ec8a4af,     0x3fed7d0b02b8ecf9,     //  -0.38835,   0.92151
    0xbfd908ef81ef7bd1,     0x3fed733f508c0dff,     //  -0.39117,   0.92032
    0xbfd9372a63bc93d7,     0x3fed696173c9e68b,     //  -0.39399,   0.91911
    0xbfd96555b7ab948f,     0x3fed5f7172888a7f,     //  -0.39681,    0.9179
    0xbfd993716141bdfe,     0x3fed556f52e93eb1,     //  -0.39962,   0.91668
    0xbfd9c17d440df9f2,     0x3fed4b5b1b187524,     //  -0.40243,   0.91545
    0xbfd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //  -0.40524,   0.91421
    0xbfda1d6543b50ac0,     0x3fed36fc7bcbfbdc,     //  -0.40804,   0.91296
    0xbfda4b4127dea1e4,     0x3fed2cb220e0ef9f,     //  -0.41084,   0.91171
    0xbfda790cd3dbf31a,     0x3fed2255c6e5a4e1,     //  -0.41364,   0.91044
    0xbfdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     //  -0.41643,   0.90917
    0xbfdad473125cdc08,     0x3fed0d672f59d2b9,     //  -0.41922,   0.90789
    0xbfdb020d6c7f4009,     0x3fed02d4feb2bd92,     //    -0.422,    0.9066
    0xbfdb2f971db31972,     0x3fecf830e8ce467b,     //  -0.42478,    0.9053
    0xbfdb5d1009e15cc0,     0x3feced7af43cc773,     //  -0.42756,   0.90399
    0xbfdb8a7814fd5693,     0x3fece2b32799a060,     //  -0.43033,   0.90267
    0xbfdbb7cf2304bd01,     0x3fecd7d9898b32f6,     //  -0.43309,   0.90135
    0xbfdbe51517ffc0d9,     0x3fecccee20c2de9f,     //  -0.43586,   0.90002
    0xbfdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     //  -0.43862,   0.89867
    0xbfdc3f6d47263129,     0x3fecb6e20a00da99,     //  -0.44137,   0.89732
    0xbfdc6c7f4997000a,     0x3fecabc169a0b901,     //  -0.44412,   0.89597
    0xbfdc997fc3865388,     0x3feca08f19b9c449,     //  -0.44687,    0.8946
    0xbfdcc66e9931c45d,     0x3fec954b213411f5,     //  -0.44961,   0.89322
    0xbfdcf34baee1cd21,     0x3fec89f587029c13,     //  -0.45235,   0.89184
    0xbfdd2016e8e9db5b,     0x3fec7e8e52233cf3,     //  -0.45508,   0.89045
    0xbfdd4cd02ba8609c,     0x3fec7315899eaad7,     //  -0.45781,   0.88905
    0xbfdd79775b86e389,     0x3fec678b3488739b,     //  -0.46054,   0.88764
    0xbfdda60c5cfa10d8,     0x3fec5bef59fef85a,     //  -0.46326,   0.88622
    0xbfddd28f1481cc58,     0x3fec5042012b6907,     //  -0.46598,    0.8848
    0xbfddfeff66a941de,     0x3fec44833141c004,     //  -0.46869,   0.88336
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   -0.4714,   0.88192
    0xbfde57a86d3cd824,     0x3fec2cd14931e3f1,     //   -0.4741,   0.88047
    0xbfde83e0eaf85113,     0x3fec20de3fa971b0,     //   -0.4768,   0.87901
    0xbfdeb00695f25620,     0x3fec14d9dc465e58,     //  -0.47949,   0.87755
    0xbfdedc1952ef78d5,     0x3fec08c426725549,     //  -0.48218,   0.87607
    0xbfdf081906bff7fd,     0x3febfc9d25a1b147,     //  -0.48487,   0.87459
    0xbfdf3405963fd068,     0x3febf064e15377dd,     //  -0.48755,   0.87309
    0xbfdf5fdee656cda3,     0x3febe41b611154c1,     //  -0.49023,    0.8716
    0xbfdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //   -0.4929,   0.87009
    0xbfdfb7575c24d2de,     0x3febcb54cb0d2327,     //  -0.49557,   0.86857
    0xbfdfe2f64be7120f,     0x3febbed7c49380ea,     //  -0.49823,   0.86705
    0xbfe00740c82b82e0,     0x3febb249a0b6c40d,     //  -0.50089,   0.86551
    0xbfe01cfc874c3eb7,     0x3feba5aa673590d2,     //  -0.50354,   0.86397
    0xbfe032ae55edbd95,     0x3feb98fa1fd9155e,     //  -0.50619,   0.86242
    0xbfe0485626ae221a,     0x3feb8c38d27504e9,     //  -0.50883,   0.86087
    0xbfe05df3ec31b8b6,     0x3feb7f6686e792ea,     //  -0.51147,    0.8593
    0xbfe073879922ffed,     0x3feb728345196e3e,     //   -0.5141,   0.85773
    0xbfe089112032b08c,     0x3feb658f14fdbc47,     //  -0.51673,   0.85615
    0xbfe09e907417c5e1,     0x3feb5889fe921405,     //  -0.51936,   0.85456
    0xbfe0b405878f85ec,     0x3feb4b7409de7925,     //  -0.52198,   0.85296
    0xbfe0c9704d5d898f,     0x3feb3e4d3ef55712,     //  -0.52459,   0.85136
    0xbfe0ded0b84bc4b5,     0x3feb3115a5f37bf4,     //   -0.5272,   0.84974
    0xbfe0f426bb2a8e7d,     0x3feb23cd470013b4,     //   -0.5298,   0.84812
    0xbfe1097248d0a956,     0x3feb16742a4ca2f5,     //   -0.5324,   0.84649
    0xbfe11eb3541b4b22,     0x3feb090a58150200,     //    -0.535,   0.84485
    0xbfe133e9cfee254e,     0x3feafb8fd89f57b6,     //  -0.53759,   0.84321
    0xbfe14915af336ceb,     0x3feaee04b43c1474,     //  -0.54017,   0.84155
    0xbfe15e36e4dbe2bc,     0x3feae068f345ecef,     //  -0.54275,   0.83989
    0xbfe1734d63dedb49,     0x3fead2bc9e21d511,     //  -0.54532,   0.83822
    0xbfe188591f3a46e5,     0x3feac4ffbd3efac8,     //  -0.54789,   0.83655
    0xbfe19d5a09f2b9b8,     0x3feab7325916c0d4,     //  -0.55046,   0.83486
    0xbfe1b250171373be,     0x3feaa9547a2cb98e,     //  -0.55302,   0.83317
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  -0.55557,   0.83147
    0xbfe1dc1b64dc4872,     0x3fea8d676e545ad2,     //  -0.55812,   0.82976
    0xbfe1f0f08bbc861b,     0x3fea7f58529fe69d,     //  -0.56066,   0.82805
    0xbfe205baa17560d6,     0x3fea7138de9d60f5,     //   -0.5632,   0.82632
    0xbfe21a799933eb58,     0x3fea63091b02fae2,     //  -0.56573,   0.82459
    0xbfe22f2d662c13e1,     0x3fea54c91090f524,     //  -0.56826,   0.82285
    0xbfe243d5fb98ac1f,     0x3fea4678c8119ac8,     //  -0.57078,    0.8211
    0xbfe258734cbb7110,     0x3fea38184a593bc6,     //   -0.5733,   0.81935
    0xbfe26d054cdd12df,     0x3fea29a7a0462782,     //  -0.57581,   0.81758
    0xbfe2818bef4d3cba,     0x3fea1b26d2c0a75e,     //  -0.57831,   0.81581
    0xbfe2960727629ca8,     0x3fea0c95eabaf937,     //  -0.58081,   0.81404
    0xbfe2aa76e87aeb58,     0x3fe9fdf4f13149de,     //  -0.58331,   0.81225
    0xbfe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //   -0.5858,   0.81046
    0xbfe2d333d34e9bb7,     0x3fe9e082edb42472,     //  -0.58828,   0.80866
    0xbfe2e780e3e8ea16,     0x3fe9d1b1f5ea80d6,     //  -0.59076,   0.80685
    0xbfe2fbc24b441015,     0x3fe9c2d110f075c3,     //  -0.59323,   0.80503
    0xbfe30ff7fce17035,     0x3fe9b3e047f38741,     //   -0.5957,   0.80321
    0xbfe32421ec49a620,     0x3fe9a4dfa42b06b2,     //  -0.59816,   0.80138
    0xbfe338400d0c8e57,     0x3fe995cf2ed80d22,     //  -0.60062,   0.79954
    0xbfe34c5252c14de1,     0x3fe986aef1457594,     //  -0.60307,   0.79769
    0xbfe36058b10659f3,     0x3fe9777ef4c7d742,     //  -0.60551,   0.79584
    0xbfe374531b817f8d,     0x3fe9683f42bd7fe1,     //  -0.60795,   0.79398
    0xbfe3884185dfeb22,     0x3fe958efe48e6dd7,     //  -0.61038,   0.79211
    0xbfe39c23e3d63029,     0x3fe94990e3ac4a6c,     //  -0.61281,   0.79023
    0xbfe3affa292050b9,     0x3fe93a22499263fc,     //  -0.61523,   0.78835
    0xbfe3c3c44981c517,     0x3fe92aa41fc5a815,     //  -0.61765,   0.78646
    0xbfe3d78238c58343,     0x3fe91b166fd49da2,     //  -0.62006,   0.78456
    0xbfe3eb33eabe0680,     0x3fe90b7943575efe,     //  -0.62246,   0.78265
    0xbfe3fed9534556d4,     0x3fe8fbcca3ef940d,     //  -0.62486,   0.78074
    0xbfe41272663d108c,     0x3fe8ec109b486c49,     //  -0.62725,   0.77882
    0xbfe425ff178e6bb1,     0x3fe8dc45331698cc,     //  -0.62964,   0.77689
    0xbfe4397f5b2a4380,     0x3fe8cc6a75184655,     //  -0.63202,   0.77495
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     //  -0.63439,   0.77301
    0xbfe4605a692b32a2,     0x3fe8ac871ede1d88,     //  -0.63676,   0.77106
    0xbfe473b51b987347,     0x3fe89c7e9a4dd4ab,     //  -0.63912,    0.7691
    0xbfe48703306091fe,     0x3fe88c66e7481ba1,     //  -0.64148,   0.76714
    0xbfe49a449b9b0938,     0x3fe87c400fba2ebf,     //  -0.64383,   0.76517
    0xbfe4ad79516722f0,     0x3fe86c0a1d9aa195,     //  -0.64618,   0.76319
    0xbfe4c0a145ec0004,     0x3fe85bc51ae958cc,     //  -0.64851,    0.7612
    0xbfe4d3bc6d589f80,     0x3fe84b7111af83f9,     //  -0.65085,   0.75921
    0xbfe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //  -0.65317,   0.75721
    0xbfe4f9cc25cca486,     0x3fe82a9c13f545ff,     //  -0.65549,    0.7552
    0xbfe50cc09f59a09b,     0x3fe81a1b33b57acc,     //  -0.65781,   0.75319
    0xbfe51fa81cd99aa6,     0x3fe8098b756e52fa,     //  -0.66011,   0.75117
    0xbfe5328292a35596,     0x3fe7f8ece3571771,     //  -0.66242,   0.74914
    0xbfe5454ff5159dfb,     0x3fe7e83f87b03686,     //  -0.66471,    0.7471
    0xbfe5581038975137,     0x3fe7d7836cc33db2,     //    -0.667,   0.74506
    0xbfe56ac35197649e,     0x3fe7c6b89ce2d333,     //  -0.66928,   0.74301
    0xbfe57d69348cec9f,     0x3fe7b5df226aafb0,     //  -0.67156,   0.74095
    0xbfe59001d5f723df,     0x3fe7a4f707bf97d2,     //  -0.67383,   0.73889
    0xbfe5a28d2a5d7250,     0x3fe79400574f55e4,     //  -0.67609,   0.73682
    0xbfe5b50b264f7448,     0x3fe782fb1b90b35b,     //  -0.67835,   0.73474
    0xbfe5c77bbe65018c,     0x3fe771e75f037261,     //   -0.6806,   0.73265
    0xbfe5d9dee73e345c,     0x3fe760c52c304764,     //  -0.68285,   0.73056
    0xbfe5ec3495837074,     0x3fe74f948da8d28d,     //  -0.68508,   0.72846
    0xbfe5fe7cbde56a0f,     0x3fe73e558e079942,     //  -0.68732,   0.72636
    0xbfe610b7551d2cde,     0x3fe72d0837efff97,     //  -0.68954,   0.72425
    0xbfe622e44fec22ff,     0x3fe71bac960e41bf,     //  -0.69176,   0.72213
    0xbfe63503a31c1be8,     0x3fe70a42b3176d7a,     //  -0.69397,      0.72
    0xbfe64715437f535b,     0x3fe6f8ca99c95b75,     //  -0.69618,   0.71787
    0xbfe6591925f0783e,     0x3fe6e74454eaa8ae,     //  -0.69838,   0.71573
    0xbfe66b0f3f52b386,     0x3fe6d5afef4aafcc,     //  -0.70057,   0.71358
    0xbfe67cf78491af10,     0x3fe6c40d73c18275,     //  -0.70275,   0.71143
    0xbfe68ed1eaa19c71,     0x3fe6b25ced2fe29c,     //  -0.70493,   0.70927
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  -0.70711,   0.70711
    0xbfe6b25ced2fe29c,     0x3fe68ed1eaa19c71,     //  -0.70927,   0.70493
    0xbfe6c40d73c18275,     0x3fe67cf78491af10,     //  -0.71143,   0.70275
    0xbfe6d5afef4aafcc,     0x3fe66b0f3f52b386,     //  -0.71358,   0.70057
    0xbfe6e74454eaa8ae,     0x3fe6591925f0783e,     //  -0.71573,   0.69838
    0xbfe6f8ca99c95b75,     0x3fe64715437f535b,     //  -0.71787,   0.69618
    0xbfe70a42b3176d7a,     0x3fe63503a31c1be8,     //     -0.72,   0.69397
    0xbfe71bac960e41bf,     0x3fe622e44fec22ff,     //  -0.72213,   0.69176
    0xbfe72d0837efff97,     0x3fe610b7551d2cde,     //  -0.72425,   0.68954
    0xbfe73e558e079942,     0x3fe5fe7cbde56a0f,     //  -0.72636,   0.68732
    0xbfe74f948da8d28d,     0x3fe5ec3495837074,     //  -0.72846,   0.68508
    0xbfe760c52c304764,     0x3fe5d9dee73e345c,     //  -0.73056,   0.68285
    0xbfe771e75f037261,     0x3fe5c77bbe65018c,     //  -0.73265,    0.6806
    0xbfe782fb1b90b35b,     0x3fe5b50b264f7448,     //  -0.73474,   0.67835
    0xbfe79400574f55e4,     0x3fe5a28d2a5d7250,     //  -0.73682,   0.67609
    0xbfe7a4f707bf97d2,     0x3fe59001d5f723df,     //  -0.73889,   0.67383
    0xbfe7b5df226aafb0,     0x3fe57d69348cec9f,     //  -0.74095,   0.67156
    0xbfe7c6b89ce2d333,     0x3fe56ac35197649e,     //  -0.74301,   0.66928
    0xbfe7d7836cc33db2,     0x3fe5581038975137,     //  -0.74506,     0.667
    0xbfe7e83f87b03686,     0x3fe5454ff5159dfb,     //   -0.7471,   0.66471
    0xbfe7f8ece3571771,     0x3fe5328292a35596,     //  -0.74914,   0.66242
    0xbfe8098b756e52fa,     0x3fe51fa81cd99aa6,     //  -0.75117,   0.66011
    0xbfe81a1b33b57acc,     0x3fe50cc09f59a09b,     //  -0.75319,   0.65781
    0xbfe82a9c13f545ff,     0x3fe4f9cc25cca486,     //   -0.7552,   0.65549
    0xbfe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //  -0.75721,   0.65317
    0xbfe84b7111af83f9,     0x3fe4d3bc6d589f80,     //  -0.75921,   0.65085
    0xbfe85bc51ae958cc,     0x3fe4c0a145ec0004,     //   -0.7612,   0.64851
    0xbfe86c0a1d9aa195,     0x3fe4ad79516722f0,     //  -0.76319,   0.64618
    0xbfe87c400fba2ebf,     0x3fe49a449b9b0938,     //  -0.76517,   0.64383
    0xbfe88c66e7481ba1,     0x3fe48703306091fe,     //  -0.76714,   0.64148
    0xbfe89c7e9a4dd4ab,     0x3fe473b51b987347,     //   -0.7691,   0.63912
    0xbfe8ac871ede1d88,     0x3fe4605a692b32a2,     //  -0.77106,   0.63676
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     //  -0.77301,   0.63439
    0xbfe8cc6a75184655,     0x3fe4397f5b2a4380,     //  -0.77495,   0.63202
    0xbfe8dc45331698cc,     0x3fe425ff178e6bb1,     //  -0.77689,   0.62964
    0xbfe8ec109b486c49,     0x3fe41272663d108c,     //  -0.77882,   0.62725
    0xbfe8fbcca3ef940d,     0x3fe3fed9534556d4,     //  -0.78074,   0.62486
    0xbfe90b7943575efe,     0x3fe3eb33eabe0680,     //  -0.78265,   0.62246
    0xbfe91b166fd49da2,     0x3fe3d78238c58343,     //  -0.78456,   0.62006
    0xbfe92aa41fc5a815,     0x3fe3c3c44981c517,     //  -0.78646,   0.61765
    0xbfe93a22499263fc,     0x3fe3affa292050b9,     //  -0.78835,   0.61523
    0xbfe94990e3ac4a6c,     0x3fe39c23e3d63029,     //  -0.79023,   0.61281
    0xbfe958efe48e6dd7,     0x3fe3884185dfeb22,     //  -0.79211,   0.61038
    0xbfe9683f42bd7fe1,     0x3fe374531b817f8d,     //  -0.79398,   0.60795
    0xbfe9777ef4c7d742,     0x3fe36058b10659f3,     //  -0.79584,   0.60551
    0xbfe986aef1457594,     0x3fe34c5252c14de1,     //  -0.79769,   0.60307
    0xbfe995cf2ed80d22,     0x3fe338400d0c8e57,     //  -0.79954,   0.60062
    0xbfe9a4dfa42b06b2,     0x3fe32421ec49a620,     //  -0.80138,   0.59816
    0xbfe9b3e047f38741,     0x3fe30ff7fce17035,     //  -0.80321,    0.5957
    0xbfe9c2d110f075c3,     0x3fe2fbc24b441015,     //  -0.80503,   0.59323
    0xbfe9d1b1f5ea80d6,     0x3fe2e780e3e8ea16,     //  -0.80685,   0.59076
    0xbfe9e082edb42472,     0x3fe2d333d34e9bb7,     //  -0.80866,   0.58828
    0xbfe9ef43ef29af94,     0x3fe2bedb25faf3ea,     //  -0.81046,    0.5858
    0xbfe9fdf4f13149de,     0x3fe2aa76e87aeb58,     //  -0.81225,   0.58331
    0xbfea0c95eabaf937,     0x3fe2960727629ca8,     //  -0.81404,   0.58081
    0xbfea1b26d2c0a75e,     0x3fe2818bef4d3cba,     //  -0.81581,   0.57831
    0xbfea29a7a0462782,     0x3fe26d054cdd12df,     //  -0.81758,   0.57581
    0xbfea38184a593bc6,     0x3fe258734cbb7110,     //  -0.81935,    0.5733
    0xbfea4678c8119ac8,     0x3fe243d5fb98ac1f,     //   -0.8211,   0.57078
    0xbfea54c91090f524,     0x3fe22f2d662c13e1,     //  -0.82285,   0.56826
    0xbfea63091b02fae2,     0x3fe21a799933eb58,     //  -0.82459,   0.56573
    0xbfea7138de9d60f5,     0x3fe205baa17560d6,     //  -0.82632,    0.5632
    0xbfea7f58529fe69d,     0x3fe1f0f08bbc861b,     //  -0.82805,   0.56066
    0xbfea8d676e545ad2,     0x3fe1dc1b64dc4872,     //  -0.82976,   0.55812
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  -0.83147,   0.55557
    0xbfeaa9547a2cb98e,     0x3fe1b250171373be,     //  -0.83317,   0.55302
    0xbfeab7325916c0d4,     0x3fe19d5a09f2b9b8,     //  -0.83486,   0.55046
    0xbfeac4ffbd3efac8,     0x3fe188591f3a46e5,     //  -0.83655,   0.54789
    0xbfead2bc9e21d511,     0x3fe1734d63dedb49,     //  -0.83822,   0.54532
    0xbfeae068f345ecef,     0x3fe15e36e4dbe2bc,     //  -0.83989,   0.54275
    0xbfeaee04b43c1474,     0x3fe14915af336ceb,     //  -0.84155,   0.54017
    0xbfeafb8fd89f57b6,     0x3fe133e9cfee254e,     //  -0.84321,   0.53759
    0xbfeb090a58150200,     0x3fe11eb3541b4b22,     //  -0.84485,     0.535
    0xbfeb16742a4ca2f5,     0x3fe1097248d0a956,     //  -0.84649,    0.5324
    0xbfeb23cd470013b4,     0x3fe0f426bb2a8e7d,     //  -0.84812,    0.5298
    0xbfeb3115a5f37bf4,     0x3fe0ded0b84bc4b5,     //  -0.84974,    0.5272
    0xbfeb3e4d3ef55712,     0x3fe0c9704d5d898f,     //  -0.85136,   0.52459
    0xbfeb4b7409de7925,     0x3fe0b405878f85ec,     //  -0.85296,   0.52198
    0xbfeb5889fe921405,     0x3fe09e907417c5e1,     //  -0.85456,   0.51936
    0xbfeb658f14fdbc47,     0x3fe089112032b08c,     //  -0.85615,   0.51673
    0xbfeb728345196e3e,     0x3fe073879922ffed,     //  -0.85773,    0.5141
    0xbfeb7f6686e792ea,     0x3fe05df3ec31b8b6,     //   -0.8593,   0.51147
    0xbfeb8c38d27504e9,     0x3fe0485626ae221a,     //  -0.86087,   0.50883
    0xbfeb98fa1fd9155e,     0x3fe032ae55edbd95,     //  -0.86242,   0.50619
    0xbfeba5aa673590d2,     0x3fe01cfc874c3eb7,     //  -0.86397,   0.50354
    0xbfebb249a0b6c40d,     0x3fe00740c82b82e0,     //  -0.86551,   0.50089
    0xbfebbed7c49380ea,     0x3fdfe2f64be7120f,     //  -0.86705,   0.49823
    0xbfebcb54cb0d2327,     0x3fdfb7575c24d2de,     //  -0.86857,   0.49557
    0xbfebd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //  -0.87009,    0.4929
    0xbfebe41b611154c1,     0x3fdf5fdee656cda3,     //   -0.8716,   0.49023
    0xbfebf064e15377dd,     0x3fdf3405963fd068,     //  -0.87309,   0.48755
    0xbfebfc9d25a1b147,     0x3fdf081906bff7fd,     //  -0.87459,   0.48487
    0xbfec08c426725549,     0x3fdedc1952ef78d5,     //  -0.87607,   0.48218
    0xbfec14d9dc465e58,     0x3fdeb00695f25620,     //  -0.87755,   0.47949
    0xbfec20de3fa971b0,     0x3fde83e0eaf85113,     //  -0.87901,    0.4768
    0xbfec2cd14931e3f1,     0x3fde57a86d3cd824,     //  -0.88047,    0.4741
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  -0.88192,    0.4714
    0xbfec44833141c004,     0x3fddfeff66a941de,     //  -0.88336,   0.46869
    0xbfec5042012b6907,     0x3fddd28f1481cc58,     //   -0.8848,   0.46598
    0xbfec5bef59fef85a,     0x3fdda60c5cfa10d8,     //  -0.88622,   0.46326
    0xbfec678b3488739b,     0x3fdd79775b86e389,     //  -0.88764,   0.46054
    0xbfec7315899eaad7,     0x3fdd4cd02ba8609c,     //  -0.88905,   0.45781
    0xbfec7e8e52233cf3,     0x3fdd2016e8e9db5b,     //  -0.89045,   0.45508
    0xbfec89f587029c13,     0x3fdcf34baee1cd21,     //  -0.89184,   0.45235
    0xbfec954b213411f5,     0x3fdcc66e9931c45d,     //  -0.89322,   0.44961
    0xbfeca08f19b9c449,     0x3fdc997fc3865388,     //   -0.8946,   0.44687
    0xbfecabc169a0b901,     0x3fdc6c7f4997000a,     //  -0.89597,   0.44412
    0xbfecb6e20a00da99,     0x3fdc3f6d47263129,     //  -0.89732,   0.44137
    0xbfecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     //  -0.89867,   0.43862
    0xbfecccee20c2de9f,     0x3fdbe51517ffc0d9,     //  -0.90002,   0.43586
    0xbfecd7d9898b32f6,     0x3fdbb7cf2304bd01,     //  -0.90135,   0.43309
    0xbfece2b32799a060,     0x3fdb8a7814fd5693,     //  -0.90267,   0.43033
    0xbfeced7af43cc773,     0x3fdb5d1009e15cc0,     //  -0.90399,   0.42756
    0xbfecf830e8ce467b,     0x3fdb2f971db31972,     //   -0.9053,   0.42478
    0xbfed02d4feb2bd92,     0x3fdb020d6c7f4009,     //   -0.9066,     0.422
    0xbfed0d672f59d2b9,     0x3fdad473125cdc08,     //  -0.90789,   0.41922
    0xbfed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     //  -0.90917,   0.41643
    0xbfed2255c6e5a4e1,     0x3fda790cd3dbf31a,     //  -0.91044,   0.41364
    0xbfed2cb220e0ef9f,     0x3fda4b4127dea1e4,     //  -0.91171,   0.41084
    0xbfed36fc7bcbfbdc,     0x3fda1d6543b50ac0,     //  -0.91296,   0.40804
    0xbfed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //  -0.91421,   0.40524
    0xbfed4b5b1b187524,     0x3fd9c17d440df9f2,     //  -0.91545,   0.40243
    0xbfed556f52e93eb1,     0x3fd993716141bdfe,     //  -0.91668,   0.39962
    0xbfed5f7172888a7f,     0x3fd96555b7ab948f,     //   -0.9179,   0.39681
    0xbfed696173c9e68b,     0x3fd9372a63bc93d7,     //  -0.91911,   0.39399
    0xbfed733f508c0dff,     0x3fd908ef81ef7bd1,     //  -0.92032,   0.39117
    0xbfed7d0b02b8ecf9,     0x3fd8daa52ec8a4af,     //  -0.92151,   0.38835
    0xbfed86c48445a450,     0x3fd8ac4b86d5ed44,     //   -0.9227,   0.38552
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     //  -0.92388,   0.38268
    0xbfed9a00dd8b3d46,     0x3fd84f6aaaf3903f,     //  -0.92505,   0.37985
    0xbfeda383a9668988,     0x3fd820e3b04eaac4,     //  -0.92621,   0.37701
    0xbfedacf42ce68ab9,     0x3fd7f24dd37341e3,     //  -0.92736,   0.37416
    0xbfedb6526238a09b,     0x3fd7c3a9311dcce7,     //  -0.92851,   0.37132
    0xbfedbf9e4395759a,     0x3fd794f5e613dfae,     //  -0.92964,   0.36847
    0xbfedc8d7cb410260,     0x3fd766340f2418f6,     //  -0.93077,   0.36561
    0xbfedd1fef38a915a,     0x3fd73763c9261092,     //  -0.93188,   0.36276
    0xbfeddb13b6ccc23d,     0x3fd7088530fa459e,     //  -0.93299,    0.3599
    0xbfede4160f6d8d81,     0x3fd6d998638a0cb5,     //  -0.93409,   0.35703
    0xbfeded05f7de47da,     0x3fd6aa9d7dc77e16,     //  -0.93518,   0.35416
    0xbfedf5e36a9ba59c,     0x3fd67b949cad63ca,     //  -0.93627,   0.35129
    0xbfedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     //  -0.93734,   0.34842
    0xbfee0766d9280f54,     0x3fd61d595c88c203,     //   -0.9384,   0.34554
    0xbfee100cca2980ac,     0x3fd5ee27379ea693,     //  -0.93946,   0.34266
    0xbfee18a02fdc66d9,     0x3fd5bee78b9db3b6,     //  -0.94051,   0.33978
    0xbfee212104f686e5,     0x3fd58f9a75ab1fdd,     //  -0.94154,   0.33689
    0xbfee298f4439197a,     0x3fd5604012f467b4,     //  -0.94257,     0.334
    0xbfee31eae870ce25,     0x3fd530d880af3c24,     //  -0.94359,   0.33111
    0xbfee3a33ec75ce85,     0x3fd50163dc197047,     //   -0.9446,   0.32821
    0xbfee426a4b2bc17e,     0x3fd4d1e24278e76a,     //  -0.94561,   0.32531
    0xbfee4a8dff81ce5e,     0x3fd4a253d11b82f3,     //   -0.9466,   0.32241
    0xbfee529f04729ffc,     0x3fd472b8a5571054,     //  -0.94759,    0.3195
    0xbfee5a9d550467d3,     0x3fd44310dc8936f0,     //  -0.94856,   0.31659
    0xbfee6288ec48e112,     0x3fd4135c94176602,     //  -0.94953,   0.31368
    0xbfee6a61c55d53a7,     0x3fd3e39be96ec271,     //  -0.95049,   0.31077
    0xbfee7227db6a9744,     0x3fd3b3cefa0414b7,     //  -0.95144,   0.30785
    0xbfee79db29a5165a,     0x3fd383f5e353b6aa,     //  -0.95238,   0.30493
    0xbfee817bab4cd10d,     0x3fd35410c2e18152,     //  -0.95331,   0.30201
    0xbfee89095bad6025,     0x3fd3241fb638baaf,     //  -0.95423,   0.29908
    0xbfee9084361df7f3,     0x3fd2f422daec0386,     //  -0.95514,   0.29615
    0xbfee97ec36016b30,     0x3fd2c41a4e954520,     //  -0.95605,   0.29322
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     //  -0.95694,   0.29028
    0xbfeea68393e65800,     0x3fd263e6995554ba,     //  -0.95783,   0.28735
    0xbfeeadb2e8e7a88e,     0x3fd233bbabc3bb72,     //   -0.9587,   0.28441
    0xbfeeb4cf515b8811,     0x3fd2038583d727bd,     //  -0.95957,   0.28146
    0xbfeebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     //  -0.96043,   0.27852
    0xbfeec2cf4b1af6b2,     0x3fd1a2f7fbe8f243,     //  -0.96128,   0.27557
    0xbfeec9b2d3c3bf84,     0x3fd172a0d7765177,     //  -0.96212,   0.27262
    0xbfeed0835e999009,     0x3fd1423eefc69378,     //  -0.96295,   0.26967
    0xbfeed740e7684963,     0x3fd111d262b1f677,     //  -0.96378,   0.26671
    0xbfeeddeb6a078651,     0x3fd0e15b4e1749cd,     //  -0.96459,   0.26375
    0xbfeee482e25a9dbc,     0x3fd0b0d9cfdbdb90,     //  -0.96539,   0.26079
    0xbfeeeb074c50a544,     0x3fd0804e05eb661e,     //  -0.96619,   0.25783
    0xbfeef178a3e473c2,     0x3fd04fb80e37fdae,     //  -0.96698,   0.25487
    0xbfeef7d6e51ca3c0,     0x3fd01f1806b9fdd2,     //  -0.96775,    0.2519
    0xbfeefe220c0b95ec,     0x3fcfdcdc1adfedf8,     //  -0.96852,   0.24893
    0xbfef045a14cf738c,     0x3fcf7b7480bd3801,     //  -0.96928,   0.24596
    0xbfef0a7efb9230d7,     0x3fcf19f97b215f1a,     //  -0.97003,   0.24298
    0xbfef1090bc898f5f,     0x3fceb86b462de348,     //  -0.97077,      0.24
    0xbfef168f53f7205d,     0x3fce56ca1e101a1b,     //   -0.9715,   0.23702
    0xbfef1c7abe284708,     0x3fcdf5163f01099a,     //  -0.97223,   0.23404
    0xbfef2252f7763ada,     0x3fcd934fe5454311,     //  -0.97294,   0.23106
    0xbfef2817fc4609ce,     0x3fcd31774d2cbdee,     //  -0.97364,   0.22807
    0xbfef2dc9c9089a9d,     0x3fcccf8cb312b286,     //  -0.97434,   0.22508
    0xbfef33685a3aaef0,     0x3fcc6d90535d74dc,     //  -0.97503,   0.22209
    0xbfef38f3ac64e589,     0x3fcc0b826a7e4f63,     //   -0.9757,    0.2191
    0xbfef3e6bbc1bbc65,     0x3fcba96334f15dad,     //  -0.97637,   0.21611
    0xbfef43d085ff92dd,     0x3fcb4732ef3d6722,     //  -0.97703,   0.21311
    0xbfef492206bcabb4,     0x3fcae4f1d5f3b9ab,     //  -0.97768,   0.21011
    0xbfef4e603b0b2f2d,     0x3fca82a025b00451,     //  -0.97832,   0.20711
    0xbfef538b1faf2d07,     0x3fca203e1b1831da,     //  -0.97895,   0.20411
    0xbfef58a2b1789e84,     0x3fc9bdcbf2dc4366,     //  -0.97957,    0.2011
    0xbfef5da6ed43685d,     0x3fc95b49e9b62af9,     //  -0.98018,    0.1981
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  -0.98079,   0.19509
    0xbfef677556883cee,     0x3fc8961727c41804,     //  -0.98138,   0.19208
    0xbfef6c3f7df5bbb7,     0x3fc83366e89c64c5,     //  -0.98196,   0.18907
    0xbfef70f6434b7eb7,     0x3fc7d0a7bbd2cb1b,     //  -0.98254,   0.18606
    0xbfef7599a3a12077,     0x3fc76dd9de50bf31,     //  -0.98311,   0.18304
    0xbfef7a299c1a322a,     0x3fc70afd8d08c4ff,     //  -0.98366,   0.18002
    0xbfef7ea629e63d6e,     0x3fc6a81304f64ab2,     //  -0.98421,     0.177
    0xbfef830f4a40c60c,     0x3fc6451a831d830d,     //  -0.98475,   0.17398
    0xbfef8764fa714ba9,     0x3fc5e214448b3fc6,     //  -0.98528,   0.17096
    0xbfef8ba737cb4b78,     0x3fc57f008654cbde,     //   -0.9858,   0.16794
    0xbfef8fd5ffae41db,     0x3fc51bdf8597c5f2,     //  -0.98631,   0.16491
    0xbfef93f14f85ac08,     0x3fc4b8b17f79fa88,     //  -0.98681,   0.16189
    0xbfef97f924c9099b,     0x3fc45576b1293e5a,     //   -0.9873,   0.15886
    0xbfef9bed7cfbde29,     0x3fc3f22f57db4893,     //  -0.98778,   0.15583
    0xbfef9fce55adb2c8,     0x3fc38edbb0cd8d14,     //  -0.98826,    0.1528
    0xbfefa39bac7a1791,     0x3fc32b7bf94516a7,     //  -0.98872,   0.14976
    0xbfefa7557f08a517,     0x3fc2c8106e8e613a,     //  -0.98918,   0.14673
    0xbfefaafbcb0cfddc,     0x3fc264994dfd340a,     //  -0.98962,    0.1437
    0xbfefae8e8e46cfbb,     0x3fc20116d4ec7bce,     //  -0.99006,   0.14066
    0xbfefb20dc681d54d,     0x3fc19d8940be24e7,     //  -0.99049,   0.13762
    0xbfefb5797195d741,     0x3fc139f0cedaf576,     //   -0.9909,   0.13458
    0xbfefb8d18d66adb7,     0x3fc0d64dbcb26786,     //  -0.99131,   0.13154
    0xbfefbc1617e44186,     0x3fc072a047ba831d,     //  -0.99171,    0.1285
    0xbfefbf470f0a8d88,     0x3fc00ee8ad6fb85b,     //   -0.9921,   0.12545
    0xbfefc26470e19fd3,     0x3fbf564e56a9730e,     //  -0.99248,   0.12241
    0xbfefc56e3b7d9af6,     0x3fbe8eb7fde4aa3e,     //  -0.99285,   0.11937
    0xbfefc8646cfeb721,     0x3fbdc70ecbae9fc8,     //  -0.99321,   0.11632
    0xbfefcb4703914354,     0x3fbcff533b307dc1,     //  -0.99356,   0.11327
    0xbfefce15fd6da67b,     0x3fbc3785c79ec2d5,     //  -0.99391,   0.11022
    0xbfefd0d158d86087,     0x3fbb6fa6ec38f64c,     //  -0.99424,   0.10717
    0xbfefd37914220b84,     0x3fbaa7b724495c04,     //  -0.99456,   0.10412
    0xbfefd60d2da75c9e,     0x3fb9dfb6eb24a85c,     //  -0.99488,   0.10107
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     //  -0.99518,  0.098017
    0xbfefdafa7514538c,     0x3fb84f8712c130a0,     //  -0.99548,  0.094963
    0xbfefdd539ff1f456,     0x3fb787586a5d5b21,     //  -0.99577,  0.091909
    0xbfefdf9922f73307,     0x3fb6bf1b3e79b129,     //  -0.99604,  0.088854
    0xbfefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     //  -0.99631,  0.085797
    0xbfefe3e92be9d886,     0x3fb52e774a4d4d0a,     //  -0.99657,   0.08274
    0xbfefe5f3af2e3940,     0x3fb4661179272096,     //  -0.99682,  0.079682
    0xbfefe7ea85482d60,     0x3fb39d9f12c5a299,     //  -0.99706,  0.076624
    0xbfefe9cdad01883a,     0x3fb2d52092ce19f6,     //  -0.99729,  0.073565
    0xbfefeb9d2530410f,     0x3fb20c9674ed444c,     //  -0.99751,  0.070505
    0xbfefed58ecb673c4,     0x3fb1440134d709b2,     //  -0.99772,  0.067444
    0xbfefef0102826191,     0x3fb07b614e463064,     //  -0.99793,  0.064383
    0xbfeff095658e71ad,     0x3faf656e79f820e0,     //  -0.99812,  0.061321
    0xbfeff21614e131ed,     0x3fadd406f9808ec8,     //   -0.9983,  0.058258
    0xbfeff3830f8d575c,     0x3fac428d12c0d7e3,     //  -0.99848,  0.055195
    0xbfeff4dc54b1bed3,     0x3faab101bd5f8317,     //  -0.99864,  0.052132
    0xbfeff621e3796d7e,     0x3fa91f65f10dd814,     //   -0.9988,  0.049068
    0xbfeff753bb1b9164,     0x3fa78dbaa5874685,     //  -0.99894,  0.046003
    0xbfeff871dadb81df,     0x3fa5fc00d290cd43,     //  -0.99908,  0.042938
    0xbfeff97c4208c014,     0x3fa46a396ff86179,     //   -0.9992,  0.039873
    0xbfeffa72effef75d,     0x3fa2d865759455cd,     //  -0.99932,  0.036807
    0xbfeffb55e425fdae,     0x3fa14685db42c17e,     //  -0.99943,  0.033741
    0xbfeffc251df1d3f8,     0x3f9f693731d1cf01,     //  -0.99953,  0.030675
    0xbfeffce09ce2a679,     0x3f9c454f4ce53b1c,     //  -0.99962,  0.027608
    0xbfeffd886084cd0d,     0x3f992155f7a3667e,     //   -0.9997,  0.024541
    0xbfeffe1c6870cb77,     0x3f95fd4d21fab226,     //  -0.99977,  0.021474
    0xbfeffe9cb44b51a1,     0x3f92d936bbe30efd,     //  -0.99983,  0.018407
    0xbfefff0943c53bd1,     0x3f8f6a296ab997ca,     //  -0.99988,  0.015339
    0xbfefff62169b92db,     0x3f8921d1fcdec784,     //  -0.99992,  0.012272
    0xbfefffa72c978c4f,     0x3f82d96b0e509703,     //  -0.99996, 0.0092038
    0xbfefffd8858e8a92,     0x3f7921f0fe670071,     //  -0.99998, 0.0061359
    0xbfeffff621621d02,     0x3f6921f8becca4ba,     //        -1,  0.003068
    0xbff0000000000000,     0x0000000000000000,     //        -1,         0
    0xbfeffff621621d02,     0xbf6921f8becca4ba,     //        -1, -0.003068
    0xbfefffd8858e8a92,     0xbf7921f0fe670071,     //  -0.99998,-0.0061359
    0xbfefffa72c978c4f,     0xbf82d96b0e509703,     //  -0.99996,-0.0092038
    0xbfefff62169b92db,     0xbf8921d1fcdec784,     //  -0.99992, -0.012272
    0xbfefff0943c53bd1,     0xbf8f6a296ab997ca,     //  -0.99988, -0.015339
    0xbfeffe9cb44b51a1,     0xbf92d936bbe30efd,     //  -0.99983, -0.018407
    0xbfeffe1c6870cb77,     0xbf95fd4d21fab226,     //  -0.99977, -0.021474
    0xbfeffd886084cd0d,     0xbf992155f7a3667e,     //   -0.9997, -0.024541
    0xbfeffce09ce2a679,     0xbf9c454f4ce53b1c,     //  -0.99962, -0.027608
    0xbfeffc251df1d3f8,     0xbf9f693731d1cf01,     //  -0.99953, -0.030675
    0xbfeffb55e425fdae,     0xbfa14685db42c17e,     //  -0.99943, -0.033741
    0xbfeffa72effef75d,     0xbfa2d865759455cd,     //  -0.99932, -0.036807
    0xbfeff97c4208c014,     0xbfa46a396ff86179,     //   -0.9992, -0.039873
    0xbfeff871dadb81df,     0xbfa5fc00d290cd43,     //  -0.99908, -0.042938
    0xbfeff753bb1b9164,     0xbfa78dbaa5874685,     //  -0.99894, -0.046003
    0xbfeff621e3796d7e,     0xbfa91f65f10dd814,     //   -0.9988, -0.049068
    0xbfeff4dc54b1bed3,     0xbfaab101bd5f8317,     //  -0.99864, -0.052132
    0xbfeff3830f8d575c,     0xbfac428d12c0d7e3,     //  -0.99848, -0.055195
    0xbfeff21614e131ed,     0xbfadd406f9808ec8,     //   -0.9983, -0.058258
    0xbfeff095658e71ad,     0xbfaf656e79f820e0,     //  -0.99812, -0.061321
    0xbfefef0102826191,     0xbfb07b614e463064,     //  -0.99793, -0.064383
    0xbfefed58ecb673c4,     0xbfb1440134d709b2,     //  -0.99772, -0.067444
    0xbfefeb9d2530410f,     0xbfb20c9674ed444c,     //  -0.99751, -0.070505
    0xbfefe9cdad01883a,     0xbfb2d52092ce19f6,     //  -0.99729, -0.073565
    0xbfefe7ea85482d60,     0xbfb39d9f12c5a299,     //  -0.99706, -0.076624
    0xbfefe5f3af2e3940,     0xbfb4661179272096,     //  -0.99682, -0.079682
    0xbfefe3e92be9d886,     0xbfb52e774a4d4d0a,     //  -0.99657,  -0.08274
    0xbfefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     //  -0.99631, -0.085797
    0xbfefdf9922f73307,     0xbfb6bf1b3e79b129,     //  -0.99604, -0.088854
    0xbfefdd539ff1f456,     0xbfb787586a5d5b21,     //  -0.99577, -0.091909
    0xbfefdafa7514538c,     0xbfb84f8712c130a0,     //  -0.99548, -0.094963
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     //  -0.99518, -0.098017
    0xbfefd60d2da75c9e,     0xbfb9dfb6eb24a85c,     //  -0.99488,  -0.10107
    0xbfefd37914220b84,     0xbfbaa7b724495c04,     //  -0.99456,  -0.10412
    0xbfefd0d158d86087,     0xbfbb6fa6ec38f64c,     //  -0.99424,  -0.10717
    0xbfefce15fd6da67b,     0xbfbc3785c79ec2d5,     //  -0.99391,  -0.11022
    0xbfefcb4703914354,     0xbfbcff533b307dc1,     //  -0.99356,  -0.11327
    0xbfefc8646cfeb721,     0xbfbdc70ecbae9fc8,     //  -0.99321,  -0.11632
    0xbfefc56e3b7d9af6,     0xbfbe8eb7fde4aa3e,     //  -0.99285,  -0.11937
    0xbfefc26470e19fd3,     0xbfbf564e56a9730e,     //  -0.99248,  -0.12241
    0xbfefbf470f0a8d88,     0xbfc00ee8ad6fb85b,     //   -0.9921,  -0.12545
    0xbfefbc1617e44186,     0xbfc072a047ba831d,     //  -0.99171,   -0.1285
    0xbfefb8d18d66adb7,     0xbfc0d64dbcb26786,     //  -0.99131,  -0.13154
    0xbfefb5797195d741,     0xbfc139f0cedaf576,     //   -0.9909,  -0.13458
    0xbfefb20dc681d54d,     0xbfc19d8940be24e7,     //  -0.99049,  -0.13762
    0xbfefae8e8e46cfbb,     0xbfc20116d4ec7bce,     //  -0.99006,  -0.14066
    0xbfefaafbcb0cfddc,     0xbfc264994dfd340a,     //  -0.98962,   -0.1437
    0xbfefa7557f08a517,     0xbfc2c8106e8e613a,     //  -0.98918,  -0.14673
    0xbfefa39bac7a1791,     0xbfc32b7bf94516a7,     //  -0.98872,  -0.14976
    0xbfef9fce55adb2c8,     0xbfc38edbb0cd8d14,     //  -0.98826,   -0.1528
    0xbfef9bed7cfbde29,     0xbfc3f22f57db4893,     //  -0.98778,  -0.15583
    0xbfef97f924c9099b,     0xbfc45576b1293e5a,     //   -0.9873,  -0.15886
    0xbfef93f14f85ac08,     0xbfc4b8b17f79fa88,     //  -0.98681,  -0.16189
    0xbfef8fd5ffae41db,     0xbfc51bdf8597c5f2,     //  -0.98631,  -0.16491
    0xbfef8ba737cb4b78,     0xbfc57f008654cbde,     //   -0.9858,  -0.16794
    0xbfef8764fa714ba9,     0xbfc5e214448b3fc6,     //  -0.98528,  -0.17096
    0xbfef830f4a40c60c,     0xbfc6451a831d830d,     //  -0.98475,  -0.17398
    0xbfef7ea629e63d6e,     0xbfc6a81304f64ab2,     //  -0.98421,    -0.177
    0xbfef7a299c1a322a,     0xbfc70afd8d08c4ff,     //  -0.98366,  -0.18002
    0xbfef7599a3a12077,     0xbfc76dd9de50bf31,     //  -0.98311,  -0.18304
    0xbfef70f6434b7eb7,     0xbfc7d0a7bbd2cb1b,     //  -0.98254,  -0.18606
    0xbfef6c3f7df5bbb7,     0xbfc83366e89c64c5,     //  -0.98196,  -0.18907
    0xbfef677556883cee,     0xbfc8961727c41804,     //  -0.98138,  -0.19208
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  -0.98079,  -0.19509
    0xbfef5da6ed43685d,     0xbfc95b49e9b62af9,     //  -0.98018,   -0.1981
    0xbfef58a2b1789e84,     0xbfc9bdcbf2dc4366,     //  -0.97957,   -0.2011
    0xbfef538b1faf2d07,     0xbfca203e1b1831da,     //  -0.97895,  -0.20411
    0xbfef4e603b0b2f2d,     0xbfca82a025b00451,     //  -0.97832,  -0.20711
    0xbfef492206bcabb4,     0xbfcae4f1d5f3b9ab,     //  -0.97768,  -0.21011
    0xbfef43d085ff92dd,     0xbfcb4732ef3d6722,     //  -0.97703,  -0.21311
    0xbfef3e6bbc1bbc65,     0xbfcba96334f15dad,     //  -0.97637,  -0.21611
    0xbfef38f3ac64e589,     0xbfcc0b826a7e4f63,     //   -0.9757,   -0.2191
    0xbfef33685a3aaef0,     0xbfcc6d90535d74dc,     //  -0.97503,  -0.22209
    0xbfef2dc9c9089a9d,     0xbfcccf8cb312b286,     //  -0.97434,  -0.22508
    0xbfef2817fc4609ce,     0xbfcd31774d2cbdee,     //  -0.97364,  -0.22807
    0xbfef2252f7763ada,     0xbfcd934fe5454311,     //  -0.97294,  -0.23106
    0xbfef1c7abe284708,     0xbfcdf5163f01099a,     //  -0.97223,  -0.23404
    0xbfef168f53f7205d,     0xbfce56ca1e101a1b,     //   -0.9715,  -0.23702
    0xbfef1090bc898f5f,     0xbfceb86b462de348,     //  -0.97077,     -0.24
    0xbfef0a7efb9230d7,     0xbfcf19f97b215f1a,     //  -0.97003,  -0.24298
    0xbfef045a14cf738c,     0xbfcf7b7480bd3801,     //  -0.96928,  -0.24596
    0xbfeefe220c0b95ec,     0xbfcfdcdc1adfedf8,     //  -0.96852,  -0.24893
    0xbfeef7d6e51ca3c0,     0xbfd01f1806b9fdd2,     //  -0.96775,   -0.2519
    0xbfeef178a3e473c2,     0xbfd04fb80e37fdae,     //  -0.96698,  -0.25487
    0xbfeeeb074c50a544,     0xbfd0804e05eb661e,     //  -0.96619,  -0.25783
    0xbfeee482e25a9dbc,     0xbfd0b0d9cfdbdb90,     //  -0.96539,  -0.26079
    0xbfeeddeb6a078651,     0xbfd0e15b4e1749cd,     //  -0.96459,  -0.26375
    0xbfeed740e7684963,     0xbfd111d262b1f677,     //  -0.96378,  -0.26671
    0xbfeed0835e999009,     0xbfd1423eefc69378,     //  -0.96295,  -0.26967
    0xbfeec9b2d3c3bf84,     0xbfd172a0d7765177,     //  -0.96212,  -0.27262
    0xbfeec2cf4b1af6b2,     0xbfd1a2f7fbe8f243,     //  -0.96128,  -0.27557
    0xbfeebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     //  -0.96043,  -0.27852
    0xbfeeb4cf515b8811,     0xbfd2038583d727bd,     //  -0.95957,  -0.28146
    0xbfeeadb2e8e7a88e,     0xbfd233bbabc3bb72,     //   -0.9587,  -0.28441
    0xbfeea68393e65800,     0xbfd263e6995554ba,     //  -0.95783,  -0.28735
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     //  -0.95694,  -0.29028
    0xbfee97ec36016b30,     0xbfd2c41a4e954520,     //  -0.95605,  -0.29322
    0xbfee9084361df7f3,     0xbfd2f422daec0386,     //  -0.95514,  -0.29615
    0xbfee89095bad6025,     0xbfd3241fb638baaf,     //  -0.95423,  -0.29908
    0xbfee817bab4cd10d,     0xbfd35410c2e18152,     //  -0.95331,  -0.30201
    0xbfee79db29a5165a,     0xbfd383f5e353b6aa,     //  -0.95238,  -0.30493
    0xbfee7227db6a9744,     0xbfd3b3cefa0414b7,     //  -0.95144,  -0.30785
    0xbfee6a61c55d53a7,     0xbfd3e39be96ec271,     //  -0.95049,  -0.31077
    0xbfee6288ec48e112,     0xbfd4135c94176602,     //  -0.94953,  -0.31368
    0xbfee5a9d550467d3,     0xbfd44310dc8936f0,     //  -0.94856,  -0.31659
    0xbfee529f04729ffc,     0xbfd472b8a5571054,     //  -0.94759,   -0.3195
    0xbfee4a8dff81ce5e,     0xbfd4a253d11b82f3,     //   -0.9466,  -0.32241
    0xbfee426a4b2bc17e,     0xbfd4d1e24278e76a,     //  -0.94561,  -0.32531
    0xbfee3a33ec75ce85,     0xbfd50163dc197047,     //   -0.9446,  -0.32821
    0xbfee31eae870ce25,     0xbfd530d880af3c24,     //  -0.94359,  -0.33111
    0xbfee298f4439197a,     0xbfd5604012f467b4,     //  -0.94257,    -0.334
    0xbfee212104f686e5,     0xbfd58f9a75ab1fdd,     //  -0.94154,  -0.33689
    0xbfee18a02fdc66d9,     0xbfd5bee78b9db3b6,     //  -0.94051,  -0.33978
    0xbfee100cca2980ac,     0xbfd5ee27379ea693,     //  -0.93946,  -0.34266
    0xbfee0766d9280f54,     0xbfd61d595c88c203,     //   -0.9384,  -0.34554
    0xbfedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     //  -0.93734,  -0.34842
    0xbfedf5e36a9ba59c,     0xbfd67b949cad63ca,     //  -0.93627,  -0.35129
    0xbfeded05f7de47da,     0xbfd6aa9d7dc77e16,     //  -0.93518,  -0.35416
    0xbfede4160f6d8d81,     0xbfd6d998638a0cb5,     //  -0.93409,  -0.35703
    0xbfeddb13b6ccc23d,     0xbfd7088530fa459e,     //  -0.93299,   -0.3599
    0xbfedd1fef38a915a,     0xbfd73763c9261092,     //  -0.93188,  -0.36276
    0xbfedc8d7cb410260,     0xbfd766340f2418f6,     //  -0.93077,  -0.36561
    0xbfedbf9e4395759a,     0xbfd794f5e613dfae,     //  -0.92964,  -0.36847
    0xbfedb6526238a09b,     0xbfd7c3a9311dcce7,     //  -0.92851,  -0.37132
    0xbfedacf42ce68ab9,     0xbfd7f24dd37341e3,     //  -0.92736,  -0.37416
    0xbfeda383a9668988,     0xbfd820e3b04eaac4,     //  -0.92621,  -0.37701
    0xbfed9a00dd8b3d46,     0xbfd84f6aaaf3903f,     //  -0.92505,  -0.37985
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     //  -0.92388,  -0.38268
    0xbfed86c48445a450,     0xbfd8ac4b86d5ed44,     //   -0.9227,  -0.38552
    0xbfed7d0b02b8ecf9,     0xbfd8daa52ec8a4af,     //  -0.92151,  -0.38835
    0xbfed733f508c0dff,     0xbfd908ef81ef7bd1,     //  -0.92032,  -0.39117
    0xbfed696173c9e68b,     0xbfd9372a63bc93d7,     //  -0.91911,  -0.39399
    0xbfed5f7172888a7f,     0xbfd96555b7ab948f,     //   -0.9179,  -0.39681
    0xbfed556f52e93eb1,     0xbfd993716141bdfe,     //  -0.91668,  -0.39962
    0xbfed4b5b1b187524,     0xbfd9c17d440df9f2,     //  -0.91545,  -0.40243
    0xbfed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //  -0.91421,  -0.40524
    0xbfed36fc7bcbfbdc,     0xbfda1d6543b50ac0,     //  -0.91296,  -0.40804
    0xbfed2cb220e0ef9f,     0xbfda4b4127dea1e4,     //  -0.91171,  -0.41084
    0xbfed2255c6e5a4e1,     0xbfda790cd3dbf31a,     //  -0.91044,  -0.41364
    0xbfed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     //  -0.90917,  -0.41643
    0xbfed0d672f59d2b9,     0xbfdad473125cdc08,     //  -0.90789,  -0.41922
    0xbfed02d4feb2bd92,     0xbfdb020d6c7f4009,     //   -0.9066,    -0.422
    0xbfecf830e8ce467b,     0xbfdb2f971db31972,     //   -0.9053,  -0.42478
    0xbfeced7af43cc773,     0xbfdb5d1009e15cc0,     //  -0.90399,  -0.42756
    0xbfece2b32799a060,     0xbfdb8a7814fd5693,     //  -0.90267,  -0.43033
    0xbfecd7d9898b32f6,     0xbfdbb7cf2304bd01,     //  -0.90135,  -0.43309
    0xbfecccee20c2de9f,     0xbfdbe51517ffc0d9,     //  -0.90002,  -0.43586
    0xbfecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     //  -0.89867,  -0.43862
    0xbfecb6e20a00da99,     0xbfdc3f6d47263129,     //  -0.89732,  -0.44137
    0xbfecabc169a0b901,     0xbfdc6c7f4997000a,     //  -0.89597,  -0.44412
    0xbfeca08f19b9c449,     0xbfdc997fc3865388,     //   -0.8946,  -0.44687
    0xbfec954b213411f5,     0xbfdcc66e9931c45d,     //  -0.89322,  -0.44961
    0xbfec89f587029c13,     0xbfdcf34baee1cd21,     //  -0.89184,  -0.45235
    0xbfec7e8e52233cf3,     0xbfdd2016e8e9db5b,     //  -0.89045,  -0.45508
    0xbfec7315899eaad7,     0xbfdd4cd02ba8609c,     //  -0.88905,  -0.45781
    0xbfec678b3488739b,     0xbfdd79775b86e389,     //  -0.88764,  -0.46054
    0xbfec5bef59fef85a,     0xbfdda60c5cfa10d8,     //  -0.88622,  -0.46326
    0xbfec5042012b6907,     0xbfddd28f1481cc58,     //   -0.8848,  -0.46598
    0xbfec44833141c004,     0xbfddfeff66a941de,     //  -0.88336,  -0.46869
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  -0.88192,   -0.4714
    0xbfec2cd14931e3f1,     0xbfde57a86d3cd824,     //  -0.88047,   -0.4741
    0xbfec20de3fa971b0,     0xbfde83e0eaf85113,     //  -0.87901,   -0.4768
    0xbfec14d9dc465e58,     0xbfdeb00695f25620,     //  -0.87755,  -0.47949
    0xbfec08c426725549,     0xbfdedc1952ef78d5,     //  -0.87607,  -0.48218
    0xbfebfc9d25a1b147,     0xbfdf081906bff7fd,     //  -0.87459,  -0.48487
    0xbfebf064e15377dd,     0xbfdf3405963fd068,     //  -0.87309,  -0.48755
    0xbfebe41b611154c1,     0xbfdf5fdee656cda3,     //   -0.8716,  -0.49023
    0xbfebd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //  -0.87009,   -0.4929
    0xbfebcb54cb0d2327,     0xbfdfb7575c24d2de,     //  -0.86857,  -0.49557
    0xbfebbed7c49380ea,     0xbfdfe2f64be7120f,     //  -0.86705,  -0.49823
    0xbfebb249a0b6c40d,     0xbfe00740c82b82e0,     //  -0.86551,  -0.50089
    0xbfeba5aa673590d2,     0xbfe01cfc874c3eb7,     //  -0.86397,  -0.50354
    0xbfeb98fa1fd9155e,     0xbfe032ae55edbd95,     //  -0.86242,  -0.50619
    0xbfeb8c38d27504e9,     0xbfe0485626ae221a,     //  -0.86087,  -0.50883
    0xbfeb7f6686e792ea,     0xbfe05df3ec31b8b6,     //   -0.8593,  -0.51147
    0xbfeb728345196e3e,     0xbfe073879922ffed,     //  -0.85773,   -0.5141
    0xbfeb658f14fdbc47,     0xbfe089112032b08c,     //  -0.85615,  -0.51673
    0xbfeb5889fe921405,     0xbfe09e907417c5e1,     //  -0.85456,  -0.51936
    0xbfeb4b7409de7925,     0xbfe0b405878f85ec,     //  -0.85296,  -0.52198
    0xbfeb3e4d3ef55712,     0xbfe0c9704d5d898f,     //  -0.85136,  -0.52459
    0xbfeb3115a5f37bf4,     0xbfe0ded0b84bc4b5,     //  -0.84974,   -0.5272
    0xbfeb23cd470013b4,     0xbfe0f426bb2a8e7d,     //  -0.84812,   -0.5298
    0xbfeb16742a4ca2f5,     0xbfe1097248d0a956,     //  -0.84649,   -0.5324
    0xbfeb090a58150200,     0xbfe11eb3541b4b22,     //  -0.84485,    -0.535
    0xbfeafb8fd89f57b6,     0xbfe133e9cfee254e,     //  -0.84321,  -0.53759
    0xbfeaee04b43c1474,     0xbfe14915af336ceb,     //  -0.84155,  -0.54017
    0xbfeae068f345ecef,     0xbfe15e36e4dbe2bc,     //  -0.83989,  -0.54275
    0xbfead2bc9e21d511,     0xbfe1734d63dedb49,     //  -0.83822,  -0.54532
    0xbfeac4ffbd3efac8,     0xbfe188591f3a46e5,     //  -0.83655,  -0.54789
    0xbfeab7325916c0d4,     0xbfe19d5a09f2b9b8,     //  -0.83486,  -0.55046
    0xbfeaa9547a2cb98e,     0xbfe1b250171373be,     //  -0.83317,  -0.55302
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  -0.83147,  -0.55557
    0xbfea8d676e545ad2,     0xbfe1dc1b64dc4872,     //  -0.82976,  -0.55812
    0xbfea7f58529fe69d,     0xbfe1f0f08bbc861b,     //  -0.82805,  -0.56066
    0xbfea7138de9d60f5,     0xbfe205baa17560d6,     //  -0.82632,   -0.5632
    0xbfea63091b02fae2,     0xbfe21a799933eb58,     //  -0.82459,  -0.56573
    0xbfea54c91090f524,     0xbfe22f2d662c13e1,     //  -0.82285,  -0.56826
    0xbfea4678c8119ac8,     0xbfe243d5fb98ac1f,     //   -0.8211,  -0.57078
    0xbfea38184a593bc6,     0xbfe258734cbb7110,     //  -0.81935,   -0.5733
    0xbfea29a7a0462782,     0xbfe26d054cdd12df,     //  -0.81758,  -0.57581
    0xbfea1b26d2c0a75e,     0xbfe2818bef4d3cba,     //  -0.81581,  -0.57831
    0xbfea0c95eabaf937,     0xbfe2960727629ca8,     //  -0.81404,  -0.58081
    0xbfe9fdf4f13149de,     0xbfe2aa76e87aeb58,     //  -0.81225,  -0.58331
    0xbfe9ef43ef29af94,     0xbfe2bedb25faf3ea,     //  -0.81046,   -0.5858
    0xbfe9e082edb42472,     0xbfe2d333d34e9bb7,     //  -0.80866,  -0.58828
    0xbfe9d1b1f5ea80d6,     0xbfe2e780e3e8ea16,     //  -0.80685,  -0.59076
    0xbfe9c2d110f075c3,     0xbfe2fbc24b441015,     //  -0.80503,  -0.59323
    0xbfe9b3e047f38741,     0xbfe30ff7fce17035,     //  -0.80321,   -0.5957
    0xbfe9a4dfa42b06b2,     0xbfe32421ec49a620,     //  -0.80138,  -0.59816
    0xbfe995cf2ed80d22,     0xbfe338400d0c8e57,     //  -0.79954,  -0.60062
    0xbfe986aef1457594,     0xbfe34c5252c14de1,     //  -0.79769,  -0.60307
    0xbfe9777ef4c7d742,     0xbfe36058b10659f3,     //  -0.79584,  -0.60551
    0xbfe9683f42bd7fe1,     0xbfe374531b817f8d,     //  -0.79398,  -0.60795
    0xbfe958efe48e6dd7,     0xbfe3884185dfeb22,     //  -0.79211,  -0.61038
    0xbfe94990e3ac4a6c,     0xbfe39c23e3d63029,     //  -0.79023,  -0.61281
    0xbfe93a22499263fc,     0xbfe3affa292050b9,     //  -0.78835,  -0.61523
    0xbfe92aa41fc5a815,     0xbfe3c3c44981c517,     //  -0.78646,  -0.61765
    0xbfe91b166fd49da2,     0xbfe3d78238c58343,     //  -0.78456,  -0.62006
    0xbfe90b7943575efe,     0xbfe3eb33eabe0680,     //  -0.78265,  -0.62246
    0xbfe8fbcca3ef940d,     0xbfe3fed9534556d4,     //  -0.78074,  -0.62486
    0xbfe8ec109b486c49,     0xbfe41272663d108c,     //  -0.77882,  -0.62725
    0xbfe8dc45331698cc,     0xbfe425ff178e6bb1,     //  -0.77689,  -0.62964
    0xbfe8cc6a75184655,     0xbfe4397f5b2a4380,     //  -0.77495,  -0.63202
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     //  -0.77301,  -0.63439
    0xbfe8ac871ede1d88,     0xbfe4605a692b32a2,     //  -0.77106,  -0.63676
    0xbfe89c7e9a4dd4ab,     0xbfe473b51b987347,     //   -0.7691,  -0.63912
    0xbfe88c66e7481ba1,     0xbfe48703306091fe,     //  -0.76714,  -0.64148
    0xbfe87c400fba2ebf,     0xbfe49a449b9b0938,     //  -0.76517,  -0.64383
    0xbfe86c0a1d9aa195,     0xbfe4ad79516722f0,     //  -0.76319,  -0.64618
    0xbfe85bc51ae958cc,     0xbfe4c0a145ec0004,     //   -0.7612,  -0.64851
    0xbfe84b7111af83f9,     0xbfe4d3bc6d589f80,     //  -0.75921,  -0.65085
    0xbfe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //  -0.75721,  -0.65317
    0xbfe82a9c13f545ff,     0xbfe4f9cc25cca486,     //   -0.7552,  -0.65549
    0xbfe81a1b33b57acc,     0xbfe50cc09f59a09b,     //  -0.75319,  -0.65781
    0xbfe8098b756e52fa,     0xbfe51fa81cd99aa6,     //  -0.75117,  -0.66011
    0xbfe7f8ece3571771,     0xbfe5328292a35596,     //  -0.74914,  -0.66242
    0xbfe7e83f87b03686,     0xbfe5454ff5159dfb,     //   -0.7471,  -0.66471
    0xbfe7d7836cc33db2,     0xbfe5581038975137,     //  -0.74506,    -0.667
    0xbfe7c6b89ce2d333,     0xbfe56ac35197649e,     //  -0.74301,  -0.66928
    0xbfe7b5df226aafb0,     0xbfe57d69348cec9f,     //  -0.74095,  -0.67156
    0xbfe7a4f707bf97d2,     0xbfe59001d5f723df,     //  -0.73889,  -0.67383
    0xbfe79400574f55e4,     0xbfe5a28d2a5d7250,     //  -0.73682,  -0.67609
    0xbfe782fb1b90b35b,     0xbfe5b50b264f7448,     //  -0.73474,  -0.67835
    0xbfe771e75f037261,     0xbfe5c77bbe65018c,     //  -0.73265,   -0.6806
    0xbfe760c52c304764,     0xbfe5d9dee73e345c,     //  -0.73056,  -0.68285
    0xbfe74f948da8d28d,     0xbfe5ec3495837074,     //  -0.72846,  -0.68508
    0xbfe73e558e079942,     0xbfe5fe7cbde56a0f,     //  -0.72636,  -0.68732
    0xbfe72d0837efff97,     0xbfe610b7551d2cde,     //  -0.72425,  -0.68954
    0xbfe71bac960e41bf,     0xbfe622e44fec22ff,     //  -0.72213,  -0.69176
    0xbfe70a42b3176d7a,     0xbfe63503a31c1be8,     //     -0.72,  -0.69397
    0xbfe6f8ca99c95b75,     0xbfe64715437f535b,     //  -0.71787,  -0.69618
    0xbfe6e74454eaa8ae,     0xbfe6591925f0783e,     //  -0.71573,  -0.69838
    0xbfe6d5afef4aafcc,     0xbfe66b0f3f52b386,     //  -0.71358,  -0.70057
    0xbfe6c40d73c18275,     0xbfe67cf78491af10,     //  -0.71143,  -0.70275
    0xbfe6b25ced2fe29c,     0xbfe68ed1eaa19c71,     //  -0.70927,  -0.70493
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  -0.70711,  -0.70711
    0xbfe68ed1eaa19c71,     0xbfe6b25ced2fe29c,     //  -0.70493,  -0.70927
    0xbfe67cf78491af10,     0xbfe6c40d73c18275,     //  -0.70275,  -0.71143
    0xbfe66b0f3f52b386,     0xbfe6d5afef4aafcc,     //  -0.70057,  -0.71358
    0xbfe6591925f0783e,     0xbfe6e74454eaa8ae,     //  -0.69838,  -0.71573
    0xbfe64715437f535b,     0xbfe6f8ca99c95b75,     //  -0.69618,  -0.71787
    0xbfe63503a31c1be8,     0xbfe70a42b3176d7a,     //  -0.69397,     -0.72
    0xbfe622e44fec22ff,     0xbfe71bac960e41bf,     //  -0.69176,  -0.72213
    0xbfe610b7551d2cde,     0xbfe72d0837efff97,     //  -0.68954,  -0.72425
    0xbfe5fe7cbde56a0f,     0xbfe73e558e079942,     //  -0.68732,  -0.72636
    0xbfe5ec3495837074,     0xbfe74f948da8d28d,     //  -0.68508,  -0.72846
    0xbfe5d9dee73e345c,     0xbfe760c52c304764,     //  -0.68285,  -0.73056
    0xbfe5c77bbe65018c,     0xbfe771e75f037261,     //   -0.6806,  -0.73265
    0xbfe5b50b264f7448,     0xbfe782fb1b90b35b,     //  -0.67835,  -0.73474
    0xbfe5a28d2a5d7250,     0xbfe79400574f55e4,     //  -0.67609,  -0.73682
    0xbfe59001d5f723df,     0xbfe7a4f707bf97d2,     //  -0.67383,  -0.73889
    0xbfe57d69348cec9f,     0xbfe7b5df226aafb0,     //  -0.67156,  -0.74095
    0xbfe56ac35197649e,     0xbfe7c6b89ce2d333,     //  -0.66928,  -0.74301
    0xbfe5581038975137,     0xbfe7d7836cc33db2,     //    -0.667,  -0.74506
    0xbfe5454ff5159dfb,     0xbfe7e83f87b03686,     //  -0.66471,   -0.7471
    0xbfe5328292a35596,     0xbfe7f8ece3571771,     //  -0.66242,  -0.74914
    0xbfe51fa81cd99aa6,     0xbfe8098b756e52fa,     //  -0.66011,  -0.75117
    0xbfe50cc09f59a09b,     0xbfe81a1b33b57acc,     //  -0.65781,  -0.75319
    0xbfe4f9cc25cca486,     0xbfe82a9c13f545ff,     //  -0.65549,   -0.7552
    0xbfe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //  -0.65317,  -0.75721
    0xbfe4d3bc6d589f80,     0xbfe84b7111af83f9,     //  -0.65085,  -0.75921
    0xbfe4c0a145ec0004,     0xbfe85bc51ae958cc,     //  -0.64851,   -0.7612
    0xbfe4ad79516722f0,     0xbfe86c0a1d9aa195,     //  -0.64618,  -0.76319
    0xbfe49a449b9b0938,     0xbfe87c400fba2ebf,     //  -0.64383,  -0.76517
    0xbfe48703306091fe,     0xbfe88c66e7481ba1,     //  -0.64148,  -0.76714
    0xbfe473b51b987347,     0xbfe89c7e9a4dd4ab,     //  -0.63912,   -0.7691
    0xbfe4605a692b32a2,     0xbfe8ac871ede1d88,     //  -0.63676,  -0.77106
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     //  -0.63439,  -0.77301
    0xbfe4397f5b2a4380,     0xbfe8cc6a75184655,     //  -0.63202,  -0.77495
    0xbfe425ff178e6bb1,     0xbfe8dc45331698cc,     //  -0.62964,  -0.77689
    0xbfe41272663d108c,     0xbfe8ec109b486c49,     //  -0.62725,  -0.77882
    0xbfe3fed9534556d4,     0xbfe8fbcca3ef940d,     //  -0.62486,  -0.78074
    0xbfe3eb33eabe0680,     0xbfe90b7943575efe,     //  -0.62246,  -0.78265
    0xbfe3d78238c58343,     0xbfe91b166fd49da2,     //  -0.62006,  -0.78456
    0xbfe3c3c44981c517,     0xbfe92aa41fc5a815,     //  -0.61765,  -0.78646
    0xbfe3affa292050b9,     0xbfe93a22499263fc,     //  -0.61523,  -0.78835
    0xbfe39c23e3d63029,     0xbfe94990e3ac4a6c,     //  -0.61281,  -0.79023
    0xbfe3884185dfeb22,     0xbfe958efe48e6dd7,     //  -0.61038,  -0.79211
    0xbfe374531b817f8d,     0xbfe9683f42bd7fe1,     //  -0.60795,  -0.79398
    0xbfe36058b10659f3,     0xbfe9777ef4c7d742,     //  -0.60551,  -0.79584
    0xbfe34c5252c14de1,     0xbfe986aef1457594,     //  -0.60307,  -0.79769
    0xbfe338400d0c8e57,     0xbfe995cf2ed80d22,     //  -0.60062,  -0.79954
    0xbfe32421ec49a620,     0xbfe9a4dfa42b06b2,     //  -0.59816,  -0.80138
    0xbfe30ff7fce17035,     0xbfe9b3e047f38741,     //   -0.5957,  -0.80321
    0xbfe2fbc24b441015,     0xbfe9c2d110f075c3,     //  -0.59323,  -0.80503
    0xbfe2e780e3e8ea16,     0xbfe9d1b1f5ea80d6,     //  -0.59076,  -0.80685
    0xbfe2d333d34e9bb7,     0xbfe9e082edb42472,     //  -0.58828,  -0.80866
    0xbfe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //   -0.5858,  -0.81046
    0xbfe2aa76e87aeb58,     0xbfe9fdf4f13149de,     //  -0.58331,  -0.81225
    0xbfe2960727629ca8,     0xbfea0c95eabaf937,     //  -0.58081,  -0.81404
    0xbfe2818bef4d3cba,     0xbfea1b26d2c0a75e,     //  -0.57831,  -0.81581
    0xbfe26d054cdd12df,     0xbfea29a7a0462782,     //  -0.57581,  -0.81758
    0xbfe258734cbb7110,     0xbfea38184a593bc6,     //   -0.5733,  -0.81935
    0xbfe243d5fb98ac1f,     0xbfea4678c8119ac8,     //  -0.57078,   -0.8211
    0xbfe22f2d662c13e1,     0xbfea54c91090f524,     //  -0.56826,  -0.82285
    0xbfe21a799933eb58,     0xbfea63091b02fae2,     //  -0.56573,  -0.82459
    0xbfe205baa17560d6,     0xbfea7138de9d60f5,     //   -0.5632,  -0.82632
    0xbfe1f0f08bbc861b,     0xbfea7f58529fe69d,     //  -0.56066,  -0.82805
    0xbfe1dc1b64dc4872,     0xbfea8d676e545ad2,     //  -0.55812,  -0.82976
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  -0.55557,  -0.83147
    0xbfe1b250171373be,     0xbfeaa9547a2cb98e,     //  -0.55302,  -0.83317
    0xbfe19d5a09f2b9b8,     0xbfeab7325916c0d4,     //  -0.55046,  -0.83486
    0xbfe188591f3a46e5,     0xbfeac4ffbd3efac8,     //  -0.54789,  -0.83655
    0xbfe1734d63dedb49,     0xbfead2bc9e21d511,     //  -0.54532,  -0.83822
    0xbfe15e36e4dbe2bc,     0xbfeae068f345ecef,     //  -0.54275,  -0.83989
    0xbfe14915af336ceb,     0xbfeaee04b43c1474,     //  -0.54017,  -0.84155
    0xbfe133e9cfee254e,     0xbfeafb8fd89f57b6,     //  -0.53759,  -0.84321
    0xbfe11eb3541b4b22,     0xbfeb090a58150200,     //    -0.535,  -0.84485
    0xbfe1097248d0a956,     0xbfeb16742a4ca2f5,     //   -0.5324,  -0.84649
    0xbfe0f426bb2a8e7d,     0xbfeb23cd470013b4,     //   -0.5298,  -0.84812
    0xbfe0ded0b84bc4b5,     0xbfeb3115a5f37bf4,     //   -0.5272,  -0.84974
    0xbfe0c9704d5d898f,     0xbfeb3e4d3ef55712,     //  -0.52459,  -0.85136
    0xbfe0b405878f85ec,     0xbfeb4b7409de7925,     //  -0.52198,  -0.85296
    0xbfe09e907417c5e1,     0xbfeb5889fe921405,     //  -0.51936,  -0.85456
    0xbfe089112032b08c,     0xbfeb658f14fdbc47,     //  -0.51673,  -0.85615
    0xbfe073879922ffed,     0xbfeb728345196e3e,     //   -0.5141,  -0.85773
    0xbfe05df3ec31b8b6,     0xbfeb7f6686e792ea,     //  -0.51147,   -0.8593
    0xbfe0485626ae221a,     0xbfeb8c38d27504e9,     //  -0.50883,  -0.86087
    0xbfe032ae55edbd95,     0xbfeb98fa1fd9155e,     //  -0.50619,  -0.86242
    0xbfe01cfc874c3eb7,     0xbfeba5aa673590d2,     //  -0.50354,  -0.86397
    0xbfe00740c82b82e0,     0xbfebb249a0b6c40d,     //  -0.50089,  -0.86551
    0xbfdfe2f64be7120f,     0xbfebbed7c49380ea,     //  -0.49823,  -0.86705
    0xbfdfb7575c24d2de,     0xbfebcb54cb0d2327,     //  -0.49557,  -0.86857
    0xbfdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //   -0.4929,  -0.87009
    0xbfdf5fdee656cda3,     0xbfebe41b611154c1,     //  -0.49023,   -0.8716
    0xbfdf3405963fd068,     0xbfebf064e15377dd,     //  -0.48755,  -0.87309
    0xbfdf081906bff7fd,     0xbfebfc9d25a1b147,     //  -0.48487,  -0.87459
    0xbfdedc1952ef78d5,     0xbfec08c426725549,     //  -0.48218,  -0.87607
    0xbfdeb00695f25620,     0xbfec14d9dc465e58,     //  -0.47949,  -0.87755
    0xbfde83e0eaf85113,     0xbfec20de3fa971b0,     //   -0.4768,  -0.87901
    0xbfde57a86d3cd824,     0xbfec2cd14931e3f1,     //   -0.4741,  -0.88047
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   -0.4714,  -0.88192
    0xbfddfeff66a941de,     0xbfec44833141c004,     //  -0.46869,  -0.88336
    0xbfddd28f1481cc58,     0xbfec5042012b6907,     //  -0.46598,   -0.8848
    0xbfdda60c5cfa10d8,     0xbfec5bef59fef85a,     //  -0.46326,  -0.88622
    0xbfdd79775b86e389,     0xbfec678b3488739b,     //  -0.46054,  -0.88764
    0xbfdd4cd02ba8609c,     0xbfec7315899eaad7,     //  -0.45781,  -0.88905
    0xbfdd2016e8e9db5b,     0xbfec7e8e52233cf3,     //  -0.45508,  -0.89045
    0xbfdcf34baee1cd21,     0xbfec89f587029c13,     //  -0.45235,  -0.89184
    0xbfdcc66e9931c45d,     0xbfec954b213411f5,     //  -0.44961,  -0.89322
    0xbfdc997fc3865388,     0xbfeca08f19b9c449,     //  -0.44687,   -0.8946
    0xbfdc6c7f4997000a,     0xbfecabc169a0b901,     //  -0.44412,  -0.89597
    0xbfdc3f6d47263129,     0xbfecb6e20a00da99,     //  -0.44137,  -0.89732
    0xbfdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     //  -0.43862,  -0.89867
    0xbfdbe51517ffc0d9,     0xbfecccee20c2de9f,     //  -0.43586,  -0.90002
    0xbfdbb7cf2304bd01,     0xbfecd7d9898b32f6,     //  -0.43309,  -0.90135
    0xbfdb8a7814fd5693,     0xbfece2b32799a060,     //  -0.43033,  -0.90267
    0xbfdb5d1009e15cc0,     0xbfeced7af43cc773,     //  -0.42756,  -0.90399
    0xbfdb2f971db31972,     0xbfecf830e8ce467b,     //  -0.42478,   -0.9053
    0xbfdb020d6c7f4009,     0xbfed02d4feb2bd92,     //    -0.422,   -0.9066
    0xbfdad473125cdc08,     0xbfed0d672f59d2b9,     //  -0.41922,  -0.90789
    0xbfdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     //  -0.41643,  -0.90917
    0xbfda790cd3dbf31a,     0xbfed2255c6e5a4e1,     //  -0.41364,  -0.91044
    0xbfda4b4127dea1e4,     0xbfed2cb220e0ef9f,     //  -0.41084,  -0.91171
    0xbfda1d6543b50ac0,     0xbfed36fc7bcbfbdc,     //  -0.40804,  -0.91296
    0xbfd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //  -0.40524,  -0.91421
    0xbfd9c17d440df9f2,     0xbfed4b5b1b187524,     //  -0.40243,  -0.91545
    0xbfd993716141bdfe,     0xbfed556f52e93eb1,     //  -0.39962,  -0.91668
    0xbfd96555b7ab948f,     0xbfed5f7172888a7f,     //  -0.39681,   -0.9179
    0xbfd9372a63bc93d7,     0xbfed696173c9e68b,     //  -0.39399,  -0.91911
    0xbfd908ef81ef7bd1,     0xbfed733f508c0dff,     //  -0.39117,  -0.92032
    0xbfd8daa52ec8a4af,     0xbfed7d0b02b8ecf9,     //  -0.38835,  -0.92151
    0xbfd8ac4b86d5ed44,     0xbfed86c48445a450,     //  -0.38552,   -0.9227
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     //  -0.38268,  -0.92388
    0xbfd84f6aaaf3903f,     0xbfed9a00dd8b3d46,     //  -0.37985,  -0.92505
    0xbfd820e3b04eaac4,     0xbfeda383a9668988,     //  -0.37701,  -0.92621
    0xbfd7f24dd37341e3,     0xbfedacf42ce68ab9,     //  -0.37416,  -0.92736
    0xbfd7c3a9311dcce7,     0xbfedb6526238a09b,     //  -0.37132,  -0.92851
    0xbfd794f5e613dfae,     0xbfedbf9e4395759a,     //  -0.36847,  -0.92964
    0xbfd766340f2418f6,     0xbfedc8d7cb410260,     //  -0.36561,  -0.93077
    0xbfd73763c9261092,     0xbfedd1fef38a915a,     //  -0.36276,  -0.93188
    0xbfd7088530fa459e,     0xbfeddb13b6ccc23d,     //   -0.3599,  -0.93299
    0xbfd6d998638a0cb5,     0xbfede4160f6d8d81,     //  -0.35703,  -0.93409
    0xbfd6aa9d7dc77e16,     0xbfeded05f7de47da,     //  -0.35416,  -0.93518
    0xbfd67b949cad63ca,     0xbfedf5e36a9ba59c,     //  -0.35129,  -0.93627
    0xbfd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     //  -0.34842,  -0.93734
    0xbfd61d595c88c203,     0xbfee0766d9280f54,     //  -0.34554,   -0.9384
    0xbfd5ee27379ea693,     0xbfee100cca2980ac,     //  -0.34266,  -0.93946
    0xbfd5bee78b9db3b6,     0xbfee18a02fdc66d9,     //  -0.33978,  -0.94051
    0xbfd58f9a75ab1fdd,     0xbfee212104f686e5,     //  -0.33689,  -0.94154
    0xbfd5604012f467b4,     0xbfee298f4439197a,     //    -0.334,  -0.94257
    0xbfd530d880af3c24,     0xbfee31eae870ce25,     //  -0.33111,  -0.94359
    0xbfd50163dc197047,     0xbfee3a33ec75ce85,     //  -0.32821,   -0.9446
    0xbfd4d1e24278e76a,     0xbfee426a4b2bc17e,     //  -0.32531,  -0.94561
    0xbfd4a253d11b82f3,     0xbfee4a8dff81ce5e,     //  -0.32241,   -0.9466
    0xbfd472b8a5571054,     0xbfee529f04729ffc,     //   -0.3195,  -0.94759
    0xbfd44310dc8936f0,     0xbfee5a9d550467d3,     //  -0.31659,  -0.94856
    0xbfd4135c94176602,     0xbfee6288ec48e112,     //  -0.31368,  -0.94953
    0xbfd3e39be96ec271,     0xbfee6a61c55d53a7,     //  -0.31077,  -0.95049
    0xbfd3b3cefa0414b7,     0xbfee7227db6a9744,     //  -0.30785,  -0.95144
    0xbfd383f5e353b6aa,     0xbfee79db29a5165a,     //  -0.30493,  -0.95238
    0xbfd35410c2e18152,     0xbfee817bab4cd10d,     //  -0.30201,  -0.95331
    0xbfd3241fb638baaf,     0xbfee89095bad6025,     //  -0.29908,  -0.95423
    0xbfd2f422daec0386,     0xbfee9084361df7f3,     //  -0.29615,  -0.95514
    0xbfd2c41a4e954520,     0xbfee97ec36016b30,     //  -0.29322,  -0.95605
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     //  -0.29028,  -0.95694
    0xbfd263e6995554ba,     0xbfeea68393e65800,     //  -0.28735,  -0.95783
    0xbfd233bbabc3bb72,     0xbfeeadb2e8e7a88e,     //  -0.28441,   -0.9587
    0xbfd2038583d727bd,     0xbfeeb4cf515b8811,     //  -0.28146,  -0.95957
    0xbfd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     //  -0.27852,  -0.96043
    0xbfd1a2f7fbe8f243,     0xbfeec2cf4b1af6b2,     //  -0.27557,  -0.96128
    0xbfd172a0d7765177,     0xbfeec9b2d3c3bf84,     //  -0.27262,  -0.96212
    0xbfd1423eefc69378,     0xbfeed0835e999009,     //  -0.26967,  -0.96295
    0xbfd111d262b1f677,     0xbfeed740e7684963,     //  -0.26671,  -0.96378
    0xbfd0e15b4e1749cd,     0xbfeeddeb6a078651,     //  -0.26375,  -0.96459
    0xbfd0b0d9cfdbdb90,     0xbfeee482e25a9dbc,     //  -0.26079,  -0.96539
    0xbfd0804e05eb661e,     0xbfeeeb074c50a544,     //  -0.25783,  -0.96619
    0xbfd04fb80e37fdae,     0xbfeef178a3e473c2,     //  -0.25487,  -0.96698
    0xbfd01f1806b9fdd2,     0xbfeef7d6e51ca3c0,     //   -0.2519,  -0.96775
    0xbfcfdcdc1adfedf8,     0xbfeefe220c0b95ec,     //  -0.24893,  -0.96852
    0xbfcf7b7480bd3801,     0xbfef045a14cf738c,     //  -0.24596,  -0.96928
    0xbfcf19f97b215f1a,     0xbfef0a7efb9230d7,     //  -0.24298,  -0.97003
    0xbfceb86b462de348,     0xbfef1090bc898f5f,     //     -0.24,  -0.97077
    0xbfce56ca1e101a1b,     0xbfef168f53f7205d,     //  -0.23702,   -0.9715
    0xbfcdf5163f01099a,     0xbfef1c7abe284708,     //  -0.23404,  -0.97223
    0xbfcd934fe5454311,     0xbfef2252f7763ada,     //  -0.23106,  -0.97294
    0xbfcd31774d2cbdee,     0xbfef2817fc4609ce,     //  -0.22807,  -0.97364
    0xbfcccf8cb312b286,     0xbfef2dc9c9089a9d,     //  -0.22508,  -0.97434
    0xbfcc6d90535d74dc,     0xbfef33685a3aaef0,     //  -0.22209,  -0.97503
    0xbfcc0b826a7e4f63,     0xbfef38f3ac64e589,     //   -0.2191,   -0.9757
    0xbfcba96334f15dad,     0xbfef3e6bbc1bbc65,     //  -0.21611,  -0.97637
    0xbfcb4732ef3d6722,     0xbfef43d085ff92dd,     //  -0.21311,  -0.97703
    0xbfcae4f1d5f3b9ab,     0xbfef492206bcabb4,     //  -0.21011,  -0.97768
    0xbfca82a025b00451,     0xbfef4e603b0b2f2d,     //  -0.20711,  -0.97832
    0xbfca203e1b1831da,     0xbfef538b1faf2d07,     //  -0.20411,  -0.97895
    0xbfc9bdcbf2dc4366,     0xbfef58a2b1789e84,     //   -0.2011,  -0.97957
    0xbfc95b49e9b62af9,     0xbfef5da6ed43685d,     //   -0.1981,  -0.98018
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  -0.19509,  -0.98079
    0xbfc8961727c41804,     0xbfef677556883cee,     //  -0.19208,  -0.98138
    0xbfc83366e89c64c5,     0xbfef6c3f7df5bbb7,     //  -0.18907,  -0.98196
    0xbfc7d0a7bbd2cb1b,     0xbfef70f6434b7eb7,     //  -0.18606,  -0.98254
    0xbfc76dd9de50bf31,     0xbfef7599a3a12077,     //  -0.18304,  -0.98311
    0xbfc70afd8d08c4ff,     0xbfef7a299c1a322a,     //  -0.18002,  -0.98366
    0xbfc6a81304f64ab2,     0xbfef7ea629e63d6e,     //    -0.177,  -0.98421
    0xbfc6451a831d830d,     0xbfef830f4a40c60c,     //  -0.17398,  -0.98475
    0xbfc5e214448b3fc6,     0xbfef8764fa714ba9,     //  -0.17096,  -0.98528
    0xbfc57f008654cbde,     0xbfef8ba737cb4b78,     //  -0.16794,   -0.9858
    0xbfc51bdf8597c5f2,     0xbfef8fd5ffae41db,     //  -0.16491,  -0.98631
    0xbfc4b8b17f79fa88,     0xbfef93f14f85ac08,     //  -0.16189,  -0.98681
    0xbfc45576b1293e5a,     0xbfef97f924c9099b,     //  -0.15886,   -0.9873
    0xbfc3f22f57db4893,     0xbfef9bed7cfbde29,     //  -0.15583,  -0.98778
    0xbfc38edbb0cd8d14,     0xbfef9fce55adb2c8,     //   -0.1528,  -0.98826
    0xbfc32b7bf94516a7,     0xbfefa39bac7a1791,     //  -0.14976,  -0.98872
    0xbfc2c8106e8e613a,     0xbfefa7557f08a517,     //  -0.14673,  -0.98918
    0xbfc264994dfd340a,     0xbfefaafbcb0cfddc,     //   -0.1437,  -0.98962
    0xbfc20116d4ec7bce,     0xbfefae8e8e46cfbb,     //  -0.14066,  -0.99006
    0xbfc19d8940be24e7,     0xbfefb20dc681d54d,     //  -0.13762,  -0.99049
    0xbfc139f0cedaf576,     0xbfefb5797195d741,     //  -0.13458,   -0.9909
    0xbfc0d64dbcb26786,     0xbfefb8d18d66adb7,     //  -0.13154,  -0.99131
    0xbfc072a047ba831d,     0xbfefbc1617e44186,     //   -0.1285,  -0.99171
    0xbfc00ee8ad6fb85b,     0xbfefbf470f0a8d88,     //  -0.12545,   -0.9921
    0xbfbf564e56a9730e,     0xbfefc26470e19fd3,     //  -0.12241,  -0.99248
    0xbfbe8eb7fde4aa3e,     0xbfefc56e3b7d9af6,     //  -0.11937,  -0.99285
    0xbfbdc70ecbae9fc8,     0xbfefc8646cfeb721,     //  -0.11632,  -0.99321
    0xbfbcff533b307dc1,     0xbfefcb4703914354,     //  -0.11327,  -0.99356
    0xbfbc3785c79ec2d5,     0xbfefce15fd6da67b,     //  -0.11022,  -0.99391
    0xbfbb6fa6ec38f64c,     0xbfefd0d158d86087,     //  -0.10717,  -0.99424
    0xbfbaa7b724495c04,     0xbfefd37914220b84,     //  -0.10412,  -0.99456
    0xbfb9dfb6eb24a85c,     0xbfefd60d2da75c9e,     //  -0.10107,  -0.99488
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     // -0.098017,  -0.99518
    0xbfb84f8712c130a0,     0xbfefdafa7514538c,     // -0.094963,  -0.99548
    0xbfb787586a5d5b21,     0xbfefdd539ff1f456,     // -0.091909,  -0.99577
    0xbfb6bf1b3e79b129,     0xbfefdf9922f73307,     // -0.088854,  -0.99604
    0xbfb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     // -0.085797,  -0.99631
    0xbfb52e774a4d4d0a,     0xbfefe3e92be9d886,     //  -0.08274,  -0.99657
    0xbfb4661179272096,     0xbfefe5f3af2e3940,     // -0.079682,  -0.99682
    0xbfb39d9f12c5a299,     0xbfefe7ea85482d60,     // -0.076624,  -0.99706
    0xbfb2d52092ce19f6,     0xbfefe9cdad01883a,     // -0.073565,  -0.99729
    0xbfb20c9674ed444c,     0xbfefeb9d2530410f,     // -0.070505,  -0.99751
    0xbfb1440134d709b2,     0xbfefed58ecb673c4,     // -0.067444,  -0.99772
    0xbfb07b614e463064,     0xbfefef0102826191,     // -0.064383,  -0.99793
    0xbfaf656e79f820e0,     0xbfeff095658e71ad,     // -0.061321,  -0.99812
    0xbfadd406f9808ec8,     0xbfeff21614e131ed,     // -0.058258,   -0.9983
    0xbfac428d12c0d7e3,     0xbfeff3830f8d575c,     // -0.055195,  -0.99848
    0xbfaab101bd5f8317,     0xbfeff4dc54b1bed3,     // -0.052132,  -0.99864
    0xbfa91f65f10dd814,     0xbfeff621e3796d7e,     // -0.049068,   -0.9988
    0xbfa78dbaa5874685,     0xbfeff753bb1b9164,     // -0.046003,  -0.99894
    0xbfa5fc00d290cd43,     0xbfeff871dadb81df,     // -0.042938,  -0.99908
    0xbfa46a396ff86179,     0xbfeff97c4208c014,     // -0.039873,   -0.9992
    0xbfa2d865759455cd,     0xbfeffa72effef75d,     // -0.036807,  -0.99932
    0xbfa14685db42c17e,     0xbfeffb55e425fdae,     // -0.033741,  -0.99943
    0xbf9f693731d1cf01,     0xbfeffc251df1d3f8,     // -0.030675,  -0.99953
    0xbf9c454f4ce53b1c,     0xbfeffce09ce2a679,     // -0.027608,  -0.99962
    0xbf992155f7a3667e,     0xbfeffd886084cd0d,     // -0.024541,   -0.9997
    0xbf95fd4d21fab226,     0xbfeffe1c6870cb77,     // -0.021474,  -0.99977
    0xbf92d936bbe30efd,     0xbfeffe9cb44b51a1,     // -0.018407,  -0.99983
    0xbf8f6a296ab997ca,     0xbfefff0943c53bd1,     // -0.015339,  -0.99988
    0xbf8921d1fcdec784,     0xbfefff62169b92db,     // -0.012272,  -0.99992
    0xbf82d96b0e509703,     0xbfefffa72c978c4f,     //-0.0092038,  -0.99996
    0xbf7921f0fe670071,     0xbfefffd8858e8a92,     //-0.0061359,  -0.99998
    0xbf6921f8becca4ba,     0xbfeffff621621d02,     // -0.003068,        -1
    0x0000000000000000,     0xbff0000000000000,     //         0,        -1
    0x3f6921f8becca4ba,     0xbfeffff621621d02,     //  0.003068,        -1
    0x3f7921f0fe670071,     0xbfefffd8858e8a92,     // 0.0061359,  -0.99998
    0x3f82d96b0e509703,     0xbfefffa72c978c4f,     // 0.0092038,  -0.99996
    0x3f8921d1fcdec784,     0xbfefff62169b92db,     //  0.012272,  -0.99992
    0x3f8f6a296ab997ca,     0xbfefff0943c53bd1,     //  0.015339,  -0.99988
    0x3f92d936bbe30efd,     0xbfeffe9cb44b51a1,     //  0.018407,  -0.99983
    0x3f95fd4d21fab226,     0xbfeffe1c6870cb77,     //  0.021474,  -0.99977
    0x3f992155f7a3667e,     0xbfeffd886084cd0d,     //  0.024541,   -0.9997
    0x3f9c454f4ce53b1c,     0xbfeffce09ce2a679,     //  0.027608,  -0.99962
    0x3f9f693731d1cf01,     0xbfeffc251df1d3f8,     //  0.030675,  -0.99953
    0x3fa14685db42c17e,     0xbfeffb55e425fdae,     //  0.033741,  -0.99943
    0x3fa2d865759455cd,     0xbfeffa72effef75d,     //  0.036807,  -0.99932
    0x3fa46a396ff86179,     0xbfeff97c4208c014,     //  0.039873,   -0.9992
    0x3fa5fc00d290cd43,     0xbfeff871dadb81df,     //  0.042938,  -0.99908
    0x3fa78dbaa5874685,     0xbfeff753bb1b9164,     //  0.046003,  -0.99894
    0x3fa91f65f10dd814,     0xbfeff621e3796d7e,     //  0.049068,   -0.9988
    0x3faab101bd5f8317,     0xbfeff4dc54b1bed3,     //  0.052132,  -0.99864
    0x3fac428d12c0d7e3,     0xbfeff3830f8d575c,     //  0.055195,  -0.99848
    0x3fadd406f9808ec8,     0xbfeff21614e131ed,     //  0.058258,   -0.9983
    0x3faf656e79f820e0,     0xbfeff095658e71ad,     //  0.061321,  -0.99812
    0x3fb07b614e463064,     0xbfefef0102826191,     //  0.064383,  -0.99793
    0x3fb1440134d709b2,     0xbfefed58ecb673c4,     //  0.067444,  -0.99772
    0x3fb20c9674ed444c,     0xbfefeb9d2530410f,     //  0.070505,  -0.99751
    0x3fb2d52092ce19f6,     0xbfefe9cdad01883a,     //  0.073565,  -0.99729
    0x3fb39d9f12c5a299,     0xbfefe7ea85482d60,     //  0.076624,  -0.99706
    0x3fb4661179272096,     0xbfefe5f3af2e3940,     //  0.079682,  -0.99682
    0x3fb52e774a4d4d0a,     0xbfefe3e92be9d886,     //   0.08274,  -0.99657
    0x3fb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     //  0.085797,  -0.99631
    0x3fb6bf1b3e79b129,     0xbfefdf9922f73307,     //  0.088854,  -0.99604
    0x3fb787586a5d5b21,     0xbfefdd539ff1f456,     //  0.091909,  -0.99577
    0x3fb84f8712c130a0,     0xbfefdafa7514538c,     //  0.094963,  -0.99548
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     //  0.098017,  -0.99518
    0x3fb9dfb6eb24a85c,     0xbfefd60d2da75c9e,     //   0.10107,  -0.99488
    0x3fbaa7b724495c04,     0xbfefd37914220b84,     //   0.10412,  -0.99456
    0x3fbb6fa6ec38f64c,     0xbfefd0d158d86087,     //   0.10717,  -0.99424
    0x3fbc3785c79ec2d5,     0xbfefce15fd6da67b,     //   0.11022,  -0.99391
    0x3fbcff533b307dc1,     0xbfefcb4703914354,     //   0.11327,  -0.99356
    0x3fbdc70ecbae9fc8,     0xbfefc8646cfeb721,     //   0.11632,  -0.99321
    0x3fbe8eb7fde4aa3e,     0xbfefc56e3b7d9af6,     //   0.11937,  -0.99285
    0x3fbf564e56a9730e,     0xbfefc26470e19fd3,     //   0.12241,  -0.99248
    0x3fc00ee8ad6fb85b,     0xbfefbf470f0a8d88,     //   0.12545,   -0.9921
    0x3fc072a047ba831d,     0xbfefbc1617e44186,     //    0.1285,  -0.99171
    0x3fc0d64dbcb26786,     0xbfefb8d18d66adb7,     //   0.13154,  -0.99131
    0x3fc139f0cedaf576,     0xbfefb5797195d741,     //   0.13458,   -0.9909
    0x3fc19d8940be24e7,     0xbfefb20dc681d54d,     //   0.13762,  -0.99049
    0x3fc20116d4ec7bce,     0xbfefae8e8e46cfbb,     //   0.14066,  -0.99006
    0x3fc264994dfd340a,     0xbfefaafbcb0cfddc,     //    0.1437,  -0.98962
    0x3fc2c8106e8e613a,     0xbfefa7557f08a517,     //   0.14673,  -0.98918
    0x3fc32b7bf94516a7,     0xbfefa39bac7a1791,     //   0.14976,  -0.98872
    0x3fc38edbb0cd8d14,     0xbfef9fce55adb2c8,     //    0.1528,  -0.98826
    0x3fc3f22f57db4893,     0xbfef9bed7cfbde29,     //   0.15583,  -0.98778
    0x3fc45576b1293e5a,     0xbfef97f924c9099b,     //   0.15886,   -0.9873
    0x3fc4b8b17f79fa88,     0xbfef93f14f85ac08,     //   0.16189,  -0.98681
    0x3fc51bdf8597c5f2,     0xbfef8fd5ffae41db,     //   0.16491,  -0.98631
    0x3fc57f008654cbde,     0xbfef8ba737cb4b78,     //   0.16794,   -0.9858
    0x3fc5e214448b3fc6,     0xbfef8764fa714ba9,     //   0.17096,  -0.98528
    0x3fc6451a831d830d,     0xbfef830f4a40c60c,     //   0.17398,  -0.98475
    0x3fc6a81304f64ab2,     0xbfef7ea629e63d6e,     //     0.177,  -0.98421
    0x3fc70afd8d08c4ff,     0xbfef7a299c1a322a,     //   0.18002,  -0.98366
    0x3fc76dd9de50bf31,     0xbfef7599a3a12077,     //   0.18304,  -0.98311
    0x3fc7d0a7bbd2cb1b,     0xbfef70f6434b7eb7,     //   0.18606,  -0.98254
    0x3fc83366e89c64c5,     0xbfef6c3f7df5bbb7,     //   0.18907,  -0.98196
    0x3fc8961727c41804,     0xbfef677556883cee,     //   0.19208,  -0.98138
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //   0.19509,  -0.98079
    0x3fc95b49e9b62af9,     0xbfef5da6ed43685d,     //    0.1981,  -0.98018
    0x3fc9bdcbf2dc4366,     0xbfef58a2b1789e84,     //    0.2011,  -0.97957
    0x3fca203e1b1831da,     0xbfef538b1faf2d07,     //   0.20411,  -0.97895
    0x3fca82a025b00451,     0xbfef4e603b0b2f2d,     //   0.20711,  -0.97832
    0x3fcae4f1d5f3b9ab,     0xbfef492206bcabb4,     //   0.21011,  -0.97768
    0x3fcb4732ef3d6722,     0xbfef43d085ff92dd,     //   0.21311,  -0.97703
    0x3fcba96334f15dad,     0xbfef3e6bbc1bbc65,     //   0.21611,  -0.97637
    0x3fcc0b826a7e4f63,     0xbfef38f3ac64e589,     //    0.2191,   -0.9757
    0x3fcc6d90535d74dc,     0xbfef33685a3aaef0,     //   0.22209,  -0.97503
    0x3fcccf8cb312b286,     0xbfef2dc9c9089a9d,     //   0.22508,  -0.97434
    0x3fcd31774d2cbdee,     0xbfef2817fc4609ce,     //   0.22807,  -0.97364
    0x3fcd934fe5454311,     0xbfef2252f7763ada,     //   0.23106,  -0.97294
    0x3fcdf5163f01099a,     0xbfef1c7abe284708,     //   0.23404,  -0.97223
    0x3fce56ca1e101a1b,     0xbfef168f53f7205d,     //   0.23702,   -0.9715
    0x3fceb86b462de348,     0xbfef1090bc898f5f,     //      0.24,  -0.97077
    0x3fcf19f97b215f1a,     0xbfef0a7efb9230d7,     //   0.24298,  -0.97003
    0x3fcf7b7480bd3801,     0xbfef045a14cf738c,     //   0.24596,  -0.96928
    0x3fcfdcdc1adfedf8,     0xbfeefe220c0b95ec,     //   0.24893,  -0.96852
    0x3fd01f1806b9fdd2,     0xbfeef7d6e51ca3c0,     //    0.2519,  -0.96775
    0x3fd04fb80e37fdae,     0xbfeef178a3e473c2,     //   0.25487,  -0.96698
    0x3fd0804e05eb661e,     0xbfeeeb074c50a544,     //   0.25783,  -0.96619
    0x3fd0b0d9cfdbdb90,     0xbfeee482e25a9dbc,     //   0.26079,  -0.96539
    0x3fd0e15b4e1749cd,     0xbfeeddeb6a078651,     //   0.26375,  -0.96459
    0x3fd111d262b1f677,     0xbfeed740e7684963,     //   0.26671,  -0.96378
    0x3fd1423eefc69378,     0xbfeed0835e999009,     //   0.26967,  -0.96295
    0x3fd172a0d7765177,     0xbfeec9b2d3c3bf84,     //   0.27262,  -0.96212
    0x3fd1a2f7fbe8f243,     0xbfeec2cf4b1af6b2,     //   0.27557,  -0.96128
    0x3fd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     //   0.27852,  -0.96043
    0x3fd2038583d727bd,     0xbfeeb4cf515b8811,     //   0.28146,  -0.95957
    0x3fd233bbabc3bb72,     0xbfeeadb2e8e7a88e,     //   0.28441,   -0.9587
    0x3fd263e6995554ba,     0xbfeea68393e65800,     //   0.28735,  -0.95783
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //   0.29028,  -0.95694
    0x3fd2c41a4e954520,     0xbfee97ec36016b30,     //   0.29322,  -0.95605
    0x3fd2f422daec0386,     0xbfee9084361df7f3,     //   0.29615,  -0.95514
    0x3fd3241fb638baaf,     0xbfee89095bad6025,     //   0.29908,  -0.95423
    0x3fd35410c2e18152,     0xbfee817bab4cd10d,     //   0.30201,  -0.95331
    0x3fd383f5e353b6aa,     0xbfee79db29a5165a,     //   0.30493,  -0.95238
    0x3fd3b3cefa0414b7,     0xbfee7227db6a9744,     //   0.30785,  -0.95144
    0x3fd3e39be96ec271,     0xbfee6a61c55d53a7,     //   0.31077,  -0.95049
    0x3fd4135c94176602,     0xbfee6288ec48e112,     //   0.31368,  -0.94953
    0x3fd44310dc8936f0,     0xbfee5a9d550467d3,     //   0.31659,  -0.94856
    0x3fd472b8a5571054,     0xbfee529f04729ffc,     //    0.3195,  -0.94759
    0x3fd4a253d11b82f3,     0xbfee4a8dff81ce5e,     //   0.32241,   -0.9466
    0x3fd4d1e24278e76a,     0xbfee426a4b2bc17e,     //   0.32531,  -0.94561
    0x3fd50163dc197047,     0xbfee3a33ec75ce85,     //   0.32821,   -0.9446
    0x3fd530d880af3c24,     0xbfee31eae870ce25,     //   0.33111,  -0.94359
    0x3fd5604012f467b4,     0xbfee298f4439197a,     //     0.334,  -0.94257
    0x3fd58f9a75ab1fdd,     0xbfee212104f686e5,     //   0.33689,  -0.94154
    0x3fd5bee78b9db3b6,     0xbfee18a02fdc66d9,     //   0.33978,  -0.94051
    0x3fd5ee27379ea693,     0xbfee100cca2980ac,     //   0.34266,  -0.93946
    0x3fd61d595c88c203,     0xbfee0766d9280f54,     //   0.34554,   -0.9384
    0x3fd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     //   0.34842,  -0.93734
    0x3fd67b949cad63ca,     0xbfedf5e36a9ba59c,     //   0.35129,  -0.93627
    0x3fd6aa9d7dc77e16,     0xbfeded05f7de47da,     //   0.35416,  -0.93518
    0x3fd6d998638a0cb5,     0xbfede4160f6d8d81,     //   0.35703,  -0.93409
    0x3fd7088530fa459e,     0xbfeddb13b6ccc23d,     //    0.3599,  -0.93299
    0x3fd73763c9261092,     0xbfedd1fef38a915a,     //   0.36276,  -0.93188
    0x3fd766340f2418f6,     0xbfedc8d7cb410260,     //   0.36561,  -0.93077
    0x3fd794f5e613dfae,     0xbfedbf9e4395759a,     //   0.36847,  -0.92964
    0x3fd7c3a9311dcce7,     0xbfedb6526238a09b,     //   0.37132,  -0.92851
    0x3fd7f24dd37341e3,     0xbfedacf42ce68ab9,     //   0.37416,  -0.92736
    0x3fd820e3b04eaac4,     0xbfeda383a9668988,     //   0.37701,  -0.92621
    0x3fd84f6aaaf3903f,     0xbfed9a00dd8b3d46,     //   0.37985,  -0.92505
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //   0.38268,  -0.92388
    0x3fd8ac4b86d5ed44,     0xbfed86c48445a450,     //   0.38552,   -0.9227
    0x3fd8daa52ec8a4af,     0xbfed7d0b02b8ecf9,     //   0.38835,  -0.92151
    0x3fd908ef81ef7bd1,     0xbfed733f508c0dff,     //   0.39117,  -0.92032
    0x3fd9372a63bc93d7,     0xbfed696173c9e68b,     //   0.39399,  -0.91911
    0x3fd96555b7ab948f,     0xbfed5f7172888a7f,     //   0.39681,   -0.9179
    0x3fd993716141bdfe,     0xbfed556f52e93eb1,     //   0.39962,  -0.91668
    0x3fd9c17d440df9f2,     0xbfed4b5b1b187524,     //   0.40243,  -0.91545
    0x3fd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //   0.40524,  -0.91421
    0x3fda1d6543b50ac0,     0xbfed36fc7bcbfbdc,     //   0.40804,  -0.91296
    0x3fda4b4127dea1e4,     0xbfed2cb220e0ef9f,     //   0.41084,  -0.91171
    0x3fda790cd3dbf31a,     0xbfed2255c6e5a4e1,     //   0.41364,  -0.91044
    0x3fdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     //   0.41643,  -0.90917
    0x3fdad473125cdc08,     0xbfed0d672f59d2b9,     //   0.41922,  -0.90789
    0x3fdb020d6c7f4009,     0xbfed02d4feb2bd92,     //     0.422,   -0.9066
    0x3fdb2f971db31972,     0xbfecf830e8ce467b,     //   0.42478,   -0.9053
    0x3fdb5d1009e15cc0,     0xbfeced7af43cc773,     //   0.42756,  -0.90399
    0x3fdb8a7814fd5693,     0xbfece2b32799a060,     //   0.43033,  -0.90267
    0x3fdbb7cf2304bd01,     0xbfecd7d9898b32f6,     //   0.43309,  -0.90135
    0x3fdbe51517ffc0d9,     0xbfecccee20c2de9f,     //   0.43586,  -0.90002
    0x3fdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     //   0.43862,  -0.89867
    0x3fdc3f6d47263129,     0xbfecb6e20a00da99,     //   0.44137,  -0.89732
    0x3fdc6c7f4997000a,     0xbfecabc169a0b901,     //   0.44412,  -0.89597
    0x3fdc997fc3865388,     0xbfeca08f19b9c449,     //   0.44687,   -0.8946
    0x3fdcc66e9931c45d,     0xbfec954b213411f5,     //   0.44961,  -0.89322
    0x3fdcf34baee1cd21,     0xbfec89f587029c13,     //   0.45235,  -0.89184
    0x3fdd2016e8e9db5b,     0xbfec7e8e52233cf3,     //   0.45508,  -0.89045
    0x3fdd4cd02ba8609c,     0xbfec7315899eaad7,     //   0.45781,  -0.88905
    0x3fdd79775b86e389,     0xbfec678b3488739b,     //   0.46054,  -0.88764
    0x3fdda60c5cfa10d8,     0xbfec5bef59fef85a,     //   0.46326,  -0.88622
    0x3fddd28f1481cc58,     0xbfec5042012b6907,     //   0.46598,   -0.8848
    0x3fddfeff66a941de,     0xbfec44833141c004,     //   0.46869,  -0.88336
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //    0.4714,  -0.88192
    0x3fde57a86d3cd824,     0xbfec2cd14931e3f1,     //    0.4741,  -0.88047
    0x3fde83e0eaf85113,     0xbfec20de3fa971b0,     //    0.4768,  -0.87901
    0x3fdeb00695f25620,     0xbfec14d9dc465e58,     //   0.47949,  -0.87755
    0x3fdedc1952ef78d5,     0xbfec08c426725549,     //   0.48218,  -0.87607
    0x3fdf081906bff7fd,     0xbfebfc9d25a1b147,     //   0.48487,  -0.87459
    0x3fdf3405963fd068,     0xbfebf064e15377dd,     //   0.48755,  -0.87309
    0x3fdf5fdee656cda3,     0xbfebe41b611154c1,     //   0.49023,   -0.8716
    0x3fdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //    0.4929,  -0.87009
    0x3fdfb7575c24d2de,     0xbfebcb54cb0d2327,     //   0.49557,  -0.86857
    0x3fdfe2f64be7120f,     0xbfebbed7c49380ea,     //   0.49823,  -0.86705
    0x3fe00740c82b82e0,     0xbfebb249a0b6c40d,     //   0.50089,  -0.86551
    0x3fe01cfc874c3eb7,     0xbfeba5aa673590d2,     //   0.50354,  -0.86397
    0x3fe032ae55edbd95,     0xbfeb98fa1fd9155e,     //   0.50619,  -0.86242
    0x3fe0485626ae221a,     0xbfeb8c38d27504e9,     //   0.50883,  -0.86087
    0x3fe05df3ec31b8b6,     0xbfeb7f6686e792ea,     //   0.51147,   -0.8593
    0x3fe073879922ffed,     0xbfeb728345196e3e,     //    0.5141,  -0.85773
    0x3fe089112032b08c,     0xbfeb658f14fdbc47,     //   0.51673,  -0.85615
    0x3fe09e907417c5e1,     0xbfeb5889fe921405,     //   0.51936,  -0.85456
    0x3fe0b405878f85ec,     0xbfeb4b7409de7925,     //   0.52198,  -0.85296
    0x3fe0c9704d5d898f,     0xbfeb3e4d3ef55712,     //   0.52459,  -0.85136
    0x3fe0ded0b84bc4b5,     0xbfeb3115a5f37bf4,     //    0.5272,  -0.84974
    0x3fe0f426bb2a8e7d,     0xbfeb23cd470013b4,     //    0.5298,  -0.84812
    0x3fe1097248d0a956,     0xbfeb16742a4ca2f5,     //    0.5324,  -0.84649
    0x3fe11eb3541b4b22,     0xbfeb090a58150200,     //     0.535,  -0.84485
    0x3fe133e9cfee254e,     0xbfeafb8fd89f57b6,     //   0.53759,  -0.84321
    0x3fe14915af336ceb,     0xbfeaee04b43c1474,     //   0.54017,  -0.84155
    0x3fe15e36e4dbe2bc,     0xbfeae068f345ecef,     //   0.54275,  -0.83989
    0x3fe1734d63dedb49,     0xbfead2bc9e21d511,     //   0.54532,  -0.83822
    0x3fe188591f3a46e5,     0xbfeac4ffbd3efac8,     //   0.54789,  -0.83655
    0x3fe19d5a09f2b9b8,     0xbfeab7325916c0d4,     //   0.55046,  -0.83486
    0x3fe1b250171373be,     0xbfeaa9547a2cb98e,     //   0.55302,  -0.83317
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //   0.55557,  -0.83147
    0x3fe1dc1b64dc4872,     0xbfea8d676e545ad2,     //   0.55812,  -0.82976
    0x3fe1f0f08bbc861b,     0xbfea7f58529fe69d,     //   0.56066,  -0.82805
    0x3fe205baa17560d6,     0xbfea7138de9d60f5,     //    0.5632,  -0.82632
    0x3fe21a799933eb58,     0xbfea63091b02fae2,     //   0.56573,  -0.82459
    0x3fe22f2d662c13e1,     0xbfea54c91090f524,     //   0.56826,  -0.82285
    0x3fe243d5fb98ac1f,     0xbfea4678c8119ac8,     //   0.57078,   -0.8211
    0x3fe258734cbb7110,     0xbfea38184a593bc6,     //    0.5733,  -0.81935
    0x3fe26d054cdd12df,     0xbfea29a7a0462782,     //   0.57581,  -0.81758
    0x3fe2818bef4d3cba,     0xbfea1b26d2c0a75e,     //   0.57831,  -0.81581
    0x3fe2960727629ca8,     0xbfea0c95eabaf937,     //   0.58081,  -0.81404
    0x3fe2aa76e87aeb58,     0xbfe9fdf4f13149de,     //   0.58331,  -0.81225
    0x3fe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //    0.5858,  -0.81046
    0x3fe2d333d34e9bb7,     0xbfe9e082edb42472,     //   0.58828,  -0.80866
    0x3fe2e780e3e8ea16,     0xbfe9d1b1f5ea80d6,     //   0.59076,  -0.80685
    0x3fe2fbc24b441015,     0xbfe9c2d110f075c3,     //   0.59323,  -0.80503
    0x3fe30ff7fce17035,     0xbfe9b3e047f38741,     //    0.5957,  -0.80321
    0x3fe32421ec49a620,     0xbfe9a4dfa42b06b2,     //   0.59816,  -0.80138
    0x3fe338400d0c8e57,     0xbfe995cf2ed80d22,     //   0.60062,  -0.79954
    0x3fe34c5252c14de1,     0xbfe986aef1457594,     //   0.60307,  -0.79769
    0x3fe36058b10659f3,     0xbfe9777ef4c7d742,     //   0.60551,  -0.79584
    0x3fe374531b817f8d,     0xbfe9683f42bd7fe1,     //   0.60795,  -0.79398
    0x3fe3884185dfeb22,     0xbfe958efe48e6dd7,     //   0.61038,  -0.79211
    0x3fe39c23e3d63029,     0xbfe94990e3ac4a6c,     //   0.61281,  -0.79023
    0x3fe3affa292050b9,     0xbfe93a22499263fc,     //   0.61523,  -0.78835
    0x3fe3c3c44981c517,     0xbfe92aa41fc5a815,     //   0.61765,  -0.78646
    0x3fe3d78238c58343,     0xbfe91b166fd49da2,     //   0.62006,  -0.78456
    0x3fe3eb33eabe0680,     0xbfe90b7943575efe,     //   0.62246,  -0.78265
    0x3fe3fed9534556d4,     0xbfe8fbcca3ef940d,     //   0.62486,  -0.78074
    0x3fe41272663d108c,     0xbfe8ec109b486c49,     //   0.62725,  -0.77882
    0x3fe425ff178e6bb1,     0xbfe8dc45331698cc,     //   0.62964,  -0.77689
    0x3fe4397f5b2a4380,     0xbfe8cc6a75184655,     //   0.63202,  -0.77495
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //   0.63439,  -0.77301
    0x3fe4605a692b32a2,     0xbfe8ac871ede1d88,     //   0.63676,  -0.77106
    0x3fe473b51b987347,     0xbfe89c7e9a4dd4ab,     //   0.63912,   -0.7691
    0x3fe48703306091fe,     0xbfe88c66e7481ba1,     //   0.64148,  -0.76714
    0x3fe49a449b9b0938,     0xbfe87c400fba2ebf,     //   0.64383,  -0.76517
    0x3fe4ad79516722f0,     0xbfe86c0a1d9aa195,     //   0.64618,  -0.76319
    0x3fe4c0a145ec0004,     0xbfe85bc51ae958cc,     //   0.64851,   -0.7612
    0x3fe4d3bc6d589f80,     0xbfe84b7111af83f9,     //   0.65085,  -0.75921
    0x3fe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //   0.65317,  -0.75721
    0x3fe4f9cc25cca486,     0xbfe82a9c13f545ff,     //   0.65549,   -0.7552
    0x3fe50cc09f59a09b,     0xbfe81a1b33b57acc,     //   0.65781,  -0.75319
    0x3fe51fa81cd99aa6,     0xbfe8098b756e52fa,     //   0.66011,  -0.75117
    0x3fe5328292a35596,     0xbfe7f8ece3571771,     //   0.66242,  -0.74914
    0x3fe5454ff5159dfb,     0xbfe7e83f87b03686,     //   0.66471,   -0.7471
    0x3fe5581038975137,     0xbfe7d7836cc33db2,     //     0.667,  -0.74506
    0x3fe56ac35197649e,     0xbfe7c6b89ce2d333,     //   0.66928,  -0.74301
    0x3fe57d69348cec9f,     0xbfe7b5df226aafb0,     //   0.67156,  -0.74095
    0x3fe59001d5f723df,     0xbfe7a4f707bf97d2,     //   0.67383,  -0.73889
    0x3fe5a28d2a5d7250,     0xbfe79400574f55e4,     //   0.67609,  -0.73682
    0x3fe5b50b264f7448,     0xbfe782fb1b90b35b,     //   0.67835,  -0.73474
    0x3fe5c77bbe65018c,     0xbfe771e75f037261,     //    0.6806,  -0.73265
    0x3fe5d9dee73e345c,     0xbfe760c52c304764,     //   0.68285,  -0.73056
    0x3fe5ec3495837074,     0xbfe74f948da8d28d,     //   0.68508,  -0.72846
    0x3fe5fe7cbde56a0f,     0xbfe73e558e079942,     //   0.68732,  -0.72636
    0x3fe610b7551d2cde,     0xbfe72d0837efff97,     //   0.68954,  -0.72425
    0x3fe622e44fec22ff,     0xbfe71bac960e41bf,     //   0.69176,  -0.72213
    0x3fe63503a31c1be8,     0xbfe70a42b3176d7a,     //   0.69397,     -0.72
    0x3fe64715437f535b,     0xbfe6f8ca99c95b75,     //   0.69618,  -0.71787
    0x3fe6591925f0783e,     0xbfe6e74454eaa8ae,     //   0.69838,  -0.71573
    0x3fe66b0f3f52b386,     0xbfe6d5afef4aafcc,     //   0.70057,  -0.71358
    0x3fe67cf78491af10,     0xbfe6c40d73c18275,     //   0.70275,  -0.71143
    0x3fe68ed1eaa19c71,     0xbfe6b25ced2fe29c,     //   0.70493,  -0.70927
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //   0.70711,  -0.70711
    0x3fe6b25ced2fe29c,     0xbfe68ed1eaa19c71,     //   0.70927,  -0.70493
    0x3fe6c40d73c18275,     0xbfe67cf78491af10,     //   0.71143,  -0.70275
    0x3fe6d5afef4aafcc,     0xbfe66b0f3f52b386,     //   0.71358,  -0.70057
    0x3fe6e74454eaa8ae,     0xbfe6591925f0783e,     //   0.71573,  -0.69838
    0x3fe6f8ca99c95b75,     0xbfe64715437f535b,     //   0.71787,  -0.69618
    0x3fe70a42b3176d7a,     0xbfe63503a31c1be8,     //      0.72,  -0.69397
    0x3fe71bac960e41bf,     0xbfe622e44fec22ff,     //   0.72213,  -0.69176
    0x3fe72d0837efff97,     0xbfe610b7551d2cde,     //   0.72425,  -0.68954
    0x3fe73e558e079942,     0xbfe5fe7cbde56a0f,     //   0.72636,  -0.68732
    0x3fe74f948da8d28d,     0xbfe5ec3495837074,     //   0.72846,  -0.68508
    0x3fe760c52c304764,     0xbfe5d9dee73e345c,     //   0.73056,  -0.68285
    0x3fe771e75f037261,     0xbfe5c77bbe65018c,     //   0.73265,   -0.6806
    0x3fe782fb1b90b35b,     0xbfe5b50b264f7448,     //   0.73474,  -0.67835
    0x3fe79400574f55e4,     0xbfe5a28d2a5d7250,     //   0.73682,  -0.67609
    0x3fe7a4f707bf97d2,     0xbfe59001d5f723df,     //   0.73889,  -0.67383
    0x3fe7b5df226aafb0,     0xbfe57d69348cec9f,     //   0.74095,  -0.67156
    0x3fe7c6b89ce2d333,     0xbfe56ac35197649e,     //   0.74301,  -0.66928
    0x3fe7d7836cc33db2,     0xbfe5581038975137,     //   0.74506,    -0.667
    0x3fe7e83f87b03686,     0xbfe5454ff5159dfb,     //    0.7471,  -0.66471
    0x3fe7f8ece3571771,     0xbfe5328292a35596,     //   0.74914,  -0.66242
    0x3fe8098b756e52fa,     0xbfe51fa81cd99aa6,     //   0.75117,  -0.66011
    0x3fe81a1b33b57acc,     0xbfe50cc09f59a09b,     //   0.75319,  -0.65781
    0x3fe82a9c13f545ff,     0xbfe4f9cc25cca486,     //    0.7552,  -0.65549
    0x3fe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //   0.75721,  -0.65317
    0x3fe84b7111af83f9,     0xbfe4d3bc6d589f80,     //   0.75921,  -0.65085
    0x3fe85bc51ae958cc,     0xbfe4c0a145ec0004,     //    0.7612,  -0.64851
    0x3fe86c0a1d9aa195,     0xbfe4ad79516722f0,     //   0.76319,  -0.64618
    0x3fe87c400fba2ebf,     0xbfe49a449b9b0938,     //   0.76517,  -0.64383
    0x3fe88c66e7481ba1,     0xbfe48703306091fe,     //   0.76714,  -0.64148
    0x3fe89c7e9a4dd4ab,     0xbfe473b51b987347,     //    0.7691,  -0.63912
    0x3fe8ac871ede1d88,     0xbfe4605a692b32a2,     //   0.77106,  -0.63676
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //   0.77301,  -0.63439
    0x3fe8cc6a75184655,     0xbfe4397f5b2a4380,     //   0.77495,  -0.63202
    0x3fe8dc45331698cc,     0xbfe425ff178e6bb1,     //   0.77689,  -0.62964
    0x3fe8ec109b486c49,     0xbfe41272663d108c,     //   0.77882,  -0.62725
    0x3fe8fbcca3ef940d,     0xbfe3fed9534556d4,     //   0.78074,  -0.62486
    0x3fe90b7943575efe,     0xbfe3eb33eabe0680,     //   0.78265,  -0.62246
    0x3fe91b166fd49da2,     0xbfe3d78238c58343,     //   0.78456,  -0.62006
    0x3fe92aa41fc5a815,     0xbfe3c3c44981c517,     //   0.78646,  -0.61765
    0x3fe93a22499263fc,     0xbfe3affa292050b9,     //   0.78835,  -0.61523
    0x3fe94990e3ac4a6c,     0xbfe39c23e3d63029,     //   0.79023,  -0.61281
    0x3fe958efe48e6dd7,     0xbfe3884185dfeb22,     //   0.79211,  -0.61038
    0x3fe9683f42bd7fe1,     0xbfe374531b817f8d,     //   0.79398,  -0.60795
    0x3fe9777ef4c7d742,     0xbfe36058b10659f3,     //   0.79584,  -0.60551
    0x3fe986aef1457594,     0xbfe34c5252c14de1,     //   0.79769,  -0.60307
    0x3fe995cf2ed80d22,     0xbfe338400d0c8e57,     //   0.79954,  -0.60062
    0x3fe9a4dfa42b06b2,     0xbfe32421ec49a620,     //   0.80138,  -0.59816
    0x3fe9b3e047f38741,     0xbfe30ff7fce17035,     //   0.80321,   -0.5957
    0x3fe9c2d110f075c3,     0xbfe2fbc24b441015,     //   0.80503,  -0.59323
    0x3fe9d1b1f5ea80d6,     0xbfe2e780e3e8ea16,     //   0.80685,  -0.59076
    0x3fe9e082edb42472,     0xbfe2d333d34e9bb7,     //   0.80866,  -0.58828
    0x3fe9ef43ef29af94,     0xbfe2bedb25faf3ea,     //   0.81046,   -0.5858
    0x3fe9fdf4f13149de,     0xbfe2aa76e87aeb58,     //   0.81225,  -0.58331
    0x3fea0c95eabaf937,     0xbfe2960727629ca8,     //   0.81404,  -0.58081
    0x3fea1b26d2c0a75e,     0xbfe2818bef4d3cba,     //   0.81581,  -0.57831
    0x3fea29a7a0462782,     0xbfe26d054cdd12df,     //   0.81758,  -0.57581
    0x3fea38184a593bc6,     0xbfe258734cbb7110,     //   0.81935,   -0.5733
    0x3fea4678c8119ac8,     0xbfe243d5fb98ac1f,     //    0.8211,  -0.57078
    0x3fea54c91090f524,     0xbfe22f2d662c13e1,     //   0.82285,  -0.56826
    0x3fea63091b02fae2,     0xbfe21a799933eb58,     //   0.82459,  -0.56573
    0x3fea7138de9d60f5,     0xbfe205baa17560d6,     //   0.82632,   -0.5632
    0x3fea7f58529fe69d,     0xbfe1f0f08bbc861b,     //   0.82805,  -0.56066
    0x3fea8d676e545ad2,     0xbfe1dc1b64dc4872,     //   0.82976,  -0.55812
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //   0.83147,  -0.55557
    0x3feaa9547a2cb98e,     0xbfe1b250171373be,     //   0.83317,  -0.55302
    0x3feab7325916c0d4,     0xbfe19d5a09f2b9b8,     //   0.83486,  -0.55046
    0x3feac4ffbd3efac8,     0xbfe188591f3a46e5,     //   0.83655,  -0.54789
    0x3fead2bc9e21d511,     0xbfe1734d63dedb49,     //   0.83822,  -0.54532
    0x3feae068f345ecef,     0xbfe15e36e4dbe2bc,     //   0.83989,  -0.54275
    0x3feaee04b43c1474,     0xbfe14915af336ceb,     //   0.84155,  -0.54017
    0x3feafb8fd89f57b6,     0xbfe133e9cfee254e,     //   0.84321,  -0.53759
    0x3feb090a58150200,     0xbfe11eb3541b4b22,     //   0.84485,    -0.535
    0x3feb16742a4ca2f5,     0xbfe1097248d0a956,     //   0.84649,   -0.5324
    0x3feb23cd470013b4,     0xbfe0f426bb2a8e7d,     //   0.84812,   -0.5298
    0x3feb3115a5f37bf4,     0xbfe0ded0b84bc4b5,     //   0.84974,   -0.5272
    0x3feb3e4d3ef55712,     0xbfe0c9704d5d898f,     //   0.85136,  -0.52459
    0x3feb4b7409de7925,     0xbfe0b405878f85ec,     //   0.85296,  -0.52198
    0x3feb5889fe921405,     0xbfe09e907417c5e1,     //   0.85456,  -0.51936
    0x3feb658f14fdbc47,     0xbfe089112032b08c,     //   0.85615,  -0.51673
    0x3feb728345196e3e,     0xbfe073879922ffed,     //   0.85773,   -0.5141
    0x3feb7f6686e792ea,     0xbfe05df3ec31b8b6,     //    0.8593,  -0.51147
    0x3feb8c38d27504e9,     0xbfe0485626ae221a,     //   0.86087,  -0.50883
    0x3feb98fa1fd9155e,     0xbfe032ae55edbd95,     //   0.86242,  -0.50619
    0x3feba5aa673590d2,     0xbfe01cfc874c3eb7,     //   0.86397,  -0.50354
    0x3febb249a0b6c40d,     0xbfe00740c82b82e0,     //   0.86551,  -0.50089
    0x3febbed7c49380ea,     0xbfdfe2f64be7120f,     //   0.86705,  -0.49823
    0x3febcb54cb0d2327,     0xbfdfb7575c24d2de,     //   0.86857,  -0.49557
    0x3febd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //   0.87009,   -0.4929
    0x3febe41b611154c1,     0xbfdf5fdee656cda3,     //    0.8716,  -0.49023
    0x3febf064e15377dd,     0xbfdf3405963fd068,     //   0.87309,  -0.48755
    0x3febfc9d25a1b147,     0xbfdf081906bff7fd,     //   0.87459,  -0.48487
    0x3fec08c426725549,     0xbfdedc1952ef78d5,     //   0.87607,  -0.48218
    0x3fec14d9dc465e58,     0xbfdeb00695f25620,     //   0.87755,  -0.47949
    0x3fec20de3fa971b0,     0xbfde83e0eaf85113,     //   0.87901,   -0.4768
    0x3fec2cd14931e3f1,     0xbfde57a86d3cd824,     //   0.88047,   -0.4741
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //   0.88192,   -0.4714
    0x3fec44833141c004,     0xbfddfeff66a941de,     //   0.88336,  -0.46869
    0x3fec5042012b6907,     0xbfddd28f1481cc58,     //    0.8848,  -0.46598
    0x3fec5bef59fef85a,     0xbfdda60c5cfa10d8,     //   0.88622,  -0.46326
    0x3fec678b3488739b,     0xbfdd79775b86e389,     //   0.88764,  -0.46054
    0x3fec7315899eaad7,     0xbfdd4cd02ba8609c,     //   0.88905,  -0.45781
    0x3fec7e8e52233cf3,     0xbfdd2016e8e9db5b,     //   0.89045,  -0.45508
    0x3fec89f587029c13,     0xbfdcf34baee1cd21,     //   0.89184,  -0.45235
    0x3fec954b213411f5,     0xbfdcc66e9931c45d,     //   0.89322,  -0.44961
    0x3feca08f19b9c449,     0xbfdc997fc3865388,     //    0.8946,  -0.44687
    0x3fecabc169a0b901,     0xbfdc6c7f4997000a,     //   0.89597,  -0.44412
    0x3fecb6e20a00da99,     0xbfdc3f6d47263129,     //   0.89732,  -0.44137
    0x3fecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     //   0.89867,  -0.43862
    0x3fecccee20c2de9f,     0xbfdbe51517ffc0d9,     //   0.90002,  -0.43586
    0x3fecd7d9898b32f6,     0xbfdbb7cf2304bd01,     //   0.90135,  -0.43309
    0x3fece2b32799a060,     0xbfdb8a7814fd5693,     //   0.90267,  -0.43033
    0x3feced7af43cc773,     0xbfdb5d1009e15cc0,     //   0.90399,  -0.42756
    0x3fecf830e8ce467b,     0xbfdb2f971db31972,     //    0.9053,  -0.42478
    0x3fed02d4feb2bd92,     0xbfdb020d6c7f4009,     //    0.9066,    -0.422
    0x3fed0d672f59d2b9,     0xbfdad473125cdc08,     //   0.90789,  -0.41922
    0x3fed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     //   0.90917,  -0.41643
    0x3fed2255c6e5a4e1,     0xbfda790cd3dbf31a,     //   0.91044,  -0.41364
    0x3fed2cb220e0ef9f,     0xbfda4b4127dea1e4,     //   0.91171,  -0.41084
    0x3fed36fc7bcbfbdc,     0xbfda1d6543b50ac0,     //   0.91296,  -0.40804
    0x3fed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //   0.91421,  -0.40524
    0x3fed4b5b1b187524,     0xbfd9c17d440df9f2,     //   0.91545,  -0.40243
    0x3fed556f52e93eb1,     0xbfd993716141bdfe,     //   0.91668,  -0.39962
    0x3fed5f7172888a7f,     0xbfd96555b7ab948f,     //    0.9179,  -0.39681
    0x3fed696173c9e68b,     0xbfd9372a63bc93d7,     //   0.91911,  -0.39399
    0x3fed733f508c0dff,     0xbfd908ef81ef7bd1,     //   0.92032,  -0.39117
    0x3fed7d0b02b8ecf9,     0xbfd8daa52ec8a4af,     //   0.92151,  -0.38835
    0x3fed86c48445a450,     0xbfd8ac4b86d5ed44,     //    0.9227,  -0.38552
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //   0.92388,  -0.38268
    0x3fed9a00dd8b3d46,     0xbfd84f6aaaf3903f,     //   0.92505,  -0.37985
    0x3feda383a9668988,     0xbfd820e3b04eaac4,     //   0.92621,  -0.37701
    0x3fedacf42ce68ab9,     0xbfd7f24dd37341e3,     //   0.92736,  -0.37416
    0x3fedb6526238a09b,     0xbfd7c3a9311dcce7,     //   0.92851,  -0.37132
    0x3fedbf9e4395759a,     0xbfd794f5e613dfae,     //   0.92964,  -0.36847
    0x3fedc8d7cb410260,     0xbfd766340f2418f6,     //   0.93077,  -0.36561
    0x3fedd1fef38a915a,     0xbfd73763c9261092,     //   0.93188,  -0.36276
    0x3feddb13b6ccc23d,     0xbfd7088530fa459e,     //   0.93299,   -0.3599
    0x3fede4160f6d8d81,     0xbfd6d998638a0cb5,     //   0.93409,  -0.35703
    0x3feded05f7de47da,     0xbfd6aa9d7dc77e16,     //   0.93518,  -0.35416
    0x3fedf5e36a9ba59c,     0xbfd67b949cad63ca,     //   0.93627,  -0.35129
    0x3fedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     //   0.93734,  -0.34842
    0x3fee0766d9280f54,     0xbfd61d595c88c203,     //    0.9384,  -0.34554
    0x3fee100cca2980ac,     0xbfd5ee27379ea693,     //   0.93946,  -0.34266
    0x3fee18a02fdc66d9,     0xbfd5bee78b9db3b6,     //   0.94051,  -0.33978
    0x3fee212104f686e5,     0xbfd58f9a75ab1fdd,     //   0.94154,  -0.33689
    0x3fee298f4439197a,     0xbfd5604012f467b4,     //   0.94257,    -0.334
    0x3fee31eae870ce25,     0xbfd530d880af3c24,     //   0.94359,  -0.33111
    0x3fee3a33ec75ce85,     0xbfd50163dc197047,     //    0.9446,  -0.32821
    0x3fee426a4b2bc17e,     0xbfd4d1e24278e76a,     //   0.94561,  -0.32531
    0x3fee4a8dff81ce5e,     0xbfd4a253d11b82f3,     //    0.9466,  -0.32241
    0x3fee529f04729ffc,     0xbfd472b8a5571054,     //   0.94759,   -0.3195
    0x3fee5a9d550467d3,     0xbfd44310dc8936f0,     //   0.94856,  -0.31659
    0x3fee6288ec48e112,     0xbfd4135c94176602,     //   0.94953,  -0.31368
    0x3fee6a61c55d53a7,     0xbfd3e39be96ec271,     //   0.95049,  -0.31077
    0x3fee7227db6a9744,     0xbfd3b3cefa0414b7,     //   0.95144,  -0.30785
    0x3fee79db29a5165a,     0xbfd383f5e353b6aa,     //   0.95238,  -0.30493
    0x3fee817bab4cd10d,     0xbfd35410c2e18152,     //   0.95331,  -0.30201
    0x3fee89095bad6025,     0xbfd3241fb638baaf,     //   0.95423,  -0.29908
    0x3fee9084361df7f3,     0xbfd2f422daec0386,     //   0.95514,  -0.29615
    0x3fee97ec36016b30,     0xbfd2c41a4e954520,     //   0.95605,  -0.29322
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //   0.95694,  -0.29028
    0x3feea68393e65800,     0xbfd263e6995554ba,     //   0.95783,  -0.28735
    0x3feeadb2e8e7a88e,     0xbfd233bbabc3bb72,     //    0.9587,  -0.28441
    0x3feeb4cf515b8811,     0xbfd2038583d727bd,     //   0.95957,  -0.28146
    0x3feebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     //   0.96043,  -0.27852
    0x3feec2cf4b1af6b2,     0xbfd1a2f7fbe8f243,     //   0.96128,  -0.27557
    0x3feec9b2d3c3bf84,     0xbfd172a0d7765177,     //   0.96212,  -0.27262
    0x3feed0835e999009,     0xbfd1423eefc69378,     //   0.96295,  -0.26967
    0x3feed740e7684963,     0xbfd111d262b1f677,     //   0.96378,  -0.26671
    0x3feeddeb6a078651,     0xbfd0e15b4e1749cd,     //   0.96459,  -0.26375
    0x3feee482e25a9dbc,     0xbfd0b0d9cfdbdb90,     //   0.96539,  -0.26079
    0x3feeeb074c50a544,     0xbfd0804e05eb661e,     //   0.96619,  -0.25783
    0x3feef178a3e473c2,     0xbfd04fb80e37fdae,     //   0.96698,  -0.25487
    0x3feef7d6e51ca3c0,     0xbfd01f1806b9fdd2,     //   0.96775,   -0.2519
    0x3feefe220c0b95ec,     0xbfcfdcdc1adfedf8,     //   0.96852,  -0.24893
    0x3fef045a14cf738c,     0xbfcf7b7480bd3801,     //   0.96928,  -0.24596
    0x3fef0a7efb9230d7,     0xbfcf19f97b215f1a,     //   0.97003,  -0.24298
    0x3fef1090bc898f5f,     0xbfceb86b462de348,     //   0.97077,     -0.24
    0x3fef168f53f7205d,     0xbfce56ca1e101a1b,     //    0.9715,  -0.23702
    0x3fef1c7abe284708,     0xbfcdf5163f01099a,     //   0.97223,  -0.23404
    0x3fef2252f7763ada,     0xbfcd934fe5454311,     //   0.97294,  -0.23106
    0x3fef2817fc4609ce,     0xbfcd31774d2cbdee,     //   0.97364,  -0.22807
    0x3fef2dc9c9089a9d,     0xbfcccf8cb312b286,     //   0.97434,  -0.22508
    0x3fef33685a3aaef0,     0xbfcc6d90535d74dc,     //   0.97503,  -0.22209
    0x3fef38f3ac64e589,     0xbfcc0b826a7e4f63,     //    0.9757,   -0.2191
    0x3fef3e6bbc1bbc65,     0xbfcba96334f15dad,     //   0.97637,  -0.21611
    0x3fef43d085ff92dd,     0xbfcb4732ef3d6722,     //   0.97703,  -0.21311
    0x3fef492206bcabb4,     0xbfcae4f1d5f3b9ab,     //   0.97768,  -0.21011
    0x3fef4e603b0b2f2d,     0xbfca82a025b00451,     //   0.97832,  -0.20711
    0x3fef538b1faf2d07,     0xbfca203e1b1831da,     //   0.97895,  -0.20411
    0x3fef58a2b1789e84,     0xbfc9bdcbf2dc4366,     //   0.97957,   -0.2011
    0x3fef5da6ed43685d,     0xbfc95b49e9b62af9,     //   0.98018,   -0.1981
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //   0.98079,  -0.19509
    0x3fef677556883cee,     0xbfc8961727c41804,     //   0.98138,  -0.19208
    0x3fef6c3f7df5bbb7,     0xbfc83366e89c64c5,     //   0.98196,  -0.18907
    0x3fef70f6434b7eb7,     0xbfc7d0a7bbd2cb1b,     //   0.98254,  -0.18606
    0x3fef7599a3a12077,     0xbfc76dd9de50bf31,     //   0.98311,  -0.18304
    0x3fef7a299c1a322a,     0xbfc70afd8d08c4ff,     //   0.98366,  -0.18002
    0x3fef7ea629e63d6e,     0xbfc6a81304f64ab2,     //   0.98421,    -0.177
    0x3fef830f4a40c60c,     0xbfc6451a831d830d,     //   0.98475,  -0.17398
    0x3fef8764fa714ba9,     0xbfc5e214448b3fc6,     //   0.98528,  -0.17096
    0x3fef8ba737cb4b78,     0xbfc57f008654cbde,     //    0.9858,  -0.16794
    0x3fef8fd5ffae41db,     0xbfc51bdf8597c5f2,     //   0.98631,  -0.16491
    0x3fef93f14f85ac08,     0xbfc4b8b17f79fa88,     //   0.98681,  -0.16189
    0x3fef97f924c9099b,     0xbfc45576b1293e5a,     //    0.9873,  -0.15886
    0x3fef9bed7cfbde29,     0xbfc3f22f57db4893,     //   0.98778,  -0.15583
    0x3fef9fce55adb2c8,     0xbfc38edbb0cd8d14,     //   0.98826,   -0.1528
    0x3fefa39bac7a1791,     0xbfc32b7bf94516a7,     //   0.98872,  -0.14976
    0x3fefa7557f08a517,     0xbfc2c8106e8e613a,     //   0.98918,  -0.14673
    0x3fefaafbcb0cfddc,     0xbfc264994dfd340a,     //   0.98962,   -0.1437
    0x3fefae8e8e46cfbb,     0xbfc20116d4ec7bce,     //   0.99006,  -0.14066
    0x3fefb20dc681d54d,     0xbfc19d8940be24e7,     //   0.99049,  -0.13762
    0x3fefb5797195d741,     0xbfc139f0cedaf576,     //    0.9909,  -0.13458
    0x3fefb8d18d66adb7,     0xbfc0d64dbcb26786,     //   0.99131,  -0.13154
    0x3fefbc1617e44186,     0xbfc072a047ba831d,     //   0.99171,   -0.1285
    0x3fefbf470f0a8d88,     0xbfc00ee8ad6fb85b,     //    0.9921,  -0.12545
    0x3fefc26470e19fd3,     0xbfbf564e56a9730e,     //   0.99248,  -0.12241
    0x3fefc56e3b7d9af6,     0xbfbe8eb7fde4aa3e,     //   0.99285,  -0.11937
    0x3fefc8646cfeb721,     0xbfbdc70ecbae9fc8,     //   0.99321,  -0.11632
    0x3fefcb4703914354,     0xbfbcff533b307dc1,     //   0.99356,  -0.11327
    0x3fefce15fd6da67b,     0xbfbc3785c79ec2d5,     //   0.99391,  -0.11022
    0x3fefd0d158d86087,     0xbfbb6fa6ec38f64c,     //   0.99424,  -0.10717
    0x3fefd37914220b84,     0xbfbaa7b724495c04,     //   0.99456,  -0.10412
    0x3fefd60d2da75c9e,     0xbfb9dfb6eb24a85c,     //   0.99488,  -0.10107
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //   0.99518, -0.098017
    0x3fefdafa7514538c,     0xbfb84f8712c130a0,     //   0.99548, -0.094963
    0x3fefdd539ff1f456,     0xbfb787586a5d5b21,     //   0.99577, -0.091909
    0x3fefdf9922f73307,     0xbfb6bf1b3e79b129,     //   0.99604, -0.088854
    0x3fefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     //   0.99631, -0.085797
    0x3fefe3e92be9d886,     0xbfb52e774a4d4d0a,     //   0.99657,  -0.08274
    0x3fefe5f3af2e3940,     0xbfb4661179272096,     //   0.99682, -0.079682
    0x3fefe7ea85482d60,     0xbfb39d9f12c5a299,     //   0.99706, -0.076624
    0x3fefe9cdad01883a,     0xbfb2d52092ce19f6,     //   0.99729, -0.073565
    0x3fefeb9d2530410f,     0xbfb20c9674ed444c,     //   0.99751, -0.070505
    0x3fefed58ecb673c4,     0xbfb1440134d709b2,     //   0.99772, -0.067444
    0x3fefef0102826191,     0xbfb07b614e463064,     //   0.99793, -0.064383
    0x3feff095658e71ad,     0xbfaf656e79f820e0,     //   0.99812, -0.061321
    0x3feff21614e131ed,     0xbfadd406f9808ec8,     //    0.9983, -0.058258
    0x3feff3830f8d575c,     0xbfac428d12c0d7e3,     //   0.99848, -0.055195
    0x3feff4dc54b1bed3,     0xbfaab101bd5f8317,     //   0.99864, -0.052132
    0x3feff621e3796d7e,     0xbfa91f65f10dd814,     //    0.9988, -0.049068
    0x3feff753bb1b9164,     0xbfa78dbaa5874685,     //   0.99894, -0.046003
    0x3feff871dadb81df,     0xbfa5fc00d290cd43,     //   0.99908, -0.042938
    0x3feff97c4208c014,     0xbfa46a396ff86179,     //    0.9992, -0.039873
    0x3feffa72effef75d,     0xbfa2d865759455cd,     //   0.99932, -0.036807
    0x3feffb55e425fdae,     0xbfa14685db42c17e,     //   0.99943, -0.033741
    0x3feffc251df1d3f8,     0xbf9f693731d1cf01,     //   0.99953, -0.030675
    0x3feffce09ce2a679,     0xbf9c454f4ce53b1c,     //   0.99962, -0.027608
    0x3feffd886084cd0d,     0xbf992155f7a3667e,     //    0.9997, -0.024541
    0x3feffe1c6870cb77,     0xbf95fd4d21fab226,     //   0.99977, -0.021474
    0x3feffe9cb44b51a1,     0xbf92d936bbe30efd,     //   0.99983, -0.018407
    0x3fefff0943c53bd1,     0xbf8f6a296ab997ca,     //   0.99988, -0.015339
    0x3fefff62169b92db,     0xbf8921d1fcdec784,     //   0.99992, -0.012272
    0x3fefffa72c978c4f,     0xbf82d96b0e509703,     //   0.99996,-0.0092038
    0x3fefffd8858e8a92,     0xbf7921f0fe670071,     //   0.99998,-0.0061359
    0x3feffff621621d02,     0xbf6921f8becca4ba,     //         1, -0.003068

};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F64_4096)
/**
  @par
  Example code for Double Precision Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 4096, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const uint64_t twiddleCoefF64_4096[8192] = {
    0x3ff0000000000000,     0x0000000000000000,     //         1,         0
    0x3feffffd88586ee6,     0x3f5921faaee6472d,     //         1,  0.001534
    0x3feffff621621d02,     0x3f6921f8becca4ba,     //         1,  0.003068
    0x3fefffe9cb1e2e8d,     0x3f72d97822f996bc,     //   0.99999, 0.0046019
    0x3fefffd8858e8a92,     0x3f7921f0fe670071,     //   0.99998, 0.0061359
    0x3fefffc250b5daef,     0x3f7f6a65f9a2a3c5,     //   0.99997, 0.0076698
    0x3fefffa72c978c4f,     0x3f82d96b0e509703,     //   0.99996, 0.0092038
    0x3fefff871937ce2f,     0x3f85fda037ac05e0,     //   0.99994,  0.010738
    0x3fefff62169b92db,     0x3f8921d1fcdec784,     //   0.99992,  0.012272
    0x3fefff3824c88f6f,     0x3f8c45ffe1e48ad9,     //    0.9999,  0.013805
    0x3fefff0943c53bd1,     0x3f8f6a296ab997ca,     //   0.99988,  0.015339
    0x3feffed57398d2b7,     0x3f9147270dad7132,     //   0.99986,  0.016873
    0x3feffe9cb44b51a1,     0x3f92d936bbe30efd,     //   0.99983,  0.018407
    0x3feffe5f05e578db,     0x3f946b4381fce81c,     //    0.9998,   0.01994
    0x3feffe1c6870cb77,     0x3f95fd4d21fab226,     //   0.99977,  0.021474
    0x3feffdd4dbf78f52,     0x3f978f535ddc9f03,     //   0.99974,  0.023008
    0x3feffd886084cd0d,     0x3f992155f7a3667e,     //    0.9997,  0.024541
    0x3feffd36f624500c,     0x3f9ab354b1504fca,     //   0.99966,  0.026075
    0x3feffce09ce2a679,     0x3f9c454f4ce53b1c,     //   0.99962,  0.027608
    0x3feffc8554cd213a,     0x3f9dd7458c64ab39,     //   0.99958,  0.029142
    0x3feffc251df1d3f8,     0x3f9f693731d1cf01,     //   0.99953,  0.030675
    0x3feffbbff85f9515,     0x3fa07d91ff984580,     //   0.99948,  0.032208
    0x3feffb55e425fdae,     0x3fa14685db42c17e,     //   0.99943,  0.033741
    0x3feffae6e1556998,     0x3fa20f770ceb11c6,     //   0.99938,  0.035274
    0x3feffa72effef75d,     0x3fa2d865759455cd,     //   0.99932,  0.036807
    0x3feff9fa10348837,     0x3fa3a150f6421afc,     //   0.99926,   0.03834
    0x3feff97c4208c014,     0x3fa46a396ff86179,     //    0.9992,  0.039873
    0x3feff8f9858f058b,     0x3fa5331ec3bba0eb,     //   0.99914,  0.041406
    0x3feff871dadb81df,     0x3fa5fc00d290cd43,     //   0.99908,  0.042938
    0x3feff7e5420320f9,     0x3fa6c4df7d7d5b84,     //   0.99901,  0.044471
    0x3feff753bb1b9164,     0x3fa78dbaa5874685,     //   0.99894,  0.046003
    0x3feff6bd463b444d,     0x3fa856922bb513c1,     //   0.99887,  0.047535
    0x3feff621e3796d7e,     0x3fa91f65f10dd814,     //    0.9988,  0.049068
    0x3feff58192ee0358,     0x3fa9e835d6993c87,     //   0.99872,    0.0506
    0x3feff4dc54b1bed3,     0x3faab101bd5f8317,     //   0.99864,  0.052132
    0x3feff43228de1b77,     0x3fab79c986698b78,     //   0.99856,  0.053664
    0x3feff3830f8d575c,     0x3fac428d12c0d7e3,     //   0.99848,  0.055195
    0x3feff2cf08da7321,     0x3fad0b4c436f91d0,     //   0.99839,  0.056727
    0x3feff21614e131ed,     0x3fadd406f9808ec8,     //    0.9983,  0.058258
    0x3feff15833be1965,     0x3fae9cbd15ff5527,     //   0.99821,   0.05979
    0x3feff095658e71ad,     0x3faf656e79f820e0,     //   0.99812,  0.061321
    0x3fefefcdaa704562,     0x3fb0170d833bf421,     //   0.99802,  0.062852
    0x3fefef0102826191,     0x3fb07b614e463064,     //   0.99793,  0.064383
    0x3fefee2f6de455ba,     0x3fb0dfb28ea201e6,     //   0.99783,  0.065913
    0x3fefed58ecb673c4,     0x3fb1440134d709b2,     //   0.99772,  0.067444
    0x3fefec7d7f19cffc,     0x3fb1a84d316d4f8a,     //   0.99762,  0.068974
    0x3fefeb9d2530410f,     0x3fb20c9674ed444c,     //   0.99751,  0.070505
    0x3fefeab7df1c6005,     0x3fb270dcefdfc45b,     //    0.9974,  0.072035
    0x3fefe9cdad01883a,     0x3fb2d52092ce19f6,     //   0.99729,  0.073565
    0x3fefe8de8f03d75c,     0x3fb339614e41ffa5,     //   0.99718,  0.075094
    0x3fefe7ea85482d60,     0x3fb39d9f12c5a299,     //   0.99706,  0.076624
    0x3fefe6f18ff42c84,     0x3fb401d9d0e3a507,     //   0.99694,  0.078153
    0x3fefe5f3af2e3940,     0x3fb4661179272096,     //   0.99682,  0.079682
    0x3fefe4f0e31d7a4a,     0x3fb4ca45fc1ba8b6,     //    0.9967,  0.081211
    0x3fefe3e92be9d886,     0x3fb52e774a4d4d0a,     //   0.99657,   0.08274
    0x3fefe2dc89bbff08,     0x3fb592a554489bc8,     //   0.99644,  0.084269
    0x3fefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     //   0.99631,  0.085797
    0x3fefe0b485181be3,     0x3fb65af75dd0f87b,     //   0.99618,  0.087326
    0x3fefdf9922f73307,     0x3fb6bf1b3e79b129,     //   0.99604,  0.088854
    0x3fefde78d68653fd,     0x3fb7233b9d236e71,     //   0.99591,  0.090381
    0x3fefdd539ff1f456,     0x3fb787586a5d5b21,     //   0.99577,  0.091909
    0x3fefdc297f674ba9,     0x3fb7eb7196b72ee4,     //   0.99563,  0.093436
    0x3fefdafa7514538c,     0x3fb84f8712c130a0,     //   0.99548,  0.094963
    0x3fefd9c68127c78c,     0x3fb8b398cf0c38e0,     //   0.99533,   0.09649
    0x3fefd88da3d12526,     0x3fb917a6bc29b42c,     //   0.99518,  0.098017
    0x3fefd74fdd40abbf,     0x3fb97bb0caaba56f,     //   0.99503,  0.099544
    0x3fefd60d2da75c9e,     0x3fb9dfb6eb24a85c,     //   0.99488,   0.10107
    0x3fefd4c59536fae4,     0x3fba43b90e27f3c4,     //   0.99472,    0.1026
    0x3fefd37914220b84,     0x3fbaa7b724495c04,     //   0.99456,   0.10412
    0x3fefd227aa9bd53b,     0x3fbb0bb11e1d5559,     //    0.9944,   0.10565
    0x3fefd0d158d86087,     0x3fbb6fa6ec38f64c,     //   0.99424,   0.10717
    0x3fefcf761f0c77a3,     0x3fbbd3987f31fa0e,     //   0.99407,    0.1087
    0x3fefce15fd6da67b,     0x3fbc3785c79ec2d5,     //   0.99391,   0.11022
    0x3fefccb0f4323aa3,     0x3fbc9b6eb6165c42,     //   0.99374,   0.11175
    0x3fefcb4703914354,     0x3fbcff533b307dc1,     //   0.99356,   0.11327
    0x3fefc9d82bc2915e,     0x3fbd633347858ce4,     //   0.99339,   0.11479
    0x3fefc8646cfeb721,     0x3fbdc70ecbae9fc8,     //   0.99321,   0.11632
    0x3fefc6ebc77f0887,     0x3fbe2ae5b8457f77,     //   0.99303,   0.11784
    0x3fefc56e3b7d9af6,     0x3fbe8eb7fde4aa3e,     //   0.99285,   0.11937
    0x3fefc3ebc935454c,     0x3fbef2858d27561b,     //   0.99267,   0.12089
    0x3fefc26470e19fd3,     0x3fbf564e56a9730e,     //   0.99248,   0.12241
    0x3fefc0d832bf043a,     0x3fbfba124b07ad85,     //   0.99229,   0.12393
    0x3fefbf470f0a8d88,     0x3fc00ee8ad6fb85b,     //    0.9921,   0.12545
    0x3fefbdb106021816,     0x3fc040c5bb67747e,     //   0.99191,   0.12698
    0x3fefbc1617e44186,     0x3fc072a047ba831d,     //   0.99171,    0.1285
    0x3fefba7644f068b5,     0x3fc0a4784ab8bf1d,     //   0.99151,   0.13002
    0x3fefb8d18d66adb7,     0x3fc0d64dbcb26786,     //   0.99131,   0.13154
    0x3fefb727f187f1c7,     0x3fc1082095f820b0,     //   0.99111,   0.13306
    0x3fefb5797195d741,     0x3fc139f0cedaf576,     //    0.9909,   0.13458
    0x3fefb3c60dd2c199,     0x3fc16bbe5fac5865,     //    0.9907,    0.1361
    0x3fefb20dc681d54d,     0x3fc19d8940be24e7,     //   0.99049,   0.13762
    0x3fefb0509be6f7db,     0x3fc1cf516a62a077,     //   0.99027,   0.13914
    0x3fefae8e8e46cfbb,     0x3fc20116d4ec7bce,     //   0.99006,   0.14066
    0x3fefacc79de6c44f,     0x3fc232d978aed413,     //   0.98984,   0.14218
    0x3fefaafbcb0cfddc,     0x3fc264994dfd340a,     //   0.98962,    0.1437
    0x3fefa92b1600657c,     0x3fc296564d2b953e,     //    0.9894,   0.14521
    0x3fefa7557f08a517,     0x3fc2c8106e8e613a,     //   0.98918,   0.14673
    0x3fefa57b066e2754,     0x3fc2f9c7aa7a72af,     //   0.98895,   0.14825
    0x3fefa39bac7a1791,     0x3fc32b7bf94516a7,     //   0.98872,   0.14976
    0x3fefa1b7717661d5,     0x3fc35d2d53440db2,     //   0.98849,   0.15128
    0x3fef9fce55adb2c8,     0x3fc38edbb0cd8d14,     //   0.98826,    0.1528
    0x3fef9de0596b77a3,     0x3fc3c0870a383ff6,     //   0.98802,   0.15431
    0x3fef9bed7cfbde29,     0x3fc3f22f57db4893,     //   0.98778,   0.15583
    0x3fef99f5c0abd496,     0x3fc423d4920e4166,     //   0.98754,   0.15734
    0x3fef97f924c9099b,     0x3fc45576b1293e5a,     //    0.9873,   0.15886
    0x3fef95f7a9a1ec47,     0x3fc48715ad84cdf5,     //   0.98706,   0.16037
    0x3fef93f14f85ac08,     0x3fc4b8b17f79fa88,     //   0.98681,   0.16189
    0x3fef91e616c43891,     0x3fc4ea4a1f624b61,     //   0.98656,    0.1634
    0x3fef8fd5ffae41db,     0x3fc51bdf8597c5f2,     //   0.98631,   0.16491
    0x3fef8dc10a95380d,     0x3fc54d71aa74ef02,     //   0.98605,   0.16643
    0x3fef8ba737cb4b78,     0x3fc57f008654cbde,     //    0.9858,   0.16794
    0x3fef898887a36c84,     0x3fc5b08c1192e381,     //   0.98554,   0.16945
    0x3fef8764fa714ba9,     0x3fc5e214448b3fc6,     //   0.98528,   0.17096
    0x3fef853c9089595e,     0x3fc61399179a6e94,     //   0.98501,   0.17247
    0x3fef830f4a40c60c,     0x3fc6451a831d830d,     //   0.98475,   0.17398
    0x3fef80dd27ed8204,     0x3fc676987f7216b8,     //   0.98448,   0.17549
    0x3fef7ea629e63d6e,     0x3fc6a81304f64ab2,     //   0.98421,     0.177
    0x3fef7c6a50826840,     0x3fc6d98a0c08c8da,     //   0.98394,   0.17851
    0x3fef7a299c1a322a,     0x3fc70afd8d08c4ff,     //   0.98366,   0.18002
    0x3fef77e40d068a90,     0x3fc73c6d8055fe0a,     //   0.98339,   0.18153
    0x3fef7599a3a12077,     0x3fc76dd9de50bf31,     //   0.98311,   0.18304
    0x3fef734a60446279,     0x3fc79f429f59e11d,     //   0.98282,   0.18455
    0x3fef70f6434b7eb7,     0x3fc7d0a7bbd2cb1b,     //   0.98254,   0.18606
    0x3fef6e9d4d1262ca,     0x3fc802092c1d744b,     //   0.98225,   0.18756
    0x3fef6c3f7df5bbb7,     0x3fc83366e89c64c5,     //   0.98196,   0.18907
    0x3fef69dcd652f5de,     0x3fc864c0e9b2b6cf,     //   0.98167,   0.19057
    0x3fef677556883cee,     0x3fc8961727c41804,     //   0.98138,   0.19208
    0x3fef6508fef47bd5,     0x3fc8c7699b34ca7e,     //   0.98108,   0.19359
    0x3fef6297cff75cb0,     0x3fc8f8b83c69a60a,     //   0.98079,   0.19509
    0x3fef6021c9f148c2,     0x3fc92a0303c8194f,     //   0.98048,   0.19659
    0x3fef5da6ed43685d,     0x3fc95b49e9b62af9,     //   0.98018,    0.1981
    0x3fef5b273a4fa2d9,     0x3fc98c8ce69a7aec,     //   0.97988,    0.1996
    0x3fef58a2b1789e84,     0x3fc9bdcbf2dc4366,     //   0.97957,    0.2011
    0x3fef56195321c090,     0x3fc9ef0706e35a35,     //   0.97926,   0.20261
    0x3fef538b1faf2d07,     0x3fca203e1b1831da,     //   0.97895,   0.20411
    0x3fef50f81785c6b9,     0x3fca517127e3dabc,     //   0.97863,   0.20561
    0x3fef4e603b0b2f2d,     0x3fca82a025b00451,     //   0.97832,   0.20711
    0x3fef4bc38aa5c694,     0x3fcab3cb0ce6fe44,     //     0.978,   0.20861
    0x3fef492206bcabb4,     0x3fcae4f1d5f3b9ab,     //   0.97768,   0.21011
    0x3fef467bafb7bbe0,     0x3fcb16147941ca2a,     //   0.97735,   0.21161
    0x3fef43d085ff92dd,     0x3fcb4732ef3d6722,     //   0.97703,   0.21311
    0x3fef412089fd8adc,     0x3fcb784d30536cda,     //    0.9767,   0.21461
    0x3fef3e6bbc1bbc65,     0x3fcba96334f15dad,     //   0.97637,   0.21611
    0x3fef3bb21cc4fe47,     0x3fcbda74f5856330,     //   0.97604,    0.2176
    0x3fef38f3ac64e589,     0x3fcc0b826a7e4f63,     //    0.9757,    0.2191
    0x3fef36306b67c556,     0x3fcc3c8b8c4b9dd7,     //   0.97536,    0.2206
    0x3fef33685a3aaef0,     0x3fcc6d90535d74dc,     //   0.97503,   0.22209
    0x3fef309b794b719f,     0x3fcc9e90b824a6a9,     //   0.97468,   0.22359
    0x3fef2dc9c9089a9d,     0x3fcccf8cb312b286,     //   0.97434,   0.22508
    0x3fef2af349e17507,     0x3fcd00843c99c5f9,     //   0.97399,   0.22658
    0x3fef2817fc4609ce,     0x3fcd31774d2cbdee,     //   0.97364,   0.22807
    0x3fef2537e0a71f9f,     0x3fcd6265dd3f27e3,     //   0.97329,   0.22957
    0x3fef2252f7763ada,     0x3fcd934fe5454311,     //   0.97294,   0.23106
    0x3fef1f6941259d7a,     0x3fcdc4355db40195,     //   0.97258,   0.23255
    0x3fef1c7abe284708,     0x3fcdf5163f01099a,     //   0.97223,   0.23404
    0x3fef19876ef1f486,     0x3fce25f281a2b684,     //   0.97187,   0.23553
    0x3fef168f53f7205d,     0x3fce56ca1e101a1b,     //    0.9715,   0.23702
    0x3fef13926dad024e,     0x3fce879d0cc0fdaf,     //   0.97114,   0.23851
    0x3fef1090bc898f5f,     0x3fceb86b462de348,     //   0.97077,      0.24
    0x3fef0d8a410379c5,     0x3fcee934c2d006c7,     //    0.9704,   0.24149
    0x3fef0a7efb9230d7,     0x3fcf19f97b215f1a,     //   0.97003,   0.24298
    0x3fef076eecade0fa,     0x3fcf4ab9679c9f5c,     //   0.96966,   0.24447
    0x3fef045a14cf738c,     0x3fcf7b7480bd3801,     //   0.96928,   0.24596
    0x3fef014074708ed3,     0x3fcfac2abeff57ff,     //    0.9689,   0.24744
    0x3feefe220c0b95ec,     0x3fcfdcdc1adfedf8,     //   0.96852,   0.24893
    0x3feefafedc1ba8b7,     0x3fd006c4466e54af,     //   0.96814,   0.25041
    0x3feef7d6e51ca3c0,     0x3fd01f1806b9fdd2,     //   0.96775,    0.2519
    0x3feef4aa278b2032,     0x3fd037694a928cac,     //   0.96737,   0.25338
    0x3feef178a3e473c2,     0x3fd04fb80e37fdae,     //   0.96698,   0.25487
    0x3feeee425aa6b09a,     0x3fd068044deab002,     //   0.96658,   0.25635
    0x3feeeb074c50a544,     0x3fd0804e05eb661e,     //   0.96619,   0.25783
    0x3feee7c77961dc9e,     0x3fd09895327b465e,     //   0.96579,   0.25931
    0x3feee482e25a9dbc,     0x3fd0b0d9cfdbdb90,     //   0.96539,   0.26079
    0x3feee13987bbebdc,     0x3fd0c91bda4f158d,     //   0.96499,   0.26227
    0x3feeddeb6a078651,     0x3fd0e15b4e1749cd,     //   0.96459,   0.26375
    0x3feeda9889bfe86a,     0x3fd0f998277733f7,     //   0.96418,   0.26523
    0x3feed740e7684963,     0x3fd111d262b1f677,     //   0.96378,   0.26671
    0x3feed3e483849c51,     0x3fd12a09fc0b1b12,     //   0.96337,   0.26819
    0x3feed0835e999009,     0x3fd1423eefc69378,     //   0.96295,   0.26967
    0x3feecd1d792c8f10,     0x3fd15a713a28b9d9,     //   0.96254,   0.27115
    0x3feec9b2d3c3bf84,     0x3fd172a0d7765177,     //   0.96212,   0.27262
    0x3feec6436ee60309,     0x3fd18acdc3f4873a,     //    0.9617,    0.2741
    0x3feec2cf4b1af6b2,     0x3fd1a2f7fbe8f243,     //   0.96128,   0.27557
    0x3feebf5668eaf2ef,     0x3fd1bb1f7b999480,     //   0.96086,   0.27705
    0x3feebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     //   0.96043,   0.27852
    0x3feeb8566b810f2a,     0x3fd1eb6643499fbb,     //      0.96,   0.27999
    0x3feeb4cf515b8811,     0x3fd2038583d727bd,     //   0.95957,   0.28146
    0x3feeb1437af9bb34,     0x3fd21ba1fd3d2623,     //   0.95914,   0.28294
    0x3feeadb2e8e7a88e,     0x3fd233bbabc3bb72,     //    0.9587,   0.28441
    0x3feeaa1d9bb20af3,     0x3fd24bd28bb37672,     //   0.95827,   0.28588
    0x3feea68393e65800,     0x3fd263e6995554ba,     //   0.95783,   0.28735
    0x3feea2e4d212c000,     0x3fd27bf7d0f2c346,     //   0.95738,   0.28882
    0x3fee9f4156c62dda,     0x3fd294062ed59f05,     //   0.95694,   0.29028
    0x3fee9b99229046f8,     0x3fd2ac11af483572,     //   0.95649,   0.29175
    0x3fee97ec36016b30,     0x3fd2c41a4e954520,     //   0.95605,   0.29322
    0x3fee943a91aab4b4,     0x3fd2dc200907fe51,     //   0.95559,   0.29469
    0x3fee9084361df7f3,     0x3fd2f422daec0386,     //   0.95514,   0.29615
    0x3fee8cc923edc388,     0x3fd30c22c08d6a13,     //   0.95469,   0.29762
    0x3fee89095bad6025,     0x3fd3241fb638baaf,     //   0.95423,   0.29908
    0x3fee8544ddf0d075,     0x3fd33c19b83af207,     //   0.95377,   0.30054
    0x3fee817bab4cd10d,     0x3fd35410c2e18152,     //   0.95331,   0.30201
    0x3fee7dadc456d850,     0x3fd36c04d27a4edf,     //   0.95284,   0.30347
    0x3fee79db29a5165a,     0x3fd383f5e353b6aa,     //   0.95238,   0.30493
    0x3fee7603dbce74e9,     0x3fd39be3f1bc8aef,     //   0.95191,   0.30639
    0x3fee7227db6a9744,     0x3fd3b3cefa0414b7,     //   0.95144,   0.30785
    0x3fee6e472911da27,     0x3fd3cbb6f87a146e,     //   0.95096,   0.30931
    0x3fee6a61c55d53a7,     0x3fd3e39be96ec271,     //   0.95049,   0.31077
    0x3fee6677b0e6d31e,     0x3fd3fb7dc932cfa4,     //   0.95001,   0.31222
    0x3fee6288ec48e112,     0x3fd4135c94176602,     //   0.94953,   0.31368
    0x3fee5e95781ebf1c,     0x3fd42b38466e2928,     //   0.94905,   0.31514
    0x3fee5a9d550467d3,     0x3fd44310dc8936f0,     //   0.94856,   0.31659
    0x3fee56a083968eb1,     0x3fd45ae652bb2800,     //   0.94807,   0.31805
    0x3fee529f04729ffc,     0x3fd472b8a5571054,     //   0.94759,    0.3195
    0x3fee4e98d836c0af,     0x3fd48a87d0b07fd7,     //   0.94709,   0.32096
    0x3fee4a8dff81ce5e,     0x3fd4a253d11b82f3,     //    0.9466,   0.32241
    0x3fee467e7af35f23,     0x3fd4ba1ca2eca31c,     //   0.94611,   0.32386
    0x3fee426a4b2bc17e,     0x3fd4d1e24278e76a,     //   0.94561,   0.32531
    0x3fee3e5170cbfc46,     0x3fd4e9a4ac15d520,     //   0.94511,   0.32676
    0x3fee3a33ec75ce85,     0x3fd50163dc197047,     //    0.9446,   0.32821
    0x3fee3611becbaf69,     0x3fd5191fceda3c35,     //    0.9441,   0.32966
    0x3fee31eae870ce25,     0x3fd530d880af3c24,     //   0.94359,   0.33111
    0x3fee2dbf6a0911d9,     0x3fd5488dedeff3be,     //   0.94308,   0.33255
    0x3fee298f4439197a,     0x3fd5604012f467b4,     //   0.94257,     0.334
    0x3fee255a77a63bb8,     0x3fd577eeec151e47,     //   0.94206,   0.33545
    0x3fee212104f686e5,     0x3fd58f9a75ab1fdd,     //   0.94154,   0.33689
    0x3fee1ce2ecd0c0d8,     0x3fd5a742ac0ff78d,     //   0.94103,   0.33833
    0x3fee18a02fdc66d9,     0x3fd5bee78b9db3b6,     //   0.94051,   0.33978
    0x3fee1458cec1ad83,     0x3fd5d68910aee686,     //   0.93998,   0.34122
    0x3fee100cca2980ac,     0x3fd5ee27379ea693,     //   0.93946,   0.34266
    0x3fee0bbc22bd8349,     0x3fd605c1fcc88f63,     //   0.93893,    0.3441
    0x3fee0766d9280f54,     0x3fd61d595c88c203,     //    0.9384,   0.34554
    0x3fee030cee1435b8,     0x3fd634ed533be58e,     //   0.93787,   0.34698
    0x3fedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     //   0.93734,   0.34842
    0x3fedfa4b3621271d,     0x3fd6640af6f03d9e,     //    0.9368,   0.34986
    0x3fedf5e36a9ba59c,     0x3fd67b949cad63ca,     //   0.93627,   0.35129
    0x3fedf177004b2534,     0x3fd6931acad55f51,     //   0.93573,   0.35273
    0x3feded05f7de47da,     0x3fd6aa9d7dc77e16,     //   0.93518,   0.35416
    0x3fede890520465ce,     0x3fd6c21cb1e39771,     //   0.93464,    0.3556
    0x3fede4160f6d8d81,     0x3fd6d998638a0cb5,     //   0.93409,   0.35703
    0x3feddf9730ca837b,     0x3fd6f1108f1bc9c5,     //   0.93354,   0.35846
    0x3feddb13b6ccc23d,     0x3fd7088530fa459e,     //   0.93299,    0.3599
    0x3fedd68ba2267a25,     0x3fd71ff6458782ec,     //   0.93244,   0.36133
    0x3fedd1fef38a915a,     0x3fd73763c9261092,     //   0.93188,   0.36276
    0x3fedcd6dabaca3a5,     0x3fd74ecdb8390a3e,     //   0.93133,   0.36418
    0x3fedc8d7cb410260,     0x3fd766340f2418f6,     //   0.93077,   0.36561
    0x3fedc43d52fcb453,     0x3fd77d96ca4b73a6,     //   0.93021,   0.36704
    0x3fedbf9e4395759a,     0x3fd794f5e613dfae,     //   0.92964,   0.36847
    0x3fedbafa9dc1b78d,     0x3fd7ac515ee2b172,     //   0.92907,   0.36989
    0x3fedb6526238a09b,     0x3fd7c3a9311dcce7,     //   0.92851,   0.37132
    0x3fedb1a591b20c38,     0x3fd7dafd592ba621,     //   0.92794,   0.37274
    0x3fedacf42ce68ab9,     0x3fd7f24dd37341e3,     //   0.92736,   0.37416
    0x3feda83e348f613b,     0x3fd8099a9c5c362d,     //   0.92679,   0.37559
    0x3feda383a9668988,     0x3fd820e3b04eaac4,     //   0.92621,   0.37701
    0x3fed9ec48c26b1f3,     0x3fd838290bb359c8,     //   0.92563,   0.37843
    0x3fed9a00dd8b3d46,     0x3fd84f6aaaf3903f,     //   0.92505,   0.37985
    0x3fed95389e50429b,     0x3fd866a88a792ea0,     //   0.92447,   0.38127
    0x3fed906bcf328d46,     0x3fd87de2a6aea963,     //   0.92388,   0.38268
    0x3fed8b9a70ef9cb4,     0x3fd89518fbff098e,     //   0.92329,    0.3841
    0x3fed86c48445a450,     0x3fd8ac4b86d5ed44,     //    0.9227,   0.38552
    0x3fed81ea09f38b63,     0x3fd8c37a439f884f,     //   0.92211,   0.38693
    0x3fed7d0b02b8ecf9,     0x3fd8daa52ec8a4af,     //   0.92151,   0.38835
    0x3fed78276f5617c6,     0x3fd8f1cc44bea329,     //   0.92092,   0.38976
    0x3fed733f508c0dff,     0x3fd908ef81ef7bd1,     //   0.92032,   0.39117
    0x3fed6e52a71c8547,     0x3fd9200ee2c9be97,     //   0.91972,   0.39258
    0x3fed696173c9e68b,     0x3fd9372a63bc93d7,     //   0.91911,   0.39399
    0x3fed646bb7574de5,     0x3fd94e420137bce3,     //   0.91851,    0.3954
    0x3fed5f7172888a7f,     0x3fd96555b7ab948f,     //    0.9179,   0.39681
    0x3fed5a72a6221e73,     0x3fd97c6583890fc2,     //   0.91729,   0.39822
    0x3fed556f52e93eb1,     0x3fd993716141bdfe,     //   0.91668,   0.39962
    0x3fed506779a3d2d9,     0x3fd9aa794d47c9ee,     //   0.91606,   0.40103
    0x3fed4b5b1b187524,     0x3fd9c17d440df9f2,     //   0.91545,   0.40243
    0x3fed464a380e7242,     0x3fd9d87d4207b0ab,     //   0.91483,   0.40384
    0x3fed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //   0.91421,   0.40524
    0x3fed3c1ae79f2b4e,     0x3fda067145664d57,     //   0.91359,   0.40664
    0x3fed36fc7bcbfbdc,     0x3fda1d6543b50ac0,     //   0.91296,   0.40804
    0x3fed31d98e9e503a,     0x3fda34553b0afee5,     //   0.91234,   0.40944
    0x3fed2cb220e0ef9f,     0x3fda4b4127dea1e4,     //   0.91171,   0.41084
    0x3fed2786335f52fc,     0x3fda622906a70b63,     //   0.91107,   0.41224
    0x3fed2255c6e5a4e1,     0x3fda790cd3dbf31a,     //   0.91044,   0.41364
    0x3fed1d20dc40c15c,     0x3fda8fec8bf5b166,     //   0.90981,   0.41503
    0x3fed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     //   0.90917,   0.41643
    0x3fed12a98fac410c,     0x3fdabd9faebc3980,     //   0.90853,   0.41782
    0x3fed0d672f59d2b9,     0x3fdad473125cdc08,     //   0.90789,   0.41922
    0x3fed082054168bac,     0x3fdaeb4252ca07ab,     //   0.90724,   0.42061
    0x3fed02d4feb2bd92,     0x3fdb020d6c7f4009,     //    0.9066,     0.422
    0x3fecfd852fff6ad4,     0x3fdb18d45bf8aca6,     //   0.90595,   0.42339
    0x3fecf830e8ce467b,     0x3fdb2f971db31972,     //    0.9053,   0.42478
    0x3fecf2d829f1b40e,     0x3fdb4655ae2bf757,     //   0.90464,   0.42617
    0x3feced7af43cc773,     0x3fdb5d1009e15cc0,     //   0.90399,   0.42756
    0x3fece819488344ce,     0x3fdb73c62d520624,     //   0.90333,   0.42894
    0x3fece2b32799a060,     0x3fdb8a7814fd5693,     //   0.90267,   0.43033
    0x3fecdd489254fe65,     0x3fdba125bd63583e,     //   0.90201,   0.43171
    0x3fecd7d9898b32f6,     0x3fdbb7cf2304bd01,     //   0.90135,   0.43309
    0x3fecd2660e12c1e6,     0x3fdbce744262deee,     //   0.90068,   0.43448
    0x3fecccee20c2de9f,     0x3fdbe51517ffc0d9,     //   0.90002,   0.43586
    0x3fecc771c2736c09,     0x3fdbfbb1a05e0edc,     //   0.89935,   0.43724
    0x3fecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     //   0.89867,   0.43862
    0x3fecbc6bb638d10b,     0x3fdc28ddbb6cf145,     //     0.898,   0.43999
    0x3fecb6e20a00da99,     0x3fdc3f6d47263129,     //   0.89732,   0.44137
    0x3fecb153f02fb87d,     0x3fdc55f877b23537,     //   0.89665,   0.44275
    0x3fecabc169a0b901,     0x3fdc6c7f4997000a,     //   0.89597,   0.44412
    0x3feca62a772fd919,     0x3fdc8301b95b40c2,     //   0.89528,    0.4455
    0x3feca08f19b9c449,     0x3fdc997fc3865388,     //    0.8946,   0.44687
    0x3fec9aef521bd480,     0x3fdcaff964a0421d,     //   0.89391,   0.44824
    0x3fec954b213411f5,     0x3fdcc66e9931c45d,     //   0.89322,   0.44961
    0x3fec8fa287e13305,     0x3fdcdcdf5dc440ce,     //   0.89253,   0.45098
    0x3fec89f587029c13,     0x3fdcf34baee1cd21,     //   0.89184,   0.45235
    0x3fec84441f785f61,     0x3fdd09b389152ec1,     //   0.89115,   0.45372
    0x3fec7e8e52233cf3,     0x3fdd2016e8e9db5b,     //   0.89045,   0.45508
    0x3fec78d41fe4a267,     0x3fdd3675caebf962,     //   0.88975,   0.45645
    0x3fec7315899eaad7,     0x3fdd4cd02ba8609c,     //   0.88905,   0.45781
    0x3fec6d5290341eb2,     0x3fdd632607ac9aa9,     //   0.88835,   0.45918
    0x3fec678b3488739b,     0x3fdd79775b86e389,     //   0.88764,   0.46054
    0x3fec61bf777fcc48,     0x3fdd8fc423c62a25,     //   0.88693,    0.4619
    0x3fec5bef59fef85a,     0x3fdda60c5cfa10d8,     //   0.88622,   0.46326
    0x3fec561adceb743e,     0x3fddbc5003b2edf8,     //   0.88551,   0.46462
    0x3fec5042012b6907,     0x3fddd28f1481cc58,     //    0.8848,   0.46598
    0x3fec4a64c7a5ac4c,     0x3fdde8c98bf86bd6,     //   0.88408,   0.46733
    0x3fec44833141c004,     0x3fddfeff66a941de,     //   0.88336,   0.46869
    0x3fec3e9d3ee7d262,     0x3fde1530a12779f4,     //   0.88264,   0.47004
    0x3fec38b2f180bdb1,     0x3fde2b5d3806f63b,     //   0.88192,    0.4714
    0x3fec32c449f60831,     0x3fde418527dc4ffa,     //    0.8812,   0.47275
    0x3fec2cd14931e3f1,     0x3fde57a86d3cd824,     //   0.88047,    0.4741
    0x3fec26d9f01f2eaf,     0x3fde6dc704be97e2,     //   0.87974,   0.47545
    0x3fec20de3fa971b0,     0x3fde83e0eaf85113,     //   0.87901,    0.4768
    0x3fec1ade38bce19b,     0x3fde99f61c817eda,     //   0.87828,   0.47815
    0x3fec14d9dc465e58,     0x3fdeb00695f25620,     //   0.87755,   0.47949
    0x3fec0ed12b3372e9,     0x3fdec61253e3c61b,     //   0.87681,   0.48084
    0x3fec08c426725549,     0x3fdedc1952ef78d5,     //   0.87607,   0.48218
    0x3fec02b2cef1e641,     0x3fdef21b8fafd3b5,     //   0.87533,   0.48353
    0x3febfc9d25a1b147,     0x3fdf081906bff7fd,     //   0.87459,   0.48487
    0x3febf6832b71ec5b,     0x3fdf1e11b4bbc35c,     //   0.87384,   0.48621
    0x3febf064e15377dd,     0x3fdf3405963fd068,     //   0.87309,   0.48755
    0x3febea424837de6d,     0x3fdf49f4a7e97729,     //   0.87235,   0.48889
    0x3febe41b611154c1,     0x3fdf5fdee656cda3,     //    0.8716,   0.49023
    0x3febddf02cd2b983,     0x3fdf75c44e26a852,     //   0.87084,   0.49156
    0x3febd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //   0.87009,    0.4929
    0x3febd18ce0dc19d6,     0x3fdfa1808c6cf7e0,     //   0.86933,   0.49423
    0x3febcb54cb0d2327,     0x3fdfb7575c24d2de,     //   0.86857,   0.49557
    0x3febc5186bf8361d,     0x3fdfcd2947c1ff57,     //   0.86781,    0.4969
    0x3febbed7c49380ea,     0x3fdfe2f64be7120f,     //   0.86705,   0.49823
    0x3febb892d5d5dad5,     0x3fdff8be6537615e,     //   0.86628,   0.49956
    0x3febb249a0b6c40d,     0x3fe00740c82b82e0,     //   0.86551,   0.50089
    0x3febabfc262e6586,     0x3fe0121fe4f56d2c,     //   0.86474,   0.50221
    0x3feba5aa673590d2,     0x3fe01cfc874c3eb7,     //   0.86397,   0.50354
    0x3feb9f5464c5bffc,     0x3fe027d6ad83287e,     //    0.8632,   0.50486
    0x3feb98fa1fd9155e,     0x3fe032ae55edbd95,     //   0.86242,   0.50619
    0x3feb929b996a5b7f,     0x3fe03d837edff370,     //   0.86165,   0.50751
    0x3feb8c38d27504e9,     0x3fe0485626ae221a,     //   0.86087,   0.50883
    0x3feb85d1cbf52c02,     0x3fe053264bad0483,     //   0.86009,   0.51015
    0x3feb7f6686e792ea,     0x3fe05df3ec31b8b6,     //    0.8593,   0.51147
    0x3feb78f70449a34b,     0x3fe068bf0691c028,     //   0.85852,   0.51279
    0x3feb728345196e3e,     0x3fe073879922ffed,     //   0.85773,    0.5141
    0x3feb6c0b4a55ac17,     0x3fe07e4da23bc102,     //   0.85694,   0.51542
    0x3feb658f14fdbc47,     0x3fe089112032b08c,     //   0.85615,   0.51673
    0x3feb5f0ea611a532,     0x3fe093d2115ee018,     //   0.85535,   0.51804
    0x3feb5889fe921405,     0x3fe09e907417c5e1,     //   0.85456,   0.51936
    0x3feb52011f805c92,     0x3fe0a94c46b53d0b,     //   0.85376,   0.52067
    0x3feb4b7409de7925,     0x3fe0b405878f85ec,     //   0.85296,   0.52198
    0x3feb44e2beaf0a61,     0x3fe0bebc34ff4646,     //   0.85216,   0.52328
    0x3feb3e4d3ef55712,     0x3fe0c9704d5d898f,     //   0.85136,   0.52459
    0x3feb37b38bb54c09,     0x3fe0d421cf03c12b,     //   0.85055,    0.5259
    0x3feb3115a5f37bf4,     0x3fe0ded0b84bc4b5,     //   0.84974,    0.5272
    0x3feb2a738eb51f33,     0x3fe0e97d078fd23b,     //   0.84893,    0.5285
    0x3feb23cd470013b4,     0x3fe0f426bb2a8e7d,     //   0.84812,    0.5298
    0x3feb1d22cfdadcc6,     0x3fe0fecdd1770537,     //   0.84731,    0.5311
    0x3feb16742a4ca2f5,     0x3fe1097248d0a956,     //   0.84649,    0.5324
    0x3feb0fc1575d33db,     0x3fe114141f935545,     //   0.84567,    0.5337
    0x3feb090a58150200,     0x3fe11eb3541b4b22,     //   0.84485,     0.535
    0x3feb024f2d7d24a9,     0x3fe1294fe4c5350a,     //   0.84403,   0.53629
    0x3feafb8fd89f57b6,     0x3fe133e9cfee254e,     //   0.84321,   0.53759
    0x3feaf4cc5a85fb73,     0x3fe13e8113f396c1,     //   0.84238,   0.53888
    0x3feaee04b43c1474,     0x3fe14915af336ceb,     //   0.84155,   0.54017
    0x3feae738e6cd4b67,     0x3fe153a7a00bf453,     //   0.84073,   0.54146
    0x3feae068f345ecef,     0x3fe15e36e4dbe2bc,     //   0.83989,   0.54275
    0x3fead994dab2e979,     0x3fe168c37c025764,     //   0.83906,   0.54404
    0x3fead2bc9e21d511,     0x3fe1734d63dedb49,     //   0.83822,   0.54532
    0x3feacbe03ea0e73b,     0x3fe17dd49ad16161,     //   0.83739,   0.54661
    0x3feac4ffbd3efac8,     0x3fe188591f3a46e5,     //   0.83655,   0.54789
    0x3feabe1b1b0b8dac,     0x3fe192daef7a5386,     //   0.83571,   0.54918
    0x3feab7325916c0d4,     0x3fe19d5a09f2b9b8,     //   0.83486,   0.55046
    0x3feab045787157ff,     0x3fe1a7d66d0516e6,     //   0.83402,   0.55174
    0x3feaa9547a2cb98e,     0x3fe1b250171373be,     //   0.83317,   0.55302
    0x3feaa25f5f5aee60,     0x3fe1bcc706804467,     //   0.83232,   0.55429
    0x3fea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //   0.83147,   0.55557
    0x3fea9468d85b20ae,     0x3fe1d1acaf012cc2,     //   0.83062,   0.55685
    0x3fea8d676e545ad2,     0x3fe1dc1b64dc4872,     //   0.82976,   0.55812
    0x3fea8661ec0ee133,     0x3fe1e68759a3e074,     //    0.8289,   0.55939
    0x3fea7f58529fe69d,     0x3fe1f0f08bbc861b,     //   0.82805,   0.56066
    0x3fea784aa31d3f55,     0x3fe1fb56f98b37b8,     //   0.82718,   0.56193
    0x3fea7138de9d60f5,     0x3fe205baa17560d6,     //   0.82632,    0.5632
    0x3fea6a230637623b,     0x3fe2101b81e0da78,     //   0.82546,   0.56447
    0x3fea63091b02fae2,     0x3fe21a799933eb58,     //   0.82459,   0.56573
    0x3fea5beb1e188375,     0x3fe224d4e5d5482e,     //   0.82372,     0.567
    0x3fea54c91090f524,     0x3fe22f2d662c13e1,     //   0.82285,   0.56826
    0x3fea4da2f385e997,     0x3fe23983189fdfd5,     //   0.82198,   0.56952
    0x3fea4678c8119ac8,     0x3fe243d5fb98ac1f,     //    0.8211,   0.57078
    0x3fea3f4a8f4ee2d2,     0x3fe24e260d7ee7c9,     //   0.82023,   0.57204
    0x3fea38184a593bc6,     0x3fe258734cbb7110,     //   0.81935,    0.5733
    0x3fea30e1fa4cbf81,     0x3fe262bdb7b795a2,     //   0.81847,   0.57455
    0x3fea29a7a0462782,     0x3fe26d054cdd12df,     //   0.81758,   0.57581
    0x3fea22693d62ccb9,     0x3fe2774a0a961612,     //    0.8167,   0.57706
    0x3fea1b26d2c0a75e,     0x3fe2818bef4d3cba,     //   0.81581,   0.57831
    0x3fea13e0617e4ec7,     0x3fe28bcaf96d94ba,     //   0.81493,   0.57956
    0x3fea0c95eabaf937,     0x3fe2960727629ca8,     //   0.81404,   0.58081
    0x3fea05476f967bb5,     0x3fe2a040779843fb,     //   0.81314,   0.58206
    0x3fe9fdf4f13149de,     0x3fe2aa76e87aeb58,     //   0.81225,   0.58331
    0x3fe9f69e70ac75bc,     0x3fe2b4aa787764c4,     //   0.81135,   0.58455
    0x3fe9ef43ef29af94,     0x3fe2bedb25faf3ea,     //   0.81046,    0.5858
    0x3fe9e7e56dcb45bd,     0x3fe2c908ef734e57,     //   0.80956,   0.58704
    0x3fe9e082edb42472,     0x3fe2d333d34e9bb7,     //   0.80866,   0.58828
    0x3fe9d91c7007d5a6,     0x3fe2dd5bcffb7616,     //   0.80775,   0.58952
    0x3fe9d1b1f5ea80d6,     0x3fe2e780e3e8ea16,     //   0.80685,   0.59076
    0x3fe9ca438080eadb,     0x3fe2f1a30d86773a,     //   0.80594,     0.592
    0x3fe9c2d110f075c3,     0x3fe2fbc24b441015,     //   0.80503,   0.59323
    0x3fe9bb5aa85f2098,     0x3fe305de9b921a94,     //   0.80412,   0.59447
    0x3fe9b3e047f38741,     0x3fe30ff7fce17035,     //   0.80321,    0.5957
    0x3fe9ac61f0d4e247,     0x3fe31a0e6da35e44,     //   0.80229,   0.59693
    0x3fe9a4dfa42b06b2,     0x3fe32421ec49a620,     //   0.80138,   0.59816
    0x3fe99d59631e65d5,     0x3fe32e3277467d6b,     //   0.80046,   0.59939
    0x3fe995cf2ed80d22,     0x3fe338400d0c8e57,     //   0.79954,   0.60062
    0x3fe98e410881a600,     0x3fe3424aac0ef7d6,     //   0.79861,   0.60184
    0x3fe986aef1457594,     0x3fe34c5252c14de1,     //   0.79769,   0.60307
    0x3fe97f18ea4e5c9e,     0x3fe35656ff9799ae,     //   0.79676,   0.60429
    0x3fe9777ef4c7d742,     0x3fe36058b10659f3,     //   0.79584,   0.60551
    0x3fe96fe111ddfce0,     0x3fe36a576582831b,     //   0.79491,   0.60673
    0x3fe9683f42bd7fe1,     0x3fe374531b817f8d,     //   0.79398,   0.60795
    0x3fe960998893ad8c,     0x3fe37e4bd1792fe2,     //   0.79304,   0.60917
    0x3fe958efe48e6dd7,     0x3fe3884185dfeb22,     //   0.79211,   0.61038
    0x3fe9514257dc4335,     0x3fe39234372c7f04,     //   0.79117,    0.6116
    0x3fe94990e3ac4a6c,     0x3fe39c23e3d63029,     //   0.79023,   0.61281
    0x3fe941db892e3a65,     0x3fe3a6108a54ba58,     //   0.78929,   0.61402
    0x3fe93a22499263fc,     0x3fe3affa292050b9,     //   0.78835,   0.61523
    0x3fe932652609b1cf,     0x3fe3b9e0beb19e18,     //    0.7874,   0.61644
    0x3fe92aa41fc5a815,     0x3fe3c3c44981c517,     //   0.78646,   0.61765
    0x3fe922df37f8646a,     0x3fe3cda4c80a6076,     //   0.78551,   0.61885
    0x3fe91b166fd49da2,     0x3fe3d78238c58343,     //   0.78456,   0.62006
    0x3fe91349c88da398,     0x3fe3e15c9a2db922,     //    0.7836,   0.62126
    0x3fe90b7943575efe,     0x3fe3eb33eabe0680,     //   0.78265,   0.62246
    0x3fe903a4e1665133,     0x3fe3f50828f1e8d2,     //   0.78169,   0.62366
    0x3fe8fbcca3ef940d,     0x3fe3fed9534556d4,     //   0.78074,   0.62486
    0x3fe8f3f08c28d9ac,     0x3fe408a76834c0c0,     //   0.77978,   0.62606
    0x3fe8ec109b486c49,     0x3fe41272663d108c,     //   0.77882,   0.62725
    0x3fe8e42cd2852e0a,     0x3fe41c3a4bdbaa26,     //   0.77785,   0.62845
    0x3fe8dc45331698cc,     0x3fe425ff178e6bb1,     //   0.77689,   0.62964
    0x3fe8d459be34bdfa,     0x3fe42fc0c7d3adbb,     //   0.77592,   0.63083
    0x3fe8cc6a75184655,     0x3fe4397f5b2a4380,     //   0.77495,   0.63202
    0x3fe8c47758fa71cb,     0x3fe4433ad0117b1d,     //   0.77398,   0.63321
    0x3fe8bc806b151741,     0x3fe44cf325091dd6,     //   0.77301,   0.63439
    0x3fe8b485aca2a468,     0x3fe456a858917046,     //   0.77204,   0.63558
    0x3fe8ac871ede1d88,     0x3fe4605a692b32a2,     //   0.77106,   0.63676
    0x3fe8a484c3031d50,     0x3fe46a095557a0f1,     //   0.77008,   0.63794
    0x3fe89c7e9a4dd4ab,     0x3fe473b51b987347,     //    0.7691,   0.63912
    0x3fe89474a5fb0a84,     0x3fe47d5dba6fde01,     //   0.76812,    0.6403
    0x3fe88c66e7481ba1,     0x3fe48703306091fe,     //   0.76714,   0.64148
    0x3fe884555f72fa6b,     0x3fe490a57bedbcdf,     //   0.76615,   0.64266
    0x3fe87c400fba2ebf,     0x3fe49a449b9b0938,     //   0.76517,   0.64383
    0x3fe87426f95cd5bd,     0x3fe4a3e08dec9ed6,     //   0.76418,     0.645
    0x3fe86c0a1d9aa195,     0x3fe4ad79516722f0,     //   0.76319,   0.64618
    0x3fe863e97db3d95a,     0x3fe4b70ee48fb869,     //    0.7622,   0.64735
    0x3fe85bc51ae958cc,     0x3fe4c0a145ec0004,     //    0.7612,   0.64851
    0x3fe8539cf67c9029,     0x3fe4ca30740218a3,     //   0.76021,   0.64968
    0x3fe84b7111af83f9,     0x3fe4d3bc6d589f80,     //   0.75921,   0.65085
    0x3fe843416dc4cce2,     0x3fe4dd453076b064,     //   0.75821,   0.65201
    0x3fe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //   0.75721,   0.65317
    0x3fe832d6eda3a3e0,     0x3fe4f04d0e2859aa,     //   0.75621,   0.65433
    0x3fe82a9c13f545ff,     0x3fe4f9cc25cca486,     //    0.7552,   0.65549
    0x3fe8225d803964e5,     0x3fe503480159ded2,     //   0.75419,   0.65665
    0x3fe81a1b33b57acc,     0x3fe50cc09f59a09b,     //   0.75319,   0.65781
    0x3fe811d52faf94dc,     0x3fe51635fe5601d7,     //   0.75218,   0.65896
    0x3fe8098b756e52fa,     0x3fe51fa81cd99aa6,     //   0.75117,   0.66011
    0x3fe8013e0638e795,     0x3fe52916f96f8388,     //   0.75015,   0.66127
    0x3fe7f8ece3571771,     0x3fe5328292a35596,     //   0.74914,   0.66242
    0x3fe7f0980e113978,     0x3fe53beae7012abe,     //   0.74812,   0.66356
    0x3fe7e83f87b03686,     0x3fe5454ff5159dfb,     //    0.7471,   0.66471
    0x3fe7dfe3517d8937,     0x3fe54eb1bb6dcb8f,     //   0.74608,   0.66586
    0x3fe7d7836cc33db2,     0x3fe5581038975137,     //   0.74506,     0.667
    0x3fe7cf1fdacbf179,     0x3fe5616b6b204e6e,     //   0.74403,   0.66814
    0x3fe7c6b89ce2d333,     0x3fe56ac35197649e,     //   0.74301,   0.66928
    0x3fe7be4db453a27c,     0x3fe57417ea8bb75c,     //   0.74198,   0.67042
    0x3fe7b5df226aafb0,     0x3fe57d69348cec9f,     //   0.74095,   0.67156
    0x3fe7ad6ce874dbb6,     0x3fe586b72e2b2cfd,     //   0.73992,   0.67269
    0x3fe7a4f707bf97d2,     0x3fe59001d5f723df,     //   0.73889,   0.67383
    0x3fe79c7d8198e56e,     0x3fe599492a81ffbc,     //   0.73785,   0.67496
    0x3fe79400574f55e4,     0x3fe5a28d2a5d7250,     //   0.73682,   0.67609
    0x3fe78b7f8a320a52,     0x3fe5abcdd41bb0d8,     //   0.73578,   0.67722
    0x3fe782fb1b90b35b,     0x3fe5b50b264f7448,     //   0.73474,   0.67835
    0x3fe77a730cbb9100,     0x3fe5be451f8bf980,     //    0.7337,   0.67948
    0x3fe771e75f037261,     0x3fe5c77bbe65018c,     //   0.73265,    0.6806
    0x3fe7695813b9b594,     0x3fe5d0af016ed1d4,     //   0.73161,   0.68172
    0x3fe760c52c304764,     0x3fe5d9dee73e345c,     //   0.73056,   0.68285
    0x3fe7582ea9b9a329,     0x3fe5e30b6e6877f3,     //   0.72951,   0.68397
    0x3fe74f948da8d28d,     0x3fe5ec3495837074,     //   0.72846,   0.68508
    0x3fe746f6d9516d59,     0x3fe5f55a5b2576f8,     //   0.72741,    0.6862
    0x3fe73e558e079942,     0x3fe5fe7cbde56a0f,     //   0.72636,   0.68732
    0x3fe735b0ad2009b2,     0x3fe6079bbc5aadfa,     //    0.7253,   0.68843
    0x3fe72d0837efff97,     0x3fe610b7551d2cde,     //   0.72425,   0.68954
    0x3fe7245c2fcd492a,     0x3fe619cf86c55702,     //   0.72319,   0.69065
    0x3fe71bac960e41bf,     0x3fe622e44fec22ff,     //   0.72213,   0.69176
    0x3fe712f96c09d18d,     0x3fe62bf5af2b0dfd,     //   0.72107,   0.69287
    0x3fe70a42b3176d7a,     0x3fe63503a31c1be8,     //      0.72,   0.69397
    0x3fe701886c8f16e6,     0x3fe63e0e2a59d7aa,     //   0.71894,   0.69508
    0x3fe6f8ca99c95b75,     0x3fe64715437f535b,     //   0.71787,   0.69618
    0x3fe6f0093c1f54de,     0x3fe65018ed28287f,     //    0.7168,   0.69728
    0x3fe6e74454eaa8ae,     0x3fe6591925f0783e,     //   0.71573,   0.69838
    0x3fe6de7be585881d,     0x3fe66215ec74eb91,     //   0.71466,   0.69947
    0x3fe6d5afef4aafcc,     0x3fe66b0f3f52b386,     //   0.71358,   0.70057
    0x3fe6cce07395679f,     0x3fe674051d27896c,     //   0.71251,   0.70166
    0x3fe6c40d73c18275,     0x3fe67cf78491af10,     //   0.71143,   0.70275
    0x3fe6bb36f12b5e06,     0x3fe685e6742feeef,     //   0.71035,   0.70385
    0x3fe6b25ced2fe29c,     0x3fe68ed1eaa19c71,     //   0.70927,   0.70493
    0x3fe6a97f692c82ea,     0x3fe697b9e686941c,     //   0.70819,   0.70602
    0x3fe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //   0.70711,   0.70711
    0x3fe697b9e686941c,     0x3fe6a97f692c82ea,     //   0.70602,   0.70819
    0x3fe68ed1eaa19c71,     0x3fe6b25ced2fe29c,     //   0.70493,   0.70927
    0x3fe685e6742feeef,     0x3fe6bb36f12b5e06,     //   0.70385,   0.71035
    0x3fe67cf78491af10,     0x3fe6c40d73c18275,     //   0.70275,   0.71143
    0x3fe674051d27896c,     0x3fe6cce07395679f,     //   0.70166,   0.71251
    0x3fe66b0f3f52b386,     0x3fe6d5afef4aafcc,     //   0.70057,   0.71358
    0x3fe66215ec74eb91,     0x3fe6de7be585881d,     //   0.69947,   0.71466
    0x3fe6591925f0783e,     0x3fe6e74454eaa8ae,     //   0.69838,   0.71573
    0x3fe65018ed28287f,     0x3fe6f0093c1f54de,     //   0.69728,    0.7168
    0x3fe64715437f535b,     0x3fe6f8ca99c95b75,     //   0.69618,   0.71787
    0x3fe63e0e2a59d7aa,     0x3fe701886c8f16e6,     //   0.69508,   0.71894
    0x3fe63503a31c1be8,     0x3fe70a42b3176d7a,     //   0.69397,      0.72
    0x3fe62bf5af2b0dfd,     0x3fe712f96c09d18d,     //   0.69287,   0.72107
    0x3fe622e44fec22ff,     0x3fe71bac960e41bf,     //   0.69176,   0.72213
    0x3fe619cf86c55702,     0x3fe7245c2fcd492a,     //   0.69065,   0.72319
    0x3fe610b7551d2cde,     0x3fe72d0837efff97,     //   0.68954,   0.72425
    0x3fe6079bbc5aadfa,     0x3fe735b0ad2009b2,     //   0.68843,    0.7253
    0x3fe5fe7cbde56a0f,     0x3fe73e558e079942,     //   0.68732,   0.72636
    0x3fe5f55a5b2576f8,     0x3fe746f6d9516d59,     //    0.6862,   0.72741
    0x3fe5ec3495837074,     0x3fe74f948da8d28d,     //   0.68508,   0.72846
    0x3fe5e30b6e6877f3,     0x3fe7582ea9b9a329,     //   0.68397,   0.72951
    0x3fe5d9dee73e345c,     0x3fe760c52c304764,     //   0.68285,   0.73056
    0x3fe5d0af016ed1d4,     0x3fe7695813b9b594,     //   0.68172,   0.73161
    0x3fe5c77bbe65018c,     0x3fe771e75f037261,     //    0.6806,   0.73265
    0x3fe5be451f8bf980,     0x3fe77a730cbb9100,     //   0.67948,    0.7337
    0x3fe5b50b264f7448,     0x3fe782fb1b90b35b,     //   0.67835,   0.73474
    0x3fe5abcdd41bb0d8,     0x3fe78b7f8a320a52,     //   0.67722,   0.73578
    0x3fe5a28d2a5d7250,     0x3fe79400574f55e4,     //   0.67609,   0.73682
    0x3fe599492a81ffbc,     0x3fe79c7d8198e56e,     //   0.67496,   0.73785
    0x3fe59001d5f723df,     0x3fe7a4f707bf97d2,     //   0.67383,   0.73889
    0x3fe586b72e2b2cfd,     0x3fe7ad6ce874dbb6,     //   0.67269,   0.73992
    0x3fe57d69348cec9f,     0x3fe7b5df226aafb0,     //   0.67156,   0.74095
    0x3fe57417ea8bb75c,     0x3fe7be4db453a27c,     //   0.67042,   0.74198
    0x3fe56ac35197649e,     0x3fe7c6b89ce2d333,     //   0.66928,   0.74301
    0x3fe5616b6b204e6e,     0x3fe7cf1fdacbf179,     //   0.66814,   0.74403
    0x3fe5581038975137,     0x3fe7d7836cc33db2,     //     0.667,   0.74506
    0x3fe54eb1bb6dcb8f,     0x3fe7dfe3517d8937,     //   0.66586,   0.74608
    0x3fe5454ff5159dfb,     0x3fe7e83f87b03686,     //   0.66471,    0.7471
    0x3fe53beae7012abe,     0x3fe7f0980e113978,     //   0.66356,   0.74812
    0x3fe5328292a35596,     0x3fe7f8ece3571771,     //   0.66242,   0.74914
    0x3fe52916f96f8388,     0x3fe8013e0638e795,     //   0.66127,   0.75015
    0x3fe51fa81cd99aa6,     0x3fe8098b756e52fa,     //   0.66011,   0.75117
    0x3fe51635fe5601d7,     0x3fe811d52faf94dc,     //   0.65896,   0.75218
    0x3fe50cc09f59a09b,     0x3fe81a1b33b57acc,     //   0.65781,   0.75319
    0x3fe503480159ded2,     0x3fe8225d803964e5,     //   0.65665,   0.75419
    0x3fe4f9cc25cca486,     0x3fe82a9c13f545ff,     //   0.65549,    0.7552
    0x3fe4f04d0e2859aa,     0x3fe832d6eda3a3e0,     //   0.65433,   0.75621
    0x3fe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //   0.65317,   0.75721
    0x3fe4dd453076b064,     0x3fe843416dc4cce2,     //   0.65201,   0.75821
    0x3fe4d3bc6d589f80,     0x3fe84b7111af83f9,     //   0.65085,   0.75921
    0x3fe4ca30740218a3,     0x3fe8539cf67c9029,     //   0.64968,   0.76021
    0x3fe4c0a145ec0004,     0x3fe85bc51ae958cc,     //   0.64851,    0.7612
    0x3fe4b70ee48fb869,     0x3fe863e97db3d95a,     //   0.64735,    0.7622
    0x3fe4ad79516722f0,     0x3fe86c0a1d9aa195,     //   0.64618,   0.76319
    0x3fe4a3e08dec9ed6,     0x3fe87426f95cd5bd,     //     0.645,   0.76418
    0x3fe49a449b9b0938,     0x3fe87c400fba2ebf,     //   0.64383,   0.76517
    0x3fe490a57bedbcdf,     0x3fe884555f72fa6b,     //   0.64266,   0.76615
    0x3fe48703306091fe,     0x3fe88c66e7481ba1,     //   0.64148,   0.76714
    0x3fe47d5dba6fde01,     0x3fe89474a5fb0a84,     //    0.6403,   0.76812
    0x3fe473b51b987347,     0x3fe89c7e9a4dd4ab,     //   0.63912,    0.7691
    0x3fe46a095557a0f1,     0x3fe8a484c3031d50,     //   0.63794,   0.77008
    0x3fe4605a692b32a2,     0x3fe8ac871ede1d88,     //   0.63676,   0.77106
    0x3fe456a858917046,     0x3fe8b485aca2a468,     //   0.63558,   0.77204
    0x3fe44cf325091dd6,     0x3fe8bc806b151741,     //   0.63439,   0.77301
    0x3fe4433ad0117b1d,     0x3fe8c47758fa71cb,     //   0.63321,   0.77398
    0x3fe4397f5b2a4380,     0x3fe8cc6a75184655,     //   0.63202,   0.77495
    0x3fe42fc0c7d3adbb,     0x3fe8d459be34bdfa,     //   0.63083,   0.77592
    0x3fe425ff178e6bb1,     0x3fe8dc45331698cc,     //   0.62964,   0.77689
    0x3fe41c3a4bdbaa26,     0x3fe8e42cd2852e0a,     //   0.62845,   0.77785
    0x3fe41272663d108c,     0x3fe8ec109b486c49,     //   0.62725,   0.77882
    0x3fe408a76834c0c0,     0x3fe8f3f08c28d9ac,     //   0.62606,   0.77978
    0x3fe3fed9534556d4,     0x3fe8fbcca3ef940d,     //   0.62486,   0.78074
    0x3fe3f50828f1e8d2,     0x3fe903a4e1665133,     //   0.62366,   0.78169
    0x3fe3eb33eabe0680,     0x3fe90b7943575efe,     //   0.62246,   0.78265
    0x3fe3e15c9a2db922,     0x3fe91349c88da398,     //   0.62126,    0.7836
    0x3fe3d78238c58343,     0x3fe91b166fd49da2,     //   0.62006,   0.78456
    0x3fe3cda4c80a6076,     0x3fe922df37f8646a,     //   0.61885,   0.78551
    0x3fe3c3c44981c517,     0x3fe92aa41fc5a815,     //   0.61765,   0.78646
    0x3fe3b9e0beb19e18,     0x3fe932652609b1cf,     //   0.61644,    0.7874
    0x3fe3affa292050b9,     0x3fe93a22499263fc,     //   0.61523,   0.78835
    0x3fe3a6108a54ba58,     0x3fe941db892e3a65,     //   0.61402,   0.78929
    0x3fe39c23e3d63029,     0x3fe94990e3ac4a6c,     //   0.61281,   0.79023
    0x3fe39234372c7f04,     0x3fe9514257dc4335,     //    0.6116,   0.79117
    0x3fe3884185dfeb22,     0x3fe958efe48e6dd7,     //   0.61038,   0.79211
    0x3fe37e4bd1792fe2,     0x3fe960998893ad8c,     //   0.60917,   0.79304
    0x3fe374531b817f8d,     0x3fe9683f42bd7fe1,     //   0.60795,   0.79398
    0x3fe36a576582831b,     0x3fe96fe111ddfce0,     //   0.60673,   0.79491
    0x3fe36058b10659f3,     0x3fe9777ef4c7d742,     //   0.60551,   0.79584
    0x3fe35656ff9799ae,     0x3fe97f18ea4e5c9e,     //   0.60429,   0.79676
    0x3fe34c5252c14de1,     0x3fe986aef1457594,     //   0.60307,   0.79769
    0x3fe3424aac0ef7d6,     0x3fe98e410881a600,     //   0.60184,   0.79861
    0x3fe338400d0c8e57,     0x3fe995cf2ed80d22,     //   0.60062,   0.79954
    0x3fe32e3277467d6b,     0x3fe99d59631e65d5,     //   0.59939,   0.80046
    0x3fe32421ec49a620,     0x3fe9a4dfa42b06b2,     //   0.59816,   0.80138
    0x3fe31a0e6da35e44,     0x3fe9ac61f0d4e247,     //   0.59693,   0.80229
    0x3fe30ff7fce17035,     0x3fe9b3e047f38741,     //    0.5957,   0.80321
    0x3fe305de9b921a94,     0x3fe9bb5aa85f2098,     //   0.59447,   0.80412
    0x3fe2fbc24b441015,     0x3fe9c2d110f075c3,     //   0.59323,   0.80503
    0x3fe2f1a30d86773a,     0x3fe9ca438080eadb,     //     0.592,   0.80594
    0x3fe2e780e3e8ea16,     0x3fe9d1b1f5ea80d6,     //   0.59076,   0.80685
    0x3fe2dd5bcffb7616,     0x3fe9d91c7007d5a6,     //   0.58952,   0.80775
    0x3fe2d333d34e9bb7,     0x3fe9e082edb42472,     //   0.58828,   0.80866
    0x3fe2c908ef734e57,     0x3fe9e7e56dcb45bd,     //   0.58704,   0.80956
    0x3fe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //    0.5858,   0.81046
    0x3fe2b4aa787764c4,     0x3fe9f69e70ac75bc,     //   0.58455,   0.81135
    0x3fe2aa76e87aeb58,     0x3fe9fdf4f13149de,     //   0.58331,   0.81225
    0x3fe2a040779843fb,     0x3fea05476f967bb5,     //   0.58206,   0.81314
    0x3fe2960727629ca8,     0x3fea0c95eabaf937,     //   0.58081,   0.81404
    0x3fe28bcaf96d94ba,     0x3fea13e0617e4ec7,     //   0.57956,   0.81493
    0x3fe2818bef4d3cba,     0x3fea1b26d2c0a75e,     //   0.57831,   0.81581
    0x3fe2774a0a961612,     0x3fea22693d62ccb9,     //   0.57706,    0.8167
    0x3fe26d054cdd12df,     0x3fea29a7a0462782,     //   0.57581,   0.81758
    0x3fe262bdb7b795a2,     0x3fea30e1fa4cbf81,     //   0.57455,   0.81847
    0x3fe258734cbb7110,     0x3fea38184a593bc6,     //    0.5733,   0.81935
    0x3fe24e260d7ee7c9,     0x3fea3f4a8f4ee2d2,     //   0.57204,   0.82023
    0x3fe243d5fb98ac1f,     0x3fea4678c8119ac8,     //   0.57078,    0.8211
    0x3fe23983189fdfd5,     0x3fea4da2f385e997,     //   0.56952,   0.82198
    0x3fe22f2d662c13e1,     0x3fea54c91090f524,     //   0.56826,   0.82285
    0x3fe224d4e5d5482e,     0x3fea5beb1e188375,     //     0.567,   0.82372
    0x3fe21a799933eb58,     0x3fea63091b02fae2,     //   0.56573,   0.82459
    0x3fe2101b81e0da78,     0x3fea6a230637623b,     //   0.56447,   0.82546
    0x3fe205baa17560d6,     0x3fea7138de9d60f5,     //    0.5632,   0.82632
    0x3fe1fb56f98b37b8,     0x3fea784aa31d3f55,     //   0.56193,   0.82718
    0x3fe1f0f08bbc861b,     0x3fea7f58529fe69d,     //   0.56066,   0.82805
    0x3fe1e68759a3e074,     0x3fea8661ec0ee133,     //   0.55939,    0.8289
    0x3fe1dc1b64dc4872,     0x3fea8d676e545ad2,     //   0.55812,   0.82976
    0x3fe1d1acaf012cc2,     0x3fea9468d85b20ae,     //   0.55685,   0.83062
    0x3fe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //   0.55557,   0.83147
    0x3fe1bcc706804467,     0x3feaa25f5f5aee60,     //   0.55429,   0.83232
    0x3fe1b250171373be,     0x3feaa9547a2cb98e,     //   0.55302,   0.83317
    0x3fe1a7d66d0516e6,     0x3feab045787157ff,     //   0.55174,   0.83402
    0x3fe19d5a09f2b9b8,     0x3feab7325916c0d4,     //   0.55046,   0.83486
    0x3fe192daef7a5386,     0x3feabe1b1b0b8dac,     //   0.54918,   0.83571
    0x3fe188591f3a46e5,     0x3feac4ffbd3efac8,     //   0.54789,   0.83655
    0x3fe17dd49ad16161,     0x3feacbe03ea0e73b,     //   0.54661,   0.83739
    0x3fe1734d63dedb49,     0x3fead2bc9e21d511,     //   0.54532,   0.83822
    0x3fe168c37c025764,     0x3fead994dab2e979,     //   0.54404,   0.83906
    0x3fe15e36e4dbe2bc,     0x3feae068f345ecef,     //   0.54275,   0.83989
    0x3fe153a7a00bf453,     0x3feae738e6cd4b67,     //   0.54146,   0.84073
    0x3fe14915af336ceb,     0x3feaee04b43c1474,     //   0.54017,   0.84155
    0x3fe13e8113f396c1,     0x3feaf4cc5a85fb73,     //   0.53888,   0.84238
    0x3fe133e9cfee254e,     0x3feafb8fd89f57b6,     //   0.53759,   0.84321
    0x3fe1294fe4c5350a,     0x3feb024f2d7d24a9,     //   0.53629,   0.84403
    0x3fe11eb3541b4b22,     0x3feb090a58150200,     //     0.535,   0.84485
    0x3fe114141f935545,     0x3feb0fc1575d33db,     //    0.5337,   0.84567
    0x3fe1097248d0a956,     0x3feb16742a4ca2f5,     //    0.5324,   0.84649
    0x3fe0fecdd1770537,     0x3feb1d22cfdadcc6,     //    0.5311,   0.84731
    0x3fe0f426bb2a8e7d,     0x3feb23cd470013b4,     //    0.5298,   0.84812
    0x3fe0e97d078fd23b,     0x3feb2a738eb51f33,     //    0.5285,   0.84893
    0x3fe0ded0b84bc4b5,     0x3feb3115a5f37bf4,     //    0.5272,   0.84974
    0x3fe0d421cf03c12b,     0x3feb37b38bb54c09,     //    0.5259,   0.85055
    0x3fe0c9704d5d898f,     0x3feb3e4d3ef55712,     //   0.52459,   0.85136
    0x3fe0bebc34ff4646,     0x3feb44e2beaf0a61,     //   0.52328,   0.85216
    0x3fe0b405878f85ec,     0x3feb4b7409de7925,     //   0.52198,   0.85296
    0x3fe0a94c46b53d0b,     0x3feb52011f805c92,     //   0.52067,   0.85376
    0x3fe09e907417c5e1,     0x3feb5889fe921405,     //   0.51936,   0.85456
    0x3fe093d2115ee018,     0x3feb5f0ea611a532,     //   0.51804,   0.85535
    0x3fe089112032b08c,     0x3feb658f14fdbc47,     //   0.51673,   0.85615
    0x3fe07e4da23bc102,     0x3feb6c0b4a55ac17,     //   0.51542,   0.85694
    0x3fe073879922ffed,     0x3feb728345196e3e,     //    0.5141,   0.85773
    0x3fe068bf0691c028,     0x3feb78f70449a34b,     //   0.51279,   0.85852
    0x3fe05df3ec31b8b6,     0x3feb7f6686e792ea,     //   0.51147,    0.8593
    0x3fe053264bad0483,     0x3feb85d1cbf52c02,     //   0.51015,   0.86009
    0x3fe0485626ae221a,     0x3feb8c38d27504e9,     //   0.50883,   0.86087
    0x3fe03d837edff370,     0x3feb929b996a5b7f,     //   0.50751,   0.86165
    0x3fe032ae55edbd95,     0x3feb98fa1fd9155e,     //   0.50619,   0.86242
    0x3fe027d6ad83287e,     0x3feb9f5464c5bffc,     //   0.50486,    0.8632
    0x3fe01cfc874c3eb7,     0x3feba5aa673590d2,     //   0.50354,   0.86397
    0x3fe0121fe4f56d2c,     0x3febabfc262e6586,     //   0.50221,   0.86474
    0x3fe00740c82b82e0,     0x3febb249a0b6c40d,     //   0.50089,   0.86551
    0x3fdff8be6537615e,     0x3febb892d5d5dad5,     //   0.49956,   0.86628
    0x3fdfe2f64be7120f,     0x3febbed7c49380ea,     //   0.49823,   0.86705
    0x3fdfcd2947c1ff57,     0x3febc5186bf8361d,     //    0.4969,   0.86781
    0x3fdfb7575c24d2de,     0x3febcb54cb0d2327,     //   0.49557,   0.86857
    0x3fdfa1808c6cf7e0,     0x3febd18ce0dc19d6,     //   0.49423,   0.86933
    0x3fdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //    0.4929,   0.87009
    0x3fdf75c44e26a852,     0x3febddf02cd2b983,     //   0.49156,   0.87084
    0x3fdf5fdee656cda3,     0x3febe41b611154c1,     //   0.49023,    0.8716
    0x3fdf49f4a7e97729,     0x3febea424837de6d,     //   0.48889,   0.87235
    0x3fdf3405963fd068,     0x3febf064e15377dd,     //   0.48755,   0.87309
    0x3fdf1e11b4bbc35c,     0x3febf6832b71ec5b,     //   0.48621,   0.87384
    0x3fdf081906bff7fd,     0x3febfc9d25a1b147,     //   0.48487,   0.87459
    0x3fdef21b8fafd3b5,     0x3fec02b2cef1e641,     //   0.48353,   0.87533
    0x3fdedc1952ef78d5,     0x3fec08c426725549,     //   0.48218,   0.87607
    0x3fdec61253e3c61b,     0x3fec0ed12b3372e9,     //   0.48084,   0.87681
    0x3fdeb00695f25620,     0x3fec14d9dc465e58,     //   0.47949,   0.87755
    0x3fde99f61c817eda,     0x3fec1ade38bce19b,     //   0.47815,   0.87828
    0x3fde83e0eaf85113,     0x3fec20de3fa971b0,     //    0.4768,   0.87901
    0x3fde6dc704be97e2,     0x3fec26d9f01f2eaf,     //   0.47545,   0.87974
    0x3fde57a86d3cd824,     0x3fec2cd14931e3f1,     //    0.4741,   0.88047
    0x3fde418527dc4ffa,     0x3fec32c449f60831,     //   0.47275,    0.8812
    0x3fde2b5d3806f63b,     0x3fec38b2f180bdb1,     //    0.4714,   0.88192
    0x3fde1530a12779f4,     0x3fec3e9d3ee7d262,     //   0.47004,   0.88264
    0x3fddfeff66a941de,     0x3fec44833141c004,     //   0.46869,   0.88336
    0x3fdde8c98bf86bd6,     0x3fec4a64c7a5ac4c,     //   0.46733,   0.88408
    0x3fddd28f1481cc58,     0x3fec5042012b6907,     //   0.46598,    0.8848
    0x3fddbc5003b2edf8,     0x3fec561adceb743e,     //   0.46462,   0.88551
    0x3fdda60c5cfa10d8,     0x3fec5bef59fef85a,     //   0.46326,   0.88622
    0x3fdd8fc423c62a25,     0x3fec61bf777fcc48,     //    0.4619,   0.88693
    0x3fdd79775b86e389,     0x3fec678b3488739b,     //   0.46054,   0.88764
    0x3fdd632607ac9aa9,     0x3fec6d5290341eb2,     //   0.45918,   0.88835
    0x3fdd4cd02ba8609c,     0x3fec7315899eaad7,     //   0.45781,   0.88905
    0x3fdd3675caebf962,     0x3fec78d41fe4a267,     //   0.45645,   0.88975
    0x3fdd2016e8e9db5b,     0x3fec7e8e52233cf3,     //   0.45508,   0.89045
    0x3fdd09b389152ec1,     0x3fec84441f785f61,     //   0.45372,   0.89115
    0x3fdcf34baee1cd21,     0x3fec89f587029c13,     //   0.45235,   0.89184
    0x3fdcdcdf5dc440ce,     0x3fec8fa287e13305,     //   0.45098,   0.89253
    0x3fdcc66e9931c45d,     0x3fec954b213411f5,     //   0.44961,   0.89322
    0x3fdcaff964a0421d,     0x3fec9aef521bd480,     //   0.44824,   0.89391
    0x3fdc997fc3865388,     0x3feca08f19b9c449,     //   0.44687,    0.8946
    0x3fdc8301b95b40c2,     0x3feca62a772fd919,     //    0.4455,   0.89528
    0x3fdc6c7f4997000a,     0x3fecabc169a0b901,     //   0.44412,   0.89597
    0x3fdc55f877b23537,     0x3fecb153f02fb87d,     //   0.44275,   0.89665
    0x3fdc3f6d47263129,     0x3fecb6e20a00da99,     //   0.44137,   0.89732
    0x3fdc28ddbb6cf145,     0x3fecbc6bb638d10b,     //   0.43999,     0.898
    0x3fdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     //   0.43862,   0.89867
    0x3fdbfbb1a05e0edc,     0x3fecc771c2736c09,     //   0.43724,   0.89935
    0x3fdbe51517ffc0d9,     0x3fecccee20c2de9f,     //   0.43586,   0.90002
    0x3fdbce744262deee,     0x3fecd2660e12c1e6,     //   0.43448,   0.90068
    0x3fdbb7cf2304bd01,     0x3fecd7d9898b32f6,     //   0.43309,   0.90135
    0x3fdba125bd63583e,     0x3fecdd489254fe65,     //   0.43171,   0.90201
    0x3fdb8a7814fd5693,     0x3fece2b32799a060,     //   0.43033,   0.90267
    0x3fdb73c62d520624,     0x3fece819488344ce,     //   0.42894,   0.90333
    0x3fdb5d1009e15cc0,     0x3feced7af43cc773,     //   0.42756,   0.90399
    0x3fdb4655ae2bf757,     0x3fecf2d829f1b40e,     //   0.42617,   0.90464
    0x3fdb2f971db31972,     0x3fecf830e8ce467b,     //   0.42478,    0.9053
    0x3fdb18d45bf8aca6,     0x3fecfd852fff6ad4,     //   0.42339,   0.90595
    0x3fdb020d6c7f4009,     0x3fed02d4feb2bd92,     //     0.422,    0.9066
    0x3fdaeb4252ca07ab,     0x3fed082054168bac,     //   0.42061,   0.90724
    0x3fdad473125cdc08,     0x3fed0d672f59d2b9,     //   0.41922,   0.90789
    0x3fdabd9faebc3980,     0x3fed12a98fac410c,     //   0.41782,   0.90853
    0x3fdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     //   0.41643,   0.90917
    0x3fda8fec8bf5b166,     0x3fed1d20dc40c15c,     //   0.41503,   0.90981
    0x3fda790cd3dbf31a,     0x3fed2255c6e5a4e1,     //   0.41364,   0.91044
    0x3fda622906a70b63,     0x3fed2786335f52fc,     //   0.41224,   0.91107
    0x3fda4b4127dea1e4,     0x3fed2cb220e0ef9f,     //   0.41084,   0.91171
    0x3fda34553b0afee5,     0x3fed31d98e9e503a,     //   0.40944,   0.91234
    0x3fda1d6543b50ac0,     0x3fed36fc7bcbfbdc,     //   0.40804,   0.91296
    0x3fda067145664d57,     0x3fed3c1ae79f2b4e,     //   0.40664,   0.91359
    0x3fd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //   0.40524,   0.91421
    0x3fd9d87d4207b0ab,     0x3fed464a380e7242,     //   0.40384,   0.91483
    0x3fd9c17d440df9f2,     0x3fed4b5b1b187524,     //   0.40243,   0.91545
    0x3fd9aa794d47c9ee,     0x3fed506779a3d2d9,     //   0.40103,   0.91606
    0x3fd993716141bdfe,     0x3fed556f52e93eb1,     //   0.39962,   0.91668
    0x3fd97c6583890fc2,     0x3fed5a72a6221e73,     //   0.39822,   0.91729
    0x3fd96555b7ab948f,     0x3fed5f7172888a7f,     //   0.39681,    0.9179
    0x3fd94e420137bce3,     0x3fed646bb7574de5,     //    0.3954,   0.91851
    0x3fd9372a63bc93d7,     0x3fed696173c9e68b,     //   0.39399,   0.91911
    0x3fd9200ee2c9be97,     0x3fed6e52a71c8547,     //   0.39258,   0.91972
    0x3fd908ef81ef7bd1,     0x3fed733f508c0dff,     //   0.39117,   0.92032
    0x3fd8f1cc44bea329,     0x3fed78276f5617c6,     //   0.38976,   0.92092
    0x3fd8daa52ec8a4af,     0x3fed7d0b02b8ecf9,     //   0.38835,   0.92151
    0x3fd8c37a439f884f,     0x3fed81ea09f38b63,     //   0.38693,   0.92211
    0x3fd8ac4b86d5ed44,     0x3fed86c48445a450,     //   0.38552,    0.9227
    0x3fd89518fbff098e,     0x3fed8b9a70ef9cb4,     //    0.3841,   0.92329
    0x3fd87de2a6aea963,     0x3fed906bcf328d46,     //   0.38268,   0.92388
    0x3fd866a88a792ea0,     0x3fed95389e50429b,     //   0.38127,   0.92447
    0x3fd84f6aaaf3903f,     0x3fed9a00dd8b3d46,     //   0.37985,   0.92505
    0x3fd838290bb359c8,     0x3fed9ec48c26b1f3,     //   0.37843,   0.92563
    0x3fd820e3b04eaac4,     0x3feda383a9668988,     //   0.37701,   0.92621
    0x3fd8099a9c5c362d,     0x3feda83e348f613b,     //   0.37559,   0.92679
    0x3fd7f24dd37341e3,     0x3fedacf42ce68ab9,     //   0.37416,   0.92736
    0x3fd7dafd592ba621,     0x3fedb1a591b20c38,     //   0.37274,   0.92794
    0x3fd7c3a9311dcce7,     0x3fedb6526238a09b,     //   0.37132,   0.92851
    0x3fd7ac515ee2b172,     0x3fedbafa9dc1b78d,     //   0.36989,   0.92907
    0x3fd794f5e613dfae,     0x3fedbf9e4395759a,     //   0.36847,   0.92964
    0x3fd77d96ca4b73a6,     0x3fedc43d52fcb453,     //   0.36704,   0.93021
    0x3fd766340f2418f6,     0x3fedc8d7cb410260,     //   0.36561,   0.93077
    0x3fd74ecdb8390a3e,     0x3fedcd6dabaca3a5,     //   0.36418,   0.93133
    0x3fd73763c9261092,     0x3fedd1fef38a915a,     //   0.36276,   0.93188
    0x3fd71ff6458782ec,     0x3fedd68ba2267a25,     //   0.36133,   0.93244
    0x3fd7088530fa459e,     0x3feddb13b6ccc23d,     //    0.3599,   0.93299
    0x3fd6f1108f1bc9c5,     0x3feddf9730ca837b,     //   0.35846,   0.93354
    0x3fd6d998638a0cb5,     0x3fede4160f6d8d81,     //   0.35703,   0.93409
    0x3fd6c21cb1e39771,     0x3fede890520465ce,     //    0.3556,   0.93464
    0x3fd6aa9d7dc77e16,     0x3feded05f7de47da,     //   0.35416,   0.93518
    0x3fd6931acad55f51,     0x3fedf177004b2534,     //   0.35273,   0.93573
    0x3fd67b949cad63ca,     0x3fedf5e36a9ba59c,     //   0.35129,   0.93627
    0x3fd6640af6f03d9e,     0x3fedfa4b3621271d,     //   0.34986,    0.9368
    0x3fd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     //   0.34842,   0.93734
    0x3fd634ed533be58e,     0x3fee030cee1435b8,     //   0.34698,   0.93787
    0x3fd61d595c88c203,     0x3fee0766d9280f54,     //   0.34554,    0.9384
    0x3fd605c1fcc88f63,     0x3fee0bbc22bd8349,     //    0.3441,   0.93893
    0x3fd5ee27379ea693,     0x3fee100cca2980ac,     //   0.34266,   0.93946
    0x3fd5d68910aee686,     0x3fee1458cec1ad83,     //   0.34122,   0.93998
    0x3fd5bee78b9db3b6,     0x3fee18a02fdc66d9,     //   0.33978,   0.94051
    0x3fd5a742ac0ff78d,     0x3fee1ce2ecd0c0d8,     //   0.33833,   0.94103
    0x3fd58f9a75ab1fdd,     0x3fee212104f686e5,     //   0.33689,   0.94154
    0x3fd577eeec151e47,     0x3fee255a77a63bb8,     //   0.33545,   0.94206
    0x3fd5604012f467b4,     0x3fee298f4439197a,     //     0.334,   0.94257
    0x3fd5488dedeff3be,     0x3fee2dbf6a0911d9,     //   0.33255,   0.94308
    0x3fd530d880af3c24,     0x3fee31eae870ce25,     //   0.33111,   0.94359
    0x3fd5191fceda3c35,     0x3fee3611becbaf69,     //   0.32966,    0.9441
    0x3fd50163dc197047,     0x3fee3a33ec75ce85,     //   0.32821,    0.9446
    0x3fd4e9a4ac15d520,     0x3fee3e5170cbfc46,     //   0.32676,   0.94511
    0x3fd4d1e24278e76a,     0x3fee426a4b2bc17e,     //   0.32531,   0.94561
    0x3fd4ba1ca2eca31c,     0x3fee467e7af35f23,     //   0.32386,   0.94611
    0x3fd4a253d11b82f3,     0x3fee4a8dff81ce5e,     //   0.32241,    0.9466
    0x3fd48a87d0b07fd7,     0x3fee4e98d836c0af,     //   0.32096,   0.94709
    0x3fd472b8a5571054,     0x3fee529f04729ffc,     //    0.3195,   0.94759
    0x3fd45ae652bb2800,     0x3fee56a083968eb1,     //   0.31805,   0.94807
    0x3fd44310dc8936f0,     0x3fee5a9d550467d3,     //   0.31659,   0.94856
    0x3fd42b38466e2928,     0x3fee5e95781ebf1c,     //   0.31514,   0.94905
    0x3fd4135c94176602,     0x3fee6288ec48e112,     //   0.31368,   0.94953
    0x3fd3fb7dc932cfa4,     0x3fee6677b0e6d31e,     //   0.31222,   0.95001
    0x3fd3e39be96ec271,     0x3fee6a61c55d53a7,     //   0.31077,   0.95049
    0x3fd3cbb6f87a146e,     0x3fee6e472911da27,     //   0.30931,   0.95096
    0x3fd3b3cefa0414b7,     0x3fee7227db6a9744,     //   0.30785,   0.95144
    0x3fd39be3f1bc8aef,     0x3fee7603dbce74e9,     //   0.30639,   0.95191
    0x3fd383f5e353b6aa,     0x3fee79db29a5165a,     //   0.30493,   0.95238
    0x3fd36c04d27a4edf,     0x3fee7dadc456d850,     //   0.30347,   0.95284
    0x3fd35410c2e18152,     0x3fee817bab4cd10d,     //   0.30201,   0.95331
    0x3fd33c19b83af207,     0x3fee8544ddf0d075,     //   0.30054,   0.95377
    0x3fd3241fb638baaf,     0x3fee89095bad6025,     //   0.29908,   0.95423
    0x3fd30c22c08d6a13,     0x3fee8cc923edc388,     //   0.29762,   0.95469
    0x3fd2f422daec0386,     0x3fee9084361df7f3,     //   0.29615,   0.95514
    0x3fd2dc200907fe51,     0x3fee943a91aab4b4,     //   0.29469,   0.95559
    0x3fd2c41a4e954520,     0x3fee97ec36016b30,     //   0.29322,   0.95605
    0x3fd2ac11af483572,     0x3fee9b99229046f8,     //   0.29175,   0.95649
    0x3fd294062ed59f05,     0x3fee9f4156c62dda,     //   0.29028,   0.95694
    0x3fd27bf7d0f2c346,     0x3feea2e4d212c000,     //   0.28882,   0.95738
    0x3fd263e6995554ba,     0x3feea68393e65800,     //   0.28735,   0.95783
    0x3fd24bd28bb37672,     0x3feeaa1d9bb20af3,     //   0.28588,   0.95827
    0x3fd233bbabc3bb72,     0x3feeadb2e8e7a88e,     //   0.28441,    0.9587
    0x3fd21ba1fd3d2623,     0x3feeb1437af9bb34,     //   0.28294,   0.95914
    0x3fd2038583d727bd,     0x3feeb4cf515b8811,     //   0.28146,   0.95957
    0x3fd1eb6643499fbb,     0x3feeb8566b810f2a,     //   0.27999,      0.96
    0x3fd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     //   0.27852,   0.96043
    0x3fd1bb1f7b999480,     0x3feebf5668eaf2ef,     //   0.27705,   0.96086
    0x3fd1a2f7fbe8f243,     0x3feec2cf4b1af6b2,     //   0.27557,   0.96128
    0x3fd18acdc3f4873a,     0x3feec6436ee60309,     //    0.2741,    0.9617
    0x3fd172a0d7765177,     0x3feec9b2d3c3bf84,     //   0.27262,   0.96212
    0x3fd15a713a28b9d9,     0x3feecd1d792c8f10,     //   0.27115,   0.96254
    0x3fd1423eefc69378,     0x3feed0835e999009,     //   0.26967,   0.96295
    0x3fd12a09fc0b1b12,     0x3feed3e483849c51,     //   0.26819,   0.96337
    0x3fd111d262b1f677,     0x3feed740e7684963,     //   0.26671,   0.96378
    0x3fd0f998277733f7,     0x3feeda9889bfe86a,     //   0.26523,   0.96418
    0x3fd0e15b4e1749cd,     0x3feeddeb6a078651,     //   0.26375,   0.96459
    0x3fd0c91bda4f158d,     0x3feee13987bbebdc,     //   0.26227,   0.96499
    0x3fd0b0d9cfdbdb90,     0x3feee482e25a9dbc,     //   0.26079,   0.96539
    0x3fd09895327b465e,     0x3feee7c77961dc9e,     //   0.25931,   0.96579
    0x3fd0804e05eb661e,     0x3feeeb074c50a544,     //   0.25783,   0.96619
    0x3fd068044deab002,     0x3feeee425aa6b09a,     //   0.25635,   0.96658
    0x3fd04fb80e37fdae,     0x3feef178a3e473c2,     //   0.25487,   0.96698
    0x3fd037694a928cac,     0x3feef4aa278b2032,     //   0.25338,   0.96737
    0x3fd01f1806b9fdd2,     0x3feef7d6e51ca3c0,     //    0.2519,   0.96775
    0x3fd006c4466e54af,     0x3feefafedc1ba8b7,     //   0.25041,   0.96814
    0x3fcfdcdc1adfedf8,     0x3feefe220c0b95ec,     //   0.24893,   0.96852
    0x3fcfac2abeff57ff,     0x3fef014074708ed3,     //   0.24744,    0.9689
    0x3fcf7b7480bd3801,     0x3fef045a14cf738c,     //   0.24596,   0.96928
    0x3fcf4ab9679c9f5c,     0x3fef076eecade0fa,     //   0.24447,   0.96966
    0x3fcf19f97b215f1a,     0x3fef0a7efb9230d7,     //   0.24298,   0.97003
    0x3fcee934c2d006c7,     0x3fef0d8a410379c5,     //   0.24149,    0.9704
    0x3fceb86b462de348,     0x3fef1090bc898f5f,     //      0.24,   0.97077
    0x3fce879d0cc0fdaf,     0x3fef13926dad024e,     //   0.23851,   0.97114
    0x3fce56ca1e101a1b,     0x3fef168f53f7205d,     //   0.23702,    0.9715
    0x3fce25f281a2b684,     0x3fef19876ef1f486,     //   0.23553,   0.97187
    0x3fcdf5163f01099a,     0x3fef1c7abe284708,     //   0.23404,   0.97223
    0x3fcdc4355db40195,     0x3fef1f6941259d7a,     //   0.23255,   0.97258
    0x3fcd934fe5454311,     0x3fef2252f7763ada,     //   0.23106,   0.97294
    0x3fcd6265dd3f27e3,     0x3fef2537e0a71f9f,     //   0.22957,   0.97329
    0x3fcd31774d2cbdee,     0x3fef2817fc4609ce,     //   0.22807,   0.97364
    0x3fcd00843c99c5f9,     0x3fef2af349e17507,     //   0.22658,   0.97399
    0x3fcccf8cb312b286,     0x3fef2dc9c9089a9d,     //   0.22508,   0.97434
    0x3fcc9e90b824a6a9,     0x3fef309b794b719f,     //   0.22359,   0.97468
    0x3fcc6d90535d74dc,     0x3fef33685a3aaef0,     //   0.22209,   0.97503
    0x3fcc3c8b8c4b9dd7,     0x3fef36306b67c556,     //    0.2206,   0.97536
    0x3fcc0b826a7e4f63,     0x3fef38f3ac64e589,     //    0.2191,    0.9757
    0x3fcbda74f5856330,     0x3fef3bb21cc4fe47,     //    0.2176,   0.97604
    0x3fcba96334f15dad,     0x3fef3e6bbc1bbc65,     //   0.21611,   0.97637
    0x3fcb784d30536cda,     0x3fef412089fd8adc,     //   0.21461,    0.9767
    0x3fcb4732ef3d6722,     0x3fef43d085ff92dd,     //   0.21311,   0.97703
    0x3fcb16147941ca2a,     0x3fef467bafb7bbe0,     //   0.21161,   0.97735
    0x3fcae4f1d5f3b9ab,     0x3fef492206bcabb4,     //   0.21011,   0.97768
    0x3fcab3cb0ce6fe44,     0x3fef4bc38aa5c694,     //   0.20861,     0.978
    0x3fca82a025b00451,     0x3fef4e603b0b2f2d,     //   0.20711,   0.97832
    0x3fca517127e3dabc,     0x3fef50f81785c6b9,     //   0.20561,   0.97863
    0x3fca203e1b1831da,     0x3fef538b1faf2d07,     //   0.20411,   0.97895
    0x3fc9ef0706e35a35,     0x3fef56195321c090,     //   0.20261,   0.97926
    0x3fc9bdcbf2dc4366,     0x3fef58a2b1789e84,     //    0.2011,   0.97957
    0x3fc98c8ce69a7aec,     0x3fef5b273a4fa2d9,     //    0.1996,   0.97988
    0x3fc95b49e9b62af9,     0x3fef5da6ed43685d,     //    0.1981,   0.98018
    0x3fc92a0303c8194f,     0x3fef6021c9f148c2,     //   0.19659,   0.98048
    0x3fc8f8b83c69a60a,     0x3fef6297cff75cb0,     //   0.19509,   0.98079
    0x3fc8c7699b34ca7e,     0x3fef6508fef47bd5,     //   0.19359,   0.98108
    0x3fc8961727c41804,     0x3fef677556883cee,     //   0.19208,   0.98138
    0x3fc864c0e9b2b6cf,     0x3fef69dcd652f5de,     //   0.19057,   0.98167
    0x3fc83366e89c64c5,     0x3fef6c3f7df5bbb7,     //   0.18907,   0.98196
    0x3fc802092c1d744b,     0x3fef6e9d4d1262ca,     //   0.18756,   0.98225
    0x3fc7d0a7bbd2cb1b,     0x3fef70f6434b7eb7,     //   0.18606,   0.98254
    0x3fc79f429f59e11d,     0x3fef734a60446279,     //   0.18455,   0.98282
    0x3fc76dd9de50bf31,     0x3fef7599a3a12077,     //   0.18304,   0.98311
    0x3fc73c6d8055fe0a,     0x3fef77e40d068a90,     //   0.18153,   0.98339
    0x3fc70afd8d08c4ff,     0x3fef7a299c1a322a,     //   0.18002,   0.98366
    0x3fc6d98a0c08c8da,     0x3fef7c6a50826840,     //   0.17851,   0.98394
    0x3fc6a81304f64ab2,     0x3fef7ea629e63d6e,     //     0.177,   0.98421
    0x3fc676987f7216b8,     0x3fef80dd27ed8204,     //   0.17549,   0.98448
    0x3fc6451a831d830d,     0x3fef830f4a40c60c,     //   0.17398,   0.98475
    0x3fc61399179a6e94,     0x3fef853c9089595e,     //   0.17247,   0.98501
    0x3fc5e214448b3fc6,     0x3fef8764fa714ba9,     //   0.17096,   0.98528
    0x3fc5b08c1192e381,     0x3fef898887a36c84,     //   0.16945,   0.98554
    0x3fc57f008654cbde,     0x3fef8ba737cb4b78,     //   0.16794,    0.9858
    0x3fc54d71aa74ef02,     0x3fef8dc10a95380d,     //   0.16643,   0.98605
    0x3fc51bdf8597c5f2,     0x3fef8fd5ffae41db,     //   0.16491,   0.98631
    0x3fc4ea4a1f624b61,     0x3fef91e616c43891,     //    0.1634,   0.98656
    0x3fc4b8b17f79fa88,     0x3fef93f14f85ac08,     //   0.16189,   0.98681
    0x3fc48715ad84cdf5,     0x3fef95f7a9a1ec47,     //   0.16037,   0.98706
    0x3fc45576b1293e5a,     0x3fef97f924c9099b,     //   0.15886,    0.9873
    0x3fc423d4920e4166,     0x3fef99f5c0abd496,     //   0.15734,   0.98754
    0x3fc3f22f57db4893,     0x3fef9bed7cfbde29,     //   0.15583,   0.98778
    0x3fc3c0870a383ff6,     0x3fef9de0596b77a3,     //   0.15431,   0.98802
    0x3fc38edbb0cd8d14,     0x3fef9fce55adb2c8,     //    0.1528,   0.98826
    0x3fc35d2d53440db2,     0x3fefa1b7717661d5,     //   0.15128,   0.98849
    0x3fc32b7bf94516a7,     0x3fefa39bac7a1791,     //   0.14976,   0.98872
    0x3fc2f9c7aa7a72af,     0x3fefa57b066e2754,     //   0.14825,   0.98895
    0x3fc2c8106e8e613a,     0x3fefa7557f08a517,     //   0.14673,   0.98918
    0x3fc296564d2b953e,     0x3fefa92b1600657c,     //   0.14521,    0.9894
    0x3fc264994dfd340a,     0x3fefaafbcb0cfddc,     //    0.1437,   0.98962
    0x3fc232d978aed413,     0x3fefacc79de6c44f,     //   0.14218,   0.98984
    0x3fc20116d4ec7bce,     0x3fefae8e8e46cfbb,     //   0.14066,   0.99006
    0x3fc1cf516a62a077,     0x3fefb0509be6f7db,     //   0.13914,   0.99027
    0x3fc19d8940be24e7,     0x3fefb20dc681d54d,     //   0.13762,   0.99049
    0x3fc16bbe5fac5865,     0x3fefb3c60dd2c199,     //    0.1361,    0.9907
    0x3fc139f0cedaf576,     0x3fefb5797195d741,     //   0.13458,    0.9909
    0x3fc1082095f820b0,     0x3fefb727f187f1c7,     //   0.13306,   0.99111
    0x3fc0d64dbcb26786,     0x3fefb8d18d66adb7,     //   0.13154,   0.99131
    0x3fc0a4784ab8bf1d,     0x3fefba7644f068b5,     //   0.13002,   0.99151
    0x3fc072a047ba831d,     0x3fefbc1617e44186,     //    0.1285,   0.99171
    0x3fc040c5bb67747e,     0x3fefbdb106021816,     //   0.12698,   0.99191
    0x3fc00ee8ad6fb85b,     0x3fefbf470f0a8d88,     //   0.12545,    0.9921
    0x3fbfba124b07ad85,     0x3fefc0d832bf043a,     //   0.12393,   0.99229
    0x3fbf564e56a9730e,     0x3fefc26470e19fd3,     //   0.12241,   0.99248
    0x3fbef2858d27561b,     0x3fefc3ebc935454c,     //   0.12089,   0.99267
    0x3fbe8eb7fde4aa3e,     0x3fefc56e3b7d9af6,     //   0.11937,   0.99285
    0x3fbe2ae5b8457f77,     0x3fefc6ebc77f0887,     //   0.11784,   0.99303
    0x3fbdc70ecbae9fc8,     0x3fefc8646cfeb721,     //   0.11632,   0.99321
    0x3fbd633347858ce4,     0x3fefc9d82bc2915e,     //   0.11479,   0.99339
    0x3fbcff533b307dc1,     0x3fefcb4703914354,     //   0.11327,   0.99356
    0x3fbc9b6eb6165c42,     0x3fefccb0f4323aa3,     //   0.11175,   0.99374
    0x3fbc3785c79ec2d5,     0x3fefce15fd6da67b,     //   0.11022,   0.99391
    0x3fbbd3987f31fa0e,     0x3fefcf761f0c77a3,     //    0.1087,   0.99407
    0x3fbb6fa6ec38f64c,     0x3fefd0d158d86087,     //   0.10717,   0.99424
    0x3fbb0bb11e1d5559,     0x3fefd227aa9bd53b,     //   0.10565,    0.9944
    0x3fbaa7b724495c04,     0x3fefd37914220b84,     //   0.10412,   0.99456
    0x3fba43b90e27f3c4,     0x3fefd4c59536fae4,     //    0.1026,   0.99472
    0x3fb9dfb6eb24a85c,     0x3fefd60d2da75c9e,     //   0.10107,   0.99488
    0x3fb97bb0caaba56f,     0x3fefd74fdd40abbf,     //  0.099544,   0.99503
    0x3fb917a6bc29b42c,     0x3fefd88da3d12526,     //  0.098017,   0.99518
    0x3fb8b398cf0c38e0,     0x3fefd9c68127c78c,     //   0.09649,   0.99533
    0x3fb84f8712c130a0,     0x3fefdafa7514538c,     //  0.094963,   0.99548
    0x3fb7eb7196b72ee4,     0x3fefdc297f674ba9,     //  0.093436,   0.99563
    0x3fb787586a5d5b21,     0x3fefdd539ff1f456,     //  0.091909,   0.99577
    0x3fb7233b9d236e71,     0x3fefde78d68653fd,     //  0.090381,   0.99591
    0x3fb6bf1b3e79b129,     0x3fefdf9922f73307,     //  0.088854,   0.99604
    0x3fb65af75dd0f87b,     0x3fefe0b485181be3,     //  0.087326,   0.99618
    0x3fb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     //  0.085797,   0.99631
    0x3fb592a554489bc8,     0x3fefe2dc89bbff08,     //  0.084269,   0.99644
    0x3fb52e774a4d4d0a,     0x3fefe3e92be9d886,     //   0.08274,   0.99657
    0x3fb4ca45fc1ba8b6,     0x3fefe4f0e31d7a4a,     //  0.081211,    0.9967
    0x3fb4661179272096,     0x3fefe5f3af2e3940,     //  0.079682,   0.99682
    0x3fb401d9d0e3a507,     0x3fefe6f18ff42c84,     //  0.078153,   0.99694
    0x3fb39d9f12c5a299,     0x3fefe7ea85482d60,     //  0.076624,   0.99706
    0x3fb339614e41ffa5,     0x3fefe8de8f03d75c,     //  0.075094,   0.99718
    0x3fb2d52092ce19f6,     0x3fefe9cdad01883a,     //  0.073565,   0.99729
    0x3fb270dcefdfc45b,     0x3fefeab7df1c6005,     //  0.072035,    0.9974
    0x3fb20c9674ed444c,     0x3fefeb9d2530410f,     //  0.070505,   0.99751
    0x3fb1a84d316d4f8a,     0x3fefec7d7f19cffc,     //  0.068974,   0.99762
    0x3fb1440134d709b2,     0x3fefed58ecb673c4,     //  0.067444,   0.99772
    0x3fb0dfb28ea201e6,     0x3fefee2f6de455ba,     //  0.065913,   0.99783
    0x3fb07b614e463064,     0x3fefef0102826191,     //  0.064383,   0.99793
    0x3fb0170d833bf421,     0x3fefefcdaa704562,     //  0.062852,   0.99802
    0x3faf656e79f820e0,     0x3feff095658e71ad,     //  0.061321,   0.99812
    0x3fae9cbd15ff5527,     0x3feff15833be1965,     //   0.05979,   0.99821
    0x3fadd406f9808ec8,     0x3feff21614e131ed,     //  0.058258,    0.9983
    0x3fad0b4c436f91d0,     0x3feff2cf08da7321,     //  0.056727,   0.99839
    0x3fac428d12c0d7e3,     0x3feff3830f8d575c,     //  0.055195,   0.99848
    0x3fab79c986698b78,     0x3feff43228de1b77,     //  0.053664,   0.99856
    0x3faab101bd5f8317,     0x3feff4dc54b1bed3,     //  0.052132,   0.99864
    0x3fa9e835d6993c87,     0x3feff58192ee0358,     //    0.0506,   0.99872
    0x3fa91f65f10dd814,     0x3feff621e3796d7e,     //  0.049068,    0.9988
    0x3fa856922bb513c1,     0x3feff6bd463b444d,     //  0.047535,   0.99887
    0x3fa78dbaa5874685,     0x3feff753bb1b9164,     //  0.046003,   0.99894
    0x3fa6c4df7d7d5b84,     0x3feff7e5420320f9,     //  0.044471,   0.99901
    0x3fa5fc00d290cd43,     0x3feff871dadb81df,     //  0.042938,   0.99908
    0x3fa5331ec3bba0eb,     0x3feff8f9858f058b,     //  0.041406,   0.99914
    0x3fa46a396ff86179,     0x3feff97c4208c014,     //  0.039873,    0.9992
    0x3fa3a150f6421afc,     0x3feff9fa10348837,     //   0.03834,   0.99926
    0x3fa2d865759455cd,     0x3feffa72effef75d,     //  0.036807,   0.99932
    0x3fa20f770ceb11c6,     0x3feffae6e1556998,     //  0.035274,   0.99938
    0x3fa14685db42c17e,     0x3feffb55e425fdae,     //  0.033741,   0.99943
    0x3fa07d91ff984580,     0x3feffbbff85f9515,     //  0.032208,   0.99948
    0x3f9f693731d1cf01,     0x3feffc251df1d3f8,     //  0.030675,   0.99953
    0x3f9dd7458c64ab39,     0x3feffc8554cd213a,     //  0.029142,   0.99958
    0x3f9c454f4ce53b1c,     0x3feffce09ce2a679,     //  0.027608,   0.99962
    0x3f9ab354b1504fca,     0x3feffd36f624500c,     //  0.026075,   0.99966
    0x3f992155f7a3667e,     0x3feffd886084cd0d,     //  0.024541,    0.9997
    0x3f978f535ddc9f03,     0x3feffdd4dbf78f52,     //  0.023008,   0.99974
    0x3f95fd4d21fab226,     0x3feffe1c6870cb77,     //  0.021474,   0.99977
    0x3f946b4381fce81c,     0x3feffe5f05e578db,     //   0.01994,    0.9998
    0x3f92d936bbe30efd,     0x3feffe9cb44b51a1,     //  0.018407,   0.99983
    0x3f9147270dad7132,     0x3feffed57398d2b7,     //  0.016873,   0.99986
    0x3f8f6a296ab997ca,     0x3fefff0943c53bd1,     //  0.015339,   0.99988
    0x3f8c45ffe1e48ad9,     0x3fefff3824c88f6f,     //  0.013805,    0.9999
    0x3f8921d1fcdec784,     0x3fefff62169b92db,     //  0.012272,   0.99992
    0x3f85fda037ac05e0,     0x3fefff871937ce2f,     //  0.010738,   0.99994
    0x3f82d96b0e509703,     0x3fefffa72c978c4f,     // 0.0092038,   0.99996
    0x3f7f6a65f9a2a3c5,     0x3fefffc250b5daef,     // 0.0076698,   0.99997
    0x3f7921f0fe670071,     0x3fefffd8858e8a92,     // 0.0061359,   0.99998
    0x3f72d97822f996bc,     0x3fefffe9cb1e2e8d,     // 0.0046019,   0.99999
    0x3f6921f8becca4ba,     0x3feffff621621d02,     //  0.003068,         1
    0x3f5921faaee6472d,     0x3feffffd88586ee6,     //  0.001534,         1
    0x0000000000000000,     0x3ff0000000000000,     //         0,         1
    0xbf5921faaee6472d,     0x3feffffd88586ee6,     // -0.001534,         1
    0xbf6921f8becca4ba,     0x3feffff621621d02,     // -0.003068,         1
    0xbf72d97822f996bc,     0x3fefffe9cb1e2e8d,     //-0.0046019,   0.99999
    0xbf7921f0fe670071,     0x3fefffd8858e8a92,     //-0.0061359,   0.99998
    0xbf7f6a65f9a2a3c5,     0x3fefffc250b5daef,     //-0.0076698,   0.99997
    0xbf82d96b0e509703,     0x3fefffa72c978c4f,     //-0.0092038,   0.99996
    0xbf85fda037ac05e0,     0x3fefff871937ce2f,     // -0.010738,   0.99994
    0xbf8921d1fcdec784,     0x3fefff62169b92db,     // -0.012272,   0.99992
    0xbf8c45ffe1e48ad9,     0x3fefff3824c88f6f,     // -0.013805,    0.9999
    0xbf8f6a296ab997ca,     0x3fefff0943c53bd1,     // -0.015339,   0.99988
    0xbf9147270dad7132,     0x3feffed57398d2b7,     // -0.016873,   0.99986
    0xbf92d936bbe30efd,     0x3feffe9cb44b51a1,     // -0.018407,   0.99983
    0xbf946b4381fce81c,     0x3feffe5f05e578db,     //  -0.01994,    0.9998
    0xbf95fd4d21fab226,     0x3feffe1c6870cb77,     // -0.021474,   0.99977
    0xbf978f535ddc9f03,     0x3feffdd4dbf78f52,     // -0.023008,   0.99974
    0xbf992155f7a3667e,     0x3feffd886084cd0d,     // -0.024541,    0.9997
    0xbf9ab354b1504fca,     0x3feffd36f624500c,     // -0.026075,   0.99966
    0xbf9c454f4ce53b1c,     0x3feffce09ce2a679,     // -0.027608,   0.99962
    0xbf9dd7458c64ab39,     0x3feffc8554cd213a,     // -0.029142,   0.99958
    0xbf9f693731d1cf01,     0x3feffc251df1d3f8,     // -0.030675,   0.99953
    0xbfa07d91ff984580,     0x3feffbbff85f9515,     // -0.032208,   0.99948
    0xbfa14685db42c17e,     0x3feffb55e425fdae,     // -0.033741,   0.99943
    0xbfa20f770ceb11c6,     0x3feffae6e1556998,     // -0.035274,   0.99938
    0xbfa2d865759455cd,     0x3feffa72effef75d,     // -0.036807,   0.99932
    0xbfa3a150f6421afc,     0x3feff9fa10348837,     //  -0.03834,   0.99926
    0xbfa46a396ff86179,     0x3feff97c4208c014,     // -0.039873,    0.9992
    0xbfa5331ec3bba0eb,     0x3feff8f9858f058b,     // -0.041406,   0.99914
    0xbfa5fc00d290cd43,     0x3feff871dadb81df,     // -0.042938,   0.99908
    0xbfa6c4df7d7d5b84,     0x3feff7e5420320f9,     // -0.044471,   0.99901
    0xbfa78dbaa5874685,     0x3feff753bb1b9164,     // -0.046003,   0.99894
    0xbfa856922bb513c1,     0x3feff6bd463b444d,     // -0.047535,   0.99887
    0xbfa91f65f10dd814,     0x3feff621e3796d7e,     // -0.049068,    0.9988
    0xbfa9e835d6993c87,     0x3feff58192ee0358,     //   -0.0506,   0.99872
    0xbfaab101bd5f8317,     0x3feff4dc54b1bed3,     // -0.052132,   0.99864
    0xbfab79c986698b78,     0x3feff43228de1b77,     // -0.053664,   0.99856
    0xbfac428d12c0d7e3,     0x3feff3830f8d575c,     // -0.055195,   0.99848
    0xbfad0b4c436f91d0,     0x3feff2cf08da7321,     // -0.056727,   0.99839
    0xbfadd406f9808ec8,     0x3feff21614e131ed,     // -0.058258,    0.9983
    0xbfae9cbd15ff5527,     0x3feff15833be1965,     //  -0.05979,   0.99821
    0xbfaf656e79f820e0,     0x3feff095658e71ad,     // -0.061321,   0.99812
    0xbfb0170d833bf421,     0x3fefefcdaa704562,     // -0.062852,   0.99802
    0xbfb07b614e463064,     0x3fefef0102826191,     // -0.064383,   0.99793
    0xbfb0dfb28ea201e6,     0x3fefee2f6de455ba,     // -0.065913,   0.99783
    0xbfb1440134d709b2,     0x3fefed58ecb673c4,     // -0.067444,   0.99772
    0xbfb1a84d316d4f8a,     0x3fefec7d7f19cffc,     // -0.068974,   0.99762
    0xbfb20c9674ed444c,     0x3fefeb9d2530410f,     // -0.070505,   0.99751
    0xbfb270dcefdfc45b,     0x3fefeab7df1c6005,     // -0.072035,    0.9974
    0xbfb2d52092ce19f6,     0x3fefe9cdad01883a,     // -0.073565,   0.99729
    0xbfb339614e41ffa5,     0x3fefe8de8f03d75c,     // -0.075094,   0.99718
    0xbfb39d9f12c5a299,     0x3fefe7ea85482d60,     // -0.076624,   0.99706
    0xbfb401d9d0e3a507,     0x3fefe6f18ff42c84,     // -0.078153,   0.99694
    0xbfb4661179272096,     0x3fefe5f3af2e3940,     // -0.079682,   0.99682
    0xbfb4ca45fc1ba8b6,     0x3fefe4f0e31d7a4a,     // -0.081211,    0.9967
    0xbfb52e774a4d4d0a,     0x3fefe3e92be9d886,     //  -0.08274,   0.99657
    0xbfb592a554489bc8,     0x3fefe2dc89bbff08,     // -0.084269,   0.99644
    0xbfb5f6d00a9aa419,     0x3fefe1cafcbd5b09,     // -0.085797,   0.99631
    0xbfb65af75dd0f87b,     0x3fefe0b485181be3,     // -0.087326,   0.99618
    0xbfb6bf1b3e79b129,     0x3fefdf9922f73307,     // -0.088854,   0.99604
    0xbfb7233b9d236e71,     0x3fefde78d68653fd,     // -0.090381,   0.99591
    0xbfb787586a5d5b21,     0x3fefdd539ff1f456,     // -0.091909,   0.99577
    0xbfb7eb7196b72ee4,     0x3fefdc297f674ba9,     // -0.093436,   0.99563
    0xbfb84f8712c130a0,     0x3fefdafa7514538c,     // -0.094963,   0.99548
    0xbfb8b398cf0c38e0,     0x3fefd9c68127c78c,     //  -0.09649,   0.99533
    0xbfb917a6bc29b42c,     0x3fefd88da3d12526,     // -0.098017,   0.99518
    0xbfb97bb0caaba56f,     0x3fefd74fdd40abbf,     // -0.099544,   0.99503
    0xbfb9dfb6eb24a85c,     0x3fefd60d2da75c9e,     //  -0.10107,   0.99488
    0xbfba43b90e27f3c4,     0x3fefd4c59536fae4,     //   -0.1026,   0.99472
    0xbfbaa7b724495c04,     0x3fefd37914220b84,     //  -0.10412,   0.99456
    0xbfbb0bb11e1d5559,     0x3fefd227aa9bd53b,     //  -0.10565,    0.9944
    0xbfbb6fa6ec38f64c,     0x3fefd0d158d86087,     //  -0.10717,   0.99424
    0xbfbbd3987f31fa0e,     0x3fefcf761f0c77a3,     //   -0.1087,   0.99407
    0xbfbc3785c79ec2d5,     0x3fefce15fd6da67b,     //  -0.11022,   0.99391
    0xbfbc9b6eb6165c42,     0x3fefccb0f4323aa3,     //  -0.11175,   0.99374
    0xbfbcff533b307dc1,     0x3fefcb4703914354,     //  -0.11327,   0.99356
    0xbfbd633347858ce4,     0x3fefc9d82bc2915e,     //  -0.11479,   0.99339
    0xbfbdc70ecbae9fc8,     0x3fefc8646cfeb721,     //  -0.11632,   0.99321
    0xbfbe2ae5b8457f77,     0x3fefc6ebc77f0887,     //  -0.11784,   0.99303
    0xbfbe8eb7fde4aa3e,     0x3fefc56e3b7d9af6,     //  -0.11937,   0.99285
    0xbfbef2858d27561b,     0x3fefc3ebc935454c,     //  -0.12089,   0.99267
    0xbfbf564e56a9730e,     0x3fefc26470e19fd3,     //  -0.12241,   0.99248
    0xbfbfba124b07ad85,     0x3fefc0d832bf043a,     //  -0.12393,   0.99229
    0xbfc00ee8ad6fb85b,     0x3fefbf470f0a8d88,     //  -0.12545,    0.9921
    0xbfc040c5bb67747e,     0x3fefbdb106021816,     //  -0.12698,   0.99191
    0xbfc072a047ba831d,     0x3fefbc1617e44186,     //   -0.1285,   0.99171
    0xbfc0a4784ab8bf1d,     0x3fefba7644f068b5,     //  -0.13002,   0.99151
    0xbfc0d64dbcb26786,     0x3fefb8d18d66adb7,     //  -0.13154,   0.99131
    0xbfc1082095f820b0,     0x3fefb727f187f1c7,     //  -0.13306,   0.99111
    0xbfc139f0cedaf576,     0x3fefb5797195d741,     //  -0.13458,    0.9909
    0xbfc16bbe5fac5865,     0x3fefb3c60dd2c199,     //   -0.1361,    0.9907
    0xbfc19d8940be24e7,     0x3fefb20dc681d54d,     //  -0.13762,   0.99049
    0xbfc1cf516a62a077,     0x3fefb0509be6f7db,     //  -0.13914,   0.99027
    0xbfc20116d4ec7bce,     0x3fefae8e8e46cfbb,     //  -0.14066,   0.99006
    0xbfc232d978aed413,     0x3fefacc79de6c44f,     //  -0.14218,   0.98984
    0xbfc264994dfd340a,     0x3fefaafbcb0cfddc,     //   -0.1437,   0.98962
    0xbfc296564d2b953e,     0x3fefa92b1600657c,     //  -0.14521,    0.9894
    0xbfc2c8106e8e613a,     0x3fefa7557f08a517,     //  -0.14673,   0.98918
    0xbfc2f9c7aa7a72af,     0x3fefa57b066e2754,     //  -0.14825,   0.98895
    0xbfc32b7bf94516a7,     0x3fefa39bac7a1791,     //  -0.14976,   0.98872
    0xbfc35d2d53440db2,     0x3fefa1b7717661d5,     //  -0.15128,   0.98849
    0xbfc38edbb0cd8d14,     0x3fef9fce55adb2c8,     //   -0.1528,   0.98826
    0xbfc3c0870a383ff6,     0x3fef9de0596b77a3,     //  -0.15431,   0.98802
    0xbfc3f22f57db4893,     0x3fef9bed7cfbde29,     //  -0.15583,   0.98778
    0xbfc423d4920e4166,     0x3fef99f5c0abd496,     //  -0.15734,   0.98754
    0xbfc45576b1293e5a,     0x3fef97f924c9099b,     //  -0.15886,    0.9873
    0xbfc48715ad84cdf5,     0x3fef95f7a9a1ec47,     //  -0.16037,   0.98706
    0xbfc4b8b17f79fa88,     0x3fef93f14f85ac08,     //  -0.16189,   0.98681
    0xbfc4ea4a1f624b61,     0x3fef91e616c43891,     //   -0.1634,   0.98656
    0xbfc51bdf8597c5f2,     0x3fef8fd5ffae41db,     //  -0.16491,   0.98631
    0xbfc54d71aa74ef02,     0x3fef8dc10a95380d,     //  -0.16643,   0.98605
    0xbfc57f008654cbde,     0x3fef8ba737cb4b78,     //  -0.16794,    0.9858
    0xbfc5b08c1192e381,     0x3fef898887a36c84,     //  -0.16945,   0.98554
    0xbfc5e214448b3fc6,     0x3fef8764fa714ba9,     //  -0.17096,   0.98528
    0xbfc61399179a6e94,     0x3fef853c9089595e,     //  -0.17247,   0.98501
    0xbfc6451a831d830d,     0x3fef830f4a40c60c,     //  -0.17398,   0.98475
    0xbfc676987f7216b8,     0x3fef80dd27ed8204,     //  -0.17549,   0.98448
    0xbfc6a81304f64ab2,     0x3fef7ea629e63d6e,     //    -0.177,   0.98421
    0xbfc6d98a0c08c8da,     0x3fef7c6a50826840,     //  -0.17851,   0.98394
    0xbfc70afd8d08c4ff,     0x3fef7a299c1a322a,     //  -0.18002,   0.98366
    0xbfc73c6d8055fe0a,     0x3fef77e40d068a90,     //  -0.18153,   0.98339
    0xbfc76dd9de50bf31,     0x3fef7599a3a12077,     //  -0.18304,   0.98311
    0xbfc79f429f59e11d,     0x3fef734a60446279,     //  -0.18455,   0.98282
    0xbfc7d0a7bbd2cb1b,     0x3fef70f6434b7eb7,     //  -0.18606,   0.98254
    0xbfc802092c1d744b,     0x3fef6e9d4d1262ca,     //  -0.18756,   0.98225
    0xbfc83366e89c64c5,     0x3fef6c3f7df5bbb7,     //  -0.18907,   0.98196
    0xbfc864c0e9b2b6cf,     0x3fef69dcd652f5de,     //  -0.19057,   0.98167
    0xbfc8961727c41804,     0x3fef677556883cee,     //  -0.19208,   0.98138
    0xbfc8c7699b34ca7e,     0x3fef6508fef47bd5,     //  -0.19359,   0.98108
    0xbfc8f8b83c69a60a,     0x3fef6297cff75cb0,     //  -0.19509,   0.98079
    0xbfc92a0303c8194f,     0x3fef6021c9f148c2,     //  -0.19659,   0.98048
    0xbfc95b49e9b62af9,     0x3fef5da6ed43685d,     //   -0.1981,   0.98018
    0xbfc98c8ce69a7aec,     0x3fef5b273a4fa2d9,     //   -0.1996,   0.97988
    0xbfc9bdcbf2dc4366,     0x3fef58a2b1789e84,     //   -0.2011,   0.97957
    0xbfc9ef0706e35a35,     0x3fef56195321c090,     //  -0.20261,   0.97926
    0xbfca203e1b1831da,     0x3fef538b1faf2d07,     //  -0.20411,   0.97895
    0xbfca517127e3dabc,     0x3fef50f81785c6b9,     //  -0.20561,   0.97863
    0xbfca82a025b00451,     0x3fef4e603b0b2f2d,     //  -0.20711,   0.97832
    0xbfcab3cb0ce6fe44,     0x3fef4bc38aa5c694,     //  -0.20861,     0.978
    0xbfcae4f1d5f3b9ab,     0x3fef492206bcabb4,     //  -0.21011,   0.97768
    0xbfcb16147941ca2a,     0x3fef467bafb7bbe0,     //  -0.21161,   0.97735
    0xbfcb4732ef3d6722,     0x3fef43d085ff92dd,     //  -0.21311,   0.97703
    0xbfcb784d30536cda,     0x3fef412089fd8adc,     //  -0.21461,    0.9767
    0xbfcba96334f15dad,     0x3fef3e6bbc1bbc65,     //  -0.21611,   0.97637
    0xbfcbda74f5856330,     0x3fef3bb21cc4fe47,     //   -0.2176,   0.97604
    0xbfcc0b826a7e4f63,     0x3fef38f3ac64e589,     //   -0.2191,    0.9757
    0xbfcc3c8b8c4b9dd7,     0x3fef36306b67c556,     //   -0.2206,   0.97536
    0xbfcc6d90535d74dc,     0x3fef33685a3aaef0,     //  -0.22209,   0.97503
    0xbfcc9e90b824a6a9,     0x3fef309b794b719f,     //  -0.22359,   0.97468
    0xbfcccf8cb312b286,     0x3fef2dc9c9089a9d,     //  -0.22508,   0.97434
    0xbfcd00843c99c5f9,     0x3fef2af349e17507,     //  -0.22658,   0.97399
    0xbfcd31774d2cbdee,     0x3fef2817fc4609ce,     //  -0.22807,   0.97364
    0xbfcd6265dd3f27e3,     0x3fef2537e0a71f9f,     //  -0.22957,   0.97329
    0xbfcd934fe5454311,     0x3fef2252f7763ada,     //  -0.23106,   0.97294
    0xbfcdc4355db40195,     0x3fef1f6941259d7a,     //  -0.23255,   0.97258
    0xbfcdf5163f01099a,     0x3fef1c7abe284708,     //  -0.23404,   0.97223
    0xbfce25f281a2b684,     0x3fef19876ef1f486,     //  -0.23553,   0.97187
    0xbfce56ca1e101a1b,     0x3fef168f53f7205d,     //  -0.23702,    0.9715
    0xbfce879d0cc0fdaf,     0x3fef13926dad024e,     //  -0.23851,   0.97114
    0xbfceb86b462de348,     0x3fef1090bc898f5f,     //     -0.24,   0.97077
    0xbfcee934c2d006c7,     0x3fef0d8a410379c5,     //  -0.24149,    0.9704
    0xbfcf19f97b215f1a,     0x3fef0a7efb9230d7,     //  -0.24298,   0.97003
    0xbfcf4ab9679c9f5c,     0x3fef076eecade0fa,     //  -0.24447,   0.96966
    0xbfcf7b7480bd3801,     0x3fef045a14cf738c,     //  -0.24596,   0.96928
    0xbfcfac2abeff57ff,     0x3fef014074708ed3,     //  -0.24744,    0.9689
    0xbfcfdcdc1adfedf8,     0x3feefe220c0b95ec,     //  -0.24893,   0.96852
    0xbfd006c4466e54af,     0x3feefafedc1ba8b7,     //  -0.25041,   0.96814
    0xbfd01f1806b9fdd2,     0x3feef7d6e51ca3c0,     //   -0.2519,   0.96775
    0xbfd037694a928cac,     0x3feef4aa278b2032,     //  -0.25338,   0.96737
    0xbfd04fb80e37fdae,     0x3feef178a3e473c2,     //  -0.25487,   0.96698
    0xbfd068044deab002,     0x3feeee425aa6b09a,     //  -0.25635,   0.96658
    0xbfd0804e05eb661e,     0x3feeeb074c50a544,     //  -0.25783,   0.96619
    0xbfd09895327b465e,     0x3feee7c77961dc9e,     //  -0.25931,   0.96579
    0xbfd0b0d9cfdbdb90,     0x3feee482e25a9dbc,     //  -0.26079,   0.96539
    0xbfd0c91bda4f158d,     0x3feee13987bbebdc,     //  -0.26227,   0.96499
    0xbfd0e15b4e1749cd,     0x3feeddeb6a078651,     //  -0.26375,   0.96459
    0xbfd0f998277733f7,     0x3feeda9889bfe86a,     //  -0.26523,   0.96418
    0xbfd111d262b1f677,     0x3feed740e7684963,     //  -0.26671,   0.96378
    0xbfd12a09fc0b1b12,     0x3feed3e483849c51,     //  -0.26819,   0.96337
    0xbfd1423eefc69378,     0x3feed0835e999009,     //  -0.26967,   0.96295
    0xbfd15a713a28b9d9,     0x3feecd1d792c8f10,     //  -0.27115,   0.96254
    0xbfd172a0d7765177,     0x3feec9b2d3c3bf84,     //  -0.27262,   0.96212
    0xbfd18acdc3f4873a,     0x3feec6436ee60309,     //   -0.2741,    0.9617
    0xbfd1a2f7fbe8f243,     0x3feec2cf4b1af6b2,     //  -0.27557,   0.96128
    0xbfd1bb1f7b999480,     0x3feebf5668eaf2ef,     //  -0.27705,   0.96086
    0xbfd1d3443f4cdb3d,     0x3feebbd8c8df0b74,     //  -0.27852,   0.96043
    0xbfd1eb6643499fbb,     0x3feeb8566b810f2a,     //  -0.27999,      0.96
    0xbfd2038583d727bd,     0x3feeb4cf515b8811,     //  -0.28146,   0.95957
    0xbfd21ba1fd3d2623,     0x3feeb1437af9bb34,     //  -0.28294,   0.95914
    0xbfd233bbabc3bb72,     0x3feeadb2e8e7a88e,     //  -0.28441,    0.9587
    0xbfd24bd28bb37672,     0x3feeaa1d9bb20af3,     //  -0.28588,   0.95827
    0xbfd263e6995554ba,     0x3feea68393e65800,     //  -0.28735,   0.95783
    0xbfd27bf7d0f2c346,     0x3feea2e4d212c000,     //  -0.28882,   0.95738
    0xbfd294062ed59f05,     0x3fee9f4156c62dda,     //  -0.29028,   0.95694
    0xbfd2ac11af483572,     0x3fee9b99229046f8,     //  -0.29175,   0.95649
    0xbfd2c41a4e954520,     0x3fee97ec36016b30,     //  -0.29322,   0.95605
    0xbfd2dc200907fe51,     0x3fee943a91aab4b4,     //  -0.29469,   0.95559
    0xbfd2f422daec0386,     0x3fee9084361df7f3,     //  -0.29615,   0.95514
    0xbfd30c22c08d6a13,     0x3fee8cc923edc388,     //  -0.29762,   0.95469
    0xbfd3241fb638baaf,     0x3fee89095bad6025,     //  -0.29908,   0.95423
    0xbfd33c19b83af207,     0x3fee8544ddf0d075,     //  -0.30054,   0.95377
    0xbfd35410c2e18152,     0x3fee817bab4cd10d,     //  -0.30201,   0.95331
    0xbfd36c04d27a4edf,     0x3fee7dadc456d850,     //  -0.30347,   0.95284
    0xbfd383f5e353b6aa,     0x3fee79db29a5165a,     //  -0.30493,   0.95238
    0xbfd39be3f1bc8aef,     0x3fee7603dbce74e9,     //  -0.30639,   0.95191
    0xbfd3b3cefa0414b7,     0x3fee7227db6a9744,     //  -0.30785,   0.95144
    0xbfd3cbb6f87a146e,     0x3fee6e472911da27,     //  -0.30931,   0.95096
    0xbfd3e39be96ec271,     0x3fee6a61c55d53a7,     //  -0.31077,   0.95049
    0xbfd3fb7dc932cfa4,     0x3fee6677b0e6d31e,     //  -0.31222,   0.95001
    0xbfd4135c94176602,     0x3fee6288ec48e112,     //  -0.31368,   0.94953
    0xbfd42b38466e2928,     0x3fee5e95781ebf1c,     //  -0.31514,   0.94905
    0xbfd44310dc8936f0,     0x3fee5a9d550467d3,     //  -0.31659,   0.94856
    0xbfd45ae652bb2800,     0x3fee56a083968eb1,     //  -0.31805,   0.94807
    0xbfd472b8a5571054,     0x3fee529f04729ffc,     //   -0.3195,   0.94759
    0xbfd48a87d0b07fd7,     0x3fee4e98d836c0af,     //  -0.32096,   0.94709
    0xbfd4a253d11b82f3,     0x3fee4a8dff81ce5e,     //  -0.32241,    0.9466
    0xbfd4ba1ca2eca31c,     0x3fee467e7af35f23,     //  -0.32386,   0.94611
    0xbfd4d1e24278e76a,     0x3fee426a4b2bc17e,     //  -0.32531,   0.94561
    0xbfd4e9a4ac15d520,     0x3fee3e5170cbfc46,     //  -0.32676,   0.94511
    0xbfd50163dc197047,     0x3fee3a33ec75ce85,     //  -0.32821,    0.9446
    0xbfd5191fceda3c35,     0x3fee3611becbaf69,     //  -0.32966,    0.9441
    0xbfd530d880af3c24,     0x3fee31eae870ce25,     //  -0.33111,   0.94359
    0xbfd5488dedeff3be,     0x3fee2dbf6a0911d9,     //  -0.33255,   0.94308
    0xbfd5604012f467b4,     0x3fee298f4439197a,     //    -0.334,   0.94257
    0xbfd577eeec151e47,     0x3fee255a77a63bb8,     //  -0.33545,   0.94206
    0xbfd58f9a75ab1fdd,     0x3fee212104f686e5,     //  -0.33689,   0.94154
    0xbfd5a742ac0ff78d,     0x3fee1ce2ecd0c0d8,     //  -0.33833,   0.94103
    0xbfd5bee78b9db3b6,     0x3fee18a02fdc66d9,     //  -0.33978,   0.94051
    0xbfd5d68910aee686,     0x3fee1458cec1ad83,     //  -0.34122,   0.93998
    0xbfd5ee27379ea693,     0x3fee100cca2980ac,     //  -0.34266,   0.93946
    0xbfd605c1fcc88f63,     0x3fee0bbc22bd8349,     //   -0.3441,   0.93893
    0xbfd61d595c88c203,     0x3fee0766d9280f54,     //  -0.34554,    0.9384
    0xbfd634ed533be58e,     0x3fee030cee1435b8,     //  -0.34698,   0.93787
    0xbfd64c7ddd3f27c6,     0x3fedfeae622dbe2b,     //  -0.34842,   0.93734
    0xbfd6640af6f03d9e,     0x3fedfa4b3621271d,     //  -0.34986,    0.9368
    0xbfd67b949cad63ca,     0x3fedf5e36a9ba59c,     //  -0.35129,   0.93627
    0xbfd6931acad55f51,     0x3fedf177004b2534,     //  -0.35273,   0.93573
    0xbfd6aa9d7dc77e16,     0x3feded05f7de47da,     //  -0.35416,   0.93518
    0xbfd6c21cb1e39771,     0x3fede890520465ce,     //   -0.3556,   0.93464
    0xbfd6d998638a0cb5,     0x3fede4160f6d8d81,     //  -0.35703,   0.93409
    0xbfd6f1108f1bc9c5,     0x3feddf9730ca837b,     //  -0.35846,   0.93354
    0xbfd7088530fa459e,     0x3feddb13b6ccc23d,     //   -0.3599,   0.93299
    0xbfd71ff6458782ec,     0x3fedd68ba2267a25,     //  -0.36133,   0.93244
    0xbfd73763c9261092,     0x3fedd1fef38a915a,     //  -0.36276,   0.93188
    0xbfd74ecdb8390a3e,     0x3fedcd6dabaca3a5,     //  -0.36418,   0.93133
    0xbfd766340f2418f6,     0x3fedc8d7cb410260,     //  -0.36561,   0.93077
    0xbfd77d96ca4b73a6,     0x3fedc43d52fcb453,     //  -0.36704,   0.93021
    0xbfd794f5e613dfae,     0x3fedbf9e4395759a,     //  -0.36847,   0.92964
    0xbfd7ac515ee2b172,     0x3fedbafa9dc1b78d,     //  -0.36989,   0.92907
    0xbfd7c3a9311dcce7,     0x3fedb6526238a09b,     //  -0.37132,   0.92851
    0xbfd7dafd592ba621,     0x3fedb1a591b20c38,     //  -0.37274,   0.92794
    0xbfd7f24dd37341e3,     0x3fedacf42ce68ab9,     //  -0.37416,   0.92736
    0xbfd8099a9c5c362d,     0x3feda83e348f613b,     //  -0.37559,   0.92679
    0xbfd820e3b04eaac4,     0x3feda383a9668988,     //  -0.37701,   0.92621
    0xbfd838290bb359c8,     0x3fed9ec48c26b1f3,     //  -0.37843,   0.92563
    0xbfd84f6aaaf3903f,     0x3fed9a00dd8b3d46,     //  -0.37985,   0.92505
    0xbfd866a88a792ea0,     0x3fed95389e50429b,     //  -0.38127,   0.92447
    0xbfd87de2a6aea963,     0x3fed906bcf328d46,     //  -0.38268,   0.92388
    0xbfd89518fbff098e,     0x3fed8b9a70ef9cb4,     //   -0.3841,   0.92329
    0xbfd8ac4b86d5ed44,     0x3fed86c48445a450,     //  -0.38552,    0.9227
    0xbfd8c37a439f884f,     0x3fed81ea09f38b63,     //  -0.38693,   0.92211
    0xbfd8daa52ec8a4af,     0x3fed7d0b02b8ecf9,     //  -0.38835,   0.92151
    0xbfd8f1cc44bea329,     0x3fed78276f5617c6,     //  -0.38976,   0.92092
    0xbfd908ef81ef7bd1,     0x3fed733f508c0dff,     //  -0.39117,   0.92032
    0xbfd9200ee2c9be97,     0x3fed6e52a71c8547,     //  -0.39258,   0.91972
    0xbfd9372a63bc93d7,     0x3fed696173c9e68b,     //  -0.39399,   0.91911
    0xbfd94e420137bce3,     0x3fed646bb7574de5,     //   -0.3954,   0.91851
    0xbfd96555b7ab948f,     0x3fed5f7172888a7f,     //  -0.39681,    0.9179
    0xbfd97c6583890fc2,     0x3fed5a72a6221e73,     //  -0.39822,   0.91729
    0xbfd993716141bdfe,     0x3fed556f52e93eb1,     //  -0.39962,   0.91668
    0xbfd9aa794d47c9ee,     0x3fed506779a3d2d9,     //  -0.40103,   0.91606
    0xbfd9c17d440df9f2,     0x3fed4b5b1b187524,     //  -0.40243,   0.91545
    0xbfd9d87d4207b0ab,     0x3fed464a380e7242,     //  -0.40384,   0.91483
    0xbfd9ef7943a8ed8a,     0x3fed4134d14dc93a,     //  -0.40524,   0.91421
    0xbfda067145664d57,     0x3fed3c1ae79f2b4e,     //  -0.40664,   0.91359
    0xbfda1d6543b50ac0,     0x3fed36fc7bcbfbdc,     //  -0.40804,   0.91296
    0xbfda34553b0afee5,     0x3fed31d98e9e503a,     //  -0.40944,   0.91234
    0xbfda4b4127dea1e4,     0x3fed2cb220e0ef9f,     //  -0.41084,   0.91171
    0xbfda622906a70b63,     0x3fed2786335f52fc,     //  -0.41224,   0.91107
    0xbfda790cd3dbf31a,     0x3fed2255c6e5a4e1,     //  -0.41364,   0.91044
    0xbfda8fec8bf5b166,     0x3fed1d20dc40c15c,     //  -0.41503,   0.90981
    0xbfdaa6c82b6d3fc9,     0x3fed17e7743e35dc,     //  -0.41643,   0.90917
    0xbfdabd9faebc3980,     0x3fed12a98fac410c,     //  -0.41782,   0.90853
    0xbfdad473125cdc08,     0x3fed0d672f59d2b9,     //  -0.41922,   0.90789
    0xbfdaeb4252ca07ab,     0x3fed082054168bac,     //  -0.42061,   0.90724
    0xbfdb020d6c7f4009,     0x3fed02d4feb2bd92,     //    -0.422,    0.9066
    0xbfdb18d45bf8aca6,     0x3fecfd852fff6ad4,     //  -0.42339,   0.90595
    0xbfdb2f971db31972,     0x3fecf830e8ce467b,     //  -0.42478,    0.9053
    0xbfdb4655ae2bf757,     0x3fecf2d829f1b40e,     //  -0.42617,   0.90464
    0xbfdb5d1009e15cc0,     0x3feced7af43cc773,     //  -0.42756,   0.90399
    0xbfdb73c62d520624,     0x3fece819488344ce,     //  -0.42894,   0.90333
    0xbfdb8a7814fd5693,     0x3fece2b32799a060,     //  -0.43033,   0.90267
    0xbfdba125bd63583e,     0x3fecdd489254fe65,     //  -0.43171,   0.90201
    0xbfdbb7cf2304bd01,     0x3fecd7d9898b32f6,     //  -0.43309,   0.90135
    0xbfdbce744262deee,     0x3fecd2660e12c1e6,     //  -0.43448,   0.90068
    0xbfdbe51517ffc0d9,     0x3fecccee20c2de9f,     //  -0.43586,   0.90002
    0xbfdbfbb1a05e0edc,     0x3fecc771c2736c09,     //  -0.43724,   0.89935
    0xbfdc1249d8011ee7,     0x3fecc1f0f3fcfc5c,     //  -0.43862,   0.89867
    0xbfdc28ddbb6cf145,     0x3fecbc6bb638d10b,     //  -0.43999,     0.898
    0xbfdc3f6d47263129,     0x3fecb6e20a00da99,     //  -0.44137,   0.89732
    0xbfdc55f877b23537,     0x3fecb153f02fb87d,     //  -0.44275,   0.89665
    0xbfdc6c7f4997000a,     0x3fecabc169a0b901,     //  -0.44412,   0.89597
    0xbfdc8301b95b40c2,     0x3feca62a772fd919,     //   -0.4455,   0.89528
    0xbfdc997fc3865388,     0x3feca08f19b9c449,     //  -0.44687,    0.8946
    0xbfdcaff964a0421d,     0x3fec9aef521bd480,     //  -0.44824,   0.89391
    0xbfdcc66e9931c45d,     0x3fec954b213411f5,     //  -0.44961,   0.89322
    0xbfdcdcdf5dc440ce,     0x3fec8fa287e13305,     //  -0.45098,   0.89253
    0xbfdcf34baee1cd21,     0x3fec89f587029c13,     //  -0.45235,   0.89184
    0xbfdd09b389152ec1,     0x3fec84441f785f61,     //  -0.45372,   0.89115
    0xbfdd2016e8e9db5b,     0x3fec7e8e52233cf3,     //  -0.45508,   0.89045
    0xbfdd3675caebf962,     0x3fec78d41fe4a267,     //  -0.45645,   0.88975
    0xbfdd4cd02ba8609c,     0x3fec7315899eaad7,     //  -0.45781,   0.88905
    0xbfdd632607ac9aa9,     0x3fec6d5290341eb2,     //  -0.45918,   0.88835
    0xbfdd79775b86e389,     0x3fec678b3488739b,     //  -0.46054,   0.88764
    0xbfdd8fc423c62a25,     0x3fec61bf777fcc48,     //   -0.4619,   0.88693
    0xbfdda60c5cfa10d8,     0x3fec5bef59fef85a,     //  -0.46326,   0.88622
    0xbfddbc5003b2edf8,     0x3fec561adceb743e,     //  -0.46462,   0.88551
    0xbfddd28f1481cc58,     0x3fec5042012b6907,     //  -0.46598,    0.8848
    0xbfdde8c98bf86bd6,     0x3fec4a64c7a5ac4c,     //  -0.46733,   0.88408
    0xbfddfeff66a941de,     0x3fec44833141c004,     //  -0.46869,   0.88336
    0xbfde1530a12779f4,     0x3fec3e9d3ee7d262,     //  -0.47004,   0.88264
    0xbfde2b5d3806f63b,     0x3fec38b2f180bdb1,     //   -0.4714,   0.88192
    0xbfde418527dc4ffa,     0x3fec32c449f60831,     //  -0.47275,    0.8812
    0xbfde57a86d3cd824,     0x3fec2cd14931e3f1,     //   -0.4741,   0.88047
    0xbfde6dc704be97e2,     0x3fec26d9f01f2eaf,     //  -0.47545,   0.87974
    0xbfde83e0eaf85113,     0x3fec20de3fa971b0,     //   -0.4768,   0.87901
    0xbfde99f61c817eda,     0x3fec1ade38bce19b,     //  -0.47815,   0.87828
    0xbfdeb00695f25620,     0x3fec14d9dc465e58,     //  -0.47949,   0.87755
    0xbfdec61253e3c61b,     0x3fec0ed12b3372e9,     //  -0.48084,   0.87681
    0xbfdedc1952ef78d5,     0x3fec08c426725549,     //  -0.48218,   0.87607
    0xbfdef21b8fafd3b5,     0x3fec02b2cef1e641,     //  -0.48353,   0.87533
    0xbfdf081906bff7fd,     0x3febfc9d25a1b147,     //  -0.48487,   0.87459
    0xbfdf1e11b4bbc35c,     0x3febf6832b71ec5b,     //  -0.48621,   0.87384
    0xbfdf3405963fd068,     0x3febf064e15377dd,     //  -0.48755,   0.87309
    0xbfdf49f4a7e97729,     0x3febea424837de6d,     //  -0.48889,   0.87235
    0xbfdf5fdee656cda3,     0x3febe41b611154c1,     //  -0.49023,    0.8716
    0xbfdf75c44e26a852,     0x3febddf02cd2b983,     //  -0.49156,   0.87084
    0xbfdf8ba4dbf89aba,     0x3febd7c0ac6f952a,     //   -0.4929,   0.87009
    0xbfdfa1808c6cf7e0,     0x3febd18ce0dc19d6,     //  -0.49423,   0.86933
    0xbfdfb7575c24d2de,     0x3febcb54cb0d2327,     //  -0.49557,   0.86857
    0xbfdfcd2947c1ff57,     0x3febc5186bf8361d,     //   -0.4969,   0.86781
    0xbfdfe2f64be7120f,     0x3febbed7c49380ea,     //  -0.49823,   0.86705
    0xbfdff8be6537615e,     0x3febb892d5d5dad5,     //  -0.49956,   0.86628
    0xbfe00740c82b82e0,     0x3febb249a0b6c40d,     //  -0.50089,   0.86551
    0xbfe0121fe4f56d2c,     0x3febabfc262e6586,     //  -0.50221,   0.86474
    0xbfe01cfc874c3eb7,     0x3feba5aa673590d2,     //  -0.50354,   0.86397
    0xbfe027d6ad83287e,     0x3feb9f5464c5bffc,     //  -0.50486,    0.8632
    0xbfe032ae55edbd95,     0x3feb98fa1fd9155e,     //  -0.50619,   0.86242
    0xbfe03d837edff370,     0x3feb929b996a5b7f,     //  -0.50751,   0.86165
    0xbfe0485626ae221a,     0x3feb8c38d27504e9,     //  -0.50883,   0.86087
    0xbfe053264bad0483,     0x3feb85d1cbf52c02,     //  -0.51015,   0.86009
    0xbfe05df3ec31b8b6,     0x3feb7f6686e792ea,     //  -0.51147,    0.8593
    0xbfe068bf0691c028,     0x3feb78f70449a34b,     //  -0.51279,   0.85852
    0xbfe073879922ffed,     0x3feb728345196e3e,     //   -0.5141,   0.85773
    0xbfe07e4da23bc102,     0x3feb6c0b4a55ac17,     //  -0.51542,   0.85694
    0xbfe089112032b08c,     0x3feb658f14fdbc47,     //  -0.51673,   0.85615
    0xbfe093d2115ee018,     0x3feb5f0ea611a532,     //  -0.51804,   0.85535
    0xbfe09e907417c5e1,     0x3feb5889fe921405,     //  -0.51936,   0.85456
    0xbfe0a94c46b53d0b,     0x3feb52011f805c92,     //  -0.52067,   0.85376
    0xbfe0b405878f85ec,     0x3feb4b7409de7925,     //  -0.52198,   0.85296
    0xbfe0bebc34ff4646,     0x3feb44e2beaf0a61,     //  -0.52328,   0.85216
    0xbfe0c9704d5d898f,     0x3feb3e4d3ef55712,     //  -0.52459,   0.85136
    0xbfe0d421cf03c12b,     0x3feb37b38bb54c09,     //   -0.5259,   0.85055
    0xbfe0ded0b84bc4b5,     0x3feb3115a5f37bf4,     //   -0.5272,   0.84974
    0xbfe0e97d078fd23b,     0x3feb2a738eb51f33,     //   -0.5285,   0.84893
    0xbfe0f426bb2a8e7d,     0x3feb23cd470013b4,     //   -0.5298,   0.84812
    0xbfe0fecdd1770537,     0x3feb1d22cfdadcc6,     //   -0.5311,   0.84731
    0xbfe1097248d0a956,     0x3feb16742a4ca2f5,     //   -0.5324,   0.84649
    0xbfe114141f935545,     0x3feb0fc1575d33db,     //   -0.5337,   0.84567
    0xbfe11eb3541b4b22,     0x3feb090a58150200,     //    -0.535,   0.84485
    0xbfe1294fe4c5350a,     0x3feb024f2d7d24a9,     //  -0.53629,   0.84403
    0xbfe133e9cfee254e,     0x3feafb8fd89f57b6,     //  -0.53759,   0.84321
    0xbfe13e8113f396c1,     0x3feaf4cc5a85fb73,     //  -0.53888,   0.84238
    0xbfe14915af336ceb,     0x3feaee04b43c1474,     //  -0.54017,   0.84155
    0xbfe153a7a00bf453,     0x3feae738e6cd4b67,     //  -0.54146,   0.84073
    0xbfe15e36e4dbe2bc,     0x3feae068f345ecef,     //  -0.54275,   0.83989
    0xbfe168c37c025764,     0x3fead994dab2e979,     //  -0.54404,   0.83906
    0xbfe1734d63dedb49,     0x3fead2bc9e21d511,     //  -0.54532,   0.83822
    0xbfe17dd49ad16161,     0x3feacbe03ea0e73b,     //  -0.54661,   0.83739
    0xbfe188591f3a46e5,     0x3feac4ffbd3efac8,     //  -0.54789,   0.83655
    0xbfe192daef7a5386,     0x3feabe1b1b0b8dac,     //  -0.54918,   0.83571
    0xbfe19d5a09f2b9b8,     0x3feab7325916c0d4,     //  -0.55046,   0.83486
    0xbfe1a7d66d0516e6,     0x3feab045787157ff,     //  -0.55174,   0.83402
    0xbfe1b250171373be,     0x3feaa9547a2cb98e,     //  -0.55302,   0.83317
    0xbfe1bcc706804467,     0x3feaa25f5f5aee60,     //  -0.55429,   0.83232
    0xbfe1c73b39ae68c8,     0x3fea9b66290ea1a3,     //  -0.55557,   0.83147
    0xbfe1d1acaf012cc2,     0x3fea9468d85b20ae,     //  -0.55685,   0.83062
    0xbfe1dc1b64dc4872,     0x3fea8d676e545ad2,     //  -0.55812,   0.82976
    0xbfe1e68759a3e074,     0x3fea8661ec0ee133,     //  -0.55939,    0.8289
    0xbfe1f0f08bbc861b,     0x3fea7f58529fe69d,     //  -0.56066,   0.82805
    0xbfe1fb56f98b37b8,     0x3fea784aa31d3f55,     //  -0.56193,   0.82718
    0xbfe205baa17560d6,     0x3fea7138de9d60f5,     //   -0.5632,   0.82632
    0xbfe2101b81e0da78,     0x3fea6a230637623b,     //  -0.56447,   0.82546
    0xbfe21a799933eb58,     0x3fea63091b02fae2,     //  -0.56573,   0.82459
    0xbfe224d4e5d5482e,     0x3fea5beb1e188375,     //    -0.567,   0.82372
    0xbfe22f2d662c13e1,     0x3fea54c91090f524,     //  -0.56826,   0.82285
    0xbfe23983189fdfd5,     0x3fea4da2f385e997,     //  -0.56952,   0.82198
    0xbfe243d5fb98ac1f,     0x3fea4678c8119ac8,     //  -0.57078,    0.8211
    0xbfe24e260d7ee7c9,     0x3fea3f4a8f4ee2d2,     //  -0.57204,   0.82023
    0xbfe258734cbb7110,     0x3fea38184a593bc6,     //   -0.5733,   0.81935
    0xbfe262bdb7b795a2,     0x3fea30e1fa4cbf81,     //  -0.57455,   0.81847
    0xbfe26d054cdd12df,     0x3fea29a7a0462782,     //  -0.57581,   0.81758
    0xbfe2774a0a961612,     0x3fea22693d62ccb9,     //  -0.57706,    0.8167
    0xbfe2818bef4d3cba,     0x3fea1b26d2c0a75e,     //  -0.57831,   0.81581
    0xbfe28bcaf96d94ba,     0x3fea13e0617e4ec7,     //  -0.57956,   0.81493
    0xbfe2960727629ca8,     0x3fea0c95eabaf937,     //  -0.58081,   0.81404
    0xbfe2a040779843fb,     0x3fea05476f967bb5,     //  -0.58206,   0.81314
    0xbfe2aa76e87aeb58,     0x3fe9fdf4f13149de,     //  -0.58331,   0.81225
    0xbfe2b4aa787764c4,     0x3fe9f69e70ac75bc,     //  -0.58455,   0.81135
    0xbfe2bedb25faf3ea,     0x3fe9ef43ef29af94,     //   -0.5858,   0.81046
    0xbfe2c908ef734e57,     0x3fe9e7e56dcb45bd,     //  -0.58704,   0.80956
    0xbfe2d333d34e9bb7,     0x3fe9e082edb42472,     //  -0.58828,   0.80866
    0xbfe2dd5bcffb7616,     0x3fe9d91c7007d5a6,     //  -0.58952,   0.80775
    0xbfe2e780e3e8ea16,     0x3fe9d1b1f5ea80d6,     //  -0.59076,   0.80685
    0xbfe2f1a30d86773a,     0x3fe9ca438080eadb,     //    -0.592,   0.80594
    0xbfe2fbc24b441015,     0x3fe9c2d110f075c3,     //  -0.59323,   0.80503
    0xbfe305de9b921a94,     0x3fe9bb5aa85f2098,     //  -0.59447,   0.80412
    0xbfe30ff7fce17035,     0x3fe9b3e047f38741,     //   -0.5957,   0.80321
    0xbfe31a0e6da35e44,     0x3fe9ac61f0d4e247,     //  -0.59693,   0.80229
    0xbfe32421ec49a620,     0x3fe9a4dfa42b06b2,     //  -0.59816,   0.80138
    0xbfe32e3277467d6b,     0x3fe99d59631e65d5,     //  -0.59939,   0.80046
    0xbfe338400d0c8e57,     0x3fe995cf2ed80d22,     //  -0.60062,   0.79954
    0xbfe3424aac0ef7d6,     0x3fe98e410881a600,     //  -0.60184,   0.79861
    0xbfe34c5252c14de1,     0x3fe986aef1457594,     //  -0.60307,   0.79769
    0xbfe35656ff9799ae,     0x3fe97f18ea4e5c9e,     //  -0.60429,   0.79676
    0xbfe36058b10659f3,     0x3fe9777ef4c7d742,     //  -0.60551,   0.79584
    0xbfe36a576582831b,     0x3fe96fe111ddfce0,     //  -0.60673,   0.79491
    0xbfe374531b817f8d,     0x3fe9683f42bd7fe1,     //  -0.60795,   0.79398
    0xbfe37e4bd1792fe2,     0x3fe960998893ad8c,     //  -0.60917,   0.79304
    0xbfe3884185dfeb22,     0x3fe958efe48e6dd7,     //  -0.61038,   0.79211
    0xbfe39234372c7f04,     0x3fe9514257dc4335,     //   -0.6116,   0.79117
    0xbfe39c23e3d63029,     0x3fe94990e3ac4a6c,     //  -0.61281,   0.79023
    0xbfe3a6108a54ba58,     0x3fe941db892e3a65,     //  -0.61402,   0.78929
    0xbfe3affa292050b9,     0x3fe93a22499263fc,     //  -0.61523,   0.78835
    0xbfe3b9e0beb19e18,     0x3fe932652609b1cf,     //  -0.61644,    0.7874
    0xbfe3c3c44981c517,     0x3fe92aa41fc5a815,     //  -0.61765,   0.78646
    0xbfe3cda4c80a6076,     0x3fe922df37f8646a,     //  -0.61885,   0.78551
    0xbfe3d78238c58343,     0x3fe91b166fd49da2,     //  -0.62006,   0.78456
    0xbfe3e15c9a2db922,     0x3fe91349c88da398,     //  -0.62126,    0.7836
    0xbfe3eb33eabe0680,     0x3fe90b7943575efe,     //  -0.62246,   0.78265
    0xbfe3f50828f1e8d2,     0x3fe903a4e1665133,     //  -0.62366,   0.78169
    0xbfe3fed9534556d4,     0x3fe8fbcca3ef940d,     //  -0.62486,   0.78074
    0xbfe408a76834c0c0,     0x3fe8f3f08c28d9ac,     //  -0.62606,   0.77978
    0xbfe41272663d108c,     0x3fe8ec109b486c49,     //  -0.62725,   0.77882
    0xbfe41c3a4bdbaa26,     0x3fe8e42cd2852e0a,     //  -0.62845,   0.77785
    0xbfe425ff178e6bb1,     0x3fe8dc45331698cc,     //  -0.62964,   0.77689
    0xbfe42fc0c7d3adbb,     0x3fe8d459be34bdfa,     //  -0.63083,   0.77592
    0xbfe4397f5b2a4380,     0x3fe8cc6a75184655,     //  -0.63202,   0.77495
    0xbfe4433ad0117b1d,     0x3fe8c47758fa71cb,     //  -0.63321,   0.77398
    0xbfe44cf325091dd6,     0x3fe8bc806b151741,     //  -0.63439,   0.77301
    0xbfe456a858917046,     0x3fe8b485aca2a468,     //  -0.63558,   0.77204
    0xbfe4605a692b32a2,     0x3fe8ac871ede1d88,     //  -0.63676,   0.77106
    0xbfe46a095557a0f1,     0x3fe8a484c3031d50,     //  -0.63794,   0.77008
    0xbfe473b51b987347,     0x3fe89c7e9a4dd4ab,     //  -0.63912,    0.7691
    0xbfe47d5dba6fde01,     0x3fe89474a5fb0a84,     //   -0.6403,   0.76812
    0xbfe48703306091fe,     0x3fe88c66e7481ba1,     //  -0.64148,   0.76714
    0xbfe490a57bedbcdf,     0x3fe884555f72fa6b,     //  -0.64266,   0.76615
    0xbfe49a449b9b0938,     0x3fe87c400fba2ebf,     //  -0.64383,   0.76517
    0xbfe4a3e08dec9ed6,     0x3fe87426f95cd5bd,     //    -0.645,   0.76418
    0xbfe4ad79516722f0,     0x3fe86c0a1d9aa195,     //  -0.64618,   0.76319
    0xbfe4b70ee48fb869,     0x3fe863e97db3d95a,     //  -0.64735,    0.7622
    0xbfe4c0a145ec0004,     0x3fe85bc51ae958cc,     //  -0.64851,    0.7612
    0xbfe4ca30740218a3,     0x3fe8539cf67c9029,     //  -0.64968,   0.76021
    0xbfe4d3bc6d589f80,     0x3fe84b7111af83f9,     //  -0.65085,   0.75921
    0xbfe4dd453076b064,     0x3fe843416dc4cce2,     //  -0.65201,   0.75821
    0xbfe4e6cabbe3e5e9,     0x3fe83b0e0bff976e,     //  -0.65317,   0.75721
    0xbfe4f04d0e2859aa,     0x3fe832d6eda3a3e0,     //  -0.65433,   0.75621
    0xbfe4f9cc25cca486,     0x3fe82a9c13f545ff,     //  -0.65549,    0.7552
    0xbfe503480159ded2,     0x3fe8225d803964e5,     //  -0.65665,   0.75419
    0xbfe50cc09f59a09b,     0x3fe81a1b33b57acc,     //  -0.65781,   0.75319
    0xbfe51635fe5601d7,     0x3fe811d52faf94dc,     //  -0.65896,   0.75218
    0xbfe51fa81cd99aa6,     0x3fe8098b756e52fa,     //  -0.66011,   0.75117
    0xbfe52916f96f8388,     0x3fe8013e0638e795,     //  -0.66127,   0.75015
    0xbfe5328292a35596,     0x3fe7f8ece3571771,     //  -0.66242,   0.74914
    0xbfe53beae7012abe,     0x3fe7f0980e113978,     //  -0.66356,   0.74812
    0xbfe5454ff5159dfb,     0x3fe7e83f87b03686,     //  -0.66471,    0.7471
    0xbfe54eb1bb6dcb8f,     0x3fe7dfe3517d8937,     //  -0.66586,   0.74608
    0xbfe5581038975137,     0x3fe7d7836cc33db2,     //    -0.667,   0.74506
    0xbfe5616b6b204e6e,     0x3fe7cf1fdacbf179,     //  -0.66814,   0.74403
    0xbfe56ac35197649e,     0x3fe7c6b89ce2d333,     //  -0.66928,   0.74301
    0xbfe57417ea8bb75c,     0x3fe7be4db453a27c,     //  -0.67042,   0.74198
    0xbfe57d69348cec9f,     0x3fe7b5df226aafb0,     //  -0.67156,   0.74095
    0xbfe586b72e2b2cfd,     0x3fe7ad6ce874dbb6,     //  -0.67269,   0.73992
    0xbfe59001d5f723df,     0x3fe7a4f707bf97d2,     //  -0.67383,   0.73889
    0xbfe599492a81ffbc,     0x3fe79c7d8198e56e,     //  -0.67496,   0.73785
    0xbfe5a28d2a5d7250,     0x3fe79400574f55e4,     //  -0.67609,   0.73682
    0xbfe5abcdd41bb0d8,     0x3fe78b7f8a320a52,     //  -0.67722,   0.73578
    0xbfe5b50b264f7448,     0x3fe782fb1b90b35b,     //  -0.67835,   0.73474
    0xbfe5be451f8bf980,     0x3fe77a730cbb9100,     //  -0.67948,    0.7337
    0xbfe5c77bbe65018c,     0x3fe771e75f037261,     //   -0.6806,   0.73265
    0xbfe5d0af016ed1d4,     0x3fe7695813b9b594,     //  -0.68172,   0.73161
    0xbfe5d9dee73e345c,     0x3fe760c52c304764,     //  -0.68285,   0.73056
    0xbfe5e30b6e6877f3,     0x3fe7582ea9b9a329,     //  -0.68397,   0.72951
    0xbfe5ec3495837074,     0x3fe74f948da8d28d,     //  -0.68508,   0.72846
    0xbfe5f55a5b2576f8,     0x3fe746f6d9516d59,     //   -0.6862,   0.72741
    0xbfe5fe7cbde56a0f,     0x3fe73e558e079942,     //  -0.68732,   0.72636
    0xbfe6079bbc5aadfa,     0x3fe735b0ad2009b2,     //  -0.68843,    0.7253
    0xbfe610b7551d2cde,     0x3fe72d0837efff97,     //  -0.68954,   0.72425
    0xbfe619cf86c55702,     0x3fe7245c2fcd492a,     //  -0.69065,   0.72319
    0xbfe622e44fec22ff,     0x3fe71bac960e41bf,     //  -0.69176,   0.72213
    0xbfe62bf5af2b0dfd,     0x3fe712f96c09d18d,     //  -0.69287,   0.72107
    0xbfe63503a31c1be8,     0x3fe70a42b3176d7a,     //  -0.69397,      0.72
    0xbfe63e0e2a59d7aa,     0x3fe701886c8f16e6,     //  -0.69508,   0.71894
    0xbfe64715437f535b,     0x3fe6f8ca99c95b75,     //  -0.69618,   0.71787
    0xbfe65018ed28287f,     0x3fe6f0093c1f54de,     //  -0.69728,    0.7168
    0xbfe6591925f0783e,     0x3fe6e74454eaa8ae,     //  -0.69838,   0.71573
    0xbfe66215ec74eb91,     0x3fe6de7be585881d,     //  -0.69947,   0.71466
    0xbfe66b0f3f52b386,     0x3fe6d5afef4aafcc,     //  -0.70057,   0.71358
    0xbfe674051d27896c,     0x3fe6cce07395679f,     //  -0.70166,   0.71251
    0xbfe67cf78491af10,     0x3fe6c40d73c18275,     //  -0.70275,   0.71143
    0xbfe685e6742feeef,     0x3fe6bb36f12b5e06,     //  -0.70385,   0.71035
    0xbfe68ed1eaa19c71,     0x3fe6b25ced2fe29c,     //  -0.70493,   0.70927
    0xbfe697b9e686941c,     0x3fe6a97f692c82ea,     //  -0.70602,   0.70819
    0xbfe6a09e667f3bcc,     0x3fe6a09e667f3bcc,     //  -0.70711,   0.70711
    0xbfe6a97f692c82ea,     0x3fe697b9e686941c,     //  -0.70819,   0.70602
    0xbfe6b25ced2fe29c,     0x3fe68ed1eaa19c71,     //  -0.70927,   0.70493
    0xbfe6bb36f12b5e06,     0x3fe685e6742feeef,     //  -0.71035,   0.70385
    0xbfe6c40d73c18275,     0x3fe67cf78491af10,     //  -0.71143,   0.70275
    0xbfe6cce07395679f,     0x3fe674051d27896c,     //  -0.71251,   0.70166
    0xbfe6d5afef4aafcc,     0x3fe66b0f3f52b386,     //  -0.71358,   0.70057
    0xbfe6de7be585881d,     0x3fe66215ec74eb91,     //  -0.71466,   0.69947
    0xbfe6e74454eaa8ae,     0x3fe6591925f0783e,     //  -0.71573,   0.69838
    0xbfe6f0093c1f54de,     0x3fe65018ed28287f,     //   -0.7168,   0.69728
    0xbfe6f8ca99c95b75,     0x3fe64715437f535b,     //  -0.71787,   0.69618
    0xbfe701886c8f16e6,     0x3fe63e0e2a59d7aa,     //  -0.71894,   0.69508
    0xbfe70a42b3176d7a,     0x3fe63503a31c1be8,     //     -0.72,   0.69397
    0xbfe712f96c09d18d,     0x3fe62bf5af2b0dfd,     //  -0.72107,   0.69287
    0xbfe71bac960e41bf,     0x3fe622e44fec22ff,     //  -0.72213,   0.69176
    0xbfe7245c2fcd492a,     0x3fe619cf86c55702,     //  -0.72319,   0.69065
    0xbfe72d0837efff97,     0x3fe610b7551d2cde,     //  -0.72425,   0.68954
    0xbfe735b0ad2009b2,     0x3fe6079bbc5aadfa,     //   -0.7253,   0.68843
    0xbfe73e558e079942,     0x3fe5fe7cbde56a0f,     //  -0.72636,   0.68732
    0xbfe746f6d9516d59,     0x3fe5f55a5b2576f8,     //  -0.72741,    0.6862
    0xbfe74f948da8d28d,     0x3fe5ec3495837074,     //  -0.72846,   0.68508
    0xbfe7582ea9b9a329,     0x3fe5e30b6e6877f3,     //  -0.72951,   0.68397
    0xbfe760c52c304764,     0x3fe5d9dee73e345c,     //  -0.73056,   0.68285
    0xbfe7695813b9b594,     0x3fe5d0af016ed1d4,     //  -0.73161,   0.68172
    0xbfe771e75f037261,     0x3fe5c77bbe65018c,     //  -0.73265,    0.6806
    0xbfe77a730cbb9100,     0x3fe5be451f8bf980,     //   -0.7337,   0.67948
    0xbfe782fb1b90b35b,     0x3fe5b50b264f7448,     //  -0.73474,   0.67835
    0xbfe78b7f8a320a52,     0x3fe5abcdd41bb0d8,     //  -0.73578,   0.67722
    0xbfe79400574f55e4,     0x3fe5a28d2a5d7250,     //  -0.73682,   0.67609
    0xbfe79c7d8198e56e,     0x3fe599492a81ffbc,     //  -0.73785,   0.67496
    0xbfe7a4f707bf97d2,     0x3fe59001d5f723df,     //  -0.73889,   0.67383
    0xbfe7ad6ce874dbb6,     0x3fe586b72e2b2cfd,     //  -0.73992,   0.67269
    0xbfe7b5df226aafb0,     0x3fe57d69348cec9f,     //  -0.74095,   0.67156
    0xbfe7be4db453a27c,     0x3fe57417ea8bb75c,     //  -0.74198,   0.67042
    0xbfe7c6b89ce2d333,     0x3fe56ac35197649e,     //  -0.74301,   0.66928
    0xbfe7cf1fdacbf179,     0x3fe5616b6b204e6e,     //  -0.74403,   0.66814
    0xbfe7d7836cc33db2,     0x3fe5581038975137,     //  -0.74506,     0.667
    0xbfe7dfe3517d8937,     0x3fe54eb1bb6dcb8f,     //  -0.74608,   0.66586
    0xbfe7e83f87b03686,     0x3fe5454ff5159dfb,     //   -0.7471,   0.66471
    0xbfe7f0980e113978,     0x3fe53beae7012abe,     //  -0.74812,   0.66356
    0xbfe7f8ece3571771,     0x3fe5328292a35596,     //  -0.74914,   0.66242
    0xbfe8013e0638e795,     0x3fe52916f96f8388,     //  -0.75015,   0.66127
    0xbfe8098b756e52fa,     0x3fe51fa81cd99aa6,     //  -0.75117,   0.66011
    0xbfe811d52faf94dc,     0x3fe51635fe5601d7,     //  -0.75218,   0.65896
    0xbfe81a1b33b57acc,     0x3fe50cc09f59a09b,     //  -0.75319,   0.65781
    0xbfe8225d803964e5,     0x3fe503480159ded2,     //  -0.75419,   0.65665
    0xbfe82a9c13f545ff,     0x3fe4f9cc25cca486,     //   -0.7552,   0.65549
    0xbfe832d6eda3a3e0,     0x3fe4f04d0e2859aa,     //  -0.75621,   0.65433
    0xbfe83b0e0bff976e,     0x3fe4e6cabbe3e5e9,     //  -0.75721,   0.65317
    0xbfe843416dc4cce2,     0x3fe4dd453076b064,     //  -0.75821,   0.65201
    0xbfe84b7111af83f9,     0x3fe4d3bc6d589f80,     //  -0.75921,   0.65085
    0xbfe8539cf67c9029,     0x3fe4ca30740218a3,     //  -0.76021,   0.64968
    0xbfe85bc51ae958cc,     0x3fe4c0a145ec0004,     //   -0.7612,   0.64851
    0xbfe863e97db3d95a,     0x3fe4b70ee48fb869,     //   -0.7622,   0.64735
    0xbfe86c0a1d9aa195,     0x3fe4ad79516722f0,     //  -0.76319,   0.64618
    0xbfe87426f95cd5bd,     0x3fe4a3e08dec9ed6,     //  -0.76418,     0.645
    0xbfe87c400fba2ebf,     0x3fe49a449b9b0938,     //  -0.76517,   0.64383
    0xbfe884555f72fa6b,     0x3fe490a57bedbcdf,     //  -0.76615,   0.64266
    0xbfe88c66e7481ba1,     0x3fe48703306091fe,     //  -0.76714,   0.64148
    0xbfe89474a5fb0a84,     0x3fe47d5dba6fde01,     //  -0.76812,    0.6403
    0xbfe89c7e9a4dd4ab,     0x3fe473b51b987347,     //   -0.7691,   0.63912
    0xbfe8a484c3031d50,     0x3fe46a095557a0f1,     //  -0.77008,   0.63794
    0xbfe8ac871ede1d88,     0x3fe4605a692b32a2,     //  -0.77106,   0.63676
    0xbfe8b485aca2a468,     0x3fe456a858917046,     //  -0.77204,   0.63558
    0xbfe8bc806b151741,     0x3fe44cf325091dd6,     //  -0.77301,   0.63439
    0xbfe8c47758fa71cb,     0x3fe4433ad0117b1d,     //  -0.77398,   0.63321
    0xbfe8cc6a75184655,     0x3fe4397f5b2a4380,     //  -0.77495,   0.63202
    0xbfe8d459be34bdfa,     0x3fe42fc0c7d3adbb,     //  -0.77592,   0.63083
    0xbfe8dc45331698cc,     0x3fe425ff178e6bb1,     //  -0.77689,   0.62964
    0xbfe8e42cd2852e0a,     0x3fe41c3a4bdbaa26,     //  -0.77785,   0.62845
    0xbfe8ec109b486c49,     0x3fe41272663d108c,     //  -0.77882,   0.62725
    0xbfe8f3f08c28d9ac,     0x3fe408a76834c0c0,     //  -0.77978,   0.62606
    0xbfe8fbcca3ef940d,     0x3fe3fed9534556d4,     //  -0.78074,   0.62486
    0xbfe903a4e1665133,     0x3fe3f50828f1e8d2,     //  -0.78169,   0.62366
    0xbfe90b7943575efe,     0x3fe3eb33eabe0680,     //  -0.78265,   0.62246
    0xbfe91349c88da398,     0x3fe3e15c9a2db922,     //   -0.7836,   0.62126
    0xbfe91b166fd49da2,     0x3fe3d78238c58343,     //  -0.78456,   0.62006
    0xbfe922df37f8646a,     0x3fe3cda4c80a6076,     //  -0.78551,   0.61885
    0xbfe92aa41fc5a815,     0x3fe3c3c44981c517,     //  -0.78646,   0.61765
    0xbfe932652609b1cf,     0x3fe3b9e0beb19e18,     //   -0.7874,   0.61644
    0xbfe93a22499263fc,     0x3fe3affa292050b9,     //  -0.78835,   0.61523
    0xbfe941db892e3a65,     0x3fe3a6108a54ba58,     //  -0.78929,   0.61402
    0xbfe94990e3ac4a6c,     0x3fe39c23e3d63029,     //  -0.79023,   0.61281
    0xbfe9514257dc4335,     0x3fe39234372c7f04,     //  -0.79117,    0.6116
    0xbfe958efe48e6dd7,     0x3fe3884185dfeb22,     //  -0.79211,   0.61038
    0xbfe960998893ad8c,     0x3fe37e4bd1792fe2,     //  -0.79304,   0.60917
    0xbfe9683f42bd7fe1,     0x3fe374531b817f8d,     //  -0.79398,   0.60795
    0xbfe96fe111ddfce0,     0x3fe36a576582831b,     //  -0.79491,   0.60673
    0xbfe9777ef4c7d742,     0x3fe36058b10659f3,     //  -0.79584,   0.60551
    0xbfe97f18ea4e5c9e,     0x3fe35656ff9799ae,     //  -0.79676,   0.60429
    0xbfe986aef1457594,     0x3fe34c5252c14de1,     //  -0.79769,   0.60307
    0xbfe98e410881a600,     0x3fe3424aac0ef7d6,     //  -0.79861,   0.60184
    0xbfe995cf2ed80d22,     0x3fe338400d0c8e57,     //  -0.79954,   0.60062
    0xbfe99d59631e65d5,     0x3fe32e3277467d6b,     //  -0.80046,   0.59939
    0xbfe9a4dfa42b06b2,     0x3fe32421ec49a620,     //  -0.80138,   0.59816
    0xbfe9ac61f0d4e247,     0x3fe31a0e6da35e44,     //  -0.80229,   0.59693
    0xbfe9b3e047f38741,     0x3fe30ff7fce17035,     //  -0.80321,    0.5957
    0xbfe9bb5aa85f2098,     0x3fe305de9b921a94,     //  -0.80412,   0.59447
    0xbfe9c2d110f075c3,     0x3fe2fbc24b441015,     //  -0.80503,   0.59323
    0xbfe9ca438080eadb,     0x3fe2f1a30d86773a,     //  -0.80594,     0.592
    0xbfe9d1b1f5ea80d6,     0x3fe2e780e3e8ea16,     //  -0.80685,   0.59076
    0xbfe9d91c7007d5a6,     0x3fe2dd5bcffb7616,     //  -0.80775,   0.58952
    0xbfe9e082edb42472,     0x3fe2d333d34e9bb7,     //  -0.80866,   0.58828
    0xbfe9e7e56dcb45bd,     0x3fe2c908ef734e57,     //  -0.80956,   0.58704
    0xbfe9ef43ef29af94,     0x3fe2bedb25faf3ea,     //  -0.81046,    0.5858
    0xbfe9f69e70ac75bc,     0x3fe2b4aa787764c4,     //  -0.81135,   0.58455
    0xbfe9fdf4f13149de,     0x3fe2aa76e87aeb58,     //  -0.81225,   0.58331
    0xbfea05476f967bb5,     0x3fe2a040779843fb,     //  -0.81314,   0.58206
    0xbfea0c95eabaf937,     0x3fe2960727629ca8,     //  -0.81404,   0.58081
    0xbfea13e0617e4ec7,     0x3fe28bcaf96d94ba,     //  -0.81493,   0.57956
    0xbfea1b26d2c0a75e,     0x3fe2818bef4d3cba,     //  -0.81581,   0.57831
    0xbfea22693d62ccb9,     0x3fe2774a0a961612,     //   -0.8167,   0.57706
    0xbfea29a7a0462782,     0x3fe26d054cdd12df,     //  -0.81758,   0.57581
    0xbfea30e1fa4cbf81,     0x3fe262bdb7b795a2,     //  -0.81847,   0.57455
    0xbfea38184a593bc6,     0x3fe258734cbb7110,     //  -0.81935,    0.5733
    0xbfea3f4a8f4ee2d2,     0x3fe24e260d7ee7c9,     //  -0.82023,   0.57204
    0xbfea4678c8119ac8,     0x3fe243d5fb98ac1f,     //   -0.8211,   0.57078
    0xbfea4da2f385e997,     0x3fe23983189fdfd5,     //  -0.82198,   0.56952
    0xbfea54c91090f524,     0x3fe22f2d662c13e1,     //  -0.82285,   0.56826
    0xbfea5beb1e188375,     0x3fe224d4e5d5482e,     //  -0.82372,     0.567
    0xbfea63091b02fae2,     0x3fe21a799933eb58,     //  -0.82459,   0.56573
    0xbfea6a230637623b,     0x3fe2101b81e0da78,     //  -0.82546,   0.56447
    0xbfea7138de9d60f5,     0x3fe205baa17560d6,     //  -0.82632,    0.5632
    0xbfea784aa31d3f55,     0x3fe1fb56f98b37b8,     //  -0.82718,   0.56193
    0xbfea7f58529fe69d,     0x3fe1f0f08bbc861b,     //  -0.82805,   0.56066
    0xbfea8661ec0ee133,     0x3fe1e68759a3e074,     //   -0.8289,   0.55939
    0xbfea8d676e545ad2,     0x3fe1dc1b64dc4872,     //  -0.82976,   0.55812
    0xbfea9468d85b20ae,     0x3fe1d1acaf012cc2,     //  -0.83062,   0.55685
    0xbfea9b66290ea1a3,     0x3fe1c73b39ae68c8,     //  -0.83147,   0.55557
    0xbfeaa25f5f5aee60,     0x3fe1bcc706804467,     //  -0.83232,   0.55429
    0xbfeaa9547a2cb98e,     0x3fe1b250171373be,     //  -0.83317,   0.55302
    0xbfeab045787157ff,     0x3fe1a7d66d0516e6,     //  -0.83402,   0.55174
    0xbfeab7325916c0d4,     0x3fe19d5a09f2b9b8,     //  -0.83486,   0.55046
    0xbfeabe1b1b0b8dac,     0x3fe192daef7a5386,     //  -0.83571,   0.54918
    0xbfeac4ffbd3efac8,     0x3fe188591f3a46e5,     //  -0.83655,   0.54789
    0xbfeacbe03ea0e73b,     0x3fe17dd49ad16161,     //  -0.83739,   0.54661
    0xbfead2bc9e21d511,     0x3fe1734d63dedb49,     //  -0.83822,   0.54532
    0xbfead994dab2e979,     0x3fe168c37c025764,     //  -0.83906,   0.54404
    0xbfeae068f345ecef,     0x3fe15e36e4dbe2bc,     //  -0.83989,   0.54275
    0xbfeae738e6cd4b67,     0x3fe153a7a00bf453,     //  -0.84073,   0.54146
    0xbfeaee04b43c1474,     0x3fe14915af336ceb,     //  -0.84155,   0.54017
    0xbfeaf4cc5a85fb73,     0x3fe13e8113f396c1,     //  -0.84238,   0.53888
    0xbfeafb8fd89f57b6,     0x3fe133e9cfee254e,     //  -0.84321,   0.53759
    0xbfeb024f2d7d24a9,     0x3fe1294fe4c5350a,     //  -0.84403,   0.53629
    0xbfeb090a58150200,     0x3fe11eb3541b4b22,     //  -0.84485,     0.535
    0xbfeb0fc1575d33db,     0x3fe114141f935545,     //  -0.84567,    0.5337
    0xbfeb16742a4ca2f5,     0x3fe1097248d0a956,     //  -0.84649,    0.5324
    0xbfeb1d22cfdadcc6,     0x3fe0fecdd1770537,     //  -0.84731,    0.5311
    0xbfeb23cd470013b4,     0x3fe0f426bb2a8e7d,     //  -0.84812,    0.5298
    0xbfeb2a738eb51f33,     0x3fe0e97d078fd23b,     //  -0.84893,    0.5285
    0xbfeb3115a5f37bf4,     0x3fe0ded0b84bc4b5,     //  -0.84974,    0.5272
    0xbfeb37b38bb54c09,     0x3fe0d421cf03c12b,     //  -0.85055,    0.5259
    0xbfeb3e4d3ef55712,     0x3fe0c9704d5d898f,     //  -0.85136,   0.52459
    0xbfeb44e2beaf0a61,     0x3fe0bebc34ff4646,     //  -0.85216,   0.52328
    0xbfeb4b7409de7925,     0x3fe0b405878f85ec,     //  -0.85296,   0.52198
    0xbfeb52011f805c92,     0x3fe0a94c46b53d0b,     //  -0.85376,   0.52067
    0xbfeb5889fe921405,     0x3fe09e907417c5e1,     //  -0.85456,   0.51936
    0xbfeb5f0ea611a532,     0x3fe093d2115ee018,     //  -0.85535,   0.51804
    0xbfeb658f14fdbc47,     0x3fe089112032b08c,     //  -0.85615,   0.51673
    0xbfeb6c0b4a55ac17,     0x3fe07e4da23bc102,     //  -0.85694,   0.51542
    0xbfeb728345196e3e,     0x3fe073879922ffed,     //  -0.85773,    0.5141
    0xbfeb78f70449a34b,     0x3fe068bf0691c028,     //  -0.85852,   0.51279
    0xbfeb7f6686e792ea,     0x3fe05df3ec31b8b6,     //   -0.8593,   0.51147
    0xbfeb85d1cbf52c02,     0x3fe053264bad0483,     //  -0.86009,   0.51015
    0xbfeb8c38d27504e9,     0x3fe0485626ae221a,     //  -0.86087,   0.50883
    0xbfeb929b996a5b7f,     0x3fe03d837edff370,     //  -0.86165,   0.50751
    0xbfeb98fa1fd9155e,     0x3fe032ae55edbd95,     //  -0.86242,   0.50619
    0xbfeb9f5464c5bffc,     0x3fe027d6ad83287e,     //   -0.8632,   0.50486
    0xbfeba5aa673590d2,     0x3fe01cfc874c3eb7,     //  -0.86397,   0.50354
    0xbfebabfc262e6586,     0x3fe0121fe4f56d2c,     //  -0.86474,   0.50221
    0xbfebb249a0b6c40d,     0x3fe00740c82b82e0,     //  -0.86551,   0.50089
    0xbfebb892d5d5dad5,     0x3fdff8be6537615e,     //  -0.86628,   0.49956
    0xbfebbed7c49380ea,     0x3fdfe2f64be7120f,     //  -0.86705,   0.49823
    0xbfebc5186bf8361d,     0x3fdfcd2947c1ff57,     //  -0.86781,    0.4969
    0xbfebcb54cb0d2327,     0x3fdfb7575c24d2de,     //  -0.86857,   0.49557
    0xbfebd18ce0dc19d6,     0x3fdfa1808c6cf7e0,     //  -0.86933,   0.49423
    0xbfebd7c0ac6f952a,     0x3fdf8ba4dbf89aba,     //  -0.87009,    0.4929
    0xbfebddf02cd2b983,     0x3fdf75c44e26a852,     //  -0.87084,   0.49156
    0xbfebe41b611154c1,     0x3fdf5fdee656cda3,     //   -0.8716,   0.49023
    0xbfebea424837de6d,     0x3fdf49f4a7e97729,     //  -0.87235,   0.48889
    0xbfebf064e15377dd,     0x3fdf3405963fd068,     //  -0.87309,   0.48755
    0xbfebf6832b71ec5b,     0x3fdf1e11b4bbc35c,     //  -0.87384,   0.48621
    0xbfebfc9d25a1b147,     0x3fdf081906bff7fd,     //  -0.87459,   0.48487
    0xbfec02b2cef1e641,     0x3fdef21b8fafd3b5,     //  -0.87533,   0.48353
    0xbfec08c426725549,     0x3fdedc1952ef78d5,     //  -0.87607,   0.48218
    0xbfec0ed12b3372e9,     0x3fdec61253e3c61b,     //  -0.87681,   0.48084
    0xbfec14d9dc465e58,     0x3fdeb00695f25620,     //  -0.87755,   0.47949
    0xbfec1ade38bce19b,     0x3fde99f61c817eda,     //  -0.87828,   0.47815
    0xbfec20de3fa971b0,     0x3fde83e0eaf85113,     //  -0.87901,    0.4768
    0xbfec26d9f01f2eaf,     0x3fde6dc704be97e2,     //  -0.87974,   0.47545
    0xbfec2cd14931e3f1,     0x3fde57a86d3cd824,     //  -0.88047,    0.4741
    0xbfec32c449f60831,     0x3fde418527dc4ffa,     //   -0.8812,   0.47275
    0xbfec38b2f180bdb1,     0x3fde2b5d3806f63b,     //  -0.88192,    0.4714
    0xbfec3e9d3ee7d262,     0x3fde1530a12779f4,     //  -0.88264,   0.47004
    0xbfec44833141c004,     0x3fddfeff66a941de,     //  -0.88336,   0.46869
    0xbfec4a64c7a5ac4c,     0x3fdde8c98bf86bd6,     //  -0.88408,   0.46733
    0xbfec5042012b6907,     0x3fddd28f1481cc58,     //   -0.8848,   0.46598
    0xbfec561adceb743e,     0x3fddbc5003b2edf8,     //  -0.88551,   0.46462
    0xbfec5bef59fef85a,     0x3fdda60c5cfa10d8,     //  -0.88622,   0.46326
    0xbfec61bf777fcc48,     0x3fdd8fc423c62a25,     //  -0.88693,    0.4619
    0xbfec678b3488739b,     0x3fdd79775b86e389,     //  -0.88764,   0.46054
    0xbfec6d5290341eb2,     0x3fdd632607ac9aa9,     //  -0.88835,   0.45918
    0xbfec7315899eaad7,     0x3fdd4cd02ba8609c,     //  -0.88905,   0.45781
    0xbfec78d41fe4a267,     0x3fdd3675caebf962,     //  -0.88975,   0.45645
    0xbfec7e8e52233cf3,     0x3fdd2016e8e9db5b,     //  -0.89045,   0.45508
    0xbfec84441f785f61,     0x3fdd09b389152ec1,     //  -0.89115,   0.45372
    0xbfec89f587029c13,     0x3fdcf34baee1cd21,     //  -0.89184,   0.45235
    0xbfec8fa287e13305,     0x3fdcdcdf5dc440ce,     //  -0.89253,   0.45098
    0xbfec954b213411f5,     0x3fdcc66e9931c45d,     //  -0.89322,   0.44961
    0xbfec9aef521bd480,     0x3fdcaff964a0421d,     //  -0.89391,   0.44824
    0xbfeca08f19b9c449,     0x3fdc997fc3865388,     //   -0.8946,   0.44687
    0xbfeca62a772fd919,     0x3fdc8301b95b40c2,     //  -0.89528,    0.4455
    0xbfecabc169a0b901,     0x3fdc6c7f4997000a,     //  -0.89597,   0.44412
    0xbfecb153f02fb87d,     0x3fdc55f877b23537,     //  -0.89665,   0.44275
    0xbfecb6e20a00da99,     0x3fdc3f6d47263129,     //  -0.89732,   0.44137
    0xbfecbc6bb638d10b,     0x3fdc28ddbb6cf145,     //    -0.898,   0.43999
    0xbfecc1f0f3fcfc5c,     0x3fdc1249d8011ee7,     //  -0.89867,   0.43862
    0xbfecc771c2736c09,     0x3fdbfbb1a05e0edc,     //  -0.89935,   0.43724
    0xbfecccee20c2de9f,     0x3fdbe51517ffc0d9,     //  -0.90002,   0.43586
    0xbfecd2660e12c1e6,     0x3fdbce744262deee,     //  -0.90068,   0.43448
    0xbfecd7d9898b32f6,     0x3fdbb7cf2304bd01,     //  -0.90135,   0.43309
    0xbfecdd489254fe65,     0x3fdba125bd63583e,     //  -0.90201,   0.43171
    0xbfece2b32799a060,     0x3fdb8a7814fd5693,     //  -0.90267,   0.43033
    0xbfece819488344ce,     0x3fdb73c62d520624,     //  -0.90333,   0.42894
    0xbfeced7af43cc773,     0x3fdb5d1009e15cc0,     //  -0.90399,   0.42756
    0xbfecf2d829f1b40e,     0x3fdb4655ae2bf757,     //  -0.90464,   0.42617
    0xbfecf830e8ce467b,     0x3fdb2f971db31972,     //   -0.9053,   0.42478
    0xbfecfd852fff6ad4,     0x3fdb18d45bf8aca6,     //  -0.90595,   0.42339
    0xbfed02d4feb2bd92,     0x3fdb020d6c7f4009,     //   -0.9066,     0.422
    0xbfed082054168bac,     0x3fdaeb4252ca07ab,     //  -0.90724,   0.42061
    0xbfed0d672f59d2b9,     0x3fdad473125cdc08,     //  -0.90789,   0.41922
    0xbfed12a98fac410c,     0x3fdabd9faebc3980,     //  -0.90853,   0.41782
    0xbfed17e7743e35dc,     0x3fdaa6c82b6d3fc9,     //  -0.90917,   0.41643
    0xbfed1d20dc40c15c,     0x3fda8fec8bf5b166,     //  -0.90981,   0.41503
    0xbfed2255c6e5a4e1,     0x3fda790cd3dbf31a,     //  -0.91044,   0.41364
    0xbfed2786335f52fc,     0x3fda622906a70b63,     //  -0.91107,   0.41224
    0xbfed2cb220e0ef9f,     0x3fda4b4127dea1e4,     //  -0.91171,   0.41084
    0xbfed31d98e9e503a,     0x3fda34553b0afee5,     //  -0.91234,   0.40944
    0xbfed36fc7bcbfbdc,     0x3fda1d6543b50ac0,     //  -0.91296,   0.40804
    0xbfed3c1ae79f2b4e,     0x3fda067145664d57,     //  -0.91359,   0.40664
    0xbfed4134d14dc93a,     0x3fd9ef7943a8ed8a,     //  -0.91421,   0.40524
    0xbfed464a380e7242,     0x3fd9d87d4207b0ab,     //  -0.91483,   0.40384
    0xbfed4b5b1b187524,     0x3fd9c17d440df9f2,     //  -0.91545,   0.40243
    0xbfed506779a3d2d9,     0x3fd9aa794d47c9ee,     //  -0.91606,   0.40103
    0xbfed556f52e93eb1,     0x3fd993716141bdfe,     //  -0.91668,   0.39962
    0xbfed5a72a6221e73,     0x3fd97c6583890fc2,     //  -0.91729,   0.39822
    0xbfed5f7172888a7f,     0x3fd96555b7ab948f,     //   -0.9179,   0.39681
    0xbfed646bb7574de5,     0x3fd94e420137bce3,     //  -0.91851,    0.3954
    0xbfed696173c9e68b,     0x3fd9372a63bc93d7,     //  -0.91911,   0.39399
    0xbfed6e52a71c8547,     0x3fd9200ee2c9be97,     //  -0.91972,   0.39258
    0xbfed733f508c0dff,     0x3fd908ef81ef7bd1,     //  -0.92032,   0.39117
    0xbfed78276f5617c6,     0x3fd8f1cc44bea329,     //  -0.92092,   0.38976
    0xbfed7d0b02b8ecf9,     0x3fd8daa52ec8a4af,     //  -0.92151,   0.38835
    0xbfed81ea09f38b63,     0x3fd8c37a439f884f,     //  -0.92211,   0.38693
    0xbfed86c48445a450,     0x3fd8ac4b86d5ed44,     //   -0.9227,   0.38552
    0xbfed8b9a70ef9cb4,     0x3fd89518fbff098e,     //  -0.92329,    0.3841
    0xbfed906bcf328d46,     0x3fd87de2a6aea963,     //  -0.92388,   0.38268
    0xbfed95389e50429b,     0x3fd866a88a792ea0,     //  -0.92447,   0.38127
    0xbfed9a00dd8b3d46,     0x3fd84f6aaaf3903f,     //  -0.92505,   0.37985
    0xbfed9ec48c26b1f3,     0x3fd838290bb359c8,     //  -0.92563,   0.37843
    0xbfeda383a9668988,     0x3fd820e3b04eaac4,     //  -0.92621,   0.37701
    0xbfeda83e348f613b,     0x3fd8099a9c5c362d,     //  -0.92679,   0.37559
    0xbfedacf42ce68ab9,     0x3fd7f24dd37341e3,     //  -0.92736,   0.37416
    0xbfedb1a591b20c38,     0x3fd7dafd592ba621,     //  -0.92794,   0.37274
    0xbfedb6526238a09b,     0x3fd7c3a9311dcce7,     //  -0.92851,   0.37132
    0xbfedbafa9dc1b78d,     0x3fd7ac515ee2b172,     //  -0.92907,   0.36989
    0xbfedbf9e4395759a,     0x3fd794f5e613dfae,     //  -0.92964,   0.36847
    0xbfedc43d52fcb453,     0x3fd77d96ca4b73a6,     //  -0.93021,   0.36704
    0xbfedc8d7cb410260,     0x3fd766340f2418f6,     //  -0.93077,   0.36561
    0xbfedcd6dabaca3a5,     0x3fd74ecdb8390a3e,     //  -0.93133,   0.36418
    0xbfedd1fef38a915a,     0x3fd73763c9261092,     //  -0.93188,   0.36276
    0xbfedd68ba2267a25,     0x3fd71ff6458782ec,     //  -0.93244,   0.36133
    0xbfeddb13b6ccc23d,     0x3fd7088530fa459e,     //  -0.93299,    0.3599
    0xbfeddf9730ca837b,     0x3fd6f1108f1bc9c5,     //  -0.93354,   0.35846
    0xbfede4160f6d8d81,     0x3fd6d998638a0cb5,     //  -0.93409,   0.35703
    0xbfede890520465ce,     0x3fd6c21cb1e39771,     //  -0.93464,    0.3556
    0xbfeded05f7de47da,     0x3fd6aa9d7dc77e16,     //  -0.93518,   0.35416
    0xbfedf177004b2534,     0x3fd6931acad55f51,     //  -0.93573,   0.35273
    0xbfedf5e36a9ba59c,     0x3fd67b949cad63ca,     //  -0.93627,   0.35129
    0xbfedfa4b3621271d,     0x3fd6640af6f03d9e,     //   -0.9368,   0.34986
    0xbfedfeae622dbe2b,     0x3fd64c7ddd3f27c6,     //  -0.93734,   0.34842
    0xbfee030cee1435b8,     0x3fd634ed533be58e,     //  -0.93787,   0.34698
    0xbfee0766d9280f54,     0x3fd61d595c88c203,     //   -0.9384,   0.34554
    0xbfee0bbc22bd8349,     0x3fd605c1fcc88f63,     //  -0.93893,    0.3441
    0xbfee100cca2980ac,     0x3fd5ee27379ea693,     //  -0.93946,   0.34266
    0xbfee1458cec1ad83,     0x3fd5d68910aee686,     //  -0.93998,   0.34122
    0xbfee18a02fdc66d9,     0x3fd5bee78b9db3b6,     //  -0.94051,   0.33978
    0xbfee1ce2ecd0c0d8,     0x3fd5a742ac0ff78d,     //  -0.94103,   0.33833
    0xbfee212104f686e5,     0x3fd58f9a75ab1fdd,     //  -0.94154,   0.33689
    0xbfee255a77a63bb8,     0x3fd577eeec151e47,     //  -0.94206,   0.33545
    0xbfee298f4439197a,     0x3fd5604012f467b4,     //  -0.94257,     0.334
    0xbfee2dbf6a0911d9,     0x3fd5488dedeff3be,     //  -0.94308,   0.33255
    0xbfee31eae870ce25,     0x3fd530d880af3c24,     //  -0.94359,   0.33111
    0xbfee3611becbaf69,     0x3fd5191fceda3c35,     //   -0.9441,   0.32966
    0xbfee3a33ec75ce85,     0x3fd50163dc197047,     //   -0.9446,   0.32821
    0xbfee3e5170cbfc46,     0x3fd4e9a4ac15d520,     //  -0.94511,   0.32676
    0xbfee426a4b2bc17e,     0x3fd4d1e24278e76a,     //  -0.94561,   0.32531
    0xbfee467e7af35f23,     0x3fd4ba1ca2eca31c,     //  -0.94611,   0.32386
    0xbfee4a8dff81ce5e,     0x3fd4a253d11b82f3,     //   -0.9466,   0.32241
    0xbfee4e98d836c0af,     0x3fd48a87d0b07fd7,     //  -0.94709,   0.32096
    0xbfee529f04729ffc,     0x3fd472b8a5571054,     //  -0.94759,    0.3195
    0xbfee56a083968eb1,     0x3fd45ae652bb2800,     //  -0.94807,   0.31805
    0xbfee5a9d550467d3,     0x3fd44310dc8936f0,     //  -0.94856,   0.31659
    0xbfee5e95781ebf1c,     0x3fd42b38466e2928,     //  -0.94905,   0.31514
    0xbfee6288ec48e112,     0x3fd4135c94176602,     //  -0.94953,   0.31368
    0xbfee6677b0e6d31e,     0x3fd3fb7dc932cfa4,     //  -0.95001,   0.31222
    0xbfee6a61c55d53a7,     0x3fd3e39be96ec271,     //  -0.95049,   0.31077
    0xbfee6e472911da27,     0x3fd3cbb6f87a146e,     //  -0.95096,   0.30931
    0xbfee7227db6a9744,     0x3fd3b3cefa0414b7,     //  -0.95144,   0.30785
    0xbfee7603dbce74e9,     0x3fd39be3f1bc8aef,     //  -0.95191,   0.30639
    0xbfee79db29a5165a,     0x3fd383f5e353b6aa,     //  -0.95238,   0.30493
    0xbfee7dadc456d850,     0x3fd36c04d27a4edf,     //  -0.95284,   0.30347
    0xbfee817bab4cd10d,     0x3fd35410c2e18152,     //  -0.95331,   0.30201
    0xbfee8544ddf0d075,     0x3fd33c19b83af207,     //  -0.95377,   0.30054
    0xbfee89095bad6025,     0x3fd3241fb638baaf,     //  -0.95423,   0.29908
    0xbfee8cc923edc388,     0x3fd30c22c08d6a13,     //  -0.95469,   0.29762
    0xbfee9084361df7f3,     0x3fd2f422daec0386,     //  -0.95514,   0.29615
    0xbfee943a91aab4b4,     0x3fd2dc200907fe51,     //  -0.95559,   0.29469
    0xbfee97ec36016b30,     0x3fd2c41a4e954520,     //  -0.95605,   0.29322
    0xbfee9b99229046f8,     0x3fd2ac11af483572,     //  -0.95649,   0.29175
    0xbfee9f4156c62dda,     0x3fd294062ed59f05,     //  -0.95694,   0.29028
    0xbfeea2e4d212c000,     0x3fd27bf7d0f2c346,     //  -0.95738,   0.28882
    0xbfeea68393e65800,     0x3fd263e6995554ba,     //  -0.95783,   0.28735
    0xbfeeaa1d9bb20af3,     0x3fd24bd28bb37672,     //  -0.95827,   0.28588
    0xbfeeadb2e8e7a88e,     0x3fd233bbabc3bb72,     //   -0.9587,   0.28441
    0xbfeeb1437af9bb34,     0x3fd21ba1fd3d2623,     //  -0.95914,   0.28294
    0xbfeeb4cf515b8811,     0x3fd2038583d727bd,     //  -0.95957,   0.28146
    0xbfeeb8566b810f2a,     0x3fd1eb6643499fbb,     //     -0.96,   0.27999
    0xbfeebbd8c8df0b74,     0x3fd1d3443f4cdb3d,     //  -0.96043,   0.27852
    0xbfeebf5668eaf2ef,     0x3fd1bb1f7b999480,     //  -0.96086,   0.27705
    0xbfeec2cf4b1af6b2,     0x3fd1a2f7fbe8f243,     //  -0.96128,   0.27557
    0xbfeec6436ee60309,     0x3fd18acdc3f4873a,     //   -0.9617,    0.2741
    0xbfeec9b2d3c3bf84,     0x3fd172a0d7765177,     //  -0.96212,   0.27262
    0xbfeecd1d792c8f10,     0x3fd15a713a28b9d9,     //  -0.96254,   0.27115
    0xbfeed0835e999009,     0x3fd1423eefc69378,     //  -0.96295,   0.26967
    0xbfeed3e483849c51,     0x3fd12a09fc0b1b12,     //  -0.96337,   0.26819
    0xbfeed740e7684963,     0x3fd111d262b1f677,     //  -0.96378,   0.26671
    0xbfeeda9889bfe86a,     0x3fd0f998277733f7,     //  -0.96418,   0.26523
    0xbfeeddeb6a078651,     0x3fd0e15b4e1749cd,     //  -0.96459,   0.26375
    0xbfeee13987bbebdc,     0x3fd0c91bda4f158d,     //  -0.96499,   0.26227
    0xbfeee482e25a9dbc,     0x3fd0b0d9cfdbdb90,     //  -0.96539,   0.26079
    0xbfeee7c77961dc9e,     0x3fd09895327b465e,     //  -0.96579,   0.25931
    0xbfeeeb074c50a544,     0x3fd0804e05eb661e,     //  -0.96619,   0.25783
    0xbfeeee425aa6b09a,     0x3fd068044deab002,     //  -0.96658,   0.25635
    0xbfeef178a3e473c2,     0x3fd04fb80e37fdae,     //  -0.96698,   0.25487
    0xbfeef4aa278b2032,     0x3fd037694a928cac,     //  -0.96737,   0.25338
    0xbfeef7d6e51ca3c0,     0x3fd01f1806b9fdd2,     //  -0.96775,    0.2519
    0xbfeefafedc1ba8b7,     0x3fd006c4466e54af,     //  -0.96814,   0.25041
    0xbfeefe220c0b95ec,     0x3fcfdcdc1adfedf8,     //  -0.96852,   0.24893
    0xbfef014074708ed3,     0x3fcfac2abeff57ff,     //   -0.9689,   0.24744
    0xbfef045a14cf738c,     0x3fcf7b7480bd3801,     //  -0.96928,   0.24596
    0xbfef076eecade0fa,     0x3fcf4ab9679c9f5c,     //  -0.96966,   0.24447
    0xbfef0a7efb9230d7,     0x3fcf19f97b215f1a,     //  -0.97003,   0.24298
    0xbfef0d8a410379c5,     0x3fcee934c2d006c7,     //   -0.9704,   0.24149
    0xbfef1090bc898f5f,     0x3fceb86b462de348,     //  -0.97077,      0.24
    0xbfef13926dad024e,     0x3fce879d0cc0fdaf,     //  -0.97114,   0.23851
    0xbfef168f53f7205d,     0x3fce56ca1e101a1b,     //   -0.9715,   0.23702
    0xbfef19876ef1f486,     0x3fce25f281a2b684,     //  -0.97187,   0.23553
    0xbfef1c7abe284708,     0x3fcdf5163f01099a,     //  -0.97223,   0.23404
    0xbfef1f6941259d7a,     0x3fcdc4355db40195,     //  -0.97258,   0.23255
    0xbfef2252f7763ada,     0x3fcd934fe5454311,     //  -0.97294,   0.23106
    0xbfef2537e0a71f9f,     0x3fcd6265dd3f27e3,     //  -0.97329,   0.22957
    0xbfef2817fc4609ce,     0x3fcd31774d2cbdee,     //  -0.97364,   0.22807
    0xbfef2af349e17507,     0x3fcd00843c99c5f9,     //  -0.97399,   0.22658
    0xbfef2dc9c9089a9d,     0x3fcccf8cb312b286,     //  -0.97434,   0.22508
    0xbfef309b794b719f,     0x3fcc9e90b824a6a9,     //  -0.97468,   0.22359
    0xbfef33685a3aaef0,     0x3fcc6d90535d74dc,     //  -0.97503,   0.22209
    0xbfef36306b67c556,     0x3fcc3c8b8c4b9dd7,     //  -0.97536,    0.2206
    0xbfef38f3ac64e589,     0x3fcc0b826a7e4f63,     //   -0.9757,    0.2191
    0xbfef3bb21cc4fe47,     0x3fcbda74f5856330,     //  -0.97604,    0.2176
    0xbfef3e6bbc1bbc65,     0x3fcba96334f15dad,     //  -0.97637,   0.21611
    0xbfef412089fd8adc,     0x3fcb784d30536cda,     //   -0.9767,   0.21461
    0xbfef43d085ff92dd,     0x3fcb4732ef3d6722,     //  -0.97703,   0.21311
    0xbfef467bafb7bbe0,     0x3fcb16147941ca2a,     //  -0.97735,   0.21161
    0xbfef492206bcabb4,     0x3fcae4f1d5f3b9ab,     //  -0.97768,   0.21011
    0xbfef4bc38aa5c694,     0x3fcab3cb0ce6fe44,     //    -0.978,   0.20861
    0xbfef4e603b0b2f2d,     0x3fca82a025b00451,     //  -0.97832,   0.20711
    0xbfef50f81785c6b9,     0x3fca517127e3dabc,     //  -0.97863,   0.20561
    0xbfef538b1faf2d07,     0x3fca203e1b1831da,     //  -0.97895,   0.20411
    0xbfef56195321c090,     0x3fc9ef0706e35a35,     //  -0.97926,   0.20261
    0xbfef58a2b1789e84,     0x3fc9bdcbf2dc4366,     //  -0.97957,    0.2011
    0xbfef5b273a4fa2d9,     0x3fc98c8ce69a7aec,     //  -0.97988,    0.1996
    0xbfef5da6ed43685d,     0x3fc95b49e9b62af9,     //  -0.98018,    0.1981
    0xbfef6021c9f148c2,     0x3fc92a0303c8194f,     //  -0.98048,   0.19659
    0xbfef6297cff75cb0,     0x3fc8f8b83c69a60a,     //  -0.98079,   0.19509
    0xbfef6508fef47bd5,     0x3fc8c7699b34ca7e,     //  -0.98108,   0.19359
    0xbfef677556883cee,     0x3fc8961727c41804,     //  -0.98138,   0.19208
    0xbfef69dcd652f5de,     0x3fc864c0e9b2b6cf,     //  -0.98167,   0.19057
    0xbfef6c3f7df5bbb7,     0x3fc83366e89c64c5,     //  -0.98196,   0.18907
    0xbfef6e9d4d1262ca,     0x3fc802092c1d744b,     //  -0.98225,   0.18756
    0xbfef70f6434b7eb7,     0x3fc7d0a7bbd2cb1b,     //  -0.98254,   0.18606
    0xbfef734a60446279,     0x3fc79f429f59e11d,     //  -0.98282,   0.18455
    0xbfef7599a3a12077,     0x3fc76dd9de50bf31,     //  -0.98311,   0.18304
    0xbfef77e40d068a90,     0x3fc73c6d8055fe0a,     //  -0.98339,   0.18153
    0xbfef7a299c1a322a,     0x3fc70afd8d08c4ff,     //  -0.98366,   0.18002
    0xbfef7c6a50826840,     0x3fc6d98a0c08c8da,     //  -0.98394,   0.17851
    0xbfef7ea629e63d6e,     0x3fc6a81304f64ab2,     //  -0.98421,     0.177
    0xbfef80dd27ed8204,     0x3fc676987f7216b8,     //  -0.98448,   0.17549
    0xbfef830f4a40c60c,     0x3fc6451a831d830d,     //  -0.98475,   0.17398
    0xbfef853c9089595e,     0x3fc61399179a6e94,     //  -0.98501,   0.17247
    0xbfef8764fa714ba9,     0x3fc5e214448b3fc6,     //  -0.98528,   0.17096
    0xbfef898887a36c84,     0x3fc5b08c1192e381,     //  -0.98554,   0.16945
    0xbfef8ba737cb4b78,     0x3fc57f008654cbde,     //   -0.9858,   0.16794
    0xbfef8dc10a95380d,     0x3fc54d71aa74ef02,     //  -0.98605,   0.16643
    0xbfef8fd5ffae41db,     0x3fc51bdf8597c5f2,     //  -0.98631,   0.16491
    0xbfef91e616c43891,     0x3fc4ea4a1f624b61,     //  -0.98656,    0.1634
    0xbfef93f14f85ac08,     0x3fc4b8b17f79fa88,     //  -0.98681,   0.16189
    0xbfef95f7a9a1ec47,     0x3fc48715ad84cdf5,     //  -0.98706,   0.16037
    0xbfef97f924c9099b,     0x3fc45576b1293e5a,     //   -0.9873,   0.15886
    0xbfef99f5c0abd496,     0x3fc423d4920e4166,     //  -0.98754,   0.15734
    0xbfef9bed7cfbde29,     0x3fc3f22f57db4893,     //  -0.98778,   0.15583
    0xbfef9de0596b77a3,     0x3fc3c0870a383ff6,     //  -0.98802,   0.15431
    0xbfef9fce55adb2c8,     0x3fc38edbb0cd8d14,     //  -0.98826,    0.1528
    0xbfefa1b7717661d5,     0x3fc35d2d53440db2,     //  -0.98849,   0.15128
    0xbfefa39bac7a1791,     0x3fc32b7bf94516a7,     //  -0.98872,   0.14976
    0xbfefa57b066e2754,     0x3fc2f9c7aa7a72af,     //  -0.98895,   0.14825
    0xbfefa7557f08a517,     0x3fc2c8106e8e613a,     //  -0.98918,   0.14673
    0xbfefa92b1600657c,     0x3fc296564d2b953e,     //   -0.9894,   0.14521
    0xbfefaafbcb0cfddc,     0x3fc264994dfd340a,     //  -0.98962,    0.1437
    0xbfefacc79de6c44f,     0x3fc232d978aed413,     //  -0.98984,   0.14218
    0xbfefae8e8e46cfbb,     0x3fc20116d4ec7bce,     //  -0.99006,   0.14066
    0xbfefb0509be6f7db,     0x3fc1cf516a62a077,     //  -0.99027,   0.13914
    0xbfefb20dc681d54d,     0x3fc19d8940be24e7,     //  -0.99049,   0.13762
    0xbfefb3c60dd2c199,     0x3fc16bbe5fac5865,     //   -0.9907,    0.1361
    0xbfefb5797195d741,     0x3fc139f0cedaf576,     //   -0.9909,   0.13458
    0xbfefb727f187f1c7,     0x3fc1082095f820b0,     //  -0.99111,   0.13306
    0xbfefb8d18d66adb7,     0x3fc0d64dbcb26786,     //  -0.99131,   0.13154
    0xbfefba7644f068b5,     0x3fc0a4784ab8bf1d,     //  -0.99151,   0.13002
    0xbfefbc1617e44186,     0x3fc072a047ba831d,     //  -0.99171,    0.1285
    0xbfefbdb106021816,     0x3fc040c5bb67747e,     //  -0.99191,   0.12698
    0xbfefbf470f0a8d88,     0x3fc00ee8ad6fb85b,     //   -0.9921,   0.12545
    0xbfefc0d832bf043a,     0x3fbfba124b07ad85,     //  -0.99229,   0.12393
    0xbfefc26470e19fd3,     0x3fbf564e56a9730e,     //  -0.99248,   0.12241
    0xbfefc3ebc935454c,     0x3fbef2858d27561b,     //  -0.99267,   0.12089
    0xbfefc56e3b7d9af6,     0x3fbe8eb7fde4aa3e,     //  -0.99285,   0.11937
    0xbfefc6ebc77f0887,     0x3fbe2ae5b8457f77,     //  -0.99303,   0.11784
    0xbfefc8646cfeb721,     0x3fbdc70ecbae9fc8,     //  -0.99321,   0.11632
    0xbfefc9d82bc2915e,     0x3fbd633347858ce4,     //  -0.99339,   0.11479
    0xbfefcb4703914354,     0x3fbcff533b307dc1,     //  -0.99356,   0.11327
    0xbfefccb0f4323aa3,     0x3fbc9b6eb6165c42,     //  -0.99374,   0.11175
    0xbfefce15fd6da67b,     0x3fbc3785c79ec2d5,     //  -0.99391,   0.11022
    0xbfefcf761f0c77a3,     0x3fbbd3987f31fa0e,     //  -0.99407,    0.1087
    0xbfefd0d158d86087,     0x3fbb6fa6ec38f64c,     //  -0.99424,   0.10717
    0xbfefd227aa9bd53b,     0x3fbb0bb11e1d5559,     //   -0.9944,   0.10565
    0xbfefd37914220b84,     0x3fbaa7b724495c04,     //  -0.99456,   0.10412
    0xbfefd4c59536fae4,     0x3fba43b90e27f3c4,     //  -0.99472,    0.1026
    0xbfefd60d2da75c9e,     0x3fb9dfb6eb24a85c,     //  -0.99488,   0.10107
    0xbfefd74fdd40abbf,     0x3fb97bb0caaba56f,     //  -0.99503,  0.099544
    0xbfefd88da3d12526,     0x3fb917a6bc29b42c,     //  -0.99518,  0.098017
    0xbfefd9c68127c78c,     0x3fb8b398cf0c38e0,     //  -0.99533,   0.09649
    0xbfefdafa7514538c,     0x3fb84f8712c130a0,     //  -0.99548,  0.094963
    0xbfefdc297f674ba9,     0x3fb7eb7196b72ee4,     //  -0.99563,  0.093436
    0xbfefdd539ff1f456,     0x3fb787586a5d5b21,     //  -0.99577,  0.091909
    0xbfefde78d68653fd,     0x3fb7233b9d236e71,     //  -0.99591,  0.090381
    0xbfefdf9922f73307,     0x3fb6bf1b3e79b129,     //  -0.99604,  0.088854
    0xbfefe0b485181be3,     0x3fb65af75dd0f87b,     //  -0.99618,  0.087326
    0xbfefe1cafcbd5b09,     0x3fb5f6d00a9aa419,     //  -0.99631,  0.085797
    0xbfefe2dc89bbff08,     0x3fb592a554489bc8,     //  -0.99644,  0.084269
    0xbfefe3e92be9d886,     0x3fb52e774a4d4d0a,     //  -0.99657,   0.08274
    0xbfefe4f0e31d7a4a,     0x3fb4ca45fc1ba8b6,     //   -0.9967,  0.081211
    0xbfefe5f3af2e3940,     0x3fb4661179272096,     //  -0.99682,  0.079682
    0xbfefe6f18ff42c84,     0x3fb401d9d0e3a507,     //  -0.99694,  0.078153
    0xbfefe7ea85482d60,     0x3fb39d9f12c5a299,     //  -0.99706,  0.076624
    0xbfefe8de8f03d75c,     0x3fb339614e41ffa5,     //  -0.99718,  0.075094
    0xbfefe9cdad01883a,     0x3fb2d52092ce19f6,     //  -0.99729,  0.073565
    0xbfefeab7df1c6005,     0x3fb270dcefdfc45b,     //   -0.9974,  0.072035
    0xbfefeb9d2530410f,     0x3fb20c9674ed444c,     //  -0.99751,  0.070505
    0xbfefec7d7f19cffc,     0x3fb1a84d316d4f8a,     //  -0.99762,  0.068974
    0xbfefed58ecb673c4,     0x3fb1440134d709b2,     //  -0.99772,  0.067444
    0xbfefee2f6de455ba,     0x3fb0dfb28ea201e6,     //  -0.99783,  0.065913
    0xbfefef0102826191,     0x3fb07b614e463064,     //  -0.99793,  0.064383
    0xbfefefcdaa704562,     0x3fb0170d833bf421,     //  -0.99802,  0.062852
    0xbfeff095658e71ad,     0x3faf656e79f820e0,     //  -0.99812,  0.061321
    0xbfeff15833be1965,     0x3fae9cbd15ff5527,     //  -0.99821,   0.05979
    0xbfeff21614e131ed,     0x3fadd406f9808ec8,     //   -0.9983,  0.058258
    0xbfeff2cf08da7321,     0x3fad0b4c436f91d0,     //  -0.99839,  0.056727
    0xbfeff3830f8d575c,     0x3fac428d12c0d7e3,     //  -0.99848,  0.055195
    0xbfeff43228de1b77,     0x3fab79c986698b78,     //  -0.99856,  0.053664
    0xbfeff4dc54b1bed3,     0x3faab101bd5f8317,     //  -0.99864,  0.052132
    0xbfeff58192ee0358,     0x3fa9e835d6993c87,     //  -0.99872,    0.0506
    0xbfeff621e3796d7e,     0x3fa91f65f10dd814,     //   -0.9988,  0.049068
    0xbfeff6bd463b444d,     0x3fa856922bb513c1,     //  -0.99887,  0.047535
    0xbfeff753bb1b9164,     0x3fa78dbaa5874685,     //  -0.99894,  0.046003
    0xbfeff7e5420320f9,     0x3fa6c4df7d7d5b84,     //  -0.99901,  0.044471
    0xbfeff871dadb81df,     0x3fa5fc00d290cd43,     //  -0.99908,  0.042938
    0xbfeff8f9858f058b,     0x3fa5331ec3bba0eb,     //  -0.99914,  0.041406
    0xbfeff97c4208c014,     0x3fa46a396ff86179,     //   -0.9992,  0.039873
    0xbfeff9fa10348837,     0x3fa3a150f6421afc,     //  -0.99926,   0.03834
    0xbfeffa72effef75d,     0x3fa2d865759455cd,     //  -0.99932,  0.036807
    0xbfeffae6e1556998,     0x3fa20f770ceb11c6,     //  -0.99938,  0.035274
    0xbfeffb55e425fdae,     0x3fa14685db42c17e,     //  -0.99943,  0.033741
    0xbfeffbbff85f9515,     0x3fa07d91ff984580,     //  -0.99948,  0.032208
    0xbfeffc251df1d3f8,     0x3f9f693731d1cf01,     //  -0.99953,  0.030675
    0xbfeffc8554cd213a,     0x3f9dd7458c64ab39,     //  -0.99958,  0.029142
    0xbfeffce09ce2a679,     0x3f9c454f4ce53b1c,     //  -0.99962,  0.027608
    0xbfeffd36f624500c,     0x3f9ab354b1504fca,     //  -0.99966,  0.026075
    0xbfeffd886084cd0d,     0x3f992155f7a3667e,     //   -0.9997,  0.024541
    0xbfeffdd4dbf78f52,     0x3f978f535ddc9f03,     //  -0.99974,  0.023008
    0xbfeffe1c6870cb77,     0x3f95fd4d21fab226,     //  -0.99977,  0.021474
    0xbfeffe5f05e578db,     0x3f946b4381fce81c,     //   -0.9998,   0.01994
    0xbfeffe9cb44b51a1,     0x3f92d936bbe30efd,     //  -0.99983,  0.018407
    0xbfeffed57398d2b7,     0x3f9147270dad7132,     //  -0.99986,  0.016873
    0xbfefff0943c53bd1,     0x3f8f6a296ab997ca,     //  -0.99988,  0.015339
    0xbfefff3824c88f6f,     0x3f8c45ffe1e48ad9,     //   -0.9999,  0.013805
    0xbfefff62169b92db,     0x3f8921d1fcdec784,     //  -0.99992,  0.012272
    0xbfefff871937ce2f,     0x3f85fda037ac05e0,     //  -0.99994,  0.010738
    0xbfefffa72c978c4f,     0x3f82d96b0e509703,     //  -0.99996, 0.0092038
    0xbfefffc250b5daef,     0x3f7f6a65f9a2a3c5,     //  -0.99997, 0.0076698
    0xbfefffd8858e8a92,     0x3f7921f0fe670071,     //  -0.99998, 0.0061359
    0xbfefffe9cb1e2e8d,     0x3f72d97822f996bc,     //  -0.99999, 0.0046019
    0xbfeffff621621d02,     0x3f6921f8becca4ba,     //        -1,  0.003068
    0xbfeffffd88586ee6,     0x3f5921faaee6472d,     //        -1,  0.001534
    0xbff0000000000000,     0x0000000000000000,     //        -1,         0
    0xbfeffffd88586ee6,     0xbf5921faaee6472d,     //        -1, -0.001534
    0xbfeffff621621d02,     0xbf6921f8becca4ba,     //        -1, -0.003068
    0xbfefffe9cb1e2e8d,     0xbf72d97822f996bc,     //  -0.99999,-0.0046019
    0xbfefffd8858e8a92,     0xbf7921f0fe670071,     //  -0.99998,-0.0061359
    0xbfefffc250b5daef,     0xbf7f6a65f9a2a3c5,     //  -0.99997,-0.0076698
    0xbfefffa72c978c4f,     0xbf82d96b0e509703,     //  -0.99996,-0.0092038
    0xbfefff871937ce2f,     0xbf85fda037ac05e0,     //  -0.99994, -0.010738
    0xbfefff62169b92db,     0xbf8921d1fcdec784,     //  -0.99992, -0.012272
    0xbfefff3824c88f6f,     0xbf8c45ffe1e48ad9,     //   -0.9999, -0.013805
    0xbfefff0943c53bd1,     0xbf8f6a296ab997ca,     //  -0.99988, -0.015339
    0xbfeffed57398d2b7,     0xbf9147270dad7132,     //  -0.99986, -0.016873
    0xbfeffe9cb44b51a1,     0xbf92d936bbe30efd,     //  -0.99983, -0.018407
    0xbfeffe5f05e578db,     0xbf946b4381fce81c,     //   -0.9998,  -0.01994
    0xbfeffe1c6870cb77,     0xbf95fd4d21fab226,     //  -0.99977, -0.021474
    0xbfeffdd4dbf78f52,     0xbf978f535ddc9f03,     //  -0.99974, -0.023008
    0xbfeffd886084cd0d,     0xbf992155f7a3667e,     //   -0.9997, -0.024541
    0xbfeffd36f624500c,     0xbf9ab354b1504fca,     //  -0.99966, -0.026075
    0xbfeffce09ce2a679,     0xbf9c454f4ce53b1c,     //  -0.99962, -0.027608
    0xbfeffc8554cd213a,     0xbf9dd7458c64ab39,     //  -0.99958, -0.029142
    0xbfeffc251df1d3f8,     0xbf9f693731d1cf01,     //  -0.99953, -0.030675
    0xbfeffbbff85f9515,     0xbfa07d91ff984580,     //  -0.99948, -0.032208
    0xbfeffb55e425fdae,     0xbfa14685db42c17e,     //  -0.99943, -0.033741
    0xbfeffae6e1556998,     0xbfa20f770ceb11c6,     //  -0.99938, -0.035274
    0xbfeffa72effef75d,     0xbfa2d865759455cd,     //  -0.99932, -0.036807
    0xbfeff9fa10348837,     0xbfa3a150f6421afc,     //  -0.99926,  -0.03834
    0xbfeff97c4208c014,     0xbfa46a396ff86179,     //   -0.9992, -0.039873
    0xbfeff8f9858f058b,     0xbfa5331ec3bba0eb,     //  -0.99914, -0.041406
    0xbfeff871dadb81df,     0xbfa5fc00d290cd43,     //  -0.99908, -0.042938
    0xbfeff7e5420320f9,     0xbfa6c4df7d7d5b84,     //  -0.99901, -0.044471
    0xbfeff753bb1b9164,     0xbfa78dbaa5874685,     //  -0.99894, -0.046003
    0xbfeff6bd463b444d,     0xbfa856922bb513c1,     //  -0.99887, -0.047535
    0xbfeff621e3796d7e,     0xbfa91f65f10dd814,     //   -0.9988, -0.049068
    0xbfeff58192ee0358,     0xbfa9e835d6993c87,     //  -0.99872,   -0.0506
    0xbfeff4dc54b1bed3,     0xbfaab101bd5f8317,     //  -0.99864, -0.052132
    0xbfeff43228de1b77,     0xbfab79c986698b78,     //  -0.99856, -0.053664
    0xbfeff3830f8d575c,     0xbfac428d12c0d7e3,     //  -0.99848, -0.055195
    0xbfeff2cf08da7321,     0xbfad0b4c436f91d0,     //  -0.99839, -0.056727
    0xbfeff21614e131ed,     0xbfadd406f9808ec8,     //   -0.9983, -0.058258
    0xbfeff15833be1965,     0xbfae9cbd15ff5527,     //  -0.99821,  -0.05979
    0xbfeff095658e71ad,     0xbfaf656e79f820e0,     //  -0.99812, -0.061321
    0xbfefefcdaa704562,     0xbfb0170d833bf421,     //  -0.99802, -0.062852
    0xbfefef0102826191,     0xbfb07b614e463064,     //  -0.99793, -0.064383
    0xbfefee2f6de455ba,     0xbfb0dfb28ea201e6,     //  -0.99783, -0.065913
    0xbfefed58ecb673c4,     0xbfb1440134d709b2,     //  -0.99772, -0.067444
    0xbfefec7d7f19cffc,     0xbfb1a84d316d4f8a,     //  -0.99762, -0.068974
    0xbfefeb9d2530410f,     0xbfb20c9674ed444c,     //  -0.99751, -0.070505
    0xbfefeab7df1c6005,     0xbfb270dcefdfc45b,     //   -0.9974, -0.072035
    0xbfefe9cdad01883a,     0xbfb2d52092ce19f6,     //  -0.99729, -0.073565
    0xbfefe8de8f03d75c,     0xbfb339614e41ffa5,     //  -0.99718, -0.075094
    0xbfefe7ea85482d60,     0xbfb39d9f12c5a299,     //  -0.99706, -0.076624
    0xbfefe6f18ff42c84,     0xbfb401d9d0e3a507,     //  -0.99694, -0.078153
    0xbfefe5f3af2e3940,     0xbfb4661179272096,     //  -0.99682, -0.079682
    0xbfefe4f0e31d7a4a,     0xbfb4ca45fc1ba8b6,     //   -0.9967, -0.081211
    0xbfefe3e92be9d886,     0xbfb52e774a4d4d0a,     //  -0.99657,  -0.08274
    0xbfefe2dc89bbff08,     0xbfb592a554489bc8,     //  -0.99644, -0.084269
    0xbfefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     //  -0.99631, -0.085797
    0xbfefe0b485181be3,     0xbfb65af75dd0f87b,     //  -0.99618, -0.087326
    0xbfefdf9922f73307,     0xbfb6bf1b3e79b129,     //  -0.99604, -0.088854
    0xbfefde78d68653fd,     0xbfb7233b9d236e71,     //  -0.99591, -0.090381
    0xbfefdd539ff1f456,     0xbfb787586a5d5b21,     //  -0.99577, -0.091909
    0xbfefdc297f674ba9,     0xbfb7eb7196b72ee4,     //  -0.99563, -0.093436
    0xbfefdafa7514538c,     0xbfb84f8712c130a0,     //  -0.99548, -0.094963
    0xbfefd9c68127c78c,     0xbfb8b398cf0c38e0,     //  -0.99533,  -0.09649
    0xbfefd88da3d12526,     0xbfb917a6bc29b42c,     //  -0.99518, -0.098017
    0xbfefd74fdd40abbf,     0xbfb97bb0caaba56f,     //  -0.99503, -0.099544
    0xbfefd60d2da75c9e,     0xbfb9dfb6eb24a85c,     //  -0.99488,  -0.10107
    0xbfefd4c59536fae4,     0xbfba43b90e27f3c4,     //  -0.99472,   -0.1026
    0xbfefd37914220b84,     0xbfbaa7b724495c04,     //  -0.99456,  -0.10412
    0xbfefd227aa9bd53b,     0xbfbb0bb11e1d5559,     //   -0.9944,  -0.10565
    0xbfefd0d158d86087,     0xbfbb6fa6ec38f64c,     //  -0.99424,  -0.10717
    0xbfefcf761f0c77a3,     0xbfbbd3987f31fa0e,     //  -0.99407,   -0.1087
    0xbfefce15fd6da67b,     0xbfbc3785c79ec2d5,     //  -0.99391,  -0.11022
    0xbfefccb0f4323aa3,     0xbfbc9b6eb6165c42,     //  -0.99374,  -0.11175
    0xbfefcb4703914354,     0xbfbcff533b307dc1,     //  -0.99356,  -0.11327
    0xbfefc9d82bc2915e,     0xbfbd633347858ce4,     //  -0.99339,  -0.11479
    0xbfefc8646cfeb721,     0xbfbdc70ecbae9fc8,     //  -0.99321,  -0.11632
    0xbfefc6ebc77f0887,     0xbfbe2ae5b8457f77,     //  -0.99303,  -0.11784
    0xbfefc56e3b7d9af6,     0xbfbe8eb7fde4aa3e,     //  -0.99285,  -0.11937
    0xbfefc3ebc935454c,     0xbfbef2858d27561b,     //  -0.99267,  -0.12089
    0xbfefc26470e19fd3,     0xbfbf564e56a9730e,     //  -0.99248,  -0.12241
    0xbfefc0d832bf043a,     0xbfbfba124b07ad85,     //  -0.99229,  -0.12393
    0xbfefbf470f0a8d88,     0xbfc00ee8ad6fb85b,     //   -0.9921,  -0.12545
    0xbfefbdb106021816,     0xbfc040c5bb67747e,     //  -0.99191,  -0.12698
    0xbfefbc1617e44186,     0xbfc072a047ba831d,     //  -0.99171,   -0.1285
    0xbfefba7644f068b5,     0xbfc0a4784ab8bf1d,     //  -0.99151,  -0.13002
    0xbfefb8d18d66adb7,     0xbfc0d64dbcb26786,     //  -0.99131,  -0.13154
    0xbfefb727f187f1c7,     0xbfc1082095f820b0,     //  -0.99111,  -0.13306
    0xbfefb5797195d741,     0xbfc139f0cedaf576,     //   -0.9909,  -0.13458
    0xbfefb3c60dd2c199,     0xbfc16bbe5fac5865,     //   -0.9907,   -0.1361
    0xbfefb20dc681d54d,     0xbfc19d8940be24e7,     //  -0.99049,  -0.13762
    0xbfefb0509be6f7db,     0xbfc1cf516a62a077,     //  -0.99027,  -0.13914
    0xbfefae8e8e46cfbb,     0xbfc20116d4ec7bce,     //  -0.99006,  -0.14066
    0xbfefacc79de6c44f,     0xbfc232d978aed413,     //  -0.98984,  -0.14218
    0xbfefaafbcb0cfddc,     0xbfc264994dfd340a,     //  -0.98962,   -0.1437
    0xbfefa92b1600657c,     0xbfc296564d2b953e,     //   -0.9894,  -0.14521
    0xbfefa7557f08a517,     0xbfc2c8106e8e613a,     //  -0.98918,  -0.14673
    0xbfefa57b066e2754,     0xbfc2f9c7aa7a72af,     //  -0.98895,  -0.14825
    0xbfefa39bac7a1791,     0xbfc32b7bf94516a7,     //  -0.98872,  -0.14976
    0xbfefa1b7717661d5,     0xbfc35d2d53440db2,     //  -0.98849,  -0.15128
    0xbfef9fce55adb2c8,     0xbfc38edbb0cd8d14,     //  -0.98826,   -0.1528
    0xbfef9de0596b77a3,     0xbfc3c0870a383ff6,     //  -0.98802,  -0.15431
    0xbfef9bed7cfbde29,     0xbfc3f22f57db4893,     //  -0.98778,  -0.15583
    0xbfef99f5c0abd496,     0xbfc423d4920e4166,     //  -0.98754,  -0.15734
    0xbfef97f924c9099b,     0xbfc45576b1293e5a,     //   -0.9873,  -0.15886
    0xbfef95f7a9a1ec47,     0xbfc48715ad84cdf5,     //  -0.98706,  -0.16037
    0xbfef93f14f85ac08,     0xbfc4b8b17f79fa88,     //  -0.98681,  -0.16189
    0xbfef91e616c43891,     0xbfc4ea4a1f624b61,     //  -0.98656,   -0.1634
    0xbfef8fd5ffae41db,     0xbfc51bdf8597c5f2,     //  -0.98631,  -0.16491
    0xbfef8dc10a95380d,     0xbfc54d71aa74ef02,     //  -0.98605,  -0.16643
    0xbfef8ba737cb4b78,     0xbfc57f008654cbde,     //   -0.9858,  -0.16794
    0xbfef898887a36c84,     0xbfc5b08c1192e381,     //  -0.98554,  -0.16945
    0xbfef8764fa714ba9,     0xbfc5e214448b3fc6,     //  -0.98528,  -0.17096
    0xbfef853c9089595e,     0xbfc61399179a6e94,     //  -0.98501,  -0.17247
    0xbfef830f4a40c60c,     0xbfc6451a831d830d,     //  -0.98475,  -0.17398
    0xbfef80dd27ed8204,     0xbfc676987f7216b8,     //  -0.98448,  -0.17549
    0xbfef7ea629e63d6e,     0xbfc6a81304f64ab2,     //  -0.98421,    -0.177
    0xbfef7c6a50826840,     0xbfc6d98a0c08c8da,     //  -0.98394,  -0.17851
    0xbfef7a299c1a322a,     0xbfc70afd8d08c4ff,     //  -0.98366,  -0.18002
    0xbfef77e40d068a90,     0xbfc73c6d8055fe0a,     //  -0.98339,  -0.18153
    0xbfef7599a3a12077,     0xbfc76dd9de50bf31,     //  -0.98311,  -0.18304
    0xbfef734a60446279,     0xbfc79f429f59e11d,     //  -0.98282,  -0.18455
    0xbfef70f6434b7eb7,     0xbfc7d0a7bbd2cb1b,     //  -0.98254,  -0.18606
    0xbfef6e9d4d1262ca,     0xbfc802092c1d744b,     //  -0.98225,  -0.18756
    0xbfef6c3f7df5bbb7,     0xbfc83366e89c64c5,     //  -0.98196,  -0.18907
    0xbfef69dcd652f5de,     0xbfc864c0e9b2b6cf,     //  -0.98167,  -0.19057
    0xbfef677556883cee,     0xbfc8961727c41804,     //  -0.98138,  -0.19208
    0xbfef6508fef47bd5,     0xbfc8c7699b34ca7e,     //  -0.98108,  -0.19359
    0xbfef6297cff75cb0,     0xbfc8f8b83c69a60a,     //  -0.98079,  -0.19509
    0xbfef6021c9f148c2,     0xbfc92a0303c8194f,     //  -0.98048,  -0.19659
    0xbfef5da6ed43685d,     0xbfc95b49e9b62af9,     //  -0.98018,   -0.1981
    0xbfef5b273a4fa2d9,     0xbfc98c8ce69a7aec,     //  -0.97988,   -0.1996
    0xbfef58a2b1789e84,     0xbfc9bdcbf2dc4366,     //  -0.97957,   -0.2011
    0xbfef56195321c090,     0xbfc9ef0706e35a35,     //  -0.97926,  -0.20261
    0xbfef538b1faf2d07,     0xbfca203e1b1831da,     //  -0.97895,  -0.20411
    0xbfef50f81785c6b9,     0xbfca517127e3dabc,     //  -0.97863,  -0.20561
    0xbfef4e603b0b2f2d,     0xbfca82a025b00451,     //  -0.97832,  -0.20711
    0xbfef4bc38aa5c694,     0xbfcab3cb0ce6fe44,     //    -0.978,  -0.20861
    0xbfef492206bcabb4,     0xbfcae4f1d5f3b9ab,     //  -0.97768,  -0.21011
    0xbfef467bafb7bbe0,     0xbfcb16147941ca2a,     //  -0.97735,  -0.21161
    0xbfef43d085ff92dd,     0xbfcb4732ef3d6722,     //  -0.97703,  -0.21311
    0xbfef412089fd8adc,     0xbfcb784d30536cda,     //   -0.9767,  -0.21461
    0xbfef3e6bbc1bbc65,     0xbfcba96334f15dad,     //  -0.97637,  -0.21611
    0xbfef3bb21cc4fe47,     0xbfcbda74f5856330,     //  -0.97604,   -0.2176
    0xbfef38f3ac64e589,     0xbfcc0b826a7e4f63,     //   -0.9757,   -0.2191
    0xbfef36306b67c556,     0xbfcc3c8b8c4b9dd7,     //  -0.97536,   -0.2206
    0xbfef33685a3aaef0,     0xbfcc6d90535d74dc,     //  -0.97503,  -0.22209
    0xbfef309b794b719f,     0xbfcc9e90b824a6a9,     //  -0.97468,  -0.22359
    0xbfef2dc9c9089a9d,     0xbfcccf8cb312b286,     //  -0.97434,  -0.22508
    0xbfef2af349e17507,     0xbfcd00843c99c5f9,     //  -0.97399,  -0.22658
    0xbfef2817fc4609ce,     0xbfcd31774d2cbdee,     //  -0.97364,  -0.22807
    0xbfef2537e0a71f9f,     0xbfcd6265dd3f27e3,     //  -0.97329,  -0.22957
    0xbfef2252f7763ada,     0xbfcd934fe5454311,     //  -0.97294,  -0.23106
    0xbfef1f6941259d7a,     0xbfcdc4355db40195,     //  -0.97258,  -0.23255
    0xbfef1c7abe284708,     0xbfcdf5163f01099a,     //  -0.97223,  -0.23404
    0xbfef19876ef1f486,     0xbfce25f281a2b684,     //  -0.97187,  -0.23553
    0xbfef168f53f7205d,     0xbfce56ca1e101a1b,     //   -0.9715,  -0.23702
    0xbfef13926dad024e,     0xbfce879d0cc0fdaf,     //  -0.97114,  -0.23851
    0xbfef1090bc898f5f,     0xbfceb86b462de348,     //  -0.97077,     -0.24
    0xbfef0d8a410379c5,     0xbfcee934c2d006c7,     //   -0.9704,  -0.24149
    0xbfef0a7efb9230d7,     0xbfcf19f97b215f1a,     //  -0.97003,  -0.24298
    0xbfef076eecade0fa,     0xbfcf4ab9679c9f5c,     //  -0.96966,  -0.24447
    0xbfef045a14cf738c,     0xbfcf7b7480bd3801,     //  -0.96928,  -0.24596
    0xbfef014074708ed3,     0xbfcfac2abeff57ff,     //   -0.9689,  -0.24744
    0xbfeefe220c0b95ec,     0xbfcfdcdc1adfedf8,     //  -0.96852,  -0.24893
    0xbfeefafedc1ba8b7,     0xbfd006c4466e54af,     //  -0.96814,  -0.25041
    0xbfeef7d6e51ca3c0,     0xbfd01f1806b9fdd2,     //  -0.96775,   -0.2519
    0xbfeef4aa278b2032,     0xbfd037694a928cac,     //  -0.96737,  -0.25338
    0xbfeef178a3e473c2,     0xbfd04fb80e37fdae,     //  -0.96698,  -0.25487
    0xbfeeee425aa6b09a,     0xbfd068044deab002,     //  -0.96658,  -0.25635
    0xbfeeeb074c50a544,     0xbfd0804e05eb661e,     //  -0.96619,  -0.25783
    0xbfeee7c77961dc9e,     0xbfd09895327b465e,     //  -0.96579,  -0.25931
    0xbfeee482e25a9dbc,     0xbfd0b0d9cfdbdb90,     //  -0.96539,  -0.26079
    0xbfeee13987bbebdc,     0xbfd0c91bda4f158d,     //  -0.96499,  -0.26227
    0xbfeeddeb6a078651,     0xbfd0e15b4e1749cd,     //  -0.96459,  -0.26375
    0xbfeeda9889bfe86a,     0xbfd0f998277733f7,     //  -0.96418,  -0.26523
    0xbfeed740e7684963,     0xbfd111d262b1f677,     //  -0.96378,  -0.26671
    0xbfeed3e483849c51,     0xbfd12a09fc0b1b12,     //  -0.96337,  -0.26819
    0xbfeed0835e999009,     0xbfd1423eefc69378,     //  -0.96295,  -0.26967
    0xbfeecd1d792c8f10,     0xbfd15a713a28b9d9,     //  -0.96254,  -0.27115
    0xbfeec9b2d3c3bf84,     0xbfd172a0d7765177,     //  -0.96212,  -0.27262
    0xbfeec6436ee60309,     0xbfd18acdc3f4873a,     //   -0.9617,   -0.2741
    0xbfeec2cf4b1af6b2,     0xbfd1a2f7fbe8f243,     //  -0.96128,  -0.27557
    0xbfeebf5668eaf2ef,     0xbfd1bb1f7b999480,     //  -0.96086,  -0.27705
    0xbfeebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     //  -0.96043,  -0.27852
    0xbfeeb8566b810f2a,     0xbfd1eb6643499fbb,     //     -0.96,  -0.27999
    0xbfeeb4cf515b8811,     0xbfd2038583d727bd,     //  -0.95957,  -0.28146
    0xbfeeb1437af9bb34,     0xbfd21ba1fd3d2623,     //  -0.95914,  -0.28294
    0xbfeeadb2e8e7a88e,     0xbfd233bbabc3bb72,     //   -0.9587,  -0.28441
    0xbfeeaa1d9bb20af3,     0xbfd24bd28bb37672,     //  -0.95827,  -0.28588
    0xbfeea68393e65800,     0xbfd263e6995554ba,     //  -0.95783,  -0.28735
    0xbfeea2e4d212c000,     0xbfd27bf7d0f2c346,     //  -0.95738,  -0.28882
    0xbfee9f4156c62dda,     0xbfd294062ed59f05,     //  -0.95694,  -0.29028
    0xbfee9b99229046f8,     0xbfd2ac11af483572,     //  -0.95649,  -0.29175
    0xbfee97ec36016b30,     0xbfd2c41a4e954520,     //  -0.95605,  -0.29322
    0xbfee943a91aab4b4,     0xbfd2dc200907fe51,     //  -0.95559,  -0.29469
    0xbfee9084361df7f3,     0xbfd2f422daec0386,     //  -0.95514,  -0.29615
    0xbfee8cc923edc388,     0xbfd30c22c08d6a13,     //  -0.95469,  -0.29762
    0xbfee89095bad6025,     0xbfd3241fb638baaf,     //  -0.95423,  -0.29908
    0xbfee8544ddf0d075,     0xbfd33c19b83af207,     //  -0.95377,  -0.30054
    0xbfee817bab4cd10d,     0xbfd35410c2e18152,     //  -0.95331,  -0.30201
    0xbfee7dadc456d850,     0xbfd36c04d27a4edf,     //  -0.95284,  -0.30347
    0xbfee79db29a5165a,     0xbfd383f5e353b6aa,     //  -0.95238,  -0.30493
    0xbfee7603dbce74e9,     0xbfd39be3f1bc8aef,     //  -0.95191,  -0.30639
    0xbfee7227db6a9744,     0xbfd3b3cefa0414b7,     //  -0.95144,  -0.30785
    0xbfee6e472911da27,     0xbfd3cbb6f87a146e,     //  -0.95096,  -0.30931
    0xbfee6a61c55d53a7,     0xbfd3e39be96ec271,     //  -0.95049,  -0.31077
    0xbfee6677b0e6d31e,     0xbfd3fb7dc932cfa4,     //  -0.95001,  -0.31222
    0xbfee6288ec48e112,     0xbfd4135c94176602,     //  -0.94953,  -0.31368
    0xbfee5e95781ebf1c,     0xbfd42b38466e2928,     //  -0.94905,  -0.31514
    0xbfee5a9d550467d3,     0xbfd44310dc8936f0,     //  -0.94856,  -0.31659
    0xbfee56a083968eb1,     0xbfd45ae652bb2800,     //  -0.94807,  -0.31805
    0xbfee529f04729ffc,     0xbfd472b8a5571054,     //  -0.94759,   -0.3195
    0xbfee4e98d836c0af,     0xbfd48a87d0b07fd7,     //  -0.94709,  -0.32096
    0xbfee4a8dff81ce5e,     0xbfd4a253d11b82f3,     //   -0.9466,  -0.32241
    0xbfee467e7af35f23,     0xbfd4ba1ca2eca31c,     //  -0.94611,  -0.32386
    0xbfee426a4b2bc17e,     0xbfd4d1e24278e76a,     //  -0.94561,  -0.32531
    0xbfee3e5170cbfc46,     0xbfd4e9a4ac15d520,     //  -0.94511,  -0.32676
    0xbfee3a33ec75ce85,     0xbfd50163dc197047,     //   -0.9446,  -0.32821
    0xbfee3611becbaf69,     0xbfd5191fceda3c35,     //   -0.9441,  -0.32966
    0xbfee31eae870ce25,     0xbfd530d880af3c24,     //  -0.94359,  -0.33111
    0xbfee2dbf6a0911d9,     0xbfd5488dedeff3be,     //  -0.94308,  -0.33255
    0xbfee298f4439197a,     0xbfd5604012f467b4,     //  -0.94257,    -0.334
    0xbfee255a77a63bb8,     0xbfd577eeec151e47,     //  -0.94206,  -0.33545
    0xbfee212104f686e5,     0xbfd58f9a75ab1fdd,     //  -0.94154,  -0.33689
    0xbfee1ce2ecd0c0d8,     0xbfd5a742ac0ff78d,     //  -0.94103,  -0.33833
    0xbfee18a02fdc66d9,     0xbfd5bee78b9db3b6,     //  -0.94051,  -0.33978
    0xbfee1458cec1ad83,     0xbfd5d68910aee686,     //  -0.93998,  -0.34122
    0xbfee100cca2980ac,     0xbfd5ee27379ea693,     //  -0.93946,  -0.34266
    0xbfee0bbc22bd8349,     0xbfd605c1fcc88f63,     //  -0.93893,   -0.3441
    0xbfee0766d9280f54,     0xbfd61d595c88c203,     //   -0.9384,  -0.34554
    0xbfee030cee1435b8,     0xbfd634ed533be58e,     //  -0.93787,  -0.34698
    0xbfedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     //  -0.93734,  -0.34842
    0xbfedfa4b3621271d,     0xbfd6640af6f03d9e,     //   -0.9368,  -0.34986
    0xbfedf5e36a9ba59c,     0xbfd67b949cad63ca,     //  -0.93627,  -0.35129
    0xbfedf177004b2534,     0xbfd6931acad55f51,     //  -0.93573,  -0.35273
    0xbfeded05f7de47da,     0xbfd6aa9d7dc77e16,     //  -0.93518,  -0.35416
    0xbfede890520465ce,     0xbfd6c21cb1e39771,     //  -0.93464,   -0.3556
    0xbfede4160f6d8d81,     0xbfd6d998638a0cb5,     //  -0.93409,  -0.35703
    0xbfeddf9730ca837b,     0xbfd6f1108f1bc9c5,     //  -0.93354,  -0.35846
    0xbfeddb13b6ccc23d,     0xbfd7088530fa459e,     //  -0.93299,   -0.3599
    0xbfedd68ba2267a25,     0xbfd71ff6458782ec,     //  -0.93244,  -0.36133
    0xbfedd1fef38a915a,     0xbfd73763c9261092,     //  -0.93188,  -0.36276
    0xbfedcd6dabaca3a5,     0xbfd74ecdb8390a3e,     //  -0.93133,  -0.36418
    0xbfedc8d7cb410260,     0xbfd766340f2418f6,     //  -0.93077,  -0.36561
    0xbfedc43d52fcb453,     0xbfd77d96ca4b73a6,     //  -0.93021,  -0.36704
    0xbfedbf9e4395759a,     0xbfd794f5e613dfae,     //  -0.92964,  -0.36847
    0xbfedbafa9dc1b78d,     0xbfd7ac515ee2b172,     //  -0.92907,  -0.36989
    0xbfedb6526238a09b,     0xbfd7c3a9311dcce7,     //  -0.92851,  -0.37132
    0xbfedb1a591b20c38,     0xbfd7dafd592ba621,     //  -0.92794,  -0.37274
    0xbfedacf42ce68ab9,     0xbfd7f24dd37341e3,     //  -0.92736,  -0.37416
    0xbfeda83e348f613b,     0xbfd8099a9c5c362d,     //  -0.92679,  -0.37559
    0xbfeda383a9668988,     0xbfd820e3b04eaac4,     //  -0.92621,  -0.37701
    0xbfed9ec48c26b1f3,     0xbfd838290bb359c8,     //  -0.92563,  -0.37843
    0xbfed9a00dd8b3d46,     0xbfd84f6aaaf3903f,     //  -0.92505,  -0.37985
    0xbfed95389e50429b,     0xbfd866a88a792ea0,     //  -0.92447,  -0.38127
    0xbfed906bcf328d46,     0xbfd87de2a6aea963,     //  -0.92388,  -0.38268
    0xbfed8b9a70ef9cb4,     0xbfd89518fbff098e,     //  -0.92329,   -0.3841
    0xbfed86c48445a450,     0xbfd8ac4b86d5ed44,     //   -0.9227,  -0.38552
    0xbfed81ea09f38b63,     0xbfd8c37a439f884f,     //  -0.92211,  -0.38693
    0xbfed7d0b02b8ecf9,     0xbfd8daa52ec8a4af,     //  -0.92151,  -0.38835
    0xbfed78276f5617c6,     0xbfd8f1cc44bea329,     //  -0.92092,  -0.38976
    0xbfed733f508c0dff,     0xbfd908ef81ef7bd1,     //  -0.92032,  -0.39117
    0xbfed6e52a71c8547,     0xbfd9200ee2c9be97,     //  -0.91972,  -0.39258
    0xbfed696173c9e68b,     0xbfd9372a63bc93d7,     //  -0.91911,  -0.39399
    0xbfed646bb7574de5,     0xbfd94e420137bce3,     //  -0.91851,   -0.3954
    0xbfed5f7172888a7f,     0xbfd96555b7ab948f,     //   -0.9179,  -0.39681
    0xbfed5a72a6221e73,     0xbfd97c6583890fc2,     //  -0.91729,  -0.39822
    0xbfed556f52e93eb1,     0xbfd993716141bdfe,     //  -0.91668,  -0.39962
    0xbfed506779a3d2d9,     0xbfd9aa794d47c9ee,     //  -0.91606,  -0.40103
    0xbfed4b5b1b187524,     0xbfd9c17d440df9f2,     //  -0.91545,  -0.40243
    0xbfed464a380e7242,     0xbfd9d87d4207b0ab,     //  -0.91483,  -0.40384
    0xbfed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //  -0.91421,  -0.40524
    0xbfed3c1ae79f2b4e,     0xbfda067145664d57,     //  -0.91359,  -0.40664
    0xbfed36fc7bcbfbdc,     0xbfda1d6543b50ac0,     //  -0.91296,  -0.40804
    0xbfed31d98e9e503a,     0xbfda34553b0afee5,     //  -0.91234,  -0.40944
    0xbfed2cb220e0ef9f,     0xbfda4b4127dea1e4,     //  -0.91171,  -0.41084
    0xbfed2786335f52fc,     0xbfda622906a70b63,     //  -0.91107,  -0.41224
    0xbfed2255c6e5a4e1,     0xbfda790cd3dbf31a,     //  -0.91044,  -0.41364
    0xbfed1d20dc40c15c,     0xbfda8fec8bf5b166,     //  -0.90981,  -0.41503
    0xbfed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     //  -0.90917,  -0.41643
    0xbfed12a98fac410c,     0xbfdabd9faebc3980,     //  -0.90853,  -0.41782
    0xbfed0d672f59d2b9,     0xbfdad473125cdc08,     //  -0.90789,  -0.41922
    0xbfed082054168bac,     0xbfdaeb4252ca07ab,     //  -0.90724,  -0.42061
    0xbfed02d4feb2bd92,     0xbfdb020d6c7f4009,     //   -0.9066,    -0.422
    0xbfecfd852fff6ad4,     0xbfdb18d45bf8aca6,     //  -0.90595,  -0.42339
    0xbfecf830e8ce467b,     0xbfdb2f971db31972,     //   -0.9053,  -0.42478
    0xbfecf2d829f1b40e,     0xbfdb4655ae2bf757,     //  -0.90464,  -0.42617
    0xbfeced7af43cc773,     0xbfdb5d1009e15cc0,     //  -0.90399,  -0.42756
    0xbfece819488344ce,     0xbfdb73c62d520624,     //  -0.90333,  -0.42894
    0xbfece2b32799a060,     0xbfdb8a7814fd5693,     //  -0.90267,  -0.43033
    0xbfecdd489254fe65,     0xbfdba125bd63583e,     //  -0.90201,  -0.43171
    0xbfecd7d9898b32f6,     0xbfdbb7cf2304bd01,     //  -0.90135,  -0.43309
    0xbfecd2660e12c1e6,     0xbfdbce744262deee,     //  -0.90068,  -0.43448
    0xbfecccee20c2de9f,     0xbfdbe51517ffc0d9,     //  -0.90002,  -0.43586
    0xbfecc771c2736c09,     0xbfdbfbb1a05e0edc,     //  -0.89935,  -0.43724
    0xbfecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     //  -0.89867,  -0.43862
    0xbfecbc6bb638d10b,     0xbfdc28ddbb6cf145,     //    -0.898,  -0.43999
    0xbfecb6e20a00da99,     0xbfdc3f6d47263129,     //  -0.89732,  -0.44137
    0xbfecb153f02fb87d,     0xbfdc55f877b23537,     //  -0.89665,  -0.44275
    0xbfecabc169a0b901,     0xbfdc6c7f4997000a,     //  -0.89597,  -0.44412
    0xbfeca62a772fd919,     0xbfdc8301b95b40c2,     //  -0.89528,   -0.4455
    0xbfeca08f19b9c449,     0xbfdc997fc3865388,     //   -0.8946,  -0.44687
    0xbfec9aef521bd480,     0xbfdcaff964a0421d,     //  -0.89391,  -0.44824
    0xbfec954b213411f5,     0xbfdcc66e9931c45d,     //  -0.89322,  -0.44961
    0xbfec8fa287e13305,     0xbfdcdcdf5dc440ce,     //  -0.89253,  -0.45098
    0xbfec89f587029c13,     0xbfdcf34baee1cd21,     //  -0.89184,  -0.45235
    0xbfec84441f785f61,     0xbfdd09b389152ec1,     //  -0.89115,  -0.45372
    0xbfec7e8e52233cf3,     0xbfdd2016e8e9db5b,     //  -0.89045,  -0.45508
    0xbfec78d41fe4a267,     0xbfdd3675caebf962,     //  -0.88975,  -0.45645
    0xbfec7315899eaad7,     0xbfdd4cd02ba8609c,     //  -0.88905,  -0.45781
    0xbfec6d5290341eb2,     0xbfdd632607ac9aa9,     //  -0.88835,  -0.45918
    0xbfec678b3488739b,     0xbfdd79775b86e389,     //  -0.88764,  -0.46054
    0xbfec61bf777fcc48,     0xbfdd8fc423c62a25,     //  -0.88693,   -0.4619
    0xbfec5bef59fef85a,     0xbfdda60c5cfa10d8,     //  -0.88622,  -0.46326
    0xbfec561adceb743e,     0xbfddbc5003b2edf8,     //  -0.88551,  -0.46462
    0xbfec5042012b6907,     0xbfddd28f1481cc58,     //   -0.8848,  -0.46598
    0xbfec4a64c7a5ac4c,     0xbfdde8c98bf86bd6,     //  -0.88408,  -0.46733
    0xbfec44833141c004,     0xbfddfeff66a941de,     //  -0.88336,  -0.46869
    0xbfec3e9d3ee7d262,     0xbfde1530a12779f4,     //  -0.88264,  -0.47004
    0xbfec38b2f180bdb1,     0xbfde2b5d3806f63b,     //  -0.88192,   -0.4714
    0xbfec32c449f60831,     0xbfde418527dc4ffa,     //   -0.8812,  -0.47275
    0xbfec2cd14931e3f1,     0xbfde57a86d3cd824,     //  -0.88047,   -0.4741
    0xbfec26d9f01f2eaf,     0xbfde6dc704be97e2,     //  -0.87974,  -0.47545
    0xbfec20de3fa971b0,     0xbfde83e0eaf85113,     //  -0.87901,   -0.4768
    0xbfec1ade38bce19b,     0xbfde99f61c817eda,     //  -0.87828,  -0.47815
    0xbfec14d9dc465e58,     0xbfdeb00695f25620,     //  -0.87755,  -0.47949
    0xbfec0ed12b3372e9,     0xbfdec61253e3c61b,     //  -0.87681,  -0.48084
    0xbfec08c426725549,     0xbfdedc1952ef78d5,     //  -0.87607,  -0.48218
    0xbfec02b2cef1e641,     0xbfdef21b8fafd3b5,     //  -0.87533,  -0.48353
    0xbfebfc9d25a1b147,     0xbfdf081906bff7fd,     //  -0.87459,  -0.48487
    0xbfebf6832b71ec5b,     0xbfdf1e11b4bbc35c,     //  -0.87384,  -0.48621
    0xbfebf064e15377dd,     0xbfdf3405963fd068,     //  -0.87309,  -0.48755
    0xbfebea424837de6d,     0xbfdf49f4a7e97729,     //  -0.87235,  -0.48889
    0xbfebe41b611154c1,     0xbfdf5fdee656cda3,     //   -0.8716,  -0.49023
    0xbfebddf02cd2b983,     0xbfdf75c44e26a852,     //  -0.87084,  -0.49156
    0xbfebd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //  -0.87009,   -0.4929
    0xbfebd18ce0dc19d6,     0xbfdfa1808c6cf7e0,     //  -0.86933,  -0.49423
    0xbfebcb54cb0d2327,     0xbfdfb7575c24d2de,     //  -0.86857,  -0.49557
    0xbfebc5186bf8361d,     0xbfdfcd2947c1ff57,     //  -0.86781,   -0.4969
    0xbfebbed7c49380ea,     0xbfdfe2f64be7120f,     //  -0.86705,  -0.49823
    0xbfebb892d5d5dad5,     0xbfdff8be6537615e,     //  -0.86628,  -0.49956
    0xbfebb249a0b6c40d,     0xbfe00740c82b82e0,     //  -0.86551,  -0.50089
    0xbfebabfc262e6586,     0xbfe0121fe4f56d2c,     //  -0.86474,  -0.50221
    0xbfeba5aa673590d2,     0xbfe01cfc874c3eb7,     //  -0.86397,  -0.50354
    0xbfeb9f5464c5bffc,     0xbfe027d6ad83287e,     //   -0.8632,  -0.50486
    0xbfeb98fa1fd9155e,     0xbfe032ae55edbd95,     //  -0.86242,  -0.50619
    0xbfeb929b996a5b7f,     0xbfe03d837edff370,     //  -0.86165,  -0.50751
    0xbfeb8c38d27504e9,     0xbfe0485626ae221a,     //  -0.86087,  -0.50883
    0xbfeb85d1cbf52c02,     0xbfe053264bad0483,     //  -0.86009,  -0.51015
    0xbfeb7f6686e792ea,     0xbfe05df3ec31b8b6,     //   -0.8593,  -0.51147
    0xbfeb78f70449a34b,     0xbfe068bf0691c028,     //  -0.85852,  -0.51279
    0xbfeb728345196e3e,     0xbfe073879922ffed,     //  -0.85773,   -0.5141
    0xbfeb6c0b4a55ac17,     0xbfe07e4da23bc102,     //  -0.85694,  -0.51542
    0xbfeb658f14fdbc47,     0xbfe089112032b08c,     //  -0.85615,  -0.51673
    0xbfeb5f0ea611a532,     0xbfe093d2115ee018,     //  -0.85535,  -0.51804
    0xbfeb5889fe921405,     0xbfe09e907417c5e1,     //  -0.85456,  -0.51936
    0xbfeb52011f805c92,     0xbfe0a94c46b53d0b,     //  -0.85376,  -0.52067
    0xbfeb4b7409de7925,     0xbfe0b405878f85ec,     //  -0.85296,  -0.52198
    0xbfeb44e2beaf0a61,     0xbfe0bebc34ff4646,     //  -0.85216,  -0.52328
    0xbfeb3e4d3ef55712,     0xbfe0c9704d5d898f,     //  -0.85136,  -0.52459
    0xbfeb37b38bb54c09,     0xbfe0d421cf03c12b,     //  -0.85055,   -0.5259
    0xbfeb3115a5f37bf4,     0xbfe0ded0b84bc4b5,     //  -0.84974,   -0.5272
    0xbfeb2a738eb51f33,     0xbfe0e97d078fd23b,     //  -0.84893,   -0.5285
    0xbfeb23cd470013b4,     0xbfe0f426bb2a8e7d,     //  -0.84812,   -0.5298
    0xbfeb1d22cfdadcc6,     0xbfe0fecdd1770537,     //  -0.84731,   -0.5311
    0xbfeb16742a4ca2f5,     0xbfe1097248d0a956,     //  -0.84649,   -0.5324
    0xbfeb0fc1575d33db,     0xbfe114141f935545,     //  -0.84567,   -0.5337
    0xbfeb090a58150200,     0xbfe11eb3541b4b22,     //  -0.84485,    -0.535
    0xbfeb024f2d7d24a9,     0xbfe1294fe4c5350a,     //  -0.84403,  -0.53629
    0xbfeafb8fd89f57b6,     0xbfe133e9cfee254e,     //  -0.84321,  -0.53759
    0xbfeaf4cc5a85fb73,     0xbfe13e8113f396c1,     //  -0.84238,  -0.53888
    0xbfeaee04b43c1474,     0xbfe14915af336ceb,     //  -0.84155,  -0.54017
    0xbfeae738e6cd4b67,     0xbfe153a7a00bf453,     //  -0.84073,  -0.54146
    0xbfeae068f345ecef,     0xbfe15e36e4dbe2bc,     //  -0.83989,  -0.54275
    0xbfead994dab2e979,     0xbfe168c37c025764,     //  -0.83906,  -0.54404
    0xbfead2bc9e21d511,     0xbfe1734d63dedb49,     //  -0.83822,  -0.54532
    0xbfeacbe03ea0e73b,     0xbfe17dd49ad16161,     //  -0.83739,  -0.54661
    0xbfeac4ffbd3efac8,     0xbfe188591f3a46e5,     //  -0.83655,  -0.54789
    0xbfeabe1b1b0b8dac,     0xbfe192daef7a5386,     //  -0.83571,  -0.54918
    0xbfeab7325916c0d4,     0xbfe19d5a09f2b9b8,     //  -0.83486,  -0.55046
    0xbfeab045787157ff,     0xbfe1a7d66d0516e6,     //  -0.83402,  -0.55174
    0xbfeaa9547a2cb98e,     0xbfe1b250171373be,     //  -0.83317,  -0.55302
    0xbfeaa25f5f5aee60,     0xbfe1bcc706804467,     //  -0.83232,  -0.55429
    0xbfea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //  -0.83147,  -0.55557
    0xbfea9468d85b20ae,     0xbfe1d1acaf012cc2,     //  -0.83062,  -0.55685
    0xbfea8d676e545ad2,     0xbfe1dc1b64dc4872,     //  -0.82976,  -0.55812
    0xbfea8661ec0ee133,     0xbfe1e68759a3e074,     //   -0.8289,  -0.55939
    0xbfea7f58529fe69d,     0xbfe1f0f08bbc861b,     //  -0.82805,  -0.56066
    0xbfea784aa31d3f55,     0xbfe1fb56f98b37b8,     //  -0.82718,  -0.56193
    0xbfea7138de9d60f5,     0xbfe205baa17560d6,     //  -0.82632,   -0.5632
    0xbfea6a230637623b,     0xbfe2101b81e0da78,     //  -0.82546,  -0.56447
    0xbfea63091b02fae2,     0xbfe21a799933eb58,     //  -0.82459,  -0.56573
    0xbfea5beb1e188375,     0xbfe224d4e5d5482e,     //  -0.82372,    -0.567
    0xbfea54c91090f524,     0xbfe22f2d662c13e1,     //  -0.82285,  -0.56826
    0xbfea4da2f385e997,     0xbfe23983189fdfd5,     //  -0.82198,  -0.56952
    0xbfea4678c8119ac8,     0xbfe243d5fb98ac1f,     //   -0.8211,  -0.57078
    0xbfea3f4a8f4ee2d2,     0xbfe24e260d7ee7c9,     //  -0.82023,  -0.57204
    0xbfea38184a593bc6,     0xbfe258734cbb7110,     //  -0.81935,   -0.5733
    0xbfea30e1fa4cbf81,     0xbfe262bdb7b795a2,     //  -0.81847,  -0.57455
    0xbfea29a7a0462782,     0xbfe26d054cdd12df,     //  -0.81758,  -0.57581
    0xbfea22693d62ccb9,     0xbfe2774a0a961612,     //   -0.8167,  -0.57706
    0xbfea1b26d2c0a75e,     0xbfe2818bef4d3cba,     //  -0.81581,  -0.57831
    0xbfea13e0617e4ec7,     0xbfe28bcaf96d94ba,     //  -0.81493,  -0.57956
    0xbfea0c95eabaf937,     0xbfe2960727629ca8,     //  -0.81404,  -0.58081
    0xbfea05476f967bb5,     0xbfe2a040779843fb,     //  -0.81314,  -0.58206
    0xbfe9fdf4f13149de,     0xbfe2aa76e87aeb58,     //  -0.81225,  -0.58331
    0xbfe9f69e70ac75bc,     0xbfe2b4aa787764c4,     //  -0.81135,  -0.58455
    0xbfe9ef43ef29af94,     0xbfe2bedb25faf3ea,     //  -0.81046,   -0.5858
    0xbfe9e7e56dcb45bd,     0xbfe2c908ef734e57,     //  -0.80956,  -0.58704
    0xbfe9e082edb42472,     0xbfe2d333d34e9bb7,     //  -0.80866,  -0.58828
    0xbfe9d91c7007d5a6,     0xbfe2dd5bcffb7616,     //  -0.80775,  -0.58952
    0xbfe9d1b1f5ea80d6,     0xbfe2e780e3e8ea16,     //  -0.80685,  -0.59076
    0xbfe9ca438080eadb,     0xbfe2f1a30d86773a,     //  -0.80594,    -0.592
    0xbfe9c2d110f075c3,     0xbfe2fbc24b441015,     //  -0.80503,  -0.59323
    0xbfe9bb5aa85f2098,     0xbfe305de9b921a94,     //  -0.80412,  -0.59447
    0xbfe9b3e047f38741,     0xbfe30ff7fce17035,     //  -0.80321,   -0.5957
    0xbfe9ac61f0d4e247,     0xbfe31a0e6da35e44,     //  -0.80229,  -0.59693
    0xbfe9a4dfa42b06b2,     0xbfe32421ec49a620,     //  -0.80138,  -0.59816
    0xbfe99d59631e65d5,     0xbfe32e3277467d6b,     //  -0.80046,  -0.59939
    0xbfe995cf2ed80d22,     0xbfe338400d0c8e57,     //  -0.79954,  -0.60062
    0xbfe98e410881a600,     0xbfe3424aac0ef7d6,     //  -0.79861,  -0.60184
    0xbfe986aef1457594,     0xbfe34c5252c14de1,     //  -0.79769,  -0.60307
    0xbfe97f18ea4e5c9e,     0xbfe35656ff9799ae,     //  -0.79676,  -0.60429
    0xbfe9777ef4c7d742,     0xbfe36058b10659f3,     //  -0.79584,  -0.60551
    0xbfe96fe111ddfce0,     0xbfe36a576582831b,     //  -0.79491,  -0.60673
    0xbfe9683f42bd7fe1,     0xbfe374531b817f8d,     //  -0.79398,  -0.60795
    0xbfe960998893ad8c,     0xbfe37e4bd1792fe2,     //  -0.79304,  -0.60917
    0xbfe958efe48e6dd7,     0xbfe3884185dfeb22,     //  -0.79211,  -0.61038
    0xbfe9514257dc4335,     0xbfe39234372c7f04,     //  -0.79117,   -0.6116
    0xbfe94990e3ac4a6c,     0xbfe39c23e3d63029,     //  -0.79023,  -0.61281
    0xbfe941db892e3a65,     0xbfe3a6108a54ba58,     //  -0.78929,  -0.61402
    0xbfe93a22499263fc,     0xbfe3affa292050b9,     //  -0.78835,  -0.61523
    0xbfe932652609b1cf,     0xbfe3b9e0beb19e18,     //   -0.7874,  -0.61644
    0xbfe92aa41fc5a815,     0xbfe3c3c44981c517,     //  -0.78646,  -0.61765
    0xbfe922df37f8646a,     0xbfe3cda4c80a6076,     //  -0.78551,  -0.61885
    0xbfe91b166fd49da2,     0xbfe3d78238c58343,     //  -0.78456,  -0.62006
    0xbfe91349c88da398,     0xbfe3e15c9a2db922,     //   -0.7836,  -0.62126
    0xbfe90b7943575efe,     0xbfe3eb33eabe0680,     //  -0.78265,  -0.62246
    0xbfe903a4e1665133,     0xbfe3f50828f1e8d2,     //  -0.78169,  -0.62366
    0xbfe8fbcca3ef940d,     0xbfe3fed9534556d4,     //  -0.78074,  -0.62486
    0xbfe8f3f08c28d9ac,     0xbfe408a76834c0c0,     //  -0.77978,  -0.62606
    0xbfe8ec109b486c49,     0xbfe41272663d108c,     //  -0.77882,  -0.62725
    0xbfe8e42cd2852e0a,     0xbfe41c3a4bdbaa26,     //  -0.77785,  -0.62845
    0xbfe8dc45331698cc,     0xbfe425ff178e6bb1,     //  -0.77689,  -0.62964
    0xbfe8d459be34bdfa,     0xbfe42fc0c7d3adbb,     //  -0.77592,  -0.63083
    0xbfe8cc6a75184655,     0xbfe4397f5b2a4380,     //  -0.77495,  -0.63202
    0xbfe8c47758fa71cb,     0xbfe4433ad0117b1d,     //  -0.77398,  -0.63321
    0xbfe8bc806b151741,     0xbfe44cf325091dd6,     //  -0.77301,  -0.63439
    0xbfe8b485aca2a468,     0xbfe456a858917046,     //  -0.77204,  -0.63558
    0xbfe8ac871ede1d88,     0xbfe4605a692b32a2,     //  -0.77106,  -0.63676
    0xbfe8a484c3031d50,     0xbfe46a095557a0f1,     //  -0.77008,  -0.63794
    0xbfe89c7e9a4dd4ab,     0xbfe473b51b987347,     //   -0.7691,  -0.63912
    0xbfe89474a5fb0a84,     0xbfe47d5dba6fde01,     //  -0.76812,   -0.6403
    0xbfe88c66e7481ba1,     0xbfe48703306091fe,     //  -0.76714,  -0.64148
    0xbfe884555f72fa6b,     0xbfe490a57bedbcdf,     //  -0.76615,  -0.64266
    0xbfe87c400fba2ebf,     0xbfe49a449b9b0938,     //  -0.76517,  -0.64383
    0xbfe87426f95cd5bd,     0xbfe4a3e08dec9ed6,     //  -0.76418,    -0.645
    0xbfe86c0a1d9aa195,     0xbfe4ad79516722f0,     //  -0.76319,  -0.64618
    0xbfe863e97db3d95a,     0xbfe4b70ee48fb869,     //   -0.7622,  -0.64735
    0xbfe85bc51ae958cc,     0xbfe4c0a145ec0004,     //   -0.7612,  -0.64851
    0xbfe8539cf67c9029,     0xbfe4ca30740218a3,     //  -0.76021,  -0.64968
    0xbfe84b7111af83f9,     0xbfe4d3bc6d589f80,     //  -0.75921,  -0.65085
    0xbfe843416dc4cce2,     0xbfe4dd453076b064,     //  -0.75821,  -0.65201
    0xbfe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //  -0.75721,  -0.65317
    0xbfe832d6eda3a3e0,     0xbfe4f04d0e2859aa,     //  -0.75621,  -0.65433
    0xbfe82a9c13f545ff,     0xbfe4f9cc25cca486,     //   -0.7552,  -0.65549
    0xbfe8225d803964e5,     0xbfe503480159ded2,     //  -0.75419,  -0.65665
    0xbfe81a1b33b57acc,     0xbfe50cc09f59a09b,     //  -0.75319,  -0.65781
    0xbfe811d52faf94dc,     0xbfe51635fe5601d7,     //  -0.75218,  -0.65896
    0xbfe8098b756e52fa,     0xbfe51fa81cd99aa6,     //  -0.75117,  -0.66011
    0xbfe8013e0638e795,     0xbfe52916f96f8388,     //  -0.75015,  -0.66127
    0xbfe7f8ece3571771,     0xbfe5328292a35596,     //  -0.74914,  -0.66242
    0xbfe7f0980e113978,     0xbfe53beae7012abe,     //  -0.74812,  -0.66356
    0xbfe7e83f87b03686,     0xbfe5454ff5159dfb,     //   -0.7471,  -0.66471
    0xbfe7dfe3517d8937,     0xbfe54eb1bb6dcb8f,     //  -0.74608,  -0.66586
    0xbfe7d7836cc33db2,     0xbfe5581038975137,     //  -0.74506,    -0.667
    0xbfe7cf1fdacbf179,     0xbfe5616b6b204e6e,     //  -0.74403,  -0.66814
    0xbfe7c6b89ce2d333,     0xbfe56ac35197649e,     //  -0.74301,  -0.66928
    0xbfe7be4db453a27c,     0xbfe57417ea8bb75c,     //  -0.74198,  -0.67042
    0xbfe7b5df226aafb0,     0xbfe57d69348cec9f,     //  -0.74095,  -0.67156
    0xbfe7ad6ce874dbb6,     0xbfe586b72e2b2cfd,     //  -0.73992,  -0.67269
    0xbfe7a4f707bf97d2,     0xbfe59001d5f723df,     //  -0.73889,  -0.67383
    0xbfe79c7d8198e56e,     0xbfe599492a81ffbc,     //  -0.73785,  -0.67496
    0xbfe79400574f55e4,     0xbfe5a28d2a5d7250,     //  -0.73682,  -0.67609
    0xbfe78b7f8a320a52,     0xbfe5abcdd41bb0d8,     //  -0.73578,  -0.67722
    0xbfe782fb1b90b35b,     0xbfe5b50b264f7448,     //  -0.73474,  -0.67835
    0xbfe77a730cbb9100,     0xbfe5be451f8bf980,     //   -0.7337,  -0.67948
    0xbfe771e75f037261,     0xbfe5c77bbe65018c,     //  -0.73265,   -0.6806
    0xbfe7695813b9b594,     0xbfe5d0af016ed1d4,     //  -0.73161,  -0.68172
    0xbfe760c52c304764,     0xbfe5d9dee73e345c,     //  -0.73056,  -0.68285
    0xbfe7582ea9b9a329,     0xbfe5e30b6e6877f3,     //  -0.72951,  -0.68397
    0xbfe74f948da8d28d,     0xbfe5ec3495837074,     //  -0.72846,  -0.68508
    0xbfe746f6d9516d59,     0xbfe5f55a5b2576f8,     //  -0.72741,   -0.6862
    0xbfe73e558e079942,     0xbfe5fe7cbde56a0f,     //  -0.72636,  -0.68732
    0xbfe735b0ad2009b2,     0xbfe6079bbc5aadfa,     //   -0.7253,  -0.68843
    0xbfe72d0837efff97,     0xbfe610b7551d2cde,     //  -0.72425,  -0.68954
    0xbfe7245c2fcd492a,     0xbfe619cf86c55702,     //  -0.72319,  -0.69065
    0xbfe71bac960e41bf,     0xbfe622e44fec22ff,     //  -0.72213,  -0.69176
    0xbfe712f96c09d18d,     0xbfe62bf5af2b0dfd,     //  -0.72107,  -0.69287
    0xbfe70a42b3176d7a,     0xbfe63503a31c1be8,     //     -0.72,  -0.69397
    0xbfe701886c8f16e6,     0xbfe63e0e2a59d7aa,     //  -0.71894,  -0.69508
    0xbfe6f8ca99c95b75,     0xbfe64715437f535b,     //  -0.71787,  -0.69618
    0xbfe6f0093c1f54de,     0xbfe65018ed28287f,     //   -0.7168,  -0.69728
    0xbfe6e74454eaa8ae,     0xbfe6591925f0783e,     //  -0.71573,  -0.69838
    0xbfe6de7be585881d,     0xbfe66215ec74eb91,     //  -0.71466,  -0.69947
    0xbfe6d5afef4aafcc,     0xbfe66b0f3f52b386,     //  -0.71358,  -0.70057
    0xbfe6cce07395679f,     0xbfe674051d27896c,     //  -0.71251,  -0.70166
    0xbfe6c40d73c18275,     0xbfe67cf78491af10,     //  -0.71143,  -0.70275
    0xbfe6bb36f12b5e06,     0xbfe685e6742feeef,     //  -0.71035,  -0.70385
    0xbfe6b25ced2fe29c,     0xbfe68ed1eaa19c71,     //  -0.70927,  -0.70493
    0xbfe6a97f692c82ea,     0xbfe697b9e686941c,     //  -0.70819,  -0.70602
    0xbfe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //  -0.70711,  -0.70711
    0xbfe697b9e686941c,     0xbfe6a97f692c82ea,     //  -0.70602,  -0.70819
    0xbfe68ed1eaa19c71,     0xbfe6b25ced2fe29c,     //  -0.70493,  -0.70927
    0xbfe685e6742feeef,     0xbfe6bb36f12b5e06,     //  -0.70385,  -0.71035
    0xbfe67cf78491af10,     0xbfe6c40d73c18275,     //  -0.70275,  -0.71143
    0xbfe674051d27896c,     0xbfe6cce07395679f,     //  -0.70166,  -0.71251
    0xbfe66b0f3f52b386,     0xbfe6d5afef4aafcc,     //  -0.70057,  -0.71358
    0xbfe66215ec74eb91,     0xbfe6de7be585881d,     //  -0.69947,  -0.71466
    0xbfe6591925f0783e,     0xbfe6e74454eaa8ae,     //  -0.69838,  -0.71573
    0xbfe65018ed28287f,     0xbfe6f0093c1f54de,     //  -0.69728,   -0.7168
    0xbfe64715437f535b,     0xbfe6f8ca99c95b75,     //  -0.69618,  -0.71787
    0xbfe63e0e2a59d7aa,     0xbfe701886c8f16e6,     //  -0.69508,  -0.71894
    0xbfe63503a31c1be8,     0xbfe70a42b3176d7a,     //  -0.69397,     -0.72
    0xbfe62bf5af2b0dfd,     0xbfe712f96c09d18d,     //  -0.69287,  -0.72107
    0xbfe622e44fec22ff,     0xbfe71bac960e41bf,     //  -0.69176,  -0.72213
    0xbfe619cf86c55702,     0xbfe7245c2fcd492a,     //  -0.69065,  -0.72319
    0xbfe610b7551d2cde,     0xbfe72d0837efff97,     //  -0.68954,  -0.72425
    0xbfe6079bbc5aadfa,     0xbfe735b0ad2009b2,     //  -0.68843,   -0.7253
    0xbfe5fe7cbde56a0f,     0xbfe73e558e079942,     //  -0.68732,  -0.72636
    0xbfe5f55a5b2576f8,     0xbfe746f6d9516d59,     //   -0.6862,  -0.72741
    0xbfe5ec3495837074,     0xbfe74f948da8d28d,     //  -0.68508,  -0.72846
    0xbfe5e30b6e6877f3,     0xbfe7582ea9b9a329,     //  -0.68397,  -0.72951
    0xbfe5d9dee73e345c,     0xbfe760c52c304764,     //  -0.68285,  -0.73056
    0xbfe5d0af016ed1d4,     0xbfe7695813b9b594,     //  -0.68172,  -0.73161
    0xbfe5c77bbe65018c,     0xbfe771e75f037261,     //   -0.6806,  -0.73265
    0xbfe5be451f8bf980,     0xbfe77a730cbb9100,     //  -0.67948,   -0.7337
    0xbfe5b50b264f7448,     0xbfe782fb1b90b35b,     //  -0.67835,  -0.73474
    0xbfe5abcdd41bb0d8,     0xbfe78b7f8a320a52,     //  -0.67722,  -0.73578
    0xbfe5a28d2a5d7250,     0xbfe79400574f55e4,     //  -0.67609,  -0.73682
    0xbfe599492a81ffbc,     0xbfe79c7d8198e56e,     //  -0.67496,  -0.73785
    0xbfe59001d5f723df,     0xbfe7a4f707bf97d2,     //  -0.67383,  -0.73889
    0xbfe586b72e2b2cfd,     0xbfe7ad6ce874dbb6,     //  -0.67269,  -0.73992
    0xbfe57d69348cec9f,     0xbfe7b5df226aafb0,     //  -0.67156,  -0.74095
    0xbfe57417ea8bb75c,     0xbfe7be4db453a27c,     //  -0.67042,  -0.74198
    0xbfe56ac35197649e,     0xbfe7c6b89ce2d333,     //  -0.66928,  -0.74301
    0xbfe5616b6b204e6e,     0xbfe7cf1fdacbf179,     //  -0.66814,  -0.74403
    0xbfe5581038975137,     0xbfe7d7836cc33db2,     //    -0.667,  -0.74506
    0xbfe54eb1bb6dcb8f,     0xbfe7dfe3517d8937,     //  -0.66586,  -0.74608
    0xbfe5454ff5159dfb,     0xbfe7e83f87b03686,     //  -0.66471,   -0.7471
    0xbfe53beae7012abe,     0xbfe7f0980e113978,     //  -0.66356,  -0.74812
    0xbfe5328292a35596,     0xbfe7f8ece3571771,     //  -0.66242,  -0.74914
    0xbfe52916f96f8388,     0xbfe8013e0638e795,     //  -0.66127,  -0.75015
    0xbfe51fa81cd99aa6,     0xbfe8098b756e52fa,     //  -0.66011,  -0.75117
    0xbfe51635fe5601d7,     0xbfe811d52faf94dc,     //  -0.65896,  -0.75218
    0xbfe50cc09f59a09b,     0xbfe81a1b33b57acc,     //  -0.65781,  -0.75319
    0xbfe503480159ded2,     0xbfe8225d803964e5,     //  -0.65665,  -0.75419
    0xbfe4f9cc25cca486,     0xbfe82a9c13f545ff,     //  -0.65549,   -0.7552
    0xbfe4f04d0e2859aa,     0xbfe832d6eda3a3e0,     //  -0.65433,  -0.75621
    0xbfe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //  -0.65317,  -0.75721
    0xbfe4dd453076b064,     0xbfe843416dc4cce2,     //  -0.65201,  -0.75821
    0xbfe4d3bc6d589f80,     0xbfe84b7111af83f9,     //  -0.65085,  -0.75921
    0xbfe4ca30740218a3,     0xbfe8539cf67c9029,     //  -0.64968,  -0.76021
    0xbfe4c0a145ec0004,     0xbfe85bc51ae958cc,     //  -0.64851,   -0.7612
    0xbfe4b70ee48fb869,     0xbfe863e97db3d95a,     //  -0.64735,   -0.7622
    0xbfe4ad79516722f0,     0xbfe86c0a1d9aa195,     //  -0.64618,  -0.76319
    0xbfe4a3e08dec9ed6,     0xbfe87426f95cd5bd,     //    -0.645,  -0.76418
    0xbfe49a449b9b0938,     0xbfe87c400fba2ebf,     //  -0.64383,  -0.76517
    0xbfe490a57bedbcdf,     0xbfe884555f72fa6b,     //  -0.64266,  -0.76615
    0xbfe48703306091fe,     0xbfe88c66e7481ba1,     //  -0.64148,  -0.76714
    0xbfe47d5dba6fde01,     0xbfe89474a5fb0a84,     //   -0.6403,  -0.76812
    0xbfe473b51b987347,     0xbfe89c7e9a4dd4ab,     //  -0.63912,   -0.7691
    0xbfe46a095557a0f1,     0xbfe8a484c3031d50,     //  -0.63794,  -0.77008
    0xbfe4605a692b32a2,     0xbfe8ac871ede1d88,     //  -0.63676,  -0.77106
    0xbfe456a858917046,     0xbfe8b485aca2a468,     //  -0.63558,  -0.77204
    0xbfe44cf325091dd6,     0xbfe8bc806b151741,     //  -0.63439,  -0.77301
    0xbfe4433ad0117b1d,     0xbfe8c47758fa71cb,     //  -0.63321,  -0.77398
    0xbfe4397f5b2a4380,     0xbfe8cc6a75184655,     //  -0.63202,  -0.77495
    0xbfe42fc0c7d3adbb,     0xbfe8d459be34bdfa,     //  -0.63083,  -0.77592
    0xbfe425ff178e6bb1,     0xbfe8dc45331698cc,     //  -0.62964,  -0.77689
    0xbfe41c3a4bdbaa26,     0xbfe8e42cd2852e0a,     //  -0.62845,  -0.77785
    0xbfe41272663d108c,     0xbfe8ec109b486c49,     //  -0.62725,  -0.77882
    0xbfe408a76834c0c0,     0xbfe8f3f08c28d9ac,     //  -0.62606,  -0.77978
    0xbfe3fed9534556d4,     0xbfe8fbcca3ef940d,     //  -0.62486,  -0.78074
    0xbfe3f50828f1e8d2,     0xbfe903a4e1665133,     //  -0.62366,  -0.78169
    0xbfe3eb33eabe0680,     0xbfe90b7943575efe,     //  -0.62246,  -0.78265
    0xbfe3e15c9a2db922,     0xbfe91349c88da398,     //  -0.62126,   -0.7836
    0xbfe3d78238c58343,     0xbfe91b166fd49da2,     //  -0.62006,  -0.78456
    0xbfe3cda4c80a6076,     0xbfe922df37f8646a,     //  -0.61885,  -0.78551
    0xbfe3c3c44981c517,     0xbfe92aa41fc5a815,     //  -0.61765,  -0.78646
    0xbfe3b9e0beb19e18,     0xbfe932652609b1cf,     //  -0.61644,   -0.7874
    0xbfe3affa292050b9,     0xbfe93a22499263fc,     //  -0.61523,  -0.78835
    0xbfe3a6108a54ba58,     0xbfe941db892e3a65,     //  -0.61402,  -0.78929
    0xbfe39c23e3d63029,     0xbfe94990e3ac4a6c,     //  -0.61281,  -0.79023
    0xbfe39234372c7f04,     0xbfe9514257dc4335,     //   -0.6116,  -0.79117
    0xbfe3884185dfeb22,     0xbfe958efe48e6dd7,     //  -0.61038,  -0.79211
    0xbfe37e4bd1792fe2,     0xbfe960998893ad8c,     //  -0.60917,  -0.79304
    0xbfe374531b817f8d,     0xbfe9683f42bd7fe1,     //  -0.60795,  -0.79398
    0xbfe36a576582831b,     0xbfe96fe111ddfce0,     //  -0.60673,  -0.79491
    0xbfe36058b10659f3,     0xbfe9777ef4c7d742,     //  -0.60551,  -0.79584
    0xbfe35656ff9799ae,     0xbfe97f18ea4e5c9e,     //  -0.60429,  -0.79676
    0xbfe34c5252c14de1,     0xbfe986aef1457594,     //  -0.60307,  -0.79769
    0xbfe3424aac0ef7d6,     0xbfe98e410881a600,     //  -0.60184,  -0.79861
    0xbfe338400d0c8e57,     0xbfe995cf2ed80d22,     //  -0.60062,  -0.79954
    0xbfe32e3277467d6b,     0xbfe99d59631e65d5,     //  -0.59939,  -0.80046
    0xbfe32421ec49a620,     0xbfe9a4dfa42b06b2,     //  -0.59816,  -0.80138
    0xbfe31a0e6da35e44,     0xbfe9ac61f0d4e247,     //  -0.59693,  -0.80229
    0xbfe30ff7fce17035,     0xbfe9b3e047f38741,     //   -0.5957,  -0.80321
    0xbfe305de9b921a94,     0xbfe9bb5aa85f2098,     //  -0.59447,  -0.80412
    0xbfe2fbc24b441015,     0xbfe9c2d110f075c3,     //  -0.59323,  -0.80503
    0xbfe2f1a30d86773a,     0xbfe9ca438080eadb,     //    -0.592,  -0.80594
    0xbfe2e780e3e8ea16,     0xbfe9d1b1f5ea80d6,     //  -0.59076,  -0.80685
    0xbfe2dd5bcffb7616,     0xbfe9d91c7007d5a6,     //  -0.58952,  -0.80775
    0xbfe2d333d34e9bb7,     0xbfe9e082edb42472,     //  -0.58828,  -0.80866
    0xbfe2c908ef734e57,     0xbfe9e7e56dcb45bd,     //  -0.58704,  -0.80956
    0xbfe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //   -0.5858,  -0.81046
    0xbfe2b4aa787764c4,     0xbfe9f69e70ac75bc,     //  -0.58455,  -0.81135
    0xbfe2aa76e87aeb58,     0xbfe9fdf4f13149de,     //  -0.58331,  -0.81225
    0xbfe2a040779843fb,     0xbfea05476f967bb5,     //  -0.58206,  -0.81314
    0xbfe2960727629ca8,     0xbfea0c95eabaf937,     //  -0.58081,  -0.81404
    0xbfe28bcaf96d94ba,     0xbfea13e0617e4ec7,     //  -0.57956,  -0.81493
    0xbfe2818bef4d3cba,     0xbfea1b26d2c0a75e,     //  -0.57831,  -0.81581
    0xbfe2774a0a961612,     0xbfea22693d62ccb9,     //  -0.57706,   -0.8167
    0xbfe26d054cdd12df,     0xbfea29a7a0462782,     //  -0.57581,  -0.81758
    0xbfe262bdb7b795a2,     0xbfea30e1fa4cbf81,     //  -0.57455,  -0.81847
    0xbfe258734cbb7110,     0xbfea38184a593bc6,     //   -0.5733,  -0.81935
    0xbfe24e260d7ee7c9,     0xbfea3f4a8f4ee2d2,     //  -0.57204,  -0.82023
    0xbfe243d5fb98ac1f,     0xbfea4678c8119ac8,     //  -0.57078,   -0.8211
    0xbfe23983189fdfd5,     0xbfea4da2f385e997,     //  -0.56952,  -0.82198
    0xbfe22f2d662c13e1,     0xbfea54c91090f524,     //  -0.56826,  -0.82285
    0xbfe224d4e5d5482e,     0xbfea5beb1e188375,     //    -0.567,  -0.82372
    0xbfe21a799933eb58,     0xbfea63091b02fae2,     //  -0.56573,  -0.82459
    0xbfe2101b81e0da78,     0xbfea6a230637623b,     //  -0.56447,  -0.82546
    0xbfe205baa17560d6,     0xbfea7138de9d60f5,     //   -0.5632,  -0.82632
    0xbfe1fb56f98b37b8,     0xbfea784aa31d3f55,     //  -0.56193,  -0.82718
    0xbfe1f0f08bbc861b,     0xbfea7f58529fe69d,     //  -0.56066,  -0.82805
    0xbfe1e68759a3e074,     0xbfea8661ec0ee133,     //  -0.55939,   -0.8289
    0xbfe1dc1b64dc4872,     0xbfea8d676e545ad2,     //  -0.55812,  -0.82976
    0xbfe1d1acaf012cc2,     0xbfea9468d85b20ae,     //  -0.55685,  -0.83062
    0xbfe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //  -0.55557,  -0.83147
    0xbfe1bcc706804467,     0xbfeaa25f5f5aee60,     //  -0.55429,  -0.83232
    0xbfe1b250171373be,     0xbfeaa9547a2cb98e,     //  -0.55302,  -0.83317
    0xbfe1a7d66d0516e6,     0xbfeab045787157ff,     //  -0.55174,  -0.83402
    0xbfe19d5a09f2b9b8,     0xbfeab7325916c0d4,     //  -0.55046,  -0.83486
    0xbfe192daef7a5386,     0xbfeabe1b1b0b8dac,     //  -0.54918,  -0.83571
    0xbfe188591f3a46e5,     0xbfeac4ffbd3efac8,     //  -0.54789,  -0.83655
    0xbfe17dd49ad16161,     0xbfeacbe03ea0e73b,     //  -0.54661,  -0.83739
    0xbfe1734d63dedb49,     0xbfead2bc9e21d511,     //  -0.54532,  -0.83822
    0xbfe168c37c025764,     0xbfead994dab2e979,     //  -0.54404,  -0.83906
    0xbfe15e36e4dbe2bc,     0xbfeae068f345ecef,     //  -0.54275,  -0.83989
    0xbfe153a7a00bf453,     0xbfeae738e6cd4b67,     //  -0.54146,  -0.84073
    0xbfe14915af336ceb,     0xbfeaee04b43c1474,     //  -0.54017,  -0.84155
    0xbfe13e8113f396c1,     0xbfeaf4cc5a85fb73,     //  -0.53888,  -0.84238
    0xbfe133e9cfee254e,     0xbfeafb8fd89f57b6,     //  -0.53759,  -0.84321
    0xbfe1294fe4c5350a,     0xbfeb024f2d7d24a9,     //  -0.53629,  -0.84403
    0xbfe11eb3541b4b22,     0xbfeb090a58150200,     //    -0.535,  -0.84485
    0xbfe114141f935545,     0xbfeb0fc1575d33db,     //   -0.5337,  -0.84567
    0xbfe1097248d0a956,     0xbfeb16742a4ca2f5,     //   -0.5324,  -0.84649
    0xbfe0fecdd1770537,     0xbfeb1d22cfdadcc6,     //   -0.5311,  -0.84731
    0xbfe0f426bb2a8e7d,     0xbfeb23cd470013b4,     //   -0.5298,  -0.84812
    0xbfe0e97d078fd23b,     0xbfeb2a738eb51f33,     //   -0.5285,  -0.84893
    0xbfe0ded0b84bc4b5,     0xbfeb3115a5f37bf4,     //   -0.5272,  -0.84974
    0xbfe0d421cf03c12b,     0xbfeb37b38bb54c09,     //   -0.5259,  -0.85055
    0xbfe0c9704d5d898f,     0xbfeb3e4d3ef55712,     //  -0.52459,  -0.85136
    0xbfe0bebc34ff4646,     0xbfeb44e2beaf0a61,     //  -0.52328,  -0.85216
    0xbfe0b405878f85ec,     0xbfeb4b7409de7925,     //  -0.52198,  -0.85296
    0xbfe0a94c46b53d0b,     0xbfeb52011f805c92,     //  -0.52067,  -0.85376
    0xbfe09e907417c5e1,     0xbfeb5889fe921405,     //  -0.51936,  -0.85456
    0xbfe093d2115ee018,     0xbfeb5f0ea611a532,     //  -0.51804,  -0.85535
    0xbfe089112032b08c,     0xbfeb658f14fdbc47,     //  -0.51673,  -0.85615
    0xbfe07e4da23bc102,     0xbfeb6c0b4a55ac17,     //  -0.51542,  -0.85694
    0xbfe073879922ffed,     0xbfeb728345196e3e,     //   -0.5141,  -0.85773
    0xbfe068bf0691c028,     0xbfeb78f70449a34b,     //  -0.51279,  -0.85852
    0xbfe05df3ec31b8b6,     0xbfeb7f6686e792ea,     //  -0.51147,   -0.8593
    0xbfe053264bad0483,     0xbfeb85d1cbf52c02,     //  -0.51015,  -0.86009
    0xbfe0485626ae221a,     0xbfeb8c38d27504e9,     //  -0.50883,  -0.86087
    0xbfe03d837edff370,     0xbfeb929b996a5b7f,     //  -0.50751,  -0.86165
    0xbfe032ae55edbd95,     0xbfeb98fa1fd9155e,     //  -0.50619,  -0.86242
    0xbfe027d6ad83287e,     0xbfeb9f5464c5bffc,     //  -0.50486,   -0.8632
    0xbfe01cfc874c3eb7,     0xbfeba5aa673590d2,     //  -0.50354,  -0.86397
    0xbfe0121fe4f56d2c,     0xbfebabfc262e6586,     //  -0.50221,  -0.86474
    0xbfe00740c82b82e0,     0xbfebb249a0b6c40d,     //  -0.50089,  -0.86551
    0xbfdff8be6537615e,     0xbfebb892d5d5dad5,     //  -0.49956,  -0.86628
    0xbfdfe2f64be7120f,     0xbfebbed7c49380ea,     //  -0.49823,  -0.86705
    0xbfdfcd2947c1ff57,     0xbfebc5186bf8361d,     //   -0.4969,  -0.86781
    0xbfdfb7575c24d2de,     0xbfebcb54cb0d2327,     //  -0.49557,  -0.86857
    0xbfdfa1808c6cf7e0,     0xbfebd18ce0dc19d6,     //  -0.49423,  -0.86933
    0xbfdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //   -0.4929,  -0.87009
    0xbfdf75c44e26a852,     0xbfebddf02cd2b983,     //  -0.49156,  -0.87084
    0xbfdf5fdee656cda3,     0xbfebe41b611154c1,     //  -0.49023,   -0.8716
    0xbfdf49f4a7e97729,     0xbfebea424837de6d,     //  -0.48889,  -0.87235
    0xbfdf3405963fd068,     0xbfebf064e15377dd,     //  -0.48755,  -0.87309
    0xbfdf1e11b4bbc35c,     0xbfebf6832b71ec5b,     //  -0.48621,  -0.87384
    0xbfdf081906bff7fd,     0xbfebfc9d25a1b147,     //  -0.48487,  -0.87459
    0xbfdef21b8fafd3b5,     0xbfec02b2cef1e641,     //  -0.48353,  -0.87533
    0xbfdedc1952ef78d5,     0xbfec08c426725549,     //  -0.48218,  -0.87607
    0xbfdec61253e3c61b,     0xbfec0ed12b3372e9,     //  -0.48084,  -0.87681
    0xbfdeb00695f25620,     0xbfec14d9dc465e58,     //  -0.47949,  -0.87755
    0xbfde99f61c817eda,     0xbfec1ade38bce19b,     //  -0.47815,  -0.87828
    0xbfde83e0eaf85113,     0xbfec20de3fa971b0,     //   -0.4768,  -0.87901
    0xbfde6dc704be97e2,     0xbfec26d9f01f2eaf,     //  -0.47545,  -0.87974
    0xbfde57a86d3cd824,     0xbfec2cd14931e3f1,     //   -0.4741,  -0.88047
    0xbfde418527dc4ffa,     0xbfec32c449f60831,     //  -0.47275,   -0.8812
    0xbfde2b5d3806f63b,     0xbfec38b2f180bdb1,     //   -0.4714,  -0.88192
    0xbfde1530a12779f4,     0xbfec3e9d3ee7d262,     //  -0.47004,  -0.88264
    0xbfddfeff66a941de,     0xbfec44833141c004,     //  -0.46869,  -0.88336
    0xbfdde8c98bf86bd6,     0xbfec4a64c7a5ac4c,     //  -0.46733,  -0.88408
    0xbfddd28f1481cc58,     0xbfec5042012b6907,     //  -0.46598,   -0.8848
    0xbfddbc5003b2edf8,     0xbfec561adceb743e,     //  -0.46462,  -0.88551
    0xbfdda60c5cfa10d8,     0xbfec5bef59fef85a,     //  -0.46326,  -0.88622
    0xbfdd8fc423c62a25,     0xbfec61bf777fcc48,     //   -0.4619,  -0.88693
    0xbfdd79775b86e389,     0xbfec678b3488739b,     //  -0.46054,  -0.88764
    0xbfdd632607ac9aa9,     0xbfec6d5290341eb2,     //  -0.45918,  -0.88835
    0xbfdd4cd02ba8609c,     0xbfec7315899eaad7,     //  -0.45781,  -0.88905
    0xbfdd3675caebf962,     0xbfec78d41fe4a267,     //  -0.45645,  -0.88975
    0xbfdd2016e8e9db5b,     0xbfec7e8e52233cf3,     //  -0.45508,  -0.89045
    0xbfdd09b389152ec1,     0xbfec84441f785f61,     //  -0.45372,  -0.89115
    0xbfdcf34baee1cd21,     0xbfec89f587029c13,     //  -0.45235,  -0.89184
    0xbfdcdcdf5dc440ce,     0xbfec8fa287e13305,     //  -0.45098,  -0.89253
    0xbfdcc66e9931c45d,     0xbfec954b213411f5,     //  -0.44961,  -0.89322
    0xbfdcaff964a0421d,     0xbfec9aef521bd480,     //  -0.44824,  -0.89391
    0xbfdc997fc3865388,     0xbfeca08f19b9c449,     //  -0.44687,   -0.8946
    0xbfdc8301b95b40c2,     0xbfeca62a772fd919,     //   -0.4455,  -0.89528
    0xbfdc6c7f4997000a,     0xbfecabc169a0b901,     //  -0.44412,  -0.89597
    0xbfdc55f877b23537,     0xbfecb153f02fb87d,     //  -0.44275,  -0.89665
    0xbfdc3f6d47263129,     0xbfecb6e20a00da99,     //  -0.44137,  -0.89732
    0xbfdc28ddbb6cf145,     0xbfecbc6bb638d10b,     //  -0.43999,    -0.898
    0xbfdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     //  -0.43862,  -0.89867
    0xbfdbfbb1a05e0edc,     0xbfecc771c2736c09,     //  -0.43724,  -0.89935
    0xbfdbe51517ffc0d9,     0xbfecccee20c2de9f,     //  -0.43586,  -0.90002
    0xbfdbce744262deee,     0xbfecd2660e12c1e6,     //  -0.43448,  -0.90068
    0xbfdbb7cf2304bd01,     0xbfecd7d9898b32f6,     //  -0.43309,  -0.90135
    0xbfdba125bd63583e,     0xbfecdd489254fe65,     //  -0.43171,  -0.90201
    0xbfdb8a7814fd5693,     0xbfece2b32799a060,     //  -0.43033,  -0.90267
    0xbfdb73c62d520624,     0xbfece819488344ce,     //  -0.42894,  -0.90333
    0xbfdb5d1009e15cc0,     0xbfeced7af43cc773,     //  -0.42756,  -0.90399
    0xbfdb4655ae2bf757,     0xbfecf2d829f1b40e,     //  -0.42617,  -0.90464
    0xbfdb2f971db31972,     0xbfecf830e8ce467b,     //  -0.42478,   -0.9053
    0xbfdb18d45bf8aca6,     0xbfecfd852fff6ad4,     //  -0.42339,  -0.90595
    0xbfdb020d6c7f4009,     0xbfed02d4feb2bd92,     //    -0.422,   -0.9066
    0xbfdaeb4252ca07ab,     0xbfed082054168bac,     //  -0.42061,  -0.90724
    0xbfdad473125cdc08,     0xbfed0d672f59d2b9,     //  -0.41922,  -0.90789
    0xbfdabd9faebc3980,     0xbfed12a98fac410c,     //  -0.41782,  -0.90853
    0xbfdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     //  -0.41643,  -0.90917
    0xbfda8fec8bf5b166,     0xbfed1d20dc40c15c,     //  -0.41503,  -0.90981
    0xbfda790cd3dbf31a,     0xbfed2255c6e5a4e1,     //  -0.41364,  -0.91044
    0xbfda622906a70b63,     0xbfed2786335f52fc,     //  -0.41224,  -0.91107
    0xbfda4b4127dea1e4,     0xbfed2cb220e0ef9f,     //  -0.41084,  -0.91171
    0xbfda34553b0afee5,     0xbfed31d98e9e503a,     //  -0.40944,  -0.91234
    0xbfda1d6543b50ac0,     0xbfed36fc7bcbfbdc,     //  -0.40804,  -0.91296
    0xbfda067145664d57,     0xbfed3c1ae79f2b4e,     //  -0.40664,  -0.91359
    0xbfd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //  -0.40524,  -0.91421
    0xbfd9d87d4207b0ab,     0xbfed464a380e7242,     //  -0.40384,  -0.91483
    0xbfd9c17d440df9f2,     0xbfed4b5b1b187524,     //  -0.40243,  -0.91545
    0xbfd9aa794d47c9ee,     0xbfed506779a3d2d9,     //  -0.40103,  -0.91606
    0xbfd993716141bdfe,     0xbfed556f52e93eb1,     //  -0.39962,  -0.91668
    0xbfd97c6583890fc2,     0xbfed5a72a6221e73,     //  -0.39822,  -0.91729
    0xbfd96555b7ab948f,     0xbfed5f7172888a7f,     //  -0.39681,   -0.9179
    0xbfd94e420137bce3,     0xbfed646bb7574de5,     //   -0.3954,  -0.91851
    0xbfd9372a63bc93d7,     0xbfed696173c9e68b,     //  -0.39399,  -0.91911
    0xbfd9200ee2c9be97,     0xbfed6e52a71c8547,     //  -0.39258,  -0.91972
    0xbfd908ef81ef7bd1,     0xbfed733f508c0dff,     //  -0.39117,  -0.92032
    0xbfd8f1cc44bea329,     0xbfed78276f5617c6,     //  -0.38976,  -0.92092
    0xbfd8daa52ec8a4af,     0xbfed7d0b02b8ecf9,     //  -0.38835,  -0.92151
    0xbfd8c37a439f884f,     0xbfed81ea09f38b63,     //  -0.38693,  -0.92211
    0xbfd8ac4b86d5ed44,     0xbfed86c48445a450,     //  -0.38552,   -0.9227
    0xbfd89518fbff098e,     0xbfed8b9a70ef9cb4,     //   -0.3841,  -0.92329
    0xbfd87de2a6aea963,     0xbfed906bcf328d46,     //  -0.38268,  -0.92388
    0xbfd866a88a792ea0,     0xbfed95389e50429b,     //  -0.38127,  -0.92447
    0xbfd84f6aaaf3903f,     0xbfed9a00dd8b3d46,     //  -0.37985,  -0.92505
    0xbfd838290bb359c8,     0xbfed9ec48c26b1f3,     //  -0.37843,  -0.92563
    0xbfd820e3b04eaac4,     0xbfeda383a9668988,     //  -0.37701,  -0.92621
    0xbfd8099a9c5c362d,     0xbfeda83e348f613b,     //  -0.37559,  -0.92679
    0xbfd7f24dd37341e3,     0xbfedacf42ce68ab9,     //  -0.37416,  -0.92736
    0xbfd7dafd592ba621,     0xbfedb1a591b20c38,     //  -0.37274,  -0.92794
    0xbfd7c3a9311dcce7,     0xbfedb6526238a09b,     //  -0.37132,  -0.92851
    0xbfd7ac515ee2b172,     0xbfedbafa9dc1b78d,     //  -0.36989,  -0.92907
    0xbfd794f5e613dfae,     0xbfedbf9e4395759a,     //  -0.36847,  -0.92964
    0xbfd77d96ca4b73a6,     0xbfedc43d52fcb453,     //  -0.36704,  -0.93021
    0xbfd766340f2418f6,     0xbfedc8d7cb410260,     //  -0.36561,  -0.93077
    0xbfd74ecdb8390a3e,     0xbfedcd6dabaca3a5,     //  -0.36418,  -0.93133
    0xbfd73763c9261092,     0xbfedd1fef38a915a,     //  -0.36276,  -0.93188
    0xbfd71ff6458782ec,     0xbfedd68ba2267a25,     //  -0.36133,  -0.93244
    0xbfd7088530fa459e,     0xbfeddb13b6ccc23d,     //   -0.3599,  -0.93299
    0xbfd6f1108f1bc9c5,     0xbfeddf9730ca837b,     //  -0.35846,  -0.93354
    0xbfd6d998638a0cb5,     0xbfede4160f6d8d81,     //  -0.35703,  -0.93409
    0xbfd6c21cb1e39771,     0xbfede890520465ce,     //   -0.3556,  -0.93464
    0xbfd6aa9d7dc77e16,     0xbfeded05f7de47da,     //  -0.35416,  -0.93518
    0xbfd6931acad55f51,     0xbfedf177004b2534,     //  -0.35273,  -0.93573
    0xbfd67b949cad63ca,     0xbfedf5e36a9ba59c,     //  -0.35129,  -0.93627
    0xbfd6640af6f03d9e,     0xbfedfa4b3621271d,     //  -0.34986,   -0.9368
    0xbfd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     //  -0.34842,  -0.93734
    0xbfd634ed533be58e,     0xbfee030cee1435b8,     //  -0.34698,  -0.93787
    0xbfd61d595c88c203,     0xbfee0766d9280f54,     //  -0.34554,   -0.9384
    0xbfd605c1fcc88f63,     0xbfee0bbc22bd8349,     //   -0.3441,  -0.93893
    0xbfd5ee27379ea693,     0xbfee100cca2980ac,     //  -0.34266,  -0.93946
    0xbfd5d68910aee686,     0xbfee1458cec1ad83,     //  -0.34122,  -0.93998
    0xbfd5bee78b9db3b6,     0xbfee18a02fdc66d9,     //  -0.33978,  -0.94051
    0xbfd5a742ac0ff78d,     0xbfee1ce2ecd0c0d8,     //  -0.33833,  -0.94103
    0xbfd58f9a75ab1fdd,     0xbfee212104f686e5,     //  -0.33689,  -0.94154
    0xbfd577eeec151e47,     0xbfee255a77a63bb8,     //  -0.33545,  -0.94206
    0xbfd5604012f467b4,     0xbfee298f4439197a,     //    -0.334,  -0.94257
    0xbfd5488dedeff3be,     0xbfee2dbf6a0911d9,     //  -0.33255,  -0.94308
    0xbfd530d880af3c24,     0xbfee31eae870ce25,     //  -0.33111,  -0.94359
    0xbfd5191fceda3c35,     0xbfee3611becbaf69,     //  -0.32966,   -0.9441
    0xbfd50163dc197047,     0xbfee3a33ec75ce85,     //  -0.32821,   -0.9446
    0xbfd4e9a4ac15d520,     0xbfee3e5170cbfc46,     //  -0.32676,  -0.94511
    0xbfd4d1e24278e76a,     0xbfee426a4b2bc17e,     //  -0.32531,  -0.94561
    0xbfd4ba1ca2eca31c,     0xbfee467e7af35f23,     //  -0.32386,  -0.94611
    0xbfd4a253d11b82f3,     0xbfee4a8dff81ce5e,     //  -0.32241,   -0.9466
    0xbfd48a87d0b07fd7,     0xbfee4e98d836c0af,     //  -0.32096,  -0.94709
    0xbfd472b8a5571054,     0xbfee529f04729ffc,     //   -0.3195,  -0.94759
    0xbfd45ae652bb2800,     0xbfee56a083968eb1,     //  -0.31805,  -0.94807
    0xbfd44310dc8936f0,     0xbfee5a9d550467d3,     //  -0.31659,  -0.94856
    0xbfd42b38466e2928,     0xbfee5e95781ebf1c,     //  -0.31514,  -0.94905
    0xbfd4135c94176602,     0xbfee6288ec48e112,     //  -0.31368,  -0.94953
    0xbfd3fb7dc932cfa4,     0xbfee6677b0e6d31e,     //  -0.31222,  -0.95001
    0xbfd3e39be96ec271,     0xbfee6a61c55d53a7,     //  -0.31077,  -0.95049
    0xbfd3cbb6f87a146e,     0xbfee6e472911da27,     //  -0.30931,  -0.95096
    0xbfd3b3cefa0414b7,     0xbfee7227db6a9744,     //  -0.30785,  -0.95144
    0xbfd39be3f1bc8aef,     0xbfee7603dbce74e9,     //  -0.30639,  -0.95191
    0xbfd383f5e353b6aa,     0xbfee79db29a5165a,     //  -0.30493,  -0.95238
    0xbfd36c04d27a4edf,     0xbfee7dadc456d850,     //  -0.30347,  -0.95284
    0xbfd35410c2e18152,     0xbfee817bab4cd10d,     //  -0.30201,  -0.95331
    0xbfd33c19b83af207,     0xbfee8544ddf0d075,     //  -0.30054,  -0.95377
    0xbfd3241fb638baaf,     0xbfee89095bad6025,     //  -0.29908,  -0.95423
    0xbfd30c22c08d6a13,     0xbfee8cc923edc388,     //  -0.29762,  -0.95469
    0xbfd2f422daec0386,     0xbfee9084361df7f3,     //  -0.29615,  -0.95514
    0xbfd2dc200907fe51,     0xbfee943a91aab4b4,     //  -0.29469,  -0.95559
    0xbfd2c41a4e954520,     0xbfee97ec36016b30,     //  -0.29322,  -0.95605
    0xbfd2ac11af483572,     0xbfee9b99229046f8,     //  -0.29175,  -0.95649
    0xbfd294062ed59f05,     0xbfee9f4156c62dda,     //  -0.29028,  -0.95694
    0xbfd27bf7d0f2c346,     0xbfeea2e4d212c000,     //  -0.28882,  -0.95738
    0xbfd263e6995554ba,     0xbfeea68393e65800,     //  -0.28735,  -0.95783
    0xbfd24bd28bb37672,     0xbfeeaa1d9bb20af3,     //  -0.28588,  -0.95827
    0xbfd233bbabc3bb72,     0xbfeeadb2e8e7a88e,     //  -0.28441,   -0.9587
    0xbfd21ba1fd3d2623,     0xbfeeb1437af9bb34,     //  -0.28294,  -0.95914
    0xbfd2038583d727bd,     0xbfeeb4cf515b8811,     //  -0.28146,  -0.95957
    0xbfd1eb6643499fbb,     0xbfeeb8566b810f2a,     //  -0.27999,     -0.96
    0xbfd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     //  -0.27852,  -0.96043
    0xbfd1bb1f7b999480,     0xbfeebf5668eaf2ef,     //  -0.27705,  -0.96086
    0xbfd1a2f7fbe8f243,     0xbfeec2cf4b1af6b2,     //  -0.27557,  -0.96128
    0xbfd18acdc3f4873a,     0xbfeec6436ee60309,     //   -0.2741,   -0.9617
    0xbfd172a0d7765177,     0xbfeec9b2d3c3bf84,     //  -0.27262,  -0.96212
    0xbfd15a713a28b9d9,     0xbfeecd1d792c8f10,     //  -0.27115,  -0.96254
    0xbfd1423eefc69378,     0xbfeed0835e999009,     //  -0.26967,  -0.96295
    0xbfd12a09fc0b1b12,     0xbfeed3e483849c51,     //  -0.26819,  -0.96337
    0xbfd111d262b1f677,     0xbfeed740e7684963,     //  -0.26671,  -0.96378
    0xbfd0f998277733f7,     0xbfeeda9889bfe86a,     //  -0.26523,  -0.96418
    0xbfd0e15b4e1749cd,     0xbfeeddeb6a078651,     //  -0.26375,  -0.96459
    0xbfd0c91bda4f158d,     0xbfeee13987bbebdc,     //  -0.26227,  -0.96499
    0xbfd0b0d9cfdbdb90,     0xbfeee482e25a9dbc,     //  -0.26079,  -0.96539
    0xbfd09895327b465e,     0xbfeee7c77961dc9e,     //  -0.25931,  -0.96579
    0xbfd0804e05eb661e,     0xbfeeeb074c50a544,     //  -0.25783,  -0.96619
    0xbfd068044deab002,     0xbfeeee425aa6b09a,     //  -0.25635,  -0.96658
    0xbfd04fb80e37fdae,     0xbfeef178a3e473c2,     //  -0.25487,  -0.96698
    0xbfd037694a928cac,     0xbfeef4aa278b2032,     //  -0.25338,  -0.96737
    0xbfd01f1806b9fdd2,     0xbfeef7d6e51ca3c0,     //   -0.2519,  -0.96775
    0xbfd006c4466e54af,     0xbfeefafedc1ba8b7,     //  -0.25041,  -0.96814
    0xbfcfdcdc1adfedf8,     0xbfeefe220c0b95ec,     //  -0.24893,  -0.96852
    0xbfcfac2abeff57ff,     0xbfef014074708ed3,     //  -0.24744,   -0.9689
    0xbfcf7b7480bd3801,     0xbfef045a14cf738c,     //  -0.24596,  -0.96928
    0xbfcf4ab9679c9f5c,     0xbfef076eecade0fa,     //  -0.24447,  -0.96966
    0xbfcf19f97b215f1a,     0xbfef0a7efb9230d7,     //  -0.24298,  -0.97003
    0xbfcee934c2d006c7,     0xbfef0d8a410379c5,     //  -0.24149,   -0.9704
    0xbfceb86b462de348,     0xbfef1090bc898f5f,     //     -0.24,  -0.97077
    0xbfce879d0cc0fdaf,     0xbfef13926dad024e,     //  -0.23851,  -0.97114
    0xbfce56ca1e101a1b,     0xbfef168f53f7205d,     //  -0.23702,   -0.9715
    0xbfce25f281a2b684,     0xbfef19876ef1f486,     //  -0.23553,  -0.97187
    0xbfcdf5163f01099a,     0xbfef1c7abe284708,     //  -0.23404,  -0.97223
    0xbfcdc4355db40195,     0xbfef1f6941259d7a,     //  -0.23255,  -0.97258
    0xbfcd934fe5454311,     0xbfef2252f7763ada,     //  -0.23106,  -0.97294
    0xbfcd6265dd3f27e3,     0xbfef2537e0a71f9f,     //  -0.22957,  -0.97329
    0xbfcd31774d2cbdee,     0xbfef2817fc4609ce,     //  -0.22807,  -0.97364
    0xbfcd00843c99c5f9,     0xbfef2af349e17507,     //  -0.22658,  -0.97399
    0xbfcccf8cb312b286,     0xbfef2dc9c9089a9d,     //  -0.22508,  -0.97434
    0xbfcc9e90b824a6a9,     0xbfef309b794b719f,     //  -0.22359,  -0.97468
    0xbfcc6d90535d74dc,     0xbfef33685a3aaef0,     //  -0.22209,  -0.97503
    0xbfcc3c8b8c4b9dd7,     0xbfef36306b67c556,     //   -0.2206,  -0.97536
    0xbfcc0b826a7e4f63,     0xbfef38f3ac64e589,     //   -0.2191,   -0.9757
    0xbfcbda74f5856330,     0xbfef3bb21cc4fe47,     //   -0.2176,  -0.97604
    0xbfcba96334f15dad,     0xbfef3e6bbc1bbc65,     //  -0.21611,  -0.97637
    0xbfcb784d30536cda,     0xbfef412089fd8adc,     //  -0.21461,   -0.9767
    0xbfcb4732ef3d6722,     0xbfef43d085ff92dd,     //  -0.21311,  -0.97703
    0xbfcb16147941ca2a,     0xbfef467bafb7bbe0,     //  -0.21161,  -0.97735
    0xbfcae4f1d5f3b9ab,     0xbfef492206bcabb4,     //  -0.21011,  -0.97768
    0xbfcab3cb0ce6fe44,     0xbfef4bc38aa5c694,     //  -0.20861,    -0.978
    0xbfca82a025b00451,     0xbfef4e603b0b2f2d,     //  -0.20711,  -0.97832
    0xbfca517127e3dabc,     0xbfef50f81785c6b9,     //  -0.20561,  -0.97863
    0xbfca203e1b1831da,     0xbfef538b1faf2d07,     //  -0.20411,  -0.97895
    0xbfc9ef0706e35a35,     0xbfef56195321c090,     //  -0.20261,  -0.97926
    0xbfc9bdcbf2dc4366,     0xbfef58a2b1789e84,     //   -0.2011,  -0.97957
    0xbfc98c8ce69a7aec,     0xbfef5b273a4fa2d9,     //   -0.1996,  -0.97988
    0xbfc95b49e9b62af9,     0xbfef5da6ed43685d,     //   -0.1981,  -0.98018
    0xbfc92a0303c8194f,     0xbfef6021c9f148c2,     //  -0.19659,  -0.98048
    0xbfc8f8b83c69a60a,     0xbfef6297cff75cb0,     //  -0.19509,  -0.98079
    0xbfc8c7699b34ca7e,     0xbfef6508fef47bd5,     //  -0.19359,  -0.98108
    0xbfc8961727c41804,     0xbfef677556883cee,     //  -0.19208,  -0.98138
    0xbfc864c0e9b2b6cf,     0xbfef69dcd652f5de,     //  -0.19057,  -0.98167
    0xbfc83366e89c64c5,     0xbfef6c3f7df5bbb7,     //  -0.18907,  -0.98196
    0xbfc802092c1d744b,     0xbfef6e9d4d1262ca,     //  -0.18756,  -0.98225
    0xbfc7d0a7bbd2cb1b,     0xbfef70f6434b7eb7,     //  -0.18606,  -0.98254
    0xbfc79f429f59e11d,     0xbfef734a60446279,     //  -0.18455,  -0.98282
    0xbfc76dd9de50bf31,     0xbfef7599a3a12077,     //  -0.18304,  -0.98311
    0xbfc73c6d8055fe0a,     0xbfef77e40d068a90,     //  -0.18153,  -0.98339
    0xbfc70afd8d08c4ff,     0xbfef7a299c1a322a,     //  -0.18002,  -0.98366
    0xbfc6d98a0c08c8da,     0xbfef7c6a50826840,     //  -0.17851,  -0.98394
    0xbfc6a81304f64ab2,     0xbfef7ea629e63d6e,     //    -0.177,  -0.98421
    0xbfc676987f7216b8,     0xbfef80dd27ed8204,     //  -0.17549,  -0.98448
    0xbfc6451a831d830d,     0xbfef830f4a40c60c,     //  -0.17398,  -0.98475
    0xbfc61399179a6e94,     0xbfef853c9089595e,     //  -0.17247,  -0.98501
    0xbfc5e214448b3fc6,     0xbfef8764fa714ba9,     //  -0.17096,  -0.98528
    0xbfc5b08c1192e381,     0xbfef898887a36c84,     //  -0.16945,  -0.98554
    0xbfc57f008654cbde,     0xbfef8ba737cb4b78,     //  -0.16794,   -0.9858
    0xbfc54d71aa74ef02,     0xbfef8dc10a95380d,     //  -0.16643,  -0.98605
    0xbfc51bdf8597c5f2,     0xbfef8fd5ffae41db,     //  -0.16491,  -0.98631
    0xbfc4ea4a1f624b61,     0xbfef91e616c43891,     //   -0.1634,  -0.98656
    0xbfc4b8b17f79fa88,     0xbfef93f14f85ac08,     //  -0.16189,  -0.98681
    0xbfc48715ad84cdf5,     0xbfef95f7a9a1ec47,     //  -0.16037,  -0.98706
    0xbfc45576b1293e5a,     0xbfef97f924c9099b,     //  -0.15886,   -0.9873
    0xbfc423d4920e4166,     0xbfef99f5c0abd496,     //  -0.15734,  -0.98754
    0xbfc3f22f57db4893,     0xbfef9bed7cfbde29,     //  -0.15583,  -0.98778
    0xbfc3c0870a383ff6,     0xbfef9de0596b77a3,     //  -0.15431,  -0.98802
    0xbfc38edbb0cd8d14,     0xbfef9fce55adb2c8,     //   -0.1528,  -0.98826
    0xbfc35d2d53440db2,     0xbfefa1b7717661d5,     //  -0.15128,  -0.98849
    0xbfc32b7bf94516a7,     0xbfefa39bac7a1791,     //  -0.14976,  -0.98872
    0xbfc2f9c7aa7a72af,     0xbfefa57b066e2754,     //  -0.14825,  -0.98895
    0xbfc2c8106e8e613a,     0xbfefa7557f08a517,     //  -0.14673,  -0.98918
    0xbfc296564d2b953e,     0xbfefa92b1600657c,     //  -0.14521,   -0.9894
    0xbfc264994dfd340a,     0xbfefaafbcb0cfddc,     //   -0.1437,  -0.98962
    0xbfc232d978aed413,     0xbfefacc79de6c44f,     //  -0.14218,  -0.98984
    0xbfc20116d4ec7bce,     0xbfefae8e8e46cfbb,     //  -0.14066,  -0.99006
    0xbfc1cf516a62a077,     0xbfefb0509be6f7db,     //  -0.13914,  -0.99027
    0xbfc19d8940be24e7,     0xbfefb20dc681d54d,     //  -0.13762,  -0.99049
    0xbfc16bbe5fac5865,     0xbfefb3c60dd2c199,     //   -0.1361,   -0.9907
    0xbfc139f0cedaf576,     0xbfefb5797195d741,     //  -0.13458,   -0.9909
    0xbfc1082095f820b0,     0xbfefb727f187f1c7,     //  -0.13306,  -0.99111
    0xbfc0d64dbcb26786,     0xbfefb8d18d66adb7,     //  -0.13154,  -0.99131
    0xbfc0a4784ab8bf1d,     0xbfefba7644f068b5,     //  -0.13002,  -0.99151
    0xbfc072a047ba831d,     0xbfefbc1617e44186,     //   -0.1285,  -0.99171
    0xbfc040c5bb67747e,     0xbfefbdb106021816,     //  -0.12698,  -0.99191
    0xbfc00ee8ad6fb85b,     0xbfefbf470f0a8d88,     //  -0.12545,   -0.9921
    0xbfbfba124b07ad85,     0xbfefc0d832bf043a,     //  -0.12393,  -0.99229
    0xbfbf564e56a9730e,     0xbfefc26470e19fd3,     //  -0.12241,  -0.99248
    0xbfbef2858d27561b,     0xbfefc3ebc935454c,     //  -0.12089,  -0.99267
    0xbfbe8eb7fde4aa3e,     0xbfefc56e3b7d9af6,     //  -0.11937,  -0.99285
    0xbfbe2ae5b8457f77,     0xbfefc6ebc77f0887,     //  -0.11784,  -0.99303
    0xbfbdc70ecbae9fc8,     0xbfefc8646cfeb721,     //  -0.11632,  -0.99321
    0xbfbd633347858ce4,     0xbfefc9d82bc2915e,     //  -0.11479,  -0.99339
    0xbfbcff533b307dc1,     0xbfefcb4703914354,     //  -0.11327,  -0.99356
    0xbfbc9b6eb6165c42,     0xbfefccb0f4323aa3,     //  -0.11175,  -0.99374
    0xbfbc3785c79ec2d5,     0xbfefce15fd6da67b,     //  -0.11022,  -0.99391
    0xbfbbd3987f31fa0e,     0xbfefcf761f0c77a3,     //   -0.1087,  -0.99407
    0xbfbb6fa6ec38f64c,     0xbfefd0d158d86087,     //  -0.10717,  -0.99424
    0xbfbb0bb11e1d5559,     0xbfefd227aa9bd53b,     //  -0.10565,   -0.9944
    0xbfbaa7b724495c04,     0xbfefd37914220b84,     //  -0.10412,  -0.99456
    0xbfba43b90e27f3c4,     0xbfefd4c59536fae4,     //   -0.1026,  -0.99472
    0xbfb9dfb6eb24a85c,     0xbfefd60d2da75c9e,     //  -0.10107,  -0.99488
    0xbfb97bb0caaba56f,     0xbfefd74fdd40abbf,     // -0.099544,  -0.99503
    0xbfb917a6bc29b42c,     0xbfefd88da3d12526,     // -0.098017,  -0.99518
    0xbfb8b398cf0c38e0,     0xbfefd9c68127c78c,     //  -0.09649,  -0.99533
    0xbfb84f8712c130a0,     0xbfefdafa7514538c,     // -0.094963,  -0.99548
    0xbfb7eb7196b72ee4,     0xbfefdc297f674ba9,     // -0.093436,  -0.99563
    0xbfb787586a5d5b21,     0xbfefdd539ff1f456,     // -0.091909,  -0.99577
    0xbfb7233b9d236e71,     0xbfefde78d68653fd,     // -0.090381,  -0.99591
    0xbfb6bf1b3e79b129,     0xbfefdf9922f73307,     // -0.088854,  -0.99604
    0xbfb65af75dd0f87b,     0xbfefe0b485181be3,     // -0.087326,  -0.99618
    0xbfb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     // -0.085797,  -0.99631
    0xbfb592a554489bc8,     0xbfefe2dc89bbff08,     // -0.084269,  -0.99644
    0xbfb52e774a4d4d0a,     0xbfefe3e92be9d886,     //  -0.08274,  -0.99657
    0xbfb4ca45fc1ba8b6,     0xbfefe4f0e31d7a4a,     // -0.081211,   -0.9967
    0xbfb4661179272096,     0xbfefe5f3af2e3940,     // -0.079682,  -0.99682
    0xbfb401d9d0e3a507,     0xbfefe6f18ff42c84,     // -0.078153,  -0.99694
    0xbfb39d9f12c5a299,     0xbfefe7ea85482d60,     // -0.076624,  -0.99706
    0xbfb339614e41ffa5,     0xbfefe8de8f03d75c,     // -0.075094,  -0.99718
    0xbfb2d52092ce19f6,     0xbfefe9cdad01883a,     // -0.073565,  -0.99729
    0xbfb270dcefdfc45b,     0xbfefeab7df1c6005,     // -0.072035,   -0.9974
    0xbfb20c9674ed444c,     0xbfefeb9d2530410f,     // -0.070505,  -0.99751
    0xbfb1a84d316d4f8a,     0xbfefec7d7f19cffc,     // -0.068974,  -0.99762
    0xbfb1440134d709b2,     0xbfefed58ecb673c4,     // -0.067444,  -0.99772
    0xbfb0dfb28ea201e6,     0xbfefee2f6de455ba,     // -0.065913,  -0.99783
    0xbfb07b614e463064,     0xbfefef0102826191,     // -0.064383,  -0.99793
    0xbfb0170d833bf421,     0xbfefefcdaa704562,     // -0.062852,  -0.99802
    0xbfaf656e79f820e0,     0xbfeff095658e71ad,     // -0.061321,  -0.99812
    0xbfae9cbd15ff5527,     0xbfeff15833be1965,     //  -0.05979,  -0.99821
    0xbfadd406f9808ec8,     0xbfeff21614e131ed,     // -0.058258,   -0.9983
    0xbfad0b4c436f91d0,     0xbfeff2cf08da7321,     // -0.056727,  -0.99839
    0xbfac428d12c0d7e3,     0xbfeff3830f8d575c,     // -0.055195,  -0.99848
    0xbfab79c986698b78,     0xbfeff43228de1b77,     // -0.053664,  -0.99856
    0xbfaab101bd5f8317,     0xbfeff4dc54b1bed3,     // -0.052132,  -0.99864
    0xbfa9e835d6993c87,     0xbfeff58192ee0358,     //   -0.0506,  -0.99872
    0xbfa91f65f10dd814,     0xbfeff621e3796d7e,     // -0.049068,   -0.9988
    0xbfa856922bb513c1,     0xbfeff6bd463b444d,     // -0.047535,  -0.99887
    0xbfa78dbaa5874685,     0xbfeff753bb1b9164,     // -0.046003,  -0.99894
    0xbfa6c4df7d7d5b84,     0xbfeff7e5420320f9,     // -0.044471,  -0.99901
    0xbfa5fc00d290cd43,     0xbfeff871dadb81df,     // -0.042938,  -0.99908
    0xbfa5331ec3bba0eb,     0xbfeff8f9858f058b,     // -0.041406,  -0.99914
    0xbfa46a396ff86179,     0xbfeff97c4208c014,     // -0.039873,   -0.9992
    0xbfa3a150f6421afc,     0xbfeff9fa10348837,     //  -0.03834,  -0.99926
    0xbfa2d865759455cd,     0xbfeffa72effef75d,     // -0.036807,  -0.99932
    0xbfa20f770ceb11c6,     0xbfeffae6e1556998,     // -0.035274,  -0.99938
    0xbfa14685db42c17e,     0xbfeffb55e425fdae,     // -0.033741,  -0.99943
    0xbfa07d91ff984580,     0xbfeffbbff85f9515,     // -0.032208,  -0.99948
    0xbf9f693731d1cf01,     0xbfeffc251df1d3f8,     // -0.030675,  -0.99953
    0xbf9dd7458c64ab39,     0xbfeffc8554cd213a,     // -0.029142,  -0.99958
    0xbf9c454f4ce53b1c,     0xbfeffce09ce2a679,     // -0.027608,  -0.99962
    0xbf9ab354b1504fca,     0xbfeffd36f624500c,     // -0.026075,  -0.99966
    0xbf992155f7a3667e,     0xbfeffd886084cd0d,     // -0.024541,   -0.9997
    0xbf978f535ddc9f03,     0xbfeffdd4dbf78f52,     // -0.023008,  -0.99974
    0xbf95fd4d21fab226,     0xbfeffe1c6870cb77,     // -0.021474,  -0.99977
    0xbf946b4381fce81c,     0xbfeffe5f05e578db,     //  -0.01994,   -0.9998
    0xbf92d936bbe30efd,     0xbfeffe9cb44b51a1,     // -0.018407,  -0.99983
    0xbf9147270dad7132,     0xbfeffed57398d2b7,     // -0.016873,  -0.99986
    0xbf8f6a296ab997ca,     0xbfefff0943c53bd1,     // -0.015339,  -0.99988
    0xbf8c45ffe1e48ad9,     0xbfefff3824c88f6f,     // -0.013805,   -0.9999
    0xbf8921d1fcdec784,     0xbfefff62169b92db,     // -0.012272,  -0.99992
    0xbf85fda037ac05e0,     0xbfefff871937ce2f,     // -0.010738,  -0.99994
    0xbf82d96b0e509703,     0xbfefffa72c978c4f,     //-0.0092038,  -0.99996
    0xbf7f6a65f9a2a3c5,     0xbfefffc250b5daef,     //-0.0076698,  -0.99997
    0xbf7921f0fe670071,     0xbfefffd8858e8a92,     //-0.0061359,  -0.99998
    0xbf72d97822f996bc,     0xbfefffe9cb1e2e8d,     //-0.0046019,  -0.99999
    0xbf6921f8becca4ba,     0xbfeffff621621d02,     // -0.003068,        -1
    0xbf5921faaee6472d,     0xbfeffffd88586ee6,     // -0.001534,        -1
    0x0000000000000000,     0xbff0000000000000,     //         0,        -1
    0x3f5921faaee6472d,     0xbfeffffd88586ee6,     //  0.001534,        -1
    0x3f6921f8becca4ba,     0xbfeffff621621d02,     //  0.003068,        -1
    0x3f72d97822f996bc,     0xbfefffe9cb1e2e8d,     // 0.0046019,  -0.99999
    0x3f7921f0fe670071,     0xbfefffd8858e8a92,     // 0.0061359,  -0.99998
    0x3f7f6a65f9a2a3c5,     0xbfefffc250b5daef,     // 0.0076698,  -0.99997
    0x3f82d96b0e509703,     0xbfefffa72c978c4f,     // 0.0092038,  -0.99996
    0x3f85fda037ac05e0,     0xbfefff871937ce2f,     //  0.010738,  -0.99994
    0x3f8921d1fcdec784,     0xbfefff62169b92db,     //  0.012272,  -0.99992
    0x3f8c45ffe1e48ad9,     0xbfefff3824c88f6f,     //  0.013805,   -0.9999
    0x3f8f6a296ab997ca,     0xbfefff0943c53bd1,     //  0.015339,  -0.99988
    0x3f9147270dad7132,     0xbfeffed57398d2b7,     //  0.016873,  -0.99986
    0x3f92d936bbe30efd,     0xbfeffe9cb44b51a1,     //  0.018407,  -0.99983
    0x3f946b4381fce81c,     0xbfeffe5f05e578db,     //   0.01994,   -0.9998
    0x3f95fd4d21fab226,     0xbfeffe1c6870cb77,     //  0.021474,  -0.99977
    0x3f978f535ddc9f03,     0xbfeffdd4dbf78f52,     //  0.023008,  -0.99974
    0x3f992155f7a3667e,     0xbfeffd886084cd0d,     //  0.024541,   -0.9997
    0x3f9ab354b1504fca,     0xbfeffd36f624500c,     //  0.026075,  -0.99966
    0x3f9c454f4ce53b1c,     0xbfeffce09ce2a679,     //  0.027608,  -0.99962
    0x3f9dd7458c64ab39,     0xbfeffc8554cd213a,     //  0.029142,  -0.99958
    0x3f9f693731d1cf01,     0xbfeffc251df1d3f8,     //  0.030675,  -0.99953
    0x3fa07d91ff984580,     0xbfeffbbff85f9515,     //  0.032208,  -0.99948
    0x3fa14685db42c17e,     0xbfeffb55e425fdae,     //  0.033741,  -0.99943
    0x3fa20f770ceb11c6,     0xbfeffae6e1556998,     //  0.035274,  -0.99938
    0x3fa2d865759455cd,     0xbfeffa72effef75d,     //  0.036807,  -0.99932
    0x3fa3a150f6421afc,     0xbfeff9fa10348837,     //   0.03834,  -0.99926
    0x3fa46a396ff86179,     0xbfeff97c4208c014,     //  0.039873,   -0.9992
    0x3fa5331ec3bba0eb,     0xbfeff8f9858f058b,     //  0.041406,  -0.99914
    0x3fa5fc00d290cd43,     0xbfeff871dadb81df,     //  0.042938,  -0.99908
    0x3fa6c4df7d7d5b84,     0xbfeff7e5420320f9,     //  0.044471,  -0.99901
    0x3fa78dbaa5874685,     0xbfeff753bb1b9164,     //  0.046003,  -0.99894
    0x3fa856922bb513c1,     0xbfeff6bd463b444d,     //  0.047535,  -0.99887
    0x3fa91f65f10dd814,     0xbfeff621e3796d7e,     //  0.049068,   -0.9988
    0x3fa9e835d6993c87,     0xbfeff58192ee0358,     //    0.0506,  -0.99872
    0x3faab101bd5f8317,     0xbfeff4dc54b1bed3,     //  0.052132,  -0.99864
    0x3fab79c986698b78,     0xbfeff43228de1b77,     //  0.053664,  -0.99856
    0x3fac428d12c0d7e3,     0xbfeff3830f8d575c,     //  0.055195,  -0.99848
    0x3fad0b4c436f91d0,     0xbfeff2cf08da7321,     //  0.056727,  -0.99839
    0x3fadd406f9808ec8,     0xbfeff21614e131ed,     //  0.058258,   -0.9983
    0x3fae9cbd15ff5527,     0xbfeff15833be1965,     //   0.05979,  -0.99821
    0x3faf656e79f820e0,     0xbfeff095658e71ad,     //  0.061321,  -0.99812
    0x3fb0170d833bf421,     0xbfefefcdaa704562,     //  0.062852,  -0.99802
    0x3fb07b614e463064,     0xbfefef0102826191,     //  0.064383,  -0.99793
    0x3fb0dfb28ea201e6,     0xbfefee2f6de455ba,     //  0.065913,  -0.99783
    0x3fb1440134d709b2,     0xbfefed58ecb673c4,     //  0.067444,  -0.99772
    0x3fb1a84d316d4f8a,     0xbfefec7d7f19cffc,     //  0.068974,  -0.99762
    0x3fb20c9674ed444c,     0xbfefeb9d2530410f,     //  0.070505,  -0.99751
    0x3fb270dcefdfc45b,     0xbfefeab7df1c6005,     //  0.072035,   -0.9974
    0x3fb2d52092ce19f6,     0xbfefe9cdad01883a,     //  0.073565,  -0.99729
    0x3fb339614e41ffa5,     0xbfefe8de8f03d75c,     //  0.075094,  -0.99718
    0x3fb39d9f12c5a299,     0xbfefe7ea85482d60,     //  0.076624,  -0.99706
    0x3fb401d9d0e3a507,     0xbfefe6f18ff42c84,     //  0.078153,  -0.99694
    0x3fb4661179272096,     0xbfefe5f3af2e3940,     //  0.079682,  -0.99682
    0x3fb4ca45fc1ba8b6,     0xbfefe4f0e31d7a4a,     //  0.081211,   -0.9967
    0x3fb52e774a4d4d0a,     0xbfefe3e92be9d886,     //   0.08274,  -0.99657
    0x3fb592a554489bc8,     0xbfefe2dc89bbff08,     //  0.084269,  -0.99644
    0x3fb5f6d00a9aa419,     0xbfefe1cafcbd5b09,     //  0.085797,  -0.99631
    0x3fb65af75dd0f87b,     0xbfefe0b485181be3,     //  0.087326,  -0.99618
    0x3fb6bf1b3e79b129,     0xbfefdf9922f73307,     //  0.088854,  -0.99604
    0x3fb7233b9d236e71,     0xbfefde78d68653fd,     //  0.090381,  -0.99591
    0x3fb787586a5d5b21,     0xbfefdd539ff1f456,     //  0.091909,  -0.99577
    0x3fb7eb7196b72ee4,     0xbfefdc297f674ba9,     //  0.093436,  -0.99563
    0x3fb84f8712c130a0,     0xbfefdafa7514538c,     //  0.094963,  -0.99548
    0x3fb8b398cf0c38e0,     0xbfefd9c68127c78c,     //   0.09649,  -0.99533
    0x3fb917a6bc29b42c,     0xbfefd88da3d12526,     //  0.098017,  -0.99518
    0x3fb97bb0caaba56f,     0xbfefd74fdd40abbf,     //  0.099544,  -0.99503
    0x3fb9dfb6eb24a85c,     0xbfefd60d2da75c9e,     //   0.10107,  -0.99488
    0x3fba43b90e27f3c4,     0xbfefd4c59536fae4,     //    0.1026,  -0.99472
    0x3fbaa7b724495c04,     0xbfefd37914220b84,     //   0.10412,  -0.99456
    0x3fbb0bb11e1d5559,     0xbfefd227aa9bd53b,     //   0.10565,   -0.9944
    0x3fbb6fa6ec38f64c,     0xbfefd0d158d86087,     //   0.10717,  -0.99424
    0x3fbbd3987f31fa0e,     0xbfefcf761f0c77a3,     //    0.1087,  -0.99407
    0x3fbc3785c79ec2d5,     0xbfefce15fd6da67b,     //   0.11022,  -0.99391
    0x3fbc9b6eb6165c42,     0xbfefccb0f4323aa3,     //   0.11175,  -0.99374
    0x3fbcff533b307dc1,     0xbfefcb4703914354,     //   0.11327,  -0.99356
    0x3fbd633347858ce4,     0xbfefc9d82bc2915e,     //   0.11479,  -0.99339
    0x3fbdc70ecbae9fc8,     0xbfefc8646cfeb721,     //   0.11632,  -0.99321
    0x3fbe2ae5b8457f77,     0xbfefc6ebc77f0887,     //   0.11784,  -0.99303
    0x3fbe8eb7fde4aa3e,     0xbfefc56e3b7d9af6,     //   0.11937,  -0.99285
    0x3fbef2858d27561b,     0xbfefc3ebc935454c,     //   0.12089,  -0.99267
    0x3fbf564e56a9730e,     0xbfefc26470e19fd3,     //   0.12241,  -0.99248
    0x3fbfba124b07ad85,     0xbfefc0d832bf043a,     //   0.12393,  -0.99229
    0x3fc00ee8ad6fb85b,     0xbfefbf470f0a8d88,     //   0.12545,   -0.9921
    0x3fc040c5bb67747e,     0xbfefbdb106021816,     //   0.12698,  -0.99191
    0x3fc072a047ba831d,     0xbfefbc1617e44186,     //    0.1285,  -0.99171
    0x3fc0a4784ab8bf1d,     0xbfefba7644f068b5,     //   0.13002,  -0.99151
    0x3fc0d64dbcb26786,     0xbfefb8d18d66adb7,     //   0.13154,  -0.99131
    0x3fc1082095f820b0,     0xbfefb727f187f1c7,     //   0.13306,  -0.99111
    0x3fc139f0cedaf576,     0xbfefb5797195d741,     //   0.13458,   -0.9909
    0x3fc16bbe5fac5865,     0xbfefb3c60dd2c199,     //    0.1361,   -0.9907
    0x3fc19d8940be24e7,     0xbfefb20dc681d54d,     //   0.13762,  -0.99049
    0x3fc1cf516a62a077,     0xbfefb0509be6f7db,     //   0.13914,  -0.99027
    0x3fc20116d4ec7bce,     0xbfefae8e8e46cfbb,     //   0.14066,  -0.99006
    0x3fc232d978aed413,     0xbfefacc79de6c44f,     //   0.14218,  -0.98984
    0x3fc264994dfd340a,     0xbfefaafbcb0cfddc,     //    0.1437,  -0.98962
    0x3fc296564d2b953e,     0xbfefa92b1600657c,     //   0.14521,   -0.9894
    0x3fc2c8106e8e613a,     0xbfefa7557f08a517,     //   0.14673,  -0.98918
    0x3fc2f9c7aa7a72af,     0xbfefa57b066e2754,     //   0.14825,  -0.98895
    0x3fc32b7bf94516a7,     0xbfefa39bac7a1791,     //   0.14976,  -0.98872
    0x3fc35d2d53440db2,     0xbfefa1b7717661d5,     //   0.15128,  -0.98849
    0x3fc38edbb0cd8d14,     0xbfef9fce55adb2c8,     //    0.1528,  -0.98826
    0x3fc3c0870a383ff6,     0xbfef9de0596b77a3,     //   0.15431,  -0.98802
    0x3fc3f22f57db4893,     0xbfef9bed7cfbde29,     //   0.15583,  -0.98778
    0x3fc423d4920e4166,     0xbfef99f5c0abd496,     //   0.15734,  -0.98754
    0x3fc45576b1293e5a,     0xbfef97f924c9099b,     //   0.15886,   -0.9873
    0x3fc48715ad84cdf5,     0xbfef95f7a9a1ec47,     //   0.16037,  -0.98706
    0x3fc4b8b17f79fa88,     0xbfef93f14f85ac08,     //   0.16189,  -0.98681
    0x3fc4ea4a1f624b61,     0xbfef91e616c43891,     //    0.1634,  -0.98656
    0x3fc51bdf8597c5f2,     0xbfef8fd5ffae41db,     //   0.16491,  -0.98631
    0x3fc54d71aa74ef02,     0xbfef8dc10a95380d,     //   0.16643,  -0.98605
    0x3fc57f008654cbde,     0xbfef8ba737cb4b78,     //   0.16794,   -0.9858
    0x3fc5b08c1192e381,     0xbfef898887a36c84,     //   0.16945,  -0.98554
    0x3fc5e214448b3fc6,     0xbfef8764fa714ba9,     //   0.17096,  -0.98528
    0x3fc61399179a6e94,     0xbfef853c9089595e,     //   0.17247,  -0.98501
    0x3fc6451a831d830d,     0xbfef830f4a40c60c,     //   0.17398,  -0.98475
    0x3fc676987f7216b8,     0xbfef80dd27ed8204,     //   0.17549,  -0.98448
    0x3fc6a81304f64ab2,     0xbfef7ea629e63d6e,     //     0.177,  -0.98421
    0x3fc6d98a0c08c8da,     0xbfef7c6a50826840,     //   0.17851,  -0.98394
    0x3fc70afd8d08c4ff,     0xbfef7a299c1a322a,     //   0.18002,  -0.98366
    0x3fc73c6d8055fe0a,     0xbfef77e40d068a90,     //   0.18153,  -0.98339
    0x3fc76dd9de50bf31,     0xbfef7599a3a12077,     //   0.18304,  -0.98311
    0x3fc79f429f59e11d,     0xbfef734a60446279,     //   0.18455,  -0.98282
    0x3fc7d0a7bbd2cb1b,     0xbfef70f6434b7eb7,     //   0.18606,  -0.98254
    0x3fc802092c1d744b,     0xbfef6e9d4d1262ca,     //   0.18756,  -0.98225
    0x3fc83366e89c64c5,     0xbfef6c3f7df5bbb7,     //   0.18907,  -0.98196
    0x3fc864c0e9b2b6cf,     0xbfef69dcd652f5de,     //   0.19057,  -0.98167
    0x3fc8961727c41804,     0xbfef677556883cee,     //   0.19208,  -0.98138
    0x3fc8c7699b34ca7e,     0xbfef6508fef47bd5,     //   0.19359,  -0.98108
    0x3fc8f8b83c69a60a,     0xbfef6297cff75cb0,     //   0.19509,  -0.98079
    0x3fc92a0303c8194f,     0xbfef6021c9f148c2,     //   0.19659,  -0.98048
    0x3fc95b49e9b62af9,     0xbfef5da6ed43685d,     //    0.1981,  -0.98018
    0x3fc98c8ce69a7aec,     0xbfef5b273a4fa2d9,     //    0.1996,  -0.97988
    0x3fc9bdcbf2dc4366,     0xbfef58a2b1789e84,     //    0.2011,  -0.97957
    0x3fc9ef0706e35a35,     0xbfef56195321c090,     //   0.20261,  -0.97926
    0x3fca203e1b1831da,     0xbfef538b1faf2d07,     //   0.20411,  -0.97895
    0x3fca517127e3dabc,     0xbfef50f81785c6b9,     //   0.20561,  -0.97863
    0x3fca82a025b00451,     0xbfef4e603b0b2f2d,     //   0.20711,  -0.97832
    0x3fcab3cb0ce6fe44,     0xbfef4bc38aa5c694,     //   0.20861,    -0.978
    0x3fcae4f1d5f3b9ab,     0xbfef492206bcabb4,     //   0.21011,  -0.97768
    0x3fcb16147941ca2a,     0xbfef467bafb7bbe0,     //   0.21161,  -0.97735
    0x3fcb4732ef3d6722,     0xbfef43d085ff92dd,     //   0.21311,  -0.97703
    0x3fcb784d30536cda,     0xbfef412089fd8adc,     //   0.21461,   -0.9767
    0x3fcba96334f15dad,     0xbfef3e6bbc1bbc65,     //   0.21611,  -0.97637
    0x3fcbda74f5856330,     0xbfef3bb21cc4fe47,     //    0.2176,  -0.97604
    0x3fcc0b826a7e4f63,     0xbfef38f3ac64e589,     //    0.2191,   -0.9757
    0x3fcc3c8b8c4b9dd7,     0xbfef36306b67c556,     //    0.2206,  -0.97536
    0x3fcc6d90535d74dc,     0xbfef33685a3aaef0,     //   0.22209,  -0.97503
    0x3fcc9e90b824a6a9,     0xbfef309b794b719f,     //   0.22359,  -0.97468
    0x3fcccf8cb312b286,     0xbfef2dc9c9089a9d,     //   0.22508,  -0.97434
    0x3fcd00843c99c5f9,     0xbfef2af349e17507,     //   0.22658,  -0.97399
    0x3fcd31774d2cbdee,     0xbfef2817fc4609ce,     //   0.22807,  -0.97364
    0x3fcd6265dd3f27e3,     0xbfef2537e0a71f9f,     //   0.22957,  -0.97329
    0x3fcd934fe5454311,     0xbfef2252f7763ada,     //   0.23106,  -0.97294
    0x3fcdc4355db40195,     0xbfef1f6941259d7a,     //   0.23255,  -0.97258
    0x3fcdf5163f01099a,     0xbfef1c7abe284708,     //   0.23404,  -0.97223
    0x3fce25f281a2b684,     0xbfef19876ef1f486,     //   0.23553,  -0.97187
    0x3fce56ca1e101a1b,     0xbfef168f53f7205d,     //   0.23702,   -0.9715
    0x3fce879d0cc0fdaf,     0xbfef13926dad024e,     //   0.23851,  -0.97114
    0x3fceb86b462de348,     0xbfef1090bc898f5f,     //      0.24,  -0.97077
    0x3fcee934c2d006c7,     0xbfef0d8a410379c5,     //   0.24149,   -0.9704
    0x3fcf19f97b215f1a,     0xbfef0a7efb9230d7,     //   0.24298,  -0.97003
    0x3fcf4ab9679c9f5c,     0xbfef076eecade0fa,     //   0.24447,  -0.96966
    0x3fcf7b7480bd3801,     0xbfef045a14cf738c,     //   0.24596,  -0.96928
    0x3fcfac2abeff57ff,     0xbfef014074708ed3,     //   0.24744,   -0.9689
    0x3fcfdcdc1adfedf8,     0xbfeefe220c0b95ec,     //   0.24893,  -0.96852
    0x3fd006c4466e54af,     0xbfeefafedc1ba8b7,     //   0.25041,  -0.96814
    0x3fd01f1806b9fdd2,     0xbfeef7d6e51ca3c0,     //    0.2519,  -0.96775
    0x3fd037694a928cac,     0xbfeef4aa278b2032,     //   0.25338,  -0.96737
    0x3fd04fb80e37fdae,     0xbfeef178a3e473c2,     //   0.25487,  -0.96698
    0x3fd068044deab002,     0xbfeeee425aa6b09a,     //   0.25635,  -0.96658
    0x3fd0804e05eb661e,     0xbfeeeb074c50a544,     //   0.25783,  -0.96619
    0x3fd09895327b465e,     0xbfeee7c77961dc9e,     //   0.25931,  -0.96579
    0x3fd0b0d9cfdbdb90,     0xbfeee482e25a9dbc,     //   0.26079,  -0.96539
    0x3fd0c91bda4f158d,     0xbfeee13987bbebdc,     //   0.26227,  -0.96499
    0x3fd0e15b4e1749cd,     0xbfeeddeb6a078651,     //   0.26375,  -0.96459
    0x3fd0f998277733f7,     0xbfeeda9889bfe86a,     //   0.26523,  -0.96418
    0x3fd111d262b1f677,     0xbfeed740e7684963,     //   0.26671,  -0.96378
    0x3fd12a09fc0b1b12,     0xbfeed3e483849c51,     //   0.26819,  -0.96337
    0x3fd1423eefc69378,     0xbfeed0835e999009,     //   0.26967,  -0.96295
    0x3fd15a713a28b9d9,     0xbfeecd1d792c8f10,     //   0.27115,  -0.96254
    0x3fd172a0d7765177,     0xbfeec9b2d3c3bf84,     //   0.27262,  -0.96212
    0x3fd18acdc3f4873a,     0xbfeec6436ee60309,     //    0.2741,   -0.9617
    0x3fd1a2f7fbe8f243,     0xbfeec2cf4b1af6b2,     //   0.27557,  -0.96128
    0x3fd1bb1f7b999480,     0xbfeebf5668eaf2ef,     //   0.27705,  -0.96086
    0x3fd1d3443f4cdb3d,     0xbfeebbd8c8df0b74,     //   0.27852,  -0.96043
    0x3fd1eb6643499fbb,     0xbfeeb8566b810f2a,     //   0.27999,     -0.96
    0x3fd2038583d727bd,     0xbfeeb4cf515b8811,     //   0.28146,  -0.95957
    0x3fd21ba1fd3d2623,     0xbfeeb1437af9bb34,     //   0.28294,  -0.95914
    0x3fd233bbabc3bb72,     0xbfeeadb2e8e7a88e,     //   0.28441,   -0.9587
    0x3fd24bd28bb37672,     0xbfeeaa1d9bb20af3,     //   0.28588,  -0.95827
    0x3fd263e6995554ba,     0xbfeea68393e65800,     //   0.28735,  -0.95783
    0x3fd27bf7d0f2c346,     0xbfeea2e4d212c000,     //   0.28882,  -0.95738
    0x3fd294062ed59f05,     0xbfee9f4156c62dda,     //   0.29028,  -0.95694
    0x3fd2ac11af483572,     0xbfee9b99229046f8,     //   0.29175,  -0.95649
    0x3fd2c41a4e954520,     0xbfee97ec36016b30,     //   0.29322,  -0.95605
    0x3fd2dc200907fe51,     0xbfee943a91aab4b4,     //   0.29469,  -0.95559
    0x3fd2f422daec0386,     0xbfee9084361df7f3,     //   0.29615,  -0.95514
    0x3fd30c22c08d6a13,     0xbfee8cc923edc388,     //   0.29762,  -0.95469
    0x3fd3241fb638baaf,     0xbfee89095bad6025,     //   0.29908,  -0.95423
    0x3fd33c19b83af207,     0xbfee8544ddf0d075,     //   0.30054,  -0.95377
    0x3fd35410c2e18152,     0xbfee817bab4cd10d,     //   0.30201,  -0.95331
    0x3fd36c04d27a4edf,     0xbfee7dadc456d850,     //   0.30347,  -0.95284
    0x3fd383f5e353b6aa,     0xbfee79db29a5165a,     //   0.30493,  -0.95238
    0x3fd39be3f1bc8aef,     0xbfee7603dbce74e9,     //   0.30639,  -0.95191
    0x3fd3b3cefa0414b7,     0xbfee7227db6a9744,     //   0.30785,  -0.95144
    0x3fd3cbb6f87a146e,     0xbfee6e472911da27,     //   0.30931,  -0.95096
    0x3fd3e39be96ec271,     0xbfee6a61c55d53a7,     //   0.31077,  -0.95049
    0x3fd3fb7dc932cfa4,     0xbfee6677b0e6d31e,     //   0.31222,  -0.95001
    0x3fd4135c94176602,     0xbfee6288ec48e112,     //   0.31368,  -0.94953
    0x3fd42b38466e2928,     0xbfee5e95781ebf1c,     //   0.31514,  -0.94905
    0x3fd44310dc8936f0,     0xbfee5a9d550467d3,     //   0.31659,  -0.94856
    0x3fd45ae652bb2800,     0xbfee56a083968eb1,     //   0.31805,  -0.94807
    0x3fd472b8a5571054,     0xbfee529f04729ffc,     //    0.3195,  -0.94759
    0x3fd48a87d0b07fd7,     0xbfee4e98d836c0af,     //   0.32096,  -0.94709
    0x3fd4a253d11b82f3,     0xbfee4a8dff81ce5e,     //   0.32241,   -0.9466
    0x3fd4ba1ca2eca31c,     0xbfee467e7af35f23,     //   0.32386,  -0.94611
    0x3fd4d1e24278e76a,     0xbfee426a4b2bc17e,     //   0.32531,  -0.94561
    0x3fd4e9a4ac15d520,     0xbfee3e5170cbfc46,     //   0.32676,  -0.94511
    0x3fd50163dc197047,     0xbfee3a33ec75ce85,     //   0.32821,   -0.9446
    0x3fd5191fceda3c35,     0xbfee3611becbaf69,     //   0.32966,   -0.9441
    0x3fd530d880af3c24,     0xbfee31eae870ce25,     //   0.33111,  -0.94359
    0x3fd5488dedeff3be,     0xbfee2dbf6a0911d9,     //   0.33255,  -0.94308
    0x3fd5604012f467b4,     0xbfee298f4439197a,     //     0.334,  -0.94257
    0x3fd577eeec151e47,     0xbfee255a77a63bb8,     //   0.33545,  -0.94206
    0x3fd58f9a75ab1fdd,     0xbfee212104f686e5,     //   0.33689,  -0.94154
    0x3fd5a742ac0ff78d,     0xbfee1ce2ecd0c0d8,     //   0.33833,  -0.94103
    0x3fd5bee78b9db3b6,     0xbfee18a02fdc66d9,     //   0.33978,  -0.94051
    0x3fd5d68910aee686,     0xbfee1458cec1ad83,     //   0.34122,  -0.93998
    0x3fd5ee27379ea693,     0xbfee100cca2980ac,     //   0.34266,  -0.93946
    0x3fd605c1fcc88f63,     0xbfee0bbc22bd8349,     //    0.3441,  -0.93893
    0x3fd61d595c88c203,     0xbfee0766d9280f54,     //   0.34554,   -0.9384
    0x3fd634ed533be58e,     0xbfee030cee1435b8,     //   0.34698,  -0.93787
    0x3fd64c7ddd3f27c6,     0xbfedfeae622dbe2b,     //   0.34842,  -0.93734
    0x3fd6640af6f03d9e,     0xbfedfa4b3621271d,     //   0.34986,   -0.9368
    0x3fd67b949cad63ca,     0xbfedf5e36a9ba59c,     //   0.35129,  -0.93627
    0x3fd6931acad55f51,     0xbfedf177004b2534,     //   0.35273,  -0.93573
    0x3fd6aa9d7dc77e16,     0xbfeded05f7de47da,     //   0.35416,  -0.93518
    0x3fd6c21cb1e39771,     0xbfede890520465ce,     //    0.3556,  -0.93464
    0x3fd6d998638a0cb5,     0xbfede4160f6d8d81,     //   0.35703,  -0.93409
    0x3fd6f1108f1bc9c5,     0xbfeddf9730ca837b,     //   0.35846,  -0.93354
    0x3fd7088530fa459e,     0xbfeddb13b6ccc23d,     //    0.3599,  -0.93299
    0x3fd71ff6458782ec,     0xbfedd68ba2267a25,     //   0.36133,  -0.93244
    0x3fd73763c9261092,     0xbfedd1fef38a915a,     //   0.36276,  -0.93188
    0x3fd74ecdb8390a3e,     0xbfedcd6dabaca3a5,     //   0.36418,  -0.93133
    0x3fd766340f2418f6,     0xbfedc8d7cb410260,     //   0.36561,  -0.93077
    0x3fd77d96ca4b73a6,     0xbfedc43d52fcb453,     //   0.36704,  -0.93021
    0x3fd794f5e613dfae,     0xbfedbf9e4395759a,     //   0.36847,  -0.92964
    0x3fd7ac515ee2b172,     0xbfedbafa9dc1b78d,     //   0.36989,  -0.92907
    0x3fd7c3a9311dcce7,     0xbfedb6526238a09b,     //   0.37132,  -0.92851
    0x3fd7dafd592ba621,     0xbfedb1a591b20c38,     //   0.37274,  -0.92794
    0x3fd7f24dd37341e3,     0xbfedacf42ce68ab9,     //   0.37416,  -0.92736
    0x3fd8099a9c5c362d,     0xbfeda83e348f613b,     //   0.37559,  -0.92679
    0x3fd820e3b04eaac4,     0xbfeda383a9668988,     //   0.37701,  -0.92621
    0x3fd838290bb359c8,     0xbfed9ec48c26b1f3,     //   0.37843,  -0.92563
    0x3fd84f6aaaf3903f,     0xbfed9a00dd8b3d46,     //   0.37985,  -0.92505
    0x3fd866a88a792ea0,     0xbfed95389e50429b,     //   0.38127,  -0.92447
    0x3fd87de2a6aea963,     0xbfed906bcf328d46,     //   0.38268,  -0.92388
    0x3fd89518fbff098e,     0xbfed8b9a70ef9cb4,     //    0.3841,  -0.92329
    0x3fd8ac4b86d5ed44,     0xbfed86c48445a450,     //   0.38552,   -0.9227
    0x3fd8c37a439f884f,     0xbfed81ea09f38b63,     //   0.38693,  -0.92211
    0x3fd8daa52ec8a4af,     0xbfed7d0b02b8ecf9,     //   0.38835,  -0.92151
    0x3fd8f1cc44bea329,     0xbfed78276f5617c6,     //   0.38976,  -0.92092
    0x3fd908ef81ef7bd1,     0xbfed733f508c0dff,     //   0.39117,  -0.92032
    0x3fd9200ee2c9be97,     0xbfed6e52a71c8547,     //   0.39258,  -0.91972
    0x3fd9372a63bc93d7,     0xbfed696173c9e68b,     //   0.39399,  -0.91911
    0x3fd94e420137bce3,     0xbfed646bb7574de5,     //    0.3954,  -0.91851
    0x3fd96555b7ab948f,     0xbfed5f7172888a7f,     //   0.39681,   -0.9179
    0x3fd97c6583890fc2,     0xbfed5a72a6221e73,     //   0.39822,  -0.91729
    0x3fd993716141bdfe,     0xbfed556f52e93eb1,     //   0.39962,  -0.91668
    0x3fd9aa794d47c9ee,     0xbfed506779a3d2d9,     //   0.40103,  -0.91606
    0x3fd9c17d440df9f2,     0xbfed4b5b1b187524,     //   0.40243,  -0.91545
    0x3fd9d87d4207b0ab,     0xbfed464a380e7242,     //   0.40384,  -0.91483
    0x3fd9ef7943a8ed8a,     0xbfed4134d14dc93a,     //   0.40524,  -0.91421
    0x3fda067145664d57,     0xbfed3c1ae79f2b4e,     //   0.40664,  -0.91359
    0x3fda1d6543b50ac0,     0xbfed36fc7bcbfbdc,     //   0.40804,  -0.91296
    0x3fda34553b0afee5,     0xbfed31d98e9e503a,     //   0.40944,  -0.91234
    0x3fda4b4127dea1e4,     0xbfed2cb220e0ef9f,     //   0.41084,  -0.91171
    0x3fda622906a70b63,     0xbfed2786335f52fc,     //   0.41224,  -0.91107
    0x3fda790cd3dbf31a,     0xbfed2255c6e5a4e1,     //   0.41364,  -0.91044
    0x3fda8fec8bf5b166,     0xbfed1d20dc40c15c,     //   0.41503,  -0.90981
    0x3fdaa6c82b6d3fc9,     0xbfed17e7743e35dc,     //   0.41643,  -0.90917
    0x3fdabd9faebc3980,     0xbfed12a98fac410c,     //   0.41782,  -0.90853
    0x3fdad473125cdc08,     0xbfed0d672f59d2b9,     //   0.41922,  -0.90789
    0x3fdaeb4252ca07ab,     0xbfed082054168bac,     //   0.42061,  -0.90724
    0x3fdb020d6c7f4009,     0xbfed02d4feb2bd92,     //     0.422,   -0.9066
    0x3fdb18d45bf8aca6,     0xbfecfd852fff6ad4,     //   0.42339,  -0.90595
    0x3fdb2f971db31972,     0xbfecf830e8ce467b,     //   0.42478,   -0.9053
    0x3fdb4655ae2bf757,     0xbfecf2d829f1b40e,     //   0.42617,  -0.90464
    0x3fdb5d1009e15cc0,     0xbfeced7af43cc773,     //   0.42756,  -0.90399
    0x3fdb73c62d520624,     0xbfece819488344ce,     //   0.42894,  -0.90333
    0x3fdb8a7814fd5693,     0xbfece2b32799a060,     //   0.43033,  -0.90267
    0x3fdba125bd63583e,     0xbfecdd489254fe65,     //   0.43171,  -0.90201
    0x3fdbb7cf2304bd01,     0xbfecd7d9898b32f6,     //   0.43309,  -0.90135
    0x3fdbce744262deee,     0xbfecd2660e12c1e6,     //   0.43448,  -0.90068
    0x3fdbe51517ffc0d9,     0xbfecccee20c2de9f,     //   0.43586,  -0.90002
    0x3fdbfbb1a05e0edc,     0xbfecc771c2736c09,     //   0.43724,  -0.89935
    0x3fdc1249d8011ee7,     0xbfecc1f0f3fcfc5c,     //   0.43862,  -0.89867
    0x3fdc28ddbb6cf145,     0xbfecbc6bb638d10b,     //   0.43999,    -0.898
    0x3fdc3f6d47263129,     0xbfecb6e20a00da99,     //   0.44137,  -0.89732
    0x3fdc55f877b23537,     0xbfecb153f02fb87d,     //   0.44275,  -0.89665
    0x3fdc6c7f4997000a,     0xbfecabc169a0b901,     //   0.44412,  -0.89597
    0x3fdc8301b95b40c2,     0xbfeca62a772fd919,     //    0.4455,  -0.89528
    0x3fdc997fc3865388,     0xbfeca08f19b9c449,     //   0.44687,   -0.8946
    0x3fdcaff964a0421d,     0xbfec9aef521bd480,     //   0.44824,  -0.89391
    0x3fdcc66e9931c45d,     0xbfec954b213411f5,     //   0.44961,  -0.89322
    0x3fdcdcdf5dc440ce,     0xbfec8fa287e13305,     //   0.45098,  -0.89253
    0x3fdcf34baee1cd21,     0xbfec89f587029c13,     //   0.45235,  -0.89184
    0x3fdd09b389152ec1,     0xbfec84441f785f61,     //   0.45372,  -0.89115
    0x3fdd2016e8e9db5b,     0xbfec7e8e52233cf3,     //   0.45508,  -0.89045
    0x3fdd3675caebf962,     0xbfec78d41fe4a267,     //   0.45645,  -0.88975
    0x3fdd4cd02ba8609c,     0xbfec7315899eaad7,     //   0.45781,  -0.88905
    0x3fdd632607ac9aa9,     0xbfec6d5290341eb2,     //   0.45918,  -0.88835
    0x3fdd79775b86e389,     0xbfec678b3488739b,     //   0.46054,  -0.88764
    0x3fdd8fc423c62a25,     0xbfec61bf777fcc48,     //    0.4619,  -0.88693
    0x3fdda60c5cfa10d8,     0xbfec5bef59fef85a,     //   0.46326,  -0.88622
    0x3fddbc5003b2edf8,     0xbfec561adceb743e,     //   0.46462,  -0.88551
    0x3fddd28f1481cc58,     0xbfec5042012b6907,     //   0.46598,   -0.8848
    0x3fdde8c98bf86bd6,     0xbfec4a64c7a5ac4c,     //   0.46733,  -0.88408
    0x3fddfeff66a941de,     0xbfec44833141c004,     //   0.46869,  -0.88336
    0x3fde1530a12779f4,     0xbfec3e9d3ee7d262,     //   0.47004,  -0.88264
    0x3fde2b5d3806f63b,     0xbfec38b2f180bdb1,     //    0.4714,  -0.88192
    0x3fde418527dc4ffa,     0xbfec32c449f60831,     //   0.47275,   -0.8812
    0x3fde57a86d3cd824,     0xbfec2cd14931e3f1,     //    0.4741,  -0.88047
    0x3fde6dc704be97e2,     0xbfec26d9f01f2eaf,     //   0.47545,  -0.87974
    0x3fde83e0eaf85113,     0xbfec20de3fa971b0,     //    0.4768,  -0.87901
    0x3fde99f61c817eda,     0xbfec1ade38bce19b,     //   0.47815,  -0.87828
    0x3fdeb00695f25620,     0xbfec14d9dc465e58,     //   0.47949,  -0.87755
    0x3fdec61253e3c61b,     0xbfec0ed12b3372e9,     //   0.48084,  -0.87681
    0x3fdedc1952ef78d5,     0xbfec08c426725549,     //   0.48218,  -0.87607
    0x3fdef21b8fafd3b5,     0xbfec02b2cef1e641,     //   0.48353,  -0.87533
    0x3fdf081906bff7fd,     0xbfebfc9d25a1b147,     //   0.48487,  -0.87459
    0x3fdf1e11b4bbc35c,     0xbfebf6832b71ec5b,     //   0.48621,  -0.87384
    0x3fdf3405963fd068,     0xbfebf064e15377dd,     //   0.48755,  -0.87309
    0x3fdf49f4a7e97729,     0xbfebea424837de6d,     //   0.48889,  -0.87235
    0x3fdf5fdee656cda3,     0xbfebe41b611154c1,     //   0.49023,   -0.8716
    0x3fdf75c44e26a852,     0xbfebddf02cd2b983,     //   0.49156,  -0.87084
    0x3fdf8ba4dbf89aba,     0xbfebd7c0ac6f952a,     //    0.4929,  -0.87009
    0x3fdfa1808c6cf7e0,     0xbfebd18ce0dc19d6,     //   0.49423,  -0.86933
    0x3fdfb7575c24d2de,     0xbfebcb54cb0d2327,     //   0.49557,  -0.86857
    0x3fdfcd2947c1ff57,     0xbfebc5186bf8361d,     //    0.4969,  -0.86781
    0x3fdfe2f64be7120f,     0xbfebbed7c49380ea,     //   0.49823,  -0.86705
    0x3fdff8be6537615e,     0xbfebb892d5d5dad5,     //   0.49956,  -0.86628
    0x3fe00740c82b82e0,     0xbfebb249a0b6c40d,     //   0.50089,  -0.86551
    0x3fe0121fe4f56d2c,     0xbfebabfc262e6586,     //   0.50221,  -0.86474
    0x3fe01cfc874c3eb7,     0xbfeba5aa673590d2,     //   0.50354,  -0.86397
    0x3fe027d6ad83287e,     0xbfeb9f5464c5bffc,     //   0.50486,   -0.8632
    0x3fe032ae55edbd95,     0xbfeb98fa1fd9155e,     //   0.50619,  -0.86242
    0x3fe03d837edff370,     0xbfeb929b996a5b7f,     //   0.50751,  -0.86165
    0x3fe0485626ae221a,     0xbfeb8c38d27504e9,     //   0.50883,  -0.86087
    0x3fe053264bad0483,     0xbfeb85d1cbf52c02,     //   0.51015,  -0.86009
    0x3fe05df3ec31b8b6,     0xbfeb7f6686e792ea,     //   0.51147,   -0.8593
    0x3fe068bf0691c028,     0xbfeb78f70449a34b,     //   0.51279,  -0.85852
    0x3fe073879922ffed,     0xbfeb728345196e3e,     //    0.5141,  -0.85773
    0x3fe07e4da23bc102,     0xbfeb6c0b4a55ac17,     //   0.51542,  -0.85694
    0x3fe089112032b08c,     0xbfeb658f14fdbc47,     //   0.51673,  -0.85615
    0x3fe093d2115ee018,     0xbfeb5f0ea611a532,     //   0.51804,  -0.85535
    0x3fe09e907417c5e1,     0xbfeb5889fe921405,     //   0.51936,  -0.85456
    0x3fe0a94c46b53d0b,     0xbfeb52011f805c92,     //   0.52067,  -0.85376
    0x3fe0b405878f85ec,     0xbfeb4b7409de7925,     //   0.52198,  -0.85296
    0x3fe0bebc34ff4646,     0xbfeb44e2beaf0a61,     //   0.52328,  -0.85216
    0x3fe0c9704d5d898f,     0xbfeb3e4d3ef55712,     //   0.52459,  -0.85136
    0x3fe0d421cf03c12b,     0xbfeb37b38bb54c09,     //    0.5259,  -0.85055
    0x3fe0ded0b84bc4b5,     0xbfeb3115a5f37bf4,     //    0.5272,  -0.84974
    0x3fe0e97d078fd23b,     0xbfeb2a738eb51f33,     //    0.5285,  -0.84893
    0x3fe0f426bb2a8e7d,     0xbfeb23cd470013b4,     //    0.5298,  -0.84812
    0x3fe0fecdd1770537,     0xbfeb1d22cfdadcc6,     //    0.5311,  -0.84731
    0x3fe1097248d0a956,     0xbfeb16742a4ca2f5,     //    0.5324,  -0.84649
    0x3fe114141f935545,     0xbfeb0fc1575d33db,     //    0.5337,  -0.84567
    0x3fe11eb3541b4b22,     0xbfeb090a58150200,     //     0.535,  -0.84485
    0x3fe1294fe4c5350a,     0xbfeb024f2d7d24a9,     //   0.53629,  -0.84403
    0x3fe133e9cfee254e,     0xbfeafb8fd89f57b6,     //   0.53759,  -0.84321
    0x3fe13e8113f396c1,     0xbfeaf4cc5a85fb73,     //   0.53888,  -0.84238
    0x3fe14915af336ceb,     0xbfeaee04b43c1474,     //   0.54017,  -0.84155
    0x3fe153a7a00bf453,     0xbfeae738e6cd4b67,     //   0.54146,  -0.84073
    0x3fe15e36e4dbe2bc,     0xbfeae068f345ecef,     //   0.54275,  -0.83989
    0x3fe168c37c025764,     0xbfead994dab2e979,     //   0.54404,  -0.83906
    0x3fe1734d63dedb49,     0xbfead2bc9e21d511,     //   0.54532,  -0.83822
    0x3fe17dd49ad16161,     0xbfeacbe03ea0e73b,     //   0.54661,  -0.83739
    0x3fe188591f3a46e5,     0xbfeac4ffbd3efac8,     //   0.54789,  -0.83655
    0x3fe192daef7a5386,     0xbfeabe1b1b0b8dac,     //   0.54918,  -0.83571
    0x3fe19d5a09f2b9b8,     0xbfeab7325916c0d4,     //   0.55046,  -0.83486
    0x3fe1a7d66d0516e6,     0xbfeab045787157ff,     //   0.55174,  -0.83402
    0x3fe1b250171373be,     0xbfeaa9547a2cb98e,     //   0.55302,  -0.83317
    0x3fe1bcc706804467,     0xbfeaa25f5f5aee60,     //   0.55429,  -0.83232
    0x3fe1c73b39ae68c8,     0xbfea9b66290ea1a3,     //   0.55557,  -0.83147
    0x3fe1d1acaf012cc2,     0xbfea9468d85b20ae,     //   0.55685,  -0.83062
    0x3fe1dc1b64dc4872,     0xbfea8d676e545ad2,     //   0.55812,  -0.82976
    0x3fe1e68759a3e074,     0xbfea8661ec0ee133,     //   0.55939,   -0.8289
    0x3fe1f0f08bbc861b,     0xbfea7f58529fe69d,     //   0.56066,  -0.82805
    0x3fe1fb56f98b37b8,     0xbfea784aa31d3f55,     //   0.56193,  -0.82718
    0x3fe205baa17560d6,     0xbfea7138de9d60f5,     //    0.5632,  -0.82632
    0x3fe2101b81e0da78,     0xbfea6a230637623b,     //   0.56447,  -0.82546
    0x3fe21a799933eb58,     0xbfea63091b02fae2,     //   0.56573,  -0.82459
    0x3fe224d4e5d5482e,     0xbfea5beb1e188375,     //     0.567,  -0.82372
    0x3fe22f2d662c13e1,     0xbfea54c91090f524,     //   0.56826,  -0.82285
    0x3fe23983189fdfd5,     0xbfea4da2f385e997,     //   0.56952,  -0.82198
    0x3fe243d5fb98ac1f,     0xbfea4678c8119ac8,     //   0.57078,   -0.8211
    0x3fe24e260d7ee7c9,     0xbfea3f4a8f4ee2d2,     //   0.57204,  -0.82023
    0x3fe258734cbb7110,     0xbfea38184a593bc6,     //    0.5733,  -0.81935
    0x3fe262bdb7b795a2,     0xbfea30e1fa4cbf81,     //   0.57455,  -0.81847
    0x3fe26d054cdd12df,     0xbfea29a7a0462782,     //   0.57581,  -0.81758
    0x3fe2774a0a961612,     0xbfea22693d62ccb9,     //   0.57706,   -0.8167
    0x3fe2818bef4d3cba,     0xbfea1b26d2c0a75e,     //   0.57831,  -0.81581
    0x3fe28bcaf96d94ba,     0xbfea13e0617e4ec7,     //   0.57956,  -0.81493
    0x3fe2960727629ca8,     0xbfea0c95eabaf937,     //   0.58081,  -0.81404
    0x3fe2a040779843fb,     0xbfea05476f967bb5,     //   0.58206,  -0.81314
    0x3fe2aa76e87aeb58,     0xbfe9fdf4f13149de,     //   0.58331,  -0.81225
    0x3fe2b4aa787764c4,     0xbfe9f69e70ac75bc,     //   0.58455,  -0.81135
    0x3fe2bedb25faf3ea,     0xbfe9ef43ef29af94,     //    0.5858,  -0.81046
    0x3fe2c908ef734e57,     0xbfe9e7e56dcb45bd,     //   0.58704,  -0.80956
    0x3fe2d333d34e9bb7,     0xbfe9e082edb42472,     //   0.58828,  -0.80866
    0x3fe2dd5bcffb7616,     0xbfe9d91c7007d5a6,     //   0.58952,  -0.80775
    0x3fe2e780e3e8ea16,     0xbfe9d1b1f5ea80d6,     //   0.59076,  -0.80685
    0x3fe2f1a30d86773a,     0xbfe9ca438080eadb,     //     0.592,  -0.80594
    0x3fe2fbc24b441015,     0xbfe9c2d110f075c3,     //   0.59323,  -0.80503
    0x3fe305de9b921a94,     0xbfe9bb5aa85f2098,     //   0.59447,  -0.80412
    0x3fe30ff7fce17035,     0xbfe9b3e047f38741,     //    0.5957,  -0.80321
    0x3fe31a0e6da35e44,     0xbfe9ac61f0d4e247,     //   0.59693,  -0.80229
    0x3fe32421ec49a620,     0xbfe9a4dfa42b06b2,     //   0.59816,  -0.80138
    0x3fe32e3277467d6b,     0xbfe99d59631e65d5,     //   0.59939,  -0.80046
    0x3fe338400d0c8e57,     0xbfe995cf2ed80d22,     //   0.60062,  -0.79954
    0x3fe3424aac0ef7d6,     0xbfe98e410881a600,     //   0.60184,  -0.79861
    0x3fe34c5252c14de1,     0xbfe986aef1457594,     //   0.60307,  -0.79769
    0x3fe35656ff9799ae,     0xbfe97f18ea4e5c9e,     //   0.60429,  -0.79676
    0x3fe36058b10659f3,     0xbfe9777ef4c7d742,     //   0.60551,  -0.79584
    0x3fe36a576582831b,     0xbfe96fe111ddfce0,     //   0.60673,  -0.79491
    0x3fe374531b817f8d,     0xbfe9683f42bd7fe1,     //   0.60795,  -0.79398
    0x3fe37e4bd1792fe2,     0xbfe960998893ad8c,     //   0.60917,  -0.79304
    0x3fe3884185dfeb22,     0xbfe958efe48e6dd7,     //   0.61038,  -0.79211
    0x3fe39234372c7f04,     0xbfe9514257dc4335,     //    0.6116,  -0.79117
    0x3fe39c23e3d63029,     0xbfe94990e3ac4a6c,     //   0.61281,  -0.79023
    0x3fe3a6108a54ba58,     0xbfe941db892e3a65,     //   0.61402,  -0.78929
    0x3fe3affa292050b9,     0xbfe93a22499263fc,     //   0.61523,  -0.78835
    0x3fe3b9e0beb19e18,     0xbfe932652609b1cf,     //   0.61644,   -0.7874
    0x3fe3c3c44981c517,     0xbfe92aa41fc5a815,     //   0.61765,  -0.78646
    0x3fe3cda4c80a6076,     0xbfe922df37f8646a,     //   0.61885,  -0.78551
    0x3fe3d78238c58343,     0xbfe91b166fd49da2,     //   0.62006,  -0.78456
    0x3fe3e15c9a2db922,     0xbfe91349c88da398,     //   0.62126,   -0.7836
    0x3fe3eb33eabe0680,     0xbfe90b7943575efe,     //   0.62246,  -0.78265
    0x3fe3f50828f1e8d2,     0xbfe903a4e1665133,     //   0.62366,  -0.78169
    0x3fe3fed9534556d4,     0xbfe8fbcca3ef940d,     //   0.62486,  -0.78074
    0x3fe408a76834c0c0,     0xbfe8f3f08c28d9ac,     //   0.62606,  -0.77978
    0x3fe41272663d108c,     0xbfe8ec109b486c49,     //   0.62725,  -0.77882
    0x3fe41c3a4bdbaa26,     0xbfe8e42cd2852e0a,     //   0.62845,  -0.77785
    0x3fe425ff178e6bb1,     0xbfe8dc45331698cc,     //   0.62964,  -0.77689
    0x3fe42fc0c7d3adbb,     0xbfe8d459be34bdfa,     //   0.63083,  -0.77592
    0x3fe4397f5b2a4380,     0xbfe8cc6a75184655,     //   0.63202,  -0.77495
    0x3fe4433ad0117b1d,     0xbfe8c47758fa71cb,     //   0.63321,  -0.77398
    0x3fe44cf325091dd6,     0xbfe8bc806b151741,     //   0.63439,  -0.77301
    0x3fe456a858917046,     0xbfe8b485aca2a468,     //   0.63558,  -0.77204
    0x3fe4605a692b32a2,     0xbfe8ac871ede1d88,     //   0.63676,  -0.77106
    0x3fe46a095557a0f1,     0xbfe8a484c3031d50,     //   0.63794,  -0.77008
    0x3fe473b51b987347,     0xbfe89c7e9a4dd4ab,     //   0.63912,   -0.7691
    0x3fe47d5dba6fde01,     0xbfe89474a5fb0a84,     //    0.6403,  -0.76812
    0x3fe48703306091fe,     0xbfe88c66e7481ba1,     //   0.64148,  -0.76714
    0x3fe490a57bedbcdf,     0xbfe884555f72fa6b,     //   0.64266,  -0.76615
    0x3fe49a449b9b0938,     0xbfe87c400fba2ebf,     //   0.64383,  -0.76517
    0x3fe4a3e08dec9ed6,     0xbfe87426f95cd5bd,     //     0.645,  -0.76418
    0x3fe4ad79516722f0,     0xbfe86c0a1d9aa195,     //   0.64618,  -0.76319
    0x3fe4b70ee48fb869,     0xbfe863e97db3d95a,     //   0.64735,   -0.7622
    0x3fe4c0a145ec0004,     0xbfe85bc51ae958cc,     //   0.64851,   -0.7612
    0x3fe4ca30740218a3,     0xbfe8539cf67c9029,     //   0.64968,  -0.76021
    0x3fe4d3bc6d589f80,     0xbfe84b7111af83f9,     //   0.65085,  -0.75921
    0x3fe4dd453076b064,     0xbfe843416dc4cce2,     //   0.65201,  -0.75821
    0x3fe4e6cabbe3e5e9,     0xbfe83b0e0bff976e,     //   0.65317,  -0.75721
    0x3fe4f04d0e2859aa,     0xbfe832d6eda3a3e0,     //   0.65433,  -0.75621
    0x3fe4f9cc25cca486,     0xbfe82a9c13f545ff,     //   0.65549,   -0.7552
    0x3fe503480159ded2,     0xbfe8225d803964e5,     //   0.65665,  -0.75419
    0x3fe50cc09f59a09b,     0xbfe81a1b33b57acc,     //   0.65781,  -0.75319
    0x3fe51635fe5601d7,     0xbfe811d52faf94dc,     //   0.65896,  -0.75218
    0x3fe51fa81cd99aa6,     0xbfe8098b756e52fa,     //   0.66011,  -0.75117
    0x3fe52916f96f8388,     0xbfe8013e0638e795,     //   0.66127,  -0.75015
    0x3fe5328292a35596,     0xbfe7f8ece3571771,     //   0.66242,  -0.74914
    0x3fe53beae7012abe,     0xbfe7f0980e113978,     //   0.66356,  -0.74812
    0x3fe5454ff5159dfb,     0xbfe7e83f87b03686,     //   0.66471,   -0.7471
    0x3fe54eb1bb6dcb8f,     0xbfe7dfe3517d8937,     //   0.66586,  -0.74608
    0x3fe5581038975137,     0xbfe7d7836cc33db2,     //     0.667,  -0.74506
    0x3fe5616b6b204e6e,     0xbfe7cf1fdacbf179,     //   0.66814,  -0.74403
    0x3fe56ac35197649e,     0xbfe7c6b89ce2d333,     //   0.66928,  -0.74301
    0x3fe57417ea8bb75c,     0xbfe7be4db453a27c,     //   0.67042,  -0.74198
    0x3fe57d69348cec9f,     0xbfe7b5df226aafb0,     //   0.67156,  -0.74095
    0x3fe586b72e2b2cfd,     0xbfe7ad6ce874dbb6,     //   0.67269,  -0.73992
    0x3fe59001d5f723df,     0xbfe7a4f707bf97d2,     //   0.67383,  -0.73889
    0x3fe599492a81ffbc,     0xbfe79c7d8198e56e,     //   0.67496,  -0.73785
    0x3fe5a28d2a5d7250,     0xbfe79400574f55e4,     //   0.67609,  -0.73682
    0x3fe5abcdd41bb0d8,     0xbfe78b7f8a320a52,     //   0.67722,  -0.73578
    0x3fe5b50b264f7448,     0xbfe782fb1b90b35b,     //   0.67835,  -0.73474
    0x3fe5be451f8bf980,     0xbfe77a730cbb9100,     //   0.67948,   -0.7337
    0x3fe5c77bbe65018c,     0xbfe771e75f037261,     //    0.6806,  -0.73265
    0x3fe5d0af016ed1d4,     0xbfe7695813b9b594,     //   0.68172,  -0.73161
    0x3fe5d9dee73e345c,     0xbfe760c52c304764,     //   0.68285,  -0.73056
    0x3fe5e30b6e6877f3,     0xbfe7582ea9b9a329,     //   0.68397,  -0.72951
    0x3fe5ec3495837074,     0xbfe74f948da8d28d,     //   0.68508,  -0.72846
    0x3fe5f55a5b2576f8,     0xbfe746f6d9516d59,     //    0.6862,  -0.72741
    0x3fe5fe7cbde56a0f,     0xbfe73e558e079942,     //   0.68732,  -0.72636
    0x3fe6079bbc5aadfa,     0xbfe735b0ad2009b2,     //   0.68843,   -0.7253
    0x3fe610b7551d2cde,     0xbfe72d0837efff97,     //   0.68954,  -0.72425
    0x3fe619cf86c55702,     0xbfe7245c2fcd492a,     //   0.69065,  -0.72319
    0x3fe622e44fec22ff,     0xbfe71bac960e41bf,     //   0.69176,  -0.72213
    0x3fe62bf5af2b0dfd,     0xbfe712f96c09d18d,     //   0.69287,  -0.72107
    0x3fe63503a31c1be8,     0xbfe70a42b3176d7a,     //   0.69397,     -0.72
    0x3fe63e0e2a59d7aa,     0xbfe701886c8f16e6,     //   0.69508,  -0.71894
    0x3fe64715437f535b,     0xbfe6f8ca99c95b75,     //   0.69618,  -0.71787
    0x3fe65018ed28287f,     0xbfe6f0093c1f54de,     //   0.69728,   -0.7168
    0x3fe6591925f0783e,     0xbfe6e74454eaa8ae,     //   0.69838,  -0.71573
    0x3fe66215ec74eb91,     0xbfe6de7be585881d,     //   0.69947,  -0.71466
    0x3fe66b0f3f52b386,     0xbfe6d5afef4aafcc,     //   0.70057,  -0.71358
    0x3fe674051d27896c,     0xbfe6cce07395679f,     //   0.70166,  -0.71251
    0x3fe67cf78491af10,     0xbfe6c40d73c18275,     //   0.70275,  -0.71143
    0x3fe685e6742feeef,     0xbfe6bb36f12b5e06,     //   0.70385,  -0.71035
    0x3fe68ed1eaa19c71,     0xbfe6b25ced2fe29c,     //   0.70493,  -0.70927
    0x3fe697b9e686941c,     0xbfe6a97f692c82ea,     //   0.70602,  -0.70819
    0x3fe6a09e667f3bcc,     0xbfe6a09e667f3bcc,     //   0.70711,  -0.70711
    0x3fe6a97f692c82ea,     0xbfe697b9e686941c,     //   0.70819,  -0.70602
    0x3fe6b25ced2fe29c,     0xbfe68ed1eaa19c71,     //   0.70927,  -0.70493
    0x3fe6bb36f12b5e06,     0xbfe685e6742feeef,     //   0.71035,  -0.70385
    0x3fe6c40d73c18275,     0xbfe67cf78491af10,     //   0.71143,  -0.70275
    0x3fe6cce07395679f,     0xbfe674051d27896c,     //   0.71251,  -0.70166
    0x3fe6d5afef4aafcc,     0xbfe66b0f3f52b386,     //   0.71358,  -0.70057
    0x3fe6de7be585881d,     0xbfe66215ec74eb91,     //   0.71466,  -0.69947
    0x3fe6e74454eaa8ae,     0xbfe6591925f0783e,     //   0.71573,  -0.69838
    0x3fe6f0093c1f54de,     0xbfe65018ed28287f,     //    0.7168,  -0.69728
    0x3fe6f8ca99c95b75,     0xbfe64715437f535b,     //   0.71787,  -0.69618
    0x3fe701886c8f16e6,     0xbfe63e0e2a59d7aa,     //   0.71894,  -0.69508
    0x3fe70a42b3176d7a,     0xbfe63503a31c1be8,     //      0.72,  -0.69397
    0x3fe712f96c09d18d,     0xbfe62bf5af2b0dfd,     //   0.72107,  -0.69287
    0x3fe71bac960e41bf,     0xbfe622e44fec22ff,     //   0.72213,  -0.69176
    0x3fe7245c2fcd492a,     0xbfe619cf86c55702,     //   0.72319,  -0.69065
    0x3fe72d0837efff97,     0xbfe610b7551d2cde,     //   0.72425,  -0.68954
    0x3fe735b0ad2009b2,     0xbfe6079bbc5aadfa,     //    0.7253,  -0.68843
    0x3fe73e558e079942,     0xbfe5fe7cbde56a0f,     //   0.72636,  -0.68732
    0x3fe746f6d9516d59,     0xbfe5f55a5b2576f8,     //   0.72741,   -0.6862
    0x3fe74f948da8d28d,     0xbfe5ec3495837074,     //   0.72846,  -0.68508
    0x3fe7582ea9b9a329,     0xbfe5e30b6e6877f3,     //   0.72951,  -0.68397
    0x3fe760c52c304764,     0xbfe5d9dee73e345c,     //   0.73056,  -0.68285
    0x3fe7695813b9b594,     0xbfe5d0af016ed1d4,     //   0.73161,  -0.68172
    0x3fe771e75f037261,     0xbfe5c77bbe65018c,     //   0.73265,   -0.6806
    0x3fe77a730cbb9100,     0xbfe5be451f8bf980,     //    0.7337,  -0.67948
    0x3fe782fb1b90b35b,     0xbfe5b50b264f7448,     //   0.73474,  -0.67835
    0x3fe78b7f8a320a52,     0xbfe5abcdd41bb0d8,     //   0.73578,  -0.67722
    0x3fe79400574f55e4,     0xbfe5a28d2a5d7250,     //   0.73682,  -0.67609
    0x3fe79c7d8198e56e,     0xbfe599492a81ffbc,     //   0.73785,  -0.67496
    0x3fe7a4f707bf97d2,     0xbfe59001d5f723df,     //   0.73889,  -0.67383
    0x3fe7ad6ce874dbb6,     0xbfe586b72e2b2cfd,     //   0.73992,  -0.67269
    0x3fe7b5df226aafb0,     0xbfe57d69348cec9f,     //   0.74095,  -0.67156
    0x3fe7be4db453a27c,     0xbfe57417ea8bb75c,     //   0.74198,  -0.67042
    0x3fe7c6b89ce2d333,     0xbfe56ac35197649e,     //   0.74301,  -0.66928
    0x3fe7cf1fdacbf179,     0xbfe5616b6b204e6e,     //   0.74403,  -0.66814
    0x3fe7d7836cc33db2,     0xbfe5581038975137,     //   0.74506,    -0.667
    0x3fe7dfe3517d8937,     0xbfe54eb1bb6dcb8f,     //   0.74608,  -0.66586
    0x3fe7e83f87b03686,     0xbfe5454ff5159dfb,     //    0.7471,  -0.66471
    0x3fe7f0980e113978,     0xbfe53beae7012abe,     //   0.74812,  -0.66356
    0x3fe7f8ece3571771,     0xbfe5328292a35596,     //   0.74914,  -0.66242
    0x3fe8013e0638e795,     0xbfe52916f96f8388,     //   0.75015,  -0.66127
    0x3fe8098b756e52fa,     0xbfe51fa81cd99aa6,     //   0.75117,  -0.66011
    0x3fe811d52faf94dc,     0xbfe51635fe5601d7,     //   0.75218,  -0.65896
    0x3fe81a1b33b57acc,     0xbfe50cc09f59a09b,     //   0.75319,  -0.65781
    0x3fe8225d803964e5,     0xbfe503480159ded2,     //   0.75419,  -0.65665
    0x3fe82a9c13f545ff,     0xbfe4f9cc25cca486,     //    0.7552,  -0.65549
    0x3fe832d6eda3a3e0,     0xbfe4f04d0e2859aa,     //   0.75621,  -0.65433
    0x3fe83b0e0bff976e,     0xbfe4e6cabbe3e5e9,     //   0.75721,  -0.65317
    0x3fe843416dc4cce2,     0xbfe4dd453076b064,     //   0.75821,  -0.65201
    0x3fe84b7111af83f9,     0xbfe4d3bc6d589f80,     //   0.75921,  -0.65085
    0x3fe8539cf67c9029,     0xbfe4ca30740218a3,     //   0.76021,  -0.64968
    0x3fe85bc51ae958cc,     0xbfe4c0a145ec0004,     //    0.7612,  -0.64851
    0x3fe863e97db3d95a,     0xbfe4b70ee48fb869,     //    0.7622,  -0.64735
    0x3fe86c0a1d9aa195,     0xbfe4ad79516722f0,     //   0.76319,  -0.64618
    0x3fe87426f95cd5bd,     0xbfe4a3e08dec9ed6,     //   0.76418,    -0.645
    0x3fe87c400fba2ebf,     0xbfe49a449b9b0938,     //   0.76517,  -0.64383
    0x3fe884555f72fa6b,     0xbfe490a57bedbcdf,     //   0.76615,  -0.64266
    0x3fe88c66e7481ba1,     0xbfe48703306091fe,     //   0.76714,  -0.64148
    0x3fe89474a5fb0a84,     0xbfe47d5dba6fde01,     //   0.76812,   -0.6403
    0x3fe89c7e9a4dd4ab,     0xbfe473b51b987347,     //    0.7691,  -0.63912
    0x3fe8a484c3031d50,     0xbfe46a095557a0f1,     //   0.77008,  -0.63794
    0x3fe8ac871ede1d88,     0xbfe4605a692b32a2,     //   0.77106,  -0.63676
    0x3fe8b485aca2a468,     0xbfe456a858917046,     //   0.77204,  -0.63558
    0x3fe8bc806b151741,     0xbfe44cf325091dd6,     //   0.77301,  -0.63439
    0x3fe8c47758fa71cb,     0xbfe4433ad0117b1d,     //   0.77398,  -0.63321
    0x3fe8cc6a75184655,     0xbfe4397f5b2a4380,     //   0.77495,  -0.63202
    0x3fe8d459be34bdfa,     0xbfe42fc0c7d3adbb,     //   0.77592,  -0.63083
    0x3fe8dc45331698cc,     0xbfe425ff178e6bb1,     //   0.77689,  -0.62964
    0x3fe8e42cd2852e0a,     0xbfe41c3a4bdbaa26,     //   0.77785,  -0.62845
    0x3fe8ec109b486c49,     0xbfe41272663d108c,     //   0.77882,  -0.62725
    0x3fe8f3f08c28d9ac,     0xbfe408a76834c0c0,     //   0.77978,  -0.62606
    0x3fe8fbcca3ef940d,     0xbfe3fed9534556d4,     //   0.78074,  -0.62486
    0x3fe903a4e1665133,     0xbfe3f50828f1e8d2,     //   0.78169,  -0.62366
    0x3fe90b7943575efe,     0xbfe3eb33eabe0680,     //   0.78265,  -0.62246
    0x3fe91349c88da398,     0xbfe3e15c9a2db922,     //    0.7836,  -0.62126
    0x3fe91b166fd49da2,     0xbfe3d78238c58343,     //   0.78456,  -0.62006
    0x3fe922df37f8646a,     0xbfe3cda4c80a6076,     //   0.78551,  -0.61885
    0x3fe92aa41fc5a815,     0xbfe3c3c44981c517,     //   0.78646,  -0.61765
    0x3fe932652609b1cf,     0xbfe3b9e0beb19e18,     //    0.7874,  -0.61644
    0x3fe93a22499263fc,     0xbfe3affa292050b9,     //   0.78835,  -0.61523
    0x3fe941db892e3a65,     0xbfe3a6108a54ba58,     //   0.78929,  -0.61402
    0x3fe94990e3ac4a6c,     0xbfe39c23e3d63029,     //   0.79023,  -0.61281
    0x3fe9514257dc4335,     0xbfe39234372c7f04,     //   0.79117,   -0.6116
    0x3fe958efe48e6dd7,     0xbfe3884185dfeb22,     //   0.79211,  -0.61038
    0x3fe960998893ad8c,     0xbfe37e4bd1792fe2,     //   0.79304,  -0.60917
    0x3fe9683f42bd7fe1,     0xbfe374531b817f8d,     //   0.79398,  -0.60795
    0x3fe96fe111ddfce0,     0xbfe36a576582831b,     //   0.79491,  -0.60673
    0x3fe9777ef4c7d742,     0xbfe36058b10659f3,     //   0.79584,  -0.60551
    0x3fe97f18ea4e5c9e,     0xbfe35656ff9799ae,     //   0.79676,  -0.60429
    0x3fe986aef1457594,     0xbfe34c5252c14de1,     //   0.79769,  -0.60307
    0x3fe98e410881a600,     0xbfe3424aac0ef7d6,     //   0.79861,  -0.60184
    0x3fe995cf2ed80d22,     0xbfe338400d0c8e57,     //   0.79954,  -0.60062
    0x3fe99d59631e65d5,     0xbfe32e3277467d6b,     //   0.80046,  -0.59939
    0x3fe9a4dfa42b06b2,     0xbfe32421ec49a620,     //   0.80138,  -0.59816
    0x3fe9ac61f0d4e247,     0xbfe31a0e6da35e44,     //   0.80229,  -0.59693
    0x3fe9b3e047f38741,     0xbfe30ff7fce17035,     //   0.80321,   -0.5957
    0x3fe9bb5aa85f2098,     0xbfe305de9b921a94,     //   0.80412,  -0.59447
    0x3fe9c2d110f075c3,     0xbfe2fbc24b441015,     //   0.80503,  -0.59323
    0x3fe9ca438080eadb,     0xbfe2f1a30d86773a,     //   0.80594,    -0.592
    0x3fe9d1b1f5ea80d6,     0xbfe2e780e3e8ea16,     //   0.80685,  -0.59076
    0x3fe9d91c7007d5a6,     0xbfe2dd5bcffb7616,     //   0.80775,  -0.58952
    0x3fe9e082edb42472,     0xbfe2d333d34e9bb7,     //   0.80866,  -0.58828
    0x3fe9e7e56dcb45bd,     0xbfe2c908ef734e57,     //   0.80956,  -0.58704
    0x3fe9ef43ef29af94,     0xbfe2bedb25faf3ea,     //   0.81046,   -0.5858
    0x3fe9f69e70ac75bc,     0xbfe2b4aa787764c4,     //   0.81135,  -0.58455
    0x3fe9fdf4f13149de,     0xbfe2aa76e87aeb58,     //   0.81225,  -0.58331
    0x3fea05476f967bb5,     0xbfe2a040779843fb,     //   0.81314,  -0.58206
    0x3fea0c95eabaf937,     0xbfe2960727629ca8,     //   0.81404,  -0.58081
    0x3fea13e0617e4ec7,     0xbfe28bcaf96d94ba,     //   0.81493,  -0.57956
    0x3fea1b26d2c0a75e,     0xbfe2818bef4d3cba,     //   0.81581,  -0.57831
    0x3fea22693d62ccb9,     0xbfe2774a0a961612,     //    0.8167,  -0.57706
    0x3fea29a7a0462782,     0xbfe26d054cdd12df,     //   0.81758,  -0.57581
    0x3fea30e1fa4cbf81,     0xbfe262bdb7b795a2,     //   0.81847,  -0.57455
    0x3fea38184a593bc6,     0xbfe258734cbb7110,     //   0.81935,   -0.5733
    0x3fea3f4a8f4ee2d2,     0xbfe24e260d7ee7c9,     //   0.82023,  -0.57204
    0x3fea4678c8119ac8,     0xbfe243d5fb98ac1f,     //    0.8211,  -0.57078
    0x3fea4da2f385e997,     0xbfe23983189fdfd5,     //   0.82198,  -0.56952
    0x3fea54c91090f524,     0xbfe22f2d662c13e1,     //   0.82285,  -0.56826
    0x3fea5beb1e188375,     0xbfe224d4e5d5482e,     //   0.82372,    -0.567
    0x3fea63091b02fae2,     0xbfe21a799933eb58,     //   0.82459,  -0.56573
    0x3fea6a230637623b,     0xbfe2101b81e0da78,     //   0.82546,  -0.56447
    0x3fea7138de9d60f5,     0xbfe205baa17560d6,     //   0.82632,   -0.5632
    0x3fea784aa31d3f55,     0xbfe1fb56f98b37b8,     //   0.82718,  -0.56193
    0x3fea7f58529fe69d,     0xbfe1f0f08bbc861b,     //   0.82805,  -0.56066
    0x3fea8661ec0ee133,     0xbfe1e68759a3e074,     //    0.8289,  -0.55939
    0x3fea8d676e545ad2,     0xbfe1dc1b64dc4872,     //   0.82976,  -0.55812
    0x3fea9468d85b20ae,     0xbfe1d1acaf012cc2,     //   0.83062,  -0.55685
    0x3fea9b66290ea1a3,     0xbfe1c73b39ae68c8,     //   0.83147,  -0.55557
    0x3feaa25f5f5aee60,     0xbfe1bcc706804467,     //   0.83232,  -0.55429
    0x3feaa9547a2cb98e,     0xbfe1b250171373be,     //   0.83317,  -0.55302
    0x3feab045787157ff,     0xbfe1a7d66d0516e6,     //   0.83402,  -0.55174
    0x3feab7325916c0d4,     0xbfe19d5a09f2b9b8,     //   0.83486,  -0.55046
    0x3feabe1b1b0b8dac,     0xbfe192daef7a5386,     //   0.83571,  -0.54918
    0x3feac4ffbd3efac8,     0xbfe188591f3a46e5,     //   0.83655,  -0.54789
    0x3feacbe03ea0e73b,     0xbfe17dd49ad16161,     //   0.83739,  -0.54661
    0x3fead2bc9e21d511,     0xbfe1734d63dedb49,     //   0.83822,  -0.54532
    0x3fead994dab2e979,     0xbfe168c37c025764,     //   0.83906,  -0.54404
    0x3feae068f345ecef,     0xbfe15e36e4dbe2bc,     //   0.83989,  -0.54275
    0x3feae738e6cd4b67,     0xbfe153a7a00bf453,     //   0.84073,  -0.54146
    0x3feaee04b43c1474,     0xbfe14915af336ceb,     //   0.84155,  -0.54017
    0x3feaf4cc5a85fb73,     0xbfe13e8113f396c1,     //   0.84238,  -0.53888
    0x3feafb8fd89f57b6,     0xbfe133e9cfee254e,     //   0.84321,  -0.53759
    0x3feb024f2d7d24a9,     0xbfe1294fe4c5350a,     //   0.84403,  -0.53629
    0x3feb090a58150200,     0xbfe11eb3541b4b22,     //   0.84485,    -0.535
    0x3feb0fc1575d33db,     0xbfe114141f935545,     //   0.84567,   -0.5337
    0x3feb16742a4ca2f5,     0xbfe1097248d0a956,     //   0.84649,   -0.5324
    0x3feb1d22cfdadcc6,     0xbfe0fecdd1770537,     //   0.84731,   -0.5311
    0x3feb23cd470013b4,     0xbfe0f426bb2a8e7d,     //   0.84812,   -0.5298
    0x3feb2a738eb51f33,     0xbfe0e97d078fd23b,     //   0.84893,   -0.5285
    0x3feb3115a5f37bf4,     0xbfe0ded0b84bc4b5,     //   0.84974,   -0.5272
    0x3feb37b38bb54c09,     0xbfe0d421cf03c12b,     //   0.85055,   -0.5259
    0x3feb3e4d3ef55712,     0xbfe0c9704d5d898f,     //   0.85136,  -0.52459
    0x3feb44e2beaf0a61,     0xbfe0bebc34ff4646,     //   0.85216,  -0.52328
    0x3feb4b7409de7925,     0xbfe0b405878f85ec,     //   0.85296,  -0.52198
    0x3feb52011f805c92,     0xbfe0a94c46b53d0b,     //   0.85376,  -0.52067
    0x3feb5889fe921405,     0xbfe09e907417c5e1,     //   0.85456,  -0.51936
    0x3feb5f0ea611a532,     0xbfe093d2115ee018,     //   0.85535,  -0.51804
    0x3feb658f14fdbc47,     0xbfe089112032b08c,     //   0.85615,  -0.51673
    0x3feb6c0b4a55ac17,     0xbfe07e4da23bc102,     //   0.85694,  -0.51542
    0x3feb728345196e3e,     0xbfe073879922ffed,     //   0.85773,   -0.5141
    0x3feb78f70449a34b,     0xbfe068bf0691c028,     //   0.85852,  -0.51279
    0x3feb7f6686e792ea,     0xbfe05df3ec31b8b6,     //    0.8593,  -0.51147
    0x3feb85d1cbf52c02,     0xbfe053264bad0483,     //   0.86009,  -0.51015
    0x3feb8c38d27504e9,     0xbfe0485626ae221a,     //   0.86087,  -0.50883
    0x3feb929b996a5b7f,     0xbfe03d837edff370,     //   0.86165,  -0.50751
    0x3feb98fa1fd9155e,     0xbfe032ae55edbd95,     //   0.86242,  -0.50619
    0x3feb9f5464c5bffc,     0xbfe027d6ad83287e,     //    0.8632,  -0.50486
    0x3feba5aa673590d2,     0xbfe01cfc874c3eb7,     //   0.86397,  -0.50354
    0x3febabfc262e6586,     0xbfe0121fe4f56d2c,     //   0.86474,  -0.50221
    0x3febb249a0b6c40d,     0xbfe00740c82b82e0,     //   0.86551,  -0.50089
    0x3febb892d5d5dad5,     0xbfdff8be6537615e,     //   0.86628,  -0.49956
    0x3febbed7c49380ea,     0xbfdfe2f64be7120f,     //   0.86705,  -0.49823
    0x3febc5186bf8361d,     0xbfdfcd2947c1ff57,     //   0.86781,   -0.4969
    0x3febcb54cb0d2327,     0xbfdfb7575c24d2de,     //   0.86857,  -0.49557
    0x3febd18ce0dc19d6,     0xbfdfa1808c6cf7e0,     //   0.86933,  -0.49423
    0x3febd7c0ac6f952a,     0xbfdf8ba4dbf89aba,     //   0.87009,   -0.4929
    0x3febddf02cd2b983,     0xbfdf75c44e26a852,     //   0.87084,  -0.49156
    0x3febe41b611154c1,     0xbfdf5fdee656cda3,     //    0.8716,  -0.49023
    0x3febea424837de6d,     0xbfdf49f4a7e97729,     //   0.87235,  -0.48889
    0x3febf064e15377dd,     0xbfdf3405963fd068,     //   0.87309,  -0.48755
    0x3febf6832b71ec5b,     0xbfdf1e11b4bbc35c,     //   0.87384,  -0.48621
    0x3febfc9d25a1b147,     0xbfdf081906bff7fd,     //   0.87459,  -0.48487
    0x3fec02b2cef1e641,     0xbfdef21b8fafd3b5,     //   0.87533,  -0.48353
    0x3fec08c426725549,     0xbfdedc1952ef78d5,     //   0.87607,  -0.48218
    0x3fec0ed12b3372e9,     0xbfdec61253e3c61b,     //   0.87681,  -0.48084
    0x3fec14d9dc465e58,     0xbfdeb00695f25620,     //   0.87755,  -0.47949
    0x3fec1ade38bce19b,     0xbfde99f61c817eda,     //   0.87828,  -0.47815
    0x3fec20de3fa971b0,     0xbfde83e0eaf85113,     //   0.87901,   -0.4768
    0x3fec26d9f01f2eaf,     0xbfde6dc704be97e2,     //   0.87974,  -0.47545
    0x3fec2cd14931e3f1,     0xbfde57a86d3cd824,     //   0.88047,   -0.4741
    0x3fec32c449f60831,     0xbfde418527dc4ffa,     //    0.8812,  -0.47275
    0x3fec38b2f180bdb1,     0xbfde2b5d3806f63b,     //   0.88192,   -0.4714
    0x3fec3e9d3ee7d262,     0xbfde1530a12779f4,     //   0.88264,  -0.47004
    0x3fec44833141c004,     0xbfddfeff66a941de,     //   0.88336,  -0.46869
    0x3fec4a64c7a5ac4c,     0xbfdde8c98bf86bd6,     //   0.88408,  -0.46733
    0x3fec5042012b6907,     0xbfddd28f1481cc58,     //    0.8848,  -0.46598
    0x3fec561adceb743e,     0xbfddbc5003b2edf8,     //   0.88551,  -0.46462
    0x3fec5bef59fef85a,     0xbfdda60c5cfa10d8,     //   0.88622,  -0.46326
    0x3fec61bf777fcc48,     0xbfdd8fc423c62a25,     //   0.88693,   -0.4619
    0x3fec678b3488739b,     0xbfdd79775b86e389,     //   0.88764,  -0.46054
    0x3fec6d5290341eb2,     0xbfdd632607ac9aa9,     //   0.88835,  -0.45918
    0x3fec7315899eaad7,     0xbfdd4cd02ba8609c,     //   0.88905,  -0.45781
    0x3fec78d41fe4a267,     0xbfdd3675caebf962,     //   0.88975,  -0.45645
    0x3fec7e8e52233cf3,     0xbfdd2016e8e9db5b,     //   0.89045,  -0.45508
    0x3fec84441f785f61,     0xbfdd09b389152ec1,     //   0.89115,  -0.45372
    0x3fec89f587029c13,     0xbfdcf34baee1cd21,     //   0.89184,  -0.45235
    0x3fec8fa287e13305,     0xbfdcdcdf5dc440ce,     //   0.89253,  -0.45098
    0x3fec954b213411f5,     0xbfdcc66e9931c45d,     //   0.89322,  -0.44961
    0x3fec9aef521bd480,     0xbfdcaff964a0421d,     //   0.89391,  -0.44824
    0x3feca08f19b9c449,     0xbfdc997fc3865388,     //    0.8946,  -0.44687
    0x3feca62a772fd919,     0xbfdc8301b95b40c2,     //   0.89528,   -0.4455
    0x3fecabc169a0b901,     0xbfdc6c7f4997000a,     //   0.89597,  -0.44412
    0x3fecb153f02fb87d,     0xbfdc55f877b23537,     //   0.89665,  -0.44275
    0x3fecb6e20a00da99,     0xbfdc3f6d47263129,     //   0.89732,  -0.44137
    0x3fecbc6bb638d10b,     0xbfdc28ddbb6cf145,     //     0.898,  -0.43999
    0x3fecc1f0f3fcfc5c,     0xbfdc1249d8011ee7,     //   0.89867,  -0.43862
    0x3fecc771c2736c09,     0xbfdbfbb1a05e0edc,     //   0.89935,  -0.43724
    0x3fecccee20c2de9f,     0xbfdbe51517ffc0d9,     //   0.90002,  -0.43586
    0x3fecd2660e12c1e6,     0xbfdbce744262deee,     //   0.90068,  -0.43448
    0x3fecd7d9898b32f6,     0xbfdbb7cf2304bd01,     //   0.90135,  -0.43309
    0x3fecdd489254fe65,     0xbfdba125bd63583e,     //   0.90201,  -0.43171
    0x3fece2b32799a060,     0xbfdb8a7814fd5693,     //   0.90267,  -0.43033
    0x3fece819488344ce,     0xbfdb73c62d520624,     //   0.90333,  -0.42894
    0x3feced7af43cc773,     0xbfdb5d1009e15cc0,     //   0.90399,  -0.42756
    0x3fecf2d829f1b40e,     0xbfdb4655ae2bf757,     //   0.90464,  -0.42617
    0x3fecf830e8ce467b,     0xbfdb2f971db31972,     //    0.9053,  -0.42478
    0x3fecfd852fff6ad4,     0xbfdb18d45bf8aca6,     //   0.90595,  -0.42339
    0x3fed02d4feb2bd92,     0xbfdb020d6c7f4009,     //    0.9066,    -0.422
    0x3fed082054168bac,     0xbfdaeb4252ca07ab,     //   0.90724,  -0.42061
    0x3fed0d672f59d2b9,     0xbfdad473125cdc08,     //   0.90789,  -0.41922
    0x3fed12a98fac410c,     0xbfdabd9faebc3980,     //   0.90853,  -0.41782
    0x3fed17e7743e35dc,     0xbfdaa6c82b6d3fc9,     //   0.90917,  -0.41643
    0x3fed1d20dc40c15c,     0xbfda8fec8bf5b166,     //   0.90981,  -0.41503
    0x3fed2255c6e5a4e1,     0xbfda790cd3dbf31a,     //   0.91044,  -0.41364
    0x3fed2786335f52fc,     0xbfda622906a70b63,     //   0.91107,  -0.41224
    0x3fed2cb220e0ef9f,     0xbfda4b4127dea1e4,     //   0.91171,  -0.41084
    0x3fed31d98e9e503a,     0xbfda34553b0afee5,     //   0.91234,  -0.40944
    0x3fed36fc7bcbfbdc,     0xbfda1d6543b50ac0,     //   0.91296,  -0.40804
    0x3fed3c1ae79f2b4e,     0xbfda067145664d57,     //   0.91359,  -0.40664
    0x3fed4134d14dc93a,     0xbfd9ef7943a8ed8a,     //   0.91421,  -0.40524
    0x3fed464a380e7242,     0xbfd9d87d4207b0ab,     //   0.91483,  -0.40384
    0x3fed4b5b1b187524,     0xbfd9c17d440df9f2,     //   0.91545,  -0.40243
    0x3fed506779a3d2d9,     0xbfd9aa794d47c9ee,     //   0.91606,  -0.40103
    0x3fed556f52e93eb1,     0xbfd993716141bdfe,     //   0.91668,  -0.39962
    0x3fed5a72a6221e73,     0xbfd97c6583890fc2,     //   0.91729,  -0.39822
    0x3fed5f7172888a7f,     0xbfd96555b7ab948f,     //    0.9179,  -0.39681
    0x3fed646bb7574de5,     0xbfd94e420137bce3,     //   0.91851,   -0.3954
    0x3fed696173c9e68b,     0xbfd9372a63bc93d7,     //   0.91911,  -0.39399
    0x3fed6e52a71c8547,     0xbfd9200ee2c9be97,     //   0.91972,  -0.39258
    0x3fed733f508c0dff,     0xbfd908ef81ef7bd1,     //   0.92032,  -0.39117
    0x3fed78276f5617c6,     0xbfd8f1cc44bea329,     //   0.92092,  -0.38976
    0x3fed7d0b02b8ecf9,     0xbfd8daa52ec8a4af,     //   0.92151,  -0.38835
    0x3fed81ea09f38b63,     0xbfd8c37a439f884f,     //   0.92211,  -0.38693
    0x3fed86c48445a450,     0xbfd8ac4b86d5ed44,     //    0.9227,  -0.38552
    0x3fed8b9a70ef9cb4,     0xbfd89518fbff098e,     //   0.92329,   -0.3841
    0x3fed906bcf328d46,     0xbfd87de2a6aea963,     //   0.92388,  -0.38268
    0x3fed95389e50429b,     0xbfd866a88a792ea0,     //   0.92447,  -0.38127
    0x3fed9a00dd8b3d46,     0xbfd84f6aaaf3903f,     //   0.92505,  -0.37985
    0x3fed9ec48c26b1f3,     0xbfd838290bb359c8,     //   0.92563,  -0.37843
    0x3feda383a9668988,     0xbfd820e3b04eaac4,     //   0.92621,  -0.37701
    0x3feda83e348f613b,     0xbfd8099a9c5c362d,     //   0.92679,  -0.37559
    0x3fedacf42ce68ab9,     0xbfd7f24dd37341e3,     //   0.92736,  -0.37416
    0x3fedb1a591b20c38,     0xbfd7dafd592ba621,     //   0.92794,  -0.37274
    0x3fedb6526238a09b,     0xbfd7c3a9311dcce7,     //   0.92851,  -0.37132
    0x3fedbafa9dc1b78d,     0xbfd7ac515ee2b172,     //   0.92907,  -0.36989
    0x3fedbf9e4395759a,     0xbfd794f5e613dfae,     //   0.92964,  -0.36847
    0x3fedc43d52fcb453,     0xbfd77d96ca4b73a6,     //   0.93021,  -0.36704
    0x3fedc8d7cb410260,     0xbfd766340f2418f6,     //   0.93077,  -0.36561
    0x3fedcd6dabaca3a5,     0xbfd74ecdb8390a3e,     //   0.93133,  -0.36418
    0x3fedd1fef38a915a,     0xbfd73763c9261092,     //   0.93188,  -0.36276
    0x3fedd68ba2267a25,     0xbfd71ff6458782ec,     //   0.93244,  -0.36133
    0x3feddb13b6ccc23d,     0xbfd7088530fa459e,     //   0.93299,   -0.3599
    0x3feddf9730ca837b,     0xbfd6f1108f1bc9c5,     //   0.93354,  -0.35846
    0x3fede4160f6d8d81,     0xbfd6d998638a0cb5,     //   0.93409,  -0.35703
    0x3fede890520465ce,     0xbfd6c21cb1e39771,     //   0.93464,   -0.3556
    0x3feded05f7de47da,     0xbfd6aa9d7dc77e16,     //   0.93518,  -0.35416
    0x3fedf177004b2534,     0xbfd6931acad55f51,     //   0.93573,  -0.35273
    0x3fedf5e36a9ba59c,     0xbfd67b949cad63ca,     //   0.93627,  -0.35129
    0x3fedfa4b3621271d,     0xbfd6640af6f03d9e,     //    0.9368,  -0.34986
    0x3fedfeae622dbe2b,     0xbfd64c7ddd3f27c6,     //   0.93734,  -0.34842
    0x3fee030cee1435b8,     0xbfd634ed533be58e,     //   0.93787,  -0.34698
    0x3fee0766d9280f54,     0xbfd61d595c88c203,     //    0.9384,  -0.34554
    0x3fee0bbc22bd8349,     0xbfd605c1fcc88f63,     //   0.93893,   -0.3441
    0x3fee100cca2980ac,     0xbfd5ee27379ea693,     //   0.93946,  -0.34266
    0x3fee1458cec1ad83,     0xbfd5d68910aee686,     //   0.93998,  -0.34122
    0x3fee18a02fdc66d9,     0xbfd5bee78b9db3b6,     //   0.94051,  -0.33978
    0x3fee1ce2ecd0c0d8,     0xbfd5a742ac0ff78d,     //   0.94103,  -0.33833
    0x3fee212104f686e5,     0xbfd58f9a75ab1fdd,     //   0.94154,  -0.33689
    0x3fee255a77a63bb8,     0xbfd577eeec151e47,     //   0.94206,  -0.33545
    0x3fee298f4439197a,     0xbfd5604012f467b4,     //   0.94257,    -0.334
    0x3fee2dbf6a0911d9,     0xbfd5488dedeff3be,     //   0.94308,  -0.33255
    0x3fee31eae870ce25,     0xbfd530d880af3c24,     //   0.94359,  -0.33111
    0x3fee3611becbaf69,     0xbfd5191fceda3c35,     //    0.9441,  -0.32966
    0x3fee3a33ec75ce85,     0xbfd50163dc197047,     //    0.9446,  -0.32821
    0x3fee3e5170cbfc46,     0xbfd4e9a4ac15d520,     //   0.94511,  -0.32676
    0x3fee426a4b2bc17e,     0xbfd4d1e24278e76a,     //   0.94561,  -0.32531
    0x3fee467e7af35f23,     0xbfd4ba1ca2eca31c,     //   0.94611,  -0.32386
    0x3fee4a8dff81ce5e,     0xbfd4a253d11b82f3,     //    0.9466,  -0.32241
    0x3fee4e98d836c0af,     0xbfd48a87d0b07fd7,     //   0.94709,  -0.32096
    0x3fee529f04729ffc,     0xbfd472b8a5571054,     //   0.94759,   -0.3195
    0x3fee56a083968eb1,     0xbfd45ae652bb2800,     //   0.94807,  -0.31805
    0x3fee5a9d550467d3,     0xbfd44310dc8936f0,     //   0.94856,  -0.31659
    0x3fee5e95781ebf1c,     0xbfd42b38466e2928,     //   0.94905,  -0.31514
    0x3fee6288ec48e112,     0xbfd4135c94176602,     //   0.94953,  -0.31368
    0x3fee6677b0e6d31e,     0xbfd3fb7dc932cfa4,     //   0.95001,  -0.31222
    0x3fee6a61c55d53a7,     0xbfd3e39be96ec271,     //   0.95049,  -0.31077
    0x3fee6e472911da27,     0xbfd3cbb6f87a146e,     //   0.95096,  -0.30931
    0x3fee7227db6a9744,     0xbfd3b3cefa0414b7,     //   0.95144,  -0.30785
    0x3fee7603dbce74e9,     0xbfd39be3f1bc8aef,     //   0.95191,  -0.30639
    0x3fee79db29a5165a,     0xbfd383f5e353b6aa,     //   0.95238,  -0.30493
    0x3fee7dadc456d850,     0xbfd36c04d27a4edf,     //   0.95284,  -0.30347
    0x3fee817bab4cd10d,     0xbfd35410c2e18152,     //   0.95331,  -0.30201
    0x3fee8544ddf0d075,     0xbfd33c19b83af207,     //   0.95377,  -0.30054
    0x3fee89095bad6025,     0xbfd3241fb638baaf,     //   0.95423,  -0.29908
    0x3fee8cc923edc388,     0xbfd30c22c08d6a13,     //   0.95469,  -0.29762
    0x3fee9084361df7f3,     0xbfd2f422daec0386,     //   0.95514,  -0.29615
    0x3fee943a91aab4b4,     0xbfd2dc200907fe51,     //   0.95559,  -0.29469
    0x3fee97ec36016b30,     0xbfd2c41a4e954520,     //   0.95605,  -0.29322
    0x3fee9b99229046f8,     0xbfd2ac11af483572,     //   0.95649,  -0.29175
    0x3fee9f4156c62dda,     0xbfd294062ed59f05,     //   0.95694,  -0.29028
    0x3feea2e4d212c000,     0xbfd27bf7d0f2c346,     //   0.95738,  -0.28882
    0x3feea68393e65800,     0xbfd263e6995554ba,     //   0.95783,  -0.28735
    0x3feeaa1d9bb20af3,     0xbfd24bd28bb37672,     //   0.95827,  -0.28588
    0x3feeadb2e8e7a88e,     0xbfd233bbabc3bb72,     //    0.9587,  -0.28441
    0x3feeb1437af9bb34,     0xbfd21ba1fd3d2623,     //   0.95914,  -0.28294
    0x3feeb4cf515b8811,     0xbfd2038583d727bd,     //   0.95957,  -0.28146
    0x3feeb8566b810f2a,     0xbfd1eb6643499fbb,     //      0.96,  -0.27999
    0x3feebbd8c8df0b74,     0xbfd1d3443f4cdb3d,     //   0.96043,  -0.27852
    0x3feebf5668eaf2ef,     0xbfd1bb1f7b999480,     //   0.96086,  -0.27705
    0x3feec2cf4b1af6b2,     0xbfd1a2f7fbe8f243,     //   0.96128,  -0.27557
    0x3feec6436ee60309,     0xbfd18acdc3f4873a,     //    0.9617,   -0.2741
    0x3feec9b2d3c3bf84,     0xbfd172a0d7765177,     //   0.96212,  -0.27262
    0x3feecd1d792c8f10,     0xbfd15a713a28b9d9,     //   0.96254,  -0.27115
    0x3feed0835e999009,     0xbfd1423eefc69378,     //   0.96295,  -0.26967
    0x3feed3e483849c51,     0xbfd12a09fc0b1b12,     //   0.96337,  -0.26819
    0x3feed740e7684963,     0xbfd111d262b1f677,     //   0.96378,  -0.26671
    0x3feeda9889bfe86a,     0xbfd0f998277733f7,     //   0.96418,  -0.26523
    0x3feeddeb6a078651,     0xbfd0e15b4e1749cd,     //   0.96459,  -0.26375
    0x3feee13987bbebdc,     0xbfd0c91bda4f158d,     //   0.96499,  -0.26227
    0x3feee482e25a9dbc,     0xbfd0b0d9cfdbdb90,     //   0.96539,  -0.26079
    0x3feee7c77961dc9e,     0xbfd09895327b465e,     //   0.96579,  -0.25931
    0x3feeeb074c50a544,     0xbfd0804e05eb661e,     //   0.96619,  -0.25783
    0x3feeee425aa6b09a,     0xbfd068044deab002,     //   0.96658,  -0.25635
    0x3feef178a3e473c2,     0xbfd04fb80e37fdae,     //   0.96698,  -0.25487
    0x3feef4aa278b2032,     0xbfd037694a928cac,     //   0.96737,  -0.25338
    0x3feef7d6e51ca3c0,     0xbfd01f1806b9fdd2,     //   0.96775,   -0.2519
    0x3feefafedc1ba8b7,     0xbfd006c4466e54af,     //   0.96814,  -0.25041
    0x3feefe220c0b95ec,     0xbfcfdcdc1adfedf8,     //   0.96852,  -0.24893
    0x3fef014074708ed3,     0xbfcfac2abeff57ff,     //    0.9689,  -0.24744
    0x3fef045a14cf738c,     0xbfcf7b7480bd3801,     //   0.96928,  -0.24596
    0x3fef076eecade0fa,     0xbfcf4ab9679c9f5c,     //   0.96966,  -0.24447
    0x3fef0a7efb9230d7,     0xbfcf19f97b215f1a,     //   0.97003,  -0.24298
    0x3fef0d8a410379c5,     0xbfcee934c2d006c7,     //    0.9704,  -0.24149
    0x3fef1090bc898f5f,     0xbfceb86b462de348,     //   0.97077,     -0.24
    0x3fef13926dad024e,     0xbfce879d0cc0fdaf,     //   0.97114,  -0.23851
    0x3fef168f53f7205d,     0xbfce56ca1e101a1b,     //    0.9715,  -0.23702
    0x3fef19876ef1f486,     0xbfce25f281a2b684,     //   0.97187,  -0.23553
    0x3fef1c7abe284708,     0xbfcdf5163f01099a,     //   0.97223,  -0.23404
    0x3fef1f6941259d7a,     0xbfcdc4355db40195,     //   0.97258,  -0.23255
    0x3fef2252f7763ada,     0xbfcd934fe5454311,     //   0.97294,  -0.23106
    0x3fef2537e0a71f9f,     0xbfcd6265dd3f27e3,     //   0.97329,  -0.22957
    0x3fef2817fc4609ce,     0xbfcd31774d2cbdee,     //   0.97364,  -0.22807
    0x3fef2af349e17507,     0xbfcd00843c99c5f9,     //   0.97399,  -0.22658
    0x3fef2dc9c9089a9d,     0xbfcccf8cb312b286,     //   0.97434,  -0.22508
    0x3fef309b794b719f,     0xbfcc9e90b824a6a9,     //   0.97468,  -0.22359
    0x3fef33685a3aaef0,     0xbfcc6d90535d74dc,     //   0.97503,  -0.22209
    0x3fef36306b67c556,     0xbfcc3c8b8c4b9dd7,     //   0.97536,   -0.2206
    0x3fef38f3ac64e589,     0xbfcc0b826a7e4f63,     //    0.9757,   -0.2191
    0x3fef3bb21cc4fe47,     0xbfcbda74f5856330,     //   0.97604,   -0.2176
    0x3fef3e6bbc1bbc65,     0xbfcba96334f15dad,     //   0.97637,  -0.21611
    0x3fef412089fd8adc,     0xbfcb784d30536cda,     //    0.9767,  -0.21461
    0x3fef43d085ff92dd,     0xbfcb4732ef3d6722,     //   0.97703,  -0.21311
    0x3fef467bafb7bbe0,     0xbfcb16147941ca2a,     //   0.97735,  -0.21161
    0x3fef492206bcabb4,     0xbfcae4f1d5f3b9ab,     //   0.97768,  -0.21011
    0x3fef4bc38aa5c694,     0xbfcab3cb0ce6fe44,     //     0.978,  -0.20861
    0x3fef4e603b0b2f2d,     0xbfca82a025b00451,     //   0.97832,  -0.20711
    0x3fef50f81785c6b9,     0xbfca517127e3dabc,     //   0.97863,  -0.20561
    0x3fef538b1faf2d07,     0xbfca203e1b1831da,     //   0.97895,  -0.20411
    0x3fef56195321c090,     0xbfc9ef0706e35a35,     //   0.97926,  -0.20261
    0x3fef58a2b1789e84,     0xbfc9bdcbf2dc4366,     //   0.97957,   -0.2011
    0x3fef5b273a4fa2d9,     0xbfc98c8ce69a7aec,     //   0.97988,   -0.1996
    0x3fef5da6ed43685d,     0xbfc95b49e9b62af9,     //   0.98018,   -0.1981
    0x3fef6021c9f148c2,     0xbfc92a0303c8194f,     //   0.98048,  -0.19659
    0x3fef6297cff75cb0,     0xbfc8f8b83c69a60a,     //   0.98079,  -0.19509
    0x3fef6508fef47bd5,     0xbfc8c7699b34ca7e,     //   0.98108,  -0.19359
    0x3fef677556883cee,     0xbfc8961727c41804,     //   0.98138,  -0.19208
    0x3fef69dcd652f5de,     0xbfc864c0e9b2b6cf,     //   0.98167,  -0.19057
    0x3fef6c3f7df5bbb7,     0xbfc83366e89c64c5,     //   0.98196,  -0.18907
    0x3fef6e9d4d1262ca,     0xbfc802092c1d744b,     //   0.98225,  -0.18756
    0x3fef70f6434b7eb7,     0xbfc7d0a7bbd2cb1b,     //   0.98254,  -0.18606
    0x3fef734a60446279,     0xbfc79f429f59e11d,     //   0.98282,  -0.18455
    0x3fef7599a3a12077,     0xbfc76dd9de50bf31,     //   0.98311,  -0.18304
    0x3fef77e40d068a90,     0xbfc73c6d8055fe0a,     //   0.98339,  -0.18153
    0x3fef7a299c1a322a,     0xbfc70afd8d08c4ff,     //   0.98366,  -0.18002
    0x3fef7c6a50826840,     0xbfc6d98a0c08c8da,     //   0.98394,  -0.17851
    0x3fef7ea629e63d6e,     0xbfc6a81304f64ab2,     //   0.98421,    -0.177
    0x3fef80dd27ed8204,     0xbfc676987f7216b8,     //   0.98448,  -0.17549
    0x3fef830f4a40c60c,     0xbfc6451a831d830d,     //   0.98475,  -0.17398
    0x3fef853c9089595e,     0xbfc61399179a6e94,     //   0.98501,  -0.17247
    0x3fef8764fa714ba9,     0xbfc5e214448b3fc6,     //   0.98528,  -0.17096
    0x3fef898887a36c84,     0xbfc5b08c1192e381,     //   0.98554,  -0.16945
    0x3fef8ba737cb4b78,     0xbfc57f008654cbde,     //    0.9858,  -0.16794
    0x3fef8dc10a95380d,     0xbfc54d71aa74ef02,     //   0.98605,  -0.16643
    0x3fef8fd5ffae41db,     0xbfc51bdf8597c5f2,     //   0.98631,  -0.16491
    0x3fef91e616c43891,     0xbfc4ea4a1f624b61,     //   0.98656,   -0.1634
    0x3fef93f14f85ac08,     0xbfc4b8b17f79fa88,     //   0.98681,  -0.16189
    0x3fef95f7a9a1ec47,     0xbfc48715ad84cdf5,     //   0.98706,  -0.16037
    0x3fef97f924c9099b,     0xbfc45576b1293e5a,     //    0.9873,  -0.15886
    0x3fef99f5c0abd496,     0xbfc423d4920e4166,     //   0.98754,  -0.15734
    0x3fef9bed7cfbde29,     0xbfc3f22f57db4893,     //   0.98778,  -0.15583
    0x3fef9de0596b77a3,     0xbfc3c0870a383ff6,     //   0.98802,  -0.15431
    0x3fef9fce55adb2c8,     0xbfc38edbb0cd8d14,     //   0.98826,   -0.1528
    0x3fefa1b7717661d5,     0xbfc35d2d53440db2,     //   0.98849,  -0.15128
    0x3fefa39bac7a1791,     0xbfc32b7bf94516a7,     //   0.98872,  -0.14976
    0x3fefa57b066e2754,     0xbfc2f9c7aa7a72af,     //   0.98895,  -0.14825
    0x3fefa7557f08a517,     0xbfc2c8106e8e613a,     //   0.98918,  -0.14673
    0x3fefa92b1600657c,     0xbfc296564d2b953e,     //    0.9894,  -0.14521
    0x3fefaafbcb0cfddc,     0xbfc264994dfd340a,     //   0.98962,   -0.1437
    0x3fefacc79de6c44f,     0xbfc232d978aed413,     //   0.98984,  -0.14218
    0x3fefae8e8e46cfbb,     0xbfc20116d4ec7bce,     //   0.99006,  -0.14066
    0x3fefb0509be6f7db,     0xbfc1cf516a62a077,     //   0.99027,  -0.13914
    0x3fefb20dc681d54d,     0xbfc19d8940be24e7,     //   0.99049,  -0.13762
    0x3fefb3c60dd2c199,     0xbfc16bbe5fac5865,     //    0.9907,   -0.1361
    0x3fefb5797195d741,     0xbfc139f0cedaf576,     //    0.9909,  -0.13458
    0x3fefb727f187f1c7,     0xbfc1082095f820b0,     //   0.99111,  -0.13306
    0x3fefb8d18d66adb7,     0xbfc0d64dbcb26786,     //   0.99131,  -0.13154
    0x3fefba7644f068b5,     0xbfc0a4784ab8bf1d,     //   0.99151,  -0.13002
    0x3fefbc1617e44186,     0xbfc072a047ba831d,     //   0.99171,   -0.1285
    0x3fefbdb106021816,     0xbfc040c5bb67747e,     //   0.99191,  -0.12698
    0x3fefbf470f0a8d88,     0xbfc00ee8ad6fb85b,     //    0.9921,  -0.12545
    0x3fefc0d832bf043a,     0xbfbfba124b07ad85,     //   0.99229,  -0.12393
    0x3fefc26470e19fd3,     0xbfbf564e56a9730e,     //   0.99248,  -0.12241
    0x3fefc3ebc935454c,     0xbfbef2858d27561b,     //   0.99267,  -0.12089
    0x3fefc56e3b7d9af6,     0xbfbe8eb7fde4aa3e,     //   0.99285,  -0.11937
    0x3fefc6ebc77f0887,     0xbfbe2ae5b8457f77,     //   0.99303,  -0.11784
    0x3fefc8646cfeb721,     0xbfbdc70ecbae9fc8,     //   0.99321,  -0.11632
    0x3fefc9d82bc2915e,     0xbfbd633347858ce4,     //   0.99339,  -0.11479
    0x3fefcb4703914354,     0xbfbcff533b307dc1,     //   0.99356,  -0.11327
    0x3fefccb0f4323aa3,     0xbfbc9b6eb6165c42,     //   0.99374,  -0.11175
    0x3fefce15fd6da67b,     0xbfbc3785c79ec2d5,     //   0.99391,  -0.11022
    0x3fefcf761f0c77a3,     0xbfbbd3987f31fa0e,     //   0.99407,   -0.1087
    0x3fefd0d158d86087,     0xbfbb6fa6ec38f64c,     //   0.99424,  -0.10717
    0x3fefd227aa9bd53b,     0xbfbb0bb11e1d5559,     //    0.9944,  -0.10565
    0x3fefd37914220b84,     0xbfbaa7b724495c04,     //   0.99456,  -0.10412
    0x3fefd4c59536fae4,     0xbfba43b90e27f3c4,     //   0.99472,   -0.1026
    0x3fefd60d2da75c9e,     0xbfb9dfb6eb24a85c,     //   0.99488,  -0.10107
    0x3fefd74fdd40abbf,     0xbfb97bb0caaba56f,     //   0.99503, -0.099544
    0x3fefd88da3d12526,     0xbfb917a6bc29b42c,     //   0.99518, -0.098017
    0x3fefd9c68127c78c,     0xbfb8b398cf0c38e0,     //   0.99533,  -0.09649
    0x3fefdafa7514538c,     0xbfb84f8712c130a0,     //   0.99548, -0.094963
    0x3fefdc297f674ba9,     0xbfb7eb7196b72ee4,     //   0.99563, -0.093436
    0x3fefdd539ff1f456,     0xbfb787586a5d5b21,     //   0.99577, -0.091909
    0x3fefde78d68653fd,     0xbfb7233b9d236e71,     //   0.99591, -0.090381
    0x3fefdf9922f73307,     0xbfb6bf1b3e79b129,     //   0.99604, -0.088854
    0x3fefe0b485181be3,     0xbfb65af75dd0f87b,     //   0.99618, -0.087326
    0x3fefe1cafcbd5b09,     0xbfb5f6d00a9aa419,     //   0.99631, -0.085797
    0x3fefe2dc89bbff08,     0xbfb592a554489bc8,     //   0.99644, -0.084269
    0x3fefe3e92be9d886,     0xbfb52e774a4d4d0a,     //   0.99657,  -0.08274
    0x3fefe4f0e31d7a4a,     0xbfb4ca45fc1ba8b6,     //    0.9967, -0.081211
    0x3fefe5f3af2e3940,     0xbfb4661179272096,     //   0.99682, -0.079682
    0x3fefe6f18ff42c84,     0xbfb401d9d0e3a507,     //   0.99694, -0.078153
    0x3fefe7ea85482d60,     0xbfb39d9f12c5a299,     //   0.99706, -0.076624
    0x3fefe8de8f03d75c,     0xbfb339614e41ffa5,     //   0.99718, -0.075094
    0x3fefe9cdad01883a,     0xbfb2d52092ce19f6,     //   0.99729, -0.073565
    0x3fefeab7df1c6005,     0xbfb270dcefdfc45b,     //    0.9974, -0.072035
    0x3fefeb9d2530410f,     0xbfb20c9674ed444c,     //   0.99751, -0.070505
    0x3fefec7d7f19cffc,     0xbfb1a84d316d4f8a,     //   0.99762, -0.068974
    0x3fefed58ecb673c4,     0xbfb1440134d709b2,     //   0.99772, -0.067444
    0x3fefee2f6de455ba,     0xbfb0dfb28ea201e6,     //   0.99783, -0.065913
    0x3fefef0102826191,     0xbfb07b614e463064,     //   0.99793, -0.064383
    0x3fefefcdaa704562,     0xbfb0170d833bf421,     //   0.99802, -0.062852
    0x3feff095658e71ad,     0xbfaf656e79f820e0,     //   0.99812, -0.061321
    0x3feff15833be1965,     0xbfae9cbd15ff5527,     //   0.99821,  -0.05979
    0x3feff21614e131ed,     0xbfadd406f9808ec8,     //    0.9983, -0.058258
    0x3feff2cf08da7321,     0xbfad0b4c436f91d0,     //   0.99839, -0.056727
    0x3feff3830f8d575c,     0xbfac428d12c0d7e3,     //   0.99848, -0.055195
    0x3feff43228de1b77,     0xbfab79c986698b78,     //   0.99856, -0.053664
    0x3feff4dc54b1bed3,     0xbfaab101bd5f8317,     //   0.99864, -0.052132
    0x3feff58192ee0358,     0xbfa9e835d6993c87,     //   0.99872,   -0.0506
    0x3feff621e3796d7e,     0xbfa91f65f10dd814,     //    0.9988, -0.049068
    0x3feff6bd463b444d,     0xbfa856922bb513c1,     //   0.99887, -0.047535
    0x3feff753bb1b9164,     0xbfa78dbaa5874685,     //   0.99894, -0.046003
    0x3feff7e5420320f9,     0xbfa6c4df7d7d5b84,     //   0.99901, -0.044471
    0x3feff871dadb81df,     0xbfa5fc00d290cd43,     //   0.99908, -0.042938
    0x3feff8f9858f058b,     0xbfa5331ec3bba0eb,     //   0.99914, -0.041406
    0x3feff97c4208c014,     0xbfa46a396ff86179,     //    0.9992, -0.039873
    0x3feff9fa10348837,     0xbfa3a150f6421afc,     //   0.99926,  -0.03834
    0x3feffa72effef75d,     0xbfa2d865759455cd,     //   0.99932, -0.036807
    0x3feffae6e1556998,     0xbfa20f770ceb11c6,     //   0.99938, -0.035274
    0x3feffb55e425fdae,     0xbfa14685db42c17e,     //   0.99943, -0.033741
    0x3feffbbff85f9515,     0xbfa07d91ff984580,     //   0.99948, -0.032208
    0x3feffc251df1d3f8,     0xbf9f693731d1cf01,     //   0.99953, -0.030675
    0x3feffc8554cd213a,     0xbf9dd7458c64ab39,     //   0.99958, -0.029142
    0x3feffce09ce2a679,     0xbf9c454f4ce53b1c,     //   0.99962, -0.027608
    0x3feffd36f624500c,     0xbf9ab354b1504fca,     //   0.99966, -0.026075
    0x3feffd886084cd0d,     0xbf992155f7a3667e,     //    0.9997, -0.024541
    0x3feffdd4dbf78f52,     0xbf978f535ddc9f03,     //   0.99974, -0.023008
    0x3feffe1c6870cb77,     0xbf95fd4d21fab226,     //   0.99977, -0.021474
    0x3feffe5f05e578db,     0xbf946b4381fce81c,     //    0.9998,  -0.01994
    0x3feffe9cb44b51a1,     0xbf92d936bbe30efd,     //   0.99983, -0.018407
    0x3feffed57398d2b7,     0xbf9147270dad7132,     //   0.99986, -0.016873
    0x3fefff0943c53bd1,     0xbf8f6a296ab997ca,     //   0.99988, -0.015339
    0x3fefff3824c88f6f,     0xbf8c45ffe1e48ad9,     //    0.9999, -0.013805
    0x3fefff62169b92db,     0xbf8921d1fcdec784,     //   0.99992, -0.012272
    0x3fefff871937ce2f,     0xbf85fda037ac05e0,     //   0.99994, -0.010738
    0x3fefffa72c978c4f,     0xbf82d96b0e509703,     //   0.99996,-0.0092038
    0x3fefffc250b5daef,     0xbf7f6a65f9a2a3c5,     //   0.99997,-0.0076698
    0x3fefffd8858e8a92,     0xbf7921f0fe670071,     //   0.99998,-0.0061359
    0x3fefffe9cb1e2e8d,     0xbf72d97822f996bc,     //   0.99999,-0.0046019
    0x3feffff621621d02,     0xbf6921f8becca4ba,     //         1, -0.003068
    0x3feffffd88586ee6,     0xbf5921faaee6472d,     //         1, -0.001534


};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_16)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 16, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_16[32] = {
    1.000000000f,  0.000000000f,
    0.923879533f,  0.382683432f,
    0.707106781f,  0.707106781f,
    0.382683432f,  0.923879533f,
    0.000000000f,  1.000000000f,
   -0.382683432f,  0.923879533f,
   -0.707106781f,  0.707106781f,
   -0.923879533f,  0.382683432f,
   -1.000000000f,  0.000000000f,
   -0.923879533f, -0.382683432f,
   -0.707106781f, -0.707106781f,
   -0.382683432f, -0.923879533f,
   -0.000000000f, -1.000000000f,
    0.382683432f, -0.923879533f,
    0.707106781f, -0.707106781f,
    0.923879533f, -0.382683432f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_32)

/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 32, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_32[64] = {
    1.000000000f,  0.000000000f,
    0.980785280f,  0.195090322f,
    0.923879533f,  0.382683432f,
    0.831469612f,  0.555570233f,
    0.707106781f,  0.707106781f,
    0.555570233f,  0.831469612f,
    0.382683432f,  0.923879533f,
    0.195090322f,  0.980785280f,
    0.000000000f,  1.000000000f,
   -0.195090322f,  0.980785280f,
   -0.382683432f,  0.923879533f,
   -0.555570233f,  0.831469612f,
   -0.707106781f,  0.707106781f,
   -0.831469612f,  0.555570233f,
   -0.923879533f,  0.382683432f,
   -0.980785280f,  0.195090322f,
   -1.000000000f,  0.000000000f,
   -0.980785280f, -0.195090322f,
   -0.923879533f, -0.382683432f,
   -0.831469612f, -0.555570233f,
   -0.707106781f, -0.707106781f,
   -0.555570233f, -0.831469612f,
   -0.382683432f, -0.923879533f,
   -0.195090322f, -0.980785280f,
   -0.000000000f, -1.000000000f,
    0.195090322f, -0.980785280f,
    0.382683432f, -0.923879533f,
    0.555570233f, -0.831469612f,
    0.707106781f, -0.707106781f,
    0.831469612f, -0.555570233f,
    0.923879533f, -0.382683432f,
    0.980785280f, -0.195090322f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_64)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 64, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_64[128] = {
    1.000000000f,  0.000000000f,
    0.995184727f,  0.098017140f,
    0.980785280f,  0.195090322f,
    0.956940336f,  0.290284677f,
    0.923879533f,  0.382683432f,
    0.881921264f,  0.471396737f,
    0.831469612f,  0.555570233f,
    0.773010453f,  0.634393284f,
    0.707106781f,  0.707106781f,
    0.634393284f,  0.773010453f,
    0.555570233f,  0.831469612f,
    0.471396737f,  0.881921264f,
    0.382683432f,  0.923879533f,
    0.290284677f,  0.956940336f,
    0.195090322f,  0.980785280f,
    0.098017140f,  0.995184727f,
    0.000000000f,  1.000000000f,
   -0.098017140f,  0.995184727f,
   -0.195090322f,  0.980785280f,
   -0.290284677f,  0.956940336f,
   -0.382683432f,  0.923879533f,
   -0.471396737f,  0.881921264f,
   -0.555570233f,  0.831469612f,
   -0.634393284f,  0.773010453f,
   -0.707106781f,  0.707106781f,
   -0.773010453f,  0.634393284f,
   -0.831469612f,  0.555570233f,
   -0.881921264f,  0.471396737f,
   -0.923879533f,  0.382683432f,
   -0.956940336f,  0.290284677f,
   -0.980785280f,  0.195090322f,
   -0.995184727f,  0.098017140f,
   -1.000000000f,  0.000000000f,
   -0.995184727f, -0.098017140f,
   -0.980785280f, -0.195090322f,
   -0.956940336f, -0.290284677f,
   -0.923879533f, -0.382683432f,
   -0.881921264f, -0.471396737f,
   -0.831469612f, -0.555570233f,
   -0.773010453f, -0.634393284f,
   -0.707106781f, -0.707106781f,
   -0.634393284f, -0.773010453f,
   -0.555570233f, -0.831469612f,
   -0.471396737f, -0.881921264f,
   -0.382683432f, -0.923879533f,
   -0.290284677f, -0.956940336f,
   -0.195090322f, -0.980785280f,
   -0.098017140f, -0.995184727f,
   -0.000000000f, -1.000000000f,
    0.098017140f, -0.995184727f,
    0.195090322f, -0.980785280f,
    0.290284677f, -0.956940336f,
    0.382683432f, -0.923879533f,
    0.471396737f, -0.881921264f,
    0.555570233f, -0.831469612f,
    0.634393284f, -0.773010453f,
    0.707106781f, -0.707106781f,
    0.773010453f, -0.634393284f,
    0.831469612f, -0.555570233f,
    0.881921264f, -0.471396737f,
    0.923879533f, -0.382683432f,
    0.956940336f, -0.290284677f,
    0.980785280f, -0.195090322f,
    0.995184727f, -0.098017140f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_128)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 128, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_128[256] = {
    1.000000000f,  0.000000000f,
    0.998795456f,  0.049067674f,
    0.995184727f,  0.098017140f,
    0.989176510f,  0.146730474f,
    0.980785280f,  0.195090322f,
    0.970031253f,  0.242980180f,
    0.956940336f,  0.290284677f,
    0.941544065f,  0.336889853f,
    0.923879533f,  0.382683432f,
    0.903989293f,  0.427555093f,
    0.881921264f,  0.471396737f,
    0.857728610f,  0.514102744f,
    0.831469612f,  0.555570233f,
    0.803207531f,  0.595699304f,
    0.773010453f,  0.634393284f,
    0.740951125f,  0.671558955f,
    0.707106781f,  0.707106781f,
    0.671558955f,  0.740951125f,
    0.634393284f,  0.773010453f,
    0.595699304f,  0.803207531f,
    0.555570233f,  0.831469612f,
    0.514102744f,  0.857728610f,
    0.471396737f,  0.881921264f,
    0.427555093f,  0.903989293f,
    0.382683432f,  0.923879533f,
    0.336889853f,  0.941544065f,
    0.290284677f,  0.956940336f,
    0.242980180f,  0.970031253f,
    0.195090322f,  0.980785280f,
    0.146730474f,  0.989176510f,
    0.098017140f,  0.995184727f,
    0.049067674f,  0.998795456f,
    0.000000000f,  1.000000000f,
   -0.049067674f,  0.998795456f,
   -0.098017140f,  0.995184727f,
   -0.146730474f,  0.989176510f,
   -0.195090322f,  0.980785280f,
   -0.242980180f,  0.970031253f,
   -0.290284677f,  0.956940336f,
   -0.336889853f,  0.941544065f,
   -0.382683432f,  0.923879533f,
   -0.427555093f,  0.903989293f,
   -0.471396737f,  0.881921264f,
   -0.514102744f,  0.857728610f,
   -0.555570233f,  0.831469612f,
   -0.595699304f,  0.803207531f,
   -0.634393284f,  0.773010453f,
   -0.671558955f,  0.740951125f,
   -0.707106781f,  0.707106781f,
   -0.740951125f,  0.671558955f,
   -0.773010453f,  0.634393284f,
   -0.803207531f,  0.595699304f,
   -0.831469612f,  0.555570233f,
   -0.857728610f,  0.514102744f,
   -0.881921264f,  0.471396737f,
   -0.903989293f,  0.427555093f,
   -0.923879533f,  0.382683432f,
   -0.941544065f,  0.336889853f,
   -0.956940336f,  0.290284677f,
   -0.970031253f,  0.242980180f,
   -0.980785280f,  0.195090322f,
   -0.989176510f,  0.146730474f,
   -0.995184727f,  0.098017140f,
   -0.998795456f,  0.049067674f,
   -1.000000000f,  0.000000000f,
   -0.998795456f, -0.049067674f,
   -0.995184727f, -0.098017140f,
   -0.989176510f, -0.146730474f,
   -0.980785280f, -0.195090322f,
   -0.970031253f, -0.242980180f,
   -0.956940336f, -0.290284677f,
   -0.941544065f, -0.336889853f,
   -0.923879533f, -0.382683432f,
   -0.903989293f, -0.427555093f,
   -0.881921264f, -0.471396737f,
   -0.857728610f, -0.514102744f,
   -0.831469612f, -0.555570233f,
   -0.803207531f, -0.595699304f,
   -0.773010453f, -0.634393284f,
   -0.740951125f, -0.671558955f,
   -0.707106781f, -0.707106781f,
   -0.671558955f, -0.740951125f,
   -0.634393284f, -0.773010453f,
   -0.595699304f, -0.803207531f,
   -0.555570233f, -0.831469612f,
   -0.514102744f, -0.857728610f,
   -0.471396737f, -0.881921264f,
   -0.427555093f, -0.903989293f,
   -0.382683432f, -0.923879533f,
   -0.336889853f, -0.941544065f,
   -0.290284677f, -0.956940336f,
   -0.242980180f, -0.970031253f,
   -0.195090322f, -0.980785280f,
   -0.146730474f, -0.989176510f,
   -0.098017140f, -0.995184727f,
   -0.049067674f, -0.998795456f,
   -0.000000000f, -1.000000000f,
    0.049067674f, -0.998795456f,
    0.098017140f, -0.995184727f,
    0.146730474f, -0.989176510f,
    0.195090322f, -0.980785280f,
    0.242980180f, -0.970031253f,
    0.290284677f, -0.956940336f,
    0.336889853f, -0.941544065f,
    0.382683432f, -0.923879533f,
    0.427555093f, -0.903989293f,
    0.471396737f, -0.881921264f,
    0.514102744f, -0.857728610f,
    0.555570233f, -0.831469612f,
    0.595699304f, -0.803207531f,
    0.634393284f, -0.773010453f,
    0.671558955f, -0.740951125f,
    0.707106781f, -0.707106781f,
    0.740951125f, -0.671558955f,
    0.773010453f, -0.634393284f,
    0.803207531f, -0.595699304f,
    0.831469612f, -0.555570233f,
    0.857728610f, -0.514102744f,
    0.881921264f, -0.471396737f,
    0.903989293f, -0.427555093f,
    0.923879533f, -0.382683432f,
    0.941544065f, -0.336889853f,
    0.956940336f, -0.290284677f,
    0.970031253f, -0.242980180f,
    0.980785280f, -0.195090322f,
    0.989176510f, -0.146730474f,
    0.995184727f, -0.098017140f,
    0.998795456f, -0.049067674f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_256)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 256, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_256[512] = {
    1.000000000f,  0.000000000f,
    0.999698819f,  0.024541229f,
    0.998795456f,  0.049067674f,
    0.997290457f,  0.073564564f,
    0.995184727f,  0.098017140f,
    0.992479535f,  0.122410675f,
    0.989176510f,  0.146730474f,
    0.985277642f,  0.170961889f,
    0.980785280f,  0.195090322f,
    0.975702130f,  0.219101240f,
    0.970031253f,  0.242980180f,
    0.963776066f,  0.266712757f,
    0.956940336f,  0.290284677f,
    0.949528181f,  0.313681740f,
    0.941544065f,  0.336889853f,
    0.932992799f,  0.359895037f,
    0.923879533f,  0.382683432f,
    0.914209756f,  0.405241314f,
    0.903989293f,  0.427555093f,
    0.893224301f,  0.449611330f,
    0.881921264f,  0.471396737f,
    0.870086991f,  0.492898192f,
    0.857728610f,  0.514102744f,
    0.844853565f,  0.534997620f,
    0.831469612f,  0.555570233f,
    0.817584813f,  0.575808191f,
    0.803207531f,  0.595699304f,
    0.788346428f,  0.615231591f,
    0.773010453f,  0.634393284f,
    0.757208847f,  0.653172843f,
    0.740951125f,  0.671558955f,
    0.724247083f,  0.689540545f,
    0.707106781f,  0.707106781f,
    0.689540545f,  0.724247083f,
    0.671558955f,  0.740951125f,
    0.653172843f,  0.757208847f,
    0.634393284f,  0.773010453f,
    0.615231591f,  0.788346428f,
    0.595699304f,  0.803207531f,
    0.575808191f,  0.817584813f,
    0.555570233f,  0.831469612f,
    0.534997620f,  0.844853565f,
    0.514102744f,  0.857728610f,
    0.492898192f,  0.870086991f,
    0.471396737f,  0.881921264f,
    0.449611330f,  0.893224301f,
    0.427555093f,  0.903989293f,
    0.405241314f,  0.914209756f,
    0.382683432f,  0.923879533f,
    0.359895037f,  0.932992799f,
    0.336889853f,  0.941544065f,
    0.313681740f,  0.949528181f,
    0.290284677f,  0.956940336f,
    0.266712757f,  0.963776066f,
    0.242980180f,  0.970031253f,
    0.219101240f,  0.975702130f,
    0.195090322f,  0.980785280f,
    0.170961889f,  0.985277642f,
    0.146730474f,  0.989176510f,
    0.122410675f,  0.992479535f,
    0.098017140f,  0.995184727f,
    0.073564564f,  0.997290457f,
    0.049067674f,  0.998795456f,
    0.024541229f,  0.999698819f,
    0.000000000f,  1.000000000f,
   -0.024541229f,  0.999698819f,
   -0.049067674f,  0.998795456f,
   -0.073564564f,  0.997290457f,
   -0.098017140f,  0.995184727f,
   -0.122410675f,  0.992479535f,
   -0.146730474f,  0.989176510f,
   -0.170961889f,  0.985277642f,
   -0.195090322f,  0.980785280f,
   -0.219101240f,  0.975702130f,
   -0.242980180f,  0.970031253f,
   -0.266712757f,  0.963776066f,
   -0.290284677f,  0.956940336f,
   -0.313681740f,  0.949528181f,
   -0.336889853f,  0.941544065f,
   -0.359895037f,  0.932992799f,
   -0.382683432f,  0.923879533f,
   -0.405241314f,  0.914209756f,
   -0.427555093f,  0.903989293f,
   -0.449611330f,  0.893224301f,
   -0.471396737f,  0.881921264f,
   -0.492898192f,  0.870086991f,
   -0.514102744f,  0.857728610f,
   -0.534997620f,  0.844853565f,
   -0.555570233f,  0.831469612f,
   -0.575808191f,  0.817584813f,
   -0.595699304f,  0.803207531f,
   -0.615231591f,  0.788346428f,
   -0.634393284f,  0.773010453f,
   -0.653172843f,  0.757208847f,
   -0.671558955f,  0.740951125f,
   -0.689540545f,  0.724247083f,
   -0.707106781f,  0.707106781f,
   -0.724247083f,  0.689540545f,
   -0.740951125f,  0.671558955f,
   -0.757208847f,  0.653172843f,
   -0.773010453f,  0.634393284f,
   -0.788346428f,  0.615231591f,
   -0.803207531f,  0.595699304f,
   -0.817584813f,  0.575808191f,
   -0.831469612f,  0.555570233f,
   -0.844853565f,  0.534997620f,
   -0.857728610f,  0.514102744f,
   -0.870086991f,  0.492898192f,
   -0.881921264f,  0.471396737f,
   -0.893224301f,  0.449611330f,
   -0.903989293f,  0.427555093f,
   -0.914209756f,  0.405241314f,
   -0.923879533f,  0.382683432f,
   -0.932992799f,  0.359895037f,
   -0.941544065f,  0.336889853f,
   -0.949528181f,  0.313681740f,
   -0.956940336f,  0.290284677f,
   -0.963776066f,  0.266712757f,
   -0.970031253f,  0.242980180f,
   -0.975702130f,  0.219101240f,
   -0.980785280f,  0.195090322f,
   -0.985277642f,  0.170961889f,
   -0.989176510f,  0.146730474f,
   -0.992479535f,  0.122410675f,
   -0.995184727f,  0.098017140f,
   -0.997290457f,  0.073564564f,
   -0.998795456f,  0.049067674f,
   -0.999698819f,  0.024541229f,
   -1.000000000f,  0.000000000f,
   -0.999698819f, -0.024541229f,
   -0.998795456f, -0.049067674f,
   -0.997290457f, -0.073564564f,
   -0.995184727f, -0.098017140f,
   -0.992479535f, -0.122410675f,
   -0.989176510f, -0.146730474f,
   -0.985277642f, -0.170961889f,
   -0.980785280f, -0.195090322f,
   -0.975702130f, -0.219101240f,
   -0.970031253f, -0.242980180f,
   -0.963776066f, -0.266712757f,
   -0.956940336f, -0.290284677f,
   -0.949528181f, -0.313681740f,
   -0.941544065f, -0.336889853f,
   -0.932992799f, -0.359895037f,
   -0.923879533f, -0.382683432f,
   -0.914209756f, -0.405241314f,
   -0.903989293f, -0.427555093f,
   -0.893224301f, -0.449611330f,
   -0.881921264f, -0.471396737f,
   -0.870086991f, -0.492898192f,
   -0.857728610f, -0.514102744f,
   -0.844853565f, -0.534997620f,
   -0.831469612f, -0.555570233f,
   -0.817584813f, -0.575808191f,
   -0.803207531f, -0.595699304f,
   -0.788346428f, -0.615231591f,
   -0.773010453f, -0.634393284f,
   -0.757208847f, -0.653172843f,
   -0.740951125f, -0.671558955f,
   -0.724247083f, -0.689540545f,
   -0.707106781f, -0.707106781f,
   -0.689540545f, -0.724247083f,
   -0.671558955f, -0.740951125f,
   -0.653172843f, -0.757208847f,
   -0.634393284f, -0.773010453f,
   -0.615231591f, -0.788346428f,
   -0.595699304f, -0.803207531f,
   -0.575808191f, -0.817584813f,
   -0.555570233f, -0.831469612f,
   -0.534997620f, -0.844853565f,
   -0.514102744f, -0.857728610f,
   -0.492898192f, -0.870086991f,
   -0.471396737f, -0.881921264f,
   -0.449611330f, -0.893224301f,
   -0.427555093f, -0.903989293f,
   -0.405241314f, -0.914209756f,
   -0.382683432f, -0.923879533f,
   -0.359895037f, -0.932992799f,
   -0.336889853f, -0.941544065f,
   -0.313681740f, -0.949528181f,
   -0.290284677f, -0.956940336f,
   -0.266712757f, -0.963776066f,
   -0.242980180f, -0.970031253f,
   -0.219101240f, -0.975702130f,
   -0.195090322f, -0.980785280f,
   -0.170961889f, -0.985277642f,
   -0.146730474f, -0.989176510f,
   -0.122410675f, -0.992479535f,
   -0.098017140f, -0.995184727f,
   -0.073564564f, -0.997290457f,
   -0.049067674f, -0.998795456f,
   -0.024541229f, -0.999698819f,
   -0.000000000f, -1.000000000f,
    0.024541229f, -0.999698819f,
    0.049067674f, -0.998795456f,
    0.073564564f, -0.997290457f,
    0.098017140f, -0.995184727f,
    0.122410675f, -0.992479535f,
    0.146730474f, -0.989176510f,
    0.170961889f, -0.985277642f,
    0.195090322f, -0.980785280f,
    0.219101240f, -0.975702130f,
    0.242980180f, -0.970031253f,
    0.266712757f, -0.963776066f,
    0.290284677f, -0.956940336f,
    0.313681740f, -0.949528181f,
    0.336889853f, -0.941544065f,
    0.359895037f, -0.932992799f,
    0.382683432f, -0.923879533f,
    0.405241314f, -0.914209756f,
    0.427555093f, -0.903989293f,
    0.449611330f, -0.893224301f,
    0.471396737f, -0.881921264f,
    0.492898192f, -0.870086991f,
    0.514102744f, -0.857728610f,
    0.534997620f, -0.844853565f,
    0.555570233f, -0.831469612f,
    0.575808191f, -0.817584813f,
    0.595699304f, -0.803207531f,
    0.615231591f, -0.788346428f,
    0.634393284f, -0.773010453f,
    0.653172843f, -0.757208847f,
    0.671558955f, -0.740951125f,
    0.689540545f, -0.724247083f,
    0.707106781f, -0.707106781f,
    0.724247083f, -0.689540545f,
    0.740951125f, -0.671558955f,
    0.757208847f, -0.653172843f,
    0.773010453f, -0.634393284f,
    0.788346428f, -0.615231591f,
    0.803207531f, -0.595699304f,
    0.817584813f, -0.575808191f,
    0.831469612f, -0.555570233f,
    0.844853565f, -0.534997620f,
    0.857728610f, -0.514102744f,
    0.870086991f, -0.492898192f,
    0.881921264f, -0.471396737f,
    0.893224301f, -0.449611330f,
    0.903989293f, -0.427555093f,
    0.914209756f, -0.405241314f,
    0.923879533f, -0.382683432f,
    0.932992799f, -0.359895037f,
    0.941544065f, -0.336889853f,
    0.949528181f, -0.313681740f,
    0.956940336f, -0.290284677f,
    0.963776066f, -0.266712757f,
    0.970031253f, -0.242980180f,
    0.975702130f, -0.219101240f,
    0.980785280f, -0.195090322f,
    0.985277642f, -0.170961889f,
    0.989176510f, -0.146730474f,
    0.992479535f, -0.122410675f,
    0.995184727f, -0.098017140f,
    0.997290457f, -0.073564564f,
    0.998795456f, -0.049067674f,
    0.999698819f, -0.024541229f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_512)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 512, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_512[1024] = {
    1.000000000f,  0.000000000f,
    0.999924702f,  0.012271538f,
    0.999698819f,  0.024541229f,
    0.999322385f,  0.036807223f,
    0.998795456f,  0.049067674f,
    0.998118113f,  0.061320736f,
    0.997290457f,  0.073564564f,
    0.996312612f,  0.085797312f,
    0.995184727f,  0.098017140f,
    0.993906970f,  0.110222207f,
    0.992479535f,  0.122410675f,
    0.990902635f,  0.134580709f,
    0.989176510f,  0.146730474f,
    0.987301418f,  0.158858143f,
    0.985277642f,  0.170961889f,
    0.983105487f,  0.183039888f,
    0.980785280f,  0.195090322f,
    0.978317371f,  0.207111376f,
    0.975702130f,  0.219101240f,
    0.972939952f,  0.231058108f,
    0.970031253f,  0.242980180f,
    0.966976471f,  0.254865660f,
    0.963776066f,  0.266712757f,
    0.960430519f,  0.278519689f,
    0.956940336f,  0.290284677f,
    0.953306040f,  0.302005949f,
    0.949528181f,  0.313681740f,
    0.945607325f,  0.325310292f,
    0.941544065f,  0.336889853f,
    0.937339012f,  0.348418680f,
    0.932992799f,  0.359895037f,
    0.928506080f,  0.371317194f,
    0.923879533f,  0.382683432f,
    0.919113852f,  0.393992040f,
    0.914209756f,  0.405241314f,
    0.909167983f,  0.416429560f,
    0.903989293f,  0.427555093f,
    0.898674466f,  0.438616239f,
    0.893224301f,  0.449611330f,
    0.887639620f,  0.460538711f,
    0.881921264f,  0.471396737f,
    0.876070094f,  0.482183772f,
    0.870086991f,  0.492898192f,
    0.863972856f,  0.503538384f,
    0.857728610f,  0.514102744f,
    0.851355193f,  0.524589683f,
    0.844853565f,  0.534997620f,
    0.838224706f,  0.545324988f,
    0.831469612f,  0.555570233f,
    0.824589303f,  0.565731811f,
    0.817584813f,  0.575808191f,
    0.810457198f,  0.585797857f,
    0.803207531f,  0.595699304f,
    0.795836905f,  0.605511041f,
    0.788346428f,  0.615231591f,
    0.780737229f,  0.624859488f,
    0.773010453f,  0.634393284f,
    0.765167266f,  0.643831543f,
    0.757208847f,  0.653172843f,
    0.749136395f,  0.662415778f,
    0.740951125f,  0.671558955f,
    0.732654272f,  0.680600998f,
    0.724247083f,  0.689540545f,
    0.715730825f,  0.698376249f,
    0.707106781f,  0.707106781f,
    0.698376249f,  0.715730825f,
    0.689540545f,  0.724247083f,
    0.680600998f,  0.732654272f,
    0.671558955f,  0.740951125f,
    0.662415778f,  0.749136395f,
    0.653172843f,  0.757208847f,
    0.643831543f,  0.765167266f,
    0.634393284f,  0.773010453f,
    0.624859488f,  0.780737229f,
    0.615231591f,  0.788346428f,
    0.605511041f,  0.795836905f,
    0.595699304f,  0.803207531f,
    0.585797857f,  0.810457198f,
    0.575808191f,  0.817584813f,
    0.565731811f,  0.824589303f,
    0.555570233f,  0.831469612f,
    0.545324988f,  0.838224706f,
    0.534997620f,  0.844853565f,
    0.524589683f,  0.851355193f,
    0.514102744f,  0.857728610f,
    0.503538384f,  0.863972856f,
    0.492898192f,  0.870086991f,
    0.482183772f,  0.876070094f,
    0.471396737f,  0.881921264f,
    0.460538711f,  0.887639620f,
    0.449611330f,  0.893224301f,
    0.438616239f,  0.898674466f,
    0.427555093f,  0.903989293f,
    0.416429560f,  0.909167983f,
    0.405241314f,  0.914209756f,
    0.393992040f,  0.919113852f,
    0.382683432f,  0.923879533f,
    0.371317194f,  0.928506080f,
    0.359895037f,  0.932992799f,
    0.348418680f,  0.937339012f,
    0.336889853f,  0.941544065f,
    0.325310292f,  0.945607325f,
    0.313681740f,  0.949528181f,
    0.302005949f,  0.953306040f,
    0.290284677f,  0.956940336f,
    0.278519689f,  0.960430519f,
    0.266712757f,  0.963776066f,
    0.254865660f,  0.966976471f,
    0.242980180f,  0.970031253f,
    0.231058108f,  0.972939952f,
    0.219101240f,  0.975702130f,
    0.207111376f,  0.978317371f,
    0.195090322f,  0.980785280f,
    0.183039888f,  0.983105487f,
    0.170961889f,  0.985277642f,
    0.158858143f,  0.987301418f,
    0.146730474f,  0.989176510f,
    0.134580709f,  0.990902635f,
    0.122410675f,  0.992479535f,
    0.110222207f,  0.993906970f,
    0.098017140f,  0.995184727f,
    0.085797312f,  0.996312612f,
    0.073564564f,  0.997290457f,
    0.061320736f,  0.998118113f,
    0.049067674f,  0.998795456f,
    0.036807223f,  0.999322385f,
    0.024541229f,  0.999698819f,
    0.012271538f,  0.999924702f,
    0.000000000f,  1.000000000f,
   -0.012271538f,  0.999924702f,
   -0.024541229f,  0.999698819f,
   -0.036807223f,  0.999322385f,
   -0.049067674f,  0.998795456f,
   -0.061320736f,  0.998118113f,
   -0.073564564f,  0.997290457f,
   -0.085797312f,  0.996312612f,
   -0.098017140f,  0.995184727f,
   -0.110222207f,  0.993906970f,
   -0.122410675f,  0.992479535f,
   -0.134580709f,  0.990902635f,
   -0.146730474f,  0.989176510f,
   -0.158858143f,  0.987301418f,
   -0.170961889f,  0.985277642f,
   -0.183039888f,  0.983105487f,
   -0.195090322f,  0.980785280f,
   -0.207111376f,  0.978317371f,
   -0.219101240f,  0.975702130f,
   -0.231058108f,  0.972939952f,
   -0.242980180f,  0.970031253f,
   -0.254865660f,  0.966976471f,
   -0.266712757f,  0.963776066f,
   -0.278519689f,  0.960430519f,
   -0.290284677f,  0.956940336f,
   -0.302005949f,  0.953306040f,
   -0.313681740f,  0.949528181f,
   -0.325310292f,  0.945607325f,
   -0.336889853f,  0.941544065f,
   -0.348418680f,  0.937339012f,
   -0.359895037f,  0.932992799f,
   -0.371317194f,  0.928506080f,
   -0.382683432f,  0.923879533f,
   -0.393992040f,  0.919113852f,
   -0.405241314f,  0.914209756f,
   -0.416429560f,  0.909167983f,
   -0.427555093f,  0.903989293f,
   -0.438616239f,  0.898674466f,
   -0.449611330f,  0.893224301f,
   -0.460538711f,  0.887639620f,
   -0.471396737f,  0.881921264f,
   -0.482183772f,  0.876070094f,
   -0.492898192f,  0.870086991f,
   -0.503538384f,  0.863972856f,
   -0.514102744f,  0.857728610f,
   -0.524589683f,  0.851355193f,
   -0.534997620f,  0.844853565f,
   -0.545324988f,  0.838224706f,
   -0.555570233f,  0.831469612f,
   -0.565731811f,  0.824589303f,
   -0.575808191f,  0.817584813f,
   -0.585797857f,  0.810457198f,
   -0.595699304f,  0.803207531f,
   -0.605511041f,  0.795836905f,
   -0.615231591f,  0.788346428f,
   -0.624859488f,  0.780737229f,
   -0.634393284f,  0.773010453f,
   -0.643831543f,  0.765167266f,
   -0.653172843f,  0.757208847f,
   -0.662415778f,  0.749136395f,
   -0.671558955f,  0.740951125f,
   -0.680600998f,  0.732654272f,
   -0.689540545f,  0.724247083f,
   -0.698376249f,  0.715730825f,
   -0.707106781f,  0.707106781f,
   -0.715730825f,  0.698376249f,
   -0.724247083f,  0.689540545f,
   -0.732654272f,  0.680600998f,
   -0.740951125f,  0.671558955f,
   -0.749136395f,  0.662415778f,
   -0.757208847f,  0.653172843f,
   -0.765167266f,  0.643831543f,
   -0.773010453f,  0.634393284f,
   -0.780737229f,  0.624859488f,
   -0.788346428f,  0.615231591f,
   -0.795836905f,  0.605511041f,
   -0.803207531f,  0.595699304f,
   -0.810457198f,  0.585797857f,
   -0.817584813f,  0.575808191f,
   -0.824589303f,  0.565731811f,
   -0.831469612f,  0.555570233f,
   -0.838224706f,  0.545324988f,
   -0.844853565f,  0.534997620f,
   -0.851355193f,  0.524589683f,
   -0.857728610f,  0.514102744f,
   -0.863972856f,  0.503538384f,
   -0.870086991f,  0.492898192f,
   -0.876070094f,  0.482183772f,
   -0.881921264f,  0.471396737f,
   -0.887639620f,  0.460538711f,
   -0.893224301f,  0.449611330f,
   -0.898674466f,  0.438616239f,
   -0.903989293f,  0.427555093f,
   -0.909167983f,  0.416429560f,
   -0.914209756f,  0.405241314f,
   -0.919113852f,  0.393992040f,
   -0.923879533f,  0.382683432f,
   -0.928506080f,  0.371317194f,
   -0.932992799f,  0.359895037f,
   -0.937339012f,  0.348418680f,
   -0.941544065f,  0.336889853f,
   -0.945607325f,  0.325310292f,
   -0.949528181f,  0.313681740f,
   -0.953306040f,  0.302005949f,
   -0.956940336f,  0.290284677f,
   -0.960430519f,  0.278519689f,
   -0.963776066f,  0.266712757f,
   -0.966976471f,  0.254865660f,
   -0.970031253f,  0.242980180f,
   -0.972939952f,  0.231058108f,
   -0.975702130f,  0.219101240f,
   -0.978317371f,  0.207111376f,
   -0.980785280f,  0.195090322f,
   -0.983105487f,  0.183039888f,
   -0.985277642f,  0.170961889f,
   -0.987301418f,  0.158858143f,
   -0.989176510f,  0.146730474f,
   -0.990902635f,  0.134580709f,
   -0.992479535f,  0.122410675f,
   -0.993906970f,  0.110222207f,
   -0.995184727f,  0.098017140f,
   -0.996312612f,  0.085797312f,
   -0.997290457f,  0.073564564f,
   -0.998118113f,  0.061320736f,
   -0.998795456f,  0.049067674f,
   -0.999322385f,  0.036807223f,
   -0.999698819f,  0.024541229f,
   -0.999924702f,  0.012271538f,
   -1.000000000f,  0.000000000f,
   -0.999924702f, -0.012271538f,
   -0.999698819f, -0.024541229f,
   -0.999322385f, -0.036807223f,
   -0.998795456f, -0.049067674f,
   -0.998118113f, -0.061320736f,
   -0.997290457f, -0.073564564f,
   -0.996312612f, -0.085797312f,
   -0.995184727f, -0.098017140f,
   -0.993906970f, -0.110222207f,
   -0.992479535f, -0.122410675f,
   -0.990902635f, -0.134580709f,
   -0.989176510f, -0.146730474f,
   -0.987301418f, -0.158858143f,
   -0.985277642f, -0.170961889f,
   -0.983105487f, -0.183039888f,
   -0.980785280f, -0.195090322f,
   -0.978317371f, -0.207111376f,
   -0.975702130f, -0.219101240f,
   -0.972939952f, -0.231058108f,
   -0.970031253f, -0.242980180f,
   -0.966976471f, -0.254865660f,
   -0.963776066f, -0.266712757f,
   -0.960430519f, -0.278519689f,
   -0.956940336f, -0.290284677f,
   -0.953306040f, -0.302005949f,
   -0.949528181f, -0.313681740f,
   -0.945607325f, -0.325310292f,
   -0.941544065f, -0.336889853f,
   -0.937339012f, -0.348418680f,
   -0.932992799f, -0.359895037f,
   -0.928506080f, -0.371317194f,
   -0.923879533f, -0.382683432f,
   -0.919113852f, -0.393992040f,
   -0.914209756f, -0.405241314f,
   -0.909167983f, -0.416429560f,
   -0.903989293f, -0.427555093f,
   -0.898674466f, -0.438616239f,
   -0.893224301f, -0.449611330f,
   -0.887639620f, -0.460538711f,
   -0.881921264f, -0.471396737f,
   -0.876070094f, -0.482183772f,
   -0.870086991f, -0.492898192f,
   -0.863972856f, -0.503538384f,
   -0.857728610f, -0.514102744f,
   -0.851355193f, -0.524589683f,
   -0.844853565f, -0.534997620f,
   -0.838224706f, -0.545324988f,
   -0.831469612f, -0.555570233f,
   -0.824589303f, -0.565731811f,
   -0.817584813f, -0.575808191f,
   -0.810457198f, -0.585797857f,
   -0.803207531f, -0.595699304f,
   -0.795836905f, -0.605511041f,
   -0.788346428f, -0.615231591f,
   -0.780737229f, -0.624859488f,
   -0.773010453f, -0.634393284f,
   -0.765167266f, -0.643831543f,
   -0.757208847f, -0.653172843f,
   -0.749136395f, -0.662415778f,
   -0.740951125f, -0.671558955f,
   -0.732654272f, -0.680600998f,
   -0.724247083f, -0.689540545f,
   -0.715730825f, -0.698376249f,
   -0.707106781f, -0.707106781f,
   -0.698376249f, -0.715730825f,
   -0.689540545f, -0.724247083f,
   -0.680600998f, -0.732654272f,
   -0.671558955f, -0.740951125f,
   -0.662415778f, -0.749136395f,
   -0.653172843f, -0.757208847f,
   -0.643831543f, -0.765167266f,
   -0.634393284f, -0.773010453f,
   -0.624859488f, -0.780737229f,
   -0.615231591f, -0.788346428f,
   -0.605511041f, -0.795836905f,
   -0.595699304f, -0.803207531f,
   -0.585797857f, -0.810457198f,
   -0.575808191f, -0.817584813f,
   -0.565731811f, -0.824589303f,
   -0.555570233f, -0.831469612f,
   -0.545324988f, -0.838224706f,
   -0.534997620f, -0.844853565f,
   -0.524589683f, -0.851355193f,
   -0.514102744f, -0.857728610f,
   -0.503538384f, -0.863972856f,
   -0.492898192f, -0.870086991f,
   -0.482183772f, -0.876070094f,
   -0.471396737f, -0.881921264f,
   -0.460538711f, -0.887639620f,
   -0.449611330f, -0.893224301f,
   -0.438616239f, -0.898674466f,
   -0.427555093f, -0.903989293f,
   -0.416429560f, -0.909167983f,
   -0.405241314f, -0.914209756f,
   -0.393992040f, -0.919113852f,
   -0.382683432f, -0.923879533f,
   -0.371317194f, -0.928506080f,
   -0.359895037f, -0.932992799f,
   -0.348418680f, -0.937339012f,
   -0.336889853f, -0.941544065f,
   -0.325310292f, -0.945607325f,
   -0.313681740f, -0.949528181f,
   -0.302005949f, -0.953306040f,
   -0.290284677f, -0.956940336f,
   -0.278519689f, -0.960430519f,
   -0.266712757f, -0.963776066f,
   -0.254865660f, -0.966976471f,
   -0.242980180f, -0.970031253f,
   -0.231058108f, -0.972939952f,
   -0.219101240f, -0.975702130f,
   -0.207111376f, -0.978317371f,
   -0.195090322f, -0.980785280f,
   -0.183039888f, -0.983105487f,
   -0.170961889f, -0.985277642f,
   -0.158858143f, -0.987301418f,
   -0.146730474f, -0.989176510f,
   -0.134580709f, -0.990902635f,
   -0.122410675f, -0.992479535f,
   -0.110222207f, -0.993906970f,
   -0.098017140f, -0.995184727f,
   -0.085797312f, -0.996312612f,
   -0.073564564f, -0.997290457f,
   -0.061320736f, -0.998118113f,
   -0.049067674f, -0.998795456f,
   -0.036807223f, -0.999322385f,
   -0.024541229f, -0.999698819f,
   -0.012271538f, -0.999924702f,
   -0.000000000f, -1.000000000f,
    0.012271538f, -0.999924702f,
    0.024541229f, -0.999698819f,
    0.036807223f, -0.999322385f,
    0.049067674f, -0.998795456f,
    0.061320736f, -0.998118113f,
    0.073564564f, -0.997290457f,
    0.085797312f, -0.996312612f,
    0.098017140f, -0.995184727f,
    0.110222207f, -0.993906970f,
    0.122410675f, -0.992479535f,
    0.134580709f, -0.990902635f,
    0.146730474f, -0.989176510f,
    0.158858143f, -0.987301418f,
    0.170961889f, -0.985277642f,
    0.183039888f, -0.983105487f,
    0.195090322f, -0.980785280f,
    0.207111376f, -0.978317371f,
    0.219101240f, -0.975702130f,
    0.231058108f, -0.972939952f,
    0.242980180f, -0.970031253f,
    0.254865660f, -0.966976471f,
    0.266712757f, -0.963776066f,
    0.278519689f, -0.960430519f,
    0.290284677f, -0.956940336f,
    0.302005949f, -0.953306040f,
    0.313681740f, -0.949528181f,
    0.325310292f, -0.945607325f,
    0.336889853f, -0.941544065f,
    0.348418680f, -0.937339012f,
    0.359895037f, -0.932992799f,
    0.371317194f, -0.928506080f,
    0.382683432f, -0.923879533f,
    0.393992040f, -0.919113852f,
    0.405241314f, -0.914209756f,
    0.416429560f, -0.909167983f,
    0.427555093f, -0.903989293f,
    0.438616239f, -0.898674466f,
    0.449611330f, -0.893224301f,
    0.460538711f, -0.887639620f,
    0.471396737f, -0.881921264f,
    0.482183772f, -0.876070094f,
    0.492898192f, -0.870086991f,
    0.503538384f, -0.863972856f,
    0.514102744f, -0.857728610f,
    0.524589683f, -0.851355193f,
    0.534997620f, -0.844853565f,
    0.545324988f, -0.838224706f,
    0.555570233f, -0.831469612f,
    0.565731811f, -0.824589303f,
    0.575808191f, -0.817584813f,
    0.585797857f, -0.810457198f,
    0.595699304f, -0.803207531f,
    0.605511041f, -0.795836905f,
    0.615231591f, -0.788346428f,
    0.624859488f, -0.780737229f,
    0.634393284f, -0.773010453f,
    0.643831543f, -0.765167266f,
    0.653172843f, -0.757208847f,
    0.662415778f, -0.749136395f,
    0.671558955f, -0.740951125f,
    0.680600998f, -0.732654272f,
    0.689540545f, -0.724247083f,
    0.698376249f, -0.715730825f,
    0.707106781f, -0.707106781f,
    0.715730825f, -0.698376249f,
    0.724247083f, -0.689540545f,
    0.732654272f, -0.680600998f,
    0.740951125f, -0.671558955f,
    0.749136395f, -0.662415778f,
    0.757208847f, -0.653172843f,
    0.765167266f, -0.643831543f,
    0.773010453f, -0.634393284f,
    0.780737229f, -0.624859488f,
    0.788346428f, -0.615231591f,
    0.795836905f, -0.605511041f,
    0.803207531f, -0.595699304f,
    0.810457198f, -0.585797857f,
    0.817584813f, -0.575808191f,
    0.824589303f, -0.565731811f,
    0.831469612f, -0.555570233f,
    0.838224706f, -0.545324988f,
    0.844853565f, -0.534997620f,
    0.851355193f, -0.524589683f,
    0.857728610f, -0.514102744f,
    0.863972856f, -0.503538384f,
    0.870086991f, -0.492898192f,
    0.876070094f, -0.482183772f,
    0.881921264f, -0.471396737f,
    0.887639620f, -0.460538711f,
    0.893224301f, -0.449611330f,
    0.898674466f, -0.438616239f,
    0.903989293f, -0.427555093f,
    0.909167983f, -0.416429560f,
    0.914209756f, -0.405241314f,
    0.919113852f, -0.393992040f,
    0.923879533f, -0.382683432f,
    0.928506080f, -0.371317194f,
    0.932992799f, -0.359895037f,
    0.937339012f, -0.348418680f,
    0.941544065f, -0.336889853f,
    0.945607325f, -0.325310292f,
    0.949528181f, -0.313681740f,
    0.953306040f, -0.302005949f,
    0.956940336f, -0.290284677f,
    0.960430519f, -0.278519689f,
    0.963776066f, -0.266712757f,
    0.966976471f, -0.254865660f,
    0.970031253f, -0.242980180f,
    0.972939952f, -0.231058108f,
    0.975702130f, -0.219101240f,
    0.978317371f, -0.207111376f,
    0.980785280f, -0.195090322f,
    0.983105487f, -0.183039888f,
    0.985277642f, -0.170961889f,
    0.987301418f, -0.158858143f,
    0.989176510f, -0.146730474f,
    0.990902635f, -0.134580709f,
    0.992479535f, -0.122410675f,
    0.993906970f, -0.110222207f,
    0.995184727f, -0.098017140f,
    0.996312612f, -0.085797312f,
    0.997290457f, -0.073564564f,
    0.998118113f, -0.061320736f,
    0.998795456f, -0.049067674f,
    0.999322385f, -0.036807223f,
    0.999698819f, -0.024541229f,
    0.999924702f, -0.012271538f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_1024)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 1024, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_1024[2048] = {
    1.000000000f,  0.000000000f,
    0.999981175f,  0.006135885f,
    0.999924702f,  0.012271538f,
    0.999830582f,  0.018406730f,
    0.999698819f,  0.024541229f,
    0.999529418f,  0.030674803f,
    0.999322385f,  0.036807223f,
    0.999077728f,  0.042938257f,
    0.998795456f,  0.049067674f,
    0.998475581f,  0.055195244f,
    0.998118113f,  0.061320736f,
    0.997723067f,  0.067443920f,
    0.997290457f,  0.073564564f,
    0.996820299f,  0.079682438f,
    0.996312612f,  0.085797312f,
    0.995767414f,  0.091908956f,
    0.995184727f,  0.098017140f,
    0.994564571f,  0.104121634f,
    0.993906970f,  0.110222207f,
    0.993211949f,  0.116318631f,
    0.992479535f,  0.122410675f,
    0.991709754f,  0.128498111f,
    0.990902635f,  0.134580709f,
    0.990058210f,  0.140658239f,
    0.989176510f,  0.146730474f,
    0.988257568f,  0.152797185f,
    0.987301418f,  0.158858143f,
    0.986308097f,  0.164913120f,
    0.985277642f,  0.170961889f,
    0.984210092f,  0.177004220f,
    0.983105487f,  0.183039888f,
    0.981963869f,  0.189068664f,
    0.980785280f,  0.195090322f,
    0.979569766f,  0.201104635f,
    0.978317371f,  0.207111376f,
    0.977028143f,  0.213110320f,
    0.975702130f,  0.219101240f,
    0.974339383f,  0.225083911f,
    0.972939952f,  0.231058108f,
    0.971503891f,  0.237023606f,
    0.970031253f,  0.242980180f,
    0.968522094f,  0.248927606f,
    0.966976471f,  0.254865660f,
    0.965394442f,  0.260794118f,
    0.963776066f,  0.266712757f,
    0.962121404f,  0.272621355f,
    0.960430519f,  0.278519689f,
    0.958703475f,  0.284407537f,
    0.956940336f,  0.290284677f,
    0.955141168f,  0.296150888f,
    0.953306040f,  0.302005949f,
    0.951435021f,  0.307849640f,
    0.949528181f,  0.313681740f,
    0.947585591f,  0.319502031f,
    0.945607325f,  0.325310292f,
    0.943593458f,  0.331106306f,
    0.941544065f,  0.336889853f,
    0.939459224f,  0.342660717f,
    0.937339012f,  0.348418680f,
    0.935183510f,  0.354163525f,
    0.932992799f,  0.359895037f,
    0.930766961f,  0.365612998f,
    0.928506080f,  0.371317194f,
    0.926210242f,  0.377007410f,
    0.923879533f,  0.382683432f,
    0.921514039f,  0.388345047f,
    0.919113852f,  0.393992040f,
    0.916679060f,  0.399624200f,
    0.914209756f,  0.405241314f,
    0.911706032f,  0.410843171f,
    0.909167983f,  0.416429560f,
    0.906595705f,  0.422000271f,
    0.903989293f,  0.427555093f,
    0.901348847f,  0.433093819f,
    0.898674466f,  0.438616239f,
    0.895966250f,  0.444122145f,
    0.893224301f,  0.449611330f,
    0.890448723f,  0.455083587f,
    0.887639620f,  0.460538711f,
    0.884797098f,  0.465976496f,
    0.881921264f,  0.471396737f,
    0.879012226f,  0.476799230f,
    0.876070094f,  0.482183772f,
    0.873094978f,  0.487550160f,
    0.870086991f,  0.492898192f,
    0.867046246f,  0.498227667f,
    0.863972856f,  0.503538384f,
    0.860866939f,  0.508830143f,
    0.857728610f,  0.514102744f,
    0.854557988f,  0.519355990f,
    0.851355193f,  0.524589683f,
    0.848120345f,  0.529803625f,
    0.844853565f,  0.534997620f,
    0.841554977f,  0.540171473f,
    0.838224706f,  0.545324988f,
    0.834862875f,  0.550457973f,
    0.831469612f,  0.555570233f,
    0.828045045f,  0.560661576f,
    0.824589303f,  0.565731811f,
    0.821102515f,  0.570780746f,
    0.817584813f,  0.575808191f,
    0.814036330f,  0.580813958f,
    0.810457198f,  0.585797857f,
    0.806847554f,  0.590759702f,
    0.803207531f,  0.595699304f,
    0.799537269f,  0.600616479f,
    0.795836905f,  0.605511041f,
    0.792106577f,  0.610382806f,
    0.788346428f,  0.615231591f,
    0.784556597f,  0.620057212f,
    0.780737229f,  0.624859488f,
    0.776888466f,  0.629638239f,
    0.773010453f,  0.634393284f,
    0.769103338f,  0.639124445f,
    0.765167266f,  0.643831543f,
    0.761202385f,  0.648514401f,
    0.757208847f,  0.653172843f,
    0.753186799f,  0.657806693f,
    0.749136395f,  0.662415778f,
    0.745057785f,  0.666999922f,
    0.740951125f,  0.671558955f,
    0.736816569f,  0.676092704f,
    0.732654272f,  0.680600998f,
    0.728464390f,  0.685083668f,
    0.724247083f,  0.689540545f,
    0.720002508f,  0.693971461f,
    0.715730825f,  0.698376249f,
    0.711432196f,  0.702754744f,
    0.707106781f,  0.707106781f,
    0.702754744f,  0.711432196f,
    0.698376249f,  0.715730825f,
    0.693971461f,  0.720002508f,
    0.689540545f,  0.724247083f,
    0.685083668f,  0.728464390f,
    0.680600998f,  0.732654272f,
    0.676092704f,  0.736816569f,
    0.671558955f,  0.740951125f,
    0.666999922f,  0.745057785f,
    0.662415778f,  0.749136395f,
    0.657806693f,  0.753186799f,
    0.653172843f,  0.757208847f,
    0.648514401f,  0.761202385f,
    0.643831543f,  0.765167266f,
    0.639124445f,  0.769103338f,
    0.634393284f,  0.773010453f,
    0.629638239f,  0.776888466f,
    0.624859488f,  0.780737229f,
    0.620057212f,  0.784556597f,
    0.615231591f,  0.788346428f,
    0.610382806f,  0.792106577f,
    0.605511041f,  0.795836905f,
    0.600616479f,  0.799537269f,
    0.595699304f,  0.803207531f,
    0.590759702f,  0.806847554f,
    0.585797857f,  0.810457198f,
    0.580813958f,  0.814036330f,
    0.575808191f,  0.817584813f,
    0.570780746f,  0.821102515f,
    0.565731811f,  0.824589303f,
    0.560661576f,  0.828045045f,
    0.555570233f,  0.831469612f,
    0.550457973f,  0.834862875f,
    0.545324988f,  0.838224706f,
    0.540171473f,  0.841554977f,
    0.534997620f,  0.844853565f,
    0.529803625f,  0.848120345f,
    0.524589683f,  0.851355193f,
    0.519355990f,  0.854557988f,
    0.514102744f,  0.857728610f,
    0.508830143f,  0.860866939f,
    0.503538384f,  0.863972856f,
    0.498227667f,  0.867046246f,
    0.492898192f,  0.870086991f,
    0.487550160f,  0.873094978f,
    0.482183772f,  0.876070094f,
    0.476799230f,  0.879012226f,
    0.471396737f,  0.881921264f,
    0.465976496f,  0.884797098f,
    0.460538711f,  0.887639620f,
    0.455083587f,  0.890448723f,
    0.449611330f,  0.893224301f,
    0.444122145f,  0.895966250f,
    0.438616239f,  0.898674466f,
    0.433093819f,  0.901348847f,
    0.427555093f,  0.903989293f,
    0.422000271f,  0.906595705f,
    0.416429560f,  0.909167983f,
    0.410843171f,  0.911706032f,
    0.405241314f,  0.914209756f,
    0.399624200f,  0.916679060f,
    0.393992040f,  0.919113852f,
    0.388345047f,  0.921514039f,
    0.382683432f,  0.923879533f,
    0.377007410f,  0.926210242f,
    0.371317194f,  0.928506080f,
    0.365612998f,  0.930766961f,
    0.359895037f,  0.932992799f,
    0.354163525f,  0.935183510f,
    0.348418680f,  0.937339012f,
    0.342660717f,  0.939459224f,
    0.336889853f,  0.941544065f,
    0.331106306f,  0.943593458f,
    0.325310292f,  0.945607325f,
    0.319502031f,  0.947585591f,
    0.313681740f,  0.949528181f,
    0.307849640f,  0.951435021f,
    0.302005949f,  0.953306040f,
    0.296150888f,  0.955141168f,
    0.290284677f,  0.956940336f,
    0.284407537f,  0.958703475f,
    0.278519689f,  0.960430519f,
    0.272621355f,  0.962121404f,
    0.266712757f,  0.963776066f,
    0.260794118f,  0.965394442f,
    0.254865660f,  0.966976471f,
    0.248927606f,  0.968522094f,
    0.242980180f,  0.970031253f,
    0.237023606f,  0.971503891f,
    0.231058108f,  0.972939952f,
    0.225083911f,  0.974339383f,
    0.219101240f,  0.975702130f,
    0.213110320f,  0.977028143f,
    0.207111376f,  0.978317371f,
    0.201104635f,  0.979569766f,
    0.195090322f,  0.980785280f,
    0.189068664f,  0.981963869f,
    0.183039888f,  0.983105487f,
    0.177004220f,  0.984210092f,
    0.170961889f,  0.985277642f,
    0.164913120f,  0.986308097f,
    0.158858143f,  0.987301418f,
    0.152797185f,  0.988257568f,
    0.146730474f,  0.989176510f,
    0.140658239f,  0.990058210f,
    0.134580709f,  0.990902635f,
    0.128498111f,  0.991709754f,
    0.122410675f,  0.992479535f,
    0.116318631f,  0.993211949f,
    0.110222207f,  0.993906970f,
    0.104121634f,  0.994564571f,
    0.098017140f,  0.995184727f,
    0.091908956f,  0.995767414f,
    0.085797312f,  0.996312612f,
    0.079682438f,  0.996820299f,
    0.073564564f,  0.997290457f,
    0.067443920f,  0.997723067f,
    0.061320736f,  0.998118113f,
    0.055195244f,  0.998475581f,
    0.049067674f,  0.998795456f,
    0.042938257f,  0.999077728f,
    0.036807223f,  0.999322385f,
    0.030674803f,  0.999529418f,
    0.024541229f,  0.999698819f,
    0.018406730f,  0.999830582f,
    0.012271538f,  0.999924702f,
    0.006135885f,  0.999981175f,
    0.000000000f,  1.000000000f,
   -0.006135885f,  0.999981175f,
   -0.012271538f,  0.999924702f,
   -0.018406730f,  0.999830582f,
   -0.024541229f,  0.999698819f,
   -0.030674803f,  0.999529418f,
   -0.036807223f,  0.999322385f,
   -0.042938257f,  0.999077728f,
   -0.049067674f,  0.998795456f,
   -0.055195244f,  0.998475581f,
   -0.061320736f,  0.998118113f,
   -0.067443920f,  0.997723067f,
   -0.073564564f,  0.997290457f,
   -0.079682438f,  0.996820299f,
   -0.085797312f,  0.996312612f,
   -0.091908956f,  0.995767414f,
   -0.098017140f,  0.995184727f,
   -0.104121634f,  0.994564571f,
   -0.110222207f,  0.993906970f,
   -0.116318631f,  0.993211949f,
   -0.122410675f,  0.992479535f,
   -0.128498111f,  0.991709754f,
   -0.134580709f,  0.990902635f,
   -0.140658239f,  0.990058210f,
   -0.146730474f,  0.989176510f,
   -0.152797185f,  0.988257568f,
   -0.158858143f,  0.987301418f,
   -0.164913120f,  0.986308097f,
   -0.170961889f,  0.985277642f,
   -0.177004220f,  0.984210092f,
   -0.183039888f,  0.983105487f,
   -0.189068664f,  0.981963869f,
   -0.195090322f,  0.980785280f,
   -0.201104635f,  0.979569766f,
   -0.207111376f,  0.978317371f,
   -0.213110320f,  0.977028143f,
   -0.219101240f,  0.975702130f,
   -0.225083911f,  0.974339383f,
   -0.231058108f,  0.972939952f,
   -0.237023606f,  0.971503891f,
   -0.242980180f,  0.970031253f,
   -0.248927606f,  0.968522094f,
   -0.254865660f,  0.966976471f,
   -0.260794118f,  0.965394442f,
   -0.266712757f,  0.963776066f,
   -0.272621355f,  0.962121404f,
   -0.278519689f,  0.960430519f,
   -0.284407537f,  0.958703475f,
   -0.290284677f,  0.956940336f,
   -0.296150888f,  0.955141168f,
   -0.302005949f,  0.953306040f,
   -0.307849640f,  0.951435021f,
   -0.313681740f,  0.949528181f,
   -0.319502031f,  0.947585591f,
   -0.325310292f,  0.945607325f,
   -0.331106306f,  0.943593458f,
   -0.336889853f,  0.941544065f,
   -0.342660717f,  0.939459224f,
   -0.348418680f,  0.937339012f,
   -0.354163525f,  0.935183510f,
   -0.359895037f,  0.932992799f,
   -0.365612998f,  0.930766961f,
   -0.371317194f,  0.928506080f,
   -0.377007410f,  0.926210242f,
   -0.382683432f,  0.923879533f,
   -0.388345047f,  0.921514039f,
   -0.393992040f,  0.919113852f,
   -0.399624200f,  0.916679060f,
   -0.405241314f,  0.914209756f,
   -0.410843171f,  0.911706032f,
   -0.416429560f,  0.909167983f,
   -0.422000271f,  0.906595705f,
   -0.427555093f,  0.903989293f,
   -0.433093819f,  0.901348847f,
   -0.438616239f,  0.898674466f,
   -0.444122145f,  0.895966250f,
   -0.449611330f,  0.893224301f,
   -0.455083587f,  0.890448723f,
   -0.460538711f,  0.887639620f,
   -0.465976496f,  0.884797098f,
   -0.471396737f,  0.881921264f,
   -0.476799230f,  0.879012226f,
   -0.482183772f,  0.876070094f,
   -0.487550160f,  0.873094978f,
   -0.492898192f,  0.870086991f,
   -0.498227667f,  0.867046246f,
   -0.503538384f,  0.863972856f,
   -0.508830143f,  0.860866939f,
   -0.514102744f,  0.857728610f,
   -0.519355990f,  0.854557988f,
   -0.524589683f,  0.851355193f,
   -0.529803625f,  0.848120345f,
   -0.534997620f,  0.844853565f,
   -0.540171473f,  0.841554977f,
   -0.545324988f,  0.838224706f,
   -0.550457973f,  0.834862875f,
   -0.555570233f,  0.831469612f,
   -0.560661576f,  0.828045045f,
   -0.565731811f,  0.824589303f,
   -0.570780746f,  0.821102515f,
   -0.575808191f,  0.817584813f,
   -0.580813958f,  0.814036330f,
   -0.585797857f,  0.810457198f,
   -0.590759702f,  0.806847554f,
   -0.595699304f,  0.803207531f,
   -0.600616479f,  0.799537269f,
   -0.605511041f,  0.795836905f,
   -0.610382806f,  0.792106577f,
   -0.615231591f,  0.788346428f,
   -0.620057212f,  0.784556597f,
   -0.624859488f,  0.780737229f,
   -0.629638239f,  0.776888466f,
   -0.634393284f,  0.773010453f,
   -0.639124445f,  0.769103338f,
   -0.643831543f,  0.765167266f,
   -0.648514401f,  0.761202385f,
   -0.653172843f,  0.757208847f,
   -0.657806693f,  0.753186799f,
   -0.662415778f,  0.749136395f,
   -0.666999922f,  0.745057785f,
   -0.671558955f,  0.740951125f,
   -0.676092704f,  0.736816569f,
   -0.680600998f,  0.732654272f,
   -0.685083668f,  0.728464390f,
   -0.689540545f,  0.724247083f,
   -0.693971461f,  0.720002508f,
   -0.698376249f,  0.715730825f,
   -0.702754744f,  0.711432196f,
   -0.707106781f,  0.707106781f,
   -0.711432196f,  0.702754744f,
   -0.715730825f,  0.698376249f,
   -0.720002508f,  0.693971461f,
   -0.724247083f,  0.689540545f,
   -0.728464390f,  0.685083668f,
   -0.732654272f,  0.680600998f,
   -0.736816569f,  0.676092704f,
   -0.740951125f,  0.671558955f,
   -0.745057785f,  0.666999922f,
   -0.749136395f,  0.662415778f,
   -0.753186799f,  0.657806693f,
   -0.757208847f,  0.653172843f,
   -0.761202385f,  0.648514401f,
   -0.765167266f,  0.643831543f,
   -0.769103338f,  0.639124445f,
   -0.773010453f,  0.634393284f,
   -0.776888466f,  0.629638239f,
   -0.780737229f,  0.624859488f,
   -0.784556597f,  0.620057212f,
   -0.788346428f,  0.615231591f,
   -0.792106577f,  0.610382806f,
   -0.795836905f,  0.605511041f,
   -0.799537269f,  0.600616479f,
   -0.803207531f,  0.595699304f,
   -0.806847554f,  0.590759702f,
   -0.810457198f,  0.585797857f,
   -0.814036330f,  0.580813958f,
   -0.817584813f,  0.575808191f,
   -0.821102515f,  0.570780746f,
   -0.824589303f,  0.565731811f,
   -0.828045045f,  0.560661576f,
   -0.831469612f,  0.555570233f,
   -0.834862875f,  0.550457973f,
   -0.838224706f,  0.545324988f,
   -0.841554977f,  0.540171473f,
   -0.844853565f,  0.534997620f,
   -0.848120345f,  0.529803625f,
   -0.851355193f,  0.524589683f,
   -0.854557988f,  0.519355990f,
   -0.857728610f,  0.514102744f,
   -0.860866939f,  0.508830143f,
   -0.863972856f,  0.503538384f,
   -0.867046246f,  0.498227667f,
   -0.870086991f,  0.492898192f,
   -0.873094978f,  0.487550160f,
   -0.876070094f,  0.482183772f,
   -0.879012226f,  0.476799230f,
   -0.881921264f,  0.471396737f,
   -0.884797098f,  0.465976496f,
   -0.887639620f,  0.460538711f,
   -0.890448723f,  0.455083587f,
   -0.893224301f,  0.449611330f,
   -0.895966250f,  0.444122145f,
   -0.898674466f,  0.438616239f,
   -0.901348847f,  0.433093819f,
   -0.903989293f,  0.427555093f,
   -0.906595705f,  0.422000271f,
   -0.909167983f,  0.416429560f,
   -0.911706032f,  0.410843171f,
   -0.914209756f,  0.405241314f,
   -0.916679060f,  0.399624200f,
   -0.919113852f,  0.393992040f,
   -0.921514039f,  0.388345047f,
   -0.923879533f,  0.382683432f,
   -0.926210242f,  0.377007410f,
   -0.928506080f,  0.371317194f,
   -0.930766961f,  0.365612998f,
   -0.932992799f,  0.359895037f,
   -0.935183510f,  0.354163525f,
   -0.937339012f,  0.348418680f,
   -0.939459224f,  0.342660717f,
   -0.941544065f,  0.336889853f,
   -0.943593458f,  0.331106306f,
   -0.945607325f,  0.325310292f,
   -0.947585591f,  0.319502031f,
   -0.949528181f,  0.313681740f,
   -0.951435021f,  0.307849640f,
   -0.953306040f,  0.302005949f,
   -0.955141168f,  0.296150888f,
   -0.956940336f,  0.290284677f,
   -0.958703475f,  0.284407537f,
   -0.960430519f,  0.278519689f,
   -0.962121404f,  0.272621355f,
   -0.963776066f,  0.266712757f,
   -0.965394442f,  0.260794118f,
   -0.966976471f,  0.254865660f,
   -0.968522094f,  0.248927606f,
   -0.970031253f,  0.242980180f,
   -0.971503891f,  0.237023606f,
   -0.972939952f,  0.231058108f,
   -0.974339383f,  0.225083911f,
   -0.975702130f,  0.219101240f,
   -0.977028143f,  0.213110320f,
   -0.978317371f,  0.207111376f,
   -0.979569766f,  0.201104635f,
   -0.980785280f,  0.195090322f,
   -0.981963869f,  0.189068664f,
   -0.983105487f,  0.183039888f,
   -0.984210092f,  0.177004220f,
   -0.985277642f,  0.170961889f,
   -0.986308097f,  0.164913120f,
   -0.987301418f,  0.158858143f,
   -0.988257568f,  0.152797185f,
   -0.989176510f,  0.146730474f,
   -0.990058210f,  0.140658239f,
   -0.990902635f,  0.134580709f,
   -0.991709754f,  0.128498111f,
   -0.992479535f,  0.122410675f,
   -0.993211949f,  0.116318631f,
   -0.993906970f,  0.110222207f,
   -0.994564571f,  0.104121634f,
   -0.995184727f,  0.098017140f,
   -0.995767414f,  0.091908956f,
   -0.996312612f,  0.085797312f,
   -0.996820299f,  0.079682438f,
   -0.997290457f,  0.073564564f,
   -0.997723067f,  0.067443920f,
   -0.998118113f,  0.061320736f,
   -0.998475581f,  0.055195244f,
   -0.998795456f,  0.049067674f,
   -0.999077728f,  0.042938257f,
   -0.999322385f,  0.036807223f,
   -0.999529418f,  0.030674803f,
   -0.999698819f,  0.024541229f,
   -0.999830582f,  0.018406730f,
   -0.999924702f,  0.012271538f,
   -0.999981175f,  0.006135885f,
   -1.000000000f,  0.000000000f,
   -0.999981175f, -0.006135885f,
   -0.999924702f, -0.012271538f,
   -0.999830582f, -0.018406730f,
   -0.999698819f, -0.024541229f,
   -0.999529418f, -0.030674803f,
   -0.999322385f, -0.036807223f,
   -0.999077728f, -0.042938257f,
   -0.998795456f, -0.049067674f,
   -0.998475581f, -0.055195244f,
   -0.998118113f, -0.061320736f,
   -0.997723067f, -0.067443920f,
   -0.997290457f, -0.073564564f,
   -0.996820299f, -0.079682438f,
   -0.996312612f, -0.085797312f,
   -0.995767414f, -0.091908956f,
   -0.995184727f, -0.098017140f,
   -0.994564571f, -0.104121634f,
   -0.993906970f, -0.110222207f,
   -0.993211949f, -0.116318631f,
   -0.992479535f, -0.122410675f,
   -0.991709754f, -0.128498111f,
   -0.990902635f, -0.134580709f,
   -0.990058210f, -0.140658239f,
   -0.989176510f, -0.146730474f,
   -0.988257568f, -0.152797185f,
   -0.987301418f, -0.158858143f,
   -0.986308097f, -0.164913120f,
   -0.985277642f, -0.170961889f,
   -0.984210092f, -0.177004220f,
   -0.983105487f, -0.183039888f,
   -0.981963869f, -0.189068664f,
   -0.980785280f, -0.195090322f,
   -0.979569766f, -0.201104635f,
   -0.978317371f, -0.207111376f,
   -0.977028143f, -0.213110320f,
   -0.975702130f, -0.219101240f,
   -0.974339383f, -0.225083911f,
   -0.972939952f, -0.231058108f,
   -0.971503891f, -0.237023606f,
   -0.970031253f, -0.242980180f,
   -0.968522094f, -0.248927606f,
   -0.966976471f, -0.254865660f,
   -0.965394442f, -0.260794118f,
   -0.963776066f, -0.266712757f,
   -0.962121404f, -0.272621355f,
   -0.960430519f, -0.278519689f,
   -0.958703475f, -0.284407537f,
   -0.956940336f, -0.290284677f,
   -0.955141168f, -0.296150888f,
   -0.953306040f, -0.302005949f,
   -0.951435021f, -0.307849640f,
   -0.949528181f, -0.313681740f,
   -0.947585591f, -0.319502031f,
   -0.945607325f, -0.325310292f,
   -0.943593458f, -0.331106306f,
   -0.941544065f, -0.336889853f,
   -0.939459224f, -0.342660717f,
   -0.937339012f, -0.348418680f,
   -0.935183510f, -0.354163525f,
   -0.932992799f, -0.359895037f,
   -0.930766961f, -0.365612998f,
   -0.928506080f, -0.371317194f,
   -0.926210242f, -0.377007410f,
   -0.923879533f, -0.382683432f,
   -0.921514039f, -0.388345047f,
   -0.919113852f, -0.393992040f,
   -0.916679060f, -0.399624200f,
   -0.914209756f, -0.405241314f,
   -0.911706032f, -0.410843171f,
   -0.909167983f, -0.416429560f,
   -0.906595705f, -0.422000271f,
   -0.903989293f, -0.427555093f,
   -0.901348847f, -0.433093819f,
   -0.898674466f, -0.438616239f,
   -0.895966250f, -0.444122145f,
   -0.893224301f, -0.449611330f,
   -0.890448723f, -0.455083587f,
   -0.887639620f, -0.460538711f,
   -0.884797098f, -0.465976496f,
   -0.881921264f, -0.471396737f,
   -0.879012226f, -0.476799230f,
   -0.876070094f, -0.482183772f,
   -0.873094978f, -0.487550160f,
   -0.870086991f, -0.492898192f,
   -0.867046246f, -0.498227667f,
   -0.863972856f, -0.503538384f,
   -0.860866939f, -0.508830143f,
   -0.857728610f, -0.514102744f,
   -0.854557988f, -0.519355990f,
   -0.851355193f, -0.524589683f,
   -0.848120345f, -0.529803625f,
   -0.844853565f, -0.534997620f,
   -0.841554977f, -0.540171473f,
   -0.838224706f, -0.545324988f,
   -0.834862875f, -0.550457973f,
   -0.831469612f, -0.555570233f,
   -0.828045045f, -0.560661576f,
   -0.824589303f, -0.565731811f,
   -0.821102515f, -0.570780746f,
   -0.817584813f, -0.575808191f,
   -0.814036330f, -0.580813958f,
   -0.810457198f, -0.585797857f,
   -0.806847554f, -0.590759702f,
   -0.803207531f, -0.595699304f,
   -0.799537269f, -0.600616479f,
   -0.795836905f, -0.605511041f,
   -0.792106577f, -0.610382806f,
   -0.788346428f, -0.615231591f,
   -0.784556597f, -0.620057212f,
   -0.780737229f, -0.624859488f,
   -0.776888466f, -0.629638239f,
   -0.773010453f, -0.634393284f,
   -0.769103338f, -0.639124445f,
   -0.765167266f, -0.643831543f,
   -0.761202385f, -0.648514401f,
   -0.757208847f, -0.653172843f,
   -0.753186799f, -0.657806693f,
   -0.749136395f, -0.662415778f,
   -0.745057785f, -0.666999922f,
   -0.740951125f, -0.671558955f,
   -0.736816569f, -0.676092704f,
   -0.732654272f, -0.680600998f,
   -0.728464390f, -0.685083668f,
   -0.724247083f, -0.689540545f,
   -0.720002508f, -0.693971461f,
   -0.715730825f, -0.698376249f,
   -0.711432196f, -0.702754744f,
   -0.707106781f, -0.707106781f,
   -0.702754744f, -0.711432196f,
   -0.698376249f, -0.715730825f,
   -0.693971461f, -0.720002508f,
   -0.689540545f, -0.724247083f,
   -0.685083668f, -0.728464390f,
   -0.680600998f, -0.732654272f,
   -0.676092704f, -0.736816569f,
   -0.671558955f, -0.740951125f,
   -0.666999922f, -0.745057785f,
   -0.662415778f, -0.749136395f,
   -0.657806693f, -0.753186799f,
   -0.653172843f, -0.757208847f,
   -0.648514401f, -0.761202385f,
   -0.643831543f, -0.765167266f,
   -0.639124445f, -0.769103338f,
   -0.634393284f, -0.773010453f,
   -0.629638239f, -0.776888466f,
   -0.624859488f, -0.780737229f,
   -0.620057212f, -0.784556597f,
   -0.615231591f, -0.788346428f,
   -0.610382806f, -0.792106577f,
   -0.605511041f, -0.795836905f,
   -0.600616479f, -0.799537269f,
   -0.595699304f, -0.803207531f,
   -0.590759702f, -0.806847554f,
   -0.585797857f, -0.810457198f,
   -0.580813958f, -0.814036330f,
   -0.575808191f, -0.817584813f,
   -0.570780746f, -0.821102515f,
   -0.565731811f, -0.824589303f,
   -0.560661576f, -0.828045045f,
   -0.555570233f, -0.831469612f,
   -0.550457973f, -0.834862875f,
   -0.545324988f, -0.838224706f,
   -0.540171473f, -0.841554977f,
   -0.534997620f, -0.844853565f,
   -0.529803625f, -0.848120345f,
   -0.524589683f, -0.851355193f,
   -0.519355990f, -0.854557988f,
   -0.514102744f, -0.857728610f,
   -0.508830143f, -0.860866939f,
   -0.503538384f, -0.863972856f,
   -0.498227667f, -0.867046246f,
   -0.492898192f, -0.870086991f,
   -0.487550160f, -0.873094978f,
   -0.482183772f, -0.876070094f,
   -0.476799230f, -0.879012226f,
   -0.471396737f, -0.881921264f,
   -0.465976496f, -0.884797098f,
   -0.460538711f, -0.887639620f,
   -0.455083587f, -0.890448723f,
   -0.449611330f, -0.893224301f,
   -0.444122145f, -0.895966250f,
   -0.438616239f, -0.898674466f,
   -0.433093819f, -0.901348847f,
   -0.427555093f, -0.903989293f,
   -0.422000271f, -0.906595705f,
   -0.416429560f, -0.909167983f,
   -0.410843171f, -0.911706032f,
   -0.405241314f, -0.914209756f,
   -0.399624200f, -0.916679060f,
   -0.393992040f, -0.919113852f,
   -0.388345047f, -0.921514039f,
   -0.382683432f, -0.923879533f,
   -0.377007410f, -0.926210242f,
   -0.371317194f, -0.928506080f,
   -0.365612998f, -0.930766961f,
   -0.359895037f, -0.932992799f,
   -0.354163525f, -0.935183510f,
   -0.348418680f, -0.937339012f,
   -0.342660717f, -0.939459224f,
   -0.336889853f, -0.941544065f,
   -0.331106306f, -0.943593458f,
   -0.325310292f, -0.945607325f,
   -0.319502031f, -0.947585591f,
   -0.313681740f, -0.949528181f,
   -0.307849640f, -0.951435021f,
   -0.302005949f, -0.953306040f,
   -0.296150888f, -0.955141168f,
   -0.290284677f, -0.956940336f,
   -0.284407537f, -0.958703475f,
   -0.278519689f, -0.960430519f,
   -0.272621355f, -0.962121404f,
   -0.266712757f, -0.963776066f,
   -0.260794118f, -0.965394442f,
   -0.254865660f, -0.966976471f,
   -0.248927606f, -0.968522094f,
   -0.242980180f, -0.970031253f,
   -0.237023606f, -0.971503891f,
   -0.231058108f, -0.972939952f,
   -0.225083911f, -0.974339383f,
   -0.219101240f, -0.975702130f,
   -0.213110320f, -0.977028143f,
   -0.207111376f, -0.978317371f,
   -0.201104635f, -0.979569766f,
   -0.195090322f, -0.980785280f,
   -0.189068664f, -0.981963869f,
   -0.183039888f, -0.983105487f,
   -0.177004220f, -0.984210092f,
   -0.170961889f, -0.985277642f,
   -0.164913120f, -0.986308097f,
   -0.158858143f, -0.987301418f,
   -0.152797185f, -0.988257568f,
   -0.146730474f, -0.989176510f,
   -0.140658239f, -0.990058210f,
   -0.134580709f, -0.990902635f,
   -0.128498111f, -0.991709754f,
   -0.122410675f, -0.992479535f,
   -0.116318631f, -0.993211949f,
   -0.110222207f, -0.993906970f,
   -0.104121634f, -0.994564571f,
   -0.098017140f, -0.995184727f,
   -0.091908956f, -0.995767414f,
   -0.085797312f, -0.996312612f,
   -0.079682438f, -0.996820299f,
   -0.073564564f, -0.997290457f,
   -0.067443920f, -0.997723067f,
   -0.061320736f, -0.998118113f,
   -0.055195244f, -0.998475581f,
   -0.049067674f, -0.998795456f,
   -0.042938257f, -0.999077728f,
   -0.036807223f, -0.999322385f,
   -0.030674803f, -0.999529418f,
   -0.024541229f, -0.999698819f,
   -0.018406730f, -0.999830582f,
   -0.012271538f, -0.999924702f,
   -0.006135885f, -0.999981175f,
   -0.000000000f, -1.000000000f,
    0.006135885f, -0.999981175f,
    0.012271538f, -0.999924702f,
    0.018406730f, -0.999830582f,
    0.024541229f, -0.999698819f,
    0.030674803f, -0.999529418f,
    0.036807223f, -0.999322385f,
    0.042938257f, -0.999077728f,
    0.049067674f, -0.998795456f,
    0.055195244f, -0.998475581f,
    0.061320736f, -0.998118113f,
    0.067443920f, -0.997723067f,
    0.073564564f, -0.997290457f,
    0.079682438f, -0.996820299f,
    0.085797312f, -0.996312612f,
    0.091908956f, -0.995767414f,
    0.098017140f, -0.995184727f,
    0.104121634f, -0.994564571f,
    0.110222207f, -0.993906970f,
    0.116318631f, -0.993211949f,
    0.122410675f, -0.992479535f,
    0.128498111f, -0.991709754f,
    0.134580709f, -0.990902635f,
    0.140658239f, -0.990058210f,
    0.146730474f, -0.989176510f,
    0.152797185f, -0.988257568f,
    0.158858143f, -0.987301418f,
    0.164913120f, -0.986308097f,
    0.170961889f, -0.985277642f,
    0.177004220f, -0.984210092f,
    0.183039888f, -0.983105487f,
    0.189068664f, -0.981963869f,
    0.195090322f, -0.980785280f,
    0.201104635f, -0.979569766f,
    0.207111376f, -0.978317371f,
    0.213110320f, -0.977028143f,
    0.219101240f, -0.975702130f,
    0.225083911f, -0.974339383f,
    0.231058108f, -0.972939952f,
    0.237023606f, -0.971503891f,
    0.242980180f, -0.970031253f,
    0.248927606f, -0.968522094f,
    0.254865660f, -0.966976471f,
    0.260794118f, -0.965394442f,
    0.266712757f, -0.963776066f,
    0.272621355f, -0.962121404f,
    0.278519689f, -0.960430519f,
    0.284407537f, -0.958703475f,
    0.290284677f, -0.956940336f,
    0.296150888f, -0.955141168f,
    0.302005949f, -0.953306040f,
    0.307849640f, -0.951435021f,
    0.313681740f, -0.949528181f,
    0.319502031f, -0.947585591f,
    0.325310292f, -0.945607325f,
    0.331106306f, -0.943593458f,
    0.336889853f, -0.941544065f,
    0.342660717f, -0.939459224f,
    0.348418680f, -0.937339012f,
    0.354163525f, -0.935183510f,
    0.359895037f, -0.932992799f,
    0.365612998f, -0.930766961f,
    0.371317194f, -0.928506080f,
    0.377007410f, -0.926210242f,
    0.382683432f, -0.923879533f,
    0.388345047f, -0.921514039f,
    0.393992040f, -0.919113852f,
    0.399624200f, -0.916679060f,
    0.405241314f, -0.914209756f,
    0.410843171f, -0.911706032f,
    0.416429560f, -0.909167983f,
    0.422000271f, -0.906595705f,
    0.427555093f, -0.903989293f,
    0.433093819f, -0.901348847f,
    0.438616239f, -0.898674466f,
    0.444122145f, -0.895966250f,
    0.449611330f, -0.893224301f,
    0.455083587f, -0.890448723f,
    0.460538711f, -0.887639620f,
    0.465976496f, -0.884797098f,
    0.471396737f, -0.881921264f,
    0.476799230f, -0.879012226f,
    0.482183772f, -0.876070094f,
    0.487550160f, -0.873094978f,
    0.492898192f, -0.870086991f,
    0.498227667f, -0.867046246f,
    0.503538384f, -0.863972856f,
    0.508830143f, -0.860866939f,
    0.514102744f, -0.857728610f,
    0.519355990f, -0.854557988f,
    0.524589683f, -0.851355193f,
    0.529803625f, -0.848120345f,
    0.534997620f, -0.844853565f,
    0.540171473f, -0.841554977f,
    0.545324988f, -0.838224706f,
    0.550457973f, -0.834862875f,
    0.555570233f, -0.831469612f,
    0.560661576f, -0.828045045f,
    0.565731811f, -0.824589303f,
    0.570780746f, -0.821102515f,
    0.575808191f, -0.817584813f,
    0.580813958f, -0.814036330f,
    0.585797857f, -0.810457198f,
    0.590759702f, -0.806847554f,
    0.595699304f, -0.803207531f,
    0.600616479f, -0.799537269f,
    0.605511041f, -0.795836905f,
    0.610382806f, -0.792106577f,
    0.615231591f, -0.788346428f,
    0.620057212f, -0.784556597f,
    0.624859488f, -0.780737229f,
    0.629638239f, -0.776888466f,
    0.634393284f, -0.773010453f,
    0.639124445f, -0.769103338f,
    0.643831543f, -0.765167266f,
    0.648514401f, -0.761202385f,
    0.653172843f, -0.757208847f,
    0.657806693f, -0.753186799f,
    0.662415778f, -0.749136395f,
    0.666999922f, -0.745057785f,
    0.671558955f, -0.740951125f,
    0.676092704f, -0.736816569f,
    0.680600998f, -0.732654272f,
    0.685083668f, -0.728464390f,
    0.689540545f, -0.724247083f,
    0.693971461f, -0.720002508f,
    0.698376249f, -0.715730825f,
    0.702754744f, -0.711432196f,
    0.707106781f, -0.707106781f,
    0.711432196f, -0.702754744f,
    0.715730825f, -0.698376249f,
    0.720002508f, -0.693971461f,
    0.724247083f, -0.689540545f,
    0.728464390f, -0.685083668f,
    0.732654272f, -0.680600998f,
    0.736816569f, -0.676092704f,
    0.740951125f, -0.671558955f,
    0.745057785f, -0.666999922f,
    0.749136395f, -0.662415778f,
    0.753186799f, -0.657806693f,
    0.757208847f, -0.653172843f,
    0.761202385f, -0.648514401f,
    0.765167266f, -0.643831543f,
    0.769103338f, -0.639124445f,
    0.773010453f, -0.634393284f,
    0.776888466f, -0.629638239f,
    0.780737229f, -0.624859488f,
    0.784556597f, -0.620057212f,
    0.788346428f, -0.615231591f,
    0.792106577f, -0.610382806f,
    0.795836905f, -0.605511041f,
    0.799537269f, -0.600616479f,
    0.803207531f, -0.595699304f,
    0.806847554f, -0.590759702f,
    0.810457198f, -0.585797857f,
    0.814036330f, -0.580813958f,
    0.817584813f, -0.575808191f,
    0.821102515f, -0.570780746f,
    0.824589303f, -0.565731811f,
    0.828045045f, -0.560661576f,
    0.831469612f, -0.555570233f,
    0.834862875f, -0.550457973f,
    0.838224706f, -0.545324988f,
    0.841554977f, -0.540171473f,
    0.844853565f, -0.534997620f,
    0.848120345f, -0.529803625f,
    0.851355193f, -0.524589683f,
    0.854557988f, -0.519355990f,
    0.857728610f, -0.514102744f,
    0.860866939f, -0.508830143f,
    0.863972856f, -0.503538384f,
    0.867046246f, -0.498227667f,
    0.870086991f, -0.492898192f,
    0.873094978f, -0.487550160f,
    0.876070094f, -0.482183772f,
    0.879012226f, -0.476799230f,
    0.881921264f, -0.471396737f,
    0.884797098f, -0.465976496f,
    0.887639620f, -0.460538711f,
    0.890448723f, -0.455083587f,
    0.893224301f, -0.449611330f,
    0.895966250f, -0.444122145f,
    0.898674466f, -0.438616239f,
    0.901348847f, -0.433093819f,
    0.903989293f, -0.427555093f,
    0.906595705f, -0.422000271f,
    0.909167983f, -0.416429560f,
    0.911706032f, -0.410843171f,
    0.914209756f, -0.405241314f,
    0.916679060f, -0.399624200f,
    0.919113852f, -0.393992040f,
    0.921514039f, -0.388345047f,
    0.923879533f, -0.382683432f,
    0.926210242f, -0.377007410f,
    0.928506080f, -0.371317194f,
    0.930766961f, -0.365612998f,
    0.932992799f, -0.359895037f,
    0.935183510f, -0.354163525f,
    0.937339012f, -0.348418680f,
    0.939459224f, -0.342660717f,
    0.941544065f, -0.336889853f,
    0.943593458f, -0.331106306f,
    0.945607325f, -0.325310292f,
    0.947585591f, -0.319502031f,
    0.949528181f, -0.313681740f,
    0.951435021f, -0.307849640f,
    0.953306040f, -0.302005949f,
    0.955141168f, -0.296150888f,
    0.956940336f, -0.290284677f,
    0.958703475f, -0.284407537f,
    0.960430519f, -0.278519689f,
    0.962121404f, -0.272621355f,
    0.963776066f, -0.266712757f,
    0.965394442f, -0.260794118f,
    0.966976471f, -0.254865660f,
    0.968522094f, -0.248927606f,
    0.970031253f, -0.242980180f,
    0.971503891f, -0.237023606f,
    0.972939952f, -0.231058108f,
    0.974339383f, -0.225083911f,
    0.975702130f, -0.219101240f,
    0.977028143f, -0.213110320f,
    0.978317371f, -0.207111376f,
    0.979569766f, -0.201104635f,
    0.980785280f, -0.195090322f,
    0.981963869f, -0.189068664f,
    0.983105487f, -0.183039888f,
    0.984210092f, -0.177004220f,
    0.985277642f, -0.170961889f,
    0.986308097f, -0.164913120f,
    0.987301418f, -0.158858143f,
    0.988257568f, -0.152797185f,
    0.989176510f, -0.146730474f,
    0.990058210f, -0.140658239f,
    0.990902635f, -0.134580709f,
    0.991709754f, -0.128498111f,
    0.992479535f, -0.122410675f,
    0.993211949f, -0.116318631f,
    0.993906970f, -0.110222207f,
    0.994564571f, -0.104121634f,
    0.995184727f, -0.098017140f,
    0.995767414f, -0.091908956f,
    0.996312612f, -0.085797312f,
    0.996820299f, -0.079682438f,
    0.997290457f, -0.073564564f,
    0.997723067f, -0.067443920f,
    0.998118113f, -0.061320736f,
    0.998475581f, -0.055195244f,
    0.998795456f, -0.049067674f,
    0.999077728f, -0.042938257f,
    0.999322385f, -0.036807223f,
    0.999529418f, -0.030674803f,
    0.999698819f, -0.024541229f,
    0.999830582f, -0.018406730f,
    0.999924702f, -0.012271538f,
    0.999981175f, -0.006135885f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_2048)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 2048, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_2048[4096] = {
    1.000000000f,  0.000000000f,
    0.999995294f,  0.003067957f,
    0.999981175f,  0.006135885f,
    0.999957645f,  0.009203755f,
    0.999924702f,  0.012271538f,
    0.999882347f,  0.015339206f,
    0.999830582f,  0.018406730f,
    0.999769405f,  0.021474080f,
    0.999698819f,  0.024541229f,
    0.999618822f,  0.027608146f,
    0.999529418f,  0.030674803f,
    0.999430605f,  0.033741172f,
    0.999322385f,  0.036807223f,
    0.999204759f,  0.039872928f,
    0.999077728f,  0.042938257f,
    0.998941293f,  0.046003182f,
    0.998795456f,  0.049067674f,
    0.998640218f,  0.052131705f,
    0.998475581f,  0.055195244f,
    0.998301545f,  0.058258265f,
    0.998118113f,  0.061320736f,
    0.997925286f,  0.064382631f,
    0.997723067f,  0.067443920f,
    0.997511456f,  0.070504573f,
    0.997290457f,  0.073564564f,
    0.997060070f,  0.076623861f,
    0.996820299f,  0.079682438f,
    0.996571146f,  0.082740265f,
    0.996312612f,  0.085797312f,
    0.996044701f,  0.088853553f,
    0.995767414f,  0.091908956f,
    0.995480755f,  0.094963495f,
    0.995184727f,  0.098017140f,
    0.994879331f,  0.101069863f,
    0.994564571f,  0.104121634f,
    0.994240449f,  0.107172425f,
    0.993906970f,  0.110222207f,
    0.993564136f,  0.113270952f,
    0.993211949f,  0.116318631f,
    0.992850414f,  0.119365215f,
    0.992479535f,  0.122410675f,
    0.992099313f,  0.125454983f,
    0.991709754f,  0.128498111f,
    0.991310860f,  0.131540029f,
    0.990902635f,  0.134580709f,
    0.990485084f,  0.137620122f,
    0.990058210f,  0.140658239f,
    0.989622017f,  0.143695033f,
    0.989176510f,  0.146730474f,
    0.988721692f,  0.149764535f,
    0.988257568f,  0.152797185f,
    0.987784142f,  0.155828398f,
    0.987301418f,  0.158858143f,
    0.986809402f,  0.161886394f,
    0.986308097f,  0.164913120f,
    0.985797509f,  0.167938295f,
    0.985277642f,  0.170961889f,
    0.984748502f,  0.173983873f,
    0.984210092f,  0.177004220f,
    0.983662419f,  0.180022901f,
    0.983105487f,  0.183039888f,
    0.982539302f,  0.186055152f,
    0.981963869f,  0.189068664f,
    0.981379193f,  0.192080397f,
    0.980785280f,  0.195090322f,
    0.980182136f,  0.198098411f,
    0.979569766f,  0.201104635f,
    0.978948175f,  0.204108966f,
    0.978317371f,  0.207111376f,
    0.977677358f,  0.210111837f,
    0.977028143f,  0.213110320f,
    0.976369731f,  0.216106797f,
    0.975702130f,  0.219101240f,
    0.975025345f,  0.222093621f,
    0.974339383f,  0.225083911f,
    0.973644250f,  0.228072083f,
    0.972939952f,  0.231058108f,
    0.972226497f,  0.234041959f,
    0.971503891f,  0.237023606f,
    0.970772141f,  0.240003022f,
    0.970031253f,  0.242980180f,
    0.969281235f,  0.245955050f,
    0.968522094f,  0.248927606f,
    0.967753837f,  0.251897818f,
    0.966976471f,  0.254865660f,
    0.966190003f,  0.257831102f,
    0.965394442f,  0.260794118f,
    0.964589793f,  0.263754679f,
    0.963776066f,  0.266712757f,
    0.962953267f,  0.269668326f,
    0.962121404f,  0.272621355f,
    0.961280486f,  0.275571819f,
    0.960430519f,  0.278519689f,
    0.959571513f,  0.281464938f,
    0.958703475f,  0.284407537f,
    0.957826413f,  0.287347460f,
    0.956940336f,  0.290284677f,
    0.956045251f,  0.293219163f,
    0.955141168f,  0.296150888f,
    0.954228095f,  0.299079826f,
    0.953306040f,  0.302005949f,
    0.952375013f,  0.304929230f,
    0.951435021f,  0.307849640f,
    0.950486074f,  0.310767153f,
    0.949528181f,  0.313681740f,
    0.948561350f,  0.316593376f,
    0.947585591f,  0.319502031f,
    0.946600913f,  0.322407679f,
    0.945607325f,  0.325310292f,
    0.944604837f,  0.328209844f,
    0.943593458f,  0.331106306f,
    0.942573198f,  0.333999651f,
    0.941544065f,  0.336889853f,
    0.940506071f,  0.339776884f,
    0.939459224f,  0.342660717f,
    0.938403534f,  0.345541325f,
    0.937339012f,  0.348418680f,
    0.936265667f,  0.351292756f,
    0.935183510f,  0.354163525f,
    0.934092550f,  0.357030961f,
    0.932992799f,  0.359895037f,
    0.931884266f,  0.362755724f,
    0.930766961f,  0.365612998f,
    0.929640896f,  0.368466830f,
    0.928506080f,  0.371317194f,
    0.927362526f,  0.374164063f,
    0.926210242f,  0.377007410f,
    0.925049241f,  0.379847209f,
    0.923879533f,  0.382683432f,
    0.922701128f,  0.385516054f,
    0.921514039f,  0.388345047f,
    0.920318277f,  0.391170384f,
    0.919113852f,  0.393992040f,
    0.917900776f,  0.396809987f,
    0.916679060f,  0.399624200f,
    0.915448716f,  0.402434651f,
    0.914209756f,  0.405241314f,
    0.912962190f,  0.408044163f,
    0.911706032f,  0.410843171f,
    0.910441292f,  0.413638312f,
    0.909167983f,  0.416429560f,
    0.907886116f,  0.419216888f,
    0.906595705f,  0.422000271f,
    0.905296759f,  0.424779681f,
    0.903989293f,  0.427555093f,
    0.902673318f,  0.430326481f,
    0.901348847f,  0.433093819f,
    0.900015892f,  0.435857080f,
    0.898674466f,  0.438616239f,
    0.897324581f,  0.441371269f,
    0.895966250f,  0.444122145f,
    0.894599486f,  0.446868840f,
    0.893224301f,  0.449611330f,
    0.891840709f,  0.452349587f,
    0.890448723f,  0.455083587f,
    0.889048356f,  0.457813304f,
    0.887639620f,  0.460538711f,
    0.886222530f,  0.463259784f,
    0.884797098f,  0.465976496f,
    0.883363339f,  0.468688822f,
    0.881921264f,  0.471396737f,
    0.880470889f,  0.474100215f,
    0.879012226f,  0.476799230f,
    0.877545290f,  0.479493758f,
    0.876070094f,  0.482183772f,
    0.874586652f,  0.484869248f,
    0.873094978f,  0.487550160f,
    0.871595087f,  0.490226483f,
    0.870086991f,  0.492898192f,
    0.868570706f,  0.495565262f,
    0.867046246f,  0.498227667f,
    0.865513624f,  0.500885383f,
    0.863972856f,  0.503538384f,
    0.862423956f,  0.506186645f,
    0.860866939f,  0.508830143f,
    0.859301818f,  0.511468850f,
    0.857728610f,  0.514102744f,
    0.856147328f,  0.516731799f,
    0.854557988f,  0.519355990f,
    0.852960605f,  0.521975293f,
    0.851355193f,  0.524589683f,
    0.849741768f,  0.527199135f,
    0.848120345f,  0.529803625f,
    0.846490939f,  0.532403128f,
    0.844853565f,  0.534997620f,
    0.843208240f,  0.537587076f,
    0.841554977f,  0.540171473f,
    0.839893794f,  0.542750785f,
    0.838224706f,  0.545324988f,
    0.836547727f,  0.547894059f,
    0.834862875f,  0.550457973f,
    0.833170165f,  0.553016706f,
    0.831469612f,  0.555570233f,
    0.829761234f,  0.558118531f,
    0.828045045f,  0.560661576f,
    0.826321063f,  0.563199344f,
    0.824589303f,  0.565731811f,
    0.822849781f,  0.568258953f,
    0.821102515f,  0.570780746f,
    0.819347520f,  0.573297167f,
    0.817584813f,  0.575808191f,
    0.815814411f,  0.578313796f,
    0.814036330f,  0.580813958f,
    0.812250587f,  0.583308653f,
    0.810457198f,  0.585797857f,
    0.808656182f,  0.588281548f,
    0.806847554f,  0.590759702f,
    0.805031331f,  0.593232295f,
    0.803207531f,  0.595699304f,
    0.801376172f,  0.598160707f,
    0.799537269f,  0.600616479f,
    0.797690841f,  0.603066599f,
    0.795836905f,  0.605511041f,
    0.793975478f,  0.607949785f,
    0.792106577f,  0.610382806f,
    0.790230221f,  0.612810082f,
    0.788346428f,  0.615231591f,
    0.786455214f,  0.617647308f,
    0.784556597f,  0.620057212f,
    0.782650596f,  0.622461279f,
    0.780737229f,  0.624859488f,
    0.778816512f,  0.627251815f,
    0.776888466f,  0.629638239f,
    0.774953107f,  0.632018736f,
    0.773010453f,  0.634393284f,
    0.771060524f,  0.636761861f,
    0.769103338f,  0.639124445f,
    0.767138912f,  0.641481013f,
    0.765167266f,  0.643831543f,
    0.763188417f,  0.646176013f,
    0.761202385f,  0.648514401f,
    0.759209189f,  0.650846685f,
    0.757208847f,  0.653172843f,
    0.755201377f,  0.655492853f,
    0.753186799f,  0.657806693f,
    0.751165132f,  0.660114342f,
    0.749136395f,  0.662415778f,
    0.747100606f,  0.664710978f,
    0.745057785f,  0.666999922f,
    0.743007952f,  0.669282588f,
    0.740951125f,  0.671558955f,
    0.738887324f,  0.673829000f,
    0.736816569f,  0.676092704f,
    0.734738878f,  0.678350043f,
    0.732654272f,  0.680600998f,
    0.730562769f,  0.682845546f,
    0.728464390f,  0.685083668f,
    0.726359155f,  0.687315341f,
    0.724247083f,  0.689540545f,
    0.722128194f,  0.691759258f,
    0.720002508f,  0.693971461f,
    0.717870045f,  0.696177131f,
    0.715730825f,  0.698376249f,
    0.713584869f,  0.700568794f,
    0.711432196f,  0.702754744f,
    0.709272826f,  0.704934080f,
    0.707106781f,  0.707106781f,
    0.704934080f,  0.709272826f,
    0.702754744f,  0.711432196f,
    0.700568794f,  0.713584869f,
    0.698376249f,  0.715730825f,
    0.696177131f,  0.717870045f,
    0.693971461f,  0.720002508f,
    0.691759258f,  0.722128194f,
    0.689540545f,  0.724247083f,
    0.687315341f,  0.726359155f,
    0.685083668f,  0.728464390f,
    0.682845546f,  0.730562769f,
    0.680600998f,  0.732654272f,
    0.678350043f,  0.734738878f,
    0.676092704f,  0.736816569f,
    0.673829000f,  0.738887324f,
    0.671558955f,  0.740951125f,
    0.669282588f,  0.743007952f,
    0.666999922f,  0.745057785f,
    0.664710978f,  0.747100606f,
    0.662415778f,  0.749136395f,
    0.660114342f,  0.751165132f,
    0.657806693f,  0.753186799f,
    0.655492853f,  0.755201377f,
    0.653172843f,  0.757208847f,
    0.650846685f,  0.759209189f,
    0.648514401f,  0.761202385f,
    0.646176013f,  0.763188417f,
    0.643831543f,  0.765167266f,
    0.641481013f,  0.767138912f,
    0.639124445f,  0.769103338f,
    0.636761861f,  0.771060524f,
    0.634393284f,  0.773010453f,
    0.632018736f,  0.774953107f,
    0.629638239f,  0.776888466f,
    0.627251815f,  0.778816512f,
    0.624859488f,  0.780737229f,
    0.622461279f,  0.782650596f,
    0.620057212f,  0.784556597f,
    0.617647308f,  0.786455214f,
    0.615231591f,  0.788346428f,
    0.612810082f,  0.790230221f,
    0.610382806f,  0.792106577f,
    0.607949785f,  0.793975478f,
    0.605511041f,  0.795836905f,
    0.603066599f,  0.797690841f,
    0.600616479f,  0.799537269f,
    0.598160707f,  0.801376172f,
    0.595699304f,  0.803207531f,
    0.593232295f,  0.805031331f,
    0.590759702f,  0.806847554f,
    0.588281548f,  0.808656182f,
    0.585797857f,  0.810457198f,
    0.583308653f,  0.812250587f,
    0.580813958f,  0.814036330f,
    0.578313796f,  0.815814411f,
    0.575808191f,  0.817584813f,
    0.573297167f,  0.819347520f,
    0.570780746f,  0.821102515f,
    0.568258953f,  0.822849781f,
    0.565731811f,  0.824589303f,
    0.563199344f,  0.826321063f,
    0.560661576f,  0.828045045f,
    0.558118531f,  0.829761234f,
    0.555570233f,  0.831469612f,
    0.553016706f,  0.833170165f,
    0.550457973f,  0.834862875f,
    0.547894059f,  0.836547727f,
    0.545324988f,  0.838224706f,
    0.542750785f,  0.839893794f,
    0.540171473f,  0.841554977f,
    0.537587076f,  0.843208240f,
    0.534997620f,  0.844853565f,
    0.532403128f,  0.846490939f,
    0.529803625f,  0.848120345f,
    0.527199135f,  0.849741768f,
    0.524589683f,  0.851355193f,
    0.521975293f,  0.852960605f,
    0.519355990f,  0.854557988f,
    0.516731799f,  0.856147328f,
    0.514102744f,  0.857728610f,
    0.511468850f,  0.859301818f,
    0.508830143f,  0.860866939f,
    0.506186645f,  0.862423956f,
    0.503538384f,  0.863972856f,
    0.500885383f,  0.865513624f,
    0.498227667f,  0.867046246f,
    0.495565262f,  0.868570706f,
    0.492898192f,  0.870086991f,
    0.490226483f,  0.871595087f,
    0.487550160f,  0.873094978f,
    0.484869248f,  0.874586652f,
    0.482183772f,  0.876070094f,
    0.479493758f,  0.877545290f,
    0.476799230f,  0.879012226f,
    0.474100215f,  0.880470889f,
    0.471396737f,  0.881921264f,
    0.468688822f,  0.883363339f,
    0.465976496f,  0.884797098f,
    0.463259784f,  0.886222530f,
    0.460538711f,  0.887639620f,
    0.457813304f,  0.889048356f,
    0.455083587f,  0.890448723f,
    0.452349587f,  0.891840709f,
    0.449611330f,  0.893224301f,
    0.446868840f,  0.894599486f,
    0.444122145f,  0.895966250f,
    0.441371269f,  0.897324581f,
    0.438616239f,  0.898674466f,
    0.435857080f,  0.900015892f,
    0.433093819f,  0.901348847f,
    0.430326481f,  0.902673318f,
    0.427555093f,  0.903989293f,
    0.424779681f,  0.905296759f,
    0.422000271f,  0.906595705f,
    0.419216888f,  0.907886116f,
    0.416429560f,  0.909167983f,
    0.413638312f,  0.910441292f,
    0.410843171f,  0.911706032f,
    0.408044163f,  0.912962190f,
    0.405241314f,  0.914209756f,
    0.402434651f,  0.915448716f,
    0.399624200f,  0.916679060f,
    0.396809987f,  0.917900776f,
    0.393992040f,  0.919113852f,
    0.391170384f,  0.920318277f,
    0.388345047f,  0.921514039f,
    0.385516054f,  0.922701128f,
    0.382683432f,  0.923879533f,
    0.379847209f,  0.925049241f,
    0.377007410f,  0.926210242f,
    0.374164063f,  0.927362526f,
    0.371317194f,  0.928506080f,
    0.368466830f,  0.929640896f,
    0.365612998f,  0.930766961f,
    0.362755724f,  0.931884266f,
    0.359895037f,  0.932992799f,
    0.357030961f,  0.934092550f,
    0.354163525f,  0.935183510f,
    0.351292756f,  0.936265667f,
    0.348418680f,  0.937339012f,
    0.345541325f,  0.938403534f,
    0.342660717f,  0.939459224f,
    0.339776884f,  0.940506071f,
    0.336889853f,  0.941544065f,
    0.333999651f,  0.942573198f,
    0.331106306f,  0.943593458f,
    0.328209844f,  0.944604837f,
    0.325310292f,  0.945607325f,
    0.322407679f,  0.946600913f,
    0.319502031f,  0.947585591f,
    0.316593376f,  0.948561350f,
    0.313681740f,  0.949528181f,
    0.310767153f,  0.950486074f,
    0.307849640f,  0.951435021f,
    0.304929230f,  0.952375013f,
    0.302005949f,  0.953306040f,
    0.299079826f,  0.954228095f,
    0.296150888f,  0.955141168f,
    0.293219163f,  0.956045251f,
    0.290284677f,  0.956940336f,
    0.287347460f,  0.957826413f,
    0.284407537f,  0.958703475f,
    0.281464938f,  0.959571513f,
    0.278519689f,  0.960430519f,
    0.275571819f,  0.961280486f,
    0.272621355f,  0.962121404f,
    0.269668326f,  0.962953267f,
    0.266712757f,  0.963776066f,
    0.263754679f,  0.964589793f,
    0.260794118f,  0.965394442f,
    0.257831102f,  0.966190003f,
    0.254865660f,  0.966976471f,
    0.251897818f,  0.967753837f,
    0.248927606f,  0.968522094f,
    0.245955050f,  0.969281235f,
    0.242980180f,  0.970031253f,
    0.240003022f,  0.970772141f,
    0.237023606f,  0.971503891f,
    0.234041959f,  0.972226497f,
    0.231058108f,  0.972939952f,
    0.228072083f,  0.973644250f,
    0.225083911f,  0.974339383f,
    0.222093621f,  0.975025345f,
    0.219101240f,  0.975702130f,
    0.216106797f,  0.976369731f,
    0.213110320f,  0.977028143f,
    0.210111837f,  0.977677358f,
    0.207111376f,  0.978317371f,
    0.204108966f,  0.978948175f,
    0.201104635f,  0.979569766f,
    0.198098411f,  0.980182136f,
    0.195090322f,  0.980785280f,
    0.192080397f,  0.981379193f,
    0.189068664f,  0.981963869f,
    0.186055152f,  0.982539302f,
    0.183039888f,  0.983105487f,
    0.180022901f,  0.983662419f,
    0.177004220f,  0.984210092f,
    0.173983873f,  0.984748502f,
    0.170961889f,  0.985277642f,
    0.167938295f,  0.985797509f,
    0.164913120f,  0.986308097f,
    0.161886394f,  0.986809402f,
    0.158858143f,  0.987301418f,
    0.155828398f,  0.987784142f,
    0.152797185f,  0.988257568f,
    0.149764535f,  0.988721692f,
    0.146730474f,  0.989176510f,
    0.143695033f,  0.989622017f,
    0.140658239f,  0.990058210f,
    0.137620122f,  0.990485084f,
    0.134580709f,  0.990902635f,
    0.131540029f,  0.991310860f,
    0.128498111f,  0.991709754f,
    0.125454983f,  0.992099313f,
    0.122410675f,  0.992479535f,
    0.119365215f,  0.992850414f,
    0.116318631f,  0.993211949f,
    0.113270952f,  0.993564136f,
    0.110222207f,  0.993906970f,
    0.107172425f,  0.994240449f,
    0.104121634f,  0.994564571f,
    0.101069863f,  0.994879331f,
    0.098017140f,  0.995184727f,
    0.094963495f,  0.995480755f,
    0.091908956f,  0.995767414f,
    0.088853553f,  0.996044701f,
    0.085797312f,  0.996312612f,
    0.082740265f,  0.996571146f,
    0.079682438f,  0.996820299f,
    0.076623861f,  0.997060070f,
    0.073564564f,  0.997290457f,
    0.070504573f,  0.997511456f,
    0.067443920f,  0.997723067f,
    0.064382631f,  0.997925286f,
    0.061320736f,  0.998118113f,
    0.058258265f,  0.998301545f,
    0.055195244f,  0.998475581f,
    0.052131705f,  0.998640218f,
    0.049067674f,  0.998795456f,
    0.046003182f,  0.998941293f,
    0.042938257f,  0.999077728f,
    0.039872928f,  0.999204759f,
    0.036807223f,  0.999322385f,
    0.033741172f,  0.999430605f,
    0.030674803f,  0.999529418f,
    0.027608146f,  0.999618822f,
    0.024541229f,  0.999698819f,
    0.021474080f,  0.999769405f,
    0.018406730f,  0.999830582f,
    0.015339206f,  0.999882347f,
    0.012271538f,  0.999924702f,
    0.009203755f,  0.999957645f,
    0.006135885f,  0.999981175f,
    0.003067957f,  0.999995294f,
    0.000000000f,  1.000000000f,
   -0.003067957f,  0.999995294f,
   -0.006135885f,  0.999981175f,
   -0.009203755f,  0.999957645f,
   -0.012271538f,  0.999924702f,
   -0.015339206f,  0.999882347f,
   -0.018406730f,  0.999830582f,
   -0.021474080f,  0.999769405f,
   -0.024541229f,  0.999698819f,
   -0.027608146f,  0.999618822f,
   -0.030674803f,  0.999529418f,
   -0.033741172f,  0.999430605f,
   -0.036807223f,  0.999322385f,
   -0.039872928f,  0.999204759f,
   -0.042938257f,  0.999077728f,
   -0.046003182f,  0.998941293f,
   -0.049067674f,  0.998795456f,
   -0.052131705f,  0.998640218f,
   -0.055195244f,  0.998475581f,
   -0.058258265f,  0.998301545f,
   -0.061320736f,  0.998118113f,
   -0.064382631f,  0.997925286f,
   -0.067443920f,  0.997723067f,
   -0.070504573f,  0.997511456f,
   -0.073564564f,  0.997290457f,
   -0.076623861f,  0.997060070f,
   -0.079682438f,  0.996820299f,
   -0.082740265f,  0.996571146f,
   -0.085797312f,  0.996312612f,
   -0.088853553f,  0.996044701f,
   -0.091908956f,  0.995767414f,
   -0.094963495f,  0.995480755f,
   -0.098017140f,  0.995184727f,
   -0.101069863f,  0.994879331f,
   -0.104121634f,  0.994564571f,
   -0.107172425f,  0.994240449f,
   -0.110222207f,  0.993906970f,
   -0.113270952f,  0.993564136f,
   -0.116318631f,  0.993211949f,
   -0.119365215f,  0.992850414f,
   -0.122410675f,  0.992479535f,
   -0.125454983f,  0.992099313f,
   -0.128498111f,  0.991709754f,
   -0.131540029f,  0.991310860f,
   -0.134580709f,  0.990902635f,
   -0.137620122f,  0.990485084f,
   -0.140658239f,  0.990058210f,
   -0.143695033f,  0.989622017f,
   -0.146730474f,  0.989176510f,
   -0.149764535f,  0.988721692f,
   -0.152797185f,  0.988257568f,
   -0.155828398f,  0.987784142f,
   -0.158858143f,  0.987301418f,
   -0.161886394f,  0.986809402f,
   -0.164913120f,  0.986308097f,
   -0.167938295f,  0.985797509f,
   -0.170961889f,  0.985277642f,
   -0.173983873f,  0.984748502f,
   -0.177004220f,  0.984210092f,
   -0.180022901f,  0.983662419f,
   -0.183039888f,  0.983105487f,
   -0.186055152f,  0.982539302f,
   -0.189068664f,  0.981963869f,
   -0.192080397f,  0.981379193f,
   -0.195090322f,  0.980785280f,
   -0.198098411f,  0.980182136f,
   -0.201104635f,  0.979569766f,
   -0.204108966f,  0.978948175f,
   -0.207111376f,  0.978317371f,
   -0.210111837f,  0.977677358f,
   -0.213110320f,  0.977028143f,
   -0.216106797f,  0.976369731f,
   -0.219101240f,  0.975702130f,
   -0.222093621f,  0.975025345f,
   -0.225083911f,  0.974339383f,
   -0.228072083f,  0.973644250f,
   -0.231058108f,  0.972939952f,
   -0.234041959f,  0.972226497f,
   -0.237023606f,  0.971503891f,
   -0.240003022f,  0.970772141f,
   -0.242980180f,  0.970031253f,
   -0.245955050f,  0.969281235f,
   -0.248927606f,  0.968522094f,
   -0.251897818f,  0.967753837f,
   -0.254865660f,  0.966976471f,
   -0.257831102f,  0.966190003f,
   -0.260794118f,  0.965394442f,
   -0.263754679f,  0.964589793f,
   -0.266712757f,  0.963776066f,
   -0.269668326f,  0.962953267f,
   -0.272621355f,  0.962121404f,
   -0.275571819f,  0.961280486f,
   -0.278519689f,  0.960430519f,
   -0.281464938f,  0.959571513f,
   -0.284407537f,  0.958703475f,
   -0.287347460f,  0.957826413f,
   -0.290284677f,  0.956940336f,
   -0.293219163f,  0.956045251f,
   -0.296150888f,  0.955141168f,
   -0.299079826f,  0.954228095f,
   -0.302005949f,  0.953306040f,
   -0.304929230f,  0.952375013f,
   -0.307849640f,  0.951435021f,
   -0.310767153f,  0.950486074f,
   -0.313681740f,  0.949528181f,
   -0.316593376f,  0.948561350f,
   -0.319502031f,  0.947585591f,
   -0.322407679f,  0.946600913f,
   -0.325310292f,  0.945607325f,
   -0.328209844f,  0.944604837f,
   -0.331106306f,  0.943593458f,
   -0.333999651f,  0.942573198f,
   -0.336889853f,  0.941544065f,
   -0.339776884f,  0.940506071f,
   -0.342660717f,  0.939459224f,
   -0.345541325f,  0.938403534f,
   -0.348418680f,  0.937339012f,
   -0.351292756f,  0.936265667f,
   -0.354163525f,  0.935183510f,
   -0.357030961f,  0.934092550f,
   -0.359895037f,  0.932992799f,
   -0.362755724f,  0.931884266f,
   -0.365612998f,  0.930766961f,
   -0.368466830f,  0.929640896f,
   -0.371317194f,  0.928506080f,
   -0.374164063f,  0.927362526f,
   -0.377007410f,  0.926210242f,
   -0.379847209f,  0.925049241f,
   -0.382683432f,  0.923879533f,
   -0.385516054f,  0.922701128f,
   -0.388345047f,  0.921514039f,
   -0.391170384f,  0.920318277f,
   -0.393992040f,  0.919113852f,
   -0.396809987f,  0.917900776f,
   -0.399624200f,  0.916679060f,
   -0.402434651f,  0.915448716f,
   -0.405241314f,  0.914209756f,
   -0.408044163f,  0.912962190f,
   -0.410843171f,  0.911706032f,
   -0.413638312f,  0.910441292f,
   -0.416429560f,  0.909167983f,
   -0.419216888f,  0.907886116f,
   -0.422000271f,  0.906595705f,
   -0.424779681f,  0.905296759f,
   -0.427555093f,  0.903989293f,
   -0.430326481f,  0.902673318f,
   -0.433093819f,  0.901348847f,
   -0.435857080f,  0.900015892f,
   -0.438616239f,  0.898674466f,
   -0.441371269f,  0.897324581f,
   -0.444122145f,  0.895966250f,
   -0.446868840f,  0.894599486f,
   -0.449611330f,  0.893224301f,
   -0.452349587f,  0.891840709f,
   -0.455083587f,  0.890448723f,
   -0.457813304f,  0.889048356f,
   -0.460538711f,  0.887639620f,
   -0.463259784f,  0.886222530f,
   -0.465976496f,  0.884797098f,
   -0.468688822f,  0.883363339f,
   -0.471396737f,  0.881921264f,
   -0.474100215f,  0.880470889f,
   -0.476799230f,  0.879012226f,
   -0.479493758f,  0.877545290f,
   -0.482183772f,  0.876070094f,
   -0.484869248f,  0.874586652f,
   -0.487550160f,  0.873094978f,
   -0.490226483f,  0.871595087f,
   -0.492898192f,  0.870086991f,
   -0.495565262f,  0.868570706f,
   -0.498227667f,  0.867046246f,
   -0.500885383f,  0.865513624f,
   -0.503538384f,  0.863972856f,
   -0.506186645f,  0.862423956f,
   -0.508830143f,  0.860866939f,
   -0.511468850f,  0.859301818f,
   -0.514102744f,  0.857728610f,
   -0.516731799f,  0.856147328f,
   -0.519355990f,  0.854557988f,
   -0.521975293f,  0.852960605f,
   -0.524589683f,  0.851355193f,
   -0.527199135f,  0.849741768f,
   -0.529803625f,  0.848120345f,
   -0.532403128f,  0.846490939f,
   -0.534997620f,  0.844853565f,
   -0.537587076f,  0.843208240f,
   -0.540171473f,  0.841554977f,
   -0.542750785f,  0.839893794f,
   -0.545324988f,  0.838224706f,
   -0.547894059f,  0.836547727f,
   -0.550457973f,  0.834862875f,
   -0.553016706f,  0.833170165f,
   -0.555570233f,  0.831469612f,
   -0.558118531f,  0.829761234f,
   -0.560661576f,  0.828045045f,
   -0.563199344f,  0.826321063f,
   -0.565731811f,  0.824589303f,
   -0.568258953f,  0.822849781f,
   -0.570780746f,  0.821102515f,
   -0.573297167f,  0.819347520f,
   -0.575808191f,  0.817584813f,
   -0.578313796f,  0.815814411f,
   -0.580813958f,  0.814036330f,
   -0.583308653f,  0.812250587f,
   -0.585797857f,  0.810457198f,
   -0.588281548f,  0.808656182f,
   -0.590759702f,  0.806847554f,
   -0.593232295f,  0.805031331f,
   -0.595699304f,  0.803207531f,
   -0.598160707f,  0.801376172f,
   -0.600616479f,  0.799537269f,
   -0.603066599f,  0.797690841f,
   -0.605511041f,  0.795836905f,
   -0.607949785f,  0.793975478f,
   -0.610382806f,  0.792106577f,
   -0.612810082f,  0.790230221f,
   -0.615231591f,  0.788346428f,
   -0.617647308f,  0.786455214f,
   -0.620057212f,  0.784556597f,
   -0.622461279f,  0.782650596f,
   -0.624859488f,  0.780737229f,
   -0.627251815f,  0.778816512f,
   -0.629638239f,  0.776888466f,
   -0.632018736f,  0.774953107f,
   -0.634393284f,  0.773010453f,
   -0.636761861f,  0.771060524f,
   -0.639124445f,  0.769103338f,
   -0.641481013f,  0.767138912f,
   -0.643831543f,  0.765167266f,
   -0.646176013f,  0.763188417f,
   -0.648514401f,  0.761202385f,
   -0.650846685f,  0.759209189f,
   -0.653172843f,  0.757208847f,
   -0.655492853f,  0.755201377f,
   -0.657806693f,  0.753186799f,
   -0.660114342f,  0.751165132f,
   -0.662415778f,  0.749136395f,
   -0.664710978f,  0.747100606f,
   -0.666999922f,  0.745057785f,
   -0.669282588f,  0.743007952f,
   -0.671558955f,  0.740951125f,
   -0.673829000f,  0.738887324f,
   -0.676092704f,  0.736816569f,
   -0.678350043f,  0.734738878f,
   -0.680600998f,  0.732654272f,
   -0.682845546f,  0.730562769f,
   -0.685083668f,  0.728464390f,
   -0.687315341f,  0.726359155f,
   -0.689540545f,  0.724247083f,
   -0.691759258f,  0.722128194f,
   -0.693971461f,  0.720002508f,
   -0.696177131f,  0.717870045f,
   -0.698376249f,  0.715730825f,
   -0.700568794f,  0.713584869f,
   -0.702754744f,  0.711432196f,
   -0.704934080f,  0.709272826f,
   -0.707106781f,  0.707106781f,
   -0.709272826f,  0.704934080f,
   -0.711432196f,  0.702754744f,
   -0.713584869f,  0.700568794f,
   -0.715730825f,  0.698376249f,
   -0.717870045f,  0.696177131f,
   -0.720002508f,  0.693971461f,
   -0.722128194f,  0.691759258f,
   -0.724247083f,  0.689540545f,
   -0.726359155f,  0.687315341f,
   -0.728464390f,  0.685083668f,
   -0.730562769f,  0.682845546f,
   -0.732654272f,  0.680600998f,
   -0.734738878f,  0.678350043f,
   -0.736816569f,  0.676092704f,
   -0.738887324f,  0.673829000f,
   -0.740951125f,  0.671558955f,
   -0.743007952f,  0.669282588f,
   -0.745057785f,  0.666999922f,
   -0.747100606f,  0.664710978f,
   -0.749136395f,  0.662415778f,
   -0.751165132f,  0.660114342f,
   -0.753186799f,  0.657806693f,
   -0.755201377f,  0.655492853f,
   -0.757208847f,  0.653172843f,
   -0.759209189f,  0.650846685f,
   -0.761202385f,  0.648514401f,
   -0.763188417f,  0.646176013f,
   -0.765167266f,  0.643831543f,
   -0.767138912f,  0.641481013f,
   -0.769103338f,  0.639124445f,
   -0.771060524f,  0.636761861f,
   -0.773010453f,  0.634393284f,
   -0.774953107f,  0.632018736f,
   -0.776888466f,  0.629638239f,
   -0.778816512f,  0.627251815f,
   -0.780737229f,  0.624859488f,
   -0.782650596f,  0.622461279f,
   -0.784556597f,  0.620057212f,
   -0.786455214f,  0.617647308f,
   -0.788346428f,  0.615231591f,
   -0.790230221f,  0.612810082f,
   -0.792106577f,  0.610382806f,
   -0.793975478f,  0.607949785f,
   -0.795836905f,  0.605511041f,
   -0.797690841f,  0.603066599f,
   -0.799537269f,  0.600616479f,
   -0.801376172f,  0.598160707f,
   -0.803207531f,  0.595699304f,
   -0.805031331f,  0.593232295f,
   -0.806847554f,  0.590759702f,
   -0.808656182f,  0.588281548f,
   -0.810457198f,  0.585797857f,
   -0.812250587f,  0.583308653f,
   -0.814036330f,  0.580813958f,
   -0.815814411f,  0.578313796f,
   -0.817584813f,  0.575808191f,
   -0.819347520f,  0.573297167f,
   -0.821102515f,  0.570780746f,
   -0.822849781f,  0.568258953f,
   -0.824589303f,  0.565731811f,
   -0.826321063f,  0.563199344f,
   -0.828045045f,  0.560661576f,
   -0.829761234f,  0.558118531f,
   -0.831469612f,  0.555570233f,
   -0.833170165f,  0.553016706f,
   -0.834862875f,  0.550457973f,
   -0.836547727f,  0.547894059f,
   -0.838224706f,  0.545324988f,
   -0.839893794f,  0.542750785f,
   -0.841554977f,  0.540171473f,
   -0.843208240f,  0.537587076f,
   -0.844853565f,  0.534997620f,
   -0.846490939f,  0.532403128f,
   -0.848120345f,  0.529803625f,
   -0.849741768f,  0.527199135f,
   -0.851355193f,  0.524589683f,
   -0.852960605f,  0.521975293f,
   -0.854557988f,  0.519355990f,
   -0.856147328f,  0.516731799f,
   -0.857728610f,  0.514102744f,
   -0.859301818f,  0.511468850f,
   -0.860866939f,  0.508830143f,
   -0.862423956f,  0.506186645f,
   -0.863972856f,  0.503538384f,
   -0.865513624f,  0.500885383f,
   -0.867046246f,  0.498227667f,
   -0.868570706f,  0.495565262f,
   -0.870086991f,  0.492898192f,
   -0.871595087f,  0.490226483f,
   -0.873094978f,  0.487550160f,
   -0.874586652f,  0.484869248f,
   -0.876070094f,  0.482183772f,
   -0.877545290f,  0.479493758f,
   -0.879012226f,  0.476799230f,
   -0.880470889f,  0.474100215f,
   -0.881921264f,  0.471396737f,
   -0.883363339f,  0.468688822f,
   -0.884797098f,  0.465976496f,
   -0.886222530f,  0.463259784f,
   -0.887639620f,  0.460538711f,
   -0.889048356f,  0.457813304f,
   -0.890448723f,  0.455083587f,
   -0.891840709f,  0.452349587f,
   -0.893224301f,  0.449611330f,
   -0.894599486f,  0.446868840f,
   -0.895966250f,  0.444122145f,
   -0.897324581f,  0.441371269f,
   -0.898674466f,  0.438616239f,
   -0.900015892f,  0.435857080f,
   -0.901348847f,  0.433093819f,
   -0.902673318f,  0.430326481f,
   -0.903989293f,  0.427555093f,
   -0.905296759f,  0.424779681f,
   -0.906595705f,  0.422000271f,
   -0.907886116f,  0.419216888f,
   -0.909167983f,  0.416429560f,
   -0.910441292f,  0.413638312f,
   -0.911706032f,  0.410843171f,
   -0.912962190f,  0.408044163f,
   -0.914209756f,  0.405241314f,
   -0.915448716f,  0.402434651f,
   -0.916679060f,  0.399624200f,
   -0.917900776f,  0.396809987f,
   -0.919113852f,  0.393992040f,
   -0.920318277f,  0.391170384f,
   -0.921514039f,  0.388345047f,
   -0.922701128f,  0.385516054f,
   -0.923879533f,  0.382683432f,
   -0.925049241f,  0.379847209f,
   -0.926210242f,  0.377007410f,
   -0.927362526f,  0.374164063f,
   -0.928506080f,  0.371317194f,
   -0.929640896f,  0.368466830f,
   -0.930766961f,  0.365612998f,
   -0.931884266f,  0.362755724f,
   -0.932992799f,  0.359895037f,
   -0.934092550f,  0.357030961f,
   -0.935183510f,  0.354163525f,
   -0.936265667f,  0.351292756f,
   -0.937339012f,  0.348418680f,
   -0.938403534f,  0.345541325f,
   -0.939459224f,  0.342660717f,
   -0.940506071f,  0.339776884f,
   -0.941544065f,  0.336889853f,
   -0.942573198f,  0.333999651f,
   -0.943593458f,  0.331106306f,
   -0.944604837f,  0.328209844f,
   -0.945607325f,  0.325310292f,
   -0.946600913f,  0.322407679f,
   -0.947585591f,  0.319502031f,
   -0.948561350f,  0.316593376f,
   -0.949528181f,  0.313681740f,
   -0.950486074f,  0.310767153f,
   -0.951435021f,  0.307849640f,
   -0.952375013f,  0.304929230f,
   -0.953306040f,  0.302005949f,
   -0.954228095f,  0.299079826f,
   -0.955141168f,  0.296150888f,
   -0.956045251f,  0.293219163f,
   -0.956940336f,  0.290284677f,
   -0.957826413f,  0.287347460f,
   -0.958703475f,  0.284407537f,
   -0.959571513f,  0.281464938f,
   -0.960430519f,  0.278519689f,
   -0.961280486f,  0.275571819f,
   -0.962121404f,  0.272621355f,
   -0.962953267f,  0.269668326f,
   -0.963776066f,  0.266712757f,
   -0.964589793f,  0.263754679f,
   -0.965394442f,  0.260794118f,
   -0.966190003f,  0.257831102f,
   -0.966976471f,  0.254865660f,
   -0.967753837f,  0.251897818f,
   -0.968522094f,  0.248927606f,
   -0.969281235f,  0.245955050f,
   -0.970031253f,  0.242980180f,
   -0.970772141f,  0.240003022f,
   -0.971503891f,  0.237023606f,
   -0.972226497f,  0.234041959f,
   -0.972939952f,  0.231058108f,
   -0.973644250f,  0.228072083f,
   -0.974339383f,  0.225083911f,
   -0.975025345f,  0.222093621f,
   -0.975702130f,  0.219101240f,
   -0.976369731f,  0.216106797f,
   -0.977028143f,  0.213110320f,
   -0.977677358f,  0.210111837f,
   -0.978317371f,  0.207111376f,
   -0.978948175f,  0.204108966f,
   -0.979569766f,  0.201104635f,
   -0.980182136f,  0.198098411f,
   -0.980785280f,  0.195090322f,
   -0.981379193f,  0.192080397f,
   -0.981963869f,  0.189068664f,
   -0.982539302f,  0.186055152f,
   -0.983105487f,  0.183039888f,
   -0.983662419f,  0.180022901f,
   -0.984210092f,  0.177004220f,
   -0.984748502f,  0.173983873f,
   -0.985277642f,  0.170961889f,
   -0.985797509f,  0.167938295f,
   -0.986308097f,  0.164913120f,
   -0.986809402f,  0.161886394f,
   -0.987301418f,  0.158858143f,
   -0.987784142f,  0.155828398f,
   -0.988257568f,  0.152797185f,
   -0.988721692f,  0.149764535f,
   -0.989176510f,  0.146730474f,
   -0.989622017f,  0.143695033f,
   -0.990058210f,  0.140658239f,
   -0.990485084f,  0.137620122f,
   -0.990902635f,  0.134580709f,
   -0.991310860f,  0.131540029f,
   -0.991709754f,  0.128498111f,
   -0.992099313f,  0.125454983f,
   -0.992479535f,  0.122410675f,
   -0.992850414f,  0.119365215f,
   -0.993211949f,  0.116318631f,
   -0.993564136f,  0.113270952f,
   -0.993906970f,  0.110222207f,
   -0.994240449f,  0.107172425f,
   -0.994564571f,  0.104121634f,
   -0.994879331f,  0.101069863f,
   -0.995184727f,  0.098017140f,
   -0.995480755f,  0.094963495f,
   -0.995767414f,  0.091908956f,
   -0.996044701f,  0.088853553f,
   -0.996312612f,  0.085797312f,
   -0.996571146f,  0.082740265f,
   -0.996820299f,  0.079682438f,
   -0.997060070f,  0.076623861f,
   -0.997290457f,  0.073564564f,
   -0.997511456f,  0.070504573f,
   -0.997723067f,  0.067443920f,
   -0.997925286f,  0.064382631f,
   -0.998118113f,  0.061320736f,
   -0.998301545f,  0.058258265f,
   -0.998475581f,  0.055195244f,
   -0.998640218f,  0.052131705f,
   -0.998795456f,  0.049067674f,
   -0.998941293f,  0.046003182f,
   -0.999077728f,  0.042938257f,
   -0.999204759f,  0.039872928f,
   -0.999322385f,  0.036807223f,
   -0.999430605f,  0.033741172f,
   -0.999529418f,  0.030674803f,
   -0.999618822f,  0.027608146f,
   -0.999698819f,  0.024541229f,
   -0.999769405f,  0.021474080f,
   -0.999830582f,  0.018406730f,
   -0.999882347f,  0.015339206f,
   -0.999924702f,  0.012271538f,
   -0.999957645f,  0.009203755f,
   -0.999981175f,  0.006135885f,
   -0.999995294f,  0.003067957f,
   -1.000000000f,  0.000000000f,
   -0.999995294f, -0.003067957f,
   -0.999981175f, -0.006135885f,
   -0.999957645f, -0.009203755f,
   -0.999924702f, -0.012271538f,
   -0.999882347f, -0.015339206f,
   -0.999830582f, -0.018406730f,
   -0.999769405f, -0.021474080f,
   -0.999698819f, -0.024541229f,
   -0.999618822f, -0.027608146f,
   -0.999529418f, -0.030674803f,
   -0.999430605f, -0.033741172f,
   -0.999322385f, -0.036807223f,
   -0.999204759f, -0.039872928f,
   -0.999077728f, -0.042938257f,
   -0.998941293f, -0.046003182f,
   -0.998795456f, -0.049067674f,
   -0.998640218f, -0.052131705f,
   -0.998475581f, -0.055195244f,
   -0.998301545f, -0.058258265f,
   -0.998118113f, -0.061320736f,
   -0.997925286f, -0.064382631f,
   -0.997723067f, -0.067443920f,
   -0.997511456f, -0.070504573f,
   -0.997290457f, -0.073564564f,
   -0.997060070f, -0.076623861f,
   -0.996820299f, -0.079682438f,
   -0.996571146f, -0.082740265f,
   -0.996312612f, -0.085797312f,
   -0.996044701f, -0.088853553f,
   -0.995767414f, -0.091908956f,
   -0.995480755f, -0.094963495f,
   -0.995184727f, -0.098017140f,
   -0.994879331f, -0.101069863f,
   -0.994564571f, -0.104121634f,
   -0.994240449f, -0.107172425f,
   -0.993906970f, -0.110222207f,
   -0.993564136f, -0.113270952f,
   -0.993211949f, -0.116318631f,
   -0.992850414f, -0.119365215f,
   -0.992479535f, -0.122410675f,
   -0.992099313f, -0.125454983f,
   -0.991709754f, -0.128498111f,
   -0.991310860f, -0.131540029f,
   -0.990902635f, -0.134580709f,
   -0.990485084f, -0.137620122f,
   -0.990058210f, -0.140658239f,
   -0.989622017f, -0.143695033f,
   -0.989176510f, -0.146730474f,
   -0.988721692f, -0.149764535f,
   -0.988257568f, -0.152797185f,
   -0.987784142f, -0.155828398f,
   -0.987301418f, -0.158858143f,
   -0.986809402f, -0.161886394f,
   -0.986308097f, -0.164913120f,
   -0.985797509f, -0.167938295f,
   -0.985277642f, -0.170961889f,
   -0.984748502f, -0.173983873f,
   -0.984210092f, -0.177004220f,
   -0.983662419f, -0.180022901f,
   -0.983105487f, -0.183039888f,
   -0.982539302f, -0.186055152f,
   -0.981963869f, -0.189068664f,
   -0.981379193f, -0.192080397f,
   -0.980785280f, -0.195090322f,
   -0.980182136f, -0.198098411f,
   -0.979569766f, -0.201104635f,
   -0.978948175f, -0.204108966f,
   -0.978317371f, -0.207111376f,
   -0.977677358f, -0.210111837f,
   -0.977028143f, -0.213110320f,
   -0.976369731f, -0.216106797f,
   -0.975702130f, -0.219101240f,
   -0.975025345f, -0.222093621f,
   -0.974339383f, -0.225083911f,
   -0.973644250f, -0.228072083f,
   -0.972939952f, -0.231058108f,
   -0.972226497f, -0.234041959f,
   -0.971503891f, -0.237023606f,
   -0.970772141f, -0.240003022f,
   -0.970031253f, -0.242980180f,
   -0.969281235f, -0.245955050f,
   -0.968522094f, -0.248927606f,
   -0.967753837f, -0.251897818f,
   -0.966976471f, -0.254865660f,
   -0.966190003f, -0.257831102f,
   -0.965394442f, -0.260794118f,
   -0.964589793f, -0.263754679f,
   -0.963776066f, -0.266712757f,
   -0.962953267f, -0.269668326f,
   -0.962121404f, -0.272621355f,
   -0.961280486f, -0.275571819f,
   -0.960430519f, -0.278519689f,
   -0.959571513f, -0.281464938f,
   -0.958703475f, -0.284407537f,
   -0.957826413f, -0.287347460f,
   -0.956940336f, -0.290284677f,
   -0.956045251f, -0.293219163f,
   -0.955141168f, -0.296150888f,
   -0.954228095f, -0.299079826f,
   -0.953306040f, -0.302005949f,
   -0.952375013f, -0.304929230f,
   -0.951435021f, -0.307849640f,
   -0.950486074f, -0.310767153f,
   -0.949528181f, -0.313681740f,
   -0.948561350f, -0.316593376f,
   -0.947585591f, -0.319502031f,
   -0.946600913f, -0.322407679f,
   -0.945607325f, -0.325310292f,
   -0.944604837f, -0.328209844f,
   -0.943593458f, -0.331106306f,
   -0.942573198f, -0.333999651f,
   -0.941544065f, -0.336889853f,
   -0.940506071f, -0.339776884f,
   -0.939459224f, -0.342660717f,
   -0.938403534f, -0.345541325f,
   -0.937339012f, -0.348418680f,
   -0.936265667f, -0.351292756f,
   -0.935183510f, -0.354163525f,
   -0.934092550f, -0.357030961f,
   -0.932992799f, -0.359895037f,
   -0.931884266f, -0.362755724f,
   -0.930766961f, -0.365612998f,
   -0.929640896f, -0.368466830f,
   -0.928506080f, -0.371317194f,
   -0.927362526f, -0.374164063f,
   -0.926210242f, -0.377007410f,
   -0.925049241f, -0.379847209f,
   -0.923879533f, -0.382683432f,
   -0.922701128f, -0.385516054f,
   -0.921514039f, -0.388345047f,
   -0.920318277f, -0.391170384f,
   -0.919113852f, -0.393992040f,
   -0.917900776f, -0.396809987f,
   -0.916679060f, -0.399624200f,
   -0.915448716f, -0.402434651f,
   -0.914209756f, -0.405241314f,
   -0.912962190f, -0.408044163f,
   -0.911706032f, -0.410843171f,
   -0.910441292f, -0.413638312f,
   -0.909167983f, -0.416429560f,
   -0.907886116f, -0.419216888f,
   -0.906595705f, -0.422000271f,
   -0.905296759f, -0.424779681f,
   -0.903989293f, -0.427555093f,
   -0.902673318f, -0.430326481f,
   -0.901348847f, -0.433093819f,
   -0.900015892f, -0.435857080f,
   -0.898674466f, -0.438616239f,
   -0.897324581f, -0.441371269f,
   -0.895966250f, -0.444122145f,
   -0.894599486f, -0.446868840f,
   -0.893224301f, -0.449611330f,
   -0.891840709f, -0.452349587f,
   -0.890448723f, -0.455083587f,
   -0.889048356f, -0.457813304f,
   -0.887639620f, -0.460538711f,
   -0.886222530f, -0.463259784f,
   -0.884797098f, -0.465976496f,
   -0.883363339f, -0.468688822f,
   -0.881921264f, -0.471396737f,
   -0.880470889f, -0.474100215f,
   -0.879012226f, -0.476799230f,
   -0.877545290f, -0.479493758f,
   -0.876070094f, -0.482183772f,
   -0.874586652f, -0.484869248f,
   -0.873094978f, -0.487550160f,
   -0.871595087f, -0.490226483f,
   -0.870086991f, -0.492898192f,
   -0.868570706f, -0.495565262f,
   -0.867046246f, -0.498227667f,
   -0.865513624f, -0.500885383f,
   -0.863972856f, -0.503538384f,
   -0.862423956f, -0.506186645f,
   -0.860866939f, -0.508830143f,
   -0.859301818f, -0.511468850f,
   -0.857728610f, -0.514102744f,
   -0.856147328f, -0.516731799f,
   -0.854557988f, -0.519355990f,
   -0.852960605f, -0.521975293f,
   -0.851355193f, -0.524589683f,
   -0.849741768f, -0.527199135f,
   -0.848120345f, -0.529803625f,
   -0.846490939f, -0.532403128f,
   -0.844853565f, -0.534997620f,
   -0.843208240f, -0.537587076f,
   -0.841554977f, -0.540171473f,
   -0.839893794f, -0.542750785f,
   -0.838224706f, -0.545324988f,
   -0.836547727f, -0.547894059f,
   -0.834862875f, -0.550457973f,
   -0.833170165f, -0.553016706f,
   -0.831469612f, -0.555570233f,
   -0.829761234f, -0.558118531f,
   -0.828045045f, -0.560661576f,
   -0.826321063f, -0.563199344f,
   -0.824589303f, -0.565731811f,
   -0.822849781f, -0.568258953f,
   -0.821102515f, -0.570780746f,
   -0.819347520f, -0.573297167f,
   -0.817584813f, -0.575808191f,
   -0.815814411f, -0.578313796f,
   -0.814036330f, -0.580813958f,
   -0.812250587f, -0.583308653f,
   -0.810457198f, -0.585797857f,
   -0.808656182f, -0.588281548f,
   -0.806847554f, -0.590759702f,
   -0.805031331f, -0.593232295f,
   -0.803207531f, -0.595699304f,
   -0.801376172f, -0.598160707f,
   -0.799537269f, -0.600616479f,
   -0.797690841f, -0.603066599f,
   -0.795836905f, -0.605511041f,
   -0.793975478f, -0.607949785f,
   -0.792106577f, -0.610382806f,
   -0.790230221f, -0.612810082f,
   -0.788346428f, -0.615231591f,
   -0.786455214f, -0.617647308f,
   -0.784556597f, -0.620057212f,
   -0.782650596f, -0.622461279f,
   -0.780737229f, -0.624859488f,
   -0.778816512f, -0.627251815f,
   -0.776888466f, -0.629638239f,
   -0.774953107f, -0.632018736f,
   -0.773010453f, -0.634393284f,
   -0.771060524f, -0.636761861f,
   -0.769103338f, -0.639124445f,
   -0.767138912f, -0.641481013f,
   -0.765167266f, -0.643831543f,
   -0.763188417f, -0.646176013f,
   -0.761202385f, -0.648514401f,
   -0.759209189f, -0.650846685f,
   -0.757208847f, -0.653172843f,
   -0.755201377f, -0.655492853f,
   -0.753186799f, -0.657806693f,
   -0.751165132f, -0.660114342f,
   -0.749136395f, -0.662415778f,
   -0.747100606f, -0.664710978f,
   -0.745057785f, -0.666999922f,
   -0.743007952f, -0.669282588f,
   -0.740951125f, -0.671558955f,
   -0.738887324f, -0.673829000f,
   -0.736816569f, -0.676092704f,
   -0.734738878f, -0.678350043f,
   -0.732654272f, -0.680600998f,
   -0.730562769f, -0.682845546f,
   -0.728464390f, -0.685083668f,
   -0.726359155f, -0.687315341f,
   -0.724247083f, -0.689540545f,
   -0.722128194f, -0.691759258f,
   -0.720002508f, -0.693971461f,
   -0.717870045f, -0.696177131f,
   -0.715730825f, -0.698376249f,
   -0.713584869f, -0.700568794f,
   -0.711432196f, -0.702754744f,
   -0.709272826f, -0.704934080f,
   -0.707106781f, -0.707106781f,
   -0.704934080f, -0.709272826f,
   -0.702754744f, -0.711432196f,
   -0.700568794f, -0.713584869f,
   -0.698376249f, -0.715730825f,
   -0.696177131f, -0.717870045f,
   -0.693971461f, -0.720002508f,
   -0.691759258f, -0.722128194f,
   -0.689540545f, -0.724247083f,
   -0.687315341f, -0.726359155f,
   -0.685083668f, -0.728464390f,
   -0.682845546f, -0.730562769f,
   -0.680600998f, -0.732654272f,
   -0.678350043f, -0.734738878f,
   -0.676092704f, -0.736816569f,
   -0.673829000f, -0.738887324f,
   -0.671558955f, -0.740951125f,
   -0.669282588f, -0.743007952f,
   -0.666999922f, -0.745057785f,
   -0.664710978f, -0.747100606f,
   -0.662415778f, -0.749136395f,
   -0.660114342f, -0.751165132f,
   -0.657806693f, -0.753186799f,
   -0.655492853f, -0.755201377f,
   -0.653172843f, -0.757208847f,
   -0.650846685f, -0.759209189f,
   -0.648514401f, -0.761202385f,
   -0.646176013f, -0.763188417f,
   -0.643831543f, -0.765167266f,
   -0.641481013f, -0.767138912f,
   -0.639124445f, -0.769103338f,
   -0.636761861f, -0.771060524f,
   -0.634393284f, -0.773010453f,
   -0.632018736f, -0.774953107f,
   -0.629638239f, -0.776888466f,
   -0.627251815f, -0.778816512f,
   -0.624859488f, -0.780737229f,
   -0.622461279f, -0.782650596f,
   -0.620057212f, -0.784556597f,
   -0.617647308f, -0.786455214f,
   -0.615231591f, -0.788346428f,
   -0.612810082f, -0.790230221f,
   -0.610382806f, -0.792106577f,
   -0.607949785f, -0.793975478f,
   -0.605511041f, -0.795836905f,
   -0.603066599f, -0.797690841f,
   -0.600616479f, -0.799537269f,
   -0.598160707f, -0.801376172f,
   -0.595699304f, -0.803207531f,
   -0.593232295f, -0.805031331f,
   -0.590759702f, -0.806847554f,
   -0.588281548f, -0.808656182f,
   -0.585797857f, -0.810457198f,
   -0.583308653f, -0.812250587f,
   -0.580813958f, -0.814036330f,
   -0.578313796f, -0.815814411f,
   -0.575808191f, -0.817584813f,
   -0.573297167f, -0.819347520f,
   -0.570780746f, -0.821102515f,
   -0.568258953f, -0.822849781f,
   -0.565731811f, -0.824589303f,
   -0.563199344f, -0.826321063f,
   -0.560661576f, -0.828045045f,
   -0.558118531f, -0.829761234f,
   -0.555570233f, -0.831469612f,
   -0.553016706f, -0.833170165f,
   -0.550457973f, -0.834862875f,
   -0.547894059f, -0.836547727f,
   -0.545324988f, -0.838224706f,
   -0.542750785f, -0.839893794f,
   -0.540171473f, -0.841554977f,
   -0.537587076f, -0.843208240f,
   -0.534997620f, -0.844853565f,
   -0.532403128f, -0.846490939f,
   -0.529803625f, -0.848120345f,
   -0.527199135f, -0.849741768f,
   -0.524589683f, -0.851355193f,
   -0.521975293f, -0.852960605f,
   -0.519355990f, -0.854557988f,
   -0.516731799f, -0.856147328f,
   -0.514102744f, -0.857728610f,
   -0.511468850f, -0.859301818f,
   -0.508830143f, -0.860866939f,
   -0.506186645f, -0.862423956f,
   -0.503538384f, -0.863972856f,
   -0.500885383f, -0.865513624f,
   -0.498227667f, -0.867046246f,
   -0.495565262f, -0.868570706f,
   -0.492898192f, -0.870086991f,
   -0.490226483f, -0.871595087f,
   -0.487550160f, -0.873094978f,
   -0.484869248f, -0.874586652f,
   -0.482183772f, -0.876070094f,
   -0.479493758f, -0.877545290f,
   -0.476799230f, -0.879012226f,
   -0.474100215f, -0.880470889f,
   -0.471396737f, -0.881921264f,
   -0.468688822f, -0.883363339f,
   -0.465976496f, -0.884797098f,
   -0.463259784f, -0.886222530f,
   -0.460538711f, -0.887639620f,
   -0.457813304f, -0.889048356f,
   -0.455083587f, -0.890448723f,
   -0.452349587f, -0.891840709f,
   -0.449611330f, -0.893224301f,
   -0.446868840f, -0.894599486f,
   -0.444122145f, -0.895966250f,
   -0.441371269f, -0.897324581f,
   -0.438616239f, -0.898674466f,
   -0.435857080f, -0.900015892f,
   -0.433093819f, -0.901348847f,
   -0.430326481f, -0.902673318f,
   -0.427555093f, -0.903989293f,
   -0.424779681f, -0.905296759f,
   -0.422000271f, -0.906595705f,
   -0.419216888f, -0.907886116f,
   -0.416429560f, -0.909167983f,
   -0.413638312f, -0.910441292f,
   -0.410843171f, -0.911706032f,
   -0.408044163f, -0.912962190f,
   -0.405241314f, -0.914209756f,
   -0.402434651f, -0.915448716f,
   -0.399624200f, -0.916679060f,
   -0.396809987f, -0.917900776f,
   -0.393992040f, -0.919113852f,
   -0.391170384f, -0.920318277f,
   -0.388345047f, -0.921514039f,
   -0.385516054f, -0.922701128f,
   -0.382683432f, -0.923879533f,
   -0.379847209f, -0.925049241f,
   -0.377007410f, -0.926210242f,
   -0.374164063f, -0.927362526f,
   -0.371317194f, -0.928506080f,
   -0.368466830f, -0.929640896f,
   -0.365612998f, -0.930766961f,
   -0.362755724f, -0.931884266f,
   -0.359895037f, -0.932992799f,
   -0.357030961f, -0.934092550f,
   -0.354163525f, -0.935183510f,
   -0.351292756f, -0.936265667f,
   -0.348418680f, -0.937339012f,
   -0.345541325f, -0.938403534f,
   -0.342660717f, -0.939459224f,
   -0.339776884f, -0.940506071f,
   -0.336889853f, -0.941544065f,
   -0.333999651f, -0.942573198f,
   -0.331106306f, -0.943593458f,
   -0.328209844f, -0.944604837f,
   -0.325310292f, -0.945607325f,
   -0.322407679f, -0.946600913f,
   -0.319502031f, -0.947585591f,
   -0.316593376f, -0.948561350f,
   -0.313681740f, -0.949528181f,
   -0.310767153f, -0.950486074f,
   -0.307849640f, -0.951435021f,
   -0.304929230f, -0.952375013f,
   -0.302005949f, -0.953306040f,
   -0.299079826f, -0.954228095f,
   -0.296150888f, -0.955141168f,
   -0.293219163f, -0.956045251f,
   -0.290284677f, -0.956940336f,
   -0.287347460f, -0.957826413f,
   -0.284407537f, -0.958703475f,
   -0.281464938f, -0.959571513f,
   -0.278519689f, -0.960430519f,
   -0.275571819f, -0.961280486f,
   -0.272621355f, -0.962121404f,
   -0.269668326f, -0.962953267f,
   -0.266712757f, -0.963776066f,
   -0.263754679f, -0.964589793f,
   -0.260794118f, -0.965394442f,
   -0.257831102f, -0.966190003f,
   -0.254865660f, -0.966976471f,
   -0.251897818f, -0.967753837f,
   -0.248927606f, -0.968522094f,
   -0.245955050f, -0.969281235f,
   -0.242980180f, -0.970031253f,
   -0.240003022f, -0.970772141f,
   -0.237023606f, -0.971503891f,
   -0.234041959f, -0.972226497f,
   -0.231058108f, -0.972939952f,
   -0.228072083f, -0.973644250f,
   -0.225083911f, -0.974339383f,
   -0.222093621f, -0.975025345f,
   -0.219101240f, -0.975702130f,
   -0.216106797f, -0.976369731f,
   -0.213110320f, -0.977028143f,
   -0.210111837f, -0.977677358f,
   -0.207111376f, -0.978317371f,
   -0.204108966f, -0.978948175f,
   -0.201104635f, -0.979569766f,
   -0.198098411f, -0.980182136f,
   -0.195090322f, -0.980785280f,
   -0.192080397f, -0.981379193f,
   -0.189068664f, -0.981963869f,
   -0.186055152f, -0.982539302f,
   -0.183039888f, -0.983105487f,
   -0.180022901f, -0.983662419f,
   -0.177004220f, -0.984210092f,
   -0.173983873f, -0.984748502f,
   -0.170961889f, -0.985277642f,
   -0.167938295f, -0.985797509f,
   -0.164913120f, -0.986308097f,
   -0.161886394f, -0.986809402f,
   -0.158858143f, -0.987301418f,
   -0.155828398f, -0.987784142f,
   -0.152797185f, -0.988257568f,
   -0.149764535f, -0.988721692f,
   -0.146730474f, -0.989176510f,
   -0.143695033f, -0.989622017f,
   -0.140658239f, -0.990058210f,
   -0.137620122f, -0.990485084f,
   -0.134580709f, -0.990902635f,
   -0.131540029f, -0.991310860f,
   -0.128498111f, -0.991709754f,
   -0.125454983f, -0.992099313f,
   -0.122410675f, -0.992479535f,
   -0.119365215f, -0.992850414f,
   -0.116318631f, -0.993211949f,
   -0.113270952f, -0.993564136f,
   -0.110222207f, -0.993906970f,
   -0.107172425f, -0.994240449f,
   -0.104121634f, -0.994564571f,
   -0.101069863f, -0.994879331f,
   -0.098017140f, -0.995184727f,
   -0.094963495f, -0.995480755f,
   -0.091908956f, -0.995767414f,
   -0.088853553f, -0.996044701f,
   -0.085797312f, -0.996312612f,
   -0.082740265f, -0.996571146f,
   -0.079682438f, -0.996820299f,
   -0.076623861f, -0.997060070f,
   -0.073564564f, -0.997290457f,
   -0.070504573f, -0.997511456f,
   -0.067443920f, -0.997723067f,
   -0.064382631f, -0.997925286f,
   -0.061320736f, -0.998118113f,
   -0.058258265f, -0.998301545f,
   -0.055195244f, -0.998475581f,
   -0.052131705f, -0.998640218f,
   -0.049067674f, -0.998795456f,
   -0.046003182f, -0.998941293f,
   -0.042938257f, -0.999077728f,
   -0.039872928f, -0.999204759f,
   -0.036807223f, -0.999322385f,
   -0.033741172f, -0.999430605f,
   -0.030674803f, -0.999529418f,
   -0.027608146f, -0.999618822f,
   -0.024541229f, -0.999698819f,
   -0.021474080f, -0.999769405f,
   -0.018406730f, -0.999830582f,
   -0.015339206f, -0.999882347f,
   -0.012271538f, -0.999924702f,
   -0.009203755f, -0.999957645f,
   -0.006135885f, -0.999981175f,
   -0.003067957f, -0.999995294f,
   -0.000000000f, -1.000000000f,
    0.003067957f, -0.999995294f,
    0.006135885f, -0.999981175f,
    0.009203755f, -0.999957645f,
    0.012271538f, -0.999924702f,
    0.015339206f, -0.999882347f,
    0.018406730f, -0.999830582f,
    0.021474080f, -0.999769405f,
    0.024541229f, -0.999698819f,
    0.027608146f, -0.999618822f,
    0.030674803f, -0.999529418f,
    0.033741172f, -0.999430605f,
    0.036807223f, -0.999322385f,
    0.039872928f, -0.999204759f,
    0.042938257f, -0.999077728f,
    0.046003182f, -0.998941293f,
    0.049067674f, -0.998795456f,
    0.052131705f, -0.998640218f,
    0.055195244f, -0.998475581f,
    0.058258265f, -0.998301545f,
    0.061320736f, -0.998118113f,
    0.064382631f, -0.997925286f,
    0.067443920f, -0.997723067f,
    0.070504573f, -0.997511456f,
    0.073564564f, -0.997290457f,
    0.076623861f, -0.997060070f,
    0.079682438f, -0.996820299f,
    0.082740265f, -0.996571146f,
    0.085797312f, -0.996312612f,
    0.088853553f, -0.996044701f,
    0.091908956f, -0.995767414f,
    0.094963495f, -0.995480755f,
    0.098017140f, -0.995184727f,
    0.101069863f, -0.994879331f,
    0.104121634f, -0.994564571f,
    0.107172425f, -0.994240449f,
    0.110222207f, -0.993906970f,
    0.113270952f, -0.993564136f,
    0.116318631f, -0.993211949f,
    0.119365215f, -0.992850414f,
    0.122410675f, -0.992479535f,
    0.125454983f, -0.992099313f,
    0.128498111f, -0.991709754f,
    0.131540029f, -0.991310860f,
    0.134580709f, -0.990902635f,
    0.137620122f, -0.990485084f,
    0.140658239f, -0.990058210f,
    0.143695033f, -0.989622017f,
    0.146730474f, -0.989176510f,
    0.149764535f, -0.988721692f,
    0.152797185f, -0.988257568f,
    0.155828398f, -0.987784142f,
    0.158858143f, -0.987301418f,
    0.161886394f, -0.986809402f,
    0.164913120f, -0.986308097f,
    0.167938295f, -0.985797509f,
    0.170961889f, -0.985277642f,
    0.173983873f, -0.984748502f,
    0.177004220f, -0.984210092f,
    0.180022901f, -0.983662419f,
    0.183039888f, -0.983105487f,
    0.186055152f, -0.982539302f,
    0.189068664f, -0.981963869f,
    0.192080397f, -0.981379193f,
    0.195090322f, -0.980785280f,
    0.198098411f, -0.980182136f,
    0.201104635f, -0.979569766f,
    0.204108966f, -0.978948175f,
    0.207111376f, -0.978317371f,
    0.210111837f, -0.977677358f,
    0.213110320f, -0.977028143f,
    0.216106797f, -0.976369731f,
    0.219101240f, -0.975702130f,
    0.222093621f, -0.975025345f,
    0.225083911f, -0.974339383f,
    0.228072083f, -0.973644250f,
    0.231058108f, -0.972939952f,
    0.234041959f, -0.972226497f,
    0.237023606f, -0.971503891f,
    0.240003022f, -0.970772141f,
    0.242980180f, -0.970031253f,
    0.245955050f, -0.969281235f,
    0.248927606f, -0.968522094f,
    0.251897818f, -0.967753837f,
    0.254865660f, -0.966976471f,
    0.257831102f, -0.966190003f,
    0.260794118f, -0.965394442f,
    0.263754679f, -0.964589793f,
    0.266712757f, -0.963776066f,
    0.269668326f, -0.962953267f,
    0.272621355f, -0.962121404f,
    0.275571819f, -0.961280486f,
    0.278519689f, -0.960430519f,
    0.281464938f, -0.959571513f,
    0.284407537f, -0.958703475f,
    0.287347460f, -0.957826413f,
    0.290284677f, -0.956940336f,
    0.293219163f, -0.956045251f,
    0.296150888f, -0.955141168f,
    0.299079826f, -0.954228095f,
    0.302005949f, -0.953306040f,
    0.304929230f, -0.952375013f,
    0.307849640f, -0.951435021f,
    0.310767153f, -0.950486074f,
    0.313681740f, -0.949528181f,
    0.316593376f, -0.948561350f,
    0.319502031f, -0.947585591f,
    0.322407679f, -0.946600913f,
    0.325310292f, -0.945607325f,
    0.328209844f, -0.944604837f,
    0.331106306f, -0.943593458f,
    0.333999651f, -0.942573198f,
    0.336889853f, -0.941544065f,
    0.339776884f, -0.940506071f,
    0.342660717f, -0.939459224f,
    0.345541325f, -0.938403534f,
    0.348418680f, -0.937339012f,
    0.351292756f, -0.936265667f,
    0.354163525f, -0.935183510f,
    0.357030961f, -0.934092550f,
    0.359895037f, -0.932992799f,
    0.362755724f, -0.931884266f,
    0.365612998f, -0.930766961f,
    0.368466830f, -0.929640896f,
    0.371317194f, -0.928506080f,
    0.374164063f, -0.927362526f,
    0.377007410f, -0.926210242f,
    0.379847209f, -0.925049241f,
    0.382683432f, -0.923879533f,
    0.385516054f, -0.922701128f,
    0.388345047f, -0.921514039f,
    0.391170384f, -0.920318277f,
    0.393992040f, -0.919113852f,
    0.396809987f, -0.917900776f,
    0.399624200f, -0.916679060f,
    0.402434651f, -0.915448716f,
    0.405241314f, -0.914209756f,
    0.408044163f, -0.912962190f,
    0.410843171f, -0.911706032f,
    0.413638312f, -0.910441292f,
    0.416429560f, -0.909167983f,
    0.419216888f, -0.907886116f,
    0.422000271f, -0.906595705f,
    0.424779681f, -0.905296759f,
    0.427555093f, -0.903989293f,
    0.430326481f, -0.902673318f,
    0.433093819f, -0.901348847f,
    0.435857080f, -0.900015892f,
    0.438616239f, -0.898674466f,
    0.441371269f, -0.897324581f,
    0.444122145f, -0.895966250f,
    0.446868840f, -0.894599486f,
    0.449611330f, -0.893224301f,
    0.452349587f, -0.891840709f,
    0.455083587f, -0.890448723f,
    0.457813304f, -0.889048356f,
    0.460538711f, -0.887639620f,
    0.463259784f, -0.886222530f,
    0.465976496f, -0.884797098f,
    0.468688822f, -0.883363339f,
    0.471396737f, -0.881921264f,
    0.474100215f, -0.880470889f,
    0.476799230f, -0.879012226f,
    0.479493758f, -0.877545290f,
    0.482183772f, -0.876070094f,
    0.484869248f, -0.874586652f,
    0.487550160f, -0.873094978f,
    0.490226483f, -0.871595087f,
    0.492898192f, -0.870086991f,
    0.495565262f, -0.868570706f,
    0.498227667f, -0.867046246f,
    0.500885383f, -0.865513624f,
    0.503538384f, -0.863972856f,
    0.506186645f, -0.862423956f,
    0.508830143f, -0.860866939f,
    0.511468850f, -0.859301818f,
    0.514102744f, -0.857728610f,
    0.516731799f, -0.856147328f,
    0.519355990f, -0.854557988f,
    0.521975293f, -0.852960605f,
    0.524589683f, -0.851355193f,
    0.527199135f, -0.849741768f,
    0.529803625f, -0.848120345f,
    0.532403128f, -0.846490939f,
    0.534997620f, -0.844853565f,
    0.537587076f, -0.843208240f,
    0.540171473f, -0.841554977f,
    0.542750785f, -0.839893794f,
    0.545324988f, -0.838224706f,
    0.547894059f, -0.836547727f,
    0.550457973f, -0.834862875f,
    0.553016706f, -0.833170165f,
    0.555570233f, -0.831469612f,
    0.558118531f, -0.829761234f,
    0.560661576f, -0.828045045f,
    0.563199344f, -0.826321063f,
    0.565731811f, -0.824589303f,
    0.568258953f, -0.822849781f,
    0.570780746f, -0.821102515f,
    0.573297167f, -0.819347520f,
    0.575808191f, -0.817584813f,
    0.578313796f, -0.815814411f,
    0.580813958f, -0.814036330f,
    0.583308653f, -0.812250587f,
    0.585797857f, -0.810457198f,
    0.588281548f, -0.808656182f,
    0.590759702f, -0.806847554f,
    0.593232295f, -0.805031331f,
    0.595699304f, -0.803207531f,
    0.598160707f, -0.801376172f,
    0.600616479f, -0.799537269f,
    0.603066599f, -0.797690841f,
    0.605511041f, -0.795836905f,
    0.607949785f, -0.793975478f,
    0.610382806f, -0.792106577f,
    0.612810082f, -0.790230221f,
    0.615231591f, -0.788346428f,
    0.617647308f, -0.786455214f,
    0.620057212f, -0.784556597f,
    0.622461279f, -0.782650596f,
    0.624859488f, -0.780737229f,
    0.627251815f, -0.778816512f,
    0.629638239f, -0.776888466f,
    0.632018736f, -0.774953107f,
    0.634393284f, -0.773010453f,
    0.636761861f, -0.771060524f,
    0.639124445f, -0.769103338f,
    0.641481013f, -0.767138912f,
    0.643831543f, -0.765167266f,
    0.646176013f, -0.763188417f,
    0.648514401f, -0.761202385f,
    0.650846685f, -0.759209189f,
    0.653172843f, -0.757208847f,
    0.655492853f, -0.755201377f,
    0.657806693f, -0.753186799f,
    0.660114342f, -0.751165132f,
    0.662415778f, -0.749136395f,
    0.664710978f, -0.747100606f,
    0.666999922f, -0.745057785f,
    0.669282588f, -0.743007952f,
    0.671558955f, -0.740951125f,
    0.673829000f, -0.738887324f,
    0.676092704f, -0.736816569f,
    0.678350043f, -0.734738878f,
    0.680600998f, -0.732654272f,
    0.682845546f, -0.730562769f,
    0.685083668f, -0.728464390f,
    0.687315341f, -0.726359155f,
    0.689540545f, -0.724247083f,
    0.691759258f, -0.722128194f,
    0.693971461f, -0.720002508f,
    0.696177131f, -0.717870045f,
    0.698376249f, -0.715730825f,
    0.700568794f, -0.713584869f,
    0.702754744f, -0.711432196f,
    0.704934080f, -0.709272826f,
    0.707106781f, -0.707106781f,
    0.709272826f, -0.704934080f,
    0.711432196f, -0.702754744f,
    0.713584869f, -0.700568794f,
    0.715730825f, -0.698376249f,
    0.717870045f, -0.696177131f,
    0.720002508f, -0.693971461f,
    0.722128194f, -0.691759258f,
    0.724247083f, -0.689540545f,
    0.726359155f, -0.687315341f,
    0.728464390f, -0.685083668f,
    0.730562769f, -0.682845546f,
    0.732654272f, -0.680600998f,
    0.734738878f, -0.678350043f,
    0.736816569f, -0.676092704f,
    0.738887324f, -0.673829000f,
    0.740951125f, -0.671558955f,
    0.743007952f, -0.669282588f,
    0.745057785f, -0.666999922f,
    0.747100606f, -0.664710978f,
    0.749136395f, -0.662415778f,
    0.751165132f, -0.660114342f,
    0.753186799f, -0.657806693f,
    0.755201377f, -0.655492853f,
    0.757208847f, -0.653172843f,
    0.759209189f, -0.650846685f,
    0.761202385f, -0.648514401f,
    0.763188417f, -0.646176013f,
    0.765167266f, -0.643831543f,
    0.767138912f, -0.641481013f,
    0.769103338f, -0.639124445f,
    0.771060524f, -0.636761861f,
    0.773010453f, -0.634393284f,
    0.774953107f, -0.632018736f,
    0.776888466f, -0.629638239f,
    0.778816512f, -0.627251815f,
    0.780737229f, -0.624859488f,
    0.782650596f, -0.622461279f,
    0.784556597f, -0.620057212f,
    0.786455214f, -0.617647308f,
    0.788346428f, -0.615231591f,
    0.790230221f, -0.612810082f,
    0.792106577f, -0.610382806f,
    0.793975478f, -0.607949785f,
    0.795836905f, -0.605511041f,
    0.797690841f, -0.603066599f,
    0.799537269f, -0.600616479f,
    0.801376172f, -0.598160707f,
    0.803207531f, -0.595699304f,
    0.805031331f, -0.593232295f,
    0.806847554f, -0.590759702f,
    0.808656182f, -0.588281548f,
    0.810457198f, -0.585797857f,
    0.812250587f, -0.583308653f,
    0.814036330f, -0.580813958f,
    0.815814411f, -0.578313796f,
    0.817584813f, -0.575808191f,
    0.819347520f, -0.573297167f,
    0.821102515f, -0.570780746f,
    0.822849781f, -0.568258953f,
    0.824589303f, -0.565731811f,
    0.826321063f, -0.563199344f,
    0.828045045f, -0.560661576f,
    0.829761234f, -0.558118531f,
    0.831469612f, -0.555570233f,
    0.833170165f, -0.553016706f,
    0.834862875f, -0.550457973f,
    0.836547727f, -0.547894059f,
    0.838224706f, -0.545324988f,
    0.839893794f, -0.542750785f,
    0.841554977f, -0.540171473f,
    0.843208240f, -0.537587076f,
    0.844853565f, -0.534997620f,
    0.846490939f, -0.532403128f,
    0.848120345f, -0.529803625f,
    0.849741768f, -0.527199135f,
    0.851355193f, -0.524589683f,
    0.852960605f, -0.521975293f,
    0.854557988f, -0.519355990f,
    0.856147328f, -0.516731799f,
    0.857728610f, -0.514102744f,
    0.859301818f, -0.511468850f,
    0.860866939f, -0.508830143f,
    0.862423956f, -0.506186645f,
    0.863972856f, -0.503538384f,
    0.865513624f, -0.500885383f,
    0.867046246f, -0.498227667f,
    0.868570706f, -0.495565262f,
    0.870086991f, -0.492898192f,
    0.871595087f, -0.490226483f,
    0.873094978f, -0.487550160f,
    0.874586652f, -0.484869248f,
    0.876070094f, -0.482183772f,
    0.877545290f, -0.479493758f,
    0.879012226f, -0.476799230f,
    0.880470889f, -0.474100215f,
    0.881921264f, -0.471396737f,
    0.883363339f, -0.468688822f,
    0.884797098f, -0.465976496f,
    0.886222530f, -0.463259784f,
    0.887639620f, -0.460538711f,
    0.889048356f, -0.457813304f,
    0.890448723f, -0.455083587f,
    0.891840709f, -0.452349587f,
    0.893224301f, -0.449611330f,
    0.894599486f, -0.446868840f,
    0.895966250f, -0.444122145f,
    0.897324581f, -0.441371269f,
    0.898674466f, -0.438616239f,
    0.900015892f, -0.435857080f,
    0.901348847f, -0.433093819f,
    0.902673318f, -0.430326481f,
    0.903989293f, -0.427555093f,
    0.905296759f, -0.424779681f,
    0.906595705f, -0.422000271f,
    0.907886116f, -0.419216888f,
    0.909167983f, -0.416429560f,
    0.910441292f, -0.413638312f,
    0.911706032f, -0.410843171f,
    0.912962190f, -0.408044163f,
    0.914209756f, -0.405241314f,
    0.915448716f, -0.402434651f,
    0.916679060f, -0.399624200f,
    0.917900776f, -0.396809987f,
    0.919113852f, -0.393992040f,
    0.920318277f, -0.391170384f,
    0.921514039f, -0.388345047f,
    0.922701128f, -0.385516054f,
    0.923879533f, -0.382683432f,
    0.925049241f, -0.379847209f,
    0.926210242f, -0.377007410f,
    0.927362526f, -0.374164063f,
    0.928506080f, -0.371317194f,
    0.929640896f, -0.368466830f,
    0.930766961f, -0.365612998f,
    0.931884266f, -0.362755724f,
    0.932992799f, -0.359895037f,
    0.934092550f, -0.357030961f,
    0.935183510f, -0.354163525f,
    0.936265667f, -0.351292756f,
    0.937339012f, -0.348418680f,
    0.938403534f, -0.345541325f,
    0.939459224f, -0.342660717f,
    0.940506071f, -0.339776884f,
    0.941544065f, -0.336889853f,
    0.942573198f, -0.333999651f,
    0.943593458f, -0.331106306f,
    0.944604837f, -0.328209844f,
    0.945607325f, -0.325310292f,
    0.946600913f, -0.322407679f,
    0.947585591f, -0.319502031f,
    0.948561350f, -0.316593376f,
    0.949528181f, -0.313681740f,
    0.950486074f, -0.310767153f,
    0.951435021f, -0.307849640f,
    0.952375013f, -0.304929230f,
    0.953306040f, -0.302005949f,
    0.954228095f, -0.299079826f,
    0.955141168f, -0.296150888f,
    0.956045251f, -0.293219163f,
    0.956940336f, -0.290284677f,
    0.957826413f, -0.287347460f,
    0.958703475f, -0.284407537f,
    0.959571513f, -0.281464938f,
    0.960430519f, -0.278519689f,
    0.961280486f, -0.275571819f,
    0.962121404f, -0.272621355f,
    0.962953267f, -0.269668326f,
    0.963776066f, -0.266712757f,
    0.964589793f, -0.263754679f,
    0.965394442f, -0.260794118f,
    0.966190003f, -0.257831102f,
    0.966976471f, -0.254865660f,
    0.967753837f, -0.251897818f,
    0.968522094f, -0.248927606f,
    0.969281235f, -0.245955050f,
    0.970031253f, -0.242980180f,
    0.970772141f, -0.240003022f,
    0.971503891f, -0.237023606f,
    0.972226497f, -0.234041959f,
    0.972939952f, -0.231058108f,
    0.973644250f, -0.228072083f,
    0.974339383f, -0.225083911f,
    0.975025345f, -0.222093621f,
    0.975702130f, -0.219101240f,
    0.976369731f, -0.216106797f,
    0.977028143f, -0.213110320f,
    0.977677358f, -0.210111837f,
    0.978317371f, -0.207111376f,
    0.978948175f, -0.204108966f,
    0.979569766f, -0.201104635f,
    0.980182136f, -0.198098411f,
    0.980785280f, -0.195090322f,
    0.981379193f, -0.192080397f,
    0.981963869f, -0.189068664f,
    0.982539302f, -0.186055152f,
    0.983105487f, -0.183039888f,
    0.983662419f, -0.180022901f,
    0.984210092f, -0.177004220f,
    0.984748502f, -0.173983873f,
    0.985277642f, -0.170961889f,
    0.985797509f, -0.167938295f,
    0.986308097f, -0.164913120f,
    0.986809402f, -0.161886394f,
    0.987301418f, -0.158858143f,
    0.987784142f, -0.155828398f,
    0.988257568f, -0.152797185f,
    0.988721692f, -0.149764535f,
    0.989176510f, -0.146730474f,
    0.989622017f, -0.143695033f,
    0.990058210f, -0.140658239f,
    0.990485084f, -0.137620122f,
    0.990902635f, -0.134580709f,
    0.991310860f, -0.131540029f,
    0.991709754f, -0.128498111f,
    0.992099313f, -0.125454983f,
    0.992479535f, -0.122410675f,
    0.992850414f, -0.119365215f,
    0.993211949f, -0.116318631f,
    0.993564136f, -0.113270952f,
    0.993906970f, -0.110222207f,
    0.994240449f, -0.107172425f,
    0.994564571f, -0.104121634f,
    0.994879331f, -0.101069863f,
    0.995184727f, -0.098017140f,
    0.995480755f, -0.094963495f,
    0.995767414f, -0.091908956f,
    0.996044701f, -0.088853553f,
    0.996312612f, -0.085797312f,
    0.996571146f, -0.082740265f,
    0.996820299f, -0.079682438f,
    0.997060070f, -0.076623861f,
    0.997290457f, -0.073564564f,
    0.997511456f, -0.070504573f,
    0.997723067f, -0.067443920f,
    0.997925286f, -0.064382631f,
    0.998118113f, -0.061320736f,
    0.998301545f, -0.058258265f,
    0.998475581f, -0.055195244f,
    0.998640218f, -0.052131705f,
    0.998795456f, -0.049067674f,
    0.998941293f, -0.046003182f,
    0.999077728f, -0.042938257f,
    0.999204759f, -0.039872928f,
    0.999322385f, -0.036807223f,
    0.999430605f, -0.033741172f,
    0.999529418f, -0.030674803f,
    0.999618822f, -0.027608146f,
    0.999698819f, -0.024541229f,
    0.999769405f, -0.021474080f,
    0.999830582f, -0.018406730f,
    0.999882347f, -0.015339206f,
    0.999924702f, -0.012271538f,
    0.999957645f, -0.009203755f,
    0.999981175f, -0.006135885f,
    0.999995294f, -0.003067957f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_F32_4096)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 4096, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
const float32_t twiddleCoef_4096[8192] = {
    1.000000000f,  0.000000000f,
    0.999998823f,  0.001533980f,
    0.999995294f,  0.003067957f,
    0.999989411f,  0.004601926f,
    0.999981175f,  0.006135885f,
    0.999970586f,  0.007669829f,
    0.999957645f,  0.009203755f,
    0.999942350f,  0.010737659f,
    0.999924702f,  0.012271538f,
    0.999904701f,  0.013805389f,
    0.999882347f,  0.015339206f,
    0.999857641f,  0.016872988f,
    0.999830582f,  0.018406730f,
    0.999801170f,  0.019940429f,
    0.999769405f,  0.021474080f,
    0.999735288f,  0.023007681f,
    0.999698819f,  0.024541229f,
    0.999659997f,  0.026074718f,
    0.999618822f,  0.027608146f,
    0.999575296f,  0.029141509f,
    0.999529418f,  0.030674803f,
    0.999481187f,  0.032208025f,
    0.999430605f,  0.033741172f,
    0.999377670f,  0.035274239f,
    0.999322385f,  0.036807223f,
    0.999264747f,  0.038340120f,
    0.999204759f,  0.039872928f,
    0.999142419f,  0.041405641f,
    0.999077728f,  0.042938257f,
    0.999010686f,  0.044470772f,
    0.998941293f,  0.046003182f,
    0.998869550f,  0.047535484f,
    0.998795456f,  0.049067674f,
    0.998719012f,  0.050599749f,
    0.998640218f,  0.052131705f,
    0.998559074f,  0.053663538f,
    0.998475581f,  0.055195244f,
    0.998389737f,  0.056726821f,
    0.998301545f,  0.058258265f,
    0.998211003f,  0.059789571f,
    0.998118113f,  0.061320736f,
    0.998022874f,  0.062851758f,
    0.997925286f,  0.064382631f,
    0.997825350f,  0.065913353f,
    0.997723067f,  0.067443920f,
    0.997618435f,  0.068974328f,
    0.997511456f,  0.070504573f,
    0.997402130f,  0.072034653f,
    0.997290457f,  0.073564564f,
    0.997176437f,  0.075094301f,
    0.997060070f,  0.076623861f,
    0.996941358f,  0.078153242f,
    0.996820299f,  0.079682438f,
    0.996696895f,  0.081211447f,
    0.996571146f,  0.082740265f,
    0.996443051f,  0.084268888f,
    0.996312612f,  0.085797312f,
    0.996179829f,  0.087325535f,
    0.996044701f,  0.088853553f,
    0.995907229f,  0.090381361f,
    0.995767414f,  0.091908956f,
    0.995625256f,  0.093436336f,
    0.995480755f,  0.094963495f,
    0.995333912f,  0.096490431f,
    0.995184727f,  0.098017140f,
    0.995033199f,  0.099543619f,
    0.994879331f,  0.101069863f,
    0.994723121f,  0.102595869f,
    0.994564571f,  0.104121634f,
    0.994403680f,  0.105647154f,
    0.994240449f,  0.107172425f,
    0.994074879f,  0.108697444f,
    0.993906970f,  0.110222207f,
    0.993736722f,  0.111746711f,
    0.993564136f,  0.113270952f,
    0.993389211f,  0.114794927f,
    0.993211949f,  0.116318631f,
    0.993032350f,  0.117842062f,
    0.992850414f,  0.119365215f,
    0.992666142f,  0.120888087f,
    0.992479535f,  0.122410675f,
    0.992290591f,  0.123932975f,
    0.992099313f,  0.125454983f,
    0.991905700f,  0.126976696f,
    0.991709754f,  0.128498111f,
    0.991511473f,  0.130019223f,
    0.991310860f,  0.131540029f,
    0.991107914f,  0.133060525f,
    0.990902635f,  0.134580709f,
    0.990695025f,  0.136100575f,
    0.990485084f,  0.137620122f,
    0.990272812f,  0.139139344f,
    0.990058210f,  0.140658239f,
    0.989841278f,  0.142176804f,
    0.989622017f,  0.143695033f,
    0.989400428f,  0.145212925f,
    0.989176510f,  0.146730474f,
    0.988950265f,  0.148247679f,
    0.988721692f,  0.149764535f,
    0.988490793f,  0.151281038f,
    0.988257568f,  0.152797185f,
    0.988022017f,  0.154312973f,
    0.987784142f,  0.155828398f,
    0.987543942f,  0.157343456f,
    0.987301418f,  0.158858143f,
    0.987056571f,  0.160372457f,
    0.986809402f,  0.161886394f,
    0.986559910f,  0.163399949f,
    0.986308097f,  0.164913120f,
    0.986053963f,  0.166425904f,
    0.985797509f,  0.167938295f,
    0.985538735f,  0.169450291f,
    0.985277642f,  0.170961889f,
    0.985014231f,  0.172473084f,
    0.984748502f,  0.173983873f,
    0.984480455f,  0.175494253f,
    0.984210092f,  0.177004220f,
    0.983937413f,  0.178513771f,
    0.983662419f,  0.180022901f,
    0.983385110f,  0.181531608f,
    0.983105487f,  0.183039888f,
    0.982823551f,  0.184547737f,
    0.982539302f,  0.186055152f,
    0.982252741f,  0.187562129f,
    0.981963869f,  0.189068664f,
    0.981672686f,  0.190574755f,
    0.981379193f,  0.192080397f,
    0.981083391f,  0.193585587f,
    0.980785280f,  0.195090322f,
    0.980484862f,  0.196594598f,
    0.980182136f,  0.198098411f,
    0.979877104f,  0.199601758f,
    0.979569766f,  0.201104635f,
    0.979260123f,  0.202607039f,
    0.978948175f,  0.204108966f,
    0.978633924f,  0.205610413f,
    0.978317371f,  0.207111376f,
    0.977998515f,  0.208611852f,
    0.977677358f,  0.210111837f,
    0.977353900f,  0.211611327f,
    0.977028143f,  0.213110320f,
    0.976700086f,  0.214608811f,
    0.976369731f,  0.216106797f,
    0.976037079f,  0.217604275f,
    0.975702130f,  0.219101240f,
    0.975364885f,  0.220597690f,
    0.975025345f,  0.222093621f,
    0.974683511f,  0.223589029f,
    0.974339383f,  0.225083911f,
    0.973992962f,  0.226578264f,
    0.973644250f,  0.228072083f,
    0.973293246f,  0.229565366f,
    0.972939952f,  0.231058108f,
    0.972584369f,  0.232550307f,
    0.972226497f,  0.234041959f,
    0.971866337f,  0.235533059f,
    0.971503891f,  0.237023606f,
    0.971139158f,  0.238513595f,
    0.970772141f,  0.240003022f,
    0.970402839f,  0.241491885f,
    0.970031253f,  0.242980180f,
    0.969657385f,  0.244467903f,
    0.969281235f,  0.245955050f,
    0.968902805f,  0.247441619f,
    0.968522094f,  0.248927606f,
    0.968139105f,  0.250413007f,
    0.967753837f,  0.251897818f,
    0.967366292f,  0.253382037f,
    0.966976471f,  0.254865660f,
    0.966584374f,  0.256348682f,
    0.966190003f,  0.257831102f,
    0.965793359f,  0.259312915f,
    0.965394442f,  0.260794118f,
    0.964993253f,  0.262274707f,
    0.964589793f,  0.263754679f,
    0.964184064f,  0.265234030f,
    0.963776066f,  0.266712757f,
    0.963365800f,  0.268190857f,
    0.962953267f,  0.269668326f,
    0.962538468f,  0.271145160f,
    0.962121404f,  0.272621355f,
    0.961702077f,  0.274096910f,
    0.961280486f,  0.275571819f,
    0.960856633f,  0.277046080f,
    0.960430519f,  0.278519689f,
    0.960002146f,  0.279992643f,
    0.959571513f,  0.281464938f,
    0.959138622f,  0.282936570f,
    0.958703475f,  0.284407537f,
    0.958266071f,  0.285877835f,
    0.957826413f,  0.287347460f,
    0.957384501f,  0.288816408f,
    0.956940336f,  0.290284677f,
    0.956493919f,  0.291752263f,
    0.956045251f,  0.293219163f,
    0.955594334f,  0.294685372f,
    0.955141168f,  0.296150888f,
    0.954685755f,  0.297615707f,
    0.954228095f,  0.299079826f,
    0.953768190f,  0.300543241f,
    0.953306040f,  0.302005949f,
    0.952841648f,  0.303467947f,
    0.952375013f,  0.304929230f,
    0.951906137f,  0.306389795f,
    0.951435021f,  0.307849640f,
    0.950961666f,  0.309308760f,
    0.950486074f,  0.310767153f,
    0.950008245f,  0.312224814f,
    0.949528181f,  0.313681740f,
    0.949045882f,  0.315137929f,
    0.948561350f,  0.316593376f,
    0.948074586f,  0.318048077f,
    0.947585591f,  0.319502031f,
    0.947094366f,  0.320955232f,
    0.946600913f,  0.322407679f,
    0.946105232f,  0.323859367f,
    0.945607325f,  0.325310292f,
    0.945107193f,  0.326760452f,
    0.944604837f,  0.328209844f,
    0.944100258f,  0.329658463f,
    0.943593458f,  0.331106306f,
    0.943084437f,  0.332553370f,
    0.942573198f,  0.333999651f,
    0.942059740f,  0.335445147f,
    0.941544065f,  0.336889853f,
    0.941026175f,  0.338333767f,
    0.940506071f,  0.339776884f,
    0.939983753f,  0.341219202f,
    0.939459224f,  0.342660717f,
    0.938932484f,  0.344101426f,
    0.938403534f,  0.345541325f,
    0.937872376f,  0.346980411f,
    0.937339012f,  0.348418680f,
    0.936803442f,  0.349856130f,
    0.936265667f,  0.351292756f,
    0.935725689f,  0.352728556f,
    0.935183510f,  0.354163525f,
    0.934639130f,  0.355597662f,
    0.934092550f,  0.357030961f,
    0.933543773f,  0.358463421f,
    0.932992799f,  0.359895037f,
    0.932439629f,  0.361325806f,
    0.931884266f,  0.362755724f,
    0.931326709f,  0.364184790f,
    0.930766961f,  0.365612998f,
    0.930205023f,  0.367040346f,
    0.929640896f,  0.368466830f,
    0.929074581f,  0.369892447f,
    0.928506080f,  0.371317194f,
    0.927935395f,  0.372741067f,
    0.927362526f,  0.374164063f,
    0.926787474f,  0.375586178f,
    0.926210242f,  0.377007410f,
    0.925630831f,  0.378427755f,
    0.925049241f,  0.379847209f,
    0.924465474f,  0.381265769f,
    0.923879533f,  0.382683432f,
    0.923291417f,  0.384100195f,
    0.922701128f,  0.385516054f,
    0.922108669f,  0.386931006f,
    0.921514039f,  0.388345047f,
    0.920917242f,  0.389758174f,
    0.920318277f,  0.391170384f,
    0.919717146f,  0.392581674f,
    0.919113852f,  0.393992040f,
    0.918508394f,  0.395401479f,
    0.917900776f,  0.396809987f,
    0.917290997f,  0.398217562f,
    0.916679060f,  0.399624200f,
    0.916064966f,  0.401029897f,
    0.915448716f,  0.402434651f,
    0.914830312f,  0.403838458f,
    0.914209756f,  0.405241314f,
    0.913587048f,  0.406643217f,
    0.912962190f,  0.408044163f,
    0.912335185f,  0.409444149f,
    0.911706032f,  0.410843171f,
    0.911074734f,  0.412241227f,
    0.910441292f,  0.413638312f,
    0.909805708f,  0.415034424f,
    0.909167983f,  0.416429560f,
    0.908528119f,  0.417823716f,
    0.907886116f,  0.419216888f,
    0.907241978f,  0.420609074f,
    0.906595705f,  0.422000271f,
    0.905947298f,  0.423390474f,
    0.905296759f,  0.424779681f,
    0.904644091f,  0.426167889f,
    0.903989293f,  0.427555093f,
    0.903332368f,  0.428941292f,
    0.902673318f,  0.430326481f,
    0.902012144f,  0.431710658f,
    0.901348847f,  0.433093819f,
    0.900683429f,  0.434475961f,
    0.900015892f,  0.435857080f,
    0.899346237f,  0.437237174f,
    0.898674466f,  0.438616239f,
    0.898000580f,  0.439994271f,
    0.897324581f,  0.441371269f,
    0.896646470f,  0.442747228f,
    0.895966250f,  0.444122145f,
    0.895283921f,  0.445496017f,
    0.894599486f,  0.446868840f,
    0.893912945f,  0.448240612f,
    0.893224301f,  0.449611330f,
    0.892533555f,  0.450980989f,
    0.891840709f,  0.452349587f,
    0.891145765f,  0.453717121f,
    0.890448723f,  0.455083587f,
    0.889749586f,  0.456448982f,
    0.889048356f,  0.457813304f,
    0.888345033f,  0.459176548f,
    0.887639620f,  0.460538711f,
    0.886932119f,  0.461899791f,
    0.886222530f,  0.463259784f,
    0.885510856f,  0.464618686f,
    0.884797098f,  0.465976496f,
    0.884081259f,  0.467333209f,
    0.883363339f,  0.468688822f,
    0.882643340f,  0.470043332f,
    0.881921264f,  0.471396737f,
    0.881197113f,  0.472749032f,
    0.880470889f,  0.474100215f,
    0.879742593f,  0.475450282f,
    0.879012226f,  0.476799230f,
    0.878279792f,  0.478147056f,
    0.877545290f,  0.479493758f,
    0.876808724f,  0.480839331f,
    0.876070094f,  0.482183772f,
    0.875329403f,  0.483527079f,
    0.874586652f,  0.484869248f,
    0.873841843f,  0.486210276f,
    0.873094978f,  0.487550160f,
    0.872346059f,  0.488888897f,
    0.871595087f,  0.490226483f,
    0.870842063f,  0.491562916f,
    0.870086991f,  0.492898192f,
    0.869329871f,  0.494232309f,
    0.868570706f,  0.495565262f,
    0.867809497f,  0.496897049f,
    0.867046246f,  0.498227667f,
    0.866280954f,  0.499557113f,
    0.865513624f,  0.500885383f,
    0.864744258f,  0.502212474f,
    0.863972856f,  0.503538384f,
    0.863199422f,  0.504863109f,
    0.862423956f,  0.506186645f,
    0.861646461f,  0.507508991f,
    0.860866939f,  0.508830143f,
    0.860085390f,  0.510150097f,
    0.859301818f,  0.511468850f,
    0.858516224f,  0.512786401f,
    0.857728610f,  0.514102744f,
    0.856938977f,  0.515417878f,
    0.856147328f,  0.516731799f,
    0.855353665f,  0.518044504f,
    0.854557988f,  0.519355990f,
    0.853760301f,  0.520666254f,
    0.852960605f,  0.521975293f,
    0.852158902f,  0.523283103f,
    0.851355193f,  0.524589683f,
    0.850549481f,  0.525895027f,
    0.849741768f,  0.527199135f,
    0.848932055f,  0.528502002f,
    0.848120345f,  0.529803625f,
    0.847306639f,  0.531104001f,
    0.846490939f,  0.532403128f,
    0.845673247f,  0.533701002f,
    0.844853565f,  0.534997620f,
    0.844031895f,  0.536292979f,
    0.843208240f,  0.537587076f,
    0.842382600f,  0.538879909f,
    0.841554977f,  0.540171473f,
    0.840725375f,  0.541461766f,
    0.839893794f,  0.542750785f,
    0.839060237f,  0.544038527f,
    0.838224706f,  0.545324988f,
    0.837387202f,  0.546610167f,
    0.836547727f,  0.547894059f,
    0.835706284f,  0.549176662f,
    0.834862875f,  0.550457973f,
    0.834017501f,  0.551737988f,
    0.833170165f,  0.553016706f,
    0.832320868f,  0.554294121f,
    0.831469612f,  0.555570233f,
    0.830616400f,  0.556845037f,
    0.829761234f,  0.558118531f,
    0.828904115f,  0.559390712f,
    0.828045045f,  0.560661576f,
    0.827184027f,  0.561931121f,
    0.826321063f,  0.563199344f,
    0.825456154f,  0.564466242f,
    0.824589303f,  0.565731811f,
    0.823720511f,  0.566996049f,
    0.822849781f,  0.568258953f,
    0.821977115f,  0.569520519f,
    0.821102515f,  0.570780746f,
    0.820225983f,  0.572039629f,
    0.819347520f,  0.573297167f,
    0.818467130f,  0.574553355f,
    0.817584813f,  0.575808191f,
    0.816700573f,  0.577061673f,
    0.815814411f,  0.578313796f,
    0.814926329f,  0.579564559f,
    0.814036330f,  0.580813958f,
    0.813144415f,  0.582061990f,
    0.812250587f,  0.583308653f,
    0.811354847f,  0.584553943f,
    0.810457198f,  0.585797857f,
    0.809557642f,  0.587040394f,
    0.808656182f,  0.588281548f,
    0.807752818f,  0.589521319f,
    0.806847554f,  0.590759702f,
    0.805940391f,  0.591996695f,
    0.805031331f,  0.593232295f,
    0.804120377f,  0.594466499f,
    0.803207531f,  0.595699304f,
    0.802292796f,  0.596930708f,
    0.801376172f,  0.598160707f,
    0.800457662f,  0.599389298f,
    0.799537269f,  0.600616479f,
    0.798614995f,  0.601842247f,
    0.797690841f,  0.603066599f,
    0.796764810f,  0.604289531f,
    0.795836905f,  0.605511041f,
    0.794907126f,  0.606731127f,
    0.793975478f,  0.607949785f,
    0.793041960f,  0.609167012f,
    0.792106577f,  0.610382806f,
    0.791169330f,  0.611597164f,
    0.790230221f,  0.612810082f,
    0.789289253f,  0.614021559f,
    0.788346428f,  0.615231591f,
    0.787401747f,  0.616440175f,
    0.786455214f,  0.617647308f,
    0.785506830f,  0.618852988f,
    0.784556597f,  0.620057212f,
    0.783604519f,  0.621259977f,
    0.782650596f,  0.622461279f,
    0.781694832f,  0.623661118f,
    0.780737229f,  0.624859488f,
    0.779777788f,  0.626056388f,
    0.778816512f,  0.627251815f,
    0.777853404f,  0.628445767f,
    0.776888466f,  0.629638239f,
    0.775921699f,  0.630829230f,
    0.774953107f,  0.632018736f,
    0.773982691f,  0.633206755f,
    0.773010453f,  0.634393284f,
    0.772036397f,  0.635578320f,
    0.771060524f,  0.636761861f,
    0.770082837f,  0.637943904f,
    0.769103338f,  0.639124445f,
    0.768122029f,  0.640303482f,
    0.767138912f,  0.641481013f,
    0.766153990f,  0.642657034f,
    0.765167266f,  0.643831543f,
    0.764178741f,  0.645004537f,
    0.763188417f,  0.646176013f,
    0.762196298f,  0.647345969f,
    0.761202385f,  0.648514401f,
    0.760206682f,  0.649681307f,
    0.759209189f,  0.650846685f,
    0.758209910f,  0.652010531f,
    0.757208847f,  0.653172843f,
    0.756206001f,  0.654333618f,
    0.755201377f,  0.655492853f,
    0.754194975f,  0.656650546f,
    0.753186799f,  0.657806693f,
    0.752176850f,  0.658961293f,
    0.751165132f,  0.660114342f,
    0.750151646f,  0.661265838f,
    0.749136395f,  0.662415778f,
    0.748119380f,  0.663564159f,
    0.747100606f,  0.664710978f,
    0.746080074f,  0.665856234f,
    0.745057785f,  0.666999922f,
    0.744033744f,  0.668142041f,
    0.743007952f,  0.669282588f,
    0.741980412f,  0.670421560f,
    0.740951125f,  0.671558955f,
    0.739920095f,  0.672694769f,
    0.738887324f,  0.673829000f,
    0.737852815f,  0.674961646f,
    0.736816569f,  0.676092704f,
    0.735778589f,  0.677222170f,
    0.734738878f,  0.678350043f,
    0.733697438f,  0.679476320f,
    0.732654272f,  0.680600998f,
    0.731609381f,  0.681724074f,
    0.730562769f,  0.682845546f,
    0.729514438f,  0.683965412f,
    0.728464390f,  0.685083668f,
    0.727412629f,  0.686200312f,
    0.726359155f,  0.687315341f,
    0.725303972f,  0.688428753f,
    0.724247083f,  0.689540545f,
    0.723188489f,  0.690650714f,
    0.722128194f,  0.691759258f,
    0.721066199f,  0.692866175f,
    0.720002508f,  0.693971461f,
    0.718937122f,  0.695075114f,
    0.717870045f,  0.696177131f,
    0.716801279f,  0.697277511f,
    0.715730825f,  0.698376249f,
    0.714658688f,  0.699473345f,
    0.713584869f,  0.700568794f,
    0.712509371f,  0.701662595f,
    0.711432196f,  0.702754744f,
    0.710353347f,  0.703845241f,
    0.709272826f,  0.704934080f,
    0.708190637f,  0.706021261f,
    0.707106781f,  0.707106781f,
    0.706021261f,  0.708190637f,
    0.704934080f,  0.709272826f,
    0.703845241f,  0.710353347f,
    0.702754744f,  0.711432196f,
    0.701662595f,  0.712509371f,
    0.700568794f,  0.713584869f,
    0.699473345f,  0.714658688f,
    0.698376249f,  0.715730825f,
    0.697277511f,  0.716801279f,
    0.696177131f,  0.717870045f,
    0.695075114f,  0.718937122f,
    0.693971461f,  0.720002508f,
    0.692866175f,  0.721066199f,
    0.691759258f,  0.722128194f,
    0.690650714f,  0.723188489f,
    0.689540545f,  0.724247083f,
    0.688428753f,  0.725303972f,
    0.687315341f,  0.726359155f,
    0.686200312f,  0.727412629f,
    0.685083668f,  0.728464390f,
    0.683965412f,  0.729514438f,
    0.682845546f,  0.730562769f,
    0.681724074f,  0.731609381f,
    0.680600998f,  0.732654272f,
    0.679476320f,  0.733697438f,
    0.678350043f,  0.734738878f,
    0.677222170f,  0.735778589f,
    0.676092704f,  0.736816569f,
    0.674961646f,  0.737852815f,
    0.673829000f,  0.738887324f,
    0.672694769f,  0.739920095f,
    0.671558955f,  0.740951125f,
    0.670421560f,  0.741980412f,
    0.669282588f,  0.743007952f,
    0.668142041f,  0.744033744f,
    0.666999922f,  0.745057785f,
    0.665856234f,  0.746080074f,
    0.664710978f,  0.747100606f,
    0.663564159f,  0.748119380f,
    0.662415778f,  0.749136395f,
    0.661265838f,  0.750151646f,
    0.660114342f,  0.751165132f,
    0.658961293f,  0.752176850f,
    0.657806693f,  0.753186799f,
    0.656650546f,  0.754194975f,
    0.655492853f,  0.755201377f,
    0.654333618f,  0.756206001f,
    0.653172843f,  0.757208847f,
    0.652010531f,  0.758209910f,
    0.650846685f,  0.759209189f,
    0.649681307f,  0.760206682f,
    0.648514401f,  0.761202385f,
    0.647345969f,  0.762196298f,
    0.646176013f,  0.763188417f,
    0.645004537f,  0.764178741f,
    0.643831543f,  0.765167266f,
    0.642657034f,  0.766153990f,
    0.641481013f,  0.767138912f,
    0.640303482f,  0.768122029f,
    0.639124445f,  0.769103338f,
    0.637943904f,  0.770082837f,
    0.636761861f,  0.771060524f,
    0.635578320f,  0.772036397f,
    0.634393284f,  0.773010453f,
    0.633206755f,  0.773982691f,
    0.632018736f,  0.774953107f,
    0.630829230f,  0.775921699f,
    0.629638239f,  0.776888466f,
    0.628445767f,  0.777853404f,
    0.627251815f,  0.778816512f,
    0.626056388f,  0.779777788f,
    0.624859488f,  0.780737229f,
    0.623661118f,  0.781694832f,
    0.622461279f,  0.782650596f,
    0.621259977f,  0.783604519f,
    0.620057212f,  0.784556597f,
    0.618852988f,  0.785506830f,
    0.617647308f,  0.786455214f,
    0.616440175f,  0.787401747f,
    0.615231591f,  0.788346428f,
    0.614021559f,  0.789289253f,
    0.612810082f,  0.790230221f,
    0.611597164f,  0.791169330f,
    0.610382806f,  0.792106577f,
    0.609167012f,  0.793041960f,
    0.607949785f,  0.793975478f,
    0.606731127f,  0.794907126f,
    0.605511041f,  0.795836905f,
    0.604289531f,  0.796764810f,
    0.603066599f,  0.797690841f,
    0.601842247f,  0.798614995f,
    0.600616479f,  0.799537269f,
    0.599389298f,  0.800457662f,
    0.598160707f,  0.801376172f,
    0.596930708f,  0.802292796f,
    0.595699304f,  0.803207531f,
    0.594466499f,  0.804120377f,
    0.593232295f,  0.805031331f,
    0.591996695f,  0.805940391f,
    0.590759702f,  0.806847554f,
    0.589521319f,  0.807752818f,
    0.588281548f,  0.808656182f,
    0.587040394f,  0.809557642f,
    0.585797857f,  0.810457198f,
    0.584553943f,  0.811354847f,
    0.583308653f,  0.812250587f,
    0.582061990f,  0.813144415f,
    0.580813958f,  0.814036330f,
    0.579564559f,  0.814926329f,
    0.578313796f,  0.815814411f,
    0.577061673f,  0.816700573f,
    0.575808191f,  0.817584813f,
    0.574553355f,  0.818467130f,
    0.573297167f,  0.819347520f,
    0.572039629f,  0.820225983f,
    0.570780746f,  0.821102515f,
    0.569520519f,  0.821977115f,
    0.568258953f,  0.822849781f,
    0.566996049f,  0.823720511f,
    0.565731811f,  0.824589303f,
    0.564466242f,  0.825456154f,
    0.563199344f,  0.826321063f,
    0.561931121f,  0.827184027f,
    0.560661576f,  0.828045045f,
    0.559390712f,  0.828904115f,
    0.558118531f,  0.829761234f,
    0.556845037f,  0.830616400f,
    0.555570233f,  0.831469612f,
    0.554294121f,  0.832320868f,
    0.553016706f,  0.833170165f,
    0.551737988f,  0.834017501f,
    0.550457973f,  0.834862875f,
    0.549176662f,  0.835706284f,
    0.547894059f,  0.836547727f,
    0.546610167f,  0.837387202f,
    0.545324988f,  0.838224706f,
    0.544038527f,  0.839060237f,
    0.542750785f,  0.839893794f,
    0.541461766f,  0.840725375f,
    0.540171473f,  0.841554977f,
    0.538879909f,  0.842382600f,
    0.537587076f,  0.843208240f,
    0.536292979f,  0.844031895f,
    0.534997620f,  0.844853565f,
    0.533701002f,  0.845673247f,
    0.532403128f,  0.846490939f,
    0.531104001f,  0.847306639f,
    0.529803625f,  0.848120345f,
    0.528502002f,  0.848932055f,
    0.527199135f,  0.849741768f,
    0.525895027f,  0.850549481f,
    0.524589683f,  0.851355193f,
    0.523283103f,  0.852158902f,
    0.521975293f,  0.852960605f,
    0.520666254f,  0.853760301f,
    0.519355990f,  0.854557988f,
    0.518044504f,  0.855353665f,
    0.516731799f,  0.856147328f,
    0.515417878f,  0.856938977f,
    0.514102744f,  0.857728610f,
    0.512786401f,  0.858516224f,
    0.511468850f,  0.859301818f,
    0.510150097f,  0.860085390f,
    0.508830143f,  0.860866939f,
    0.507508991f,  0.861646461f,
    0.506186645f,  0.862423956f,
    0.504863109f,  0.863199422f,
    0.503538384f,  0.863972856f,
    0.502212474f,  0.864744258f,
    0.500885383f,  0.865513624f,
    0.499557113f,  0.866280954f,
    0.498227667f,  0.867046246f,
    0.496897049f,  0.867809497f,
    0.495565262f,  0.868570706f,
    0.494232309f,  0.869329871f,
    0.492898192f,  0.870086991f,
    0.491562916f,  0.870842063f,
    0.490226483f,  0.871595087f,
    0.488888897f,  0.872346059f,
    0.487550160f,  0.873094978f,
    0.486210276f,  0.873841843f,
    0.484869248f,  0.874586652f,
    0.483527079f,  0.875329403f,
    0.482183772f,  0.876070094f,
    0.480839331f,  0.876808724f,
    0.479493758f,  0.877545290f,
    0.478147056f,  0.878279792f,
    0.476799230f,  0.879012226f,
    0.475450282f,  0.879742593f,
    0.474100215f,  0.880470889f,
    0.472749032f,  0.881197113f,
    0.471396737f,  0.881921264f,
    0.470043332f,  0.882643340f,
    0.468688822f,  0.883363339f,
    0.467333209f,  0.884081259f,
    0.465976496f,  0.884797098f,
    0.464618686f,  0.885510856f,
    0.463259784f,  0.886222530f,
    0.461899791f,  0.886932119f,
    0.460538711f,  0.887639620f,
    0.459176548f,  0.888345033f,
    0.457813304f,  0.889048356f,
    0.456448982f,  0.889749586f,
    0.455083587f,  0.890448723f,
    0.453717121f,  0.891145765f,
    0.452349587f,  0.891840709f,
    0.450980989f,  0.892533555f,
    0.449611330f,  0.893224301f,
    0.448240612f,  0.893912945f,
    0.446868840f,  0.894599486f,
    0.445496017f,  0.895283921f,
    0.444122145f,  0.895966250f,
    0.442747228f,  0.896646470f,
    0.441371269f,  0.897324581f,
    0.439994271f,  0.898000580f,
    0.438616239f,  0.898674466f,
    0.437237174f,  0.899346237f,
    0.435857080f,  0.900015892f,
    0.434475961f,  0.900683429f,
    0.433093819f,  0.901348847f,
    0.431710658f,  0.902012144f,
    0.430326481f,  0.902673318f,
    0.428941292f,  0.903332368f,
    0.427555093f,  0.903989293f,
    0.426167889f,  0.904644091f,
    0.424779681f,  0.905296759f,
    0.423390474f,  0.905947298f,
    0.422000271f,  0.906595705f,
    0.420609074f,  0.907241978f,
    0.419216888f,  0.907886116f,
    0.417823716f,  0.908528119f,
    0.416429560f,  0.909167983f,
    0.415034424f,  0.909805708f,
    0.413638312f,  0.910441292f,
    0.412241227f,  0.911074734f,
    0.410843171f,  0.911706032f,
    0.409444149f,  0.912335185f,
    0.408044163f,  0.912962190f,
    0.406643217f,  0.913587048f,
    0.405241314f,  0.914209756f,
    0.403838458f,  0.914830312f,
    0.402434651f,  0.915448716f,
    0.401029897f,  0.916064966f,
    0.399624200f,  0.916679060f,
    0.398217562f,  0.917290997f,
    0.396809987f,  0.917900776f,
    0.395401479f,  0.918508394f,
    0.393992040f,  0.919113852f,
    0.392581674f,  0.919717146f,
    0.391170384f,  0.920318277f,
    0.389758174f,  0.920917242f,
    0.388345047f,  0.921514039f,
    0.386931006f,  0.922108669f,
    0.385516054f,  0.922701128f,
    0.384100195f,  0.923291417f,
    0.382683432f,  0.923879533f,
    0.381265769f,  0.924465474f,
    0.379847209f,  0.925049241f,
    0.378427755f,  0.925630831f,
    0.377007410f,  0.926210242f,
    0.375586178f,  0.926787474f,
    0.374164063f,  0.927362526f,
    0.372741067f,  0.927935395f,
    0.371317194f,  0.928506080f,
    0.369892447f,  0.929074581f,
    0.368466830f,  0.929640896f,
    0.367040346f,  0.930205023f,
    0.365612998f,  0.930766961f,
    0.364184790f,  0.931326709f,
    0.362755724f,  0.931884266f,
    0.361325806f,  0.932439629f,
    0.359895037f,  0.932992799f,
    0.358463421f,  0.933543773f,
    0.357030961f,  0.934092550f,
    0.355597662f,  0.934639130f,
    0.354163525f,  0.935183510f,
    0.352728556f,  0.935725689f,
    0.351292756f,  0.936265667f,
    0.349856130f,  0.936803442f,
    0.348418680f,  0.937339012f,
    0.346980411f,  0.937872376f,
    0.345541325f,  0.938403534f,
    0.344101426f,  0.938932484f,
    0.342660717f,  0.939459224f,
    0.341219202f,  0.939983753f,
    0.339776884f,  0.940506071f,
    0.338333767f,  0.941026175f,
    0.336889853f,  0.941544065f,
    0.335445147f,  0.942059740f,
    0.333999651f,  0.942573198f,
    0.332553370f,  0.943084437f,
    0.331106306f,  0.943593458f,
    0.329658463f,  0.944100258f,
    0.328209844f,  0.944604837f,
    0.326760452f,  0.945107193f,
    0.325310292f,  0.945607325f,
    0.323859367f,  0.946105232f,
    0.322407679f,  0.946600913f,
    0.320955232f,  0.947094366f,
    0.319502031f,  0.947585591f,
    0.318048077f,  0.948074586f,
    0.316593376f,  0.948561350f,
    0.315137929f,  0.949045882f,
    0.313681740f,  0.949528181f,
    0.312224814f,  0.950008245f,
    0.310767153f,  0.950486074f,
    0.309308760f,  0.950961666f,
    0.307849640f,  0.951435021f,
    0.306389795f,  0.951906137f,
    0.304929230f,  0.952375013f,
    0.303467947f,  0.952841648f,
    0.302005949f,  0.953306040f,
    0.300543241f,  0.953768190f,
    0.299079826f,  0.954228095f,
    0.297615707f,  0.954685755f,
    0.296150888f,  0.955141168f,
    0.294685372f,  0.955594334f,
    0.293219163f,  0.956045251f,
    0.291752263f,  0.956493919f,
    0.290284677f,  0.956940336f,
    0.288816408f,  0.957384501f,
    0.287347460f,  0.957826413f,
    0.285877835f,  0.958266071f,
    0.284407537f,  0.958703475f,
    0.282936570f,  0.959138622f,
    0.281464938f,  0.959571513f,
    0.279992643f,  0.960002146f,
    0.278519689f,  0.960430519f,
    0.277046080f,  0.960856633f,
    0.275571819f,  0.961280486f,
    0.274096910f,  0.961702077f,
    0.272621355f,  0.962121404f,
    0.271145160f,  0.962538468f,
    0.269668326f,  0.962953267f,
    0.268190857f,  0.963365800f,
    0.266712757f,  0.963776066f,
    0.265234030f,  0.964184064f,
    0.263754679f,  0.964589793f,
    0.262274707f,  0.964993253f,
    0.260794118f,  0.965394442f,
    0.259312915f,  0.965793359f,
    0.257831102f,  0.966190003f,
    0.256348682f,  0.966584374f,
    0.254865660f,  0.966976471f,
    0.253382037f,  0.967366292f,
    0.251897818f,  0.967753837f,
    0.250413007f,  0.968139105f,
    0.248927606f,  0.968522094f,
    0.247441619f,  0.968902805f,
    0.245955050f,  0.969281235f,
    0.244467903f,  0.969657385f,
    0.242980180f,  0.970031253f,
    0.241491885f,  0.970402839f,
    0.240003022f,  0.970772141f,
    0.238513595f,  0.971139158f,
    0.237023606f,  0.971503891f,
    0.235533059f,  0.971866337f,
    0.234041959f,  0.972226497f,
    0.232550307f,  0.972584369f,
    0.231058108f,  0.972939952f,
    0.229565366f,  0.973293246f,
    0.228072083f,  0.973644250f,
    0.226578264f,  0.973992962f,
    0.225083911f,  0.974339383f,
    0.223589029f,  0.974683511f,
    0.222093621f,  0.975025345f,
    0.220597690f,  0.975364885f,
    0.219101240f,  0.975702130f,
    0.217604275f,  0.976037079f,
    0.216106797f,  0.976369731f,
    0.214608811f,  0.976700086f,
    0.213110320f,  0.977028143f,
    0.211611327f,  0.977353900f,
    0.210111837f,  0.977677358f,
    0.208611852f,  0.977998515f,
    0.207111376f,  0.978317371f,
    0.205610413f,  0.978633924f,
    0.204108966f,  0.978948175f,
    0.202607039f,  0.979260123f,
    0.201104635f,  0.979569766f,
    0.199601758f,  0.979877104f,
    0.198098411f,  0.980182136f,
    0.196594598f,  0.980484862f,
    0.195090322f,  0.980785280f,
    0.193585587f,  0.981083391f,
    0.192080397f,  0.981379193f,
    0.190574755f,  0.981672686f,
    0.189068664f,  0.981963869f,
    0.187562129f,  0.982252741f,
    0.186055152f,  0.982539302f,
    0.184547737f,  0.982823551f,
    0.183039888f,  0.983105487f,
    0.181531608f,  0.983385110f,
    0.180022901f,  0.983662419f,
    0.178513771f,  0.983937413f,
    0.177004220f,  0.984210092f,
    0.175494253f,  0.984480455f,
    0.173983873f,  0.984748502f,
    0.172473084f,  0.985014231f,
    0.170961889f,  0.985277642f,
    0.169450291f,  0.985538735f,
    0.167938295f,  0.985797509f,
    0.166425904f,  0.986053963f,
    0.164913120f,  0.986308097f,
    0.163399949f,  0.986559910f,
    0.161886394f,  0.986809402f,
    0.160372457f,  0.987056571f,
    0.158858143f,  0.987301418f,
    0.157343456f,  0.987543942f,
    0.155828398f,  0.987784142f,
    0.154312973f,  0.988022017f,
    0.152797185f,  0.988257568f,
    0.151281038f,  0.988490793f,
    0.149764535f,  0.988721692f,
    0.148247679f,  0.988950265f,
    0.146730474f,  0.989176510f,
    0.145212925f,  0.989400428f,
    0.143695033f,  0.989622017f,
    0.142176804f,  0.989841278f,
    0.140658239f,  0.990058210f,
    0.139139344f,  0.990272812f,
    0.137620122f,  0.990485084f,
    0.136100575f,  0.990695025f,
    0.134580709f,  0.990902635f,
    0.133060525f,  0.991107914f,
    0.131540029f,  0.991310860f,
    0.130019223f,  0.991511473f,
    0.128498111f,  0.991709754f,
    0.126976696f,  0.991905700f,
    0.125454983f,  0.992099313f,
    0.123932975f,  0.992290591f,
    0.122410675f,  0.992479535f,
    0.120888087f,  0.992666142f,
    0.119365215f,  0.992850414f,
    0.117842062f,  0.993032350f,
    0.116318631f,  0.993211949f,
    0.114794927f,  0.993389211f,
    0.113270952f,  0.993564136f,
    0.111746711f,  0.993736722f,
    0.110222207f,  0.993906970f,
    0.108697444f,  0.994074879f,
    0.107172425f,  0.994240449f,
    0.105647154f,  0.994403680f,
    0.104121634f,  0.994564571f,
    0.102595869f,  0.994723121f,
    0.101069863f,  0.994879331f,
    0.099543619f,  0.995033199f,
    0.098017140f,  0.995184727f,
    0.096490431f,  0.995333912f,
    0.094963495f,  0.995480755f,
    0.093436336f,  0.995625256f,
    0.091908956f,  0.995767414f,
    0.090381361f,  0.995907229f,
    0.088853553f,  0.996044701f,
    0.087325535f,  0.996179829f,
    0.085797312f,  0.996312612f,
    0.084268888f,  0.996443051f,
    0.082740265f,  0.996571146f,
    0.081211447f,  0.996696895f,
    0.079682438f,  0.996820299f,
    0.078153242f,  0.996941358f,
    0.076623861f,  0.997060070f,
    0.075094301f,  0.997176437f,
    0.073564564f,  0.997290457f,
    0.072034653f,  0.997402130f,
    0.070504573f,  0.997511456f,
    0.068974328f,  0.997618435f,
    0.067443920f,  0.997723067f,
    0.065913353f,  0.997825350f,
    0.064382631f,  0.997925286f,
    0.062851758f,  0.998022874f,
    0.061320736f,  0.998118113f,
    0.059789571f,  0.998211003f,
    0.058258265f,  0.998301545f,
    0.056726821f,  0.998389737f,
    0.055195244f,  0.998475581f,
    0.053663538f,  0.998559074f,
    0.052131705f,  0.998640218f,
    0.050599749f,  0.998719012f,
    0.049067674f,  0.998795456f,
    0.047535484f,  0.998869550f,
    0.046003182f,  0.998941293f,
    0.044470772f,  0.999010686f,
    0.042938257f,  0.999077728f,
    0.041405641f,  0.999142419f,
    0.039872928f,  0.999204759f,
    0.038340120f,  0.999264747f,
    0.036807223f,  0.999322385f,
    0.035274239f,  0.999377670f,
    0.033741172f,  0.999430605f,
    0.032208025f,  0.999481187f,
    0.030674803f,  0.999529418f,
    0.029141509f,  0.999575296f,
    0.027608146f,  0.999618822f,
    0.026074718f,  0.999659997f,
    0.024541229f,  0.999698819f,
    0.023007681f,  0.999735288f,
    0.021474080f,  0.999769405f,
    0.019940429f,  0.999801170f,
    0.018406730f,  0.999830582f,
    0.016872988f,  0.999857641f,
    0.015339206f,  0.999882347f,
    0.013805389f,  0.999904701f,
    0.012271538f,  0.999924702f,
    0.010737659f,  0.999942350f,
    0.009203755f,  0.999957645f,
    0.007669829f,  0.999970586f,
    0.006135885f,  0.999981175f,
    0.004601926f,  0.999989411f,
    0.003067957f,  0.999995294f,
    0.001533980f,  0.999998823f,
    0.000000000f,  1.000000000f,
   -0.001533980f,  0.999998823f,
   -0.003067957f,  0.999995294f,
   -0.004601926f,  0.999989411f,
   -0.006135885f,  0.999981175f,
   -0.007669829f,  0.999970586f,
   -0.009203755f,  0.999957645f,
   -0.010737659f,  0.999942350f,
   -0.012271538f,  0.999924702f,
   -0.013805389f,  0.999904701f,
   -0.015339206f,  0.999882347f,
   -0.016872988f,  0.999857641f,
   -0.018406730f,  0.999830582f,
   -0.019940429f,  0.999801170f,
   -0.021474080f,  0.999769405f,
   -0.023007681f,  0.999735288f,
   -0.024541229f,  0.999698819f,
   -0.026074718f,  0.999659997f,
   -0.027608146f,  0.999618822f,
   -0.029141509f,  0.999575296f,
   -0.030674803f,  0.999529418f,
   -0.032208025f,  0.999481187f,
   -0.033741172f,  0.999430605f,
   -0.035274239f,  0.999377670f,
   -0.036807223f,  0.999322385f,
   -0.038340120f,  0.999264747f,
   -0.039872928f,  0.999204759f,
   -0.041405641f,  0.999142419f,
   -0.042938257f,  0.999077728f,
   -0.044470772f,  0.999010686f,
   -0.046003182f,  0.998941293f,
   -0.047535484f,  0.998869550f,
   -0.049067674f,  0.998795456f,
   -0.050599749f,  0.998719012f,
   -0.052131705f,  0.998640218f,
   -0.053663538f,  0.998559074f,
   -0.055195244f,  0.998475581f,
   -0.056726821f,  0.998389737f,
   -0.058258265f,  0.998301545f,
   -0.059789571f,  0.998211003f,
   -0.061320736f,  0.998118113f,
   -0.062851758f,  0.998022874f,
   -0.064382631f,  0.997925286f,
   -0.065913353f,  0.997825350f,
   -0.067443920f,  0.997723067f,
   -0.068974328f,  0.997618435f,
   -0.070504573f,  0.997511456f,
   -0.072034653f,  0.997402130f,
   -0.073564564f,  0.997290457f,
   -0.075094301f,  0.997176437f,
   -0.076623861f,  0.997060070f,
   -0.078153242f,  0.996941358f,
   -0.079682438f,  0.996820299f,
   -0.081211447f,  0.996696895f,
   -0.082740265f,  0.996571146f,
   -0.084268888f,  0.996443051f,
   -0.085797312f,  0.996312612f,
   -0.087325535f,  0.996179829f,
   -0.088853553f,  0.996044701f,
   -0.090381361f,  0.995907229f,
   -0.091908956f,  0.995767414f,
   -0.093436336f,  0.995625256f,
   -0.094963495f,  0.995480755f,
   -0.096490431f,  0.995333912f,
   -0.098017140f,  0.995184727f,
   -0.099543619f,  0.995033199f,
   -0.101069863f,  0.994879331f,
   -0.102595869f,  0.994723121f,
   -0.104121634f,  0.994564571f,
   -0.105647154f,  0.994403680f,
   -0.107172425f,  0.994240449f,
   -0.108697444f,  0.994074879f,
   -0.110222207f,  0.993906970f,
   -0.111746711f,  0.993736722f,
   -0.113270952f,  0.993564136f,
   -0.114794927f,  0.993389211f,
   -0.116318631f,  0.993211949f,
   -0.117842062f,  0.993032350f,
   -0.119365215f,  0.992850414f,
   -0.120888087f,  0.992666142f,
   -0.122410675f,  0.992479535f,
   -0.123932975f,  0.992290591f,
   -0.125454983f,  0.992099313f,
   -0.126976696f,  0.991905700f,
   -0.128498111f,  0.991709754f,
   -0.130019223f,  0.991511473f,
   -0.131540029f,  0.991310860f,
   -0.133060525f,  0.991107914f,
   -0.134580709f,  0.990902635f,
   -0.136100575f,  0.990695025f,
   -0.137620122f,  0.990485084f,
   -0.139139344f,  0.990272812f,
   -0.140658239f,  0.990058210f,
   -0.142176804f,  0.989841278f,
   -0.143695033f,  0.989622017f,
   -0.145212925f,  0.989400428f,
   -0.146730474f,  0.989176510f,
   -0.148247679f,  0.988950265f,
   -0.149764535f,  0.988721692f,
   -0.151281038f,  0.988490793f,
   -0.152797185f,  0.988257568f,
   -0.154312973f,  0.988022017f,
   -0.155828398f,  0.987784142f,
   -0.157343456f,  0.987543942f,
   -0.158858143f,  0.987301418f,
   -0.160372457f,  0.987056571f,
   -0.161886394f,  0.986809402f,
   -0.163399949f,  0.986559910f,
   -0.164913120f,  0.986308097f,
   -0.166425904f,  0.986053963f,
   -0.167938295f,  0.985797509f,
   -0.169450291f,  0.985538735f,
   -0.170961889f,  0.985277642f,
   -0.172473084f,  0.985014231f,
   -0.173983873f,  0.984748502f,
   -0.175494253f,  0.984480455f,
   -0.177004220f,  0.984210092f,
   -0.178513771f,  0.983937413f,
   -0.180022901f,  0.983662419f,
   -0.181531608f,  0.983385110f,
   -0.183039888f,  0.983105487f,
   -0.184547737f,  0.982823551f,
   -0.186055152f,  0.982539302f,
   -0.187562129f,  0.982252741f,
   -0.189068664f,  0.981963869f,
   -0.190574755f,  0.981672686f,
   -0.192080397f,  0.981379193f,
   -0.193585587f,  0.981083391f,
   -0.195090322f,  0.980785280f,
   -0.196594598f,  0.980484862f,
   -0.198098411f,  0.980182136f,
   -0.199601758f,  0.979877104f,
   -0.201104635f,  0.979569766f,
   -0.202607039f,  0.979260123f,
   -0.204108966f,  0.978948175f,
   -0.205610413f,  0.978633924f,
   -0.207111376f,  0.978317371f,
   -0.208611852f,  0.977998515f,
   -0.210111837f,  0.977677358f,
   -0.211611327f,  0.977353900f,
   -0.213110320f,  0.977028143f,
   -0.214608811f,  0.976700086f,
   -0.216106797f,  0.976369731f,
   -0.217604275f,  0.976037079f,
   -0.219101240f,  0.975702130f,
   -0.220597690f,  0.975364885f,
   -0.222093621f,  0.975025345f,
   -0.223589029f,  0.974683511f,
   -0.225083911f,  0.974339383f,
   -0.226578264f,  0.973992962f,
   -0.228072083f,  0.973644250f,
   -0.229565366f,  0.973293246f,
   -0.231058108f,  0.972939952f,
   -0.232550307f,  0.972584369f,
   -0.234041959f,  0.972226497f,
   -0.235533059f,  0.971866337f,
   -0.237023606f,  0.971503891f,
   -0.238513595f,  0.971139158f,
   -0.240003022f,  0.970772141f,
   -0.241491885f,  0.970402839f,
   -0.242980180f,  0.970031253f,
   -0.244467903f,  0.969657385f,
   -0.245955050f,  0.969281235f,
   -0.247441619f,  0.968902805f,
   -0.248927606f,  0.968522094f,
   -0.250413007f,  0.968139105f,
   -0.251897818f,  0.967753837f,
   -0.253382037f,  0.967366292f,
   -0.254865660f,  0.966976471f,
   -0.256348682f,  0.966584374f,
   -0.257831102f,  0.966190003f,
   -0.259312915f,  0.965793359f,
   -0.260794118f,  0.965394442f,
   -0.262274707f,  0.964993253f,
   -0.263754679f,  0.964589793f,
   -0.265234030f,  0.964184064f,
   -0.266712757f,  0.963776066f,
   -0.268190857f,  0.963365800f,
   -0.269668326f,  0.962953267f,
   -0.271145160f,  0.962538468f,
   -0.272621355f,  0.962121404f,
   -0.274096910f,  0.961702077f,
   -0.275571819f,  0.961280486f,
   -0.277046080f,  0.960856633f,
   -0.278519689f,  0.960430519f,
   -0.279992643f,  0.960002146f,
   -0.281464938f,  0.959571513f,
   -0.282936570f,  0.959138622f,
   -0.284407537f,  0.958703475f,
   -0.285877835f,  0.958266071f,
   -0.287347460f,  0.957826413f,
   -0.288816408f,  0.957384501f,
   -0.290284677f,  0.956940336f,
   -0.291752263f,  0.956493919f,
   -0.293219163f,  0.956045251f,
   -0.294685372f,  0.955594334f,
   -0.296150888f,  0.955141168f,
   -0.297615707f,  0.954685755f,
   -0.299079826f,  0.954228095f,
   -0.300543241f,  0.953768190f,
   -0.302005949f,  0.953306040f,
   -0.303467947f,  0.952841648f,
   -0.304929230f,  0.952375013f,
   -0.306389795f,  0.951906137f,
   -0.307849640f,  0.951435021f,
   -0.309308760f,  0.950961666f,
   -0.310767153f,  0.950486074f,
   -0.312224814f,  0.950008245f,
   -0.313681740f,  0.949528181f,
   -0.315137929f,  0.949045882f,
   -0.316593376f,  0.948561350f,
   -0.318048077f,  0.948074586f,
   -0.319502031f,  0.947585591f,
   -0.320955232f,  0.947094366f,
   -0.322407679f,  0.946600913f,
   -0.323859367f,  0.946105232f,
   -0.325310292f,  0.945607325f,
   -0.326760452f,  0.945107193f,
   -0.328209844f,  0.944604837f,
   -0.329658463f,  0.944100258f,
   -0.331106306f,  0.943593458f,
   -0.332553370f,  0.943084437f,
   -0.333999651f,  0.942573198f,
   -0.335445147f,  0.942059740f,
   -0.336889853f,  0.941544065f,
   -0.338333767f,  0.941026175f,
   -0.339776884f,  0.940506071f,
   -0.341219202f,  0.939983753f,
   -0.342660717f,  0.939459224f,
   -0.344101426f,  0.938932484f,
   -0.345541325f,  0.938403534f,
   -0.346980411f,  0.937872376f,
   -0.348418680f,  0.937339012f,
   -0.349856130f,  0.936803442f,
   -0.351292756f,  0.936265667f,
   -0.352728556f,  0.935725689f,
   -0.354163525f,  0.935183510f,
   -0.355597662f,  0.934639130f,
   -0.357030961f,  0.934092550f,
   -0.358463421f,  0.933543773f,
   -0.359895037f,  0.932992799f,
   -0.361325806f,  0.932439629f,
   -0.362755724f,  0.931884266f,
   -0.364184790f,  0.931326709f,
   -0.365612998f,  0.930766961f,
   -0.367040346f,  0.930205023f,
   -0.368466830f,  0.929640896f,
   -0.369892447f,  0.929074581f,
   -0.371317194f,  0.928506080f,
   -0.372741067f,  0.927935395f,
   -0.374164063f,  0.927362526f,
   -0.375586178f,  0.926787474f,
   -0.377007410f,  0.926210242f,
   -0.378427755f,  0.925630831f,
   -0.379847209f,  0.925049241f,
   -0.381265769f,  0.924465474f,
   -0.382683432f,  0.923879533f,
   -0.384100195f,  0.923291417f,
   -0.385516054f,  0.922701128f,
   -0.386931006f,  0.922108669f,
   -0.388345047f,  0.921514039f,
   -0.389758174f,  0.920917242f,
   -0.391170384f,  0.920318277f,
   -0.392581674f,  0.919717146f,
   -0.393992040f,  0.919113852f,
   -0.395401479f,  0.918508394f,
   -0.396809987f,  0.917900776f,
   -0.398217562f,  0.917290997f,
   -0.399624200f,  0.916679060f,
   -0.401029897f,  0.916064966f,
   -0.402434651f,  0.915448716f,
   -0.403838458f,  0.914830312f,
   -0.405241314f,  0.914209756f,
   -0.406643217f,  0.913587048f,
   -0.408044163f,  0.912962190f,
   -0.409444149f,  0.912335185f,
   -0.410843171f,  0.911706032f,
   -0.412241227f,  0.911074734f,
   -0.413638312f,  0.910441292f,
   -0.415034424f,  0.909805708f,
   -0.416429560f,  0.909167983f,
   -0.417823716f,  0.908528119f,
   -0.419216888f,  0.907886116f,
   -0.420609074f,  0.907241978f,
   -0.422000271f,  0.906595705f,
   -0.423390474f,  0.905947298f,
   -0.424779681f,  0.905296759f,
   -0.426167889f,  0.904644091f,
   -0.427555093f,  0.903989293f,
   -0.428941292f,  0.903332368f,
   -0.430326481f,  0.902673318f,
   -0.431710658f,  0.902012144f,
   -0.433093819f,  0.901348847f,
   -0.434475961f,  0.900683429f,
   -0.435857080f,  0.900015892f,
   -0.437237174f,  0.899346237f,
   -0.438616239f,  0.898674466f,
   -0.439994271f,  0.898000580f,
   -0.441371269f,  0.897324581f,
   -0.442747228f,  0.896646470f,
   -0.444122145f,  0.895966250f,
   -0.445496017f,  0.895283921f,
   -0.446868840f,  0.894599486f,
   -0.448240612f,  0.893912945f,
   -0.449611330f,  0.893224301f,
   -0.450980989f,  0.892533555f,
   -0.452349587f,  0.891840709f,
   -0.453717121f,  0.891145765f,
   -0.455083587f,  0.890448723f,
   -0.456448982f,  0.889749586f,
   -0.457813304f,  0.889048356f,
   -0.459176548f,  0.888345033f,
   -0.460538711f,  0.887639620f,
   -0.461899791f,  0.886932119f,
   -0.463259784f,  0.886222530f,
   -0.464618686f,  0.885510856f,
   -0.465976496f,  0.884797098f,
   -0.467333209f,  0.884081259f,
   -0.468688822f,  0.883363339f,
   -0.470043332f,  0.882643340f,
   -0.471396737f,  0.881921264f,
   -0.472749032f,  0.881197113f,
   -0.474100215f,  0.880470889f,
   -0.475450282f,  0.879742593f,
   -0.476799230f,  0.879012226f,
   -0.478147056f,  0.878279792f,
   -0.479493758f,  0.877545290f,
   -0.480839331f,  0.876808724f,
   -0.482183772f,  0.876070094f,
   -0.483527079f,  0.875329403f,
   -0.484869248f,  0.874586652f,
   -0.486210276f,  0.873841843f,
   -0.487550160f,  0.873094978f,
   -0.488888897f,  0.872346059f,
   -0.490226483f,  0.871595087f,
   -0.491562916f,  0.870842063f,
   -0.492898192f,  0.870086991f,
   -0.494232309f,  0.869329871f,
   -0.495565262f,  0.868570706f,
   -0.496897049f,  0.867809497f,
   -0.498227667f,  0.867046246f,
   -0.499557113f,  0.866280954f,
   -0.500885383f,  0.865513624f,
   -0.502212474f,  0.864744258f,
   -0.503538384f,  0.863972856f,
   -0.504863109f,  0.863199422f,
   -0.506186645f,  0.862423956f,
   -0.507508991f,  0.861646461f,
   -0.508830143f,  0.860866939f,
   -0.510150097f,  0.860085390f,
   -0.511468850f,  0.859301818f,
   -0.512786401f,  0.858516224f,
   -0.514102744f,  0.857728610f,
   -0.515417878f,  0.856938977f,
   -0.516731799f,  0.856147328f,
   -0.518044504f,  0.855353665f,
   -0.519355990f,  0.854557988f,
   -0.520666254f,  0.853760301f,
   -0.521975293f,  0.852960605f,
   -0.523283103f,  0.852158902f,
   -0.524589683f,  0.851355193f,
   -0.525895027f,  0.850549481f,
   -0.527199135f,  0.849741768f,
   -0.528502002f,  0.848932055f,
   -0.529803625f,  0.848120345f,
   -0.531104001f,  0.847306639f,
   -0.532403128f,  0.846490939f,
   -0.533701002f,  0.845673247f,
   -0.534997620f,  0.844853565f,
   -0.536292979f,  0.844031895f,
   -0.537587076f,  0.843208240f,
   -0.538879909f,  0.842382600f,
   -0.540171473f,  0.841554977f,
   -0.541461766f,  0.840725375f,
   -0.542750785f,  0.839893794f,
   -0.544038527f,  0.839060237f,
   -0.545324988f,  0.838224706f,
   -0.546610167f,  0.837387202f,
   -0.547894059f,  0.836547727f,
   -0.549176662f,  0.835706284f,
   -0.550457973f,  0.834862875f,
   -0.551737988f,  0.834017501f,
   -0.553016706f,  0.833170165f,
   -0.554294121f,  0.832320868f,
   -0.555570233f,  0.831469612f,
   -0.556845037f,  0.830616400f,
   -0.558118531f,  0.829761234f,
   -0.559390712f,  0.828904115f,
   -0.560661576f,  0.828045045f,
   -0.561931121f,  0.827184027f,
   -0.563199344f,  0.826321063f,
   -0.564466242f,  0.825456154f,
   -0.565731811f,  0.824589303f,
   -0.566996049f,  0.823720511f,
   -0.568258953f,  0.822849781f,
   -0.569520519f,  0.821977115f,
   -0.570780746f,  0.821102515f,
   -0.572039629f,  0.820225983f,
   -0.573297167f,  0.819347520f,
   -0.574553355f,  0.818467130f,
   -0.575808191f,  0.817584813f,
   -0.577061673f,  0.816700573f,
   -0.578313796f,  0.815814411f,
   -0.579564559f,  0.814926329f,
   -0.580813958f,  0.814036330f,
   -0.582061990f,  0.813144415f,
   -0.583308653f,  0.812250587f,
   -0.584553943f,  0.811354847f,
   -0.585797857f,  0.810457198f,
   -0.587040394f,  0.809557642f,
   -0.588281548f,  0.808656182f,
   -0.589521319f,  0.807752818f,
   -0.590759702f,  0.806847554f,
   -0.591996695f,  0.805940391f,
   -0.593232295f,  0.805031331f,
   -0.594466499f,  0.804120377f,
   -0.595699304f,  0.803207531f,
   -0.596930708f,  0.802292796f,
   -0.598160707f,  0.801376172f,
   -0.599389298f,  0.800457662f,
   -0.600616479f,  0.799537269f,
   -0.601842247f,  0.798614995f,
   -0.603066599f,  0.797690841f,
   -0.604289531f,  0.796764810f,
   -0.605511041f,  0.795836905f,
   -0.606731127f,  0.794907126f,
   -0.607949785f,  0.793975478f,
   -0.609167012f,  0.793041960f,
   -0.610382806f,  0.792106577f,
   -0.611597164f,  0.791169330f,
   -0.612810082f,  0.790230221f,
   -0.614021559f,  0.789289253f,
   -0.615231591f,  0.788346428f,
   -0.616440175f,  0.787401747f,
   -0.617647308f,  0.786455214f,
   -0.618852988f,  0.785506830f,
   -0.620057212f,  0.784556597f,
   -0.621259977f,  0.783604519f,
   -0.622461279f,  0.782650596f,
   -0.623661118f,  0.781694832f,
   -0.624859488f,  0.780737229f,
   -0.626056388f,  0.779777788f,
   -0.627251815f,  0.778816512f,
   -0.628445767f,  0.777853404f,
   -0.629638239f,  0.776888466f,
   -0.630829230f,  0.775921699f,
   -0.632018736f,  0.774953107f,
   -0.633206755f,  0.773982691f,
   -0.634393284f,  0.773010453f,
   -0.635578320f,  0.772036397f,
   -0.636761861f,  0.771060524f,
   -0.637943904f,  0.770082837f,
   -0.639124445f,  0.769103338f,
   -0.640303482f,  0.768122029f,
   -0.641481013f,  0.767138912f,
   -0.642657034f,  0.766153990f,
   -0.643831543f,  0.765167266f,
   -0.645004537f,  0.764178741f,
   -0.646176013f,  0.763188417f,
   -0.647345969f,  0.762196298f,
   -0.648514401f,  0.761202385f,
   -0.649681307f,  0.760206682f,
   -0.650846685f,  0.759209189f,
   -0.652010531f,  0.758209910f,
   -0.653172843f,  0.757208847f,
   -0.654333618f,  0.756206001f,
   -0.655492853f,  0.755201377f,
   -0.656650546f,  0.754194975f,
   -0.657806693f,  0.753186799f,
   -0.658961293f,  0.752176850f,
   -0.660114342f,  0.751165132f,
   -0.661265838f,  0.750151646f,
   -0.662415778f,  0.749136395f,
   -0.663564159f,  0.748119380f,
   -0.664710978f,  0.747100606f,
   -0.665856234f,  0.746080074f,
   -0.666999922f,  0.745057785f,
   -0.668142041f,  0.744033744f,
   -0.669282588f,  0.743007952f,
   -0.670421560f,  0.741980412f,
   -0.671558955f,  0.740951125f,
   -0.672694769f,  0.739920095f,
   -0.673829000f,  0.738887324f,
   -0.674961646f,  0.737852815f,
   -0.676092704f,  0.736816569f,
   -0.677222170f,  0.735778589f,
   -0.678350043f,  0.734738878f,
   -0.679476320f,  0.733697438f,
   -0.680600998f,  0.732654272f,
   -0.681724074f,  0.731609381f,
   -0.682845546f,  0.730562769f,
   -0.683965412f,  0.729514438f,
   -0.685083668f,  0.728464390f,
   -0.686200312f,  0.727412629f,
   -0.687315341f,  0.726359155f,
   -0.688428753f,  0.725303972f,
   -0.689540545f,  0.724247083f,
   -0.690650714f,  0.723188489f,
   -0.691759258f,  0.722128194f,
   -0.692866175f,  0.721066199f,
   -0.693971461f,  0.720002508f,
   -0.695075114f,  0.718937122f,
   -0.696177131f,  0.717870045f,
   -0.697277511f,  0.716801279f,
   -0.698376249f,  0.715730825f,
   -0.699473345f,  0.714658688f,
   -0.700568794f,  0.713584869f,
   -0.701662595f,  0.712509371f,
   -0.702754744f,  0.711432196f,
   -0.703845241f,  0.710353347f,
   -0.704934080f,  0.709272826f,
   -0.706021261f,  0.708190637f,
   -0.707106781f,  0.707106781f,
   -0.708190637f,  0.706021261f,
   -0.709272826f,  0.704934080f,
   -0.710353347f,  0.703845241f,
   -0.711432196f,  0.702754744f,
   -0.712509371f,  0.701662595f,
   -0.713584869f,  0.700568794f,
   -0.714658688f,  0.699473345f,
   -0.715730825f,  0.698376249f,
   -0.716801279f,  0.697277511f,
   -0.717870045f,  0.696177131f,
   -0.718937122f,  0.695075114f,
   -0.720002508f,  0.693971461f,
   -0.721066199f,  0.692866175f,
   -0.722128194f,  0.691759258f,
   -0.723188489f,  0.690650714f,
   -0.724247083f,  0.689540545f,
   -0.725303972f,  0.688428753f,
   -0.726359155f,  0.687315341f,
   -0.727412629f,  0.686200312f,
   -0.728464390f,  0.685083668f,
   -0.729514438f,  0.683965412f,
   -0.730562769f,  0.682845546f,
   -0.731609381f,  0.681724074f,
   -0.732654272f,  0.680600998f,
   -0.733697438f,  0.679476320f,
   -0.734738878f,  0.678350043f,
   -0.735778589f,  0.677222170f,
   -0.736816569f,  0.676092704f,
   -0.737852815f,  0.674961646f,
   -0.738887324f,  0.673829000f,
   -0.739920095f,  0.672694769f,
   -0.740951125f,  0.671558955f,
   -0.741980412f,  0.670421560f,
   -0.743007952f,  0.669282588f,
   -0.744033744f,  0.668142041f,
   -0.745057785f,  0.666999922f,
   -0.746080074f,  0.665856234f,
   -0.747100606f,  0.664710978f,
   -0.748119380f,  0.663564159f,
   -0.749136395f,  0.662415778f,
   -0.750151646f,  0.661265838f,
   -0.751165132f,  0.660114342f,
   -0.752176850f,  0.658961293f,
   -0.753186799f,  0.657806693f,
   -0.754194975f,  0.656650546f,
   -0.755201377f,  0.655492853f,
   -0.756206001f,  0.654333618f,
   -0.757208847f,  0.653172843f,
   -0.758209910f,  0.652010531f,
   -0.759209189f,  0.650846685f,
   -0.760206682f,  0.649681307f,
   -0.761202385f,  0.648514401f,
   -0.762196298f,  0.647345969f,
   -0.763188417f,  0.646176013f,
   -0.764178741f,  0.645004537f,
   -0.765167266f,  0.643831543f,
   -0.766153990f,  0.642657034f,
   -0.767138912f,  0.641481013f,
   -0.768122029f,  0.640303482f,
   -0.769103338f,  0.639124445f,
   -0.770082837f,  0.637943904f,
   -0.771060524f,  0.636761861f,
   -0.772036397f,  0.635578320f,
   -0.773010453f,  0.634393284f,
   -0.773982691f,  0.633206755f,
   -0.774953107f,  0.632018736f,
   -0.775921699f,  0.630829230f,
   -0.776888466f,  0.629638239f,
   -0.777853404f,  0.628445767f,
   -0.778816512f,  0.627251815f,
   -0.779777788f,  0.626056388f,
   -0.780737229f,  0.624859488f,
   -0.781694832f,  0.623661118f,
   -0.782650596f,  0.622461279f,
   -0.783604519f,  0.621259977f,
   -0.784556597f,  0.620057212f,
   -0.785506830f,  0.618852988f,
   -0.786455214f,  0.617647308f,
   -0.787401747f,  0.616440175f,
   -0.788346428f,  0.615231591f,
   -0.789289253f,  0.614021559f,
   -0.790230221f,  0.612810082f,
   -0.791169330f,  0.611597164f,
   -0.792106577f,  0.610382806f,
   -0.793041960f,  0.609167012f,
   -0.793975478f,  0.607949785f,
   -0.794907126f,  0.606731127f,
   -0.795836905f,  0.605511041f,
   -0.796764810f,  0.604289531f,
   -0.797690841f,  0.603066599f,
   -0.798614995f,  0.601842247f,
   -0.799537269f,  0.600616479f,
   -0.800457662f,  0.599389298f,
   -0.801376172f,  0.598160707f,
   -0.802292796f,  0.596930708f,
   -0.803207531f,  0.595699304f,
   -0.804120377f,  0.594466499f,
   -0.805031331f,  0.593232295f,
   -0.805940391f,  0.591996695f,
   -0.806847554f,  0.590759702f,
   -0.807752818f,  0.589521319f,
   -0.808656182f,  0.588281548f,
   -0.809557642f,  0.587040394f,
   -0.810457198f,  0.585797857f,
   -0.811354847f,  0.584553943f,
   -0.812250587f,  0.583308653f,
   -0.813144415f,  0.582061990f,
   -0.814036330f,  0.580813958f,
   -0.814926329f,  0.579564559f,
   -0.815814411f,  0.578313796f,
   -0.816700573f,  0.577061673f,
   -0.817584813f,  0.575808191f,
   -0.818467130f,  0.574553355f,
   -0.819347520f,  0.573297167f,
   -0.820225983f,  0.572039629f,
   -0.821102515f,  0.570780746f,
   -0.821977115f,  0.569520519f,
   -0.822849781f,  0.568258953f,
   -0.823720511f,  0.566996049f,
   -0.824589303f,  0.565731811f,
   -0.825456154f,  0.564466242f,
   -0.826321063f,  0.563199344f,
   -0.827184027f,  0.561931121f,
   -0.828045045f,  0.560661576f,
   -0.828904115f,  0.559390712f,
   -0.829761234f,  0.558118531f,
   -0.830616400f,  0.556845037f,
   -0.831469612f,  0.555570233f,
   -0.832320868f,  0.554294121f,
   -0.833170165f,  0.553016706f,
   -0.834017501f,  0.551737988f,
   -0.834862875f,  0.550457973f,
   -0.835706284f,  0.549176662f,
   -0.836547727f,  0.547894059f,
   -0.837387202f,  0.546610167f,
   -0.838224706f,  0.545324988f,
   -0.839060237f,  0.544038527f,
   -0.839893794f,  0.542750785f,
   -0.840725375f,  0.541461766f,
   -0.841554977f,  0.540171473f,
   -0.842382600f,  0.538879909f,
   -0.843208240f,  0.537587076f,
   -0.844031895f,  0.536292979f,
   -0.844853565f,  0.534997620f,
   -0.845673247f,  0.533701002f,
   -0.846490939f,  0.532403128f,
   -0.847306639f,  0.531104001f,
   -0.848120345f,  0.529803625f,
   -0.848932055f,  0.528502002f,
   -0.849741768f,  0.527199135f,
   -0.850549481f,  0.525895027f,
   -0.851355193f,  0.524589683f,
   -0.852158902f,  0.523283103f,
   -0.852960605f,  0.521975293f,
   -0.853760301f,  0.520666254f,
   -0.854557988f,  0.519355990f,
   -0.855353665f,  0.518044504f,
   -0.856147328f,  0.516731799f,
   -0.856938977f,  0.515417878f,
   -0.857728610f,  0.514102744f,
   -0.858516224f,  0.512786401f,
   -0.859301818f,  0.511468850f,
   -0.860085390f,  0.510150097f,
   -0.860866939f,  0.508830143f,
   -0.861646461f,  0.507508991f,
   -0.862423956f,  0.506186645f,
   -0.863199422f,  0.504863109f,
   -0.863972856f,  0.503538384f,
   -0.864744258f,  0.502212474f,
   -0.865513624f,  0.500885383f,
   -0.866280954f,  0.499557113f,
   -0.867046246f,  0.498227667f,
   -0.867809497f,  0.496897049f,
   -0.868570706f,  0.495565262f,
   -0.869329871f,  0.494232309f,
   -0.870086991f,  0.492898192f,
   -0.870842063f,  0.491562916f,
   -0.871595087f,  0.490226483f,
   -0.872346059f,  0.488888897f,
   -0.873094978f,  0.487550160f,
   -0.873841843f,  0.486210276f,
   -0.874586652f,  0.484869248f,
   -0.875329403f,  0.483527079f,
   -0.876070094f,  0.482183772f,
   -0.876808724f,  0.480839331f,
   -0.877545290f,  0.479493758f,
   -0.878279792f,  0.478147056f,
   -0.879012226f,  0.476799230f,
   -0.879742593f,  0.475450282f,
   -0.880470889f,  0.474100215f,
   -0.881197113f,  0.472749032f,
   -0.881921264f,  0.471396737f,
   -0.882643340f,  0.470043332f,
   -0.883363339f,  0.468688822f,
   -0.884081259f,  0.467333209f,
   -0.884797098f,  0.465976496f,
   -0.885510856f,  0.464618686f,
   -0.886222530f,  0.463259784f,
   -0.886932119f,  0.461899791f,
   -0.887639620f,  0.460538711f,
   -0.888345033f,  0.459176548f,
   -0.889048356f,  0.457813304f,
   -0.889749586f,  0.456448982f,
   -0.890448723f,  0.455083587f,
   -0.891145765f,  0.453717121f,
   -0.891840709f,  0.452349587f,
   -0.892533555f,  0.450980989f,
   -0.893224301f,  0.449611330f,
   -0.893912945f,  0.448240612f,
   -0.894599486f,  0.446868840f,
   -0.895283921f,  0.445496017f,
   -0.895966250f,  0.444122145f,
   -0.896646470f,  0.442747228f,
   -0.897324581f,  0.441371269f,
   -0.898000580f,  0.439994271f,
   -0.898674466f,  0.438616239f,
   -0.899346237f,  0.437237174f,
   -0.900015892f,  0.435857080f,
   -0.900683429f,  0.434475961f,
   -0.901348847f,  0.433093819f,
   -0.902012144f,  0.431710658f,
   -0.902673318f,  0.430326481f,
   -0.903332368f,  0.428941292f,
   -0.903989293f,  0.427555093f,
   -0.904644091f,  0.426167889f,
   -0.905296759f,  0.424779681f,
   -0.905947298f,  0.423390474f,
   -0.906595705f,  0.422000271f,
   -0.907241978f,  0.420609074f,
   -0.907886116f,  0.419216888f,
   -0.908528119f,  0.417823716f,
   -0.909167983f,  0.416429560f,
   -0.909805708f,  0.415034424f,
   -0.910441292f,  0.413638312f,
   -0.911074734f,  0.412241227f,
   -0.911706032f,  0.410843171f,
   -0.912335185f,  0.409444149f,
   -0.912962190f,  0.408044163f,
   -0.913587048f,  0.406643217f,
   -0.914209756f,  0.405241314f,
   -0.914830312f,  0.403838458f,
   -0.915448716f,  0.402434651f,
   -0.916064966f,  0.401029897f,
   -0.916679060f,  0.399624200f,
   -0.917290997f,  0.398217562f,
   -0.917900776f,  0.396809987f,
   -0.918508394f,  0.395401479f,
   -0.919113852f,  0.393992040f,
   -0.919717146f,  0.392581674f,
   -0.920318277f,  0.391170384f,
   -0.920917242f,  0.389758174f,
   -0.921514039f,  0.388345047f,
   -0.922108669f,  0.386931006f,
   -0.922701128f,  0.385516054f,
   -0.923291417f,  0.384100195f,
   -0.923879533f,  0.382683432f,
   -0.924465474f,  0.381265769f,
   -0.925049241f,  0.379847209f,
   -0.925630831f,  0.378427755f,
   -0.926210242f,  0.377007410f,
   -0.926787474f,  0.375586178f,
   -0.927362526f,  0.374164063f,
   -0.927935395f,  0.372741067f,
   -0.928506080f,  0.371317194f,
   -0.929074581f,  0.369892447f,
   -0.929640896f,  0.368466830f,
   -0.930205023f,  0.367040346f,
   -0.930766961f,  0.365612998f,
   -0.931326709f,  0.364184790f,
   -0.931884266f,  0.362755724f,
   -0.932439629f,  0.361325806f,
   -0.932992799f,  0.359895037f,
   -0.933543773f,  0.358463421f,
   -0.934092550f,  0.357030961f,
   -0.934639130f,  0.355597662f,
   -0.935183510f,  0.354163525f,
   -0.935725689f,  0.352728556f,
   -0.936265667f,  0.351292756f,
   -0.936803442f,  0.349856130f,
   -0.937339012f,  0.348418680f,
   -0.937872376f,  0.346980411f,
   -0.938403534f,  0.345541325f,
   -0.938932484f,  0.344101426f,
   -0.939459224f,  0.342660717f,
   -0.939983753f,  0.341219202f,
   -0.940506071f,  0.339776884f,
   -0.941026175f,  0.338333767f,
   -0.941544065f,  0.336889853f,
   -0.942059740f,  0.335445147f,
   -0.942573198f,  0.333999651f,
   -0.943084437f,  0.332553370f,
   -0.943593458f,  0.331106306f,
   -0.944100258f,  0.329658463f,
   -0.944604837f,  0.328209844f,
   -0.945107193f,  0.326760452f,
   -0.945607325f,  0.325310292f,
   -0.946105232f,  0.323859367f,
   -0.946600913f,  0.322407679f,
   -0.947094366f,  0.320955232f,
   -0.947585591f,  0.319502031f,
   -0.948074586f,  0.318048077f,
   -0.948561350f,  0.316593376f,
   -0.949045882f,  0.315137929f,
   -0.949528181f,  0.313681740f,
   -0.950008245f,  0.312224814f,
   -0.950486074f,  0.310767153f,
   -0.950961666f,  0.309308760f,
   -0.951435021f,  0.307849640f,
   -0.951906137f,  0.306389795f,
   -0.952375013f,  0.304929230f,
   -0.952841648f,  0.303467947f,
   -0.953306040f,  0.302005949f,
   -0.953768190f,  0.300543241f,
   -0.954228095f,  0.299079826f,
   -0.954685755f,  0.297615707f,
   -0.955141168f,  0.296150888f,
   -0.955594334f,  0.294685372f,
   -0.956045251f,  0.293219163f,
   -0.956493919f,  0.291752263f,
   -0.956940336f,  0.290284677f,
   -0.957384501f,  0.288816408f,
   -0.957826413f,  0.287347460f,
   -0.958266071f,  0.285877835f,
   -0.958703475f,  0.284407537f,
   -0.959138622f,  0.282936570f,
   -0.959571513f,  0.281464938f,
   -0.960002146f,  0.279992643f,
   -0.960430519f,  0.278519689f,
   -0.960856633f,  0.277046080f,
   -0.961280486f,  0.275571819f,
   -0.961702077f,  0.274096910f,
   -0.962121404f,  0.272621355f,
   -0.962538468f,  0.271145160f,
   -0.962953267f,  0.269668326f,
   -0.963365800f,  0.268190857f,
   -0.963776066f,  0.266712757f,
   -0.964184064f,  0.265234030f,
   -0.964589793f,  0.263754679f,
   -0.964993253f,  0.262274707f,
   -0.965394442f,  0.260794118f,
   -0.965793359f,  0.259312915f,
   -0.966190003f,  0.257831102f,
   -0.966584374f,  0.256348682f,
   -0.966976471f,  0.254865660f,
   -0.967366292f,  0.253382037f,
   -0.967753837f,  0.251897818f,
   -0.968139105f,  0.250413007f,
   -0.968522094f,  0.248927606f,
   -0.968902805f,  0.247441619f,
   -0.969281235f,  0.245955050f,
   -0.969657385f,  0.244467903f,
   -0.970031253f,  0.242980180f,
   -0.970402839f,  0.241491885f,
   -0.970772141f,  0.240003022f,
   -0.971139158f,  0.238513595f,
   -0.971503891f,  0.237023606f,
   -0.971866337f,  0.235533059f,
   -0.972226497f,  0.234041959f,
   -0.972584369f,  0.232550307f,
   -0.972939952f,  0.231058108f,
   -0.973293246f,  0.229565366f,
   -0.973644250f,  0.228072083f,
   -0.973992962f,  0.226578264f,
   -0.974339383f,  0.225083911f,
   -0.974683511f,  0.223589029f,
   -0.975025345f,  0.222093621f,
   -0.975364885f,  0.220597690f,
   -0.975702130f,  0.219101240f,
   -0.976037079f,  0.217604275f,
   -0.976369731f,  0.216106797f,
   -0.976700086f,  0.214608811f,
   -0.977028143f,  0.213110320f,
   -0.977353900f,  0.211611327f,
   -0.977677358f,  0.210111837f,
   -0.977998515f,  0.208611852f,
   -0.978317371f,  0.207111376f,
   -0.978633924f,  0.205610413f,
   -0.978948175f,  0.204108966f,
   -0.979260123f,  0.202607039f,
   -0.979569766f,  0.201104635f,
   -0.979877104f,  0.199601758f,
   -0.980182136f,  0.198098411f,
   -0.980484862f,  0.196594598f,
   -0.980785280f,  0.195090322f,
   -0.981083391f,  0.193585587f,
   -0.981379193f,  0.192080397f,
   -0.981672686f,  0.190574755f,
   -0.981963869f,  0.189068664f,
   -0.982252741f,  0.187562129f,
   -0.982539302f,  0.186055152f,
   -0.982823551f,  0.184547737f,
   -0.983105487f,  0.183039888f,
   -0.983385110f,  0.181531608f,
   -0.983662419f,  0.180022901f,
   -0.983937413f,  0.178513771f,
   -0.984210092f,  0.177004220f,
   -0.984480455f,  0.175494253f,
   -0.984748502f,  0.173983873f,
   -0.985014231f,  0.172473084f,
   -0.985277642f,  0.170961889f,
   -0.985538735f,  0.169450291f,
   -0.985797509f,  0.167938295f,
   -0.986053963f,  0.166425904f,
   -0.986308097f,  0.164913120f,
   -0.986559910f,  0.163399949f,
   -0.986809402f,  0.161886394f,
   -0.987056571f,  0.160372457f,
   -0.987301418f,  0.158858143f,
   -0.987543942f,  0.157343456f,
   -0.987784142f,  0.155828398f,
   -0.988022017f,  0.154312973f,
   -0.988257568f,  0.152797185f,
   -0.988490793f,  0.151281038f,
   -0.988721692f,  0.149764535f,
   -0.988950265f,  0.148247679f,
   -0.989176510f,  0.146730474f,
   -0.989400428f,  0.145212925f,
   -0.989622017f,  0.143695033f,
   -0.989841278f,  0.142176804f,
   -0.990058210f,  0.140658239f,
   -0.990272812f,  0.139139344f,
   -0.990485084f,  0.137620122f,
   -0.990695025f,  0.136100575f,
   -0.990902635f,  0.134580709f,
   -0.991107914f,  0.133060525f,
   -0.991310860f,  0.131540029f,
   -0.991511473f,  0.130019223f,
   -0.991709754f,  0.128498111f,
   -0.991905700f,  0.126976696f,
   -0.992099313f,  0.125454983f,
   -0.992290591f,  0.123932975f,
   -0.992479535f,  0.122410675f,
   -0.992666142f,  0.120888087f,
   -0.992850414f,  0.119365215f,
   -0.993032350f,  0.117842062f,
   -0.993211949f,  0.116318631f,
   -0.993389211f,  0.114794927f,
   -0.993564136f,  0.113270952f,
   -0.993736722f,  0.111746711f,
   -0.993906970f,  0.110222207f,
   -0.994074879f,  0.108697444f,
   -0.994240449f,  0.107172425f,
   -0.994403680f,  0.105647154f,
   -0.994564571f,  0.104121634f,
   -0.994723121f,  0.102595869f,
   -0.994879331f,  0.101069863f,
   -0.995033199f,  0.099543619f,
   -0.995184727f,  0.098017140f,
   -0.995333912f,  0.096490431f,
   -0.995480755f,  0.094963495f,
   -0.995625256f,  0.093436336f,
   -0.995767414f,  0.091908956f,
   -0.995907229f,  0.090381361f,
   -0.996044701f,  0.088853553f,
   -0.996179829f,  0.087325535f,
   -0.996312612f,  0.085797312f,
   -0.996443051f,  0.084268888f,
   -0.996571146f,  0.082740265f,
   -0.996696895f,  0.081211447f,
   -0.996820299f,  0.079682438f,
   -0.996941358f,  0.078153242f,
   -0.997060070f,  0.076623861f,
   -0.997176437f,  0.075094301f,
   -0.997290457f,  0.073564564f,
   -0.997402130f,  0.072034653f,
   -0.997511456f,  0.070504573f,
   -0.997618435f,  0.068974328f,
   -0.997723067f,  0.067443920f,
   -0.997825350f,  0.065913353f,
   -0.997925286f,  0.064382631f,
   -0.998022874f,  0.062851758f,
   -0.998118113f,  0.061320736f,
   -0.998211003f,  0.059789571f,
   -0.998301545f,  0.058258265f,
   -0.998389737f,  0.056726821f,
   -0.998475581f,  0.055195244f,
   -0.998559074f,  0.053663538f,
   -0.998640218f,  0.052131705f,
   -0.998719012f,  0.050599749f,
   -0.998795456f,  0.049067674f,
   -0.998869550f,  0.047535484f,
   -0.998941293f,  0.046003182f,
   -0.999010686f,  0.044470772f,
   -0.999077728f,  0.042938257f,
   -0.999142419f,  0.041405641f,
   -0.999204759f,  0.039872928f,
   -0.999264747f,  0.038340120f,
   -0.999322385f,  0.036807223f,
   -0.999377670f,  0.035274239f,
   -0.999430605f,  0.033741172f,
   -0.999481187f,  0.032208025f,
   -0.999529418f,  0.030674803f,
   -0.999575296f,  0.029141509f,
   -0.999618822f,  0.027608146f,
   -0.999659997f,  0.026074718f,
   -0.999698819f,  0.024541229f,
   -0.999735288f,  0.023007681f,
   -0.999769405f,  0.021474080f,
   -0.999801170f,  0.019940429f,
   -0.999830582f,  0.018406730f,
   -0.999857641f,  0.016872988f,
   -0.999882347f,  0.015339206f,
   -0.999904701f,  0.013805389f,
   -0.999924702f,  0.012271538f,
   -0.999942350f,  0.010737659f,
   -0.999957645f,  0.009203755f,
   -0.999970586f,  0.007669829f,
   -0.999981175f,  0.006135885f,
   -0.999989411f,  0.004601926f,
   -0.999995294f,  0.003067957f,
   -0.999998823f,  0.001533980f,
   -1.000000000f,  0.000000000f,
   -0.999998823f, -0.001533980f,
   -0.999995294f, -0.003067957f,
   -0.999989411f, -0.004601926f,
   -0.999981175f, -0.006135885f,
   -0.999970586f, -0.007669829f,
   -0.999957645f, -0.009203755f,
   -0.999942350f, -0.010737659f,
   -0.999924702f, -0.012271538f,
   -0.999904701f, -0.013805389f,
   -0.999882347f, -0.015339206f,
   -0.999857641f, -0.016872988f,
   -0.999830582f, -0.018406730f,
   -0.999801170f, -0.019940429f,
   -0.999769405f, -0.021474080f,
   -0.999735288f, -0.023007681f,
   -0.999698819f, -0.024541229f,
   -0.999659997f, -0.026074718f,
   -0.999618822f, -0.027608146f,
   -0.999575296f, -0.029141509f,
   -0.999529418f, -0.030674803f,
   -0.999481187f, -0.032208025f,
   -0.999430605f, -0.033741172f,
   -0.999377670f, -0.035274239f,
   -0.999322385f, -0.036807223f,
   -0.999264747f, -0.038340120f,
   -0.999204759f, -0.039872928f,
   -0.999142419f, -0.041405641f,
   -0.999077728f, -0.042938257f,
   -0.999010686f, -0.044470772f,
   -0.998941293f, -0.046003182f,
   -0.998869550f, -0.047535484f,
   -0.998795456f, -0.049067674f,
   -0.998719012f, -0.050599749f,
   -0.998640218f, -0.052131705f,
   -0.998559074f, -0.053663538f,
   -0.998475581f, -0.055195244f,
   -0.998389737f, -0.056726821f,
   -0.998301545f, -0.058258265f,
   -0.998211003f, -0.059789571f,
   -0.998118113f, -0.061320736f,
   -0.998022874f, -0.062851758f,
   -0.997925286f, -0.064382631f,
   -0.997825350f, -0.065913353f,
   -0.997723067f, -0.067443920f,
   -0.997618435f, -0.068974328f,
   -0.997511456f, -0.070504573f,
   -0.997402130f, -0.072034653f,
   -0.997290457f, -0.073564564f,
   -0.997176437f, -0.075094301f,
   -0.997060070f, -0.076623861f,
   -0.996941358f, -0.078153242f,
   -0.996820299f, -0.079682438f,
   -0.996696895f, -0.081211447f,
   -0.996571146f, -0.082740265f,
   -0.996443051f, -0.084268888f,
   -0.996312612f, -0.085797312f,
   -0.996179829f, -0.087325535f,
   -0.996044701f, -0.088853553f,
   -0.995907229f, -0.090381361f,
   -0.995767414f, -0.091908956f,
   -0.995625256f, -0.093436336f,
   -0.995480755f, -0.094963495f,
   -0.995333912f, -0.096490431f,
   -0.995184727f, -0.098017140f,
   -0.995033199f, -0.099543619f,
   -0.994879331f, -0.101069863f,
   -0.994723121f, -0.102595869f,
   -0.994564571f, -0.104121634f,
   -0.994403680f, -0.105647154f,
   -0.994240449f, -0.107172425f,
   -0.994074879f, -0.108697444f,
   -0.993906970f, -0.110222207f,
   -0.993736722f, -0.111746711f,
   -0.993564136f, -0.113270952f,
   -0.993389211f, -0.114794927f,
   -0.993211949f, -0.116318631f,
   -0.993032350f, -0.117842062f,
   -0.992850414f, -0.119365215f,
   -0.992666142f, -0.120888087f,
   -0.992479535f, -0.122410675f,
   -0.992290591f, -0.123932975f,
   -0.992099313f, -0.125454983f,
   -0.991905700f, -0.126976696f,
   -0.991709754f, -0.128498111f,
   -0.991511473f, -0.130019223f,
   -0.991310860f, -0.131540029f,
   -0.991107914f, -0.133060525f,
   -0.990902635f, -0.134580709f,
   -0.990695025f, -0.136100575f,
   -0.990485084f, -0.137620122f,
   -0.990272812f, -0.139139344f,
   -0.990058210f, -0.140658239f,
   -0.989841278f, -0.142176804f,
   -0.989622017f, -0.143695033f,
   -0.989400428f, -0.145212925f,
   -0.989176510f, -0.146730474f,
   -0.988950265f, -0.148247679f,
   -0.988721692f, -0.149764535f,
   -0.988490793f, -0.151281038f,
   -0.988257568f, -0.152797185f,
   -0.988022017f, -0.154312973f,
   -0.987784142f, -0.155828398f,
   -0.987543942f, -0.157343456f,
   -0.987301418f, -0.158858143f,
   -0.987056571f, -0.160372457f,
   -0.986809402f, -0.161886394f,
   -0.986559910f, -0.163399949f,
   -0.986308097f, -0.164913120f,
   -0.986053963f, -0.166425904f,
   -0.985797509f, -0.167938295f,
   -0.985538735f, -0.169450291f,
   -0.985277642f, -0.170961889f,
   -0.985014231f, -0.172473084f,
   -0.984748502f, -0.173983873f,
   -0.984480455f, -0.175494253f,
   -0.984210092f, -0.177004220f,
   -0.983937413f, -0.178513771f,
   -0.983662419f, -0.180022901f,
   -0.983385110f, -0.181531608f,
   -0.983105487f, -0.183039888f,
   -0.982823551f, -0.184547737f,
   -0.982539302f, -0.186055152f,
   -0.982252741f, -0.187562129f,
   -0.981963869f, -0.189068664f,
   -0.981672686f, -0.190574755f,
   -0.981379193f, -0.192080397f,
   -0.981083391f, -0.193585587f,
   -0.980785280f, -0.195090322f,
   -0.980484862f, -0.196594598f,
   -0.980182136f, -0.198098411f,
   -0.979877104f, -0.199601758f,
   -0.979569766f, -0.201104635f,
   -0.979260123f, -0.202607039f,
   -0.978948175f, -0.204108966f,
   -0.978633924f, -0.205610413f,
   -0.978317371f, -0.207111376f,
   -0.977998515f, -0.208611852f,
   -0.977677358f, -0.210111837f,
   -0.977353900f, -0.211611327f,
   -0.977028143f, -0.213110320f,
   -0.976700086f, -0.214608811f,
   -0.976369731f, -0.216106797f,
   -0.976037079f, -0.217604275f,
   -0.975702130f, -0.219101240f,
   -0.975364885f, -0.220597690f,
   -0.975025345f, -0.222093621f,
   -0.974683511f, -0.223589029f,
   -0.974339383f, -0.225083911f,
   -0.973992962f, -0.226578264f,
   -0.973644250f, -0.228072083f,
   -0.973293246f, -0.229565366f,
   -0.972939952f, -0.231058108f,
   -0.972584369f, -0.232550307f,
   -0.972226497f, -0.234041959f,
   -0.971866337f, -0.235533059f,
   -0.971503891f, -0.237023606f,
   -0.971139158f, -0.238513595f,
   -0.970772141f, -0.240003022f,
   -0.970402839f, -0.241491885f,
   -0.970031253f, -0.242980180f,
   -0.969657385f, -0.244467903f,
   -0.969281235f, -0.245955050f,
   -0.968902805f, -0.247441619f,
   -0.968522094f, -0.248927606f,
   -0.968139105f, -0.250413007f,
   -0.967753837f, -0.251897818f,
   -0.967366292f, -0.253382037f,
   -0.966976471f, -0.254865660f,
   -0.966584374f, -0.256348682f,
   -0.966190003f, -0.257831102f,
   -0.965793359f, -0.259312915f,
   -0.965394442f, -0.260794118f,
   -0.964993253f, -0.262274707f,
   -0.964589793f, -0.263754679f,
   -0.964184064f, -0.265234030f,
   -0.963776066f, -0.266712757f,
   -0.963365800f, -0.268190857f,
   -0.962953267f, -0.269668326f,
   -0.962538468f, -0.271145160f,
   -0.962121404f, -0.272621355f,
   -0.961702077f, -0.274096910f,
   -0.961280486f, -0.275571819f,
   -0.960856633f, -0.277046080f,
   -0.960430519f, -0.278519689f,
   -0.960002146f, -0.279992643f,
   -0.959571513f, -0.281464938f,
   -0.959138622f, -0.282936570f,
   -0.958703475f, -0.284407537f,
   -0.958266071f, -0.285877835f,
   -0.957826413f, -0.287347460f,
   -0.957384501f, -0.288816408f,
   -0.956940336f, -0.290284677f,
   -0.956493919f, -0.291752263f,
   -0.956045251f, -0.293219163f,
   -0.955594334f, -0.294685372f,
   -0.955141168f, -0.296150888f,
   -0.954685755f, -0.297615707f,
   -0.954228095f, -0.299079826f,
   -0.953768190f, -0.300543241f,
   -0.953306040f, -0.302005949f,
   -0.952841648f, -0.303467947f,
   -0.952375013f, -0.304929230f,
   -0.951906137f, -0.306389795f,
   -0.951435021f, -0.307849640f,
   -0.950961666f, -0.309308760f,
   -0.950486074f, -0.310767153f,
   -0.950008245f, -0.312224814f,
   -0.949528181f, -0.313681740f,
   -0.949045882f, -0.315137929f,
   -0.948561350f, -0.316593376f,
   -0.948074586f, -0.318048077f,
   -0.947585591f, -0.319502031f,
   -0.947094366f, -0.320955232f,
   -0.946600913f, -0.322407679f,
   -0.946105232f, -0.323859367f,
   -0.945607325f, -0.325310292f,
   -0.945107193f, -0.326760452f,
   -0.944604837f, -0.328209844f,
   -0.944100258f, -0.329658463f,
   -0.943593458f, -0.331106306f,
   -0.943084437f, -0.332553370f,
   -0.942573198f, -0.333999651f,
   -0.942059740f, -0.335445147f,
   -0.941544065f, -0.336889853f,
   -0.941026175f, -0.338333767f,
   -0.940506071f, -0.339776884f,
   -0.939983753f, -0.341219202f,
   -0.939459224f, -0.342660717f,
   -0.938932484f, -0.344101426f,
   -0.938403534f, -0.345541325f,
   -0.937872376f, -0.346980411f,
   -0.937339012f, -0.348418680f,
   -0.936803442f, -0.349856130f,
   -0.936265667f, -0.351292756f,
   -0.935725689f, -0.352728556f,
   -0.935183510f, -0.354163525f,
   -0.934639130f, -0.355597662f,
   -0.934092550f, -0.357030961f,
   -0.933543773f, -0.358463421f,
   -0.932992799f, -0.359895037f,
   -0.932439629f, -0.361325806f,
   -0.931884266f, -0.362755724f,
   -0.931326709f, -0.364184790f,
   -0.930766961f, -0.365612998f,
   -0.930205023f, -0.367040346f,
   -0.929640896f, -0.368466830f,
   -0.929074581f, -0.369892447f,
   -0.928506080f, -0.371317194f,
   -0.927935395f, -0.372741067f,
   -0.927362526f, -0.374164063f,
   -0.926787474f, -0.375586178f,
   -0.926210242f, -0.377007410f,
   -0.925630831f, -0.378427755f,
   -0.925049241f, -0.379847209f,
   -0.924465474f, -0.381265769f,
   -0.923879533f, -0.382683432f,
   -0.923291417f, -0.384100195f,
   -0.922701128f, -0.385516054f,
   -0.922108669f, -0.386931006f,
   -0.921514039f, -0.388345047f,
   -0.920917242f, -0.389758174f,
   -0.920318277f, -0.391170384f,
   -0.919717146f, -0.392581674f,
   -0.919113852f, -0.393992040f,
   -0.918508394f, -0.395401479f,
   -0.917900776f, -0.396809987f,
   -0.917290997f, -0.398217562f,
   -0.916679060f, -0.399624200f,
   -0.916064966f, -0.401029897f,
   -0.915448716f, -0.402434651f,
   -0.914830312f, -0.403838458f,
   -0.914209756f, -0.405241314f,
   -0.913587048f, -0.406643217f,
   -0.912962190f, -0.408044163f,
   -0.912335185f, -0.409444149f,
   -0.911706032f, -0.410843171f,
   -0.911074734f, -0.412241227f,
   -0.910441292f, -0.413638312f,
   -0.909805708f, -0.415034424f,
   -0.909167983f, -0.416429560f,
   -0.908528119f, -0.417823716f,
   -0.907886116f, -0.419216888f,
   -0.907241978f, -0.420609074f,
   -0.906595705f, -0.422000271f,
   -0.905947298f, -0.423390474f,
   -0.905296759f, -0.424779681f,
   -0.904644091f, -0.426167889f,
   -0.903989293f, -0.427555093f,
   -0.903332368f, -0.428941292f,
   -0.902673318f, -0.430326481f,
   -0.902012144f, -0.431710658f,
   -0.901348847f, -0.433093819f,
   -0.900683429f, -0.434475961f,
   -0.900015892f, -0.435857080f,
   -0.899346237f, -0.437237174f,
   -0.898674466f, -0.438616239f,
   -0.898000580f, -0.439994271f,
   -0.897324581f, -0.441371269f,
   -0.896646470f, -0.442747228f,
   -0.895966250f, -0.444122145f,
   -0.895283921f, -0.445496017f,
   -0.894599486f, -0.446868840f,
   -0.893912945f, -0.448240612f,
   -0.893224301f, -0.449611330f,
   -0.892533555f, -0.450980989f,
   -0.891840709f, -0.452349587f,
   -0.891145765f, -0.453717121f,
   -0.890448723f, -0.455083587f,
   -0.889749586f, -0.456448982f,
   -0.889048356f, -0.457813304f,
   -0.888345033f, -0.459176548f,
   -0.887639620f, -0.460538711f,
   -0.886932119f, -0.461899791f,
   -0.886222530f, -0.463259784f,
   -0.885510856f, -0.464618686f,
   -0.884797098f, -0.465976496f,
   -0.884081259f, -0.467333209f,
   -0.883363339f, -0.468688822f,
   -0.882643340f, -0.470043332f,
   -0.881921264f, -0.471396737f,
   -0.881197113f, -0.472749032f,
   -0.880470889f, -0.474100215f,
   -0.879742593f, -0.475450282f,
   -0.879012226f, -0.476799230f,
   -0.878279792f, -0.478147056f,
   -0.877545290f, -0.479493758f,
   -0.876808724f, -0.480839331f,
   -0.876070094f, -0.482183772f,
   -0.875329403f, -0.483527079f,
   -0.874586652f, -0.484869248f,
   -0.873841843f, -0.486210276f,
   -0.873094978f, -0.487550160f,
   -0.872346059f, -0.488888897f,
   -0.871595087f, -0.490226483f,
   -0.870842063f, -0.491562916f,
   -0.870086991f, -0.492898192f,
   -0.869329871f, -0.494232309f,
   -0.868570706f, -0.495565262f,
   -0.867809497f, -0.496897049f,
   -0.867046246f, -0.498227667f,
   -0.866280954f, -0.499557113f,
   -0.865513624f, -0.500885383f,
   -0.864744258f, -0.502212474f,
   -0.863972856f, -0.503538384f,
   -0.863199422f, -0.504863109f,
   -0.862423956f, -0.506186645f,
   -0.861646461f, -0.507508991f,
   -0.860866939f, -0.508830143f,
   -0.860085390f, -0.510150097f,
   -0.859301818f, -0.511468850f,
   -0.858516224f, -0.512786401f,
   -0.857728610f, -0.514102744f,
   -0.856938977f, -0.515417878f,
   -0.856147328f, -0.516731799f,
   -0.855353665f, -0.518044504f,
   -0.854557988f, -0.519355990f,
   -0.853760301f, -0.520666254f,
   -0.852960605f, -0.521975293f,
   -0.852158902f, -0.523283103f,
   -0.851355193f, -0.524589683f,
   -0.850549481f, -0.525895027f,
   -0.849741768f, -0.527199135f,
   -0.848932055f, -0.528502002f,
   -0.848120345f, -0.529803625f,
   -0.847306639f, -0.531104001f,
   -0.846490939f, -0.532403128f,
   -0.845673247f, -0.533701002f,
   -0.844853565f, -0.534997620f,
   -0.844031895f, -0.536292979f,
   -0.843208240f, -0.537587076f,
   -0.842382600f, -0.538879909f,
   -0.841554977f, -0.540171473f,
   -0.840725375f, -0.541461766f,
   -0.839893794f, -0.542750785f,
   -0.839060237f, -0.544038527f,
   -0.838224706f, -0.545324988f,
   -0.837387202f, -0.546610167f,
   -0.836547727f, -0.547894059f,
   -0.835706284f, -0.549176662f,
   -0.834862875f, -0.550457973f,
   -0.834017501f, -0.551737988f,
   -0.833170165f, -0.553016706f,
   -0.832320868f, -0.554294121f,
   -0.831469612f, -0.555570233f,
   -0.830616400f, -0.556845037f,
   -0.829761234f, -0.558118531f,
   -0.828904115f, -0.559390712f,
   -0.828045045f, -0.560661576f,
   -0.827184027f, -0.561931121f,
   -0.826321063f, -0.563199344f,
   -0.825456154f, -0.564466242f,
   -0.824589303f, -0.565731811f,
   -0.823720511f, -0.566996049f,
   -0.822849781f, -0.568258953f,
   -0.821977115f, -0.569520519f,
   -0.821102515f, -0.570780746f,
   -0.820225983f, -0.572039629f,
   -0.819347520f, -0.573297167f,
   -0.818467130f, -0.574553355f,
   -0.817584813f, -0.575808191f,
   -0.816700573f, -0.577061673f,
   -0.815814411f, -0.578313796f,
   -0.814926329f, -0.579564559f,
   -0.814036330f, -0.580813958f,
   -0.813144415f, -0.582061990f,
   -0.812250587f, -0.583308653f,
   -0.811354847f, -0.584553943f,
   -0.810457198f, -0.585797857f,
   -0.809557642f, -0.587040394f,
   -0.808656182f, -0.588281548f,
   -0.807752818f, -0.589521319f,
   -0.806847554f, -0.590759702f,
   -0.805940391f, -0.591996695f,
   -0.805031331f, -0.593232295f,
   -0.804120377f, -0.594466499f,
   -0.803207531f, -0.595699304f,
   -0.802292796f, -0.596930708f,
   -0.801376172f, -0.598160707f,
   -0.800457662f, -0.599389298f,
   -0.799537269f, -0.600616479f,
   -0.798614995f, -0.601842247f,
   -0.797690841f, -0.603066599f,
   -0.796764810f, -0.604289531f,
   -0.795836905f, -0.605511041f,
   -0.794907126f, -0.606731127f,
   -0.793975478f, -0.607949785f,
   -0.793041960f, -0.609167012f,
   -0.792106577f, -0.610382806f,
   -0.791169330f, -0.611597164f,
   -0.790230221f, -0.612810082f,
   -0.789289253f, -0.614021559f,
   -0.788346428f, -0.615231591f,
   -0.787401747f, -0.616440175f,
   -0.786455214f, -0.617647308f,
   -0.785506830f, -0.618852988f,
   -0.784556597f, -0.620057212f,
   -0.783604519f, -0.621259977f,
   -0.782650596f, -0.622461279f,
   -0.781694832f, -0.623661118f,
   -0.780737229f, -0.624859488f,
   -0.779777788f, -0.626056388f,
   -0.778816512f, -0.627251815f,
   -0.777853404f, -0.628445767f,
   -0.776888466f, -0.629638239f,
   -0.775921699f, -0.630829230f,
   -0.774953107f, -0.632018736f,
   -0.773982691f, -0.633206755f,
   -0.773010453f, -0.634393284f,
   -0.772036397f, -0.635578320f,
   -0.771060524f, -0.636761861f,
   -0.770082837f, -0.637943904f,
   -0.769103338f, -0.639124445f,
   -0.768122029f, -0.640303482f,
   -0.767138912f, -0.641481013f,
   -0.766153990f, -0.642657034f,
   -0.765167266f, -0.643831543f,
   -0.764178741f, -0.645004537f,
   -0.763188417f, -0.646176013f,
   -0.762196298f, -0.647345969f,
   -0.761202385f, -0.648514401f,
   -0.760206682f, -0.649681307f,
   -0.759209189f, -0.650846685f,
   -0.758209910f, -0.652010531f,
   -0.757208847f, -0.653172843f,
   -0.756206001f, -0.654333618f,
   -0.755201377f, -0.655492853f,
   -0.754194975f, -0.656650546f,
   -0.753186799f, -0.657806693f,
   -0.752176850f, -0.658961293f,
   -0.751165132f, -0.660114342f,
   -0.750151646f, -0.661265838f,
   -0.749136395f, -0.662415778f,
   -0.748119380f, -0.663564159f,
   -0.747100606f, -0.664710978f,
   -0.746080074f, -0.665856234f,
   -0.745057785f, -0.666999922f,
   -0.744033744f, -0.668142041f,
   -0.743007952f, -0.669282588f,
   -0.741980412f, -0.670421560f,
   -0.740951125f, -0.671558955f,
   -0.739920095f, -0.672694769f,
   -0.738887324f, -0.673829000f,
   -0.737852815f, -0.674961646f,
   -0.736816569f, -0.676092704f,
   -0.735778589f, -0.677222170f,
   -0.734738878f, -0.678350043f,
   -0.733697438f, -0.679476320f,
   -0.732654272f, -0.680600998f,
   -0.731609381f, -0.681724074f,
   -0.730562769f, -0.682845546f,
   -0.729514438f, -0.683965412f,
   -0.728464390f, -0.685083668f,
   -0.727412629f, -0.686200312f,
   -0.726359155f, -0.687315341f,
   -0.725303972f, -0.688428753f,
   -0.724247083f, -0.689540545f,
   -0.723188489f, -0.690650714f,
   -0.722128194f, -0.691759258f,
   -0.721066199f, -0.692866175f,
   -0.720002508f, -0.693971461f,
   -0.718937122f, -0.695075114f,
   -0.717870045f, -0.696177131f,
   -0.716801279f, -0.697277511f,
   -0.715730825f, -0.698376249f,
   -0.714658688f, -0.699473345f,
   -0.713584869f, -0.700568794f,
   -0.712509371f, -0.701662595f,
   -0.711432196f, -0.702754744f,
   -0.710353347f, -0.703845241f,
   -0.709272826f, -0.704934080f,
   -0.708190637f, -0.706021261f,
   -0.707106781f, -0.707106781f,
   -0.706021261f, -0.708190637f,
   -0.704934080f, -0.709272826f,
   -0.703845241f, -0.710353347f,
   -0.702754744f, -0.711432196f,
   -0.701662595f, -0.712509371f,
   -0.700568794f, -0.713584869f,
   -0.699473345f, -0.714658688f,
   -0.698376249f, -0.715730825f,
   -0.697277511f, -0.716801279f,
   -0.696177131f, -0.717870045f,
   -0.695075114f, -0.718937122f,
   -0.693971461f, -0.720002508f,
   -0.692866175f, -0.721066199f,
   -0.691759258f, -0.722128194f,
   -0.690650714f, -0.723188489f,
   -0.689540545f, -0.724247083f,
   -0.688428753f, -0.725303972f,
   -0.687315341f, -0.726359155f,
   -0.686200312f, -0.727412629f,
   -0.685083668f, -0.728464390f,
   -0.683965412f, -0.729514438f,
   -0.682845546f, -0.730562769f,
   -0.681724074f, -0.731609381f,
   -0.680600998f, -0.732654272f,
   -0.679476320f, -0.733697438f,
   -0.678350043f, -0.734738878f,
   -0.677222170f, -0.735778589f,
   -0.676092704f, -0.736816569f,
   -0.674961646f, -0.737852815f,
   -0.673829000f, -0.738887324f,
   -0.672694769f, -0.739920095f,
   -0.671558955f, -0.740951125f,
   -0.670421560f, -0.741980412f,
   -0.669282588f, -0.743007952f,
   -0.668142041f, -0.744033744f,
   -0.666999922f, -0.745057785f,
   -0.665856234f, -0.746080074f,
   -0.664710978f, -0.747100606f,
   -0.663564159f, -0.748119380f,
   -0.662415778f, -0.749136395f,
   -0.661265838f, -0.750151646f,
   -0.660114342f, -0.751165132f,
   -0.658961293f, -0.752176850f,
   -0.657806693f, -0.753186799f,
   -0.656650546f, -0.754194975f,
   -0.655492853f, -0.755201377f,
   -0.654333618f, -0.756206001f,
   -0.653172843f, -0.757208847f,
   -0.652010531f, -0.758209910f,
   -0.650846685f, -0.759209189f,
   -0.649681307f, -0.760206682f,
   -0.648514401f, -0.761202385f,
   -0.647345969f, -0.762196298f,
   -0.646176013f, -0.763188417f,
   -0.645004537f, -0.764178741f,
   -0.643831543f, -0.765167266f,
   -0.642657034f, -0.766153990f,
   -0.641481013f, -0.767138912f,
   -0.640303482f, -0.768122029f,
   -0.639124445f, -0.769103338f,
   -0.637943904f, -0.770082837f,
   -0.636761861f, -0.771060524f,
   -0.635578320f, -0.772036397f,
   -0.634393284f, -0.773010453f,
   -0.633206755f, -0.773982691f,
   -0.632018736f, -0.774953107f,
   -0.630829230f, -0.775921699f,
   -0.629638239f, -0.776888466f,
   -0.628445767f, -0.777853404f,
   -0.627251815f, -0.778816512f,
   -0.626056388f, -0.779777788f,
   -0.624859488f, -0.780737229f,
   -0.623661118f, -0.781694832f,
   -0.622461279f, -0.782650596f,
   -0.621259977f, -0.783604519f,
   -0.620057212f, -0.784556597f,
   -0.618852988f, -0.785506830f,
   -0.617647308f, -0.786455214f,
   -0.616440175f, -0.787401747f,
   -0.615231591f, -0.788346428f,
   -0.614021559f, -0.789289253f,
   -0.612810082f, -0.790230221f,
   -0.611597164f, -0.791169330f,
   -0.610382806f, -0.792106577f,
   -0.609167012f, -0.793041960f,
   -0.607949785f, -0.793975478f,
   -0.606731127f, -0.794907126f,
   -0.605511041f, -0.795836905f,
   -0.604289531f, -0.796764810f,
   -0.603066599f, -0.797690841f,
   -0.601842247f, -0.798614995f,
   -0.600616479f, -0.799537269f,
   -0.599389298f, -0.800457662f,
   -0.598160707f, -0.801376172f,
   -0.596930708f, -0.802292796f,
   -0.595699304f, -0.803207531f,
   -0.594466499f, -0.804120377f,
   -0.593232295f, -0.805031331f,
   -0.591996695f, -0.805940391f,
   -0.590759702f, -0.806847554f,
   -0.589521319f, -0.807752818f,
   -0.588281548f, -0.808656182f,
   -0.587040394f, -0.809557642f,
   -0.585797857f, -0.810457198f,
   -0.584553943f, -0.811354847f,
   -0.583308653f, -0.812250587f,
   -0.582061990f, -0.813144415f,
   -0.580813958f, -0.814036330f,
   -0.579564559f, -0.814926329f,
   -0.578313796f, -0.815814411f,
   -0.577061673f, -0.816700573f,
   -0.575808191f, -0.817584813f,
   -0.574553355f, -0.818467130f,
   -0.573297167f, -0.819347520f,
   -0.572039629f, -0.820225983f,
   -0.570780746f, -0.821102515f,
   -0.569520519f, -0.821977115f,
   -0.568258953f, -0.822849781f,
   -0.566996049f, -0.823720511f,
   -0.565731811f, -0.824589303f,
   -0.564466242f, -0.825456154f,
   -0.563199344f, -0.826321063f,
   -0.561931121f, -0.827184027f,
   -0.560661576f, -0.828045045f,
   -0.559390712f, -0.828904115f,
   -0.558118531f, -0.829761234f,
   -0.556845037f, -0.830616400f,
   -0.555570233f, -0.831469612f,
   -0.554294121f, -0.832320868f,
   -0.553016706f, -0.833170165f,
   -0.551737988f, -0.834017501f,
   -0.550457973f, -0.834862875f,
   -0.549176662f, -0.835706284f,
   -0.547894059f, -0.836547727f,
   -0.546610167f, -0.837387202f,
   -0.545324988f, -0.838224706f,
   -0.544038527f, -0.839060237f,
   -0.542750785f, -0.839893794f,
   -0.541461766f, -0.840725375f,
   -0.540171473f, -0.841554977f,
   -0.538879909f, -0.842382600f,
   -0.537587076f, -0.843208240f,
   -0.536292979f, -0.844031895f,
   -0.534997620f, -0.844853565f,
   -0.533701002f, -0.845673247f,
   -0.532403128f, -0.846490939f,
   -0.531104001f, -0.847306639f,
   -0.529803625f, -0.848120345f,
   -0.528502002f, -0.848932055f,
   -0.527199135f, -0.849741768f,
   -0.525895027f, -0.850549481f,
   -0.524589683f, -0.851355193f,
   -0.523283103f, -0.852158902f,
   -0.521975293f, -0.852960605f,
   -0.520666254f, -0.853760301f,
   -0.519355990f, -0.854557988f,
   -0.518044504f, -0.855353665f,
   -0.516731799f, -0.856147328f,
   -0.515417878f, -0.856938977f,
   -0.514102744f, -0.857728610f,
   -0.512786401f, -0.858516224f,
   -0.511468850f, -0.859301818f,
   -0.510150097f, -0.860085390f,
   -0.508830143f, -0.860866939f,
   -0.507508991f, -0.861646461f,
   -0.506186645f, -0.862423956f,
   -0.504863109f, -0.863199422f,
   -0.503538384f, -0.863972856f,
   -0.502212474f, -0.864744258f,
   -0.500885383f, -0.865513624f,
   -0.499557113f, -0.866280954f,
   -0.498227667f, -0.867046246f,
   -0.496897049f, -0.867809497f,
   -0.495565262f, -0.868570706f,
   -0.494232309f, -0.869329871f,
   -0.492898192f, -0.870086991f,
   -0.491562916f, -0.870842063f,
   -0.490226483f, -0.871595087f,
   -0.488888897f, -0.872346059f,
   -0.487550160f, -0.873094978f,
   -0.486210276f, -0.873841843f,
   -0.484869248f, -0.874586652f,
   -0.483527079f, -0.875329403f,
   -0.482183772f, -0.876070094f,
   -0.480839331f, -0.876808724f,
   -0.479493758f, -0.877545290f,
   -0.478147056f, -0.878279792f,
   -0.476799230f, -0.879012226f,
   -0.475450282f, -0.879742593f,
   -0.474100215f, -0.880470889f,
   -0.472749032f, -0.881197113f,
   -0.471396737f, -0.881921264f,
   -0.470043332f, -0.882643340f,
   -0.468688822f, -0.883363339f,
   -0.467333209f, -0.884081259f,
   -0.465976496f, -0.884797098f,
   -0.464618686f, -0.885510856f,
   -0.463259784f, -0.886222530f,
   -0.461899791f, -0.886932119f,
   -0.460538711f, -0.887639620f,
   -0.459176548f, -0.888345033f,
   -0.457813304f, -0.889048356f,
   -0.456448982f, -0.889749586f,
   -0.455083587f, -0.890448723f,
   -0.453717121f, -0.891145765f,
   -0.452349587f, -0.891840709f,
   -0.450980989f, -0.892533555f,
   -0.449611330f, -0.893224301f,
   -0.448240612f, -0.893912945f,
   -0.446868840f, -0.894599486f,
   -0.445496017f, -0.895283921f,
   -0.444122145f, -0.895966250f,
   -0.442747228f, -0.896646470f,
   -0.441371269f, -0.897324581f,
   -0.439994271f, -0.898000580f,
   -0.438616239f, -0.898674466f,
   -0.437237174f, -0.899346237f,
   -0.435857080f, -0.900015892f,
   -0.434475961f, -0.900683429f,
   -0.433093819f, -0.901348847f,
   -0.431710658f, -0.902012144f,
   -0.430326481f, -0.902673318f,
   -0.428941292f, -0.903332368f,
   -0.427555093f, -0.903989293f,
   -0.426167889f, -0.904644091f,
   -0.424779681f, -0.905296759f,
   -0.423390474f, -0.905947298f,
   -0.422000271f, -0.906595705f,
   -0.420609074f, -0.907241978f,
   -0.419216888f, -0.907886116f,
   -0.417823716f, -0.908528119f,
   -0.416429560f, -0.909167983f,
   -0.415034424f, -0.909805708f,
   -0.413638312f, -0.910441292f,
   -0.412241227f, -0.911074734f,
   -0.410843171f, -0.911706032f,
   -0.409444149f, -0.912335185f,
   -0.408044163f, -0.912962190f,
   -0.406643217f, -0.913587048f,
   -0.405241314f, -0.914209756f,
   -0.403838458f, -0.914830312f,
   -0.402434651f, -0.915448716f,
   -0.401029897f, -0.916064966f,
   -0.399624200f, -0.916679060f,
   -0.398217562f, -0.917290997f,
   -0.396809987f, -0.917900776f,
   -0.395401479f, -0.918508394f,
   -0.393992040f, -0.919113852f,
   -0.392581674f, -0.919717146f,
   -0.391170384f, -0.920318277f,
   -0.389758174f, -0.920917242f,
   -0.388345047f, -0.921514039f,
   -0.386931006f, -0.922108669f,
   -0.385516054f, -0.922701128f,
   -0.384100195f, -0.923291417f,
   -0.382683432f, -0.923879533f,
   -0.381265769f, -0.924465474f,
   -0.379847209f, -0.925049241f,
   -0.378427755f, -0.925630831f,
   -0.377007410f, -0.926210242f,
   -0.375586178f, -0.926787474f,
   -0.374164063f, -0.927362526f,
   -0.372741067f, -0.927935395f,
   -0.371317194f, -0.928506080f,
   -0.369892447f, -0.929074581f,
   -0.368466830f, -0.929640896f,
   -0.367040346f, -0.930205023f,
   -0.365612998f, -0.930766961f,
   -0.364184790f, -0.931326709f,
   -0.362755724f, -0.931884266f,
   -0.361325806f, -0.932439629f,
   -0.359895037f, -0.932992799f,
   -0.358463421f, -0.933543773f,
   -0.357030961f, -0.934092550f,
   -0.355597662f, -0.934639130f,
   -0.354163525f, -0.935183510f,
   -0.352728556f, -0.935725689f,
   -0.351292756f, -0.936265667f,
   -0.349856130f, -0.936803442f,
   -0.348418680f, -0.937339012f,
   -0.346980411f, -0.937872376f,
   -0.345541325f, -0.938403534f,
   -0.344101426f, -0.938932484f,
   -0.342660717f, -0.939459224f,
   -0.341219202f, -0.939983753f,
   -0.339776884f, -0.940506071f,
   -0.338333767f, -0.941026175f,
   -0.336889853f, -0.941544065f,
   -0.335445147f, -0.942059740f,
   -0.333999651f, -0.942573198f,
   -0.332553370f, -0.943084437f,
   -0.331106306f, -0.943593458f,
   -0.329658463f, -0.944100258f,
   -0.328209844f, -0.944604837f,
   -0.326760452f, -0.945107193f,
   -0.325310292f, -0.945607325f,
   -0.323859367f, -0.946105232f,
   -0.322407679f, -0.946600913f,
   -0.320955232f, -0.947094366f,
   -0.319502031f, -0.947585591f,
   -0.318048077f, -0.948074586f,
   -0.316593376f, -0.948561350f,
   -0.315137929f, -0.949045882f,
   -0.313681740f, -0.949528181f,
   -0.312224814f, -0.950008245f,
   -0.310767153f, -0.950486074f,
   -0.309308760f, -0.950961666f,
   -0.307849640f, -0.951435021f,
   -0.306389795f, -0.951906137f,
   -0.304929230f, -0.952375013f,
   -0.303467947f, -0.952841648f,
   -0.302005949f, -0.953306040f,
   -0.300543241f, -0.953768190f,
   -0.299079826f, -0.954228095f,
   -0.297615707f, -0.954685755f,
   -0.296150888f, -0.955141168f,
   -0.294685372f, -0.955594334f,
   -0.293219163f, -0.956045251f,
   -0.291752263f, -0.956493919f,
   -0.290284677f, -0.956940336f,
   -0.288816408f, -0.957384501f,
   -0.287347460f, -0.957826413f,
   -0.285877835f, -0.958266071f,
   -0.284407537f, -0.958703475f,
   -0.282936570f, -0.959138622f,
   -0.281464938f, -0.959571513f,
   -0.279992643f, -0.960002146f,
   -0.278519689f, -0.960430519f,
   -0.277046080f, -0.960856633f,
   -0.275571819f, -0.961280486f,
   -0.274096910f, -0.961702077f,
   -0.272621355f, -0.962121404f,
   -0.271145160f, -0.962538468f,
   -0.269668326f, -0.962953267f,
   -0.268190857f, -0.963365800f,
   -0.266712757f, -0.963776066f,
   -0.265234030f, -0.964184064f,
   -0.263754679f, -0.964589793f,
   -0.262274707f, -0.964993253f,
   -0.260794118f, -0.965394442f,
   -0.259312915f, -0.965793359f,
   -0.257831102f, -0.966190003f,
   -0.256348682f, -0.966584374f,
   -0.254865660f, -0.966976471f,
   -0.253382037f, -0.967366292f,
   -0.251897818f, -0.967753837f,
   -0.250413007f, -0.968139105f,
   -0.248927606f, -0.968522094f,
   -0.247441619f, -0.968902805f,
   -0.245955050f, -0.969281235f,
   -0.244467903f, -0.969657385f,
   -0.242980180f, -0.970031253f,
   -0.241491885f, -0.970402839f,
   -0.240003022f, -0.970772141f,
   -0.238513595f, -0.971139158f,
   -0.237023606f, -0.971503891f,
   -0.235533059f, -0.971866337f,
   -0.234041959f, -0.972226497f,
   -0.232550307f, -0.972584369f,
   -0.231058108f, -0.972939952f,
   -0.229565366f, -0.973293246f,
   -0.228072083f, -0.973644250f,
   -0.226578264f, -0.973992962f,
   -0.225083911f, -0.974339383f,
   -0.223589029f, -0.974683511f,
   -0.222093621f, -0.975025345f,
   -0.220597690f, -0.975364885f,
   -0.219101240f, -0.975702130f,
   -0.217604275f, -0.976037079f,
   -0.216106797f, -0.976369731f,
   -0.214608811f, -0.976700086f,
   -0.213110320f, -0.977028143f,
   -0.211611327f, -0.977353900f,
   -0.210111837f, -0.977677358f,
   -0.208611852f, -0.977998515f,
   -0.207111376f, -0.978317371f,
   -0.205610413f, -0.978633924f,
   -0.204108966f, -0.978948175f,
   -0.202607039f, -0.979260123f,
   -0.201104635f, -0.979569766f,
   -0.199601758f, -0.979877104f,
   -0.198098411f, -0.980182136f,
   -0.196594598f, -0.980484862f,
   -0.195090322f, -0.980785280f,
   -0.193585587f, -0.981083391f,
   -0.192080397f, -0.981379193f,
   -0.190574755f, -0.981672686f,
   -0.189068664f, -0.981963869f,
   -0.187562129f, -0.982252741f,
   -0.186055152f, -0.982539302f,
   -0.184547737f, -0.982823551f,
   -0.183039888f, -0.983105487f,
   -0.181531608f, -0.983385110f,
   -0.180022901f, -0.983662419f,
   -0.178513771f, -0.983937413f,
   -0.177004220f, -0.984210092f,
   -0.175494253f, -0.984480455f,
   -0.173983873f, -0.984748502f,
   -0.172473084f, -0.985014231f,
   -0.170961889f, -0.985277642f,
   -0.169450291f, -0.985538735f,
   -0.167938295f, -0.985797509f,
   -0.166425904f, -0.986053963f,
   -0.164913120f, -0.986308097f,
   -0.163399949f, -0.986559910f,
   -0.161886394f, -0.986809402f,
   -0.160372457f, -0.987056571f,
   -0.158858143f, -0.987301418f,
   -0.157343456f, -0.987543942f,
   -0.155828398f, -0.987784142f,
   -0.154312973f, -0.988022017f,
   -0.152797185f, -0.988257568f,
   -0.151281038f, -0.988490793f,
   -0.149764535f, -0.988721692f,
   -0.148247679f, -0.988950265f,
   -0.146730474f, -0.989176510f,
   -0.145212925f, -0.989400428f,
   -0.143695033f, -0.989622017f,
   -0.142176804f, -0.989841278f,
   -0.140658239f, -0.990058210f,
   -0.139139344f, -0.990272812f,
   -0.137620122f, -0.990485084f,
   -0.136100575f, -0.990695025f,
   -0.134580709f, -0.990902635f,
   -0.133060525f, -0.991107914f,
   -0.131540029f, -0.991310860f,
   -0.130019223f, -0.991511473f,
   -0.128498111f, -0.991709754f,
   -0.126976696f, -0.991905700f,
   -0.125454983f, -0.992099313f,
   -0.123932975f, -0.992290591f,
   -0.122410675f, -0.992479535f,
   -0.120888087f, -0.992666142f,
   -0.119365215f, -0.992850414f,
   -0.117842062f, -0.993032350f,
   -0.116318631f, -0.993211949f,
   -0.114794927f, -0.993389211f,
   -0.113270952f, -0.993564136f,
   -0.111746711f, -0.993736722f,
   -0.110222207f, -0.993906970f,
   -0.108697444f, -0.994074879f,
   -0.107172425f, -0.994240449f,
   -0.105647154f, -0.994403680f,
   -0.104121634f, -0.994564571f,
   -0.102595869f, -0.994723121f,
   -0.101069863f, -0.994879331f,
   -0.099543619f, -0.995033199f,
   -0.098017140f, -0.995184727f,
   -0.096490431f, -0.995333912f,
   -0.094963495f, -0.995480755f,
   -0.093436336f, -0.995625256f,
   -0.091908956f, -0.995767414f,
   -0.090381361f, -0.995907229f,
   -0.088853553f, -0.996044701f,
   -0.087325535f, -0.996179829f,
   -0.085797312f, -0.996312612f,
   -0.084268888f, -0.996443051f,
   -0.082740265f, -0.996571146f,
   -0.081211447f, -0.996696895f,
   -0.079682438f, -0.996820299f,
   -0.078153242f, -0.996941358f,
   -0.076623861f, -0.997060070f,
   -0.075094301f, -0.997176437f,
   -0.073564564f, -0.997290457f,
   -0.072034653f, -0.997402130f,
   -0.070504573f, -0.997511456f,
   -0.068974328f, -0.997618435f,
   -0.067443920f, -0.997723067f,
   -0.065913353f, -0.997825350f,
   -0.064382631f, -0.997925286f,
   -0.062851758f, -0.998022874f,
   -0.061320736f, -0.998118113f,
   -0.059789571f, -0.998211003f,
   -0.058258265f, -0.998301545f,
   -0.056726821f, -0.998389737f,
   -0.055195244f, -0.998475581f,
   -0.053663538f, -0.998559074f,
   -0.052131705f, -0.998640218f,
   -0.050599749f, -0.998719012f,
   -0.049067674f, -0.998795456f,
   -0.047535484f, -0.998869550f,
   -0.046003182f, -0.998941293f,
   -0.044470772f, -0.999010686f,
   -0.042938257f, -0.999077728f,
   -0.041405641f, -0.999142419f,
   -0.039872928f, -0.999204759f,
   -0.038340120f, -0.999264747f,
   -0.036807223f, -0.999322385f,
   -0.035274239f, -0.999377670f,
   -0.033741172f, -0.999430605f,
   -0.032208025f, -0.999481187f,
   -0.030674803f, -0.999529418f,
   -0.029141509f, -0.999575296f,
   -0.027608146f, -0.999618822f,
   -0.026074718f, -0.999659997f,
   -0.024541229f, -0.999698819f,
   -0.023007681f, -0.999735288f,
   -0.021474080f, -0.999769405f,
   -0.019940429f, -0.999801170f,
   -0.018406730f, -0.999830582f,
   -0.016872988f, -0.999857641f,
   -0.015339206f, -0.999882347f,
   -0.013805389f, -0.999904701f,
   -0.012271538f, -0.999924702f,
   -0.010737659f, -0.999942350f,
   -0.009203755f, -0.999957645f,
   -0.007669829f, -0.999970586f,
   -0.006135885f, -0.999981175f,
   -0.004601926f, -0.999989411f,
   -0.003067957f, -0.999995294f,
   -0.001533980f, -0.999998823f,
   -0.000000000f, -1.000000000f,
    0.001533980f, -0.999998823f,
    0.003067957f, -0.999995294f,
    0.004601926f, -0.999989411f,
    0.006135885f, -0.999981175f,
    0.007669829f, -0.999970586f,
    0.009203755f, -0.999957645f,
    0.010737659f, -0.999942350f,
    0.012271538f, -0.999924702f,
    0.013805389f, -0.999904701f,
    0.015339206f, -0.999882347f,
    0.016872988f, -0.999857641f,
    0.018406730f, -0.999830582f,
    0.019940429f, -0.999801170f,
    0.021474080f, -0.999769405f,
    0.023007681f, -0.999735288f,
    0.024541229f, -0.999698819f,
    0.026074718f, -0.999659997f,
    0.027608146f, -0.999618822f,
    0.029141509f, -0.999575296f,
    0.030674803f, -0.999529418f,
    0.032208025f, -0.999481187f,
    0.033741172f, -0.999430605f,
    0.035274239f, -0.999377670f,
    0.036807223f, -0.999322385f,
    0.038340120f, -0.999264747f,
    0.039872928f, -0.999204759f,
    0.041405641f, -0.999142419f,
    0.042938257f, -0.999077728f,
    0.044470772f, -0.999010686f,
    0.046003182f, -0.998941293f,
    0.047535484f, -0.998869550f,
    0.049067674f, -0.998795456f,
    0.050599749f, -0.998719012f,
    0.052131705f, -0.998640218f,
    0.053663538f, -0.998559074f,
    0.055195244f, -0.998475581f,
    0.056726821f, -0.998389737f,
    0.058258265f, -0.998301545f,
    0.059789571f, -0.998211003f,
    0.061320736f, -0.998118113f,
    0.062851758f, -0.998022874f,
    0.064382631f, -0.997925286f,
    0.065913353f, -0.997825350f,
    0.067443920f, -0.997723067f,
    0.068974328f, -0.997618435f,
    0.070504573f, -0.997511456f,
    0.072034653f, -0.997402130f,
    0.073564564f, -0.997290457f,
    0.075094301f, -0.997176437f,
    0.076623861f, -0.997060070f,
    0.078153242f, -0.996941358f,
    0.079682438f, -0.996820299f,
    0.081211447f, -0.996696895f,
    0.082740265f, -0.996571146f,
    0.084268888f, -0.996443051f,
    0.085797312f, -0.996312612f,
    0.087325535f, -0.996179829f,
    0.088853553f, -0.996044701f,
    0.090381361f, -0.995907229f,
    0.091908956f, -0.995767414f,
    0.093436336f, -0.995625256f,
    0.094963495f, -0.995480755f,
    0.096490431f, -0.995333912f,
    0.098017140f, -0.995184727f,
    0.099543619f, -0.995033199f,
    0.101069863f, -0.994879331f,
    0.102595869f, -0.994723121f,
    0.104121634f, -0.994564571f,
    0.105647154f, -0.994403680f,
    0.107172425f, -0.994240449f,
    0.108697444f, -0.994074879f,
    0.110222207f, -0.993906970f,
    0.111746711f, -0.993736722f,
    0.113270952f, -0.993564136f,
    0.114794927f, -0.993389211f,
    0.116318631f, -0.993211949f,
    0.117842062f, -0.993032350f,
    0.119365215f, -0.992850414f,
    0.120888087f, -0.992666142f,
    0.122410675f, -0.992479535f,
    0.123932975f, -0.992290591f,
    0.125454983f, -0.992099313f,
    0.126976696f, -0.991905700f,
    0.128498111f, -0.991709754f,
    0.130019223f, -0.991511473f,
    0.131540029f, -0.991310860f,
    0.133060525f, -0.991107914f,
    0.134580709f, -0.990902635f,
    0.136100575f, -0.990695025f,
    0.137620122f, -0.990485084f,
    0.139139344f, -0.990272812f,
    0.140658239f, -0.990058210f,
    0.142176804f, -0.989841278f,
    0.143695033f, -0.989622017f,
    0.145212925f, -0.989400428f,
    0.146730474f, -0.989176510f,
    0.148247679f, -0.988950265f,
    0.149764535f, -0.988721692f,
    0.151281038f, -0.988490793f,
    0.152797185f, -0.988257568f,
    0.154312973f, -0.988022017f,
    0.155828398f, -0.987784142f,
    0.157343456f, -0.987543942f,
    0.158858143f, -0.987301418f,
    0.160372457f, -0.987056571f,
    0.161886394f, -0.986809402f,
    0.163399949f, -0.986559910f,
    0.164913120f, -0.986308097f,
    0.166425904f, -0.986053963f,
    0.167938295f, -0.985797509f,
    0.169450291f, -0.985538735f,
    0.170961889f, -0.985277642f,
    0.172473084f, -0.985014231f,
    0.173983873f, -0.984748502f,
    0.175494253f, -0.984480455f,
    0.177004220f, -0.984210092f,
    0.178513771f, -0.983937413f,
    0.180022901f, -0.983662419f,
    0.181531608f, -0.983385110f,
    0.183039888f, -0.983105487f,
    0.184547737f, -0.982823551f,
    0.186055152f, -0.982539302f,
    0.187562129f, -0.982252741f,
    0.189068664f, -0.981963869f,
    0.190574755f, -0.981672686f,
    0.192080397f, -0.981379193f,
    0.193585587f, -0.981083391f,
    0.195090322f, -0.980785280f,
    0.196594598f, -0.980484862f,
    0.198098411f, -0.980182136f,
    0.199601758f, -0.979877104f,
    0.201104635f, -0.979569766f,
    0.202607039f, -0.979260123f,
    0.204108966f, -0.978948175f,
    0.205610413f, -0.978633924f,
    0.207111376f, -0.978317371f,
    0.208611852f, -0.977998515f,
    0.210111837f, -0.977677358f,
    0.211611327f, -0.977353900f,
    0.213110320f, -0.977028143f,
    0.214608811f, -0.976700086f,
    0.216106797f, -0.976369731f,
    0.217604275f, -0.976037079f,
    0.219101240f, -0.975702130f,
    0.220597690f, -0.975364885f,
    0.222093621f, -0.975025345f,
    0.223589029f, -0.974683511f,
    0.225083911f, -0.974339383f,
    0.226578264f, -0.973992962f,
    0.228072083f, -0.973644250f,
    0.229565366f, -0.973293246f,
    0.231058108f, -0.972939952f,
    0.232550307f, -0.972584369f,
    0.234041959f, -0.972226497f,
    0.235533059f, -0.971866337f,
    0.237023606f, -0.971503891f,
    0.238513595f, -0.971139158f,
    0.240003022f, -0.970772141f,
    0.241491885f, -0.970402839f,
    0.242980180f, -0.970031253f,
    0.244467903f, -0.969657385f,
    0.245955050f, -0.969281235f,
    0.247441619f, -0.968902805f,
    0.248927606f, -0.968522094f,
    0.250413007f, -0.968139105f,
    0.251897818f, -0.967753837f,
    0.253382037f, -0.967366292f,
    0.254865660f, -0.966976471f,
    0.256348682f, -0.966584374f,
    0.257831102f, -0.966190003f,
    0.259312915f, -0.965793359f,
    0.260794118f, -0.965394442f,
    0.262274707f, -0.964993253f,
    0.263754679f, -0.964589793f,
    0.265234030f, -0.964184064f,
    0.266712757f, -0.963776066f,
    0.268190857f, -0.963365800f,
    0.269668326f, -0.962953267f,
    0.271145160f, -0.962538468f,
    0.272621355f, -0.962121404f,
    0.274096910f, -0.961702077f,
    0.275571819f, -0.961280486f,
    0.277046080f, -0.960856633f,
    0.278519689f, -0.960430519f,
    0.279992643f, -0.960002146f,
    0.281464938f, -0.959571513f,
    0.282936570f, -0.959138622f,
    0.284407537f, -0.958703475f,
    0.285877835f, -0.958266071f,
    0.287347460f, -0.957826413f,
    0.288816408f, -0.957384501f,
    0.290284677f, -0.956940336f,
    0.291752263f, -0.956493919f,
    0.293219163f, -0.956045251f,
    0.294685372f, -0.955594334f,
    0.296150888f, -0.955141168f,
    0.297615707f, -0.954685755f,
    0.299079826f, -0.954228095f,
    0.300543241f, -0.953768190f,
    0.302005949f, -0.953306040f,
    0.303467947f, -0.952841648f,
    0.304929230f, -0.952375013f,
    0.306389795f, -0.951906137f,
    0.307849640f, -0.951435021f,
    0.309308760f, -0.950961666f,
    0.310767153f, -0.950486074f,
    0.312224814f, -0.950008245f,
    0.313681740f, -0.949528181f,
    0.315137929f, -0.949045882f,
    0.316593376f, -0.948561350f,
    0.318048077f, -0.948074586f,
    0.319502031f, -0.947585591f,
    0.320955232f, -0.947094366f,
    0.322407679f, -0.946600913f,
    0.323859367f, -0.946105232f,
    0.325310292f, -0.945607325f,
    0.326760452f, -0.945107193f,
    0.328209844f, -0.944604837f,
    0.329658463f, -0.944100258f,
    0.331106306f, -0.943593458f,
    0.332553370f, -0.943084437f,
    0.333999651f, -0.942573198f,
    0.335445147f, -0.942059740f,
    0.336889853f, -0.941544065f,
    0.338333767f, -0.941026175f,
    0.339776884f, -0.940506071f,
    0.341219202f, -0.939983753f,
    0.342660717f, -0.939459224f,
    0.344101426f, -0.938932484f,
    0.345541325f, -0.938403534f,
    0.346980411f, -0.937872376f,
    0.348418680f, -0.937339012f,
    0.349856130f, -0.936803442f,
    0.351292756f, -0.936265667f,
    0.352728556f, -0.935725689f,
    0.354163525f, -0.935183510f,
    0.355597662f, -0.934639130f,
    0.357030961f, -0.934092550f,
    0.358463421f, -0.933543773f,
    0.359895037f, -0.932992799f,
    0.361325806f, -0.932439629f,
    0.362755724f, -0.931884266f,
    0.364184790f, -0.931326709f,
    0.365612998f, -0.930766961f,
    0.367040346f, -0.930205023f,
    0.368466830f, -0.929640896f,
    0.369892447f, -0.929074581f,
    0.371317194f, -0.928506080f,
    0.372741067f, -0.927935395f,
    0.374164063f, -0.927362526f,
    0.375586178f, -0.926787474f,
    0.377007410f, -0.926210242f,
    0.378427755f, -0.925630831f,
    0.379847209f, -0.925049241f,
    0.381265769f, -0.924465474f,
    0.382683432f, -0.923879533f,
    0.384100195f, -0.923291417f,
    0.385516054f, -0.922701128f,
    0.386931006f, -0.922108669f,
    0.388345047f, -0.921514039f,
    0.389758174f, -0.920917242f,
    0.391170384f, -0.920318277f,
    0.392581674f, -0.919717146f,
    0.393992040f, -0.919113852f,
    0.395401479f, -0.918508394f,
    0.396809987f, -0.917900776f,
    0.398217562f, -0.917290997f,
    0.399624200f, -0.916679060f,
    0.401029897f, -0.916064966f,
    0.402434651f, -0.915448716f,
    0.403838458f, -0.914830312f,
    0.405241314f, -0.914209756f,
    0.406643217f, -0.913587048f,
    0.408044163f, -0.912962190f,
    0.409444149f, -0.912335185f,
    0.410843171f, -0.911706032f,
    0.412241227f, -0.911074734f,
    0.413638312f, -0.910441292f,
    0.415034424f, -0.909805708f,
    0.416429560f, -0.909167983f,
    0.417823716f, -0.908528119f,
    0.419216888f, -0.907886116f,
    0.420609074f, -0.907241978f,
    0.422000271f, -0.906595705f,
    0.423390474f, -0.905947298f,
    0.424779681f, -0.905296759f,
    0.426167889f, -0.904644091f,
    0.427555093f, -0.903989293f,
    0.428941292f, -0.903332368f,
    0.430326481f, -0.902673318f,
    0.431710658f, -0.902012144f,
    0.433093819f, -0.901348847f,
    0.434475961f, -0.900683429f,
    0.435857080f, -0.900015892f,
    0.437237174f, -0.899346237f,
    0.438616239f, -0.898674466f,
    0.439994271f, -0.898000580f,
    0.441371269f, -0.897324581f,
    0.442747228f, -0.896646470f,
    0.444122145f, -0.895966250f,
    0.445496017f, -0.895283921f,
    0.446868840f, -0.894599486f,
    0.448240612f, -0.893912945f,
    0.449611330f, -0.893224301f,
    0.450980989f, -0.892533555f,
    0.452349587f, -0.891840709f,
    0.453717121f, -0.891145765f,
    0.455083587f, -0.890448723f,
    0.456448982f, -0.889749586f,
    0.457813304f, -0.889048356f,
    0.459176548f, -0.888345033f,
    0.460538711f, -0.887639620f,
    0.461899791f, -0.886932119f,
    0.463259784f, -0.886222530f,
    0.464618686f, -0.885510856f,
    0.465976496f, -0.884797098f,
    0.467333209f, -0.884081259f,
    0.468688822f, -0.883363339f,
    0.470043332f, -0.882643340f,
    0.471396737f, -0.881921264f,
    0.472749032f, -0.881197113f,
    0.474100215f, -0.880470889f,
    0.475450282f, -0.879742593f,
    0.476799230f, -0.879012226f,
    0.478147056f, -0.878279792f,
    0.479493758f, -0.877545290f,
    0.480839331f, -0.876808724f,
    0.482183772f, -0.876070094f,
    0.483527079f, -0.875329403f,
    0.484869248f, -0.874586652f,
    0.486210276f, -0.873841843f,
    0.487550160f, -0.873094978f,
    0.488888897f, -0.872346059f,
    0.490226483f, -0.871595087f,
    0.491562916f, -0.870842063f,
    0.492898192f, -0.870086991f,
    0.494232309f, -0.869329871f,
    0.495565262f, -0.868570706f,
    0.496897049f, -0.867809497f,
    0.498227667f, -0.867046246f,
    0.499557113f, -0.866280954f,
    0.500885383f, -0.865513624f,
    0.502212474f, -0.864744258f,
    0.503538384f, -0.863972856f,
    0.504863109f, -0.863199422f,
    0.506186645f, -0.862423956f,
    0.507508991f, -0.861646461f,
    0.508830143f, -0.860866939f,
    0.510150097f, -0.860085390f,
    0.511468850f, -0.859301818f,
    0.512786401f, -0.858516224f,
    0.514102744f, -0.857728610f,
    0.515417878f, -0.856938977f,
    0.516731799f, -0.856147328f,
    0.518044504f, -0.855353665f,
    0.519355990f, -0.854557988f,
    0.520666254f, -0.853760301f,
    0.521975293f, -0.852960605f,
    0.523283103f, -0.852158902f,
    0.524589683f, -0.851355193f,
    0.525895027f, -0.850549481f,
    0.527199135f, -0.849741768f,
    0.528502002f, -0.848932055f,
    0.529803625f, -0.848120345f,
    0.531104001f, -0.847306639f,
    0.532403128f, -0.846490939f,
    0.533701002f, -0.845673247f,
    0.534997620f, -0.844853565f,
    0.536292979f, -0.844031895f,
    0.537587076f, -0.843208240f,
    0.538879909f, -0.842382600f,
    0.540171473f, -0.841554977f,
    0.541461766f, -0.840725375f,
    0.542750785f, -0.839893794f,
    0.544038527f, -0.839060237f,
    0.545324988f, -0.838224706f,
    0.546610167f, -0.837387202f,
    0.547894059f, -0.836547727f,
    0.549176662f, -0.835706284f,
    0.550457973f, -0.834862875f,
    0.551737988f, -0.834017501f,
    0.553016706f, -0.833170165f,
    0.554294121f, -0.832320868f,
    0.555570233f, -0.831469612f,
    0.556845037f, -0.830616400f,
    0.558118531f, -0.829761234f,
    0.559390712f, -0.828904115f,
    0.560661576f, -0.828045045f,
    0.561931121f, -0.827184027f,
    0.563199344f, -0.826321063f,
    0.564466242f, -0.825456154f,
    0.565731811f, -0.824589303f,
    0.566996049f, -0.823720511f,
    0.568258953f, -0.822849781f,
    0.569520519f, -0.821977115f,
    0.570780746f, -0.821102515f,
    0.572039629f, -0.820225983f,
    0.573297167f, -0.819347520f,
    0.574553355f, -0.818467130f,
    0.575808191f, -0.817584813f,
    0.577061673f, -0.816700573f,
    0.578313796f, -0.815814411f,
    0.579564559f, -0.814926329f,
    0.580813958f, -0.814036330f,
    0.582061990f, -0.813144415f,
    0.583308653f, -0.812250587f,
    0.584553943f, -0.811354847f,
    0.585797857f, -0.810457198f,
    0.587040394f, -0.809557642f,
    0.588281548f, -0.808656182f,
    0.589521319f, -0.807752818f,
    0.590759702f, -0.806847554f,
    0.591996695f, -0.805940391f,
    0.593232295f, -0.805031331f,
    0.594466499f, -0.804120377f,
    0.595699304f, -0.803207531f,
    0.596930708f, -0.802292796f,
    0.598160707f, -0.801376172f,
    0.599389298f, -0.800457662f,
    0.600616479f, -0.799537269f,
    0.601842247f, -0.798614995f,
    0.603066599f, -0.797690841f,
    0.604289531f, -0.796764810f,
    0.605511041f, -0.795836905f,
    0.606731127f, -0.794907126f,
    0.607949785f, -0.793975478f,
    0.609167012f, -0.793041960f,
    0.610382806f, -0.792106577f,
    0.611597164f, -0.791169330f,
    0.612810082f, -0.790230221f,
    0.614021559f, -0.789289253f,
    0.615231591f, -0.788346428f,
    0.616440175f, -0.787401747f,
    0.617647308f, -0.786455214f,
    0.618852988f, -0.785506830f,
    0.620057212f, -0.784556597f,
    0.621259977f, -0.783604519f,
    0.622461279f, -0.782650596f,
    0.623661118f, -0.781694832f,
    0.624859488f, -0.780737229f,
    0.626056388f, -0.779777788f,
    0.627251815f, -0.778816512f,
    0.628445767f, -0.777853404f,
    0.629638239f, -0.776888466f,
    0.630829230f, -0.775921699f,
    0.632018736f, -0.774953107f,
    0.633206755f, -0.773982691f,
    0.634393284f, -0.773010453f,
    0.635578320f, -0.772036397f,
    0.636761861f, -0.771060524f,
    0.637943904f, -0.770082837f,
    0.639124445f, -0.769103338f,
    0.640303482f, -0.768122029f,
    0.641481013f, -0.767138912f,
    0.642657034f, -0.766153990f,
    0.643831543f, -0.765167266f,
    0.645004537f, -0.764178741f,
    0.646176013f, -0.763188417f,
    0.647345969f, -0.762196298f,
    0.648514401f, -0.761202385f,
    0.649681307f, -0.760206682f,
    0.650846685f, -0.759209189f,
    0.652010531f, -0.758209910f,
    0.653172843f, -0.757208847f,
    0.654333618f, -0.756206001f,
    0.655492853f, -0.755201377f,
    0.656650546f, -0.754194975f,
    0.657806693f, -0.753186799f,
    0.658961293f, -0.752176850f,
    0.660114342f, -0.751165132f,
    0.661265838f, -0.750151646f,
    0.662415778f, -0.749136395f,
    0.663564159f, -0.748119380f,
    0.664710978f, -0.747100606f,
    0.665856234f, -0.746080074f,
    0.666999922f, -0.745057785f,
    0.668142041f, -0.744033744f,
    0.669282588f, -0.743007952f,
    0.670421560f, -0.741980412f,
    0.671558955f, -0.740951125f,
    0.672694769f, -0.739920095f,
    0.673829000f, -0.738887324f,
    0.674961646f, -0.737852815f,
    0.676092704f, -0.736816569f,
    0.677222170f, -0.735778589f,
    0.678350043f, -0.734738878f,
    0.679476320f, -0.733697438f,
    0.680600998f, -0.732654272f,
    0.681724074f, -0.731609381f,
    0.682845546f, -0.730562769f,
    0.683965412f, -0.729514438f,
    0.685083668f, -0.728464390f,
    0.686200312f, -0.727412629f,
    0.687315341f, -0.726359155f,
    0.688428753f, -0.725303972f,
    0.689540545f, -0.724247083f,
    0.690650714f, -0.723188489f,
    0.691759258f, -0.722128194f,
    0.692866175f, -0.721066199f,
    0.693971461f, -0.720002508f,
    0.695075114f, -0.718937122f,
    0.696177131f, -0.717870045f,
    0.697277511f, -0.716801279f,
    0.698376249f, -0.715730825f,
    0.699473345f, -0.714658688f,
    0.700568794f, -0.713584869f,
    0.701662595f, -0.712509371f,
    0.702754744f, -0.711432196f,
    0.703845241f, -0.710353347f,
    0.704934080f, -0.709272826f,
    0.706021261f, -0.708190637f,
    0.707106781f, -0.707106781f,
    0.708190637f, -0.706021261f,
    0.709272826f, -0.704934080f,
    0.710353347f, -0.703845241f,
    0.711432196f, -0.702754744f,
    0.712509371f, -0.701662595f,
    0.713584869f, -0.700568794f,
    0.714658688f, -0.699473345f,
    0.715730825f, -0.698376249f,
    0.716801279f, -0.697277511f,
    0.717870045f, -0.696177131f,
    0.718937122f, -0.695075114f,
    0.720002508f, -0.693971461f,
    0.721066199f, -0.692866175f,
    0.722128194f, -0.691759258f,
    0.723188489f, -0.690650714f,
    0.724247083f, -0.689540545f,
    0.725303972f, -0.688428753f,
    0.726359155f, -0.687315341f,
    0.727412629f, -0.686200312f,
    0.728464390f, -0.685083668f,
    0.729514438f, -0.683965412f,
    0.730562769f, -0.682845546f,
    0.731609381f, -0.681724074f,
    0.732654272f, -0.680600998f,
    0.733697438f, -0.679476320f,
    0.734738878f, -0.678350043f,
    0.735778589f, -0.677222170f,
    0.736816569f, -0.676092704f,
    0.737852815f, -0.674961646f,
    0.738887324f, -0.673829000f,
    0.739920095f, -0.672694769f,
    0.740951125f, -0.671558955f,
    0.741980412f, -0.670421560f,
    0.743007952f, -0.669282588f,
    0.744033744f, -0.668142041f,
    0.745057785f, -0.666999922f,
    0.746080074f, -0.665856234f,
    0.747100606f, -0.664710978f,
    0.748119380f, -0.663564159f,
    0.749136395f, -0.662415778f,
    0.750151646f, -0.661265838f,
    0.751165132f, -0.660114342f,
    0.752176850f, -0.658961293f,
    0.753186799f, -0.657806693f,
    0.754194975f, -0.656650546f,
    0.755201377f, -0.655492853f,
    0.756206001f, -0.654333618f,
    0.757208847f, -0.653172843f,
    0.758209910f, -0.652010531f,
    0.759209189f, -0.650846685f,
    0.760206682f, -0.649681307f,
    0.761202385f, -0.648514401f,
    0.762196298f, -0.647345969f,
    0.763188417f, -0.646176013f,
    0.764178741f, -0.645004537f,
    0.765167266f, -0.643831543f,
    0.766153990f, -0.642657034f,
    0.767138912f, -0.641481013f,
    0.768122029f, -0.640303482f,
    0.769103338f, -0.639124445f,
    0.770082837f, -0.637943904f,
    0.771060524f, -0.636761861f,
    0.772036397f, -0.635578320f,
    0.773010453f, -0.634393284f,
    0.773982691f, -0.633206755f,
    0.774953107f, -0.632018736f,
    0.775921699f, -0.630829230f,
    0.776888466f, -0.629638239f,
    0.777853404f, -0.628445767f,
    0.778816512f, -0.627251815f,
    0.779777788f, -0.626056388f,
    0.780737229f, -0.624859488f,
    0.781694832f, -0.623661118f,
    0.782650596f, -0.622461279f,
    0.783604519f, -0.621259977f,
    0.784556597f, -0.620057212f,
    0.785506830f, -0.618852988f,
    0.786455214f, -0.617647308f,
    0.787401747f, -0.616440175f,
    0.788346428f, -0.615231591f,
    0.789289253f, -0.614021559f,
    0.790230221f, -0.612810082f,
    0.791169330f, -0.611597164f,
    0.792106577f, -0.610382806f,
    0.793041960f, -0.609167012f,
    0.793975478f, -0.607949785f,
    0.794907126f, -0.606731127f,
    0.795836905f, -0.605511041f,
    0.796764810f, -0.604289531f,
    0.797690841f, -0.603066599f,
    0.798614995f, -0.601842247f,
    0.799537269f, -0.600616479f,
    0.800457662f, -0.599389298f,
    0.801376172f, -0.598160707f,
    0.802292796f, -0.596930708f,
    0.803207531f, -0.595699304f,
    0.804120377f, -0.594466499f,
    0.805031331f, -0.593232295f,
    0.805940391f, -0.591996695f,
    0.806847554f, -0.590759702f,
    0.807752818f, -0.589521319f,
    0.808656182f, -0.588281548f,
    0.809557642f, -0.587040394f,
    0.810457198f, -0.585797857f,
    0.811354847f, -0.584553943f,
    0.812250587f, -0.583308653f,
    0.813144415f, -0.582061990f,
    0.814036330f, -0.580813958f,
    0.814926329f, -0.579564559f,
    0.815814411f, -0.578313796f,
    0.816700573f, -0.577061673f,
    0.817584813f, -0.575808191f,
    0.818467130f, -0.574553355f,
    0.819347520f, -0.573297167f,
    0.820225983f, -0.572039629f,
    0.821102515f, -0.570780746f,
    0.821977115f, -0.569520519f,
    0.822849781f, -0.568258953f,
    0.823720511f, -0.566996049f,
    0.824589303f, -0.565731811f,
    0.825456154f, -0.564466242f,
    0.826321063f, -0.563199344f,
    0.827184027f, -0.561931121f,
    0.828045045f, -0.560661576f,
    0.828904115f, -0.559390712f,
    0.829761234f, -0.558118531f,
    0.830616400f, -0.556845037f,
    0.831469612f, -0.555570233f,
    0.832320868f, -0.554294121f,
    0.833170165f, -0.553016706f,
    0.834017501f, -0.551737988f,
    0.834862875f, -0.550457973f,
    0.835706284f, -0.549176662f,
    0.836547727f, -0.547894059f,
    0.837387202f, -0.546610167f,
    0.838224706f, -0.545324988f,
    0.839060237f, -0.544038527f,
    0.839893794f, -0.542750785f,
    0.840725375f, -0.541461766f,
    0.841554977f, -0.540171473f,
    0.842382600f, -0.538879909f,
    0.843208240f, -0.537587076f,
    0.844031895f, -0.536292979f,
    0.844853565f, -0.534997620f,
    0.845673247f, -0.533701002f,
    0.846490939f, -0.532403128f,
    0.847306639f, -0.531104001f,
    0.848120345f, -0.529803625f,
    0.848932055f, -0.528502002f,
    0.849741768f, -0.527199135f,
    0.850549481f, -0.525895027f,
    0.851355193f, -0.524589683f,
    0.852158902f, -0.523283103f,
    0.852960605f, -0.521975293f,
    0.853760301f, -0.520666254f,
    0.854557988f, -0.519355990f,
    0.855353665f, -0.518044504f,
    0.856147328f, -0.516731799f,
    0.856938977f, -0.515417878f,
    0.857728610f, -0.514102744f,
    0.858516224f, -0.512786401f,
    0.859301818f, -0.511468850f,
    0.860085390f, -0.510150097f,
    0.860866939f, -0.508830143f,
    0.861646461f, -0.507508991f,
    0.862423956f, -0.506186645f,
    0.863199422f, -0.504863109f,
    0.863972856f, -0.503538384f,
    0.864744258f, -0.502212474f,
    0.865513624f, -0.500885383f,
    0.866280954f, -0.499557113f,
    0.867046246f, -0.498227667f,
    0.867809497f, -0.496897049f,
    0.868570706f, -0.495565262f,
    0.869329871f, -0.494232309f,
    0.870086991f, -0.492898192f,
    0.870842063f, -0.491562916f,
    0.871595087f, -0.490226483f,
    0.872346059f, -0.488888897f,
    0.873094978f, -0.487550160f,
    0.873841843f, -0.486210276f,
    0.874586652f, -0.484869248f,
    0.875329403f, -0.483527079f,
    0.876070094f, -0.482183772f,
    0.876808724f, -0.480839331f,
    0.877545290f, -0.479493758f,
    0.878279792f, -0.478147056f,
    0.879012226f, -0.476799230f,
    0.879742593f, -0.475450282f,
    0.880470889f, -0.474100215f,
    0.881197113f, -0.472749032f,
    0.881921264f, -0.471396737f,
    0.882643340f, -0.470043332f,
    0.883363339f, -0.468688822f,
    0.884081259f, -0.467333209f,
    0.884797098f, -0.465976496f,
    0.885510856f, -0.464618686f,
    0.886222530f, -0.463259784f,
    0.886932119f, -0.461899791f,
    0.887639620f, -0.460538711f,
    0.888345033f, -0.459176548f,
    0.889048356f, -0.457813304f,
    0.889749586f, -0.456448982f,
    0.890448723f, -0.455083587f,
    0.891145765f, -0.453717121f,
    0.891840709f, -0.452349587f,
    0.892533555f, -0.450980989f,
    0.893224301f, -0.449611330f,
    0.893912945f, -0.448240612f,
    0.894599486f, -0.446868840f,
    0.895283921f, -0.445496017f,
    0.895966250f, -0.444122145f,
    0.896646470f, -0.442747228f,
    0.897324581f, -0.441371269f,
    0.898000580f, -0.439994271f,
    0.898674466f, -0.438616239f,
    0.899346237f, -0.437237174f,
    0.900015892f, -0.435857080f,
    0.900683429f, -0.434475961f,
    0.901348847f, -0.433093819f,
    0.902012144f, -0.431710658f,
    0.902673318f, -0.430326481f,
    0.903332368f, -0.428941292f,
    0.903989293f, -0.427555093f,
    0.904644091f, -0.426167889f,
    0.905296759f, -0.424779681f,
    0.905947298f, -0.423390474f,
    0.906595705f, -0.422000271f,
    0.907241978f, -0.420609074f,
    0.907886116f, -0.419216888f,
    0.908528119f, -0.417823716f,
    0.909167983f, -0.416429560f,
    0.909805708f, -0.415034424f,
    0.910441292f, -0.413638312f,
    0.911074734f, -0.412241227f,
    0.911706032f, -0.410843171f,
    0.912335185f, -0.409444149f,
    0.912962190f, -0.408044163f,
    0.913587048f, -0.406643217f,
    0.914209756f, -0.405241314f,
    0.914830312f, -0.403838458f,
    0.915448716f, -0.402434651f,
    0.916064966f, -0.401029897f,
    0.916679060f, -0.399624200f,
    0.917290997f, -0.398217562f,
    0.917900776f, -0.396809987f,
    0.918508394f, -0.395401479f,
    0.919113852f, -0.393992040f,
    0.919717146f, -0.392581674f,
    0.920318277f, -0.391170384f,
    0.920917242f, -0.389758174f,
    0.921514039f, -0.388345047f,
    0.922108669f, -0.386931006f,
    0.922701128f, -0.385516054f,
    0.923291417f, -0.384100195f,
    0.923879533f, -0.382683432f,
    0.924465474f, -0.381265769f,
    0.925049241f, -0.379847209f,
    0.925630831f, -0.378427755f,
    0.926210242f, -0.377007410f,
    0.926787474f, -0.375586178f,
    0.927362526f, -0.374164063f,
    0.927935395f, -0.372741067f,
    0.928506080f, -0.371317194f,
    0.929074581f, -0.369892447f,
    0.929640896f, -0.368466830f,
    0.930205023f, -0.367040346f,
    0.930766961f, -0.365612998f,
    0.931326709f, -0.364184790f,
    0.931884266f, -0.362755724f,
    0.932439629f, -0.361325806f,
    0.932992799f, -0.359895037f,
    0.933543773f, -0.358463421f,
    0.934092550f, -0.357030961f,
    0.934639130f, -0.355597662f,
    0.935183510f, -0.354163525f,
    0.935725689f, -0.352728556f,
    0.936265667f, -0.351292756f,
    0.936803442f, -0.349856130f,
    0.937339012f, -0.348418680f,
    0.937872376f, -0.346980411f,
    0.938403534f, -0.345541325f,
    0.938932484f, -0.344101426f,
    0.939459224f, -0.342660717f,
    0.939983753f, -0.341219202f,
    0.940506071f, -0.339776884f,
    0.941026175f, -0.338333767f,
    0.941544065f, -0.336889853f,
    0.942059740f, -0.335445147f,
    0.942573198f, -0.333999651f,
    0.943084437f, -0.332553370f,
    0.943593458f, -0.331106306f,
    0.944100258f, -0.329658463f,
    0.944604837f, -0.328209844f,
    0.945107193f, -0.326760452f,
    0.945607325f, -0.325310292f,
    0.946105232f, -0.323859367f,
    0.946600913f, -0.322407679f,
    0.947094366f, -0.320955232f,
    0.947585591f, -0.319502031f,
    0.948074586f, -0.318048077f,
    0.948561350f, -0.316593376f,
    0.949045882f, -0.315137929f,
    0.949528181f, -0.313681740f,
    0.950008245f, -0.312224814f,
    0.950486074f, -0.310767153f,
    0.950961666f, -0.309308760f,
    0.951435021f, -0.307849640f,
    0.951906137f, -0.306389795f,
    0.952375013f, -0.304929230f,
    0.952841648f, -0.303467947f,
    0.953306040f, -0.302005949f,
    0.953768190f, -0.300543241f,
    0.954228095f, -0.299079826f,
    0.954685755f, -0.297615707f,
    0.955141168f, -0.296150888f,
    0.955594334f, -0.294685372f,
    0.956045251f, -0.293219163f,
    0.956493919f, -0.291752263f,
    0.956940336f, -0.290284677f,
    0.957384501f, -0.288816408f,
    0.957826413f, -0.287347460f,
    0.958266071f, -0.285877835f,
    0.958703475f, -0.284407537f,
    0.959138622f, -0.282936570f,
    0.959571513f, -0.281464938f,
    0.960002146f, -0.279992643f,
    0.960430519f, -0.278519689f,
    0.960856633f, -0.277046080f,
    0.961280486f, -0.275571819f,
    0.961702077f, -0.274096910f,
    0.962121404f, -0.272621355f,
    0.962538468f, -0.271145160f,
    0.962953267f, -0.269668326f,
    0.963365800f, -0.268190857f,
    0.963776066f, -0.266712757f,
    0.964184064f, -0.265234030f,
    0.964589793f, -0.263754679f,
    0.964993253f, -0.262274707f,
    0.965394442f, -0.260794118f,
    0.965793359f, -0.259312915f,
    0.966190003f, -0.257831102f,
    0.966584374f, -0.256348682f,
    0.966976471f, -0.254865660f,
    0.967366292f, -0.253382037f,
    0.967753837f, -0.251897818f,
    0.968139105f, -0.250413007f,
    0.968522094f, -0.248927606f,
    0.968902805f, -0.247441619f,
    0.969281235f, -0.245955050f,
    0.969657385f, -0.244467903f,
    0.970031253f, -0.242980180f,
    0.970402839f, -0.241491885f,
    0.970772141f, -0.240003022f,
    0.971139158f, -0.238513595f,
    0.971503891f, -0.237023606f,
    0.971866337f, -0.235533059f,
    0.972226497f, -0.234041959f,
    0.972584369f, -0.232550307f,
    0.972939952f, -0.231058108f,
    0.973293246f, -0.229565366f,
    0.973644250f, -0.228072083f,
    0.973992962f, -0.226578264f,
    0.974339383f, -0.225083911f,
    0.974683511f, -0.223589029f,
    0.975025345f, -0.222093621f,
    0.975364885f, -0.220597690f,
    0.975702130f, -0.219101240f,
    0.976037079f, -0.217604275f,
    0.976369731f, -0.216106797f,
    0.976700086f, -0.214608811f,
    0.977028143f, -0.213110320f,
    0.977353900f, -0.211611327f,
    0.977677358f, -0.210111837f,
    0.977998515f, -0.208611852f,
    0.978317371f, -0.207111376f,
    0.978633924f, -0.205610413f,
    0.978948175f, -0.204108966f,
    0.979260123f, -0.202607039f,
    0.979569766f, -0.201104635f,
    0.979877104f, -0.199601758f,
    0.980182136f, -0.198098411f,
    0.980484862f, -0.196594598f,
    0.980785280f, -0.195090322f,
    0.981083391f, -0.193585587f,
    0.981379193f, -0.192080397f,
    0.981672686f, -0.190574755f,
    0.981963869f, -0.189068664f,
    0.982252741f, -0.187562129f,
    0.982539302f, -0.186055152f,
    0.982823551f, -0.184547737f,
    0.983105487f, -0.183039888f,
    0.983385110f, -0.181531608f,
    0.983662419f, -0.180022901f,
    0.983937413f, -0.178513771f,
    0.984210092f, -0.177004220f,
    0.984480455f, -0.175494253f,
    0.984748502f, -0.173983873f,
    0.985014231f, -0.172473084f,
    0.985277642f, -0.170961889f,
    0.985538735f, -0.169450291f,
    0.985797509f, -0.167938295f,
    0.986053963f, -0.166425904f,
    0.986308097f, -0.164913120f,
    0.986559910f, -0.163399949f,
    0.986809402f, -0.161886394f,
    0.987056571f, -0.160372457f,
    0.987301418f, -0.158858143f,
    0.987543942f, -0.157343456f,
    0.987784142f, -0.155828398f,
    0.988022017f, -0.154312973f,
    0.988257568f, -0.152797185f,
    0.988490793f, -0.151281038f,
    0.988721692f, -0.149764535f,
    0.988950265f, -0.148247679f,
    0.989176510f, -0.146730474f,
    0.989400428f, -0.145212925f,
    0.989622017f, -0.143695033f,
    0.989841278f, -0.142176804f,
    0.990058210f, -0.140658239f,
    0.990272812f, -0.139139344f,
    0.990485084f, -0.137620122f,
    0.990695025f, -0.136100575f,
    0.990902635f, -0.134580709f,
    0.991107914f, -0.133060525f,
    0.991310860f, -0.131540029f,
    0.991511473f, -0.130019223f,
    0.991709754f, -0.128498111f,
    0.991905700f, -0.126976696f,
    0.992099313f, -0.125454983f,
    0.992290591f, -0.123932975f,
    0.992479535f, -0.122410675f,
    0.992666142f, -0.120888087f,
    0.992850414f, -0.119365215f,
    0.993032350f, -0.117842062f,
    0.993211949f, -0.116318631f,
    0.993389211f, -0.114794927f,
    0.993564136f, -0.113270952f,
    0.993736722f, -0.111746711f,
    0.993906970f, -0.110222207f,
    0.994074879f, -0.108697444f,
    0.994240449f, -0.107172425f,
    0.994403680f, -0.105647154f,
    0.994564571f, -0.104121634f,
    0.994723121f, -0.102595869f,
    0.994879331f, -0.101069863f,
    0.995033199f, -0.099543619f,
    0.995184727f, -0.098017140f,
    0.995333912f, -0.096490431f,
    0.995480755f, -0.094963495f,
    0.995625256f, -0.093436336f,
    0.995767414f, -0.091908956f,
    0.995907229f, -0.090381361f,
    0.996044701f, -0.088853553f,
    0.996179829f, -0.087325535f,
    0.996312612f, -0.085797312f,
    0.996443051f, -0.084268888f,
    0.996571146f, -0.082740265f,
    0.996696895f, -0.081211447f,
    0.996820299f, -0.079682438f,
    0.996941358f, -0.078153242f,
    0.997060070f, -0.076623861f,
    0.997176437f, -0.075094301f,
    0.997290457f, -0.073564564f,
    0.997402130f, -0.072034653f,
    0.997511456f, -0.070504573f,
    0.997618435f, -0.068974328f,
    0.997723067f, -0.067443920f,
    0.997825350f, -0.065913353f,
    0.997925286f, -0.064382631f,
    0.998022874f, -0.062851758f,
    0.998118113f, -0.061320736f,
    0.998211003f, -0.059789571f,
    0.998301545f, -0.058258265f,
    0.998389737f, -0.056726821f,
    0.998475581f, -0.055195244f,
    0.998559074f, -0.053663538f,
    0.998640218f, -0.052131705f,
    0.998719012f, -0.050599749f,
    0.998795456f, -0.049067674f,
    0.998869550f, -0.047535484f,
    0.998941293f, -0.046003182f,
    0.999010686f, -0.044470772f,
    0.999077728f, -0.042938257f,
    0.999142419f, -0.041405641f,
    0.999204759f, -0.039872928f,
    0.999264747f, -0.038340120f,
    0.999322385f, -0.036807223f,
    0.999377670f, -0.035274239f,
    0.999430605f, -0.033741172f,
    0.999481187f, -0.032208025f,
    0.999529418f, -0.030674803f,
    0.999575296f, -0.029141509f,
    0.999618822f, -0.027608146f,
    0.999659997f, -0.026074718f,
    0.999698819f, -0.024541229f,
    0.999735288f, -0.023007681f,
    0.999769405f, -0.021474080f,
    0.999801170f, -0.019940429f,
    0.999830582f, -0.018406730f,
    0.999857641f, -0.016872988f,
    0.999882347f, -0.015339206f,
    0.999904701f, -0.013805389f,
    0.999924702f, -0.012271538f,
    0.999942350f, -0.010737659f,
    0.999957645f, -0.009203755f,
    0.999970586f, -0.007669829f,
    0.999981175f, -0.006135885f,
    0.999989411f, -0.004601926f,
    0.999995294f, -0.003067957f,
    0.999998823f, -0.001533980f
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */



/**
  @brief  Q31 Twiddle factors Table
*/

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_16)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre> for(i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 16, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 */
const q31_t twiddleCoef_16_q31[24] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000,
    (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
    (q31_t)0x5A82799A, (q31_t)0x5A82799A,
    (q31_t)0x30FBC54D, (q31_t)0x7641AF3C,
    (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
    (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
    (q31_t)0xA57D8666, (q31_t)0x5A82799A,
    (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
    (q31_t)0x80000000, (q31_t)0x00000000,
    (q31_t)0x89BE50C3, (q31_t)0xCF043AB2,
    (q31_t)0xA57D8666, (q31_t)0xA57D8666,
    (q31_t)0xCF043AB2, (q31_t)0x89BE50C3
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_32)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 32, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 */
const q31_t twiddleCoef_32_q31[48] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000,
    (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
    (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
    (q31_t)0x6A6D98A4, (q31_t)0x471CECE6,
    (q31_t)0x5A82799A, (q31_t)0x5A82799A,
    (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
    (q31_t)0x30FBC54D, (q31_t)0x7641AF3C,
    (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
    (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
    (q31_t)0xE70747C3, (q31_t)0x7D8A5F3F,
    (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
    (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
    (q31_t)0xA57D8666, (q31_t)0x5A82799A,
    (q31_t)0x9592675B, (q31_t)0x471CECE6,
    (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
    (q31_t)0x8275A0C0, (q31_t)0x18F8B83C,
    (q31_t)0x80000000, (q31_t)0x00000000,
    (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
    (q31_t)0x89BE50C3, (q31_t)0xCF043AB2,
    (q31_t)0x9592675B, (q31_t)0xB8E31319,
    (q31_t)0xA57D8666, (q31_t)0xA57D8666,
    (q31_t)0xB8E31319, (q31_t)0x9592675B,
    (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
    (q31_t)0xE70747C3, (q31_t)0x8275A0C0
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_64)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 64, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 */
const q31_t twiddleCoef_64_q31[96] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xF3742CA1, (q31_t)0x809DC970
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_128)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i < 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 128, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 */
const q31_t twiddleCoef_128_q31[192] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xAECC336B,
	(q31_t)0x62F201AC, (q31_t)0xAA0A5B2D, (q31_t)0x5ED77C89,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA1288376,
	(q31_t)0x55F5A4D2, (q31_t)0x9D0DFE53, (q31_t)0x5133CC94,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9592675B,
	(q31_t)0x471CECE6, (q31_t)0x9235F2EB, (q31_t)0x41CE1E64,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8C4A142F,
	(q31_t)0x36BA2013, (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8582FAA4,
	(q31_t)0x25280C5D, (q31_t)0x83D60411, (q31_t)0x1F19F97B,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x8162AA03,
	(q31_t)0x12C8106E, (q31_t)0x809DC970, (q31_t)0x0C8BD35E,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x80000000,
	(q31_t)0x00000000, (q31_t)0x80277872, (q31_t)0xF9B82683,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x8162AA03,
	(q31_t)0xED37EF91, (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x8582FAA4,
	(q31_t)0xDAD7F3A2, (q31_t)0x877B7BEC, (q31_t)0xD4E0CB14,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8C4A142F,
	(q31_t)0xC945DFEC, (q31_t)0x8F1D343A, (q31_t)0xC3A9458F,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x9592675B,
	(q31_t)0xB8E31319, (q31_t)0x99307EE0, (q31_t)0xB3C0200C,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0xA1288376,
	(q31_t)0xAA0A5B2D, (q31_t)0xA57D8666, (q31_t)0xA57D8666,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAECC336B,
	(q31_t)0x9D0DFE53, (q31_t)0xB3C0200C, (q31_t)0x99307EE0,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBE31E19B,
	(q31_t)0x9235F2EB, (q31_t)0xC3A9458F, (q31_t)0x8F1D343A,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCF043AB2,
	(q31_t)0x89BE50C3, (q31_t)0xD4E0CB14, (q31_t)0x877B7BEC,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xE0E60684,
	(q31_t)0x83D60411, (q31_t)0xE70747C3, (q31_t)0x8275A0C0,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xF3742CA1,
	(q31_t)0x809DC970, (q31_t)0xF9B82683, (q31_t)0x80277872
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_256)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 256, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 
 */
const q31_t twiddleCoef_256_q31[384] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FF62182,
	(q31_t)0x03242ABF, (q31_t)0x7FD8878D, (q31_t)0x0647D97C,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F0991C3, (q31_t)0x0FAB272B,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E1D93E9,
	(q31_t)0x15E21444, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7C29FBEE,
	(q31_t)0x1F19F97B, (q31_t)0x7B5D039D, (q31_t)0x2223A4C5,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x798A23B1,
	(q31_t)0x2826B928, (q31_t)0x78848413, (q31_t)0x2B1F34EB,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x7504D345, (q31_t)0x33DEF287,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x72552C84,
	(q31_t)0x398CDD32, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6DCA0D14,
	(q31_t)0x41CE1E64, (q31_t)0x6C242960, (q31_t)0x447ACD50,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x68A69E81,
	(q31_t)0x49B41533, (q31_t)0x66CF811F, (q31_t)0x4C3FDFF3,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x60EC3830, (q31_t)0x539B2AEF,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5CB420DF,
	(q31_t)0x5842DD54, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x55F5A4D2,
	(q31_t)0x5ED77C89, (q31_t)0x539B2AEF, (q31_t)0x60EC3830,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4EBFE8A4,
	(q31_t)0x64E88926, (q31_t)0x4C3FDFF3, (q31_t)0x66CF811F,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x447ACD50, (q31_t)0x6C242960,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x3F1749B7,
	(q31_t)0x6F5F02B1, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x36BA2013,
	(q31_t)0x73B5EBD0, (q31_t)0x33DEF287, (q31_t)0x7504D345,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2E110A62,
	(q31_t)0x776C4EDB, (q31_t)0x2B1F34EB, (q31_t)0x78848413,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x2223A4C5, (q31_t)0x7B5D039D,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1C0B826A,
	(q31_t)0x7CE3CEB1, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x12C8106E,
	(q31_t)0x7E9D55FC, (q31_t)0x0FAB272B, (q31_t)0x7F0991C3,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x096A9049,
	(q31_t)0x7FA736B4, (q31_t)0x0647D97C, (q31_t)0x7FD8878D,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xFCDBD541, (q31_t)0x7FF62182,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF6956FB6,
	(q31_t)0x7FA736B4, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xED37EF91,
	(q31_t)0x7E9D55FC, (q31_t)0xEA1DEBBB, (q31_t)0x7E1D93E9,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE3F47D95,
	(q31_t)0x7CE3CEB1, (q31_t)0xE0E60684, (q31_t)0x7C29FBEE,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xD7D946D7, (q31_t)0x798A23B1,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD1EEF59E,
	(q31_t)0x776C4EDB, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xC945DFEC,
	(q31_t)0x73B5EBD0, (q31_t)0xC67322CD, (q31_t)0x72552C84,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC0E8B648,
	(q31_t)0x6F5F02B1, (q31_t)0xBE31E19B, (q31_t)0x6DCA0D14,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xB64BEACC, (q31_t)0x68A69E81,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB140175B,
	(q31_t)0x64E88926, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAA0A5B2D,
	(q31_t)0x5ED77C89, (q31_t)0xA7BD22AB, (q31_t)0x5CB420DF,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA34BDF20,
	(q31_t)0x5842DD54, (q31_t)0xA1288376, (q31_t)0x55F5A4D2,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9B1776D9, (q31_t)0x4EBFE8A4,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9759617E,
	(q31_t)0x49B41533, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9235F2EB,
	(q31_t)0x41CE1E64, (q31_t)0x90A0FD4E, (q31_t)0x3F1749B7,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8DAAD37B,
	(q31_t)0x398CDD32, (q31_t)0x8C4A142F, (q31_t)0x36BA2013,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8893B124, (q31_t)0x2E110A62,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8675DC4E,
	(q31_t)0x2826B928, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x83D60411,
	(q31_t)0x1F19F97B, (q31_t)0x831C314E, (q31_t)0x1C0B826A,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x81E26C16,
	(q31_t)0x15E21444, (q31_t)0x8162AA03, (q31_t)0x12C8106E,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x8058C94C, (q31_t)0x096A9049,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x8009DE7D,
	(q31_t)0x03242ABF, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x80277872,
	(q31_t)0xF9B82683, (q31_t)0x8058C94C, (q31_t)0xF6956FB6,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80F66E3C,
	(q31_t)0xF054D8D4, (q31_t)0x8162AA03, (q31_t)0xED37EF91,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x831C314E, (q31_t)0xE3F47D95,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x84A2FC62,
	(q31_t)0xDDDC5B3A, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x877B7BEC,
	(q31_t)0xD4E0CB14, (q31_t)0x8893B124, (q31_t)0xD1EEF59E,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC210D78, (q31_t)0x8C4A142F, (q31_t)0xC945DFEC,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x90A0FD4E, (q31_t)0xC0E8B648,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x93DBD69F,
	(q31_t)0xBB8532AF, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x99307EE0,
	(q31_t)0xB3C0200C, (q31_t)0x9B1776D9, (q31_t)0xB140175B,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9F13C7D0,
	(q31_t)0xAC64D510, (q31_t)0xA1288376, (q31_t)0xAA0A5B2D,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xA7BD22AB, (q31_t)0xA34BDF20,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAC64D510,
	(q31_t)0x9F13C7D0, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB3C0200C,
	(q31_t)0x99307EE0, (q31_t)0xB64BEACC, (q31_t)0x9759617E,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBB8532AF,
	(q31_t)0x93DBD69F, (q31_t)0xBE31E19B, (q31_t)0x9235F2EB,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xC67322CD, (q31_t)0x8DAAD37B,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCC210D78,
	(q31_t)0x8AFB2CBA, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD4E0CB14,
	(q31_t)0x877B7BEC, (q31_t)0xD7D946D7, (q31_t)0x8675DC4E,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDDDC5B3A,
	(q31_t)0x84A2FC62, (q31_t)0xE0E60684, (q31_t)0x83D60411,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xEA1DEBBB, (q31_t)0x81E26C16,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xF054D8D4,
	(q31_t)0x80F66E3C, (q31_t)0xF3742CA1, (q31_t)0x809DC970,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF9B82683,
	(q31_t)0x80277872, (q31_t)0xFCDBD541, (q31_t)0x8009DE7D
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_512)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 512, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 
 */
const q31_t twiddleCoef_512_q31[768] = {
    (q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFD885A,
	(q31_t)0x01921D1F, (q31_t)0x7FF62182, (q31_t)0x03242ABF,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7FC25596, (q31_t)0x07D95B9E,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F872BF3,
	(q31_t)0x0AFB6805, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F0991C3,
	(q31_t)0x0FAB272B, (q31_t)0x7ED5E5C6, (q31_t)0x1139F0CE,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E5FE493,
	(q31_t)0x145576B1, (q31_t)0x7E1D93E9, (q31_t)0x15E21444,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7D3980EC, (q31_t)0x1A82A025,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7C894BDD,
	(q31_t)0x1D934FE5, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7B5D039D,
	(q31_t)0x2223A4C5, (q31_t)0x7AEF6323, (q31_t)0x23A6887E,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A05EEAD,
	(q31_t)0x26A82185, (q31_t)0x798A23B1, (q31_t)0x2826B928,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x77FAB988, (q31_t)0x2C98FBBA,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x76D94988,
	(q31_t)0x2F875262, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x7504D345,
	(q31_t)0x33DEF287, (q31_t)0x745F9DD1, (q31_t)0x354D9056,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x7307C3D0,
	(q31_t)0x382493B0, (q31_t)0x72552C84, (q31_t)0x398CDD32,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x70231099, (q31_t)0x3DB832A5,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6E96A99C,
	(q31_t)0x4073F21D, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6C242960,
	(q31_t)0x447ACD50, (q31_t)0x6B4AF278, (q31_t)0x45CD358F,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x698C246C,
	(q31_t)0x4869E664, (q31_t)0x68A69E81, (q31_t)0x49B41533,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x65DDFBD3, (q31_t)0x4D8162C4,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x63EF328F,
	(q31_t)0x4FFB654D, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x60EC3830,
	(q31_t)0x539B2AEF, (q31_t)0x5FE3B38D, (q31_t)0x54CA0A4A,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5DC79D7C,
	(q31_t)0x571DEEF9, (q31_t)0x5CB420DF, (q31_t)0x5842DD54,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x59646497, (q31_t)0x5B9D1153,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x571DEEF9,
	(q31_t)0x5DC79D7C, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x539B2AEF,
	(q31_t)0x60EC3830, (q31_t)0x5269126E, (q31_t)0x61F1003E,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x4FFB654D,
	(q31_t)0x63EF328F, (q31_t)0x4EBFE8A4, (q31_t)0x64E88926,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x4AFB6C97, (q31_t)0x67BD0FBC,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x4869E664,
	(q31_t)0x698C246C, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x447ACD50,
	(q31_t)0x6C242960, (q31_t)0x4325C135, (q31_t)0x6CF934FB,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4073F21D,
	(q31_t)0x6E96A99C, (q31_t)0x3F1749B7, (q31_t)0x6F5F02B1,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x3AF2EEB7, (q31_t)0x719E2CD2,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x382493B0,
	(q31_t)0x7307C3D0, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x33DEF287,
	(q31_t)0x7504D345, (q31_t)0x326E54C7, (q31_t)0x75A585CF,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x2F875262,
	(q31_t)0x76D94988, (q31_t)0x2E110A62, (q31_t)0x776C4EDB,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x29A3C484, (q31_t)0x7909A92C,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x26A82185,
	(q31_t)0x7A05EEAD, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x2223A4C5,
	(q31_t)0x7B5D039D, (q31_t)0x209F701C, (q31_t)0x7BC5E28F,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1D934FE5,
	(q31_t)0x7C894BDD, (q31_t)0x1C0B826A, (q31_t)0x7CE3CEB1,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x176DD9DE, (q31_t)0x7DD6668E,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x145576B1,
	(q31_t)0x7E5FE493, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x0FAB272B,
	(q31_t)0x7F0991C3, (q31_t)0x0E1BC2E3, (q31_t)0x7F3857F5,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0AFB6805,
	(q31_t)0x7F872BF3, (q31_t)0x096A9049, (q31_t)0x7FA736B4,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x04B6195D, (q31_t)0x7FE9CBC0,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x01921D1F,
	(q31_t)0x7FFD885A, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFCDBD541,
	(q31_t)0x7FF62182, (q31_t)0xFB49E6A2, (q31_t)0x7FE9CBC0,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF826A461,
	(q31_t)0x7FC25596, (q31_t)0xF6956FB6, (q31_t)0x7FA736B4,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xF1E43D1C, (q31_t)0x7F3857F5,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEEC60F31,
	(q31_t)0x7ED5E5C6, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEA1DEBBB,
	(q31_t)0x7E1D93E9, (q31_t)0xE8922621, (q31_t)0x7DD6668E,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE57D5FDA,
	(q31_t)0x7D3980EC, (q31_t)0xE3F47D95, (q31_t)0x7CE3CEB1,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xDF608FE3, (q31_t)0x7BC5E28F,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDC597781,
	(q31_t)0x7AEF6323, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD7D946D7,
	(q31_t)0x798A23B1, (q31_t)0xD65C3B7B, (q31_t)0x7909A92C,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD3670445,
	(q31_t)0x77FAB988, (q31_t)0xD1EEF59E, (q31_t)0x776C4EDB,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xCD91AB38, (q31_t)0x75A585CF,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCAB26FA9,
	(q31_t)0x745F9DD1, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC67322CD,
	(q31_t)0x72552C84, (q31_t)0xC50D1148, (q31_t)0x719E2CD2,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC247CD5A,
	(q31_t)0x70231099, (q31_t)0xC0E8B648, (q31_t)0x6F5F02B1,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xBCDA3ECA, (q31_t)0x6CF934FB,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBA32CA70,
	(q31_t)0x6B4AF278, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB64BEACC,
	(q31_t)0x68A69E81, (q31_t)0xB5049368, (q31_t)0x67BD0FBC,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB27E9D3B,
	(q31_t)0x65DDFBD3, (q31_t)0xB140175B, (q31_t)0x64E88926,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAECC336B,
	(q31_t)0x62F201AC, (q31_t)0xAD96ED91, (q31_t)0x61F1003E,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xAB35F5B5,
	(q31_t)0x5FE3B38D, (q31_t)0xAA0A5B2D, (q31_t)0x5ED77C89,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA7BD22AB,
	(q31_t)0x5CB420DF, (q31_t)0xA69B9B68, (q31_t)0x5B9D1153,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA462EEAC,
	(q31_t)0x59646497, (q31_t)0xA34BDF20, (q31_t)0x5842DD54,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA1288376,
	(q31_t)0x55F5A4D2, (q31_t)0xA01C4C72, (q31_t)0x54CA0A4A,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9E0EFFC1,
	(q31_t)0x5269126E, (q31_t)0x9D0DFE53, (q31_t)0x5133CC94,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9B1776D9,
	(q31_t)0x4EBFE8A4, (q31_t)0x9A22042C, (q31_t)0x4D8162C4,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x9842F043,
	(q31_t)0x4AFB6C97, (q31_t)0x9759617E, (q31_t)0x49B41533,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x9592675B,
	(q31_t)0x471CECE6, (q31_t)0x94B50D87, (q31_t)0x45CD358F,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9306CB04,
	(q31_t)0x4325C135, (q31_t)0x9235F2EB, (q31_t)0x41CE1E64,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x90A0FD4E,
	(q31_t)0x3F1749B7, (q31_t)0x8FDCEF66, (q31_t)0x3DB832A5,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8E61D32D,
	(q31_t)0x3AF2EEB7, (q31_t)0x8DAAD37B, (q31_t)0x398CDD32,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8C4A142F,
	(q31_t)0x36BA2013, (q31_t)0x8BA0622F, (q31_t)0x354D9056,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8A5A7A30,
	(q31_t)0x326E54C7, (q31_t)0x89BE50C3, (q31_t)0x30FBC54D,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x8893B124,
	(q31_t)0x2E110A62, (q31_t)0x88054677, (q31_t)0x2C98FBBA,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x86F656D3,
	(q31_t)0x29A3C484, (q31_t)0x8675DC4E, (q31_t)0x2826B928,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x8582FAA4,
	(q31_t)0x25280C5D, (q31_t)0x85109CDC, (q31_t)0x23A6887E,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x843A1D70,
	(q31_t)0x209F701C, (q31_t)0x83D60411, (q31_t)0x1F19F97B,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x831C314E,
	(q31_t)0x1C0B826A, (q31_t)0x82C67F13, (q31_t)0x1A82A025,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x82299971,
	(q31_t)0x176DD9DE, (q31_t)0x81E26C16, (q31_t)0x15E21444,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x8162AA03,
	(q31_t)0x12C8106E, (q31_t)0x812A1A39, (q31_t)0x1139F0CE,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80C7A80A,
	(q31_t)0x0E1BC2E3, (q31_t)0x809DC970, (q31_t)0x0C8BD35E,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x8058C94C,
	(q31_t)0x096A9049, (q31_t)0x803DAA69, (q31_t)0x07D95B9E,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x80163440,
	(q31_t)0x04B6195D, (q31_t)0x8009DE7D, (q31_t)0x03242ABF,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x80000000,
	(q31_t)0x00000000, (q31_t)0x800277A5, (q31_t)0xFE6DE2E0,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x80163440,
	(q31_t)0xFB49E6A2, (q31_t)0x80277872, (q31_t)0xF9B82683,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x8058C94C,
	(q31_t)0xF6956FB6, (q31_t)0x8078D40D, (q31_t)0xF50497FA,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80C7A80A,
	(q31_t)0xF1E43D1C, (q31_t)0x80F66E3C, (q31_t)0xF054D8D4,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8162AA03,
	(q31_t)0xED37EF91, (q31_t)0x81A01B6C, (q31_t)0xEBAA894E,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x82299971,
	(q31_t)0xE8922621, (q31_t)0x8275A0C0, (q31_t)0xE70747C3,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x831C314E,
	(q31_t)0xE3F47D95, (q31_t)0x8376B422, (q31_t)0xE26CB01A,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x843A1D70,
	(q31_t)0xDF608FE3, (q31_t)0x84A2FC62, (q31_t)0xDDDC5B3A,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x8582FAA4,
	(q31_t)0xDAD7F3A2, (q31_t)0x85FA1152, (q31_t)0xD957DE7A,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x86F656D3,
	(q31_t)0xD65C3B7B, (q31_t)0x877B7BEC, (q31_t)0xD4E0CB14,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x8893B124,
	(q31_t)0xD1EEF59E, (q31_t)0x8926B677, (q31_t)0xD078AD9D,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8A5A7A30,
	(q31_t)0xCD91AB38, (q31_t)0x8AFB2CBA, (q31_t)0xCC210D78,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8C4A142F,
	(q31_t)0xC945DFEC, (q31_t)0x8CF83C30, (q31_t)0xC7DB6C50,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8E61D32D,
	(q31_t)0xC50D1148, (q31_t)0x8F1D343A, (q31_t)0xC3A9458F,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x90A0FD4E,
	(q31_t)0xC0E8B648, (q31_t)0x91695663, (q31_t)0xBF8C0DE2,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x9306CB04,
	(q31_t)0xBCDA3ECA, (q31_t)0x93DBD69F, (q31_t)0xBB8532AF,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x9592675B,
	(q31_t)0xB8E31319, (q31_t)0x9673DB94, (q31_t)0xB796199B,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x9842F043,
	(q31_t)0xB5049368, (q31_t)0x99307EE0, (q31_t)0xB3C0200C,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9B1776D9,
	(q31_t)0xB140175B, (q31_t)0x9C10CD70, (q31_t)0xB0049AB2,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9E0EFFC1,
	(q31_t)0xAD96ED91, (q31_t)0x9F13C7D0, (q31_t)0xAC64D510,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA1288376,
	(q31_t)0xAA0A5B2D, (q31_t)0xA2386283, (q31_t)0xA8E21106,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA462EEAC,
	(q31_t)0xA69B9B68, (q31_t)0xA57D8666, (q31_t)0xA57D8666,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA7BD22AB,
	(q31_t)0xA34BDF20, (q31_t)0xA8E21106, (q31_t)0xA2386283,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAB35F5B5,
	(q31_t)0xA01C4C72, (q31_t)0xAC64D510, (q31_t)0x9F13C7D0,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xAECC336B,
	(q31_t)0x9D0DFE53, (q31_t)0xB0049AB2, (q31_t)0x9C10CD70,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB27E9D3B,
	(q31_t)0x9A22042C, (q31_t)0xB3C0200C, (q31_t)0x99307EE0,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB64BEACC,
	(q31_t)0x9759617E, (q31_t)0xB796199B, (q31_t)0x9673DB94,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xBA32CA70,
	(q31_t)0x94B50D87, (q31_t)0xBB8532AF, (q31_t)0x93DBD69F,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBE31E19B,
	(q31_t)0x9235F2EB, (q31_t)0xBF8C0DE2, (q31_t)0x91695663,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC247CD5A,
	(q31_t)0x8FDCEF66, (q31_t)0xC3A9458F, (q31_t)0x8F1D343A,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC67322CD,
	(q31_t)0x8DAAD37B, (q31_t)0xC7DB6C50, (q31_t)0x8CF83C30,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xCAB26FA9,
	(q31_t)0x8BA0622F, (q31_t)0xCC210D78, (q31_t)0x8AFB2CBA,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCF043AB2,
	(q31_t)0x89BE50C3, (q31_t)0xD078AD9D, (q31_t)0x8926B677,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD3670445,
	(q31_t)0x88054677, (q31_t)0xD4E0CB14, (q31_t)0x877B7BEC,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD7D946D7,
	(q31_t)0x8675DC4E, (q31_t)0xD957DE7A, (q31_t)0x85FA1152,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDC597781,
	(q31_t)0x85109CDC, (q31_t)0xDDDC5B3A, (q31_t)0x84A2FC62,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xE0E60684,
	(q31_t)0x83D60411, (q31_t)0xE26CB01A, (q31_t)0x8376B422,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE57D5FDA,
	(q31_t)0x82C67F13, (q31_t)0xE70747C3, (q31_t)0x8275A0C0,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xEA1DEBBB,
	(q31_t)0x81E26C16, (q31_t)0xEBAA894E, (q31_t)0x81A01B6C,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xEEC60F31,
	(q31_t)0x812A1A39, (q31_t)0xF054D8D4, (q31_t)0x80F66E3C,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF3742CA1,
	(q31_t)0x809DC970, (q31_t)0xF50497FA, (q31_t)0x8078D40D,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF826A461,
	(q31_t)0x803DAA69, (q31_t)0xF9B82683, (q31_t)0x80277872,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFCDBD541,
	(q31_t)0x8009DE7D, (q31_t)0xFE6DE2E0, (q31_t)0x800277A5
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_1024)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 1024, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 
 */
const q31_t twiddleCoef_1024_q31[1536] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFF6216,
	(q31_t)0x00C90F88, (q31_t)0x7FFD885A, (q31_t)0x01921D1F,
	(q31_t)0x7FFA72D1, (q31_t)0x025B26D7, (q31_t)0x7FF62182,
	(q31_t)0x03242ABF, (q31_t)0x7FF09477, (q31_t)0x03ED26E6,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FE1C76B,
	(q31_t)0x057F0034, (q31_t)0x7FD8878D, (q31_t)0x0647D97C,
	(q31_t)0x7FCE0C3E, (q31_t)0x0710A344, (q31_t)0x7FC25596,
	(q31_t)0x07D95B9E, (q31_t)0x7FB563B2, (q31_t)0x08A2009A,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F97CEBC,
	(q31_t)0x0A3308BC, (q31_t)0x7F872BF3, (q31_t)0x0AFB6805,
	(q31_t)0x7F754E7F, (q31_t)0x0BC3AC35, (q31_t)0x7F62368F,
	(q31_t)0x0C8BD35E, (q31_t)0x7F4DE450, (q31_t)0x0D53DB92,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F2191B4,
	(q31_t)0x0EE38765, (q31_t)0x7F0991C3, (q31_t)0x0FAB272B,
	(q31_t)0x7EF0585F, (q31_t)0x1072A047, (q31_t)0x7ED5E5C6,
	(q31_t)0x1139F0CE, (q31_t)0x7EBA3A39, (q31_t)0x120116D4,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E7F3956,
	(q31_t)0x138EDBB0, (q31_t)0x7E5FE493, (q31_t)0x145576B1,
	(q31_t)0x7E3F57FE, (q31_t)0x151BDF85, (q31_t)0x7E1D93E9,
	(q31_t)0x15E21444, (q31_t)0x7DFA98A7, (q31_t)0x16A81305,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7DB0FDF7,
	(q31_t)0x183366E8, (q31_t)0x7D8A5F3F, (q31_t)0x18F8B83C,
	(q31_t)0x7D628AC5, (q31_t)0x19BDCBF2, (q31_t)0x7D3980EC,
	(q31_t)0x1A82A025, (q31_t)0x7D0F4218, (q31_t)0x1B4732EF,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7CB72724,
	(q31_t)0x1CCF8CB3, (q31_t)0x7C894BDD, (q31_t)0x1D934FE5,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E56CA1E, (q31_t)0x7C29FBEE,
	(q31_t)0x1F19F97B, (q31_t)0x7BF88830, (q31_t)0x1FDCDC1A,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7B920B89,
	(q31_t)0x2161B39F, (q31_t)0x7B5D039D, (q31_t)0x2223A4C5,
	(q31_t)0x7B26CB4F, (q31_t)0x22E541AE, (q31_t)0x7AEF6323,
	(q31_t)0x23A6887E, (q31_t)0x7AB6CBA3, (q31_t)0x24677757,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A4210D8,
	(q31_t)0x25E845B5, (q31_t)0x7A05EEAD, (q31_t)0x26A82185,
	(q31_t)0x79C89F6D, (q31_t)0x27679DF4, (q31_t)0x798A23B1,
	(q31_t)0x2826B928, (q31_t)0x794A7C11, (q31_t)0x28E5714A,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78C7ABA1,
	(q31_t)0x2A61B101, (q31_t)0x78848413, (q31_t)0x2B1F34EB,
	(q31_t)0x78403328, (q31_t)0x2BDC4E6F, (q31_t)0x77FAB988,
	(q31_t)0x2C98FBBA, (q31_t)0x77B417DF, (q31_t)0x2D553AFB,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x77235F2D,
	(q31_t)0x2ECC681E, (q31_t)0x76D94988, (q31_t)0x2F875262,
	(q31_t)0x768E0EA5, (q31_t)0x3041C760, (q31_t)0x7641AF3C,
	(q31_t)0x30FBC54D, (q31_t)0x75F42C0A, (q31_t)0x31B54A5D,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x7555BD4B,
	(q31_t)0x3326E2C2, (q31_t)0x7504D345, (q31_t)0x33DEF287,
	(q31_t)0x74B2C883, (q31_t)0x3496824F, (q31_t)0x745F9DD1,
	(q31_t)0x354D9056, (q31_t)0x740B53FA, (q31_t)0x36041AD9,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x735F6626,
	(q31_t)0x376F9E46, (q31_t)0x7307C3D0, (q31_t)0x382493B0,
	(q31_t)0x72AF05A6, (q31_t)0x38D8FE93, (q31_t)0x72552C84,
	(q31_t)0x398CDD32, (q31_t)0x71FA3948, (q31_t)0x3A402DD1,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x71410804,
	(q31_t)0x3BA51E29, (q31_t)0x70E2CBC6, (q31_t)0x3C56BA70,
	(q31_t)0x708378FE, (q31_t)0x3D07C1D5, (q31_t)0x70231099,
	(q31_t)0x3DB832A5, (q31_t)0x6FC19385, (q31_t)0x3E680B2C,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6EFB5F12,
	(q31_t)0x3FC5EC97, (q31_t)0x6E96A99C, (q31_t)0x4073F21D,
	(q31_t)0x6E30E349, (q31_t)0x4121589A, (q31_t)0x6DCA0D14,
	(q31_t)0x41CE1E64, (q31_t)0x6D6227FA, (q31_t)0x427A41D0,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6C8F351C,
	(q31_t)0x43D09AEC, (q31_t)0x6C242960, (q31_t)0x447ACD50,
	(q31_t)0x6BB812D0, (q31_t)0x452456BC, (q31_t)0x6B4AF278,
	(q31_t)0x45CD358F, (q31_t)0x6ADCC964, (q31_t)0x46756827,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x69FD614A,
	(q31_t)0x47C3C22E, (q31_t)0x698C246C, (q31_t)0x4869E664,
	(q31_t)0x6919E320, (q31_t)0x490F57EE, (q31_t)0x68A69E81,
	(q31_t)0x49B41533, (q31_t)0x683257AA, (q31_t)0x4A581C9D,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x6746C7D7,
	(q31_t)0x4B9E038F, (q31_t)0x66CF811F, (q31_t)0x4C3FDFF3,
	(q31_t)0x66573CBB, (q31_t)0x4CE10034, (q31_t)0x65DDFBD3,
	(q31_t)0x4D8162C4, (q31_t)0x6563BF92, (q31_t)0x4E210617,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x646C59BF,
	(q31_t)0x4F5E08E3, (q31_t)0x63EF328F, (q31_t)0x4FFB654D,
	(q31_t)0x637114CC, (q31_t)0x5097FC5E, (q31_t)0x62F201AC,
	(q31_t)0x5133CC94, (q31_t)0x6271FA69, (q31_t)0x51CED46E,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x616F146B,
	(q31_t)0x53028517, (q31_t)0x60EC3830, (q31_t)0x539B2AEF,
	(q31_t)0x60686CCE, (q31_t)0x5433027D, (q31_t)0x5FE3B38D,
	(q31_t)0x54CA0A4A, (q31_t)0x5F5E0DB3, (q31_t)0x556040E2,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5E50015D,
	(q31_t)0x568A34A9, (q31_t)0x5DC79D7C, (q31_t)0x571DEEF9,
	(q31_t)0x5D3E5236, (q31_t)0x57B0D256, (q31_t)0x5CB420DF,
	(q31_t)0x5842DD54, (q31_t)0x5C290ACC, (q31_t)0x58D40E8C,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5B1035CF,
	(q31_t)0x59F3DE12, (q31_t)0x5A82799A, (q31_t)0x5A82799A,
	(q31_t)0x59F3DE12, (q31_t)0x5B1035CF, (q31_t)0x59646497,
	(q31_t)0x5B9D1153, (q31_t)0x58D40E8C, (q31_t)0x5C290ACC,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x57B0D256,
	(q31_t)0x5D3E5236, (q31_t)0x571DEEF9, (q31_t)0x5DC79D7C,
	(q31_t)0x568A34A9, (q31_t)0x5E50015D, (q31_t)0x55F5A4D2,
	(q31_t)0x5ED77C89, (q31_t)0x556040E2, (q31_t)0x5F5E0DB3,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x5433027D,
	(q31_t)0x60686CCE, (q31_t)0x539B2AEF, (q31_t)0x60EC3830,
	(q31_t)0x53028517, (q31_t)0x616F146B, (q31_t)0x5269126E,
	(q31_t)0x61F1003E, (q31_t)0x51CED46E, (q31_t)0x6271FA69,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x5097FC5E,
	(q31_t)0x637114CC, (q31_t)0x4FFB654D, (q31_t)0x63EF328F,
	(q31_t)0x4F5E08E3, (q31_t)0x646C59BF, (q31_t)0x4EBFE8A4,
	(q31_t)0x64E88926, (q31_t)0x4E210617, (q31_t)0x6563BF92,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4CE10034,
	(q31_t)0x66573CBB, (q31_t)0x4C3FDFF3, (q31_t)0x66CF811F,
	(q31_t)0x4B9E038F, (q31_t)0x6746C7D7, (q31_t)0x4AFB6C97,
	(q31_t)0x67BD0FBC, (q31_t)0x4A581C9D, (q31_t)0x683257AA,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x490F57EE,
	(q31_t)0x6919E320, (q31_t)0x4869E664, (q31_t)0x698C246C,
	(q31_t)0x47C3C22E, (q31_t)0x69FD614A, (q31_t)0x471CECE6,
	(q31_t)0x6A6D98A4, (q31_t)0x46756827, (q31_t)0x6ADCC964,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x452456BC,
	(q31_t)0x6BB812D0, (q31_t)0x447ACD50, (q31_t)0x6C242960,
	(q31_t)0x43D09AEC, (q31_t)0x6C8F351C, (q31_t)0x4325C135,
	(q31_t)0x6CF934FB, (q31_t)0x427A41D0, (q31_t)0x6D6227FA,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4121589A,
	(q31_t)0x6E30E349, (q31_t)0x4073F21D, (q31_t)0x6E96A99C,
	(q31_t)0x3FC5EC97, (q31_t)0x6EFB5F12, (q31_t)0x3F1749B7,
	(q31_t)0x6F5F02B1, (q31_t)0x3E680B2C, (q31_t)0x6FC19385,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3D07C1D5,
	(q31_t)0x708378FE, (q31_t)0x3C56BA70, (q31_t)0x70E2CBC6,
	(q31_t)0x3BA51E29, (q31_t)0x71410804, (q31_t)0x3AF2EEB7,
	(q31_t)0x719E2CD2, (q31_t)0x3A402DD1, (q31_t)0x71FA3948,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x38D8FE93,
	(q31_t)0x72AF05A6, (q31_t)0x382493B0, (q31_t)0x7307C3D0,
	(q31_t)0x376F9E46, (q31_t)0x735F6626, (q31_t)0x36BA2013,
	(q31_t)0x73B5EBD0, (q31_t)0x36041AD9, (q31_t)0x740B53FA,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x3496824F,
	(q31_t)0x74B2C883, (q31_t)0x33DEF287, (q31_t)0x7504D345,
	(q31_t)0x3326E2C2, (q31_t)0x7555BD4B, (q31_t)0x326E54C7,
	(q31_t)0x75A585CF, (q31_t)0x31B54A5D, (q31_t)0x75F42C0A,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x3041C760,
	(q31_t)0x768E0EA5, (q31_t)0x2F875262, (q31_t)0x76D94988,
	(q31_t)0x2ECC681E, (q31_t)0x77235F2D, (q31_t)0x2E110A62,
	(q31_t)0x776C4EDB, (q31_t)0x2D553AFB, (q31_t)0x77B417DF,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2BDC4E6F,
	(q31_t)0x78403328, (q31_t)0x2B1F34EB, (q31_t)0x78848413,
	(q31_t)0x2A61B101, (q31_t)0x78C7ABA1, (q31_t)0x29A3C484,
	(q31_t)0x7909A92C, (q31_t)0x28E5714A, (q31_t)0x794A7C11,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x27679DF4,
	(q31_t)0x79C89F6D, (q31_t)0x26A82185, (q31_t)0x7A05EEAD,
	(q31_t)0x25E845B5, (q31_t)0x7A4210D8, (q31_t)0x25280C5D,
	(q31_t)0x7A7D055B, (q31_t)0x24677757, (q31_t)0x7AB6CBA3,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x22E541AE,
	(q31_t)0x7B26CB4F, (q31_t)0x2223A4C5, (q31_t)0x7B5D039D,
	(q31_t)0x2161B39F, (q31_t)0x7B920B89, (q31_t)0x209F701C,
	(q31_t)0x7BC5E28F, (q31_t)0x1FDCDC1A, (q31_t)0x7BF88830,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1E56CA1E,
	(q31_t)0x7C5A3D4F, (q31_t)0x1D934FE5, (q31_t)0x7C894BDD,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CB72724, (q31_t)0x1C0B826A,
	(q31_t)0x7CE3CEB1, (q31_t)0x1B4732EF, (q31_t)0x7D0F4218,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x19BDCBF2,
	(q31_t)0x7D628AC5, (q31_t)0x18F8B83C, (q31_t)0x7D8A5F3F,
	(q31_t)0x183366E8, (q31_t)0x7DB0FDF7, (q31_t)0x176DD9DE,
	(q31_t)0x7DD6668E, (q31_t)0x16A81305, (q31_t)0x7DFA98A7,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x151BDF85,
	(q31_t)0x7E3F57FE, (q31_t)0x145576B1, (q31_t)0x7E5FE493,
	(q31_t)0x138EDBB0, (q31_t)0x7E7F3956, (q31_t)0x12C8106E,
	(q31_t)0x7E9D55FC, (q31_t)0x120116D4, (q31_t)0x7EBA3A39,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x1072A047,
	(q31_t)0x7EF0585F, (q31_t)0x0FAB272B, (q31_t)0x7F0991C3,
	(q31_t)0x0EE38765, (q31_t)0x7F2191B4, (q31_t)0x0E1BC2E3,
	(q31_t)0x7F3857F5, (q31_t)0x0D53DB92, (q31_t)0x7F4DE450,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0BC3AC35,
	(q31_t)0x7F754E7F, (q31_t)0x0AFB6805, (q31_t)0x7F872BF3,
	(q31_t)0x0A3308BC, (q31_t)0x7F97CEBC, (q31_t)0x096A9049,
	(q31_t)0x7FA736B4, (q31_t)0x08A2009A, (q31_t)0x7FB563B2,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x0710A344,
	(q31_t)0x7FCE0C3E, (q31_t)0x0647D97C, (q31_t)0x7FD8878D,
	(q31_t)0x057F0034, (q31_t)0x7FE1C76B, (q31_t)0x04B6195D,
	(q31_t)0x7FE9CBC0, (q31_t)0x03ED26E6, (q31_t)0x7FF09477,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x025B26D7,
	(q31_t)0x7FFA72D1, (q31_t)0x01921D1F, (q31_t)0x7FFD885A,
	(q31_t)0x00C90F88, (q31_t)0x7FFF6216, (q31_t)0x00000000,
	(q31_t)0x7FFFFFFF, (q31_t)0xFF36F078, (q31_t)0x7FFF6216,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFDA4D928,
	(q31_t)0x7FFA72D1, (q31_t)0xFCDBD541, (q31_t)0x7FF62182,
	(q31_t)0xFC12D919, (q31_t)0x7FF09477, (q31_t)0xFB49E6A2,
	(q31_t)0x7FE9CBC0, (q31_t)0xFA80FFCB, (q31_t)0x7FE1C76B,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF8EF5CBB,
	(q31_t)0x7FCE0C3E, (q31_t)0xF826A461, (q31_t)0x7FC25596,
	(q31_t)0xF75DFF65, (q31_t)0x7FB563B2, (q31_t)0xF6956FB6,
	(q31_t)0x7FA736B4, (q31_t)0xF5CCF743, (q31_t)0x7F97CEBC,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF43C53CA,
	(q31_t)0x7F754E7F, (q31_t)0xF3742CA1, (q31_t)0x7F62368F,
	(q31_t)0xF2AC246D, (q31_t)0x7F4DE450, (q31_t)0xF1E43D1C,
	(q31_t)0x7F3857F5, (q31_t)0xF11C789A, (q31_t)0x7F2191B4,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEF8D5FB8,
	(q31_t)0x7EF0585F, (q31_t)0xEEC60F31, (q31_t)0x7ED5E5C6,
	(q31_t)0xEDFEE92B, (q31_t)0x7EBA3A39, (q31_t)0xED37EF91,
	(q31_t)0x7E9D55FC, (q31_t)0xEC71244F, (q31_t)0x7E7F3956,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEAE4207A,
	(q31_t)0x7E3F57FE, (q31_t)0xEA1DEBBB, (q31_t)0x7E1D93E9,
	(q31_t)0xE957ECFB, (q31_t)0x7DFA98A7, (q31_t)0xE8922621,
	(q31_t)0x7DD6668E, (q31_t)0xE7CC9917, (q31_t)0x7DB0FDF7,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE642340D,
	(q31_t)0x7D628AC5, (q31_t)0xE57D5FDA, (q31_t)0x7D3980EC,
	(q31_t)0xE4B8CD10, (q31_t)0x7D0F4218, (q31_t)0xE3F47D95,
	(q31_t)0x7CE3CEB1, (q31_t)0xE330734C, (q31_t)0x7CB72724,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE1A935E1,
	(q31_t)0x7C5A3D4F, (q31_t)0xE0E60684, (q31_t)0x7C29FBEE,
	(q31_t)0xE02323E5, (q31_t)0x7BF88830, (q31_t)0xDF608FE3,
	(q31_t)0x7BC5E28F, (q31_t)0xDE9E4C60, (q31_t)0x7B920B89,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDD1ABE51,
	(q31_t)0x7B26CB4F, (q31_t)0xDC597781, (q31_t)0x7AEF6323,
	(q31_t)0xDB9888A8, (q31_t)0x7AB6CBA3, (q31_t)0xDAD7F3A2,
	(q31_t)0x7A7D055B, (q31_t)0xDA17BA4A, (q31_t)0x7A4210D8,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD898620C,
	(q31_t)0x79C89F6D, (q31_t)0xD7D946D7, (q31_t)0x798A23B1,
	(q31_t)0xD71A8EB5, (q31_t)0x794A7C11, (q31_t)0xD65C3B7B,
	(q31_t)0x7909A92C, (q31_t)0xD59E4EFE, (q31_t)0x78C7ABA1,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD423B190,
	(q31_t)0x78403328, (q31_t)0xD3670445, (q31_t)0x77FAB988,
	(q31_t)0xD2AAC504, (q31_t)0x77B417DF, (q31_t)0xD1EEF59E,
	(q31_t)0x776C4EDB, (q31_t)0xD13397E1, (q31_t)0x77235F2D,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xCFBE389F,
	(q31_t)0x768E0EA5, (q31_t)0xCF043AB2, (q31_t)0x7641AF3C,
	(q31_t)0xCE4AB5A2, (q31_t)0x75F42C0A, (q31_t)0xCD91AB38,
	(q31_t)0x75A585CF, (q31_t)0xCCD91D3D, (q31_t)0x7555BD4B,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCB697DB0,
	(q31_t)0x74B2C883, (q31_t)0xCAB26FA9, (q31_t)0x745F9DD1,
	(q31_t)0xC9FBE527, (q31_t)0x740B53FA, (q31_t)0xC945DFEC,
	(q31_t)0x73B5EBD0, (q31_t)0xC89061BA, (q31_t)0x735F6626,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC727016C,
	(q31_t)0x72AF05A6, (q31_t)0xC67322CD, (q31_t)0x72552C84,
	(q31_t)0xC5BFD22E, (q31_t)0x71FA3948, (q31_t)0xC50D1148,
	(q31_t)0x719E2CD2, (q31_t)0xC45AE1D7, (q31_t)0x71410804,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC2F83E2A,
	(q31_t)0x708378FE, (q31_t)0xC247CD5A, (q31_t)0x70231099,
	(q31_t)0xC197F4D3, (q31_t)0x6FC19385, (q31_t)0xC0E8B648,
	(q31_t)0x6F5F02B1, (q31_t)0xC03A1368, (q31_t)0x6EFB5F12,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBEDEA765,
	(q31_t)0x6E30E349, (q31_t)0xBE31E19B, (q31_t)0x6DCA0D14,
	(q31_t)0xBD85BE2F, (q31_t)0x6D6227FA, (q31_t)0xBCDA3ECA,
	(q31_t)0x6CF934FB, (q31_t)0xBC2F6513, (q31_t)0x6C8F351C,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBADBA943,
	(q31_t)0x6BB812D0, (q31_t)0xBA32CA70, (q31_t)0x6B4AF278,
	(q31_t)0xB98A97D8, (q31_t)0x6ADCC964, (q31_t)0xB8E31319,
	(q31_t)0x6A6D98A4, (q31_t)0xB83C3DD1, (q31_t)0x69FD614A,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB6F0A811,
	(q31_t)0x6919E320, (q31_t)0xB64BEACC, (q31_t)0x68A69E81,
	(q31_t)0xB5A7E362, (q31_t)0x683257AA, (q31_t)0xB5049368,
	(q31_t)0x67BD0FBC, (q31_t)0xB461FC70, (q31_t)0x6746C7D7,
	(q31_t)0xB3C0200C, (q31_t)0x66CF811F, (q31_t)0xB31EFFCB,
	(q31_t)0x66573CBB, (q31_t)0xB27E9D3B, (q31_t)0x65DDFBD3,
	(q31_t)0xB1DEF9E8, (q31_t)0x6563BF92, (q31_t)0xB140175B,
	(q31_t)0x64E88926, (q31_t)0xB0A1F71C, (q31_t)0x646C59BF,
	(q31_t)0xB0049AB2, (q31_t)0x63EF328F, (q31_t)0xAF6803A1,
	(q31_t)0x637114CC, (q31_t)0xAECC336B, (q31_t)0x62F201AC,
	(q31_t)0xAE312B91, (q31_t)0x6271FA69, (q31_t)0xAD96ED91,
	(q31_t)0x61F1003E, (q31_t)0xACFD7AE8, (q31_t)0x616F146B,
	(q31_t)0xAC64D510, (q31_t)0x60EC3830, (q31_t)0xABCCFD82,
	(q31_t)0x60686CCE, (q31_t)0xAB35F5B5, (q31_t)0x5FE3B38D,
	(q31_t)0xAA9FBF1D, (q31_t)0x5F5E0DB3, (q31_t)0xAA0A5B2D,
	(q31_t)0x5ED77C89, (q31_t)0xA975CB56, (q31_t)0x5E50015D,
	(q31_t)0xA8E21106, (q31_t)0x5DC79D7C, (q31_t)0xA84F2DA9,
	(q31_t)0x5D3E5236, (q31_t)0xA7BD22AB, (q31_t)0x5CB420DF,
	(q31_t)0xA72BF173, (q31_t)0x5C290ACC, (q31_t)0xA69B9B68,
	(q31_t)0x5B9D1153, (q31_t)0xA60C21ED, (q31_t)0x5B1035CF,
	(q31_t)0xA57D8666, (q31_t)0x5A82799A, (q31_t)0xA4EFCA31,
	(q31_t)0x59F3DE12, (q31_t)0xA462EEAC, (q31_t)0x59646497,
	(q31_t)0xA3D6F533, (q31_t)0x58D40E8C, (q31_t)0xA34BDF20,
	(q31_t)0x5842DD54, (q31_t)0xA2C1ADC9, (q31_t)0x57B0D256,
	(q31_t)0xA2386283, (q31_t)0x571DEEF9, (q31_t)0xA1AFFEA2,
	(q31_t)0x568A34A9, (q31_t)0xA1288376, (q31_t)0x55F5A4D2,
	(q31_t)0xA0A1F24C, (q31_t)0x556040E2, (q31_t)0xA01C4C72,
	(q31_t)0x54CA0A4A, (q31_t)0x9F979331, (q31_t)0x5433027D,
	(q31_t)0x9F13C7D0, (q31_t)0x539B2AEF, (q31_t)0x9E90EB94,
	(q31_t)0x53028517, (q31_t)0x9E0EFFC1, (q31_t)0x5269126E,
	(q31_t)0x9D8E0596, (q31_t)0x51CED46E, (q31_t)0x9D0DFE53,
	(q31_t)0x5133CC94, (q31_t)0x9C8EEB33, (q31_t)0x5097FC5E,
	(q31_t)0x9C10CD70, (q31_t)0x4FFB654D, (q31_t)0x9B93A640,
	(q31_t)0x4F5E08E3, (q31_t)0x9B1776D9, (q31_t)0x4EBFE8A4,
	(q31_t)0x9A9C406D, (q31_t)0x4E210617, (q31_t)0x9A22042C,
	(q31_t)0x4D8162C4, (q31_t)0x99A8C344, (q31_t)0x4CE10034,
	(q31_t)0x99307EE0, (q31_t)0x4C3FDFF3, (q31_t)0x98B93828,
	(q31_t)0x4B9E038F, (q31_t)0x9842F043, (q31_t)0x4AFB6C97,
	(q31_t)0x97CDA855, (q31_t)0x4A581C9D, (q31_t)0x9759617E,
	(q31_t)0x49B41533, (q31_t)0x96E61CDF, (q31_t)0x490F57EE,
	(q31_t)0x9673DB94, (q31_t)0x4869E664, (q31_t)0x96029EB5,
	(q31_t)0x47C3C22E, (q31_t)0x9592675B, (q31_t)0x471CECE6,
	(q31_t)0x9523369B, (q31_t)0x46756827, (q31_t)0x94B50D87,
	(q31_t)0x45CD358F, (q31_t)0x9447ED2F, (q31_t)0x452456BC,
	(q31_t)0x93DBD69F, (q31_t)0x447ACD50, (q31_t)0x9370CAE4,
	(q31_t)0x43D09AEC, (q31_t)0x9306CB04, (q31_t)0x4325C135,
	(q31_t)0x929DD805, (q31_t)0x427A41D0, (q31_t)0x9235F2EB,
	(q31_t)0x41CE1E64, (q31_t)0x91CF1CB6, (q31_t)0x4121589A,
	(q31_t)0x91695663, (q31_t)0x4073F21D, (q31_t)0x9104A0ED,
	(q31_t)0x3FC5EC97, (q31_t)0x90A0FD4E, (q31_t)0x3F1749B7,
	(q31_t)0x903E6C7A, (q31_t)0x3E680B2C, (q31_t)0x8FDCEF66,
	(q31_t)0x3DB832A5, (q31_t)0x8F7C8701, (q31_t)0x3D07C1D5,
	(q31_t)0x8F1D343A, (q31_t)0x3C56BA70, (q31_t)0x8EBEF7FB,
	(q31_t)0x3BA51E29, (q31_t)0x8E61D32D, (q31_t)0x3AF2EEB7,
	(q31_t)0x8E05C6B7, (q31_t)0x3A402DD1, (q31_t)0x8DAAD37B,
	(q31_t)0x398CDD32, (q31_t)0x8D50FA59, (q31_t)0x38D8FE93,
	(q31_t)0x8CF83C30, (q31_t)0x382493B0, (q31_t)0x8CA099D9,
	(q31_t)0x376F9E46, (q31_t)0x8C4A142F, (q31_t)0x36BA2013,
	(q31_t)0x8BF4AC05, (q31_t)0x36041AD9, (q31_t)0x8BA0622F,
	(q31_t)0x354D9056, (q31_t)0x8B4D377C, (q31_t)0x3496824F,
	(q31_t)0x8AFB2CBA, (q31_t)0x33DEF287, (q31_t)0x8AAA42B4,
	(q31_t)0x3326E2C2, (q31_t)0x8A5A7A30, (q31_t)0x326E54C7,
	(q31_t)0x8A0BD3F5, (q31_t)0x31B54A5D, (q31_t)0x89BE50C3,
	(q31_t)0x30FBC54D, (q31_t)0x8971F15A, (q31_t)0x3041C760,
	(q31_t)0x8926B677, (q31_t)0x2F875262, (q31_t)0x88DCA0D3,
	(q31_t)0x2ECC681E, (q31_t)0x8893B124, (q31_t)0x2E110A62,
	(q31_t)0x884BE820, (q31_t)0x2D553AFB, (q31_t)0x88054677,
	(q31_t)0x2C98FBBA, (q31_t)0x87BFCCD7, (q31_t)0x2BDC4E6F,
	(q31_t)0x877B7BEC, (q31_t)0x2B1F34EB, (q31_t)0x8738545E,
	(q31_t)0x2A61B101, (q31_t)0x86F656D3, (q31_t)0x29A3C484,
	(q31_t)0x86B583EE, (q31_t)0x28E5714A, (q31_t)0x8675DC4E,
	(q31_t)0x2826B928, (q31_t)0x86376092, (q31_t)0x27679DF4,
	(q31_t)0x85FA1152, (q31_t)0x26A82185, (q31_t)0x85BDEF27,
	(q31_t)0x25E845B5, (q31_t)0x8582FAA4, (q31_t)0x25280C5D,
	(q31_t)0x8549345C, (q31_t)0x24677757, (q31_t)0x85109CDC,
	(q31_t)0x23A6887E, (q31_t)0x84D934B0, (q31_t)0x22E541AE,
	(q31_t)0x84A2FC62, (q31_t)0x2223A4C5, (q31_t)0x846DF476,
	(q31_t)0x2161B39F, (q31_t)0x843A1D70, (q31_t)0x209F701C,
	(q31_t)0x840777CF, (q31_t)0x1FDCDC1A, (q31_t)0x83D60411,
	(q31_t)0x1F19F97B, (q31_t)0x83A5C2B0, (q31_t)0x1E56CA1E,
	(q31_t)0x8376B422, (q31_t)0x1D934FE5, (q31_t)0x8348D8DB,
	(q31_t)0x1CCF8CB3, (q31_t)0x831C314E, (q31_t)0x1C0B826A,
	(q31_t)0x82F0BDE8, (q31_t)0x1B4732EF, (q31_t)0x82C67F13,
	(q31_t)0x1A82A025, (q31_t)0x829D753A, (q31_t)0x19BDCBF2,
	(q31_t)0x8275A0C0, (q31_t)0x18F8B83C, (q31_t)0x824F0208,
	(q31_t)0x183366E8, (q31_t)0x82299971, (q31_t)0x176DD9DE,
	(q31_t)0x82056758, (q31_t)0x16A81305, (q31_t)0x81E26C16,
	(q31_t)0x15E21444, (q31_t)0x81C0A801, (q31_t)0x151BDF85,
	(q31_t)0x81A01B6C, (q31_t)0x145576B1, (q31_t)0x8180C6A9,
	(q31_t)0x138EDBB0, (q31_t)0x8162AA03, (q31_t)0x12C8106E,
	(q31_t)0x8145C5C6, (q31_t)0x120116D4, (q31_t)0x812A1A39,
	(q31_t)0x1139F0CE, (q31_t)0x810FA7A0, (q31_t)0x1072A047,
	(q31_t)0x80F66E3C, (q31_t)0x0FAB272B, (q31_t)0x80DE6E4C,
	(q31_t)0x0EE38765, (q31_t)0x80C7A80A, (q31_t)0x0E1BC2E3,
	(q31_t)0x80B21BAF, (q31_t)0x0D53DB92, (q31_t)0x809DC970,
	(q31_t)0x0C8BD35E, (q31_t)0x808AB180, (q31_t)0x0BC3AC35,
	(q31_t)0x8078D40D, (q31_t)0x0AFB6805, (q31_t)0x80683143,
	(q31_t)0x0A3308BC, (q31_t)0x8058C94C, (q31_t)0x096A9049,
	(q31_t)0x804A9C4D, (q31_t)0x08A2009A, (q31_t)0x803DAA69,
	(q31_t)0x07D95B9E, (q31_t)0x8031F3C1, (q31_t)0x0710A344,
	(q31_t)0x80277872, (q31_t)0x0647D97C, (q31_t)0x801E3894,
	(q31_t)0x057F0034, (q31_t)0x80163440, (q31_t)0x04B6195D,
	(q31_t)0x800F6B88, (q31_t)0x03ED26E6, (q31_t)0x8009DE7D,
	(q31_t)0x03242ABF, (q31_t)0x80058D2E, (q31_t)0x025B26D7,
	(q31_t)0x800277A5, (q31_t)0x01921D1F, (q31_t)0x80009DE9,
	(q31_t)0x00C90F88, (q31_t)0x80000000, (q31_t)0x00000000,
	(q31_t)0x80009DE9, (q31_t)0xFF36F078, (q31_t)0x800277A5,
	(q31_t)0xFE6DE2E0, (q31_t)0x80058D2E, (q31_t)0xFDA4D928,
	(q31_t)0x8009DE7D, (q31_t)0xFCDBD541, (q31_t)0x800F6B88,
	(q31_t)0xFC12D919, (q31_t)0x80163440, (q31_t)0xFB49E6A2,
	(q31_t)0x801E3894, (q31_t)0xFA80FFCB, (q31_t)0x80277872,
	(q31_t)0xF9B82683, (q31_t)0x8031F3C1, (q31_t)0xF8EF5CBB,
	(q31_t)0x803DAA69, (q31_t)0xF826A461, (q31_t)0x804A9C4D,
	(q31_t)0xF75DFF65, (q31_t)0x8058C94C, (q31_t)0xF6956FB6,
	(q31_t)0x80683143, (q31_t)0xF5CCF743, (q31_t)0x8078D40D,
	(q31_t)0xF50497FA, (q31_t)0x808AB180, (q31_t)0xF43C53CA,
	(q31_t)0x809DC970, (q31_t)0xF3742CA1, (q31_t)0x80B21BAF,
	(q31_t)0xF2AC246D, (q31_t)0x80C7A80A, (q31_t)0xF1E43D1C,
	(q31_t)0x80DE6E4C, (q31_t)0xF11C789A, (q31_t)0x80F66E3C,
	(q31_t)0xF054D8D4, (q31_t)0x810FA7A0, (q31_t)0xEF8D5FB8,
	(q31_t)0x812A1A39, (q31_t)0xEEC60F31, (q31_t)0x8145C5C6,
	(q31_t)0xEDFEE92B, (q31_t)0x8162AA03, (q31_t)0xED37EF91,
	(q31_t)0x8180C6A9, (q31_t)0xEC71244F, (q31_t)0x81A01B6C,
	(q31_t)0xEBAA894E, (q31_t)0x81C0A801, (q31_t)0xEAE4207A,
	(q31_t)0x81E26C16, (q31_t)0xEA1DEBBB, (q31_t)0x82056758,
	(q31_t)0xE957ECFB, (q31_t)0x82299971, (q31_t)0xE8922621,
	(q31_t)0x824F0208, (q31_t)0xE7CC9917, (q31_t)0x8275A0C0,
	(q31_t)0xE70747C3, (q31_t)0x829D753A, (q31_t)0xE642340D,
	(q31_t)0x82C67F13, (q31_t)0xE57D5FDA, (q31_t)0x82F0BDE8,
	(q31_t)0xE4B8CD10, (q31_t)0x831C314E, (q31_t)0xE3F47D95,
	(q31_t)0x8348D8DB, (q31_t)0xE330734C, (q31_t)0x8376B422,
	(q31_t)0xE26CB01A, (q31_t)0x83A5C2B0, (q31_t)0xE1A935E1,
	(q31_t)0x83D60411, (q31_t)0xE0E60684, (q31_t)0x840777CF,
	(q31_t)0xE02323E5, (q31_t)0x843A1D70, (q31_t)0xDF608FE3,
	(q31_t)0x846DF476, (q31_t)0xDE9E4C60, (q31_t)0x84A2FC62,
	(q31_t)0xDDDC5B3A, (q31_t)0x84D934B0, (q31_t)0xDD1ABE51,
	(q31_t)0x85109CDC, (q31_t)0xDC597781, (q31_t)0x8549345C,
	(q31_t)0xDB9888A8, (q31_t)0x8582FAA4, (q31_t)0xDAD7F3A2,
	(q31_t)0x85BDEF27, (q31_t)0xDA17BA4A, (q31_t)0x85FA1152,
	(q31_t)0xD957DE7A, (q31_t)0x86376092, (q31_t)0xD898620C,
	(q31_t)0x8675DC4E, (q31_t)0xD7D946D7, (q31_t)0x86B583EE,
	(q31_t)0xD71A8EB5, (q31_t)0x86F656D3, (q31_t)0xD65C3B7B,
	(q31_t)0x8738545E, (q31_t)0xD59E4EFE, (q31_t)0x877B7BEC,
	(q31_t)0xD4E0CB14, (q31_t)0x87BFCCD7, (q31_t)0xD423B190,
	(q31_t)0x88054677, (q31_t)0xD3670445, (q31_t)0x884BE820,
	(q31_t)0xD2AAC504, (q31_t)0x8893B124, (q31_t)0xD1EEF59E,
	(q31_t)0x88DCA0D3, (q31_t)0xD13397E1, (q31_t)0x8926B677,
	(q31_t)0xD078AD9D, (q31_t)0x8971F15A, (q31_t)0xCFBE389F,
	(q31_t)0x89BE50C3, (q31_t)0xCF043AB2, (q31_t)0x8A0BD3F5,
	(q31_t)0xCE4AB5A2, (q31_t)0x8A5A7A30, (q31_t)0xCD91AB38,
	(q31_t)0x8AAA42B4, (q31_t)0xCCD91D3D, (q31_t)0x8AFB2CBA,
	(q31_t)0xCC210D78, (q31_t)0x8B4D377C, (q31_t)0xCB697DB0,
	(q31_t)0x8BA0622F, (q31_t)0xCAB26FA9, (q31_t)0x8BF4AC05,
	(q31_t)0xC9FBE527, (q31_t)0x8C4A142F, (q31_t)0xC945DFEC,
	(q31_t)0x8CA099D9, (q31_t)0xC89061BA, (q31_t)0x8CF83C30,
	(q31_t)0xC7DB6C50, (q31_t)0x8D50FA59, (q31_t)0xC727016C,
	(q31_t)0x8DAAD37B, (q31_t)0xC67322CD, (q31_t)0x8E05C6B7,
	(q31_t)0xC5BFD22E, (q31_t)0x8E61D32D, (q31_t)0xC50D1148,
	(q31_t)0x8EBEF7FB, (q31_t)0xC45AE1D7, (q31_t)0x8F1D343A,
	(q31_t)0xC3A9458F, (q31_t)0x8F7C8701, (q31_t)0xC2F83E2A,
	(q31_t)0x8FDCEF66, (q31_t)0xC247CD5A, (q31_t)0x903E6C7A,
	(q31_t)0xC197F4D3, (q31_t)0x90A0FD4E, (q31_t)0xC0E8B648,
	(q31_t)0x9104A0ED, (q31_t)0xC03A1368, (q31_t)0x91695663,
	(q31_t)0xBF8C0DE2, (q31_t)0x91CF1CB6, (q31_t)0xBEDEA765,
	(q31_t)0x9235F2EB, (q31_t)0xBE31E19B, (q31_t)0x929DD805,
	(q31_t)0xBD85BE2F, (q31_t)0x9306CB04, (q31_t)0xBCDA3ECA,
	(q31_t)0x9370CAE4, (q31_t)0xBC2F6513, (q31_t)0x93DBD69F,
	(q31_t)0xBB8532AF, (q31_t)0x9447ED2F, (q31_t)0xBADBA943,
	(q31_t)0x94B50D87, (q31_t)0xBA32CA70, (q31_t)0x9523369B,
	(q31_t)0xB98A97D8, (q31_t)0x9592675B, (q31_t)0xB8E31319,
	(q31_t)0x96029EB5, (q31_t)0xB83C3DD1, (q31_t)0x9673DB94,
	(q31_t)0xB796199B, (q31_t)0x96E61CDF, (q31_t)0xB6F0A811,
	(q31_t)0x9759617E, (q31_t)0xB64BEACC, (q31_t)0x97CDA855,
	(q31_t)0xB5A7E362, (q31_t)0x9842F043, (q31_t)0xB5049368,
	(q31_t)0x98B93828, (q31_t)0xB461FC70, (q31_t)0x99307EE0,
	(q31_t)0xB3C0200C, (q31_t)0x99A8C344, (q31_t)0xB31EFFCB,
	(q31_t)0x9A22042C, (q31_t)0xB27E9D3B, (q31_t)0x9A9C406D,
	(q31_t)0xB1DEF9E8, (q31_t)0x9B1776D9, (q31_t)0xB140175B,
	(q31_t)0x9B93A640, (q31_t)0xB0A1F71C, (q31_t)0x9C10CD70,
	(q31_t)0xB0049AB2, (q31_t)0x9C8EEB33, (q31_t)0xAF6803A1,
	(q31_t)0x9D0DFE53, (q31_t)0xAECC336B, (q31_t)0x9D8E0596,
	(q31_t)0xAE312B91, (q31_t)0x9E0EFFC1, (q31_t)0xAD96ED91,
	(q31_t)0x9E90EB94, (q31_t)0xACFD7AE8, (q31_t)0x9F13C7D0,
	(q31_t)0xAC64D510, (q31_t)0x9F979331, (q31_t)0xABCCFD82,
	(q31_t)0xA01C4C72, (q31_t)0xAB35F5B5, (q31_t)0xA0A1F24C,
	(q31_t)0xAA9FBF1D, (q31_t)0xA1288376, (q31_t)0xAA0A5B2D,
	(q31_t)0xA1AFFEA2, (q31_t)0xA975CB56, (q31_t)0xA2386283,
	(q31_t)0xA8E21106, (q31_t)0xA2C1ADC9, (q31_t)0xA84F2DA9,
	(q31_t)0xA34BDF20, (q31_t)0xA7BD22AB, (q31_t)0xA3D6F533,
	(q31_t)0xA72BF173, (q31_t)0xA462EEAC, (q31_t)0xA69B9B68,
	(q31_t)0xA4EFCA31, (q31_t)0xA60C21ED, (q31_t)0xA57D8666,
	(q31_t)0xA57D8666, (q31_t)0xA60C21ED, (q31_t)0xA4EFCA31,
	(q31_t)0xA69B9B68, (q31_t)0xA462EEAC, (q31_t)0xA72BF173,
	(q31_t)0xA3D6F533, (q31_t)0xA7BD22AB, (q31_t)0xA34BDF20,
	(q31_t)0xA84F2DA9, (q31_t)0xA2C1ADC9, (q31_t)0xA8E21106,
	(q31_t)0xA2386283, (q31_t)0xA975CB56, (q31_t)0xA1AFFEA2,
	(q31_t)0xAA0A5B2D, (q31_t)0xA1288376, (q31_t)0xAA9FBF1D,
	(q31_t)0xA0A1F24C, (q31_t)0xAB35F5B5, (q31_t)0xA01C4C72,
	(q31_t)0xABCCFD82, (q31_t)0x9F979331, (q31_t)0xAC64D510,
	(q31_t)0x9F13C7D0, (q31_t)0xACFD7AE8, (q31_t)0x9E90EB94,
	(q31_t)0xAD96ED91, (q31_t)0x9E0EFFC1, (q31_t)0xAE312B91,
	(q31_t)0x9D8E0596, (q31_t)0xAECC336B, (q31_t)0x9D0DFE53,
	(q31_t)0xAF6803A1, (q31_t)0x9C8EEB33, (q31_t)0xB0049AB2,
	(q31_t)0x9C10CD70, (q31_t)0xB0A1F71C, (q31_t)0x9B93A640,
	(q31_t)0xB140175B, (q31_t)0x9B1776D9, (q31_t)0xB1DEF9E8,
	(q31_t)0x9A9C406D, (q31_t)0xB27E9D3B, (q31_t)0x9A22042C,
	(q31_t)0xB31EFFCB, (q31_t)0x99A8C344, (q31_t)0xB3C0200C,
	(q31_t)0x99307EE0, (q31_t)0xB461FC70, (q31_t)0x98B93828,
	(q31_t)0xB5049368, (q31_t)0x9842F043, (q31_t)0xB5A7E362,
	(q31_t)0x97CDA855, (q31_t)0xB64BEACC, (q31_t)0x9759617E,
	(q31_t)0xB6F0A811, (q31_t)0x96E61CDF, (q31_t)0xB796199B,
	(q31_t)0x9673DB94, (q31_t)0xB83C3DD1, (q31_t)0x96029EB5,
	(q31_t)0xB8E31319, (q31_t)0x9592675B, (q31_t)0xB98A97D8,
	(q31_t)0x9523369B, (q31_t)0xBA32CA70, (q31_t)0x94B50D87,
	(q31_t)0xBADBA943, (q31_t)0x9447ED2F, (q31_t)0xBB8532AF,
	(q31_t)0x93DBD69F, (q31_t)0xBC2F6513, (q31_t)0x9370CAE4,
	(q31_t)0xBCDA3ECA, (q31_t)0x9306CB04, (q31_t)0xBD85BE2F,
	(q31_t)0x929DD805, (q31_t)0xBE31E19B, (q31_t)0x9235F2EB,
	(q31_t)0xBEDEA765, (q31_t)0x91CF1CB6, (q31_t)0xBF8C0DE2,
	(q31_t)0x91695663, (q31_t)0xC03A1368, (q31_t)0x9104A0ED,
	(q31_t)0xC0E8B648, (q31_t)0x90A0FD4E, (q31_t)0xC197F4D3,
	(q31_t)0x903E6C7A, (q31_t)0xC247CD5A, (q31_t)0x8FDCEF66,
	(q31_t)0xC2F83E2A, (q31_t)0x8F7C8701, (q31_t)0xC3A9458F,
	(q31_t)0x8F1D343A, (q31_t)0xC45AE1D7, (q31_t)0x8EBEF7FB,
	(q31_t)0xC50D1148, (q31_t)0x8E61D32D, (q31_t)0xC5BFD22E,
	(q31_t)0x8E05C6B7, (q31_t)0xC67322CD, (q31_t)0x8DAAD37B,
	(q31_t)0xC727016C, (q31_t)0x8D50FA59, (q31_t)0xC7DB6C50,
	(q31_t)0x8CF83C30, (q31_t)0xC89061BA, (q31_t)0x8CA099D9,
	(q31_t)0xC945DFEC, (q31_t)0x8C4A142F, (q31_t)0xC9FBE527,
	(q31_t)0x8BF4AC05, (q31_t)0xCAB26FA9, (q31_t)0x8BA0622F,
	(q31_t)0xCB697DB0, (q31_t)0x8B4D377C, (q31_t)0xCC210D78,
	(q31_t)0x8AFB2CBA, (q31_t)0xCCD91D3D, (q31_t)0x8AAA42B4,
	(q31_t)0xCD91AB38, (q31_t)0x8A5A7A30, (q31_t)0xCE4AB5A2,
	(q31_t)0x8A0BD3F5, (q31_t)0xCF043AB2, (q31_t)0x89BE50C3,
	(q31_t)0xCFBE389F, (q31_t)0x8971F15A, (q31_t)0xD078AD9D,
	(q31_t)0x8926B677, (q31_t)0xD13397E1, (q31_t)0x88DCA0D3,
	(q31_t)0xD1EEF59E, (q31_t)0x8893B124, (q31_t)0xD2AAC504,
	(q31_t)0x884BE820, (q31_t)0xD3670445, (q31_t)0x88054677,
	(q31_t)0xD423B190, (q31_t)0x87BFCCD7, (q31_t)0xD4E0CB14,
	(q31_t)0x877B7BEC, (q31_t)0xD59E4EFE, (q31_t)0x8738545E,
	(q31_t)0xD65C3B7B, (q31_t)0x86F656D3, (q31_t)0xD71A8EB5,
	(q31_t)0x86B583EE, (q31_t)0xD7D946D7, (q31_t)0x8675DC4E,
	(q31_t)0xD898620C, (q31_t)0x86376092, (q31_t)0xD957DE7A,
	(q31_t)0x85FA1152, (q31_t)0xDA17BA4A, (q31_t)0x85BDEF27,
	(q31_t)0xDAD7F3A2, (q31_t)0x8582FAA4, (q31_t)0xDB9888A8,
	(q31_t)0x8549345C, (q31_t)0xDC597781, (q31_t)0x85109CDC,
	(q31_t)0xDD1ABE51, (q31_t)0x84D934B0, (q31_t)0xDDDC5B3A,
	(q31_t)0x84A2FC62, (q31_t)0xDE9E4C60, (q31_t)0x846DF476,
	(q31_t)0xDF608FE3, (q31_t)0x843A1D70, (q31_t)0xE02323E5,
	(q31_t)0x840777CF, (q31_t)0xE0E60684, (q31_t)0x83D60411,
	(q31_t)0xE1A935E1, (q31_t)0x83A5C2B0, (q31_t)0xE26CB01A,
	(q31_t)0x8376B422, (q31_t)0xE330734C, (q31_t)0x8348D8DB,
	(q31_t)0xE3F47D95, (q31_t)0x831C314E, (q31_t)0xE4B8CD10,
	(q31_t)0x82F0BDE8, (q31_t)0xE57D5FDA, (q31_t)0x82C67F13,
	(q31_t)0xE642340D, (q31_t)0x829D753A, (q31_t)0xE70747C3,
	(q31_t)0x8275A0C0, (q31_t)0xE7CC9917, (q31_t)0x824F0208,
	(q31_t)0xE8922621, (q31_t)0x82299971, (q31_t)0xE957ECFB,
	(q31_t)0x82056758, (q31_t)0xEA1DEBBB, (q31_t)0x81E26C16,
	(q31_t)0xEAE4207A, (q31_t)0x81C0A801, (q31_t)0xEBAA894E,
	(q31_t)0x81A01B6C, (q31_t)0xEC71244F, (q31_t)0x8180C6A9,
	(q31_t)0xED37EF91, (q31_t)0x8162AA03, (q31_t)0xEDFEE92B,
	(q31_t)0x8145C5C6, (q31_t)0xEEC60F31, (q31_t)0x812A1A39,
	(q31_t)0xEF8D5FB8, (q31_t)0x810FA7A0, (q31_t)0xF054D8D4,
	(q31_t)0x80F66E3C, (q31_t)0xF11C789A, (q31_t)0x80DE6E4C,
	(q31_t)0xF1E43D1C, (q31_t)0x80C7A80A, (q31_t)0xF2AC246D,
	(q31_t)0x80B21BAF, (q31_t)0xF3742CA1, (q31_t)0x809DC970,
	(q31_t)0xF43C53CA, (q31_t)0x808AB180, (q31_t)0xF50497FA,
	(q31_t)0x8078D40D, (q31_t)0xF5CCF743, (q31_t)0x80683143,
	(q31_t)0xF6956FB6, (q31_t)0x8058C94C, (q31_t)0xF75DFF65,
	(q31_t)0x804A9C4D, (q31_t)0xF826A461, (q31_t)0x803DAA69,
	(q31_t)0xF8EF5CBB, (q31_t)0x8031F3C1, (q31_t)0xF9B82683,
	(q31_t)0x80277872, (q31_t)0xFA80FFCB, (q31_t)0x801E3894,
	(q31_t)0xFB49E6A2, (q31_t)0x80163440, (q31_t)0xFC12D919,
	(q31_t)0x800F6B88, (q31_t)0xFCDBD541, (q31_t)0x8009DE7D,
	(q31_t)0xFDA4D928, (q31_t)0x80058D2E, (q31_t)0xFE6DE2E0,
	(q31_t)0x800277A5, (q31_t)0xFF36F078, (q31_t)0x80009DE9
};

#endif /* !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) */

#if !defined(ARM_DSP_CONFIG_TABLES) || defined(ARM_ALL_FFT_TABLES) || defined(ARM_TABLE_TWIDDLECOEF_Q31_2048)
/**
  @par
  Example code for Q31 Twiddle factors Generation::
  @par
  <pre>for (i = 0; i< 3N/4; i++)
  {
     twiddleCoefQ31[2*i]   = cos(i * 2*PI/(float)N);
     twiddleCoefQ31[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 2048, PI = 3.14159265358979
  @par
  Cos and Sin values are interleaved fashion
  @par
  Convert Floating point to Q31(Fixed point 1.31):
 	round(twiddleCoefQ31(i) * pow(2, 31))
 */
const q31_t twiddleCoef_2048_q31[3072] = {
	(q31_t)0x7FFFFFFF, (q31_t)0x00000000, (q31_t)0x7FFFD885,
	(q31_t)0x006487E3, (q31_t)0x7FFF6216, (q31_t)0x00C90F88,
	(q31_t)0x7FFE9CB2, (q31_t)0x012D96B0, (q31_t)0x7FFD885A,
	(q31_t)0x01921D1F, (q31_t)0x7FFC250F, (q31_t)0x01F6A296,
	(q31_t)0x7FFA72D1, (q31_t)0x025B26D7, (q31_t)0x7FF871A1,
	(q31_t)0x02BFA9A4, (q31_t)0x7FF62182, (q31_t)0x03242ABF,
	(q31_t)0x7FF38273, (q31_t)0x0388A9E9, (q31_t)0x7FF09477,
	(q31_t)0x03ED26E6, (q31_t)0x7FED5790, (q31_t)0x0451A176,
	(q31_t)0x7FE9CBC0, (q31_t)0x04B6195D, (q31_t)0x7FE5F108,
	(q31_t)0x051A8E5C, (q31_t)0x7FE1C76B, (q31_t)0x057F0034,
	(q31_t)0x7FDD4EEC, (q31_t)0x05E36EA9, (q31_t)0x7FD8878D,
	(q31_t)0x0647D97C, (q31_t)0x7FD37152, (q31_t)0x06AC406F,
	(q31_t)0x7FCE0C3E, (q31_t)0x0710A344, (q31_t)0x7FC85853,
	(q31_t)0x077501BE, (q31_t)0x7FC25596, (q31_t)0x07D95B9E,
	(q31_t)0x7FBC040A, (q31_t)0x083DB0A7, (q31_t)0x7FB563B2,
	(q31_t)0x08A2009A, (q31_t)0x7FAE7494, (q31_t)0x09064B3A,
	(q31_t)0x7FA736B4, (q31_t)0x096A9049, (q31_t)0x7F9FAA15,
	(q31_t)0x09CECF89, (q31_t)0x7F97CEBC, (q31_t)0x0A3308BC,
	(q31_t)0x7F8FA4AF, (q31_t)0x0A973BA5, (q31_t)0x7F872BF3,
	(q31_t)0x0AFB6805, (q31_t)0x7F7E648B, (q31_t)0x0B5F8D9F,
	(q31_t)0x7F754E7F, (q31_t)0x0BC3AC35, (q31_t)0x7F6BE9D4,
	(q31_t)0x0C27C389, (q31_t)0x7F62368F, (q31_t)0x0C8BD35E,
	(q31_t)0x7F5834B6, (q31_t)0x0CEFDB75, (q31_t)0x7F4DE450,
	(q31_t)0x0D53DB92, (q31_t)0x7F434563, (q31_t)0x0DB7D376,
	(q31_t)0x7F3857F5, (q31_t)0x0E1BC2E3, (q31_t)0x7F2D1C0E,
	(q31_t)0x0E7FA99D, (q31_t)0x7F2191B4, (q31_t)0x0EE38765,
	(q31_t)0x7F15B8EE, (q31_t)0x0F475BFE, (q31_t)0x7F0991C3,
	(q31_t)0x0FAB272B, (q31_t)0x7EFD1C3C, (q31_t)0x100EE8AD,
	(q31_t)0x7EF0585F, (q31_t)0x1072A047, (q31_t)0x7EE34635,
	(q31_t)0x10D64DBC, (q31_t)0x7ED5E5C6, (q31_t)0x1139F0CE,
	(q31_t)0x7EC8371A, (q31_t)0x119D8940, (q31_t)0x7EBA3A39,
	(q31_t)0x120116D4, (q31_t)0x7EABEF2C, (q31_t)0x1264994E,
	(q31_t)0x7E9D55FC, (q31_t)0x12C8106E, (q31_t)0x7E8E6EB1,
	(q31_t)0x132B7BF9, (q31_t)0x7E7F3956, (q31_t)0x138EDBB0,
	(q31_t)0x7E6FB5F3, (q31_t)0x13F22F57, (q31_t)0x7E5FE493,
	(q31_t)0x145576B1, (q31_t)0x7E4FC53E, (q31_t)0x14B8B17F,
	(q31_t)0x7E3F57FE, (q31_t)0x151BDF85, (q31_t)0x7E2E9CDF,
	(q31_t)0x157F0086, (q31_t)0x7E1D93E9, (q31_t)0x15E21444,
	(q31_t)0x7E0C3D29, (q31_t)0x16451A83, (q31_t)0x7DFA98A7,
	(q31_t)0x16A81305, (q31_t)0x7DE8A670, (q31_t)0x170AFD8D,
	(q31_t)0x7DD6668E, (q31_t)0x176DD9DE, (q31_t)0x7DC3D90D,
	(q31_t)0x17D0A7BB, (q31_t)0x7DB0FDF7, (q31_t)0x183366E8,
	(q31_t)0x7D9DD55A, (q31_t)0x18961727, (q31_t)0x7D8A5F3F,
	(q31_t)0x18F8B83C, (q31_t)0x7D769BB5, (q31_t)0x195B49E9,
	(q31_t)0x7D628AC5, (q31_t)0x19BDCBF2, (q31_t)0x7D4E2C7E,
	(q31_t)0x1A203E1B, (q31_t)0x7D3980EC, (q31_t)0x1A82A025,
	(q31_t)0x7D24881A, (q31_t)0x1AE4F1D6, (q31_t)0x7D0F4218,
	(q31_t)0x1B4732EF, (q31_t)0x7CF9AEF0, (q31_t)0x1BA96334,
	(q31_t)0x7CE3CEB1, (q31_t)0x1C0B826A, (q31_t)0x7CCDA168,
	(q31_t)0x1C6D9053, (q31_t)0x7CB72724, (q31_t)0x1CCF8CB3,
	(q31_t)0x7CA05FF1, (q31_t)0x1D31774D, (q31_t)0x7C894BDD,
	(q31_t)0x1D934FE5, (q31_t)0x7C71EAF8, (q31_t)0x1DF5163F,
	(q31_t)0x7C5A3D4F, (q31_t)0x1E56CA1E, (q31_t)0x7C4242F2,
	(q31_t)0x1EB86B46, (q31_t)0x7C29FBEE, (q31_t)0x1F19F97B,
	(q31_t)0x7C116853, (q31_t)0x1F7B7480, (q31_t)0x7BF88830,
	(q31_t)0x1FDCDC1A, (q31_t)0x7BDF5B94, (q31_t)0x203E300D,
	(q31_t)0x7BC5E28F, (q31_t)0x209F701C, (q31_t)0x7BAC1D31,
	(q31_t)0x21009C0B, (q31_t)0x7B920B89, (q31_t)0x2161B39F,
	(q31_t)0x7B77ADA8, (q31_t)0x21C2B69C, (q31_t)0x7B5D039D,
	(q31_t)0x2223A4C5, (q31_t)0x7B420D7A, (q31_t)0x22847DDF,
	(q31_t)0x7B26CB4F, (q31_t)0x22E541AE, (q31_t)0x7B0B3D2C,
	(q31_t)0x2345EFF7, (q31_t)0x7AEF6323, (q31_t)0x23A6887E,
	(q31_t)0x7AD33D45, (q31_t)0x24070B07, (q31_t)0x7AB6CBA3,
	(q31_t)0x24677757, (q31_t)0x7A9A0E4F, (q31_t)0x24C7CD32,
	(q31_t)0x7A7D055B, (q31_t)0x25280C5D, (q31_t)0x7A5FB0D8,
	(q31_t)0x2588349D, (q31_t)0x7A4210D8, (q31_t)0x25E845B5,
	(q31_t)0x7A24256E, (q31_t)0x26483F6C, (q31_t)0x7A05EEAD,
	(q31_t)0x26A82185, (q31_t)0x79E76CA6, (q31_t)0x2707EBC6,
	(q31_t)0x79C89F6D, (q31_t)0x27679DF4, (q31_t)0x79A98715,
	(q31_t)0x27C737D2, (q31_t)0x798A23B1, (q31_t)0x2826B928,
	(q31_t)0x796A7554, (q31_t)0x288621B9, (q31_t)0x794A7C11,
	(q31_t)0x28E5714A, (q31_t)0x792A37FE, (q31_t)0x2944A7A2,
	(q31_t)0x7909A92C, (q31_t)0x29A3C484, (q31_t)0x78E8CFB1,
	(q31_t)0x2A02C7B8, (q31_t)0x78C7ABA1, (q31_t)0x2A61B101,
	(q31_t)0x78A63D10, (q31_t)0x2AC08025, (q31_t)0x78848413,
	(q31_t)0x2B1F34EB, (q31_t)0x786280BF, (q31_t)0x2B7DCF17,
	(q31_t)0x78403328, (q31_t)0x2BDC4E6F, (q31_t)0x781D9B64,
	(q31_t)0x2C3AB2B9, (q31_t)0x77FAB988, (q31_t)0x2C98FBBA,
	(q31_t)0x77D78DAA, (q31_t)0x2CF72939, (q31_t)0x77B417DF,
	(q31_t)0x2D553AFB, (q31_t)0x7790583D, (q31_t)0x2DB330C7,
	(q31_t)0x776C4EDB, (q31_t)0x2E110A62, (q31_t)0x7747FBCE,
	(q31_t)0x2E6EC792, (q31_t)0x77235F2D, (q31_t)0x2ECC681E,
	(q31_t)0x76FE790E, (q31_t)0x2F29EBCC, (q31_t)0x76D94988,
	(q31_t)0x2F875262, (q31_t)0x76B3D0B3, (q31_t)0x2FE49BA6,
	(q31_t)0x768E0EA5, (q31_t)0x3041C760, (q31_t)0x76680376,
	(q31_t)0x309ED555, (q31_t)0x7641AF3C, (q31_t)0x30FBC54D,
	(q31_t)0x761B1211, (q31_t)0x3158970D, (q31_t)0x75F42C0A,
	(q31_t)0x31B54A5D, (q31_t)0x75CCFD42, (q31_t)0x3211DF03,
	(q31_t)0x75A585CF, (q31_t)0x326E54C7, (q31_t)0x757DC5CA,
	(q31_t)0x32CAAB6F, (q31_t)0x7555BD4B, (q31_t)0x3326E2C2,
	(q31_t)0x752D6C6C, (q31_t)0x3382FA88, (q31_t)0x7504D345,
	(q31_t)0x33DEF287, (q31_t)0x74DBF1EF, (q31_t)0x343ACA87,
	(q31_t)0x74B2C883, (q31_t)0x3496824F, (q31_t)0x7489571B,
	(q31_t)0x34F219A7, (q31_t)0x745F9DD1, (q31_t)0x354D9056,
	(q31_t)0x74359CBD, (q31_t)0x35A8E624, (q31_t)0x740B53FA,
	(q31_t)0x36041AD9, (q31_t)0x73E0C3A3, (q31_t)0x365F2E3B,
	(q31_t)0x73B5EBD0, (q31_t)0x36BA2013, (q31_t)0x738ACC9E,
	(q31_t)0x3714F02A, (q31_t)0x735F6626, (q31_t)0x376F9E46,
	(q31_t)0x7333B883, (q31_t)0x37CA2A30, (q31_t)0x7307C3D0,
	(q31_t)0x382493B0, (q31_t)0x72DB8828, (q31_t)0x387EDA8E,
	(q31_t)0x72AF05A6, (q31_t)0x38D8FE93, (q31_t)0x72823C66,
	(q31_t)0x3932FF87, (q31_t)0x72552C84, (q31_t)0x398CDD32,
	(q31_t)0x7227D61C, (q31_t)0x39E6975D, (q31_t)0x71FA3948,
	(q31_t)0x3A402DD1, (q31_t)0x71CC5626, (q31_t)0x3A99A057,
	(q31_t)0x719E2CD2, (q31_t)0x3AF2EEB7, (q31_t)0x716FBD68,
	(q31_t)0x3B4C18BA, (q31_t)0x71410804, (q31_t)0x3BA51E29,
	(q31_t)0x71120CC5, (q31_t)0x3BFDFECD, (q31_t)0x70E2CBC6,
	(q31_t)0x3C56BA70, (q31_t)0x70B34524, (q31_t)0x3CAF50DA,
	(q31_t)0x708378FE, (q31_t)0x3D07C1D5, (q31_t)0x70536771,
	(q31_t)0x3D600D2B, (q31_t)0x70231099, (q31_t)0x3DB832A5,
	(q31_t)0x6FF27496, (q31_t)0x3E10320D, (q31_t)0x6FC19385,
	(q31_t)0x3E680B2C, (q31_t)0x6F906D84, (q31_t)0x3EBFBDCC,
	(q31_t)0x6F5F02B1, (q31_t)0x3F1749B7, (q31_t)0x6F2D532C,
	(q31_t)0x3F6EAEB8, (q31_t)0x6EFB5F12, (q31_t)0x3FC5EC97,
	(q31_t)0x6EC92682, (q31_t)0x401D0320, (q31_t)0x6E96A99C,
	(q31_t)0x4073F21D, (q31_t)0x6E63E87F, (q31_t)0x40CAB957,
	(q31_t)0x6E30E349, (q31_t)0x4121589A, (q31_t)0x6DFD9A1B,
	(q31_t)0x4177CFB0, (q31_t)0x6DCA0D14, (q31_t)0x41CE1E64,
	(q31_t)0x6D963C54, (q31_t)0x42244480, (q31_t)0x6D6227FA,
	(q31_t)0x427A41D0, (q31_t)0x6D2DD027, (q31_t)0x42D0161E,
	(q31_t)0x6CF934FB, (q31_t)0x4325C135, (q31_t)0x6CC45697,
	(q31_t)0x437B42E1, (q31_t)0x6C8F351C, (q31_t)0x43D09AEC,
	(q31_t)0x6C59D0A9, (q31_t)0x4425C923, (q31_t)0x6C242960,
	(q31_t)0x447ACD50, (q31_t)0x6BEE3F62, (q31_t)0x44CFA73F,
	(q31_t)0x6BB812D0, (q31_t)0x452456BC, (q31_t)0x6B81A3CD,
	(q31_t)0x4578DB93, (q31_t)0x6B4AF278, (q31_t)0x45CD358F,
	(q31_t)0x6B13FEF5, (q31_t)0x4621647C, (q31_t)0x6ADCC964,
	(q31_t)0x46756827, (q31_t)0x6AA551E8, (q31_t)0x46C9405C,
	(q31_t)0x6A6D98A4, (q31_t)0x471CECE6, (q31_t)0x6A359DB9,
	(q31_t)0x47706D93, (q31_t)0x69FD614A, (q31_t)0x47C3C22E,
	(q31_t)0x69C4E37A, (q31_t)0x4816EA85, (q31_t)0x698C246C,
	(q31_t)0x4869E664, (q31_t)0x69532442, (q31_t)0x48BCB598,
	(q31_t)0x6919E320, (q31_t)0x490F57EE, (q31_t)0x68E06129,
	(q31_t)0x4961CD32, (q31_t)0x68A69E81, (q31_t)0x49B41533,
	(q31_t)0x686C9B4B, (q31_t)0x4A062FBD, (q31_t)0x683257AA,
	(q31_t)0x4A581C9D, (q31_t)0x67F7D3C4, (q31_t)0x4AA9DBA1,
	(q31_t)0x67BD0FBC, (q31_t)0x4AFB6C97, (q31_t)0x67820BB6,
	(q31_t)0x4B4CCF4D, (q31_t)0x6746C7D7, (q31_t)0x4B9E038F,
	(q31_t)0x670B4443, (q31_t)0x4BEF092D, (q31_t)0x66CF811F,
	(q31_t)0x4C3FDFF3, (q31_t)0x66937E90, (q31_t)0x4C9087B1,
	(q31_t)0x66573CBB, (q31_t)0x4CE10034, (q31_t)0x661ABBC5,
	(q31_t)0x4D31494B, (q31_t)0x65DDFBD3, (q31_t)0x4D8162C4,
	(q31_t)0x65A0FD0B, (q31_t)0x4DD14C6E, (q31_t)0x6563BF92,
	(q31_t)0x4E210617, (q31_t)0x6526438E, (q31_t)0x4E708F8F,
	(q31_t)0x64E88926, (q31_t)0x4EBFE8A4, (q31_t)0x64AA907F,
	(q31_t)0x4F0F1126, (q31_t)0x646C59BF, (q31_t)0x4F5E08E3,
	(q31_t)0x642DE50D, (q31_t)0x4FACCFAB, (q31_t)0x63EF328F,
	(q31_t)0x4FFB654D, (q31_t)0x63B0426D, (q31_t)0x5049C999,
	(q31_t)0x637114CC, (q31_t)0x5097FC5E, (q31_t)0x6331A9D4,
	(q31_t)0x50E5FD6C, (q31_t)0x62F201AC, (q31_t)0x5133CC94,
	(q31_t)0x62B21C7B, (q31_t)0x518169A4, (q31_t)0x6271FA69,
	(q31_t)0x51CED46E, (q31_t)0x62319B9D, (q31_t)0x521C0CC1,
	(q31_t)0x61F1003E, (q31_t)0x5269126E, (q31_t)0x61B02876,
	(q31_t)0x52B5E545, (q31_t)0x616F146B, (q31_t)0x53028517,
	(q31_t)0x612DC446, (q31_t)0x534EF1B5, (q31_t)0x60EC3830,
	(q31_t)0x539B2AEF, (q31_t)0x60AA704F, (q31_t)0x53E73097,
	(q31_t)0x60686CCE, (q31_t)0x5433027D, (q31_t)0x60262DD5,
	(q31_t)0x547EA073, (q31_t)0x5FE3B38D, (q31_t)0x54CA0A4A,
	(q31_t)0x5FA0FE1E, (q31_t)0x55153FD4, (q31_t)0x5F5E0DB3,
	(q31_t)0x556040E2, (q31_t)0x5F1AE273, (q31_t)0x55AB0D46,
	(q31_t)0x5ED77C89, (q31_t)0x55F5A4D2, (q31_t)0x5E93DC1F,
	(q31_t)0x56400757, (q31_t)0x5E50015D, (q31_t)0x568A34A9,
	(q31_t)0x5E0BEC6E, (q31_t)0x56D42C99, (q31_t)0x5DC79D7C,
	(q31_t)0x571DEEF9, (q31_t)0x5D8314B0, (q31_t)0x57677B9D,
	(q31_t)0x5D3E5236, (q31_t)0x57B0D256, (q31_t)0x5CF95638,
	(q31_t)0x57F9F2F7, (q31_t)0x5CB420DF, (q31_t)0x5842DD54,
	(q31_t)0x5C6EB258, (q31_t)0x588B913F, (q31_t)0x5C290ACC,
	(q31_t)0x58D40E8C, (q31_t)0x5BE32A67, (q31_t)0x591C550E,
	(q31_t)0x5B9D1153, (q31_t)0x59646497, (q31_t)0x5B56BFBD,
	(q31_t)0x59AC3CFD, (q31_t)0x5B1035CF, (q31_t)0x59F3DE12,
	(q31_t)0x5AC973B4, (q31_t)0x5A3B47AA, (q31_t)0x5A82799A,
	(q31_t)0x5A82799A, (q31_t)0x5A3B47AA, (q31_t)0x5AC973B4,
	(q31_t)0x59F3DE12, (q31_t)0x5B1035CF, (q31_t)0x59AC3CFD,
	(q31_t)0x5B56BFBD, (q31_t)0x59646497, (q31_t)0x5B9D1153,
	(q31_t)0x591C550E, (q31_t)0x5BE32A67, (q31_t)0x58D40E8C,
	(q31_t)0x5C290ACC, (q31_t)0x588B913F, (q31_t)0x5C6EB258,
	(q31_t)0x5842DD54, (q31_t)0x5CB420DF, (q31_t)0x57F9F2F7,
	(q31_t)0x5CF95638, (q31_t)0x57B0D256, (q31_t)0x5D3E5236,
	(q31_t)0x57677B9D, (q31_t)0x5D8314B0, (q31_t)0x571DEEF9,
	(q31_t)0x5DC79D7C, (q31_t)0x56D42C99, (q31_t)0x5E0BEC6E,
	(q31_t)0x568A34A9, (q31_t)0x5E50015D, (q31_t)0x56400757,
	(q31_t)0x5E93DC1F, (q31_t)0x55F5A4D2, (q31_t)0x5ED77C89,
	(q31_t)0x55AB0D46, (q31_t)0x5F1AE273, (q31_t)0x556040E2,
	(q31_t)0x5F5E0DB3, (q31_t)0x55153FD4, (q31_t)0x5FA0FE1E,
	(q31_t)0x54CA0A4A, (q31_t)0x5FE3B38D, (q31_t)0x547EA073,
	(q31_t)0x60262DD5, (q31_t)0x5433027D, (q31_t)0x60686CCE,
	(q31_t)0x53E73097, (q31_t)0x60AA704F, (q31_t)0x539B2AEF,
	(q31_t)0x60EC3830, (q31_t)0x534EF1B5, (q31_t)0x612DC446,
	(q31_t)0x53028517, (q31_t)0x616F146B, (q31_t)0x52B5E545,
	(q31_t)0x61B02876, (q31_t)0x5269126E, (q31_t)0x61F1003E,
	(q31_t)0x521C0CC1, (q31_t)0x62319B9D, (q31_t)0x51CED46E,
	(q31_t)0x6271FA69, (q31_t)0x518169A4, (q31_t)0x62B21C7B,
	(q31_t)0x5133CC94, (q31_t)0x62F201AC, (q31_t)0x50E5FD6C,
	(q31_t)0x6331A9D4, (q31_t)0x5097FC5E, (q31_t)0x637114CC,
	(q31_t)0x5049C999, (q31_t)0x63B0426D, (q31_t)0x4FFB654D,
	(q31_t)0x63EF328F, (q31_t)0x4FACCFAB, (q31_t)0x642DE50D,
	(q31_t)0x4F5E08E3, (q31_t)0x646C59BF, (q31_t)0x4F0F1126,
	(q31_t)0x64AA907F, (q31_t)0x4EBFE8A4, (q31_t)0x64E88926,
	(q31_t)0x4E708F8F, (q31_t)0x6526438E, (q31_t)0x4E210617,
	(q31_t)0x6563BF92, (q31_t)0x4DD14C6E, (q31_t)0x65A0FD0B,
	(q31_t)0x4D8162C4, (q31_t)0x65DDFBD3, (q31_t)0x4D31494B,
	(q31_t)0x661ABBC5, (q31_t)0x4CE10034, (q31_t)0x66573CBB,
	(q31_t)0x4C9087B1, (q31_t)0x66937E90, (q31_t)0x4C3FDFF3,
	(q31_t)0x66CF811F, (q31_t)0x4BEF092D, (q31_t)0x670B4443,
	(q31_t)0x4B9E038F, (q31_t)0x6746C7D7, (q31_t)0x4B4CCF4D,
	(q31_t)0x67820BB6, (q31_t)0x4AFB6C97, (q31_t)0x67BD0FBC,
	(q31_t)0x4AA9DBA1, (q31_t)0x67F7D3C4, (q31_t)0x4A581C9D,
	(q31_t)0x683257AA, (q31_t)0x4A062FBD, (q31_t)0x686C9B4B,
	(q31_t)0x49B41533, (q31_t)0x68A69E81, (q31_t)0x4961CD32,
	(q31_t)0x68E06129, (q31_t)0x490F57EE, (q31_t)0x6919E320,
	(q31_t)0x48BCB598, (q31_t)0x69532442, (q31_t)0x4869E664,
	(q31_t)0x698C246C, (q31_t)0x4816EA85, (q31_t)0x69C4E37A,
	(q31_t)0x47C3C22E, (q31_t)0x69FD614A, (q31_t)0x47706D93,
	(q31_t)0x6A359DB9, (q31_t)0x471CECE6, (q31_t)0x6A6D98A4,
	(q31_t)0x46C9405C, (q31_t)0x6AA551E8, (q31_t)0x46756827,
	(q31_t)0x6ADCC964, (q31_t)0x4621647C, (q31_t)0x6B13FEF5,
	(q31_t)0x45CD358F, (q31_t)0x6B4AF278, (q31_t)0x4578DB93,
	(q31_t)0x6B81A3CD, (q31_t)0x452456BC, (q31_t)0x6BB812D0,
	(q31_t)0x44CFA73F, (q31_t)0x6BEE3F62, (q31_t)0x447ACD50,
	(q31_t)0x6C242960, (q31_t)0x4425C923, (q31_t)0x6C59D0A9,
	(q31_t)0x43D09AEC, (q31_t)0x6C8F351C, (q31_t)0x437B42E1,
	(q31_t)0x6CC45697, (q31_t)0x4325C135, (q31_t)0x6CF934FB,
	(q31_t)0x42D0161E, (q31_t)0x6D2DD027, (q31_t)0x427A41D0,
	(q31_t)0x6D6227FA, (q31_t)0x42244480, (q31_t)0x6D963C54,
	(q31_t)0x41CE1E64, (q31_t)0x6DCA0D14, (q31_t)0x4177CFB0,
	(q31_t)0x6DFD9A1B, (q31_t)0x4121589A, (q31_t)0x6E30E349,
	(q31_t)0x40CAB957, (q31_t)0x6E63E87F, (q31_t)0x4073F21D,
	(q31_t)0x6E96A99C, (q31_t)0x401D0320, (q31_t)0x6EC92682,
	(q31_t)0x3FC5EC97, (q31_t)0x6EFB5F12, (q31_t)0x3F6EAEB8,
	(q31_t)0x6F2D532C, (q31_t)0x3F1749B7, (q31_t)0x6F5F02B1,
	(q31_t)0x3EBFBDCC, (q31_t)0x6F906D84, (q31_t)0x3E680B2C,
	(q31_t)0x6FC19385, (q31_t)0x3E10320D, (q31_t)0x6FF27496,
	(q31_t)0x3DB832A5, (q31_t)0x70231099, (q31_t)0x3D600D2B,
	(q31_t)0x70536771, (q31_t)0x3D07C1D5, (q31_t)0x708378FE,
	(q31_t)0x3CAF50DA, (q31_t)0x70B34524, (q31_t)0x3C56BA70,
	(q31_t)0x70E2CBC6, (q31_t)0x3BFDFECD, (q31_t)0x71120CC5,
	(q31_t)0x3BA51E29, (q31_t)0x71410804, (q31_t)0x3B4C18BA,
	(q31_t)0x716FBD68, (q31_t)0x3AF2EEB7, (q31_t)0x719E2CD2,
	(q31_t)0x3A99A057, (q31_t)0x71CC5626, (q31_t)0x3A402DD1,
	(q31_t)0x71FA3948, (q31_t)0x39E6975D, (q31_t)0x7227D61C,
	(q31_t)0x398CDD32, (q31_t)0x72552C84, (q31_t)0x3932FF87,
	(q31_t)0x72823C66, (q31_t)0x38D8FE93, (q31_t)0x72AF05A6,
	(q31_t)0x387EDA8E, (q31_t)0x72DB8828, (q31_t)0x382493B0,
	(q31_t)0x7307C3D0, (q31_t)0x37CA2A30, (q31_t)0x7333B883,
	(q31_t)0x376F9E46, (q31_t)0x735F6626, (q31_t)0x3714F02A,
	(q31_t)0x738ACC9E, (q31_t)0x36BA2013, (q31_t)0x73B5EBD0,
	(q31_t)0x365F2E3B, (q31_t)0x73E0C3A3, (q31_t)0x36041AD9,
	(q31_t)0x740B53FA, (q31_t)0x35A8E624, (q31_t)0x74359CBD,
	(q31_t)0x354D9056, (q31_t)0x745F9DD1, (q31_t)0x34F219A7,
	(q31_t)0x7489571B, (q31_t)0x3496824F, (q31_t)0x74B2C883,
	(q31_t)0x343ACA87, (q31_t)0x74DBF1EF, (q31_t)0x33DEF287,
	(q31_t)0x7504D345, (q31_t)0x3382FA88, (q31_t)0x752D6C6C,
	(q31_t)0x3326E2C2, (q31_t)0x7555BD4B, (q31_t)0x32CAAB6F,
	(q31_t)0x757DC5CA, (q31_t)0x326E54C7, (q31_t)0x75A585CF,
	(q31_t)0x3211DF03, (q31_t)0x75CCFD42, (q31_t)0x31B54A5D,
	(q31_t)0x75F42C0A, (q31_t)0x3158970D, (q31_t)0x761B1211,
	(q31_t)0x30FBC54D, (q31_t)0x7641AF3C, (q31_t)0x309ED555,
	(q31_t)0x76680376, (q31_t)0x3041C760, (q31_t)0x768E0EA5,
	(q31_t)0x2FE49BA6, (q31_t)0x76B3D0B3, (q31_t)0x2F875262,
	(q31_t)0x76D94988, (q31_t)0x2F29EBCC, (q31_t)0x76FE790E,
	(q31_t)0x2ECC681E, (q31_t)0x77235F2D, (q31_t)0x2E6EC792,
	(q31_t)0x7747FBCE, (q31_t)0x2E110A62, (q31_t)0x776C4EDB,
	(q31_t)0x2DB330C7, (q31_t)0x7790583D, (q31_t)0x2D553AFB,
	(q31_t)0x77B417DF, (q31_t)0x2CF72939, (q31_t)0x77D78DAA,
	(q31_t)0x2C98FBBA, (q31_t)0x77FAB988, (q31_t)0x2C3AB2B9,
	(q31_t)0x781D9B64, (q31_t)0x2BDC4E6F, (q31_t)0x78403328,
	(q31_t)0x2B7DCF17, (q31_t)0x786280BF, (q31_t)0x2B1F34EB,
	(q31_t)0x78848413, (q31_t)0x2AC08025, (q31_t)0x78A63D10,
	(q31_t)0x2A61B101, (q31_t)0x78C7ABA1, (q31_t)0x2A02C7B8,
	(q31_t)0x78E8CFB1, (q31_t)0x29A3C484, (q31_t)0x7909A92C,
	(q31_t)0x2944A7A2, (q31_t)0x792A37FE, (q31_t)0x28E5714A,
	(q31_t)0x794A7C11, (q31_t)0x288621B9, (q31_t)0x796A7554,
	(q31_t)0x2826B928, (q31_t)0x798A23B1, (q31_t)0x27C737D2,
	(q31_t)0x79A98715, (q31_t)0x27679DF4, (q31_t)0x79C89F6D,
	(q31_t)0x2707EBC6, (q31_t)0x79E76CA6, (q31_t)0x26A82185,
	(q31_t)0x7A05EEAD, (q31_t)0x26483F6C, (q31_t)0x7A24256E,
	(q31_t)0x25E845B5, (q31_t)0x7A4210D8, (q31_t)0x2588349D,
	(q31_t)0x7A5FB0D8, (q31_t)0x25280C5D, (q31_t)0x7A7D055B,
	(q31_t)0x24C7CD32, (q31_t)0x7A9A0E4F, (q31_t)0x24677757,
	(q31_t)0x7AB6CBA3, (q31_t)0x24070B07, (q31_t)0x7AD33D45,
	(q31_t)0x23A6887E, (q31_t)0x7AEF6323, (q31_t)0x2345EFF7,
	(q31_t)0x7B0B3D2C, (q31_t)0x22E541AE, (q31_t)0x7B26CB4F,
	(q31_t)0x22847DDF, (q31_t)0x7B420D7A, (q31_t)0x2223A4C5,
	(q31_t)0x7B5D039D, (q31_t)0x21C2B69C, (q31_t)0x7B77ADA8,
	(q31_t)0x2161B39F, (q31_t)0x7B920B89, (q31_t)0x21009C0B,
	(q31_t)0x7BAC1D31, (q31_t)0x209F701C, (q31_t)0x7BC5E28F,
	(q31_t)0x203E300D, (q31_t)0x7BDF5B94, (q31_t)0x1FDCDC1A,
	(q31_t)0x7BF88830, (q31_t)0x1F7B7480, (q31_t)0x7C116853,
	(q31_t)0x1F19F97B, (q31_t)0x7C29FBEE, (q31_t)0x1EB86B46,
	(q31_t)0x7C4242F2, (q31_t)0x1E56CA1E, (q31_t)0x7C5A3D4F,
	(q31_t)0x1DF5163F, (q31_t)0x7C71EAF8, (q31_t)0x1D934FE5,
	(q31_t)0x7C894BDD, (q31_t)0x1D31774D, (q31_t)0x7CA05FF1,
	(q31_t)0x1CCF8CB3, (q31_t)0x7CB72724, (q31_t)0x1C6D9053,
	(q31_t)0x7CCDA168, (q31_t)0x1C0B826A, (q31_t)0x7CE3CEB1,
	(q31_t)0x1BA96334, (q31_t)0x7CF9AEF0, (q31_t)0x1B4732EF,
	(q31_t)0x7D0F4218, (q31_t)0x1AE4F1D6, (q31_t)0x7D24881A,
	(q31_t)0x1A82A025, (q31_t)0x7D3980EC, (q31_t)0x1A203E1B,
	(q31_t)0x7D4E2C7E, (q31_t)0x19BDCBF2, (q31_t)0x7D628AC5,
	(q31_t)0x195B49E9, (q31_t)0x7D769BB5, (q31_t)0x18F8B83C,
	(q31_t)0x7D8A5F3F, (q31_t)0x18961727, (q31_t)0x7D9DD55A,
	(q31_t)0x183366E8, (q31_t)0x7DB0FDF7, (q31_t)0x17D0A7BB,
	(q31_t)0x7DC3D90D, (q31_t)0x176DD9DE, (q31_t)0x7DD6668E,
	(q31_t)0x170AFD8D, (q31_t)0x7DE8A670, (q31_t)0x16A81305,
	(q31_t)0x7DFA98A7, (q31_t)0x16451A83, (q31_t)0x7E0C3D29,
	(q31_t)0x15E21444, (q31_t)0x7E1D93E9, (q31_t)0x157F0086,
	(q31_t)0x7E2E9CDF, (q31_t)0x151BDF85, (q31_t)0x7E3F57FE,
	(q31_t)0x14B8B17F, (q31_t)0x7E4FC53E, (q31_t)0x145576B1,
	(q31_t)0x7E5FE493, (q31_t)0x13F22F57, (q31_t)0x7E6FB5F3,
	(q31_t)0x138EDBB0, (q31_t)0x7E7F3956, (q31_t)0x132B7BF9,
	(q31_t)0x7E8E6EB1, (q31_t)0x12C8106E, (q31_t)0x7E9D55FC,
	(q31_t)0x1264994E, (q31_t)0x7EABEF2C, (q31_t)0x120116D4,
	(q31_t)0x7EBA3A39, (q31_t)0x119D8940, (q31_t)0x7EC8371A,
	(q31_t)0x1139F0CE, (q31_t)0x7ED5E5C6, (q31_t)0x10D64DBC,
	(q31_t)0x7EE34635, (q31_t)0x1072A047, (q31_t)0x7EF0585F,
	(q31_t)0x100EE8AD, (q31_t)0x7EFD1C3C, (q31_t)0x0FAB272B,
	(q31_t)0x7F0991C3, (q31_t)0x0F475BFE, (q31_t)0x7F15B8EE,
	(q31_t)0x0EE38765, (q31_t)0x7F2191B4, (q31_t)0x0E7FA99D,
	(q31_t)0x7F2D1C0E, (q31_t)0x0E1BC2E3, (q31_t)0x7F3857F5,
	(q31_t)0x0DB7D376, (q31_t)0x7F434563, (q31_t)0x0D53DB92,
	(q31_t)0x7F4DE450, (q31_t)0x0CEFDB75, (q31_t)0x7F5834B6,
	(q31_t)0x0C8BD35E, (q31_t)0x7F62368F, (q31_t)0x0C27C389,
	(q31_t)0x7F6BE9D4, (q31_t)0x0BC3AC35, (q31_t)0x7F754E7F,
	(q31_t)0x0B5F8D9F, (q31_t)0x7F7E648B, (q31_t)0x0AFB6805,
	(q31_t)0x7F872BF3, (q31_t)0x0A973BA5, (q31_t)0x7F8FA4AF,
	(q31_t)0x0A3308BC, (q31_t)0x7F97CEBC, (q31_t)0x09CECF89,
	(q31_t)0x7F9FAA15, (q31_t)0x096A9049, (q31_t)0x7FA736B4,
	(q31_t)0x09064B3A, (q31_t)0x7FAE7494, (q31_t)0x08A2009A,
	(q31_t)0x7FB563B2, (q31_t)0x083DB0A7, (q31_t)0x7FBC040A,
	(q31_t)0x07D95B9E, (q31_t)0x7FC25596, (q31_t)0x077501BE,
	(q31_t)0x7FC85853, (q31_t)0x0710A344, (q31_t)0x7FCE0C3E,
	(q31_t)0x06AC406F, (q31_t)0x7FD37152, (q31_t)0x0647D97C,
	(q31_t)0x7FD8878D, (q31_t)0x05E36EA9, (q31_t)0x7FDD4EEC,
	(q31_t)0x057F0034, (q31_t)0x7FE1C76B, (q31_t)0x051A8E5C,
	(q31_t)0x7FE5F108, (q31_t)0x04B6195D, (q31_t)0x7FE9CBC0,
	(q31_t)0x0451A176, (q31_t)0x7FED5790, (q31_t)0x03ED26E6,
	(q31_t)0x7FF09477, (q31_t)0x0388A9E9, (q31_t)0x7FF38273,
	(q31_t)0x03242ABF, (q31_t)0x7FF62182, (q31_t)0x02BFA9A4,
	(q31_t)0x7FF871A1, (q31_t)0x025B26D7, (q31_t)0x7FFA72D1,
	(q31_t)0x01F6A296, (q31_t)0x7FFC250F, (q31_t)0x01921D1F,
	(q31_t)0x7FFD885A, (q31_t)0x012D96B0, (q31_t)0x7FFE9CB2,
	(q31_t)0x00C90F88, (q31_t)0x7FFF6216, (q31_t)0x006487E3,
	(q31_t)0x7FFFD885, (q31_t)0x00000000, (q31_t)0x7FFFFFFF,
	(q31_t)0xFF9B781D, (q31_t)0x7FFFD885, (q31_t)0xFF36F078,
	(q31_t)0x7FFF6216, (q31_t)0xFED2694F, (q31_t)0x7FFE9CB2,
	(q31_t)0xFE6DE2E0, (q31_t)0x7FFD885A, (q31_t)0xFE095D69,
	(q31_t)0x7FFC250F, (q31_t)0xFDA4D928, (q31_t)0x7FFA72D1,
	(q31_t)0xFD40565B, (q31_t)0x7FF871A1, (q31_t)0xFCDBD541,
	(q31_t)0x7FF62182, (q31_t)0xFC775616, (q31_t)0x7FF38273,
	(q31_t)0xFC12D919, (q31_t)0x7FF09477, (q31_t)0xFBAE5E89,
	(q31_t)0x7FED5790, (q31_t)0xFB49E6A2, (q31_t)0x7FE9CBC0,
	(q31_t)0xFAE571A4, (q31_t)0x7FE5F108, (q31_t)0xFA80FFCB,
	(q31_t)0x7FE1C76B, (q31_t)0xFA1C9156, (q31_t)0x7FDD4EEC,
	(q31_t)0xF9B82683, (q31_t)0x7FD8878D, (q31_t)0xF953BF90,
	(q31_t)0x7FD37152, (q31_t)0xF8EF5CBB, (q31_t)0x7FCE0C3E,
	(q31_t)0xF88AFE41, (q31_t)0x7FC85853, (q31_t)0xF826A461,
	(q31_t)0x7FC25596, (q31_t)0xF7C24F58, (q31_t)0x7FBC040A,
	(q31_t)0xF75DFF65, (q31_t)0x7FB563B2, (q31_t)0xF6F9B4C5,
	(q31_t)0x7FAE7494, (q31_t)0xF6956FB6, (q31_t)0x7FA736B4,
	(q31_t)0xF6313076, (q31_t)0x7F9FAA15, (q31_t)0xF5CCF743,
	(q31_t)0x7F97CEBC, (q31_t)0xF568C45A, (q31_t)0x7F8FA4AF,
	(q31_t)0xF50497FA, (q31_t)0x7F872BF3, (q31_t)0xF4A07260,
	(q31_t)0x7F7E648B, (q31_t)0xF43C53CA, (q31_t)0x7F754E7F,
	(q31_t)0xF3D83C76, (q31_t)0x7F6BE9D4, (q31_t)0xF3742CA1,
	(q31_t)0x7F62368F, (q31_t)0xF310248A, (q31_t)0x7F5834B6,
	(q31_t)0xF2AC246D, (q31_t)0x7F4DE450, (q31_t)0xF2482C89,
	(q31_t)0x7F434563, (q31_t)0xF1E43D1C, (q31_t)0x7F3857F5,
	(q31_t)0xF1805662, (q31_t)0x7F2D1C0E, (q31_t)0xF11C789A,
	(q31_t)0x7F2191B4, (q31_t)0xF0B8A401, (q31_t)0x7F15B8EE,
	(q31_t)0xF054D8D4, (q31_t)0x7F0991C3, (q31_t)0xEFF11752,
	(q31_t)0x7EFD1C3C, (q31_t)0xEF8D5FB8, (q31_t)0x7EF0585F,
	(q31_t)0xEF29B243, (q31_t)0x7EE34635, (q31_t)0xEEC60F31,
	(q31_t)0x7ED5E5C6, (q31_t)0xEE6276BF, (q31_t)0x7EC8371A,
	(q31_t)0xEDFEE92B, (q31_t)0x7EBA3A39, (q31_t)0xED9B66B2,
	(q31_t)0x7EABEF2C, (q31_t)0xED37EF91, (q31_t)0x7E9D55FC,
	(q31_t)0xECD48406, (q31_t)0x7E8E6EB1, (q31_t)0xEC71244F,
	(q31_t)0x7E7F3956, (q31_t)0xEC0DD0A8, (q31_t)0x7E6FB5F3,
	(q31_t)0xEBAA894E, (q31_t)0x7E5FE493, (q31_t)0xEB474E80,
	(q31_t)0x7E4FC53E, (q31_t)0xEAE4207A, (q31_t)0x7E3F57FE,
	(q31_t)0xEA80FF79, (q31_t)0x7E2E9CDF, (q31_t)0xEA1DEBBB,
	(q31_t)0x7E1D93E9, (q31_t)0xE9BAE57C, (q31_t)0x7E0C3D29,
	(q31_t)0xE957ECFB, (q31_t)0x7DFA98A7, (q31_t)0xE8F50273,
	(q31_t)0x7DE8A670, (q31_t)0xE8922621, (q31_t)0x7DD6668E,
	(q31_t)0xE82F5844, (q31_t)0x7DC3D90D, (q31_t)0xE7CC9917,
	(q31_t)0x7DB0FDF7, (q31_t)0xE769E8D8, (q31_t)0x7D9DD55A,
	(q31_t)0xE70747C3, (q31_t)0x7D8A5F3F, (q31_t)0xE6A4B616,
	(q31_t)0x7D769BB5, (q31_t)0xE642340D, (q31_t)0x7D628AC5,
	(q31_t)0xE5DFC1E4, (q31_t)0x7D4E2C7E, (q31_t)0xE57D5FDA,
	(q31_t)0x7D3980EC, (q31_t)0xE51B0E2A, (q31_t)0x7D24881A,
	(q31_t)0xE4B8CD10, (q31_t)0x7D0F4218, (q31_t)0xE4569CCB,
	(q31_t)0x7CF9AEF0, (q31_t)0xE3F47D95, (q31_t)0x7CE3CEB1,
	(q31_t)0xE3926FAC, (q31_t)0x7CCDA168, (q31_t)0xE330734C,
	(q31_t)0x7CB72724, (q31_t)0xE2CE88B2, (q31_t)0x7CA05FF1,
	(q31_t)0xE26CB01A, (q31_t)0x7C894BDD, (q31_t)0xE20AE9C1,
	(q31_t)0x7C71EAF8, (q31_t)0xE1A935E1, (q31_t)0x7C5A3D4F,
	(q31_t)0xE14794B9, (q31_t)0x7C4242F2, (q31_t)0xE0E60684,
	(q31_t)0x7C29FBEE, (q31_t)0xE0848B7F, (q31_t)0x7C116853,
	(q31_t)0xE02323E5, (q31_t)0x7BF88830, (q31_t)0xDFC1CFF2,
	(q31_t)0x7BDF5B94, (q31_t)0xDF608FE3, (q31_t)0x7BC5E28F,
	(q31_t)0xDEFF63F4, (q31_t)0x7BAC1D31, (q31_t)0xDE9E4C60,
	(q31_t)0x7B920B89, (q31_t)0xDE3D4963, (q31_t)0x7B77ADA8,
	(q31_t)0xDDDC5B3A, (q31_t)0x7B5D039D, (q31_t)0xDD7B8220,
	(q31_t)0x7B420D7A, (q31_t)0xDD1ABE51, (q31_t)0x7B26CB4F,
	(q31_t)0xDCBA1008, (q31_t)0x7B0B3D2C, (q31_t)0xDC597781,
	(q31_t)0x7AEF6323, (q31_t)0xDBF8F4F8, (q31_t)0x7AD33D45,
	(q31_t)0xDB9888A8, (q31_t)0x7AB6CBA3, (q31_t)0xDB3832CD,
	(q31_t)0x7A9A0E4F, (q31_t)0xDAD7F3A2, (q31_t)0x7A7D055B,
	(q31_t)0xDA77CB62, (q31_t)0x7A5FB0D8, (q31_t)0xDA17BA4A,
	(q31_t)0x7A4210D8, (q31_t)0xD9B7C093, (q31_t)0x7A24256E,
	(q31_t)0xD957DE7A, (q31_t)0x7A05EEAD, (q31_t)0xD8F81439,
	(q31_t)0x79E76CA6, (q31_t)0xD898620C, (q31_t)0x79C89F6D,
	(q31_t)0xD838C82D, (q31_t)0x79A98715, (q31_t)0xD7D946D7,
	(q31_t)0x798A23B1, (q31_t)0xD779DE46, (q31_t)0x796A7554,
	(q31_t)0xD71A8EB5, (q31_t)0x794A7C11, (q31_t)0xD6BB585D,
	(q31_t)0x792A37FE, (q31_t)0xD65C3B7B, (q31_t)0x7909A92C,
	(q31_t)0xD5FD3847, (q31_t)0x78E8CFB1, (q31_t)0xD59E4EFE,
	(q31_t)0x78C7ABA1, (q31_t)0xD53F7FDA, (q31_t)0x78A63D10,
	(q31_t)0xD4E0CB14, (q31_t)0x78848413, (q31_t)0xD48230E8,
	(q31_t)0x786280BF, (q31_t)0xD423B190, (q31_t)0x78403328,
	(q31_t)0xD3C54D46, (q31_t)0x781D9B64, (q31_t)0xD3670445,
	(q31_t)0x77FAB988, (q31_t)0xD308D6C6, (q31_t)0x77D78DAA,
	(q31_t)0xD2AAC504, (q31_t)0x77B417DF, (q31_t)0xD24CCF38,
	(q31_t)0x7790583D, (q31_t)0xD1EEF59E, (q31_t)0x776C4EDB,
	(q31_t)0xD191386D, (q31_t)0x7747FBCE, (q31_t)0xD13397E1,
	(q31_t)0x77235F2D, (q31_t)0xD0D61433, (q31_t)0x76FE790E,
	(q31_t)0xD078AD9D, (q31_t)0x76D94988, (q31_t)0xD01B6459,
	(q31_t)0x76B3D0B3, (q31_t)0xCFBE389F, (q31_t)0x768E0EA5,
	(q31_t)0xCF612AAA, (q31_t)0x76680376, (q31_t)0xCF043AB2,
	(q31_t)0x7641AF3C, (q31_t)0xCEA768F2, (q31_t)0x761B1211,
	(q31_t)0xCE4AB5A2, (q31_t)0x75F42C0A, (q31_t)0xCDEE20FC,
	(q31_t)0x75CCFD42, (q31_t)0xCD91AB38, (q31_t)0x75A585CF,
	(q31_t)0xCD355490, (q31_t)0x757DC5CA, (q31_t)0xCCD91D3D,
	(q31_t)0x7555BD4B, (q31_t)0xCC7D0577, (q31_t)0x752D6C6C,
	(q31_t)0xCC210D78, (q31_t)0x7504D345, (q31_t)0xCBC53578,
	(q31_t)0x74DBF1EF, (q31_t)0xCB697DB0, (q31_t)0x74B2C883,
	(q31_t)0xCB0DE658, (q31_t)0x7489571B, (q31_t)0xCAB26FA9,
	(q31_t)0x745F9DD1, (q31_t)0xCA5719DB, (q31_t)0x74359CBD,
	(q31_t)0xC9FBE527, (q31_t)0x740B53FA, (q31_t)0xC9A0D1C4,
	(q31_t)0x73E0C3A3, (q31_t)0xC945DFEC, (q31_t)0x73B5EBD0,
	(q31_t)0xC8EB0FD6, (q31_t)0x738ACC9E, (q31_t)0xC89061BA,
	(q31_t)0x735F6626, (q31_t)0xC835D5D0, (q31_t)0x7333B883,
	(q31_t)0xC7DB6C50, (q31_t)0x7307C3D0, (q31_t)0xC7812571,
	(q31_t)0x72DB8828, (q31_t)0xC727016C, (q31_t)0x72AF05A6,
	(q31_t)0xC6CD0079, (q31_t)0x72823C66, (q31_t)0xC67322CD,
	(q31_t)0x72552C84, (q31_t)0xC61968A2, (q31_t)0x7227D61C,
	(q31_t)0xC5BFD22E, (q31_t)0x71FA3948, (q31_t)0xC5665FA8,
	(q31_t)0x71CC5626, (q31_t)0xC50D1148, (q31_t)0x719E2CD2,
	(q31_t)0xC4B3E746, (q31_t)0x716FBD68, (q31_t)0xC45AE1D7,
	(q31_t)0x71410804, (q31_t)0xC4020132, (q31_t)0x71120CC5,
	(q31_t)0xC3A9458F, (q31_t)0x70E2CBC6, (q31_t)0xC350AF25,
	(q31_t)0x70B34524, (q31_t)0xC2F83E2A, (q31_t)0x708378FE,
	(q31_t)0xC29FF2D4, (q31_t)0x70536771, (q31_t)0xC247CD5A,
	(q31_t)0x70231099, (q31_t)0xC1EFCDF2, (q31_t)0x6FF27496,
	(q31_t)0xC197F4D3, (q31_t)0x6FC19385, (q31_t)0xC1404233,
	(q31_t)0x6F906D84, (q31_t)0xC0E8B648, (q31_t)0x6F5F02B1,
	(q31_t)0xC0915147, (q31_t)0x6F2D532C, (q31_t)0xC03A1368,
	(q31_t)0x6EFB5F12, (q31_t)0xBFE2FCDF, (q31_t)0x6EC92682,
	(q31_t)0xBF8C0DE2, (q31_t)0x6E96A99C, (q31_t)0xBF3546A8,
	(q31_t)0x6E63E87F, (q31_t)0xBEDEA765, (q31_t)0x6E30E349,
	(q31_t)0xBE88304F, (q31_t)0x6DFD9A1B, (q31_t)0xBE31E19B,
	(q31_t)0x6DCA0D14, (q31_t)0xBDDBBB7F, (q31_t)0x6D963C54,
	(q31_t)0xBD85BE2F, (q31_t)0x6D6227FA, (q31_t)0xBD2FE9E1,
	(q31_t)0x6D2DD027, (q31_t)0xBCDA3ECA, (q31_t)0x6CF934FB,
	(q31_t)0xBC84BD1E, (q31_t)0x6CC45697, (q31_t)0xBC2F6513,
	(q31_t)0x6C8F351C, (q31_t)0xBBDA36DC, (q31_t)0x6C59D0A9,
	(q31_t)0xBB8532AF, (q31_t)0x6C242960, (q31_t)0xBB3058C0,
	(q31_t)0x6BEE3F62, (q31_t)0xBADBA943, (q31_t)0x6BB812D0,
	(q31_t)0xBA87246C, (q31_t)0x6B81A3CD, (q31_t)0xBA32CA70,
	(q31_t)0x6B4AF278, (q31_t)0xB9DE9B83, (q31_t)0x6B13FEF5,
	(q31_t)0xB98A97D8, (q31_t)0x6ADCC964, (q31_t)0xB936BFA3,
	(q31_t)0x6AA551E8, (q31_t)0xB8E31319, (q31_t)0x6A6D98A4,
	(q31_t)0xB88F926C, (q31_t)0x6A359DB9, (q31_t)0xB83C3DD1,
	(q31_t)0x69FD614A, (q31_t)0xB7E9157A, (q31_t)0x69C4E37A,
	(q31_t)0xB796199B, (q31_t)0x698C246C, (q31_t)0xB7434A67,
	(q31_t)0x69532442, (q31_t)0xB6F0A811, (q31_t)0x6919E320,
	(q31_t)0xB69E32CD, (q31_t)0x68E06129, (q31_t)0xB64BEACC,
	(q31_t)0x68A69E81, (q31_t)0xB5F9D042, (q31_t)0x686C9B4B,
	(q31_t)0xB5A7E362, (q31_t)0x683257AA, (q31_t)0xB556245E,
	(q31_t)0x67F7D3C4, (q31_t)0xB5049368, (q31_t)0x67BD0FBC,
	(q31_t)0xB4B330B2, (q31_t)0x67820BB6, (q31_t)0xB461FC70,
	(q31_t)0x6746C7D7, (q31_t)0xB410F6D2, (q31_t)0x670B4443,
	(q31_t)0xB3C02