/*-------------------------------------------------------------------------
 *
 * altertablenodes.h
 *	  definitions for ALTER TABLE subcommands
 *
 * In PostgreSQL, these are in tablecmds.c, but in GPDB, they need to be
 * exposed to out/readfunc.c, because we need support for serializing
 * these. Also, the structs are Nodes in GPDB for the same reason.
 *
 * NOTE: If new fields are added to these structs, remember to update
 * outfuncs.c/readfuncs.c
 *
 * Portions Copyright (c) 2006-2009, Greenplum inc
 * Portions Copyright (c) 2012-Present VMware, Inc. or its affiliates.
 * Portions Copyright (c) 1996-2019, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 * src/include/nodes/altertablenodes.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef ALTERTABLENODES_H
#define ALTERTABLENODES_H

#include "cdb/cdbaocsam.h"
/*
 * GPDB: Moved here from tablecmds.c
 *
 * State information for ALTER TABLE
 *
 * The pending-work queue for an ALTER TABLE is a List of AlteredTableInfo
 * structs, one for each table modified by the operation (the named table
 * plus any child tables that are affected).  We save lists of subcommands
 * to apply to this table (possibly modified by parse transformation steps);
 * these lists will be executed in Phase 2.  If a Phase 3 step is needed,
 * necessary information is stored in the constraints and newcolvals lists.
 *
 * Phase 2 is divided into multiple passes; subcommands are executed in
 * a pass determined by subcommand type.
 */

#define AT_PASS_UNSET			-1	/* UNSET will cause ERROR */
#define AT_PASS_DROP			0	/* DROP (all flavors) */
#define AT_PASS_ALTER_TYPE		1	/* ALTER COLUMN TYPE */
#define AT_PASS_OLD_INDEX		2	/* re-add existing indexes */
#define AT_PASS_OLD_CONSTR		3	/* re-add existing constraints */
/* We could support a RENAME COLUMN pass here, but not currently used */
#define AT_PASS_ADD_COL			4	/* ADD COLUMN */
#define AT_PASS_COL_ATTRS		5	/* set other column attributes */
#define AT_PASS_ADD_INDEX		6	/* ADD indexes */
#define AT_PASS_ADD_CONSTR		7	/* ADD constraints, defaults */
#define AT_PASS_SET_ENCODING		8	/* ALTER COLUMN SET ENCODING. Note: if you are going to reorder the macros, this has to be after ALTER COLUMN TYPE. */
#define AT_PASS_MISC			9	/* other stuff */
#define AT_NUM_PASSES			10

typedef struct AlteredTableInfo
{
	NodeTag		type;

	/* Information saved before any work commences: */
	Oid			relid;			/* Relation to work on */
	char		relkind;		/* Its relkind */
	TupleDesc	oldDesc;		/* Pre-modification tuple descriptor */
	/* Information saved by Phase 1 for Phase 2: */
	List	   *subcmds[AT_NUM_PASSES]; /* Lists of AlterTableCmd */
	/* Information saved by Phases 1/2 for Phase 3: */
	List	   *constraints;	/* List of NewConstraint */
	List	   *newvals;		/* List of NewColumnValue */
	bool		verify_new_notnull; /* T if we should recheck NOT NULL */
	int			rewrite;		/* Reason for forced rewrite, if any */
	Oid 		newAccessMethod; /* new access method; 0 means no change */
	Datum 		newOptions; /* new reloptions, in case when we need a table rewrite */
	bool		dist_opfamily_changed; /* T if changing datatype of distribution key column and new opclass is in different opfamily than old one */
	Oid			new_opclass;		/* new opclass, if changing a distribution key column */
	Oid			newTableSpace;	/* new tablespace; 0 means no change */
	bool		chgPersistence; /* T if SET LOGGED/UNLOGGED is used */
	char		newrelpersistence;	/* if above is true */
	Expr	   *partition_constraint;	/* for attach partition validation */
	/* true, if validating default due to some other attach/detach */
	bool		validate_default;
	/* Objects to rebuild after completing ALTER TYPE operations */
	List	   *changedConstraintOids;	/* OIDs of constraints to rebuild */
	List	   *changedConstraintDefs;	/* string definitions of same */
	List	   *changedIndexOids;	/* OIDs of indexes to rebuild */
	List	   *changedIndexDefs;	/* string definitions of same */
	char	   *replicaIdentityIndex;	/* index to reset as REPLICA IDENTITY */
	char	   *clusterOnIndex;	/* index to use for CLUSTER */
	List       *new_crsds; /* new column reference storage directives */
	List	   *repack_cols; /* List of SortBy nodes for REPACK */
} AlteredTableInfo;

/* Struct describing one new constraint to check in Phase 3 scan */
/* Note: new NOT NULL constraints are handled elsewhere */
typedef struct NewConstraint
{
	NodeTag		type;

	char	   *name;			/* Constraint name, or NULL if none */
	ConstrType	contype;		/* CHECK or FOREIGN */
	Oid			refrelid;		/* PK rel, if FOREIGN */
	Oid			refindid;		/* OID of PK's index, if FOREIGN */
	Oid			conid;			/* OID of pg_constraint entry, if FOREIGN */
	Node	   *qual;			/* Check expr or CONSTR_FOREIGN Constraint */
	ExprState  *qualstate;		/* Execution state for CHECK expr */
} NewConstraint;

/*
 * Struct describing one new column value that needs to be computed during
 * Phase 3 copy (this could be either a new column with a non-null default, or
 * a column that we're changing the type of).  Columns without such an entry
 * are just copied from the old table during ATRewriteTable.  Note that the
 * expr is an expression over *old* table values, except when is_generated
 * is true; then it is an expression over columns of the *new* tuple.
 */
typedef struct NewColumnValue
{
	NodeTag		type;

	AttrNumber	attnum;			/* which column */
	Expr	   *expr;			/* expression to compute */
	ExprState  *exprstate;		/* execution state */
	bool		is_generated;	/* is it a GENERATED expression? */
	List 		*new_encoding; 	/* new column encoding options */
	AOCSWriteColumnOperation op;/* operation being performed */
} NewColumnValue;

#endif							/* ALTERTABLENODES_H */
