@rem = '--*-Perl-*--
@set "ErrorLevel="
@if "%OS%" == "Windows_NT" @goto WinNT
@perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
@set ErrorLevel=%ErrorLevel%
@goto endofperl
:WinNT
@perl -x -S %0 %*
@set ErrorLevel=%ErrorLevel%
@if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" @goto endofperl
@if %ErrorLevel% == 9009 @echo You do not have Perl in your PATH.
@goto endofperl
@rem ';
#!perl
#line 30

	## sha3sum: filter for computing SHA-3 digests (ref. shasum)
	##
	## Copyright (C) 2012-2022 Mark Shelor, All Rights Reserved
	##
	## Version: 1.05
	## Mon 08 Aug 2022 11:30:32 AM MST

	## sha3sum SYNOPSIS adapted from GNU Coreutils sha1sum. Add
	## "-a" option for algorithm selection,
	## "-U" option for Universal Newlines support, and
	## "-0" option for reading bit strings.

BEGIN { pop @INC if $INC[-1] eq '.' }

use strict;
use warnings;
use Fcntl;
use Getopt::Long;
use Digest::SHA3 qw($errmsg);

my $POD = <<'END_OF_POD';

=head1 NAME

sha3sum - Print or Check SHA-3 Checksums

=head1 SYNOPSIS

 Usage: sha3sum [OPTION]... [FILE]...
 Print or check SHA-3 checksums.
 With no FILE, or when FILE is -, read standard input.

   -a, --algorithm   224 (default), 256, 384, 512, 128000, 256000
   -b, --binary      read in binary mode
   -c, --check       read SHA-3 sums from the FILEs and check them
       --tag         create a BSD-style checksum
   -t, --text        read in text mode (default)
   -U, --UNIVERSAL   read in Universal Newlines mode
                         produces same digest on Windows/Unix/Mac
   -0, --01          read in BITS mode
                         ASCII '0' interpreted as 0-bit,
                         ASCII '1' interpreted as 1-bit,
                         all other characters ignored

 The following five options are useful only when verifying checksums:
       --ignore-missing  don't fail or report status for missing files
   -q, --quiet           don't print OK for each successfully verified file
   -s, --status          don't output anything, status code shows success
       --strict          exit non-zero for improperly formatted checksum lines
   -w, --warn            warn about improperly formatted checksum lines

   -h, --help        display this help and exit
   -v, --version     output version information and exit

 The sums are computed as described in the FIPS 202 SHA-3 submission.
 When checking, the input should be a former output of this program.
 The default mode is to print a line with checksum, a character
 indicating type (`*' for binary, ` ' for text, `U' for UNIVERSAL,
 `^' for BITS), and name for each FILE.  The line starts with a `\'
 character if the FILE name contains either newlines or backslashes,
 which are then replaced by the two-character sequences `\n' and
 `\\' respectively.

 Report sha3sum bugs to mshelor@cpan.org

=head1 DESCRIPTION

Running I<sha3sum> is often the quickest way to compute SHA-3 message
digests.  The user simply feeds data to the script through files or
standard input, and then collects the results from standard output.

The following command shows how to compute digests for typical inputs
such as the NIST test vector "abc":

	perl -e "print qq(abc)" | sha3sum

Or, if you want to use SHA3-256 instead of the default SHA3-224,
simply say:

	perl -e "print qq(abc)" | sha3sum -a 256

Unlike many other digest computation programs, I<sha3sum> implements
the full SHA-3 standard by allowing partial-byte inputs, which can
be recognized through the BITS option (I<-0>).  The following example
computes the SHA3-384 digest of the 7-bit message I<0001100>:

	perl -e "print qq(0001100)" | sha3sum -0 -a 384

=head1 AUTHOR

Copyright (C) 2012-2022 Mark Shelor <mshelor@cpan.org>.

=head1 SEE ALSO

I<sha3sum> is implemented using the Perl module L<Digest::SHA3>.

=cut

END_OF_POD

my $VERSION = "1.05";

sub usage {
	my($err, $msg) = @_;

	$msg = "" unless defined $msg;
	if ($err) {
		warn($msg . "Type sha3sum -h for help\n");
		exit($err);
	}
	my($USAGE) = $POD =~ /SYNOPSIS(.+?)^=/sm;
	$USAGE =~ s/^\s*//;
	$USAGE =~ s/\s*$//;
	$USAGE =~ s/^ //gm;
	print $USAGE, "\n";
	exit($err);
}


	## Sync stdout and stderr by forcing a flush after every write

select((select(STDOUT), $| = 1)[0]);
select((select(STDERR), $| = 1)[0]);


	## Collect options from command line

my ($alg, $binary, $check, $text, $status, $quiet, $warn, $help);
my ($version, $BITS, $UNIVERSAL, $tag, $strict, $ignore_missing);

eval { Getopt::Long::Configure ("bundling") };
GetOptions(
	'b|binary' => \$binary, 'c|check' => \$check,
	't|text' => \$text, 'a|algorithm=i' => \$alg,
	's|status' => \$status, 'w|warn' => \$warn,
	'q|quiet' => \$quiet,
	'h|help' => \$help, 'v|version' => \$version,
	'U|UNIVERSAL' => \$UNIVERSAL,
	'0|01' => \$BITS,
	'tag' => \$tag,
	'strict' => \$strict,
	'ignore-missing' => \$ignore_missing,
) or usage(1, "");


	## Deal with help requests and incorrect uses

usage(0)
	if $help;
usage(1, "sha3sum: Ambiguous file mode\n")
	if scalar(grep {defined $_}
		($binary, $text, $BITS, $UNIVERSAL)) > 1;
usage(1, "sha3sum: --warn option used only when verifying checksums\n")
	if $warn && !$check;
usage(1, "sha3sum: --status option used only when verifying checksums\n")
	if $status && !$check;
usage(1, "sha3sum: --quiet option used only when verifying checksums\n")
	if $quiet && !$check;
usage(1, "sha3sum: --ignore-missing option used only when verifying checksums\n")
	if $ignore_missing && !$check;
usage(1, "sha3sum: --strict option used only when verifying checksums\n")
	if $strict && !$check;
usage(1, "sha3sum: --tag does not support --text mode\n")
	if $tag && $text;
usage(1, "sha3sum: --tag does not support Universal Newlines mode\n")
	if $tag && $UNIVERSAL;
usage(1, "sha3sum: --tag does not support BITS mode\n")
	if $tag && $BITS;


	## Default to SHA3-224 unless overridden by command line option

my %isAlg = map { $_ => 1 } (224, 256, 384, 512, 128000, 256000);
$alg = 224 unless defined $alg;
usage(1, "sha3sum: Unrecognized algorithm\n") unless $isAlg{$alg};

my %Tag = map { $_ => "SHA3-$_" } (224, 256, 384, 512);
$Tag{128000} = "SHAKE128";
$Tag{256000} = "SHAKE256";


	## Display version information if requested

if ($version) {
	print "$VERSION\n";
	exit(0);
}


	## Try to figure out if the OS is DOS-like.  If it is,
	## default to binary mode when reading files, unless
	## explicitly overridden by command line "--text" or
	## "--UNIVERSAL" options.

my $isDOSish = ($^O =~ /^(MSWin\d\d|os2|dos|mint|cygwin)$/);
if ($isDOSish) { $binary = 1 unless $text || $UNIVERSAL }

my $modesym = $binary ? '*' : ($UNIVERSAL ? 'U' : ($BITS ? '^' : ' '));


	## Read from STDIN (-) if no files listed on command line

@ARGV = ("-") unless @ARGV;


	## sumfile($file): computes SHA-3 digest of $file

sub sumfile {
	my $file = shift;

	my $mode = $binary ? 'b' : ($UNIVERSAL ? 'U' : ($BITS ? '0' : ''));
	my $digest = eval { Digest::SHA3->new($alg)->addfile($file, $mode) };
	if ($@) { warn "sha3sum: $file: $errmsg\n"; return }
	$digest->hexdigest;
}


	## %len2alg: maps hex digest length to SHA-3 algorithm

my %len2alg = (56 => 224, 64 => 256, 96 => 384, 128 => 512,
		336 => 128000, 272 => 256000);


	## unescape: convert backslashed filename to plain filename

sub unescape {
	$_ = shift;
	s/\\\\/\0/g;
	s/\\n/\n/g;
	s/\0/\\/g;
	return $_;
}


	## verify: confirm the digest values in a checksum file

sub verify {
	my $checkfile = shift;
	my ($err, $fmt_errs, $read_errs, $match_errs) = (0, 0, 0, 0);
	my ($num_fmt_OK, $num_OK) = (0, 0);
	my ($bslash, $fcn, $sum, $fname, $rsp, $digest, $isOK);

	local *FH;
	$checkfile eq '-' and open(FH, '< -')
		and $checkfile = 'standard input'
	or sysopen(FH, $checkfile, O_RDONLY)
		or die "sha3sum: $checkfile: $!\n";
	while (<FH>) {
		next if /^#/;
		if (/^[ \t]*\\?(SHA3-|SHAKE)/) {
			$modesym = '*';
			($bslash, $fcn, $alg, $fname, $sum) =
		/^[ \t]*(\\?)(SHA3-|SHAKE)(\d+) \((.+)\) = ([\da-fA-F]+)/;
			if (defined $fcn and $fcn eq 'SHAKE') {
				$alg *= 1000 if defined $alg;
			}
		}
		else {
			($bslash, $sum, $modesym, $fname) =
			/^[ \t]*(\\?)([\da-fA-F]+)[ \t]([ *^U])(.+)/;
			$alg = defined $sum ? $len2alg{length($sum)} : undef;
		}
		if (grep { ! defined $_ } ($alg, $sum, $modesym, $fname) or
			! $isAlg{$alg}) {
			warn("sha3sum: $checkfile: $.: improperly formatted" .
				" SHA3 checksum line\n") if $warn;
			$fmt_errs++;
			$err = 1 if $strict;
			next;
		}
		$num_fmt_OK++;
		$fname = unescape($fname) if $bslash;
		next if $ignore_missing && ! -e $fname;
		$rsp = "$fname: ";
		($binary, $text, $UNIVERSAL, $BITS) =
			map { $_ eq $modesym } ('*', ' ', 'U', '^');
		$isOK = 0;
		unless ($digest = sumfile($fname)) {
			$rsp .= "FAILED open or read\n";
			$err = 1; $read_errs++;
		}
		elsif (lc($sum) eq $digest) {
			$rsp .= "OK\n";
			$isOK = 1;
			$num_OK++;
		}
		else { $rsp .= "FAILED\n"; $err = 1; $match_errs++ }
		print $rsp unless ($status || ($quiet && $isOK));
	}
	close(FH);
	if (! $num_fmt_OK) {
		warn("sha3sum: $checkfile: no properly formatted " .
			"SHA3 checksum lines found\n");
		$err = 1;
	}
	elsif (! $status) {
		warn("sha3sum: WARNING: $fmt_errs line" . ($fmt_errs>1?
		's are':' is') . " improperly formatted\n") if $fmt_errs;
		warn("sha3sum: WARNING: $read_errs listed file" .
		($read_errs>1?'s':'') . " could not be read\n") if $read_errs;
		warn("sha3sum: WARNING: $match_errs computed checksum" .
		($match_errs>1?'s':'') . " did NOT match\n") if $match_errs;
	}
	if ($ignore_missing && ! $num_OK && $num_fmt_OK) {
		warn("sha3sum: $checkfile: no file was verified\n")
			unless $status;
		$err = 1;
	}
	return($err == 0);
}


	## Verify or compute SHA-3 checksums of requested files

my($file, $digest);
my $STATUS = 0;
for $file (@ARGV) {
	if ($check) { $STATUS = 1 unless verify($file) }
	elsif ($digest = sumfile($file)) {
		if ($file =~ /[\n\\]/) {
			$file =~ s/\\/\\\\/g; $file =~ s/\n/\\n/g;
			print "\\";
		}
		unless ($tag) { print "$digest $modesym$file\n" }
		else          { print "$Tag{$alg} ($file) = $digest\n" }
	}
	else { $STATUS = 1 }
}
exit($STATUS);
__END__
:endofperl
@set "ErrorLevel=" & @goto _undefined_label_ 2>NUL || @"%COMSPEC%" /d/c @exit %ErrorLevel%
