/******************************************************************************
* Copyright (C) 2024-2025 Advanced Micro Devices, Inc.  All rights reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

/*****************************************************************************/
/**
*
* @file	xilsecure_versal_net_rsa_client_example.c
* @addtogroup xsecure_rsa_client_apis XilSecure RSA CLIENT APIs
* @{
* This example illustrates RSA APIs usage for CRT and Expopt.
*- Firstly data will be decrypted with Private key and generates RSA
* signature also verifies with expected signature.
* - Then signature will be encrypted with public key and generates the actual
* data also verifies with actual data.
*
* To build this application, xilmailbox library must be included in BSP and xilsecure
* must be in client mode
* This example is supported for Versal and Versal Net devices.
*
* Procedure to link and compile the example for the default ddr less designs
* ------------------------------------------------------------------------------------------------------------
* The default linker settings places a software stack, heap and data in DDR memory. For this example to work,
* any data shared between client running on A72/R5/PL and server running on PMC, should be placed in area
* which is accessible to both client and server.
*
* Following is the procedure to compile the example on OCM or any memory region which can be accessed by server
*
*		1. Open example linker script(lscript.ld) in Vitis project and section to memory mapping should
*			be updated to point all the required sections to shared memory(OCM or TCM)
*			using a memory region drop down selection
*
*						OR
*
*		1. In linker script(lscript.ld) user can add new memory section in source tab as shown below
*			.sharedmemory : {
*   			. = ALIGN(4);
*   			__sharedmemory_start = .;
*   			*(.sharedmemory)
*   			*(.sharedmemory.*)
*   			*(.gnu.linkonce.d.*)
*   			__sharedmemory_end = .;
* 			} > versal_cips_0_pspmc_0_psv_ocm_ram_0_psv_ocm_ram_0
*
* 		2. In this example ".data" section elements that are passed by reference to the server-side should
* 		   be stored in the above shared memory section. To make it happen in below example,
*		   replace ".data" in attribute section with ".sharedmemory". For example,
* 		   static const u8 Data __attribute__ ((section (".data.Data")))
* 					should be changed to
* 		   static const u8 Data __attribute__ ((section (".sharedmemory.Data")))
*
* To keep things simple, by default the cache is disabled for this example
*
* MODIFICATION HISTORY:
* <pre>
* Ver   Who    Date     Changes
* ----- ------ -------- -------------------------------------------------
* 1.0   kpt    05/22/24 Initial release.
* 5.4   kpt    06/28/24 Fix compilation failure and remove unused varialbe
*       kpt    06/30/24 Removed endianness changes
*
* </pre>
******************************************************************************/

/***************************** Include Files *********************************/
#include "xil_cache.h"
#include "xparameters.h"
#include "xil_util.h"
#include "xsecure_rsaclient.h"
#include "xsecure_katclient.h"
#ifndef VERSAL_2VE_2VM
#include "xsecure_plat_client.h"
#endif
/************************** Constant Definitions *****************************/
#define XSECURE_RSA_SIZE		(512U)	/**< 512 bytes for 4096 bit data */
#define XSECURE_RSA_SHARED_DATA_SIZE (XSECURE_RSA_SIZE + XSECURE_RSA_SIZE+\
									XSECURE_RSA_SIZE + XSECURE_RSA_SIZE)
#define XSECURE_SHARED_BUF_SIZE		(XSECURE_SHARED_MEM_SIZE +\
						XSECURE_RSA_SHARED_DATA_SIZE)

#define XSECURE_PRIME_FACTOR_P_SIZE	(256U)  /**< 256 bytes size of first prime factor(P) */
#define XSECURE_PRIME_FACTOR_Q_SIZE	(256U)  /**< 256 bytes size of first prime factor(Q) */

/**************************** Type Definitions *******************************/

/*
 * Below defined data is in Big Endian order and
 * it will be converted to little endian in the example
 * as the API expects to be in that order.
 */
static const u8 P[XSECURE_PRIME_FACTOR_P_SIZE] = {
0xE6U, 0x16U, 0x56U, 0xE6U, 0x71U, 0x20U, 0xA4U, 0xB9U, 0x9AU, 0xFAU, 0x59U, 0xD9U, 0x31U, 0x1FU, 0xABU, 0xBFU,
0x0EU, 0x6BU, 0xD1U, 0xF0U, 0xC1U, 0xBEU, 0x4EU, 0x14U, 0xF4U, 0x3FU, 0x9CU, 0x8CU, 0xFAU, 0x70U, 0x88U, 0x3AU,
0x6AU, 0x98U, 0x55U, 0x43U, 0xCCU, 0x95U, 0xECU, 0x64U, 0x3FU, 0xA2U, 0xFDU, 0xE0U, 0xBAU, 0xA9U, 0xE3U, 0xFCU,
0xB6U, 0x4CU, 0xB8U, 0xD0U, 0x92U, 0x57U, 0x02U, 0x7AU, 0x44U, 0xEDU, 0xEEU, 0xFBU, 0xD8U, 0xE9U, 0xD2U, 0x63U,
0x9AU, 0xD2U, 0x0EU, 0x8AU, 0xC7U, 0x40U, 0x32U, 0x79U, 0x67U, 0x5DU, 0xEBU, 0xF1U, 0xDCU, 0xBFU, 0xC0U, 0x63U,
0x08U, 0x51U, 0x2CU, 0x86U, 0x98U, 0x27U, 0xEDU, 0x47U, 0xB5U, 0x7CU, 0xDBU, 0x20U, 0x60U, 0x5CU, 0x4AU, 0xEBU,
0xCAU, 0x26U, 0xD4U, 0xE4U, 0x10U, 0x80U, 0x96U, 0x6FU, 0xBAU, 0x13U, 0x46U, 0x95U, 0xBBU, 0xE7U, 0x16U, 0x1CU,
0x3DU, 0xC4U, 0x55U, 0x1FU, 0x09U, 0x66U, 0xA3U, 0xA5U, 0x30U, 0x44U, 0xFEU, 0x00U, 0x15U, 0x25U, 0xFBU, 0x0BU,
0x7DU, 0x3EU, 0x8DU, 0x89U, 0x2DU, 0x79U, 0xD6U, 0x7FU, 0xCDU, 0x4AU, 0x1DU, 0x6CU, 0xEEU, 0x3EU, 0xD9U, 0xDDU,
0xB3U, 0x12U, 0x7AU, 0xFCU, 0x82U, 0xAFU, 0xD1U, 0xABU, 0x36U, 0xFCU, 0x93U, 0xFDU, 0xFFU, 0x01U, 0x9DU, 0x21U,
0x03U, 0xB4U, 0x01U, 0x48U, 0x3FU, 0x99U, 0xD2U, 0x81U, 0xDCU, 0x03U, 0x45U, 0xB4U, 0x10U, 0xF7U, 0x08U, 0x8CU,
0xE3U, 0x5DU, 0x3BU, 0xB3U, 0x1CU, 0x96U, 0x46U, 0xB7U, 0x83U, 0x2DU, 0xCFU, 0x94U, 0xD3U, 0x5EU, 0xCAU, 0x66U,
0x38U, 0x55U, 0x82U, 0xAEU, 0x5FU, 0xB8U, 0x39U, 0xFEU, 0xA7U, 0xECU, 0xB0U, 0x79U, 0x0FU, 0x71U, 0x20U, 0x0DU,
0x02U, 0x55U, 0xBCU, 0x82U, 0x86U, 0x6BU, 0xA7U, 0x1FU, 0x07U, 0xDFU, 0x36U, 0xF1U, 0xF7U, 0x5BU, 0xD7U, 0xCEU,
0x3CU, 0xA3U, 0xB0U, 0x30U, 0xF7U, 0xA6U, 0x47U, 0xBAU, 0x2AU, 0xABU, 0x4FU, 0xAFU, 0x83U, 0xD4U, 0x17U, 0x37U,
0x94U, 0xB8U, 0x78U, 0x1DU, 0xD3U, 0xB6U, 0x58U, 0x4DU, 0xC4U, 0x91U, 0xC1U, 0xABU, 0xDBU, 0xEDU, 0xE7U, 0x9FU
};

static const u8 Q[XSECURE_PRIME_FACTOR_Q_SIZE] = {
0xE3U, 0x9AU, 0x8AU, 0x30U, 0x2AU, 0x0AU, 0xBBU, 0x94U, 0xA4U, 0x62U, 0x58U, 0x37U, 0xDFU, 0x9DU, 0xEAU, 0x1FU,
0x6DU, 0x7BU, 0x27U, 0xC5U, 0x0AU, 0x11U, 0xB8U, 0xD9U, 0xA4U, 0x72U, 0x25U, 0xD7U, 0x3DU, 0x6DU, 0x73U, 0xC2U,
0x03U, 0xD0U, 0x51U, 0x50U, 0xB3U, 0x58U, 0xC6U, 0x4BU, 0xBAU, 0xAAU, 0x69U, 0x24U, 0x73U, 0x37U, 0x50U, 0x31U,
0xB5U, 0x83U, 0x86U, 0x12U, 0xEFU, 0xC0U, 0x26U, 0x35U, 0x5EU, 0x57U, 0xA7U, 0x64U, 0x63U, 0x4BU, 0x62U, 0x71U,
0xBFU, 0xF3U, 0x9CU, 0x43U, 0x05U, 0xC8U, 0xCCU, 0x19U, 0x6EU, 0xB1U, 0x6CU, 0xEFU, 0xBEU, 0xD5U, 0xA9U, 0xBDU,
0x79U, 0xFFU, 0x49U, 0x8CU, 0xC5U, 0xD9U, 0x95U, 0x7BU, 0xDBU, 0xA6U, 0x0AU, 0x28U, 0x7DU, 0xAFU, 0x46U, 0xEEU,
0xB8U, 0x63U, 0x86U, 0x05U, 0x44U, 0xF8U, 0xE4U, 0x5DU, 0xB6U, 0x44U, 0x74U, 0x25U, 0x37U, 0x04U, 0x32U, 0xE9U,
0xE0U, 0x30U, 0x2BU, 0x47U, 0x03U, 0x5CU, 0x3CU, 0x80U, 0x60U, 0xFEU, 0x81U, 0x11U, 0x8EU, 0x90U, 0x8EU, 0xDDU,
0x46U, 0xB5U, 0xACU, 0xC5U, 0x10U, 0x4EU, 0xE9U, 0x96U, 0xCAU, 0x92U, 0x1FU, 0xE5U, 0x4AU, 0x30U, 0x0DU, 0xB4U,
0x1DU, 0x47U, 0xF1U, 0xCAU, 0x94U, 0x4BU, 0xD9U, 0x3EU, 0x55U, 0x77U, 0xF0U, 0x3EU, 0xA1U, 0xCEU, 0x1FU, 0x25U,
0x6AU, 0x21U, 0x25U, 0x50U, 0x82U, 0x21U, 0x24U, 0xC2U, 0x8EU, 0xF7U, 0x68U, 0x78U, 0x47U, 0x69U, 0x2EU, 0xB7U,
0x36U, 0x6AU, 0x6BU, 0x71U, 0x9CU, 0xC0U, 0x1DU, 0x59U, 0x46U, 0xE4U, 0xA8U, 0x40U, 0xB0U, 0xE7U, 0x87U, 0x73U,
0xA4U, 0x7FU, 0x92U, 0xC4U, 0xF0U, 0xEDU, 0x89U, 0xCCU, 0x52U, 0x14U, 0x23U, 0x72U, 0x9AU, 0x17U, 0xC6U, 0x0DU,
0xCAU, 0x40U, 0x7DU, 0xB9U, 0x21U, 0x13U, 0x94U, 0xA9U, 0x82U, 0x25U, 0x9BU, 0x88U, 0x01U, 0x95U, 0xF3U, 0xA3U,
0x9CU, 0xD0U, 0x4EU, 0x75U, 0x2BU, 0x91U, 0x35U, 0x84U, 0xA2U, 0xFCU, 0x1FU, 0xBBU, 0x1EU, 0x25U, 0xC8U, 0x50U,
0xE5U, 0xCBU, 0x90U, 0x96U, 0x10U, 0x3DU, 0x50U, 0x43U, 0x00U, 0x59U, 0x2AU, 0xF5U, 0xB1U, 0x25U, 0x28U, 0x3BU
};

static const u8 Tot[XSECURE_RSA_SIZE] = {
0xCCU, 0x90U, 0xB4U, 0xB1U, 0xF1U, 0xC1U, 0xD3U, 0x9DU, 0x98U, 0xB1U, 0x34U, 0x49U, 0x92U, 0xCDU, 0xFDU, 0xD1U,
0xCFU, 0x87U, 0x43U, 0x52U, 0xAFU, 0x9CU, 0x5CU, 0xFBU, 0x62U, 0xC9U, 0xAAU, 0x7EU, 0xCAU, 0xFDU, 0x0EU, 0x2AU,
0xF9U, 0xBFU, 0x86U, 0xBCU, 0xDEU, 0xFDU, 0x10U, 0x42U, 0x79U, 0xF2U, 0x18U, 0x6BU, 0xB1U, 0xCFU, 0x8DU, 0x66U,
0xDEU, 0xB0U, 0x64U, 0x1EU, 0xB1U, 0x45U, 0x45U, 0x20U, 0x80U, 0xCCU, 0xCBU, 0x6AU, 0xADU, 0x1AU, 0x0AU, 0x2AU,
0xA9U, 0xE2U, 0xA6U, 0x41U, 0xABU, 0xD3U, 0x3AU, 0xEDU, 0x34U, 0x71U, 0x2EU, 0x2DU, 0x08U, 0xF5U, 0x16U, 0x7BU,
0x89U, 0xEBU, 0xBAU, 0xD5U, 0x86U, 0x33U, 0x0FU, 0x26U, 0x80U, 0x9FU, 0x3CU, 0x98U, 0x11U, 0x62U, 0xDCU, 0x12U,
0x41U, 0xE9U, 0x7AU, 0x3CU, 0xD9U, 0x1BU, 0x0CU, 0xE5U, 0x33U, 0x55U, 0x25U, 0x51U, 0xEFU, 0x2DU, 0xF2U, 0x3EU,
0x6EU, 0x7FU, 0xF9U, 0x76U, 0x86U, 0x6CU, 0xA9U, 0x3CU, 0x04U, 0xFFU, 0xADU, 0xD1U, 0x9DU, 0x93U, 0xC1U, 0x04U,
0x4CU, 0x55U, 0xABU, 0xFDU, 0xE7U, 0x71U, 0xAFU, 0xF5U, 0xA6U, 0xE1U, 0x51U, 0xE7U, 0xBAU, 0x4DU, 0x94U, 0x22U,
0xD2U, 0x07U, 0x7BU, 0x64U, 0x11U, 0xC2U, 0x56U, 0xFAU, 0xBEU, 0x7CU, 0x1DU, 0x5FU, 0x57U, 0x66U, 0x4EU, 0x88U,
0xD4U, 0x58U, 0x1FU, 0x12U, 0x07U, 0x65U, 0x77U, 0xA3U, 0x2AU, 0xF4U, 0x8DU, 0xDEU, 0xDAU, 0x3CU, 0xA1U, 0x0DU,
0x41U, 0x3EU, 0x58U, 0x77U, 0x1EU, 0x59U, 0x71U, 0xDBU, 0xD2U, 0xDDU, 0x48U, 0x20U, 0xE8U, 0xB9U, 0x09U, 0xB0U,
0xD5U, 0x95U, 0x26U, 0xCFU, 0xAEU, 0x4FU, 0x31U, 0xDFU, 0x10U, 0xE7U, 0x7BU, 0x34U, 0x44U, 0x3EU, 0x2DU, 0x4EU,
0x7EU, 0x9BU, 0xBAU, 0x9CU, 0xBDU, 0xB4U, 0xC9U, 0xD9U, 0x8AU, 0x09U, 0x74U, 0x75U, 0x3DU, 0xD2U, 0x9FU, 0xDFU,
0x1AU, 0xE8U, 0x6AU, 0xA0U, 0x09U, 0xD3U, 0x66U, 0x04U, 0x8DU, 0xA8U, 0x7BU, 0x4BU, 0x2EU, 0x3AU, 0x42U, 0xACU,
0xD7U, 0x24U, 0xFFU, 0x5FU, 0x9CU, 0x09U, 0x7FU, 0xEFU, 0x22U, 0xA5U, 0x5AU, 0x7EU, 0x3BU, 0x63U, 0x36U, 0x37U,
0x2FU, 0x3BU, 0x1EU, 0x98U, 0x62U, 0xB2U, 0xB9U, 0x67U, 0xF9U, 0x65U, 0xDFU, 0x72U, 0x64U, 0xA1U, 0xF0U, 0xC0U,
0x2BU, 0xB2U, 0xA8U, 0x0AU, 0xA9U, 0xAEU, 0x3CU, 0x9EU, 0x4EU, 0x99U, 0xA3U, 0xE8U, 0x03U, 0xDDU, 0x15U, 0x42U,
0x84U, 0x34U, 0xDDU, 0xB5U, 0xAFU, 0xBEU, 0x0CU, 0x7CU, 0xC8U, 0x5CU, 0x30U, 0xD7U, 0x96U, 0x2DU, 0xD4U, 0xFEU,
0x60U, 0x4DU, 0x18U, 0xF2U, 0xB2U, 0x99U, 0x1BU, 0x0AU, 0x58U, 0x00U, 0x32U, 0xF1U, 0xACU, 0x52U, 0x04U, 0xBAU,
0xEEU, 0xBBU, 0x74U, 0x40U, 0xA0U, 0x11U, 0x51U, 0x2FU, 0x2EU, 0xDFU, 0x11U, 0x0EU, 0xCCU, 0x39U, 0x72U, 0x72U,
0x43U, 0x3CU, 0x58U, 0x7EU, 0x2BU, 0x2AU, 0xD4U, 0x38U, 0x67U, 0x4DU, 0xAAU, 0x70U, 0xF1U, 0x95U, 0x2BU, 0xC5U,
0x08U, 0x6DU, 0x6BU, 0x04U, 0x4BU, 0xDEU, 0x59U, 0x59U, 0xFFU, 0xDDU, 0x1AU, 0x9EU, 0x77U, 0xA7U, 0xFFU, 0x0AU,
0xE2U, 0x0EU, 0x50U, 0x2CU, 0xD9U, 0x58U, 0xC3U, 0xD7U, 0x66U, 0xC0U, 0xE4U, 0xF6U, 0x68U, 0x24U, 0xBFU, 0xABU,
0x1FU, 0xC0U, 0x13U, 0x1AU, 0xADU, 0xD6U, 0x2AU, 0x94U, 0x81U, 0x5AU, 0x66U, 0x7DU, 0x00U, 0xFBU, 0xF5U, 0x92U,
0x49U, 0xF4U, 0x03U, 0x9DU, 0xDDU, 0x10U, 0x81U, 0x6BU, 0x42U, 0x3EU, 0xF4U, 0x17U, 0x49U, 0x5DU, 0xF4U, 0x95U,
0x80U, 0x83U, 0x5FU, 0x67U, 0x80U, 0xDEU, 0x98U, 0x8AU, 0xC3U, 0x25U, 0x1DU, 0x79U, 0x07U, 0xE4U, 0x9DU, 0x79U,
0xEBU, 0x65U, 0x2DU, 0x3EU, 0x1FU, 0x5DU, 0xD4U, 0xBAU, 0xB6U, 0x26U, 0x00U, 0x6CU, 0xBFU, 0x66U, 0x99U, 0xCBU,
0xE0U, 0x30U, 0xE6U, 0xA0U, 0x56U, 0x85U, 0xA3U, 0xB7U, 0x08U, 0x4BU, 0xA0U, 0xA3U, 0x01U, 0x47U, 0x42U, 0xB6U,
0x6CU, 0xF6U, 0xB5U, 0x26U, 0x0BU, 0x80U, 0x24U, 0x78U, 0xF9U, 0x94U, 0xAEU, 0xD3U, 0xD6U, 0x8FU, 0xB1U, 0x5BU,
0x13U, 0x6FU, 0xF7U, 0x4AU, 0xF3U, 0xF3U, 0xB5U, 0xB3U, 0x78U, 0xE3U, 0x55U, 0x28U, 0x7EU, 0xC7U, 0xF9U, 0x74U,
0x4DU, 0x9FU, 0x69U, 0x0BU, 0x53U, 0x91U, 0x57U, 0xECU, 0x28U, 0xCBU, 0x98U, 0x09U, 0xB7U, 0xEDU, 0x29U, 0xCCU
};

static const u8 DP[XSECURE_RSA_SIZE/2U] = {
0x8BU, 0xC9U, 0x36U, 0xB2U, 0x7AU, 0x38U, 0x62U, 0xDBU, 0x80U, 0x94U, 0x68U, 0xD1U, 0xD3U, 0x4EU, 0x72U, 0x81U,
0x78U, 0xB9U, 0x78U, 0xAAU, 0xD4U, 0x8AU, 0x51U, 0xD5U, 0xD9U, 0x2EU, 0x6CU, 0x66U, 0x1AU, 0x21U, 0x14U, 0x22U,
0xA9U, 0x70U, 0x33U, 0x1DU, 0x25U, 0xC6U, 0x4FU, 0xFFU, 0xF7U, 0xC9U, 0x86U, 0x84U, 0x69U, 0xE3U, 0x9DU, 0x67U,
0x83U, 0x39U, 0xC9U, 0x6EU, 0xADU, 0xF9U, 0xBEU, 0xBFU, 0x99U, 0xF1U, 0x14U, 0x28U, 0x5DU, 0xF8U, 0x70U, 0xA1U,
0xA3U, 0x05U, 0xCBU, 0x5FU, 0xF4U, 0xFDU, 0xC3U, 0xACU, 0x8EU, 0x68U, 0x39U, 0x58U, 0x8FU, 0xCCU, 0xCFU, 0x8DU,
0x5BU, 0x18U, 0x35U, 0xFCU, 0xEFU, 0xA4U, 0x12U, 0x3CU, 0x66U, 0xBEU, 0x9CU, 0x42U, 0x67U, 0x48U, 0xABU, 0x1BU,
0xE5U, 0x8DU, 0x01U, 0xFAU, 0x2DU, 0x2CU, 0x72U, 0x39U, 0x1EU, 0xCBU, 0x67U, 0x2EU, 0xC1U, 0x21U, 0xFCU, 0x04U,
0xDCU, 0x61U, 0xD3U, 0x75U, 0x2FU, 0xB8U, 0xF3U, 0xE3U, 0x00U, 0x18U, 0x6AU, 0x58U, 0x92U, 0x80U, 0xA8U, 0xACU,
0xAAU, 0x42U, 0xBEU, 0xAAU, 0x80U, 0xBEU, 0x6CU, 0xDAU, 0xE8U, 0x56U, 0x06U, 0xDAU, 0x8FU, 0x44U, 0x10U, 0x08U,
0xE9U, 0x2AU, 0x69U, 0x1DU, 0xD0U, 0x28U, 0x11U, 0x31U, 0xFCU, 0x0EU, 0x27U, 0xBBU, 0x40U, 0x9AU, 0x32U, 0x64U,
0x44U, 0x63U, 0xB0U, 0x03U, 0xBCU, 0xE0U, 0x2FU, 0x4CU, 0x45U, 0xE6U, 0x97U, 0x0AU, 0x76U, 0xE4U, 0x1EU, 0x85U,
0xB9U, 0xFCU, 0x82U, 0xB0U, 0xA6U, 0xC5U, 0x85U, 0xE1U, 0x28U, 0x09U, 0x44U, 0xFCU, 0x1EU, 0x0EU, 0x90U, 0xD8U,
0xD9U, 0x90U, 0xD9U, 0xFCU, 0x26U, 0x9BU, 0x3EU, 0x60U, 0xC0U, 0x94U, 0xC4U, 0x02U, 0x48U, 0xA7U, 0x69U, 0xD9U,
0x7DU, 0x79U, 0xA9U, 0x35U, 0x83U, 0x85U, 0x13U, 0xDBU, 0x7EU, 0x64U, 0xD6U, 0x74U, 0xDAU, 0xDBU, 0x29U, 0x99U,
0x5AU, 0xADU, 0xF7U, 0xF5U, 0x47U, 0xFFU, 0xF5U, 0xC3U, 0xDCU, 0xF8U, 0x87U, 0x49U, 0x12U, 0xC6U, 0x5FU, 0xF4U,
0x9CU, 0x9EU, 0x16U, 0x1BU, 0xC8U, 0xECU, 0x17U, 0x69U, 0xE7U, 0xEAU, 0xC5U, 0xC7U, 0x2BU, 0xAEU, 0xB5U, 0xF1U
};

static const u8 DQ[XSECURE_RSA_SIZE/2U] = {
0x14U, 0x43U, 0xAFU, 0x28U, 0x08U, 0xA6U, 0x39U, 0xFBU, 0x4BU, 0x03U, 0x82U, 0x8DU, 0x3EU, 0xF8U, 0xA5U, 0x5EU,
0xB4U, 0xF5U, 0xB1U, 0x1DU, 0xC6U, 0xFEU, 0xBAU, 0xF1U, 0x8DU, 0xC5U, 0xA7U, 0xFEU, 0xD8U, 0xF6U, 0x44U, 0xC7U,
0x32U, 0x65U, 0xBAU, 0xC7U, 0xAFU, 0xA0U, 0x28U, 0xCFU, 0x97U, 0x48U, 0x12U, 0xE2U, 0x6EU, 0xEBU, 0xAEU, 0xD5U,
0x2EU, 0x2FU, 0x0EU, 0xA1U, 0xE2U, 0xFCU, 0xBDU, 0xAAU, 0x24U, 0xC3U, 0x33U, 0x21U, 0x2AU, 0x02U, 0x0DU, 0x3AU,
0xC9U, 0x77U, 0x5CU, 0x23U, 0x4FU, 0x81U, 0x88U, 0x42U, 0x7FU, 0x6BU, 0x86U, 0xA2U, 0x75U, 0xE0U, 0x3AU, 0x23U,
0xA7U, 0x98U, 0x06U, 0x29U, 0x65U, 0x78U, 0x30U, 0xE9U, 0xFDU, 0x2DU, 0x74U, 0x5CU, 0x14U, 0x8CU, 0xA5U, 0x89U,
0x1DU, 0x9BU, 0xA6U, 0xD4U, 0x13U, 0x47U, 0x01U, 0xB6U, 0x22U, 0x43U, 0x48U, 0x00U, 0x04U, 0x3CU, 0xF8U, 0x77U,
0x80U, 0x41U, 0x69U, 0xA9U, 0x01U, 0xF0U, 0x95U, 0x66U, 0x6CU, 0xBCU, 0x5CU, 0x2BU, 0x7FU, 0x00U, 0xF9U, 0xE7U,
0x53U, 0xA3U, 0x5BU, 0x45U, 0x94U, 0x65U, 0x22U, 0x41U, 0x0DU, 0xBAU, 0x22U, 0xE2U, 0xDCU, 0x4FU, 0x18U, 0xF9U,
0x3FU, 0x5EU, 0x27U, 0x75U, 0xFCU, 0xECU, 0x1FU, 0xE4U, 0x7FU, 0xE1U, 0x94U, 0xDFU, 0x4EU, 0xB2U, 0xB6U, 0x6DU,
0x33U, 0x5CU, 0xCEU, 0x20U, 0xB3U, 0xE5U, 0x55U, 0x8BU, 0x89U, 0x05U, 0x18U, 0x25U, 0x15U, 0x5EU, 0x95U, 0x0DU,
0x34U, 0xE8U, 0x16U, 0xB6U, 0xDFU, 0xD8U, 0xEFU, 0x04U, 0x02U, 0xDEU, 0xDAU, 0xE8U, 0x1FU, 0x98U, 0x0DU, 0xA4U,
0x4BU, 0x32U, 0x1CU, 0x94U, 0x32U, 0x4BU, 0x43U, 0xE8U, 0x90U, 0x2AU, 0x40U, 0xD9U, 0xB4U, 0x44U, 0x87U, 0x9AU,
0xB4U, 0xB6U, 0x37U, 0x3BU, 0x63U, 0x69U, 0x85U, 0xE3U, 0xEDU, 0xB5U, 0x81U, 0x7AU, 0x83U, 0x81U, 0xA1U, 0x56U,
0xA0U, 0xE0U, 0x9FU, 0x0BU, 0xA9U, 0x9CU, 0x5DU, 0x22U, 0x79U, 0x0CU, 0x6FU, 0x9AU, 0xCEU, 0x5DU, 0x58U, 0xCFU,
0x73U, 0x16U, 0x43U, 0xC2U, 0x48U, 0xA1U, 0xDCU, 0xE2U, 0x8AU, 0x46U, 0x66U, 0x1FU, 0xC7U, 0xECU, 0xE1U, 0xFFU
};

static const u8 QInv[XSECURE_RSA_SIZE/2U] = {
0xB2U, 0x2AU, 0x6FU, 0xD0U, 0x30U, 0x00U, 0x71U, 0xFAU, 0xCBU, 0xC3U, 0x7EU, 0x1DU, 0x7BU, 0x93U, 0x90U, 0xF3U,
0x81U, 0x58U, 0xC2U, 0xA4U, 0x16U, 0xB7U, 0xD9U, 0x8BU, 0x52U, 0xF2U, 0x63U, 0x8DU, 0xD6U, 0x96U, 0xFBU, 0xFEU,
0x1EU, 0x86U, 0xB3U, 0xA6U, 0x57U, 0x3DU, 0x0CU, 0x44U, 0x70U, 0x38U, 0xADU, 0xFAU, 0x75U, 0x54U, 0x7AU, 0x0BU,
0xB6U, 0xB4U, 0xF8U, 0xE4U, 0x8EU, 0xD9U, 0xBAU, 0x68U, 0x63U, 0x71U, 0x18U, 0xDCU, 0x7AU, 0x43U, 0x02U, 0x95U,
0xB6U, 0xA8U, 0x17U, 0x72U, 0x3DU, 0xDAU, 0x06U, 0x0DU, 0xA4U, 0x05U, 0x14U, 0x8DU, 0x4EU, 0xC1U, 0x94U, 0x12U,
0x25U, 0x6BU, 0x7EU, 0x30U, 0x87U, 0xC9U, 0xEDU, 0x94U, 0xDCU, 0xB6U, 0xD6U, 0x0AU, 0x14U, 0x71U, 0x72U, 0x95U,
0xF9U, 0x49U, 0x4DU, 0x2FU, 0x10U, 0xC8U, 0x33U, 0x91U, 0x71U, 0x95U, 0x71U, 0x1BU, 0x7BU, 0x61U, 0x1BU, 0x2DU,
0x70U, 0xACU, 0x49U, 0xB7U, 0xABU, 0xA8U, 0x73U, 0x25U, 0x77U, 0x25U, 0x55U, 0xE4U, 0xC7U, 0x23U, 0xF3U, 0xFBU,
0xADU, 0xE2U, 0x1AU, 0xD9U, 0x4CU, 0xF6U, 0x1AU, 0x74U, 0x4EU, 0x89U, 0x87U, 0xF2U, 0xF1U, 0x92U, 0xE5U, 0x89U,
0x49U, 0x99U, 0xE3U, 0x3AU, 0x70U, 0xEFU, 0x29U, 0x3EU, 0xACU, 0x47U, 0x46U, 0xF2U, 0x34U, 0xD9U, 0xFBU, 0x36U,
0x23U, 0xB2U, 0xD4U, 0x8EU, 0x6DU, 0xA2U, 0x0CU, 0x51U, 0x2FU, 0xE8U, 0xFDU, 0x18U, 0x8BU, 0xEFU, 0x56U, 0xE7U,
0x24U, 0x06U, 0x76U, 0x23U, 0xBEU, 0x7EU, 0x7AU, 0xDBU, 0x9FU, 0x70U, 0x6BU, 0xE7U, 0x90U, 0x1AU, 0xD1U, 0x76U,
0xB5U, 0xADU, 0xC1U, 0x3BU, 0xD9U, 0x52U, 0x86U, 0xFBU, 0xC5U, 0xBAU, 0x87U, 0x60U, 0x65U, 0xE4U, 0x7EU, 0x54U,
0x6FU, 0x11U, 0x6BU, 0xB7U, 0x5EU, 0x9AU, 0x87U, 0xEBU, 0x59U, 0xF7U, 0x51U, 0x17U, 0x45U, 0x6CU, 0x8DU, 0xFBU,
0x7BU, 0x2CU, 0x4FU, 0x65U, 0xA9U, 0x89U, 0xF8U, 0x3CU, 0x0BU, 0xC1U, 0x87U, 0x89U, 0x71U, 0x77U, 0x12U, 0x8CU,
0xB0U, 0x2EU, 0xF1U, 0xB5U, 0x92U, 0xACU, 0xC7U, 0x5AU, 0x0BU, 0x8CU, 0xEAU, 0x97U, 0x77U, 0x01U, 0xE0U, 0x6DU
};

static const u8 RN[XSECURE_RSA_SIZE] = {
0x5DU, 0x04U, 0x4EU, 0xDDU, 0xF2U, 0x51U, 0xFEU, 0xA7U, 0x57U, 0x9EU, 0xD6U, 0xF4U, 0xE4U, 0xAFU, 0x2CU, 0xF7U,
0x45U, 0x6BU, 0x4BU, 0x86U, 0x35U, 0xDDU, 0x73U, 0x99U, 0xCAU, 0x27U, 0xBFU, 0x83U, 0x24U, 0x86U, 0xD7U, 0xE0U,
0xBBU, 0x6FU, 0x3CU, 0xBEU, 0xF5U, 0x5BU, 0x80U, 0x39U, 0x2AU, 0x6AU, 0x0FU, 0xC3U, 0xACU, 0x85U, 0x66U, 0x6FU,
0x48U, 0xBDU, 0x95U, 0x08U, 0xFBU, 0x04U, 0x12U, 0x26U, 0x7EU, 0xF4U, 0xA0U, 0xE6U, 0xADU, 0x72U, 0xC3U, 0x66U,
0x6EU, 0x31U, 0x36U, 0x44U, 0x29U, 0x8DU, 0x97U, 0x29U, 0x2AU, 0xB1U, 0x74U, 0x62U, 0x13U, 0x7FU, 0xF1U, 0xA4U,
0x48U, 0xE9U, 0xC0U, 0x4FU, 0xC4U, 0x54U, 0x07U, 0x04U, 0xB4U, 0xE3U, 0xBCU, 0xE6U, 0x88U, 0xD2U, 0x9FU, 0xE8U,
0x53U, 0xF5U, 0xD9U, 0x46U, 0xEFU, 0xC2U, 0xAEU, 0xF5U, 0xADU, 0xF8U, 0x48U, 0xD2U, 0x96U, 0x42U, 0xAEU, 0xB2U,
0xA5U, 0x6DU, 0xF5U, 0x43U, 0x22U, 0x17U, 0x2DU, 0x58U, 0x39U, 0x59U, 0xE2U, 0x00U, 0xE7U, 0x14U, 0x64U, 0xFDU,
0x9DU, 0xE1U, 0x6FU, 0x76U, 0x97U, 0x35U, 0x44U, 0x55U, 0x9BU, 0xC4U, 0xF2U, 0x43U, 0xBCU, 0x13U, 0x6DU, 0xD3U,
0xF1U, 0xB3U, 0xB0U, 0xD1U, 0xF5U, 0x9CU, 0x8FU, 0x4AU, 0x0AU, 0x6EU, 0xBDU, 0x6AU, 0xDBU, 0xE9U, 0x79U, 0xECU,
0x99U, 0xECU, 0x54U, 0x27U, 0xB8U, 0x6BU, 0xA2U, 0x89U, 0x93U, 0xBDU, 0x9BU, 0x61U, 0x91U, 0x14U, 0x7CU, 0x85U,
0xEAU, 0xD8U, 0x3BU, 0xEDU, 0x7FU, 0x3AU, 0x70U, 0x28U, 0x2CU, 0x8BU, 0x5FU, 0xA8U, 0x71U, 0x49U, 0x6BU, 0xCEU,
0xB5U, 0xD3U, 0x45U, 0x07U, 0x8BU, 0xDAU, 0xF8U, 0xC1U, 0xBBU, 0x73U, 0xBFU, 0x8DU, 0xF5U, 0xCEU, 0x77U, 0xD0U,
0x9BU, 0x82U, 0x34U, 0x30U, 0x22U, 0xAAU, 0x8FU, 0x8DU, 0x76U, 0x55U, 0xA1U, 0xDDU, 0xD2U, 0x2BU, 0x57U, 0x60U,
0x87U, 0xA2U, 0x32U, 0x5FU, 0xCDU, 0xC2U, 0xAEU, 0xF9U, 0x6FU, 0x86U, 0xF7U, 0x6EU, 0xC2U, 0x83U, 0x7DU, 0x10U,
0x48U, 0xD9U, 0x01U, 0x3FU, 0x31U, 0x92U, 0x2BU, 0xDBU, 0x37U, 0x1EU, 0x15U, 0x5FU, 0x82U, 0x53U, 0xEAU, 0x19U,
0x3EU, 0x15U, 0x41U, 0x72U, 0x0AU, 0xA8U, 0x86U, 0x53U, 0x1BU, 0x12U, 0x91U, 0xD6U, 0x7AU, 0x73U, 0x98U, 0x0AU,
0x4FU, 0xFEU, 0x8DU, 0x2FU, 0xC7U, 0x14U, 0xF1U, 0x7AU, 0x25U, 0x7DU, 0x65U, 0x25U, 0x55U, 0xB1U, 0xF0U, 0xC8U,
0xF2U, 0x13U, 0x49U, 0x23U, 0x5DU, 0x60U, 0x20U, 0x9CU, 0xD9U, 0x9BU, 0x16U, 0x71U, 0x53U, 0x20U, 0x6AU, 0xCDU,
0xDDU, 0x63U, 0xEBU, 0x97U, 0x7EU, 0xABU, 0x69U, 0xB8U, 0xD3U, 0xF2U, 0xE2U, 0x6FU, 0xCAU, 0x84U, 0xF6U, 0x7EU,
0x93U, 0xEDU, 0x33U, 0x8EU, 0xBEU, 0x37U, 0x2DU, 0x21U, 0x89U, 0x33U, 0x56U, 0xC9U, 0xB7U, 0x52U, 0x91U, 0x98U,
0xE5U, 0x45U, 0x99U, 0x2DU, 0x8AU, 0x91U, 0x5FU, 0xC3U, 0x83U, 0x75U, 0xAEU, 0x93U, 0x71U, 0xA0U, 0xABU, 0xF3U,
0x50U, 0x6CU, 0x2DU, 0xF3U, 0xDAU, 0x3CU, 0x12U, 0x16U, 0x88U, 0x3AU, 0xA4U, 0x94U, 0x98U, 0x2BU, 0x22U, 0xD4U,
0xADU, 0x84U, 0x15U, 0xD2U, 0xE8U, 0x0DU, 0xBFU, 0x50U, 0x73U, 0xD7U, 0xCAU, 0x3CU, 0x8AU, 0x22U, 0x27U, 0x5FU,
0xCEU, 0xDDU, 0x3EU, 0xA0U, 0x97U, 0x98U, 0x04U, 0x99U, 0xF7U, 0xADU, 0x56U, 0xC0U, 0x94U, 0x3CU, 0x27U, 0x77U,
0x89U, 0x58U, 0xDDU, 0xF1U, 0xCBU, 0xE1U, 0x5EU, 0x80U, 0x65U, 0x0FU, 0x07U, 0x55U, 0x77U, 0x16U, 0x8AU, 0xDAU,
0x90U, 0x17U, 0xCEU, 0xE0U, 0x54U, 0xDBU, 0xC6U, 0x55U, 0x40U, 0x78U, 0x4DU, 0x6AU, 0x10U, 0x6EU, 0x29U, 0xAEU,
0x94U, 0x19U, 0xB8U, 0x8BU, 0x24U, 0x3BU, 0xC9U, 0xF6U, 0x9BU, 0xFEU, 0x70U, 0x69U, 0xCDU, 0x22U, 0xFCU, 0x65U,
0x66U, 0xDFU, 0xF8U, 0xEFU, 0x19U, 0xD9U, 0x7CU, 0x8FU, 0xDFU, 0x51U, 0xE8U, 0x97U, 0x56U, 0x49U, 0x4CU, 0xE9U,
0x92U, 0x05U, 0x22U, 0xB2U, 0x03U, 0x5AU, 0x6BU, 0xD0U, 0xE5U, 0xC4U, 0x86U, 0xDAU, 0x9DU, 0x38U, 0xCEU, 0xE7U,
0x86U, 0x80U, 0x0CU, 0x1EU, 0x2DU, 0xD7U, 0x9EU, 0x6EU, 0x77U, 0xECU, 0x15U, 0xC0U, 0x54U, 0x89U, 0xB9U, 0x23U,
0x86U, 0x66U, 0x82U, 0xE1U, 0x52U, 0xC1U, 0xBBU, 0xF2U, 0x77U, 0x51U, 0x1FU, 0x9AU, 0xA1U, 0xF7U, 0x33U, 0x39U
};

static const u8 RRN[XSECURE_RSA_SIZE] = {
0x1FU, 0x77U, 0xC1U, 0x73U, 0x9EU, 0x03U, 0x3DU, 0xF2U, 0xBEU, 0x5EU, 0xC7U, 0x2EU, 0x81U, 0xD1U, 0x72U, 0xEFU,
0xBAU, 0x37U, 0x53U, 0x59U, 0x30U, 0x62U, 0x26U, 0x54U, 0x44U, 0xBDU, 0xFAU, 0xB1U, 0xE6U, 0x89U, 0x41U, 0xA4U,
0x51U, 0x7EU, 0x5CU, 0xD3U, 0x87U, 0xEFU, 0x94U, 0xF1U, 0x93U, 0xC8U, 0x7AU, 0x05U, 0xE6U, 0xD4U, 0x28U, 0xAFU,
0x72U, 0xC1U, 0x7CU, 0x07U, 0x94U, 0x8CU, 0xE9U, 0x34U, 0xE2U, 0x98U, 0xB6U, 0x51U, 0xA7U, 0xDDU, 0x2EU, 0x2EU,
0x39U, 0xA8U, 0xBEU, 0x85U, 0xC8U, 0xC5U, 0x58U, 0x0FU, 0x26U, 0xA6U, 0x1BU, 0xB3U, 0x00U, 0x11U, 0x4AU, 0x50U,
0x83U, 0x7FU, 0x12U, 0x3CU, 0x3AU, 0x2AU, 0x85U, 0x4AU, 0xA8U, 0xABU, 0x1EU, 0x52U, 0x1EU, 0x33U, 0xECU, 0xAFU,
0x87U, 0xB9U, 0x18U, 0x7BU, 0x19U, 0x99U, 0x5FU, 0xAFU, 0x25U, 0x7FU, 0x40U, 0x35U, 0xFAU, 0x43U, 0x46U, 0xA1U,
0xB4U, 0x68U, 0x32U, 0xF3U, 0x22U, 0x69U, 0x0CU, 0xC7U, 0xABU, 0x30U, 0xE2U, 0x2EU, 0x9CU, 0x60U, 0x90U, 0xA7U,
0x74U, 0xFAU, 0x96U, 0x28U, 0xC7U, 0x06U, 0x9EU, 0xE0U, 0x8AU, 0xA9U, 0xC9U, 0x98U, 0xEDU, 0xE5U, 0x0BU, 0x55U,
0xC7U, 0x4AU, 0xF9U, 0xE8U, 0xCFU, 0x42U, 0x46U, 0xE4U, 0x54U, 0xEAU, 0x8CU, 0x90U, 0xCEU, 0x5DU, 0x02U, 0x62U,
0x7EU, 0xF4U, 0x23U, 0x5DU, 0xA4U, 0xCBU, 0xE3U, 0x4DU, 0x8EU, 0x60U, 0x78U, 0x83U, 0x57U, 0x21U, 0x01U, 0x5BU,
0xC9U, 0x24U, 0x81U, 0x5CU, 0xEDU, 0x5DU, 0xC4U, 0xD1U, 0x5BU, 0x42U, 0x40U, 0x0CU, 0x8BU, 0x85U, 0x05U, 0xEFU,
0x55U, 0xB8U, 0x7FU, 0xA6U, 0x4BU, 0x55U, 0xD7U, 0xD5U, 0x39U, 0xE1U, 0xDAU, 0xF3U, 0xACU, 0x6BU, 0xADU, 0xA8U,
0x10U, 0x7BU, 0x5DU, 0x47U, 0xA4U, 0x05U, 0x58U, 0x12U, 0xD3U, 0xDDU, 0xF6U, 0xBFU, 0x18U, 0xD4U, 0x6FU, 0xB8U,
0x5CU, 0x41U, 0x76U, 0x3BU, 0xEAU, 0x4CU, 0xB6U, 0x8EU, 0x08U, 0xBCU, 0x8AU, 0xB9U, 0x5BU, 0x62U, 0xCFU, 0x05U,
0xEFU, 0x54U, 0x17U, 0x9EU, 0x7BU, 0x14U, 0x8FU, 0x60U, 0xBCU, 0x99U, 0x85U, 0x28U, 0xB3U, 0x76U, 0xDBU, 0xF0U,
0x41U, 0x40U, 0xC8U, 0x95U, 0xF1U, 0x18U, 0xA7U, 0x96U, 0xCFU, 0x85U, 0x69U, 0x25U, 0x5AU, 0x43U, 0x04U, 0xA1U,
0x78U, 0xA8U, 0x5DU, 0x82U, 0x70U, 0xF4U, 0xE6U, 0x31U, 0x84U, 0xC5U, 0x67U, 0x5EU, 0x92U, 0x2FU, 0x7CU, 0xEDU,
0x17U, 0xD8U, 0xC6U, 0x57U, 0x58U, 0x33U, 0xC5U, 0x81U, 0xFFU, 0x7CU, 0x73U, 0x17U, 0xDAU, 0xA6U, 0x53U, 0xAAU,
0xFFU, 0x22U, 0x50U, 0xC4U, 0x18U, 0x6EU, 0x6EU, 0x9CU, 0x97U, 0xBAU, 0xE4U, 0x54U, 0x7FU, 0x81U, 0xFCU, 0xE9U,
0x0AU, 0x1DU, 0x33U, 0x07U, 0x18U, 0x14U, 0x71U, 0x2AU, 0x96U, 0x27U, 0x34U, 0xD9U, 0xFBU, 0xF0U, 0xB1U, 0x33U,
0xB9U, 0x30U, 0x25U, 0xA3U, 0x4DU, 0x8FU, 0x68U, 0x00U, 0x68U, 0xCAU, 0xCCU, 0xB5U, 0x21U, 0x82U, 0x3AU, 0x37U,
0xE3U, 0xA9U, 0xB4U, 0xC9U, 0xF6U, 0xAEU, 0x99U, 0x53U, 0x68U, 0x25U, 0x39U, 0x9DU, 0x5EU, 0xFCU, 0xD0U, 0x89U,
0x26U, 0x08U, 0x8DU, 0xBAU, 0x23U, 0x14U, 0x91U, 0x47U, 0x14U, 0x70U, 0x7CU, 0xF2U, 0x2FU, 0x18U, 0xEDU, 0xECU,
0xD3U, 0xE8U, 0x18U, 0x44U, 0x69U, 0x83U, 0xA7U, 0x57U, 0x83U, 0xDEU, 0x06U, 0x7FU, 0xBCU, 0xEEU, 0x7CU, 0x20U,
0x7FU, 0x1CU, 0xC6U, 0x59U, 0xB5U, 0xC8U, 0xD5U, 0x49U, 0x83U, 0x3BU, 0xBCU, 0x36U, 0xF1U, 0xE4U, 0x56U, 0x00U,
0xE4U, 0x98U, 0x64U, 0x43U, 0xC0U, 0xF1U, 0x48U, 0x25U, 0xAAU, 0x49U, 0x55U, 0x87U, 0x36U, 0x55U, 0x3DU, 0xA7U,
0x04U, 0xF9U, 0xAEU, 0x6BU, 0xA1U, 0x6EU, 0x99U, 0x2FU, 0x66U, 0x3FU, 0x05U, 0xDAU, 0x3FU, 0x7BU, 0x06U, 0xEDU,
0x19U, 0xE5U, 0x24U, 0x20U, 0xFBU, 0x1FU, 0x3CU, 0x48U, 0xE1U, 0xD2U, 0xB2U, 0xE2U, 0x01U, 0x50U, 0xCDU, 0x40U,
0x41U, 0x67U, 0xB8U, 0x70U, 0xE2U, 0x3EU, 0x73U, 0x5EU, 0xFCU, 0x7EU, 0x14U, 0x3DU, 0x00U, 0xD6U, 0xDEU, 0x94U,
0xA6U, 0x61U, 0xBAU, 0xE5U, 0x89U, 0x2BU, 0x5FU, 0x66U, 0xAAU, 0x7AU, 0xDAU, 0xA5U, 0x07U, 0x29U, 0xB5U, 0x02U,
0x4BU, 0x69U, 0x49U, 0x2BU, 0x13U, 0x3CU, 0x8CU, 0xAAU, 0x17U, 0x6CU, 0xB4U, 0x28U, 0x81U, 0xC3U, 0xE5U, 0xA3U
};

static const u8 Data[XSECURE_RSA_SIZE]  = {
0x78U, 0x45U, 0x42U, 0x51U, 0xF0U, 0x0AU, 0xF6U, 0x1FU, 0x62U, 0x16U, 0x75U, 0xE2U, 0x7DU, 0xD1U, 0x9BU, 0x7DU,
0x6AU, 0xE2U, 0x16U, 0x66U, 0x03U, 0x6AU, 0x45U, 0x59U, 0xEBU, 0x7DU, 0x4CU, 0x09U, 0x1FU, 0x46U, 0xEDU, 0x24U,
0xD6U, 0x5CU, 0xADU, 0x09U, 0xB7U, 0x1AU, 0xC9U, 0xF7U, 0xEFU, 0x5CU, 0xF1U, 0x3CU, 0x43U, 0xBCU, 0x04U, 0xA3U,
0xE0U, 0xC9U, 0x82U, 0xABU, 0x67U, 0x25U, 0x5DU, 0x69U, 0xBDU, 0x53U, 0x34U, 0xF5U, 0x58U, 0xEDU, 0x35U, 0xD7U,
0x6AU, 0x1EU, 0xE2U, 0xCFU, 0x0FU, 0xC4U, 0x35U, 0x41U, 0xE6U, 0xD1U, 0x12U, 0x16U, 0xC0U, 0x7DU, 0x17U, 0x88U,
0xBFU, 0xC9U, 0x91U, 0xEAU, 0xD9U, 0x99U, 0x3DU, 0xD4U, 0x27U, 0xEBU, 0xB1U, 0x75U, 0x99U, 0xF6U, 0x13U, 0x28U,
0x8DU, 0x8DU, 0xA1U, 0x64U, 0xD7U, 0xE7U, 0xC9U, 0xCDU, 0x37U, 0xF6U, 0x36U, 0x09U, 0x71U, 0xB8U, 0x33U, 0x26U,
0xDEU, 0xD5U, 0xADU, 0xFBU, 0x29U, 0xC7U, 0x86U, 0x6AU, 0xD3U, 0xC7U, 0x42U, 0x81U, 0xCAU, 0xD9U, 0xBBU, 0x91U,
0xA0U, 0x8CU, 0x68U, 0xF8U, 0xD8U, 0x70U, 0x02U, 0xC6U, 0x9EU, 0x39U, 0xC1U, 0xE0U, 0x82U, 0x81U, 0xD4U, 0x04U,
0x63U, 0x3EU, 0x86U, 0x19U, 0x71U, 0x99U, 0x5EU, 0xA7U, 0x48U, 0x92U, 0x71U, 0x9BU, 0x41U, 0xB7U, 0x57U, 0x61U,
0xE7U, 0xBBU, 0x35U, 0xEBU, 0x6CU, 0x55U, 0xF3U, 0xA8U, 0xD9U, 0xE8U, 0x86U, 0x92U, 0x66U, 0xBEU, 0x57U, 0x13U,
0x08U, 0xF9U, 0x4DU, 0x58U, 0xB6U, 0xCEU, 0x6BU, 0x99U, 0xC6U, 0x8BU, 0x4CU, 0xD9U, 0x98U, 0x62U, 0x4BU, 0x5EU,
0x64U, 0xBAU, 0x2FU, 0x49U, 0x17U, 0x93U, 0x1BU, 0xB1U, 0xF0U, 0xABU, 0xD8U, 0xC2U, 0x03U, 0x0EU, 0x8CU, 0x30U,
0x85U, 0xFCU, 0x66U, 0x30U, 0xECU, 0x1BU, 0x97U, 0x82U, 0xF7U, 0xC5U, 0x55U, 0x30U, 0x85U, 0xD8U, 0xF5U, 0xD6U,
0x3FU, 0x2DU, 0xC9U, 0xC5U, 0xCBU, 0x2CU, 0x6BU, 0x63U, 0xB2U, 0x48U, 0xEEU, 0x77U, 0x21U, 0x35U, 0x3AU, 0xC9U,
0x91U, 0xAAU, 0x25U, 0x7DU, 0xE9U, 0x73U, 0x15U, 0x56U, 0x9AU, 0x12U, 0x2EU, 0x6BU, 0x23U, 0x00U, 0xBEU, 0x92U,
0x1DU, 0xADU, 0xA4U, 0xACU, 0x1DU, 0xABU, 0x69U, 0x48U, 0x83U, 0x1AU, 0x4FU, 0x16U, 0xF2U, 0x4AU, 0x41U, 0xE5U,
0xC1U, 0x1EU, 0xBEU, 0x17U, 0x30U, 0x6EU, 0x30U, 0xF4U, 0x2BU, 0x6AU, 0xB5U, 0x70U, 0xF7U, 0x5DU, 0xC4U, 0xA0U,
0xAEU, 0x73U, 0xB0U, 0x0DU, 0xB8U, 0xE0U, 0x42U, 0x9AU, 0x8EU, 0xACU, 0x4BU, 0x11U, 0x54U, 0x6AU, 0xC6U, 0x84U,
0x59U, 0xCBU, 0x6AU, 0x1EU, 0x1FU, 0xF2U, 0x4FU, 0xBCU, 0x8BU, 0xF9U, 0xD5U, 0x06U, 0x1AU, 0xE8U, 0x88U, 0xC4U,
0xDCU, 0x89U, 0xEFU, 0x1BU, 0x46U, 0x68U, 0xD2U, 0xBAU, 0xDEU, 0xA9U, 0x90U, 0x7FU, 0xF7U, 0xCDU, 0x1EU, 0xF6U,
0x89U, 0x30U, 0xE2U, 0x41U, 0x95U, 0xE7U, 0xC1U, 0xDAU, 0x3DU, 0xC2U, 0xA9U, 0xCFU, 0x62U, 0x03U, 0x83U, 0xB4U,
0x21U, 0x5AU, 0xE4U, 0x7CU, 0xD0U, 0x7FU, 0x12U, 0xE8U, 0xDFU, 0x5FU, 0x97U, 0xFBU, 0x21U, 0xD4U, 0x80U, 0x6BU,
0x13U, 0x6AU, 0x4EU, 0x64U, 0xF0U, 0xABU, 0xD4U, 0x60U, 0xD3U, 0x4FU, 0xB3U, 0xACU, 0x45U, 0xF3U, 0x2EU, 0x6CU,
0x53U, 0x48U, 0xFBU, 0xAAU, 0xB0U, 0x2EU, 0x96U, 0xECU, 0xA7U, 0x20U, 0xDDU, 0xE5U, 0xCEU, 0xDEU, 0x4DU, 0x2DU,
0xC0U, 0x4EU, 0x43U, 0x03U, 0xC9U, 0x7CU, 0x9CU, 0x74U, 0x9BU, 0xF9U, 0xD4U, 0x46U, 0x69U, 0xDEU, 0x95U, 0xEBU,
0xEBU, 0x26U, 0x0EU, 0x36U, 0x35U, 0x62U, 0x69U, 0xDBU, 0x67U, 0x29U, 0x23U, 0x57U, 0xFBU, 0xA2U, 0xE0U, 0x03U,
0x0FU, 0xBFU, 0x35U, 0xF8U, 0x45U, 0x21U, 0xE9U, 0x91U, 0x09U, 0xE5U, 0xB2U, 0x4CU, 0x1CU, 0x10U, 0x83U, 0x20U,
0x66U, 0xB7U, 0xA4U, 0xE5U, 0xDEU, 0x06U, 0x07U, 0xB6U, 0x25U, 0x3CU, 0xF1U, 0x62U, 0x7EU, 0xD3U, 0x9CU, 0x27U,
0xEEU, 0x0DU, 0x26U, 0x65U, 0xD0U, 0x0EU, 0x42U, 0x66U, 0x15U, 0x75U, 0x5FU, 0x33U, 0x6FU, 0x3CU, 0x10U, 0x5CU,
0xB4U, 0x39U, 0x8DU, 0xD5U, 0xD9U, 0xEEU, 0xEAU, 0xEAU, 0xFBU, 0xDDU, 0x38U, 0xA2U, 0xCCU, 0x54U, 0x90U, 0xB8U,
0xB0U, 0x80U, 0xF2U, 0xF4U, 0x9EU, 0x29U, 0xBEU, 0x84U, 0x74U, 0xEBU, 0x9AU, 0x03U, 0xF4U, 0x55U, 0x39U, 0x69U
};

static const u8 Modulus[XSECURE_RSA_SIZE] = {
0xCCU, 0x90U, 0xB4U, 0xB1U, 0xF1U, 0xC1U, 0xD3U, 0x9DU, 0x98U, 0xB1U, 0x34U, 0x49U, 0x92U, 0xCDU, 0xFDU, 0xD1U,
0xCFU, 0x87U, 0x43U, 0x52U, 0xAFU, 0x9CU, 0x5CU, 0xFBU, 0x62U, 0xC9U, 0xAAU, 0x7EU, 0xCAU, 0xFDU, 0x0EU, 0x2AU,
0xF9U, 0xBFU, 0x86U, 0xBCU, 0xDEU, 0xFDU, 0x10U, 0x42U, 0x79U, 0xF2U, 0x18U, 0x6BU, 0xB1U, 0xCFU, 0x8DU, 0x66U,
0xDEU, 0xB0U, 0x64U, 0x1EU, 0xB1U, 0x45U, 0x45U, 0x20U, 0x80U, 0xCCU, 0xCBU, 0x6AU, 0xADU, 0x1AU, 0x0AU, 0x2AU,
0xA9U, 0xE2U, 0xA6U, 0x41U, 0xABU, 0xD3U, 0x3AU, 0xEDU, 0x34U, 0x71U, 0x2EU, 0x2DU, 0x08U, 0xF5U, 0x16U, 0x7BU,
0x89U, 0xEBU, 0xBAU, 0xD5U, 0x86U, 0x33U, 0x0FU, 0x26U, 0x80U, 0x9FU, 0x3CU, 0x98U, 0x11U, 0x62U, 0xDCU, 0x12U,
0x41U, 0xE9U, 0x7AU, 0x3CU, 0xD9U, 0x1BU, 0x0CU, 0xE5U, 0x33U, 0x55U, 0x25U, 0x51U, 0xEFU, 0x2DU, 0xF2U, 0x3EU,
0x6EU, 0x7FU, 0xF9U, 0x76U, 0x86U, 0x6CU, 0xA9U, 0x3CU, 0x04U, 0xFFU, 0xADU, 0xD1U, 0x9DU, 0x93U, 0xC1U, 0x04U,
0x4CU, 0x55U, 0xABU, 0xFDU, 0xE7U, 0x71U, 0xAFU, 0xF5U, 0xA6U, 0xE1U, 0x51U, 0xE7U, 0xBAU, 0x4DU, 0x94U, 0x22U,
0xD2U, 0x07U, 0x7BU, 0x64U, 0x11U, 0xC2U, 0x56U, 0xFAU, 0xBEU, 0x7CU, 0x1DU, 0x5FU, 0x57U, 0x66U, 0x4EU, 0x88U,
0xD4U, 0x58U, 0x1FU, 0x12U, 0x07U, 0x65U, 0x77U, 0xA3U, 0x2AU, 0xF4U, 0x8DU, 0xDEU, 0xDAU, 0x3CU, 0xA1U, 0x0DU,
0x41U, 0x3EU, 0x58U, 0x77U, 0x1EU, 0x59U, 0x71U, 0xDBU, 0xD2U, 0xDDU, 0x48U, 0x20U, 0xE8U, 0xB9U, 0x09U, 0xB0U,
0xD5U, 0x95U, 0x26U, 0xCFU, 0xAEU, 0x4FU, 0x31U, 0xDFU, 0x10U, 0xE7U, 0x7BU, 0x34U, 0x44U, 0x3EU, 0x2DU, 0x4EU,
0x7EU, 0x9BU, 0xBAU, 0x9CU, 0xBDU, 0xB4U, 0xC9U, 0xD9U, 0x8AU, 0x09U, 0x74U, 0x75U, 0x3DU, 0xD2U, 0x9FU, 0xDFU,
0x1AU, 0xE8U, 0x6AU, 0xA0U, 0x09U, 0xD3U, 0x66U, 0x04U, 0x8DU, 0xA8U, 0x7BU, 0x4BU, 0x2EU, 0x3AU, 0x42U, 0xACU,
0xD7U, 0x24U, 0xFFU, 0x5FU, 0x9CU, 0x09U, 0x7FU, 0xEFU, 0x22U, 0xA5U, 0x5AU, 0x7EU, 0x3BU, 0x63U, 0x36U, 0x38U,
0xF8U, 0xEBU, 0xFFU, 0xAEU, 0xFDU, 0xDEU, 0x19U, 0xB6U, 0x38U, 0xC2U, 0x91U, 0x83U, 0x75U, 0x5FU, 0x86U, 0x9EU,
0xA7U, 0x99U, 0xA1U, 0xC0U, 0x75U, 0x7EU, 0x43U, 0x8CU, 0xE7U, 0x4BU, 0x66U, 0x4CU, 0x3BU, 0xBBU, 0x11U, 0x3EU,
0xF2U, 0x9DU, 0x84U, 0x4AU, 0x2FU, 0xACU, 0xBFU, 0x2CU, 0xC2U, 0xA9U, 0x97U, 0xDCU, 0xC4U, 0x0FU, 0x09U, 0x2CU,
0xCCU, 0x1DU, 0x57U, 0xD6U, 0x34U, 0xB0U, 0x43U, 0xB9U, 0xFBU, 0x45U, 0xC9U, 0x51U, 0xE8U, 0x87U, 0x39U, 0x90U,
0x49U, 0x81U, 0x1FU, 0x0EU, 0x6DU, 0x1AU, 0x4FU, 0xC2U, 0x04U, 0xEEU, 0x69U, 0xF0U, 0x67U, 0xCEU, 0xDCU, 0x92U,
0xC5U, 0x8CU, 0xCEU, 0x91U, 0x89U, 0x2CU, 0x56U, 0xFBU, 0xF8U, 0x70U, 0x8FU, 0xB9U, 0xCFU, 0xA0U, 0xBDU, 0x9FU,
0x8AU, 0xF7U, 0xC5U, 0xEDU, 0xA1U, 0x57U, 0xD4U, 0x27U, 0x70U, 0x34U, 0xD5U, 0x59U, 0x6AU, 0x93U, 0x48U, 0x11U,
0x00U, 0x02U, 0xD0U, 0x92U, 0xE6U, 0x1BU, 0xA3U, 0xFCU, 0xF8U, 0x04U, 0x64U, 0x08U, 0x0BU, 0xDBU, 0x49U, 0x93U,
0xE3U, 0xB4U, 0x4DU, 0x68U, 0xEBU, 0x9EU, 0xEAU, 0xABU, 0x19U, 0x36U, 0xA3U, 0xCFU, 0x39U, 0x6AU, 0xDDU, 0x24U,
0x1AU, 0x4EU, 0x70U, 0x64U, 0xF4U, 0x0CU, 0x2CU, 0x54U, 0xCEU, 0xB3U, 0x78U, 0x53U, 0xEAU, 0x2DU, 0xB0U, 0xDBU,
0xEEU, 0x58U, 0x86U, 0x00U, 0x42U, 0x99U, 0x8FU, 0xCFU, 0x2EU, 0x1FU, 0xCBU, 0xA5U, 0x60U, 0x44U, 0xD4U, 0xBEU,
0x05U, 0x2CU, 0xD4U, 0x62U, 0xD8U, 0xB4U, 0x38U, 0xCBU, 0x80U, 0x38U, 0x78U, 0x42U, 0x43U, 0xACU, 0xEBU, 0xA5U,
0xBDU, 0x05U, 0xFCU, 0x13U, 0xA7U, 0x2BU, 0x67U, 0x82U, 0x02U, 0x4CU, 0x74U, 0x8EU, 0xAAU, 0xD0U, 0x28U, 0xD1U,
0x39U, 0x8CU, 0xEFU, 0x61U, 0xB2U, 0xFFU, 0x60U, 0x41U, 0x83U, 0x99U, 0x81U, 0x4DU, 0xCFU, 0x81U, 0x7CU, 0xCCU,
0xECU, 0xE3U, 0xF5U, 0xF1U, 0x17U, 0x2BU, 0x32U, 0xF2U, 0x46U, 0x8AU, 0xC4U, 0x93U, 0x20U, 0xC1U, 0xD8U, 0xFCU,
0xC8U, 0x23U, 0x71U, 0xBFU, 0x37U, 0x85U, 0x00U, 0x7CU, 0xEDU, 0xB6U, 0x84U, 0xABU, 0x45U, 0x00U, 0x39U, 0xA5U
};

static const u8 PrivateExp[XSECURE_RSA_SIZE] = {
0x07U, 0x31U, 0xDBU, 0xB9U, 0x1AU, 0x78U, 0xA7U, 0xB9U, 0xB6U, 0x42U, 0x1DU, 0x45U, 0xADU, 0x2DU, 0x23U, 0x8DU,
0x3AU, 0x90U, 0x56U, 0x54U, 0xD5U, 0x2AU, 0x59U, 0xB6U, 0xD9U, 0xBDU, 0xA0U, 0x0AU, 0x7FU, 0x97U, 0x30U, 0xE5U,
0x60U, 0x0DU, 0x55U, 0x6EU, 0xD5U, 0x24U, 0xEFU, 0x6AU, 0x77U, 0x21U, 0x85U, 0xACU, 0x5BU, 0xFFU, 0xA1U, 0xC8U,
0x89U, 0x72U, 0x88U, 0xC2U, 0xEFU, 0x97U, 0x31U, 0x1CU, 0x38U, 0x8BU, 0x7BU, 0x68U, 0x10U, 0x18U, 0x87U, 0x5DU,
0x02U, 0xC6U, 0x9EU, 0xF4U, 0x56U, 0x57U, 0xC2U, 0x8DU, 0xCFU, 0x37U, 0x5FU, 0xD9U, 0x63U, 0xA4U, 0x43U, 0x1BU,
0x2BU, 0x39U, 0xA8U, 0x43U, 0x94U, 0x49U, 0xBDU, 0x71U, 0xACU, 0x3AU, 0x6DU, 0x86U, 0x27U, 0xAEU, 0x63U, 0x71U,
0x1CU, 0xF2U, 0x5AU, 0x43U, 0x43U, 0x9BU, 0x88U, 0xF3U, 0x92U, 0xBFU, 0x9EU, 0xE5U, 0x67U, 0xD5U, 0x24U, 0xDCU,
0xFFU, 0x43U, 0xEFU, 0x01U, 0x35U, 0x2FU, 0x23U, 0x30U, 0xA2U, 0x38U, 0x5FU, 0xE3U, 0xACU, 0x78U, 0x22U, 0xE4U,
0xC4U, 0xCEU, 0x8AU, 0x93U, 0x0EU, 0x8DU, 0xD8U, 0x13U, 0x7AU, 0xC0U, 0x18U, 0x02U, 0x5FU, 0x72U, 0x15U, 0x10U,
0xB8U, 0x74U, 0x5CU, 0xE9U, 0xA3U, 0x1AU, 0x43U, 0xB6U, 0xE3U, 0xF6U, 0x37U, 0x8EU, 0x3FU, 0xE3U, 0xB0U, 0x45U,
0x6DU, 0xBAU, 0x7FU, 0xB5U, 0x41U, 0x9FU, 0x56U, 0xFAU, 0xDDU, 0x2AU, 0xE6U, 0xC6U, 0x7CU, 0xC4U, 0x7FU, 0x21U,
0x99U, 0x36U, 0xD9U, 0x23U, 0xAFU, 0x64U, 0x93U, 0xF5U, 0x98U, 0x50U, 0x02U, 0x16U, 0x6EU, 0x38U, 0xFBU, 0xD7U,
0x45U, 0x5BU, 0xCEU, 0x96U, 0xA5U, 0x59U, 0xD1U, 0xB6U, 0x37U, 0xC0U, 0xFCU, 0x7BU, 0x55U, 0x1FU, 0x1EU, 0xB6U,
0xD1U, 0x0BU, 0x27U, 0x20U, 0x16U, 0x28U, 0x02U, 0xA5U, 0x6DU, 0x0EU, 0x72U, 0x13U, 0x21U, 0x8EU, 0x83U, 0xE2U,
0xF3U, 0xEEU, 0x52U, 0xB9U, 0xB8U, 0x3EU, 0xC0U, 0x2AU, 0xCEU, 0xD4U, 0xABU, 0x29U, 0x74U, 0xC1U, 0xC5U, 0xD0U,
0x90U, 0x6CU, 0x93U, 0xB2U, 0xC8U, 0x29U, 0x29U, 0x5FU, 0xBEU, 0xC7U, 0x34U, 0x0CU, 0x96U, 0x88U, 0x21U, 0xC3U,
0x05U, 0x1DU, 0x3EU, 0x31U, 0x3BU, 0xDFU, 0x6FU, 0x57U, 0xF1U, 0xD4U, 0x9CU, 0x6CU, 0x48U, 0x8FU, 0xCDU, 0x88U,
0xE4U, 0xC0U, 0x8EU, 0xDAU, 0x79U, 0x27U, 0x50U, 0xA8U, 0x7CU, 0xB9U, 0x38U, 0xA3U, 0x93U, 0x67U, 0x36U, 0x35U,
0x35U, 0xB3U, 0x2AU, 0x4DU, 0x16U, 0x96U, 0x47U, 0x98U, 0x27U, 0xEFU, 0xDEU, 0x23U, 0xE6U, 0xFBU, 0x4BU, 0xAFU,
0x7AU, 0xB1U, 0x9BU, 0x7CU, 0x05U, 0xBEU, 0x0EU, 0xCEU, 0x69U, 0x53U, 0xF0U, 0x76U, 0xC1U, 0x89U, 0xCCU, 0xF2U,
0xCAU, 0x2CU, 0xBBU, 0xA4U, 0xFEU, 0x3CU, 0x3DU, 0xAFU, 0xBCU, 0x25U, 0x4AU, 0x53U, 0x4BU, 0xE9U, 0x85U, 0x55U,
0xF1U, 0x79U, 0x70U, 0xDFU, 0x57U, 0x23U, 0x55U, 0x7DU, 0x7AU, 0x5CU, 0xA7U, 0xEEU, 0xFBU, 0x00U, 0x35U, 0x1EU,
0xE3U, 0xF1U, 0xFDU, 0x3EU, 0x94U, 0x70U, 0x88U, 0x91U, 0xFAU, 0xC6U, 0xCBU, 0x05U, 0xE1U, 0xCDU, 0x7DU, 0xD1U,
0xE3U, 0x2BU, 0x7FU, 0xB4U, 0x64U, 0x1CU, 0x94U, 0x1EU, 0xC2U, 0x55U, 0x6CU, 0x78U, 0x22U, 0x27U, 0x90U, 0xBAU,
0x33U, 0x0EU, 0xADU, 0x5DU, 0x55U, 0xD9U, 0xDFU, 0x7BU, 0x83U, 0xA5U, 0x2BU, 0x4FU, 0xA0U, 0x36U, 0x3CU, 0x67U,
0xDDU, 0xC4U, 0x00U, 0x50U, 0x90U, 0x12U, 0xE1U, 0x8AU, 0x65U, 0x35U, 0x13U, 0x9EU, 0xB2U, 0x0CU, 0xE4U, 0xEAU,
0x51U, 0x2FU, 0xCDU, 0xAEU, 0x35U, 0x41U, 0x1EU, 0xFAU, 0x5AU, 0x6CU, 0xB6U, 0xC1U, 0xA7U, 0xFEU, 0x69U, 0x6FU,
0x7DU, 0x50U, 0xFCU, 0xABU, 0x5FU, 0xADU, 0x0CU, 0x2BU, 0x58U, 0xF6U, 0xB3U, 0x33U, 0x1FU, 0xF4U, 0x3AU, 0xDAU,
0x89U, 0xD3U, 0x0EU, 0x79U, 0x7BU, 0x31U, 0x8EU, 0x15U, 0x84U, 0xECU, 0x2CU, 0x05U, 0x2FU, 0xA9U, 0x52U, 0xF6U,
0x59U, 0x94U, 0xBFU, 0xC0U, 0x4BU, 0xCDU, 0x40U, 0xFBU, 0x24U, 0x44U, 0x0FU, 0x76U, 0x11U, 0xE8U, 0xD1U, 0xE3U,
0x13U, 0x26U, 0xEFU, 0xFAU, 0xA9U, 0xE3U, 0xDBU, 0x73U, 0x29U, 0x80U, 0x4DU, 0x61U, 0x74U, 0x3CU, 0x12U, 0x50U,
0xFDU, 0xDDU, 0xEAU, 0x72U, 0xE4U, 0x88U, 0x8BU, 0x20U, 0x1AU, 0x3BU, 0x36U, 0xE8U, 0xB8U, 0x98U, 0x55U, 0xCDU
};

static const u8 ExpectedSign[XSECURE_RSA_SIZE] = {
0x05U, 0x95U, 0x87U, 0x3FU, 0xA5U, 0x76U, 0x50U, 0xBFU, 0x76U, 0x6FU, 0x2CU, 0xB3U, 0x97U, 0x80U, 0x8BU, 0x7BU,
0x99U, 0xE8U, 0x56U, 0x2FU, 0x4BU, 0xCDU, 0x66U, 0x25U, 0x6FU, 0xA9U, 0x51U, 0x7CU, 0x42U, 0x57U, 0x95U, 0x95U,
0x6AU, 0xF0U, 0xF8U, 0x7EU, 0x56U, 0x0DU, 0x28U, 0x07U, 0xE7U, 0x1CU, 0x7CU, 0xD8U, 0x9EU, 0xF2U, 0xD6U, 0x37U,
0xE8U, 0x7AU, 0x5AU, 0xDBU, 0xA5U, 0xB5U, 0xD2U, 0x93U, 0xFAU, 0x29U, 0x28U, 0x76U, 0xB8U, 0xCEU, 0xC3U, 0x66U,
0xFFU, 0x15U, 0xFBU, 0x28U, 0x3BU, 0xC2U, 0x82U, 0x32U, 0xB7U, 0x5FU, 0xE4U, 0xD5U, 0x54U, 0x8EU, 0xE5U, 0x43U,
0x7DU, 0xAEU, 0x02U, 0x21U, 0x5BU, 0x55U, 0x9CU, 0xD0U, 0xCEU, 0x65U, 0x05U, 0x47U, 0xD2U, 0x5AU, 0x6FU, 0x2CU,
0x34U, 0x3EU, 0x48U, 0x6BU, 0xE1U, 0xCAU, 0x63U, 0xECU, 0x65U, 0xA0U, 0xD6U, 0x62U, 0xA0U, 0x59U, 0x17U, 0x85U,
0xA2U, 0xA3U, 0xB4U, 0x6CU, 0xD1U, 0x58U, 0x5CU, 0xEEU, 0xF9U, 0x2BU, 0x86U, 0x10U, 0xFCU, 0xDBU, 0x14U, 0x59U,
0x49U, 0x6AU, 0xA1U, 0xBCU, 0x46U, 0x51U, 0xFAU, 0xCFU, 0xFEU, 0x75U, 0x76U, 0xA7U, 0xFBU, 0xD3U, 0x07U, 0xB0U,
0xC1U, 0x66U, 0x97U, 0x8DU, 0x46U, 0x5CU, 0xDAU, 0x20U, 0x5CU, 0xE4U, 0xD4U, 0x61U, 0xFDU, 0x24U, 0x7EU, 0x61U,
0xFBU, 0x61U, 0x80U, 0x99U, 0x4FU, 0x16U, 0xB4U, 0x30U, 0xDAU, 0xBDU, 0x70U, 0xA0U, 0xFEU, 0xA3U, 0x5CU, 0xC9U,
0xF6U, 0x39U, 0xEFU, 0x20U, 0x25U, 0xBCU, 0xD4U, 0x28U, 0x6FU, 0x88U, 0x1AU, 0xD2U, 0xF0U, 0x02U, 0x42U, 0xA1U,
0xE3U, 0x45U, 0x2BU, 0xADU, 0xABU, 0x5BU, 0x93U, 0xBDU, 0x26U, 0x7DU, 0xAEU, 0xDCU, 0x48U, 0x86U, 0x8AU, 0x41U,
0x9FU, 0x75U, 0xA4U, 0x33U, 0x23U, 0x8CU, 0x9BU, 0xB9U, 0x3BU, 0x80U, 0xBDU, 0xDBU, 0x34U, 0x4FU, 0x6FU, 0x8EU,
0xC4U, 0xFAU, 0xB8U, 0xB7U, 0x48U, 0x53U, 0x17U, 0x8EU, 0x55U, 0x3BU, 0xC7U, 0xBFU, 0x2CU, 0x53U, 0x05U, 0xEBU,
0xADU, 0x96U, 0xA9U, 0x37U, 0xB0U, 0x5DU, 0x42U, 0xD5U, 0xE1U, 0x91U, 0x58U, 0x56U, 0x62U, 0x6BU, 0xE4U, 0xE5U,
0x5DU, 0xFFU, 0x47U, 0x7AU, 0x42U, 0xE7U, 0x51U, 0xA0U, 0xE2U, 0x1BU, 0x49U, 0xEFU, 0x54U, 0xD1U, 0x7EU, 0xABU,
0xF9U, 0xB9U, 0x5CU, 0x5EU, 0x12U, 0x8FU, 0x38U, 0x6AU, 0xDFU, 0xA4U, 0xAEU, 0x2FU, 0x97U, 0x29U, 0xC0U, 0x9BU,
0x46U, 0xA2U, 0x9CU, 0x95U, 0xCBU, 0x94U, 0x97U, 0x1FU, 0x80U, 0x4CU, 0x62U, 0x6DU, 0x18U, 0xF0U, 0xFDU, 0x55U,
0x4CU, 0x92U, 0x45U, 0xDDU, 0x8AU, 0x0BU, 0x38U, 0x4AU, 0x11U, 0x2EU, 0xE8U, 0x3BU, 0xBBU, 0x66U, 0xDEU, 0x25U,
0xB9U, 0xC3U, 0xB6U, 0xA9U, 0xCFU, 0xE1U, 0xC1U, 0x43U, 0xD6U, 0x03U, 0x7CU, 0xB1U, 0x05U, 0x39U, 0x54U, 0xF8U,
0x3AU, 0x4CU, 0xD8U, 0x94U, 0xB9U, 0xA7U, 0xD6U, 0xF1U, 0x99U, 0x0EU, 0x55U, 0x07U, 0x56U, 0x72U, 0x5CU, 0xCDU,
0x0CU, 0x8FU, 0x00U, 0x51U, 0x4CU, 0xC5U, 0x39U, 0x7BU, 0x6AU, 0x7EU, 0x1DU, 0xA1U, 0x2BU, 0x5EU, 0x66U, 0x7EU,
0x07U, 0x9BU, 0xCFU, 0x55U, 0xB1U, 0xFFU, 0x95U, 0xFEU, 0x4CU, 0x18U, 0x2CU, 0xD7U, 0xBCU, 0x47U, 0xDCU, 0xB1U,
0x08U, 0xB9U, 0x6FU, 0x43U, 0x70U, 0xFCU, 0xEAU, 0x4EU, 0xD3U, 0xB7U, 0x0DU, 0xF8U, 0x66U, 0x03U, 0xBDU, 0xC9U,
0xFCU, 0x36U, 0x42U, 0x32U, 0x66U, 0x37U, 0x56U, 0x86U, 0x82U, 0x16U, 0x12U, 0x29U, 0x44U, 0x67U, 0x57U, 0x0CU,
0xA0U, 0x3CU, 0xEEU, 0xA3U, 0x88U, 0x07U, 0x33U, 0xDBU, 0x25U, 0x60U, 0x8BU, 0x3EU, 0x9FU, 0x46U, 0x1FU, 0x72U,
0x49U, 0x4FU, 0xC7U, 0x74U, 0xD3U, 0x09U, 0x5BU, 0x7FU, 0x3FU, 0x01U, 0x77U, 0x29U, 0xEBU, 0x6DU, 0xEAU, 0xCEU,
0x62U, 0xE2U, 0xDBU, 0xF7U, 0x42U, 0x05U, 0xF9U, 0xD6U, 0x26U, 0x0CU, 0x32U, 0x9AU, 0x21U, 0x99U, 0x01U, 0x12U,
0xBEU, 0xC8U, 0xDCU, 0xA5U, 0x3CU, 0x30U, 0x7FU, 0xC5U, 0x30U, 0x38U, 0x2EU, 0x1BU, 0x14U, 0x76U, 0xF4U, 0x84U,
0x60U, 0x0BU, 0x8BU, 0x5FU, 0x3DU, 0x9EU, 0xCAU, 0x22U, 0x79U, 0x55U, 0xD6U, 0x7EU, 0x8BU, 0xBEU, 0xB9U, 0xD7U,
0xBEU, 0xF7U, 0x23U, 0xA4U, 0x04U, 0x45U, 0xC3U, 0xCEU, 0xB3U, 0xC5U, 0x0CU, 0x70U, 0xB8U, 0xAFU, 0xD2U, 0x1FU
};

/* Exponent of Public key */
static const u32 PublicExp = 0x1000100;

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Function Prototypes ******************************/

static u32 SecureRsaExample(void);
static int XSecure_RsaCrtTest(XSecure_ClientInstance *SecureClientInstance);
static int XSecure_RsaExpoptTest(XSecure_ClientInstance *SecureClientInstance);
static int XSecure_RsaPublicEncryptTest(XSecure_ClientInstance *SecureClientInstance);

/************************** Variable Definitions *****************************/

static u32 Size = XSECURE_RSA_SIZE;

/* shared memory allocation */
static u8 SharedMem[XSECURE_SHARED_BUF_SIZE] __attribute__((aligned(64U)))
										__attribute__ ((section (".data.SharedMem")));
XSecure_RsaKeyParam RsaKeyParam __attribute__((aligned(64U))) __attribute__ ((section (".data.RsaKeyParam")));

/*****************************************************************************/
/**
*
* This example illustrates the usage of RSA APIs to encrypt or decrypt the
* signature
*
* @return
* 		- XST_SUCCESS if the boot image authentication is successful.
*		- ErrorCode if the boot image authentication Failed.
*
******************************************************************************/
int main(void)
{
	u32 Status;

	#ifdef XSECURE_CACHE_DISABLE
		Xil_DCacheDisable();
	#endif

	Status = SecureRsaExample();
	if(Status != XST_SUCCESS)
	{
		xil_printf("\r\nRSA client example failed %d \r\n",
				Status);
		Status = XST_FAILURE;
		goto END;
	}

	xil_printf("\r\nSuccessfully ran RSA client example\r\n ");
END:
	return Status;
}

/****************************************************************************/
/**
*
* This function authenticates boot image located in DDR using RSA-4096
* algorithm. The decrypted hash is matched against the calculated boot image
* hash.
* The purpose of this function is to illustrate how to use the XSecure_Rsa
* driver.
*
*
* @return	- ErrorCode if RSA authentication is failed
*		- XST_SUCCESS if RSA authentication is successful.
*
* @note		None.
*
****************************************************************************/
/** //! [RSA generic example] */
static u32 SecureRsaExample(void)
{
	u32 Status = XST_FAILURE;
	XMailbox MailboxInstance;
	XSecure_ClientInstance SecureClientInstance;
	u8 *Key = (u8*)(UINTPTR)&SharedMem[0U] + XSECURE_RSA_SIZE + XSECURE_RSA_SIZE;

	Xil_DCacheFlushRange((UINTPTR)Data, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Modulus, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)PrivateExp, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)P, XSECURE_PRIME_FACTOR_P_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Q, XSECURE_PRIME_FACTOR_Q_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Tot, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)DP, XSECURE_PRIME_FACTOR_P_SIZE);
	Xil_DCacheFlushRange((UINTPTR)DQ, XSECURE_PRIME_FACTOR_Q_SIZE);
	Xil_DCacheFlushRange((UINTPTR)QInv, XSECURE_PRIME_FACTOR_P_SIZE);
	Xil_DCacheFlushRange((UINTPTR)RN, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)RRN, XSECURE_RSA_SIZE);

	Status = XMailbox_Initialize(&MailboxInstance, 0U);
	if (Status != XST_SUCCESS) {
		xil_printf("Mailbox initialize failed:%08x \r\n", Status);
		goto END;
	}

	Status = XSecure_ClientInit(&SecureClientInstance, &MailboxInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("Client initialize failed:%08x \r\n", Status);
		goto END;
	}

	/* Set shared memory */
	Status = XMailbox_SetSharedMem(&MailboxInstance, (u64)(UINTPTR)(Key + XSECURE_RSA_SIZE +
				XSECURE_RSA_SIZE), XSECURE_SHARED_MEM_SIZE);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n shared memory initialization failed");
		goto END;
	}

	Status = XSecure_RsaPrivateDecKat(&SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("RSA private decrypt KAT failed\n\r");
	}

	/* RSA decrypt with private key using CRT */
	Status = XSecure_RsaCrtTest(&SecureClientInstance);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	/* RSA decrypt with private key using pre-calculated modulus */
	Status = XSecure_RsaExpoptTest(&SecureClientInstance);

END:
	Status |= XMailbox_ReleaseSharedMem(&MailboxInstance);
	return Status;
}

/****************************************************************************/
/**
*
* This function does the RSA CRT test.
*
* @param	SecureClientInstance - Pointer to XSecure_ClientInstance
*
* @return	- ErrorCode if RSA authentication is failed
*		- XST_SUCCESS if RSA authentication is successful.
*
****************************************************************************/
static int XSecure_RsaCrtTest(XSecure_ClientInstance *SecureClientInstance)
{
	u64 KeyAddr;
	u64 InputMsgAddr;
	u32 Status = XST_FAILURE;
	u32 Index;
	u8 *Signature = &SharedMem[0U];

	/* RSA parameters for CRT operation */
	RsaKeyParam.ModAddr = (u64)(UINTPTR)&Modulus;
	RsaKeyParam.PAddr   = (u64)(UINTPTR)&P;
	RsaKeyParam.PSize   = XSECURE_PRIME_FACTOR_P_SIZE;
	RsaKeyParam.QAddr   = (u64)(UINTPTR)&Q;
	RsaKeyParam.QSize   = XSECURE_PRIME_FACTOR_Q_SIZE;
	RsaKeyParam.DPAddr = (u64)(UINTPTR)&DP;
	RsaKeyParam.DQAddr = (u64)(UINTPTR)&DQ;
	RsaKeyParam.QInvAddr = (u64)(UINTPTR)&QInv;
	RsaKeyParam.PubExp  = PublicExp;
	RsaKeyParam.IsPubExpAvail = TRUE;
	RsaKeyParam.OpMode = XSECURE_RSA_CRT_MODE;

	KeyAddr = (u64)(UINTPTR)&RsaKeyParam;
	InputMsgAddr = (u64)(UINTPTR)&Data;

	/* RSA signature decrypt with private key */
	Status = XSecure_RsaPrivateDecrypt(SecureClientInstance, KeyAddr, InputMsgAddr,
			Size, (UINTPTR)Signature);
	if(XST_SUCCESS != Status) {
		xil_printf("Failed at RSA signature decryption\n\r");
		goto END;
	}

	Xil_DCacheInvalidateRange((UINTPTR)Signature, XSECURE_RSA_SIZE);

	xil_printf("\r\n Decrypted Signature with private key\r\n ");
	for(Index = 0; Index < Size; Index++) {
		xil_printf(" %02x ", Signature[Index]);
	}
	xil_printf(" \r\n ");

	/* Verification if Data is expected */
	Status = Xil_SMemCmp_CT(Signature, Size, ExpectedSign, Size, Size);
	if (Status != XST_SUCCESS) {
		xil_printf(
				"\r\nError at verification of RSA signature"
				" Decryption\n\r");
		goto END;
	}

	Status = XSecure_RsaPublicEncryptTest(SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n RSA CRT authentication failed");
	}
	else {
		xil_printf("\r\n Successfully ran RSA CRT authentication");
	}

END:
	return Status;
}

/****************************************************************************/
/**
*
* This function does the RSA private encryption using pre-calculated modulus.
*
* @param	SecureClientInstance - Pointer to XSecure_ClientInstance
*
* @return	- ErrorCode if RSA authentication is failed
*		- XST_SUCCESS if RSA authentication is successful.
*
****************************************************************************/
static int XSecure_RsaExpoptTest(XSecure_ClientInstance *SecureClientInstance)
{
	u64 KeyAddr;
	u64 InputMsgAddr;
	u32 Status = XST_FAILURE;
	u32 Index;
	u8 *Signature = &SharedMem[0U];

	/* RSA parameters for RRN operation */
	RsaKeyParam.ExpAddr = (u64)(UINTPTR)&PrivateExp;
	RsaKeyParam.ModAddr = (u64)(UINTPTR)&Modulus;
	RsaKeyParam.PAddr   = (u64)(UINTPTR)&P;
	RsaKeyParam.PSize   = XSECURE_PRIME_FACTOR_P_SIZE;
	RsaKeyParam.QAddr   = (u64)(UINTPTR)&Q;
	RsaKeyParam.QSize   = XSECURE_PRIME_FACTOR_Q_SIZE;
	RsaKeyParam.RNAddr = (u64)(UINTPTR)&RN;
	RsaKeyParam.RRNAddr = (u64)(UINTPTR)&RRN;
	RsaKeyParam.IsPrimeAvail = TRUE;
	RsaKeyParam.PubExp  = PublicExp;
	RsaKeyParam.IsPubExpAvail = TRUE;
	RsaKeyParam.TotAddr = (u64)(UINTPTR)&Tot;
	RsaKeyParam.IsTotAvail = FALSE;
	RsaKeyParam.OpMode = XSECURE_RSA_EXPOPT_MODE;

	KeyAddr = (u64)(UINTPTR)&RsaKeyParam;
	InputMsgAddr = (u64)(UINTPTR)&Data;

	/* RSA signature decrypt with private key */
	Status = XSecure_RsaPrivateDecrypt(SecureClientInstance, KeyAddr, InputMsgAddr,
			Size, (UINTPTR)Signature);
	if(XST_SUCCESS != Status) {
		xil_printf("Failed at RSA signature decryption\n\r");
		goto END;
	}

	Xil_DCacheInvalidateRange((UINTPTR)Signature, XSECURE_RSA_SIZE);

	xil_printf("\r\n Decrypted Signature with private key\r\n ");
	for(Index = 0; Index < Size; Index++) {
		xil_printf(" %02x ", Signature[Index]);
	}
	xil_printf(" \r\n ");

	/* Verification if Data is expected */
	Status = Xil_SMemCmp_CT(Signature, Size, ExpectedSign, Size, Size);
	if (Status != XST_SUCCESS) {
		xil_printf(
				"\r\nError at verification of RSA signature"
				" Decryption\n\r");
		goto END;
	}

	Status = XSecure_RsaPublicEncryptTest(SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n RSA Expopt authentication failed");
	}
	else {
		xil_printf("\r\n Successfully ran RSA Expopt authentication");
	}

END:
	return Status;
}

/****************************************************************************/
/**
*
* This function does RSA public encryption.
*
* @param	SecureClientInstance - Pointer to XSecure_ClientInstance
*
* @return	- ErrorCode if RSA encryption is failed
*		- XST_SUCCESS if RSA encryption is successful.
*
****************************************************************************/
static int XSecure_RsaPublicEncryptTest(XSecure_ClientInstance *SecureClientInstance)
{
	volatile int Status = XST_FAILURE;
	u32 Index;
	u8 *Signature = &SharedMem[0];
	u8 *EncryptSignatureOut = Signature + XSECURE_RSA_SIZE;
	u8 *Key = EncryptSignatureOut + XSECURE_RSA_SIZE;

	Status = Xil_SMemCpy(Key, XSECURE_RSA_SIZE, Modulus, XSECURE_RSA_SIZE,
			XSECURE_RSA_SIZE);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	Xil_DCacheFlushRange((UINTPTR)Key, XSECURE_RSA_SIZE);
	Xil_DCacheInvalidateRange((UINTPTR)Signature, XSECURE_RSA_SIZE);

	/* RSA signature encrypt with Public key components */
	Status = Xil_SMemCpy(Key + XSECURE_RSA_SIZE, 4U, &PublicExp, 4U, 4U);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	Xil_DCacheFlushRange((UINTPTR)Key, XSECURE_RSA_SIZE + XSECURE_RSA_SIZE);
	Xil_DCacheInvalidateRange((UINTPTR)EncryptSignatureOut, XSECURE_RSA_SIZE);

	Status = XSecure_RsaPublicEncKat(SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("RSA public encrypt KAT failed\n\r");
	}

	Status = XSecure_RsaPublicEncrypt(SecureClientInstance, (UINTPTR)Key, (UINTPTR)Signature,
			Size, (UINTPTR)EncryptSignatureOut);
	if(XST_SUCCESS != Status) {
		xil_printf("\r\nFailed at RSA signature encryption\n\r");
		goto END;
	}

	Xil_DCacheInvalidateRange((UINTPTR)EncryptSignatureOut, XSECURE_RSA_SIZE);

	xil_printf("\r\n Encrypted Signature with public key\r\n ");

	for(Index = 0; Index < Size; Index++) {
		xil_printf(" %02x ", EncryptSignatureOut[Index]);
	}

	/* Verification if Data is expected */
	Status = Xil_SMemCmp_CT(EncryptSignatureOut, Size, Data, Size, Size);
	if (Status != XST_SUCCESS) {
		xil_printf(
				"\r\nError at verification of RSA signature"
				" encryption\n\r");
	}

END:
	return Status;
}

/** //! [RSA generic example] */
/** @} */
