/******************************************************************************
* Copyright (c) 2019 - 2022 Xilinx, Inc.  All rights reserved.
* Copyright (c) 2022 - 2025, Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/


#include "xil_util.h"
#include "xpm_api.h"
#include "xpm_core.h"
#include "xpm_ipi.h"
#include "xpm_notifier.h"
#include "xpm_power.h"
#include "xplmi_err_common.h"
#include "xpm_common.h"
#include "xplmi_task.h"
#include "xplmi_scheduler.h"

#ifdef XILPM_NOTIFIER_LIST_SIZE
#define XPM_NOTIFIERS_COUNT XILPM_NOTIFIER_LIST_SIZE /* Provide by user */
#else
#define XPM_NOTIFIERS_COUNT    20U /* Default size */
#endif

#define XILPM_NOTIFIER_INTERVAL	(10U)
#define PRESENT		(1)
#define NOT_PRESENT	(0)

/* Use Time out count 0 to just check for IPI ack without holding the PLM */
#define XPM_NOTIFY_TIMEOUTCOUNT	(0U)

/* Array to maintain sequence of event happen */
#if (XPM_NOTIFIERS_COUNT < (256U)) /* u8 max limit is 255 */
#define DATA_TYPE	u8
#else
#define DATA_TYPE	u16
#endif

 /*
  * EventeSeq[] : This array store event Generated by PmNotifier[]
  * as part of memory optimization unsigned type of array are
  * getting used so to indicate empty element in array used value
  * are 0. but 0 is also the valid index.In XPmNotifier_AddInEventSeq()
  * PmNotifier[] Index store Index+1 in EventeSeq[] instead of Index.
  * when got form EventeSeq[] to PmNotifier[] so Index is Index-1
  * instead of Index.
  */
static DATA_TYPE EventSeq[XPM_NOTIFIERS_COUNT] = {0};

static u32 PendingEvent = (u32)NOT_PRESENT;

/* Variable to record the position of first empty space in EventSeq */
static volatile u32 PosEmptySpace = 0U;

typedef struct {
	u32 SubsystemId;
	u32 NodeId;
	u32 EventMask;
	u32 WakeMask;
	u32 IpiMask;  /* TODO: Remove this when IPI mask support in CDO is available*/
	u32 PendEvent;
	u32 PendEventCnt;
} XPmNotifier;

static XPmNotifier PmNotifiers[XPM_NOTIFIERS_COUNT];

static volatile u32 SchedulerTask = (u32)NOT_PRESENT;

#ifdef VERSAL_NET
#ifndef VERSAL_2VE_2VM
#include "xpm_update.h"
#include "xpm_update_data.h"

EXPORT_DS(EventSeq, \
	XPLMI_MODULE_XILPM_ID, XPM_EVENTSEQ_DS_ID, \
	XPM_DATA_STRUCT_VERSION, XPM_DATA_STRUCT_LCVERSION, \
	sizeof(EventSeq), (u32)(UINTPTR)EventSeq);

EXPORT_DS(PendingEvent, \
	XPLMI_MODULE_XILPM_ID, XPM_PENDINGEVENT_DS_ID, \
	XPM_DATA_STRUCT_VERSION, XPM_DATA_STRUCT_LCVERSION, \
	sizeof(PendingEvent), (u32)(UINTPTR)&PendingEvent);

EXPORT_DS(PosEmptySpace, \
	XPLMI_MODULE_XILPM_ID, XPM_POSEMPTYSPACE_DS_ID, \
	XPM_DATA_STRUCT_VERSION, XPM_DATA_STRUCT_LCVERSION, \
	sizeof(PosEmptySpace), (u32)(UINTPTR)&PosEmptySpace);

EXPORT_DS(PmNotifiers, \
	XPLMI_MODULE_XILPM_ID, XPM_PMNOTIFIERS_DS_ID, \
	XPM_DATA_STRUCT_VERSION, XPM_DATA_STRUCT_LCVERSION, \
	sizeof(PmNotifiers), (u32)(UINTPTR)PmNotifiers);

EXPORT_DS(SchedulerTask, \
	XPLMI_MODULE_XILPM_ID, XPM_SCHEDULERTASK_DS_ID, \
	XPM_DATA_STRUCT_VERSION, XPM_DATA_STRUCT_LCVERSION, \
	sizeof(SchedulerTask), (u32)(UINTPTR)&SchedulerTask);

/***************************************************************************/
/**
 * @brief This function will re-register error event handlers of event nodes.
 * This function called after PLM's module initialization after plm update done.
 *
 ***************************************************************************/
XStatus XPmNotifier_RestoreErrorEvents(void)
{
	XStatus Status = XST_FAILURE;
	/* Going through all PMNotifiers and re-register those event nodes*/
	for (u32 Index = 0U; Index < ARRAY_SIZE(PmNotifiers); Index ++) {
		u32 SsId = PmNotifiers[Index].SubsystemId;
		u32 NodeId = PmNotifiers[Index].NodeId;
		if ((0U != SsId) && ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId))) {
			Status = XPlmi_EmSetAction(NodeId, PmNotifiers[Index].EventMask , XPLMI_EM_ACTION_CUSTOM,
				   &XPmNotifier_Event, INVALID_SUBSYSID);
			if (XST_SUCCESS != Status) {
				goto done;
			}
		}
	}
	/* If there are no PmNotifiers, return OK. */
	Status = XST_SUCCESS;
done:
	return Status;
}
#endif
#endif

static int XPmNotifier_SchedulerTask(void *Arg);

/****************************************************************************/
/**
 * @brief  Register the notifier for given subsystem, NodeId and event
 *
 * @param  Subsystem  Subsystem to be notified
 * @param  NodeId     NodeId related to event
 * @param  Event      Event to be notified about
 * @param  Wake       Flag specifying whether the subsystem should be woken
 *		      upon event notification
 *
 * @return None
 *
 ****************************************************************************/
XStatus XPmNotifier_Register(XPm_Subsystem* const Subsystem,
			 const u32 NodeId,
			 const u32 Event, const u32 Wake, const u32 IpiMask)
{
	XStatus Status = XST_FAILURE;
	u32 Idx;
	u32 EmptyIdx = ARRAY_SIZE(PmNotifiers);

	for (Idx = 0U; Idx < ARRAY_SIZE(PmNotifiers); Idx++) {
		const XPm_Subsystem *NotifierSubsystem = XPmSubsystem_GetById(PmNotifiers[Idx].SubsystemId);

		if (NULL == NotifierSubsystem) {
			/* Empty entry found in PmNotifiers array */
			if (EmptyIdx > Idx) {
				/* Remember only first found empty entry */
				EmptyIdx = Idx;
			}
			continue;
		}

		if ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId)) {
			if ((Subsystem == NotifierSubsystem) &&
			    (NodeId == PmNotifiers[Idx].NodeId)) {
				/* Drop empty index - existing entry found */
				EmptyIdx = ARRAY_SIZE(PmNotifiers);
				break;
			}
		} else {
			if ((Subsystem == NotifierSubsystem) &&
			    (Event == PmNotifiers[Idx].EventMask) &&
			    (NodeId == PmNotifiers[Idx].NodeId)) {
				Status = XST_SUCCESS;
				goto done;
			}
		}
	}

	if ((EmptyIdx == ARRAY_SIZE(PmNotifiers)) &&
	    (Idx >= ARRAY_SIZE(PmNotifiers))) {
		/* There is no free entry in PmNotifiers array, report error */
		Status = XST_FAILURE;
		goto done;
	}

	/*
	 * Check if Node Class is EVENT and enable error action.
	 */
	if ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId)) {
		Status = XPlmi_EmSetAction(NodeId, Event, XPLMI_EM_ACTION_CUSTOM,
					   &XPmNotifier_Event, INVALID_SUBSYSID);
		if (XST_SUCCESS != Status) {
			goto done;
		}
	}

	if (EmptyIdx != ARRAY_SIZE(PmNotifiers)) {
		/* Add new entry in empty place if no notifier found for given pair */
		PmNotifiers[EmptyIdx].SubsystemId = Subsystem->Id;
		PmNotifiers[EmptyIdx].NodeId = NodeId;
		PmNotifiers[EmptyIdx].IpiMask = IpiMask;
		PmNotifiers[EmptyIdx].PendEvent = 0U;
		PmNotifiers[EmptyIdx].PendEventCnt = 0U;
		Idx = EmptyIdx;
	}

	/* Update event and wake mask for given entry */
	PmNotifiers[Idx].EventMask |= Event;
	if (0U != Wake) {
		/* Wake subsystem for this event */
		PmNotifiers[Idx].WakeMask |= Event;
	}

	if ((u8)EVENT_CPU_IDLE_FORCE_PWRDWN == Event) {
		XPm_Core* const Core = (XPm_Core *)XPmDevice_GetById(NodeId);
		if (NULL == Core) {
			goto done;
		}

		Core->IsCoreIdleSupported = 1U;
		Subsystem->Flags |= (u8)SUBSYSTEM_IDLE_SUPPORTED;
	}

	Status = XST_SUCCESS;
done:
	return Status;
}

static XStatus XPmNotifier_GetNotifyCbData(const u32 Idx, u32 *Payload)
{
	XStatus Status = XST_FAILURE;
	const XPmNotifier *Notifier = NULL;
	const XPm_Device *Device = NULL;
	const XPm_Power *Power = NULL;

	Notifier = &PmNotifiers[Idx];
	Payload[0] = (u32)PM_NOTIFY_CB;
	Payload[1] = Notifier->NodeId;

	switch (NODECLASS(Notifier->NodeId)) {
	case (u32)XPM_NODECLASS_EVENT:
		Status = XST_SUCCESS;
		Payload[2] = Notifier->PendEvent;
		Payload[3] = 0U;
		break;
	case (u32)XPM_NODECLASS_DEVICE:
		Device = XPmDevice_GetById(Notifier->NodeId);
		if (NULL == Device) {
			goto done;
		}
		if ((u32)EVENT_STATE_CHANGE ==
		    ((u32)EVENT_STATE_CHANGE & Notifier->PendEvent)) {
			Payload[2] = (u32)EVENT_STATE_CHANGE;
		} else if ((u32)EVENT_CPU_IDLE_FORCE_PWRDWN ==
			   ((u32)EVENT_CPU_IDLE_FORCE_PWRDWN &
			   Notifier->PendEvent)) {
			Payload[2] = (u32)EVENT_CPU_IDLE_FORCE_PWRDWN;
		} else {
			Payload[2] = (u32)EVENT_ZERO_USERS;
		}
		Payload[3] = Device->Node.State;
		Status = XST_SUCCESS;
		break;
	case (u32)XPM_NODECLASS_POWER:
		Power = XPmPower_GetById(Notifier->NodeId);
		if (NULL == Power) {
			goto done;
		}
		if ((u32)EVENT_STATE_CHANGE ==
		    ((u32)EVENT_STATE_CHANGE & Notifier->PendEvent)) {
			Payload[2] = (u32)EVENT_STATE_CHANGE;
		} else {
			Payload[2] = (u32)EVENT_ZERO_USERS;
		}
		Payload[3] = Power->Node.State;
		Status = XST_SUCCESS;
		break;
	default:
		PmErr("Unsupported Node Class: %d\r\n", NODECLASS(Notifier->NodeId));
		break;
	}
done:
	return Status;
}

static void XPmNotifier_SendPendingSuspendCb(XPm_Subsystem *SubSystem)
{
	XStatus IpiAck;
	u32 Payload[PAYLOAD_ARG_CNT] = {0};

	if (0U != SubSystem->PendCb.Reason) {
		PendingEvent = (u32)PRESENT;

		IpiAck = XPm_IpiPollForAck(SubSystem->IpiMask, XPM_NOTIFY_TIMEOUTCOUNT);
		if ((XST_SUCCESS == IpiAck) && ((u8)ONLINE == SubSystem->State)) {
			Payload[0] = (u32)PM_INIT_SUSPEND_CB;
			Payload[1] = SubSystem->PendCb.Reason;
			Payload[2] = SubSystem->PendCb.Latency;
			Payload[3] = SubSystem->PendCb.State;

			(*PmRequestCb)(SubSystem->IpiMask, (u32)PM_INIT_SUSPEND_CB,
				       Payload);

			SubSystem->PendCb.Reason = 0U;
		}
		if ((u8)ONLINE != SubSystem->State) {
			SubSystem->PendCb.Reason = 0U;
		}
	}
}

static void XPmNotifier_RmvFromEventSeq(const u32 Index)
{
	u32 Idx = Index;

	/* remove from list */
	while (Idx < (PosEmptySpace - 1U)) {
		/* Doing left shifting of data in list by 1 position */
		EventSeq[Idx] = EventSeq[Idx + 1U];
		Idx++;
	}

	EventSeq[Idx] = 0U; /* clear the data in list */

	PosEmptySpace -= 1U;
}

static XStatus XPmNotifier_AddInEventSeq(const u32 Index, const u32 NodeId)
{
	XStatus Status = XST_FAILURE;
	u32 Idx = 0U;
	u32 EmptyIdx;

	/* search for already present in list or not */
	for (Idx = 0U; Idx < XPM_NOTIFIERS_COUNT; Idx++) {
		if (((Index + 1U) == EventSeq[Idx]) &&
		    ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId))) {
			Status = XST_SUCCESS;
			goto done;
		}
	}

	EmptyIdx = PosEmptySpace;
	if (EmptyIdx < (u32)XPM_NOTIFIERS_COUNT) {
		EventSeq[EmptyIdx] = (DATA_TYPE)(Index + 1U);
		PosEmptySpace += 1U;
		Status = XST_SUCCESS;
	}
done:
	return Status;
}

static void XPmNotifier_SendPendingNotifyEvent(const XPm_Subsystem *SubSystem)
{
	XStatus Status = XST_FAILURE;
	u32 Payload[PAYLOAD_ARG_CNT] = {0};
	XStatus IpiAck;
	u32 NodeId;
	u32 Event;
	u32 Index = 0U;
	u32 Idx = 0;
	XPmNotifier* Notifier = NULL;
	u32 TempSubSystemId ;

	/* Check Subsystem is ready or not */
	IpiAck = XPm_IpiPollForAck(SubSystem->IpiMask, XPM_NOTIFY_TIMEOUTCOUNT);
	if (XST_SUCCESS == IpiAck) {
		/* Serch and get pending event for given SubSystem from event
		 * sequence.
		 */
		while (Idx < PosEmptySpace) {
			Index = ((u32)EventSeq[Idx] - 1U);
			Notifier = &PmNotifiers[Index];
			TempSubSystemId = Notifier->SubsystemId;
			if ((SubSystem->Id == TempSubSystemId) &&
			    (0U != Notifier->PendEventCnt)) {
				break;
			}
			Idx++;
		}

		if (Idx != PosEmptySpace) {
			PendingEvent = (u32)PRESENT;
			Status = XPmNotifier_GetNotifyCbData(Index, Payload);
			if (XST_SUCCESS != Status) {
				XPmNotifier_RmvFromEventSeq(Idx);
				Notifier->PendEvent = 0U;
				Notifier->PendEventCnt = 0U;
				goto done;
			}

			NodeId = Payload[1];
			Event = Payload[2];
			const XPm_Subsystem* NotifierSubsystem = XPmSubsystem_GetById(Notifier->SubsystemId);
			if (((u8)ONLINE ==NotifierSubsystem->State) ||
			    ((u8)PENDING_POWER_OFF ==  NotifierSubsystem->State) ||
			    ((u8)PENDING_RESTART == NotifierSubsystem->State) ||
			    (0U != (Event & Notifier->WakeMask))) {
				/* Send Notification */
				(*PmRequestCb)(Notifier->IpiMask,
					       (u32)PM_NOTIFY_CB,
					       Payload);
			}

			if ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId)) {
				Notifier->PendEventCnt = 0U;
			} else {
				Notifier->PendEventCnt -= 1U;
			}

			if (0U == Notifier->PendEventCnt) {
				Notifier->PendEvent = 0U;
				XPmNotifier_RmvFromEventSeq(Idx);
			}
		}
	} else {
		/* Mark this Flag as present to avoid removal of scheduler
		 * task as subsystem is busy.
		 */
		PendingEvent = (u32)PRESENT;
	}

done:
	return;
}

static int XPmNotifier_SchedulerTask(void *Arg)
{
	(void)Arg;
	int Status = XST_FAILURE;
	u32 Index = 0U;
	XPm_Subsystem *SubSystem = NULL;
	u32 MaxSubIdx = XPmSubsystem_GetMaxSubsysIdx();
	PendingEvent = (u32)NOT_PRESENT;

	/* Find and Send Notification */
	for (Index = 0; Index <= MaxSubIdx; Index++) {
		SubSystem = XPmSubsystem_GetByIndex(Index);
		if (NULL == SubSystem) {
			continue;
		}

		/* Check for suspend callback priority */
		if ( SUBSYSTEM_SUSCB_PRIORITIZE == (SubSystem->Flags & SUBSYSTEM_SUSCB_PRIORITIZE )) {
			/* Remove priority flag for suspend call back for next scheduler cycle */
			SubSystem->Flags &= (u8)(~(SUBSYSTEM_SUSCB_PRIORITIZE));

			/* Send pending suspend callback */
			XPmNotifier_SendPendingSuspendCb(SubSystem);

			/* Send pending notify events per subsystem */
			XPmNotifier_SendPendingNotifyEvent(SubSystem);
		} else {
			/* Set priority flag for suspend call back for next scheduler cycle */
			SubSystem->Flags |= (u8)(SUBSYSTEM_SUSCB_PRIORITIZE);

			/* Send pending notify events per subsystem */
			XPmNotifier_SendPendingNotifyEvent(SubSystem);

			/* Send pending suspend callback */
			XPmNotifier_SendPendingSuspendCb(SubSystem);
		}
	}

	/*
	 * Remove the scheduler task when the loops is in the beginning of the
	 * function start at 0 and don't find any event.
	 */
	if ((u32)NOT_PRESENT == PendingEvent) {
		Status = XPlmi_SchedulerRemoveTask(XPLMI_MODULE_XILPM_ID,
						   &XPmNotifier_SchedulerTask,
						   XILPM_NOTIFIER_INTERVAL,
						   NULL);
		if (Status != XST_SUCCESS) {
			PmErr("[%s] Failed to remove task\r\n", __func__);
			goto done;
		}
		SchedulerTask = (u32)NOT_PRESENT;
	}
	Status = XST_SUCCESS;
done:
	return Status;
}

static XStatus XPmNotifier_AddSuspEvent(const u32 IpiMask, const u32 *Payload)
{
	XStatus Status = XST_FAILURE;
	XPm_Subsystem *Subsystem = NULL;
	u32 SubsystemId;

	SubsystemId = XPmSubsystem_GetSubSysIdByIpiMask(IpiMask);
	Subsystem = XPmSubsystem_GetById(SubsystemId);
	if (NULL == Subsystem) {
		Status = XPM_INVALID_SUBSYSID;
		goto done;
	}

	/* Store pending suspend callback entry in empty subsystem */
	Subsystem->PendCb.Reason = Payload[1];
	Subsystem->PendCb.Latency = Payload[2];
	Subsystem->PendCb.State = Payload[3];

	/* Set priority flag for suspend call back */
	Subsystem->Flags |= (u8)(SUBSYSTEM_SUSCB_PRIORITIZE);

	Status = XST_SUCCESS;
done:
	return Status;
}

static XStatus XPmNotifier_AddPendingEvent(const u32 IpiMask, const u32 *Payload)
{
	XStatus Status = XST_FAILURE;
	XPmNotifier* Notifier = NULL;
	u32 CbType;
	u32 NodeId;
	u32 Event;
	u32 Idx;

	if (NULL == Payload) {
		goto done;
	}

	CbType = Payload[0];

	switch (CbType) {
	case (u32)PM_INIT_SUSPEND_CB:
		Status = XPmNotifier_AddSuspEvent(IpiMask, Payload);
		break;
	case (u32)PM_NOTIFY_CB:
		NodeId = Payload[1];
		Event = Payload[2];
		for (Idx = 0U; Idx < ARRAY_SIZE(PmNotifiers); Idx++) {
			/* Search for the given NodeId */
			if ((NodeId == PmNotifiers[Idx].NodeId) &&
			    (Event == (Event & PmNotifiers[Idx].EventMask))) {
				Notifier = &PmNotifiers[Idx];
				Notifier->PendEvent |= Event;
				Notifier->PendEventCnt += 1U;

				/* Store Index of PmNotifiers[] in Event sequence to get
				 * order of event occurence in terms of Node-Id
				 */
				Status = XPmNotifier_AddInEventSeq(Idx, NodeId);
				if (XST_SUCCESS != Status) {
					break;
				}
			}
		}
		break;
	default:
		PmErr("Invalid callback type %d\r\n", CbType);
		break;
	}

	if (XST_SUCCESS != Status) {
		goto done;
	}
	if ((u32)NOT_PRESENT == SchedulerTask) {
		Status = XPlmi_SchedulerAddTask(XPLMI_MODULE_XILPM_ID,
						&XPmNotifier_SchedulerTask, NULL,
						XILPM_NOTIFIER_INTERVAL,
						XPLM_TASK_PRIORITY_0,
						NULL, XPLMI_PERIODIC_TASK);
		if (Status != XST_SUCCESS) {
			PmErr("[%s] Failed to create task\r\n",__func__);
			goto done;
		}
		SchedulerTask = (u32)PRESENT;
	}
done:
	return Status;
}

/****************************************************************************/
/**
 * @brief  Notify target only if IPI acked for previous request else add as
 *	   pending event.
 *
 * @param  IpiMask    IpiMask of the target
 * @param  Payload    Payload with callback data
 *
 ****************************************************************************/
void XPmNotifier_NotifyTarget(u32 IpiMask, u32 *Payload)
{
	XStatus IpiAck;
	u32 CbType = Payload[0];
	u32 TaskPresent = SchedulerTask;

	IpiAck = XPm_IpiPollForAck(IpiMask, XPM_NOTIFY_TIMEOUTCOUNT);
	if ((XST_SUCCESS == IpiAck) &&
	    (NULL != PmRequestCb) &&
	    ((u32)NOT_PRESENT == TaskPresent)) {
		(*PmRequestCb)(IpiMask, CbType, Payload);
	} else if (NULL != PmRequestCb){
		(void)XPmNotifier_AddPendingEvent(IpiMask, Payload);
	} else {
		PmInfo("Invalid Call back Handler \r\n");
	}
}

/****************************************************************************/
/**
 * @brief  Unregister the notifier for given subsystem, NodeId and event
 *
 * @param  Subsystem  Subsystem which was registered for notification
 * @param  NodeId     NodeId related to event
 * @param  Event      Notification event
 *
 * @return XST_SUCCESS if successful else XST_FAILURE or an error code
 * or a reason code
 *
 ****************************************************************************/
XStatus XPmNotifier_Unregister(XPm_Subsystem* const Subsystem,
			    const u32 NodeId,
			    const u32 Event)
{
	u32 Idx;
	u32 Pos;
	XStatus Status = XST_FAILURE;

	for (Idx = 0U; Idx < ARRAY_SIZE(PmNotifiers); Idx++) {

		if ((Subsystem->Id == PmNotifiers[Idx].SubsystemId) &&
		    (Event == (Event & PmNotifiers[Idx].EventMask)) &&
		    (NodeId == PmNotifiers[Idx].NodeId)) {
			/* Entry for subsystem/NodeId pair found */
			PmNotifiers[Idx].EventMask &= ~Event;
			PmNotifiers[Idx].WakeMask &= ~Event;
			if (0U != (PmNotifiers[Idx].PendEvent & Event)) {
				PmNotifiers[Idx].PendEvent &= ~Event;
			}
			if (0U == PmNotifiers[Idx].EventMask) {
				for (Pos = 0; Pos < XPM_NOTIFIERS_COUNT; Pos++ ) {
					if ((Idx + 1U) == EventSeq[Pos]) {
						XPmNotifier_RmvFromEventSeq(Pos);
					}
				}
				Status = Xil_SMemSet(&PmNotifiers[Idx],
						     sizeof(XPmNotifier),
						     0,
						     sizeof(XPmNotifier));
				if (XST_SUCCESS != Status) {
					goto done;
				}
			}
			/*
			 * Check if Node Class is EVENT and disable error action.
			 */
			if ((u32)XPM_NODECLASS_EVENT == NODECLASS(NodeId)) {
				(void)XPlmi_EmDisable(NodeId, Event);
			}

			if ((u8)EVENT_CPU_IDLE_FORCE_PWRDWN == Event) {
				XPm_Core* const Core = (XPm_Core *)XPmDevice_GetById(NodeId);
				if (NULL == Core) {
					goto done;
				}

				Core->IsCoreIdleSupported = 0U;
				Subsystem->Flags &= (u8)(~SUBSYSTEM_IDLE_SUPPORTED);
			}
			break;
		}
	}
	Status = XST_SUCCESS;

done:
	return Status;
}

/****************************************************************************/
/**
 * @brief  This function unregisters all notifiers of the given subsystem
 *
 * @param  Subsystem  Subsystem for which notifiers to be unregistered
 *
 * @return XST_SUCCESS if successful else XST_FAILURE or an error code
 * or a reason code
 *
 ****************************************************************************/
XStatus XPmNotifier_UnregisterAll(const XPm_Subsystem* const Subsystem)
{
	u32 Idx;
	u32 Pos;
	XStatus Status = XST_FAILURE;

	for (Idx = 0U; Idx < ARRAY_SIZE(PmNotifiers); Idx++) {
		if (Subsystem->Id == PmNotifiers[Idx].SubsystemId) {
			for (Pos = 0; Pos < XPM_NOTIFIERS_COUNT; Pos++ ) {
				if ((Idx + 1U) == EventSeq[Pos]) {
					XPmNotifier_RmvFromEventSeq(Pos);
				}
			}
			Status = Xil_SMemSet(&PmNotifiers[Idx], sizeof(XPmNotifier),
					     0, sizeof(XPmNotifier));
			if (XST_SUCCESS != Status) {
				goto done;
			}
		}
	}
	Status = XST_SUCCESS;

done:
	return Status;
}

/****************************************************************************/
/**
 * @brief  This function triggers the notification if enabled for current
 *	   NodeId and current event.
 *
 * @param  NodeId  NodeId for which event is occurred
 * @param  Event   Event type
 *
 * @return None
 *
 ****************************************************************************/
void XPmNotifier_Event(const u32 NodeId, const u32 Event)
{
	u32 Idx;
	const XPmNotifier* Notifier = NULL;
	u32 Payload[PAYLOAD_ARG_CNT] = {0};
	const XPm_Device* Device;
	const XPm_Power *Power;
	XStatus Status = XST_FAILURE;

	for (Idx = 0U; Idx < ARRAY_SIZE(PmNotifiers); Idx++) {
		/* Search for the given NodeId */
		if ((NodeId != PmNotifiers[Idx].NodeId) ||
		    (0U == (Event & PmNotifiers[Idx].EventMask))) {
			continue; /* Match not found */
		}

		Notifier = &PmNotifiers[Idx];

		if (NULL == PmRequestCb) {
			goto done;
		}

		/* Populate the PayLoad */
		Payload[0] = (u32)PM_NOTIFY_CB;
		Payload[1] = Notifier->NodeId;
		Payload[2] = Event;

		switch (NODECLASS(NodeId)) {
		case (u32)XPM_NODECLASS_EVENT:
			/* Disable the error event. Agent will re-register for
			 * notification if needed */
			(void)XPlmi_EmDisable(NodeId, Event);
			Payload[3] = 0U;
			Status = XST_SUCCESS;
			break;
		case (u32)XPM_NODECLASS_DEVICE:
			Device = XPmDevice_GetById(NodeId);
			if (NULL != Device) {
				Payload[3] = Device->Node.State;
				Status = XST_SUCCESS;
			}
			break;
		case (u32)XPM_NODECLASS_POWER:
			Power = XPmPower_GetById(NodeId);
			if (NULL != Power) {
				Payload[3] = Power->Node.State;
				Status = XST_SUCCESS;
			}
			break;
		default:
			PmErr("Unsupported Node Class: %d\r\n", NODECLASS(NodeId));
			break;
		}
		if (XST_SUCCESS != Status) {
			goto done;
		}

		/*
		 * If subsystem is OFFLINE then it should be notified about
		 * the event only if it requested to be woken up.
		 */
		const XPm_Subsystem* NotifierSubsystem = XPmSubsystem_GetById(Notifier->SubsystemId);
		if (((u8)OFFLINE != NotifierSubsystem->State) ||
		    (0U != (Event & Notifier->WakeMask))) {
			XPmNotifier_NotifyTarget(Notifier->IpiMask, Payload);
		}
	}

 done:
	return;
}
