/**************************************************************************************************
* Copyright (c) 2024 - 2025 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
**************************************************************************************************/

/*************************************************************************************************/
/**
*
* @file	xasufw_rsa_client_example.c
* @addtogroup xasufw_rsa_client_example XilAsu RSA API Example Generic Usage
* @{
*
* @note
* This example illustrates the usage of ASU RSA APIs using below tests
* 	Encrypt the data with provided key. The output result is then decrypted
* 	back to get original data. The test fails, if decryption not
* 	produce the original data.
*
* MODIFICATION HISTORY:
* <pre>
* Ver   Who    Date     Changes
* ----- ------ -------- -------------------------------------------------
* 1.0   ss    08/20/24 Initial Release
*       ss    09/19/24 Added print for client init failure
*       am    09/24/24 Added SDT support
*       ss    12/24/24 Added Keysize parameter
*
* </pre>
**************************************************************************************************/

/*************************************** Include Files *******************************************/
#include "xil_cache.h"
#include "xil_util.h"
#include "xasu_rsa.h"
#include "xasu_client.h"

/************************************ Constant Definitions ***************************************/
#define XASU_RSA_4096_KEY_SIZE_IN_BYTES	(512U)	/**< 512 bytes for 4096 bit data */
#define XASU_CACHE_DISABLE
/************************************** Type Definitions *****************************************/

/*************************** Macros (Inline Functions) Definitions *******************************/

/************************************ Function Prototypes ****************************************/
static void XAsu_RsaCallBackRef(void *CallBackRef, u32 Status);

/************************************ Variable Definitions ***************************************/
static const u8 XAsu_RsaCt[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((
		section (".data.XAsu_RsaCt"))) = {
	0x78U, 0x45U, 0x42U, 0x51U, 0xF0U, 0x0AU, 0xF6U, 0x1FU, 0x62U, 0x16U, 0x75U, 0xE2U, 0x7DU, 0xD1U,
	0x9BU, 0x7DU, 0x6AU, 0xE2U, 0x16U, 0x66U, 0x03U, 0x6AU, 0x45U, 0x59U, 0xEBU, 0x7DU, 0x4CU, 0x09U,
	0x1FU, 0x46U, 0xEDU, 0x24U, 0xD6U, 0x5CU, 0xADU, 0x09U, 0xB7U, 0x1AU, 0xC9U, 0xF7U, 0xEFU, 0x5CU,
	0xF1U, 0x3CU, 0x43U, 0xBCU, 0x04U, 0xA3U, 0xE0U, 0xC9U, 0x82U, 0xABU, 0x67U, 0x25U, 0x5DU, 0x69U,
	0xBDU, 0x53U, 0x34U, 0xF5U, 0x58U, 0xEDU, 0x35U, 0xD7U, 0x6AU, 0x1EU, 0xE2U, 0xCFU, 0x0FU, 0xC4U,
	0x35U, 0x41U, 0xE6U, 0xD1U, 0x12U, 0x16U, 0xC0U, 0x7DU, 0x17U, 0x88U, 0xBFU, 0xC9U, 0x91U, 0xEAU,
	0xD9U, 0x99U, 0x3DU, 0xD4U, 0x27U, 0xEBU, 0xB1U, 0x75U, 0x99U, 0xF6U, 0x13U, 0x28U, 0x8DU, 0x8DU,
	0xA1U, 0x64U, 0xD7U, 0xE7U, 0xC9U, 0xCDU, 0x37U, 0xF6U, 0x36U, 0x09U, 0x71U, 0xB8U, 0x33U, 0x26U,
	0xDEU, 0xD5U, 0xADU, 0xFBU, 0x29U, 0xC7U, 0x86U, 0x6AU, 0xD3U, 0xC7U, 0x42U, 0x81U, 0xCAU, 0xD9U,
	0xBBU, 0x91U, 0xA0U, 0x8CU, 0x68U, 0xF8U, 0xD8U, 0x70U, 0x02U, 0xC6U, 0x9EU, 0x39U, 0xC1U, 0xE0U,
	0x82U, 0x81U, 0xD4U, 0x04U, 0x63U, 0x3EU, 0x86U, 0x19U, 0x71U, 0x99U, 0x5EU, 0xA7U, 0x48U, 0x92U,
	0x71U, 0x9BU, 0x41U, 0xB7U, 0x57U, 0x61U, 0xE7U, 0xBBU, 0x35U, 0xEBU, 0x6CU, 0x55U, 0xF3U, 0xA8U,
	0xD9U, 0xE8U, 0x86U, 0x92U, 0x66U, 0xBEU, 0x57U, 0x13U, 0x08U, 0xF9U, 0x4DU, 0x58U, 0xB6U, 0xCEU,
	0x6BU, 0x99U, 0xC6U, 0x8BU, 0x4CU, 0xD9U, 0x98U, 0x62U, 0x4BU, 0x5EU, 0x64U, 0xBAU, 0x2FU, 0x49U,
	0x17U, 0x93U, 0x1BU, 0xB1U, 0xF0U, 0xABU, 0xD8U, 0xC2U, 0x03U, 0x0EU, 0x8CU, 0x30U, 0x85U, 0xFCU,
	0x66U, 0x30U, 0xECU, 0x1BU, 0x97U, 0x82U, 0xF7U, 0xC5U, 0x55U, 0x30U, 0x85U, 0xD8U, 0xF5U, 0xD6U,
	0x3FU, 0x2DU, 0xC9U, 0xC5U, 0xCBU, 0x2CU, 0x6BU, 0x63U, 0xB2U, 0x48U, 0xEEU, 0x77U, 0x21U, 0x35U,
	0x3AU, 0xC9U, 0x91U, 0xAAU, 0x25U, 0x7DU, 0xE9U, 0x73U, 0x15U, 0x56U, 0x9AU, 0x12U, 0x2EU, 0x6BU,
	0x23U, 0x00U, 0xBEU, 0x92U, 0x1DU, 0xADU, 0xA4U, 0xACU, 0x1DU, 0xABU, 0x69U, 0x48U, 0x83U, 0x1AU,
	0x4FU, 0x16U, 0xF2U, 0x4AU, 0x41U, 0xE5U, 0xC1U, 0x1EU, 0xBEU, 0x17U, 0x30U, 0x6EU, 0x30U, 0xF4U,
	0x2BU, 0x6AU, 0xB5U, 0x70U, 0xF7U, 0x5DU, 0xC4U, 0xA0U, 0xAEU, 0x73U, 0xB0U, 0x0DU, 0xB8U, 0xE0U,
	0x42U, 0x9AU, 0x8EU, 0xACU, 0x4BU, 0x11U, 0x54U, 0x6AU, 0xC6U, 0x84U, 0x59U, 0xCBU, 0x6AU, 0x1EU,
	0x1FU, 0xF2U, 0x4FU, 0xBCU, 0x8BU, 0xF9U, 0xD5U, 0x06U, 0x1AU, 0xE8U, 0x88U, 0xC4U, 0xDCU, 0x89U,
	0xEFU, 0x1BU, 0x46U, 0x68U, 0xD2U, 0xBAU, 0xDEU, 0xA9U, 0x90U, 0x7FU, 0xF7U, 0xCDU, 0x1EU, 0xF6U,
	0x89U, 0x30U, 0xE2U, 0x41U, 0x95U, 0xE7U, 0xC1U, 0xDAU, 0x3DU, 0xC2U, 0xA9U, 0xCFU, 0x62U, 0x03U,
	0x83U, 0xB4U, 0x21U, 0x5AU, 0xE4U, 0x7CU, 0xD0U, 0x7FU, 0x12U, 0xE8U, 0xDFU, 0x5FU, 0x97U, 0xFBU,
	0x21U, 0xD4U, 0x80U, 0x6BU, 0x13U, 0x6AU, 0x4EU, 0x64U, 0xF0U, 0xABU, 0xD4U, 0x60U, 0xD3U, 0x4FU,
	0xB3U, 0xACU, 0x45U, 0xF3U, 0x2EU, 0x6CU, 0x53U, 0x48U, 0xFBU, 0xAAU, 0xB0U, 0x2EU, 0x96U, 0xECU,
	0xA7U, 0x20U, 0xDDU, 0xE5U, 0xCEU, 0xDEU, 0x4DU, 0x2DU, 0xC0U, 0x4EU, 0x43U, 0x03U, 0xC9U, 0x7CU,
	0x9CU, 0x74U, 0x9BU, 0xF9U, 0xD4U, 0x46U, 0x69U, 0xDEU, 0x95U, 0xEBU, 0xEBU, 0x26U, 0x0EU, 0x36U,
	0x35U, 0x62U, 0x69U, 0xDBU, 0x67U, 0x29U, 0x23U, 0x57U, 0xFBU, 0xA2U, 0xE0U, 0x03U, 0x0FU, 0xBFU,
	0x35U, 0xF8U, 0x45U, 0x21U, 0xE9U, 0x91U, 0x09U, 0xE5U, 0xB2U, 0x4CU, 0x1CU, 0x10U, 0x83U, 0x20U,
	0x66U, 0xB7U, 0xA4U, 0xE5U, 0xDEU, 0x06U, 0x07U, 0xB6U, 0x25U, 0x3CU, 0xF1U, 0x62U, 0x7EU, 0xD3U,
	0x9CU, 0x27U, 0xEEU, 0x0DU, 0x26U, 0x65U, 0xD0U, 0x0EU, 0x42U, 0x66U, 0x15U, 0x75U, 0x5FU, 0x33U,
	0x6FU, 0x3CU, 0x10U, 0x5CU, 0xB4U, 0x39U, 0x8DU, 0xD5U, 0xD9U, 0xEEU, 0xEAU, 0xEAU, 0xFBU, 0xDDU,
	0x38U, 0xA2U, 0xCCU, 0x54U, 0x90U, 0xB8U, 0xB0U, 0x80U, 0xF2U, 0xF4U, 0x9EU, 0x29U, 0xBEU, 0x84U,
	0x74U, 0xEBU, 0x9AU, 0x03U, 0xF4U, 0x55U, 0x39U, 0x69U
};

static const u8 XAsu_RsaModulus[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((
		section (".data.XAsu_RsaModulus"))) = {
	0xCCU, 0x90U, 0xB4U, 0xB1U, 0xF1U, 0xC1U, 0xD3U, 0x9DU, 0x98U, 0xB1U, 0x34U, 0x49U, 0x92U, 0xCDU,
	0xFDU, 0xD1U, 0xCFU, 0x87U, 0x43U, 0x52U, 0xAFU, 0x9CU, 0x5CU, 0xFBU, 0x62U, 0xC9U, 0xAAU, 0x7EU,
	0xCAU, 0xFDU, 0x0EU, 0x2AU, 0xF9U, 0xBFU, 0x86U, 0xBCU, 0xDEU, 0xFDU, 0x10U, 0x42U, 0x79U, 0xF2U,
	0x18U, 0x6BU, 0xB1U, 0xCFU, 0x8DU, 0x66U, 0xDEU, 0xB0U, 0x64U, 0x1EU, 0xB1U, 0x45U, 0x45U, 0x20U,
	0x80U, 0xCCU, 0xCBU, 0x6AU, 0xADU, 0x1AU, 0x0AU, 0x2AU, 0xA9U, 0xE2U, 0xA6U, 0x41U, 0xABU, 0xD3U,
	0x3AU, 0xEDU, 0x34U, 0x71U, 0x2EU, 0x2DU, 0x08U, 0xF5U, 0x16U, 0x7BU, 0x89U, 0xEBU, 0xBAU, 0xD5U,
	0x86U, 0x33U, 0x0FU, 0x26U, 0x80U, 0x9FU, 0x3CU, 0x98U, 0x11U, 0x62U, 0xDCU, 0x12U, 0x41U, 0xE9U,
	0x7AU, 0x3CU, 0xD9U, 0x1BU, 0x0CU, 0xE5U, 0x33U, 0x55U, 0x25U, 0x51U, 0xEFU, 0x2DU, 0xF2U, 0x3EU,
	0x6EU, 0x7FU, 0xF9U, 0x76U, 0x86U, 0x6CU, 0xA9U, 0x3CU, 0x04U, 0xFFU, 0xADU, 0xD1U, 0x9DU, 0x93U,
	0xC1U, 0x04U, 0x4CU, 0x55U, 0xABU, 0xFDU, 0xE7U, 0x71U, 0xAFU, 0xF5U, 0xA6U, 0xE1U, 0x51U, 0xE7U,
	0xBAU, 0x4DU, 0x94U, 0x22U, 0xD2U, 0x07U, 0x7BU, 0x64U, 0x11U, 0xC2U, 0x56U, 0xFAU, 0xBEU, 0x7CU,
	0x1DU, 0x5FU, 0x57U, 0x66U, 0x4EU, 0x88U, 0xD4U, 0x58U, 0x1FU, 0x12U, 0x07U, 0x65U, 0x77U, 0xA3U,
	0x2AU, 0xF4U, 0x8DU, 0xDEU, 0xDAU, 0x3CU, 0xA1U, 0x0DU, 0x41U, 0x3EU, 0x58U, 0x77U, 0x1EU, 0x59U,
	0x71U, 0xDBU, 0xD2U, 0xDDU, 0x48U, 0x20U, 0xE8U, 0xB9U, 0x09U, 0xB0U, 0xD5U, 0x95U, 0x26U, 0xCFU,
	0xAEU, 0x4FU, 0x31U, 0xDFU, 0x10U, 0xE7U, 0x7BU, 0x34U, 0x44U, 0x3EU, 0x2DU, 0x4EU, 0x7EU, 0x9BU,
	0xBAU, 0x9CU, 0xBDU, 0xB4U, 0xC9U, 0xD9U, 0x8AU, 0x09U, 0x74U, 0x75U, 0x3DU, 0xD2U, 0x9FU, 0xDFU,
	0x1AU, 0xE8U, 0x6AU, 0xA0U, 0x09U, 0xD3U, 0x66U, 0x04U, 0x8DU, 0xA8U, 0x7BU, 0x4BU, 0x2EU, 0x3AU,
	0x42U, 0xACU, 0xD7U, 0x24U, 0xFFU, 0x5FU, 0x9CU, 0x09U, 0x7FU, 0xEFU, 0x22U, 0xA5U, 0x5AU, 0x7EU,
	0x3BU, 0x63U, 0x36U, 0x38U, 0xF8U, 0xEBU, 0xFFU, 0xAEU, 0xFDU, 0xDEU, 0x19U, 0xB6U, 0x38U, 0xC2U,
	0x91U, 0x83U, 0x75U, 0x5FU, 0x86U, 0x9EU, 0xA7U, 0x99U, 0xA1U, 0xC0U, 0x75U, 0x7EU, 0x43U, 0x8CU,
	0xE7U, 0x4BU, 0x66U, 0x4CU, 0x3BU, 0xBBU, 0x11U, 0x3EU, 0xF2U, 0x9DU, 0x84U, 0x4AU, 0x2FU, 0xACU,
	0xBFU, 0x2CU, 0xC2U, 0xA9U, 0x97U, 0xDCU, 0xC4U, 0x0FU, 0x09U, 0x2CU, 0xCCU, 0x1DU, 0x57U, 0xD6U,
	0x34U, 0xB0U, 0x43U, 0xB9U, 0xFBU, 0x45U, 0xC9U, 0x51U, 0xE8U, 0x87U, 0x39U, 0x90U, 0x49U, 0x81U,
	0x1FU, 0x0EU, 0x6DU, 0x1AU, 0x4FU, 0xC2U, 0x04U, 0xEEU, 0x69U, 0xF0U, 0x67U, 0xCEU, 0xDCU, 0x92U,
	0xC5U, 0x8CU, 0xCEU, 0x91U, 0x89U, 0x2CU, 0x56U, 0xFBU, 0xF8U, 0x70U, 0x8FU, 0xB9U, 0xCFU, 0xA0U,
	0xBDU, 0x9FU, 0x8AU, 0xF7U, 0xC5U, 0xEDU, 0xA1U, 0x57U, 0xD4U, 0x27U, 0x70U, 0x34U, 0xD5U, 0x59U,
	0x6AU, 0x93U, 0x48U, 0x11U, 0x00U, 0x02U, 0xD0U, 0x92U, 0xE6U, 0x1BU, 0xA3U, 0xFCU, 0xF8U, 0x04U,
	0x64U, 0x08U, 0x0BU, 0xDBU, 0x49U, 0x93U, 0xE3U, 0xB4U, 0x4DU, 0x68U, 0xEBU, 0x9EU, 0xEAU, 0xABU,
	0x19U, 0x36U, 0xA3U, 0xCFU, 0x39U, 0x6AU, 0xDDU, 0x24U, 0x1AU, 0x4EU, 0x70U, 0x64U, 0xF4U, 0x0CU,
	0x2CU, 0x54U, 0xCEU, 0xB3U, 0x78U, 0x53U, 0xEAU, 0x2DU, 0xB0U, 0xDBU, 0xEEU, 0x58U, 0x86U, 0x00U,
	0x42U, 0x99U, 0x8FU, 0xCFU, 0x2EU, 0x1FU, 0xCBU, 0xA5U, 0x60U, 0x44U, 0xD4U, 0xBEU, 0x05U, 0x2CU,
	0xD4U, 0x62U, 0xD8U, 0xB4U, 0x38U, 0xCBU, 0x80U, 0x38U, 0x78U, 0x42U, 0x43U, 0xACU, 0xEBU, 0xA5U,
	0xBDU, 0x05U, 0xFCU, 0x13U, 0xA7U, 0x2BU, 0x67U, 0x82U, 0x02U, 0x4CU, 0x74U, 0x8EU, 0xAAU, 0xD0U,
	0x28U, 0xD1U, 0x39U, 0x8CU, 0xEFU, 0x61U, 0xB2U, 0xFFU, 0x60U, 0x41U, 0x83U, 0x99U, 0x81U, 0x4DU,
	0xCFU, 0x81U, 0x7CU, 0xCCU, 0xECU, 0xE3U, 0xF5U, 0xF1U, 0x17U, 0x2BU, 0x32U, 0xF2U, 0x46U, 0x8AU,
	0xC4U, 0x93U, 0x20U, 0xC1U, 0xD8U, 0xFCU, 0xC8U, 0x23U, 0x71U, 0xBFU, 0x37U, 0x85U, 0x00U, 0x7CU,
	0xEDU, 0xB6U, 0x84U, 0xABU, 0x45U, 0x00U, 0x39U, 0xA5U
};
static const u8 XAsu_RsaPt[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((
		section (".data.XAsu_RsaPt"))) = {
	0x05U, 0x95U, 0x87U, 0x3FU, 0xA5U, 0x76U, 0x50U, 0xBFU, 0x76U, 0x6FU, 0x2CU, 0xB3U, 0x97U, 0x80U,
	0x8BU, 0x7BU, 0x99U, 0xE8U, 0x56U, 0x2FU, 0x4BU, 0xCDU, 0x66U, 0x25U, 0x6FU, 0xA9U, 0x51U, 0x7CU,
	0x42U, 0x57U, 0x95U, 0x95U, 0x6AU, 0xF0U, 0xF8U, 0x7EU, 0x56U, 0x0DU, 0x28U, 0x07U, 0xE7U, 0x1CU,
	0x7CU, 0xD8U, 0x9EU, 0xF2U, 0xD6U, 0x37U, 0xE8U, 0x7AU, 0x5AU, 0xDBU, 0xA5U, 0xB5U, 0xD2U, 0x93U,
	0xFAU, 0x29U, 0x28U, 0x76U, 0xB8U, 0xCEU, 0xC3U, 0x66U, 0xFFU, 0x15U, 0xFBU, 0x28U, 0x3BU, 0xC2U,
	0x82U, 0x32U, 0xB7U, 0x5FU, 0xE4U, 0xD5U, 0x54U, 0x8EU, 0xE5U, 0x43U, 0x7DU, 0xAEU, 0x02U, 0x21U,
	0x5BU, 0x55U, 0x9CU, 0xD0U, 0xCEU, 0x65U, 0x05U, 0x47U, 0xD2U, 0x5AU, 0x6FU, 0x2CU, 0x34U, 0x3EU,
	0x48U, 0x6BU, 0xE1U, 0xCAU, 0x63U, 0xECU, 0x65U, 0xA0U, 0xD6U, 0x62U, 0xA0U, 0x59U, 0x17U, 0x85U,
	0xA2U, 0xA3U, 0xB4U, 0x6CU, 0xD1U, 0x58U, 0x5CU, 0xEEU, 0xF9U, 0x2BU, 0x86U, 0x10U, 0xFCU, 0xDBU,
	0x14U, 0x59U, 0x49U, 0x6AU, 0xA1U, 0xBCU, 0x46U, 0x51U, 0xFAU, 0xCFU, 0xFEU, 0x75U, 0x76U, 0xA7U,
	0xFBU, 0xD3U, 0x07U, 0xB0U, 0xC1U, 0x66U, 0x97U, 0x8DU, 0x46U, 0x5CU, 0xDAU, 0x20U, 0x5CU, 0xE4U,
	0xD4U, 0x61U, 0xFDU, 0x24U, 0x7EU, 0x61U, 0xFBU, 0x61U, 0x80U, 0x99U, 0x4FU, 0x16U, 0xB4U, 0x30U,
	0xDAU, 0xBDU, 0x70U, 0xA0U, 0xFEU, 0xA3U, 0x5CU, 0xC9U, 0xF6U, 0x39U, 0xEFU, 0x20U, 0x25U, 0xBCU,
	0xD4U, 0x28U, 0x6FU, 0x88U, 0x1AU, 0xD2U, 0xF0U, 0x02U, 0x42U, 0xA1U, 0xE3U, 0x45U, 0x2BU, 0xADU,
	0xABU, 0x5BU, 0x93U, 0xBDU, 0x26U, 0x7DU, 0xAEU, 0xDCU, 0x48U, 0x86U, 0x8AU, 0x41U, 0x9FU, 0x75U,
	0xA4U, 0x33U, 0x23U, 0x8CU, 0x9BU, 0xB9U, 0x3BU, 0x80U, 0xBDU, 0xDBU, 0x34U, 0x4FU, 0x6FU, 0x8EU,
	0xC4U, 0xFAU, 0xB8U, 0xB7U, 0x48U, 0x53U, 0x17U, 0x8EU, 0x55U, 0x3BU, 0xC7U, 0xBFU, 0x2CU, 0x53U,
	0x05U, 0xEBU, 0xADU, 0x96U, 0xA9U, 0x37U, 0xB0U, 0x5DU, 0x42U, 0xD5U, 0xE1U, 0x91U, 0x58U, 0x56U,
	0x62U, 0x6BU, 0xE4U, 0xE5U, 0x5DU, 0xFFU, 0x47U, 0x7AU, 0x42U, 0xE7U, 0x51U, 0xA0U, 0xE2U, 0x1BU,
	0x49U, 0xEFU, 0x54U, 0xD1U, 0x7EU, 0xABU, 0xF9U, 0xB9U, 0x5CU, 0x5EU, 0x12U, 0x8FU, 0x38U, 0x6AU,
	0xDFU, 0xA4U, 0xAEU, 0x2FU, 0x97U, 0x29U, 0xC0U, 0x9BU, 0x46U, 0xA2U, 0x9CU, 0x95U, 0xCBU, 0x94U,
	0x97U, 0x1FU, 0x80U, 0x4CU, 0x62U, 0x6DU, 0x18U, 0xF0U, 0xFDU, 0x55U, 0x4CU, 0x92U, 0x45U, 0xDDU,
	0x8AU, 0x0BU, 0x38U, 0x4AU, 0x11U, 0x2EU, 0xE8U, 0x3BU, 0xBBU, 0x66U, 0xDEU, 0x25U, 0xB9U, 0xC3U,
	0xB6U, 0xA9U, 0xCFU, 0xE1U, 0xC1U, 0x43U, 0xD6U, 0x03U, 0x7CU, 0xB1U, 0x05U, 0x39U, 0x54U, 0xF8U,
	0x3AU, 0x4CU, 0xD8U, 0x94U, 0xB9U, 0xA7U, 0xD6U, 0xF1U, 0x99U, 0x0EU, 0x55U, 0x07U, 0x56U, 0x72U,
	0x5CU, 0xCDU, 0x0CU, 0x8FU, 0x00U, 0x51U, 0x4CU, 0xC5U, 0x39U, 0x7BU, 0x6AU, 0x7EU, 0x1DU, 0xA1U,
	0x2BU, 0x5EU, 0x66U, 0x7EU, 0x07U, 0x9BU, 0xCFU, 0x55U, 0xB1U, 0xFFU, 0x95U, 0xFEU, 0x4CU, 0x18U,
	0x2CU, 0xD7U, 0xBCU, 0x47U, 0xDCU, 0xB1U, 0x08U, 0xB9U, 0x6FU, 0x43U, 0x70U, 0xFCU, 0xEAU, 0x4EU,
	0xD3U, 0xB7U, 0x0DU, 0xF8U, 0x66U, 0x03U, 0xBDU, 0xC9U, 0xFCU, 0x36U, 0x42U, 0x32U, 0x66U, 0x37U,
	0x56U, 0x86U, 0x82U, 0x16U, 0x12U, 0x29U, 0x44U, 0x67U, 0x57U, 0x0CU, 0xA0U, 0x3CU, 0xEEU, 0xA3U,
	0x88U, 0x07U, 0x33U, 0xDBU, 0x25U, 0x60U, 0x8BU, 0x3EU, 0x9FU, 0x46U, 0x1FU, 0x72U, 0x49U, 0x4FU,
	0xC7U, 0x74U, 0xD3U, 0x09U, 0x5BU, 0x7FU, 0x3FU, 0x01U, 0x77U, 0x29U, 0xEBU, 0x6DU, 0xEAU, 0xCEU,
	0x62U, 0xE2U, 0xDBU, 0xF7U, 0x42U, 0x05U, 0xF9U, 0xD6U, 0x26U, 0x0CU, 0x32U, 0x9AU, 0x21U, 0x99U,
	0x01U, 0x12U, 0xBEU, 0xC8U, 0xDCU, 0xA5U, 0x3CU, 0x30U, 0x7FU, 0xC5U, 0x30U, 0x38U, 0x2EU, 0x1BU,
	0x14U, 0x76U, 0xF4U, 0x84U, 0x60U, 0x0BU, 0x8BU, 0x5FU, 0x3DU, 0x9EU, 0xCAU, 0x22U, 0x79U, 0x55U,
	0xD6U, 0x7EU, 0x8BU, 0xBEU, 0xB9U, 0xD7U, 0xBEU, 0xF7U, 0x23U, 0xA4U, 0x04U, 0x45U, 0xC3U, 0xCEU,
	0xB3U, 0xC5U, 0x0CU, 0x70U, 0xB8U, 0xAFU, 0xD2U, 0x1FU
};

static const u8 XAsu_RsaPrivateExp[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((
		section (".data.XAsu_RsaPrivateExp"))) = {
	0x07U, 0x31U, 0xDBU, 0xB9U, 0x1AU, 0x78U, 0xA7U, 0xB9U, 0xB6U, 0x42U, 0x1DU, 0x45U, 0xADU, 0x2DU,
	0x23U, 0x8DU, 0x3AU, 0x90U, 0x56U, 0x54U, 0xD5U, 0x2AU, 0x59U, 0xB6U, 0xD9U, 0xBDU, 0xA0U, 0x0AU,
	0x7FU, 0x97U, 0x30U, 0xE5U, 0x60U, 0x0DU, 0x55U, 0x6EU, 0xD5U, 0x24U, 0xEFU, 0x6AU, 0x77U, 0x21U,
	0x85U, 0xACU, 0x5BU, 0xFFU, 0xA1U, 0xC8U, 0x89U, 0x72U, 0x88U, 0xC2U, 0xEFU, 0x97U, 0x31U, 0x1CU,
	0x38U, 0x8BU, 0x7BU, 0x68U, 0x10U, 0x18U, 0x87U, 0x5DU, 0x02U, 0xC6U, 0x9EU, 0xF4U, 0x56U, 0x57U,
	0xC2U, 0x8DU, 0xCFU, 0x37U, 0x5FU, 0xD9U, 0x63U, 0xA4U, 0x43U, 0x1BU, 0x2BU, 0x39U, 0xA8U, 0x43U,
	0x94U, 0x49U, 0xBDU, 0x71U, 0xACU, 0x3AU, 0x6DU, 0x86U, 0x27U, 0xAEU, 0x63U, 0x71U, 0x1CU, 0xF2U,
	0x5AU, 0x43U, 0x43U, 0x9BU, 0x88U, 0xF3U, 0x92U, 0xBFU, 0x9EU, 0xE5U, 0x67U, 0xD5U, 0x24U, 0xDCU,
	0xFFU, 0x43U, 0xEFU, 0x01U, 0x35U, 0x2FU, 0x23U, 0x30U, 0xA2U, 0x38U, 0x5FU, 0xE3U, 0xACU, 0x78U,
	0x22U, 0xE4U, 0xC4U, 0xCEU, 0x8AU, 0x93U, 0x0EU, 0x8DU, 0xD8U, 0x13U, 0x7AU, 0xC0U, 0x18U, 0x02U,
	0x5FU, 0x72U, 0x15U, 0x10U, 0xB8U, 0x74U, 0x5CU, 0xE9U, 0xA3U, 0x1AU, 0x43U, 0xB6U, 0xE3U, 0xF6U,
	0x37U, 0x8EU, 0x3FU, 0xE3U, 0xB0U, 0x45U, 0x6DU, 0xBAU, 0x7FU, 0xB5U, 0x41U, 0x9FU, 0x56U, 0xFAU,
	0xDDU, 0x2AU, 0xE6U, 0xC6U, 0x7CU, 0xC4U, 0x7FU, 0x21U, 0x99U, 0x36U, 0xD9U, 0x23U, 0xAFU, 0x64U,
	0x93U, 0xF5U, 0x98U, 0x50U, 0x02U, 0x16U, 0x6EU, 0x38U, 0xFBU, 0xD7U, 0x45U, 0x5BU, 0xCEU, 0x96U,
	0xA5U, 0x59U, 0xD1U, 0xB6U, 0x37U, 0xC0U, 0xFCU, 0x7BU, 0x55U, 0x1FU, 0x1EU, 0xB6U, 0xD1U, 0x0BU,
	0x27U, 0x20U, 0x16U, 0x28U, 0x02U, 0xA5U, 0x6DU, 0x0EU, 0x72U, 0x13U, 0x21U, 0x8EU, 0x83U, 0xE2U,
	0xF3U, 0xEEU, 0x52U, 0xB9U, 0xB8U, 0x3EU, 0xC0U, 0x2AU, 0xCEU, 0xD4U, 0xABU, 0x29U, 0x74U, 0xC1U,
	0xC5U, 0xD0U, 0x90U, 0x6CU, 0x93U, 0xB2U, 0xC8U, 0x29U, 0x29U, 0x5FU, 0xBEU, 0xC7U, 0x34U, 0x0CU,
	0x96U, 0x88U, 0x21U, 0xC3U, 0x05U, 0x1DU, 0x3EU, 0x31U, 0x3BU, 0xDFU, 0x6FU, 0x57U, 0xF1U, 0xD4U,
	0x9CU, 0x6CU, 0x48U, 0x8FU, 0xCDU, 0x88U, 0xE4U, 0xC0U, 0x8EU, 0xDAU, 0x79U, 0x27U, 0x50U, 0xA8U,
	0x7CU, 0xB9U, 0x38U, 0xA3U, 0x93U, 0x67U, 0x36U, 0x35U, 0x35U, 0xB3U, 0x2AU, 0x4DU, 0x16U, 0x96U,
	0x47U, 0x98U, 0x27U, 0xEFU, 0xDEU, 0x23U, 0xE6U, 0xFBU, 0x4BU, 0xAFU, 0x7AU, 0xB1U, 0x9BU, 0x7CU,
	0x05U, 0xBEU, 0x0EU, 0xCEU, 0x69U, 0x53U, 0xF0U, 0x76U, 0xC1U, 0x89U, 0xCCU, 0xF2U, 0xCAU, 0x2CU,
	0xBBU, 0xA4U, 0xFEU, 0x3CU, 0x3DU, 0xAFU, 0xBCU, 0x25U, 0x4AU, 0x53U, 0x4BU, 0xE9U, 0x85U, 0x55U,
	0xF1U, 0x79U, 0x70U, 0xDFU, 0x57U, 0x23U, 0x55U, 0x7DU, 0x7AU, 0x5CU, 0xA7U, 0xEEU, 0xFBU, 0x00U,
	0x35U, 0x1EU, 0xE3U, 0xF1U, 0xFDU, 0x3EU, 0x94U, 0x70U, 0x88U, 0x91U, 0xFAU, 0xC6U, 0xCBU, 0x05U,
	0xE1U, 0xCDU, 0x7DU, 0xD1U, 0xE3U, 0x2BU, 0x7FU, 0xB4U, 0x64U, 0x1CU, 0x94U, 0x1EU, 0xC2U, 0x55U,
	0x6CU, 0x78U, 0x22U, 0x27U, 0x90U, 0xBAU, 0x33U, 0x0EU, 0xADU, 0x5DU, 0x55U, 0xD9U, 0xDFU, 0x7BU,
	0x83U, 0xA5U, 0x2BU, 0x4FU, 0xA0U, 0x36U, 0x3CU, 0x67U, 0xDDU, 0xC4U, 0x00U, 0x50U, 0x90U, 0x12U,
	0xE1U, 0x8AU, 0x65U, 0x35U, 0x13U, 0x9EU, 0xB2U, 0x0CU, 0xE4U, 0xEAU, 0x51U, 0x2FU, 0xCDU, 0xAEU,
	0x35U, 0x41U, 0x1EU, 0xFAU, 0x5AU, 0x6CU, 0xB6U, 0xC1U, 0xA7U, 0xFEU, 0x69U, 0x6FU, 0x7DU, 0x50U,
	0xFCU, 0xABU, 0x5FU, 0xADU, 0x0CU, 0x2BU, 0x58U, 0xF6U, 0xB3U, 0x33U, 0x1FU, 0xF4U, 0x3AU, 0xDAU,
	0x89U, 0xD3U, 0x0EU, 0x79U, 0x7BU, 0x31U, 0x8EU, 0x15U, 0x84U, 0xECU, 0x2CU, 0x05U, 0x2FU, 0xA9U,
	0x52U, 0xF6U, 0x59U, 0x94U, 0xBFU, 0xC0U, 0x4BU, 0xCDU, 0x40U, 0xFBU, 0x24U, 0x44U, 0x0FU, 0x76U,
	0x11U, 0xE8U, 0xD1U, 0xE3U, 0x13U, 0x26U, 0xEFU, 0xFAU, 0xA9U, 0xE3U, 0xDBU, 0x73U, 0x29U, 0x80U,
	0x4DU, 0x61U, 0x74U, 0x3CU, 0x12U, 0x50U, 0xFDU, 0xDDU, 0xEAU, 0x72U, 0xE4U, 0x88U, 0x8BU, 0x20U,
	0x1AU, 0x3BU, 0x36U, 0xE8U, 0xB8U, 0x98U, 0x55U, 0xCDU
};

static const u32 XAsu_RsaPublicExp
__attribute__ ((section (".data.XAsu_RsaPublicExp"))) = 0x1000100U;

static u8 EncResult[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((section (".data.EncResult")));
static u8 DecResult[XASU_RSA_4096_KEY_SIZE_IN_BYTES] __attribute__ ((section (".data.DecResult")));

static u8 Notify = 0; /**< To notify the call back from client library */
volatile u32 ErrorStatus = XST_FAILURE; /**< Status variable to store the error returned from
						server. */

/************************************ Function Definitions ***************************************/

/*************************************************************************************************/
/**
 * @brief	Main function to call the XAsu_RsaEnc() and XAsu_RsaDec() API's.
 *
 * @return
 *		- XST_SUCCESS if example runs successfully.
 *		- ErrorCode if the example fails.
 *
 *************************************************************************************************/
int main(void)
{
	s32 Status = XST_FAILURE;
	XAsu_RsaParams RsaClientParam;
	XAsu_ClientParams ClientParam = {0U};
	XAsu_RsaPubKeyComp PubKeyParam;
	XAsu_RsaPvtKeyComp PvtKeyParam;
	XMailbox MailboxInstance;

#ifdef	XASU_CACHE_DISABLE
	Xil_DCacheDisable();
#endif

	ClientParam.Priority = XASU_PRIORITY_HIGH;
	ClientParam.CallBackFuncPtr = (XAsuClient_ResponseHandler)((void *)XAsu_RsaCallBackRef);
	ClientParam.CallBackRefPtr = (void *)&ClientParam;

	/** Initialize mailbox instance. */
	Status = (s32)XMailbox_Initialize(&MailboxInstance, XPAR_XIPIPSU_0_BASEADDR);
	if (Status != XST_SUCCESS) {
		xil_printf("Mailbox initialize failed: %08x \r\n", Status);
		goto END;
	}

	/* Initialize the client instance */
	Status = XAsu_ClientInit(&MailboxInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("Client initialize failed:%08x \r\n", Status);
		goto END;
	}

	PubKeyParam.Keysize = XASU_RSA_4096_KEY_SIZE_IN_BYTES;
	PubKeyParam.PubExp = XAsu_RsaPublicExp;

	Status = Xil_SMemCpy(PubKeyParam.Modulus, XASU_RSA_4096_KEY_SIZE_IN_BYTES, XAsu_RsaModulus,
		    XASU_RSA_4096_KEY_SIZE_IN_BYTES, XASU_RSA_4096_KEY_SIZE_IN_BYTES);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	PvtKeyParam.PubKeyComp = PubKeyParam;
	PvtKeyParam.PrimeCompOrTotientPrsnt = 0U;

	Status = Xil_SMemCpy(PvtKeyParam.PvtExp, XASU_RSA_4096_KEY_SIZE_IN_BYTES, XAsu_RsaPrivateExp,
		    XASU_RSA_4096_KEY_SIZE_IN_BYTES, XASU_RSA_4096_KEY_SIZE_IN_BYTES);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	ErrorStatus = XST_FAILURE;
	RsaClientParam.Len = XASU_RSA_4096_KEY_SIZE_IN_BYTES;
	RsaClientParam.KeySize = XASU_RSA_4096_KEY_SIZE_IN_BYTES;
	RsaClientParam.InputDataAddr = (u64)(UINTPTR)XAsu_RsaPt;
	RsaClientParam.OutputDataAddr = (u64)(UINTPTR)EncResult;
	RsaClientParam.KeyCompAddr = (u64)(UINTPTR)&PubKeyParam;
	RsaClientParam.ExpoCompAddr = 0U;
	Status = XAsu_RsaEnc(&ClientParam, &RsaClientParam);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n Encrypt operation Status = %08x", Status);
		goto END;
	}
	while (!Notify);
	Notify = 0;
	if (ErrorStatus != XST_SUCCESS) {
		goto END;
	}

	Status = Xil_SMemCmp_CT(XAsu_RsaCt, XASU_RSA_4096_KEY_SIZE_IN_BYTES, EncResult,
				XASU_RSA_4096_KEY_SIZE_IN_BYTES, XASU_RSA_4096_KEY_SIZE_IN_BYTES);
	if (Status != XST_SUCCESS) {
		xil_printf("ASU RSA Example Failed at Encrypted data Comparison \r\n");
		goto END;
	}

	ClientParam.Priority = XASU_PRIORITY_HIGH;
	ClientParam.CallBackFuncPtr = (XAsuClient_ResponseHandler)((void *)XAsu_RsaCallBackRef);
	ClientParam.CallBackRefPtr = (void *)&ClientParam;

	ErrorStatus = XST_FAILURE;
	RsaClientParam.Len = XASU_RSA_4096_KEY_SIZE_IN_BYTES;
	RsaClientParam.KeySize = XASU_RSA_4096_KEY_SIZE_IN_BYTES;
	RsaClientParam.InputDataAddr = (u64)(UINTPTR)EncResult ;
	RsaClientParam.OutputDataAddr = (u64)(UINTPTR)DecResult;
	RsaClientParam.KeyCompAddr =  (u64)(UINTPTR)&PvtKeyParam;
	RsaClientParam.ExpoCompAddr = 0U;
	Status = XAsu_RsaDec(&ClientParam, &RsaClientParam);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n Decrypt operation Status = %08x", Status);
		goto END;
	}
	while (!Notify);
	Notify = 0;
	if (ErrorStatus != XST_SUCCESS) {
		goto END;
	}

	Status = Xil_SMemCmp_CT(XAsu_RsaPt, XASU_RSA_4096_KEY_SIZE_IN_BYTES, DecResult,
				XASU_RSA_4096_KEY_SIZE_IN_BYTES, XASU_RSA_4096_KEY_SIZE_IN_BYTES);
	if (Status != XST_SUCCESS) {
		xil_printf("ASU RSA Example Failed at Decrypted data Comparison \r\n");
	}

END:
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n RSA client example failed with Status = %08x", Status);
	} else if (ErrorStatus != XST_SUCCESS) {
		xil_printf("\r\n RSA client example failed with error from server = %08x", ErrorStatus);
	} else {
		xil_printf("\r\n Successfully ran RSA client example ");
	}

	return Status;
}

/*************************************************************************************************/
/**
 * @brief	Call back function which will be registered with library to notify the completion
 * 			of request
 *
 * @param	CallBackRef	Pointer to the call back reference.
 * @param	Status		Status of the request will be passed as an argument during callback
 * 			- 0 Upon success
 * 			- Error code from ASUFW application upon any error
 *
 *************************************************************************************************/
  static void XAsu_RsaCallBackRef(void *CallBackRef, u32 Status)
 {
	(void)CallBackRef;

	ErrorStatus = Status;
	/* Update the variable to notify the callback */
	Notify = 1U;

 }