/******************************************************************************
* Copyright (C) 2022 Xilinx, Inc.  All rights reserved.
* Copyright (C) 2022 - 2024 Advanced Micro Devices, Inc. All Rights Reserved
* SPDX-License-Identifier: MIT
******************************************************************************/

/*****************************************************************************/
/**
*
* @file xi3cpsx_hw.h
* @addtogroup Overview
* @{
*
* This file contains low level access functions using the base address
* directly without an instance.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who Date     Changes
* ----- --- -------- -----------------------------------------------.
* 1.00  sd  06/10/22 First release
* 1.4   gm  10/07/24 Added macros for reading write and read fifo leves,
* 		     enable and disable interrupts, and register offset.
*
* </pre>
*
******************************************************************************/
#ifndef XI3CPSX_HW_H_		/**< prevent circular inclusions */
#define XI3CPSX_HW_H_		/**< by using protection macros */

#ifdef __cplusplus
extern "C" {
#endif

/***************************** Include Files *********************************/

#include "xil_types.h"
#include "xil_assert.h"
#include "xil_io.h"
#include "xparameters.h"

/************************** Constant Definitions *****************************/

/**
 * PS_I2C_I3C0 Base Address
#define XI3CPSX_BASEADDR      0xF1000000
#define XI3CPSX_BASEADDR      0xF1940000
#define XI3CPSX_BASEADDR      0xF1950000
 */
#ifdef SDT
#define XI3CPSX_BASEADDR      0x0
#else
#define XI3CPSX_BASEADDR      0x8000
#endif
#define XI3cPsx_In32  Xil_In32
#define XI3cPsx_Out32 Xil_Out32

/****************************************************************************/
/**
* Read an I3C register.
*
* @param	BaseAddress contains the base address of the device.
* @param	RegOffset contains the offset from the 1st register of the
*		device to select the specific register.
*
* @return	The value read from the register.
*
* @note		C-Style signature:
*		u32 XI3Psxl_ReadReg(u32 BaseAddress. int RegOffset)
*
******************************************************************************/
#define XI3cPsx_ReadReg(BaseAddress, RegOffset) \
	XI3cPsx_In32((BaseAddress) + (u32)(RegOffset))

/***************************************************************************/
/**
* Write an I3C register.
*
* @param	BaseAddress contains the base address of the device.
* @param	RegOffset contains the offset from the 1st register of the
*		device to select the specific register.
* @param	RegisterValue is the value to be written to the register.
*
* @return	None.
*
* @note	C-Style signature:
*	void XI3cPsx_WriteReg(u32 BaseAddress, int RegOffset, u32 RegisterValue)
*
******************************************************************************/
#define XI3cPsx_WriteReg(BaseAddress, RegOffset, RegisterValue) \
	XI3cPsx_Out32((BaseAddress) + (u32)(RegOffset), (u32)(RegisterValue))

/****************************************************************************/
/**
*
* Read Tx FIFO level.
*
* @param        InstancePtr is the instance of I3cPs
*
* @return       None.
*
* @note         C-Style signature:
*               void XI3cPsx_WrFifoLevel(XI3cPsx *InstancePtr)
*
*****************************************************************************/
#define XI3cPsx_WrFifoLevel(InstancePtr)				     \
	(u8)(XI3cPsx_ReadReg(InstancePtr->Config.BaseAddress,		     \
			     XI3CPSX_DATA_BUFFER_STATUS_LEVEL)		     \
	     & XI3CPSX_DATA_BUFFER_STATUS_LEVEL_TX_BUF_EMPTY_LOC_MASK)

/****************************************************************************/
/**
*
* Read Rx FIFO level.
*
* @param        InstancePtr is the instance of I3cPs
*
* @return       None.
*
* @note         C-Style signature:
*               void XI3cPsx_RdFifoLevel(XI3cPsx *InstancePtr)
*
*****************************************************************************/
#define XI3cPsx_RdFifoLevel(InstancePtr)				     \
	(u8)((XI3cPsx_ReadReg(InstancePtr->Config.BaseAddress,		     \
			      XI3CPSX_DATA_BUFFER_STATUS_LEVEL)	     	     \
	     & XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_MASK)	     \
	     >> XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_SHIFT)

/*****************************************************************************/
/**
*
* Enable interrupts
*
* @param        Base address of the XI3cPsx core instance.
* @param        interrupt mask value.
*
* @return       None.
*
* @note         C-style signature:
*               u16 XI3cPsx_EnableInterrupts(XI3cPsx *InstancePtr, u32 IntrMask)
*
******************************************************************************/
#define XI3cPsx_EnableInterrupts(BaseAddress, IntrMask) 		          \
	XI3cPsx_WriteReg((BaseAddress), XI3CPSX_INTR_STATUS_EN,		          \
			 ((XI3cPsx_ReadReg(BaseAddress, XI3CPSX_INTR_STATUS_EN))| \
			 (IntrMask)));					          \
	XI3cPsx_WriteReg((BaseAddress), XI3CPSX_INTR_SIGNAL_EN,		          \
			 ((XI3cPsx_ReadReg(BaseAddress, XI3CPSX_INTR_SIGNAL_EN))| \
			 (IntrMask)))

/*****************************************************************************/
/**
*
* Disable interrupts
*
* @param        Base address of the XI3cPsx core instance.
* @param        interrupt mask value.
*
* @return       None.
*
* @note         C-style signature:
*               u16 XI3cPsx_DisableInterrupts(XI3cPsx *InstancePtr, u32 IntrMask)
*
******************************************************************************/
#define XI3cPsx_DisableInterrupts(BaseAddress, IntrMask)			\
	XI3cPsx_WriteReg((BaseAddress), XI3CPSX_INTR_STATUS_EN,			\
			 ((XI3cPsx_ReadReg(BaseAddress, XI3CPSX_INTR_STATUS_EN))\
			 & ~(IntrMask)));					\
	XI3cPsx_WriteReg((BaseAddress), XI3CPSX_INTR_SIGNAL_EN,			\
			 ((XI3cPsx_ReadReg(BaseAddress, XI3CPSX_INTR_SIGNAL_EN))\
			 & ~(IntrMask)))


#define XI3CPSX_NS_1SEC			1000000000

/**
 * Register: XI3CPSX_DEVICE_CTRL
 */
#define XI3CPSX_DEVICE_CTRL     XI3CPSX_BASEADDR
#define XI3CPSX_DEVICE_CTRL_FULLMASK     0xeb000181
#define XI3CPSX_DEVICE_CTRL_FULLRWMASK  0xeb000181
#define XI3CPSX_DEVICE_CTRL_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_ENABLE_SHIFT   31
#define XI3CPSX_DEVICE_CTRL_ENABLE_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_ENABLE_MASK    0x80000000
#define XI3CPSX_DEVICE_CTRL_ENABLE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_RESUME_SHIFT   30
#define XI3CPSX_DEVICE_CTRL_RESUME_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_RESUME_MASK    0x40000000
#define XI3CPSX_DEVICE_CTRL_RESUME_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_ABORT_SHIFT   29
#define XI3CPSX_DEVICE_CTRL_ABORT_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_ABORT_MASK    0x20000000
#define XI3CPSX_DEVICE_CTRL_ABORT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_ADAPTIVE_I2C_I3C_SHIFT   27
#define XI3CPSX_DEVICE_CTRL_ADAPTIVE_I2C_I3C_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_ADAPTIVE_I2C_I3C_MASK    0x08000000
#define XI3CPSX_DEVICE_CTRL_ADAPTIVE_I2C_I3C_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_IDLE_CNT_MULTPLIER_SHIFT   24
#define XI3CPSX_DEVICE_CTRL_IDLE_CNT_MULTPLIER_WIDTH   2
#define XI3CPSX_DEVICE_CTRL_IDLE_CNT_MULTPLIER_MASK    0x03000000
#define XI3CPSX_DEVICE_CTRL_IDLE_CNT_MULTPLIER_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_HOT_JOIN_CTRL_SHIFT   8
#define XI3CPSX_DEVICE_CTRL_HOT_JOIN_CTRL_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_HOT_JOIN_CTRL_MASK    0x00000100
#define XI3CPSX_DEVICE_CTRL_HOT_JOIN_CTRL_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_I2C_SLAVE_PRESENT_SHIFT   7
#define XI3CPSX_DEVICE_CTRL_I2C_SLAVE_PRESENT_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_I2C_SLAVE_PRESENT_MASK    0x00000080
#define XI3CPSX_DEVICE_CTRL_I2C_SLAVE_PRESENT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_IBA_INCLUDE_SHIFT   0
#define XI3CPSX_DEVICE_CTRL_IBA_INCLUDE_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_IBA_INCLUDE_MASK    0x00000001
#define XI3CPSX_DEVICE_CTRL_IBA_INCLUDE_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEVICE_ADDR
 */
#define XI3CPSX_DEVICE_ADDR    ( ( XI3CPSX_BASEADDR ) + 0x00000004 )
#define XI3CPSX_DEVICE_ADDR_FULLMASK     0x807f807f
#define XI3CPSX_DEVICE_ADDR_FULLRWMASK  0x807f807f
#define XI3CPSX_DEVICE_ADDR_DEFVAL   0x0

#define XI3CPSX_DEVICE_ADDR_SELF_DYNAMIC_ADDR   0x45

/* access_type: rw */
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_VALID_SHIFT   31
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_VALID_WIDTH   1
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_VALID_MASK    0x80000000
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_VALID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_WIDTH   7
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_MASK    0x007f0000
#define XI3CPSX_DEVICE_ADDR_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_VALID_SHIFT   15
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_VALID_WIDTH   1
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_VALID_MASK    0x00008000
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_VALID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEVICE_ADDR_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_HW_CAPABILITY
 */
#define XI3CPSX_HW_CAPABILITY    ( ( XI3CPSX_BASEADDR ) + 0x00000008 )
#define XI3CPSX_HW_CAPABILITY_FULLMASK     0x000fffff
#define XI3CPSX_HW_CAPABILITY_FULLRWMASK  0x00000000
#define XI3CPSX_HW_CAPABILITY_DEFVAL   0xd40ab

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_SLV_IBI_CAP_SHIFT   19
#define XI3CPSX_HW_CAPABILITY_SLV_IBI_CAP_WIDTH   1
#define XI3CPSX_HW_CAPABILITY_SLV_IBI_CAP_MASK    0x00080000
#define XI3CPSX_HW_CAPABILITY_SLV_IBI_CAP_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_SLV_HJ_CAP_SHIFT   18
#define XI3CPSX_HW_CAPABILITY_SLV_HJ_CAP_WIDTH   1
#define XI3CPSX_HW_CAPABILITY_SLV_HJ_CAP_MASK    0x00040000
#define XI3CPSX_HW_CAPABILITY_SLV_HJ_CAP_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_DMA_EN_SHIFT   17
#define XI3CPSX_HW_CAPABILITY_DMA_EN_WIDTH   1
#define XI3CPSX_HW_CAPABILITY_DMA_EN_MASK    0x00020000
#define XI3CPSX_HW_CAPABILITY_DMA_EN_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_HDR_TX_CLOCK_PERIOD_SHIFT   11
#define XI3CPSX_HW_CAPABILITY_HDR_TX_CLOCK_PERIOD_WIDTH   6
#define XI3CPSX_HW_CAPABILITY_HDR_TX_CLOCK_PERIOD_MASK    0x0001f800
#define XI3CPSX_HW_CAPABILITY_HDR_TX_CLOCK_PERIOD_DEFVAL  0x28

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_CLOCK_PERIOD_SHIFT   5
#define XI3CPSX_HW_CAPABILITY_CLOCK_PERIOD_WIDTH   6
#define XI3CPSX_HW_CAPABILITY_CLOCK_PERIOD_MASK    0x000007e0
#define XI3CPSX_HW_CAPABILITY_CLOCK_PERIOD_DEFVAL  0x5

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_HDR_TS_EN_SHIFT   4
#define XI3CPSX_HW_CAPABILITY_HDR_TS_EN_WIDTH   1
#define XI3CPSX_HW_CAPABILITY_HDR_TS_EN_MASK    0x00000010
#define XI3CPSX_HW_CAPABILITY_HDR_TS_EN_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_HDR_DDR_EN_SHIFT   3
#define XI3CPSX_HW_CAPABILITY_HDR_DDR_EN_WIDTH   1
#define XI3CPSX_HW_CAPABILITY_HDR_DDR_EN_MASK    0x00000008
#define XI3CPSX_HW_CAPABILITY_HDR_DDR_EN_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_HW_CAPABILITY_DEVICE_ROLE_CONFIG_SHIFT   0
#define XI3CPSX_HW_CAPABILITY_DEVICE_ROLE_CONFIG_WIDTH   3
#define XI3CPSX_HW_CAPABILITY_DEVICE_ROLE_CONFIG_MASK    0x00000007
#define XI3CPSX_HW_CAPABILITY_DEVICE_ROLE_CONFIG_DEFVAL  0x3

/**
 * Register: XI3CPSX_COMMAND_QUEUE_PORT
 */
#define XI3CPSX_COMMAND_QUEUE_PORT    ( ( XI3CPSX_BASEADDR ) + 0x0000000C )
#define XI3CPSX_COMMAND_QUEUE_PORT_FULLMASK     0xffffffff
#define XI3CPSX_COMMAND_QUEUE_PORT_FULLRWMASK  0x00000000
#define XI3CPSX_COMMAND_QUEUE_PORT_DEFVAL   0x0

/* access_type: wo */
#define XI3CPSX_COMMAND_QUEUE_PORT_COMMAND_SHIFT   0
#define XI3CPSX_COMMAND_QUEUE_PORT_COMMAND_WIDTH   32
#define XI3CPSX_COMMAND_QUEUE_PORT_COMMAND_MASK    0xffffffff
#define XI3CPSX_COMMAND_QUEUE_PORT_COMMAND_DEFVAL  0x0

/**
 * Register: XI3CPSX_RESPONSE_QUEUE_PORT
 */
#define XI3CPSX_RESPONSE_QUEUE_PORT    ( ( XI3CPSX_BASEADDR ) + 0x00000010 )
#define XI3CPSX_RESPONSE_QUEUE_PORT_FULLMASK     0xffffffff
#define XI3CPSX_RESPONSE_QUEUE_PORT_FULLRWMASK  0x00000000
#define XI3CPSX_RESPONSE_QUEUE_PORT_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_RESPONSE_QUEUE_PORT_RESPONSE_SHIFT   0
#define XI3CPSX_RESPONSE_QUEUE_PORT_RESPONSE_WIDTH   32
#define XI3CPSX_RESPONSE_QUEUE_PORT_RESPONSE_MASK    0xffffffff
#define XI3CPSX_RESPONSE_QUEUE_PORT_RESPONSE_DEFVAL  0x0

#define XI3CPSX_RESPONSE_ERR_STS_SHIFT			28
#define XI3CPSX_RESPONSE_ERR_STS_MASK			0xf0000000

/**
 * Register: XI3CPSX_TX_RX_DATA_PORT
 */
#define XI3CPSX_TX_RX_DATA_PORT    ( ( XI3CPSX_BASEADDR ) + 0x00000014 )
#define XI3CPSX_TX_RX_DATA_PORT_FULLMASK     0xffffffff
#define XI3CPSX_TX_RX_DATA_PORT_FULLRWMASK  0xffffffff
#define XI3CPSX_TX_RX_DATA_PORT_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_TX_RX_DATA_PORT_TX_RX_DATA_PORT_SHIFT   0
#define XI3CPSX_TX_RX_DATA_PORT_TX_RX_DATA_PORT_WIDTH   32
#define XI3CPSX_TX_RX_DATA_PORT_TX_RX_DATA_PORT_MASK    0xffffffff
#define XI3CPSX_TX_RX_DATA_PORT_TX_RX_DATA_PORT_DEFVAL  0x0

/**
 * Register: XI3CPSX_IBI_QUEUE_STATUS
 */
#define XI3CPSX_IBI_QUEUE_STATUS    ( ( XI3CPSX_BASEADDR ) + 0x00000018 )
#define XI3CPSX_IBI_QUEUE_STATUS_FULLMASK     0xf000ffff
#define XI3CPSX_IBI_QUEUE_STATUS_FULLRWMASK  0x00000000
#define XI3CPSX_IBI_QUEUE_STATUS_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_STS_SHIFT   28
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_STS_WIDTH   4
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_STS_MASK    0xf0000000
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_ID_SHIFT   8
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_ID_WIDTH   8
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_ID_MASK    0x0000ff00
#define XI3CPSX_IBI_QUEUE_STATUS_IBI_ID_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_IBI_QUEUE_STATUS_DATA_LENGTH_SHIFT   0
#define XI3CPSX_IBI_QUEUE_STATUS_DATA_LENGTH_WIDTH   8
#define XI3CPSX_IBI_QUEUE_STATUS_DATA_LENGTH_MASK    0x000000ff
#define XI3CPSX_IBI_QUEUE_STATUS_DATA_LENGTH_DEFVAL  0x0

/**
 * Register: XI3CPSX_QUEUE_THLD_CTRL
 */
#define XI3CPSX_QUEUE_THLD_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x0000001C )
#define XI3CPSX_QUEUE_THLD_CTRL_FULLMASK     0xff00ffff
#define XI3CPSX_QUEUE_THLD_CTRL_FULLRWMASK  0xff00ffff
#define XI3CPSX_QUEUE_THLD_CTRL_DEFVAL   0x1000101

/* access_type: rw */
#define XI3CPSX_QUEUE_THLD_CTRL_IBI_STATUS_THLD_SHIFT   24
#define XI3CPSX_QUEUE_THLD_CTRL_IBI_STATUS_THLD_WIDTH   8
#define XI3CPSX_QUEUE_THLD_CTRL_IBI_STATUS_THLD_MASK    0xff000000
#define XI3CPSX_QUEUE_THLD_CTRL_IBI_STATUS_THLD_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_QUEUE_THLD_CTRL_RESP_BUF_THLD_SHIFT   8
#define XI3CPSX_QUEUE_THLD_CTRL_RESP_BUF_THLD_WIDTH   8
#define XI3CPSX_QUEUE_THLD_CTRL_RESP_BUF_THLD_MASK    0x0000ff00
#define XI3CPSX_QUEUE_THLD_CTRL_RESP_BUF_THLD_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_QUEUE_THLD_CTRL_CMD_EMPTY_BUF_THLD_SHIFT   0
#define XI3CPSX_QUEUE_THLD_CTRL_CMD_EMPTY_BUF_THLD_WIDTH   8
#define XI3CPSX_QUEUE_THLD_CTRL_CMD_EMPTY_BUF_THLD_MASK    0x000000ff
#define XI3CPSX_QUEUE_THLD_CTRL_CMD_EMPTY_BUF_THLD_DEFVAL  0x1

/**
 * Register: XI3CPSX_DATA_BUFFER_THLD_CTRL
 */
#define XI3CPSX_DATA_BUFFER_THLD_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x00000020 )
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_FULLMASK     0x07070707
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_FULLRWMASK  0x07070707
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_DEFVAL   0x1010101

/* access_type: rw */
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_START_THLD_SHIFT   24
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_START_THLD_WIDTH   3
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_START_THLD_MASK    0x07000000
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_START_THLD_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_START_THLD_SHIFT   16
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_START_THLD_WIDTH   3
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_START_THLD_MASK    0x00070000
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_START_THLD_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_BUF_THLD_SHIFT   8
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_BUF_THLD_WIDTH   3
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_BUF_THLD_MASK    0x00000700
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_RX_BUF_THLD_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_EMPTY_BUF_THLD_SHIFT   0
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_EMPTY_BUF_THLD_WIDTH   3
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_EMPTY_BUF_THLD_MASK    0x00000007
#define XI3CPSX_DATA_BUFFER_THLD_CTRL_TX_EMPTY_BUF_THLD_DEFVAL  0x1

/**
 * Register: XI3CPSX_IBI_QUEUE_CTRL
 */
#define XI3CPSX_IBI_QUEUE_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x00000024 )
#define XI3CPSX_IBI_QUEUE_CTRL_FULLMASK     0x0000000b
#define XI3CPSX_IBI_QUEUE_CTRL_FULLRWMASK  0x0000000b
#define XI3CPSX_IBI_QUEUE_CTRL_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_SIR_REJECTED_SHIFT   3
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_SIR_REJECTED_WIDTH   1
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_SIR_REJECTED_MASK    0x00000008
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_SIR_REJECTED_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_MR_REJECTED_SHIFT   1
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_MR_REJECTED_WIDTH   1
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_MR_REJECTED_MASK    0x00000002
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_MR_REJECTED_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_HJ_REJECTED_SHIFT   0
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_HJ_REJECTED_WIDTH   1
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_HJ_REJECTED_MASK    0x00000001
#define XI3CPSX_IBI_QUEUE_CTRL_NOTIFY_HJ_REJECTED_DEFVAL  0x0

/**
 * Register: XI3CPSX_IBI_MR_REQ_REJECT
 */
#define XI3CPSX_IBI_MR_REQ_REJECT    ( ( XI3CPSX_BASEADDR ) + 0x0000002C )
#define XI3CPSX_IBI_MR_REQ_REJECT_FULLMASK     0xffffffff
#define XI3CPSX_IBI_MR_REQ_REJECT_FULLRWMASK  0xffffffff
#define XI3CPSX_IBI_MR_REQ_REJECT_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_IBI_MR_REQ_REJECT_MR_REQ_REJECT_SHIFT   0
#define XI3CPSX_IBI_MR_REQ_REJECT_MR_REQ_REJECT_WIDTH   32
#define XI3CPSX_IBI_MR_REQ_REJECT_MR_REQ_REJECT_MASK    0xffffffff
#define XI3CPSX_IBI_MR_REQ_REJECT_MR_REQ_REJECT_DEFVAL  0x0

/**
 * Register: XI3CPSX_IBI_SIR_REQ_REJECT
 */
#define XI3CPSX_IBI_SIR_REQ_REJECT    ( ( XI3CPSX_BASEADDR ) + 0x00000030 )
#define XI3CPSX_IBI_SIR_REQ_REJECT_FULLMASK     0xffffffff
#define XI3CPSX_IBI_SIR_REQ_REJECT_FULLRWMASK  0xffffffff
#define XI3CPSX_IBI_SIR_REQ_REJECT_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_IBI_SIR_REQ_REJECT_SIR_REQ_REJECT_SHIFT   0
#define XI3CPSX_IBI_SIR_REQ_REJECT_SIR_REQ_REJECT_WIDTH   32
#define XI3CPSX_IBI_SIR_REQ_REJECT_SIR_REQ_REJECT_MASK    0xffffffff
#define XI3CPSX_IBI_SIR_REQ_REJECT_SIR_REQ_REJECT_DEFVAL  0x0

/**
 * Register: XI3CPSX_RESET_CTRL
 */
#define XI3CPSX_RESET_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x00000034 )
#define XI3CPSX_RESET_CTRL_FULLMASK     0xe000003f
#define XI3CPSX_RESET_CTRL_FULLRWMASK  0xe000003f
#define XI3CPSX_RESET_CTRL_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_BUS_RESET_SHIFT   31
#define XI3CPSX_RESET_CTRL_BUS_RESET_WIDTH   1
#define XI3CPSX_RESET_CTRL_BUS_RESET_MASK    0x80000000
#define XI3CPSX_RESET_CTRL_BUS_RESET_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_BUS_RESET_TYPE_SHIFT   29
#define XI3CPSX_RESET_CTRL_BUS_RESET_TYPE_WIDTH   2
#define XI3CPSX_RESET_CTRL_BUS_RESET_TYPE_MASK    0x60000000
#define XI3CPSX_RESET_CTRL_BUS_RESET_TYPE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_BI_QUEUE_RST_SHIFT   5
#define XI3CPSX_RESET_CTRL_BI_QUEUE_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_BI_QUEUE_RST_MASK    0x00000020
#define XI3CPSX_RESET_CTRL_BI_QUEUE_RST_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_RX_FIFO_RST_SHIFT   4
#define XI3CPSX_RESET_CTRL_RX_FIFO_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_RX_FIFO_RST_MASK    0x00000010
#define XI3CPSX_RESET_CTRL_RX_FIFO_RST_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_TX_FIFO_RST_SHIFT   3
#define XI3CPSX_RESET_CTRL_TX_FIFO_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_TX_FIFO_RST_MASK    0x00000008
#define XI3CPSX_RESET_CTRL_TX_FIFO_RST_DEFVAL  0x0

#define XI3CPSX_RESET_CTRL_ALL_FIFOS_RST_VAL   0x1E

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_RESP_QUEUE_RST_SHIFT   2
#define XI3CPSX_RESET_CTRL_RESP_QUEUE_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_RESP_QUEUE_RST_MASK    0x00000004
#define XI3CPSX_RESET_CTRL_RESP_QUEUE_RST_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_CMD_QUEUE_RST_SHIFT   1
#define XI3CPSX_RESET_CTRL_CMD_QUEUE_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_CMD_QUEUE_RST_MASK    0x00000002
#define XI3CPSX_RESET_CTRL_CMD_QUEUE_RST_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_RESET_CTRL_SOFT_RST_SHIFT   0
#define XI3CPSX_RESET_CTRL_SOFT_RST_WIDTH   1
#define XI3CPSX_RESET_CTRL_SOFT_RST_MASK    0x00000001
#define XI3CPSX_RESET_CTRL_SOFT_RST_DEFVAL  0x0
#define XI3CPSX_RESET_CTRL_SOFT_RST_VAL     0x1

/**
 * Register: XI3CPSX_SLV_EVENT_STATUS
 */
#define XI3CPSX_SLV_EVENT_STATUS    ( ( XI3CPSX_BASEADDR ) + 0x00000038 )
#define XI3CPSX_SLV_EVENT_STATUS_FULLMASK     0x000000fb
#define XI3CPSX_SLV_EVENT_STATUS_FULLRWMASK  0x00000008
#define XI3CPSX_SLV_EVENT_STATUS_DEFVAL   0xb

/* access_type: wtc */
#define XI3CPSX_SLV_EVENT_STATUS_MWL_UPDATED_SHIFT   7
#define XI3CPSX_SLV_EVENT_STATUS_MWL_UPDATED_WIDTH   1
#define XI3CPSX_SLV_EVENT_STATUS_MWL_UPDATED_MASK    0x00000080
#define XI3CPSX_SLV_EVENT_STATUS_MWL_UPDATED_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_SLV_EVENT_STATUS_MRL_UPDATED_SHIFT   6
#define XI3CPSX_SLV_EVENT_STATUS_MRL_UPDATED_WIDTH   1
#define XI3CPSX_SLV_EVENT_STATUS_MRL_UPDATED_MASK    0x00000040
#define XI3CPSX_SLV_EVENT_STATUS_MRL_UPDATED_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_SLV_EVENT_STATUS_ACTIVITY_STATE_SHIFT   4
#define XI3CPSX_SLV_EVENT_STATUS_ACTIVITY_STATE_WIDTH   2
#define XI3CPSX_SLV_EVENT_STATUS_ACTIVITY_STATE_MASK    0x00000030
#define XI3CPSX_SLV_EVENT_STATUS_ACTIVITY_STATE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_EVENT_STATUS_HJ_EN_SHIFT   3
#define XI3CPSX_SLV_EVENT_STATUS_HJ_EN_WIDTH   1
#define XI3CPSX_SLV_EVENT_STATUS_HJ_EN_MASK    0x00000008
#define XI3CPSX_SLV_EVENT_STATUS_HJ_EN_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_SLV_EVENT_STATUS_MR_EN_SHIFT   1
#define XI3CPSX_SLV_EVENT_STATUS_MR_EN_WIDTH   1
#define XI3CPSX_SLV_EVENT_STATUS_MR_EN_MASK    0x00000002
#define XI3CPSX_SLV_EVENT_STATUS_MR_EN_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_SLV_EVENT_STATUS_SIR_EN_SHIFT   0
#define XI3CPSX_SLV_EVENT_STATUS_SIR_EN_WIDTH   1
#define XI3CPSX_SLV_EVENT_STATUS_SIR_EN_MASK    0x00000001
#define XI3CPSX_SLV_EVENT_STATUS_SIR_EN_DEFVAL  0x1

/**
 * Register: XI3CPSX_INTR_STATUS
 */
#define XI3CPSX_INTR_STATUS    ( ( XI3CPSX_BASEADDR ) + 0x0000003C )
#define XI3CPSX_INTR_STATUS_FULLMASK     0x0000bf7f
#define XI3CPSX_INTR_STATUS_FULLRWMASK  0x00000000
#define XI3CPSX_INTR_STATUS_DEFVAL   0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_BUS_RESET_DONE_STS_SHIFT   15
#define XI3CPSX_INTR_STATUS_BUS_RESET_DONE_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_BUS_RESET_DONE_STS_MASK    0x00008000
#define XI3CPSX_INTR_STATUS_BUS_RESET_DONE_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_BUSOWNER_UPDATED_STS_SHIFT   13
#define XI3CPSX_INTR_STATUS_BUSOWNER_UPDATED_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_BUSOWNER_UPDATED_STS_MASK    0x00002000
#define XI3CPSX_INTR_STATUS_BUSOWNER_UPDATED_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_IBI_UPDATED_STS_SHIFT   12
#define XI3CPSX_INTR_STATUS_IBI_UPDATED_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_IBI_UPDATED_STS_MASK    0x00001000
#define XI3CPSX_INTR_STATUS_IBI_UPDATED_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_READ_REQ_RECV_STS_SHIFT   11
#define XI3CPSX_INTR_STATUS_READ_REQ_RECV_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_READ_REQ_RECV_STS_MASK    0x00000800
#define XI3CPSX_INTR_STATUS_READ_REQ_RECV_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_DEFSLV_STS_SHIFT   10
#define XI3CPSX_INTR_STATUS_DEFSLV_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_DEFSLV_STS_MASK    0x00000400
#define XI3CPSX_INTR_STATUS_DEFSLV_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_TRANSFER_ERR_STS_SHIFT   9
#define XI3CPSX_INTR_STATUS_TRANSFER_ERR_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_TRANSFER_ERR_STS_MASK    0x00000200
#define XI3CPSX_INTR_STATUS_TRANSFER_ERR_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_DYN_ADDR_ASSGN_STS_SHIFT   8
#define XI3CPSX_INTR_STATUS_DYN_ADDR_ASSGN_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_DYN_ADDR_ASSGN_STS_MASK    0x00000100
#define XI3CPSX_INTR_STATUS_DYN_ADDR_ASSGN_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_CCC_UPDATED_STS_SHIFT   6
#define XI3CPSX_INTR_STATUS_CCC_UPDATED_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_CCC_UPDATED_STS_MASK    0x00000040
#define XI3CPSX_INTR_STATUS_CCC_UPDATED_STS_DEFVAL  0x0

/* access_type: wtc */
#define XI3CPSX_INTR_STATUS_TRANSFER_ABORT_STS_SHIFT   5
#define XI3CPSX_INTR_STATUS_TRANSFER_ABORT_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_TRANSFER_ABORT_STS_MASK    0x00000020
#define XI3CPSX_INTR_STATUS_TRANSFER_ABORT_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_INTR_STATUS_RESP_READY_STS_SHIFT   4
#define XI3CPSX_INTR_STATUS_RESP_READY_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_RESP_READY_STS_MASK    0x00000010
#define XI3CPSX_INTR_STATUS_RESP_READY_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_INTR_STATUS_CMD_QUEUE_READY_STS_SHIFT   3
#define XI3CPSX_INTR_STATUS_CMD_QUEUE_READY_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_CMD_QUEUE_READY_STS_MASK    0x00000008
#define XI3CPSX_INTR_STATUS_CMD_QUEUE_READY_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_INTR_STATUS_IBI_THLD_STS_SHIFT   2
#define XI3CPSX_INTR_STATUS_IBI_THLD_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_IBI_THLD_STS_MASK    0x00000004
#define XI3CPSX_INTR_STATUS_IBI_THLD_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_INTR_STATUS_RX_THLD_STS_SHIFT   1
#define XI3CPSX_INTR_STATUS_RX_THLD_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_RX_THLD_STS_MASK    0x00000002
#define XI3CPSX_INTR_STATUS_RX_THLD_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_INTR_STATUS_TX_THLD_STS_SHIFT   0
#define XI3CPSX_INTR_STATUS_TX_THLD_STS_WIDTH   1
#define XI3CPSX_INTR_STATUS_TX_THLD_STS_MASK    0x00000001
#define XI3CPSX_INTR_STATUS_TX_THLD_STS_DEFVAL  0x0

/**
 * Register: XI3CPSX_INTR_STATUS_EN
 */
#define XI3CPSX_INTR_STATUS_EN    ( ( XI3CPSX_BASEADDR ) + 0x00000040 )
#define XI3CPSX_INTR_STATUS_EN_FULLMASK     0x0000bf7f
#define XI3CPSX_INTR_STATUS_EN_FULLRWMASK  0x0000bf7f
#define XI3CPSX_INTR_STATUS_EN_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_BUS_RESET_DONE_STS_EN_SHIFT   15
#define XI3CPSX_INTR_STATUS_EN_BUS_RESET_DONE_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_BUS_RESET_DONE_STS_EN_MASK    0x00008000
#define XI3CPSX_INTR_STATUS_EN_BUS_RESET_DONE_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_BUSOWNER_UPDATED_STS_EN_SHIFT   13
#define XI3CPSX_INTR_STATUS_EN_BUSOWNER_UPDATED_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_BUSOWNER_UPDATED_STS_EN_MASK    0x00002000
#define XI3CPSX_INTR_STATUS_EN_BUSOWNER_UPDATED_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_IBI_UPDATED_STS_EN_SHIFT   12
#define XI3CPSX_INTR_STATUS_EN_IBI_UPDATED_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_IBI_UPDATED_STS_EN_MASK    0x00001000
#define XI3CPSX_INTR_STATUS_EN_IBI_UPDATED_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_READ_REQ_RECV_STS_EN_SHIFT   11
#define XI3CPSX_INTR_STATUS_EN_READ_REQ_RECV_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_READ_REQ_RECV_STS_EN_MASK    0x00000800
#define XI3CPSX_INTR_STATUS_EN_READ_REQ_RECV_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_DEFSLV_STS_EN_SHIFT   10
#define XI3CPSX_INTR_STATUS_EN_DEFSLV_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_DEFSLV_STS_EN_MASK    0x00000400
#define XI3CPSX_INTR_STATUS_EN_DEFSLV_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ERR_STS_EN_SHIFT   9
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ERR_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ERR_STS_EN_MASK    0x00000200
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ERR_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_DYN_ADDR_ASSGN_STS_EN_SHIFT   8
#define XI3CPSX_INTR_STATUS_EN_DYN_ADDR_ASSGN_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_DYN_ADDR_ASSGN_STS_EN_MASK    0x00000100
#define XI3CPSX_INTR_STATUS_EN_DYN_ADDR_ASSGN_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_CCC_UPDATED_STS_EN_SHIFT   6
#define XI3CPSX_INTR_STATUS_EN_CCC_UPDATED_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_CCC_UPDATED_STS_EN_MASK    0x00000040
#define XI3CPSX_INTR_STATUS_EN_CCC_UPDATED_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ABORT_STS_EN_SHIFT   5
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ABORT_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ABORT_STS_EN_MASK    0x00000020
#define XI3CPSX_INTR_STATUS_EN_TRANSFER_ABORT_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_RESP_READY_STS_EN_SHIFT   4
#define XI3CPSX_INTR_STATUS_EN_RESP_READY_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_RESP_READY_STS_EN_MASK    0x00000010
#define XI3CPSX_INTR_STATUS_EN_RESP_READY_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_CMD_QUEUE_READY_STS_EN_SHIFT   3
#define XI3CPSX_INTR_STATUS_EN_CMD_QUEUE_READY_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_CMD_QUEUE_READY_STS_EN_MASK    0x00000008
#define XI3CPSX_INTR_STATUS_EN_CMD_QUEUE_READY_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_IBI_THLD_STS_EN_SHIFT   2
#define XI3CPSX_INTR_STATUS_EN_IBI_THLD_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_IBI_THLD_STS_EN_MASK    0x00000004
#define XI3CPSX_INTR_STATUS_EN_IBI_THLD_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_RX_THLD_STS_EN_SHIFT   1
#define XI3CPSX_INTR_STATUS_EN_RX_THLD_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_RX_THLD_STS_EN_MASK    0x00000002
#define XI3CPSX_INTR_STATUS_EN_RX_THLD_STS_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_STATUS_EN_TX_THLD_STS_EN_SHIFT   0
#define XI3CPSX_INTR_STATUS_EN_TX_THLD_STS_EN_WIDTH   1
#define XI3CPSX_INTR_STATUS_EN_TX_THLD_STS_EN_MASK    0x00000001
#define XI3CPSX_INTR_STATUS_EN_TX_THLD_STS_EN_DEFVAL  0x0

/**
 * Register: XI3CPSX_INTR_SIGNAL_EN
 */
#define XI3CPSX_INTR_SIGNAL_EN    ( ( XI3CPSX_BASEADDR ) + 0x00000044 )
#define XI3CPSX_INTR_SIGNAL_EN_FULLMASK     0x0000bf7f
#define XI3CPSX_INTR_SIGNAL_EN_FULLRWMASK  0x0000bf7f
#define XI3CPSX_INTR_SIGNAL_EN_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_BUS_RESET_DONE_SIGNAL_EN_SHIFT   15
#define XI3CPSX_INTR_SIGNAL_EN_BUS_RESET_DONE_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_BUS_RESET_DONE_SIGNAL_EN_MASK    0x00008000
#define XI3CPSX_INTR_SIGNAL_EN_BUS_RESET_DONE_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_BUSOWNER_UPDATED_SIGNAL_EN_SHIFT   13
#define XI3CPSX_INTR_SIGNAL_EN_BUSOWNER_UPDATED_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_BUSOWNER_UPDATED_SIGNAL_EN_MASK    0x00002000
#define XI3CPSX_INTR_SIGNAL_EN_BUSOWNER_UPDATED_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_IBI_UPDATED_SIGNAL_EN_SHIFT   12
#define XI3CPSX_INTR_SIGNAL_EN_IBI_UPDATED_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_IBI_UPDATED_SIGNAL_EN_MASK    0x00001000
#define XI3CPSX_INTR_SIGNAL_EN_IBI_UPDATED_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_READ_REQ_RECV_SIGNAL_EN_SHIFT   11
#define XI3CPSX_INTR_SIGNAL_EN_READ_REQ_RECV_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_READ_REQ_RECV_SIGNAL_EN_MASK    0x00000800
#define XI3CPSX_INTR_SIGNAL_EN_READ_REQ_RECV_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_DEFSLV_SIGNAL_EN_SHIFT   10
#define XI3CPSX_INTR_SIGNAL_EN_DEFSLV_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_DEFSLV_SIGNAL_EN_MASK    0x00000400
#define XI3CPSX_INTR_SIGNAL_EN_DEFSLV_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ERR_SIGNAL_EN_SHIFT   9
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ERR_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ERR_SIGNAL_EN_MASK    0x00000200
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ERR_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_DYN_ADDR_ASSGN_SIGNAL_EN_SHIFT   8
#define XI3CPSX_INTR_SIGNAL_EN_DYN_ADDR_ASSGN_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_DYN_ADDR_ASSGN_SIGNAL_EN_MASK    0x00000100
#define XI3CPSX_INTR_SIGNAL_EN_DYN_ADDR_ASSGN_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_CCC_UPDATED_SIGNAL_EN_SHIFT   6
#define XI3CPSX_INTR_SIGNAL_EN_CCC_UPDATED_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_CCC_UPDATED_SIGNAL_EN_MASK    0x00000040
#define XI3CPSX_INTR_SIGNAL_EN_CCC_UPDATED_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ABORT_SIGNAL_EN_SHIFT   5
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ABORT_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ABORT_SIGNAL_EN_MASK    0x00000020
#define XI3CPSX_INTR_SIGNAL_EN_TRANSFER_ABORT_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_RESP_READY_SIGNAL_EN_SHIFT   4
#define XI3CPSX_INTR_SIGNAL_EN_RESP_READY_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_RESP_READY_SIGNAL_EN_MASK    0x00000010
#define XI3CPSX_INTR_SIGNAL_EN_RESP_READY_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_CMD_QUEUE_READY_SIGNAL_EN_SHIFT   3
#define XI3CPSX_INTR_SIGNAL_EN_CMD_QUEUE_READY_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_CMD_QUEUE_READY_SIGNAL_EN_MASK    0x00000008
#define XI3CPSX_INTR_SIGNAL_EN_CMD_QUEUE_READY_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_IBI_THLD_SIGNAL_EN_SHIFT   2
#define XI3CPSX_INTR_SIGNAL_EN_IBI_THLD_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_IBI_THLD_SIGNAL_EN_MASK    0x00000004
#define XI3CPSX_INTR_SIGNAL_EN_IBI_THLD_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_RX_THLD_SIGNAL_EN_SHIFT   1
#define XI3CPSX_INTR_SIGNAL_EN_RX_THLD_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_RX_THLD_SIGNAL_EN_MASK    0x00000002
#define XI3CPSX_INTR_SIGNAL_EN_RX_THLD_SIGNAL_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_INTR_SIGNAL_EN_TX_THLD_SIGNAL_EN_SHIFT   0
#define XI3CPSX_INTR_SIGNAL_EN_TX_THLD_SIGNAL_EN_WIDTH   1
#define XI3CPSX_INTR_SIGNAL_EN_TX_THLD_SIGNAL_EN_MASK    0x00000001
#define XI3CPSX_INTR_SIGNAL_EN_TX_THLD_SIGNAL_EN_DEFVAL  0x0

/**
 * Register: XI3CPSX_INTR_FORCE
 */
#define XI3CPSX_INTR_FORCE    ( ( XI3CPSX_BASEADDR ) + 0x00000048 )
#define XI3CPSX_INTR_FORCE_FULLMASK     0x0000bf7f
#define XI3CPSX_INTR_FORCE_FULLRWMASK  0x00000000
#define XI3CPSX_INTR_FORCE_DEFVAL   0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_BUS_RESET_DONE_FORCE_E_SHIFT   15
#define XI3CPSX_INTR_FORCE_BUS_RESET_DONE_FORCE_E_WIDTH   1
#define XI3CPSX_INTR_FORCE_BUS_RESET_DONE_FORCE_E_MASK    0x00008000
#define XI3CPSX_INTR_FORCE_BUS_RESET_DONE_FORCE_E_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_BUSOWNER_UPDATED_FORCE_EN_SHIFT   13
#define XI3CPSX_INTR_FORCE_BUSOWNER_UPDATED_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_BUSOWNER_UPDATED_FORCE_EN_MASK    0x00002000
#define XI3CPSX_INTR_FORCE_BUSOWNER_UPDATED_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_IBI_UPDATED_FORCE_EN_SHIFT   12
#define XI3CPSX_INTR_FORCE_IBI_UPDATED_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_IBI_UPDATED_FORCE_EN_MASK    0x00001000
#define XI3CPSX_INTR_FORCE_IBI_UPDATED_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_READ_REQ_FORCE_EN_SHIFT   11
#define XI3CPSX_INTR_FORCE_READ_REQ_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_READ_REQ_FORCE_EN_MASK    0x00000800
#define XI3CPSX_INTR_FORCE_READ_REQ_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_DEFSLV_FORCE_EN_SHIFT   10
#define XI3CPSX_INTR_FORCE_DEFSLV_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_DEFSLV_FORCE_EN_MASK    0x00000400
#define XI3CPSX_INTR_FORCE_DEFSLV_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_TRANSFER_ERR_FORCE_EN_SHIFT   9
#define XI3CPSX_INTR_FORCE_TRANSFER_ERR_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_TRANSFER_ERR_FORCE_EN_MASK    0x00000200
#define XI3CPSX_INTR_FORCE_TRANSFER_ERR_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_DYN_ADDR_ASSGN_FORCE_EN_SHIFT   8
#define XI3CPSX_INTR_FORCE_DYN_ADDR_ASSGN_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_DYN_ADDR_ASSGN_FORCE_EN_MASK    0x00000100
#define XI3CPSX_INTR_FORCE_DYN_ADDR_ASSGN_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_CCC_UPDATED_FORCE_EN_SHIFT   6
#define XI3CPSX_INTR_FORCE_CCC_UPDATED_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_CCC_UPDATED_FORCE_EN_MASK    0x00000040
#define XI3CPSX_INTR_FORCE_CCC_UPDATED_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_TRANSFER_ABORT_FORCE_EN_SHIFT   5
#define XI3CPSX_INTR_FORCE_TRANSFER_ABORT_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_TRANSFER_ABORT_FORCE_EN_MASK    0x00000020
#define XI3CPSX_INTR_FORCE_TRANSFER_ABORT_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_RESP_READY_FORCE_EN_SHIFT   4
#define XI3CPSX_INTR_FORCE_RESP_READY_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_RESP_READY_FORCE_EN_MASK    0x00000010
#define XI3CPSX_INTR_FORCE_RESP_READY_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_CMD_QUEUE_READY_FORCE_EN_SHIFT   3
#define XI3CPSX_INTR_FORCE_CMD_QUEUE_READY_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_CMD_QUEUE_READY_FORCE_EN_MASK    0x00000008
#define XI3CPSX_INTR_FORCE_CMD_QUEUE_READY_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_IBI_THLD_FORCE_EN_SHIFT   2
#define XI3CPSX_INTR_FORCE_IBI_THLD_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_IBI_THLD_FORCE_EN_MASK    0x00000004
#define XI3CPSX_INTR_FORCE_IBI_THLD_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_RX_THLD_FORCE_EN_SHIFT   1
#define XI3CPSX_INTR_FORCE_RX_THLD_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_RX_THLD_FORCE_EN_MASK    0x00000002
#define XI3CPSX_INTR_FORCE_RX_THLD_FORCE_EN_DEFVAL  0x0

/* access_type: wo */
#define XI3CPSX_INTR_FORCE_TX_THLD_FORCE_EN_SHIFT   0
#define XI3CPSX_INTR_FORCE_TX_THLD_FORCE_EN_WIDTH   1
#define XI3CPSX_INTR_FORCE_TX_THLD_FORCE_EN_MASK    0x00000001
#define XI3CPSX_INTR_FORCE_TX_THLD_FORCE_EN_DEFVAL  0x0

/**
 * @name interrupt Register  (INTR) mask(s)
 * @{
 */
#define XI3CPSX_INTR_TX_THLD			0x00000001	/**< BIT 0 - Transmit Buffer Threshold */
#define XI3CPSX_INTR_RX_THLD			0x00000002	/**< BIT 1 - Receive Buffer Threshold */
#define XI3CPSX_INTR_IBI_THLD			0x00000004	/**< BIT 2 - IBI Buffer Threshold */
#define XI3CPSX_INTR_CMD_QUEUE_READY		0x00000008	/**< BIT 3 - Command Queue Ready */
#define XI3CPSX_INTR_RESP_READY			0x00000010	/**< BIT 4 - Response Queue Ready */
#define XI3CPSX_INTR_TRANSFER_ABORT		0x00000020	/**< BIT 5 - Transfer Abort */
#define XI3CPSX_INTR_CCC_UPDATED		0x00000040	/**< BIT 6 - CCC Table Updated */
#define XI3CPSX_INTR_DYN_ADDR_ASSGN		0x00000100	/**< BIT 8 - Dynamic Address Assigned - only in slave mode */
#define XI3CPSX_INTR_TRANSFER_ERR		0x00000200	/**< BIT 9 - Transfer Error */
#define XI3CPSX_INTR_DEFSLV			0x00000400	/**< BIT 10 - Define Slave CCC Received */
#define XI3CPSX_INTR_READ_REQ_RECV		0x00000800	/**< BIT 11 - Read Request Received */
#define XI3CPSX_INTR_IBI_UPDATED		0x00001000	/**< BIT 12 - IBI status is updated */
#define XI3CPSX_INTR_BUSOWNER_UPDATED		0x00002000	/**< BIT 13 - Role of the controller changes from being
								  a Master to Slave or vice versa */
#define XI3CPSX_INTR_BUS_RESET_DONE		0x00008000	/**< BIT 15 - Bus Reset Pattern Generation Done */

/**
 * Register: XI3CPSX_QUEUE_STATUS_LEVEL
 */
#define XI3CPSX_QUEUE_STATUS_LEVEL    ( ( XI3CPSX_BASEADDR ) + 0x0000004C )
#define XI3CPSX_QUEUE_STATUS_LEVEL_FULLMASK     0x1fffffff
#define XI3CPSX_QUEUE_STATUS_LEVEL_FULLRWMASK  0x00000000
#define XI3CPSX_QUEUE_STATUS_LEVEL_DEFVAL   0x8

/* access_type: ro */
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_STS_CNT_SHIFT   24
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_STS_CNT_WIDTH   5
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_STS_CNT_MASK    0x1f000000
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_STS_CNT_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_BUF_BLR_SHIFT   16
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_BUF_BLR_WIDTH   8
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_BUF_BLR_MASK    0x00ff0000
#define XI3CPSX_QUEUE_STATUS_LEVEL_IBI_BUF_BLR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_QUEUE_STATUS_LEVEL_RESP_BUF_BLR_SHIFT   8
#define XI3CPSX_QUEUE_STATUS_LEVEL_RESP_BUF_BLR_WIDTH   8
#define XI3CPSX_QUEUE_STATUS_LEVEL_RESP_BUF_BLR_MASK    0x0000ff00
#define XI3CPSX_QUEUE_STATUS_LEVEL_RESP_BUF_BLR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_QUEUE_STATUS_LEVEL_CMD_QUEUE_EMPTY_LOC_SHIFT   0
#define XI3CPSX_QUEUE_STATUS_LEVEL_CMD_QUEUE_EMPTY_LOC_WIDTH   8
#define XI3CPSX_QUEUE_STATUS_LEVEL_CMD_QUEUE_EMPTY_LOC_MASK    0x000000ff
#define XI3CPSX_QUEUE_STATUS_LEVEL_CMD_QUEUE_EMPTY_LOC_DEFVAL  0x8

/**
 * Register: XI3CPSX_DATA_BUFFER_STATUS_LEVEL
 */
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL    ( ( XI3CPSX_BASEADDR ) + 0x00000050 )
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_FULLMASK     0x00ff00ff
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_FULLRWMASK  0x00000000
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_DEFVAL   0x20

/* access_type: ro */
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_SHIFT   16
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_WIDTH   8
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_MASK    0x00ff0000
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_RX_BUF_BLR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_TX_BUF_EMPTY_LOC_SHIFT   0
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_TX_BUF_EMPTY_LOC_WIDTH   8
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_TX_BUF_EMPTY_LOC_MASK    0x000000ff
#define XI3CPSX_DATA_BUFFER_STATUS_LEVEL_TX_BUF_EMPTY_LOC_DEFVAL  0x20

/**
 * Register: XI3CPSX_PRESENT_STATE
 */
#define XI3CPSX_PRESENT_STATE    ( ( XI3CPSX_BASEADDR ) + 0x00000054 )
#define XI3CPSX_PRESENT_STATE_FULLMASK     0x1f3f3f07
#define XI3CPSX_PRESENT_STATE_FULLRWMASK  0x00000000
#define XI3CPSX_PRESENT_STATE_DEFVAL   0x10000007

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_MASTER_IDLE_SHIFT   28
#define XI3CPSX_PRESENT_STATE_MASTER_IDLE_WIDTH   1
#define XI3CPSX_PRESENT_STATE_MASTER_IDLE_MASK    0x10000000
#define XI3CPSX_PRESENT_STATE_MASTER_IDLE_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_CMD_TID_SHIFT   24
#define XI3CPSX_PRESENT_STATE_CMD_TID_WIDTH   4
#define XI3CPSX_PRESENT_STATE_CMD_TID_MASK    0x0f000000
#define XI3CPSX_PRESENT_STATE_CMD_TID_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_CM_TFR_ST_STS_SHIFT   16
#define XI3CPSX_PRESENT_STATE_CM_TFR_ST_STS_WIDTH   6
#define XI3CPSX_PRESENT_STATE_CM_TFR_ST_STS_MASK    0x003f0000
#define XI3CPSX_PRESENT_STATE_CM_TFR_ST_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_CM_TFR_STS_SHIFT   8
#define XI3CPSX_PRESENT_STATE_CM_TFR_STS_WIDTH   6
#define XI3CPSX_PRESENT_STATE_CM_TFR_STS_MASK    0x00003f00
#define XI3CPSX_PRESENT_STATE_CM_TFR_STS_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_CURRENT_MASTER_SHIFT   2
#define XI3CPSX_PRESENT_STATE_CURRENT_MASTER_WIDTH   1
#define XI3CPSX_PRESENT_STATE_CURRENT_MASTER_MASK    0x00000004
#define XI3CPSX_PRESENT_STATE_CURRENT_MASTER_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_SDA_LINE_SIGNAL_LEVEL_SHIFT   1
#define XI3CPSX_PRESENT_STATE_SDA_LINE_SIGNAL_LEVEL_WIDTH   1
#define XI3CPSX_PRESENT_STATE_SDA_LINE_SIGNAL_LEVEL_MASK    0x00000002
#define XI3CPSX_PRESENT_STATE_SDA_LINE_SIGNAL_LEVEL_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_PRESENT_STATE_SCL_LINE_SIGNAL_LEVEL_SHIFT   0
#define XI3CPSX_PRESENT_STATE_SCL_LINE_SIGNAL_LEVEL_WIDTH   1
#define XI3CPSX_PRESENT_STATE_SCL_LINE_SIGNAL_LEVEL_MASK    0x00000001
#define XI3CPSX_PRESENT_STATE_SCL_LINE_SIGNAL_LEVEL_DEFVAL  0x1

/**
 * Register: XI3CPSX_CCC_DEVICE_STATUS
 */
#define XI3CPSX_CCC_DEVICE_STATUS    ( ( XI3CPSX_BASEADDR ) + 0x00000058 )
#define XI3CPSX_CCC_DEVICE_STATUS_FULLMASK     0x00003fef
#define XI3CPSX_CCC_DEVICE_STATUS_FULLRWMASK  0x00000000
#define XI3CPSX_CCC_DEVICE_STATUS_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_FRAME_ERROR_SHIFT   13
#define XI3CPSX_CCC_DEVICE_STATUS_FRAME_ERROR_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_FRAME_ERROR_MASK    0x00002000
#define XI3CPSX_CCC_DEVICE_STATUS_FRAME_ERROR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_BUFFER_NOT_AVAIL_SHIFT   12
#define XI3CPSX_CCC_DEVICE_STATUS_BUFFER_NOT_AVAIL_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_BUFFER_NOT_AVAIL_MASK    0x00001000
#define XI3CPSX_CCC_DEVICE_STATUS_BUFFER_NOT_AVAIL_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_DATA_NOT_READY_SHIFT   11
#define XI3CPSX_CCC_DEVICE_STATUS_DATA_NOT_READY_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_DATA_NOT_READY_MASK    0x00000800
#define XI3CPSX_CCC_DEVICE_STATUS_DATA_NOT_READY_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_OVERFLOW_ERR_SHIFT   10
#define XI3CPSX_CCC_DEVICE_STATUS_OVERFLOW_ERR_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_OVERFLOW_ERR_MASK    0x00000400
#define XI3CPSX_CCC_DEVICE_STATUS_OVERFLOW_ERR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_SLAVE_BUSY_SHIFT   9
#define XI3CPSX_CCC_DEVICE_STATUS_SLAVE_BUSY_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_SLAVE_BUSY_MASK    0x00000200
#define XI3CPSX_CCC_DEVICE_STATUS_SLAVE_BUSY_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_UNDERFLOW_ERR_SHIFT   8
#define XI3CPSX_CCC_DEVICE_STATUS_UNDERFLOW_ERR_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_UNDERFLOW_ERR_MASK    0x00000100
#define XI3CPSX_CCC_DEVICE_STATUS_UNDERFLOW_ERR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_ACTIVITY_MODE_SHIFT   6
#define XI3CPSX_CCC_DEVICE_STATUS_ACTIVITY_MODE_WIDTH   2
#define XI3CPSX_CCC_DEVICE_STATUS_ACTIVITY_MODE_MASK    0x000000c0
#define XI3CPSX_CCC_DEVICE_STATUS_ACTIVITY_MODE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_PROTOCOL_ERR_SHIFT   5
#define XI3CPSX_CCC_DEVICE_STATUS_PROTOCOL_ERR_WIDTH   1
#define XI3CPSX_CCC_DEVICE_STATUS_PROTOCOL_ERR_MASK    0x00000020
#define XI3CPSX_CCC_DEVICE_STATUS_PROTOCOL_ERR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_CCC_DEVICE_STATUS_PENDING_INTR_SHIFT   0
#define XI3CPSX_CCC_DEVICE_STATUS_PENDING_INTR_WIDTH   4
#define XI3CPSX_CCC_DEVICE_STATUS_PENDING_INTR_MASK    0x0000000f
#define XI3CPSX_CCC_DEVICE_STATUS_PENDING_INTR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEVICE_ADDR_TABLE_POINTER
 */
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER    ( ( XI3CPSX_BASEADDR ) + 0x0000005C )
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_FULLMASK     0xffffffff
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_FULLRWMASK  0x00000000
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_DEFVAL   0xb02c0

/* access_type: ro */
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_DEV_ADDR_TABLE_DEPTH_SHIFT   16
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_DEV_ADDR_TABLE_DEPTH_WIDTH   16
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_DEV_ADDR_TABLE_DEPTH_MASK    0xffff0000
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_DEV_ADDR_TABLE_DEPTH_DEFVAL  0xb

/* access_type: ro */
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_P_DEV_ADDR_TABLE_START_ADDR_SHIFT   0
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_P_DEV_ADDR_TABLE_START_ADDR_WIDTH   16
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_P_DEV_ADDR_TABLE_START_ADDR_MASK    0x0000ffff
#define XI3CPSX_DEVICE_ADDR_TABLE_POINTER_P_DEV_ADDR_TABLE_START_ADDR_DEFVAL  0x2c0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE_POINTER
 */
#define XI3CPSX_DEV_CHAR_TABLE_POINTER    ( ( XI3CPSX_BASEADDR ) + 0x00000060 )
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_FULLMASK     0x007fffff
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_FULLRWMASK  0x00780000
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_DEFVAL   0x2c200

/* access_type: rw */
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_PRESENT_DEV_CHAR_TABLE_INDX_SHIFT   19
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_PRESENT_DEV_CHAR_TABLE_INDX_WIDTH   4
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_PRESENT_DEV_CHAR_TABLE_INDX_MASK    0x00780000
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_PRESENT_DEV_CHAR_TABLE_INDX_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_DEV_CHAR_TABLE_DEPTH_SHIFT   12
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_DEV_CHAR_TABLE_DEPTH_WIDTH   7
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_DEV_CHAR_TABLE_DEPTH_MASK    0x0007f000
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_DEV_CHAR_TABLE_DEPTH_DEFVAL  0x2c

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_P_DEV_CHAR_TABLE_START_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_P_DEV_CHAR_TABLE_START_ADDR_WIDTH   12
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_P_DEV_CHAR_TABLE_START_ADDR_MASK    0x00000fff
#define XI3CPSX_DEV_CHAR_TABLE_POINTER_P_DEV_CHAR_TABLE_START_ADDR_DEFVAL  0x200

/**
 * Register: XI3CPSX_VENDOR_SPECIFIC_REG_POINTER
 */
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER    ( ( XI3CPSX_BASEADDR ) + 0x0000006C )
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_FULLMASK     0x0000ffff
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_FULLRWMASK  0x00000000
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_DEFVAL   0xb0

/* access_type: ro */
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_P_VENDOR_REG_START_ADDR_SHIFT   0
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_P_VENDOR_REG_START_ADDR_WIDTH   16
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_P_VENDOR_REG_START_ADDR_MASK    0x0000ffff
#define XI3CPSX_VENDOR_SPECIFIC_REG_POINTER_P_VENDOR_REG_START_ADDR_DEFVAL  0xb0

/**
 * Register: XI3CPSX_SLV_MIPI_ID_VALUE
 */
#define XI3CPSX_SLV_MIPI_ID_VALUE    ( ( XI3CPSX_BASEADDR ) + 0x00000070 )
#define XI3CPSX_SLV_MIPI_ID_VALUE_FULLMASK     0x0000ffff
#define XI3CPSX_SLV_MIPI_ID_VALUE_FULLRWMASK  0x0000ffff
#define XI3CPSX_SLV_MIPI_ID_VALUE_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_MIPI_MFG_ID_SHIFT   1
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_MIPI_MFG_ID_WIDTH   15
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_MIPI_MFG_ID_MASK    0x0000fffe
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_MIPI_MFG_ID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_PROV_ID_SEL_SHIFT     0
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_PROV_ID_SEL_WIDTH     1
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_PROV_ID_SEL_MASK      0x00000001
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_PROV_ID_SEL_DEFVAL    0x0
#define XI3CPSX_SLV_MIPI_ID_VALUE_SLV_PROV_ID_SEL_VAL       0x2

/**
 * Register: XI3CPSX_SLV_PID_VALUE
 */
#define XI3CPSX_SLV_PID_VALUE    ( ( XI3CPSX_BASEADDR ) + 0x00000074 )
#define XI3CPSX_SLV_PID_VALUE_FULLMASK     0xffffffff
#define XI3CPSX_SLV_PID_VALUE_FULLRWMASK  0xffffffff
#define XI3CPSX_SLV_PID_VALUE_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_SLV_PID_VALUE_SLV_PART_ID_SHIFT   16
#define XI3CPSX_SLV_PID_VALUE_SLV_PART_ID_WIDTH   16
#define XI3CPSX_SLV_PID_VALUE_SLV_PART_ID_MASK    0xffff0000
#define XI3CPSX_SLV_PID_VALUE_SLV_PART_ID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_PID_VALUE_SLV_INST_ID_SHIFT   12
#define XI3CPSX_SLV_PID_VALUE_SLV_INST_ID_WIDTH   4
#define XI3CPSX_SLV_PID_VALUE_SLV_INST_ID_MASK    0x0000f000
#define XI3CPSX_SLV_PID_VALUE_SLV_INST_ID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_PID_VALUE_SLV_PID_DCR_SHIFT   0
#define XI3CPSX_SLV_PID_VALUE_SLV_PID_DCR_WIDTH   12
#define XI3CPSX_SLV_PID_VALUE_SLV_PID_DCR_MASK    0x00000fff
#define XI3CPSX_SLV_PID_VALUE_SLV_PID_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_SLV_CHAR_CTRL
 */
#define XI3CPSX_SLV_CHAR_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x00000078 )
#define XI3CPSX_SLV_CHAR_CTRL_FULLMASK     0x00ffffff
#define XI3CPSX_SLV_CHAR_CTRL_FULLRWMASK  0x0000ffe1
#define XI3CPSX_SLV_CHAR_CTRL_DEFVAL   0x10062

/* access_type: ro */
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAP_SHIFT   16
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAP_WIDTH   8
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAP_MASK    0x00ff0000
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAP_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_SLV_CHAR_CTRL_DCR_SHIFT        8
#define XI3CPSX_SLV_CHAR_CTRL_DCR_WIDTH        8
#define XI3CPSX_SLV_CHAR_CTRL_DCR_MASK         0x0000ff00
#define XI3CPSX_SLV_CHAR_CTRL_DCR_DEFVAL       0x0
#define XI3CPSX_SLV_CHAR_CTRL_DCR_VAL          0xC4

/* access_type: rw */
#define XI3CPSX_SLV_CHAR_CTRL_DEVICE_ROLE_SHIFT   6
#define XI3CPSX_SLV_CHAR_CTRL_DEVICE_ROLE_WIDTH   2
#define XI3CPSX_SLV_CHAR_CTRL_DEVICE_ROLE_MASK    0x000000c0
#define XI3CPSX_SLV_CHAR_CTRL_DEVICE_ROLE_DEFVAL  0x1
#define XI3CPSX_SLV_CHAR_CTRL_DEVICE_ROLE_SLAVE  0x2

/* access_type: rw */
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAPABLE_SHIFT   5
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAPABLE_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAPABLE_MASK    0x00000020
#define XI3CPSX_SLV_CHAR_CTRL_HDR_CAPABLE_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_SLV_CHAR_CTRL_BRIDGE_IDENTIFIER_SHIFT   4
#define XI3CPSX_SLV_CHAR_CTRL_BRIDGE_IDENTIFIER_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_BRIDGE_IDENTIFIER_MASK    0x00000010
#define XI3CPSX_SLV_CHAR_CTRL_BRIDGE_IDENTIFIER_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_SLV_CHAR_CTRL_OFFLINE_CAPABLE_SHIFT   3
#define XI3CPSX_SLV_CHAR_CTRL_OFFLINE_CAPABLE_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_OFFLINE_CAPABLE_MASK    0x00000008
#define XI3CPSX_SLV_CHAR_CTRL_OFFLINE_CAPABLE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_SLV_CHAR_CTRL_IBI_PAYLOAD_SHIFT   2
#define XI3CPSX_SLV_CHAR_CTRL_IBI_PAYLOAD_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_IBI_PAYLOAD_MASK    0x00000004
#define XI3CPSX_SLV_CHAR_CTRL_IBI_PAYLOAD_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_SLV_CHAR_CTRL_IBI_REQUEST_CAPABLE_SHIFT   1
#define XI3CPSX_SLV_CHAR_CTRL_IBI_REQUEST_CAPABLE_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_IBI_REQUEST_CAPABLE_MASK    0x00000002
#define XI3CPSX_SLV_CHAR_CTRL_IBI_REQUEST_CAPABLE_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_SLV_CHAR_CTRL_MAX_DATA_SPEED_LIMIT_SHIFT   0
#define XI3CPSX_SLV_CHAR_CTRL_MAX_DATA_SPEED_LIMIT_WIDTH   1
#define XI3CPSX_SLV_CHAR_CTRL_MAX_DATA_SPEED_LIMIT_MASK    0x00000001
#define XI3CPSX_SLV_CHAR_CTRL_MAX_DATA_SPEED_LIMIT_DEFVAL  0x0

/**
 * Register: XI3CPSX_SLV_MAX_LEN
 */
#define XI3CPSX_SLV_MAX_LEN    ( ( XI3CPSX_BASEADDR ) + 0x0000007C )
#define XI3CPSX_SLV_MAX_LEN_FULLMASK     0xffffffff
#define XI3CPSX_SLV_MAX_LEN_FULLRWMASK  0x00000000
#define XI3CPSX_SLV_MAX_LEN_DEFVAL   0xff00ff

/* access_type: ro */
#define XI3CPSX_SLV_MAX_LEN_MRL_SHIFT   16
#define XI3CPSX_SLV_MAX_LEN_MRL_WIDTH   16
#define XI3CPSX_SLV_MAX_LEN_MRL_MASK    0xffff0000
#define XI3CPSX_SLV_MAX_LEN_MRL_DEFVAL  0xff

/* access_type: ro */
#define XI3CPSX_SLV_MAX_LEN_MWL_SHIFT   0
#define XI3CPSX_SLV_MAX_LEN_MWL_WIDTH   16
#define XI3CPSX_SLV_MAX_LEN_MWL_MASK    0x0000ffff
#define XI3CPSX_SLV_MAX_LEN_MWL_DEFVAL  0xff

/**
 * Register: XI3CPSX_MAX_READ_TURNAROUND
 */
#define XI3CPSX_MAX_READ_TURNAROUND    ( ( XI3CPSX_BASEADDR ) + 0x00000080 )
#define XI3CPSX_MAX_READ_TURNAROUND_FULLMASK     0x00ffffff
#define XI3CPSX_MAX_READ_TURNAROUND_FULLRWMASK  0x00000000
#define XI3CPSX_MAX_READ_TURNAROUND_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_MAX_READ_TURNAROUND_MXDS_MAX_RD_TURN_SHIFT   0
#define XI3CPSX_MAX_READ_TURNAROUND_MXDS_MAX_RD_TURN_WIDTH   24
#define XI3CPSX_MAX_READ_TURNAROUND_MXDS_MAX_RD_TURN_MASK    0x00ffffff
#define XI3CPSX_MAX_READ_TURNAROUND_MXDS_MAX_RD_TURN_DEFVAL  0x0

/**
 * Register: XI3CPSX_MAX_DATA_SPEED
 */
#define XI3CPSX_MAX_DATA_SPEED    ( ( XI3CPSX_BASEADDR ) + 0x00000084 )
#define XI3CPSX_MAX_DATA_SPEED_FULLMASK     0x00070707
#define XI3CPSX_MAX_DATA_SPEED_FULLRWMASK  0x00070707
#define XI3CPSX_MAX_DATA_SPEED_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_MAX_DATA_SPEED_MXDS_CLK_DATA_TURN_SHIFT   16
#define XI3CPSX_MAX_DATA_SPEED_MXDS_CLK_DATA_TURN_WIDTH   3
#define XI3CPSX_MAX_DATA_SPEED_MXDS_CLK_DATA_TURN_MASK    0x00070000
#define XI3CPSX_MAX_DATA_SPEED_MXDS_CLK_DATA_TURN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_RD_SPEED_SHIFT   8
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_RD_SPEED_WIDTH   3
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_RD_SPEED_MASK    0x00000700
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_RD_SPEED_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_WR_SPEED_SHIFT   0
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_WR_SPEED_WIDTH   3
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_WR_SPEED_MASK    0x00000007
#define XI3CPSX_MAX_DATA_SPEED_MXDS_MAX_WR_SPEED_DEFVAL  0x0

/**
 * Register: XI3CPSX_SLV_INTR_REQ
 */
#define XI3CPSX_SLV_INTR_REQ    ( ( XI3CPSX_BASEADDR ) + 0x0000008C )
#define XI3CPSX_SLV_INTR_REQ_FULLMASK     0x0000030f
#define XI3CPSX_SLV_INTR_REQ_FULLRWMASK  0x0000000f
#define XI3CPSX_SLV_INTR_REQ_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_SLV_INTR_REQ_IBI_STS_SHIFT   8
#define XI3CPSX_SLV_INTR_REQ_IBI_STS_WIDTH   2
#define XI3CPSX_SLV_INTR_REQ_IBI_STS_MASK    0x00000300
#define XI3CPSX_SLV_INTR_REQ_IBI_STS_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_INTR_REQ_MR_SHIFT   3
#define XI3CPSX_SLV_INTR_REQ_MR_WIDTH   1
#define XI3CPSX_SLV_INTR_REQ_MR_MASK    0x00000008
#define XI3CPSX_SLV_INTR_REQ_MR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_INTR_REQ_SIR_CTRL_SHIFT   1
#define XI3CPSX_SLV_INTR_REQ_SIR_CTRL_WIDTH   2
#define XI3CPSX_SLV_INTR_REQ_SIR_CTRL_MASK    0x00000006
#define XI3CPSX_SLV_INTR_REQ_SIR_CTRL_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_SLV_INTR_REQ_SIR_SHIFT   0
#define XI3CPSX_SLV_INTR_REQ_SIR_WIDTH   1
#define XI3CPSX_SLV_INTR_REQ_SIR_MASK    0x00000001
#define XI3CPSX_SLV_INTR_REQ_SIR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEVICE_CTRL_EXTENDED
 */
#define XI3CPSX_DEVICE_CTRL_EXTENDED    ( ( XI3CPSX_BASEADDR ) + 0x000000B0 )
#define XI3CPSX_DEVICE_CTRL_EXTENDED_FULLMASK     0x0000000b
#define XI3CPSX_DEVICE_CTRL_EXTENDED_FULLRWMASK  0x0000000b
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_EXTENDED_REQMST_ACK_CTRL_SHIFT   3
#define XI3CPSX_DEVICE_CTRL_EXTENDED_REQMST_ACK_CTRL_WIDTH   1
#define XI3CPSX_DEVICE_CTRL_EXTENDED_REQMST_ACK_CTRL_MASK    0x00000008
#define XI3CPSX_DEVICE_CTRL_EXTENDED_REQMST_ACK_CTRL_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_SHIFT    0
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_MASTER   0x0
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_SLAVE    0x1
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_WIDTH    2
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_MASK     0x00000003
#define XI3CPSX_DEVICE_CTRL_EXTENDED_DEV_OPERATION_MODE_DEFVAL   0x0

/**
 * Register: XI3CPSX_SCL_I3C_OD_TIMING
 */
#define XI3CPSX_SCL_I3C_OD_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000B4 )
#define XI3CPSX_SCL_I3C_OD_TIMING_FULLMASK     0x00ff00ff
#define XI3CPSX_SCL_I3C_OD_TIMING_FULLRWMASK  0x00ff00ff
#define XI3CPSX_SCL_I3C_OD_TIMING_DEFVAL   0xa0010

/* access_type: rw */
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_HCNT_SHIFT   16
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_HCNT_WIDTH   8
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_HCNT_MASK    0x00ff0000
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_HCNT_DEFVAL  0xa
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_HCNT_VAL     8

/* access_type: rw */
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_LCNT_SHIFT   0
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_LCNT_WIDTH   8
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_LCNT_MASK    0x000000ff
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_LCNT_DEFVAL  0x10
#define XI3CPSX_SCL_I3C_OD_TIMING_I3C_OD_LCNT_VAL     50

/**
 * Register: XI3CPSX_SCL_I3C_PP_TIMING
 */
#define XI3CPSX_SCL_I3C_PP_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000B8 )
#define XI3CPSX_SCL_I3C_PP_TIMING_FULLMASK     0x00ff00ff
#define XI3CPSX_SCL_I3C_PP_TIMING_FULLRWMASK  0x00ff00ff
#define XI3CPSX_SCL_I3C_PP_TIMING_DEFVAL   0xa000a

/* access_type: rw */
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_HCNT_SHIFT   16
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_HCNT_WIDTH   8
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_HCNT_MASK    0x00ff0000
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_HCNT_DEFVAL  0xa
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_HCNT_VAL     8

/* access_type: rw */
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_LCNT_SHIFT   0
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_LCNT_WIDTH   8
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_LCNT_MASK    0x000000ff
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_LCNT_DEFVAL  0xa
#define XI3CPSX_SCL_I3C_PP_TIMING_I3C_PP_LCNT_VAL     8

/**
 * Register: XI3CPSX_SCL_I2C_FM_TIMING
 */
#define XI3CPSX_SCL_I2C_FM_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000BC )
#define XI3CPSX_SCL_I2C_FM_TIMING_FULLMASK     0xffffffff
#define XI3CPSX_SCL_I2C_FM_TIMING_FULLRWMASK  0xffffffff
#define XI3CPSX_SCL_I2C_FM_TIMING_DEFVAL   0x100010

/* access_type: rw */
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_HCNT_SHIFT   16
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_HCNT_WIDTH   16
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_HCNT_MASK    0xffff0000
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_HCNT_DEFVAL  0x10

/* access_type: rw */
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_LCNT_SHIFT   0
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_LCNT_WIDTH   16
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_LCNT_MASK    0x0000ffff
#define XI3CPSX_SCL_I2C_FM_TIMING_I2C_FM_LCNT_DEFVAL  0x10

/**
 * Register: XI3CPSX_SCL_I2C_FMP_TIMING
 */
#define XI3CPSX_SCL_I2C_FMP_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000C0 )
#define XI3CPSX_SCL_I2C_FMP_TIMING_FULLMASK     0x00ffffff
#define XI3CPSX_SCL_I2C_FMP_TIMING_FULLRWMASK  0x00ffffff
#define XI3CPSX_SCL_I2C_FMP_TIMING_DEFVAL   0x100010

/* access_type: rw */
#define XI3CPSX_SCL_I2C_FMP_TIMING_I2C_FMP_HCNT_SHIFT   16
#define XI3CPSX_SCL_I2C_FMP_TIMING_I2C_FMP_HCNT_WIDTH   8
#define XI3CPSX_SCL_I2C_FMP_TIMING_I2C_FMP_HCNT_MASK    0x00ff0000
#define XI3CPSX_SCL_I2C_FMP_TIMING_I2C_FMP_HCNT_DEFVAL  0x10

/* access_type: rw */
#define XI3CPSX_SCL_I2C_FMP_TIMING_2C_FMP_LCNT_SHIFT   0
#define XI3CPSX_SCL_I2C_FMP_TIMING_2C_FMP_LCNT_WIDTH   16
#define XI3CPSX_SCL_I2C_FMP_TIMING_2C_FMP_LCNT_MASK    0x0000ffff
#define XI3CPSX_SCL_I2C_FMP_TIMING_2C_FMP_LCNT_DEFVAL  0x10

/**
 * Register: XI3CPSX_SCL_EXT_LCNT_TIMING
 */
#define XI3CPSX_SCL_EXT_LCNT_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000C8 )
#define XI3CPSX_SCL_EXT_LCNT_TIMING_FULLMASK     0xffffffff
#define XI3CPSX_SCL_EXT_LCNT_TIMING_FULLRWMASK  0xffffffff
#define XI3CPSX_SCL_EXT_LCNT_TIMING_DEFVAL   0x20202020

/* access_type: rw */
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_4_SHIFT   24
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_4_WIDTH   8
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_4_MASK    0xff000000
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_4_DEFVAL  0x20

/* access_type: rw */
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_3_SHIFT   16
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_3_WIDTH   8
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_3_MASK    0x00ff0000
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_3_DEFVAL  0x20

/* access_type: rw */
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_2_SHIFT   8
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_2_WIDTH   8
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_2_MASK    0x0000ff00
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_2_DEFVAL  0x20

/* access_type: rw */
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_1_SHIFT   0
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_1_WIDTH   8
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_1_MASK    0x000000ff
#define XI3CPSX_SCL_EXT_LCNT_TIMING_I3C_EXT_LCNT_1_DEFVAL  0x20

/**
 * Register: XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING
 */
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000CC )
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_FULLMASK     0x0000000f
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_FULLRWMASK  0x0000000f
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_I3C_EXT_TERMN_LCNT_SHIFT   0
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_I3C_EXT_TERMN_LCNT_WIDTH   4
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_I3C_EXT_TERMN_LCNT_MASK    0x0000000f
#define XI3CPSX_SCL_EXT_TERMN_LCNT_TIMING_I3C_EXT_TERMN_LCNT_DEFVAL  0x0

/**
 * Register: XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING
 */
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000D0 )
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_FULLMASK     0x00070000
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_FULLRWMASK  0x00070000
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_DEFVAL   0x10000

/* access_type: rw */
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_SDA_TX_HOLD_SHIFT   16
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_SDA_TX_HOLD_WIDTH   3
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_SDA_TX_HOLD_MASK    0x00070000
#define XI3CPSX_SDA_HOLD_SWITCH_DLY_TIMING_SDA_TX_HOLD_DEFVAL  0x1

/**
 * Register: XI3CPSX_BUS_FREE_AVAIL_TIMING
 */
#define XI3CPSX_BUS_FREE_AVAIL_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000D4 )
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_FULLMASK     0xffffffff
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_FULLRWMASK  0xffffffff
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_DEFVAL   0x200020

/* access_type: rw */
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_AVAILABLE_TIME_SHIFT       16
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_AVAILABLE_TIME_WIDTH       16
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_AVAILABLE_TIME_MASK        0xffff0000
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_AVAILABLE_TIME_DEFVAL      0x20
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_AVAILABLE_TIME_VAL         0xa0

/* access_type: rw */
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_FREE_TIME_SHIFT   0
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_FREE_TIME_WIDTH   16
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_FREE_TIME_MASK    0x0000ffff
#define XI3CPSX_BUS_FREE_AVAIL_TIMING_BUS_FREE_TIME_DEFVAL  0x20

/**
 * Register: XI3CPSX_BUS_IDLE_TIMING
 */
#define XI3CPSX_BUS_IDLE_TIMING    ( ( XI3CPSX_BASEADDR ) + 0x000000D8 )
#define XI3CPSX_BUS_IDLE_TIMING_FULLMASK     0x000fffff
#define XI3CPSX_BUS_IDLE_TIMING_FULLRWMASK  0x000fffff
#define XI3CPSX_BUS_IDLE_TIMING_DEFVAL   0x20

/* access_type: rw */
#define XI3CPSX_BUS_IDLE_TIMING_BUS_IDLE_TIME_SHIFT   0
#define XI3CPSX_BUS_IDLE_TIMING_BUS_IDLE_TIME_WIDTH   20
#define XI3CPSX_BUS_IDLE_TIMING_BUS_IDLE_TIME_MASK    0x000fffff
#define XI3CPSX_BUS_IDLE_TIMING_BUS_IDLE_TIME_DEFVAL  0x20

/**
 * Register: XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT
 */
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT    ( ( XI3CPSX_BASEADDR ) + 0x000000DC )
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_FULLMASK     0x03ffffff
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_FULLRWMASK  0x03ffffff
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_DEFVAL   0x3567e0

/* access_type: rw */
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_SCL_LOW_MST_TIMEOUT_COUNT_SHIFT   0
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_SCL_LOW_MST_TIMEOUT_COUNT_WIDTH   26
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_SCL_LOW_MST_TIMEOUT_COUNT_MASK    0x03ffffff
#define XI3CPSX_SCL_LOW_MST_EXT_TIMEOUT_SCL_LOW_MST_TIMEOUT_COUNT_DEFVAL  0x3567e0

/**
 * Register: XI3CPSX_I3C_VER_ID
 */
#define XI3CPSX_I3C_VER_ID    ( ( XI3CPSX_BASEADDR ) + 0x000000E0 )
#define XI3CPSX_I3C_VER_ID_FULLMASK     0xffffffff
#define XI3CPSX_I3C_VER_ID_FULLRWMASK  0x00000000
#define XI3CPSX_I3C_VER_ID_DEFVAL   0x3130302a

/* access_type: ro */
#define XI3CPSX_I3C_VER_ID_I3C_VER_ID_SHIFT   0
#define XI3CPSX_I3C_VER_ID_I3C_VER_ID_WIDTH   32
#define XI3CPSX_I3C_VER_ID_I3C_VER_ID_MASK    0xffffffff
#define XI3CPSX_I3C_VER_ID_I3C_VER_ID_DEFVAL  0x3130302a

/**
 * Register: XI3CPSX_I3C_VER_TYPE
 */
#define XI3CPSX_I3C_VER_TYPE    ( ( XI3CPSX_BASEADDR ) + 0x000000E4 )
#define XI3CPSX_I3C_VER_TYPE_FULLMASK     0xffffffff
#define XI3CPSX_I3C_VER_TYPE_FULLRWMASK  0x00000000
#define XI3CPSX_I3C_VER_TYPE_DEFVAL   0x6c633033

/* access_type: ro */
#define XI3CPSX_I3C_VER_TYPE_I3C_VER_TYPE_SHIFT   0
#define XI3CPSX_I3C_VER_TYPE_I3C_VER_TYPE_WIDTH   32
#define XI3CPSX_I3C_VER_TYPE_I3C_VER_TYPE_MASK    0xffffffff
#define XI3CPSX_I3C_VER_TYPE_I3C_VER_TYPE_DEFVAL  0x6c633033

/**
 * Register: XI3CPSX_QUEUE_SIZE_CAPABILITY
 */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY    ( ( XI3CPSX_BASEADDR ) + 0x000000E8 )
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_FULLMASK     0x000fffff
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_FULLRWMASK  0x00000000
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_DEFVAL   0x21244

/* access_type: ro */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_IBI_BUF_SIZE_SHIFT   16
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_IBI_BUF_SIZE_WIDTH   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_IBI_BUF_SIZE_MASK    0x000f0000
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_IBI_BUF_SIZE_DEFVAL  0x2

/* access_type: ro */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RESP_BUF_SIZE_SHIFT   12
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RESP_BUF_SIZE_WIDTH   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RESP_BUF_SIZE_MASK    0x0000f000
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RESP_BUF_SIZE_DEFVAL  0x1

/* access_type: ro */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_CMD_BUF_SIZE_SHIFT   8
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_CMD_BUF_SIZE_WIDTH   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_CMD_BUF_SIZE_MASK    0x00000f00
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_CMD_BUF_SIZE_DEFVAL  0x2

/* access_type: ro */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RX_BUF_SIZE_SHIFT   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RX_BUF_SIZE_WIDTH   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RX_BUF_SIZE_MASK    0x000000f0
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_RX_BUF_SIZE_DEFVAL  0x4

/* access_type: ro */
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_TX_BUF_SIZE_SHIFT   0
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_TX_BUF_SIZE_WIDTH   4
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_TX_BUF_SIZE_MASK    0x0000000f
#define XI3CPSX_QUEUE_SIZE_CAPABILITY_TX_BUF_SIZE_DEFVAL  0x4

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE1_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000200 )
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE1_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000204 )
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE1_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000208 )
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE1_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000020C )
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE1_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE2_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000210 )
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE2_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000214 )
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE2_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000218 )
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE2_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000021C )
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE2_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE3_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000220 )
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE3_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000224 )
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE3_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000228 )
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE3_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000022C )
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE3_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE4_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000230 )
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE4_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000234 )
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE4_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000238 )
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE4_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000023C )
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE4_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE5_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000240 )
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE5_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000244 )
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE5_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000248 )
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE5_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000024C )
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE5_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE6_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000250 )
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE6_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000254 )
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE6_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000258 )
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE6_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000025C )
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE6_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE7_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000260 )
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE7_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000264 )
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE7_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000268 )
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE7_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000026C )
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE7_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE8_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000270 )
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE8_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000274 )
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE8_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000278 )
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE8_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000027C )
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE8_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE9_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000280 )
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE9_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000284 )
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE9_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000288 )
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE9_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000028C )
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE9_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE10_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x00000290 )
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE10_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x00000294 )
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE10_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x00000298 )
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE10_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x0000029C )
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE10_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE11_LOC1
 */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x000002A0 )
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_FULLMASK     0xffffffff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_STATIC_ADDR_SHIFT   24
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_STATIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_STATIC_ADDR_MASK    0xff000000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_STATIC_ADDR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_BCR_TYPE_SHIFT   16
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_BCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_BCR_TYPE_MASK    0x00ff0000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_BCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DCR_TYPE_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DCR_TYPE_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DCR_TYPE_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DCR_TYPE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC1_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE11_LOC2
 */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x000002A4 )
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_MSB_PROVISIONAL_ID_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_MSB_PROVISIONAL_ID_WIDTH   16
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_MSB_PROVISIONAL_ID_MASK    0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC2_MSB_PROVISIONAL_ID_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE11_LOC3
 */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x000002A8 )
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_FULLMASK     0x0000ffff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_BCR_SHIFT   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_BCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_BCR_MASK    0x0000ff00
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_BCR_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_DCR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_DCR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_DCR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC3_DCR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_CHAR_TABLE11_LOC4
 */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x000002AC )
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_FULLMASK     0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_FULLRWMASK  0x00000000
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_DEV_DYNAMIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_DEV_DYNAMIC_ADDR_MASK    0x000000ff
#define XI3CPSX_DEV_CHAR_TABLE11_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC1
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC1    ( ( XI3CPSX_BASEADDR ) + 0x000002C0 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC1_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC2
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC2    ( ( XI3CPSX_BASEADDR ) + 0x000002C4 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC2_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC3
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC3    ( ( XI3CPSX_BASEADDR ) + 0x000002C8 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC3_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC4
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC4    ( ( XI3CPSX_BASEADDR ) + 0x000002CC )
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC4_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC5
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC5    ( ( XI3CPSX_BASEADDR ) + 0x000002D0 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC5_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC6
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC6    ( ( XI3CPSX_BASEADDR ) + 0x000002D4 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC6_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC7
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC7    ( ( XI3CPSX_BASEADDR ) + 0x000002D8 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC7_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC8
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC8    ( ( XI3CPSX_BASEADDR ) + 0x000002DC )
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC8_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC9
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC9    ( ( XI3CPSX_BASEADDR ) + 0x000002E0 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC9_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC10
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC10    ( ( XI3CPSX_BASEADDR ) + 0x000002E4 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC10_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_DEV_ADDR_TABLE_LOC11
 */
#define XI3CPSX_DEV_ADDR_TABLE_LOC11    ( ( XI3CPSX_BASEADDR ) + 0x000002E8 )
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_FULLMASK     0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_FULLRWMASK  0xe0ff007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_LEGACY_I2C_DEVICE_SHIFT   31
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_LEGACY_I2C_DEVICE_WIDTH   1
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_LEGACY_I2C_DEVICE_MASK    0x80000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_LEGACY_I2C_DEVICE_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_NACK_RETRY_CNT_SHIFT   29
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_NACK_RETRY_CNT_WIDTH   2
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_NACK_RETRY_CNT_MASK    0x60000000
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_NACK_RETRY_CNT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_DYNAMIC_ADDR_SHIFT   16
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_DYNAMIC_ADDR_WIDTH   8
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_DYNAMIC_ADDR_MASK    0x00ff0000
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_DYNAMIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_STATIC_ADDR_SHIFT   0
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_STATIC_ADDR_WIDTH   7
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_STATIC_ADDR_MASK    0x0000007f
#define XI3CPSX_DEV_ADDR_TABLE_LOC11_DEV_STATIC_ADDR_DEFVAL  0x0

/**
 * Register: XI3CPSX_EXT_REG_1
 */
#define XI3CPSX_EXT_REG_1    ( ( XI3CPSX_BASEADDR ) + 0x00004000 )
#define XI3CPSX_EXT_REG_1_FULLMASK     0x3fffffff
#define XI3CPSX_EXT_REG_1_FULLRWMASK  0x3feffffd
#define XI3CPSX_EXT_REG_1_DEFVAL   0x2

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_SLV_CLK_DATA_TURN_TIME_SHIFT   27
#define XI3CPSX_EXT_REG_1_SLV_CLK_DATA_TURN_TIME_WIDTH   3
#define XI3CPSX_EXT_REG_1_SLV_CLK_DATA_TURN_TIME_MASK    0x38000000
#define XI3CPSX_EXT_REG_1_SLV_CLK_DATA_TURN_TIME_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_SLV_MAX_WR_SPEED_SHIFT   24
#define XI3CPSX_EXT_REG_1_SLV_MAX_WR_SPEED_WIDTH   3
#define XI3CPSX_EXT_REG_1_SLV_MAX_WR_SPEED_MASK    0x07000000
#define XI3CPSX_EXT_REG_1_SLV_MAX_WR_SPEED_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_SLV_MAX_RD_SPEED_SHIFT   21
#define XI3CPSX_EXT_REG_1_SLV_MAX_RD_SPEED_WIDTH   3
#define XI3CPSX_EXT_REG_1_SLV_MAX_RD_SPEED_MASK    0x00e00000
#define XI3CPSX_EXT_REG_1_SLV_MAX_RD_SPEED_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_EXT_REG_1_WAKEUP_SHIFT   20
#define XI3CPSX_EXT_REG_1_WAKEUP_WIDTH   1
#define XI3CPSX_EXT_REG_1_WAKEUP_MASK    0x00100000
#define XI3CPSX_EXT_REG_1_WAKEUP_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_INST_ID_SHIFT   16
#define XI3CPSX_EXT_REG_1_INST_ID_WIDTH   4
#define XI3CPSX_EXT_REG_1_INST_ID_MASK    0x000f0000
#define XI3CPSX_EXT_REG_1_INST_ID_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_SHIFT   9
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_WIDTH   7
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_MASK    0x0000fe00
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_EN_SHIFT   8
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_EN_WIDTH   1
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_EN_MASK    0x00000100
#define XI3CPSX_EXT_REG_1_STATIC_ADDR_EN_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_PENDING_INT_SHIFT   4
#define XI3CPSX_EXT_REG_1_PENDING_INT_WIDTH   4
#define XI3CPSX_EXT_REG_1_PENDING_INT_MASK    0x000000f0
#define XI3CPSX_EXT_REG_1_PENDING_INT_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_ACT_MODE_SHIFT   2
#define XI3CPSX_EXT_REG_1_ACT_MODE_WIDTH   2
#define XI3CPSX_EXT_REG_1_ACT_MODE_MASK    0x0000000c
#define XI3CPSX_EXT_REG_1_ACT_MODE_DEFVAL  0x0

/* access_type: ro */
#define XI3CPSX_EXT_REG_1_I2C_GLITCH_FILTER_EN_SHIFT   1
#define XI3CPSX_EXT_REG_1_I2C_GLITCH_FILTER_EN_WIDTH   1
#define XI3CPSX_EXT_REG_1_I2C_GLITCH_FILTER_EN_MASK    0x00000002
#define XI3CPSX_EXT_REG_1_I2C_GLITCH_FILTER_EN_DEFVAL  0x1

/* access_type: rw */
#define XI3CPSX_EXT_REG_1_MODE_I2C_SHIFT   0
#define XI3CPSX_EXT_REG_1_MODE_I2C_WIDTH   1
#define XI3CPSX_EXT_REG_1_MODE_I2C_MASK    0x00000001
#define XI3CPSX_EXT_REG_1_MODE_I2C_DEFVAL  0x0

/**
 * Register: XI3CPSX_EXT_REG_2
 */
#define XI3CPSX_EXT_REG_2    ( ( XI3CPSX_BASEADDR ) + 0x00004004 )
#define XI3CPSX_EXT_REG_2_FULLMASK     0xffffffff
#define XI3CPSX_EXT_REG_2_FULLRWMASK  0xffffffff
#define XI3CPSX_EXT_REG_2_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_2_SLV_PID_L_SHIFT   0
#define XI3CPSX_EXT_REG_2_SLV_PID_L_WIDTH   32
#define XI3CPSX_EXT_REG_2_SLV_PID_L_MASK    0xffffffff
#define XI3CPSX_EXT_REG_2_SLV_PID_L_DEFVAL  0x0

/**
 * Register: XI3CPSX_EXT_REG_3
 */
#define XI3CPSX_EXT_REG_3    ( ( XI3CPSX_BASEADDR ) + 0x00004008 )
#define XI3CPSX_EXT_REG_3_FULLMASK     0x00ffffff
#define XI3CPSX_EXT_REG_3_FULLRWMASK  0x00ffffff
#define XI3CPSX_EXT_REG_3_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_3_SLV_DCR_SHIFT   16
#define XI3CPSX_EXT_REG_3_SLV_DCR_WIDTH   8
#define XI3CPSX_EXT_REG_3_SLV_DCR_MASK    0x00ff0000
#define XI3CPSX_EXT_REG_3_SLV_DCR_DEFVAL  0x0

/* access_type: rw */
#define XI3CPSX_EXT_REG_3_SLV_PID_U_SHIFT   0
#define XI3CPSX_EXT_REG_3_SLV_PID_U_WIDTH   16
#define XI3CPSX_EXT_REG_3_SLV_PID_U_MASK    0x0000ffff
#define XI3CPSX_EXT_REG_3_SLV_PID_U_DEFVAL  0x0

/**
 * Register: XI3CPSX_I3C_GLITCHFILTER_CTRL
 */
#define XI3CPSX_I3C_GLITCHFILTER_CTRL    ( ( XI3CPSX_BASEADDR ) + 0x0000400C )
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_FULLMASK     0x0000003f
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_FULLRWMASK  0x0000003f
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_DEFVAL   0x10

/* access_type: rw */
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_GF_COUNTER_SHIFT   1
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_GF_COUNTER_WIDTH   5
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_GF_COUNTER_MASK    0x0000003e
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_GF_COUNTER_DEFVAL  0x8

/* access_type: rw */
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_USE_GF_SHIFT   0
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_USE_GF_WIDTH   1
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_USE_GF_MASK    0x00000001
#define XI3CPSX_I3C_GLITCHFILTER_CTRL_USE_GF_DEFVAL  0x0

/**
 * Register: XI3CPSX_ECO
 */
#define XI3CPSX_ECO    ( ( XI3CPSX_BASEADDR ) + 0x00004010 )
#define XI3CPSX_ECO_FULLMASK     0xffffffff
#define XI3CPSX_ECO_FULLRWMASK  0xffffffff
#define XI3CPSX_ECO_DEFVAL   0x0

/* access_type: rw */
#define XI3CPSX_ECO_FIELD_NAME_SHIFT   0
#define XI3CPSX_ECO_FIELD_NAME_WIDTH   32
#define XI3CPSX_ECO_FIELD_NAME_MASK    0xffffffff
#define XI3CPSX_ECO_FIELD_NAME_DEFVAL  0x0

/**
 * Register: XI3CPSX_EXT_DEBUG_REG_1
 */
#define XI3CPSX_EXT_DEBUG_REG_1    ( ( XI3CPSX_BASEADDR ) + 0x00004014 )
#define XI3CPSX_EXT_DEBUG_REG_1_FULLMASK     0xffffffff
#define XI3CPSX_EXT_DEBUG_REG_1_FULLRWMASK  0x00000000
#define XI3CPSX_EXT_DEBUG_REG_1_DEFVAL   0x80005500

/* access_type: ro */
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_31_8_SHIFT   8
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_31_8_WIDTH   24
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_31_8_MASK    0xffffff00
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_31_8_DEFVAL  0x800055

/* access_type: ro */
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_7_0_SHIFT   0
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_7_0_WIDTH   8
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_7_0_MASK    0x000000ff
#define XI3CPSX_EXT_DEBUG_REG_1_DEBUG_PORT_7_0_DEFVAL  0x0

/**
 * Register: XI3CPSX_EXT_DEBUG_REG_2
 */
#define XI3CPSX_EXT_DEBUG_REG_2    ( ( XI3CPSX_BASEADDR ) + 0x00004018 )
#define XI3CPSX_EXT_DEBUG_REG_2_FULLMASK     0x0000ffff
#define XI3CPSX_EXT_DEBUG_REG_2_FULLRWMASK  0x00000000
#define XI3CPSX_EXT_DEBUG_REG_2_DEFVAL   0x0

/* access_type: ro */
#define XI3CPSX_EXT_DEBUG_REG_2_DEBUG_PORT_47_32_SHIFT   0
#define XI3CPSX_EXT_DEBUG_REG_2_DEBUG_PORT_47_32_WIDTH   16
#define XI3CPSX_EXT_DEBUG_REG_2_DEBUG_PORT_47_32_MASK    0x0000ffff
#define XI3CPSX_EXT_DEBUG_REG_2_DEBUG_PORT_47_32_DEFVAL  0x0

#ifdef __cplusplus
}
#endif

#endif /* XI3CPSX_HW_H_ */
/** @} */
