/*******************************************************************************
* Copyright (C) 2017 - 2020 Xilinx, Inc.  All rights reserved.
* SPDX-License-Identifier: MIT
*******************************************************************************/

/******************************************************************************/
/**
 *
 * @file xdpdma_intr.c
 *
 * This file contains functions related to XDpPsu interrupt handling.
 *
 * @note	None.
 *
 * <pre>
 * MODIFICATION HISTORY:
 *
 * Ver   Who  Date     Changes
 * ----- ---- -------- -----------------------------------------------
 * 1.0   aad  01/17/17 Initial release.
 * </pre>
 *
*******************************************************************************/

/******************************* Include Files ********************************/
#include "xdpdma.h"


/*************************************************************************/
/**
 *
 * This function enables the interrupts that are required.
 *
 * @param    InstancePtr is pointer to the instance of DPDMA
 *
 * @param    Mask is mask to XDPDMA_IEN register
 *
 * @return   None.
 *
 * @note     None.
 *
 * **************************************************************************/
void XDpDma_InterruptEnable(XDpDma *InstancePtr, u32 Mask)
{
	XDpDma_WriteReg(InstancePtr->Config.BaseAddr, XDPDMA_IEN, Mask);
}

/*************************************************************************/
/**
 *
 * This function handles the interrupts generated by DPDMA
 *
 * @param    InstancePtr is pointer to the instance of the DPDMA
 *
 * @return   None.
 *
 * @note     None.
 *
 * **************************************************************************/
void XDpDma_InterruptHandler(XDpDma *InstancePtr)
{
	u32 RegVal;
	RegVal = XDpDma_ReadReg(InstancePtr->Config.BaseAddr,
				XDPDMA_ISR);
	if ((RegVal & XDPDMA_ISR_VSYNC_INT_MASK) != 0U) {
		XDpDma_VSyncHandler(InstancePtr);
	}

	if ((RegVal & XDPDMA_ISR_DSCR_DONE4_MASK) != 0U) {
		XDpDma_SetChannelState(InstancePtr, AudioChan0, XDPDMA_DISABLE);
		InstancePtr->Audio[0].Current = NULL;
		XDpDma_WriteReg(InstancePtr->Config.BaseAddr, XDPDMA_ISR,
				XDPDMA_ISR_DSCR_DONE4_MASK);
	}

	if ((RegVal & XDPDMA_ISR_DSCR_DONE5_MASK) != 0U) {
		XDpDma_SetChannelState(InstancePtr, AudioChan1, XDPDMA_DISABLE);
		InstancePtr->Audio[1].Current = NULL;
		XDpDma_WriteReg(InstancePtr->Config.BaseAddr, XDPDMA_ISR,
				XDPDMA_ISR_DSCR_DONE5_MASK);
	}
}

/*************************************************************************/
/**
 *
 * This function handles frame new frames on VSync
 *
 * @param    InstancePtr is pointer to the instance of the driver.
 *
 * @return   None.
 *
 * @note     None.
 *
 * **************************************************************************/
void XDpDma_VSyncHandler(XDpDma *InstancePtr)
{
	Xil_AssertVoid(InstancePtr != NULL);

	/* Video Channel Trigger/Retrigger Handler */
	if(InstancePtr->Video.TriggerStatus == XDPDMA_TRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, VideoChan);
		XDpDma_SetChannelState(InstancePtr, VideoChan,
				       XDPDMA_ENABLE);
		XDpDma_Trigger(InstancePtr, VideoChan);
	}
	else if(InstancePtr->Video.TriggerStatus == XDPDMA_RETRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, VideoChan);
		XDpDma_ReTrigger(InstancePtr, VideoChan);
	} else {
		/* Do nothing if TriggerStatus is XDPDMA_TRIGGER_DONE or
		 * XDPDMA_RETRIGGER_DONE
		 */
	}

	/* Graphics Channel Trigger/Retrigger Handler */
	if(InstancePtr->Gfx.TriggerStatus == XDPDMA_TRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, GraphicsChan);
		XDpDma_SetChannelState(InstancePtr, GraphicsChan,
				       XDPDMA_ENABLE);
		XDpDma_Trigger(InstancePtr, GraphicsChan);
	}
	else if(InstancePtr->Gfx.TriggerStatus == XDPDMA_RETRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, GraphicsChan);
		XDpDma_ReTrigger(InstancePtr, GraphicsChan);
	} else {
		/* Do nothing if TriggerStatus is XDPDMA_TRIGGER_DONE or
		 * XDPDMA_RETRIGGER_DONE
		 */
	}

	/* Audio Channel 0 Trigger Handler */
	if(InstancePtr->Audio[0].TriggerStatus == XDPDMA_TRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, AudioChan0);
		XDpDma_SetChannelState(InstancePtr, AudioChan0,
				       XDPDMA_ENABLE);
		XDpDma_Trigger(InstancePtr, AudioChan0);
	}

	/* Audio Channel 1 Trigger Handler */
	if(InstancePtr->Audio[1].TriggerStatus == XDPDMA_TRIGGER_EN) {
		XDpDma_SetupChannel(InstancePtr, AudioChan1);
		XDpDma_SetChannelState(InstancePtr, AudioChan1,
				       XDPDMA_ENABLE);
		XDpDma_Trigger(InstancePtr, AudioChan1);
	}
	/* Clear VSync Interrupt */
	XDpDma_WriteReg(InstancePtr->Config.BaseAddr, XDPDMA_ISR,
			XDPDMA_ISR_VSYNC_INT_MASK);
}
