/******************************************************************************
* Copyright (C) 2021 Xilinx, Inc. All rights reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

/*****************************************************************************/
/**
*
* @file xrfclk_LMX_conf.h
* @addtogroup xrfclk_LMX_conf_v1_2
* @{
*
* Contains the configuration data for LMX.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who    Date     Changes
* ----- ---    -------- -----------------------------------------------
* 1.0   dc     08/28/19 Initial version
* 1.1   dc     11/21/19 Remove xil dependencies from linux build
*       dc     11/25/19 update LMX and LMK configs
*       dc     12/05/19 adjust LMX and LMK configs to a rftool needs
* 1.3   dc     03/10/20 update LMK/LMX config for MTS
* 1.4   dc     03/30/20 new LMX config suppressing RF noise on dual output
* 1.5   dc     13/05/21 Make clear LMK/LMX configuration comments
*
* </pre>
*
******************************************************************************/
#ifndef __XRFCLK_LMX_CONF_H_
#define __XRFCLK_LMX_CONF_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "xrfclk.h"

#ifdef XPS_BOARD_ZCU111

/* ADC Frequency list for LMX2594(A/B) PLL */
const u8 ADC_FREQ_LIST[LMX_ADC_NUM][FREQ_LIST_STR_SIZE] = {
	"102.40",  "204.80",  "245.76",  "409.60",  "491.52",  "737.28",
	"1474.56", "1966_08", "2048.00", "2457.60", "2949.12", "3072.00",
	"3194.88", "3276.80", "3686.40", "3932.16", "4096.00"
};

/* DAC Frequency list for LMX2594(C) PLL */
const u8 DAC_FREQ_LIST[LMX_DAC_NUM][FREQ_LIST_STR_SIZE] = {
	"102.40",  "204.80",  "245.76",  "409.60",  "491.52",  "737.28",
	"1474.56", "1966_08", "2048.00", "2457.60", "2949.12", "3072.00",
	"3194.88", "3276.80", "3686.40", "3932.16", "4096.00", "4423.68",
	"4669.44", "4915.20", "5734.40", "5898.24", "6144.00", "6389.76",
	"6400.00", "6553.60"
};

/* Frequency values for LMX2594(A/B/C) PLL */
const u32 LMX2594[][LMX2594_COUNT] = {
	/* 0 PLL_102_4_MHZ LMK output 122.88MHz, LMX2594 ouput 102.4 MHz. Use
	   with internal PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0AC0, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240050,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 1 PLL_204_8_MHZ LMK output 122.88MHz, LMX2594 ouput 204.8. Use with
	   internal PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0A00, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240050,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 2 PLL_245_76_MHZ: LMK output 122.88MHz, LMX2594 ouput 245.76. Use
	   with internal PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B09C0, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240040,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 3 PLL_409_6_MHZ: LMK output 122.88MHz, LMX2594 ouput 409.6. Use with
	   external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0980, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240050,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 4 PLL_491_52_MHZ: LMK output 122.88MHz, LMX2594 ouput 491.52. Use
	   with internal PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0940, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240040,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 5 PLL_737_28_MHZ: LMK output 122.88MHz, LMX2594 ouput 737.28. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0900, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240048,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 6 PLL_1474_56_MHZ: LMK output 122M8, LMX outputs 1.47456 MHz. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0880, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240048,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },
	/* program R0 one additional time as specified by the reset sequence */

	/* 7 PLL_1966_08_MHZ: LMK output 122.88MHz, LMX2594 ouput 1966.08. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240040,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 8 PLL_2048_MHZ: LMK output 122.88MHz, LMX2594 ouput 2048. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0002,
	  0x2A0000, 0x290000, 0x280000, 0x270003, 0x260000, 0x250304, 0x240042,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 9 PLL_2457_6_MHZ: LMK output 122.88MHz, LMX2594 ouput 2457.6. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240050,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 10 PLL_2949_12_MHZ: LMK output 122.88MHz, LMX2594 ouput 2949.12. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240060,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 11 PLL_3072_MHZ: LMK output 122.88MHz, LMX2594 ouput 3072. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240064,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 12 PLL_3194_88_MHZ LMK output 122.88MHz, LMX2594 ouput 3194.88. Use
	   with internal PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240068,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 13 PLL_3276_8_MHZ: LMK output 122.88MHz, LMX2594 ouput 3276.8. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0002,
	  0x2A0000, 0x290000, 0x280000, 0x270003, 0x260000, 0x250404, 0x24006A,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 14 PLL_3686_4_MHZ: LMK output 122.88MHz, LMX2594 ouput 3686.4. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240078,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 15 PLL_3932_16_MHZ: LMK output 122M8, LMX outputs 3.92316 GHz */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240040,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 16 PLL_4096_MHZ: LMK output 122.88MHz, LMX2594 ouput 4096. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0002,
	  0x2A0000, 0x290000, 0x280000, 0x270003, 0x260000, 0x250304, 0x240042,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 17 PLL_4423_68_MHZ: LMK output 122.88MHz, LMX2594 ouput 4423.68. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240048,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 18 PLL_4669_44_MHZ: LMK output 122.88MHz, LMX2594 ouput 4669.44. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x24004C,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 19 PLL_4915_2_MHZ: LMK output 122.88MHz, LMX2594 ouput 4915.2. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250304, 0x240050,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 20 PLL_5734_4_MHZ: LMK output 122.88MHz, LMX2594 ouput 5734.4. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0001,
	  0x2A0000, 0x290000, 0x280000, 0x270003, 0x260000, 0x250404, 0x24005D,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 21 PLL_5898_24_MHZ: LMK output 122.88MHz, LMX2594 ouput 5898.24. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240060,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 22 PLL_6144_MHZ: LMK output 122.88MHz, LMX2594 ouput 6144. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240064,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 23 PLL_6389_76_MHZ: LMK output 122.88MHz, LMX2594 ouput 6389.76. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250404, 0x240068,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 24 PLL_6400_MHZ: LMK output 122.88MHz, LMX2594 ouput 6400. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0001,
	  0x2A0000, 0x290000, 0x280000, 0x270006, 0x260000, 0x250404, 0x240068,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C },

	/* 25 PLL_6553_6_MHZ: LMK output 122.88MHz, LMX2594 ouput 6553.6. Use
	   with external PLL */
	{ 0x700000, 0x6F0000, 0x6E0000, 0x6D0000, 0x6C0000, 0x6B0000, 0x6A0000,
	  0x690021, 0x680000, 0x670000, 0x663F80, 0x650011, 0x640000, 0x630000,
	  0x620200, 0x610888, 0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000,
	  0x5B0000, 0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
	  0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026, 0x4E0003,
	  0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F, 0x480001, 0x470081,
	  0x46C350, 0x450000, 0x4403E8, 0x430000, 0x4201F4, 0x410000, 0x401388,
	  0x3F0000, 0x3E0322, 0x3D00A8, 0x3C0000, 0x3B0001, 0x3A8001, 0x390020,
	  0x380000, 0x370000, 0x360000, 0x350000, 0x340820, 0x330080, 0x320000,
	  0x314180, 0x300300, 0x2F0300, 0x2E07FC, 0x2DC0CC, 0x2C0C23, 0x2B0000,
	  0x2A4000, 0x290000, 0x280000, 0x270001, 0x266000, 0x250404, 0x24006A,
	  0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC, 0x1E318C, 0x1D318C,
	  0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624, 0x18071A, 0x17007C, 0x160001,
	  0x150401, 0x14E048, 0x1327B7, 0x120064, 0x11012C, 0x100080, 0x0F064F,
	  0x0E1E70, 0x0D4000, 0x0C5001, 0x0B0018, 0x0A10D8, 0x090604, 0x082000,
	  0x0740B2, 0x06C802, 0x0500C8, 0x040A43, 0x030642, 0x020500, 0x010808,
	  0x00249C }
	/* program R0 one additional time as specified by the reset sequence */
};

#else

/* ADC Frequency list for LMX2594(A) PLL */
const u8 ADC_FREQ_LIST[LMX_ADC_NUM][FREQ_LIST_STR_SIZE] = {
	"LMX2594_REF-245M76__491M52",  "LMX2594_REF-245M76__614M4",
	"LMX2594_REF-245M76__737M28",  "LMX2594_REF-245M76__983M04",
	"LMX2594_REF-245M76__1966M08", "LMX2594_REF-245M76__2211M84",
	"LMX2594_REF-245M76__2457M6",  "LMX2594_REF-250M__2500M",
};

/* DAC Frequency list for LMX2594(A) PLL */
const u8 DAC_FREQ_LIST[LMX_DAC_NUM][FREQ_LIST_STR_SIZE] = {
	"LMX2594_REF-245M76__491M52",  "LMX2594_REF-245M76__614M4",
	"LMX2594_REF-245M76__737M28",  "LMX2594_REF-245M76__983M04",
	"LMX2594_REF-245M76__1966M08", "LMX2594_REF-245M76__2211M84",
	"LMX2594_REF-245M76__2457M6",  "LMX2594_REF-250M__2500M",
	"LMX2594_REF-250M__3000M",     "LMX2594_REF-245M76__3072M",
	"LMX2594_REF-245M76__3932M16", "LMX2594_REF-245M76__4915M2",
	"LMX2594_REF-250M__5000M",     "LMX2594_REF-245M76__5898M24",
	"LMX2594_REF-250M__6000M",     "LMX2594_REF-245M76__6144M",
	"LMX2594_REF-245M76__6881M28", "LMX2594_REF-250M__7000M",
	"LMX2594_REF-245M76__7372M8",  "LMX2594_REF-245M76__7864M32",
	"LMX2594_REF-250M__8000M",     "LMX2594_REF-245M76__8847M36",
	"LMX2594_REF-245M76__9830M4",  "LMX2594_REF-250M__10000M",
};

/* Frequency configurations for LMX2594(A/B) PLL */
/* Definition of the file name:
   REF corresponds to the input expected by the LMX 2594
   Last frequency is the output of the LMX
   Example LMX2594_REF-245M76__491M52.txt means LMX reference input expected is 245.76MHz, LMX output 491.52MHz
*/

const u32 LMX2594[][LMX2594_COUNT] = {

	/* LMX2594_REF-245M76__491M52.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00E5, 0x4D0000, 0x4C000C, 0x4B0940, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110117, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__614M4.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0135, 0x4D0000, 0x4C000C, 0x4B0940, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110141, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__737M28.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E010D, 0x4D0000, 0x4C000C, 0x4B0900, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240168, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D048, 0x1327B7,
		0x120064, 0x11014E, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__983M04.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00E5, 0x4D0000, 0x4C000C, 0x4B08C0, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110117, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__1966M08.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00E5, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110117, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__2211M84.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E010D, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240168, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D048, 0x1327B7,
		0x120064, 0x11014E, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__2457M6.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0135, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110141, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__2500M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0105, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110130, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__3000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0001, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x2401E0, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14E048, 0x1327B7,
		0x120064, 0x110064, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__3072M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0111, 0x4D0000, 0x4C000C, 0x4B0840, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x2401F4, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F43EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14E848, 0x1327B7,
		0x120064, 0x1100BA, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__3932M16.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00E5, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110117, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__4915M2.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0135, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110141, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__5000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0105, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110130, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__5898M24.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0031, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x2401E0, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14E048, 0x1327B7,
		0x120064, 0x11010C, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__6000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0001, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x2401E0, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14E048, 0x1327B7,
		0x120064, 0x110064, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__6144M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0111, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x2401F4, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14E848, 0x1327B7,
		0x120064, 0x1100BA, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__6881M28.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0035, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250204,
		0x240230, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14F048, 0x1327B7,
		0x120064, 0x1100AE, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__7000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0143, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250204,
		0x240230, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14F848, 0x1327B7,
		0x120064, 0x11013C, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__7372M8.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E006F, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FC, 0x2DC0DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250204,
		0x240258, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14F848, 0x1327B7,
		0x120064, 0x110106, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__7864M32.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00E5, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FD, 0x2DC8DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110117, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__8000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E00BF, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FD, 0x2DC8DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240140, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14C848, 0x1327B7,
		0x120064, 0x110110, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__8847M36.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E010D, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FD, 0x2DC8DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240168, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D048, 0x1327B7,
		0x120064, 0x11014E, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-245M76__9830M4.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0135, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FD, 0x2DC8DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110141, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},

	/* LMX2594_REF-250M__10000M.txt */
	{
		0x000002, 0x000000, 0x700000, 0x6F0000, 0x6E0000, 0x6D0000,
		0x6C0000, 0x6B0000, 0x6A0000, 0x690021, 0x680000, 0x670000,
		0x663F80, 0x650011, 0x640000, 0x630000, 0x620200, 0x610888,
		0x600000, 0x5F0000, 0x5E0000, 0x5D0000, 0x5C0000, 0x5B0000,
		0x5A0000, 0x590000, 0x580000, 0x570000, 0x560000, 0x55D300,
		0x540001, 0x530000, 0x521E00, 0x510000, 0x506666, 0x4F0026,
		0x4E0105, 0x4D0000, 0x4C000C, 0x4B0800, 0x4A0000, 0x49003F,
		0x480001, 0x470081, 0x46C350, 0x450000, 0x4403E8, 0x430000,
		0x4201F4, 0x410000, 0x401388, 0x3F0000, 0x3E0322, 0x3D00A8,
		0x3C0000, 0x3B0001, 0x3A8001, 0x390020, 0x380000, 0x370000,
		0x360000, 0x350000, 0x340820, 0x330080, 0x320000, 0x314180,
		0x300300, 0x2F0300, 0x2E07FD, 0x2DC8DF, 0x2C1FA0, 0x2B0000,
		0x2A0000, 0x290000, 0x280000, 0x270001, 0x260000, 0x250104,
		0x240190, 0x230004, 0x220000, 0x211E21, 0x200393, 0x1F03EC,
		0x1E318C, 0x1D318C, 0x1C0488, 0x1B0002, 0x1A0DB0, 0x190624,
		0x18071A, 0x17007C, 0x160001, 0x150401, 0x14D848, 0x1327B7,
		0x120064, 0x110130, 0x100080, 0x0F064F, 0x0E1E40, 0x0D4000,
		0x0C5001, 0x0B00A8, 0x0A10D8, 0x090604, 0x082000, 0x0740B2,
		0x06C802, 0x0500C8, 0x040C43, 0x030642, 0x020500, 0x010809,
		0x00241C, 0x00241C,
	},
};
#endif

#ifdef __cplusplus
}
#endif

#endif
/** @} */
