# Copyright (C) 2023 - 2024 Advanced Micro Devices, Inc.  All rights reserved.
# SPDX-License-Identifier: MIT
cmake_minimum_required(VERSION 3.3)

find_package(common)
set(lwip220_api_mode RAW_API CACHE STRING "Mode of operation for lwIP (RAW API/Sockets API)")
set_property(CACHE lwip220_api_mode PROPERTY STRINGS RAW_API SOCKET_API)
option(lwip220_no_sys_no_timers "Drops support for sys_timeout when NO_SYS==1" ON)
set(lwip220_socket_mode_thread_prio 2 CACHE STRING "Priority of threads in socket mode")
option(lwip220_tcp_keepalive "Enable keepalive processing with default interval" OFF)
set(sgmii_fixed_link 0 CACHE STRING "Enable fixed link for GEM SGMII at 1Gbps")
set_property(CACHE sgmii_fixed_link PROPERTY STRINGS 0 1)

set(lwip220_n_tx_descriptors 64 CACHE STRING "Number of TX Buffer Descriptors to be used in SDMA mode")
set(lwip220_n_rx_descriptors 64 CACHE STRING "Number of RX Buffer Descriptors to be used in SDMA mode")
set(lwip220_n_tx_coalesce 1 CACHE STRING "Setting for TX Interrupt coalescing.")
set(lwip220_n_rx_coalesce 1 CACHE STRING "Setting for RX Interrupt coalescing.")
option(lwip220_temac_tcp_rx_checksum_offload "Offload TCP Receive checksum calculation (hardware support required)" OFF)
option(lwip220_temac_tcp_tx_checksum_offload "Offload TCP Transmit checksum calculation (hardware support required)" OFF)
option(lwip220_temac_tcp_ip_rx_checksum_offload "Offload TCP and IP Receive checksum calculation (hardware support required)" OFF)
option(lwip220_temac_tcp_ip_tx_checksum_offload "Offload TCP and IP Transmit checksum calculation (hardware support required)" OFF)
set(lwip220_temac_phy_link_speed CONFIG_LINKSPEED_AUTODETECT CACHE STRING "link speed as negotiated by the PHY")
set_property(CACHE lwip220_temac_phy_link_speed PROPERTY STRINGS CONFIG_LINKSPEED10 CONFIG_LINKSPEED100 CONFIG_LINKSPEED1000 CONFIG_LINKSPEED_AUTODETECT)
option(lwip220_temac_use_jumbo_frames "use jumbo frames" OFF)
set(lwip220_temac_emac_number 0 CACHE STRING "Zynq Ethernet Interface number")

set(lwip220_mem_size 131072 CACHE STRING "Size of the heap memory (bytes).")
set(lwip220_memp_n_pbuf 16 CACHE STRING "Number of memp struct pbufs. Set this high if application sends lot of data out of ROM")
set(lwip220_memp_n_udp_pcb 4 CACHE STRING "Number of active UDP PCBs. One per active UDP connection")
set(lwip220_memp_n_tcp_pcb 32 CACHE STRING "Number of active TCP PCBs. One per active TCP connection")
set(lwip220_memp_n_tcp_pcb_listen 8 CACHE STRING "Number of listening TCP connections")
set(lwip220_memp_n_tcp_seg 256 CACHE STRING "Number of simultaneously queued TCP segments")
set(lwip220_memp_n_sys_timeout 8 CACHE STRING "Number of simultaneously active timeouts")
set(lwip220_memp_num_netbuf 8 CACHE STRING "Number of struct netbufs (socket mode only)")
set(lwip220_memp_num_netconn 16 CACHE STRING "Number of struct netconns (socket mode only)")
set(lwip220_memp_num_api_msg 16 CACHE STRING "Number of api msg structures (socket mode only)")
set(lwip220_memp_num_tcpip_msg 64 CACHE STRING "Number of tcpip msg structures (socket mode only)")

set(lwip220_pbuf_pool_size 256 CACHE STRING "Number of buffers in pbuf pool.")
set(lwip220_pbuf_pool_bufsize 1700 CACHE STRING "Size of each pbuf in pbuf pool.")
set(lwip220_pbuf_link_hlen 16 CACHE STRING "Number of bytes that should be allocated for a link level header.")

option(lwip220_lwip_tcpip_core_locking_input "TCPIP input core locking" OFF)
set(lwip220_tcpip_mbox_size 200 CACHE STRING "Size of TCPIP mbox queue.")
set(lwip220_default_tcp_recvmbox_size 200 CACHE STRING "Size of TCP receive mbox queue.")
set(lwip220_default_udp_recvmbox_size 100 CACHE STRING "Size of UDP receive mbox queue.")

set(lwip220_arp_table_size 10 CACHE STRING "Number of active hardware address IP address pairs cached.")
set(lwip220_arp_queueing 1 CACHE STRING "If enabled outgoing packets are queued during hardware address resolution.")
set_property(CACHE lwip220_arp_queueing PROPERTY STRINGS 0 1)

set(lwip220_ip_forward 0 CACHE STRING "Enable forwarding IP packets across network interfaces.")
set_property(CACHE lwip220_ip_forward PROPERTY STRINGS 0 1)
set(lwip220_ip_options 0 CACHE STRING "1 = IP options are allowed (but not parsed). 0 = packets with IP options are dropped")
set_property(CACHE lwip220_ip_options PROPERTY STRINGS 0 1)
set(lwip220_ip_reassembly 1 CACHE STRING "Reassemble incoming fragmented IP packets")
set_property(CACHE lwip220_ip_reassembly PROPERTY STRINGS 0 1)
set(lwip220_ip_frag 1 CACHE STRING "Fragment outgoing IP packets if their size exceeds MTU")
set_property(CACHE lwip220_ip_frag PROPERTY STRINGS 0 1)
set(lwip220_ip_reass_max_pbufs 128 CACHE STRING "Reassembly PBUF Queue Length")
set(lwip220_ip_frag_max_mtu 1500 CACHE STRING "Assumed max MTU on any interface for IP frag buffer")
set(lwip220_ip_default_ttl 255 CACHE STRING "Global default TTL used by transport layers")

option(lwip220_icmp_options "ICMP Options" ON)
set(lwip220_icmp_ttl 255 CACHE STRING "ICMP TTL value")

option(lwip220_ipv6_options "IPv6 Options" ON)
option(lwip220_ipv6_enable "IPv6 enable value" OFF)

option(lwip220_igmp_options "IGMP Options" OFF)

option(lwip220_udp_options "UDP Options" ON)
option(lwip220_udp "Is UDP required" ON)
set(lwip220_udp_ttl 255 CACHE STRING "UDP TTL value")
option(lwip220_udp_block_tx "Application sending a UDP packet blocks till the pkt is txed" OFF)

option(lwip220_tcp_options "Is TCP required ?" ON)
option(lwip220_tcp "Is TCP required ?" ON)
set(lwip220_tcp_wnd 2048 CACHE STRING "TCP Window (bytes)")
set(lwip220_tcp_snd_buf 8192 CACHE STRING "TCP sender buffer space (bytes)")
set(lwip220_tcp_mss 1460 CACHE STRING "TCP Maximum segment size (bytes)")
set(lwip220_tcp_ttl 255 CACHE STRING "TCP TTL value")
set(lwip220_tcp_maxrtx 12 CACHE STRING "TCP Maximum retransmission value")
set(lwip220_tcp_synmaxrtx 4 CACHE STRING "TCP Maximum SYN retransmission value")
set(lwip220_tcp_queue_ooseq 1 CACHE STRING "Should TCP queue segments arriving out of order. Set to 0 if your device is low on memory")
set_property(CACHE lwip220_tcp_queue_ooseq PROPERTY STRINGS 0 1)

option(lwip220_dhcp_options "Is DHCP required?" ON)
option(lwip220_dhcp "Is DHCP required?" OFF)
option(lwip220_lwip_dhcp_does_acd_check "ACD check on offered addresses?" OFF)

option(lwip220_stats_options "Turn on lwIP statistics?" ON)
option(lwip220_stats "Turn on lwIP statistics?" OFF)

option(lwip220_debug_options "Turn on lwIP Debug?" ON)
option(lwip220_debug "Turn on lwIP Debug?" OFF)
option(lwip220_ip_debug "Debug IP layer" OFF)
option(lwip220_tcp_debug "Debug TCP layer" OFF)
option(lwip220_udp_debug "Debug UDP layer" OFF)
option(lwip220_icmp_debug "Debug ICMP protocol" OFF)
option(lwip220_igmp_debug "Debug IGMP protocol" OFF)
option(lwip220_netif_debug "Debug network interface layer" OFF)
option(lwip220_sys_debug "Debug sys arch layer" OFF)
option(lwip220_pbuf_debug "Debug pbuf layer" OFF)
option(lwip220_dhcp_debug "Debug dhcp protocol" OFF)
option(lwip220_acd_debug "Debug acd feature" OFF)

if("${lwip220_api_mode}" STREQUAL RAW_API)
    set(NO_SYS " ")
endif()

if(${lwip220_no_sys_no_timers})
    set(NO_SYS_NO_TIMERS " ")
endif()

if(${lwip220_tcp_keepalive})
    set(LWIP_TCP_KEEPALIVE " ")
endif()

if(${sgmii_fixed_link})
    set(SGMII_FIXED_LINK   " ")
endif()

if(("${CMAKE_SYSTEM_NAME}" STREQUAL "FreeRTOS") AND
    ("${lwip220_api_mode}" STREQUAL RAW_API))
    cmake_error("FreeRTOS is not compatible with RAW_API mode")
endif()


if(("${CMAKE_SYSTEM_NAME}" STREQUAL "Generic") AND
    ("${lwip220_api_mode}" STREQUAL SOCKET_API))
    cmake_error("Standalone is not compatible with SOCKET_API mode")
endif()

set(MEM_SIZE ${lwip220_mem_size})
set(MEMP_NUM_PBUF ${lwip220_memp_n_pbuf})
set(MEMP_NUM_UDP_PCB ${lwip220_memp_n_udp_pcb})
set(MEMP_NUM_TCP_PCB ${lwip220_memp_n_tcp_pcb})
set(MEMP_NUM_TCP_PCB_LISTEN ${lwip220_memp_n_tcp_pcb_listen})
set(MEMP_NUM_TCP_SEG ${lwip220_memp_n_tcp_seg})
set(MEMP_NUM_SYS_TIMEOUT ${lwip220_memp_n_sys_timeout})
set(MEMP_NUM_NETBUF ${lwip220_memp_num_netbuf})
set(MEMP_NUM_NETCONN ${lwip220_memp_num_netconn})
set(MEMP_NUM_TCPIP_MSG_API ${lwip220_memp_num_api_msg})
set(MEMP_NUM_TCPIP_MSG_INPKT ${lwip220_memp_num_tcpip_msg})

# PBUF options
set(PBUF_POOL_SIZE ${lwip220_pbuf_pool_size})
set(PBUF_POOL_BUFSIZE ${lwip220_pbuf_pool_bufsize})
set(PBUF_LINK_HLEN ${lwip220_pbuf_link_hlen})

# ARP options
set(ARP_TABLE_SIZE ${lwip220_arp_table_size})
set(ARP_QUEUEING ${lwip220_arp_queueing})

#ICMP options
set(ICMP_TTL ${lwip220_icmp_ttl})

#IPV6 options
if (${lwip220_ipv6_enable})
   set(LWIP_IPV6 " ")
endif()

#IGMP options
if (${lwip220_igmp_options})
    set(LWIP_IGMP 1)
endif()

#IP options
set(IP_OPTIONS ${lwip220_ip_options})
set(IP_FORWARD ${lwip220_ip_forward})
set(IP_REASSEMBLY ${lwip220_ip_reassembly})
set(IP_FRAG ${lwip220_ip_frag})
set(IP_REASS_MAX_PBUFS ${lwip220_ip_reass_max_pbufs})
set(IP_FRAG_MAX_MTU ${lwip220_ip_frag_max_mtu})
set(IP_DEFAULT_TTL ${lwip220_ip_default_ttl})

#UDP options
if (${lwip220_udp})
    set(LWIP_UDP 1)
else()
    set(LWIP_UDP 0)
endif()

set(UDP_TTL ${lwip220_udp_ttl})

# Check for UDP Block TX till complete
if (${LWIP_UDP} EQUAL 0)
    if (${lwip220_udp_block_tx})
        message(WARNING "UDP block on TX is set but UDP is disabled. Disabling the block on TX flag.")
        set(LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE 0)
    else()
        set(LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE 0)
    endif()
else()
    if (${lwip220_udp_block_tx})
        set(LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE 1)
    else()
        set(LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE 0)
    endif()
endif()

#TCP options
if (${lwip220_tcp})
    set(LWIP_TCP 1)
endif()
set(TCP_MSS ${lwip220_tcp_mss})
set(TCP_SND_BUF ${lwip220_tcp_snd_buf})
set(TCP_WND ${lwip220_tcp_wnd})
set(TCP_TTL ${lwip220_tcp_ttl})
set(TCP_MAXRTX ${lwip220_tcp_maxrtx})
set(TCP_SYNMAXRTX ${lwip220_tcp_synmaxrtx})
set(TCP_QUEUE_OOSEQ ${lwip220_tcp_queue_ooseq})

## Checksum handling should be based on the MAC_INSTANCE variable
list(LENGTH MAC_INSTANCES _len)
if (${_len} GREATER 1)
    list(GET MAC_INSTANCES 0 MAC_INSTANCES)
endif()

if (${CONFIG_AXIETHERNET})
    if (MAC_INSTANCES IN_LIST AXIETHERNET_NUM_DRIVER_INSTANCES)
        if (${lwip220_temac_tcp_ip_tx_checksum_offload})
            if (${axieth_txcsum} EQUAL 0x2)
                set(CHECKSUM_GEN_TCP " ")
                set(CHECKSUM_GEN_UDP " ")
                set(CHECKSUM_GEN_IP " ")
                set(LWIP_FULL_CSUM_OFFLOAD_TX 1)
            else()
                message(FATAL_ERROR "Wrong Tx checksum options. The selected Tx checksum does not match with the HW supported Tx csum offload option")
            endif()
        endif()
        if (${lwip220_temac_tcp_ip_rx_checksum_offload})
            if (${axieth_rxcsum} EQUAL 0x2)
                set(CHECKSUM_CHECK_TCP " ")
                set(CHECKSUM_CHECK_UDP " ")
                set(CHECKSUM_CHECK_IP " ")
                set(LWIP_FULL_CSUM_OFFLOAD_RX 1)
            else()
                message(FATAL_ERROR "Wrong Rx checksum options. The selected Rx checksum does not match with the HW supported Rx csum offload option")
            endif()
        endif()
        if (${lwip220_temac_tcp_tx_checksum_offload})
            if (${axieth_txcsum} EQUAL 0x1)
                set(CHECKSUM_GEN_TCP " ")
                set(LWIP_PARTIAL_CSUM_OFFLOAD_TX 1)
            else()
                message(FATAL_ERROR "Wrong Tx checksum options. The selected Tx checksum does not match with the HW supported Tx csum offload option")
            endif()
        endif()
        if (${lwip220_temac_tcp_rx_checksum_offload})
            if (${axieth_rxcsum} EQUAL 0x1)
                set(CHECKSUM_CHECK_TCP " ")
                set(LWIP_PARTIAL_CSUM_OFFLOAD_RX 1)
            else()
                message(FATAL_ERROR "Wrong Rx checksum options. The selected Rx checksum does not match with the HW supported Rx csum offload option")
            endif()
        endif()
        if (NOT ${lwip220_temac_tcp_ip_tx_checksum_offload} AND NOT ${lwip220_temac_tcp_tx_checksum_offload})
            set(CHECKSUM_GEN_TCP " ")
            set(CHECKSUM_GEN_UDP " ")
            set(CHECKSUM_GEN_IP " ")
        endif()
        if (NOT ${lwip220_temac_tcp_ip_rx_checksum_offload} AND NOT ${lwip220_temac_tcp_rx_checksum_offload})
            set(CHECKSUM_CHECK_TCP " ")
            set(CHECKSUM_CHECK_UDP " ")
            set(CHECKSUM_CHECK_IP " ")
        endif()
    endif()
endif()

if(${CONFIG_EMACLITE})
    if (MAC_INSTANCES IN_LIST EMACLITE_NUM_DRIVER_INSTANCES)
        set(CHECKSUM_GEN_TCP " ")
	set(CHECKSUM_GEN_UDP " ")
	set(CHECKSUM_GEN_IP " ")
	set(CHECKSUM_CHECK_TCP " ")
	set(CHECKSUM_CHECK_UDP " ")
	set(CHECKSUM_CHECK_IP " ")
    endif()
endif()

if (${CONFIG_EMACPS})
    if (MAC_INSTANCES IN_LIST EMACPS_NUM_DRIVER_INSTANCES)
	set(LWIP_FULL_CSUM_OFFLOAD_RX 1)
	set(LWIP_FULL_CSUM_OFFLOAD_TX 1)
        set(index 0)
        LIST_INDEX(${index} ${MAC_INSTANCES} "${EMACPS_NUM_DRIVER_INSTANCES}")
        list(GET TOTAL_EMACPS_PROP_LIST ${index} x)
        set(y ${${x}})
        list(GET y -1 phy_type)
        if (${phy_type} EQUAL 0x5)
            set(XLWIP_CONFIG_PCS_PMA_1000BASEX_CORE_PRESENT 1)
        elseif (${phy_type} EQUAL 0x4)
            set(XLWIP_CONFIG_PCS_PMA_SGMII_CORE_PRESENT 1)
        endif()
    endif()
endif()

if(("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "microblaze") OR
   ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "microblazeel") OR
   ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "plm_microblaze") OR
   ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "pmu_microblaze"))
    set(TCP_OVERSIZE 0x0)
else()
    set(TCP_OVERSIZE TCP_MSS)
endif()

if (${lwip220_temac_use_jumbo_frames})
    set(USE_JUMBO_FRAMES 1)
endif()

if (${lwip220_dhcp})
   set(LWIP_DHCP " ")
endif()
if (${lwip220_lwip_dhcp_does_acd_check})
    set(LWIP_DHCP_DOES_ACD_CHECK " ")
    set(LWIP_ACD " ")
endif()

set(linkspeed "#define ${lwip220_temac_phy_link_speed} 1")
if (${lwip220_stats})
    set(LWIP_STATS 1)
    set(LWIP_STATS_DISPLAY 1)
endif()

if (${lwip220_debug})
    set(LWIP_DEBUG 1)
    set(DBG_TYPES_ON DBG_LEVEL_WARNING)
    if (${lwip220_ip_debug})
        set(IP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_tcp_debug})
        set(TCP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
        set(TCP_INPUT_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
        set(TCP_OUTPUT_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
        set(TCPIP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_udp_debug})
        set(UDP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_icmp_debug})
        set(ICMP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_igmp_debug})
        set(IGMP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_netif_debug})
        set(NETIF_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_sys_debug})
        set(SYS_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
        set(API_MSG_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_pbuf_debug})
        set(PBUF_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_dhcp_debug})
        set(DHCP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    if (${lwip220_acd_debug})
        set(ACD_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
    endif()
    set(MEMP_DEBUG LWIP_DBG_LEVEL_SEVERE|LWIP_DBG_ON)
endif()

## xlwipconfig.h.in file handling
foreach(x IN LISTS TOTAL_AXIETHERNET_PROP_LIST)
    set(XLWIP_CONFIG_INCLUDE_AXI_ETHERNET 1)
    set(y ${${x}})
    list(GET y -1 connected_ip)
    math(EXPR connected_ip "${connected_ip} & 0xF" OUTPUT_FORMAT HEXADECIMAL)
    if (${connected_ip} EQUAL 0x1)
       set(XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_FIFO 1)
    elseif (${connected_ip} EQUAL 0x2)
       set(XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_DMA 1)
    elseif (${connected_ip} EQUAL 0x3)
       set(XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_MCDMA 1)
    endif()
endforeach()

if (${CONFIG_EMACLITE})
    set(XLWIP_CONFIG_INCLUDE_EMACLITE 1)
endif()

if (${CONFIG_EMACPS})
    set(XLWIP_CONFIG_INCLUDE_GEM 1)
endif()

set(XLWIP_CONFIG_N_TX_DESC ${lwip220_n_tx_descriptors})
set(XLWIP_CONFIG_N_RX_DESC ${lwip220_n_rx_descriptors})
set(XLWIP_CONFIG_N_TX_COALESCE ${lwip220_n_tx_coalesce})
set(XLWIP_CONFIG_N_RX_COALESCE ${lwip220_n_rx_coalesce})

if(("${CMAKE_SYSTEM_NAME}" STREQUAL "FreeRTOS") AND
   ("${lwip220_api_mode}" STREQUAL SOCKET_API))
    set(OS_IS_FREERTOS " ")
    set(LWIP_PROVIDE_ERRNO 1)
    set(DEFAULT_THREAD_PRIO ${lwip220_socket_mode_thread_prio})
    set(LWIP_SOCKET " ")
    set(LWIP_COMPAT_SOCKETS " ")
    set(LWIP_NETCONN " ")
    if(("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "cortexa72")
        OR ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "cortexa53")
        OR ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "cortexa53-32")
	OR ("${CMAKE_SYSTEM_PROCESSOR}" STREQUAL "aarch64"))
        set(TCPIP_THREAD_PRIO ${lwip220_socket_mode_thread_prio})
    else()
        MATH(EXPR TCPIP_THREAD_PRIO "${lwip220_socket_mode_thread_prio}+1")
    endif()
    set(TCPIP_THREAD_STACKSIZE 1024)
    set(DEFAULT_TCP_RECVMBOX_SIZE ${lwip220_default_tcp_recvmbox_size})
    set(DEFAULT_ACCEPTMBOX_SIZE 5)
    set(TCPIP_MBOX_SIZE ${lwip220_tcpip_mbox_size})
    set(DEFAULT_UDP_RECVMBOX_SIZE ${lwip220_default_udp_recvmbox_size})
    set(DEFAULT_RAW_RECVMBOX_SIZE 30)
    set(LWIP_COMPAT_MUTEX 0x0)
    set(LWIP_ALLOW_MEM_FREE_FROM_OTHER_CONTEXT 1)
    if (${lwip220_lwip_tcpip_core_locking_input})
        set(LWIP_TCPIP_CORE_LOCKING_INPUT 1)
    endif()
endif()

set(LWIP_DIR ${CMAKE_CURRENT_SOURCE_DIR}/lwip-2.2.0)

configure_file(${LWIP_DIR}/contrib/ports/xilinx/include/lwipopts.h.in ${CMAKE_BINARY_DIR}/include/lwipopts.h)
configure_file(${LWIP_DIR}/contrib/ports/xilinx/include/xlwipconfig.h.in ${CMAKE_BINARY_DIR}/include/xlwipconfig.h)
