/******************************************************************************
* Copyright (c) 2021 - 2023 Xilinx, Inc.  All rights reserved.
* Copyright (C) 2022 - 2025 Advanced Micro Devices, Inc.  All rights reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

/*****************************************************************************/
/**
*
* @file	xilsecure_rsa_client_example.c
* @addtogroup xsecure_rsa_client_apis XilSecure RSA CLIENT APIs
* @{
* This example illustrates RSA APIs usage, by taking some hash with PKCS
* padding.
*- Firstly data will be decrypted with Private key and generates RSA
* signature also verifies with expected signature.
* - Then signature will be encrypted with public key and generates the actual
* data also verifies with actual data.
*
* To build this application, xilmailbox library must be included in BSP and xilsecure
* must be in client mode
* This example is supported for Versal and Versal Net devices.
*
* Procedure to link and compile the example for the default ddr less designs
* ------------------------------------------------------------------------------------------------------------
* The default linker settings places a software stack, heap and data in DDR memory. For this example to work,
* any data shared between client running on A72/R5/PL and server running on PMC, should be placed in area
* which is accessible to both client and server.
*
* Following is the procedure to compile the example on OCM or any memory region which can be accessed by server
*
*		1. Open example linker script(lscript.ld) in Vitis project and section to memory mapping should
*			be updated to point all the required sections to shared memory(OCM or TCM)
*			using a memory region drop down selection
*
*						OR
*
*		1. In linker script(lscript.ld) user can add new memory section in source tab as shown below
*			.sharedmemory : {
*   			. = ALIGN(4);
*   			__sharedmemory_start = .;
*   			*(.sharedmemory)
*   			*(.sharedmemory.*)
*   			*(.gnu.linkonce.d.*)
*   			__sharedmemory_end = .;
* 			} > versal_cips_0_pspmc_0_psv_ocm_ram_0_psv_ocm_ram_0
*
* 		2. In this example ".data" section elements that are passed by reference to the server-side should
* 		   be stored in the above shared memory section. To make it happen in below example,
*		   replace ".data" in attribute section with ".sharedmemory". For example,
* 		   static const u8 Data __attribute__ ((section (".data.Data")))
* 					should be changed to
* 		   static const u8 Data __attribute__ ((section (".sharedmemory.Data")))
*
* To keep things simple, by default the cache is disabled for this example
*
* MODIFICATION HISTORY:
* <pre>
* Ver   Who    Date     Changes
* ----- ------ -------- -------------------------------------------------
* 1.0   kal    03/23/21 First release.
* 4.5   kal    03/23/21 Updated file version to sync with library version
*       har    06/02/21 Fixed GCC warnings for R5 compiler
* 4.7   kpt    12/01/21 Replaced library specific,standard utility functions
*                       with xilinx maintained functions
*       kpt    01/13/22 Added support for PL microblaze
*       kpt    03/16/22 Removed IPI related code and added mailbox support
*       kpt    04/11/22 Added comment on usage of shared memory
* 5.2   am     05/03/23 Added KAT before crypto usage
* 5.3   kpt    12/13/23 Add RSA quiet mode support
*       kpt    01/19/23 Enable totient calculation and PWCT for XSecure_RsaExpQOperation
* 5.4   kpt    06/30/24 Removed endianness changes
*       pre    08/29/24 Added SSIT support
*
* </pre>
******************************************************************************/

/***************************** Include Files *********************************/
#include "xil_cache.h"
#include "xparameters.h"
#include "xil_util.h"
#include "xsecure_rsaclient.h"
#include "xsecure_katclient.h"
#if defined(VERSAL_NET) && !defined(VERSAL_2VE_2VM)
#include "xsecure_plat_client.h"
#endif

/************************** Constant Definitions *****************************/
#define SLR_INDEX XSECURE_SLR_INDEX_0 /* Change this for other SLRs */
#define XSECURE_RSA_SIZE		(512U)	/**< 512 bytes for 4096 bit data */
#define XSECURE_RSA_SHARED_DATA_SIZE (XSECURE_RSA_SIZE + XSECURE_RSA_SIZE+\
									XSECURE_RSA_SIZE + XSECURE_RSA_SIZE)
#define XSECURE_SHARED_BUF_SIZE		(XSECURE_SHARED_MEM_SIZE +\
						XSECURE_RSA_SHARED_DATA_SIZE)
#if defined(VERSAL_NET) || defined(VERSAL_2VE_2VM)
#define XSECURE_PRIME_FACTOR_P_SIZE	(256U)  /**< 256 bytes size of first prime factor(P) */
#define XSECURE_PRIME_FACTOR_Q_SIZE	(256U)  /**< 256 bytes size of first prime factor(Q) */
/**************************** Type Definitions *******************************/

static const u8 P[XSECURE_PRIME_FACTOR_P_SIZE] = {
0xCA, 0xDB, 0x49, 0xD4, 0xFE, 0x4C, 0xEE, 0xBD,
0x8A, 0x82, 0x52, 0xBD, 0xCB, 0xEE, 0x63, 0x5D,
0xD9, 0x55, 0xD6, 0xD6, 0x01, 0x76, 0xC8, 0xF8,
0x59, 0xD1, 0xE1, 0x6C, 0x89, 0x2D, 0xF2, 0xF4,
0x78, 0x9D, 0xC7, 0x91, 0x20, 0x3D, 0x7B, 0x90,
0x24, 0xA3, 0xD8, 0x25, 0x16, 0x90, 0x6D, 0x50,
0x7A, 0xCD, 0x1E, 0xA3, 0x70, 0xDC, 0xA6, 0x89,
0x8C, 0xAA, 0x48, 0x4C, 0x2B, 0xF3, 0xE8, 0x75,
0x51, 0x1E, 0x48, 0x5E, 0x1A, 0xFE, 0xEB, 0x71,
0xC6, 0xCE, 0x55, 0x4E, 0xC7, 0x24, 0x3C, 0x41,
0x3E, 0xD6, 0x4E, 0x90, 0xA4, 0x29, 0xB0, 0xE9,
0x29, 0xCD, 0xDF, 0xEA, 0xD6, 0x3F, 0xB3, 0x3B,
0x69, 0x5C, 0x2F, 0x01, 0x10, 0x52, 0x82, 0x85,
0x5C, 0x62, 0x8D, 0x83, 0xFF, 0xC4, 0x75, 0x3C,
0xF6, 0x23, 0xF5, 0x06, 0x60, 0xEC, 0xBB, 0x25,
0xED, 0x1B, 0xB8, 0xF2, 0xE2, 0x66, 0x90, 0x40,
0xDD, 0x34, 0xA1, 0xA7, 0x74, 0x30, 0xC6, 0x54,
0x87, 0x82, 0x21, 0xB0, 0x46, 0x95, 0x8A, 0x1B,
0xF6, 0x62, 0x92, 0x6D, 0x0A, 0x0D, 0x3C, 0x77,
0x49, 0x30, 0xCC, 0xD6, 0x23, 0x83, 0x5E, 0xE7,
0xD5, 0x69, 0x71, 0xA8, 0x08, 0xF2, 0x2E, 0x7C,
0xB3, 0x7E, 0x82, 0x2B, 0xAB, 0xE6, 0xBC, 0xBC,
0xF5, 0xA9, 0x75, 0x88, 0x9C, 0xE4, 0xFB, 0x0B,
0x73, 0x7C, 0x99, 0x45, 0xC6, 0x4A, 0x84, 0xD1,
0x42, 0x8C, 0xDF, 0x61, 0x4A, 0x90, 0x82, 0xE5,
0x83, 0x09, 0x53, 0x93, 0x7F, 0xBC, 0x00, 0x66,
0xDA, 0x48, 0x83, 0x8D, 0x80, 0xBB, 0x6F, 0x45,
0x81, 0x16, 0x97, 0xC9, 0xAA, 0x9A, 0x5A, 0x7D,
0xFF, 0xD1, 0x2D, 0xA5, 0x03, 0x3C, 0x0C, 0xAA,
0x7B, 0xEE, 0x3D, 0x23, 0xBA, 0x9D, 0xE4, 0x35,
0x2F, 0x42, 0xFC, 0x88, 0x07, 0x8A, 0x5C, 0x6B,
0x3D, 0xEF, 0xC8, 0x61, 0x85, 0xF8, 0xD3, 0x8B};

static const u8 Q[XSECURE_PRIME_FACTOR_Q_SIZE] = {
0xC5, 0x77, 0x01, 0xFB, 0x6B, 0xE1, 0x95, 0x7D,
0xFB, 0x96, 0xF9, 0x08, 0xC8, 0xAA, 0x4C, 0xFE,
0x41, 0xA2, 0x6A, 0x13, 0x92, 0x91, 0x4C, 0x88,
0xD1, 0x84, 0x55, 0x94, 0x68, 0xCB, 0xD5, 0x2C,
0x50, 0x54, 0x6B, 0xA3, 0x1B, 0x34, 0x25, 0x77,
0x3E, 0x59, 0xEF, 0x86, 0x2E, 0xA1, 0x4D, 0x6A,
0xB0, 0x07, 0xA2, 0xBE, 0x47, 0xE6, 0x8D, 0x12,
0x6F, 0xCC, 0xCC, 0xE7, 0x92, 0x1C, 0x56, 0x1F,
0x4D, 0x5B, 0xF7, 0x6C, 0x08, 0xED, 0x19, 0x21,
0xA5, 0x96, 0x22, 0xC4, 0x0F, 0x37, 0x03, 0x59,
0xF9, 0x8D, 0xFC, 0x64, 0x23, 0x0E, 0x95, 0xFE,
0x48, 0x2B, 0xF0, 0x10, 0xDB, 0x78, 0x49, 0xF9,
0x9E, 0xFC, 0x8C, 0x10, 0xFD, 0xEE, 0xAB, 0x59,
0x52, 0x70, 0x82, 0x65, 0xFE, 0x52, 0x34, 0xF9,
0x22, 0xB3, 0xEE, 0x9B, 0xE1, 0x8E, 0x71, 0xB6,
0x9D, 0x0A, 0xE1, 0x8F, 0x08, 0x7F, 0x6C, 0x24,
0xD2, 0xCB, 0xB6, 0x06, 0xE8, 0x37, 0x5A, 0x80,
0x42, 0xF8, 0x71, 0xA7, 0xEC, 0x8F, 0xA5, 0xC8,
0x83, 0x8B, 0xF1, 0x38, 0x8B, 0x1D, 0xF5, 0x01,
0x42, 0xE8, 0x32, 0xDA, 0x43, 0x84, 0x21, 0xD3,
0x4D, 0x95, 0xF9, 0x7E, 0xC8, 0xB2, 0x31, 0x6A,
0xEE, 0xCD, 0xB0, 0x03, 0xEA, 0x71, 0xA9, 0x0E,
0x82, 0x11, 0x59, 0x4F, 0xDC, 0x60, 0xC9, 0xB5,
0xC7, 0x6A, 0x49, 0x9C, 0xE4, 0x17, 0xE2, 0xE2,
0xC1, 0xF8, 0xEE, 0x06, 0x03, 0xB8, 0x10, 0xC7,
0x8B, 0x72, 0x5F, 0x0F, 0xEF, 0x59, 0xAE, 0x5C,
0xD0, 0x97, 0x83, 0x61, 0xD4, 0xA8, 0x81, 0x59,
0xF9, 0x82, 0xE9, 0xE7, 0x81, 0x33, 0x0D, 0x53,
0xE6, 0x1E, 0xAF, 0x27, 0xE1, 0x93, 0xC9, 0xCD,
0xB2, 0x38, 0x01, 0x97, 0xA2, 0xCE, 0xFD, 0xA0,
0x5D, 0x7B, 0x53, 0x03, 0xEF, 0x72, 0xB7, 0x3A,
0x48, 0x03, 0x12, 0x60, 0x58, 0x2B, 0x52, 0x1D};

static const u8 Tot[XSECURE_RSA_SIZE] = {
0x9C, 0x79, 0x0D, 0x52, 0x4F, 0x7B, 0x0B, 0x5B,
0xD8, 0xF4, 0xCF, 0x43, 0x2D, 0x11, 0x62, 0xED,
0x54, 0xEE, 0x36, 0xA4, 0x66, 0x80, 0x49, 0x62,
0x58, 0xA9, 0xDF, 0x1F, 0x4E, 0xC4, 0x12, 0x1F,
0x0C, 0x66, 0xB6, 0x58, 0xF8, 0xC5, 0xA8, 0x7F,
0x4B, 0xD9, 0x9E, 0x3C, 0xFB, 0x35, 0x30, 0x0E,
0xFE, 0x0A, 0xA0, 0x4B, 0xB6, 0xD9, 0x97, 0xBA,
0xD1, 0x0C, 0x10, 0x18, 0x70, 0xE0, 0xE5, 0x4F,
0x8E, 0x3A, 0xC5, 0x1D, 0x4E, 0xFB, 0x4E, 0xE0,
0x1D, 0x05, 0xA0, 0x52, 0x6D, 0x0F, 0xD5, 0x16,
0xF3, 0x5A, 0x5C, 0xB7, 0x4E, 0x12, 0xEC, 0x56,
0x6A, 0x8A, 0x84, 0xDD, 0x4F, 0x30, 0xC9, 0x08,
0x7E, 0xF0, 0xBF, 0x5F, 0xE4, 0xFD, 0x53, 0x59,
0x52, 0x0C, 0x6D, 0x78, 0xDC, 0xA8, 0x7A, 0x73,
0xDB, 0x8B, 0x1A, 0x07, 0xD7, 0x79, 0x53, 0x7B,
0x58, 0x6B, 0x69, 0x78, 0x81, 0x8F, 0x09, 0x61,
0xB4, 0x70, 0x82, 0x87, 0xBE, 0x37, 0xEA, 0xAB,
0x28, 0x10, 0xE5, 0x34, 0x49, 0x5C, 0x79, 0xC2,
0x3F, 0xFC, 0x6E, 0x28, 0x64, 0x01, 0xC9, 0x34,
0xD3, 0xF5, 0xC3, 0xCD, 0x3D, 0x14, 0xFD, 0xC5,
0x58, 0xBE, 0xB2, 0x36, 0x65, 0xB6, 0xDE, 0x46,
0x73, 0xD9, 0xA5, 0x0C, 0xF6, 0x76, 0xC1, 0x41,
0xD2, 0x32, 0x19, 0xF7, 0x37, 0x41, 0xF2, 0x7C,
0x5A, 0x56, 0x44, 0xC8, 0x74, 0x1B, 0xE2, 0xDD,
0x59, 0xE1, 0xA6, 0x23, 0x17, 0x04, 0xF8, 0x97,
0x77, 0x31, 0x1C, 0x09, 0x16, 0x88, 0xAC, 0x00,
0x32, 0x03, 0xFD, 0x4D, 0xEC, 0x92, 0x79, 0xB1,
0xE9, 0xDB, 0x26, 0xB4, 0x69, 0xEA, 0x4D, 0x72,
0xB6, 0xEE, 0x6A, 0x30, 0xEF, 0x12, 0xA7, 0xC6,
0x51, 0x15, 0x57, 0x54, 0xAD, 0xA0, 0xB2, 0xCA,
0x9E, 0x09, 0xBA, 0xE0, 0xC5, 0xA5, 0x88, 0x45,
0x56, 0xF2, 0x52, 0x30, 0xF8, 0xB8, 0xFF, 0x8D,
0xF6, 0x38, 0xA4, 0x55, 0x4E, 0x1D, 0xAB, 0x67,
0xB4, 0x0A, 0x7D, 0x94, 0xE0, 0x5E, 0xFD, 0xB0,
0x0F, 0x03, 0xC0, 0xC5, 0xE6, 0x5B, 0x84, 0xBC,
0x70, 0x08, 0x3E, 0x45, 0x81, 0x22, 0x99, 0x9F,
0x16, 0x8A, 0x55, 0x76, 0xBF, 0xD7, 0xCA, 0x50,
0xDC, 0x23, 0xA5, 0x43, 0x3A, 0x1A, 0xE5, 0x7E,
0xCE, 0x1B, 0x03, 0x6F, 0xE3, 0x08, 0x90, 0x85,
0x31, 0xE7, 0x99, 0x58, 0x35, 0xED, 0x61, 0xB7,
0x32, 0x6C, 0xE2, 0x7A, 0x5B, 0xF2, 0x17, 0x66,
0xDE, 0x29, 0x33, 0x45, 0xAC, 0xDE, 0xAB, 0x7A,
0x1F, 0x2A, 0x62, 0xF9, 0xAF, 0xBB, 0x91, 0x64,
0x4C, 0xF7, 0x34, 0x35, 0xFD, 0x27, 0x75, 0xC6,
0x65, 0x9C, 0xEB, 0xE2, 0x22, 0x71, 0x2D, 0x3D,
0x4C, 0xD1, 0xEF, 0x77, 0xDA, 0x7B, 0x02, 0xB7,
0x5A, 0x82, 0x47, 0x63, 0x1B, 0x16, 0x05, 0xA8,
0x2E, 0x93, 0x83, 0x51, 0x14, 0x30, 0x04, 0xF7,
0x0F, 0xAC, 0x8D, 0x00, 0x65, 0x06, 0x61, 0xAD,
0xF2, 0xCB, 0xC8, 0x68, 0x1B, 0x70, 0x0F, 0xBC,
0xE1, 0x59, 0x17, 0x42, 0x2D, 0x9C, 0x64, 0xED,
0x9E, 0x29, 0x2F, 0x48, 0xE6, 0xD0, 0x0E, 0x07,
0x05, 0xD7, 0xC2, 0x31, 0xD8, 0x3A, 0x2F, 0x05,
0xE3, 0x23, 0xED, 0xF9, 0xAA, 0x40, 0x52, 0x69,
0x32, 0xF6, 0x6A, 0xF6, 0x04, 0xB1, 0xEE, 0x39,
0x4D, 0x56, 0x0B, 0xA4, 0x0B, 0x33, 0x44, 0xD9,
0x88, 0x63, 0x1B, 0x5D, 0xCD, 0x63, 0x2D, 0x9F,
0x51, 0xFD, 0x17, 0xBC, 0x89, 0x2F, 0x49, 0x31,
0xC9, 0xA2, 0xD2, 0x5C, 0x38, 0x97, 0xBA, 0xC8,
0x72, 0x59, 0xB4, 0xD9, 0xD7, 0x52, 0xF5, 0x26,
0xEF, 0xEA, 0x67, 0x91, 0xF6, 0x77, 0xF5, 0x82,
0x59, 0xD2, 0x5E, 0x04, 0x82, 0x0E, 0x97, 0xD3,
0x56, 0x97, 0xAD, 0xD1, 0xA7, 0xEA, 0x6A, 0x64,
0xC9, 0xA5, 0x38, 0xD7, 0x53, 0x27, 0x57, 0x18};
#endif

static const u8 Data[XSECURE_RSA_SIZE] __attribute__ ((section (".data.Data")))  = {
0x31, 0xF6, 0xA0, 0x58, 0x20, 0x34, 0x6F, 0xE9,
0xCC, 0x9A, 0x55, 0x5B, 0x92, 0x88, 0x03, 0x74,
0xE3, 0x2B, 0x89, 0x32, 0xE5, 0x65, 0x02, 0xA5,
0x33, 0x1A, 0x24, 0xCF, 0x77, 0x20, 0x46, 0xED,
0x20, 0xD8, 0x80, 0x0B, 0x95, 0x1F, 0x2D, 0xA7,
0x1B, 0x19, 0x7A, 0x66, 0x51, 0xD7, 0x39, 0x5B,
0x82, 0x6D, 0x92, 0xA6, 0x24, 0xD6, 0x1A, 0xF1,
0xE6, 0xEF, 0x17, 0xEB, 0x2E, 0x59, 0x51, 0x5D,
0x0E, 0x72, 0x9F, 0x89, 0x1E, 0x18, 0x65, 0xC0,
0x9C, 0x19, 0xAE, 0x64, 0x27, 0xCB, 0xD8, 0xF2,
0x0A, 0xC2, 0x24, 0xCA, 0x71, 0xF5, 0x63, 0x17,
0x2A, 0xC1, 0x32, 0xAA, 0x2F, 0xE1, 0xA5, 0x09,
0x76, 0xB3, 0x77, 0xDE, 0xCC, 0x40, 0xBB, 0x36,
0x19, 0x21, 0xC8, 0xD2, 0xD1, 0x4F, 0x1E, 0x1F,
0x3B, 0x95, 0xC8, 0xE8, 0xB0, 0x74, 0xD7, 0xDA,
0xA7, 0xD8, 0x99, 0x3F, 0x9E, 0xD5, 0xFE, 0x63,
0x4F, 0x6E, 0xD9, 0xA1, 0xF8, 0xF5, 0x15, 0x0E,
0x71, 0x22, 0xDD, 0xFB, 0x7E, 0x7C, 0xB0, 0xD2,
0x8D, 0xD8, 0x53, 0xBE, 0x49, 0xB0, 0xBF, 0xEE,
0x26, 0xDD, 0x2F, 0x8D, 0x91, 0xFD, 0xE4, 0xB7,
0xD4, 0x73, 0xD3, 0x46, 0x0E, 0xD7, 0x5B, 0x72,
0x19, 0x41, 0x56, 0x40, 0x90, 0xE0, 0x23, 0xCD,
0x5E, 0xAC, 0xB2, 0xF6, 0xDC, 0xED, 0x66, 0x92,
0x22, 0x8A, 0x0F, 0x76, 0x83, 0x1C, 0x02, 0x5D,
0xFC, 0x3C, 0x5F, 0xEE, 0x0B, 0xD6, 0x0E, 0x8C,
0x67, 0x8E, 0x7C, 0xBB, 0xC2, 0x45, 0xA4, 0x63,
0x51, 0xE0, 0x71, 0x27, 0xF5, 0xFB, 0x6E, 0x76,
0xF0, 0xD9, 0xD3, 0x41, 0xDD, 0xFC, 0x68, 0x16,
0x43, 0x8D, 0x39, 0x78, 0x52, 0x81, 0x26, 0x61,
0x14, 0x34, 0x1C, 0x14, 0x6E, 0x2A, 0x4B, 0x44,
0xED, 0x90, 0x7C, 0xFD, 0x42, 0x86, 0xCF, 0xAC,
0x6F, 0xD4, 0xA4, 0x16, 0x0C, 0x51, 0x12, 0x3D,
0x03, 0x73, 0x4E, 0xE5, 0x30, 0x8C, 0xE1, 0xF7,
0x1F, 0xCA, 0xCD, 0x95, 0x37, 0xFD, 0x71, 0x5E,
0x47, 0x16, 0x20, 0x7C, 0x08, 0x55, 0x5E, 0x24,
0x50, 0x00, 0x80, 0x2F, 0x83, 0xFF, 0x3C, 0x65,
0xE4, 0x37, 0xDF, 0xA1, 0x65, 0x92, 0x23, 0x14,
0x79, 0x22, 0x45, 0x44, 0x10, 0xF3, 0x4D, 0xEB,
0x85, 0x96, 0x0D, 0xB8, 0x34, 0x30, 0xF1, 0xF9,
0x27, 0xD7, 0x3D, 0xD4, 0xF8, 0x05, 0xBD, 0x98,
0xB1, 0xEF, 0x0C, 0xE9, 0xAF, 0xB3, 0xE4, 0x36,
0xA4, 0x59, 0xD1, 0xDB, 0xC7, 0x8B, 0x08, 0xB5,
0x2E, 0x0A, 0x72, 0xA9, 0x07, 0x82, 0x19, 0x62,
0x30, 0x70, 0x4F, 0x43, 0xF3, 0xE9, 0xE3, 0x8B,
0x87, 0x2F, 0xEC, 0x4E, 0xCB, 0xC1, 0xE5, 0x9B,
0xBF, 0xA0, 0xCB, 0x1E, 0xC8, 0xF1, 0x05, 0xD5,
0xFA, 0x92, 0xA0, 0xF5, 0x00, 0xE0, 0xAD, 0xBD,
0xE4, 0x4E, 0xDA, 0x83, 0x72, 0x85, 0x65, 0x45,
0x37, 0x05, 0x3D, 0xC9, 0x7C, 0xE7, 0xFB, 0xC2,
0x66, 0x79, 0xB9, 0x12, 0x33, 0x68, 0xFB, 0xEE,
0x13, 0xB5, 0x7D, 0x8D, 0x81, 0xF1, 0xFF, 0x7B,
0x76, 0x10, 0x1C, 0xFF, 0x36, 0xC9, 0xF5, 0x0D,
0x48, 0x5E, 0x6A, 0xE4, 0xCA, 0xF4, 0x09, 0x4F,
0x6A, 0xBC, 0x4E, 0x49, 0x59, 0x0C, 0x3F, 0xE9,
0x8E, 0x46, 0x23, 0x8B, 0xF1, 0xB8, 0x68, 0xF5,
0x81, 0x35, 0xB0, 0x94, 0xE1, 0x29, 0xB6, 0xE7,
0xCB, 0xD9, 0xEC, 0xAF, 0x00, 0x79, 0x52, 0x37,
0xA0, 0x4F, 0x32, 0x7A, 0xBC, 0x33, 0xB1, 0x01,
0x02, 0xEA, 0x1C, 0xD1, 0x57, 0x1D, 0xD2, 0x58,
0x81, 0x9A, 0x5E, 0xE5, 0x49, 0x75, 0xD4, 0x74,
0x70, 0xB1, 0x3D, 0x1D, 0x11, 0xA4, 0x4A, 0x72,
0x6D, 0xF6, 0xA5, 0x4A, 0xDF, 0x25, 0x69, 0x73,
0xC0, 0x98, 0x36, 0x4E, 0x2D, 0xD9, 0x09, 0xAE,
0x70, 0x4E, 0xC6, 0x70, 0x8F, 0x4F, 0x83, 0xA3};

static const u8 Modulus[XSECURE_RSA_SIZE] = {
0x9C, 0x79, 0x0D, 0x52, 0x4F, 0x7B, 0x0B, 0x5B,
0xD8, 0xF4, 0xCF, 0x43, 0x2D, 0x11, 0x62, 0xED,
0x54, 0xEE, 0x36, 0xA4, 0x66, 0x80, 0x49, 0x62,
0x58, 0xA9, 0xDF, 0x1F, 0x4E, 0xC4, 0x12, 0x1F,
0x0C, 0x66, 0xB6, 0x58, 0xF8, 0xC5, 0xA8, 0x7F,
0x4B, 0xD9, 0x9E, 0x3C, 0xFB, 0x35, 0x30, 0x0E,
0xFE, 0x0A, 0xA0, 0x4B, 0xB6, 0xD9, 0x97, 0xBA,
0xD1, 0x0C, 0x10, 0x18, 0x70, 0xE0, 0xE5, 0x4F,
0x8E, 0x3A, 0xC5, 0x1D, 0x4E, 0xFB, 0x4E, 0xE0,
0x1D, 0x05, 0xA0, 0x52, 0x6D, 0x0F, 0xD5, 0x16,
0xF3, 0x5A, 0x5C, 0xB7, 0x4E, 0x12, 0xEC, 0x56,
0x6A, 0x8A, 0x84, 0xDD, 0x4F, 0x30, 0xC9, 0x08,
0x7E, 0xF0, 0xBF, 0x5F, 0xE4, 0xFD, 0x53, 0x59,
0x52, 0x0C, 0x6D, 0x78, 0xDC, 0xA8, 0x7A, 0x73,
0xDB, 0x8B, 0x1A, 0x07, 0xD7, 0x79, 0x53, 0x7B,
0x58, 0x6B, 0x69, 0x78, 0x81, 0x8F, 0x09, 0x61,
0xB4, 0x70, 0x82, 0x87, 0xBE, 0x37, 0xEA, 0xAB,
0x28, 0x10, 0xE5, 0x34, 0x49, 0x5C, 0x79, 0xC2,
0x3F, 0xFC, 0x6E, 0x28, 0x64, 0x01, 0xC9, 0x34,
0xD3, 0xF5, 0xC3, 0xCD, 0x3D, 0x14, 0xFD, 0xC5,
0x58, 0xBE, 0xB2, 0x36, 0x65, 0xB6, 0xDE, 0x46,
0x73, 0xD9, 0xA5, 0x0C, 0xF6, 0x76, 0xC1, 0x41,
0xD2, 0x32, 0x19, 0xF7, 0x37, 0x41, 0xF2, 0x7C,
0x5A, 0x56, 0x44, 0xC8, 0x74, 0x1B, 0xE2, 0xDD,
0x59, 0xE1, 0xA6, 0x23, 0x17, 0x04, 0xF8, 0x97,
0x77, 0x31, 0x1C, 0x09, 0x16, 0x88, 0xAC, 0x00,
0x32, 0x03, 0xFD, 0x4D, 0xEC, 0x92, 0x79, 0xB1,
0xE9, 0xDB, 0x26, 0xB4, 0x69, 0xEA, 0x4D, 0x72,
0xB6, 0xEE, 0x6A, 0x30, 0xEF, 0x12, 0xA7, 0xC6,
0x51, 0x15, 0x57, 0x54, 0xAD, 0xA0, 0xB2, 0xCA,
0x9E, 0x09, 0xBA, 0xE0, 0xC5, 0xA5, 0x88, 0x45,
0x56, 0xF2, 0x52, 0x30, 0xF8, 0xB8, 0xFF, 0x8F,
0x86, 0x8A, 0xF0, 0x25, 0xB8, 0x4C, 0x2F, 0xA3,
0x3A, 0x23, 0xC9, 0x5B, 0x74, 0xF7, 0xAE, 0x0C,
0x29, 0xFC, 0x01, 0xAF, 0x7A, 0x63, 0x9A, 0x3D,
0x9B, 0x5E, 0x75, 0x46, 0x73, 0x1C, 0x61, 0xBF,
0xDF, 0x7C, 0x88, 0xAA, 0xFB, 0x49, 0x6B, 0x58,
0x3F, 0x21, 0x6C, 0xEE, 0x7F, 0x4C, 0xA0, 0x39,
0xF8, 0xEF, 0xC4, 0xD1, 0x9B, 0xCB, 0xC4, 0x21,
0x2E, 0x5E, 0xAE, 0x8B, 0xF3, 0xFD, 0xA0, 0x4B,
0xD0, 0xE7, 0x22, 0x44, 0x7F, 0xDE, 0x1B, 0xFA,
0x4A, 0x8D, 0xAB, 0x58, 0x83, 0x39, 0xEB, 0x15,
0x57, 0x8E, 0xAD, 0xEE, 0x76, 0xF3, 0xD8, 0x4B,
0xBE, 0xF1, 0x04, 0x31, 0xAE, 0xDF, 0x72, 0xFB,
0x6D, 0xF5, 0xA6, 0xF4, 0x30, 0xB2, 0x5B, 0x1B,
0xFB, 0xA4, 0xFF, 0x61, 0xD8, 0x91, 0xAC, 0xED,
0x73, 0x5A, 0x2B, 0x05, 0x5D, 0x91, 0x32, 0x84,
0xB8, 0xBA, 0x1D, 0xD2, 0xFF, 0x16, 0x01, 0x5C,
0xBF, 0xAC, 0xE4, 0xAE, 0xC1, 0x6E, 0x82, 0x82,
0xBD, 0x46, 0x5B, 0xC0, 0x4E, 0x95, 0x3F, 0xA1,
0x5B, 0x47, 0x9A, 0xE7, 0xC2, 0xC7, 0x96, 0x66,
0x2A, 0x42, 0x2E, 0xF9, 0x4D, 0xD7, 0x8E, 0xC2,
0x28, 0xD7, 0x2D, 0x58, 0xA9, 0xDE, 0x8E, 0xED,
0x85, 0x70, 0x20, 0x29, 0x40, 0x98, 0xB8, 0x34,
0xAA, 0xB1, 0x39, 0xCE, 0x7D, 0xF7, 0xB2, 0xFA,
0x88, 0x3C, 0xEE, 0x86, 0xB5, 0x95, 0xAC, 0x8D,
0x8C, 0xE8, 0xE8, 0xC5, 0x1B, 0xAB, 0xC1, 0x4C,
0x60, 0x78, 0xCA, 0x5F, 0xF8, 0x44, 0xF7, 0xF5,
0x74, 0x82, 0xD9, 0x4B, 0x8D, 0xFB, 0xAB, 0x67,
0xEC, 0xF3, 0x36, 0x8B, 0x03, 0x20, 0x5C, 0xF8,
0xD5, 0xDA, 0x44, 0x5E, 0xDB, 0x47, 0xCB, 0xFA,
0x87, 0xF8, 0x9C, 0xBF, 0xDF, 0x7B, 0x79, 0xA8,
0xE3, 0x55, 0xFD, 0x5D, 0x9E, 0xE7, 0x7E, 0x0A,
0x4F, 0x98, 0x13, 0x99, 0x31, 0x4B, 0x7C, 0xBF};

static const u8 PrivateExp[XSECURE_RSA_SIZE] = {
0x36, 0x4B, 0x97, 0x00, 0xC9, 0xCD, 0xA3, 0x7A,
0x62, 0x85, 0x48, 0x6A, 0x41, 0xF8, 0x37, 0x80,
0xAD, 0x28, 0xDF, 0xD3, 0x1A, 0x32, 0x47, 0xC4,
0x95, 0x89, 0x94, 0x7F, 0xD3, 0xA0, 0x1B, 0xBD,
0xA4, 0x5A, 0xFB, 0xC9, 0x51, 0xC1, 0x9C, 0x9F,
0x57, 0x5D, 0x93, 0x18, 0x0B, 0xFF, 0x3F, 0xC6,
0xE4, 0x06, 0x21, 0xEB, 0x43, 0xEE, 0xAB, 0x44,
0xCB, 0x07, 0x49, 0x8A, 0x81, 0x98, 0xB8, 0x59,
0x6B, 0xBC, 0x1E, 0xEB, 0xF5, 0x9D, 0x24, 0x98,
0x85, 0xD9, 0x8E, 0xF1, 0x2B, 0x21, 0x04, 0x4B,
0x2B, 0x74, 0x5E, 0x99, 0xC8, 0xC0, 0xAE, 0x3B,
0xAA, 0x4A, 0x98, 0xA2, 0x06, 0xB9, 0xA1, 0xF8,
0x8D, 0xBB, 0xD2, 0x5C, 0x5F, 0x24, 0x3C, 0x48,
0xD1, 0x3C, 0xB8, 0xBB, 0xE8, 0xAD, 0x99, 0x98,
0x1C, 0x45, 0x5F, 0x69, 0xFB, 0x2E, 0x01, 0x76,
0xDA, 0x8E, 0xA3, 0x11, 0x8A, 0xBB, 0x6C, 0x80,
0xF7, 0xD0, 0xD6, 0xAA, 0x6D, 0xAA, 0x0F, 0x87,
0x06, 0xA7, 0x1E, 0x31, 0x8F, 0x87, 0x44, 0x44,
0xD0, 0x59, 0x32, 0x8C, 0x57, 0x6F, 0x7B, 0x37,
0x16, 0x99, 0xC0, 0x30, 0xB6, 0x92, 0x53, 0x20,
0xA8, 0x93, 0xBB, 0x59, 0x3E, 0xE0, 0x3E, 0xAE,
0xE9, 0xC2, 0x4B, 0x10, 0x7D, 0x71, 0x65, 0xCC,
0x06, 0x39, 0x16, 0x63, 0x7F, 0x4C, 0x25, 0xED,
0x5A, 0x97, 0x76, 0xD4, 0xAB, 0xDD, 0x85, 0x52,
0x53, 0xBF, 0x06, 0x1C, 0x44, 0xFF, 0x97, 0x97,
0x4F, 0x67, 0xBE, 0x17, 0xB5, 0x37, 0x05, 0xA0,
0x27, 0xBA, 0xD4, 0xA9, 0xB6, 0x7E, 0x79, 0x35,
0xF0, 0x34, 0xF8, 0x70, 0x40, 0x14, 0x6A, 0x7D,
0x74, 0xD6, 0xB4, 0x08, 0x78, 0xDC, 0xDB, 0x54,
0xE8, 0x8B, 0xE4, 0x3A, 0xE0, 0xE8, 0xD3, 0x41,
0x95, 0xA7, 0x40, 0xAE, 0x80, 0x46, 0xD9, 0x4B,
0x5E, 0x49, 0x45, 0x95, 0xF2, 0xA8, 0x97, 0x1C,
0xD6, 0xA4, 0x7E, 0xF9, 0x8E, 0xDA, 0xA2, 0xC0,
0xA0, 0x6B, 0xC7, 0x79, 0xE2, 0x85, 0x73, 0xB8,
0x24, 0xED, 0xA3, 0x75, 0xC8, 0x62, 0x57, 0x63,
0xDF, 0xE8, 0x2C, 0x5B, 0x87, 0xDC, 0xCE, 0xBB,
0xB2, 0x61, 0x9C, 0x18, 0x51, 0xA7, 0x7A, 0x6F,
0xA6, 0x83, 0xC3, 0xF3, 0xD8, 0xF4, 0xBE, 0x4F,
0xCB, 0x0A, 0xFA, 0x9D, 0x57, 0x85, 0x80, 0x48,
0x87, 0xB1, 0x9A, 0xB7, 0x4E, 0x37, 0x29, 0xE2,
0x1C, 0xE0, 0x45, 0x92, 0x39, 0xD7, 0x7E, 0xA1,
0x56, 0x53, 0xAB, 0x93, 0xF0, 0xD1, 0x6F, 0x65,
0x37, 0xED, 0x48, 0x5C, 0xE5, 0xCA, 0xCB, 0x3F,
0xA7, 0xA9, 0x5B, 0xF7, 0xA3, 0xF9, 0x8E, 0x57,
0xA2, 0xA8, 0xD9, 0xFD, 0x25, 0x00, 0x6A, 0xB7,
0x4B, 0xB1, 0xC9, 0x4B, 0xA8, 0x87, 0x68, 0xC0,
0xE0, 0xEA, 0x36, 0x05, 0x40, 0xBF, 0xDA, 0x84,
0xAA, 0x6C, 0xCB, 0x73, 0x55, 0xF2, 0xF7, 0xB6,
0x17, 0x3D, 0x43, 0xCC, 0x30, 0x42, 0x0F, 0xA1,
0x01, 0x90, 0x0E, 0xE1, 0x79, 0x2B, 0xE0, 0x7A,
0x25, 0x23, 0xF9, 0xFB, 0x1F, 0xC2, 0x96, 0x0F,
0x12, 0x03, 0xAE, 0x81, 0xB8, 0x7D, 0xD7, 0x70,
0x3E, 0x69, 0xC8, 0xDB, 0xEC, 0xF0, 0xFB, 0xA2,
0x2A, 0x58, 0x2C, 0x5A, 0xA6, 0xE6, 0x1C, 0xF2,
0x9E, 0x19, 0x7D, 0xB2, 0x12, 0x9C, 0xFF, 0x51,
0xE8, 0xF1, 0x03, 0xA7, 0x0F, 0xAF, 0xE1, 0xA3,
0x2E, 0x3D, 0x5D, 0xA0, 0xA6, 0x05, 0x55, 0x2E,
0x64, 0x8F, 0xD3, 0x2D, 0xBB, 0xD8, 0xA9, 0xA8,
0x5A, 0x14, 0x62, 0xB4, 0x57, 0x7A, 0xE6, 0xF8,
0x69, 0x1D, 0x91, 0xB0, 0xC9, 0x93, 0xE2, 0xB7,
0x79, 0x26, 0xB8, 0x7B, 0xA0, 0xAF, 0xAD, 0x84,
0x66, 0xD0, 0xA5, 0x89, 0x20, 0xEC, 0x19, 0x68,
0xDF, 0x32, 0x50, 0xC1, 0x60, 0x67, 0xEA, 0x35,
0x40, 0xF7, 0x51, 0x0A, 0xFE, 0xB2, 0xB6, 0xF9};

static const u8 ExpectedSign[XSECURE_RSA_SIZE] = {
0x8C, 0xEC, 0xBD, 0x2E, 0x71, 0x3D, 0xB1, 0xE2,
0x2C, 0x8C, 0xA0, 0xC6, 0x2C, 0xE3, 0x70, 0x0A,
0x91, 0xB9, 0xFE, 0xEA, 0x93, 0x6F, 0xFB, 0xE5,
0x6F, 0xF8, 0xF0, 0xE3, 0x41, 0x90, 0x19, 0xFC,
0x3F, 0xCD, 0x5D, 0x07, 0xFB, 0x54, 0xC9, 0x6B,
0x6F, 0xE9, 0x11, 0xEA, 0x47, 0x1F, 0x0F, 0x75,
0x23, 0x46, 0xCD, 0xE7, 0x30, 0x84, 0x72, 0x2D,
0xBC, 0x6D, 0xCC, 0xAB, 0xF0, 0x1E, 0x82, 0x9B,
0xF1, 0x16, 0x4E, 0xAF, 0x86, 0x2D, 0x4A, 0xAF,
0x77, 0x5A, 0x4B, 0x4F, 0x6F, 0xD9, 0xB3, 0x1E,
0xAD, 0x3A, 0xAC, 0x12, 0xC0, 0xFA, 0x98, 0x6A,
0x09, 0x0F, 0xA7, 0x2B, 0xC4, 0x6C, 0x14, 0xA0,
0x29, 0xE8, 0x0D, 0x2B, 0x78, 0x43, 0x9C, 0x31,
0x59, 0x18, 0x72, 0x66, 0xCB, 0x50, 0x10, 0x4A,
0x13, 0x34, 0x7B, 0x19, 0xD8, 0x8F, 0x78, 0x1B,
0x94, 0xB1, 0xEA, 0x84, 0xD3, 0x6D, 0x50, 0xCC,
0xA8, 0x77, 0xF0, 0x81, 0xFD, 0x36, 0x16, 0x33,
0x28, 0x42, 0x02, 0xF2, 0x28, 0xB2, 0x81, 0x92,
0xD0, 0x0F, 0x4C, 0xDF, 0xD6, 0x75, 0xDB, 0xE8,
0x1B, 0x7E, 0xA1, 0xCD, 0x22, 0xE4, 0x2E, 0x1D,
0x22, 0x32, 0x88, 0x5B, 0x18, 0x82, 0xA5, 0xED,
0xF0, 0xFC, 0x00, 0x42, 0x65, 0xAC, 0x6C, 0x78,
0x6A, 0xCA, 0xEB, 0x37, 0x5B, 0x6C, 0xBF, 0x8B,
0xD0, 0x8B, 0xF6, 0x77, 0x8F, 0x74, 0x28, 0x9D,
0x45, 0xBA, 0x05, 0x01, 0xA6, 0xF4, 0x46, 0x12,
0x4C, 0x51, 0x8A, 0xA6, 0xB1, 0xA6, 0xBE, 0x11,
0x2E, 0xA8, 0x67, 0x51, 0x65, 0x28, 0x2F, 0xA1,
0xD1, 0x9E, 0x38, 0x45, 0x91, 0x96, 0xAA, 0x9E,
0x10, 0x2D, 0x8D, 0xD4, 0xFB, 0x82, 0xB2, 0x43,
0x19, 0x06, 0xFB, 0x18, 0x47, 0x49, 0x06, 0x4A,
0x89, 0x72, 0x3F, 0x2A, 0x96, 0x76, 0x4C, 0x3B,
0xA1, 0x7F, 0x7C, 0x7C, 0x39, 0x23, 0x88, 0xAA,
0x2E, 0x89, 0x90, 0x58, 0xE8, 0xB2, 0xFA, 0x12,
0x04, 0x20, 0x36, 0x52, 0x47, 0x0F, 0x69, 0xD0,
0xFD, 0xD4, 0xA7, 0xDB, 0x59, 0xE4, 0xC0, 0xAA,
0xD4, 0x36, 0x6B, 0x92, 0x2C, 0xAD, 0xA2, 0x95,
0x36, 0x0A, 0xB2, 0xFA, 0x60, 0xB6, 0xF3, 0x79,
0x24, 0xDE, 0x9C, 0x94, 0x15, 0xBC, 0x4C, 0x3C,
0xDB, 0xAC, 0x17, 0x5B, 0xF8, 0x3B, 0xE0, 0x65,
0xB2, 0xEA, 0x1E, 0x01, 0x16, 0x28, 0x37, 0x85,
0xD0, 0xC9, 0xD8, 0x23, 0x7F, 0x22, 0x78, 0x7D,
0x84, 0x97, 0xB5, 0x13, 0x8C, 0x59, 0xF7, 0x67,
0x06, 0x7C, 0x57, 0x20, 0xB9, 0xDC, 0xD5, 0x01,
0xCF, 0x25, 0xD0, 0x0A, 0xBA, 0x06, 0x7B, 0x7C,
0x84, 0x56, 0xF4, 0xDA, 0xB3, 0xF8, 0xC9, 0x15,
0xAA, 0x62, 0xAA, 0x45, 0xA1, 0x1C, 0x1F, 0xD2,
0x0D, 0x75, 0x5F, 0xE5, 0xEC, 0xA6, 0x1F, 0x47,
0x70, 0xC8, 0xEF, 0x6A, 0x5A, 0xC4, 0xFB, 0x4C,
0x9F, 0x43, 0xB9, 0x44, 0x56, 0x1E, 0x46, 0xA0,
0xF7, 0x51, 0xE5, 0x71, 0xA8, 0xBC, 0xF7, 0xEA,
0x6B, 0xB8, 0x7A, 0x10, 0x35, 0x51, 0x9F, 0x3E,
0xD3, 0x8C, 0xCB, 0xF8, 0x59, 0x08, 0x7D, 0x9F,
0xBA, 0xD4, 0x66, 0xC9, 0x6E, 0xBA, 0x3E, 0xDB,
0x25, 0xAE, 0x6F, 0x95, 0x5D, 0xB8, 0x66, 0x2D,
0xAD, 0xE4, 0x5A, 0x7E, 0x9D, 0xC7, 0x60, 0x06,
0x1A, 0xD1, 0x0E, 0xA7, 0x1E, 0xC9, 0xF2, 0xD1,
0x19, 0x0F, 0xD8, 0x2C, 0x0F, 0x44, 0x13, 0xDF,
0x54, 0x55, 0xEB, 0x5F, 0x0E, 0xAF, 0x89, 0x80,
0x94, 0x1A, 0x57, 0x02, 0x2C, 0xA2, 0xCC, 0x69,
0x14, 0xC2, 0x74, 0x0B, 0x92, 0x7A, 0x37, 0xCC,
0x0B, 0x7E, 0x2E, 0x9C, 0x4C, 0xCC, 0x2B, 0x4C,
0xBB, 0x11, 0x72, 0x9B, 0x16, 0x44, 0x3D, 0x2E,
0x6B, 0x7B, 0xBC, 0xB0, 0x46, 0x6F, 0x02, 0x9D,
0x8E, 0xDA, 0xAE, 0xE3, 0x5F, 0xD9, 0xDB, 0x5D};

/* Exponent of Public key */
static const u32 PublicExp = 0x1000100;

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Function Prototypes ******************************/

static u32 SecureRsaExample(void);
#if defined(VERSAL_NET) || defined(VERSAL_2VE_2VM)
void ReverseArr(u8 *Arr, u8 *dst, u32 Size);
#endif

/************************** Variable Definitions *****************************/

static u32 Size = XSECURE_RSA_SIZE;

/* shared memory allocation */
static u8 SharedMem[XSECURE_SHARED_BUF_SIZE] __attribute__((aligned(64U)))
										__attribute__ ((section (".data.SharedMem")));
#if defined(VERSAL_NET) || defined(VERSAL_2VE_2VM)
XSecure_RsaKeyParam RsaKeyParam __attribute__((aligned(64U))) __attribute__ ((section (".data.RsaKeyParam")));
#endif

/*****************************************************************************/
/**
*
* This example illustrates the usage of RSA APIs to encrypt or decrypt the
* signature
*
* @return
* 		- XST_SUCCESS if the boot image authentication is successful.
*		- ErrorCode if the boot image authentication Failed.
*
******************************************************************************/
int main(void)
{
	u32 Status;

	#ifdef XSECURE_CACHE_DISABLE
		Xil_DCacheDisable();
	#endif

	Status = SecureRsaExample();
	if(Status != XST_SUCCESS)
	{
		xil_printf("\r\nRSA client example failed %d \r\n",
				Status);
		Status = XST_FAILURE;
		goto END;
	}

	xil_printf("\r\nSuccessfully ran RSA client example\r\n ");
END:
	return Status;
}

/****************************************************************************/
/**
*
* This function authenticates boot image located in DDR using RSA-4096
* algorithm. The decrypted hash is matched against the calculated boot image
* hash.
* The purpose of this function is to illustrate how to use the XSecure_Rsa
* driver.
*
*
* @return	- ErrorCode if RSA authentication is failed
*		- XST_SUCCESS if RSA authentication is successful.
*
* @note		None.
*
****************************************************************************/
/** //! [RSA generic example] */
static u32 SecureRsaExample(void)
{
	u64 KeyAddr;
	u32 Status = XST_FAILURE;
	u8 *Signature = &SharedMem[0];
	u8 *EncryptSignatureOut = Signature + XSECURE_RSA_SIZE;
	u8 *Key = EncryptSignatureOut + XSECURE_RSA_SIZE;
	u8 *InputMsg;
	u32 Index;
	XMailbox MailboxInstance;
	XSecure_ClientInstance SecureClientInstance;
#if defined(VERSAL_NET) || defined(VERSAL_2VE_2VM)

	Xil_DCacheFlushRange((UINTPTR)Data, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Modulus, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)PrivateExp, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)P, XSECURE_PRIME_FACTOR_P_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Q, XSECURE_PRIME_FACTOR_Q_SIZE);
	Xil_DCacheFlushRange((UINTPTR)Tot, XSECURE_RSA_SIZE);

	RsaKeyParam.ExpAddr = (u64)(UINTPTR)&PrivateExp;
	RsaKeyParam.ModAddr = (u64)(UINTPTR)&Modulus;
	RsaKeyParam.PAddr   = (u64)(UINTPTR)&P;
	RsaKeyParam.PSize   = XSECURE_PRIME_FACTOR_P_SIZE;
	RsaKeyParam.QAddr   = (u64)(UINTPTR)&Q;
	RsaKeyParam.QSize   = XSECURE_PRIME_FACTOR_Q_SIZE;
	RsaKeyParam.IsPrimeAvail = TRUE;
	RsaKeyParam.PubExp  = PublicExp;
	RsaKeyParam.IsPubExpAvail = TRUE;
	RsaKeyParam.TotAddr = (u64)(UINTPTR)&Tot;
	RsaKeyParam.IsTotAvail = FALSE;
	RsaKeyParam.OpMode = XSECURE_RSA_EXPQ_MODE;

	KeyAddr = (u64)(UINTPTR)&RsaKeyParam;
#else
	KeyAddr = (u64)(UINTPTR)Key;
#endif
	InputMsg = (u8*)(UINTPTR)&Data;

	Status = XMailbox_Initialize(&MailboxInstance, 0U);
	if (Status != XST_SUCCESS) {
		xil_printf("Mailbox initialize failed:%08x \r\n", Status);
		goto END;
	}

	Status = XSecure_ClientInit(&SecureClientInstance, &MailboxInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("Client initialize failed:%08x \r\n", Status);
		goto END;
	}

	/* Set shared memory */
	Status = XMailbox_SetSharedMem(&MailboxInstance, (u64)(UINTPTR)(Key + XSECURE_RSA_SIZE +
				XSECURE_RSA_SIZE), XSECURE_SHARED_MEM_SIZE);
	if (Status != XST_SUCCESS) {
		xil_printf("\r\n shared memory initialization failed");
		goto END;
	}

	Status = Xil_SMemCpy(Key, XSECURE_RSA_SIZE, Modulus, XSECURE_RSA_SIZE,
			XSECURE_RSA_SIZE);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	Xil_DCacheFlushRange((UINTPTR)Key, XSECURE_RSA_SIZE);

#if !defined(VERSAL_NET) && !defined(VERSAL_2VE_2VM)
	Status = Xil_SMemCpy(Key + XSECURE_RSA_SIZE, XSECURE_RSA_SIZE, PrivateExp,
			XSECURE_RSA_SIZE, XSECURE_RSA_SIZE);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	Xil_DCacheFlushRange((UINTPTR)Data, XSECURE_RSA_SIZE);
	Xil_DCacheFlushRange((UINTPTR)(Key + XSECURE_RSA_SIZE), XSECURE_RSA_SIZE);
#endif

	Xil_DCacheInvalidateRange((UINTPTR)Signature, XSECURE_RSA_SIZE);

	Status = XSecure_SetSlrIndex(&SecureClientInstance, SLR_INDEX);
	if (Status != XST_SUCCESS) {
			xil_printf("invalid SlrIndex \r\n");
			goto END;
	}

	Status = XSecure_RsaPrivateDecKat(&SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("RSA private decrypt KAT failed\n\r");
	}

	/* RSA signature decrypt with private key */
	Status = XSecure_RsaPrivateDecrypt(&SecureClientInstance, KeyAddr, (UINTPTR)InputMsg,
			Size, (UINTPTR)Signature);
	if(XST_SUCCESS != Status) {
		xil_printf("Failed at RSA signature decryption\n\r");
		goto END;
	}

	Xil_DCacheInvalidateRange((UINTPTR)Signature, XSECURE_RSA_SIZE);
	xil_printf("\r\n Decrypted Signature with private key\r\n ");

	for(Index = 0; Index < Size; Index++) {
		xil_printf(" %02x ", Signature[Index]);
	}
	xil_printf(" \r\n ");

	/* Verification if Data is expected */
	for(Index = 0; Index < Size; Index++) {
		if (Signature[Index] != ExpectedSign[Index]) {
			xil_printf(
				"\r\nError at verification of RSA signature"
				" Decryption\n\r");
			Status = XST_FAILURE;
			goto END;
		}
	}

	/* RSA signature encrypt with Public key components */
	Status = Xil_SMemCpy(Key + XSECURE_RSA_SIZE, 4U, &PublicExp, 4U, 4U);
	if (Status != XST_SUCCESS) {
		goto END;
	}

	Xil_DCacheFlushRange((UINTPTR)Key, XSECURE_RSA_SIZE + XSECURE_RSA_SIZE);
	Xil_DCacheInvalidateRange((UINTPTR)EncryptSignatureOut, XSECURE_RSA_SIZE);

	Status = XSecure_RsaPublicEncKat(&SecureClientInstance);
	if (Status != XST_SUCCESS) {
		xil_printf("RSA public encrypt KAT failed\n\r");
	}

	Status = XSecure_RsaPublicEncrypt(&SecureClientInstance, (UINTPTR)Key, (UINTPTR)Signature,
			Size, (UINTPTR)EncryptSignatureOut);
	if(XST_SUCCESS != Status)	{
		xil_printf("\r\nFailed at RSA signature encryption\n\r");
		goto END;
	}

	Xil_DCacheInvalidateRange((UINTPTR)EncryptSignatureOut, XSECURE_RSA_SIZE);

	xil_printf("\r\n Encrypted Signature with public key\r\n ");

	for(Index = 0; Index < Size; Index++) {
		xil_printf(" %02x ", EncryptSignatureOut[Index]);
	}

	/* Verification if Data is expected */
	for(Index = 0; Index < Size; Index++) {
		if (EncryptSignatureOut[Index] != Data[Index]) {
			xil_printf(
				"\r\nError at verification of RSA signature"
				" encryption\n\r");
			Status = XST_FAILURE;
		}
	}

END:
	Status |= XMailbox_ReleaseSharedMem(&MailboxInstance);
	return Status;
}

/** //! [RSA generic example] */
/** @} */
