/**************************************************************************************************
* Copyright (c) 2024 - 2025 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
**************************************************************************************************/

/*************************************************************************************************/
/**
 *
 * @file xasu_sharedmem.h
 *
 * This header file defines the structures that are used for shared memory
 * communication between the client and the server. The structures include
 * definitions for message formats, shared memory control blocks, and
 * any other relevant data structures needed for inter-process communication
 *
 * <pre>
 * MODIFICATION HISTORY:
 *
 * Ver   Who  Date     Changes
 * ----- ---- -------- ----------------------------------------------------------------------------
 * 1.0   vns  06/04/24 Initial release
 *       ma   07/08/24 Add task based approach at queue level
 *       ma   07/23/24 Update XASU_RESPONSE_STATUS_INDEX with 0
 * 1.1   ma   12/12/24 Updated command states
 *       ma   02/19/25 Updated handling of same priority queue requests in round robin scheduling
 *       ma   02/26/25 Corrected XASU_CHANNEL_RESERVED_MEM define value
 *
 * </pre>
 *
 *************************************************************************************************/
/**
 * @addtogroup xasu_common_defs Common Defs
 * @{
*/
#ifndef XASU_SHAREDMEM_H_
#define XASU_SHAREDMEM_H_

#ifdef __cplusplus
extern "C" {
#endif

/*************************************** Include Files *******************************************/
#include "xil_types.h"

/************************************ Constant Definitions ***************************************/
#define XASU_MAX_BUFFERS				(8U) /**< Maximum request and response buffers */
#define XASU_CHANNEL_RESERVED_MEM			(1188U) /**< Reserved memory in channel */

#define XASU_COMMAND_IS_PRESENT				(0x1U) /**< Command is written by client */
#define XASU_COMMAND_IN_PROGRESS			(0x2U) /**< Command is in progress by ASUFW */
#define XASU_COMMAND_WAITING_FOR_RESOURCE	(0x3U) /**< Command is waiting for required resources */
#define XASU_COMMAND_DMA_WAIT_COMPLETE		(0x4U) /**< Command DMA wait operation is complete */
#define XASU_COMMAND_EXECUTION_COMPLETE		(0x5U) /**< Command execution is complete by ASUFW */
/* Response buffers status */
#define XASU_RESPONSE_IS_PRESENT			(0x1U) /**< Response is written by ASUFW */

#define XASU_RESPONSE_STATUS_INDEX			(0U) /**< Response status index in response buffer */
#define XASU_RESPONSE_BUFF_ADDR_INDEX			(1U) /**< Response buffer address index
								* in response buffer */

#define XASU_COMMAND_ID_MASK				(0x0000003FU) /**< Mask for command ID in header */
#define XASU_UNIQUE_REQ_ID_MASK				(0x00000FC0U) /**< Mask for command unique reqest ID */
#define XASU_UNIQUE_REQ_ID_SHIFT			(6U) /**< Shift value for unique request ID */
#define XASU_UNIQUE_ID_MAX					(XASU_MAX_BUFFERS << 1U) /**< Maximum unique IDs generated by ASU */
#define XASU_MODULE_ID_MASK					(0x0003F000U) /**< Mask for module ID */
#define XASU_MODULE_ID_SHIFT				(12U) /**< Shift value for module ID */
#define XASU_COMMAND_LENGTH_MASK			(0x00FC0000U) /**< Mask for command length in header */
#define XASU_COMMAND_LENGTH_SHIFT			(18U) /**< Shift for command length */

#define XASU_COMMAND_REQ_ARGS				(22U) /**< Command arguments count */
#define XASU_COMMAND_RESP_ARGS				(17U) /**< Response arguments count */

#define XASU_RTCA_BASEADDR			(0xEBE40000U) /**< ASUFW run time configuration area
									base address */
#define XASU_RTCA_COMM_CHANNEL_INFO_ADDR	(XASU_RTCA_BASEADDR + 0x10U) /**< Communication
										channel info register
										address */
#define XASU_MAX_IPI_CHANNELS			(8U) /**< Maximum IPI channels supported by ASUFW */
#define XASU_CHANNEL_MEMORY_OFFSET		(0x1000U) /**<Channel memory offset */
#define XASU_CHANNEL_MEMORY_BASEADDR		(XASU_RTCA_BASEADDR + XASU_CHANNEL_MEMORY_OFFSET)

/************************************** Type Definitions *****************************************/
/** @brief This structure is the request buffer */
typedef struct {
	u32 Header; /**< Command header */
	u32 Arg[XASU_COMMAND_REQ_ARGS]; /**< Command arguments */
	u32 Reserved; /**< Reserved */
} XAsu_ReqBuf;

/** @brief  This structure is the response buffer */
typedef struct {
	u32 Header; /**< Command header */
	u32 Arg[XASU_COMMAND_RESP_ARGS]; /**< Response arguments */
	u32 AdditionalStatus; /**< Redundant variable to hold additional status from server */
	u32 Reserved; /**< Reserved */
} XAsu_RespBuf;

/** @brief This structure is the channel's queue buffer */
typedef struct {
	u8 ReqBufStatus; /**< Request buffer status */
	u8 RespBufStatus; /**< Response buffer status */
	u16 Reserved; /**< Reserved */
	XAsu_ReqBuf ReqBuf; /**< Request buffer */
	XAsu_RespBuf RespBuf; /**< Response buffer */
} XAsu_ChannelQueueBuf;

/** @brief This structure is the channel's queue which is of 8 buffers */
typedef struct {
	u32 IsCmdPresent; /**< Cmd present status of the queue */
	u32 ReqSent; /**< Number of requests sent from client to ASUFW */
	u32 ReqServed; /**< Number of requests served by ASUFW */
	XAsu_ChannelQueueBuf ChannelQueueBufs[XASU_MAX_BUFFERS]; /**< Channel's queue of 8 buffers */
} XAsu_ChannelQueue;

/** @brief This struture is the channel's memory */
typedef struct {
	u32 Version; /**< Structure version */
	u8 Reserved[XASU_CHANNEL_RESERVED_MEM]; /**< Reserved memory */
	XAsu_ChannelQueue P0ChannelQueue; /**< P0 channel queue */
	XAsu_ChannelQueue P1ChannelQueue; /**< P1 channel queue */
} XAsu_ChannelMemory;

/** @brief This structure contains information of each communication channel. */
typedef struct {
	u8 P0QueuePriority; /**< P0 Queue Priority of the channel */
	u8 P1QueuePriority; /**< P1 Queue Priority of the channel */
	u16 IpiBitMask; /**< Bit mask of the IPI channel to enable interrupts */
	u32 Reserved; /**< Reserved */
} XAsu_CommChannel;

/** @brief This structure contains information of all communication channels. */
typedef struct {
	u32 CommChannelInfoHeader; /**< Communication channel information header */
	u32 NumOfIpiChannels; /**< Number of IPI channels to be enabled */
	XAsu_CommChannel Channel[XASU_MAX_IPI_CHANNELS]; /**< Communication channels information */
} XAsu_CommChannelInfo;

/*************************** Macros (Inline Functions) Definitions *******************************/

/************************************ Function Prototypes ****************************************/

/************************************ Variable Definitions ***************************************/

#ifdef __cplusplus
}
#endif

#endif  /* XASU_SHAREDMEM_H_ */
/** @} */
