/**************************************************************************************************
* Copyright (c) 2023 - 2025, Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
**************************************************************************************************/

/*************************************************************************************************/
/**
 *
 * @file xtask.h
 *
 * This file contains declarations for xtask.c file.
 *
 * <pre>
 * MODIFICATION HISTORY:
 *
 * Ver   Who  Date     Changes
 * ----- ---- -------- ----------------------------------------------------------------------------
 * 1.0   ma   10/11/23 Initial release
 * 1.1   ma   02/02/24 Update task functionality to support periodic tasks
 *       ma   03/16/24 Added error codes at required places
 *       ma   07/08/24 Add task based approach at queue level
 *       yog  09/26/24 Added doxygen groupings and fixed doxygen comments.
 * 1.2   ma   12/12/24 Added support for DMA non-blocking wait
 *
 * </pre>
 *
 *************************************************************************************************/
/**
* @addtogroup xtask_apis Task Related APIs
* @{
*/
#ifndef XTASK_H_
#define XTASK_H_

#ifdef __cplusplus
extern "C" {
#endif

/*************************************** Include Files *******************************************/
#include "xlinklist.h"

/************************************ Constant Definitions ***************************************/
#define XTASK_MAX               (32U) /**< Maximum tasks allowed to be created */
#define XTASK_PRIORITIES        (16U) /**< Number of task priorities allowed */

#define XTASK_NUM_BITS_IN_U32   (32U) /**< Maximum bits in unsigned int */

/************************************** Type Definitions *****************************************/

/** Type definition for task handler */
typedef s32 (*XTask_Handler_t)(void *Arg);

/** @brief This structure contains task metadata. */
typedef struct {
	u32 Priority; /**< Priority of the task */
	u32 Delay; /**< Task delay */
	XLinkList TaskNode; /**< Linked list for the task */
	XTask_Handler_t TaskHandler; /**< Handler for the task */
	void *PrivData; /**< Private data of the task */
	u32 Interval; /**< Task interval. For non-periodic tasks, interval should be 0 */
} XTask_TaskNode;

/** @brief This structure contains the events list. */
typedef struct {
	u32 Tasks[(XTASK_MAX + XTASK_NUM_BITS_IN_U32 - 1U) / XTASK_NUM_BITS_IN_U32]; /**< Task events */
} XTask_TaskEvent;

/*************************** Macros (Inline Functions) Definitions *******************************/

/************************************ Function Prototypes ****************************************/
void XTask_Init(void);
XTask_TaskNode *XTask_Create(u32 Priority, XTask_Handler_t TaskHandler, void *PrivData,
			     u32 Interval);
void XTask_Delete(XTask_TaskNode *Task);
void XTask_TriggerNow(XTask_TaskNode *Task);
s32 XTask_TriggerAfterDelay(XTask_TaskNode *Task, u32 Delay);
s32 XTask_TriggerOnEvent(const XTask_TaskNode *Task, XTask_TaskEvent *Event);
void XTask_EventNotify(XTask_TaskEvent *Event);
u32 XTask_DelayTime(const XTask_TaskNode *Task);
void XTask_DispatchLoop(void);
XTask_TaskNode *XTask_GetInstance(void *PrivData);

/************************************ Variable Definitions ***************************************/
/** Current time in ms */
extern u32 TaskTimeNow;

#ifdef __cplusplus
}
#endif

#endif  /* XTASK_H_ */
/** @} */
