/**************************************************************************************************
* Copyright (c) 2024 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
**************************************************************************************************/

/*************************************************************************************************/
/**
*
* @file xrsa_eccinfo.c
*
* This file contains the curve information of all the ECDSA curves supported
* in Versal.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who  Date     Changes
* ----- ---- -------- -----------------------------------------------------------------------------
* 1.0   yog  07/11/24 Initial release
*       yog  09/26/24 Added doxygen groupings.
*       ss   12/02/24 Added support for NIST curves P-256,P-384
*
* </pre>
*
* @note
*
***************************************************************************************************/
/**
* @addtogroup xrsa_ecc_server_apis RSA ECC Server APIs
* @{
*/
/*************************************** Include Files *******************************************/
#include "xrsa_eccinfo.h"

/************************************ Constant Definitions ***************************************/
#ifdef XRSA_ECC_SUPPORT_NIST_P521
static const u8 XRsa_EccP521D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xCAU, 0xA8U, 0x3DU, 0x0FU,
	0x39U, 0xF6U, 0x25U, 0xD7U, 0x97U, 0xECU, 0x95U, 0x9CU,
	0xF9U, 0x6EU, 0xB2U, 0x88U, 0x3EU, 0x1CU, 0x43U, 0x6AU,
	0x3AU, 0x49U, 0xE9U, 0x6CU, 0x33U, 0xBAU, 0xC9U, 0x67U,
	0xE5U, 0x14U, 0xC5U, 0x4CU, 0xFDU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0x8DU, 0x2BU, 0x53U, 0xF3U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D1[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D2[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0x7FU, 0xFEU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D3[] = {
	0xA6U, 0xE4U, 0x87U, 0x26U, 0x87U, 0x63U, 0x58U, 0xAEU,
	0x4FU, 0x65U, 0x14U, 0x80U, 0x1FU, 0xA8U, 0x35U, 0xEAU,
	0x8FU, 0xA2U, 0xF7U, 0x78U, 0x1AU, 0x96U, 0x1EU, 0xC4U,
	0xEFU, 0xB9U, 0x9AU, 0x83U, 0xDFU, 0xD8U, 0x9DU, 0x5EU,
	0x60U, 0x29U, 0x8BU, 0xBDU, 0x49U, 0x3FU, 0xF6U, 0xA8U,
	0x9CU, 0x0CU, 0xABU, 0xF0U, 0x84U, 0x78U, 0xC7U, 0xC8U,
	0x44U, 0x5AU, 0xDCU, 0xF9U, 0x8AU, 0xD9U, 0xCCU, 0x2DU,
	0x39U, 0x6DU, 0x51U, 0x77U, 0xA0U, 0x42U, 0x5BU, 0xD0U,
	0x10U, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D4[] = {
	0x66U, 0xBDU, 0xE5U, 0xC2U, 0x31U, 0x7EU, 0x7EU, 0xF9U,
	0x9BU, 0x42U, 0x6AU, 0x85U, 0xC1U, 0xB3U, 0x48U, 0x33U,
	0xDEU, 0xA8U, 0xFFU, 0xA2U, 0x27U, 0xC1U, 0x1DU, 0xFEU,
	0x28U, 0x59U, 0xE7U, 0xEFU, 0x77U, 0x5EU, 0x4BU, 0xA1U,
	0xBAU, 0x3DU, 0x4DU, 0x6BU, 0x60U, 0xAFU, 0x28U, 0xF8U,
	0x21U, 0xB5U, 0x3FU, 0x05U, 0x39U, 0x81U, 0x64U, 0x9CU,
	0x42U, 0xB4U, 0x95U, 0x23U, 0x66U, 0xCBU, 0x3EU, 0x9EU,
	0xCDU, 0xE9U, 0x04U, 0x04U, 0xB7U, 0x06U, 0x8EU, 0x85U,
	0xC6U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D5[] = {
	0x50U, 0x66U, 0xD1U, 0x9FU, 0x76U, 0x94U, 0xBEU, 0x88U,
	0x40U, 0xC2U, 0x72U, 0xA2U, 0x86U, 0x70U, 0x3CU, 0x35U,
	0x61U, 0x07U, 0xADU, 0x3FU, 0x01U, 0xB9U, 0x50U, 0xC5U,
	0x40U, 0x26U, 0xF4U, 0x5EU, 0x99U, 0x72U, 0xEEU, 0x97U,
	0x2CU, 0x66U, 0x3EU, 0x27U, 0x17U, 0xBDU, 0xAFU, 0x17U,
	0x68U, 0x44U, 0x9BU, 0x57U, 0x49U, 0x44U, 0xF5U, 0x98U,
	0xD9U, 0x1BU, 0x7DU, 0x2CU, 0xB4U, 0x5FU, 0x8AU, 0x5CU,
	0x04U, 0xC0U, 0x3BU, 0x9AU, 0x78U, 0x6AU, 0x29U, 0x39U,
	0x18U, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D7[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x80U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D8[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x40U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D9[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x80U, 0xFBU, 0xCDU, 0x63U, 0xB7U, 0x70U,
	0x24U, 0x48U, 0xA2U, 0x28U, 0xDCU, 0x31U, 0xBBU, 0x23U,
	0x1BU, 0x25U, 0xE2U, 0x17U, 0x2DU, 0x7BU, 0x84U, 0x5BU,
	0xFFU, 0x19U, 0x40U, 0xCAU, 0x34U, 0x68U, 0x20U, 0x3EU,
	0xBCU, 0x3CU, 0xD7U, 0x02U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D10[] = {
	0x4AU, 0x5DU, 0x1AU, 0xB4U, 0x17U, 0xB7U, 0xC8U, 0x58U,
	0x70U, 0x77U, 0xC4U, 0x7BU, 0xB2U, 0x49U, 0x87U, 0x57U,
	0xB9U, 0x82U, 0xD7U, 0xA1U, 0x16U, 0x7FU, 0xEEU, 0x90U,
	0x97U, 0x60U, 0x0BU, 0x59U, 0xFCU, 0xE0U, 0x17U, 0xE4U,
	0x97U, 0x05U, 0x45U, 0xC5U, 0xEBU, 0x00U, 0x3DU, 0x81U,
	0xDCU, 0xC7U, 0x49U, 0x2EU, 0xD8U, 0x23U, 0x6EU, 0xDDU,
	0x3EU, 0x6EU, 0x75U, 0xB7U, 0x42U, 0xD1U, 0xF3U, 0xCFU,
	0xBCU, 0x67U, 0xE5U, 0xA8U, 0x61U, 0x6DU, 0xCCU, 0x5BU,
	0x45U, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D12[] = {
	0x09U, 0x64U, 0x38U, 0x91U, 0x1EU, 0xB7U, 0x6FU, 0xBBU,
	0xAEU, 0x47U, 0x9CU, 0x89U, 0xB8U, 0xC9U, 0xB5U, 0x3BU,
	0xD0U, 0xA5U, 0x09U, 0xF7U, 0x48U, 0x01U, 0xCCU, 0x7FU,
	0x6BU, 0x96U, 0x2FU, 0xBFU, 0x83U, 0x87U, 0x86U, 0x51U,
	0xFAU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0x01U, 0x00U, 0x00U
};

static const u8 XRsa_EccP521D13[] = {
	18U, 6U, 3U, 1U,
};
#endif /* XRSA_ECC_SUPPORT_NIST_P521 */

#ifdef XRSA_ECC_SUPPORT_NIST_P192
static const u8 XRsa_EccP192D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xA8U, 0x14U, 0x15U, 0x79U,
	0xE4U, 0x3DU, 0xF0U, 0x9FU, 0xB1U, 0xA1U, 0x78U, 0xFAU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x2EU, 0xCFU, 0xDBU, 0x0DU,
};

static const u8 XRsa_EccP192D1[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFEU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP192D2[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFBU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFCU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP192D3[] = {
	0x31U, 0xB1U, 0xC0U, 0x57U, 0xC5U, 0xEEU, 0xC8U, 0x73U,
	0xC9U, 0xEBU, 0xA3U, 0xFEU, 0xCBU, 0xD2U, 0xA2U, 0xD6U,
	0xE2U, 0x6AU, 0x07U, 0x19U, 0xB2U, 0xCDU, 0x81U, 0x72U,
};

static const u8 XRsa_EccP192D4[] = {
	0x12U, 0x10U, 0xFFU, 0x82U, 0xFDU, 0x0AU, 0xFFU, 0xF4U,
	0x00U, 0x88U, 0xA1U, 0x43U, 0xEBU, 0x20U, 0xBFU, 0x7CU,
	0xF6U, 0x90U, 0x30U, 0xB0U, 0x0EU, 0xA8U, 0x8DU, 0x18U,
};

static const u8 XRsa_EccP192D5[] = {
	0x11U, 0x48U, 0x79U, 0x1EU, 0xA1U, 0x77U, 0xF9U, 0x73U,
	0xD5U, 0xCDU, 0x24U, 0x6BU, 0xEDU, 0x11U, 0x10U, 0x63U,
	0x78U, 0xDAU, 0xC8U, 0xFFU, 0x95U, 0x2BU, 0x19U, 0x07U,
};

static const u8 XRsa_EccP192D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFEU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP192D7[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP192D8[] = {
	0x02U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x03U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x02U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP192D9[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0xCFU, 0xD7U, 0x2DU, 0x4BU, 0x4EU, 0x36U, 0x94U, 0xEBU,
	0xC9U, 0x07U, 0x21U, 0x66U, 0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP192D10[] = {
	0x27U, 0x4CU, 0x13U, 0x83U, 0x0AU, 0x77U, 0xD1U, 0x01U,
	0x7FU, 0x68U, 0xAFU, 0xCAU, 0x61U, 0x69U, 0x9CU, 0xD6U,
	0xC5U, 0xD8U, 0xF5U, 0xCEU, 0xC4U, 0x92U, 0x67U, 0x12U,
};

static const u8 XRsa_EccP192D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP192D12[] = {
	0x31U, 0x28U, 0xD2U, 0xB4U, 0xB1U, 0xC9U, 0x6BU, 0x14U,
	0x36U, 0xF8U, 0xDEU, 0x99U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP192D13[] = {
	7U, 2U, 1U, 2U,
};
#endif /* XRSA_ECC_SUPPORT_NIST_P192 */

#ifdef XRSA_ECC_SUPPORT_NIST_P224
static const u8 XRsa_EccP224D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x4FU, 0xCBU, 0xEFU, 0x8CU,
	0xE8U, 0x8EU, 0x91U, 0x3BU, 0xAFU, 0xFFU, 0x16U, 0x7DU,
	0x42U, 0x9FU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xEAU, 0xC2U, 0x1FU, 0x6AU,
};

static const u8 XRsa_EccP224D1[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP224D2[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x03U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFCU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP224D3[] = {
	0x6DU, 0xD0U, 0x3FU, 0x80U, 0xCCU, 0x59U, 0xC0U, 0x63U,
	0xF6U, 0xCDU, 0x68U, 0xE7U, 0xA3U, 0x42U, 0xB0U, 0x4CU,
	0xF4U, 0xC2U, 0x7AU, 0x10U, 0x51U, 0x81U, 0x52U, 0xC8U,
	0x98U, 0xBAU, 0xCEU, 0x3DU,
};

static const u8 XRsa_EccP224D4[] = {
	0x21U, 0x1DU, 0x5CU, 0x11U, 0xD6U, 0x80U, 0x32U, 0x34U,
	0x22U, 0x11U, 0xC2U, 0x56U, 0xD3U, 0xC1U, 0x03U, 0x4AU,
	0xB9U, 0x90U, 0x13U, 0x32U, 0x7FU, 0xBFU, 0xB4U, 0x6BU,
	0xBDU, 0x0CU, 0x0EU, 0xB7U,
};

static const u8 XRsa_EccP224D5[] = {
	0x34U, 0x7EU, 0x00U, 0x85U, 0x99U, 0x81U, 0xD5U, 0x44U,
	0x64U, 0x47U, 0x07U, 0x5AU, 0xA0U, 0x75U, 0x43U, 0xCDU,
	0xE6U, 0xDFU, 0x22U, 0x4CU, 0xFBU, 0x23U, 0xF7U, 0xB5U,
	0x88U, 0x63U, 0x37U, 0xBDU,
};

static const u8 XRsa_EccP224D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x01U, 0x00U, 0x00U, 0x00U,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP224D7[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP224D8[] = {
	0x02U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFDU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP224D9[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0xC3U, 0xD5U, 0xA3U, 0xA3U, 0xBAU, 0xD6U, 0x22U, 0xECU,
	0xC1U, 0x0FU, 0x47U, 0x1FU, 0x5DU, 0xE9U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP224D10[] = {
	0x74U, 0xE7U, 0xE1U, 0x51U, 0x48U, 0xDEU, 0x48U, 0xB7U,
	0xF8U, 0x15U, 0x63U, 0x79U, 0xEDU, 0xAAU, 0xBCU, 0x70U,
	0x4EU, 0xAAU, 0xDDU, 0x68U, 0x87U, 0xA1U, 0xC9U, 0xABU,
	0x56U, 0x48U, 0x71U, 0xD9U,
};

static const u8 XRsa_EccP224D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U,
};

static const u8 XRsa_EccP224D12[] = {
	0x3DU, 0x2AU, 0x5CU, 0x5CU, 0x45U, 0x29U, 0xDDU, 0x13U,
	0x3EU, 0xF0U, 0xB8U, 0xE0U, 0xA2U, 0x16U, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU,
};

static const u8 XRsa_EccP224D13[] = {
	8U, 3U, 1U, 0U,
};
#endif /* XRSA_ECC_SUPPORT_NIST_P224 */

#ifdef XRSA_ECC_SUPPORT_NIST_P256
static const u8 XRsa_EccP256D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x35U, 0xF0U, 0xE0U, 0xEDU,
	0x84U, 0x48U, 0x4AU, 0x83U, 0xABU, 0xD7U, 0xB6U, 0x5CU,
	0xB2U, 0x8BU, 0x9EU, 0xC1U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x4EU, 0xBCU, 0x00U, 0xEEU, 0xB1U, 0x43U, 0xFFU, 0x11U,
	0x4EU, 0xBCU, 0x00U, 0xEEU,
};
static const u8 XRsa_EccP256D1[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
};
static const u8 XRsa_EccP256D2[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0xFDU, 0xFFU, 0xFFU, 0xFFU,
	0xFCU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x02U, 0x00U, 0x00U, 0x00U, 0x03U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x03U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccP256D3[] = {
	0x51U, 0x4EU, 0xB7U, 0xE0U, 0x1DU, 0x06U, 0x30U, 0xDCU,
	0xDFU, 0xBDU, 0xC4U, 0x29U, 0x11U, 0x91U, 0xE5U, 0xF7U,
	0x72U, 0x2AU, 0x54U, 0x9CU, 0xCCU, 0x05U, 0xF0U, 0xACU,
	0x85U, 0xE0U, 0x69U, 0x16U, 0xDFU, 0x68U, 0x29U, 0xEAU,
};
static const u8 XRsa_EccP256D4[] = {
	0x96U, 0xC2U, 0x98U, 0xD8U, 0x45U, 0x39U, 0xA1U, 0xF4U,
	0xA0U, 0x33U, 0xEBU, 0x2DU, 0x81U, 0x7DU, 0x03U, 0x77U,
	0xF2U, 0x40U, 0xA4U, 0x63U, 0xE5U, 0xE6U, 0xBCU, 0xF8U,
	0x47U, 0x42U, 0x2CU, 0xE1U, 0xF2U, 0xD1U, 0x17U, 0x6BU,
};
static const u8 XRsa_EccP256D5[] = {
	0xF5U, 0x51U, 0xBFU, 0x37U, 0x68U, 0x40U, 0xB6U, 0xCBU,
	0xCEU, 0x5EU, 0x31U, 0x6BU, 0x57U, 0x33U, 0xCEU, 0x2BU,
	0x16U, 0x9EU, 0x0FU, 0x7CU, 0x4AU, 0xEBU, 0xE7U, 0x8EU,
	0x9BU, 0x7FU, 0x1AU, 0xFEU, 0xE2U, 0x42U, 0xE3U, 0x4FU,
};
static const u8 XRsa_EccP256D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccP256D7[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0x01U, 0x00U, 0x00U, 0x00U,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
};
static const u8 XRsa_EccP256D8[] = {
	0xF9U, 0xFFU, 0xFFU, 0xFFU, 0xFDU, 0xFFU, 0xFFU, 0xFFU,
	0x02U, 0x00U, 0x00U, 0x00U, 0x0CU, 0x00U, 0x00U, 0x00U,
	0x05U, 0x00U, 0x00U, 0x00U, 0xFDU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0xF7U, 0xFFU, 0xFFU, 0xFFU,
};
static const u8 XRsa_EccP256D9[] = {
	0x51U, 0x25U, 0x63U, 0xFCU, 0x71U, 0xA5U, 0x56U, 0xF7U,
	0x70U, 0xAEU, 0xFAU, 0xB6U, 0x65U, 0x91U, 0x15U, 0x22U,
	0xCDU, 0x66U, 0x01U, 0x9CU, 0x52U, 0x05U, 0x19U, 0x43U,
	0x01U, 0x00U, 0x00U, 0x00U, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
};
static const u8 XRsa_EccP256D10[] = {
	0xF3U, 0xAEU, 0xE0U, 0x33U, 0x14U, 0x7CU, 0x86U, 0x1DU,
	0xD4U, 0xD5U, 0x21U, 0xEFU, 0xA5U, 0xF1U, 0x1EU, 0xECU,
	0x83U, 0x16U, 0x32U, 0xEEU, 0xE7U, 0x7DU, 0x70U, 0xE6U,
	0xDEU, 0xFAU, 0x00U, 0x65U, 0x4FU, 0xDFU, 0x41U, 0x32U,
};
static const u8 XRsa_EccP256D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccP256D12[] = {
	0x51U, 0x25U, 0x63U, 0xFCU, 0xC2U, 0xCAU, 0xB9U, 0xF3U,
	0x84U, 0x9EU, 0x17U, 0xA7U, 0xADU, 0xFAU, 0xE6U, 0xBCU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
};
static const u8 XRsa_EccP256D13[] = {
	0x09U, 0x03U, 0x01U, 0x01u,
};
#endif /* XRSA_ECC_SUPPORT_NIST_P256 */

#ifdef XRSA_ECC_SUPPORT_NIST_P384
static const u8 XRsa_EccP384D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x74U, 0xB1U, 0xB5U, 0xBDU,
	0xA7U, 0x16U, 0xEBU, 0x5BU, 0xFAU, 0x99U, 0x41U, 0x5BU,
	0xE0U, 0x1FU, 0xE5U, 0x45U, 0x4BU, 0xFEU, 0x48U, 0x25U,
	0x00U, 0x2EU, 0x92U, 0xCCU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x44U, 0xDCU, 0x8FU, 0xE8U
};

static const u8 XRsa_EccP384D1[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFEU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU
};

static const u8 XRsa_EccP384D2[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0xFDU, 0xFFU, 0xFFU, 0xFFU, 0x01U, 0x00U, 0x00U, 0x00U,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFCU, 0xFFU, 0xFFU, 0xFFU,
	0xFCU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU
};

static const u8 XRsa_EccP384D3[] = {
	0xF9U, 0xBFU, 0x4FU, 0x60U, 0x52U, 0x51U, 0xB8U, 0x6CU,
	0x81U, 0x1CU, 0x39U, 0xD0U, 0x69U, 0x48U, 0x61U, 0x68U,
	0x30U, 0x04U, 0xA4U, 0xA7U, 0x24U, 0xBFU, 0x31U, 0xC4U,
	0x2FU, 0x02U, 0x20U, 0x19U, 0x9BU, 0x20U, 0xF2U, 0x77U,
	0xE2U, 0x8AU, 0x93U, 0x94U, 0xEEU, 0x4BU, 0x37U, 0xE3U,
	0x94U, 0x20U, 0x02U, 0x1FU, 0xF4U, 0x21U, 0x2BU, 0xB6U
};

static const u8 XRsa_EccP384D4[] = {
	0xB7U, 0x0AU, 0x76U, 0x72U, 0x38U, 0x5EU, 0x54U, 0x3AU,
	0x6CU, 0x29U, 0x55U, 0xBFU, 0x5DU, 0xF2U, 0x02U, 0x55U,
	0x38U, 0x2AU, 0x54U, 0x82U, 0xE0U, 0x41U, 0xF7U, 0x59U,
	0x98U, 0x9BU, 0xA7U, 0x8BU, 0x62U, 0x3BU, 0x1DU, 0x6EU,
	0x74U, 0xADU, 0x20U, 0xF3U, 0x1EU, 0xC7U, 0xB1U, 0x8EU,
	0x37U, 0x05U, 0x8BU, 0xBEU, 0x22U, 0xCAU, 0x87U, 0xAAU
};

static const u8 XRsa_EccP384D5[] = {
	0x5FU, 0x0EU, 0xEAU, 0x90U, 0x7CU, 0x1DU, 0x43U, 0x7AU,
	0x9DU, 0x81U, 0x7EU, 0x1DU, 0xCEU, 0xB1U, 0x60U, 0x0AU,
	0xC0U, 0xB8U, 0xF0U, 0xB5U, 0x13U, 0x31U, 0xDAU, 0xE9U,
	0x7CU, 0x14U, 0x9AU, 0x28U, 0xBDU, 0x1DU, 0xF4U, 0xF8U,
	0x29U, 0xDCU, 0x92U, 0x92U, 0xBFU, 0x98U, 0x9EU, 0x5DU,
	0x6FU, 0x2CU, 0x26U, 0x96U, 0x4AU, 0xDEU, 0x17U, 0x36U
};

static const u8 XRsa_EccP384D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFEU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP384D7[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP384D8[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x00U, 0x00U, 0x00U, 0x00U,
	0x02U, 0x00U, 0x00U, 0x00U, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0x02U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0xFEU, 0xFFU, 0xFFU, 0xFFU,
	0x00U, 0x00U, 0x00U, 0x00U, 0x02U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP384D9[] = {
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x8DU, 0xD6U, 0x3AU, 0x33U, 0x95U, 0xE6U, 0x13U, 0x13U,
	0x85U, 0x58U, 0x4FU, 0xB7U, 0x4DU, 0xF2U, 0xE5U, 0xA7U,
	0x20U, 0xD2U, 0xC8U, 0x0BU, 0x7EU, 0xB2U, 0x9CU, 0x38U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP384D10[] = {
	0x11U, 0x28U, 0x56U, 0x99U, 0x1AU, 0xFCU, 0xA6U, 0x42U,
	0xEDU, 0x00U, 0x90U, 0x39U, 0x15U, 0xF7U, 0xBCU, 0x43U,
	0x8EU, 0x2DU, 0x2DU, 0xA7U, 0x8AU, 0xC9U, 0x42U, 0x0CU,
	0x92U, 0xE7U, 0xA0U, 0xF4U, 0xF9U, 0x01U, 0x96U, 0xCCU,
	0x43U, 0x9FU, 0x60U, 0xDEU, 0x06U, 0x06U, 0x03U, 0xBFU,
	0xC5U, 0x1CU, 0xABU, 0x4AU, 0x17U, 0x49U, 0x0DU, 0xD4U
};

static const u8 XRsa_EccP384D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U
};

static const u8 XRsa_EccP384D12[] = {
	0x73U, 0x29U, 0xC5U, 0xCCU, 0x6AU, 0x19U, 0xECU, 0xECU,
	0x7AU, 0xA7U, 0xB0U, 0x48U, 0xB2U, 0x0DU, 0x1AU, 0x58U,
	0xDFU, 0x2DU, 0x37U, 0xF4U, 0x81U, 0x4DU, 0x63U, 0xC7U,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU,
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFFU
};

static const u8 XRsa_EccP384D13[] = {
	13U, 4U, 2U, 2U,
};
#endif /* XRSA_ECC_SUPPORT_NIST_P384 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P256
static const u8 XRsa_EccBrainPoolP256D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xB4U, 0x14U, 0xEFU, 0x91U,
	0xD7U, 0xABU, 0x5BU, 0x22U, 0xFBU, 0x8AU, 0xF5U, 0xD0U,
	0xADU, 0x1EU, 0x80U, 0x0CU, 0xE5U, 0x97U, 0x09U, 0x7AU,
	0xDEU, 0x91U, 0x57U, 0x0DU, 0x29U, 0x16U, 0x67U, 0x64U,
	0x4BU, 0xDDU, 0x41U, 0x87U,
};
static const u8 XRsa_EccBrainPoolP256D1[] = {
	0x77U, 0x53U, 0x6EU, 0x1FU, 0x1DU, 0x48U, 0x13U, 0x20U,
	0x28U, 0x20U, 0x26U, 0xD5U, 0x23U, 0xF6U, 0x3BU, 0x6EU,
	0x72U, 0x8DU, 0x83U, 0x9DU, 0x90U, 0x0AU, 0x66U, 0x3EU,
	0xBCU, 0xA9U, 0xEEU, 0xA1U, 0xDBU, 0x57U, 0xFBU, 0xA9U,
};
static const u8 XRsa_EccBrainPoolP256D2[] = {
	0xBCU, 0x0DU, 0x8AU, 0xC0U, 0xF8U, 0x39U, 0x08U, 0x91U,
	0x77U, 0x44U, 0x04U, 0x61U, 0x6EU, 0xBFU, 0x3AU, 0xF7U,
	0xE3U, 0x0CU, 0xE0U, 0xADU, 0x27U, 0x13U, 0xBAU, 0x30U,
	0x15U, 0x5FU, 0xD7U, 0x7AU, 0x10U, 0xF6U, 0x53U, 0x69U,
};
static const u8 XRsa_EccBrainPoolP256D3[] = {
	0x05U, 0xB6U, 0xB6U, 0x13U, 0x56U, 0xE7U, 0x46U, 0x9CU,
	0x60U, 0xCAU, 0x6FU, 0xECU, 0x5DU, 0xF6U, 0xFEU, 0x4EU,
	0x62U, 0xC5U, 0x17U, 0xCAU, 0xFFU, 0x5BU, 0xF4U, 0xE4U,
	0xB6U, 0x32U, 0xD3U, 0x8CU, 0xD6U, 0xDFU, 0xB2U, 0x40U,
};
static const u8 XRsa_EccBrainPoolP256D4[] = {
	0x62U, 0x32U, 0xCEU, 0x9AU, 0xBDU, 0x53U, 0x44U, 0x3AU,
	0xC2U, 0x23U, 0xBDU, 0xE3U, 0xE1U, 0x27U, 0xDEU, 0xB9U,
	0xAFU, 0xB7U, 0x81U, 0xFCU, 0x2FU, 0x48U, 0x4BU, 0x2CU,
	0xCBU, 0x57U, 0x7EU, 0xCBU, 0xB9U, 0xAEU, 0xD2U, 0x8BU,
};
static const u8 XRsa_EccBrainPoolP256D5[] = {
	0x97U, 0x69U, 0x04U, 0x2FU, 0xC7U, 0x54U, 0x1DU, 0x5CU,
	0x54U, 0x8EU, 0xEDU, 0x2DU, 0x13U, 0x45U, 0x77U, 0xC2U,
	0xC9U, 0x1DU, 0x61U, 0x14U, 0x1AU, 0x46U, 0xF8U, 0x97U,
	0xFDU, 0xC4U, 0xDAU, 0xC3U, 0x35U, 0xF8U, 0x7EU, 0x54U,
};
static const u8 XRsa_EccBrainPoolP256D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x0FU, 0x8AU, 0x99U, 0x33U,
	0x3FU, 0xEDU, 0x39U, 0x81U, 0x84U, 0x61U, 0xFCU, 0xD6U,
	0x02U, 0xA6U, 0x8EU, 0x19U, 0xA2U, 0x8DU, 0x10U, 0x83U,
	0xC6U, 0x2DU, 0x8EU, 0xF7U, 0xD3U, 0x1CU, 0x04U, 0xE9U,
	0x8AU, 0x99U, 0x70U, 0x89U,
};
static const u8 XRsa_EccBrainPoolP256D7[] = {
	0x6EU, 0x9EU, 0x74U, 0x3EU, 0x53U, 0xB4U, 0xF5U, 0x98U,
	0x23U, 0x41U, 0x14U, 0xE7U, 0x39U, 0xC1U, 0xC0U, 0x38U,
	0x54U, 0xC5U, 0x9CU, 0x51U, 0x74U, 0x50U, 0xBDU, 0x9AU,
	0x31U, 0xBEU, 0xE3U, 0x78U, 0x10U, 0x67U, 0x77U, 0x7CU,
};
static const u8 XRsa_EccBrainPoolP256D8[] = {
	0x57U, 0x1DU, 0x85U, 0x74U, 0x48U, 0xC4U, 0x56U, 0x89U,
	0x89U, 0xEFU, 0xDAU, 0x26U, 0xBBU, 0x74U, 0xDBU, 0x88U,
	0xB9U, 0x56U, 0x4FU, 0x16U, 0x54U, 0xF6U, 0x43U, 0x6CU,
	0x9BU, 0x65U, 0x34U, 0x37U, 0x46U, 0x27U, 0x8BU, 0x59U,
};
static const u8 XRsa_EccBrainPoolP256D9[] = {
	0xCEU, 0xFCU, 0x14U, 0x1EU, 0xD2U, 0xD9U, 0x7BU, 0xC9U,
	0xBFU, 0x0AU, 0xCBU, 0xEBU, 0x94U, 0x09U, 0x29U, 0x65U,
	0x05U, 0x91U, 0xF3U, 0x86U, 0xB9U, 0xB4U, 0x1EU, 0xEFU,
	0x32U, 0xBEU, 0xE3U, 0x78U, 0x10U, 0x67U, 0x77U, 0x7CU,
};
static const u8 XRsa_EccBrainPoolP256D10[] = {
	0x5DU, 0x8FU, 0xE7U, 0x58U, 0x5EU, 0x13U, 0xF6U, 0x2BU,
	0xEBU, 0x7AU, 0xDCU, 0x10U, 0x8EU, 0xFFU, 0xA2U, 0xAFU,
	0x9FU, 0x93U, 0xFEU, 0xA2U, 0x94U, 0xE9U, 0xB0U, 0x0BU,
	0xF3U, 0xF7U, 0x61U, 0x9BU, 0x01U, 0x66U, 0x32U, 0x68U,
};
static const u8 XRsa_EccBrainPoolP256D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccBrainPoolP256D12[] = {
	0xA7U, 0x56U, 0x48U, 0x97U, 0x82U, 0x0EU, 0x1EU, 0x90U,
	0xF7U, 0xA6U, 0x61U, 0xB5U, 0xA3U, 0x7AU, 0x39U, 0x8CU,
	0x71U, 0x8DU, 0x83U, 0x9DU, 0x90U, 0x0AU, 0x66U, 0x3EU,
	0xBCU, 0xA9U, 0xEEU, 0xA1U, 0xDBU, 0x57U, 0xFBU, 0xA9U,
};
static const u8 XRsa_EccBrainPoolP256D13[] = {
	0x09U,  0x03U,  0x01U,	0x01U,
};
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P256 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P320
static const u8 XRsa_EccBrainPoolP320D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x7AU, 0x58U, 0x28U, 0xDFU,
	0x08U, 0xFEU, 0xA7U, 0x86U, 0x29U, 0xD8U, 0x9FU, 0xFAU,
	0xEAU, 0x8CU, 0x3AU, 0x58U, 0xE4U, 0x9CU, 0x1AU, 0x50U,
	0xC5U, 0x89U, 0xA9U, 0x12U, 0x4FU, 0xBDU, 0x99U, 0x34U,
	0x7BU, 0xD3U, 0xB1U, 0x47U, 0x8FU, 0xCBU, 0x12U, 0xA1U,
	0xA5U, 0xEEU, 0x61U, 0xD0U,
};
static const u8 XRsa_EccBrainPoolP320D1[] = {
	0x27U, 0x2EU, 0xB3U, 0xF1U, 0xB1U, 0x12U, 0xD4U, 0xFCU,
	0x28U, 0xECU, 0x93U, 0x78U, 0xECU, 0xB9U, 0x92U, 0x4FU,
	0xEFU, 0x0DU, 0xF4U, 0xF6U, 0xA6U, 0xCFU, 0x8FU, 0xF9U,
	0x65U, 0xE0U, 0x01U, 0xD2U, 0x5EU, 0x78U, 0x3CU, 0xE1U,
	0xB7U, 0x4FU, 0xBCU, 0x36U, 0x20U, 0x47U, 0x5EU, 0xD3U,
};
static const u8 XRsa_EccBrainPoolP320D2[] = {
	0xA3U, 0xD6U, 0x84U, 0xE8U, 0xB0U, 0xDDU, 0x5EU, 0xFAU,
	0xFCU, 0xA7U, 0xB1U, 0xC8U, 0x16U, 0x85U, 0x9DU, 0x5BU,
	0x25U, 0xD0U, 0xFEU, 0x9CU, 0xA4U, 0xB8U, 0x29U, 0x12U,
	0x3DU, 0xFEU, 0xC6U, 0xCCU, 0xAAU, 0x85U, 0x69U, 0x2EU,
	0x34U, 0xF6U, 0x97U, 0x76U, 0xDEU, 0xEAU, 0x43U, 0x9EU,
};
static const u8 XRsa_EccBrainPoolP320D3[] = {
	0xE5U, 0xD6U, 0xB5U, 0x0CU, 0xDEU, 0x35U, 0x9AU, 0x5AU,
	0x76U, 0x3CU, 0x69U, 0xCFU, 0xE9U, 0xF5U, 0xAEU, 0xD3U,
	0x36U, 0x65U, 0xAAU, 0xB0U, 0x8BU, 0x46U, 0x08U, 0xA0U,
	0x34U, 0x0BU, 0x98U, 0x4BU, 0x40U, 0x2FU, 0x17U, 0xE4U,
	0x98U, 0x17U, 0x63U, 0x25U, 0xD4U, 0x26U, 0xFAU, 0x2CU,
};
static const u8 XRsa_EccBrainPoolP320D4[] = {
	0x11U, 0x06U, 0xE2U, 0x39U, 0x0DU, 0x8DU, 0xAFU, 0x10U,
	0xC7U, 0x99U, 0xA5U, 0x10U, 0x2AU, 0x1EU, 0x87U, 0xE7U,
	0xB6U, 0x7EU, 0x08U, 0x0AU, 0xD1U, 0x37U, 0x01U, 0xF2U,
	0xE6U, 0xBFU, 0xE5U, 0x8EU, 0xC4U, 0xBCU, 0x89U, 0x52U,
	0xB8U, 0xD8U, 0x53U, 0xFBU, 0x9AU, 0x7EU, 0xBDU, 0x43U,
};
static const u8 XRsa_EccBrainPoolP320D5[] = {
	0xE1U, 0x8EU, 0x2EU, 0x69U, 0xD1U, 0x45U, 0x52U, 0xD3U,
	0xC7U, 0x6AU, 0xACU, 0xAAU, 0x77U, 0x78U, 0xC7U, 0xA9U,
	0xEAU, 0x82U, 0x71U, 0x11U, 0xEDU, 0xFFU, 0x43U, 0x07U,
	0x5EU, 0x27U, 0x77U, 0x7FU, 0x24U, 0x93U, 0x40U, 0xABU,
	0xC8U, 0x1CU, 0xECU, 0x45U, 0x55U, 0xD0U, 0xFDU, 0x14U,
};
static const u8 XRsa_EccBrainPoolP320D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x51U, 0x2EU, 0x19U, 0xDCU,
	0x3DU, 0x3EU, 0x00U, 0xC8U, 0xE4U, 0x78U, 0x2FU, 0x37U,
	0xDDU, 0x61U, 0xC2U, 0xB0U, 0xBEU, 0x61U, 0xAAU, 0x72U,
	0x71U, 0x19U, 0x2EU, 0xFDU, 0x54U, 0x68U, 0xA9U, 0x25U,
	0xC7U, 0x85U, 0x83U, 0xD0U, 0x95U, 0x74U, 0xE9U, 0x81U,
	0x48U, 0xEBU, 0x1FU, 0x23U,
};
static const u8 XRsa_EccBrainPoolP320D7[] = {
	0x20U, 0xB4U, 0x9DU, 0xC0U, 0xDDU, 0x05U, 0x7EU, 0x74U,
	0xECU, 0xB9U, 0xAEU, 0x05U, 0x8CU, 0x38U, 0x22U, 0xDDU,
	0x87U, 0x4DU, 0xC8U, 0x1CU, 0xADU, 0xA7U, 0x48U, 0x61U,
	0xEAU, 0xD8U, 0xDEU, 0xEDU, 0x93U, 0x6CU, 0x0DU, 0x09U,
	0x42U, 0x5BU, 0x3FU, 0x47U, 0x8FU, 0x95U, 0x26U, 0x36U,
};
static const u8 XRsa_EccBrainPoolP320D8[] = {
	0x0DU, 0x40U, 0x8BU, 0xADU, 0x50U, 0x40U, 0xE0U, 0xB4U,
	0x48U, 0x14U, 0x82U, 0xEFU, 0x24U, 0x0BU, 0x26U, 0x10U,
	0xF4U, 0x00U, 0x40U, 0x5CU, 0x67U, 0xA9U, 0x3AU, 0xC5U,
	0xD4U, 0xA6U, 0x27U, 0x77U, 0x81U, 0x1AU, 0xC2U, 0xDBU,
	0x9CU, 0xC0U, 0xA2U, 0x97U, 0x6EU, 0x05U, 0x3AU, 0x40U,
};
static const u8 XRsa_EccBrainPoolP320D9[] = {
	0xE0U, 0xECU, 0x15U, 0x97U, 0xC8U, 0x73U, 0xC4U, 0x29U,
	0x84U, 0xACU, 0x4CU, 0xD9U, 0x11U, 0x28U, 0x5CU, 0x87U,
	0xBDU, 0xD6U, 0x09U, 0xF2U, 0x65U, 0x67U, 0xA5U, 0x34U,
	0xEFU, 0x11U, 0xEBU, 0x71U, 0x95U, 0x6CU, 0x0DU, 0x09U,
	0x42U, 0x5BU, 0x3FU, 0x47U, 0x8FU, 0x95U, 0x26U, 0x36U,
};
static const u8 XRsa_EccBrainPoolP320D10[] = {
	0x17U, 0x3FU, 0x16U, 0x66U, 0x1DU, 0xACU, 0xB6U, 0xB9U,
	0xA0U, 0xF1U, 0xA3U, 0xE4U, 0x95U, 0x02U, 0x86U, 0xDDU,
	0x15U, 0xCCU, 0x8BU, 0x45U, 0x66U, 0x16U, 0xADU, 0x07U,
	0x14U, 0x95U, 0x0BU, 0xB0U, 0xFEU, 0x96U, 0x53U, 0x3BU,
	0x95U, 0x8AU, 0x66U, 0xD2U, 0xB6U, 0x3DU, 0x0DU, 0xA7U,
};
static const u8 XRsa_EccBrainPoolP320D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccBrainPoolP320D12[] = {
	0x11U, 0x93U, 0xC5U, 0x44U, 0x5BU, 0x55U, 0x91U, 0x86U,
	0xE9U, 0x58U, 0x86U, 0xEEU, 0xC7U, 0x2EU, 0x48U, 0x2DU,
	0xA3U, 0x12U, 0x8FU, 0xB6U, 0xA5U, 0xCFU, 0x8FU, 0xF9U,
	0x65U, 0xE0U, 0x01U, 0xD2U, 0x5EU, 0x78U, 0x3CU, 0xE1U,
	0xB7U, 0x4FU, 0xBCU, 0x36U, 0x20U, 0x47U, 0x5EU, 0xD3U,
};
static const u8 XRsa_EccBrainPoolP320D13[] = {
	0x0BU, 0x04U, 0x01U, 0x00U,
};
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P320 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P384
static const u8 XRsa_EccBrainPoolP384D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x6DU, 0x2CU, 0x89U, 0x13U,
	0x13U, 0xE7U, 0xB8U, 0xCDU, 0xAFU, 0xF5U, 0xD1U, 0x16U,
	0xE6U, 0xD1U, 0x01U, 0xBCU, 0x84U, 0xE6U, 0x31U, 0x5DU,
	0x1BU, 0xABU, 0x93U, 0x8EU, 0xD9U, 0x44U, 0x1FU, 0x4FU,
	0xBDU, 0xCEU, 0xACU, 0x64U, 0x19U, 0x3DU, 0x7AU, 0x4CU,
	0x5BU, 0x62U, 0xFBU, 0x83U, 0x5DU, 0xA0U, 0x9BU, 0x8CU,
	0xF5U, 0x6CU, 0xF6U, 0x32U
};
static const u8 XRsa_EccBrainPoolP384D1[] = {
	0x53U, 0xECU, 0x07U, 0x31U, 0x13U, 0x00U, 0x47U, 0x87U,
	0x71U, 0x1AU, 0x1DU, 0x90U, 0x29U, 0xA7U, 0xD3U, 0xACU,
	0x23U, 0x11U, 0xB7U, 0x7FU, 0x19U, 0xDAU, 0xB1U, 0x12U,
	0xB4U, 0x56U, 0x54U, 0xEDU, 0x09U, 0x71U, 0x2FU, 0x15U,
	0xDFU, 0x41U, 0xE6U, 0x50U, 0x7EU, 0x6FU, 0x5DU, 0x0FU,
	0x28U, 0x6DU, 0x38U, 0xA3U, 0x82U, 0x1EU, 0xB9U, 0x8CU
};
static const u8 XRsa_EccBrainPoolP384D2[] = {
	0x6FU, 0x1EU, 0xFEU, 0x94U, 0xA8U, 0x56U, 0xEBU, 0xA5U,
	0x82U, 0xFDU, 0x0AU, 0x3FU, 0x62U, 0x8EU, 0x11U, 0x33U,
	0xB9U, 0x33U, 0xB8U, 0x7FU, 0xD1U, 0xD4U, 0xB4U, 0x6BU,
	0x46U, 0xA0U, 0x9EU, 0x41U, 0xB9U, 0x37U, 0x43U, 0x68U,
	0xBBU, 0x20U, 0xF8U, 0x15U, 0xD9U, 0x5DU, 0xF0U, 0xAFU,
	0x07U, 0xB9U, 0xFDU, 0xC9U, 0xDFU, 0x6CU, 0xC1U, 0x38U
};
static const u8 XRsa_EccBrainPoolP384D3[] = {
	0xF4U, 0xA8U, 0x5EU, 0xB4U, 0x43U, 0x2BU, 0x0FU, 0x31U,
	0xCCU, 0x86U, 0x93U, 0x62U, 0xFDU, 0x43U, 0x77U, 0x88U,
	0xA6U, 0x86U, 0x11U, 0x70U, 0xE7U, 0x2FU, 0x0FU, 0x0FU,
	0x7BU, 0xF1U, 0x1BU, 0xC8U, 0x38U, 0x29U, 0xE0U, 0x09U,
	0x07U, 0x4FU, 0x53U, 0x44U, 0x81U, 0x51U, 0x1CU, 0x67U,
	0x92U, 0xABU, 0xA6U, 0x9CU, 0x7BU, 0x12U, 0x3BU, 0x57U
};
static const u8 XRsa_EccBrainPoolP384D4[] = {
	0x1EU, 0xAFU, 0xD4U, 0x47U, 0xE2U, 0xB2U, 0x87U, 0xEFU,
	0xAAU, 0x46U, 0xD6U, 0x36U, 0x34U, 0xE0U, 0x26U, 0xE8U,
	0xE8U, 0x10U, 0xBDU, 0x0CU, 0xFEU, 0xCAU, 0x7FU, 0xDBU,
	0xE3U, 0x4FU, 0xF1U, 0x7EU, 0xE7U, 0xA3U, 0x47U, 0x88U,
	0x6BU, 0x3FU, 0xC1U, 0xB7U, 0x81U, 0x3AU, 0xA6U, 0xA2U,
	0xFFU, 0x45U, 0xCFU, 0x68U, 0xF0U, 0x64U, 0x1CU, 0x1DU
};
static const u8 XRsa_EccBrainPoolP384D5[] = {
	0x15U, 0x53U, 0x3CU, 0x26U, 0x41U, 0x03U, 0x82U, 0x42U,
	0x11U, 0x81U, 0x91U, 0x77U, 0x21U, 0x46U, 0x46U, 0x0EU,
	0x28U, 0x29U, 0x91U, 0xF9U, 0x4FU, 0x05U, 0x9CU, 0xE1U,
	0x64U, 0x58U, 0xECU, 0xFEU, 0x29U, 0x0BU, 0xB7U, 0x62U,
	0x52U, 0xD5U, 0xCFU, 0x95U, 0x8EU, 0xEBU, 0xB1U, 0x5CU,
	0xA4U, 0xC2U, 0xF9U, 0x20U, 0x75U, 0x1DU, 0xBEU, 0x8AU
};
static const u8 XRsa_EccBrainPoolP384D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0xFBU, 0x81U, 0xFBU, 0x9BU,
	0xFBU, 0xE7U, 0x95U, 0xA8U, 0xBAU, 0x2DU, 0x91U, 0x7EU,
	0x9AU, 0x7AU, 0xD7U, 0xAAU, 0x1DU, 0x9AU, 0x92U, 0x90U,
	0xC1U, 0x51U, 0x27U, 0x4DU, 0x27U, 0xA9U, 0x3AU, 0xF7U,
	0x65U, 0x3EU, 0xCAU, 0x25U, 0xC8U, 0x32U, 0x84U, 0x7DU,
	0x5FU, 0xF6U, 0x33U, 0xA9U, 0xFFU, 0xC8U, 0x69U, 0x34U,
	0x29U, 0x7EU, 0xF8U, 0x80U
};
static const u8 XRsa_EccBrainPoolP384D7[] = {
	0x58U, 0xF2U, 0x7BU, 0x34U, 0xE1U, 0x33U, 0x38U, 0x45U,
	0xD1U, 0x09U, 0x4EU, 0x02U, 0x9BU, 0x98U, 0xF9U, 0xCEU,
	0xE6U, 0xC6U, 0xE5U, 0x01U, 0x1EU, 0xEFU, 0x67U, 0x48U,
	0x08U, 0x0AU, 0xD2U, 0xB1U, 0xA7U, 0x01U, 0xB4U, 0x20U,
	0x61U, 0x7AU, 0x39U, 0xF5U, 0x14U, 0x9AU, 0x95U, 0x1CU,
	0x44U, 0xFFU, 0x70U, 0xD7U, 0xC9U, 0x1BU, 0x41U, 0x8CU
};
static const u8 XRsa_EccBrainPoolP384D8[] = {
	0xC4U, 0x06U, 0x13U, 0xE5U, 0x4EU, 0x5FU, 0xB8U, 0x41U,
	0x95U, 0x5CU, 0x44U, 0x64U, 0x87U, 0x0BU, 0x8BU, 0x3CU,
	0x57U, 0x63U, 0x1FU, 0xCEU, 0x30U, 0x43U, 0x46U, 0x06U,
	0x01U, 0x4DU, 0xA4U, 0x0EU, 0xD0U, 0x5BU, 0x76U, 0x5AU,
	0x2EU, 0xABU, 0x3CU, 0xC4U, 0xDEU, 0x5FU, 0x8DU, 0xFFU,
	0xFCU, 0xA8U, 0x6AU, 0xE2U, 0x61U, 0x6DU, 0xD2U, 0x76U
};
static const u8 XRsa_EccBrainPoolP384D9[] = {
	0x68U, 0x01U, 0x65U, 0xDFU, 0x3DU, 0x6AU, 0xCFU, 0xA0U,
	0xA8U, 0x96U, 0x9DU, 0xC1U, 0x48U, 0x3AU, 0x55U, 0x80U,
	0xB1U, 0xA4U, 0x33U, 0xE8U, 0x6FU, 0x81U, 0xD0U, 0x85U,
	0xABU, 0x2EU, 0x78U, 0x9EU, 0x1EU, 0xFDU, 0xDDU, 0xDBU,
	0x62U, 0x7AU, 0x39U, 0xF5U, 0x14U, 0x9AU, 0x95U, 0x1CU,
	0x44U, 0xFFU, 0x70U, 0xD7U, 0xC9U, 0x1BU, 0x41U, 0x8CU
};
static const u8 XRsa_EccBrainPoolP384D10[] = {
	0xB9U, 0xCFU, 0x9DU, 0xC9U, 0x73U, 0x8FU, 0xC9U, 0xA0U,
	0xF5U, 0x91U, 0x58U, 0xBDU, 0xDFU, 0x75U, 0x9BU, 0x70U,
	0x60U, 0x41U, 0x41U, 0xFCU, 0x3FU, 0x85U, 0xD1U, 0xAFU,
	0x03U, 0x3FU, 0xD1U, 0x9FU, 0x0CU, 0xB1U, 0x12U, 0x18U,
	0x44U, 0x8EU, 0x65U, 0x7CU, 0x38U, 0x01U, 0x23U, 0x63U,
	0xDDU, 0x9BU, 0x36U, 0xCEU, 0xF3U, 0x54U, 0x8EU, 0x5AU
};
static const u8 XRsa_EccBrainPoolP384D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U
};
static const u8 XRsa_EccBrainPoolP384D12[] = {
	0x65U, 0x65U, 0x04U, 0xE9U, 0x02U, 0x32U, 0x88U, 0x3BU,
	0x10U, 0xC3U, 0x7FU, 0x6BU, 0xAFU, 0xB6U, 0x3AU, 0xCFU,
	0xA7U, 0x25U, 0x04U, 0xACU, 0x6CU, 0x6EU, 0x16U, 0x1FU,
	0xB3U, 0x56U, 0x54U, 0xEDU, 0x09U, 0x71U, 0x2FU, 0x15U,
	0xDFU, 0x41U, 0xE6U, 0x50U, 0x7EU, 0x6FU, 0x5DU, 0x0FU,
	0x28U, 0x6DU, 0x38U, 0xA3U, 0x82U, 0x1EU, 0xB9U, 0x8CU
};
static const u8 XRsa_EccBrainPoolP384D13[] = {
	0x0DU, 0x04U, 0x02U, 0x02U,
};
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P384 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P512
static const u8 XRsa_EccBrainPoolP512D0[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x48U, 0x7FU, 0x75U, 0x80U,
	0x50U, 0x6EU, 0x12U, 0x69U, 0x0AU, 0x0EU, 0xB9U, 0x30U,
	0xF9U, 0x24U, 0x93U, 0x9CU, 0xD7U, 0x7EU, 0x1FU, 0x82U,
	0x60U, 0xA3U, 0xF0U, 0x2DU, 0x8FU, 0x9EU, 0x3DU, 0xD3U,
	0x63U, 0x13U, 0x4FU, 0xE4U, 0xCDU, 0x46U, 0xB2U, 0xA3U,
	0xBFU, 0xF0U, 0xC5U, 0xB5U, 0xA0U, 0xFEU, 0xE0U, 0xABU,
	0x81U, 0x14U, 0xC3U, 0x4BU, 0x44U, 0xA7U, 0xC3U, 0x67U,
	0x1CU, 0x12U, 0x07U, 0xAAU, 0x8BU, 0xCBU, 0x25U, 0x74U,
	0x78U, 0x4CU, 0x15U, 0x0AU,
};
static const u8 XRsa_EccBrainPoolP512D1[] = {
	0xF3U, 0x48U, 0x3AU, 0x58U, 0x56U, 0x60U, 0xAAU, 0x28U,
	0x85U, 0xC6U, 0x82U, 0x2DU, 0x2FU, 0xFFU, 0x81U, 0x28U,
	0xE6U, 0x80U, 0xA3U, 0xE6U, 0x2AU, 0xA1U, 0xCDU, 0xAEU,
	0x42U, 0x68U, 0xC6U, 0x9BU, 0x00U, 0x9BU, 0x4DU, 0x7DU,
	0x71U, 0x08U, 0x33U, 0x70U, 0xCAU, 0x9CU, 0x63U, 0xD6U,
	0x0EU, 0xD2U, 0xC9U, 0xB3U, 0xB3U, 0x8DU, 0x30U, 0xCBU,
	0x07U, 0xFCU, 0xC9U, 0x33U, 0xAEU, 0xE6U, 0xD4U, 0x3FU,
	0x8BU, 0xC4U, 0xE9U, 0xDBU, 0xB8U, 0x9DU, 0xDDU, 0xAAU,
};
static const u8 XRsa_EccBrainPoolP512D2[] = {
	0x3AU, 0x19U, 0x72U, 0x48U, 0x18U, 0x2AU, 0x87U, 0xA8U,
	0x46U, 0xCDU, 0xD6U, 0x31U, 0x38U, 0x96U, 0x03U, 0xE2U,
	0x19U, 0xD9U, 0x92U, 0xB2U, 0x9DU, 0x77U, 0x44U, 0xE0U,
	0x4FU, 0xC5U, 0x84U, 0x7FU, 0xCBU, 0xA6U, 0xCEU, 0xB9U,
	0x35U, 0x80U, 0x22U, 0x56U, 0xDDU, 0x09U, 0xF8U, 0x83U,
	0xC3U, 0x57U, 0x4DU, 0x10U, 0x04U, 0x4CU, 0x74U, 0x4AU,
	0x66U, 0x65U, 0x30U, 0x2DU, 0x73U, 0x6EU, 0xA4U, 0xC0U,
	0x61U, 0xD1U, 0xC8U, 0x59U, 0x0CU, 0x5FU, 0xA4U, 0x6BU,
};
static const u8 XRsa_EccBrainPoolP512D3[] = {
	0xEDU, 0xBFU, 0x33U, 0x15U, 0xBFU, 0xD2U, 0x76U, 0x1AU,
	0xD3U, 0x3BU, 0xC4U, 0x97U, 0xB9U, 0x08U, 0x20U, 0xB8U,
	0x34U, 0x2FU, 0xF3U, 0xA0U, 0xF3U, 0xB6U, 0x54U, 0xB8U,
	0xA3U, 0x36U, 0xDEU, 0x66U, 0x46U, 0x1FU, 0x84U, 0xCDU,
	0x3CU, 0x18U, 0x5DU, 0xB7U, 0x04U, 0x4CU, 0x81U, 0x6DU,
	0x1CU, 0x8EU, 0x34U, 0x44U, 0xAEU, 0x60U, 0xDFU, 0x86U,
	0x77U, 0x73U, 0xC0U, 0xB7U, 0xE2U, 0xE2U, 0x12U, 0x7CU,
	0xBAU, 0x3FU, 0xB3U, 0x9DU, 0xC1U, 0xDCU, 0x74U, 0x6BU,
};
static const u8 XRsa_EccBrainPoolP512D4[] = {
	0x22U, 0xF8U, 0xB9U, 0xBCU, 0x09U, 0x22U, 0x35U, 0x8BU,
	0x68U, 0x5EU, 0x6AU, 0x40U, 0x47U, 0x50U, 0x6DU, 0x7CU,
	0x5FU, 0x7DU, 0xB9U, 0x93U, 0x7BU, 0x68U, 0xD1U, 0x50U,
	0x8DU, 0xD4U, 0xD0U, 0xE2U, 0x78U, 0x1FU, 0x3BU, 0xFFU,
	0x8EU, 0x09U, 0xD0U, 0xF4U, 0xEEU, 0x62U, 0x3BU, 0xB4U,
	0xC1U, 0x16U, 0xD9U, 0xB5U, 0x70U, 0x9FU, 0xEDU, 0x85U,
	0x93U, 0x6AU, 0x4CU, 0x9CU, 0x2EU, 0x32U, 0x21U, 0x5AU,
	0x64U, 0xD9U, 0x2EU, 0xD8U, 0xBDU, 0xE4U, 0xAEU, 0x81U,
};
static const u8 XRsa_EccBrainPoolP512D5[] = {
	0x92U, 0x08U, 0xD8U, 0x3AU, 0x0FU, 0x1EU, 0xCDU, 0x78U,
	0x06U, 0x54U, 0xF0U, 0xA8U, 0x2FU, 0x2BU, 0xCAU, 0xD1U,
	0xAEU, 0x63U, 0x27U, 0x8AU, 0xD8U, 0x4BU, 0xCAU, 0x5BU,
	0x5EU, 0x48U, 0x5FU, 0x4AU, 0x49U, 0xDEU, 0xDCU, 0xB2U,
	0x11U, 0x81U, 0x1FU, 0x88U, 0x5BU, 0xC5U, 0x00U, 0xA0U,
	0x1AU, 0x7BU, 0xA5U, 0x24U, 0x00U, 0xF7U, 0x09U, 0xF2U,
	0xFDU, 0x22U, 0x78U, 0xCFU, 0xA9U, 0xBFU, 0xEAU, 0xC0U,
	0xECU, 0x32U, 0x63U, 0x56U, 0x5DU, 0x38U, 0xDEU, 0x7DU,
};
static const u8 XRsa_EccBrainPoolP512D6[] = {
	0xFFU, 0xFFU, 0xFFU, 0xFFU, 0x9FU, 0x6BU, 0x56U, 0x12U,
	0xA7U, 0x09U, 0x17U, 0x5FU, 0xB6U, 0xA0U, 0xBEU, 0x22U,
	0x1AU, 0x38U, 0x1CU, 0x7EU, 0xC6U, 0x56U, 0x47U, 0xB0U,
	0x3FU, 0x70U, 0x0EU, 0x76U, 0xFCU, 0x16U, 0xD1U, 0x1FU,
	0x88U, 0x66U, 0x12U, 0xEBU, 0xF6U, 0x9CU, 0x27U, 0xBBU,
	0x6FU, 0xDEU, 0x09U, 0x6EU, 0xB5U, 0x92U, 0x9CU, 0x44U,
	0xDEU, 0xAAU, 0x1CU, 0x1BU, 0x8FU, 0x7DU, 0xB5U, 0x64U,
	0x52U, 0x5DU, 0x12U, 0xA6U, 0xA2U, 0xC9U, 0x18U, 0x0EU,
	0xA5U, 0x46U, 0xCAU, 0x53U,
};
static const u8 XRsa_EccBrainPoolP512D7[] = {
	0x02U, 0x9CU, 0xAFU, 0xAEU, 0xCFU, 0x2CU, 0xA7U, 0xEFU,
	0xF3U, 0xB6U, 0x9FU, 0xE6U, 0xC2U, 0x36U, 0xF1U, 0xC8U,
	0xBDU, 0x32U, 0x67U, 0x73U, 0x91U, 0x67U, 0xBDU, 0x2EU,
	0x5EU, 0xCCU, 0x22U, 0x25U, 0x26U, 0x9CU, 0xB7U, 0xA1U,
	0x69U, 0x87U, 0x6BU, 0xA6U, 0xFAU, 0x72U, 0x7BU, 0x4DU,
	0x39U, 0xF6U, 0x08U, 0x93U, 0x50U, 0x5EU, 0xCFU, 0x36U,
	0x1BU, 0x14U, 0x39U, 0xA3U, 0xF3U, 0x93U, 0xFBU, 0xA2U,
	0x20U, 0x7AU, 0xE3U, 0x57U, 0xA2U, 0xF4U, 0x92U, 0x0EU,
};
static const u8 XRsa_EccBrainPoolP512D8[] = {
	0x2CU, 0x1CU, 0x4DU, 0xB0U, 0xADU, 0xCCU, 0x12U, 0xE9U,
	0x09U, 0x9BU, 0x14U, 0xBFU, 0xE9U, 0x43U, 0x0DU, 0xE2U,
	0x45U, 0x88U, 0x22U, 0xE8U, 0x90U, 0x4EU, 0xBAU, 0x91U,
	0x94U, 0x2CU, 0xD7U, 0x40U, 0x3FU, 0x16U, 0x89U, 0xABU,
	0xCEU, 0x03U, 0x0CU, 0x37U, 0xBCU, 0x4AU, 0xA1U, 0x97U,
	0xC9U, 0xBCU, 0xEEU, 0xC5U, 0xE9U, 0xA6U, 0xB7U, 0xB9U,
	0x1AU, 0xB1U, 0x3DU, 0x40U, 0x42U, 0x45U, 0x09U, 0x41U,
	0xF9U, 0x94U, 0x1BU, 0x2DU, 0x35U, 0x38U, 0x77U, 0x30U,
};
static const u8 XRsa_EccBrainPoolP512D9[] = {
	0x66U, 0x64U, 0x1BU, 0x4AU, 0x88U, 0x12U, 0x09U, 0x70U,
	0x5BU, 0xC9U, 0xA6U, 0xBDU, 0x6AU, 0x72U, 0x3DU, 0x6DU,
	0x91U, 0x86U, 0x3AU, 0x1DU, 0x27U, 0x0EU, 0xE4U, 0xB2U,
	0x67U, 0x2BU, 0x65U, 0x96U, 0x0DU, 0x60U, 0xA0U, 0x9DU,
	0xE1U, 0xD2U, 0x66U, 0x2AU, 0x7BU, 0xE5U, 0x0DU, 0x09U,
	0x3BU, 0xF6U, 0x08U, 0x93U, 0x50U, 0x5EU, 0xCFU, 0x36U,
	0x1BU, 0x14U, 0x39U, 0xA3U, 0xF3U, 0x93U, 0xFBU, 0xA2U,
	0x20U, 0x7AU, 0xE3U, 0x57U, 0xA2U, 0xF4U, 0x92U, 0x0EU,
};
static const u8 XRsa_EccBrainPoolP512D10[] = {
	0xD3U, 0xC4U, 0xD7U, 0x82U, 0xA5U, 0xF2U, 0x22U, 0xFCU,
	0x09U, 0x65U, 0x05U, 0xFEU, 0xDCU, 0xA5U, 0xA3U, 0xE3U,
	0xF8U, 0x91U, 0x48U, 0x41U, 0x73U, 0xAAU, 0xEEU, 0x20U,
	0xF8U, 0x03U, 0x61U, 0x1EU, 0xB2U, 0x7FU, 0x42U, 0x88U,
	0x73U, 0xAFU, 0xB1U, 0x18U, 0x11U, 0x79U, 0x07U, 0x4DU,
	0x9BU, 0x89U, 0x29U, 0xEBU, 0x14U, 0xE3U, 0x7FU, 0xFDU,
	0x34U, 0xA3U, 0x72U, 0x74U, 0x5BU, 0x2BU, 0xDBU, 0x46U,
	0x2CU, 0xF6U, 0x27U, 0xC6U, 0xC2U, 0x1FU, 0xB4U, 0x55U,
};
static const u8 XRsa_EccBrainPoolP512D11[] = {
	0x01U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
	0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U, 0x00U,
};
static const u8 XRsa_EccBrainPoolP512D12[] = {
	0x69U, 0x00U, 0xA9U, 0x9CU, 0x82U, 0x96U, 0x87U, 0xB5U,
	0xDDU, 0xDAU, 0x5DU, 0x08U, 0x81U, 0xD3U, 0xB1U, 0x1DU,
	0x47U, 0x10U, 0xACU, 0x7FU, 0x19U, 0x61U, 0x86U, 0x41U,
	0x19U, 0x26U, 0xA9U, 0x4CU, 0x41U, 0x5CU, 0x3EU, 0x55U,
	0x70U, 0x08U, 0x33U, 0x70U, 0xCAU, 0x9CU, 0x63U, 0xD6U,
	0x0EU, 0xD2U, 0xC9U, 0xB3U, 0xB3U, 0x8DU, 0x30U, 0xCBU,
	0x07U, 0xFCU, 0xC9U, 0x33U, 0xAEU, 0xE6U, 0xD4U, 0x3FU,
	0x8BU, 0xC4U, 0xE9U, 0xDBU, 0xB8U, 0x9DU, 0xDDU, 0xAAU,
};
static const u8 XRsa_EccBrainPoolP512D13[] = {
	0x11U, 0x06U, 0x02U, 0x00U,
};
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P512 */

/**************************************** Type Definitions ***************************************/
EcdsaCrvInfo XRsa_EccCrvsDb[] = {
#ifdef XRSA_ECC_SUPPORT_NIST_P521
	{
		ECDSA_NIST_P521,
		521U,
		ECDSA_PRIME,
		XRsa_EccP521D0,
		XRsa_EccP521D1,
		XRsa_EccP521D2,
		XRsa_EccP521D3,
		XRsa_EccP521D4,
		XRsa_EccP521D5,
		XRsa_EccP521D6,
		XRsa_EccP521D7,
		XRsa_EccP521D8,
		XRsa_EccP521D9,
		XRsa_EccP521D10,
		XRsa_EccP521D11,
		XRsa_EccP521D12,
		XRsa_EccP521D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_NIST_P521 */

#ifdef XRSA_ECC_SUPPORT_NIST_P192
	{
		ECDSA_NIST_P192,
		192U,
		ECDSA_PRIME,
		XRsa_EccP192D0,
		XRsa_EccP192D1,
		XRsa_EccP192D2,
		XRsa_EccP192D3,
		XRsa_EccP192D4,
		XRsa_EccP192D5,
		XRsa_EccP192D6,
		XRsa_EccP192D7,
		XRsa_EccP192D8,
		XRsa_EccP192D9,
		XRsa_EccP192D10,
		XRsa_EccP192D11,
		XRsa_EccP192D12,
		XRsa_EccP192D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_NIST_P192 */

#ifdef XRSA_ECC_SUPPORT_NIST_P224
	{
		ECDSA_NIST_P224,
		224U,
		ECDSA_PRIME,
		XRsa_EccP224D0,
		XRsa_EccP224D1,
		XRsa_EccP224D2,
		XRsa_EccP224D3,
		XRsa_EccP224D4,
		XRsa_EccP224D5,
		XRsa_EccP224D6,
		XRsa_EccP224D7,
		XRsa_EccP224D8,
		XRsa_EccP224D9,
		XRsa_EccP224D10,
		XRsa_EccP224D11,
		XRsa_EccP224D12,
		XRsa_EccP224D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_NIST_P224 */

#ifdef XRSA_ECC_SUPPORT_NIST_P256
	{
		ECDSA_NIST_P256,
		256U,
		ECDSA_PRIME,
		XRsa_EccP256D0,
		XRsa_EccP256D1,
		XRsa_EccP256D2,
		XRsa_EccP256D3,
		XRsa_EccP256D4,
		XRsa_EccP256D5,
		XRsa_EccP256D6,
		XRsa_EccP256D7,
		XRsa_EccP256D8,
		XRsa_EccP256D9,
		XRsa_EccP256D10,
		XRsa_EccP256D11,
		XRsa_EccP256D12,
		XRsa_EccP256D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_NIST_P256 */

#ifdef XRSA_ECC_SUPPORT_NIST_P384
	{
		ECDSA_NIST_P384,
		384U,
		ECDSA_PRIME,
		XRsa_EccP384D0,
		XRsa_EccP384D1,
		XRsa_EccP384D2,
		XRsa_EccP384D3,
		XRsa_EccP384D4,
		XRsa_EccP384D5,
		XRsa_EccP384D6,
		XRsa_EccP384D7,
		XRsa_EccP384D8,
		XRsa_EccP384D9,
		XRsa_EccP384D10,
		XRsa_EccP384D11,
		XRsa_EccP384D12,
		XRsa_EccP384D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_NIST_P384 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P256
	{
		ECDSA_BRAINPOOL_P256,
		256U,
		ECDSA_PRIME,
		XRsa_EccBrainPoolP256D0,
		XRsa_EccBrainPoolP256D1,
		XRsa_EccBrainPoolP256D2,
		XRsa_EccBrainPoolP256D3,
		XRsa_EccBrainPoolP256D4,
		XRsa_EccBrainPoolP256D5,
		XRsa_EccBrainPoolP256D6,
		XRsa_EccBrainPoolP256D7,
		XRsa_EccBrainPoolP256D8,
		XRsa_EccBrainPoolP256D9,
		XRsa_EccBrainPoolP256D10,
		XRsa_EccBrainPoolP256D11,
		XRsa_EccBrainPoolP256D12,
		XRsa_EccBrainPoolP256D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P256 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P320
	{
		ECDSA_BRAINPOOL_P320,
		320U,
		ECDSA_PRIME,
		XRsa_EccBrainPoolP320D0,
		XRsa_EccBrainPoolP320D1,
		XRsa_EccBrainPoolP320D2,
		XRsa_EccBrainPoolP320D3,
		XRsa_EccBrainPoolP320D4,
		XRsa_EccBrainPoolP320D5,
		XRsa_EccBrainPoolP320D6,
		XRsa_EccBrainPoolP320D7,
		XRsa_EccBrainPoolP320D8,
		XRsa_EccBrainPoolP320D9,
		XRsa_EccBrainPoolP320D10,
		XRsa_EccBrainPoolP320D11,
		XRsa_EccBrainPoolP320D12,
		XRsa_EccBrainPoolP320D13,
		NULL,
		NULL,
		0U
	},
#endif /* XRSA_ECC_SUPPORT_BRAINPOOL_P320 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P384
	{
		ECDSA_BRAINPOOL_P384,
		384U,
		ECDSA_PRIME,
		XRsa_EccBrainPoolP384D0,
		XRsa_EccBrainPoolP384D1,
		XRsa_EccBrainPoolP384D2,
		XRsa_EccBrainPoolP384D3,
		XRsa_EccBrainPoolP384D4,
		XRsa_EccBrainPoolP384D5,
		XRsa_EccBrainPoolP384D6,
		XRsa_EccBrainPoolP384D7,
		XRsa_EccBrainPoolP384D8,
		XRsa_EccBrainPoolP384D9,
		XRsa_EccBrainPoolP384D10,
		XRsa_EccBrainPoolP384D11,
		XRsa_EccBrainPoolP384D12,
		XRsa_EccBrainPoolP384D13,
		NULL,
		NULL,
		0U
	},
#endif /*XRSA_ECC_SUPPORT_BRAINPOOL_P384 */

#ifdef XRSA_ECC_SUPPORT_BRAINPOOL_P512
	{
		ECDSA_BRAINPOOL_P512,
		512U,
		ECDSA_PRIME,
		XRsa_EccBrainPoolP512D0,
		XRsa_EccBrainPoolP512D1,
		XRsa_EccBrainPoolP512D2,
		XRsa_EccBrainPoolP512D3,
		XRsa_EccBrainPoolP512D4,
		XRsa_EccBrainPoolP512D5,
		XRsa_EccBrainPoolP512D6,
		XRsa_EccBrainPoolP512D7,
		XRsa_EccBrainPoolP512D8,
		XRsa_EccBrainPoolP512D9,
		XRsa_EccBrainPoolP512D10,
		XRsa_EccBrainPoolP512D11,
		XRsa_EccBrainPoolP512D12,
		XRsa_EccBrainPoolP512D13,
		NULL,
		NULL,
		0U
	},
#endif
}; /* XRSA_ECC_SUPPORT_BRAINPOOL_P512 */

/****************************** Macros (Inline Functions) Definitions ****************************/

/*************************************************************************************************/
/**
 * @brief	This function calculates the total number of curves supported
 *
 * @return
 *	-	Number of curves supported
 *
 *************************************************************************************************/
u32 XRsa_EccCrvsGetCount(void)
{
	return sizeof(XRsa_EccCrvsDb) / sizeof(EcdsaCrvInfo);
}
/** @} */
