/******************************************************************************
* Copyright (C) 2011 - 2020 Xilinx, Inc.  All rights reserved.
* Copyright (C) 2022 - 2023 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

/******************************************************************************/
/**
*
* @file xscugic_tapp_example.c
*
* This file contains a self test example using the Interrupt Controller driver
* (XScuGic) and hardware device. Please reference other device driver examples to
* see more examples of how the Intc and interrupts can be used by a software
* application.
*
* The TestApp Gen utility uses this file to perform the self test and setup
* of Intc for interrupts.
*
* @note
*
* None
*
* <pre>
*
* MODIFICATION HISTORY:
*
* Ver   Who  Date	Changes
* ----- ---- -------- 	--------------------------------------------------------
* 1.00a sdm  05/29/11	Created for Test App Integration
* 3.6   ms   01/23/17   Modified xil_printf statement in main function to
*                       ensure that "Successfully ran" and "Failed" strings
*                       are available in all examples. This is a fix for
*                       CR-965028.
* </pre>
******************************************************************************/

/***************************** Include Files *********************************/

#include "xparameters.h"
#include "xstatus.h"
#include "xscugic.h"
#include "xil_exception.h"

/************************** Constant Definitions *****************************/

/*
 * The following constants map to the XPAR parameters created in the
 * xparameters.h file. They are defined here such that a user can easily
 * change all the needed parameters in one place. This definition is not
 * included if the example is generated from the TestAppGen test tool.
 */
#ifndef TESTAPP_GEN
#ifndef SDT
#define INTC_DEVICE_ID		  XPAR_SCUGIC_SINGLE_DEVICE_ID
#else
#define XSCUGIC_DIST_BASEADDR XPAR_XSCUGIC_0_BASEADDR
#endif
#endif

/**************************** Type Definitions *******************************/

/***************** Macros (Inline Functions) Definitions *********************/

/************************** Function Prototypes ******************************/
#ifndef SDT
int ScuGicSelfTestExample(u16 DeviceId);
int ScuGicInterruptSetup(XScuGic *IntcInstancePtr, u16 DeviceId);
#else
int ScuGicSelfTestExample(u32 BaseAddr);
int ScuGicInterruptSetup(XScuGic *IntcInstancePtr, u32 BaseAddr);
#endif

/************************** Variable Definitions *****************************/

static XScuGic IntcInstance; /* Instance of the Interrupt Controller */

/*****************************************************************************/
/**
*
* This is the main function for the Interrupt Controller example. This
* function is not included if the example is generated from the TestAppGen
* test tool.
*
* @param	None.
*
* @return	XST_SUCCESS to indicate success, otherwise XST_FAILURE.
*
* @note		None.
*
******************************************************************************/
#ifndef TESTAPP_GEN
int main(void)
{
	int Status;

	/*
	 *  Run the Scu Gic Intc example, specify the Device ID generated in
	 *  xparameters.h.
	 */
	xil_printf("Starting Scugic self test Example \r\n");
#ifndef SDT
	Status = ScuGicSelfTestExample(INTC_DEVICE_ID);
#else
	Status = ScuGicSelfTestExample(XSCUGIC_DIST_BASEADDR);
#endif
	if (Status != XST_SUCCESS) {
		xil_printf("Scugic self test Example failed \n!");
		return XST_FAILURE;
	}
	xil_printf("Successfully ran Scugic self test Example\n!");
	return XST_SUCCESS;

}
#endif

/*****************************************************************************/
/**
*
* This function runs a self-test on the driver/device. This is a destructive
* test. This function is an example of how to use the interrupt controller
* driver component (XScuGic) and the hardware device.  This function is designed
* to work without any hardware devices to cause interrupts.  It may not return
* if the interrupt controller is not properly connected to the processor in
* either software or hardware.
*
* This function relies on the fact that the interrupt controller hardware
* has come out of the reset state such that it will allow interrupts to be
* simulated by the software.
*
* @param	DeviceId is device ID of the Interrupt Controller Device,
*		typically XPAR_<INTC_instance>_DEVICE_ID value from
*		xparameters.h.
*
* @return	XST_SUCCESS to indicate success, otherwise XST_FAILURE.
*
* @note		None.
*
******************************************************************************/
#ifndef SDT
int ScuGicSelfTestExample(u16 DeviceId)
#else
int ScuGicSelfTestExample(u32 BaseAddr)
#endif
{
	int Status;
	static XScuGic_Config *GicConfig;

	/*
	 * Initialize the interrupt controller driver so that it is ready to
	 * use.
	 */
#ifndef SDT
	GicConfig = XScuGic_LookupConfig(DeviceId);
#else
	GicConfig = XScuGic_LookupConfig(BaseAddr);
#endif
	if (NULL == GicConfig) {
		return XST_FAILURE;
	}

	Status = XScuGic_CfgInitialize(&IntcInstance, GicConfig,
				       GicConfig->CpuBaseAddress);
	if (Status != XST_SUCCESS) {
		return XST_FAILURE;
	}

	return XST_SUCCESS;

}

/*****************************************************************************/
/**
*
* This function is used by the TestAppGen generated application to setup
* the interrupt controller.
*
* @param	IntcInstancePtr is the reference to the Interrupt Controller
*		instance.
* @param	DeviceId is device ID of the Interrupt Controller Device,
*		typically XPAR_<INTC_instance>_DEVICE_ID value from
*		xparameters.h.
*
* @return	XST_SUCCESS to indicate success, otherwise XST_FAILURE.
*
* @note		None.
*
******************************************************************************/
#ifndef SDT
int ScuGicInterruptSetup(XScuGic *IntcInstancePtr, u16 DeviceId)
#else
int ScuGicInterruptSetup(XScuGic *IntcInstancePtr, u32 BaseAddr)
#endif
{

	int Status;
	static XScuGic_Config *GicConfig;

	/*
	 * Initialize the interrupt controller driver so that it is ready to
	 * use.
	 */
#ifndef SDT
	GicConfig = XScuGic_LookupConfig(DeviceId);
#else
	GicConfig = XScuGic_LookupConfig(BaseAddr);
#endif
	if (NULL == GicConfig) {
		return XST_FAILURE;
	}

	Status = XScuGic_CfgInitialize(IntcInstancePtr, GicConfig,
				       GicConfig->CpuBaseAddress);
	if (Status != XST_SUCCESS) {
		return XST_FAILURE;
	}

	/*
	 * Initialize the exception table.
	 */
	Xil_ExceptionInit();

	/*
	 * Connect the interrupt controller interrupt handler to the hardware
	 * interrupt handling logic in the processor.
	 */
	Xil_ExceptionRegisterHandler(XIL_EXCEPTION_ID_IRQ_INT,
				     (Xil_ExceptionHandler)XScuGic_InterruptHandler,
				     IntcInstancePtr);

	/*
	 * Enable exceptions.
	 */
	Xil_ExceptionEnable();

	return XST_SUCCESS;

}
