/******************************************************************************
* Copyright (C) 2014 - 2020 Xilinx, Inc.  All rights reserved.
* Copyright 2022-2023 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
******************************************************************************/

/*****************************************************************************/
/**
*
* @file xhdcp22_tx_test.c
* @addtogroup hdcp22_tx Overview
* @{
* @details
*
* This is a test and logging file for the Xilinx HDCP 2.2 transmitter core.
* Testing is done by acting as a stub for the DDC handlers. Some
* functions return a test-vector as specified by the
* "Errata to HDCP on HDMI Specification Revision 2.2" document.
* Further logging functionality is available, using a buffer for logging
* events. The log buffer also acts as a results buffer for unit tests.
* A unit test can execute authentication, and check if the results in the
* log-buffer match the expected results. Using test flags, some errors can be
* simulated.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who    Date     Changes
* ----- ------ -------- --------------------------------------------------
* 1.00  JO     06/17/15 Initial release.
* 2.00  MH     06/28/16 Updated for repeater downstream support.
* </pre>
*
******************************************************************************/
#ifdef _XHDCP22_TX_TEST_

/***************************** Include Files *********************************/
#include "xhdcp22_tx_i.h"
#include "string.h"
#include "stdlib.h"

/************************** Constant Definitions *****************************/
#define XHDCP22_TX_TEST_VECTOR_R1_INDEX 0
#define XHDCP22_TX_TEST_VECTOR_R2_INDEX 1


/*
 * All described testvectors are for testing purposes only and can be found in
 * "Errata to HDCP on HDMI specification appendix E.pdf"
 */

/** TX: Global constant */
static const u8 XHdcp22_Tx_Test_LC128[] =
{
	0x93, 0xCE, 0x5A, 0x56, 0xA0, 0xA1, 0xF4, 0xF7, 0x3C, 0x65, 0x8A, 0x1B, 0xD2, 0xAE,
	0xF0, 0xF7
};

/** TX: public key to use for signature verification (modulus and exponent)*/
static const u8 XHdcp22_Tx_Test_Kpubdcp[] =
{
	0xA2, 0xC7, 0x55, 0x57, 0x54, 0xCB, 0xAA, 0xA7, 0x7A, 0x27, 0x92, 0xC3, 0x1A, 0x6D, 0xC2, 0x31, 0xCF, 0x12,
	0xC2, 0x24, 0xBF, 0x89, 0x72, 0x46, 0xA4, 0x8D, 0x20, 0x83, 0xB2, 0xDD, 0x04, 0xDA, 0x7E, 0x01, 0xA9, 0x19,
	0xEF, 0x7E, 0x8C, 0x47, 0x54, 0xC8, 0x59, 0x72, 0x5C, 0x89, 0x60, 0x62, 0x9F, 0x39, 0xD0, 0xE4, 0x80, 0xCA,
	0xA8, 0xD4, 0x1E, 0x91, 0xE3, 0x0E, 0x2C, 0x77, 0x55, 0x6D, 0x58, 0xA8, 0x9E, 0x3E, 0xF2, 0xDA, 0x78, 0x3E,
	0xBA, 0xD1, 0x05, 0x37, 0x07, 0xF2, 0x88, 0x74, 0x0C, 0xBC, 0xFB, 0x68, 0xA4, 0x7A, 0x27, 0xAD, 0x63, 0xA5,
	0x1F, 0x67, 0xF1, 0x45, 0x85, 0x16, 0x49, 0x8A, 0xE6, 0x34, 0x1C, 0x6E, 0x80, 0xF5, 0xFF, 0x13, 0x72, 0x85,
	0x5D, 0xC1, 0xDE, 0x5F, 0x01, 0x86, 0x55, 0x86, 0x71, 0xE8, 0x10, 0x33, 0x14, 0x70, 0x2A, 0x5F, 0x15, 0x7B,
	0x5C, 0x65, 0x3C, 0x46, 0x3A, 0x17, 0x79, 0xED, 0x54, 0x6A, 0xA6, 0xC9, 0xDF, 0xEB, 0x2A, 0x81, 0x2A, 0x80,
	0x2A, 0x46, 0xA2, 0x06, 0xDB, 0xFD, 0xD5, 0xF3, 0xCF, 0x74, 0xBB, 0x66, 0x56, 0x48, 0xD7, 0x7C, 0x6A, 0x03,
	0x14, 0x1E, 0x55, 0x56, 0xE4, 0xB6, 0xFA, 0x38, 0x2B, 0x5D, 0xFB, 0x87, 0x9F, 0x9E, 0x78, 0x21, 0x87, 0xC0,
	0x0C, 0x63, 0x3E, 0x8D, 0x0F, 0xE2, 0xA7, 0x19, 0x10, 0x9B, 0x15, 0xE1, 0x11, 0x87, 0x49, 0x33, 0x49, 0xB8,
	0x66, 0x32, 0x28, 0x7C, 0x87, 0xF5, 0xD2, 0x2E, 0xC5, 0xF3, 0x66, 0x2F, 0x79, 0xEF, 0x40, 0x5A, 0xD4, 0x14,
	0x85, 0x74, 0x5F, 0x06, 0x43, 0x50, 0xCD, 0xDE, 0x84, 0xE7, 0x3C, 0x7D, 0x8E, 0x8A, 0x49, 0xCC, 0x5A, 0xCF,
	0x73, 0xA1, 0x8A, 0x13, 0xFF, 0x37, 0x13, 0x3D, 0xAD, 0x57, 0xD8, 0x51, 0x22, 0xD6, 0x32, 0x1F, 0xC0, 0x68,
	0x4C, 0xA0, 0x5B, 0xDD, 0x5F, 0x78, 0xC8, 0x9F, 0x2D, 0x3A, 0xA2, 0xB8, 0x1E, 0x4A, 0xE4, 0x08, 0x55, 0x64,
	0x05, 0xE6, 0x94, 0xFB, 0xEB, 0x03, 0x6A, 0x0A, 0xBE, 0x83, 0x18, 0x94, 0xD4, 0xB6, 0xC3, 0xF2, 0x58, 0x9C,
	0x7A, 0x24, 0xDD, 0xD1, 0x3A, 0xB7, 0x3A, 0xB0, 0xBB, 0xE5, 0xD1, 0x28, 0xAB, 0xAD, 0x24, 0x54, 0x72, 0x0E,
	0x76, 0xD2, 0x89, 0x32, 0xEA, 0x46, 0xD3, 0x78, 0xD0, 0xA9, 0x67, 0x78, 0xC1, 0x2D, 0x18, 0xB0, 0x33, 0xDE,
	0xDB, 0x27, 0xCC, 0xB0, 0x7C, 0xC9, 0xA4, 0xBD, 0xDF, 0x2B, 0x64, 0x10, 0x32, 0x44, 0x06, 0x81, 0x21, 0xB3,
	0xBA, 0xCF, 0x33, 0x85, 0x49, 0x1E, 0x86, 0x4C, 0xBD, 0xF2, 0x3D, 0x34, 0xEF, 0xD6, 0x23, 0x7A, 0x9F, 0x2C,
	0xDA, 0x84, 0xF0, 0x83, 0x83, 0x71, 0x7D, 0xDA, 0x6E, 0x44, 0x96, 0xCD, 0x1D, 0x05, 0xDE, 0x30, 0xF6, 0x1E,
	0x2F, 0x9C, 0x99, 0x9C, 0x60, 0x07, 0x03
};

/** Authentication and key exchange
 * Tx->Rx AKE_INIT: hardcoded rtx is normally generated by TX
 */
static const u8 XHdcp22_Tx_Test_Rtx[2][XHDCP22_TX_RTX_SIZE] =
{{ 0x18, 0xFA, 0xE4, 0x20, 0x6A, 0xFB, 0x51, 0x49 },  // R1
 { 0xF9, 0xF1, 0x30, 0xA8, 0x2D, 0x5B, 0xE5, 0xC3 }}; // R2

/** Tx->Rx AKE_INIT: tx caps */
static const u8 XHdcp22_Tx_Test_TxCaps[] = { 0x02, 0x00, 0x00 };

/** Rx->Tx AKE_Send_Cert: Rx certificate including receiver ID */
static const u8 XHdcp22_Tx_Test_CertRx[2][XHDCP22_TX_CERT_SIZE] =
{{ // R1
	0x74, 0x5b, 0xb8, 0xbd, 0x04, 0xaf, 0xb5, 0xc5, 0xc6, 0x7b, 0xc5, 0x3a, 0x34, 0x90, 0xa9,
	0x54, 0xc0, 0x8f, 0xb7, 0xeb, 0xa1, 0x54, 0xd2, 0x4f, 0x22, 0xde, 0x83, 0xf5, 0x03, 0xa6,
	0xc6, 0x68, 0x46, 0x9b, 0xc0, 0xb8, 0xc8, 0x6c, 0xdb, 0x26, 0xf9, 0x3c, 0x49, 0x2f, 0x02,
	0xe1, 0x71, 0xdf, 0x4e, 0xf3, 0x0e, 0xc8, 0xbf, 0x22, 0x9d, 0x04, 0xcf, 0xbf, 0xa9, 0x0d,
	0xff, 0x68, 0xab, 0x05, 0x6f, 0x1f, 0x12, 0x8a, 0x68, 0x62, 0xeb, 0xfe, 0xc9, 0xea, 0x9f,
	0xa7, 0xfb, 0x8c, 0xba, 0xb1, 0xbd, 0x65, 0xac, 0x35, 0x9c, 0xa0, 0x33, 0xb1, 0xdd, 0xa6,
	0x05, 0x36, 0xaf, 0x00, 0xa2, 0x7f, 0xbc, 0x07, 0xb2, 0xdd, 0xb5, 0xcc, 0x57, 0x5c, 0xdc,
	0xc0, 0x95, 0x50, 0xe5, 0xff, 0x1f, 0x20, 0xdb, 0x59, 0x46, 0xfa, 0x47, 0xc4, 0xed, 0x12,
	0x2e, 0x9e, 0x22, 0xbd, 0x95, 0xa9, 0x85, 0x59, 0xa1, 0x59, 0x3c, 0xc7, 0x83, 0x01, 0x00,
	0x01, 0x10, 0x00, 0x0b, 0xa3, 0x73, 0x77, 0xdd, 0x03, 0x18, 0x03, 0x8a, 0x91, 0x63, 0x29,
	0x1e, 0xa2, 0x95, 0x74, 0x42, 0x90, 0x78, 0xd0, 0x67, 0x25, 0xb6, 0x32, 0x2f, 0xcc, 0x23,
	0x2b, 0xad, 0x21, 0x39, 0x3d, 0x14, 0xba, 0x37, 0xa3, 0x65, 0x14, 0x6b, 0x9c, 0xcf, 0x61,
	0x20, 0x44, 0xa1, 0x07, 0xbb, 0xcf, 0xc3, 0x4e, 0x95, 0x5b, 0x10, 0xcf, 0xc7, 0x6f, 0xf1,
	0xc3, 0x53, 0x7c, 0x63, 0xa1, 0x8c, 0xb2, 0xe8, 0xab, 0x2e, 0x96, 0x97, 0xc3, 0x83, 0x99,
	0x70, 0xd3, 0xdc, 0x21, 0x41, 0xf6, 0x0a, 0xd1, 0x1a, 0xee, 0xf4, 0xcc, 0xeb, 0xfb, 0xa6,
	0xaa, 0xb6, 0x9a, 0xaf, 0x1d, 0x16, 0x5e, 0xe2, 0x83, 0xa0, 0x4a, 0x41, 0xf6, 0x7b, 0x07,
	0xbf, 0x47, 0x85, 0x28, 0x6c, 0xa0, 0x77, 0xa6, 0xa3, 0xd7, 0x85, 0xa5, 0xc4, 0xa7, 0xe7,
	0x6e, 0xb5, 0x1f, 0x40, 0x72, 0x97, 0xfe, 0xc4, 0x81, 0x23, 0xa0, 0xc2, 0x90, 0xb3, 0x49,
	0x24, 0xf5, 0xb7, 0x90, 0x2c, 0xbf, 0xfe, 0x04, 0x2e, 0x00, 0xa9, 0x5f, 0x86, 0x04, 0xca,
	0xc5, 0x3a, 0xcc, 0x26, 0xd9, 0x39, 0x7e, 0xa9, 0x2d, 0x28, 0x6d, 0xc0, 0xcc, 0x6e, 0x81,
	0x9f, 0xb9, 0xb7, 0x11, 0x33, 0x32, 0x23, 0x47, 0x98, 0x43, 0x0d, 0xa5, 0x1c, 0x59, 0xf3,
	0xcd, 0xd2, 0x4a, 0xb7, 0x3e, 0x69, 0xd9, 0x21, 0x53, 0x9a, 0xf2, 0x6e, 0x77, 0x62, 0xae,
	0x50, 0xda, 0x85, 0xc6, 0xaa, 0xc4, 0xb5, 0x1c, 0xcd, 0xa8, 0xa5, 0xdd, 0x6e, 0x62, 0x73,
	0xff, 0x5f, 0x7b, 0xd7, 0x3c, 0x17, 0xba, 0x47, 0x0c, 0x89, 0x0e, 0x62, 0x79, 0x43, 0x94,
	0xaa, 0xa8, 0x47, 0xf4, 0x4c, 0x38, 0x89, 0xa8, 0x81, 0xad, 0x23, 0x13, 0x27, 0x0c, 0x17,
	0xcf, 0x3d, 0x83, 0x84, 0x57, 0x36, 0xe7, 0x22, 0x26, 0x2e, 0x76, 0xfd, 0x56, 0x80, 0x83,
	0xf6, 0x70, 0xd4, 0x5c, 0x91, 0x48, 0x84, 0x7b, 0x18, 0xdb, 0x0e, 0x15, 0x3b, 0x49, 0x26,
	0x23, 0xe6, 0xa3, 0xe2, 0xc6, 0x3a, 0x23, 0x57, 0x66, 0xb0, 0x72, 0xb8, 0x12, 0x17, 0x4f,
	0x86, 0xfe, 0x48, 0x0d, 0x53, 0xea, 0xfe, 0x31, 0x48, 0x7d, 0x86, 0xde, 0xeb, 0x82, 0x86,
	0x1e, 0x62, 0x03, 0x98, 0x59, 0x00, 0x37, 0xeb, 0x61, 0xe9, 0xf9, 0x7a, 0x40, 0x78, 0x1c,
	0xba, 0xbc, 0x0b, 0x88, 0xfb, 0xfd, 0x9d, 0xd5, 0x01, 0x11, 0x94, 0xe0, 0x35, 0xbe, 0x33,
	0xe8, 0xe5, 0x36, 0xfb, 0x9c, 0x45, 0xcb, 0x75, 0xaf, 0xd6, 0x35, 0xff, 0x78, 0x92, 0x7f,
	0xa1, 0x7c, 0xa8, 0xfc, 0xb7, 0xf7, 0xa8, 0x52, 0xa9, 0xc6, 0x84, 0x72, 0x3d, 0x1c, 0xc9,
	0xdf, 0x35, 0xc6, 0xe6, 0x00, 0xe1, 0x48, 0x72, 0xce, 0x83, 0x1b, 0xcc, 0xf8, 0x33, 0x2d,
	0x4f, 0x98, 0x75, 0x00, 0x3c, 0x41, 0xdf, 0x7a, 0xed, 0x38, 0x53, 0xb1
 },
 { // R2
	0x8B, 0xA4, 0x47, 0x42, 0xFB, 0xE4, 0x68, 0x63, 0x8A, 0xDA, 0x97, 0x2D, 0xDE, 0x9A, 0x8D,
	0x1C, 0xB1, 0x65, 0x4B, 0x85, 0x8D, 0xE5, 0x46, 0xD6, 0xDB, 0x95, 0xA5, 0xF6, 0x66, 0x74,
	0xEA, 0x81, 0x0B, 0x9A, 0x58, 0x58, 0x66, 0x26, 0x86, 0xA6, 0xB4, 0x56, 0x2B, 0x29, 0x43,
	0xE5, 0xBB, 0x81, 0x74, 0x86, 0xA7, 0xB7, 0x16, 0x2F, 0x07, 0xEC, 0xD1, 0xB5, 0xF9, 0xAE,
	0x4F, 0x98, 0x89, 0xA9, 0x91, 0x7D, 0x58, 0x5B, 0x8D, 0x20, 0xD5, 0xC5, 0x08, 0x40, 0x3B,
	0x86, 0xAF, 0xF4, 0xD6, 0xB9, 0x20, 0x95, 0xE8, 0x90, 0x3B, 0x8F, 0x9F, 0x36, 0x5B, 0x46,
	0xB6, 0xD4, 0x1E, 0xF5, 0x05, 0x88, 0x80, 0x14, 0xE7, 0x2C, 0x77, 0x5D, 0x6E, 0x54, 0xE9,
	0x65, 0x81, 0x5A, 0x68, 0x92, 0xA5, 0xD6, 0x40, 0x78, 0x11, 0x97, 0x65, 0xD7, 0x64, 0x36,
	0x5E, 0x8D, 0x2A, 0x87, 0xA8, 0xEB, 0x7D, 0x06, 0x2C, 0x10, 0xF8, 0x0A, 0x7D, 0x01, 0x00,
	0x01, 0x10, 0x00, 0x06, 0x40, 0x99, 0x8F, 0x5A, 0x54, 0x71, 0x23, 0xA7, 0x6A, 0x64, 0x3F,
	0xBD, 0xDD, 0x52, 0xB2, 0x79, 0x6F, 0x88, 0x26, 0x94, 0x9E, 0xAF, 0xA4, 0xDE, 0x7D, 0x8D,
	0x88, 0x10, 0xC8, 0xF6, 0x56, 0xF0, 0x8F, 0x46, 0x28, 0x48, 0x55, 0x51, 0xC5, 0xAF, 0xA1,
	0xA9, 0x9D, 0xAC, 0x9F, 0xB1, 0x26, 0x4B, 0xEB, 0x39, 0xAD, 0x88, 0x46, 0xAF, 0xBC, 0x61,
	0xA8, 0x7B, 0xF9, 0x7B, 0x3E, 0xE4, 0x95, 0xD9, 0xA8, 0x79, 0x48, 0x51, 0x00, 0xBE, 0xA4,
	0xB6, 0x96, 0x7F, 0x3D, 0xFD, 0x76, 0xA6, 0xB7, 0xBB, 0xB9, 0x77, 0xDC, 0x54, 0xFB, 0x52,
	0x9C, 0x79, 0x8F, 0xED, 0xD4, 0xB1, 0xBC, 0x0F, 0x7E, 0xB1, 0x7E, 0x70, 0x6D, 0xFC, 0xB9,
	0x7E, 0x66, 0x9A, 0x86, 0x23, 0x3A, 0x98, 0x5E, 0x32, 0x8D, 0x75, 0x18, 0x54, 0x64, 0x36,
	0xDD, 0x92, 0x01, 0x39, 0x90, 0xB9, 0xE3, 0xAF, 0x6F, 0x98, 0xA5, 0xC0, 0x80, 0xC6, 0x2F,
	0xA1, 0x02, 0xAD, 0x8D, 0xF4, 0xD6, 0x66, 0x7B, 0x45, 0xE5, 0x74, 0x18, 0xB1, 0x27, 0x24,
	0x01, 0x1E, 0xEA, 0xD8, 0xF3, 0x79, 0x92, 0xE9, 0x03, 0xF5, 0x57, 0x8D, 0x65, 0x2A, 0x8D,
	0x1B, 0xF0, 0xDA, 0x58, 0x3F, 0x58, 0xA0, 0xF4, 0xB4, 0xBE, 0xCB, 0x21, 0x66, 0xE9, 0x21,
	0x7C, 0x76, 0xF3, 0xC1, 0x7E, 0x2E, 0x7C, 0x3D, 0x61, 0x20, 0x1D, 0xC5, 0xC0, 0x71, 0x28,
	0x2E, 0xB7, 0x0F, 0x1F, 0x7A, 0xC1, 0xD3, 0x6A, 0x1E, 0xA3, 0x54, 0x34, 0x8E, 0x0D, 0xD7,
	0x96, 0x93, 0x78, 0x50, 0xC1, 0xEE, 0x27, 0x72, 0x3A, 0xBD, 0x57, 0x22, 0xF0, 0xD7, 0x6D,
	0x9D, 0x65, 0xC4, 0x07, 0x9C, 0x82, 0xA6, 0xD4, 0xF7, 0x6B, 0x9A, 0xE9, 0xC0, 0x6C, 0x4A,
	0x4F, 0x6F, 0xBE, 0x8E, 0x01, 0x37, 0x50, 0x3A, 0x66, 0xD9, 0xE9, 0xD9, 0xF9, 0x06, 0x9E,
	0x00, 0xA9, 0x84, 0xA0, 0x18, 0xB3, 0x44, 0x21, 0x24, 0xA3, 0x6C, 0xCD, 0xB7, 0x0F, 0x31,
	0x2A, 0xE8, 0x15, 0xB6, 0x93, 0x6F, 0xB9, 0x86, 0xE5, 0x28, 0x01, 0x1A, 0x5E, 0x10, 0x3F,
	0x1F, 0x4D, 0x35, 0xA2, 0x8D, 0xB8, 0x54, 0x26, 0x68, 0x3A, 0xCD, 0xCB, 0x5F, 0xFA, 0x37,
	0x4A, 0x60, 0x10, 0xB1, 0x0A, 0xFE, 0xBA, 0x9B, 0x96, 0x5D, 0x7E, 0x99, 0xCF, 0x01, 0x98,
	0x65, 0x87, 0xAD, 0x40, 0xD5, 0x82, 0x1D, 0x61, 0x54, 0xA2, 0xD3, 0x16, 0x3E, 0xF7, 0xE3,
	0x05, 0x89, 0x8D, 0x8A, 0x50, 0x87, 0x47, 0xBE, 0x29, 0x18, 0x01, 0xB7, 0xC3, 0xDD, 0x43,
	0x23, 0x7A, 0xCD, 0x85, 0x1D, 0x4E, 0xA9, 0xC0, 0x1A, 0xA4, 0x77, 0xAB, 0xE7, 0x31, 0x9A,
	0x33, 0x1B, 0x7A, 0x86, 0xE1, 0xE5, 0xCA, 0x0C, 0x43, 0x1A, 0xFA, 0xEC, 0x4C, 0x05, 0xC6,
	0xD1, 0x43, 0x12, 0xF9, 0x4D, 0x3E, 0xF7, 0xD6, 0x05, 0x9C, 0x1C, 0xDD
}};

/** Rx->Tx AKE_Send_Cert: RxCaps */
static const u8 XHdcp22_Tx_Test_RxCaps[2][XHDCP22_TX_RXCAPS_SIZE] =
{{ 0x02, 0x00, 0x01 },  // R1
 { 0x02, 0x00, 0x00 }}; // R2

/** Rx->Tx AKE_Send_Cert: Rrx */
static const u8 XHdcp22_Tx_Test_Rrx[2][XHDCP22_TX_RRX_SIZE] =
{{ 0x3B, 0xA0, 0xBE, 0xDE, 0x0C, 0x46, 0xA9, 0x91 },  // R1
 { 0xE1, 0x7A, 0xB0, 0xFD, 0x0F, 0x54, 0x40, 0x52 }}; // R2

static const u8 XHdcp22_Tx_Test_Rrx_R1[] = { 0x3B, 0xA0, 0xBE, 0xDE, 0x0C, 0x46, 0xA9, 0x91 };

/** Tx->Rx AKE_No_Stored_km: hardcoded Km is normally generated by TX */
static const u8 XHdcp22_Tx_Test_Km[2][XHDCP22_TX_KM_SIZE] =
{{ // R1
	0x68, 0xbc, 0xc5, 0x1b, 0xa9, 0xdb, 0x1b, 0xd0, 0xfa, 0xf1, 0x5e, 0x9a, 0xd8, 0xa5, 0xaf, 0xb9
 },
 { // R2
	0xCA, 0x9F, 0x83, 0x95, 0x70, 0xD0, 0xD0, 0xF9, 0xCF, 0xE4, 0xEB, 0x54, 0x7E, 0x09, 0xFA, 0x3B
}};

/** Tx->Rx AKE_NoStored_km: encrypted Km send by Tx */
static const u8 XHdcp22_Tx_Test_EkpubKm[2][XHDCP22_TX_E_KPUB_KM_SIZE] =
{{ // R1
	0x9b, 0x9f, 0x80, 0x19, 0xad, 0x0e, 0xa2, 0xf0, 0xdd, 0xa0, 0x29, 0x33,
	0xd9, 0x6d, 0x1c, 0x77, 0x31, 0x37, 0x57, 0xe0, 0xe5, 0xb2, 0xbd, 0xdd,
	0x36, 0x3e, 0x38, 0x4e, 0x7d, 0x40, 0x78, 0x66, 0x97, 0x7a, 0x4c, 0xce,
	0xc5, 0xc7, 0x5d, 0x01, 0x57, 0x26, 0xcc, 0xa2, 0xf6, 0xde, 0x34, 0xdd,
	0x29, 0xbe, 0x5e, 0x31, 0xe8, 0xf1, 0x34, 0xe8, 0x1a, 0x63, 0xa3, 0x6d,
	0x46, 0xdc, 0x0a, 0x06, 0x08, 0x99, 0x9d, 0xdb, 0x3c, 0xa2, 0x9c, 0x04,
	0xdd, 0x4e, 0xd9, 0x02, 0x7d, 0x20, 0x54, 0xec, 0xca, 0x86, 0x42, 0x1b,
	0x18, 0xda, 0x30, 0x9c, 0xc4, 0xcb, 0xac, 0xb4, 0x54, 0xde, 0x84, 0x68,
	0x71, 0x53, 0x6d, 0x92, 0x17, 0xca, 0x08, 0x8a, 0x7a, 0xf9, 0x98, 0x9a,
	0xb6, 0x7b, 0x22, 0x92, 0xac, 0x7d, 0x0d, 0x6b, 0xd6, 0x7f, 0x31, 0xab,
	0xf0, 0x10, 0xc5, 0x2a, 0x0f, 0x6d, 0x27, 0xa0
 },
 { // R2
	0xA8, 0x55, 0xC2, 0xC4, 0xC6, 0xBE, 0xEF, 0xCD, 0xCB, 0x9F, 0xE3, 0x9F,
	0x2A, 0xB7, 0x29, 0x76, 0xFE, 0xD8, 0xDA, 0xC9, 0x38, 0xFA, 0x39, 0xF0,
	0xAB, 0xCA, 0x8A, 0xED, 0x95, 0x7B, 0x93, 0xB2, 0xDF, 0xD0, 0x7D, 0x09,
	0x9D, 0x05, 0x96, 0x66, 0x03, 0x6E, 0xBA, 0xE0, 0x63, 0x0F, 0x30, 0x77,
	0xC2, 0xBB, 0xE2, 0x11, 0x39, 0xE5, 0x27, 0x78, 0xEE, 0x64, 0xF2, 0x85,
	0x36, 0x57, 0xC3, 0x39, 0xD2, 0x7B, 0x79, 0x03, 0xB7, 0xCC, 0x82, 0xCB,
	0xF0, 0x62, 0x82, 0x43, 0x38, 0x09, 0x9B, 0x71, 0xAA, 0x38, 0xA6, 0x3F,
	0x48, 0x12, 0x6D, 0x8C, 0x5E, 0x07, 0x90, 0x76, 0xAC, 0x90, 0x99, 0x51,
	0x5B, 0x06, 0xA5, 0xFA, 0x50, 0xE4, 0xF9, 0x25, 0xC3, 0x07, 0x12, 0x37,
	0x64, 0x92, 0xD7, 0xDB, 0xD3, 0x34, 0x1C, 0xE4, 0xFA, 0xDD, 0x09, 0xE6,
	0x28, 0x3D, 0x0C, 0xAD, 0xA9, 0xD8, 0xE1, 0xB5
}};

/** Tx-Rx AKE_No_Stored_km: hardcoded masking seed */
static const u8 XHdcp22_Tx_Test_Km_MaskingSeed[] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
	0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B,
	0x1C, 0x1D, 0x1E, 0x1F };

/** Rx->Tx AKE_Send_H_Prime: H' */
static const u8 XHdcp22_Tx_Test_H1[2][XHDCP22_TX_H_PRIME_SIZE] =
{{ // R1
	0x69, 0xe0, 0xab, 0x21, 0x2f, 0xdb, 0x57, 0xe6, 0x7e, 0xfc, 0x43, 0x76,
	0x1a, 0x2c, 0x5c, 0xce, 0x76, 0xc3, 0x65, 0xf1, 0x9b, 0x75, 0xc3, 0xea,
	0xc2, 0xd2, 0x77, 0xdd, 0x5c, 0x7e, 0x4a, 0xc4
 },
 {  // R2
	0x4F, 0xF1, 0xA2, 0xA5, 0x61, 0x67, 0xC8, 0xE0, 0xAD, 0x16, 0xC8, 0x95,
	0x99, 0x1B, 0x1A, 0x21, 0xA8, 0x80, 0xC6, 0x27, 0x39, 0x3F, 0xC7, 0xBB,
	0x83, 0xED, 0xA7, 0xE5, 0x69, 0x07, 0xA5, 0xDC
}};

/** Rx->Tx AKE_Send_Pairing_Info: Ekh(Km)  */
static const u8 XHdcp22_Tx_Test_Ekh_Km[2][XHDCP22_TX_EKH_KM_SIZE] =
{{ // R1
	0xb8, 0x9f, 0xf9, 0x72, 0x6a, 0x6f, 0x2c, 0x1e, 0x29, 0xb6, 0x44, 0x8d, 0xdc, 0xa3,
	0x10, 0xbd
 },
 { // R2
	0xE6, 0x57, 0x8E, 0xBC, 0xC7, 0x68, 0x44, 0x87, 0x88, 0x8A, 0x9B, 0xD7, 0xD6, 0xAE,
	0x38, 0xBE
}};

/** Locality check
 * Tx->Rx LC_Init: Rn hardcoded Rn is normally generated by TX*/
static const u8 XHdcp22_Tx_Test_Rn[2][XHDCP22_TX_RN_SIZE] =
{{ // R1
	0x32, 0x75, 0x3e, 0xa8, 0x78, 0xa6, 0x38, 0x1c
 },
 { // R2
	0xA0, 0xFE, 0x9B, 0xB8, 0x20, 0x60, 0x58, 0xCA
}};

/* Session key exchange */

/** Tx->Rx SKE_Send_Eks */
static const u8 XHdcp22_Tx_Test_Riv[2][XHDCP22_TX_RIV_SIZE] =
{{ // R1
	0x40, 0x2b, 0x6b, 0x43, 0xc5, 0xe8, 0x86, 0xd8
 },
 { // R2
	0x9A, 0x6D, 0x11, 0x00, 0xA9, 0xB7, 0x6F, 0x64
}};

/** Tx->Rx SKE_Send_Eks */
static const u8 XHdcp22_Tx_Test_Ks[] = {
	0xF3, 0xDF, 0x1D, 0xD9, 0x57, 0x96, 0x12, 0x3F,
	0x98, 0x97, 0x89, 0xB4, 0x21, 0xE1, 0x2D, 0xE1
};

/** Tx->Rx SKE_Send_Eks */
static const u8 XHdcp22_Tx_Test_EdkeyKs[2][XHDCP22_TX_EDKEY_KS_SIZE] =
{{ // R1
	0x4c, 0x32, 0x47, 0x12, 0xc4, 0xbe, 0xc6, 0x69,
	0x0a, 0xc2, 0x19, 0x64, 0xde, 0x91, 0xf1, 0x83
 },
 { // R2
	0xB6, 0x8B, 0x8A, 0xA4, 0xD2, 0xCB, 0xBA, 0xFF,
	0x53, 0x33, 0xC1, 0xD9, 0xBB, 0xB7, 0x10, 0xA9
}};

/** Rx->Tx LC_Send_L_Prime: L' */
static const u8 XHdcp22_Tx_Test_L1[2][XHDCP22_TX_L_PRIME_SIZE] =
{{ // R1
	0xbc, 0x20, 0x92, 0x33, 0x54, 0x91, 0xc1, 0x9e, 0xa4, 0xde, 0x8b, 0x30,
	0x49, 0xc2, 0x06, 0x6a, 0xd8, 0x11, 0xa2, 0x2a, 0xb1, 0x46, 0xdf, 0x74,
	0x58, 0x47, 0x05, 0xa8, 0xb7, 0x67, 0xfb, 0xdd
 },
 { // R2
	0xF2, 0x0F, 0x13, 0x6E, 0x85, 0x53, 0xC1, 0x0C, 0xD3, 0xDD, 0xB2, 0xF9,
	0x6D, 0x33, 0x31, 0xF9, 0xCB, 0x6E, 0x97, 0x8C, 0xCD, 0x5E, 0xDA, 0x13,
	0xDD, 0xEA, 0x41, 0x44, 0x10, 0x9B, 0x51, 0xB0
}};

/** Tx->Rx RepeaterAuth_Send_Ack: V */
static const u8 XHdcp22_Tx_Test_V[XHDCP22_TX_V_PRIME_SIZE] =
{ // R1
	0x63, 0x6d, 0xc5, 0x08, 0x4d, 0x6c, 0xb1, 0x0e, 0x93, 0xa5, 0x28, 0x67,
	0x0f, 0x34, 0x1f, 0x88
};

/** Rx->Tx RepeaterAuth_Send_RecieverID_List: V' */
static const u8 XHdcp22_Tx_Test_V1[XHDCP22_TX_V_PRIME_SIZE] =
{ // R1
	0xbc, 0xcc, 0x7d, 0x16, 0xe6, 0xbc, 0xb9, 0x02, 0x60, 0x08, 0x1d, 0xf7,
	0x4a, 0xb4, 0x5c, 0x8a
};

/** Rx->Tx RepeaterAuth_Send_RecieverID_List: RxInfo */
static const u8 XHdcp22_Tx_Test_RxInfo[XHDCP22_TX_RXINFO_SIZE] =
{
	0x02, 0x31
};

/** Rx->Tx RepeaterAuth_Send_RecieverID_List: seq_num_V */
static const u8 XHdcp22_Tx_Test_SeqNum_V[XHDCP22_TX_SEQ_NUM_V_SIZE] =
{
	0x00, 0x00, 0x00
};

/** Rx->Tx RepeaterAuth_Send_RecieverID_List: ReceiverID list */
static const u8 XHdcp22_Tx_Test_RcvIDList[3*XHDCP22_TX_RCVID_SIZE] =
{
	0x47, 0x8e, 0x71, 0xe2, 0x0f,
	0x35, 0x79, 0x6a, 0x17, 0x0e,
	0x74, 0xe8, 0x53, 0x97, 0xa2
};

/** Tx->Rx RepeaterAuth_Stream_Manage: seq_num_M */
static const u8 XHdcp22_Tx_Test_SeqNum_M[XHDCP22_TX_SEQ_NUM_M_SIZE] =
{
	0x00, 0x00, 0x00
};

/** Rx->Tx RepeaterAuth_Stream_Ready: M' */
static const u8 XHdcp22_Tx_Test_M1[XHDCP22_TX_M_PRIME_SIZE] =
{
	0xdd, 0x26, 0xe9, 0x52, 0x6e, 0x0e, 0x1d, 0x69, 0xc8, 0x84, 0xe4, 0xcc, 0xc8, 0x09, 0xaa, 0xc7,
	0x71, 0xe9, 0x97, 0xb5, 0x61, 0x89, 0x09, 0x6e, 0x4d, 0x94, 0x24, 0xc2, 0x1b, 0x64, 0x58, 0xc6
};

/** HDCP2 SRM from errata
 * Receiver IDs revoked: 74 5B B8 BD 04, 8B A4 47 42 FB */
static const u8 XHdcp22_Tx_Test_SrmErrata[] = {
	0x91, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x91, 0x00, 0x80, 0x00, 0x00, 0x74, 0x5B, 0xB8, 0xBD,
	0x04, 0x8B, 0xA4, 0x47, 0x42, 0xFB, 0x17, 0x07, 0xE9, 0xEA, 0x61, 0xAD, 0xB4, 0x2E, 0x9A, 0x44,
	0xA9, 0x1E, 0x44, 0xBA, 0xAB, 0x6F, 0x6B, 0x37, 0x27, 0x50, 0xBB, 0x17, 0x8E, 0xAD, 0xC5, 0x7F,
	0xE8, 0xF5, 0x21, 0xB4, 0x60, 0x1E, 0x54, 0x80, 0xDA, 0x2A, 0x1A, 0x59, 0xF3, 0x9D, 0xE3, 0x98,
	0x54, 0x43, 0x24, 0x70, 0xCA, 0x83, 0x47, 0x64, 0x2D, 0xC6, 0x26, 0x6D, 0x30, 0x05, 0xB4, 0xEE,
	0x9B, 0xB6, 0x69, 0xA2, 0xF3, 0x7C, 0x7D, 0x13, 0xCF, 0xF3, 0xA7, 0xC7, 0x89, 0xEF, 0x50, 0x0D,
	0x32, 0xE1, 0xD2, 0x2C, 0xD1, 0xB5, 0x46, 0xD6, 0x36, 0x44, 0x25, 0x52, 0x65, 0x06, 0xB6, 0x31,
	0xE7, 0x26, 0xD2, 0x5C, 0x1C, 0x17, 0xB7, 0x26, 0x6F, 0x73, 0xE5, 0x8A, 0x6E, 0xE9, 0xDB, 0x00,
	0x27, 0x70, 0x11, 0xBB, 0x75, 0xA0, 0x49, 0x30, 0xFF, 0x38, 0xB3, 0xD8, 0x2D, 0x03, 0xC1, 0x78,
	0x50, 0x74, 0xCA, 0x60, 0xAA, 0x32, 0x03, 0x28, 0x8D, 0xE4, 0xA1, 0x8C, 0x62, 0x2D, 0xF3, 0x61,
	0x6C, 0xBF, 0xA1, 0x9C, 0xAB, 0xB3, 0xD7, 0xE7, 0xD6, 0xDC, 0x31, 0xF6, 0x74, 0x1A, 0xAB, 0xC0,
	0x9C, 0x6A, 0x0A, 0xC2, 0x65, 0xE4, 0x29, 0xBD, 0x4E, 0x22, 0x73, 0x5A, 0x2C, 0xC7, 0x75, 0x96,
	0xE9, 0x7A, 0x16, 0xA6, 0x70, 0x6B, 0x3C, 0xBA, 0x50, 0x60, 0x5A, 0x33, 0xD5, 0xD7, 0xF2, 0x76,
	0x5B, 0x5E, 0x2D, 0x45, 0xE2, 0x0E, 0xF1, 0x4C, 0x6D, 0x8D, 0xCF, 0xAF, 0x39, 0x23, 0x79, 0xA5,
	0xFB, 0xAE, 0xCB, 0x8F, 0xD8, 0xDE, 0xDF, 0xDB, 0x24, 0x20, 0x52, 0x10, 0x74, 0xBA, 0x42, 0x6F,
	0xAD, 0xBB, 0x3E, 0x3A, 0xA5, 0xCE, 0x99, 0x0D, 0xFF, 0x41, 0xA6, 0x0F, 0x60, 0x7F, 0xAE, 0x05,
	0x00, 0x3C, 0x2C, 0xF3, 0xBA, 0x5A, 0x86, 0x1B, 0x04, 0x7F, 0x53, 0xE5, 0xE3, 0x68, 0xDC, 0x6B,
	0x36, 0x25, 0x73, 0x69, 0x95, 0x5D, 0x15, 0x37, 0xD6, 0x98, 0xD6, 0x6B, 0xA8, 0xD0, 0x35, 0x37,
	0x2D, 0x2F, 0xF1, 0x53, 0x90, 0xAA, 0x32, 0x87, 0x99, 0x3B, 0xB7, 0x33, 0x48, 0x0E, 0xCE, 0xE4,
	0xF1, 0xD8, 0x93, 0x51, 0xEB, 0x98, 0x92, 0xE7, 0x2D, 0xAC, 0xA2, 0x32, 0xFB, 0xE1, 0xF9, 0xF1,
	0x7D, 0x92, 0x26, 0x37, 0x5D, 0x4C, 0x0C, 0xF8, 0xA2, 0x11, 0x4C, 0x2A, 0x49, 0xB6, 0x48, 0xBC,
	0xCC, 0x44, 0x01, 0x06, 0xCB, 0x74, 0xDA, 0x0D, 0x70, 0xFA, 0x06, 0x64, 0xA6, 0x54, 0xE6, 0x7C,
	0x65, 0x34, 0xFF, 0x97, 0xED, 0xDB, 0x8A, 0x8D, 0x40, 0xBE, 0xFE, 0x60, 0xE5, 0x0C, 0xBF, 0x50,
	0x60, 0x7B, 0x16, 0x71, 0xBD, 0xFF, 0xC3, 0x1B, 0xFB, 0x15, 0xA5, 0x10, 0x07, 0x6A, 0x5D, 0x1A,
	0x6C, 0xFA, 0x67, 0xDC, 0xB3, 0xCD, 0xA3, 0x85, 0x6E, 0xEB, 0x77, 0x5F, 0x92, 0x8F, 0xEE, 0xFB,
	0xD5, 0x34, 0x58, 0x72, 0xB1, 0x55
};

/** TEST HDCP2 SRM from DCP LLC website
 * Receiver IDs revoked: 04 BD B8 5B 74, FB 42 47 A4 8B */
static const u8 XHdcp22_Tx_Test_SrmDCPSiteTest[] = {
	0x91, 0x00, 0x00, 0x02, 0x01, 0x00, 0x01, 0x91, 0x00, 0x80, 0x00, 0x00, 0x04, 0xBD, 0xB8, 0x5B,
	0x74, 0xFB, 0x42, 0x47, 0xA4, 0x8B, 0x2A, 0x53, 0x45, 0xA0, 0x9E, 0xF7, 0x76, 0xD9, 0xC8, 0xF4,
	0xC8, 0x92, 0x58, 0x08, 0xDD, 0x71, 0x89, 0xAC, 0x22, 0x79, 0x75, 0x43, 0xC9, 0xD0, 0x4F, 0x9C,
	0xE2, 0xB8, 0x29, 0x5F, 0xBE, 0x22, 0x4B, 0x3B, 0xD2, 0x4D, 0x09, 0x55, 0x1D, 0x5C, 0xBF, 0x21,
	0x7D, 0x0E, 0x62, 0x76, 0xC3, 0xBD, 0xC7, 0xFC, 0xC9, 0x9E, 0x54, 0x3F, 0x94, 0x59, 0x51, 0x4B,
	0x8F, 0x85, 0x99, 0x75, 0xC8, 0xBC, 0x1D, 0x84, 0xCC, 0xBA, 0xE4, 0x36, 0x03, 0x69, 0xD8, 0x01,
	0x76, 0x97, 0xDB, 0xF9, 0x2D, 0x3F, 0xB8, 0x66, 0xE5, 0x5E, 0x8B, 0x85, 0x0B, 0xAC, 0xBE, 0x0D,
	0xDD, 0x06, 0xD4, 0x9D, 0x28, 0x9E, 0x51, 0x62, 0xCD, 0xCE, 0x36, 0x23, 0x84, 0x65, 0x0C, 0x74,
	0xA3, 0x7B, 0xA0, 0x59, 0xB1, 0x84, 0xB7, 0xC1, 0x27, 0x8F, 0x62, 0xAD, 0x1B, 0x16, 0x26, 0x70,
	0x9F, 0x08, 0x91, 0x1B, 0x1B, 0x64, 0x90, 0xE8, 0x1E, 0x28, 0x84, 0xF7, 0x53, 0x15, 0x2C, 0x88,
	0x87, 0x53, 0x19, 0x53, 0x29, 0x75, 0x9F, 0x4B, 0x95, 0x11, 0x6A, 0x1C, 0x43, 0x6F, 0x11, 0x6A,
	0x19, 0xBA, 0xEF, 0x3D, 0xBD, 0x1F, 0x8B, 0x00, 0x39, 0x9B, 0x7D, 0xAB, 0x97, 0x0B, 0x36, 0x26,
	0x04, 0x82, 0x7B, 0x60, 0x0E, 0xC5, 0x4F, 0x7E, 0x2B, 0x21, 0x96, 0xDC, 0xED, 0x8E, 0x15, 0x8F,
	0x34, 0x86, 0x58, 0x3C, 0x91, 0x75, 0xF4, 0xD1, 0x0A, 0x47, 0xB1, 0x54, 0x18, 0xF1, 0x46, 0x17,
	0x28, 0x80, 0x73, 0xD5, 0x98, 0x97, 0x75, 0xAD, 0x61, 0x80, 0xC8, 0x0F, 0x80, 0xAA, 0x98, 0xCE,
	0x5F, 0x53, 0x03, 0xFA, 0xCC, 0x46, 0x7B, 0x2C, 0x8C, 0x99, 0x5C, 0xA1, 0x2B, 0x36, 0x36, 0x39,
	0xB0, 0x21, 0x26, 0x1B, 0xE7, 0x76, 0x29, 0x02, 0x52, 0x1A, 0x9C, 0xE3, 0x97, 0xE9, 0xB7, 0x24,
	0x5B, 0xDE, 0x8E, 0x86, 0x5D, 0x09, 0x1C, 0x87, 0xBA, 0x6C, 0x82, 0xF8, 0x44, 0xC2, 0xC0, 0x8A,
	0xBF, 0x2D, 0x7D, 0x79, 0x78, 0x11, 0xB4, 0x0A, 0x68, 0xF7, 0xCB, 0xDE, 0xF5, 0xEC, 0xD5, 0xB0,
	0x9C, 0x57, 0x0B, 0x20, 0x1C, 0x2C, 0xD3, 0x36, 0xB3, 0x44, 0xE2, 0xFF, 0xD2, 0x27, 0x80, 0xC3,
	0xB5, 0xCF, 0x0B, 0xC2, 0xD6, 0x9B, 0x49, 0x13, 0x22, 0x56, 0x91, 0x40, 0x73, 0xEE, 0x67, 0x6C,
	0x49, 0x67, 0xE7, 0x5F, 0xDA, 0xFD, 0x76, 0x57, 0x75, 0x95, 0xC3, 0x09, 0x3B, 0xBB, 0xAB, 0xEB,
	0x1D, 0x01, 0xE3, 0x3F, 0xE3, 0x87, 0x89, 0x2A, 0x8B, 0xF6, 0x2B, 0x98, 0x2E, 0x19, 0x70, 0xFE,
	0xAB, 0xBA, 0x24, 0x1F, 0x44, 0x5A, 0x42, 0xDB, 0xB9, 0x9C, 0xE0, 0x3D, 0xB4, 0x68, 0xB5, 0x0D,
	0xAE, 0xBA, 0x24, 0x51, 0xE1, 0x39, 0xAC, 0x06, 0x9E, 0x57, 0xBD, 0x7C, 0x98, 0xCC, 0x82, 0x52,
	0x28, 0x8B, 0xCF, 0x09, 0xCA, 0x09
};

/** Production HDCP2 SRM from DCP LLC website
 * Receiver IDs revoked: none */
static const u8 XHdcp22_Tx_Test_SrmDCPSiteProduction[] = {
	0x91, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x87, 0x00, 0x00, 0x00, 0x00, 0x8B, 0xBE, 0x2D, 0x46,
	0x05, 0x9F, 0x00, 0x78, 0x7B, 0xF2, 0x84, 0x79, 0x7F, 0xC4, 0xF5, 0xF6, 0xC4, 0x06, 0x36, 0xA1,
	0x20, 0x2E, 0x57, 0xEC, 0x8C, 0xA6, 0x5C, 0xF0, 0x3A, 0x14, 0x38, 0xF0, 0xB7, 0xE3, 0x68, 0xF8,
	0xB3, 0x64, 0x22, 0x55, 0x6B, 0x3E, 0xA9, 0xA8, 0x08, 0x24, 0x86, 0x55, 0x3E, 0x20, 0x0A, 0xDB,
	0x0E, 0x5F, 0x4F, 0xD5, 0x0F, 0x33, 0x52, 0x01, 0xF3, 0x62, 0x54, 0x40, 0xF3, 0x43, 0x0C, 0xFA,
	0xCD, 0x98, 0x1B, 0xA8, 0xB3, 0x77, 0xB7, 0xF8, 0xFA, 0xF7, 0x4D, 0x71, 0xFB, 0xB5, 0xBF, 0x98,
	0x9F, 0x1A, 0x1E, 0x2F, 0xF2, 0xBA, 0x80, 0xAD, 0x20, 0xB5, 0x08, 0xBA, 0xF6, 0xB5, 0x08, 0x08,
	0xCF, 0xBA, 0x49, 0x8D, 0xA5, 0x73, 0xD5, 0xDE, 0x2B, 0xEA, 0x07, 0x58, 0xA8, 0x08, 0x05, 0x66,
	0xB8, 0xD5, 0x2B, 0x9C, 0x0B, 0x32, 0xF6, 0x5A, 0x61, 0xE4, 0x9B, 0xC2, 0xF6, 0xD1, 0xF6, 0x2D,
	0x0C, 0x19, 0x06, 0x0E, 0x3E, 0xCE, 0x62, 0x97, 0x80, 0xFC, 0x50, 0x56, 0x15, 0xCB, 0xE1, 0xC7,
	0x23, 0x4B, 0x52, 0x34, 0xC0, 0x9F, 0x85, 0xEA, 0xA9, 0x15, 0x8C, 0xDD, 0x7C, 0x78, 0xD6, 0xAD,
	0x1B, 0xB8, 0x28, 0x1F, 0x50, 0xD4, 0xD5, 0x42, 0x29, 0xEC, 0xDC, 0xB9, 0xA1, 0xF4, 0x26, 0xFA,
	0x43, 0xCC, 0xCC, 0xE7, 0xEA, 0xA5, 0xD1, 0x76, 0x4C, 0xDD, 0x92, 0x9B, 0x1B, 0x1E, 0x07, 0x89,
	0x33, 0xFE, 0xD2, 0x35, 0x2E, 0x21, 0xDB, 0xF0, 0x31, 0x8A, 0x52, 0xC7, 0x1B, 0x81, 0x2E, 0x43,
	0xF6, 0x59, 0xE4, 0xAD, 0x9C, 0xDB, 0x1E, 0x80, 0x4C, 0x8D, 0x3D, 0x9C, 0xC8, 0x2D, 0x96, 0x23,
	0x2E, 0x7C, 0x14, 0x13, 0xEF, 0x4D, 0x57, 0xA2, 0x64, 0xDB, 0x33, 0xF8, 0xA9, 0x10, 0x56, 0xF4,
	0x59, 0x87, 0x43, 0xCA, 0xFC, 0x54, 0xEA, 0x2B, 0x46, 0x7F, 0x8A, 0x32, 0x86, 0x25, 0x9B, 0x2D,
	0x54, 0xC0, 0xF2, 0xEF, 0x8F, 0xE7, 0xCC, 0xFD, 0x5A, 0xB3, 0x3C, 0x4C, 0xBC, 0x51, 0x89, 0x4F,
	0x41, 0x20, 0x7E, 0xF3, 0x2A, 0x90, 0x49, 0x5A, 0xED, 0x3C, 0x8B, 0x3D, 0x9E, 0xF7, 0xC1, 0xA8,
	0x21, 0x99, 0xCF, 0x20, 0xCC, 0x17, 0xFC, 0xC7, 0xB6, 0x5F, 0xCE, 0xB3, 0x75, 0xB5, 0x27, 0x76,
	0xCA, 0x90, 0x99, 0x2F, 0x80, 0x98, 0x9B, 0x19, 0x21, 0x6D, 0x53, 0x7E, 0x1E, 0xB9, 0xE6, 0xF3,
	0xFD, 0xCB, 0x69, 0x0B, 0x10, 0xD6, 0x2A, 0xB0, 0x10, 0x5B, 0x43, 0x47, 0x11, 0xA4, 0x60, 0x28,
	0x77, 0x1D, 0xB4, 0xB2, 0xC8, 0x22, 0xDB, 0x74, 0x3E, 0x64, 0x9D, 0xA8, 0xD9, 0xAA, 0xEA, 0xFC,
	0xA8, 0xA5, 0xA7, 0xD0, 0x06, 0x88, 0xBB, 0xD7, 0x35, 0x4D, 0xDA, 0xC0, 0xB2, 0x11, 0x2B, 0xFA,
	0xED, 0xBF, 0x2A, 0x34, 0xED, 0xA4, 0x30, 0x7E, 0xFD, 0xC5, 0x21, 0xB6
};

/***************** Macros (Inline Functions) Definitions *********************/

/** Testing macro to set a specific flag */
#define TEST_FLAG_SET(Flag, Flags) (Flags&Flag)==Flag

/** Case replacement to copy a case Id to a string */
#define XHDCP22_TX_CASE_TO_STR(arg) case arg : strcpy(str, #arg); break;

/**************************** Type Definitions *******************************/

/************************** Function Prototypes ******************************/

static void XHdcp22Tx_TestReadMsg(u8* BufferPtr, u32 TestFlags);

/************************** Variable Definitions *****************************/


/************************** Function Definitions *****************************/

/*****************************************************************************/
/**
*
* This function dumps a buffer with a name (string) to the output (UART).
*
* @param  String is a string printed before the buffer contents.
* @param  Buf is the buffer to print..
* @param  Buflen is the length of the buffer.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_Dump(const char *String, const u8 *Buf, u32 Buflen)
{
	u32 Offset;

	xil_printf("\n##########################################\r\n");
	xil_printf("INFO::%s::Byte[0:%0d]", String, Buflen-1);
	for(Offset=0; Offset<Buflen; Offset++)
	{
		if((Offset%20) == 0) {
			xil_printf("\r\n");
		}

		xil_printf("%02x ", Buf[Offset]);
	}
	xil_printf("\r\n##########################################\r\n");
}

/*****************************************************************************/
/**
*
* This function simulates a read from the DDC message buffer.
* It returns hardcoded messages with valid content unless a test flag is used
* to force invalid messages.
*
* @param  BufferPtr is a pointer to the buffer that will receive a message.
* @param  TestFlags.are possible flags to influence the read.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
static void XHdcp22Tx_TestReadMsg(u8* BufferPtr, u32 TestFlags)
{
	int BytesWritten = 0;
	u8 MsgId = BufferPtr[0]; /* MessageId */

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	/* Write MessageId*/
	BufferPtr[BytesWritten++] = MsgId;
	switch (MsgId)
	{
		case XHDCP22_TX_AKE_SEND_CERT:
			/* Write HDCP22 RX certificate */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_CertRx[Index],
			       XHDCP22_TX_CERT_SIZE);
			BytesWritten += XHDCP22_TX_CERT_SIZE;

			/* Write Rrx */
			memcpy(&BufferPtr[BytesWritten], XHdcp22_Tx_Test_Rrx[Index],
			       XHDCP22_TX_RRX_SIZE);
			BytesWritten += XHDCP22_TX_RRX_SIZE;

			/* Write RxCaps */
			memcpy(&BufferPtr[BytesWritten], XHdcp22_Tx_Test_RxCaps[Index],
			       XHDCP22_TX_RXCAPS_SIZE);
			if (TEST_FLAG_SET(XHDCP22_TX_TEST_CERT_RX, TestFlags) &&
			    TEST_FLAG_SET(XHDCP22_TX_TEST_INVALID_VALUE, TestFlags)) {
				BufferPtr[1] = ~BufferPtr[1];
			}
		break;
		case XHDCP22_TX_AKE_SEND_H_PRIME:
			/* Write HDCP22 H'Prime */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_H1[Index], XHDCP22_TX_H_PRIME_SIZE);
			if (TEST_FLAG_SET(XHDCP22_TX_TEST_H1, TestFlags) &&
			    TEST_FLAG_SET(XHDCP22_TX_TEST_INVALID_VALUE, TestFlags)) {
				BufferPtr[1] = ~BufferPtr[1];
			}
		break;
		case XHDCP22_TX_AKE_SEND_PAIRING_INFO:
			/* Write Ekh(Km) */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_Ekh_Km[Index],
			       XHDCP22_TX_EKH_KM_SIZE);
		break;
		case XHDCP22_TX_LC_SEND_L_PRIME:
			/* Write LC Prime */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_L1[Index],
				   XHDCP22_TX_L_PRIME_SIZE);
			if (TEST_FLAG_SET(XHDCP22_TX_TEST_L1, TestFlags) &&
			    TEST_FLAG_SET(XHDCP22_TX_TEST_INVALID_VALUE, TestFlags)) {
				BufferPtr[1] = ~BufferPtr[1];
			}
		break;
		case XHDCP22_TX_REPEATAUTH_SEND_RECVID_LIST:
			/* Write RxInfo */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_RxInfo, XHDCP22_TX_RXINFO_SIZE);
			BytesWritten += XHDCP22_TX_RXINFO_SIZE;

			/* Write seq_num_V */
			memcpy(&BufferPtr[BytesWritten], XHdcp22_Tx_Test_SeqNum_V, XHDCP22_TX_SEQ_NUM_V_SIZE);
			BytesWritten += XHDCP22_TX_SEQ_NUM_V_SIZE;

			/* Write V'Prime */
			memcpy(&BufferPtr[BytesWritten], XHdcp22_Tx_Test_V1, XHDCP22_TX_V_PRIME_SIZE);
			BytesWritten += XHDCP22_TX_V_PRIME_SIZE;

			/* Write Receiver ID List */
			memcpy(&BufferPtr[BytesWritten], XHdcp22_Tx_Test_RcvIDList, sizeof(XHdcp22_Tx_Test_RcvIDList));
			BytesWritten += sizeof(XHdcp22_Tx_Test_RcvIDList);

			if (TEST_FLAG_SET(XHDCP22_TX_TEST_V1, TestFlags) &&
				TEST_FLAG_SET(XHDCP22_TX_TEST_INVALID_VALUE, TestFlags)) {
				BufferPtr[1] = ~BufferPtr[1];
			}
		break;
		case XHDCP22_TX_REPEATAUTH_STREAM_READY:
			/* Write M'Prime */
			memcpy(&BufferPtr[1], XHdcp22_Tx_Test_M1, XHDCP22_TX_M_PRIME_SIZE);
			if (TEST_FLAG_SET(XHDCP22_TX_TEST_M1, TestFlags) &&
				TEST_FLAG_SET(XHDCP22_TX_TEST_INVALID_VALUE, TestFlags)) {
				BufferPtr[1] = ~BufferPtr[1];
			}
		break;
		default:
		break;
	};
}

/*****************************************************************************/
/**
*
* DDCRead handler that is used in test mode, and emulates responses from a
* HDCP22 receiver.
*
* @param  DeviceAddress is the addres to use. Normally set to 0x74.
* @param  ByteCount is the amount of bytes to read.
* @param  BufferPtr contains room for dsata to read. The first byte always refers to
*         the address to read.
* @param  RefPtr is a reference to the instance and in test mode always a #XHdcp22_Tx
*         structure.
*
* @return
*         - XST_SUCCESS if reading succeeded.
*         - XST_FAILURE if reading failed.
*
* @note     None.
*
******************************************************************************/
static int XHdcp22Tx_TestDdcRead(u8 DeviceAddress, u16 ByteCount, u8* BufferPtr,
                                 u8 Stop, void *RefPtr)
{
	XHdcp22_Tx *InstancePtr = (XHdcp22_Tx *)RefPtr;
	u16  RxStatus = 0;

	if (DeviceAddress == 0x3A)
	{
		switch (InstancePtr->Test.CurrentDdcAddress)
		{
		case XHDCP22_TX_HDCPPORT_VERSION_OFFSET:
			BufferPtr[0] = 0x04;
			break;
		case XHDCP22_TX_HDCPPORT_RXSTATUS_OFFSET:
			switch(InstancePtr->Timer.ReasonId)
			{
				case XHDCP22_TX_AKE_SEND_CERT:
					RxStatus|=XHDCP22_TX_AKE_SEND_CERT_SIZE;
				break;
				case XHDCP22_TX_AKE_SEND_H_PRIME:
					RxStatus|=XHDCP22_TX_AKE_SEND_H_PRIME_SIZE;
				break;
				case XHDCP22_TX_AKE_SEND_PAIRING_INFO:
					RxStatus|=XHDCP22_TX_AKE_SEND_PAIRING_INFO_SIZE;
				break;
				case XHDCP22_TX_LC_SEND_L_PRIME:
					RxStatus|=XHDCP22_TX_LC_SEND_L_PRIME_SIZE;
				break;
				case XHDCP22_TX_REPEATAUTH_SEND_RECVID_LIST:
					RxStatus |= XHDCP22_TX_RXSTATUS_READY_MASK;
					RxStatus |= (XHDCP22_TX_RXINFO_SIZE +
					             XHDCP22_TX_SEQ_NUM_V_SIZE +
					             XHDCP22_TX_V_PRIME_SIZE +
					             sizeof(XHdcp22_Tx_Test_RcvIDList) +
					             1);
				break;
				case XHDCP22_TX_REPEATAUTH_STREAM_READY:
					RxStatus |= XHDCP22_TX_REPEATAUTH_STREAM_READY_SIZE;
				break;
			}
			*(u16 *)BufferPtr = RxStatus;
			break;
		case XHDCP22_TX_HDCPPORT_READ_MSG_OFFSET:
			XHdcp22Tx_TestReadMsg(BufferPtr, InstancePtr->Test.TestFlags);
			break;
		case XHDCP22_TX_HDCPPORT_WRITE_MSG_OFFSET:
			break;
		default:
			BufferPtr[0] = 0;
			break;
		}
	};
	return XST_SUCCESS;
}

/*****************************************************************************/
/**
*
* DDCWrite handler that is used in test mode, and emulates responses from a
* HDCP22 receiver. It also checks when possible on errors, which are logged.
*
* @param  DeviceAddress is the addres to use.
* @param  ByteCount is the amount of bytes to write.
* @param  BufferPtr contains data to write. The first byte always refers to
*         the address to write to.
* @param  RefPtr is a reference to the instance and in test mode always a #XHdcp22_Tx
*         structure.
*
* @return
*         - XST_SUCCESS if reading succeeded.
*         - XST_FAILURE if reading failed.
*
* @note   None.
*
******************************************************************************/
static int XHdcp22Tx_TestDdcWrite(u8 DeviceAddress, u16 ByteCount, u8* BufferPtr,
                                  u8 Stop, void *RefPtr)
{
	XHdcp22_Tx *InstancePtr = (XHdcp22_Tx *)RefPtr;
	InstancePtr->Test.CurrentDdcAddress = BufferPtr[0];

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, InstancePtr->Test.TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	if (DeviceAddress == XHDCP22_TX_DDC_BASE_ADDRESS)
	{
		switch (InstancePtr->Test.CurrentDdcAddress)
		{
		case XHDCP22_TX_HDCPPORT_WRITE_MSG_OFFSET:
			switch(BufferPtr[1]) {
			case XHDCP22_TX_AKE_NO_STORED_KM:
				if (memcmp(&BufferPtr[2], XHdcp22_Tx_Test_EkpubKm[Index],
					XHDCP22_TX_E_KPUB_KM_SIZE) != 0) {
						XHdcp22Tx_LogWr(InstancePtr, XHDCP22_TX_LOG_EVT_TEST_ERROR,
										XHDCP22_TX_AKE_NO_STORED_KM);
				}
				break;
			case XHDCP22_TX_SKE_SEND_EKS:
				if (memcmp(&BufferPtr[2], XHdcp22_Tx_Test_EdkeyKs[Index],
					XHDCP22_TX_EDKEY_KS_SIZE) != 0) {
						XHdcp22Tx_LogWr(InstancePtr, XHDCP22_TX_LOG_EVT_TEST_ERROR,
										XHDCP22_TX_SKE_SEND_EKS);
				}
				break;
			case XHDCP22_TX_REPEATAUTH_SEND_ACK:
				if (memcmp(&BufferPtr[2], XHdcp22_Tx_Test_V,
					XHDCP22_TX_V_PRIME_SIZE) != 0) {
					XHdcp22Tx_LogWr(InstancePtr, XHDCP22_TX_LOG_EVT_TEST_ERROR,
						XHDCP22_TX_REPEATAUTH_SEND_ACK);
				}
			break;
			case XHDCP22_TX_AKE_INIT:
			case XHDCP22_TX_AKE_STORED_KM:
			case XHDCP22_TX_LC_INIT:
			case XHDCP22_TX_REPEATAUTH_STREAM_MANAGE:
			/* No need to check these, these messages are just assembled from
			   hardcoded values from this test */
			break;
			default:
				/* Write an error to the log if the message is undefined */
				XHdcp22Tx_LogWr(InstancePtr, XHDCP22_TX_LOG_EVT_TEST_ERROR,
								XHDCP22_TX_MSG_UNDEFINED);
				break;
			};
			break;
		case XHDCP22_TX_HDCPPORT_VERSION_OFFSET:
		case XHDCP22_TX_HDCPPORT_RXSTATUS_OFFSET:
		case XHDCP22_TX_HDCPPORT_READ_MSG_OFFSET:
			/* No need to check these */
			break;
		default:
			/* Write an error to the log if the message is undefined */
			XHdcp22Tx_LogWr(InstancePtr, XHDCP22_TX_LOG_EVT_TEST_ERROR,
								XHDCP22_TX_MSG_UNDEFINED);
			break;
		}
	};
	return XST_SUCCESS;
}

/*****************************************************************************/
/**
*
* Set core into test mode for validation.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
* @param  Mode is the mode to use refer to #XHdcp22_Tx_TestMode.
* @param  TestFlags are the flags that allow to trigger test conditions.
*
* @return None.
*
* @note   Test flags should be combined for a certain test.
*         Some testflags force the use of a test item as described in:
*         "Errata to HDCP on HDMI Specification revision 2.2."
*         Examples:
*         - Receiving an invalid receiver certificate requires:
*           XHDCP22_TX_TEST_CERT_RX | XHDCP22_TX_TEST_INVALID_VALUE
*         - Receiving of the certificate times out
*           XHDCP22_TX_TEST_CERT_RX | XHDCP22_TX_TEST_RECV_TIMEOUT
*         - Receiving of HPrime in a stored Km scenario with timeout error
*           XHDCP22_TX_TEST_CERT_RX | XHDCP22_TX_TEST_STORED_KM |
*           XHDCP22_TX_TEST_RECV_TIMEOUT
*
******************************************************************************/
void XHdcp22Tx_TestSetMode(XHdcp22_Tx *InstancePtr, XHdcp22_Tx_TestMode Mode,
                           u32 TestFlags)
{
	/* Verify argument. */
	Xil_AssertVoid(InstancePtr != NULL);
	Xil_AssertVoid(Mode < XHDCP22_TX_TESTMODE_INVALID);

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	if ((TestFlags & XHDCP22_TX_TEST_CLR_PAIRINGINFO) == XHDCP22_TX_TEST_CLR_PAIRINGINFO) {
			memset(InstancePtr->Info.PairingInfo, 0, sizeof(InstancePtr->Info.PairingInfo));
	}

	/* Handle test scenarios */
	if (Mode == XHDCP22_TX_TESTMODE_NO_RX || Mode == XHDCP22_TX_TESTMODE_UNIT) {
		/* Set test handlers for DDC */
		XHdcp22Tx_SetCallback(InstancePtr, XHDCP22_TX_HANDLER_DDC_WRITE, XHdcp22Tx_TestDdcWrite, InstancePtr);
		XHdcp22Tx_SetCallback(InstancePtr, XHDCP22_TX_HANDLER_DDC_READ, XHdcp22Tx_TestDdcRead, InstancePtr);

		if ((TestFlags & XHDCP22_TX_TEST_STORED_KM) == XHDCP22_TX_TEST_STORED_KM) {
			memcpy(InstancePtr->Info.PairingInfo[0].ReceiverId, XHdcp22_Tx_Test_CertRx[Index], XHDCP22_TX_CERT_RCVID_SIZE);
			memcpy(InstancePtr->Info.PairingInfo[0].Km, XHdcp22_Tx_Test_Km[Index], XHDCP22_TX_KM_SIZE);
			memcpy(InstancePtr->Info.PairingInfo[0].Rrx, XHdcp22_Tx_Test_Rrx[Index], XHDCP22_TX_RRX_SIZE);
			memcpy(InstancePtr->Info.PairingInfo[0].Rtx, XHdcp22_Tx_Test_Rtx[Index], XHDCP22_TX_RTX_SIZE);
			memcpy(InstancePtr->Info.PairingInfo[0].RxCaps, XHdcp22_Tx_Test_RxCaps[Index], XHDCP22_TX_RXCAPS_SIZE);
			memcpy(InstancePtr->Info.PairingInfo[0].Ekh_Km, XHdcp22_Tx_Test_Ekh_Km[Index], XHDCP22_TX_EKH_KM_SIZE);
		}
	}

	InstancePtr->Test.TestMode = Mode;
	InstancePtr->Test.TestFlags = TestFlags;
}

/*****************************************************************************/
/**
*
* Compare logged results with expected results as a validation method.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
* @param  Expected specifies expected logitems to compare with.
* @param  nExpected specifies number of items to compare with.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
u8 XHdcp22Tx_TestCheckResults(XHdcp22_Tx* InstancePtr,
                              XHdcp22_Tx_LogItem *Expected, u32 nExpected)
{
	u32 iLog = 0;
	u32 iExpected = 0;
	u8  Result = (TRUE);

	for (iExpected=0; iExpected<nExpected; iExpected++)
	{
		if (InstancePtr->Log.LogItems[iLog].LogEvent != Expected[iExpected].LogEvent ||
			InstancePtr->Log.LogItems[iLog].Data != Expected[iExpected].Data) {
				xil_printf("FAIL in step %d: \r\n\tExpected log event=%d, actual=%d\r\n"
				           "\tExpected data=0x%.2x, actual=0x%.2x\r\n",
				           iExpected,  Expected[iExpected].LogEvent,
				           InstancePtr->Log.LogItems[iLog].LogEvent,
				           Expected[iExpected].Data,
				           InstancePtr->Log.LogItems[iLog].Data);

				Result = (FALSE);
				break;
		}
		iLog++;
	}
	return Result;
}

/*****************************************************************************/
/**
*
* Determine if a timeout should be simulated in uni test mode (NO_RX).
*
* @param  InstancePtr is a pointer to the HDcp22_Tx core instance.
*
* @return TRUE if a timeout must be simulated, FALSE if not.
*
* @note   None.
*
******************************************************************************/
u8 XHdcp22Tx_TestSimulateTimeout(XHdcp22_Tx* InstancePtr)
{
	u8 SimTimeout = FALSE;

	/* Simulate a timeout if in testmode without RX (unit test) */
	if ((InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_NO_RX ||
	     InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_UNIT)&&
	     InstancePtr->Test.TestFlags&XHDCP22_TX_TEST_RCV_TIMEOUT)
	{
		switch(InstancePtr->Timer.ReasonId)
		{
			case XHDCP22_TX_AKE_SEND_CERT:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_CERT_RX, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			case XHDCP22_TX_AKE_SEND_H_PRIME:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_H1, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			case XHDCP22_TX_AKE_SEND_PAIRING_INFO:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_EKH_KM, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			case XHDCP22_TX_LC_SEND_L_PRIME:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_L1, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			case XHDCP22_TX_REPEATAUTH_SEND_RECVID_LIST:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_V1, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			case XHDCP22_TX_REPEATAUTH_STREAM_READY:
				if (TEST_FLAG_SET(XHDCP22_TX_TEST_M1, InstancePtr->Test.TestFlags)) {
					SimTimeout = TRUE;
				}
				break;
			default:
				SimTimeout = FALSE;
				break;
		}
	}
	return SimTimeout;
}

/*****************************************************************************/
/**
*
* This function replaces Rtx with a test vector if the testmode is
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
* @param  RtxPtr is a pointer to Rtx.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateRtx(XHdcp22_Tx *InstancePtr, u8* RtxPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, InstancePtr->Test.TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	memcpy(RtxPtr, XHdcp22_Tx_Test_Rtx[Index], XHDCP22_TX_RTX_SIZE);
}

/*****************************************************************************/
/**
*
* This function replaces Km with a test vector if the testmode is
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance..
* @param  KmPtr is a pointer to Km.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateKm(XHdcp22_Tx *InstancePtr, u8* KmPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, InstancePtr->Test.TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	memcpy(KmPtr, XHdcp22_Tx_Test_Km[Index], XHDCP22_TX_KM_SIZE);
}

/*****************************************************************************/
/**
*
* This function replaces the masking seed used in RSA-OEAP with a test vector
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance..
* @param  SeedPtr is a pointer to masking seed.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateKmMaskingSeed(XHdcp22_Tx *InstancePtr, u8* SeedPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}
	memcpy(SeedPtr, XHdcp22_Tx_Test_Km_MaskingSeed, XHDCP22_TX_KM_MSK_SEED_SIZE);
}

/*****************************************************************************/
/**
*
* This function replaces Rn with a test vector if the testmode is
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance..
* @param  RnPtr is a pointer to Rn.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateRn(XHdcp22_Tx *InstancePtr, u8* RnPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, InstancePtr->Test.TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	memcpy(RnPtr, XHdcp22_Tx_Test_Rn[Index], XHDCP22_TX_RN_SIZE);
}

/*****************************************************************************/
/**
*
* This function replaces Riv with a test vector if the testmode is
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance..
* @param  RivPtr is a pointer to Riv.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateRiv(XHdcp22_Tx *InstancePtr, u8* RivPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}

	u8 Index = XHDCP22_TX_TEST_VECTOR_R2_INDEX;
	if (TEST_FLAG_SET(XHDCP22_TX_TEST_USE_TEST_VECTOR_R1, InstancePtr->Test.TestFlags)) {
		Index = XHDCP22_TX_TEST_VECTOR_R1_INDEX;
	}

	memcpy(RivPtr, XHdcp22_Tx_Test_Riv[Index], XHDCP22_TX_RIV_SIZE);
}

/*****************************************************************************/
/**
*
* This function replaces Ks with a test vector if the testmode is
* #XHDCP22_TX_TESTMODE_NO_RX, #XHDCP22_TX_TESTMODE_UNIT or
* #XHDCP22_TX_TESTMODE_USE_TESTKEYS.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
* @param  KsPtr is a pointer to Ks.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_TestGenerateKs(XHdcp22_Tx *InstancePtr, u8* KsPtr)
{
	if (InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_NO_RX &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_UNIT &&
	    InstancePtr->Test.TestMode != XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return;
	}
	memcpy(KsPtr, XHdcp22_Tx_Test_Ks, XHDCP22_TX_KS_SIZE);
}

/*****************************************************************************/
/**
*
* This function returns a pointer to the KPubDpc testvector, or NULL if
* not in test mode.
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
*
* @return A pointer to the LLC public test vector KPubDpc or NULL if not in
*         test mode without RX.
* @note   None.
*
******************************************************************************/
const u8* XHdcp22Tx_TestGetKPubDpc(XHdcp22_Tx *InstancePtr)
{
	if (InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_SW_RX ||
	    InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_NO_RX ||
	    InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_UNIT  ||
	    InstancePtr->Test.TestMode == XHDCP22_TX_TESTMODE_USE_TESTKEYS) {
		return XHdcp22_Tx_Test_Kpubdcp;
	}

	return NULL;
}

/*****************************************************************************/
/**
*
* This function returns a pointer to the SRM testvector
*
* @param  InstancePtr is a pointer to the HDCP22 TX core instance.
* @param  Select specifies the SRM to return
*    - 0 : HDCP2 SRM from the HDCP 2.2 Errata (February 9th, 2015)
*    - 1 : TEST HDCP2 SRM from the DCP LLC website (TEST HDCP2.SRM)
*    - 2 : Production HDCP2 SRM from the DCP LLC website (HDCP2.SRM)
* @return A pointer to the SRM test vector or NULL if invalid an
*         invalid selection was specified.
*
* @note   None.
*
******************************************************************************/
const u8* XHdcp22Tx_TestGetSrm(XHdcp22_Tx *InstancePtr, u8 Select)
{
	switch (Select) {
		case 0 :
			return XHdcp22_Tx_Test_SrmErrata;

		case 1 :
			return XHdcp22_Tx_Test_SrmDCPSiteTest;

		case 2 :
			return XHdcp22_Tx_Test_SrmDCPSiteProduction;

		default :
			return NULL;
	}
}

/*****************************************************************************/
/**
*
* This function prints the content of log buffer in a unit test result
* usable form. The displayed test is source that can be copied. If
* the test mode is set toXHDCP22_TX_TESTMODE_NO_RX, this function is executed.
* See also #XHdcp22Tx_LogDisplay.
*
* @param  InstancePtr is a pointer to the XHdcp22_Tx core instance.
*
* @return None.
*
* @note   None.
*
******************************************************************************/
void XHdcp22Tx_LogDisplayUnitTest(XHdcp22_Tx *InstancePtr)
{
	XHdcp22_Tx_LogItem* LogPtr;
	char str[255];
	u32 Index = 0;

	xil_printf("\r\n--------------------------------------\r\n");
	do {
		/* Read log data */
		LogPtr = XHdcp22Tx_LogRd(InstancePtr);

		switch (LogPtr->LogEvent)
		{
		case XHDCP22_TX_LOG_EVT_STATE:
			switch(LogPtr->Data)
			{
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_H0)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_H1)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A0)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A1)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A1_1)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A1_NSK0)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A1_NSK1)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A1_SK0)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A2)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A2_1)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A3)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A4)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A5)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A6)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A7)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A8)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A6_A7_A8)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A9)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_STATE_A9_1)
				default: break;
			};
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_STATE;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %s;\r\n", Index, str);
		break;
		case XHDCP22_TX_LOG_EVT_POLL_RESULT:
			switch(LogPtr->Data)
			{
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_INCOMPATIBLE_RX)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_AUTHENTICATION_BUSY)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_AUTHENTICATED)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_UNAUTHENTICATED)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_REAUTHENTICATE_REQUESTED)
				default: break;
			}
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_POLL_RESULT;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %s;\r\n", Index, str);
		break;
		case XHDCP22_TX_LOG_EVT_ENABLED:
			if (LogPtr->Data == (FALSE)) {
				strcpy(str, "FALSE");
			} else {
				strcpy(str, "TRUE");
			}
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_ENABLED;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %s;\r\n", Index, str);
		break;
		case XHDCP22_TX_LOG_EVT_RESET:
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_RESET;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %d;\r\n", Index, LogPtr->Data);
		break;
		case XHDCP22_TX_LOG_EVT_TEST_ERROR:
			switch(LogPtr->Data)
			{
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_AKE_NO_STORED_KM)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_SKE_SEND_EKS)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_REPEATAUTH_SEND_ACK)
				XHDCP22_TX_CASE_TO_STR(XHDCP22_TX_MSG_UNDEFINED)
				default: break;
			};
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_TEST_ERROR;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %s;\r\n", Index, str);
		break;
		case XHDCP22_TX_LOG_EVT_ENCR_ENABLED:
			if (LogPtr->Data == (FALSE)) {
				strcpy(str, "FALSE");
			} else {
				strcpy(str, "TRUE");
			}
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_ENCR_ENABLED;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %s;\r\n", Index, str);
			break;
		case XHDCP22_TX_LOG_EVT_LCCHK_COUNT:
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_LCCHK_COUNT;\r\n",
			           Index);
			xil_printf("TestRes[%d].Data = %d;\r\n", Index, LogPtr->Data);
			break;
		case XHDCP22_TX_LOG_EVT_STRMMNGCHK_COUNT:
			xil_printf("TestRes[%d].LogEvent = XHDCP22_TX_LOG_EVT_STRMMNGCHK_COUNT;\r\n",
				Index);
			xil_printf("TestRes[%d].Data = %d;\r\n", Index, LogPtr->Data);
			break;
		default: break;
		};
		Index++;
	} while (LogPtr->LogEvent != XHDCP22_TX_LOG_EVT_NONE);
	xil_printf("NumTestResItems = %d;\r\n", Index - 1);
	xil_printf("\r\n--------------------------------------\r\n");
}

#endif // _XHDCP22_TX_TEST_

/** @} */
