/*******************************************************************************
* Copyright (C) 2020-2021 Xilinx, Inc.  All rights reserved.
* Copyright 2022-2025 Advanced Micro Devices, Inc. All Rights Reserved.
* SPDX-License-Identifier: MIT
*******************************************************************************/



/*****************************************************************************/
/**
*
* @file si5328drv.h
*
* This file contains definitions for low-level driver functions for
* controlling the SiliconLabs Si5328 clock generator as mounted on the KC705
* demo board.
* The user should refer to the hardware device specification for more details
* of the device operation.
*
* <pre>
* MODIFICATION HISTORY:
*
* Ver   Who  Date        Changes
* ----- --- ----------   -----------------------------------------------
* 1.00  hf  2014/10/10   First release
* 1.1   gm  2017/06/23   Added Si5328_Reset in API declaration
*       yh  2017/07/19   Added SI5328 control using PS IIC
* </pre>
*
******************************************************************************/

#ifndef SI5328DRV_H_
#define SI5328DRV_H_
#ifdef __cplusplus
extern "C" {
#endif

#include "xil_types.h"
#include "xparameters.h"
#if defined (ARMR5) || (__aarch64__)
#include "xiicps.h"
#endif

/******************************************************************************
 * User settable defines that depend on the specific board design.
 * The defaults are for the Xilinx KC705 board.
 *****************************************************************************/

/**
 * The frequency of the crystal connected to the XA/XB pins of the Si5328 in Hz.
 */
#define SI5328_XTAL_FREQ 114285000


/******************************************************************************
 * Defines independent on the specific board design. Should not be changed.
 *****************************************************************************/

/**
 * Debug output enable. Set to TRUE to enable debug prints,
 * to FALSE to disable debug prints.
 */
#define SI5328_DEBUG FALSE

/**
 * The following constants are error codes generated by the functions in
 * this driver.
 */
#define SI5328_SUCCESS      0 /**< Operation was successful */
#define SI5328_ERR_IIC     -1 /**< IIC error occurred */
#define SI5328_ERR_FREQ    -2 /**< Could not calculate frequency setting */
#define SI5328_ERR_PARM    -3 /**< Invalid parameter */

/**
 * The following constants define the clock input select values.
 */
#define SI5328_CLKSRC_CLK1 1  /**< Use clock input 1 */
#define SI5328_CLKSRC_CLK2 2  /**< Use clock input 2 */
#define SI5328_CLKSRC_XTAL 3  /**< Use crystal (free running mode) */

/**
 * The following constants define the limits of the Si5328 frequencies.
 */
#define SI5328_FOSC_MIN 4850000000 /**< Minimum oscillator frequency */
#define SI5328_FOSC_MAX 5670000000 /**< Maximum oscillator frequency */
#define SI5328_F3_MIN        10000
// 2000 /**< Minimum phase detector frequency */
#define SI5328_F3_MAX      2000000 /**< Maximum phase detector frequency */
#define SI5328_FIN_MIN        2000 /**< Minimum input frequency */
#define SI5328_FIN_MAX   710000000 /**< Maximum input frequency */
#define SI5328_FOUT_MIN       2000 /**< Minimum output frequency */
#define SI5328_FOUT_MAX  945000000 /**< Maximum output frequency */

/**
 * The following constants define the limits of the divider settings.
 */
#define SI5328_N1_HS_MIN  6        /**< Minimum N1_HS setting
                  (4 and 5 are for higher output frequencies than we support */
#define SI5328_N1_HS_MAX 11        /**< Maximum N1_HS setting */
#define SI5328_NC_LS_MIN  1        /**< Minimum NCn_LS setting
										(1 and even values) */
#define SI5328_NC_LS_MAX 0x100000  /**< Maximum NCn_LS setting
										(1 and even values) */
#define SI5328_N2_HS_MIN  4        /**< Minimum NC2_HS setting */
#define SI5328_N2_HS_MAX 11        /**< Maximum NC2_HS setting */
#define SI5328_N2_LS_MIN  2        /**< Minimum NC2_LS setting
										(even values only) */
#define SI5328_N2_LS_MAX 0x100000  /**< Maximum NC2_LS setting
										(even values only) */
#define SI5328_N3_MIN     1        /**< Minimum N3n setting */
#define SI5328_N3_MAX    0x080000  /**< Maximum N3n setting */


typedef struct {
    // Current Si5342 parameters
    u32 n1_min;
    u32 n1_max;
    u32 n1_hs;
    u32 nc_ls_min;
    u32 nc_ls_max;
    u32 nc_ls;
    u32 n2_hs;
    u32 n2_ls_min;
    u32 n2_ls_max;
    u32 n2_ls;
    u32 n3_min;
    u32 n3_max;
    u32 n3;
    // Current frequencies (fixed point 36.28 notation)
    u64 fin;
    u64 fout;
    u64 fosc;
    // Best settings found
    u64 best_delta_fout;
    u64 best_fout;
    u32 best_n1_hs;
    u32 best_nc_ls;
    u32 best_n2_hs;
    u32 best_n2_ls;
    u32 best_n3;
} si5328_settings_t;

/*****************************************************************************/
/**
 * Initialize the SiliconLabs Si5328 clock generator. After initialization,
 * the clock generator is not generating a clock yet. Call si5328_set_clock
 * to start the clock generator.
 *
 * @param    IICBaseAddress contains the base address of the IIC master
 *           device.
 * @param    IICAddress contains the 7 bit IIC address of the Si5328 device.
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_IIC for IIC access failure.
 *****************************************************************************/
int Si5328_Init(u32 IICBaseAddress, u8 IICAddress);

/*****************************************************************************/
/**
 * Calculate the frequency settings for the desired output frequency.
 *
 * @param    ClkInFreq contains the frequency of the input clock.
 * @param    ClkOutFreq contains the desired output clock frequency.
 * @param    N1_hs  will be set to the value for the N1_HS register.
 * @param    NCn_ls will be set to the value for the NCn_LS register.
 * @param    N2_hs  will be set to the value for the N2_HS register.
 * @param    N2_ls  will be set to the value for the N2_LS register.
 * @param    N3n    will be set to the value for the N3n register.
 * @param    BwSel  will be set to the value for the BW_SEL register.
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_FREQ when the requested
 *           frequency cannot be generated.
 * @note     Private function.
 *****************************************************************************/
int Si5328_CalcFreqSettings(u32 ClkInFreq, u32 ClkOutFreq,
                        u8  *N1_hs, u32 *NCn_ls,
                        u8  *N2_hs, u32 *N2_ls,
                        u32 *N3n,   u8  *BwSel);

#if defined (ARMR5) || (__aarch64__)
/*****************************************************************************/
/**
 * Initialize the SiliconLabs Si5328 clock generator. After initialization,
 * the clock generator is not generating a clock yet. Call si5328_set_clock
 * to start the clock generator.
 *
 * @param    InstancePtr contains the pointer to PS IIC master device
 * @param    IICAddress contains the 7 bit IIC address of the Si5328 device.
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_IIC for IIC access failure.
 *****************************************************************************/

s32 Si5328_Init_Ps(XIicPs *InstancePtr, u16 IICAddress);
#endif

/**
 * Set the output frequency
 *
 * @param    IICBaseAddress contains the base address of the IIC master
 *           device.
 * @param    IICAddress contains the 7 bit IIC address of the Si5328 device.
 * @param    ClkSrc selects the clock input to use.
 * @param    ClkInFreq contains the frequency of the input clock.
 * @param    ClkOutFreq contains the desired output clock frequency
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_IIC for IIC access failure,
 *           SI5328_ERR_FREQ when the requested frequency cannot be generated.
 */
int Si5328_SetClock(u32 IICBaseAddress, u8 IICAddress, u8 ClkSrc,
                    u32 ClkInFreq, u32 ClkOutFreq);

#if defined (ARMR5) || (__aarch64__)
/**
 * Set the output frequency
 *
 * @param    InstancePtr contains the pointer to PS IIC master device
 * @param    IICAddress contains the 7 bit IIC address of the Si5328 device.
 * @param    ClkSrc selects the clock input to use.
 * @param    ClkInFreq contains the frequency of the input clock.
 * @param    ClkOutFreq contains the desired output clock frequency
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_IIC for IIC access failure,
 *           SI5328_ERR_FREQ when the requested frequency cannot be generated.
 */
s32 Si5328_SetClock_Ps(XIicPs *InstancePtr, u16 IICAddress, u8 ClkSrc,
                    u32 ClkInFreq, u32 ClkOutFreq);
#endif

/*****************************************************************************/
/**
 * Reset the SiliconLabs Si5328 clock generator.
 *
 * @param    IICBaseAddress contains the base address of the IIC master
 *           device.
 * @param    IICAddress contains the 7 bit IIC address of the Si5328 device.
 *
 * @return   SI5328_SUCCESS for success, SI5328_ERR_IIC for IIC access failure.
 *
 * @note     Private function.
 *****************************************************************************/
int Si5328_Reset(u32 IICBaseAddress, u8 IICAddress);
#ifdef __cplusplus
}
#endif
#endif /* SI5328DRV_H_ */
