/*
 * Copyright (C) 2010 - 2022 Xilinx, Inc.
 * Copyright (C) 2022 - 2024 Advanced Micro Devices, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 */

#include <stdio.h>
#include <string.h>

#include <xparameters.h>

#include "xlwipconfig.h"
#include "lwip/opt.h"
#include "lwip/def.h"
#include "lwip/mem.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include "lwip/stats.h"
#include "lwip/igmp.h"

#include "netif/etharp.h"
#include "netif/xaxiemacif.h"
#include "netif/xadapter.h"
#include "netif/xpqueue.h"

#include "xaxiemacif_fifo.h"
#include "xaxiemacif_hw.h"

#include "xparameters.h"
#ifndef SDT
#if XLWIP_CONFIG_INCLUDE_AXIETH_ON_ZYNQ == 1
#include "xscugic.h"
#else
#include "xintc.h"
#endif
#else
#include "xinterrupt_wrap.h"
#endif

#if LWIP_IPV6
#include "lwip/ethip6.h"
#endif

/* Define those to better describe your network interface. */
#define IFNAME0 't'
#define IFNAME1 'e'

#if LWIP_IGMP
static err_t xaxiemacif_mac_filter_update (struct netif *netif,
								ip_addr_t *group, u8_t action);

static u8_t xaxiemac_mcast_entry_mask = 0;
#endif

#if LWIP_IPV6 && LWIP_IPV6_MLD
#define XAXIEMAC_MAX_MAC_ADDR	4
static err_t xaxiemacif_mld6_mac_filter_update (struct netif *netif,
								ip_addr_t *group, u8_t action);

static u8_t xaxiemac_mld6_mcast_entry_mask;
#endif

#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
extern volatile u32_t notifyinfo[XLWIP_CONFIG_N_TX_DESC];
#endif

/*
 * this function is always called with interrupts off
 * this function also assumes that there are available BD's
 */
#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
static err_t _unbuffered_low_level_output(xaxiemacif_s *xaxiemacif,
        struct pbuf *p, u32_t block_till_tx_complete, u32_t *to_block_index )
#else
static err_t _unbuffered_low_level_output(xaxiemacif_s *xaxiemacif,
														struct pbuf *p)
#endif
{
	XStatus status = 0;
    err_t err = ERR_MEM;

#if ETH_PAD_SIZE
	pbuf_header(p, -ETH_PAD_SIZE);			/* drop the padding word */
#endif
	if (XAxiEthernet_IsDma(&xaxiemacif->axi_ethernet)) {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_DMA
#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
    if (block_till_tx_complete == 1) {
        status = axidma_sgsend(xaxiemacif, p, 1, to_block_index);
    } else {
        status = axidma_sgsend(xaxiemacif, p, 0, to_block_index);
    }
#else
    status = axidma_sgsend(xaxiemacif, p);
#endif
#endif
	} else if (XAxiEthernet_IsMcDma(&xaxiemacif->axi_ethernet)) {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_MCDMA
		xil_printf("lwip support with mcdma is deprecated\n");
		status = axi_mcdma_sgsend(xaxiemacif, p);
#endif
	} else {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_FIFO
		status = axififo_send(xaxiemacif, p);
#endif
	}

	if (status != XST_SUCCESS) {
#if LINK_STATS
		lwip_stats.link.drop++;
#endif
	} else {
        err = ERR_OK;
    }

#if ETH_PAD_SIZE
	pbuf_header(p, ETH_PAD_SIZE);	/* reclaim the padding word */
#endif

#if LINK_STATS
	lwip_stats.link.xmit++;
#endif /* LINK_STATS */

	return err;

}

/*
 * low_level_output():
 *
 * Should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 */

static err_t low_level_output(struct netif *netif, struct pbuf *p)
{
    err_t err = ERR_MEM;
#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
    u32_t notfifyblocksleepcntr;
    u32_t to_block_index;
#endif

        SYS_ARCH_DECL_PROTECT(lev);
        struct xemac_s *xemac = (struct xemac_s *)(netif->state);
        xaxiemacif_s *xaxiemacif = (xaxiemacif_s *)(xemac->state);

#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_DMA
	/*
	 * With AXI Ethernet on Zynq, we observed unexplained delays for
	 * BD Status update. As a result, we are hitting a condition where
	 * there are no BDs free to transmit packets. So, we have added
	 * this logic where we look for the status update in a definite
	 * loop.
	 */
	XAxiDma_BdRing *txring = XAxiDma_GetTxRing(&xaxiemacif->axidma);
#endif
        int count = 100;

        SYS_ARCH_PROTECT(lev);

        while (count) {

		/* check if space is available to send */
		if (xaxiemac_is_tx_space_available(xaxiemacif)) {
#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
    if (netif_is_opt_block_tx_set(netif, NETIF_ENABLE_BLOCKING_TX_FOR_PACKET)) {
        err = _unbuffered_low_level_output(xaxiemacif, p, 1, &to_block_index);
		break;
    } else {
        err = _unbuffered_low_level_output(xaxiemacif, p, 0, &to_block_index);
		break;
    }
#else
    err = _unbuffered_low_level_output(xaxiemacif, p);
	break;
#endif
    } else {
#if LINK_STATS
			lwip_stats.link.drop++;
#endif
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_DMA
			process_sent_bds(txring);
#endif
			count--;
    }
    }

	if (count == 0) {
		xil_printf("pack dropped, no space\r\n");
        SYS_ARCH_UNPROTECT(lev);
        goto return_pack_dropped;
	}

        SYS_ARCH_UNPROTECT(lev);
#if LWIP_UDP_OPT_BLOCK_TX_TILL_COMPLETE
    if (netif_is_opt_block_tx_set(netif, NETIF_ENABLE_BLOCKING_TX_FOR_PACKET)) {
        /* Wait for approx 1 second before timing out */
        notfifyblocksleepcntr = 900000;
        while(notifyinfo[to_block_index] == 1) {
            usleep(1);
            notfifyblocksleepcntr--;
            if (notfifyblocksleepcntr <= 0) {
                err = ERR_TIMEOUT;
                break;
            }
        }
    }
    netif_clear_opt_block_tx(netif, NETIF_ENABLE_BLOCKING_TX_FOR_PACKET);
#endif
return_pack_dropped:
        return err;
}

/*
 * low_level_input():
 *
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 */
static struct pbuf *low_level_input(struct netif *netif)
{
	struct xemac_s *xemac = (struct xemac_s *)(netif->state);
	xaxiemacif_s *xaxiemacif = (xaxiemacif_s *)(xemac->state);
	struct pbuf *p;

	/* see if there is data to process */
	if (pq_qlength(xaxiemacif->recv_q) == 0)
		return NULL;

	/* return one packet from receive q */
	p = (struct pbuf *)pq_dequeue(xaxiemacif->recv_q);
	return p;
}

/*
 * xaxiemacif_output():
 *
 * This function is called by the TCP/IP stack when an IP packet
 * should be sent. It calls the function called low_level_output() to
 * do the actual transmission of the packet.
 *
 */

static err_t xaxiemacif_output(struct netif *netif, struct pbuf *p,
		const ip_addr_t *ipaddr)
{
	/* resolve hardware address, then send (or queue) packet */
	return etharp_output(netif, p, ipaddr);
}

/*
 * xaxiemacif_input():
 *
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface.
 *
 * Returns the number of packets read (max 1 packet on success,
 * 0 if there are no packets)
 *
 */

int xaxiemacif_input(struct netif *netif)
{
	struct eth_hdr *ethhdr;
	struct pbuf *p;
	SYS_ARCH_DECL_PROTECT(lev);

#if !NO_SYS
	while (1)
#endif
	{
		/* move received packet into a new pbuf */
		SYS_ARCH_PROTECT(lev);
		p = low_level_input(netif);
		SYS_ARCH_UNPROTECT(lev);

		/* no packet could be read, silently ignore this */
		if (p == NULL)
			return 0;

		/* points to packet payload, which starts with an Ethernet header */
		ethhdr = p->payload;

#if LINK_STATS
		lwip_stats.link.recv++;
#endif /* LINK_STATS */

		switch (htons(ethhdr->type)) {
			/* IP or ARP packet? */
			case ETHTYPE_IP:
			case ETHTYPE_ARP:
#if LWIP_IPV6
			/*IPv6 Packet?*/
			case ETHTYPE_IPV6:
#endif
#if PPPOE_SUPPORT
				/* PPPoE packet? */
			case ETHTYPE_PPPOEDISC:
			case ETHTYPE_PPPOE:
#endif /* PPPOE_SUPPORT */
				/* full packet send to tcpip_thread to process */
				if (netif->input(p, netif) != ERR_OK) {
					LWIP_DEBUGF(NETIF_DEBUG, ("xaxiemacif_input: IP input error\r\n"));
					pbuf_free(p);
					p = NULL;
				}
				break;

			default:
				pbuf_free(p);
				p = NULL;
				break;
		}
	}
	return 1;
}

static err_t low_level_init(struct netif *netif)
{
	unsigned mac_address = (unsigned)(UINTPTR)(netif->state);
	struct xemac_s *xemac;
	xaxiemacif_s *xaxiemacif;
	XAxiEthernet_Config *mac_config;

	xaxiemacif = mem_malloc(sizeof *xaxiemacif);
	if (xaxiemacif == NULL) {
		LWIP_DEBUGF(NETIF_DEBUG, ("xaxiemacif_init: out of memory\r\n"));
		return ERR_MEM;
	}

	xemac = mem_malloc(sizeof *xemac);
	if (xemac == NULL) {
		LWIP_DEBUGF(NETIF_DEBUG, ("xaxiemacif_init: out of memory\r\n"));
		return ERR_MEM;
	}

	xemac->state = (void *)xaxiemacif;
	xemac->topology_index = xtopology_find_index(mac_address);
	xemac->type = xemac_type_axi_ethernet;

	xaxiemacif->send_q = NULL;
	xaxiemacif->recv_q = pq_create_queue();
	if (!xaxiemacif->recv_q)
		return ERR_MEM;

	/* maximum transfer unit */
#ifdef USE_JUMBO_FRAMES
	netif->mtu = XAE_JUMBO_MTU - XAE_HDR_SIZE;
#else
	netif->mtu = XAE_MTU - XAE_HDR_SIZE;
#endif

#if LWIP_IGMP
	netif->igmp_mac_filter = xaxiemacif_mac_filter_update;
#endif

#if LWIP_IPV6 && LWIP_IPV6_MLD
 netif->mld_mac_filter = xaxiemacif_mld6_mac_filter_update;
#endif

	netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP |
				   NETIF_FLAG_LINK_UP;

#if LWIP_IPV6 && LWIP_IPV6_MLD
	netif->flags |= NETIF_FLAG_MLD6;
#endif

#if LWIP_IGMP
	netif->flags |= NETIF_FLAG_IGMP;
#endif

#if !NO_SYS
	sys_sem_new(&xemac->sem_rx_data_available, 0);
#endif

	/* obtain config of this emac */
	mac_config = xaxiemac_lookup_config((unsigned)(UINTPTR)netif->state);

	XAxiEthernet_Initialize(&xaxiemacif->axi_ethernet, mac_config,
				mac_config->BaseAddress);

#ifdef XPAR_GIGE_PCS_PMA_SGMII_CORE_PRESENT
	enable_sgmii_clock(&xaxiemacif->axi_ethernet);
#endif

	/* figure out if the system has DMA */
	if (XAxiEthernet_IsDma(&xaxiemacif->axi_ethernet)) {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_DMA
		/* initialize the DMA engine */
		init_axi_dma(xemac);
#endif
	} else if (XAxiEthernet_IsFifo(&xaxiemacif->axi_ethernet)) {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_FIFO
		/* initialize the locallink FIFOs */
		init_axi_fifo(xemac);
#endif
	} else if (XAxiEthernet_IsMcDma(&xaxiemacif->axi_ethernet)) {
#ifdef XLWIP_CONFIG_INCLUDE_AXI_ETHERNET_MCDMA
		/* Initialize MCDMA engine */
		init_axi_mcdma(xemac);
#endif
	} else {
		/* should not occur */
		LWIP_DEBUGF(NETIF_DEBUG, ("xaxiemacif_init: mac is not configured with DMA, MCDMA or FIFO\r\n"));
		return ERR_IF;
	}

	/* initialize the mac */
	init_axiemac(xaxiemacif, netif);

	/* replace the state in netif (currently the emac baseaddress)
	 * with the mac instance pointer.
	 */
	netif->state = (void *)xemac;

	return ERR_OK;
}
#if LWIP_IPV6 && LWIP_IPV6_MLD
static u8_t xaxiemacif_ip6_addr_ismulticast(ip6_addr_t* ip_addr)
{
	if(ip6_addr_ismulticast_linklocal(ip_addr)||
           ip6_addr_ismulticast_iflocal(ip_addr)   ||
           ip6_addr_ismulticast_adminlocal(ip_addr)||
           ip6_addr_ismulticast_sitelocal(ip_addr) ||
           ip6_addr_ismulticast_orglocal(ip_addr)  ||
           ip6_addr_ismulticast_global(ip_addr)) {
	/*Return TRUE if IPv6 is Multicast type*/
		return TRUE;
	} else {
		return FALSE;
	}
}
static void xaxiemacif_mld6_mac_hash_update (struct netif *netif, u8_t *ip_addr,
		u8_t action,u8_t entry)
{
	u8_t multicast_mac_addr[6];
	u8_t multicast_mac_addr_to_clr[6];
	struct xemac_s *xemac = (struct xemac_s *)(netif->state);
	xaxiemacif_s *xaxiemacif = (xaxiemacif_s *)(xemac->state);


	if (action == NETIF_ADD_MAC_FILTER) {
		/* Set Mulitcast mac address in hash table */
		multicast_mac_addr[0] = LL_IP6_MULTICAST_ADDR_0;
		multicast_mac_addr[1] = LL_IP6_MULTICAST_ADDR_1;
		multicast_mac_addr[2] = ip_addr[12];
		multicast_mac_addr[3] = ip_addr[13];
		multicast_mac_addr[4] = ip_addr[14];
		multicast_mac_addr[5] = ip_addr[15];

		XAxiEthernet_Stop(&xaxiemacif->axi_ethernet);

		XAxiEthernet_MulticastAdd(&xaxiemacif->axi_ethernet,multicast_mac_addr, entry);

		XAxiEthernet_Start(&xaxiemacif->axi_ethernet);

	} else if (action == NETIF_DEL_MAC_FILTER) {
		/* Remove Mulitcast mac address in hash table */

		XAxiEthernet_MulticastGet(&xaxiemacif->axi_ethernet,multicast_mac_addr_to_clr, entry);

		XAxiEthernet_Stop(&xaxiemacif->axi_ethernet);

		XAxiEthernet_MulticastClear(&xaxiemacif->axi_ethernet, entry);

		XAxiEthernet_Start(&xaxiemacif->axi_ethernet);
	}
}
static err_t
xaxiemacif_mld6_mac_filter_update (struct netif *netif, ip_addr_t *group,
								u8_t action)
{
	u8_t temp_mask;
	unsigned int i;
	u8_t entry;
	u8_t * ip_addr = (u8_t *) group;

	if(!(xaxiemacif_ip6_addr_ismulticast((ip6_addr_t*) ip_addr))) {
		  LWIP_DEBUGF(NETIF_DEBUG,
                                ("%s: The requested MAC address is not a multicast address.\r\n", __func__));									LWIP_DEBUGF(NETIF_DEBUG,
                                ("Multicast address add operation failure !!\r\n"));
                        return ERR_ARG;

	}
	if (action == NETIF_ADD_MAC_FILTER) {

		for (i = 0; i < XAXIEMAC_MAX_MAC_ADDR; i++) {
			temp_mask = (0x01) << i;
			if ((xaxiemac_mld6_mcast_entry_mask & temp_mask) == temp_mask) {
				continue;
			}
			entry = i;
			xaxiemac_mld6_mcast_entry_mask |= temp_mask;

			/* Update mac address in hash table */
			xaxiemacif_mld6_mac_hash_update(netif, ip_addr, action,entry);

			LWIP_DEBUGF(NETIF_DEBUG,
					("%s: Multicast MAC address successfully added.\r\n", __func__));

			return ERR_OK;
		}
		LWIP_DEBUGF(NETIF_DEBUG,
			("%s: No multicast address registers left.\r\n", __func__));
		LWIP_DEBUGF(NETIF_DEBUG,
			("Multicast MAC address add operation failure !!\r\n"));
		return ERR_MEM;

	} else if (action == NETIF_DEL_MAC_FILTER) {
		for (i = 0; i < XAXIEMAC_MAX_MAC_ADDR; i++) {
			temp_mask = (0x01) << i;
			if ((xaxiemac_mld6_mcast_entry_mask & temp_mask) == temp_mask) {
				entry = i;
				xaxiemacif_mld6_mac_hash_update(netif, ip_addr,action, entry);

				xaxiemac_mld6_mcast_entry_mask &= (~temp_mask);

				LWIP_DEBUGF(NETIF_DEBUG,
					("%s: Multicast MAC address successfully removed.\r\n", __func__));

				return ERR_OK;
			} else {
				continue;
			}
		}
		LWIP_DEBUGF(NETIF_DEBUG,
			("%s: No multicast address registers present with\r\n", __func__));
		LWIP_DEBUGF(NETIF_DEBUG,
			("the requested Multicast MAC address.\r\n"));
		LWIP_DEBUGF(NETIF_DEBUG,
			("Multicast MAC address removal failure!!.\r\n"));
		return ERR_MEM;
	}
	return ERR_ARG;
}
#endif

#if LWIP_IGMP
static err_t
xaxiemacif_mac_filter_update (struct netif *netif, ip_addr_t *group,
								u8_t action)
{
	err_t return_val = ERR_OK;
	u8_t multicast_mac_addr[6];
	u8_t multicast_mac_addr_to_clr[6];
	u8_t temp_mask;
	int entry;
	int i;
	u8_t * ip_addr_temp  = (u8_t *)group;
	struct xemac_s *xemac = (struct xemac_s *)(netif->state);
	xaxiemacif_s *xaxiemacif = (xaxiemacif_s *)(xemac->state);

	if (action == IGMP_ADD_MAC_FILTER) {
		if ((ip_addr_temp[0] >= 224) && (ip_addr_temp[0] <= 239)) {
			if (xaxiemac_mcast_entry_mask >= 0x0F) {
				LWIP_DEBUGF(NETIF_DEBUG,
				("xaxiemacif_mac_filter_update: No multicast address registers left.\r\n"));
				LWIP_DEBUGF(NETIF_DEBUG,
				("				 Multicast MAC address add operation failure !!\r\n"));

				return_val = ERR_MEM;
			} else {
				for (i = 0; i < 4; i++) {
					temp_mask = (0x01) << i;
					if ((xaxiemac_mcast_entry_mask &
						temp_mask) == temp_mask) {

						continue;

					} else {
						entry = i;
						xaxiemac_mcast_entry_mask
								|= temp_mask;
						multicast_mac_addr[0] = 0x01;
						multicast_mac_addr[1] = 0x00;
						multicast_mac_addr[2] = 0x5E;

						multicast_mac_addr[3] =
							ip_addr_temp[1] & 0x7F;
						multicast_mac_addr[4] =
							ip_addr_temp[2];
						multicast_mac_addr[5] =
							ip_addr_temp[3];

						XAxiEthernet_Stop
						(&xaxiemacif->axi_ethernet);

						XAxiEthernet_MulticastAdd
						(&xaxiemacif->axi_ethernet,
						multicast_mac_addr,entry);

						XAxiEthernet_Start
						(&xaxiemacif->axi_ethernet);

						LWIP_DEBUGF(NETIF_DEBUG,
						("xaxiemacif_mac_filter_update: Multicast MAC address successfully added.\r\n"));

						return_val = ERR_OK;
						break;
					}
				}
				if (i == 4) {
					LWIP_DEBUGF(NETIF_DEBUG,
					("xaxiemacif_mac_filter_update: No multicast address registers left.\r\n"));
					LWIP_DEBUGF(NETIF_DEBUG,
					("				Multicast MAC address add operation failure !!\r\n"));

					return_val = ERR_MEM;
				}
			}
		} else {
			LWIP_DEBUGF(NETIF_DEBUG,
			("xaxiemacif_mac_filter_update: The requested MAC address is not a multicast address.\r\n"));
			LWIP_DEBUGF(NETIF_DEBUG,
			("				 Multicast address add operation failure !!\r\n"));

			return_val = ERR_ARG;
		}
	} else if (action == IGMP_DEL_MAC_FILTER) {
		if ((ip_addr_temp[0] < 224) || (ip_addr_temp[0] > 239)) {

			LWIP_DEBUGF(NETIF_DEBUG,
			("xaxiemacif_mac_filter_update: The requested MAC address is not a multicast address.\r\n"));
			LWIP_DEBUGF(NETIF_DEBUG,
			("				 Multicast address add operation failure !!\r\n"));

			return_val = ERR_ARG;
		} else {
			for (i = 0; i < 4; i++) {
				temp_mask = (0x01) << i;
				if ((xaxiemac_mcast_entry_mask & temp_mask)
								== temp_mask) {
					XAxiEthernet_MulticastGet
					(&xaxiemacif->axi_ethernet,
					multicast_mac_addr_to_clr, i);

					if ((ip_addr_temp[3] ==
					multicast_mac_addr_to_clr[5]) &&
					(ip_addr_temp[2] ==
					multicast_mac_addr_to_clr[4]) &&
					((ip_addr_temp[1] & 0x7f) ==
					multicast_mac_addr_to_clr[3])) {

						XAxiEthernet_Stop
						(&xaxiemacif->axi_ethernet);

						XAxiEthernet_MulticastClear
						(&xaxiemacif->axi_ethernet, i);

						XAxiEthernet_Start
						(&xaxiemacif->axi_ethernet);

						LWIP_DEBUGF(NETIF_DEBUG,
						("xaxiemacif_mac_filter_update: Multicast MAC address successfully removed.\r\n"));

						return_val = ERR_OK;
						xaxiemac_mcast_entry_mask &=
								(~temp_mask);
						break;
					} else {
						continue;
					}
				} else {
					continue;
				}
			}
			if (i == 4) {
				LWIP_DEBUGF(NETIF_DEBUG,
				("xaxiemacif_mac_filter_update: No multicast address registers present with\r\n"));
				LWIP_DEBUGF(NETIF_DEBUG,
				("				 the requested Multicast MAC address.\r\n"));
				LWIP_DEBUGF(NETIF_DEBUG,
				("				 Multicast MAC address removal failure!!.\r\n"));

				return_val = ERR_MEM;
			}
		}
	}
	return return_val;
}
#endif

/*
 * xaxiemacif_init():
 *
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 */

err_t
xaxiemacif_init(struct netif *netif)
{
#if LWIP_SNMP
	/* ifType ethernetCsmacd(6) @see RFC1213 */
	netif->link_type = 6;
	/* your link speed here */
	netif->link_speed = ;
	netif->ts = 0;
	netif->ifinoctets = 0;
	netif->ifinucastpkts = 0;
	netif->ifinnucastpkts = 0;
	netif->ifindiscards = 0;
	netif->ifoutoctets = 0;
	netif->ifoutucastpkts = 0;
	netif->ifoutnucastpkts = 0;
	netif->ifoutdiscards = 0;
#endif

	netif->name[0] = IFNAME0;
	netif->name[1] = IFNAME1;
	netif->output = xaxiemacif_output;
	netif->linkoutput = low_level_output;
#if LWIP_IPV6
	netif->output_ip6 = ethip6_output;
#endif

	low_level_init(netif);

	return ERR_OK;
}
