#!/usr/bin/env python -B
"""
Generates a JS schema file from sandbox/grist/schema.py.
"""

import schema   # pylint: disable=import-error

# These are the types that appear in Grist metadata columns.
_ts_types = {
  "Bool":           "boolean",
  "DateTime":       "number",
  "Int":            "number",
  "PositionNumber": "number",
  "Ref":            "number",
  "RefList":        "[GristObjCode.List, ...number[]]|null",  # Non-primitive values are encoded
  "ChoiceList":     "[GristObjCode.List, ...string[]]|null",
  "Text":           "string",
}

def get_ts_type(col_type):
  col_type = col_type.split(':', 1)[0]      # Strip suffix for Ref:, DateTime:, etc.
  return _ts_types.get(col_type, "CellValue")

def main():
  print("""\
/*** THIS FILE IS AUTO-GENERATED BY %s ***/

import { GristObjCode } from "app/plugin/GristData";

// tslint:disable:object-literal-key-quotes

export const SCHEMA_VERSION = %d;

export const schema = {
""" % ('core/sandbox/gen_js_schema.py', schema.SCHEMA_VERSION))
  # The script name is hardcoded since the Grist sandbox can be
  # at different paths depending on how Grist is installed, and
  # we don't want unnecessary changes to generated files.

  for table in schema.schema_create_actions():
    print('  "%s": {' % table.table_id)
    for column in table.columns:
      print('    %-20s: "%s",' % (column['id'], column['type']))
    print('  },\n')

  print("""};

export interface SchemaTypes {
""")
  for table in schema.schema_create_actions():
    print('  "%s": {' % table.table_id)
    for column in table.columns:
      print('    %s: %s;' % (column['id'], get_ts_type(column['type'])))
    print('  };\n')
  print("}")

if __name__ == '__main__':
  main()
