/*
 * Copyright (c) 2023 OpenLuat & AirM2M
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


#include "common_api.h"
#include "luat_rtos.h"
#include "luat_mem.h"
#include "luat_debug.h"

luat_rtos_task_handle task_handle;
#include "mbedtls/md5.h"
#include "mbedtls/sha1.h"
#include "mbedtls/sha256.h"
#include "mbedtls/base64.h"
#include "mbedtls/des.h"
#include "mbedtls/aes.h"
#include "mbedtls/rsa.h"
#include "luat_crypto.h"

static unsigned char md5_test_buf[8][99] =
	{
		{" "},
		{"123"},
		{"ABC3E333"},
		{"aaaaaa"},
		{"abcasdgasdgf"},
		{"pokhaw;leh;lashdgf;lk1546123"},
		{"zxcbwep;j3827319734lhalsdkv4 asDEFL01DC5A3W231A"},
		{"kc254awZXCDSV142547zcx 35874dasdf3A214354C3BA4SWE84FAWDFASDF0....65DA6SDVGF12MJDASDFasd"}};

static const int md5_test_buflen[8] = {1, 3, 8, 6, 12, 28, 47, 87};

static const unsigned char md5_test_sum[8][16] =
	{
		{0x72, 0x15, 0xee, 0x9c, 0x7d, 0x9d, 0xc2, 0x29, 0xd2, 0x92, 0x1a, 0x40, 0xe8, 0x99, 0xec, 0x5f},
		{0x20, 0x2c, 0xb9, 0x62, 0xac, 0x59, 0x07, 0x5b, 0x96, 0x4b, 0x07, 0x15, 0x2d, 0x23, 0x4b, 0x70},
		{0x99, 0xa6, 0xac, 0x88, 0x0c, 0x73, 0xaa, 0xac, 0x7c, 0xaf, 0x5e, 0xd6, 0x3a, 0x30, 0xdb, 0x3e},
		{0x0b, 0x4e, 0x7a, 0x0e, 0x5f, 0xe8, 0x4a, 0xd3, 0x5f, 0xb5, 0xf9, 0x5b, 0x9c, 0xee, 0xac, 0x79},
		{0x33, 0x57, 0x7e, 0x23, 0x23, 0xf1, 0xc1, 0x12, 0x3e, 0x90, 0xbe, 0xee, 0x7f, 0xdb, 0x43, 0xdf},
		{0x4c, 0xe3, 0xd7, 0x6c, 0xb4, 0xfd, 0xfb, 0xcf, 0xe3, 0x92, 0x1f, 0x76, 0x14, 0x84, 0xd1, 0x5c},
		{0x2d, 0x7f, 0xdb, 0x1e, 0x1f, 0xec, 0xe9, 0xe5, 0x4a, 0xc2, 0x4a, 0x78, 0x92, 0x60, 0x4d, 0x6b},
		{0x9b, 0xdf, 0xc2, 0x7c, 0xfa, 0xa1, 0x59, 0x88, 0x8e, 0xe8, 0xee, 0xf3, 0xf6, 0x24, 0x17, 0xcb}};

void md5_test(void)
{
	unsigned char output[16] = {0};
	//调用封装好的接口
	for (int i = 0; i < 8; i++)
	{
		luat_crypto_md5_simple(md5_test_buf[i], md5_test_buflen[i], output);
		if (memcmp(output, md5_test_sum[i], 16) == 0)
		{
			LUAT_DEBUG_PRINT("md5 test success1 %d", i);
		}
		else
		{
			LUAT_DEBUG_PRINT("md5 test fail1 %d", i);
		}
		memset(output, 0, sizeof(output));
	}

}

static const unsigned char base64_test_decode[88] =
	{
		0x61, 0x69, 0x78, 0x69, 0x61, 0x48, 0x70, 0x34, 0x50, 0x57, 0x4e,
		0x72, 0x62, 0x47, 0x6f, 0x67, 0x4f, 0x32, 0x78, 0x68, 0x63, 0x32,
		0x52, 0x72, 0x5a, 0x69, 0x42, 0x4d, 0x4f, 0x30, 0x46, 0x4c, 0x4c,
		0x43, 0x38, 0x67, 0x62, 0x47, 0x46, 0x73, 0x59, 0x57, 0x78, 0x68,
		0x62, 0x44, 0x45, 0x75, 0x4d, 0x6a, 0x51, 0x7a, 0x4e, 0x54, 0x52,
		0x54, 0x52, 0x45, 0x56, 0x4d, 0x54, 0x6b, 0x74, 0x42, 0x55, 0x30,
		0x51, 0x75, 0x4d, 0x6a, 0x42, 0x68, 0x63, 0x32, 0x52, 0x6d, 0x59,
		0x58, 0x63, 0x7a, 0x4e, 0x47, 0x55, 0x39, 0x50, 0x51, 0x3d, 0x3d};

static const unsigned char base64_test_encode[] = "j,bhzx=cklj ;lasdkf L;AK,/ lalalal1.24354SDELNKASD.20asdfaw34e==";

void base64_test(void)
{
	size_t len;
	unsigned char tmpbuf[128] = {0};
	if (luat_crypto_base64_encode(tmpbuf, sizeof(tmpbuf), &len, base64_test_encode, 64) == 0)
	{
		if (memcmp(tmpbuf, base64_test_decode, 88) == 0)
			LUAT_DEBUG_PRINT("base64 encoding test success");
		else
			LUAT_DEBUG_PRINT("base64 encoding test fail");
	}
	memset(tmpbuf, 0, sizeof(tmpbuf));
	if (luat_crypto_base64_decode(tmpbuf, sizeof(tmpbuf), &len, base64_test_decode, 88) == 0)
	{
		if (memcmp(tmpbuf, base64_test_encode, 64) == 0)
			LUAT_DEBUG_PRINT("base64 decoding test success");
		else
			LUAT_DEBUG_PRINT("base64 decoding test fail");
	}
}

static const unsigned char sha1_test_buf[3][100] = {
	{"abc"},
	{"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"},
	{"j,bchzx=cklj ;lasdkf L;AK,/ lalalal1.24354SDELNKASD.20asdfaw34eDBZ D2SALCHDAKCDASDEFRT=="}};

static const size_t sha1_test_buflen[3] = {3, 56, 88};

static const unsigned char sha1_test_sum[3][20] =
	{
		{0xA9, 0x99, 0x3E, 0x36, 0x47, 0x06, 0x81, 0x6A, 0xBA, 0x3E,
		 0x25, 0x71, 0x78, 0x50, 0xC2, 0x6C, 0x9C, 0xD0, 0xD8, 0x9D},
		{0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E, 0xBA, 0xAE,
		 0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5, 0xE5, 0x46, 0x70, 0xF1},
		{0x56, 0x0a, 0x8c, 0x99, 0xc3, 0x25, 0x8f, 0x07, 0x14, 0xf1,
		 0xd3, 0x2e, 0x11, 0x2a, 0x6c, 0x9c, 0x98, 0x6c, 0x8d, 0x9b}};
void sha1_test(void)
{
	unsigned char output[20] = {0};
	//调用封装好的接口
	for (int i = 0; i < 3; i++)
	{
		luat_crypto_sha1_simple(sha1_test_buf[i], sha1_test_buflen[i], output);
		if (memcmp(output, sha1_test_sum[i], 20) == 0)
		{
			LUAT_DEBUG_PRINT("sha1 test success1 %d", i);
		}
		else
		{
			LUAT_DEBUG_PRINT("sha1 test fail1 %d", i);
		}
		memset(output, 0, sizeof(output));
	}
}

static const unsigned char sha256_test_buf[2][57] = {
	{"abc"},
	{"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"},
};

static const size_t sha256_test_buflen[2] = {3, 56};

static const unsigned char sha256_224_test_sum[6][32] =
	{
		/*
		 * SHA-224 test vectors
		 */
		{0x23, 0x09, 0x7D, 0x22, 0x34, 0x05, 0xD8, 0x22,
		 0x86, 0x42, 0xA4, 0x77, 0xBD, 0xA2, 0x55, 0xB3,
		 0x2A, 0xAD, 0xBC, 0xE4, 0xBD, 0xA0, 0xB3, 0xF7,
		 0xE3, 0x6C, 0x9D, 0xA7},
		{0x75, 0x38, 0x8B, 0x16, 0x51, 0x27, 0x76, 0xCC,
		 0x5D, 0xBA, 0x5D, 0xA1, 0xFD, 0x89, 0x01, 0x50,
		 0xB0, 0xC6, 0x45, 0x5C, 0xB4, 0xF5, 0x8B, 0x19,
		 0x52, 0x52, 0x25, 0x25},

		/*
		 * SHA-256 test vectors
		 */
		{0xBA, 0x78, 0x16, 0xBF, 0x8F, 0x01, 0xCF, 0xEA,
		 0x41, 0x41, 0x40, 0xDE, 0x5D, 0xAE, 0x22, 0x23,
		 0xB0, 0x03, 0x61, 0xA3, 0x96, 0x17, 0x7A, 0x9C,
		 0xB4, 0x10, 0xFF, 0x61, 0xF2, 0x00, 0x15, 0xAD},
		{0x24, 0x8D, 0x6A, 0x61, 0xD2, 0x06, 0x38, 0xB8,
		 0xE5, 0xC0, 0x26, 0x93, 0x0C, 0x3E, 0x60, 0x39,
		 0xA3, 0x3C, 0xE4, 0x59, 0x64, 0xFF, 0x21, 0x67,
		 0xF6, 0xEC, 0xED, 0xD4, 0x19, 0xDB, 0x06, 0xC1},
};

static const unsigned char sha224_1000a[32] = {0x20, 0x79, 0x46, 0x55, 0x98, 0x0C, 0x91, 0xD8,
											   0xBB, 0xB4, 0xC1, 0xEA, 0x97, 0x61, 0x8A, 0x4B,
											   0xF0, 0x3F, 0x42, 0x58, 0x19, 0x48, 0xB2, 0xEE,
											   0x4E, 0xE7, 0xAD, 0x67};

static const unsigned char sha256_1000a[32] = {0xCD, 0xC7, 0x6E, 0x5C, 0x99, 0x14, 0xFB, 0x92,
											   0x81, 0xA1, 0xC7, 0xE2, 0x84, 0xD7, 0x3E, 0x67,
											   0xF1, 0x80, 0x9A, 0x48, 0xA4, 0x97, 0x20, 0x0E,
											   0x04, 0x6D, 0x39, 0xCC, 0xC7, 0x11, 0x2C, 0xD0};

void sha256_test(void)
{
	unsigned char output[32] = {0};
	for (int i = 0; i < 2; i++)
	{
		//sha256
		luat_crypto_sha256_simple(sha256_test_buf[i], sha256_test_buflen[i], output);
		if (memcmp(output, sha256_224_test_sum[i + 2], 32) == 0)
			LUAT_DEBUG_PRINT("sha256 test success1 %d", i);
		else
			LUAT_DEBUG_PRINT("sha256 test fail1 %d", i);
		memset(output, 0, 32);
	}
}




void trng_test()
{
    uint8_t *buff = NULL;
    buff = malloc(24 * 100);
    for (size_t i = 0; i < 100; i++)
    {
        luat_crypto_trng(buff + i*24, 24);
    }
	LUAT_DEBUG_PRINT("The Trng string is [%u]", buff);
}

static void demo_init_crypto()
{
	md5_test();
	base64_test();
	sha1_test();
	sha256_test();
	trng_test();
}

static void task(void *param)
{
	while(1)
	{
		luat_rtos_task_sleep(1000);
		LUAT_DEBUG_PRINT("==================crypto is running==================");
		demo_init_crypto();
		LUAT_DEBUG_PRINT("==================crypto is done==================");
	}
}


static void task_demoE_init(void)
{
	luat_rtos_task_create(&task_handle, 2*1024, 50, "task", task, NULL, 0);
}

//启动task_demoE_init，启动位置任务1级
INIT_TASK_EXPORT(task_demoE_init, "1");
