(ns metabase.query-processor.middleware.enterprise
  "Wrappers for enterprise-only QP middleware using [[defenterprise]]. Pre-processing and post-processing middleware can
  use [[defenterprise]] directly, since the top-level function is applied directly each during each QP run, meaning it
  gets the chance to dispatch correctly every time it is run; 'around' middleware (including 'execution' middleware)
  needs a helper function that invokes the [[defenterprise]] function during every QP run, rather than just once when
  all middleware is combined. See [[handle-audit-app-internal-queries]]
  and [[handle-audit-app-internal-queries-middleware]] for example."
  (:require
   [metabase.premium-features.core :refer [defenterprise]]
   [metabase.query-processor.error-type :as qp.error-type]
   [metabase.query-processor.schema :as qp.schema]
   [metabase.util.i18n :as i18n]
   [metabase.util.malli :as mu]))

;;;; Pre-processing middleware

;;; (f query) => query

(defenterprise attach-destination-db-middleware
  "Pre-processing middleware. Calculates the destination database that should be used for this query, e.g. for caching
  purposes. Does not make any changes to the query besides (possibly) adding a `:destination-database/id` key."
  metabase-enterprise.database-routing.middleware
  [query]
  query)

(defenterprise apply-sandboxing
  "Pre-processing middleware. Replaces source tables a User was querying against with source queries that (presumably)
  restrict the rows returned, based on presence of sandboxes."
  metabase-enterprise.sandbox.query-processor.middleware.row-level-restrictions
  [query]
  query)

(defenterprise apply-impersonation
  "Pre-processing middleware. Adds a key to the query if the user will be impersonated. Currently used solely for cache."
  metabase-enterprise.impersonation.middleware
  [query]
  query)

(defenterprise apply-impersonation-postprocessing
  "Post-processing middleware to actually bind what's needed for impersonation"
  metabase-enterprise.impersonation.middleware
  [query]
  query)

(defn apply-impersonation-postprocessing-middleware
  "Helper middleware wrapper for [[apply-impersonation-postprocessing]] to make sure we do [[defenterprise]] dispatch
  correctly on each QP run rather than just once when we combine all of the QP middleware"
  [qp]
  (fn [query rff]
    ((apply-impersonation-postprocessing qp) query rff)))

(defenterprise apply-download-limit
  "Pre-processing middleware to apply row limits to MBQL export queries if the user has `limited` download perms. This
  does not apply to native queries, which are instead limited by the [[limit-download-result-rows]] post-processing
  middleware."
  metabase-enterprise.advanced-permissions.query-processor.middleware.permissions
  [query]
  query)

;;;; Execution middleware

;;; (f qp) => qp

(defenterprise swap-destination-db
  "Must be the last middleware before we actually hit the database. If a Router Database is specified, swaps out the
   Metadata Provider for one that has the appropriate destination database."
  metabase-enterprise.database-routing.middleware
  [qp]
  qp)

(mu/defn swap-destination-db-middleware :- ::qp.schema/qp
  "Helper middleware wrapper for [[swap-destination-db]] to make sure we do [[defenterprise]] dispatch correctly on each QP run rather than just once when we combine all of the QP middleware"
  [qp :- ::qp.schema/qp]
  (fn [query rff]
    ((swap-destination-db qp) query rff)))

(defenterprise check-download-permissions
  "Middleware for queries that generate downloads, which checks that the user has permissions to download the results
  of the query, and aborts the query or limits the number of results if necessary.

  If this query is not run to generate an export (e.g. :export-format is :api) we return user's download permissions in
  the query metadata so that the frontend can determine whether to show the download option on the UI."
  metabase-enterprise.advanced-permissions.query-processor.middleware.permissions
  [qp]
  qp)

(defn check-download-permissions-middleware
  "Helper middleware wrapper for [[check-download-permissions]] to make sure we do [[defenterprise]] dispatch
  correctly on each QP run rather than just once when we combine all of the QP middleware."
  [qp]
  (fn [query rff]
    ((check-download-permissions qp) query rff)))

(defenterprise maybe-apply-column-level-perms-check
  "Execution middleware. Check column-level permissions if applicable."
  metabase-enterprise.sandbox.query-processor.middleware.column-level-perms-check
  [qp]
  qp)

(defn maybe-apply-column-level-perms-check-middleware
  "Helper middleware wrapper for [[maybe-apply-column-level-perms-check]] to make sure we do [[defenterprise]] dispatch
  correctly on each QP run rather than just once when we combine all of the QP middleware."
  [qp]
  (fn [query rff]
    ((maybe-apply-column-level-perms-check qp) query rff)))

;;;; Post-processing middleware

;;; (f query rff) => rff

(defenterprise limit-download-result-rows
  "Post-processing middleware to limit the number of rows included in downloads if the user has `limited` download
  perms. Mainly useful for native queries, which are not modified by the [[apply-download-limit]] pre-processing
  middleware."
  metabase-enterprise.advanced-permissions.query-processor.middleware.permissions
  [_query rff]
  rff)

(defenterprise merge-sandboxing-metadata
  "Post-processing middleware. Merges in column metadata from the original, unsandboxed version of the query."
  metabase-enterprise.sandbox.query-processor.middleware.row-level-restrictions
  [_query rff]
  rff)

;;;; Around middleware

;;; (f qp) => qp

(defenterprise handle-audit-app-internal-queries
  "'Around' middleware that handles `:internal` (Audit App) type queries."
  metabase-enterprise.audit-app.query-processor.middleware.handle-audit-queries
  [qp]
  (fn [{query-type :type, :as query} rff]
    (when (= (keyword query-type) :internal)
      (throw (ex-info (i18n/tru "Audit App queries are not enabled on this instance.")
                      {:type qp.error-type/invalid-query})))
    (qp query rff)))

(defn handle-audit-app-internal-queries-middleware
  "Helper middleware wrapper for [[handle-audit-app-internal-queries]] to make sure we do [[defenterprise]] dispatch
  correctly on each QP run rather than just once when we combine all of the QP middleware."
  [qp]
  (fn [query rff]
    ((handle-audit-app-internal-queries qp) query rff)))
