(ns metabase.notification.payload.impl.card
  (:require
   [metabase.channel.render.core :as channel.render]
   [metabase.events.core :as events]
   [metabase.notification.models :as models.notification]
   [metabase.notification.payload.core :as notification.payload]
   [metabase.notification.payload.execute :as notification.execute]
   [metabase.notification.payload.impl.dashboard :as notification.dashboard]
   [metabase.notification.send :as notification.send]
   [metabase.util.log :as log]
   [metabase.util.malli :as mu]
   [metabase.util.ui-logic :as ui-logic]
   [toucan2.core :as t2]))

(mu/defmethod notification.payload/payload :notification/card
  [{:keys [creator_id payload subscriptions] :as _notification-info} :- ::notification.payload/Notification]
  (log/with-context {:card_id (:card_id payload)}
    (let [card-id     (:card_id payload)
          part        (notification.execute/execute-card creator_id card-id)
          card-result (:result part)]
      (when (not= :completed (:status card-result))
        (throw (ex-info (format "Failed to execute card with error: %s" (:error card-result))
                        {:card_id card-id
                         :status (:status card-result)
                         :error  (:error card-result)})))
      {:card_part        part
       :card             (t2/select-one :model/Card card-id)
       :style            {:color_text_dark   channel.render/color-text-dark
                          :color_text_light  channel.render/color-text-light
                          :color_text_medium channel.render/color-text-medium}
       :notification_card payload
       :subscriptions     subscriptions})))

(defn- goal-met? [{:keys [send_condition], :as notification_card} card_part]
  (let [goal-comparison      (if (= :goal_above (keyword send_condition)) >= <)
        goal-val             (ui-logic/find-goal-value card_part)
        comparison-col-rowfn (ui-logic/make-goal-comparison-rowfn (:card card_part)
                                                                  (get-in card_part [:result :data]))]

    (when-not (and goal-val comparison-col-rowfn)
      (throw (ex-info "Unable to compare results to goal for notificationt_card"
                      {:notification_card  notification_card
                       :result card_part})))
    (boolean
     (some (fn [row]
             (goal-comparison (comparison-col-rowfn row) goal-val))
           (get-in card_part [:result :data :rows])))))

(mu/defmethod notification.payload/skip-reason :notification/card
  [{:keys [payload]}]
  (let [{:keys [notification_card card_part]} payload
        send-condition                        (:send_condition notification_card)]
    (cond
      (-> notification_card :card :archived true?)
      :archived

      (= :has_result send-condition)
      (when (notification.execute/is-card-empty? card_part)
        :empty)

      (#{:goal_above :goal_below} send-condition)
      (when (not (goal-met? notification_card card_part))
        :goal-not-met)

      :else
      (let [^String error-text (format "Unrecognized alert with condition '%s'" send-condition)]
        (throw (IllegalArgumentException. error-text))))))

(mu/defmethod notification.send/do-after-notification-sent :notification/card
  [{:keys [id creator_id handlers] :as notification-info} :- ::models.notification/FullyHydratedNotification
   notification-payload
   skipped?]
  (when (and (-> notification-info :payload :send_once)
             (not skipped?))
    (log/info "Archiving due to send_once")
    (t2/update! :model/Notification (:id notification-info) {:active false}))
  (try
    (when-let [rows (-> notification-payload :payload :card_part :result :data :rows)]
      (notification.payload/cleanup! rows))
    (catch Exception e
      (log/warn e "Error cleaning up temp files for notification" id)))
  (when-not skipped?
    (events/publish-event! :event/alert-send
                           {:id      id
                            :user-id creator_id
                            :object  {:recipients (notification.dashboard/handlers->audit-recipients handlers)
                                      :filters    (-> notification-info :alert :parameters)}})))
