import type {
  ApiKey,
  ApiKeyId,
  CreateApiKeyRequest,
  CreateApiKeyResponse,
  RegenerateApiKeyResponse,
  UpdateApiKeyRequest,
  UpdateApiKeyResponse,
} from "metabase-types/api/admin";

import { Api } from "./api";
import { idTag, invalidateTags, listTag, provideApiKeyListTags } from "./tags";

export const apiKeyApi = Api.injectEndpoints({
  endpoints: (builder) => ({
    listApiKeys: builder.query<ApiKey[], void>({
      query: () => `/api/api-key`,
      providesTags: (apiKeys = []) => provideApiKeyListTags(apiKeys),
    }),
    countApiKeys: builder.query<number, void>({
      query: () => `/api/api-key/count`,
      providesTags: provideApiKeyListTags([]),
    }),
    createApiKey: builder.mutation<CreateApiKeyResponse, CreateApiKeyRequest>({
      query: (body) => ({
        method: "POST",
        url: `/api/api-key`,
        body,
      }),
      invalidatesTags: (_, error) =>
        invalidateTags(error, [listTag("api-key")]),
    }),
    updateApiKey: builder.mutation<UpdateApiKeyResponse, UpdateApiKeyRequest>({
      query: ({ id, ...body }) => ({
        method: "PUT",
        url: `/api/api-key/${id}`,
        body,
      }),
      invalidatesTags: (_, error, { id }) =>
        invalidateTags(error, [listTag("api-key"), idTag("api-key", id)]),
    }),
    deleteApiKey: builder.mutation<void, ApiKeyId>({
      query: (id) => ({ method: "DELETE", url: `/api/api-key/${id}` }),
      invalidatesTags: (_, error, id) =>
        invalidateTags(error, [listTag("api-key"), idTag("api-key", id)]),
    }),
    regenerateApiKey: builder.mutation<RegenerateApiKeyResponse, ApiKeyId>({
      query: (id) => ({ method: "PUT", url: `/api/api-key/${id}/regenerate` }),
      invalidatesTags: (_, error, id) =>
        invalidateTags(error, [listTag("api-key"), idTag("api-key", id)]),
    }),
  }),
});

export const {
  useListApiKeysQuery,
  useCountApiKeysQuery,
  useCreateApiKeyMutation,
  useRegenerateApiKeyMutation,
  useUpdateApiKeyMutation,
  useDeleteApiKeyMutation,
} = apiKeyApi;
