const forge = require('node-forge');

/**
 * This function is used to generate the serial number
 * @param {string} hexString
 * @returns
 */
function toPositiveHex (hexString) {
  var mostSiginficativeHexAsInt = parseInt(hexString[0], 16);
  if (mostSiginficativeHexAsInt < 8) {
    return hexString;
  }

  mostSiginficativeHexAsInt -= 8;
  return mostSiginficativeHexAsInt.toString() + hexString.substring(1);
}

/**
 * Thus function is used to generate certificates using the postman root ca
 * @param {Object} attr
 * @param {Object} options
 * @param {Function} callback
 */
const generateCert = (attr, options, callback) => {
  try {
    const privateCAKey = forge.pki.privateKeyFromPem(options.keyPair.privateKey);
    const keys = forge.pki.rsa.generateKeyPair(options.keySize);
    const cert = forge.pki.createCertificate();

    const caCrt = forge.pki.certificateFromPem(options.keyPair.publicKey);

    cert.publicKey = keys.publicKey;

    cert.serialNumber = toPositiveHex(forge.util.bytesToHex(forge.random.getBytesSync(9)));
    cert.validity.notBefore = new Date();
    cert.validity.notAfter = new Date();
    cert.validity.notAfter.setDate(cert.validity.notBefore.getDate() + options.days);

    cert.setSubject(attr);
    cert.setIssuer(caCrt.subject.attributes);
    cert.setExtensions(options.extensions);
    cert.sign(privateCAKey, forge.md.sha256.create());

    const pem = {
      private: forge.pki.privateKeyToPem(keys.privateKey),
      public: forge.pki.publicKeyToPem(keys.publicKey),
      cert: forge.pki.certificateToPem(cert)
    };
    return callback(null, pem);
  }
  catch (err) {
    return callback(err);
  }

};

/**
 * This function generates the root certificate
 *
 * @param {Object} attr
 * @param {Object} options
 * @param {Function} callback
 */
const generateRootCa = (attrs, options, callback) => {
  try {
    const keys = forge.pki.rsa.generateKeyPair(options.keySize);
    const cert = forge.pki.createCertificate();
    cert.publicKey = keys.publicKey;
    cert.serialNumber = toPositiveHex(forge.util.bytesToHex(forge.random.getBytesSync(9)));
    cert.validity.notBefore = new Date();
    cert.validity.notAfter = new Date();
    cert.validity.notAfter.setDate(cert.validity.notBefore.getDate() + options.days);

    cert.setSubject(attrs);
    cert.setIssuer(attrs);
    cert.setExtensions(options.extensions);

    cert.sign(keys.privateKey, forge.md.sha256.create());
    const pem = {
      private: forge.pki.privateKeyToPem(keys.privateKey),
      public: forge.pki.publicKeyToPem(keys.publicKey),
      cert: forge.pki.certificateToPem(cert)
    };
    return callback(null, pem);
  }
  catch (err) {
    return callback(err);
  }
};

/**
 * This function provides the certificate information.
 *
 * @param {Object} cert
 * @param {Function} callback
 */
const getCertInfo = (cert, callback) => {
  try {
    const certDetails = forge.pki.certificateFromPem(cert);
    return callback(null, certDetails);
  }
  catch (err) {
    return callback(err);
  }
};

module.exports = { generateCert, generateRootCa, getCertInfo };
