const { exec } = require('child_process'),
      path = require('path'),
      MKDIR_DIRECTORY_FAILED = 'MKDIR_DIRECTORY_FAILED',
      INVALID_ARGUMENT_ERROR = 'INVALID_ARGUMENT_ERROR',
      EXTRACTION_FAILED = 'EXTRACTION_FAILED',
      EXTRACTION_SUCCESSFUL = 'EXTRACTION_SUCCESSFUL';

/**
 * @typedef {downloadOutputInfo} extractInputData
 * The below object defines the input object for this module for extraction.
 *
 * @example
 * {
 *   eID: 'pack-xxxx',
 *   packageName: 'pack',
 *   downloadURL: 'http://sample.com/foo.tar.gz',
 *   downloadChecksum: '2927b7c6ed350687446b7eba58bd74535afd411b'
 *   downloadChecksumAlgorith: 'sha1',
 *   downloadInfo: {
 *     status: 'DOWNLOAD_COMPLETED',
 *     error: null,
 *     downloadDirectory: 'User/PathForDownloadDirectory'
 *   }
 * }
 */

/**
 * @typedef {String} extractStatus
 *
 * They holds the value of either one of
 * 'MKDIR_DIRECTORY_FAILED','INVALID_ARGUMENT_ERROR','EXTRACTION_FAILED','EXTRACTION_SUCCESSFUL'
 *
 */

/**
 * @typedef extractInfo
 * The below object holds key properties for defining the extracted information
 *
 * @property {extractStatus} status The status of the extraction process.
 * @property {?Error} [error] Will be null initially, initialized to the error in case of failures
 * @property {String} [extractDirectory] Path where the .tar file is extracted
 *
 * @example
 * {
 *   status: 'EXTRACTION_SUCCESSFUL',
 *   error: null,
 *   extractDirectory: 'User/PathForExtractedDirectory'
 * }
 */

/**
 * @typedef extractOutputData
 * The below object defines all the properties present in the return value
 *
 * @extends extractInputData
 * @property {extractInfo} extractInfo
 *
 * @example
 * {
 *   eID: 'pack-xxxx',
 *   packageName: 'pack',
 *   downloadURL: 'http://sample.com/foo.tar.gz',
 *   downloadChecksum: '2927b7c6ed350687446b7eba58bd74535afd411b'
 *   downloadChecksumAlgorithm: 'sha1',
 *   downloadInfo: {
 *     status: 'DOWNLOAD_COMPLETED',
 *     error: null,
 *     downloadDirectory: 'User/PathForDownloadDirectory'
 *   },
 *   extractInfo: {
 *     status: 'EXTRACTION_SUCCESSFUL',
 *     error: null,
 *     extractDirectory: 'User/PathForExtractedDirectory'
 *   }
 * }
 *
 */

/**
 * @callback responseCallback
 * @param {?Error} error
 * @param {extractOutputData} data
 */

/**
 * @method extract
 * @description Used to extract the .tar file downloaded in the desired directory
 * @param  {extractInputData} extractInputData
 * @param  {responseCallback} callback
 * @throws {InvalidParamsException}
 */
module.exports = function extract(extractInputData, callback) {
  // Bail out if invoked without callback
  if (!callback) {
    throw new Error('InvalidParamsException: Hermes~extract - callback should not be empty');
  }

  // Bail out if callback is not a function
  if (typeof callback !== 'function') {
    throw new Error('InvalidParamsException: Hermes~extract - callback should be of type function');
  }

  const extractOutputData = Object.assign({}, extractInputData, {
    extractInfo: {
      status: null,
      error: null,
      extractDirectory: null,
    },
  });

  // Bail out if invoked without extractInputData
  if (!extractInputData) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData should not be empty or false');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData is not a object
  if (typeof extractInputData !== 'object' || Array.isArray(extractInputData)) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData should be of type object');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if invoked without extractInputData.eID
  if (!extractInputData.eID) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.eId should not be empty');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData.eID is not of type string.
  if (typeof extractInputData.eID !== 'string') {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.eId should be of type string');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if invoked without extractInputData.packageName
  if (!extractInputData.packageName) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.packageName should not be empty');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData.packageName is not of type string.
  if (typeof extractInputData.packageName !== 'string') {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.packageName should be of type string');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if invoked without extractInputData.downloadInfo
  if (!extractInputData.downloadInfo) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo should not be empty');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData.downloadInfo is not a object
  if (typeof extractInputData.downloadInfo !== 'object' || Array.isArray(extractInputData.downloadInfo)) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo should be of type object');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if invoked without extractInputData.downloadInfo.downloadDirectory
  if (!extractInputData.downloadInfo.downloadDirectory) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo.downloadDirectory should not be empty');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData.downloadInfo.downloadDirectory is not a string
  if (typeof extractInputData.downloadInfo.downloadDirectory !== 'string') {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo.downloadDirectory should be a string');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if invoked without extractInputData.downloadInfo.status
  if (!extractInputData.downloadInfo.status) {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo.status should not be empty');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  // Bail out if extractInputData.downloadInfo.status is not equal to 'DOWNLOAD_COMPLETED'
  if (extractInputData.downloadInfo.status !== 'DOWNLOAD_COMPLETED') {
    const err = new Error('InvalidParamsException: Hermes~extract - extractInputData.downloadInfo.status should be equal to DOWNLOAD_COMPLETED');
    extractOutputData.extractInfo = { status: INVALID_ARGUMENT_ERROR, error: err };
    return callback(err, extractOutputData);
  }

  const filePath = extractInputData.downloadInfo.downloadDirectory,
        { eID } = extractInputData,
        packName = extractInputData.packageName;

  // should create a new directory in the filePath
  // (extractInputData.downloadInfo.downloadDirectory)
  exec(`mkdir "${path.resolve(filePath, eID)}"`, (mkdirError) => {
    if (mkdirError) {
      extractOutputData.extractInfo = {
        status: MKDIR_DIRECTORY_FAILED,
        error: mkdirError,
        extractDirectory: filePath,
      };
      return callback(mkdirError, extractOutputData);
    }

    // should extract the downloaded file in the directory made in the above exec command
    // xzf are arguments to the tar command:
    // x: --extract (extract files from an archive)
    // z: --gzip
    // f: --file use archive file
    // --strip-components=NUMBER: strip NUMBER leading components from file names on extraction

    exec(`tar -xzf "${path.resolve(filePath, `${eID}.tar.gz`)}" -C "${path.resolve(filePath, eID)}" --strip-components=2 "${packName}/app/"`, { maxBuffer: Infinity }, (extractError, stdout, stderr) => {
      if (extractError) {
        extractOutputData.extractInfo = {
          status: EXTRACTION_FAILED,
          error: extractError,
          extractDirectory: filePath,
          standardOutput: stdout,
          standardError: stderr,
        };
        return callback(extractError, extractOutputData);
      }

      // reached this state when extraction is successfully done
      extractOutputData.extractInfo = {
        status: EXTRACTION_SUCCESSFUL,
        error: null,
        extractDirectory: filePath,
        standardOutput: stdout,
        standardError: stderr,
      };
      return callback(null, extractOutputData);
    });
  });
};
