const hermes = require('@postman/hermes'),
    { EventEmitter } = require('events');

/**
 * @class LinuxAutoUpdater
 * @extends EventEmitter
 * @description It orchestrate the `@postman/hermes` for the linux auto update.
 */
class LinuxAutoUpdater extends EventEmitter {
    /**
     * @method constructor
     * @param {Object} options
     * @param {Function} options.quitApp Function to quit the app
     * @param {Function} options.logger Function to push the logs from the module
     * @param {Function} options.appInstallationPath Function to get app installation path
     * @param {Function} options.appName Function to get app name
     */
    constructor (options) {
        super();
        this.quitApp = options.quitApp;
        this.logger = options.logger;
        this.updateInfo = null;
        this.originalFs = options.originalFs;

        this.cleanUp({
            appInstallationPath: options.appInstallationPath,
            appName: options.appName
        }, (err) => {
            if (err) {
                this.logger.error('LinuxAutoUpdater~Cleanup - Error in initial cleanup');
            }
            else {
                this.logger.info('LinuxAutoUpdater~Cleanup - Initial cleanup successful');
            }
        });
    }

    /**
     * @method downloadUpdate
     * @param {String} appName The appName for which update need to perform.
     * @param {String} downloadURL The downloadURL of the latest version app.
     * @param {String} downloadVersion The latest version to be downloaded.
     * @param {String} downloadDirectory The directory where the latest app will be downloaded
     * @fires LinuxAutoUpdater#event:update-downloaded
     * @fires LinuxAutoUpdater#event:error
     *
     * In ideal case, the flow would be.
     * In any case of failure, it emits error with the error and the info it holds.
     *   hermes.clear
     *         |
     *         |
     *   hermes.download
     *         |
     *         |
     *   hermes.extract
     *         |
     *         |
     *  this.emit('update-downloaded)
     */
    downloadUpdate (appName, downloadURL, downloadVersion, downloadDirectory) {
        this.updateInfo = hermes.init({ packageName: appName });

        // Cleanup before downloading
        this.cleanUp({
            appInstallationPath: downloadDirectory,
            appName: appName
        }, (err) => {
            if (err) {
                this.logger.error('LinuxAutoUpdater~Cleanup - Error in cleanup before download');
            }

            // Download & Extract
            let inputDownloadData = Object.assign({}, this.updateInfo, { downloadURL, downloadDirectory });

            hermes.download(inputDownloadData, (err, downloadInfo) => {
                this.updateInfo = downloadInfo;
                if (err) {
                    this.emit('error', err, downloadInfo); // downloadInfo will hold the error info as well
                    return;
                }

                let event = {},
                    notes = '',
                    date = '';

                hermes.extract(downloadInfo, (extractError, extractInfo) => {
                    this.updateInfo = extractInfo;
                    if (extractError) {
                        this.emit('error', extractError, extractInfo); // ExtractInfo will have the error info as well.
                        return;
                    }

                    // Adding logs to capture the standard output of child process created inside hermes
                    this.logger.info('LinuxAutoUpdater~Extraction completed', extractInfo);

                    // Reaching here means, the file is downloaded and extracted
                    // For consistency with windows and mac, we are emitting this event at the end.

                    // appName = 'Postman'
                    // downloadVersion ='x.y.z'
                    // releaseName = 'Postman x.y.z'
                    const releaseName = appName + ' ' + downloadVersion;
                    this.emit('update-downloaded', event, notes, releaseName, date, downloadURL);
                });
            });
        });

    }

    /**
     * @method cleanUp
     * @description Calls the hermes to clear the older downloads and files
     **/
    cleanUp (options, cb) { // eslint-disable-line class-methods-use-this

        // eslint-disable-next-line no-unreachable
        if (!options || !options.appInstallationPath || !options.appName) {
            this.logger.info('LinuxAutoUpdater~cleanUp: Missing parameters, bailing out', options);
            // eslint-disable-next-line no-useless-return
            return cb(new Error('LinuxAutoUpdater~cleanUp: Missing parameters, bailing out'));
        }

        let clearInputData = {
            appName: options.appName,
            directory: options.appInstallationPath,
            fileSystemModule: this.originalFs
        };

        hermes.clear(clearInputData, (err) => {
            if (err) {
                this.logger.error('LinuxAutoUpdater~cleanUp - Error in cleanup old downloads', err);
            }
            cb(err);
        });
    }

    /**
     * @method quitAndInstall
     * @param {Object} canRestart
     * @fires LinuxAutoUpdater#event:error
     *
     * It calls the `hermes.swap` module with the updateInfo & canRestart flag it holds and quits the app.
     * Incase of any error before quitting the app, it emits the `error` and won't quit the app.
     */
    quitAndInstall (canRestart) {
        this.logger.info('LinuxAutoUpdater~quitAndInstall - Initiated');
        hermes.swap(Object.assign({}, this.updateInfo, canRestart), (err, extractInfo) => {
            if (err) {
                this.emit('error', extractInfo);
                return;
            }
            this.updateInfo = extractInfo;
            this.logger.info('LinuxAutoUpdater~quitAndInstall - Quitting for update');
            this.quitApp();
        });
    }
}

module.exports = LinuxAutoUpdater;
