import type { FastifyReply, FastifyRequest } from 'fastify';
import { checkJSONPQuery, sendJSONResponse } from './json.js';
import { errorText } from './errors.js';

type CallbackResult = object | number;

/**
 * Handle JSON API response generated by a callback
 */
export function handleJSONResponse(
	req: FastifyRequest,
	res: FastifyReply,
	callback: (query: Record<string, string>) => CallbackResult | Promise<CallbackResult>
) {
	const q = (req.query || {}) as Record<string, string>;

	// Check for JSONP
	const wrap = checkJSONPQuery(q);
	if (!wrap) {
		// Invalid JSONP callback
		res.code(400).send(errorText(400));
		return;
	}

	// Function to send response
	const respond = (result: CallbackResult) => {
		if (typeof result === 'number') {
			res.code(result).send(errorText(result));
		} else {
			sendJSONResponse(result, q, wrap, res);
		}
	};

	// Get result
	const result = callback(q);
	if (result instanceof Promise) {
		result.then(respond).catch((err) => {
			console.error(err);
			respond(500);
		});
	} else {
		respond(result);
	}
}
